/*
Copyright 2021.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package main

import (
	"flag"
	"os"

	// Import all Kubernetes client auth plugins (e.g. Azure, GCP, OIDC, etc.)
	// to ensure that exec-entrypoint and run can make use of them.
	_ "k8s.io/client-go/plugin/pkg/client/auth"

	buildv1 "github.com/openshift/api/build/v1"
	"k8s.io/apimachinery/pkg/runtime"
	utilruntime "k8s.io/apimachinery/pkg/util/runtime"
	clientgoscheme "k8s.io/client-go/kubernetes/scheme"

	imagev1 "github.com/openshift/api/image/v1"

	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/healthz"
	"sigs.k8s.io/controller-runtime/pkg/log/zap"

	"github.com/quay/quay-bridge-operator/pkg/constants"
	"github.com/quay/quay-bridge-operator/pkg/core"

	quayv1 "github.com/quay/quay-bridge-operator/api/v1"
	"github.com/quay/quay-bridge-operator/controllers"
	quaywebhook "github.com/quay/quay-bridge-operator/pkg/webhook"
	"github.com/redhat-cop/operator-utils/pkg/util"
	"k8s.io/apimachinery/pkg/types"
	"sigs.k8s.io/controller-runtime/pkg/webhook"
	//+kubebuilder:scaffold:imports
)

var (
	scheme   = runtime.NewScheme()
	setupLog = ctrl.Log.WithName("setup")
)

func init() {
	utilruntime.Must(clientgoscheme.AddToScheme(scheme))

	utilruntime.Must(quayv1.AddToScheme(scheme))
	utilruntime.Must(buildv1.AddToScheme(scheme))
	utilruntime.Must(imagev1.AddToScheme(scheme))
	//+kubebuilder:scaffold:scheme
}

func main() {
	var metricsAddr string
	var enableLeaderElection bool
	var probeAddr string
	flag.StringVar(&metricsAddr, "metrics-bind-address", ":8080", "The address the metric endpoint binds to.")
	flag.StringVar(&probeAddr, "health-probe-bind-address", ":8081", "The address the probe endpoint binds to.")
	flag.BoolVar(&enableLeaderElection, "leader-elect", false,
		"Enable leader election for controller manager. "+
			"Enabling this will ensure there is only one active controller manager.")
	opts := zap.Options{
		Development: true,
	}
	opts.BindFlags(flag.CommandLine)
	flag.Parse()

	ctrl.SetLogger(zap.New(zap.UseFlagOptions(&opts)))

	mgr, err := ctrl.NewManager(ctrl.GetConfigOrDie(), ctrl.Options{
		Scheme:                     scheme,
		MetricsBindAddress:         metricsAddr,
		Port:                       9443,
		HealthProbeBindAddress:     probeAddr,
		LeaderElection:             enableLeaderElection,
		LeaderElectionResourceLock: "configmaps",
		LeaderElectionID:           "0111fb36.redhat.com",
	})
	if err != nil {
		setupLog.Error(err, "unable to start manager")
		os.Exit(1)
	}

	if err = (&controllers.QuayIntegrationReconciler{
		ReconcilerBase: util.NewReconcilerBase(mgr.GetClient(), mgr.GetScheme(), mgr.GetConfig(), mgr.GetEventRecorderFor("QuayIntegration_controller"), mgr.GetAPIReader()),
		Log:            ctrl.Log.WithName("controllers").WithName("QuayIntegration"),
		LastSeenSpec:   map[types.NamespacedName]string{},
	}).SetupWithManager(mgr); err != nil {
		setupLog.Error(err, "unable to create controller", "controller", "QuayIntegration")
		os.Exit(1)
	}

	if err = (&controllers.NamespaceIntegrationReconciler{
		CoreComponents: core.NewCoreComponents(util.NewReconcilerBase(mgr.GetClient(), mgr.GetScheme(), mgr.GetConfig(), mgr.GetEventRecorderFor("NamespaceIntegration_controller"), mgr.GetAPIReader())),
		Log:            ctrl.Log.WithName("controllers").WithName("NamespaceIntegration"),
	}).SetupWithManager(mgr); err != nil {
		setupLog.Error(err, "unable to create controller", "controller", "NamespaceIntegration")
		os.Exit(1)
	}

	if err = (&controllers.BuildIntegrationReconciler{
		CoreComponents: core.NewCoreComponents(util.NewReconcilerBase(mgr.GetClient(), mgr.GetScheme(), mgr.GetConfig(), mgr.GetEventRecorderFor("BuildIntegration_controller"), mgr.GetAPIReader())),
		Log:            ctrl.Log.WithName("controllers").WithName("BuildIntegration"),
	}).SetupWithManager(mgr); err != nil {
		setupLog.Error(err, "unable to create controller", "controller", "BuildIntegration")
		os.Exit(1)
	}

	// Enable Webhook support
	_, disableWebhookEnv := os.LookupEnv(constants.DisableWebhookEnvVar)

	if !disableWebhookEnv {

		// Register Webhook
		webhookSvr := mgr.GetWebhookServer()
		webhookSvr.CertDir = getWebhookCertDir()
		webhookSvr.CertName = constants.WebhookCertName
		webhookSvr.KeyName = constants.WebhookKeyName
		webhookSvr.Register("/admissionwebhook", &webhook.Admission{Handler: &quaywebhook.QuayIntegrationMutator{Client: mgr.GetClient(), Log: ctrl.Log.WithName("webhook").WithName("QuayIntegration")}})

	}

	//+kubebuilder:scaffold:builder

	if err := mgr.AddHealthzCheck("healthz", healthz.Ping); err != nil {
		setupLog.Error(err, "unable to set up health check")
		os.Exit(1)
	}
	if err := mgr.AddReadyzCheck("readyz", healthz.Ping); err != nil {
		setupLog.Error(err, "unable to set up ready check")
		os.Exit(1)
	}

	setupLog.Info("starting manager")
	if err := mgr.Start(ctrl.SetupSignalHandler()); err != nil {
		setupLog.Error(err, "problem running manager")
		os.Exit(1)
	}

}

func getWebhookCertDir() string {
	webhookCertDir := os.Getenv(constants.WebHookCertDirEnv)
	if webhookCertDir != "" {
		return webhookCertDir
	}

	return constants.DefaultWebhookCertDir
}
