/**
 * @module optical-properties
 */
'use strict'

module.exports = measure

var canvas = document.createElement('canvas'),
	ctx = canvas.getContext('2d')

canvas.width = 200, canvas.height = 200

measure.canvas = canvas

//returns character [x, y, scale] optical params
function measure (char, options) {
	var data, w, h, params

	//figure out argument imageData
	if (typeof char === 'string') {
		data = getCharImageData(char, options)
		w = data.width, h = data.height
	}
	else if (char instanceof HTMLCanvasElement) {
		w = char.width, h = char.height
		char = char.getContext('2d')
		data = char.getImageData(0, 0, w, h)
	}
	else if (char instanceof ImageData) {
		w = char.width, h = char.height
		data = char
	}

	params = getOpticalParams(data)

	return params
}

//draw character in canvas and get it's imagedata
function getCharImageData (char, options) {
	if (!options) options = {}
	var family = options.family || 'sans-serif'
	var w = canvas.width, h = canvas.height

	var size = options.width || options.height || options.size
	if (size && size != w) {
		w = h = canvas.width = canvas.height = size
	}

	var fs = options.fontSize || w/2

	ctx.fillStyle = '#000'
	ctx.fillRect(0, 0, w, h)

	ctx.font = fs + 'px ' + family
	ctx.textBaseline = 'middle'
	ctx.textAlign = 'center'
	ctx.fillStyle = 'white'
	ctx.fillText(char, w/2, h/2)

	return ctx.getImageData(0, 0, w, h)
}


//walks over imagedata, returns params
function getOpticalParams (data) {
	var buf = data.data, w = data.width, h = data.height

	var x, y, r, i, j, sum, xSum, ySum, rowAvg = Array(h), rowAvgX = Array(h), cx, cy, bounds, avg, top = 0, bottom = 0, left = w, right = 0, maxR = 0, rowBounds = Array(h), r2

	for (y = 0; y < h; y++) {
		sum = 0, xSum = 0, j = y*4*w

		bounds = getBounds(buf.subarray(j, j + 4*w), 4)

		if (bounds[0] === bounds[1]) {
			continue
		}
		else {
			if (!top) top = y
			bottom = y
		}

		for (x = bounds[0]; x < bounds[1]; x++) {
			i = x*4
			r = buf[j + i]
			sum += r
			xSum += x*r
		}

		rowAvg[y] = sum === 0 ? 0 : sum/w
		rowAvgX[y] = sum === 0 ? 0 : xSum/sum

		if (bounds[0] < left) left = bounds[0]
		if (bounds[1] > right) right = bounds[1]

		rowBounds[y] = bounds
	}

	sum = 0, ySum = 0, xSum = 0
	for (y = 0; y < h; y++) {
		avg = rowAvg[y]
		if (!avg) continue;

		ySum += avg*y
		sum += avg
		xSum += rowAvgX[y]*avg
	}

	cy = ySum/sum
	cx = xSum/sum

	maxR = 0, r2 = 0
	for (y = 0; y < h; y++) {
		bounds = rowBounds[y]
		if (!bounds) continue

		r2 = Math.max(
			dist2(cx - bounds[0], cy - y),
			dist2(cx - bounds[1], cy - y)
		)
		if (r2 > maxR) {
			maxR = r2
		}
	}

	return {
		center: [cx, cy],
		bounds: [left, top, right, bottom+1],
		radius: Math.sqrt(maxR)
	}
}

//get [leftId, rightId] pair of bounding values for an array
function getBounds (arr, stride) {
	var left = 0, right = arr.length, i = 0

	if (!stride) stride = 4

	//find left non-zero value
	while (!arr[i] && i < right) {
		i+=stride
	}
	left = i

	//find right non-zero value
	i = arr.length
	while (!arr[i] && i > left) {
		i-=stride
	}
	right = i

	return [left/stride, right/stride]
}

function dist2 (x, y) {
	return x*x + y*y
}
