"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.prefixedId = prefixedId;
exports.getResource = exports.getBootDeviceIndex = exports.generateDiskName = exports.setNativeValue = exports.getSequence = exports.parseUrl = void 0;

var _models = require("../models");

function prefixedId(idPrefix, id) {
  return idPrefix && id ? `${idPrefix}-${id}` : null;
}

const parseUrl = url => {
  try {
    return new URL(url);
  } catch (e) {
    return null;
  }
};

exports.parseUrl = parseUrl;

const getSequence = (from, to) => Array.from({
  length: to - from + 1
}, (v, i) => i + from);

exports.getSequence = getSequence;

const setNativeValue = (element, value) => {
  const valueSetter = Object.getOwnPropertyDescriptor(element, 'value').set;
  const prototype = Object.getPrototypeOf(element);
  const prototypeValueSetter = Object.getOwnPropertyDescriptor(prototype, 'value').set;

  if (valueSetter && valueSetter !== prototypeValueSetter) {
    prototypeValueSetter.call(element, value);
  } else {
    valueSetter.call(element, value);
  }
};

exports.setNativeValue = setNativeValue;

const generateDiskName = (vmName, diskName, clone) => {
  const name = [vmName, diskName];

  if (clone) {
    name.push('clone');
  }

  return name.join('-');
};

exports.generateDiskName = generateDiskName;

const getBootDeviceIndex = (devices, bootOrder) => devices.findIndex(device => device.bootOrder === bootOrder);

exports.getBootDeviceIndex = getBootDeviceIndex;

const getResource = (model, {
  name,
  namespaced = true,
  namespace,
  isList = true,
  matchLabels,
  matchExpressions,
  prop
} = {
  namespaced: true,
  isList: true
}) => {
  const res = {
    // non-admin user cannot list namespaces (k8s wont return only namespaces available to user but 403 forbidden, ).
    // Instead we need to use ProjectModel which will return available projects (namespaces)
    kind: model.kind === _models.NamespaceModel.kind ? _models.ProjectModel.kind : model.kind,
    namespaced,
    namespace,
    isList,
    prop: prop || model.kind
  };

  if (name) {
    res.name = name;
  }

  if (matchLabels) {
    res.selector = {
      matchLabels
    };
  }

  if (matchExpressions) {
    res.selector = {
      matchExpressions
    };
  }

  return res;
};

exports.getResource = getResource;