"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Integer = void 0;

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _patternflyReact = require("patternfly-react");

var _utils = require("./utils");

var _utils2 = require("../../utils/utils");

var _keys = require("../../constants/keys");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const NON_NEGATIVE_INTEGER_KEYS = [..._keys.INPUT_NAVIGATION_KEYS, ...(0, _utils2.getSequence)(_keys.KEY_CODES[0], _keys.KEY_CODES[9]), ...(0, _utils2.getSequence)(_keys.KEY_CODES.NUMPAD[0], _keys.KEY_CODES.NUMPAD[9])];
const INTEGER_KEYS = [...NON_NEGATIVE_INTEGER_KEYS, _keys.KEY_CODES.MINUS, _keys.KEY_CODES.HYPHEN_MINUS, _keys.KEY_CODES.NUMPAD.SUBTRACT];
const POSITIVE_INT_REGEX = /^[1-9]\d*$/;
const NON_NEGATIVE_INT_REGEX = /^(0|[1-9]\d*)$/;
const INT_REGEX = /^(-?0|-?[1-9]\d*)$/;
const PRECEEDING_ZEROS_POSITIVE_INT_REGEX = /^0*([1-9]\d*)$/;
const PRECEEDING_ZEROS_INT_REGEX = /^(-?)0*([1-9]\d*)$/;

const fixPrecedingZerosPositiveInt = value => {
  const match = PRECEEDING_ZEROS_POSITIVE_INT_REGEX.exec(value);
  return match && match.length === 2 ? match[1] : '';
};

const fixPrecedingZerosNonNegativeInt = value => {
  const match = PRECEEDING_ZEROS_POSITIVE_INT_REGEX.exec(value);
  return match && match.length === 2 ? match[1] : '0';
};

const fixInt = (value, oldValue, keyCode) => {
  if (value.length === 0 && (0, _keys.isMinus)(keyCode)) {
    if (oldValue) {
      return oldValue < 0 || oldValue === '-0' ? oldValue : `-${oldValue}`;
    }

    return '-0';
  }

  const match = PRECEEDING_ZEROS_INT_REGEX.exec(value);

  if (match && match.length === 3) {
    if (match[1]) {
      return `${match[1]}${match[2]}`;
    }

    return match[2];
  }

  return '0';
};

const isInputValid = (allowedKeys, keyCode, targetValue, additionalValidation) => {
  if (allowedKeys.includes(keyCode)) {
    return additionalValidation ? additionalValidation(keyCode, targetValue) : true;
  }

  return false;
};

const Integer = ({
  id,
  value,
  disabled,
  defaultValue,
  onChange,
  onBlur,
  positive,
  nonNegative,
  className
}) => {
  let allowedKeys;
  let validRegex;
  let fixAfterValue;
  let min;
  let additionalValidation;

  if (positive) {
    allowedKeys = NON_NEGATIVE_INTEGER_KEYS;
    validRegex = POSITIVE_INT_REGEX;
    fixAfterValue = fixPrecedingZerosPositiveInt;
    min = 1;

    additionalValidation = (keyCode, targetValue) => !(targetValue === '' && (keyCode === _keys.KEY_CODES[0] || keyCode === _keys.KEY_CODES.NUMPAD[0]));
  } else if (nonNegative) {
    allowedKeys = NON_NEGATIVE_INTEGER_KEYS;
    validRegex = NON_NEGATIVE_INT_REGEX;
    fixAfterValue = fixPrecedingZerosNonNegativeInt;
    min = 0;
  } else {
    allowedKeys = INTEGER_KEYS;
    validRegex = INT_REGEX;
    fixAfterValue = fixInt;
  }

  const onKeydown = e => {
    const keyCode = e.keyCode,
          target = e.target;

    if (!isInputValid(allowedKeys, keyCode, target.value, additionalValidation)) {
      e.preventDefault();
      return false;
    }

    const oldValue = target.value;
    window.setTimeout(() => {
      if (!validRegex.test(target.value)) {
        const v = fixAfterValue(target.value, oldValue, keyCode);
        (0, _utils2.setNativeValue)(target, v);
        target.dispatchEvent(new Event('input', {
          bubbles: true,
          cancelable: true
        }));
      }
    }, 0);
    return true;
  };

  return _react.default.createElement(_patternflyReact.FormControl, {
    id: id,
    type: "number",
    onKeyDown: onKeydown,
    min: min,
    value: value,
    defaultValue: defaultValue,
    onBlur: (0, _utils.eventValueHandler)(onBlur),
    onChange: (0, _utils.eventValueHandler)(onChange),
    className: className,
    disabled: disabled
  });
};

exports.Integer = Integer;
Integer.defaultProps = {
  value: undefined,
  defaultValue: undefined,
  onChange: _patternflyReact.noop,
  onBlur: _patternflyReact.noop,
  positive: false,
  nonNegative: false,
  // is ignored when positive == true
  className: undefined,
  disabled: false
};
Integer.propTypes = {
  id: _propTypes.default.string.isRequired,
  value: _propTypes.default.oneOfType([_propTypes.default.string, _propTypes.default.number]),
  defaultValue: _propTypes.default.oneOfType([_propTypes.default.string, _propTypes.default.number]),
  onChange: _propTypes.default.func,
  onBlur: _propTypes.default.func,
  positive: _propTypes.default.bool,
  nonNegative: _propTypes.default.bool,
  className: _propTypes.default.string,
  disabled: _propTypes.default.bool
};