'use strict'

module.exports = createFancyScatter2D

var createShader = require('gl-shader')
var createBuffer = require('gl-buffer')
var pool = require('typedarray-pool')
var shaders = require('./lib/shaders')
var snapPoints = require('snap-points-2d')
var atlas = require('font-atlas-sdf')
var createTexture = require('gl-texture2d')
var colorId = require('color-id')
var ndarray = require('ndarray')
var clamp = require('clamp')
var search = require('binary-search-bounds')

function GLScatterFancy(
    plot,
    shader,
    pickShader,
    positionBuffer,
    sizeBuffer,
    colorBuffer,
    idBuffer,
    charBuffer) {
  this.plot           = plot
  this.shader         = shader
  this.pickShader     = pickShader

  //buffers
  this.positionBuffer = positionBuffer
  this.sizeBuffer     = sizeBuffer
  this.colorBuffer    = colorBuffer
  this.idBuffer       = idBuffer
  this.charBuffer      = charBuffer

  this.pointCount     = 0
  this.pickOffset     = 0

  //positions data
  this.points         = null

  //lod scales
  this.scales         = []
  this.xCoords        = []

  //font atlas texture
  this.charCanvas     = document.createElement('canvas')
  this.charTexture    = createTexture(this.plot.gl, this.charCanvas)
  this.charStep       = 400
  this.charFit        = .255

  //snapping loses points sorting, so disable snapping on small number of points
  this.snapThreshold  = 1e4

  //border/char colors texture
  this.paletteTexture   = createTexture(this.plot.gl, [256, 1])
}

var proto = GLScatterFancy.prototype

var SCALE_HI = new Float32Array([0, 0])
var SCALE_LO = new Float32Array([0, 0])
var TRANSLATE_HI = new Float32Array([0, 0])
var TRANSLATE_LO = new Float32Array([0, 0])

var PIXEL_SCALE = [0, 0]

var pixelSize, xStart, xEnd


function calcScales() {
  var plot       = this.plot

  var viewBox    = plot.viewBox
  var dataBox    = plot.dataBox
  var pixelRatio = plot.pixelRatio

  var dataX  = dataBox[2] - dataBox[0]
  var dataY  = dataBox[3] - dataBox[1]

  var scaleX = 2 / dataX
  var scaleY = 2 / dataY
  var translateX = (- dataBox[0] - 0.5 * dataX)
  var translateY = (- dataBox[1] - 0.5 * dataY)

  SCALE_HI[0] = scaleX
  SCALE_LO[0] = scaleX - SCALE_HI[0]
  SCALE_HI[1] = scaleY
  SCALE_LO[1] = scaleY - SCALE_HI[1]

  TRANSLATE_HI[0] = translateX
  TRANSLATE_LO[0] = translateX - TRANSLATE_HI[0]
  TRANSLATE_HI[1] = translateY
  TRANSLATE_LO[1] = translateY - TRANSLATE_HI[1]

  var screenX = viewBox[2] - viewBox[0]
  var screenY = viewBox[3] - viewBox[1]

  pixelSize   = Math.min(dataX / screenX, dataY / screenY)

  //FIXME: why twice?
  PIXEL_SCALE[0] = 2 * pixelRatio / screenX
  PIXEL_SCALE[1] = 2 * pixelRatio / screenY

  xStart = dataBox[0]
  xEnd = dataBox[2]
}

var PICK_OFFSET = [0, 0, 0, 0]

proto.drawPick = function(offset) {
  var pick = offset !== undefined
  var plot = this.plot
  var pointCount = this.pointCount
  var snap = pointCount > this.snapThreshold

  if(!pointCount) {
    return offset
  }

  calcScales.call(this)

  var gl = plot.gl
  var shader = pick ? this.pickShader : this.shader
  var blend = gl.isEnabled(gl.BLEND)

  shader.bind()

  if(pick) {
    this.pickOffset = offset

    for (var i = 0; i < 4; ++i) {
      PICK_OFFSET[i] = (offset >> (i * 8)) & 0xff
    }

    shader.uniforms.pickOffset = PICK_OFFSET

    this.idBuffer.bind()
    shader.attributes.id.pointer(gl.UNSIGNED_BYTE, false)

  } else {
    gl.blendFuncSeparate(gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA, gl.ONE, gl.ONE_MINUS_SRC_ALPHA);
    gl.blendColor(0,0,0,1);
    if (!blend) gl.enable(gl.BLEND)

    this.colorBuffer.bind()
    shader.attributes.color.pointer(gl.UNSIGNED_BYTE, false)

    this.charBuffer.bind()
    shader.attributes.char.pointer(gl.UNSIGNED_BYTE, false)

    shader.uniforms.chars = this.charTexture.bind(0)
    shader.uniforms.charsShape = [this.charCanvas.width, this.charCanvas.height]
    shader.uniforms.charsStep = this.charStep
    shader.uniforms.palette = this.paletteTexture.bind(1)
  }

  this.sizeBuffer.bind()
  shader.attributes.size.pointer(gl.FLOAT, false, 8, 0)
  if (!pick) shader.attributes.border.pointer(gl.FLOAT, false, 8, 4)

  this.positionBuffer.bind()
  shader.attributes.positionHi.pointer(gl.FLOAT, false, 16, 0)
  shader.attributes.positionLo.pointer(gl.FLOAT, false, 16, 8)

  shader.uniforms.pixelRatio  = plot.pixelRatio
  shader.uniforms.scaleHi     = SCALE_HI
  shader.uniforms.scaleLo     = SCALE_LO
  shader.uniforms.translateHi = TRANSLATE_HI
  shader.uniforms.translateLo = TRANSLATE_LO
  shader.uniforms.viewBox = plot.viewBox

  var scales = this.scales

  if (snap) {
    for (var scaleNum = scales.length - 1; scaleNum >= 0; scaleNum--) {
        var lod = scales[scaleNum]
        if(lod.pixelSize && (lod.pixelSize < pixelSize * 1.25) && scaleNum > 1) {
          continue
        }

        var intervalStart = lod.offset
        var intervalEnd   = lod.count + intervalStart

        var startOffset = search.ge(this.xCoords, xStart, intervalStart, intervalEnd - 1)
        var endOffset   = search.lt(this.xCoords, xEnd, startOffset, intervalEnd - 1) + 1

        if (endOffset > startOffset) {
          gl.drawArrays(gl.POINTS, startOffset, (endOffset - startOffset))
        }
    }
  }
  else {
    gl.drawArrays(gl.POINTS, 0, pointCount)
  }

  if (pick) return offset + pointCount
  else {
    if (!blend) gl.disable(gl.BLEND)
    else {
      gl.blendFunc(gl.ONE, gl.ONE_MINUS_SRC_ALPHA)
    }
  }

}

proto.draw = proto.drawPick

proto.pick = function(x, y, value) {
  var pickOffset = this.pickOffset
  var pointCount = this.pointCount
  if(value < pickOffset || value >= pickOffset + pointCount) {
    return null
  }
  var pointId = value - pickOffset
  var points  = this.points
  return {
    object:    this,
    pointId:   pointId,
    dataCoord: [points[2 * pointId], points[2 * pointId + 1]]
  }
}

proto.update = function(options) {
  options = options || {}

  var positions     = options.positions || []
  var colors        = options.colors       || []
  var glyphs        = options.glyphs       || []
  var sizes         = options.sizes        || []
  var borderWidths  = options.borderWidths || []
  var borderColors  = options.borderColors || []
  var gl = this.plot.gl
  var pointCount = this.pointCount
  var snap = pointCount > this.snapThreshold

  //update positions
  if (options.positions != null) {
    this.points = positions

    pointCount = this.points.length / 2

    snap = pointCount > this.snapThreshold

    //create packed positions here
    var packedW            = pool.mallocFloat32(2 * pointCount)
    var packed             = pool.mallocFloat64(2 * pointCount)
    var v_ids       = pool.mallocUint32(pointCount)
    var v_position  = pool.mallocFloat32(4 * pointCount)

    packed.set(this.points)

    if (snap) {
      if (this.i2idx) pool.free(this.i2idx)
      this.i2idx = pool.mallocInt32(pointCount)
      this.scales = snapPoints(packed, this.i2idx, packedW)
    }

    this.pointCount = pointCount


    for(var i = 0; i < pointCount; ++i) {
      var id = snap ? this.i2idx[i] : i

      v_ids[i] = id

      //collect buffers data
      var x = positions[2 * id]
      var y = positions[2 * id + 1]

      //write hi- and lo- position parts
      v_position[4 * i]      = x
      v_position[4 * i + 1]  = y
      v_position[4 * i + 2]  = x - v_position[4 * i]
      v_position[4 * i + 3]  = y - v_position[4 * i + 1]

      this.xCoords[i] = x
    }

    this.idBuffer.update(v_ids)
    this.positionBuffer.update(v_position)
    pool.free(v_position)
    pool.free(v_ids)
    pool.free(packed)
    pool.free(packedW)
  }

  var v_sizeWidth = pool.mallocFloat32(2 * pointCount)
  var v_color     = pool.mallocUint8(2 * pointCount)
  var v_chars     = pool.mallocUint8(2 * pointCount)

  //aggregate colors
  var paletteIds = {}, colorIds = [], paletteColors = [], bColorIds = []
  for (var i = 0, l = pointCount, k = 0; i < l; ++i) {
    var channels = [colors[4 * i] * 255, colors[4 * i + 1] * 255, colors[4 * i + 2] * 255, colors[4 * i + 3] * 255]
    var cId = colorId(channels, false)
    if (paletteIds[cId] == null) {
      paletteIds[cId] = k++
      paletteColors.push(channels[0])
      paletteColors.push(channels[1])
      paletteColors.push(channels[2])
      paletteColors.push(channels[3])
    }
    colorIds.push(cId)

    if (borderColors && borderColors.length) {
      channels = [borderColors[4 * i] * 255, borderColors[4 * i + 1] * 255, borderColors[4 * i + 2] * 255, borderColors[4 * i + 3] * 255]
      cId = colorId(channels, false)
      if (paletteIds[cId] == null) {
        paletteIds[cId] = k++
        paletteColors.push(channels[0])
        paletteColors.push(channels[1])
        paletteColors.push(channels[2])
        paletteColors.push(channels[3])
      }
      bColorIds.push(cId)
    }
  }

  //aggregate glyphs
  var glyphChars = {}
  for (var i = 0, l = pointCount, k = 0; i < l; i++) {
    var char = glyphs[i]
    if (glyphChars[char] == null) {
      glyphChars[char] = k++
    }
  }

  //generate font atlas
  var maxSize = 0
  for (var i = 0, l = sizes.length; i < l; ++i) {
    if (sizes[i] > maxSize) maxSize = sizes[i]
  }
  var oldStep = this.charStep
  this.charStep = clamp(Math.ceil(maxSize*4), 128, 768)

  var chars = Object.keys(glyphChars)
  var step = this.charStep
  var charSize = Math.floor(step / 2)
  var maxW = gl.getParameter(gl.MAX_TEXTURE_SIZE)
  var maxChars = (maxW / step) * (maxW / step)
  var atlasW = Math.min(maxW, step*chars.length)
  var atlasH = Math.min(maxW, step*Math.ceil(step*chars.length/maxW))
  var cols = Math.floor(atlasW / step)
  if (chars.length > maxChars) {
    console.warn('gl-scatter2d-fancy: number of characters is more than maximum texture size. Try reducing it.')
  }

  //do not overupdate atlas
  if (!this.chars || (this.chars+'' !== chars+'') || this.charStep != oldStep) {
    this.charCanvas = atlas({
      canvas: this.charCanvas,
      family: 'sans-serif',
      size: charSize,
      shape: [atlasW, atlasH],
      step: [step, step],
      chars: chars,
      align: true,
      fit: this.charFit
    })
    this.chars = chars
  }

  for(var i = 0; i < pointCount; ++i) {
    var id = snap ? this.i2idx[i] : i

    var s = sizes[id]
    var w = borderWidths[id]

    //size is doubled bc character SDF is twice less than character step
    v_sizeWidth[2 * i]     = s*2
    v_sizeWidth[2 * i + 1] = w

    //color/bufferColor indexes
    var cId = colorIds[id]
    var pcId = paletteIds[cId]
    v_color[2 * i] = pcId
    var bcId = bColorIds[id]
    var pbcId = paletteIds[bcId]
    v_color[2 * i + 1] = pbcId

    //char indexes
    var char = glyphs[id]
    var charId = glyphChars[char]
    v_chars[2 * i + 1] = Math.floor(charId / cols)
    v_chars[2 * i] = charId % cols
  }

  //fill buffes
  this.sizeBuffer.update(v_sizeWidth)
  this.colorBuffer.update(v_color)
  this.charBuffer.update(v_chars)

  //update char/color textures
  this.charTexture.shape = [this.charCanvas.width, this.charCanvas.height]
  if (this.charCanvas && this.charCanvas.width) {
    this.charTexture.setPixels(this.charCanvas)
  }
  this.paletteTexture.setPixels(ndarray(paletteColors.slice(0, 256*4), [256, 1, 4]))

  pool.free(v_sizeWidth)
  pool.free(v_color)
  pool.free(v_chars)
}

proto.dispose = function() {
  this.shader.dispose()
  this.pickShader.dispose()
  this.positionBuffer.dispose()
  this.sizeBuffer.dispose()
  this.colorBuffer.dispose()
  this.idBuffer.dispose()
  this.charBuffer.dispose()
  this.plot.removeObject(this)
}

function createFancyScatter2D(plot, options) {
  var gl = plot.gl

  var shader     = createShader(gl, shaders.vertex,     shaders.fragment)
  var pickShader = createShader(gl, shaders.pickVertex, shaders.pickFragment)

  var positionBuffer   = createBuffer(gl)
  var sizeBuffer       = createBuffer(gl)
  var colorBuffer      = createBuffer(gl)
  var idBuffer         = createBuffer(gl)
  var charBuffer       = createBuffer(gl)

  var scatter = new GLScatterFancy(
    plot,
    shader,
    pickShader,
    positionBuffer,
    sizeBuffer,
    colorBuffer,
    idBuffer,
    charBuffer)

  scatter.update(options)

  plot.addObject(scatter)

  return scatter
}
