package batch

// Copyright (c) Microsoft and contributors.  All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Code generated by Microsoft (R) AutoRest Code Generator.
// Changes may cause incorrect behavior and will be lost if the code is regenerated.

import (
	"context"
	"github.com/Azure/go-autorest/autorest"
	"github.com/Azure/go-autorest/autorest/azure"
	"github.com/Azure/go-autorest/autorest/date"
	"github.com/Azure/go-autorest/autorest/validation"
	"net/http"
)

// PoolClient is the a client for issuing REST requests to the Azure Batch service.
type PoolClient struct {
	BaseClient
}

// NewPoolClient creates an instance of the PoolClient client.
func NewPoolClient() PoolClient {
	return NewPoolClientWithBaseURI(DefaultBaseURI)
}

// NewPoolClientWithBaseURI creates an instance of the PoolClient client.
func NewPoolClientWithBaseURI(baseURI string) PoolClient {
	return PoolClient{NewWithBaseURI(baseURI)}
}

// Add when naming pools, avoid including sensitive information such as user names or secret project names. This
// information may appear in telemetry logs accessible to Microsoft Support engineers.
//
// pool is the pool to be added. timeout is the maximum time that the server can spend processing the request, in
// seconds. The default is 30 seconds. clientRequestID is the caller-generated request identity, in the form of a
// GUID with no decoration such as curly braces, e.g. 9C4D50EE-2D56-4CD3-8152-34347DC9F2B0. returnClientRequestID
// is whether the server should return the client-request-id in the response. ocpDate is the time the request was
// issued. If not specified, this header will be automatically populated with the current system clock time.
func (client PoolClient) Add(ctx context.Context, pool PoolAddParameter, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: pool,
			Constraints: []validation.Constraint{{Target: "pool.ID", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "pool.VMSize", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "pool.CloudServiceConfiguration", Name: validation.Null, Rule: false,
					Chain: []validation.Constraint{{Target: "pool.CloudServiceConfiguration.OsFamily", Name: validation.Null, Rule: true, Chain: nil}}},
				{Target: "pool.VirtualMachineConfiguration", Name: validation.Null, Rule: false,
					Chain: []validation.Constraint{{Target: "pool.VirtualMachineConfiguration.ImageReference", Name: validation.Null, Rule: true,
						Chain: []validation.Constraint{{Target: "pool.VirtualMachineConfiguration.ImageReference.Publisher", Name: validation.Null, Rule: true, Chain: nil},
							{Target: "pool.VirtualMachineConfiguration.ImageReference.Offer", Name: validation.Null, Rule: true, Chain: nil},
							{Target: "pool.VirtualMachineConfiguration.ImageReference.Sku", Name: validation.Null, Rule: true, Chain: nil},
						}},
						{Target: "pool.VirtualMachineConfiguration.NodeAgentSKUID", Name: validation.Null, Rule: true, Chain: nil},
					}},
				{Target: "pool.StartTask", Name: validation.Null, Rule: false,
					Chain: []validation.Constraint{{Target: "pool.StartTask.CommandLine", Name: validation.Null, Rule: true, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("batch.PoolClient", "Add", err.Error())
	}

	req, err := client.AddPreparer(ctx, pool, timeout, clientRequestID, returnClientRequestID, ocpDate)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "Add", nil, "Failure preparing request")
		return
	}

	resp, err := client.AddSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "Add", resp, "Failure sending request")
		return
	}

	result, err = client.AddResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "Add", resp, "Failure responding to request")
	}

	return
}

// AddPreparer prepares the Add request.
func (client PoolClient) AddPreparer(ctx context.Context, pool PoolAddParameter, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123) (*http.Request, error) {
	const APIVersion = "2016-07-01.3.1"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 30)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; odata=minimalmetadata; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/pools"),
		autorest.WithJSON(pool),
		autorest.WithQueryParameters(queryParameters))
	if len(clientRequestID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("client-request-id", autorest.String(clientRequestID)))
	}
	if returnClientRequestID != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(returnClientRequestID)))
	} else {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(false)))
	}
	if ocpDate != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("ocp-date", autorest.String(ocpDate)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// AddSender sends the Add request. The method will close the
// http.Response Body if it receives an error.
func (client PoolClient) AddSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// AddResponder handles the response to the Add request. The method always
// closes the http.Response Body.
func (client PoolClient) AddResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusCreated),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Delete when you request that a pool be deleted, the following actions occur: the pool state is set to deleting; any
// ongoing resize operation on the pool are stopped; the Batch service starts resizing the pool to zero nodes; any
// tasks running on existing nodes are terminated and requeued (as if a resize pool operation had been requested with
// the default requeue option); finally, the pool is removed from the system. Because running tasks are requeued, the
// user can rerun these tasks by updating their job to target a different pool. The tasks can then run on the new pool.
// If you want to override the requeue behavior, then you should call resize pool explicitly to shrink the pool to zero
// size before deleting the pool. If you call an Update, Patch or Delete API on a pool in the deleting state, it will
// fail with HTTP status code 409 with error code PoolBeingDeleted.
//
// poolID is the ID of the pool to delete. timeout is the maximum time that the server can spend processing the
// request, in seconds. The default is 30 seconds. clientRequestID is the caller-generated request identity, in the
// form of a GUID with no decoration such as curly braces, e.g. 9C4D50EE-2D56-4CD3-8152-34347DC9F2B0.
// returnClientRequestID is whether the server should return the client-request-id in the response. ocpDate is the
// time the request was issued. If not specified, this header will be automatically populated with the current
// system clock time. ifMatch is an ETag is specified. Specify this header to perform the operation only if the
// resource's ETag is an exact match as specified. ifNoneMatch is an ETag is specified. Specify this header to
// perform the operation only if the resource's ETag does not match the specified ETag. ifModifiedSince is specify
// this header to perform the operation only if the resource has been modified since the specified date/time.
// ifUnmodifiedSince is specify this header to perform the operation only if the resource has not been modified
// since the specified date/time.
func (client PoolClient) Delete(ctx context.Context, poolID string, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123, ifMatch string, ifNoneMatch string, ifModifiedSince *date.TimeRFC1123, ifUnmodifiedSince *date.TimeRFC1123) (result autorest.Response, err error) {
	req, err := client.DeletePreparer(ctx, poolID, timeout, clientRequestID, returnClientRequestID, ocpDate, ifMatch, ifNoneMatch, ifModifiedSince, ifUnmodifiedSince)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "Delete", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "Delete", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "Delete", resp, "Failure responding to request")
	}

	return
}

// DeletePreparer prepares the Delete request.
func (client PoolClient) DeletePreparer(ctx context.Context, poolID string, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123, ifMatch string, ifNoneMatch string, ifModifiedSince *date.TimeRFC1123, ifUnmodifiedSince *date.TimeRFC1123) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"poolId": autorest.Encode("path", poolID),
	}

	const APIVersion = "2016-07-01.3.1"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 30)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/pools/{poolId}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	if len(clientRequestID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("client-request-id", autorest.String(clientRequestID)))
	}
	if returnClientRequestID != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(returnClientRequestID)))
	} else {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(false)))
	}
	if ocpDate != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("ocp-date", autorest.String(ocpDate)))
	}
	if len(ifMatch) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Match", autorest.String(ifMatch)))
	}
	if len(ifNoneMatch) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-None-Match", autorest.String(ifNoneMatch)))
	}
	if ifModifiedSince != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Modified-Since", autorest.String(ifModifiedSince)))
	}
	if ifUnmodifiedSince != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Unmodified-Since", autorest.String(ifUnmodifiedSince)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteSender sends the Delete request. The method will close the
// http.Response Body if it receives an error.
func (client PoolClient) DeleteSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DeleteResponder handles the response to the Delete request. The method always
// closes the http.Response Body.
func (client PoolClient) DeleteResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByClosing())
	result.Response = resp
	return
}

// DisableAutoScale sends the disable auto scale request.
//
// poolID is the ID of the pool on which to disable automatic scaling. timeout is the maximum time that the server
// can spend processing the request, in seconds. The default is 30 seconds. clientRequestID is the caller-generated
// request identity, in the form of a GUID with no decoration such as curly braces, e.g.
// 9C4D50EE-2D56-4CD3-8152-34347DC9F2B0. returnClientRequestID is whether the server should return the
// client-request-id in the response. ocpDate is the time the request was issued. If not specified, this header
// will be automatically populated with the current system clock time.
func (client PoolClient) DisableAutoScale(ctx context.Context, poolID string, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123) (result autorest.Response, err error) {
	req, err := client.DisableAutoScalePreparer(ctx, poolID, timeout, clientRequestID, returnClientRequestID, ocpDate)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "DisableAutoScale", nil, "Failure preparing request")
		return
	}

	resp, err := client.DisableAutoScaleSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "DisableAutoScale", resp, "Failure sending request")
		return
	}

	result, err = client.DisableAutoScaleResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "DisableAutoScale", resp, "Failure responding to request")
	}

	return
}

// DisableAutoScalePreparer prepares the DisableAutoScale request.
func (client PoolClient) DisableAutoScalePreparer(ctx context.Context, poolID string, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"poolId": autorest.Encode("path", poolID),
	}

	const APIVersion = "2016-07-01.3.1"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 30)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/pools/{poolId}/disableautoscale", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	if len(clientRequestID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("client-request-id", autorest.String(clientRequestID)))
	}
	if returnClientRequestID != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(returnClientRequestID)))
	} else {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(false)))
	}
	if ocpDate != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("ocp-date", autorest.String(ocpDate)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DisableAutoScaleSender sends the DisableAutoScale request. The method will close the
// http.Response Body if it receives an error.
func (client PoolClient) DisableAutoScaleSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DisableAutoScaleResponder handles the response to the DisableAutoScale request. The method always
// closes the http.Response Body.
func (client PoolClient) DisableAutoScaleResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// EnableAutoScale you cannot enable automatic scaling on a pool if a resize operation is in progress on the pool. If
// automatic scaling of the pool is currently disabled, you must specify a valid autoscale formula as part of the
// request. If automatic scaling of the pool is already enabled, you may specify a new autoscale formula and/or a new
// evaluation interval. You cannot call this API for the same pool more than once every 30 seconds.
//
// poolID is the ID of the pool on which to enable automatic scaling. poolEnableAutoScaleParameter is the
// parameters for the request. timeout is the maximum time that the server can spend processing the request, in
// seconds. The default is 30 seconds. clientRequestID is the caller-generated request identity, in the form of a
// GUID with no decoration such as curly braces, e.g. 9C4D50EE-2D56-4CD3-8152-34347DC9F2B0. returnClientRequestID
// is whether the server should return the client-request-id in the response. ocpDate is the time the request was
// issued. If not specified, this header will be automatically populated with the current system clock time.
// ifMatch is an ETag is specified. Specify this header to perform the operation only if the resource's ETag is an
// exact match as specified. ifNoneMatch is an ETag is specified. Specify this header to perform the operation only
// if the resource's ETag does not match the specified ETag. ifModifiedSince is specify this header to perform the
// operation only if the resource has been modified since the specified date/time. ifUnmodifiedSince is specify
// this header to perform the operation only if the resource has not been modified since the specified date/time.
func (client PoolClient) EnableAutoScale(ctx context.Context, poolID string, poolEnableAutoScaleParameter PoolEnableAutoScaleParameter, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123, ifMatch string, ifNoneMatch string, ifModifiedSince *date.TimeRFC1123, ifUnmodifiedSince *date.TimeRFC1123) (result autorest.Response, err error) {
	req, err := client.EnableAutoScalePreparer(ctx, poolID, poolEnableAutoScaleParameter, timeout, clientRequestID, returnClientRequestID, ocpDate, ifMatch, ifNoneMatch, ifModifiedSince, ifUnmodifiedSince)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "EnableAutoScale", nil, "Failure preparing request")
		return
	}

	resp, err := client.EnableAutoScaleSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "EnableAutoScale", resp, "Failure sending request")
		return
	}

	result, err = client.EnableAutoScaleResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "EnableAutoScale", resp, "Failure responding to request")
	}

	return
}

// EnableAutoScalePreparer prepares the EnableAutoScale request.
func (client PoolClient) EnableAutoScalePreparer(ctx context.Context, poolID string, poolEnableAutoScaleParameter PoolEnableAutoScaleParameter, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123, ifMatch string, ifNoneMatch string, ifModifiedSince *date.TimeRFC1123, ifUnmodifiedSince *date.TimeRFC1123) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"poolId": autorest.Encode("path", poolID),
	}

	const APIVersion = "2016-07-01.3.1"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 30)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; odata=minimalmetadata; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/pools/{poolId}/enableautoscale", pathParameters),
		autorest.WithJSON(poolEnableAutoScaleParameter),
		autorest.WithQueryParameters(queryParameters))
	if len(clientRequestID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("client-request-id", autorest.String(clientRequestID)))
	}
	if returnClientRequestID != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(returnClientRequestID)))
	} else {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(false)))
	}
	if ocpDate != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("ocp-date", autorest.String(ocpDate)))
	}
	if len(ifMatch) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Match", autorest.String(ifMatch)))
	}
	if len(ifNoneMatch) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-None-Match", autorest.String(ifNoneMatch)))
	}
	if ifModifiedSince != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Modified-Since", autorest.String(ifModifiedSince)))
	}
	if ifUnmodifiedSince != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Unmodified-Since", autorest.String(ifUnmodifiedSince)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// EnableAutoScaleSender sends the EnableAutoScale request. The method will close the
// http.Response Body if it receives an error.
func (client PoolClient) EnableAutoScaleSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// EnableAutoScaleResponder handles the response to the EnableAutoScale request. The method always
// closes the http.Response Body.
func (client PoolClient) EnableAutoScaleResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// EvaluateAutoScale this API is primarily for validating an autoscale formula, as it simply returns the result without
// applying the formula to the pool.
//
// poolID is the ID of the pool on which to evaluate the automatic scaling formula. poolEvaluateAutoScaleParameter
// is the parameters for the request. timeout is the maximum time that the server can spend processing the request,
// in seconds. The default is 30 seconds. clientRequestID is the caller-generated request identity, in the form of
// a GUID with no decoration such as curly braces, e.g. 9C4D50EE-2D56-4CD3-8152-34347DC9F2B0. returnClientRequestID
// is whether the server should return the client-request-id in the response. ocpDate is the time the request was
// issued. If not specified, this header will be automatically populated with the current system clock time.
func (client PoolClient) EvaluateAutoScale(ctx context.Context, poolID string, poolEvaluateAutoScaleParameter PoolEvaluateAutoScaleParameter, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123) (result AutoScaleRun, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: poolEvaluateAutoScaleParameter,
			Constraints: []validation.Constraint{{Target: "poolEvaluateAutoScaleParameter.AutoScaleFormula", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("batch.PoolClient", "EvaluateAutoScale", err.Error())
	}

	req, err := client.EvaluateAutoScalePreparer(ctx, poolID, poolEvaluateAutoScaleParameter, timeout, clientRequestID, returnClientRequestID, ocpDate)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "EvaluateAutoScale", nil, "Failure preparing request")
		return
	}

	resp, err := client.EvaluateAutoScaleSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "EvaluateAutoScale", resp, "Failure sending request")
		return
	}

	result, err = client.EvaluateAutoScaleResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "EvaluateAutoScale", resp, "Failure responding to request")
	}

	return
}

// EvaluateAutoScalePreparer prepares the EvaluateAutoScale request.
func (client PoolClient) EvaluateAutoScalePreparer(ctx context.Context, poolID string, poolEvaluateAutoScaleParameter PoolEvaluateAutoScaleParameter, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"poolId": autorest.Encode("path", poolID),
	}

	const APIVersion = "2016-07-01.3.1"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 30)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; odata=minimalmetadata; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/pools/{poolId}/evaluateautoscale", pathParameters),
		autorest.WithJSON(poolEvaluateAutoScaleParameter),
		autorest.WithQueryParameters(queryParameters))
	if len(clientRequestID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("client-request-id", autorest.String(clientRequestID)))
	}
	if returnClientRequestID != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(returnClientRequestID)))
	} else {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(false)))
	}
	if ocpDate != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("ocp-date", autorest.String(ocpDate)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// EvaluateAutoScaleSender sends the EvaluateAutoScale request. The method will close the
// http.Response Body if it receives an error.
func (client PoolClient) EvaluateAutoScaleSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// EvaluateAutoScaleResponder handles the response to the EvaluateAutoScale request. The method always
// closes the http.Response Body.
func (client PoolClient) EvaluateAutoScaleResponder(resp *http.Response) (result AutoScaleRun, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Exists gets basic properties of a pool.
//
// poolID is the ID of the pool to get. timeout is the maximum time that the server can spend processing the
// request, in seconds. The default is 30 seconds. clientRequestID is the caller-generated request identity, in the
// form of a GUID with no decoration such as curly braces, e.g. 9C4D50EE-2D56-4CD3-8152-34347DC9F2B0.
// returnClientRequestID is whether the server should return the client-request-id in the response. ocpDate is the
// time the request was issued. If not specified, this header will be automatically populated with the current
// system clock time. ifMatch is an ETag is specified. Specify this header to perform the operation only if the
// resource's ETag is an exact match as specified. ifNoneMatch is an ETag is specified. Specify this header to
// perform the operation only if the resource's ETag does not match the specified ETag. ifModifiedSince is specify
// this header to perform the operation only if the resource has been modified since the specified date/time.
// ifUnmodifiedSince is specify this header to perform the operation only if the resource has not been modified
// since the specified date/time.
func (client PoolClient) Exists(ctx context.Context, poolID string, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123, ifMatch string, ifNoneMatch string, ifModifiedSince *date.TimeRFC1123, ifUnmodifiedSince *date.TimeRFC1123) (result autorest.Response, err error) {
	req, err := client.ExistsPreparer(ctx, poolID, timeout, clientRequestID, returnClientRequestID, ocpDate, ifMatch, ifNoneMatch, ifModifiedSince, ifUnmodifiedSince)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "Exists", nil, "Failure preparing request")
		return
	}

	resp, err := client.ExistsSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "Exists", resp, "Failure sending request")
		return
	}

	result, err = client.ExistsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "Exists", resp, "Failure responding to request")
	}

	return
}

// ExistsPreparer prepares the Exists request.
func (client PoolClient) ExistsPreparer(ctx context.Context, poolID string, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123, ifMatch string, ifNoneMatch string, ifModifiedSince *date.TimeRFC1123, ifUnmodifiedSince *date.TimeRFC1123) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"poolId": autorest.Encode("path", poolID),
	}

	const APIVersion = "2016-07-01.3.1"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 30)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsHead(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/pools/{poolId}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	if len(clientRequestID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("client-request-id", autorest.String(clientRequestID)))
	}
	if returnClientRequestID != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(returnClientRequestID)))
	} else {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(false)))
	}
	if ocpDate != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("ocp-date", autorest.String(ocpDate)))
	}
	if len(ifMatch) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Match", autorest.String(ifMatch)))
	}
	if len(ifNoneMatch) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-None-Match", autorest.String(ifNoneMatch)))
	}
	if ifModifiedSince != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Modified-Since", autorest.String(ifModifiedSince)))
	}
	if ifUnmodifiedSince != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Unmodified-Since", autorest.String(ifUnmodifiedSince)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ExistsSender sends the Exists request. The method will close the
// http.Response Body if it receives an error.
func (client PoolClient) ExistsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ExistsResponder handles the response to the Exists request. The method always
// closes the http.Response Body.
func (client PoolClient) ExistsResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusNotFound),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Get gets information about the specified pool.
//
// poolID is the ID of the pool to get. selectParameter is an OData $select clause. expand is an OData $expand
// clause. timeout is the maximum time that the server can spend processing the request, in seconds. The default is
// 30 seconds. clientRequestID is the caller-generated request identity, in the form of a GUID with no decoration
// such as curly braces, e.g. 9C4D50EE-2D56-4CD3-8152-34347DC9F2B0. returnClientRequestID is whether the server
// should return the client-request-id in the response. ocpDate is the time the request was issued. If not
// specified, this header will be automatically populated with the current system clock time. ifMatch is an ETag is
// specified. Specify this header to perform the operation only if the resource's ETag is an exact match as
// specified. ifNoneMatch is an ETag is specified. Specify this header to perform the operation only if the
// resource's ETag does not match the specified ETag. ifModifiedSince is specify this header to perform the
// operation only if the resource has been modified since the specified date/time. ifUnmodifiedSince is specify
// this header to perform the operation only if the resource has not been modified since the specified date/time.
func (client PoolClient) Get(ctx context.Context, poolID string, selectParameter string, expand string, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123, ifMatch string, ifNoneMatch string, ifModifiedSince *date.TimeRFC1123, ifUnmodifiedSince *date.TimeRFC1123) (result CloudPool, err error) {
	req, err := client.GetPreparer(ctx, poolID, selectParameter, expand, timeout, clientRequestID, returnClientRequestID, ocpDate, ifMatch, ifNoneMatch, ifModifiedSince, ifUnmodifiedSince)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "Get", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "Get", resp, "Failure sending request")
		return
	}

	result, err = client.GetResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "Get", resp, "Failure responding to request")
	}

	return
}

// GetPreparer prepares the Get request.
func (client PoolClient) GetPreparer(ctx context.Context, poolID string, selectParameter string, expand string, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123, ifMatch string, ifNoneMatch string, ifModifiedSince *date.TimeRFC1123, ifUnmodifiedSince *date.TimeRFC1123) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"poolId": autorest.Encode("path", poolID),
	}

	const APIVersion = "2016-07-01.3.1"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(expand) > 0 {
		queryParameters["$expand"] = autorest.Encode("query", expand)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 30)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/pools/{poolId}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	if len(clientRequestID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("client-request-id", autorest.String(clientRequestID)))
	}
	if returnClientRequestID != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(returnClientRequestID)))
	} else {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(false)))
	}
	if ocpDate != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("ocp-date", autorest.String(ocpDate)))
	}
	if len(ifMatch) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Match", autorest.String(ifMatch)))
	}
	if len(ifNoneMatch) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-None-Match", autorest.String(ifNoneMatch)))
	}
	if ifModifiedSince != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Modified-Since", autorest.String(ifModifiedSince)))
	}
	if ifUnmodifiedSince != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Unmodified-Since", autorest.String(ifUnmodifiedSince)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetSender sends the Get request. The method will close the
// http.Response Body if it receives an error.
func (client PoolClient) GetSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetResponder handles the response to the Get request. The method always
// closes the http.Response Body.
func (client PoolClient) GetResponder(resp *http.Response) (result CloudPool, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetAllPoolsLifetimeStatistics statistics are aggregated across all pools that have ever existed in the account, from
// account creation to the last update time of the statistics.
//
// timeout is the maximum time that the server can spend processing the request, in seconds. The default is 30
// seconds. clientRequestID is the caller-generated request identity, in the form of a GUID with no decoration such
// as curly braces, e.g. 9C4D50EE-2D56-4CD3-8152-34347DC9F2B0. returnClientRequestID is whether the server should
// return the client-request-id in the response. ocpDate is the time the request was issued. If not specified, this
// header will be automatically populated with the current system clock time.
func (client PoolClient) GetAllPoolsLifetimeStatistics(ctx context.Context, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123) (result PoolStatistics, err error) {
	req, err := client.GetAllPoolsLifetimeStatisticsPreparer(ctx, timeout, clientRequestID, returnClientRequestID, ocpDate)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "GetAllPoolsLifetimeStatistics", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetAllPoolsLifetimeStatisticsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "GetAllPoolsLifetimeStatistics", resp, "Failure sending request")
		return
	}

	result, err = client.GetAllPoolsLifetimeStatisticsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "GetAllPoolsLifetimeStatistics", resp, "Failure responding to request")
	}

	return
}

// GetAllPoolsLifetimeStatisticsPreparer prepares the GetAllPoolsLifetimeStatistics request.
func (client PoolClient) GetAllPoolsLifetimeStatisticsPreparer(ctx context.Context, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123) (*http.Request, error) {
	const APIVersion = "2016-07-01.3.1"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 30)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/lifetimepoolstats"),
		autorest.WithQueryParameters(queryParameters))
	if len(clientRequestID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("client-request-id", autorest.String(clientRequestID)))
	}
	if returnClientRequestID != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(returnClientRequestID)))
	} else {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(false)))
	}
	if ocpDate != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("ocp-date", autorest.String(ocpDate)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetAllPoolsLifetimeStatisticsSender sends the GetAllPoolsLifetimeStatistics request. The method will close the
// http.Response Body if it receives an error.
func (client PoolClient) GetAllPoolsLifetimeStatisticsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetAllPoolsLifetimeStatisticsResponder handles the response to the GetAllPoolsLifetimeStatistics request. The method always
// closes the http.Response Body.
func (client PoolClient) GetAllPoolsLifetimeStatisticsResponder(resp *http.Response) (result PoolStatistics, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// List sends the list request.
//
// filter is an OData $filter clause. selectParameter is an OData $select clause. expand is an OData $expand
// clause. maxResults is the maximum number of items to return in the response. A maximum of 1000 pools can be
// returned. timeout is the maximum time that the server can spend processing the request, in seconds. The default
// is 30 seconds. clientRequestID is the caller-generated request identity, in the form of a GUID with no
// decoration such as curly braces, e.g. 9C4D50EE-2D56-4CD3-8152-34347DC9F2B0. returnClientRequestID is whether the
// server should return the client-request-id in the response. ocpDate is the time the request was issued. If not
// specified, this header will be automatically populated with the current system clock time.
func (client PoolClient) List(ctx context.Context, filter string, selectParameter string, expand string, maxResults *int32, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123) (result CloudPoolListResultPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxResults,
			Constraints: []validation.Constraint{{Target: "maxResults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxResults", Name: validation.InclusiveMaximum, Rule: 1000, Chain: nil},
					{Target: "maxResults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("batch.PoolClient", "List", err.Error())
	}

	result.fn = client.listNextResults
	req, err := client.ListPreparer(ctx, filter, selectParameter, expand, maxResults, timeout, clientRequestID, returnClientRequestID, ocpDate)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "List", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListSender(req)
	if err != nil {
		result.cplr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "List", resp, "Failure sending request")
		return
	}

	result.cplr, err = client.ListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "List", resp, "Failure responding to request")
	}

	return
}

// ListPreparer prepares the List request.
func (client PoolClient) ListPreparer(ctx context.Context, filter string, selectParameter string, expand string, maxResults *int32, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123) (*http.Request, error) {
	const APIVersion = "2016-07-01.3.1"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(expand) > 0 {
		queryParameters["$expand"] = autorest.Encode("query", expand)
	}
	if maxResults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxResults)
	} else {
		queryParameters["maxresults"] = autorest.Encode("query", 1000)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 30)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/pools"),
		autorest.WithQueryParameters(queryParameters))
	if len(clientRequestID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("client-request-id", autorest.String(clientRequestID)))
	}
	if returnClientRequestID != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(returnClientRequestID)))
	} else {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(false)))
	}
	if ocpDate != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("ocp-date", autorest.String(ocpDate)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListSender sends the List request. The method will close the
// http.Response Body if it receives an error.
func (client PoolClient) ListSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListResponder handles the response to the List request. The method always
// closes the http.Response Body.
func (client PoolClient) ListResponder(resp *http.Response) (result CloudPoolListResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listNextResults retrieves the next set of results, if any.
func (client PoolClient) listNextResults(lastResults CloudPoolListResult) (result CloudPoolListResult, err error) {
	req, err := lastResults.cloudPoolListResultPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "batch.PoolClient", "listNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "batch.PoolClient", "listNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "listNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListComplete enumerates all values, automatically crossing page boundaries as required.
func (client PoolClient) ListComplete(ctx context.Context, filter string, selectParameter string, expand string, maxResults *int32, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123) (result CloudPoolListResultIterator, err error) {
	result.page, err = client.List(ctx, filter, selectParameter, expand, maxResults, timeout, clientRequestID, returnClientRequestID, ocpDate)
	return
}

// ListPoolUsageMetrics if you do not specify a $filter clause including a poolId, the response includes all pools that
// existed in the account in the time range of the returned aggregation intervals.
//
// startTime is the earliest time from which to include metrics. This must be at least two and a half hours before
// the current time. If not specified this defaults to the start time of the last aggregation interval currently
// available. endTime is the latest time from which to include metrics. This must be at least two hours before the
// current time. If not specified this defaults to the end time of the last aggregation interval currently
// available. filter is an OData $filter clause. If this is not specified the response includes all pools that
// existed in the account in the time range of the returned aggregation intervals. maxResults is the maximum number
// of items to return in the response. A maximum of 1000 results will be returned. timeout is the maximum time that
// the server can spend processing the request, in seconds. The default is 30 seconds. clientRequestID is the
// caller-generated request identity, in the form of a GUID with no decoration such as curly braces, e.g.
// 9C4D50EE-2D56-4CD3-8152-34347DC9F2B0. returnClientRequestID is whether the server should return the
// client-request-id in the response. ocpDate is the time the request was issued. If not specified, this header
// will be automatically populated with the current system clock time.
func (client PoolClient) ListPoolUsageMetrics(ctx context.Context, startTime *date.Time, endTime *date.Time, filter string, maxResults *int32, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123) (result PoolListPoolUsageMetricsResultPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: maxResults,
			Constraints: []validation.Constraint{{Target: "maxResults", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "maxResults", Name: validation.InclusiveMaximum, Rule: 1000, Chain: nil},
					{Target: "maxResults", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil},
				}}}}}); err != nil {
		return result, validation.NewError("batch.PoolClient", "ListPoolUsageMetrics", err.Error())
	}

	result.fn = client.listPoolUsageMetricsNextResults
	req, err := client.ListPoolUsageMetricsPreparer(ctx, startTime, endTime, filter, maxResults, timeout, clientRequestID, returnClientRequestID, ocpDate)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "ListPoolUsageMetrics", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListPoolUsageMetricsSender(req)
	if err != nil {
		result.plpumr.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "ListPoolUsageMetrics", resp, "Failure sending request")
		return
	}

	result.plpumr, err = client.ListPoolUsageMetricsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "ListPoolUsageMetrics", resp, "Failure responding to request")
	}

	return
}

// ListPoolUsageMetricsPreparer prepares the ListPoolUsageMetrics request.
func (client PoolClient) ListPoolUsageMetricsPreparer(ctx context.Context, startTime *date.Time, endTime *date.Time, filter string, maxResults *int32, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123) (*http.Request, error) {
	const APIVersion = "2016-07-01.3.1"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if startTime != nil {
		queryParameters["starttime"] = autorest.Encode("query", *startTime)
	}
	if endTime != nil {
		queryParameters["endtime"] = autorest.Encode("query", *endTime)
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if maxResults != nil {
		queryParameters["maxresults"] = autorest.Encode("query", *maxResults)
	} else {
		queryParameters["maxresults"] = autorest.Encode("query", 1000)
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 30)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPath("/poolusagemetrics"),
		autorest.WithQueryParameters(queryParameters))
	if len(clientRequestID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("client-request-id", autorest.String(clientRequestID)))
	}
	if returnClientRequestID != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(returnClientRequestID)))
	} else {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(false)))
	}
	if ocpDate != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("ocp-date", autorest.String(ocpDate)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListPoolUsageMetricsSender sends the ListPoolUsageMetrics request. The method will close the
// http.Response Body if it receives an error.
func (client PoolClient) ListPoolUsageMetricsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListPoolUsageMetricsResponder handles the response to the ListPoolUsageMetrics request. The method always
// closes the http.Response Body.
func (client PoolClient) ListPoolUsageMetricsResponder(resp *http.Response) (result PoolListPoolUsageMetricsResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listPoolUsageMetricsNextResults retrieves the next set of results, if any.
func (client PoolClient) listPoolUsageMetricsNextResults(lastResults PoolListPoolUsageMetricsResult) (result PoolListPoolUsageMetricsResult, err error) {
	req, err := lastResults.poolListPoolUsageMetricsResultPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "batch.PoolClient", "listPoolUsageMetricsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListPoolUsageMetricsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "batch.PoolClient", "listPoolUsageMetricsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListPoolUsageMetricsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "listPoolUsageMetricsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListPoolUsageMetricsComplete enumerates all values, automatically crossing page boundaries as required.
func (client PoolClient) ListPoolUsageMetricsComplete(ctx context.Context, startTime *date.Time, endTime *date.Time, filter string, maxResults *int32, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123) (result PoolListPoolUsageMetricsResultIterator, err error) {
	result.page, err = client.ListPoolUsageMetrics(ctx, startTime, endTime, filter, maxResults, timeout, clientRequestID, returnClientRequestID, ocpDate)
	return
}

// Patch this only replaces the pool properties specified in the request. For example, if the pool has a start task
// associated with it, and a request does not specify a start task element, then the pool keeps the existing start
// task.
//
// poolID is the ID of the pool to update. poolPatchParameter is the parameters for the request. timeout is the
// maximum time that the server can spend processing the request, in seconds. The default is 30 seconds.
// clientRequestID is the caller-generated request identity, in the form of a GUID with no decoration such as curly
// braces, e.g. 9C4D50EE-2D56-4CD3-8152-34347DC9F2B0. returnClientRequestID is whether the server should return the
// client-request-id in the response. ocpDate is the time the request was issued. If not specified, this header
// will be automatically populated with the current system clock time. ifMatch is an ETag is specified. Specify
// this header to perform the operation only if the resource's ETag is an exact match as specified. ifNoneMatch is
// an ETag is specified. Specify this header to perform the operation only if the resource's ETag does not match
// the specified ETag. ifModifiedSince is specify this header to perform the operation only if the resource has
// been modified since the specified date/time. ifUnmodifiedSince is specify this header to perform the operation
// only if the resource has not been modified since the specified date/time.
func (client PoolClient) Patch(ctx context.Context, poolID string, poolPatchParameter PoolPatchParameter, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123, ifMatch string, ifNoneMatch string, ifModifiedSince *date.TimeRFC1123, ifUnmodifiedSince *date.TimeRFC1123) (result autorest.Response, err error) {
	req, err := client.PatchPreparer(ctx, poolID, poolPatchParameter, timeout, clientRequestID, returnClientRequestID, ocpDate, ifMatch, ifNoneMatch, ifModifiedSince, ifUnmodifiedSince)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "Patch", nil, "Failure preparing request")
		return
	}

	resp, err := client.PatchSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "Patch", resp, "Failure sending request")
		return
	}

	result, err = client.PatchResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "Patch", resp, "Failure responding to request")
	}

	return
}

// PatchPreparer prepares the Patch request.
func (client PoolClient) PatchPreparer(ctx context.Context, poolID string, poolPatchParameter PoolPatchParameter, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123, ifMatch string, ifNoneMatch string, ifModifiedSince *date.TimeRFC1123, ifUnmodifiedSince *date.TimeRFC1123) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"poolId": autorest.Encode("path", poolID),
	}

	const APIVersion = "2016-07-01.3.1"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 30)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; odata=minimalmetadata; charset=utf-8"),
		autorest.AsPatch(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/pools/{poolId}", pathParameters),
		autorest.WithJSON(poolPatchParameter),
		autorest.WithQueryParameters(queryParameters))
	if len(clientRequestID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("client-request-id", autorest.String(clientRequestID)))
	}
	if returnClientRequestID != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(returnClientRequestID)))
	} else {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(false)))
	}
	if ocpDate != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("ocp-date", autorest.String(ocpDate)))
	}
	if len(ifMatch) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Match", autorest.String(ifMatch)))
	}
	if len(ifNoneMatch) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-None-Match", autorest.String(ifNoneMatch)))
	}
	if ifModifiedSince != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Modified-Since", autorest.String(ifModifiedSince)))
	}
	if ifUnmodifiedSince != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Unmodified-Since", autorest.String(ifUnmodifiedSince)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// PatchSender sends the Patch request. The method will close the
// http.Response Body if it receives an error.
func (client PoolClient) PatchSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// PatchResponder handles the response to the Patch request. The method always
// closes the http.Response Body.
func (client PoolClient) PatchResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// RemoveNodes this operation can only run when the allocation state of the pool is steady. When this operation runs,
// the allocation state changes from steady to resizing.
//
// poolID is the ID of the pool from which you want to remove nodes. nodeRemoveParameter is the parameters for the
// request. timeout is the maximum time that the server can spend processing the request, in seconds. The default
// is 30 seconds. clientRequestID is the caller-generated request identity, in the form of a GUID with no
// decoration such as curly braces, e.g. 9C4D50EE-2D56-4CD3-8152-34347DC9F2B0. returnClientRequestID is whether the
// server should return the client-request-id in the response. ocpDate is the time the request was issued. If not
// specified, this header will be automatically populated with the current system clock time. ifMatch is an ETag is
// specified. Specify this header to perform the operation only if the resource's ETag is an exact match as
// specified. ifNoneMatch is an ETag is specified. Specify this header to perform the operation only if the
// resource's ETag does not match the specified ETag. ifModifiedSince is specify this header to perform the
// operation only if the resource has been modified since the specified date/time. ifUnmodifiedSince is specify
// this header to perform the operation only if the resource has not been modified since the specified date/time.
func (client PoolClient) RemoveNodes(ctx context.Context, poolID string, nodeRemoveParameter NodeRemoveParameter, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123, ifMatch string, ifNoneMatch string, ifModifiedSince *date.TimeRFC1123, ifUnmodifiedSince *date.TimeRFC1123) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: nodeRemoveParameter,
			Constraints: []validation.Constraint{{Target: "nodeRemoveParameter.NodeList", Name: validation.Null, Rule: true,
				Chain: []validation.Constraint{{Target: "nodeRemoveParameter.NodeList", Name: validation.MaxItems, Rule: 100, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("batch.PoolClient", "RemoveNodes", err.Error())
	}

	req, err := client.RemoveNodesPreparer(ctx, poolID, nodeRemoveParameter, timeout, clientRequestID, returnClientRequestID, ocpDate, ifMatch, ifNoneMatch, ifModifiedSince, ifUnmodifiedSince)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "RemoveNodes", nil, "Failure preparing request")
		return
	}

	resp, err := client.RemoveNodesSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "RemoveNodes", resp, "Failure sending request")
		return
	}

	result, err = client.RemoveNodesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "RemoveNodes", resp, "Failure responding to request")
	}

	return
}

// RemoveNodesPreparer prepares the RemoveNodes request.
func (client PoolClient) RemoveNodesPreparer(ctx context.Context, poolID string, nodeRemoveParameter NodeRemoveParameter, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123, ifMatch string, ifNoneMatch string, ifModifiedSince *date.TimeRFC1123, ifUnmodifiedSince *date.TimeRFC1123) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"poolId": autorest.Encode("path", poolID),
	}

	const APIVersion = "2016-07-01.3.1"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 30)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; odata=minimalmetadata; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/pools/{poolId}/removenodes", pathParameters),
		autorest.WithJSON(nodeRemoveParameter),
		autorest.WithQueryParameters(queryParameters))
	if len(clientRequestID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("client-request-id", autorest.String(clientRequestID)))
	}
	if returnClientRequestID != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(returnClientRequestID)))
	} else {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(false)))
	}
	if ocpDate != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("ocp-date", autorest.String(ocpDate)))
	}
	if len(ifMatch) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Match", autorest.String(ifMatch)))
	}
	if len(ifNoneMatch) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-None-Match", autorest.String(ifNoneMatch)))
	}
	if ifModifiedSince != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Modified-Since", autorest.String(ifModifiedSince)))
	}
	if ifUnmodifiedSince != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Unmodified-Since", autorest.String(ifUnmodifiedSince)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RemoveNodesSender sends the RemoveNodes request. The method will close the
// http.Response Body if it receives an error.
func (client PoolClient) RemoveNodesSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RemoveNodesResponder handles the response to the RemoveNodes request. The method always
// closes the http.Response Body.
func (client PoolClient) RemoveNodesResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Resize you can only resize a pool when its allocation state is steady. If the pool is already resizing, the request
// fails with status code 409. When you resize a pool, the pool's allocation state changes from steady to resizing. You
// cannot resize pools which are configured for automatic scaling. If you try to do this, the Batch service returns an
// error 409. If you resize a pool downwards, the Batch service chooses which nodes to remove. To remove specific
// nodes, use the pool remove nodes API instead.
//
// poolID is the ID of the pool to resize. poolResizeParameter is the parameters for the request. timeout is the
// maximum time that the server can spend processing the request, in seconds. The default is 30 seconds.
// clientRequestID is the caller-generated request identity, in the form of a GUID with no decoration such as curly
// braces, e.g. 9C4D50EE-2D56-4CD3-8152-34347DC9F2B0. returnClientRequestID is whether the server should return the
// client-request-id in the response. ocpDate is the time the request was issued. If not specified, this header
// will be automatically populated with the current system clock time. ifMatch is an ETag is specified. Specify
// this header to perform the operation only if the resource's ETag is an exact match as specified. ifNoneMatch is
// an ETag is specified. Specify this header to perform the operation only if the resource's ETag does not match
// the specified ETag. ifModifiedSince is specify this header to perform the operation only if the resource has
// been modified since the specified date/time. ifUnmodifiedSince is specify this header to perform the operation
// only if the resource has not been modified since the specified date/time.
func (client PoolClient) Resize(ctx context.Context, poolID string, poolResizeParameter PoolResizeParameter, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123, ifMatch string, ifNoneMatch string, ifModifiedSince *date.TimeRFC1123, ifUnmodifiedSince *date.TimeRFC1123) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: poolResizeParameter,
			Constraints: []validation.Constraint{{Target: "poolResizeParameter.TargetDedicated", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("batch.PoolClient", "Resize", err.Error())
	}

	req, err := client.ResizePreparer(ctx, poolID, poolResizeParameter, timeout, clientRequestID, returnClientRequestID, ocpDate, ifMatch, ifNoneMatch, ifModifiedSince, ifUnmodifiedSince)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "Resize", nil, "Failure preparing request")
		return
	}

	resp, err := client.ResizeSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "Resize", resp, "Failure sending request")
		return
	}

	result, err = client.ResizeResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "Resize", resp, "Failure responding to request")
	}

	return
}

// ResizePreparer prepares the Resize request.
func (client PoolClient) ResizePreparer(ctx context.Context, poolID string, poolResizeParameter PoolResizeParameter, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123, ifMatch string, ifNoneMatch string, ifModifiedSince *date.TimeRFC1123, ifUnmodifiedSince *date.TimeRFC1123) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"poolId": autorest.Encode("path", poolID),
	}

	const APIVersion = "2016-07-01.3.1"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 30)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; odata=minimalmetadata; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/pools/{poolId}/resize", pathParameters),
		autorest.WithJSON(poolResizeParameter),
		autorest.WithQueryParameters(queryParameters))
	if len(clientRequestID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("client-request-id", autorest.String(clientRequestID)))
	}
	if returnClientRequestID != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(returnClientRequestID)))
	} else {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(false)))
	}
	if ocpDate != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("ocp-date", autorest.String(ocpDate)))
	}
	if len(ifMatch) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Match", autorest.String(ifMatch)))
	}
	if len(ifNoneMatch) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-None-Match", autorest.String(ifNoneMatch)))
	}
	if ifModifiedSince != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Modified-Since", autorest.String(ifModifiedSince)))
	}
	if ifUnmodifiedSince != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Unmodified-Since", autorest.String(ifUnmodifiedSince)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ResizeSender sends the Resize request. The method will close the
// http.Response Body if it receives an error.
func (client PoolClient) ResizeSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ResizeResponder handles the response to the Resize request. The method always
// closes the http.Response Body.
func (client PoolClient) ResizeResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByClosing())
	result.Response = resp
	return
}

// StopResize this does not restore the pool to its previous state before the resize operation: it only stops any
// further changes being made, and the pool maintains its current state. A resize operation need not be an explicit
// resize pool request; this API can also be used to halt the initial sizing of the pool when it is created.
//
// poolID is the ID of the pool whose resizing you want to stop. timeout is the maximum time that the server can
// spend processing the request, in seconds. The default is 30 seconds. clientRequestID is the caller-generated
// request identity, in the form of a GUID with no decoration such as curly braces, e.g.
// 9C4D50EE-2D56-4CD3-8152-34347DC9F2B0. returnClientRequestID is whether the server should return the
// client-request-id in the response. ocpDate is the time the request was issued. If not specified, this header
// will be automatically populated with the current system clock time. ifMatch is an ETag is specified. Specify
// this header to perform the operation only if the resource's ETag is an exact match as specified. ifNoneMatch is
// an ETag is specified. Specify this header to perform the operation only if the resource's ETag does not match
// the specified ETag. ifModifiedSince is specify this header to perform the operation only if the resource has
// been modified since the specified date/time. ifUnmodifiedSince is specify this header to perform the operation
// only if the resource has not been modified since the specified date/time.
func (client PoolClient) StopResize(ctx context.Context, poolID string, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123, ifMatch string, ifNoneMatch string, ifModifiedSince *date.TimeRFC1123, ifUnmodifiedSince *date.TimeRFC1123) (result autorest.Response, err error) {
	req, err := client.StopResizePreparer(ctx, poolID, timeout, clientRequestID, returnClientRequestID, ocpDate, ifMatch, ifNoneMatch, ifModifiedSince, ifUnmodifiedSince)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "StopResize", nil, "Failure preparing request")
		return
	}

	resp, err := client.StopResizeSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "StopResize", resp, "Failure sending request")
		return
	}

	result, err = client.StopResizeResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "StopResize", resp, "Failure responding to request")
	}

	return
}

// StopResizePreparer prepares the StopResize request.
func (client PoolClient) StopResizePreparer(ctx context.Context, poolID string, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123, ifMatch string, ifNoneMatch string, ifModifiedSince *date.TimeRFC1123, ifUnmodifiedSince *date.TimeRFC1123) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"poolId": autorest.Encode("path", poolID),
	}

	const APIVersion = "2016-07-01.3.1"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 30)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/pools/{poolId}/stopresize", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	if len(clientRequestID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("client-request-id", autorest.String(clientRequestID)))
	}
	if returnClientRequestID != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(returnClientRequestID)))
	} else {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(false)))
	}
	if ocpDate != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("ocp-date", autorest.String(ocpDate)))
	}
	if len(ifMatch) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Match", autorest.String(ifMatch)))
	}
	if len(ifNoneMatch) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-None-Match", autorest.String(ifNoneMatch)))
	}
	if ifModifiedSince != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Modified-Since", autorest.String(ifModifiedSince)))
	}
	if ifUnmodifiedSince != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Unmodified-Since", autorest.String(ifUnmodifiedSince)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// StopResizeSender sends the StopResize request. The method will close the
// http.Response Body if it receives an error.
func (client PoolClient) StopResizeSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// StopResizeResponder handles the response to the StopResize request. The method always
// closes the http.Response Body.
func (client PoolClient) StopResizeResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByClosing())
	result.Response = resp
	return
}

// UpdateProperties this fully replaces all the updateable properties of the pool. For example, if the pool has a start
// task associated with it and if start task is not specified with this request, then the Batch service will remove the
// existing start task.
//
// poolID is the ID of the pool to update. poolUpdatePropertiesParameter is the parameters for the request. timeout
// is the maximum time that the server can spend processing the request, in seconds. The default is 30 seconds.
// clientRequestID is the caller-generated request identity, in the form of a GUID with no decoration such as curly
// braces, e.g. 9C4D50EE-2D56-4CD3-8152-34347DC9F2B0. returnClientRequestID is whether the server should return the
// client-request-id in the response. ocpDate is the time the request was issued. If not specified, this header
// will be automatically populated with the current system clock time.
func (client PoolClient) UpdateProperties(ctx context.Context, poolID string, poolUpdatePropertiesParameter PoolUpdatePropertiesParameter, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: poolUpdatePropertiesParameter,
			Constraints: []validation.Constraint{{Target: "poolUpdatePropertiesParameter.StartTask", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "poolUpdatePropertiesParameter.StartTask.CommandLine", Name: validation.Null, Rule: true, Chain: nil}}},
				{Target: "poolUpdatePropertiesParameter.CertificateReferences", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "poolUpdatePropertiesParameter.ApplicationPackageReferences", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "poolUpdatePropertiesParameter.Metadata", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("batch.PoolClient", "UpdateProperties", err.Error())
	}

	req, err := client.UpdatePropertiesPreparer(ctx, poolID, poolUpdatePropertiesParameter, timeout, clientRequestID, returnClientRequestID, ocpDate)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "UpdateProperties", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdatePropertiesSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "UpdateProperties", resp, "Failure sending request")
		return
	}

	result, err = client.UpdatePropertiesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "UpdateProperties", resp, "Failure responding to request")
	}

	return
}

// UpdatePropertiesPreparer prepares the UpdateProperties request.
func (client PoolClient) UpdatePropertiesPreparer(ctx context.Context, poolID string, poolUpdatePropertiesParameter PoolUpdatePropertiesParameter, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"poolId": autorest.Encode("path", poolID),
	}

	const APIVersion = "2016-07-01.3.1"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 30)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; odata=minimalmetadata; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/pools/{poolId}/updateproperties", pathParameters),
		autorest.WithJSON(poolUpdatePropertiesParameter),
		autorest.WithQueryParameters(queryParameters))
	if len(clientRequestID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("client-request-id", autorest.String(clientRequestID)))
	}
	if returnClientRequestID != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(returnClientRequestID)))
	} else {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(false)))
	}
	if ocpDate != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("ocp-date", autorest.String(ocpDate)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdatePropertiesSender sends the UpdateProperties request. The method will close the
// http.Response Body if it receives an error.
func (client PoolClient) UpdatePropertiesSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// UpdatePropertiesResponder handles the response to the UpdateProperties request. The method always
// closes the http.Response Body.
func (client PoolClient) UpdatePropertiesResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusNoContent),
		autorest.ByClosing())
	result.Response = resp
	return
}

// UpgradeOS during an upgrade, the Batch service upgrades each compute node in the pool. When a compute node is chosen
// for upgrade, any tasks running on that node are removed from the node and returned to the queue to be rerun later
// (or on a different compute node). The node will be unavailable until the upgrade is complete. This operation results
// in temporarily reduced pool capacity as nodes are taken out of service to be upgraded. Although the Batch service
// tries to avoid upgrading all compute nodes at the same time, it does not guarantee to do this (particularly on small
// pools); therefore, the pool may be temporarily unavailable to run tasks. When this operation runs, the pool state
// changes to upgrading. When all compute nodes have finished upgrading, the pool state returns to active.
//
// poolID is the ID of the pool to upgrade. poolUpgradeOSParameter is the parameters for the request. timeout is
// the maximum time that the server can spend processing the request, in seconds. The default is 30 seconds.
// clientRequestID is the caller-generated request identity, in the form of a GUID with no decoration such as curly
// braces, e.g. 9C4D50EE-2D56-4CD3-8152-34347DC9F2B0. returnClientRequestID is whether the server should return the
// client-request-id in the response. ocpDate is the time the request was issued. If not specified, this header
// will be automatically populated with the current system clock time. ifMatch is an ETag is specified. Specify
// this header to perform the operation only if the resource's ETag is an exact match as specified. ifNoneMatch is
// an ETag is specified. Specify this header to perform the operation only if the resource's ETag does not match
// the specified ETag. ifModifiedSince is specify this header to perform the operation only if the resource has
// been modified since the specified date/time. ifUnmodifiedSince is specify this header to perform the operation
// only if the resource has not been modified since the specified date/time.
func (client PoolClient) UpgradeOS(ctx context.Context, poolID string, poolUpgradeOSParameter PoolUpgradeOSParameter, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123, ifMatch string, ifNoneMatch string, ifModifiedSince *date.TimeRFC1123, ifUnmodifiedSince *date.TimeRFC1123) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: poolUpgradeOSParameter,
			Constraints: []validation.Constraint{{Target: "poolUpgradeOSParameter.TargetOSVersion", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("batch.PoolClient", "UpgradeOS", err.Error())
	}

	req, err := client.UpgradeOSPreparer(ctx, poolID, poolUpgradeOSParameter, timeout, clientRequestID, returnClientRequestID, ocpDate, ifMatch, ifNoneMatch, ifModifiedSince, ifUnmodifiedSince)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "UpgradeOS", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpgradeOSSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "UpgradeOS", resp, "Failure sending request")
		return
	}

	result, err = client.UpgradeOSResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "batch.PoolClient", "UpgradeOS", resp, "Failure responding to request")
	}

	return
}

// UpgradeOSPreparer prepares the UpgradeOS request.
func (client PoolClient) UpgradeOSPreparer(ctx context.Context, poolID string, poolUpgradeOSParameter PoolUpgradeOSParameter, timeout *int32, clientRequestID string, returnClientRequestID *bool, ocpDate *date.TimeRFC1123, ifMatch string, ifNoneMatch string, ifModifiedSince *date.TimeRFC1123, ifUnmodifiedSince *date.TimeRFC1123) (*http.Request, error) {
	pathParameters := map[string]interface{}{
		"poolId": autorest.Encode("path", poolID),
	}

	const APIVersion = "2016-07-01.3.1"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if timeout != nil {
		queryParameters["timeout"] = autorest.Encode("query", *timeout)
	} else {
		queryParameters["timeout"] = autorest.Encode("query", 30)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsContentType("application/json; odata=minimalmetadata; charset=utf-8"),
		autorest.AsPost(),
		autorest.WithBaseURL(client.BaseURI),
		autorest.WithPathParameters("/pools/{poolId}/upgradeos", pathParameters),
		autorest.WithJSON(poolUpgradeOSParameter),
		autorest.WithQueryParameters(queryParameters))
	if len(clientRequestID) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("client-request-id", autorest.String(clientRequestID)))
	}
	if returnClientRequestID != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(returnClientRequestID)))
	} else {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("return-client-request-id", autorest.String(false)))
	}
	if ocpDate != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("ocp-date", autorest.String(ocpDate)))
	}
	if len(ifMatch) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Match", autorest.String(ifMatch)))
	}
	if len(ifNoneMatch) > 0 {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-None-Match", autorest.String(ifNoneMatch)))
	}
	if ifModifiedSince != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Modified-Since", autorest.String(ifModifiedSince)))
	}
	if ifUnmodifiedSince != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithHeader("If-Unmodified-Since", autorest.String(ifUnmodifiedSince)))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpgradeOSSender sends the UpgradeOS request. The method will close the
// http.Response Body if it receives an error.
func (client PoolClient) UpgradeOSSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// UpgradeOSResponder handles the response to the UpgradeOS request. The method always
// closes the http.Response Body.
func (client PoolClient) UpgradeOSResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusAccepted),
		autorest.ByClosing())
	result.Response = resp
	return
}
