package manifests

import (
	"fmt"
	"os"
	"path/filepath"

	"github.com/ghodss/yaml"
	"github.com/pkg/errors"

	"github.com/openshift/installer/pkg/asset"
	"github.com/openshift/installer/pkg/asset/installconfig"
	"github.com/openshift/installer/pkg/asset/templates/content"

	configv1 "github.com/openshift/api/config/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
)

var (
	ingCrdFilename = "cluster-ingress-01-crd.yaml"
	ingCfgFilename = filepath.Join(manifestDir, "cluster-ingress-02-config.yml")
)

// Ingress generates the cluster-ingress-*.yml files.
type Ingress struct {
	config   *configv1.Ingress
	FileList []*asset.File
}

var _ asset.WritableAsset = (*Ingress)(nil)

// Name returns a human friendly name for the asset.
func (*Ingress) Name() string {
	return "Ingress Config"
}

// Dependencies returns all of the dependencies directly needed to generate
// the asset.
func (*Ingress) Dependencies() []asset.Asset {
	return []asset.Asset{
		&installconfig.InstallConfig{},
	}
}

// Generate generates the ingress config and its CRD.
func (ing *Ingress) Generate(dependencies asset.Parents) error {
	installConfig := &installconfig.InstallConfig{}
	dependencies.Get(installConfig)

	ing.config = &configv1.Ingress{
		TypeMeta: metav1.TypeMeta{
			APIVersion: configv1.SchemeGroupVersion.String(),
			Kind:       "Ingress",
		},
		ObjectMeta: metav1.ObjectMeta{
			Name: "cluster",
			// not namespaced
		},
		Spec: configv1.IngressSpec{
			Domain: fmt.Sprintf("apps.%s.%s", installConfig.Config.ObjectMeta.Name, installConfig.Config.BaseDomain),
		},
	}

	configData, err := yaml.Marshal(ing.config)
	if err != nil {
		return errors.Wrapf(err, "failed to create %s manifests from InstallConfig", ing.Name())
	}

	crdData, err := content.GetBootkubeTemplate(ingCrdFilename)
	if err != nil {
		return err
	}

	ing.FileList = []*asset.File{
		{
			Filename: filepath.Join(manifestDir, ingCrdFilename),
			Data:     []byte(crdData),
		},
		{
			Filename: ingCfgFilename,
			Data:     configData,
		},
	}

	return nil
}

// Files returns the files generated by the asset.
func (ing *Ingress) Files() []*asset.File {
	return ing.FileList
}

// Load loads the already-rendered files back from disk.
func (ing *Ingress) Load(f asset.FileFetcher) (bool, error) {
	crdFile, err := f.FetchByName(filepath.Join(manifestDir, ingCrdFilename))
	if err != nil {
		if os.IsNotExist(err) {
			return false, nil
		}
		return false, err
	}

	cfgFile, err := f.FetchByName(ingCfgFilename)
	if err != nil {
		if os.IsNotExist(err) {
			return false, nil
		}

		return false, err
	}

	ingressConfig := &configv1.Ingress{}
	if err := yaml.Unmarshal(cfgFile.Data, ingressConfig); err != nil {
		return false, errors.Wrapf(err, "failed to unmarshal %s", ingCfgFilename)
	}

	fileList := []*asset.File{crdFile, cfgFile}

	ing.FileList, ing.config = fileList, ingressConfig

	return true, nil
}
