package openshift

import (
	"os"
	"path/filepath"

	"github.com/openshift/installer/pkg/asset"
	"github.com/openshift/installer/pkg/asset/templates/content"
)

const (
	kubeadminPasswordSecretFileName = "kubeadmin-password-secret.yaml.template"
)

var _ asset.WritableAsset = (*KubeadminPasswordSecret)(nil)

// KubeadminPasswordSecret is the constant to represent contents of kubeadmin-password-secret.yaml.template file
type KubeadminPasswordSecret struct {
	fileName string
	FileList []*asset.File
}

// Dependencies returns all of the dependencies directly needed by the asset
func (t *KubeadminPasswordSecret) Dependencies() []asset.Asset {
	return []asset.Asset{}
}

// Name returns the human-friendly name of the asset.
func (t *KubeadminPasswordSecret) Name() string {
	return "KubeadminPasswordSecret"
}

// Generate generates the actual files by this asset
func (t *KubeadminPasswordSecret) Generate(parents asset.Parents) error {
	t.fileName = kubeadminPasswordSecretFileName
	data, err := content.GetOpenshiftTemplate(t.fileName)
	if err != nil {
		return err
	}
	t.FileList = []*asset.File{
		{
			Filename: filepath.Join(content.TemplateDir, t.fileName),
			Data:     []byte(data),
		},
	}
	return nil
}

// Files returns the files generated by the asset.
func (t *KubeadminPasswordSecret) Files() []*asset.File {
	return t.FileList
}

// Load returns the asset from disk.
func (t *KubeadminPasswordSecret) Load(f asset.FileFetcher) (bool, error) {
	file, err := f.FetchByName(filepath.Join(content.TemplateDir, kubeadminPasswordSecretFileName))
	if err != nil {
		if os.IsNotExist(err) {
			return false, nil
		}
		return false, err
	}
	t.FileList = []*asset.File{file}
	return true, nil
}
