package catalog

// Copyright (c) Microsoft and contributors.  All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Code generated by Microsoft (R) AutoRest Code Generator.
// Changes may cause incorrect behavior and will be lost if the code is regenerated.

import (
	"context"
	"github.com/Azure/go-autorest/autorest"
	"github.com/Azure/go-autorest/autorest/azure"
	"github.com/Azure/go-autorest/autorest/validation"
	"net/http"
)

// Client is the creates an Azure Data Lake Analytics catalog client.
type Client struct {
	BaseClient
}

// NewClient creates an instance of the Client client.
func NewClient() Client {
	return Client{New()}
}

// CreateCredential creates the specified credential for use with external data sources in the specified database.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database in which to create the credential. Note: This is NOT an external database name, but the
// name of an existing U-SQL database that should contain the new credential object. credentialName is the name of
// the credential. parameters is the parameters required to create the credential (name and password)
func (client Client) CreateCredential(ctx context.Context, accountName string, databaseName string, credentialName string, parameters DataLakeAnalyticsCatalogCredentialCreateParameters) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Password", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "parameters.URI", Name: validation.Null, Rule: true, Chain: nil},
				{Target: "parameters.UserID", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "CreateCredential", err.Error())
	}

	req, err := client.CreateCredentialPreparer(ctx, accountName, databaseName, credentialName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "CreateCredential", nil, "Failure preparing request")
		return
	}

	resp, err := client.CreateCredentialSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "catalog.Client", "CreateCredential", resp, "Failure sending request")
		return
	}

	result, err = client.CreateCredentialResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "CreateCredential", resp, "Failure responding to request")
	}

	return
}

// CreateCredentialPreparer prepares the CreateCredential request.
func (client Client) CreateCredentialPreparer(ctx context.Context, accountName string, databaseName string, credentialName string, parameters DataLakeAnalyticsCatalogCredentialCreateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"credentialName": autorest.Encode("path", credentialName),
		"databaseName":   autorest.Encode("path", databaseName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/credentials/{credentialName}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// CreateCredentialSender sends the CreateCredential request. The method will close the
// http.Response Body if it receives an error.
func (client Client) CreateCredentialSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// CreateCredentialResponder handles the response to the CreateCredential request. The method always
// closes the http.Response Body.
func (client Client) CreateCredentialResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// CreateSecret creates the specified secret for use with external data sources in the specified database. This is
// deprecated and will be removed in the next release. Please use CreateCredential instead.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database in which to create the secret. secretName is the name of the secret. parameters is the
// parameters required to create the secret (name and password)
func (client Client) CreateSecret(ctx context.Context, accountName string, databaseName string, secretName string, parameters DataLakeAnalyticsCatalogSecretCreateOrUpdateParameters) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.Password", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "CreateSecret", err.Error())
	}

	req, err := client.CreateSecretPreparer(ctx, accountName, databaseName, secretName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "CreateSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.CreateSecretSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "catalog.Client", "CreateSecret", resp, "Failure sending request")
		return
	}

	result, err = client.CreateSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "CreateSecret", resp, "Failure responding to request")
	}

	return
}

// CreateSecretPreparer prepares the CreateSecret request.
func (client Client) CreateSecretPreparer(ctx context.Context, accountName string, databaseName string, secretName string, parameters DataLakeAnalyticsCatalogSecretCreateOrUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
		"secretName":   autorest.Encode("path", secretName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/secrets/{secretName}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// CreateSecretSender sends the CreateSecret request. The method will close the
// http.Response Body if it receives an error.
func (client Client) CreateSecretSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// CreateSecretResponder handles the response to the CreateSecret request. The method always
// closes the http.Response Body.
func (client Client) CreateSecretResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// DeleteAllSecrets deletes all secrets in the specified database. This is deprecated and will be removed in the next
// release. In the future, please only drop individual credentials using DeleteCredential
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the secret.
func (client Client) DeleteAllSecrets(ctx context.Context, accountName string, databaseName string) (result autorest.Response, err error) {
	req, err := client.DeleteAllSecretsPreparer(ctx, accountName, databaseName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "DeleteAllSecrets", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteAllSecretsSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "catalog.Client", "DeleteAllSecrets", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteAllSecretsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "DeleteAllSecrets", resp, "Failure responding to request")
	}

	return
}

// DeleteAllSecretsPreparer prepares the DeleteAllSecrets request.
func (client Client) DeleteAllSecretsPreparer(ctx context.Context, accountName string, databaseName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/secrets", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteAllSecretsSender sends the DeleteAllSecrets request. The method will close the
// http.Response Body if it receives an error.
func (client Client) DeleteAllSecretsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DeleteAllSecretsResponder handles the response to the DeleteAllSecrets request. The method always
// closes the http.Response Body.
func (client Client) DeleteAllSecretsResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// DeleteCredential deletes the specified credential in the specified database
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the credential. credentialName is the name of the credential to delete
// parameters is the parameters to delete a credential if the current user is not the account owner. cascade is
// indicates if the delete should be a cascading delete (which deletes all resources dependent on the credential as
// well as the credential) or not. If false will fail if there are any resources relying on the credential.
func (client Client) DeleteCredential(ctx context.Context, accountName string, databaseName string, credentialName string, parameters *DataLakeAnalyticsCatalogCredentialDeleteParameters, cascade *bool) (result autorest.Response, err error) {
	req, err := client.DeleteCredentialPreparer(ctx, accountName, databaseName, credentialName, parameters, cascade)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "DeleteCredential", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteCredentialSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "catalog.Client", "DeleteCredential", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteCredentialResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "DeleteCredential", resp, "Failure responding to request")
	}

	return
}

// DeleteCredentialPreparer prepares the DeleteCredential request.
func (client Client) DeleteCredentialPreparer(ctx context.Context, accountName string, databaseName string, credentialName string, parameters *DataLakeAnalyticsCatalogCredentialDeleteParameters, cascade *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"credentialName": autorest.Encode("path", credentialName),
		"databaseName":   autorest.Encode("path", databaseName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if cascade != nil {
		queryParameters["cascade"] = autorest.Encode("query", *cascade)
	} else {
		queryParameters["cascade"] = autorest.Encode("query", false)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/credentials/{credentialName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	if parameters != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithJSON(parameters))
	}
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteCredentialSender sends the DeleteCredential request. The method will close the
// http.Response Body if it receives an error.
func (client Client) DeleteCredentialSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DeleteCredentialResponder handles the response to the DeleteCredential request. The method always
// closes the http.Response Body.
func (client Client) DeleteCredentialResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// DeleteSecret deletes the specified secret in the specified database. This is deprecated and will be removed in the
// next release. Please use DeleteCredential instead.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the secret. secretName is the name of the secret to delete
func (client Client) DeleteSecret(ctx context.Context, accountName string, databaseName string, secretName string) (result autorest.Response, err error) {
	req, err := client.DeleteSecretPreparer(ctx, accountName, databaseName, secretName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "DeleteSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteSecretSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "catalog.Client", "DeleteSecret", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "DeleteSecret", resp, "Failure responding to request")
	}

	return
}

// DeleteSecretPreparer prepares the DeleteSecret request.
func (client Client) DeleteSecretPreparer(ctx context.Context, accountName string, databaseName string, secretName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
		"secretName":   autorest.Encode("path", secretName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/secrets/{secretName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// DeleteSecretSender sends the DeleteSecret request. The method will close the
// http.Response Body if it receives an error.
func (client Client) DeleteSecretSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DeleteSecretResponder handles the response to the DeleteSecret request. The method always
// closes the http.Response Body.
func (client Client) DeleteSecretResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// GetAssembly retrieves the specified assembly from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the assembly. assemblyName is the name of the assembly.
func (client Client) GetAssembly(ctx context.Context, accountName string, databaseName string, assemblyName string) (result USQLAssembly, err error) {
	req, err := client.GetAssemblyPreparer(ctx, accountName, databaseName, assemblyName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetAssembly", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetAssemblySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetAssembly", resp, "Failure sending request")
		return
	}

	result, err = client.GetAssemblyResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetAssembly", resp, "Failure responding to request")
	}

	return
}

// GetAssemblyPreparer prepares the GetAssembly request.
func (client Client) GetAssemblyPreparer(ctx context.Context, accountName string, databaseName string, assemblyName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"assemblyName": autorest.Encode("path", assemblyName),
		"databaseName": autorest.Encode("path", databaseName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/assemblies/{assemblyName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetAssemblySender sends the GetAssembly request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GetAssemblySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetAssemblyResponder handles the response to the GetAssembly request. The method always
// closes the http.Response Body.
func (client Client) GetAssemblyResponder(resp *http.Response) (result USQLAssembly, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetCredential retrieves the specified credential from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the schema. credentialName is the name of the credential.
func (client Client) GetCredential(ctx context.Context, accountName string, databaseName string, credentialName string) (result USQLCredential, err error) {
	req, err := client.GetCredentialPreparer(ctx, accountName, databaseName, credentialName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetCredential", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetCredentialSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetCredential", resp, "Failure sending request")
		return
	}

	result, err = client.GetCredentialResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetCredential", resp, "Failure responding to request")
	}

	return
}

// GetCredentialPreparer prepares the GetCredential request.
func (client Client) GetCredentialPreparer(ctx context.Context, accountName string, databaseName string, credentialName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"credentialName": autorest.Encode("path", credentialName),
		"databaseName":   autorest.Encode("path", databaseName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/credentials/{credentialName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetCredentialSender sends the GetCredential request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GetCredentialSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetCredentialResponder handles the response to the GetCredential request. The method always
// closes the http.Response Body.
func (client Client) GetCredentialResponder(resp *http.Response) (result USQLCredential, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetDatabase retrieves the specified database from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database.
func (client Client) GetDatabase(ctx context.Context, accountName string, databaseName string) (result USQLDatabase, err error) {
	req, err := client.GetDatabasePreparer(ctx, accountName, databaseName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetDatabase", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetDatabaseSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetDatabase", resp, "Failure sending request")
		return
	}

	result, err = client.GetDatabaseResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetDatabase", resp, "Failure responding to request")
	}

	return
}

// GetDatabasePreparer prepares the GetDatabase request.
func (client Client) GetDatabasePreparer(ctx context.Context, accountName string, databaseName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetDatabaseSender sends the GetDatabase request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GetDatabaseSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetDatabaseResponder handles the response to the GetDatabase request. The method always
// closes the http.Response Body.
func (client Client) GetDatabaseResponder(resp *http.Response) (result USQLDatabase, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetExternalDataSource retrieves the specified external data source from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the external data source. externalDataSourceName is the name of the external
// data source.
func (client Client) GetExternalDataSource(ctx context.Context, accountName string, databaseName string, externalDataSourceName string) (result USQLExternalDataSource, err error) {
	req, err := client.GetExternalDataSourcePreparer(ctx, accountName, databaseName, externalDataSourceName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetExternalDataSource", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetExternalDataSourceSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetExternalDataSource", resp, "Failure sending request")
		return
	}

	result, err = client.GetExternalDataSourceResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetExternalDataSource", resp, "Failure responding to request")
	}

	return
}

// GetExternalDataSourcePreparer prepares the GetExternalDataSource request.
func (client Client) GetExternalDataSourcePreparer(ctx context.Context, accountName string, databaseName string, externalDataSourceName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName":           autorest.Encode("path", databaseName),
		"externalDataSourceName": autorest.Encode("path", externalDataSourceName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/externaldatasources/{externalDataSourceName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetExternalDataSourceSender sends the GetExternalDataSource request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GetExternalDataSourceSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetExternalDataSourceResponder handles the response to the GetExternalDataSource request. The method always
// closes the http.Response Body.
func (client Client) GetExternalDataSourceResponder(resp *http.Response) (result USQLExternalDataSource, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetPackage retrieves the specified package from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the package. schemaName is the name of the schema containing the package.
// packageName is the name of the package.
func (client Client) GetPackage(ctx context.Context, accountName string, databaseName string, schemaName string, packageName string) (result USQLPackage, err error) {
	req, err := client.GetPackagePreparer(ctx, accountName, databaseName, schemaName, packageName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetPackage", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetPackageSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetPackage", resp, "Failure sending request")
		return
	}

	result, err = client.GetPackageResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetPackage", resp, "Failure responding to request")
	}

	return
}

// GetPackagePreparer prepares the GetPackage request.
func (client Client) GetPackagePreparer(ctx context.Context, accountName string, databaseName string, schemaName string, packageName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
		"packageName":  autorest.Encode("path", packageName),
		"schemaName":   autorest.Encode("path", schemaName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}/packages/{packageName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetPackageSender sends the GetPackage request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GetPackageSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetPackageResponder handles the response to the GetPackage request. The method always
// closes the http.Response Body.
func (client Client) GetPackageResponder(resp *http.Response) (result USQLPackage, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetProcedure retrieves the specified procedure from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the procedure. schemaName is the name of the schema containing the
// procedure. procedureName is the name of the procedure.
func (client Client) GetProcedure(ctx context.Context, accountName string, databaseName string, schemaName string, procedureName string) (result USQLProcedure, err error) {
	req, err := client.GetProcedurePreparer(ctx, accountName, databaseName, schemaName, procedureName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetProcedure", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetProcedureSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetProcedure", resp, "Failure sending request")
		return
	}

	result, err = client.GetProcedureResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetProcedure", resp, "Failure responding to request")
	}

	return
}

// GetProcedurePreparer prepares the GetProcedure request.
func (client Client) GetProcedurePreparer(ctx context.Context, accountName string, databaseName string, schemaName string, procedureName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName":  autorest.Encode("path", databaseName),
		"procedureName": autorest.Encode("path", procedureName),
		"schemaName":    autorest.Encode("path", schemaName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}/procedures/{procedureName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetProcedureSender sends the GetProcedure request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GetProcedureSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetProcedureResponder handles the response to the GetProcedure request. The method always
// closes the http.Response Body.
func (client Client) GetProcedureResponder(resp *http.Response) (result USQLProcedure, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetSchema retrieves the specified schema from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the schema. schemaName is the name of the schema.
func (client Client) GetSchema(ctx context.Context, accountName string, databaseName string, schemaName string) (result USQLSchema, err error) {
	req, err := client.GetSchemaPreparer(ctx, accountName, databaseName, schemaName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetSchema", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetSchemaSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetSchema", resp, "Failure sending request")
		return
	}

	result, err = client.GetSchemaResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetSchema", resp, "Failure responding to request")
	}

	return
}

// GetSchemaPreparer prepares the GetSchema request.
func (client Client) GetSchemaPreparer(ctx context.Context, accountName string, databaseName string, schemaName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
		"schemaName":   autorest.Encode("path", schemaName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetSchemaSender sends the GetSchema request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GetSchemaSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetSchemaResponder handles the response to the GetSchema request. The method always
// closes the http.Response Body.
func (client Client) GetSchemaResponder(resp *http.Response) (result USQLSchema, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetSecret gets the specified secret in the specified database. This is deprecated and will be removed in the next
// release. Please use GetCredential instead.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the secret. secretName is the name of the secret to get
func (client Client) GetSecret(ctx context.Context, accountName string, databaseName string, secretName string) (result USQLSecret, err error) {
	req, err := client.GetSecretPreparer(ctx, accountName, databaseName, secretName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetSecretSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetSecret", resp, "Failure sending request")
		return
	}

	result, err = client.GetSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetSecret", resp, "Failure responding to request")
	}

	return
}

// GetSecretPreparer prepares the GetSecret request.
func (client Client) GetSecretPreparer(ctx context.Context, accountName string, databaseName string, secretName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
		"secretName":   autorest.Encode("path", secretName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/secrets/{secretName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetSecretSender sends the GetSecret request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GetSecretSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetSecretResponder handles the response to the GetSecret request. The method always
// closes the http.Response Body.
func (client Client) GetSecretResponder(resp *http.Response) (result USQLSecret, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetTable retrieves the specified table from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the table. schemaName is the name of the schema containing the table.
// tableName is the name of the table.
func (client Client) GetTable(ctx context.Context, accountName string, databaseName string, schemaName string, tableName string) (result USQLTable, err error) {
	req, err := client.GetTablePreparer(ctx, accountName, databaseName, schemaName, tableName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetTable", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetTableSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetTable", resp, "Failure sending request")
		return
	}

	result, err = client.GetTableResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetTable", resp, "Failure responding to request")
	}

	return
}

// GetTablePreparer prepares the GetTable request.
func (client Client) GetTablePreparer(ctx context.Context, accountName string, databaseName string, schemaName string, tableName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
		"schemaName":   autorest.Encode("path", schemaName),
		"tableName":    autorest.Encode("path", tableName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}/tables/{tableName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetTableSender sends the GetTable request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GetTableSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetTableResponder handles the response to the GetTable request. The method always
// closes the http.Response Body.
func (client Client) GetTableResponder(resp *http.Response) (result USQLTable, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetTablePartition retrieves the specified table partition from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the partition. schemaName is the name of the schema containing the
// partition. tableName is the name of the table containing the partition. partitionName is the name of the table
// partition.
func (client Client) GetTablePartition(ctx context.Context, accountName string, databaseName string, schemaName string, tableName string, partitionName string) (result USQLTablePartition, err error) {
	req, err := client.GetTablePartitionPreparer(ctx, accountName, databaseName, schemaName, tableName, partitionName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetTablePartition", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetTablePartitionSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetTablePartition", resp, "Failure sending request")
		return
	}

	result, err = client.GetTablePartitionResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetTablePartition", resp, "Failure responding to request")
	}

	return
}

// GetTablePartitionPreparer prepares the GetTablePartition request.
func (client Client) GetTablePartitionPreparer(ctx context.Context, accountName string, databaseName string, schemaName string, tableName string, partitionName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName":  autorest.Encode("path", databaseName),
		"partitionName": autorest.Encode("path", partitionName),
		"schemaName":    autorest.Encode("path", schemaName),
		"tableName":     autorest.Encode("path", tableName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}/tables/{tableName}/partitions/{partitionName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetTablePartitionSender sends the GetTablePartition request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GetTablePartitionSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetTablePartitionResponder handles the response to the GetTablePartition request. The method always
// closes the http.Response Body.
func (client Client) GetTablePartitionResponder(resp *http.Response) (result USQLTablePartition, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetTableStatistic retrieves the specified table statistics from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the statistics. schemaName is the name of the schema containing the
// statistics. tableName is the name of the table containing the statistics. statisticsName is the name of the
// table statistics.
func (client Client) GetTableStatistic(ctx context.Context, accountName string, databaseName string, schemaName string, tableName string, statisticsName string) (result USQLTableStatistics, err error) {
	req, err := client.GetTableStatisticPreparer(ctx, accountName, databaseName, schemaName, tableName, statisticsName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetTableStatistic", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetTableStatisticSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetTableStatistic", resp, "Failure sending request")
		return
	}

	result, err = client.GetTableStatisticResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetTableStatistic", resp, "Failure responding to request")
	}

	return
}

// GetTableStatisticPreparer prepares the GetTableStatistic request.
func (client Client) GetTableStatisticPreparer(ctx context.Context, accountName string, databaseName string, schemaName string, tableName string, statisticsName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName":   autorest.Encode("path", databaseName),
		"schemaName":     autorest.Encode("path", schemaName),
		"statisticsName": autorest.Encode("path", statisticsName),
		"tableName":      autorest.Encode("path", tableName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}/tables/{tableName}/statistics/{statisticsName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetTableStatisticSender sends the GetTableStatistic request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GetTableStatisticSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetTableStatisticResponder handles the response to the GetTableStatistic request. The method always
// closes the http.Response Body.
func (client Client) GetTableStatisticResponder(resp *http.Response) (result USQLTableStatistics, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetTableType retrieves the specified table type from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the table type. schemaName is the name of the schema containing the table
// type. tableTypeName is the name of the table type to retrieve.
func (client Client) GetTableType(ctx context.Context, accountName string, databaseName string, schemaName string, tableTypeName string) (result USQLTableType, err error) {
	req, err := client.GetTableTypePreparer(ctx, accountName, databaseName, schemaName, tableTypeName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetTableType", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetTableTypeSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetTableType", resp, "Failure sending request")
		return
	}

	result, err = client.GetTableTypeResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetTableType", resp, "Failure responding to request")
	}

	return
}

// GetTableTypePreparer prepares the GetTableType request.
func (client Client) GetTableTypePreparer(ctx context.Context, accountName string, databaseName string, schemaName string, tableTypeName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName":  autorest.Encode("path", databaseName),
		"schemaName":    autorest.Encode("path", schemaName),
		"tableTypeName": autorest.Encode("path", tableTypeName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}/tabletypes/{tableTypeName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetTableTypeSender sends the GetTableType request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GetTableTypeSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetTableTypeResponder handles the response to the GetTableType request. The method always
// closes the http.Response Body.
func (client Client) GetTableTypeResponder(resp *http.Response) (result USQLTableType, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetTableValuedFunction retrieves the specified table valued function from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the table valued function. schemaName is the name of the schema containing
// the table valued function. tableValuedFunctionName is the name of the tableValuedFunction.
func (client Client) GetTableValuedFunction(ctx context.Context, accountName string, databaseName string, schemaName string, tableValuedFunctionName string) (result USQLTableValuedFunction, err error) {
	req, err := client.GetTableValuedFunctionPreparer(ctx, accountName, databaseName, schemaName, tableValuedFunctionName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetTableValuedFunction", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetTableValuedFunctionSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetTableValuedFunction", resp, "Failure sending request")
		return
	}

	result, err = client.GetTableValuedFunctionResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetTableValuedFunction", resp, "Failure responding to request")
	}

	return
}

// GetTableValuedFunctionPreparer prepares the GetTableValuedFunction request.
func (client Client) GetTableValuedFunctionPreparer(ctx context.Context, accountName string, databaseName string, schemaName string, tableValuedFunctionName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName":            autorest.Encode("path", databaseName),
		"schemaName":              autorest.Encode("path", schemaName),
		"tableValuedFunctionName": autorest.Encode("path", tableValuedFunctionName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}/tablevaluedfunctions/{tableValuedFunctionName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetTableValuedFunctionSender sends the GetTableValuedFunction request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GetTableValuedFunctionSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetTableValuedFunctionResponder handles the response to the GetTableValuedFunction request. The method always
// closes the http.Response Body.
func (client Client) GetTableValuedFunctionResponder(resp *http.Response) (result USQLTableValuedFunction, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetView retrieves the specified view from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the view. schemaName is the name of the schema containing the view. viewName
// is the name of the view.
func (client Client) GetView(ctx context.Context, accountName string, databaseName string, schemaName string, viewName string) (result USQLView, err error) {
	req, err := client.GetViewPreparer(ctx, accountName, databaseName, schemaName, viewName)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetView", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetViewSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetView", resp, "Failure sending request")
		return
	}

	result, err = client.GetViewResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GetView", resp, "Failure responding to request")
	}

	return
}

// GetViewPreparer prepares the GetView request.
func (client Client) GetViewPreparer(ctx context.Context, accountName string, databaseName string, schemaName string, viewName string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
		"schemaName":   autorest.Encode("path", schemaName),
		"viewName":     autorest.Encode("path", viewName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}/views/{viewName}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GetViewSender sends the GetView request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GetViewSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetViewResponder handles the response to the GetView request. The method always
// closes the http.Response Body.
func (client Client) GetViewResponder(resp *http.Response) (result USQLView, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GrantACL grants an access control list (ACL) entry to the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. parameters is
// parameters supplied to create or update an access control list (ACL) entry for a Data Lake Analytics catalog.
func (client Client) GrantACL(ctx context.Context, accountName string, parameters ACLCreateOrUpdateParameters) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.PrincipalID", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "GrantACL", err.Error())
	}

	req, err := client.GrantACLPreparer(ctx, accountName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GrantACL", nil, "Failure preparing request")
		return
	}

	resp, err := client.GrantACLSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "catalog.Client", "GrantACL", resp, "Failure sending request")
		return
	}

	result, err = client.GrantACLResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GrantACL", resp, "Failure responding to request")
	}

	return
}

// GrantACLPreparer prepares the GrantACL request.
func (client Client) GrantACLPreparer(ctx context.Context, accountName string, parameters ACLCreateOrUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "GRANTACE"),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPath("/catalog/usql/acl"),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GrantACLSender sends the GrantACL request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GrantACLSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GrantACLResponder handles the response to the GrantACL request. The method always
// closes the http.Response Body.
func (client Client) GrantACLResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// GrantACLToDatabase grants an access control list (ACL) entry to the database from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database. parameters is parameters supplied to create or update an access control list (ACL)
// entry for a database.
func (client Client) GrantACLToDatabase(ctx context.Context, accountName string, databaseName string, parameters ACLCreateOrUpdateParameters) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.PrincipalID", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "GrantACLToDatabase", err.Error())
	}

	req, err := client.GrantACLToDatabasePreparer(ctx, accountName, databaseName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GrantACLToDatabase", nil, "Failure preparing request")
		return
	}

	resp, err := client.GrantACLToDatabaseSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "catalog.Client", "GrantACLToDatabase", resp, "Failure sending request")
		return
	}

	result, err = client.GrantACLToDatabaseResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "GrantACLToDatabase", resp, "Failure responding to request")
	}

	return
}

// GrantACLToDatabasePreparer prepares the GrantACLToDatabase request.
func (client Client) GrantACLToDatabasePreparer(ctx context.Context, accountName string, databaseName string, parameters ACLCreateOrUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "GRANTACE"),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/acl", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// GrantACLToDatabaseSender sends the GrantACLToDatabase request. The method will close the
// http.Response Body if it receives an error.
func (client Client) GrantACLToDatabaseSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GrantACLToDatabaseResponder handles the response to the GrantACLToDatabase request. The method always
// closes the http.Response Body.
func (client Client) GrantACLToDatabaseResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// ListAcls retrieves the list of access control list (ACL) entries for the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. filter is oData
// filter. Optional. top is the number of items to return. Optional. skip is the number of items to skip over
// before returning elements. Optional. selectParameter is oData Select statement. Limits the properties on each
// entry to just those requested, e.g. Categories?$select=CategoryName,Description. Optional. orderby is orderBy
// clause. One or more comma-separated expressions with an optional "asc" (the default) or "desc" depending on the
// order you'd like the values sorted, e.g. Categories?$orderby=CategoryName desc. Optional. count is the Boolean
// value of true or false to request a count of the matching resources included with the resources in the response,
// e.g. Categories?$count=true. Optional.
func (client Client) ListAcls(ctx context.Context, accountName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result ACLListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListAcls", err.Error())
	}

	result.fn = client.listAclsNextResults
	req, err := client.ListAclsPreparer(ctx, accountName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListAcls", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListAclsSender(req)
	if err != nil {
		result.al.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListAcls", resp, "Failure sending request")
		return
	}

	result.al, err = client.ListAclsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListAcls", resp, "Failure responding to request")
	}

	return
}

// ListAclsPreparer prepares the ListAcls request.
func (client Client) ListAclsPreparer(ctx context.Context, accountName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPath("/catalog/usql/acl"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListAclsSender sends the ListAcls request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListAclsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListAclsResponder handles the response to the ListAcls request. The method always
// closes the http.Response Body.
func (client Client) ListAclsResponder(resp *http.Response) (result ACLList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listAclsNextResults retrieves the next set of results, if any.
func (client Client) listAclsNextResults(lastResults ACLList) (result ACLList, err error) {
	req, err := lastResults.aCLListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listAclsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListAclsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listAclsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListAclsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listAclsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListAclsComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListAclsComplete(ctx context.Context, accountName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result ACLListIterator, err error) {
	result.page, err = client.ListAcls(ctx, accountName, filter, top, skip, selectParameter, orderby, count)
	return
}

// ListAclsByDatabase retrieves the list of access control list (ACL) entries for the database from the Data Lake
// Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database. filter is oData filter. Optional. top is the number of items to return. Optional. skip
// is the number of items to skip over before returning elements. Optional. selectParameter is oData Select
// statement. Limits the properties on each entry to just those requested, e.g.
// Categories?$select=CategoryName,Description. Optional. orderby is orderBy clause. One or more comma-separated
// expressions with an optional "asc" (the default) or "desc" depending on the order you'd like the values sorted,
// e.g. Categories?$orderby=CategoryName desc. Optional. count is the Boolean value of true or false to request a
// count of the matching resources included with the resources in the response, e.g. Categories?$count=true.
// Optional.
func (client Client) ListAclsByDatabase(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result ACLListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListAclsByDatabase", err.Error())
	}

	result.fn = client.listAclsByDatabaseNextResults
	req, err := client.ListAclsByDatabasePreparer(ctx, accountName, databaseName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListAclsByDatabase", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListAclsByDatabaseSender(req)
	if err != nil {
		result.al.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListAclsByDatabase", resp, "Failure sending request")
		return
	}

	result.al, err = client.ListAclsByDatabaseResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListAclsByDatabase", resp, "Failure responding to request")
	}

	return
}

// ListAclsByDatabasePreparer prepares the ListAclsByDatabase request.
func (client Client) ListAclsByDatabasePreparer(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/acl", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListAclsByDatabaseSender sends the ListAclsByDatabase request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListAclsByDatabaseSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListAclsByDatabaseResponder handles the response to the ListAclsByDatabase request. The method always
// closes the http.Response Body.
func (client Client) ListAclsByDatabaseResponder(resp *http.Response) (result ACLList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listAclsByDatabaseNextResults retrieves the next set of results, if any.
func (client Client) listAclsByDatabaseNextResults(lastResults ACLList) (result ACLList, err error) {
	req, err := lastResults.aCLListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listAclsByDatabaseNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListAclsByDatabaseSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listAclsByDatabaseNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListAclsByDatabaseResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listAclsByDatabaseNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListAclsByDatabaseComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListAclsByDatabaseComplete(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result ACLListIterator, err error) {
	result.page, err = client.ListAclsByDatabase(ctx, accountName, databaseName, filter, top, skip, selectParameter, orderby, count)
	return
}

// ListAssemblies retrieves the list of assemblies from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the assembly. filter is oData filter. Optional. top is the number of items
// to return. Optional. skip is the number of items to skip over before returning elements. Optional.
// selectParameter is oData Select statement. Limits the properties on each entry to just those requested, e.g.
// Categories?$select=CategoryName,Description. Optional. orderby is orderBy clause. One or more comma-separated
// expressions with an optional "asc" (the default) or "desc" depending on the order you'd like the values sorted,
// e.g. Categories?$orderby=CategoryName desc. Optional. count is the Boolean value of true or false to request a
// count of the matching resources included with the resources in the response, e.g. Categories?$count=true.
// Optional.
func (client Client) ListAssemblies(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLAssemblyListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListAssemblies", err.Error())
	}

	result.fn = client.listAssembliesNextResults
	req, err := client.ListAssembliesPreparer(ctx, accountName, databaseName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListAssemblies", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListAssembliesSender(req)
	if err != nil {
		result.ual.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListAssemblies", resp, "Failure sending request")
		return
	}

	result.ual, err = client.ListAssembliesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListAssemblies", resp, "Failure responding to request")
	}

	return
}

// ListAssembliesPreparer prepares the ListAssemblies request.
func (client Client) ListAssembliesPreparer(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/assemblies", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListAssembliesSender sends the ListAssemblies request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListAssembliesSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListAssembliesResponder handles the response to the ListAssemblies request. The method always
// closes the http.Response Body.
func (client Client) ListAssembliesResponder(resp *http.Response) (result USQLAssemblyList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listAssembliesNextResults retrieves the next set of results, if any.
func (client Client) listAssembliesNextResults(lastResults USQLAssemblyList) (result USQLAssemblyList, err error) {
	req, err := lastResults.uSQLAssemblyListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listAssembliesNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListAssembliesSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listAssembliesNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListAssembliesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listAssembliesNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListAssembliesComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListAssembliesComplete(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLAssemblyListIterator, err error) {
	result.page, err = client.ListAssemblies(ctx, accountName, databaseName, filter, top, skip, selectParameter, orderby, count)
	return
}

// ListCredentials retrieves the list of credentials from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the schema. filter is oData filter. Optional. top is the number of items to
// return. Optional. skip is the number of items to skip over before returning elements. Optional. selectParameter
// is oData Select statement. Limits the properties on each entry to just those requested, e.g.
// Categories?$select=CategoryName,Description. Optional. orderby is orderBy clause. One or more comma-separated
// expressions with an optional "asc" (the default) or "desc" depending on the order you'd like the values sorted,
// e.g. Categories?$orderby=CategoryName desc. Optional. count is the Boolean value of true or false to request a
// count of the matching resources included with the resources in the response, e.g. Categories?$count=true.
// Optional.
func (client Client) ListCredentials(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLCredentialListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListCredentials", err.Error())
	}

	result.fn = client.listCredentialsNextResults
	req, err := client.ListCredentialsPreparer(ctx, accountName, databaseName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListCredentials", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListCredentialsSender(req)
	if err != nil {
		result.ucl.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListCredentials", resp, "Failure sending request")
		return
	}

	result.ucl, err = client.ListCredentialsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListCredentials", resp, "Failure responding to request")
	}

	return
}

// ListCredentialsPreparer prepares the ListCredentials request.
func (client Client) ListCredentialsPreparer(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/credentials", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListCredentialsSender sends the ListCredentials request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListCredentialsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListCredentialsResponder handles the response to the ListCredentials request. The method always
// closes the http.Response Body.
func (client Client) ListCredentialsResponder(resp *http.Response) (result USQLCredentialList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listCredentialsNextResults retrieves the next set of results, if any.
func (client Client) listCredentialsNextResults(lastResults USQLCredentialList) (result USQLCredentialList, err error) {
	req, err := lastResults.uSQLCredentialListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listCredentialsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListCredentialsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listCredentialsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListCredentialsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listCredentialsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListCredentialsComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListCredentialsComplete(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLCredentialListIterator, err error) {
	result.page, err = client.ListCredentials(ctx, accountName, databaseName, filter, top, skip, selectParameter, orderby, count)
	return
}

// ListDatabases retrieves the list of databases from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. filter is oData
// filter. Optional. top is the number of items to return. Optional. skip is the number of items to skip over
// before returning elements. Optional. selectParameter is oData Select statement. Limits the properties on each
// entry to just those requested, e.g. Categories?$select=CategoryName,Description. Optional. orderby is orderBy
// clause. One or more comma-separated expressions with an optional "asc" (the default) or "desc" depending on the
// order you'd like the values sorted, e.g. Categories?$orderby=CategoryName desc. Optional. count is the Boolean
// value of true or false to request a count of the matching resources included with the resources in the response,
// e.g. Categories?$count=true. Optional.
func (client Client) ListDatabases(ctx context.Context, accountName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLDatabaseListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListDatabases", err.Error())
	}

	result.fn = client.listDatabasesNextResults
	req, err := client.ListDatabasesPreparer(ctx, accountName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListDatabases", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListDatabasesSender(req)
	if err != nil {
		result.udl.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListDatabases", resp, "Failure sending request")
		return
	}

	result.udl, err = client.ListDatabasesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListDatabases", resp, "Failure responding to request")
	}

	return
}

// ListDatabasesPreparer prepares the ListDatabases request.
func (client Client) ListDatabasesPreparer(ctx context.Context, accountName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPath("/catalog/usql/databases"),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListDatabasesSender sends the ListDatabases request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListDatabasesSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListDatabasesResponder handles the response to the ListDatabases request. The method always
// closes the http.Response Body.
func (client Client) ListDatabasesResponder(resp *http.Response) (result USQLDatabaseList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listDatabasesNextResults retrieves the next set of results, if any.
func (client Client) listDatabasesNextResults(lastResults USQLDatabaseList) (result USQLDatabaseList, err error) {
	req, err := lastResults.uSQLDatabaseListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listDatabasesNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListDatabasesSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listDatabasesNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListDatabasesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listDatabasesNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListDatabasesComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListDatabasesComplete(ctx context.Context, accountName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLDatabaseListIterator, err error) {
	result.page, err = client.ListDatabases(ctx, accountName, filter, top, skip, selectParameter, orderby, count)
	return
}

// ListExternalDataSources retrieves the list of external data sources from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the external data sources. filter is oData filter. Optional. top is the
// number of items to return. Optional. skip is the number of items to skip over before returning elements.
// Optional. selectParameter is oData Select statement. Limits the properties on each entry to just those
// requested, e.g. Categories?$select=CategoryName,Description. Optional. orderby is orderBy clause. One or more
// comma-separated expressions with an optional "asc" (the default) or "desc" depending on the order you'd like the
// values sorted, e.g. Categories?$orderby=CategoryName desc. Optional. count is the Boolean value of true or false
// to request a count of the matching resources included with the resources in the response, e.g.
// Categories?$count=true. Optional.
func (client Client) ListExternalDataSources(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLExternalDataSourceListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListExternalDataSources", err.Error())
	}

	result.fn = client.listExternalDataSourcesNextResults
	req, err := client.ListExternalDataSourcesPreparer(ctx, accountName, databaseName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListExternalDataSources", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListExternalDataSourcesSender(req)
	if err != nil {
		result.uedsl.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListExternalDataSources", resp, "Failure sending request")
		return
	}

	result.uedsl, err = client.ListExternalDataSourcesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListExternalDataSources", resp, "Failure responding to request")
	}

	return
}

// ListExternalDataSourcesPreparer prepares the ListExternalDataSources request.
func (client Client) ListExternalDataSourcesPreparer(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/externaldatasources", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListExternalDataSourcesSender sends the ListExternalDataSources request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListExternalDataSourcesSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListExternalDataSourcesResponder handles the response to the ListExternalDataSources request. The method always
// closes the http.Response Body.
func (client Client) ListExternalDataSourcesResponder(resp *http.Response) (result USQLExternalDataSourceList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listExternalDataSourcesNextResults retrieves the next set of results, if any.
func (client Client) listExternalDataSourcesNextResults(lastResults USQLExternalDataSourceList) (result USQLExternalDataSourceList, err error) {
	req, err := lastResults.uSQLExternalDataSourceListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listExternalDataSourcesNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListExternalDataSourcesSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listExternalDataSourcesNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListExternalDataSourcesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listExternalDataSourcesNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListExternalDataSourcesComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListExternalDataSourcesComplete(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLExternalDataSourceListIterator, err error) {
	result.page, err = client.ListExternalDataSources(ctx, accountName, databaseName, filter, top, skip, selectParameter, orderby, count)
	return
}

// ListPackages retrieves the list of packages from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the packages. schemaName is the name of the schema containing the packages.
// filter is oData filter. Optional. top is the number of items to return. Optional. skip is the number of items to
// skip over before returning elements. Optional. selectParameter is oData Select statement. Limits the properties
// on each entry to just those requested, e.g. Categories?$select=CategoryName,Description. Optional. orderby is
// orderBy clause. One or more comma-separated expressions with an optional "asc" (the default) or "desc" depending
// on the order you'd like the values sorted, e.g. Categories?$orderby=CategoryName desc. Optional. count is the
// Boolean value of true or false to request a count of the matching resources included with the resources in the
// response, e.g. Categories?$count=true. Optional.
func (client Client) ListPackages(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLPackageListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListPackages", err.Error())
	}

	result.fn = client.listPackagesNextResults
	req, err := client.ListPackagesPreparer(ctx, accountName, databaseName, schemaName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListPackages", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListPackagesSender(req)
	if err != nil {
		result.upl.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListPackages", resp, "Failure sending request")
		return
	}

	result.upl, err = client.ListPackagesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListPackages", resp, "Failure responding to request")
	}

	return
}

// ListPackagesPreparer prepares the ListPackages request.
func (client Client) ListPackagesPreparer(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
		"schemaName":   autorest.Encode("path", schemaName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}/packages", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListPackagesSender sends the ListPackages request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListPackagesSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListPackagesResponder handles the response to the ListPackages request. The method always
// closes the http.Response Body.
func (client Client) ListPackagesResponder(resp *http.Response) (result USQLPackageList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listPackagesNextResults retrieves the next set of results, if any.
func (client Client) listPackagesNextResults(lastResults USQLPackageList) (result USQLPackageList, err error) {
	req, err := lastResults.uSQLPackageListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listPackagesNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListPackagesSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listPackagesNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListPackagesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listPackagesNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListPackagesComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListPackagesComplete(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLPackageListIterator, err error) {
	result.page, err = client.ListPackages(ctx, accountName, databaseName, schemaName, filter, top, skip, selectParameter, orderby, count)
	return
}

// ListProcedures retrieves the list of procedures from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the procedures. schemaName is the name of the schema containing the
// procedures. filter is oData filter. Optional. top is the number of items to return. Optional. skip is the number
// of items to skip over before returning elements. Optional. selectParameter is oData Select statement. Limits the
// properties on each entry to just those requested, e.g. Categories?$select=CategoryName,Description. Optional.
// orderby is orderBy clause. One or more comma-separated expressions with an optional "asc" (the default) or
// "desc" depending on the order you'd like the values sorted, e.g. Categories?$orderby=CategoryName desc.
// Optional. count is the Boolean value of true or false to request a count of the matching resources included with
// the resources in the response, e.g. Categories?$count=true. Optional.
func (client Client) ListProcedures(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLProcedureListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListProcedures", err.Error())
	}

	result.fn = client.listProceduresNextResults
	req, err := client.ListProceduresPreparer(ctx, accountName, databaseName, schemaName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListProcedures", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListProceduresSender(req)
	if err != nil {
		result.upl.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListProcedures", resp, "Failure sending request")
		return
	}

	result.upl, err = client.ListProceduresResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListProcedures", resp, "Failure responding to request")
	}

	return
}

// ListProceduresPreparer prepares the ListProcedures request.
func (client Client) ListProceduresPreparer(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
		"schemaName":   autorest.Encode("path", schemaName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}/procedures", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListProceduresSender sends the ListProcedures request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListProceduresSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListProceduresResponder handles the response to the ListProcedures request. The method always
// closes the http.Response Body.
func (client Client) ListProceduresResponder(resp *http.Response) (result USQLProcedureList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listProceduresNextResults retrieves the next set of results, if any.
func (client Client) listProceduresNextResults(lastResults USQLProcedureList) (result USQLProcedureList, err error) {
	req, err := lastResults.uSQLProcedureListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listProceduresNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListProceduresSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listProceduresNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListProceduresResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listProceduresNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListProceduresComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListProceduresComplete(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLProcedureListIterator, err error) {
	result.page, err = client.ListProcedures(ctx, accountName, databaseName, schemaName, filter, top, skip, selectParameter, orderby, count)
	return
}

// ListSchemas retrieves the list of schemas from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the schema. filter is oData filter. Optional. top is the number of items to
// return. Optional. skip is the number of items to skip over before returning elements. Optional. selectParameter
// is oData Select statement. Limits the properties on each entry to just those requested, e.g.
// Categories?$select=CategoryName,Description. Optional. orderby is orderBy clause. One or more comma-separated
// expressions with an optional "asc" (the default) or "desc" depending on the order you'd like the values sorted,
// e.g. Categories?$orderby=CategoryName desc. Optional. count is the Boolean value of true or false to request a
// count of the matching resources included with the resources in the response, e.g. Categories?$count=true.
// Optional.
func (client Client) ListSchemas(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLSchemaListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListSchemas", err.Error())
	}

	result.fn = client.listSchemasNextResults
	req, err := client.ListSchemasPreparer(ctx, accountName, databaseName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListSchemas", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListSchemasSender(req)
	if err != nil {
		result.usl.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListSchemas", resp, "Failure sending request")
		return
	}

	result.usl, err = client.ListSchemasResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListSchemas", resp, "Failure responding to request")
	}

	return
}

// ListSchemasPreparer prepares the ListSchemas request.
func (client Client) ListSchemasPreparer(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListSchemasSender sends the ListSchemas request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListSchemasSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListSchemasResponder handles the response to the ListSchemas request. The method always
// closes the http.Response Body.
func (client Client) ListSchemasResponder(resp *http.Response) (result USQLSchemaList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listSchemasNextResults retrieves the next set of results, if any.
func (client Client) listSchemasNextResults(lastResults USQLSchemaList) (result USQLSchemaList, err error) {
	req, err := lastResults.uSQLSchemaListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listSchemasNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListSchemasSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listSchemasNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListSchemasResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listSchemasNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListSchemasComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListSchemasComplete(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLSchemaListIterator, err error) {
	result.page, err = client.ListSchemas(ctx, accountName, databaseName, filter, top, skip, selectParameter, orderby, count)
	return
}

// ListTablePartitions retrieves the list of table partitions from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the partitions. schemaName is the name of the schema containing the
// partitions. tableName is the name of the table containing the partitions. filter is oData filter. Optional. top
// is the number of items to return. Optional. skip is the number of items to skip over before returning elements.
// Optional. selectParameter is oData Select statement. Limits the properties on each entry to just those
// requested, e.g. Categories?$select=CategoryName,Description. Optional. orderby is orderBy clause. One or more
// comma-separated expressions with an optional "asc" (the default) or "desc" depending on the order you'd like the
// values sorted, e.g. Categories?$orderby=CategoryName desc. Optional. count is the Boolean value of true or false
// to request a count of the matching resources included with the resources in the response, e.g.
// Categories?$count=true. Optional.
func (client Client) ListTablePartitions(ctx context.Context, accountName string, databaseName string, schemaName string, tableName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLTablePartitionListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListTablePartitions", err.Error())
	}

	result.fn = client.listTablePartitionsNextResults
	req, err := client.ListTablePartitionsPreparer(ctx, accountName, databaseName, schemaName, tableName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTablePartitions", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListTablePartitionsSender(req)
	if err != nil {
		result.utpl.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTablePartitions", resp, "Failure sending request")
		return
	}

	result.utpl, err = client.ListTablePartitionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTablePartitions", resp, "Failure responding to request")
	}

	return
}

// ListTablePartitionsPreparer prepares the ListTablePartitions request.
func (client Client) ListTablePartitionsPreparer(ctx context.Context, accountName string, databaseName string, schemaName string, tableName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
		"schemaName":   autorest.Encode("path", schemaName),
		"tableName":    autorest.Encode("path", tableName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}/tables/{tableName}/partitions", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListTablePartitionsSender sends the ListTablePartitions request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListTablePartitionsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListTablePartitionsResponder handles the response to the ListTablePartitions request. The method always
// closes the http.Response Body.
func (client Client) ListTablePartitionsResponder(resp *http.Response) (result USQLTablePartitionList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listTablePartitionsNextResults retrieves the next set of results, if any.
func (client Client) listTablePartitionsNextResults(lastResults USQLTablePartitionList) (result USQLTablePartitionList, err error) {
	req, err := lastResults.uSQLTablePartitionListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTablePartitionsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListTablePartitionsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTablePartitionsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListTablePartitionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listTablePartitionsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListTablePartitionsComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListTablePartitionsComplete(ctx context.Context, accountName string, databaseName string, schemaName string, tableName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLTablePartitionListIterator, err error) {
	result.page, err = client.ListTablePartitions(ctx, accountName, databaseName, schemaName, tableName, filter, top, skip, selectParameter, orderby, count)
	return
}

// ListTables retrieves the list of tables from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the tables. schemaName is the name of the schema containing the tables.
// filter is oData filter. Optional. top is the number of items to return. Optional. skip is the number of items to
// skip over before returning elements. Optional. selectParameter is oData Select statement. Limits the properties
// on each entry to just those requested, e.g. Categories?$select=CategoryName,Description. Optional. orderby is
// orderBy clause. One or more comma-separated expressions with an optional "asc" (the default) or "desc" depending
// on the order you'd like the values sorted, e.g. Categories?$orderby=CategoryName desc. Optional. count is the
// Boolean value of true or false to request a count of the matching resources included with the resources in the
// response, e.g. Categories?$count=true. Optional. basic is the basic switch indicates what level of information
// to return when listing tables. When basic is true, only database_name, schema_name, table_name and version are
// returned for each table, otherwise all table metadata is returned. By default, it is false. Optional.
func (client Client) ListTables(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool, basic *bool) (result USQLTableListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListTables", err.Error())
	}

	result.fn = client.listTablesNextResults
	req, err := client.ListTablesPreparer(ctx, accountName, databaseName, schemaName, filter, top, skip, selectParameter, orderby, count, basic)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTables", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListTablesSender(req)
	if err != nil {
		result.utl.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTables", resp, "Failure sending request")
		return
	}

	result.utl, err = client.ListTablesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTables", resp, "Failure responding to request")
	}

	return
}

// ListTablesPreparer prepares the ListTables request.
func (client Client) ListTablesPreparer(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool, basic *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
		"schemaName":   autorest.Encode("path", schemaName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}
	if basic != nil {
		queryParameters["basic"] = autorest.Encode("query", *basic)
	} else {
		queryParameters["basic"] = autorest.Encode("query", false)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}/tables", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListTablesSender sends the ListTables request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListTablesSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListTablesResponder handles the response to the ListTables request. The method always
// closes the http.Response Body.
func (client Client) ListTablesResponder(resp *http.Response) (result USQLTableList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listTablesNextResults retrieves the next set of results, if any.
func (client Client) listTablesNextResults(lastResults USQLTableList) (result USQLTableList, err error) {
	req, err := lastResults.uSQLTableListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTablesNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListTablesSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTablesNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListTablesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listTablesNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListTablesComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListTablesComplete(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool, basic *bool) (result USQLTableListIterator, err error) {
	result.page, err = client.ListTables(ctx, accountName, databaseName, schemaName, filter, top, skip, selectParameter, orderby, count, basic)
	return
}

// ListTablesByDatabase retrieves the list of all tables in a database from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the tables. filter is oData filter. Optional. top is the number of items to
// return. Optional. skip is the number of items to skip over before returning elements. Optional. selectParameter
// is oData Select statement. Limits the properties on each entry to just those requested, e.g.
// Categories?$select=CategoryName,Description. Optional. orderby is orderBy clause. One or more comma-separated
// expressions with an optional "asc" (the default) or "desc" depending on the order you'd like the values sorted,
// e.g. Categories?$orderby=CategoryName desc. Optional. count is the Boolean value of true or false to request a
// count of the matching resources included with the resources in the response, e.g. Categories?$count=true.
// Optional. basic is the basic switch indicates what level of information to return when listing tables. When
// basic is true, only database_name, schema_name, table_name and version are returned for each table, otherwise
// all table metadata is returned. By default, it is false
func (client Client) ListTablesByDatabase(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool, basic *bool) (result USQLTableListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListTablesByDatabase", err.Error())
	}

	result.fn = client.listTablesByDatabaseNextResults
	req, err := client.ListTablesByDatabasePreparer(ctx, accountName, databaseName, filter, top, skip, selectParameter, orderby, count, basic)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTablesByDatabase", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListTablesByDatabaseSender(req)
	if err != nil {
		result.utl.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTablesByDatabase", resp, "Failure sending request")
		return
	}

	result.utl, err = client.ListTablesByDatabaseResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTablesByDatabase", resp, "Failure responding to request")
	}

	return
}

// ListTablesByDatabasePreparer prepares the ListTablesByDatabase request.
func (client Client) ListTablesByDatabasePreparer(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool, basic *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}
	if basic != nil {
		queryParameters["basic"] = autorest.Encode("query", *basic)
	} else {
		queryParameters["basic"] = autorest.Encode("query", false)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/tables", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListTablesByDatabaseSender sends the ListTablesByDatabase request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListTablesByDatabaseSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListTablesByDatabaseResponder handles the response to the ListTablesByDatabase request. The method always
// closes the http.Response Body.
func (client Client) ListTablesByDatabaseResponder(resp *http.Response) (result USQLTableList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listTablesByDatabaseNextResults retrieves the next set of results, if any.
func (client Client) listTablesByDatabaseNextResults(lastResults USQLTableList) (result USQLTableList, err error) {
	req, err := lastResults.uSQLTableListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTablesByDatabaseNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListTablesByDatabaseSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTablesByDatabaseNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListTablesByDatabaseResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listTablesByDatabaseNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListTablesByDatabaseComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListTablesByDatabaseComplete(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool, basic *bool) (result USQLTableListIterator, err error) {
	result.page, err = client.ListTablesByDatabase(ctx, accountName, databaseName, filter, top, skip, selectParameter, orderby, count, basic)
	return
}

// ListTableStatistics retrieves the list of table statistics from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the statistics. schemaName is the name of the schema containing the
// statistics. tableName is the name of the table containing the statistics. filter is oData filter. Optional. top
// is the number of items to return. Optional. skip is the number of items to skip over before returning elements.
// Optional. selectParameter is oData Select statement. Limits the properties on each entry to just those
// requested, e.g. Categories?$select=CategoryName,Description. Optional. orderby is orderBy clause. One or more
// comma-separated expressions with an optional "asc" (the default) or "desc" depending on the order you'd like the
// values sorted, e.g. Categories?$orderby=CategoryName desc. Optional. count is the Boolean value of true or false
// to request a count of the matching resources included with the resources in the response, e.g.
// Categories?$count=true. Optional.
func (client Client) ListTableStatistics(ctx context.Context, accountName string, databaseName string, schemaName string, tableName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLTableStatisticsListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListTableStatistics", err.Error())
	}

	result.fn = client.listTableStatisticsNextResults
	req, err := client.ListTableStatisticsPreparer(ctx, accountName, databaseName, schemaName, tableName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTableStatistics", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListTableStatisticsSender(req)
	if err != nil {
		result.utsl.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTableStatistics", resp, "Failure sending request")
		return
	}

	result.utsl, err = client.ListTableStatisticsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTableStatistics", resp, "Failure responding to request")
	}

	return
}

// ListTableStatisticsPreparer prepares the ListTableStatistics request.
func (client Client) ListTableStatisticsPreparer(ctx context.Context, accountName string, databaseName string, schemaName string, tableName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
		"schemaName":   autorest.Encode("path", schemaName),
		"tableName":    autorest.Encode("path", tableName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}/tables/{tableName}/statistics", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListTableStatisticsSender sends the ListTableStatistics request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListTableStatisticsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListTableStatisticsResponder handles the response to the ListTableStatistics request. The method always
// closes the http.Response Body.
func (client Client) ListTableStatisticsResponder(resp *http.Response) (result USQLTableStatisticsList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listTableStatisticsNextResults retrieves the next set of results, if any.
func (client Client) listTableStatisticsNextResults(lastResults USQLTableStatisticsList) (result USQLTableStatisticsList, err error) {
	req, err := lastResults.uSQLTableStatisticsListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTableStatisticsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListTableStatisticsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTableStatisticsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListTableStatisticsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listTableStatisticsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListTableStatisticsComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListTableStatisticsComplete(ctx context.Context, accountName string, databaseName string, schemaName string, tableName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLTableStatisticsListIterator, err error) {
	result.page, err = client.ListTableStatistics(ctx, accountName, databaseName, schemaName, tableName, filter, top, skip, selectParameter, orderby, count)
	return
}

// ListTableStatisticsByDatabase retrieves the list of all statistics in a database from the Data Lake Analytics
// catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the table statistics. filter is oData filter. Optional. top is the number of
// items to return. Optional. skip is the number of items to skip over before returning elements. Optional.
// selectParameter is oData Select statement. Limits the properties on each entry to just those requested, e.g.
// Categories?$select=CategoryName,Description. Optional. orderby is orderBy clause. One or more comma-separated
// expressions with an optional "asc" (the default) or "desc" depending on the order you'd like the values sorted,
// e.g. Categories?$orderby=CategoryName desc. Optional. count is the Boolean value of true or false to request a
// count of the matching resources included with the resources in the response, e.g. Categories?$count=true.
// Optional.
func (client Client) ListTableStatisticsByDatabase(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLTableStatisticsListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListTableStatisticsByDatabase", err.Error())
	}

	result.fn = client.listTableStatisticsByDatabaseNextResults
	req, err := client.ListTableStatisticsByDatabasePreparer(ctx, accountName, databaseName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTableStatisticsByDatabase", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListTableStatisticsByDatabaseSender(req)
	if err != nil {
		result.utsl.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTableStatisticsByDatabase", resp, "Failure sending request")
		return
	}

	result.utsl, err = client.ListTableStatisticsByDatabaseResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTableStatisticsByDatabase", resp, "Failure responding to request")
	}

	return
}

// ListTableStatisticsByDatabasePreparer prepares the ListTableStatisticsByDatabase request.
func (client Client) ListTableStatisticsByDatabasePreparer(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/statistics", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListTableStatisticsByDatabaseSender sends the ListTableStatisticsByDatabase request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListTableStatisticsByDatabaseSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListTableStatisticsByDatabaseResponder handles the response to the ListTableStatisticsByDatabase request. The method always
// closes the http.Response Body.
func (client Client) ListTableStatisticsByDatabaseResponder(resp *http.Response) (result USQLTableStatisticsList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listTableStatisticsByDatabaseNextResults retrieves the next set of results, if any.
func (client Client) listTableStatisticsByDatabaseNextResults(lastResults USQLTableStatisticsList) (result USQLTableStatisticsList, err error) {
	req, err := lastResults.uSQLTableStatisticsListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTableStatisticsByDatabaseNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListTableStatisticsByDatabaseSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTableStatisticsByDatabaseNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListTableStatisticsByDatabaseResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listTableStatisticsByDatabaseNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListTableStatisticsByDatabaseComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListTableStatisticsByDatabaseComplete(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLTableStatisticsListIterator, err error) {
	result.page, err = client.ListTableStatisticsByDatabase(ctx, accountName, databaseName, filter, top, skip, selectParameter, orderby, count)
	return
}

// ListTableStatisticsByDatabaseAndSchema retrieves the list of all table statistics within the specified schema from
// the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the statistics. schemaName is the name of the schema containing the
// statistics. filter is oData filter. Optional. top is the number of items to return. Optional. skip is the number
// of items to skip over before returning elements. Optional. selectParameter is oData Select statement. Limits the
// properties on each entry to just those requested, e.g. Categories?$select=CategoryName,Description. Optional.
// orderby is orderBy clause. One or more comma-separated expressions with an optional "asc" (the default) or
// "desc" depending on the order you'd like the values sorted, e.g. Categories?$orderby=CategoryName desc.
// Optional. count is the Boolean value of true or false to request a count of the matching resources included with
// the resources in the response, e.g. Categories?$count=true. Optional.
func (client Client) ListTableStatisticsByDatabaseAndSchema(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLTableStatisticsListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListTableStatisticsByDatabaseAndSchema", err.Error())
	}

	result.fn = client.listTableStatisticsByDatabaseAndSchemaNextResults
	req, err := client.ListTableStatisticsByDatabaseAndSchemaPreparer(ctx, accountName, databaseName, schemaName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTableStatisticsByDatabaseAndSchema", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListTableStatisticsByDatabaseAndSchemaSender(req)
	if err != nil {
		result.utsl.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTableStatisticsByDatabaseAndSchema", resp, "Failure sending request")
		return
	}

	result.utsl, err = client.ListTableStatisticsByDatabaseAndSchemaResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTableStatisticsByDatabaseAndSchema", resp, "Failure responding to request")
	}

	return
}

// ListTableStatisticsByDatabaseAndSchemaPreparer prepares the ListTableStatisticsByDatabaseAndSchema request.
func (client Client) ListTableStatisticsByDatabaseAndSchemaPreparer(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
		"schemaName":   autorest.Encode("path", schemaName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}/statistics", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListTableStatisticsByDatabaseAndSchemaSender sends the ListTableStatisticsByDatabaseAndSchema request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListTableStatisticsByDatabaseAndSchemaSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListTableStatisticsByDatabaseAndSchemaResponder handles the response to the ListTableStatisticsByDatabaseAndSchema request. The method always
// closes the http.Response Body.
func (client Client) ListTableStatisticsByDatabaseAndSchemaResponder(resp *http.Response) (result USQLTableStatisticsList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listTableStatisticsByDatabaseAndSchemaNextResults retrieves the next set of results, if any.
func (client Client) listTableStatisticsByDatabaseAndSchemaNextResults(lastResults USQLTableStatisticsList) (result USQLTableStatisticsList, err error) {
	req, err := lastResults.uSQLTableStatisticsListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTableStatisticsByDatabaseAndSchemaNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListTableStatisticsByDatabaseAndSchemaSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTableStatisticsByDatabaseAndSchemaNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListTableStatisticsByDatabaseAndSchemaResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listTableStatisticsByDatabaseAndSchemaNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListTableStatisticsByDatabaseAndSchemaComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListTableStatisticsByDatabaseAndSchemaComplete(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLTableStatisticsListIterator, err error) {
	result.page, err = client.ListTableStatisticsByDatabaseAndSchema(ctx, accountName, databaseName, schemaName, filter, top, skip, selectParameter, orderby, count)
	return
}

// ListTableTypes retrieves the list of table types from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the table types. schemaName is the name of the schema containing the table
// types. filter is oData filter. Optional. top is the number of items to return. Optional. skip is the number of
// items to skip over before returning elements. Optional. selectParameter is oData Select statement. Limits the
// properties on each entry to just those requested, e.g. Categories?$select=CategoryName,Description. Optional.
// orderby is orderBy clause. One or more comma-separated expressions with an optional "asc" (the default) or
// "desc" depending on the order you'd like the values sorted, e.g. Categories?$orderby=CategoryName desc.
// Optional. count is the Boolean value of true or false to request a count of the matching resources included with
// the resources in the response, e.g. Categories?$count=true. Optional.
func (client Client) ListTableTypes(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLTableTypeListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListTableTypes", err.Error())
	}

	result.fn = client.listTableTypesNextResults
	req, err := client.ListTableTypesPreparer(ctx, accountName, databaseName, schemaName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTableTypes", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListTableTypesSender(req)
	if err != nil {
		result.uttl.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTableTypes", resp, "Failure sending request")
		return
	}

	result.uttl, err = client.ListTableTypesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTableTypes", resp, "Failure responding to request")
	}

	return
}

// ListTableTypesPreparer prepares the ListTableTypes request.
func (client Client) ListTableTypesPreparer(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
		"schemaName":   autorest.Encode("path", schemaName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}/tabletypes", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListTableTypesSender sends the ListTableTypes request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListTableTypesSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListTableTypesResponder handles the response to the ListTableTypes request. The method always
// closes the http.Response Body.
func (client Client) ListTableTypesResponder(resp *http.Response) (result USQLTableTypeList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listTableTypesNextResults retrieves the next set of results, if any.
func (client Client) listTableTypesNextResults(lastResults USQLTableTypeList) (result USQLTableTypeList, err error) {
	req, err := lastResults.uSQLTableTypeListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTableTypesNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListTableTypesSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTableTypesNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListTableTypesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listTableTypesNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListTableTypesComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListTableTypesComplete(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLTableTypeListIterator, err error) {
	result.page, err = client.ListTableTypes(ctx, accountName, databaseName, schemaName, filter, top, skip, selectParameter, orderby, count)
	return
}

// ListTableValuedFunctions retrieves the list of table valued functions from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the table valued functions. schemaName is the name of the schema containing
// the table valued functions. filter is oData filter. Optional. top is the number of items to return. Optional.
// skip is the number of items to skip over before returning elements. Optional. selectParameter is oData Select
// statement. Limits the properties on each entry to just those requested, e.g.
// Categories?$select=CategoryName,Description. Optional. orderby is orderBy clause. One or more comma-separated
// expressions with an optional "asc" (the default) or "desc" depending on the order you'd like the values sorted,
// e.g. Categories?$orderby=CategoryName desc. Optional. count is the Boolean value of true or false to request a
// count of the matching resources included with the resources in the response, e.g. Categories?$count=true.
// Optional.
func (client Client) ListTableValuedFunctions(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLTableValuedFunctionListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListTableValuedFunctions", err.Error())
	}

	result.fn = client.listTableValuedFunctionsNextResults
	req, err := client.ListTableValuedFunctionsPreparer(ctx, accountName, databaseName, schemaName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTableValuedFunctions", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListTableValuedFunctionsSender(req)
	if err != nil {
		result.utvfl.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTableValuedFunctions", resp, "Failure sending request")
		return
	}

	result.utvfl, err = client.ListTableValuedFunctionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTableValuedFunctions", resp, "Failure responding to request")
	}

	return
}

// ListTableValuedFunctionsPreparer prepares the ListTableValuedFunctions request.
func (client Client) ListTableValuedFunctionsPreparer(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
		"schemaName":   autorest.Encode("path", schemaName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}/tablevaluedfunctions", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListTableValuedFunctionsSender sends the ListTableValuedFunctions request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListTableValuedFunctionsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListTableValuedFunctionsResponder handles the response to the ListTableValuedFunctions request. The method always
// closes the http.Response Body.
func (client Client) ListTableValuedFunctionsResponder(resp *http.Response) (result USQLTableValuedFunctionList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listTableValuedFunctionsNextResults retrieves the next set of results, if any.
func (client Client) listTableValuedFunctionsNextResults(lastResults USQLTableValuedFunctionList) (result USQLTableValuedFunctionList, err error) {
	req, err := lastResults.uSQLTableValuedFunctionListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTableValuedFunctionsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListTableValuedFunctionsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTableValuedFunctionsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListTableValuedFunctionsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listTableValuedFunctionsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListTableValuedFunctionsComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListTableValuedFunctionsComplete(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLTableValuedFunctionListIterator, err error) {
	result.page, err = client.ListTableValuedFunctions(ctx, accountName, databaseName, schemaName, filter, top, skip, selectParameter, orderby, count)
	return
}

// ListTableValuedFunctionsByDatabase retrieves the list of all table valued functions in a database from the Data Lake
// Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the table valued functions. filter is oData filter. Optional. top is the
// number of items to return. Optional. skip is the number of items to skip over before returning elements.
// Optional. selectParameter is oData Select statement. Limits the properties on each entry to just those
// requested, e.g. Categories?$select=CategoryName,Description. Optional. orderby is orderBy clause. One or more
// comma-separated expressions with an optional "asc" (the default) or "desc" depending on the order you'd like the
// values sorted, e.g. Categories?$orderby=CategoryName desc. Optional. count is the Boolean value of true or false
// to request a count of the matching resources included with the resources in the response, e.g.
// Categories?$count=true. Optional.
func (client Client) ListTableValuedFunctionsByDatabase(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLTableValuedFunctionListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListTableValuedFunctionsByDatabase", err.Error())
	}

	result.fn = client.listTableValuedFunctionsByDatabaseNextResults
	req, err := client.ListTableValuedFunctionsByDatabasePreparer(ctx, accountName, databaseName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTableValuedFunctionsByDatabase", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListTableValuedFunctionsByDatabaseSender(req)
	if err != nil {
		result.utvfl.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTableValuedFunctionsByDatabase", resp, "Failure sending request")
		return
	}

	result.utvfl, err = client.ListTableValuedFunctionsByDatabaseResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTableValuedFunctionsByDatabase", resp, "Failure responding to request")
	}

	return
}

// ListTableValuedFunctionsByDatabasePreparer prepares the ListTableValuedFunctionsByDatabase request.
func (client Client) ListTableValuedFunctionsByDatabasePreparer(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/tablevaluedfunctions", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListTableValuedFunctionsByDatabaseSender sends the ListTableValuedFunctionsByDatabase request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListTableValuedFunctionsByDatabaseSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListTableValuedFunctionsByDatabaseResponder handles the response to the ListTableValuedFunctionsByDatabase request. The method always
// closes the http.Response Body.
func (client Client) ListTableValuedFunctionsByDatabaseResponder(resp *http.Response) (result USQLTableValuedFunctionList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listTableValuedFunctionsByDatabaseNextResults retrieves the next set of results, if any.
func (client Client) listTableValuedFunctionsByDatabaseNextResults(lastResults USQLTableValuedFunctionList) (result USQLTableValuedFunctionList, err error) {
	req, err := lastResults.uSQLTableValuedFunctionListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTableValuedFunctionsByDatabaseNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListTableValuedFunctionsByDatabaseSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTableValuedFunctionsByDatabaseNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListTableValuedFunctionsByDatabaseResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listTableValuedFunctionsByDatabaseNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListTableValuedFunctionsByDatabaseComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListTableValuedFunctionsByDatabaseComplete(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLTableValuedFunctionListIterator, err error) {
	result.page, err = client.ListTableValuedFunctionsByDatabase(ctx, accountName, databaseName, filter, top, skip, selectParameter, orderby, count)
	return
}

// ListTypes retrieves the list of types within the specified database and schema from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the types. schemaName is the name of the schema containing the types. filter
// is oData filter. Optional. top is the number of items to return. Optional. skip is the number of items to skip
// over before returning elements. Optional. selectParameter is oData Select statement. Limits the properties on
// each entry to just those requested, e.g. Categories?$select=CategoryName,Description. Optional. orderby is
// orderBy clause. One or more comma-separated expressions with an optional "asc" (the default) or "desc" depending
// on the order you'd like the values sorted, e.g. Categories?$orderby=CategoryName desc. Optional. count is the
// Boolean value of true or false to request a count of the matching resources included with the resources in the
// response, e.g. Categories?$count=true. Optional.
func (client Client) ListTypes(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLTypeListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListTypes", err.Error())
	}

	result.fn = client.listTypesNextResults
	req, err := client.ListTypesPreparer(ctx, accountName, databaseName, schemaName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTypes", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListTypesSender(req)
	if err != nil {
		result.utl.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTypes", resp, "Failure sending request")
		return
	}

	result.utl, err = client.ListTypesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListTypes", resp, "Failure responding to request")
	}

	return
}

// ListTypesPreparer prepares the ListTypes request.
func (client Client) ListTypesPreparer(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
		"schemaName":   autorest.Encode("path", schemaName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}/types", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListTypesSender sends the ListTypes request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListTypesSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListTypesResponder handles the response to the ListTypes request. The method always
// closes the http.Response Body.
func (client Client) ListTypesResponder(resp *http.Response) (result USQLTypeList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listTypesNextResults retrieves the next set of results, if any.
func (client Client) listTypesNextResults(lastResults USQLTypeList) (result USQLTypeList, err error) {
	req, err := lastResults.uSQLTypeListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTypesNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListTypesSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listTypesNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListTypesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listTypesNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListTypesComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListTypesComplete(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLTypeListIterator, err error) {
	result.page, err = client.ListTypes(ctx, accountName, databaseName, schemaName, filter, top, skip, selectParameter, orderby, count)
	return
}

// ListViews retrieves the list of views from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the views. schemaName is the name of the schema containing the views. filter
// is oData filter. Optional. top is the number of items to return. Optional. skip is the number of items to skip
// over before returning elements. Optional. selectParameter is oData Select statement. Limits the properties on
// each entry to just those requested, e.g. Categories?$select=CategoryName,Description. Optional. orderby is
// orderBy clause. One or more comma-separated expressions with an optional "asc" (the default) or "desc" depending
// on the order you'd like the values sorted, e.g. Categories?$orderby=CategoryName desc. Optional. count is the
// Boolean value of true or false to request a count of the matching resources included with the resources in the
// response, e.g. Categories?$count=true. Optional.
func (client Client) ListViews(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLViewListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListViews", err.Error())
	}

	result.fn = client.listViewsNextResults
	req, err := client.ListViewsPreparer(ctx, accountName, databaseName, schemaName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListViews", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListViewsSender(req)
	if err != nil {
		result.uvl.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListViews", resp, "Failure sending request")
		return
	}

	result.uvl, err = client.ListViewsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListViews", resp, "Failure responding to request")
	}

	return
}

// ListViewsPreparer prepares the ListViews request.
func (client Client) ListViewsPreparer(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
		"schemaName":   autorest.Encode("path", schemaName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/schemas/{schemaName}/views", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListViewsSender sends the ListViews request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListViewsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListViewsResponder handles the response to the ListViews request. The method always
// closes the http.Response Body.
func (client Client) ListViewsResponder(resp *http.Response) (result USQLViewList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listViewsNextResults retrieves the next set of results, if any.
func (client Client) listViewsNextResults(lastResults USQLViewList) (result USQLViewList, err error) {
	req, err := lastResults.uSQLViewListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listViewsNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListViewsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listViewsNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListViewsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listViewsNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListViewsComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListViewsComplete(ctx context.Context, accountName string, databaseName string, schemaName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLViewListIterator, err error) {
	result.page, err = client.ListViews(ctx, accountName, databaseName, schemaName, filter, top, skip, selectParameter, orderby, count)
	return
}

// ListViewsByDatabase retrieves the list of all views in a database from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the views. filter is oData filter. Optional. top is the number of items to
// return. Optional. skip is the number of items to skip over before returning elements. Optional. selectParameter
// is oData Select statement. Limits the properties on each entry to just those requested, e.g.
// Categories?$select=CategoryName,Description. Optional. orderby is orderBy clause. One or more comma-separated
// expressions with an optional "asc" (the default) or "desc" depending on the order you'd like the values sorted,
// e.g. Categories?$orderby=CategoryName desc. Optional. count is the Boolean value of true or false to request a
// count of the matching resources included with the resources in the response, e.g. Categories?$count=true.
// Optional.
func (client Client) ListViewsByDatabase(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLViewListPage, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: top,
			Constraints: []validation.Constraint{{Target: "top", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "top", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}},
		{TargetValue: skip,
			Constraints: []validation.Constraint{{Target: "skip", Name: validation.Null, Rule: false,
				Chain: []validation.Constraint{{Target: "skip", Name: validation.InclusiveMinimum, Rule: 1, Chain: nil}}}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "ListViewsByDatabase", err.Error())
	}

	result.fn = client.listViewsByDatabaseNextResults
	req, err := client.ListViewsByDatabasePreparer(ctx, accountName, databaseName, filter, top, skip, selectParameter, orderby, count)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListViewsByDatabase", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListViewsByDatabaseSender(req)
	if err != nil {
		result.uvl.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListViewsByDatabase", resp, "Failure sending request")
		return
	}

	result.uvl, err = client.ListViewsByDatabaseResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "ListViewsByDatabase", resp, "Failure responding to request")
	}

	return
}

// ListViewsByDatabasePreparer prepares the ListViewsByDatabase request.
func (client Client) ListViewsByDatabasePreparer(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}
	if len(filter) > 0 {
		queryParameters["$filter"] = autorest.Encode("query", filter)
	}
	if top != nil {
		queryParameters["$top"] = autorest.Encode("query", *top)
	}
	if skip != nil {
		queryParameters["$skip"] = autorest.Encode("query", *skip)
	}
	if len(selectParameter) > 0 {
		queryParameters["$select"] = autorest.Encode("query", selectParameter)
	}
	if len(orderby) > 0 {
		queryParameters["$orderby"] = autorest.Encode("query", orderby)
	}
	if count != nil {
		queryParameters["$count"] = autorest.Encode("query", *count)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/views", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// ListViewsByDatabaseSender sends the ListViewsByDatabase request. The method will close the
// http.Response Body if it receives an error.
func (client Client) ListViewsByDatabaseSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListViewsByDatabaseResponder handles the response to the ListViewsByDatabase request. The method always
// closes the http.Response Body.
func (client Client) ListViewsByDatabaseResponder(resp *http.Response) (result USQLViewList, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// listViewsByDatabaseNextResults retrieves the next set of results, if any.
func (client Client) listViewsByDatabaseNextResults(lastResults USQLViewList) (result USQLViewList, err error) {
	req, err := lastResults.uSQLViewListPreparer()
	if err != nil {
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listViewsByDatabaseNextResults", nil, "Failure preparing next results request")
	}
	if req == nil {
		return
	}
	resp, err := client.ListViewsByDatabaseSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		return result, autorest.NewErrorWithError(err, "catalog.Client", "listViewsByDatabaseNextResults", resp, "Failure sending next results request")
	}
	result, err = client.ListViewsByDatabaseResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "listViewsByDatabaseNextResults", resp, "Failure responding to next results request")
	}
	return
}

// ListViewsByDatabaseComplete enumerates all values, automatically crossing page boundaries as required.
func (client Client) ListViewsByDatabaseComplete(ctx context.Context, accountName string, databaseName string, filter string, top *int32, skip *int32, selectParameter string, orderby string, count *bool) (result USQLViewListIterator, err error) {
	result.page, err = client.ListViewsByDatabase(ctx, accountName, databaseName, filter, top, skip, selectParameter, orderby, count)
	return
}

// RevokeACL revokes an access control list (ACL) entry from the Data Lake Analytics catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. parameters is
// parameters supplied to delete an access control list (ACL) entry from a Data Lake Analytics catalog.
func (client Client) RevokeACL(ctx context.Context, accountName string, parameters ACLDeleteParameters) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.PrincipalID", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "RevokeACL", err.Error())
	}

	req, err := client.RevokeACLPreparer(ctx, accountName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "RevokeACL", nil, "Failure preparing request")
		return
	}

	resp, err := client.RevokeACLSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "catalog.Client", "RevokeACL", resp, "Failure sending request")
		return
	}

	result, err = client.RevokeACLResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "RevokeACL", resp, "Failure responding to request")
	}

	return
}

// RevokeACLPreparer prepares the RevokeACL request.
func (client Client) RevokeACLPreparer(ctx context.Context, accountName string, parameters ACLDeleteParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "REVOKEACE"),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPath("/catalog/usql/acl"),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RevokeACLSender sends the RevokeACL request. The method will close the
// http.Response Body if it receives an error.
func (client Client) RevokeACLSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RevokeACLResponder handles the response to the RevokeACL request. The method always
// closes the http.Response Body.
func (client Client) RevokeACLResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// RevokeACLFromDatabase revokes an access control list (ACL) entry for the database from the Data Lake Analytics
// catalog.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database. parameters is parameters supplied to delete an access control list (ACL) entry for a
// database.
func (client Client) RevokeACLFromDatabase(ctx context.Context, accountName string, databaseName string, parameters ACLDeleteParameters) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: parameters,
			Constraints: []validation.Constraint{{Target: "parameters.PrincipalID", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewError("catalog.Client", "RevokeACLFromDatabase", err.Error())
	}

	req, err := client.RevokeACLFromDatabasePreparer(ctx, accountName, databaseName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "RevokeACLFromDatabase", nil, "Failure preparing request")
		return
	}

	resp, err := client.RevokeACLFromDatabaseSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "catalog.Client", "RevokeACLFromDatabase", resp, "Failure sending request")
		return
	}

	result, err = client.RevokeACLFromDatabaseResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "RevokeACLFromDatabase", resp, "Failure responding to request")
	}

	return
}

// RevokeACLFromDatabasePreparer prepares the RevokeACLFromDatabase request.
func (client Client) RevokeACLFromDatabasePreparer(ctx context.Context, accountName string, databaseName string, parameters ACLDeleteParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", "REVOKEACE"),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPost(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/acl", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// RevokeACLFromDatabaseSender sends the RevokeACLFromDatabase request. The method will close the
// http.Response Body if it receives an error.
func (client Client) RevokeACLFromDatabaseSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RevokeACLFromDatabaseResponder handles the response to the RevokeACLFromDatabase request. The method always
// closes the http.Response Body.
func (client Client) RevokeACLFromDatabaseResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// UpdateCredential modifies the specified credential for use with external data sources in the specified database
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the credential. credentialName is the name of the credential. parameters is
// the parameters required to modify the credential (name and password)
func (client Client) UpdateCredential(ctx context.Context, accountName string, databaseName string, credentialName string, parameters DataLakeAnalyticsCatalogCredentialUpdateParameters) (result autorest.Response, err error) {
	req, err := client.UpdateCredentialPreparer(ctx, accountName, databaseName, credentialName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "UpdateCredential", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateCredentialSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "catalog.Client", "UpdateCredential", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateCredentialResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "UpdateCredential", resp, "Failure responding to request")
	}

	return
}

// UpdateCredentialPreparer prepares the UpdateCredential request.
func (client Client) UpdateCredentialPreparer(ctx context.Context, accountName string, databaseName string, credentialName string, parameters DataLakeAnalyticsCatalogCredentialUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"credentialName": autorest.Encode("path", credentialName),
		"databaseName":   autorest.Encode("path", databaseName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/credentials/{credentialName}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateCredentialSender sends the UpdateCredential request. The method will close the
// http.Response Body if it receives an error.
func (client Client) UpdateCredentialSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// UpdateCredentialResponder handles the response to the UpdateCredential request. The method always
// closes the http.Response Body.
func (client Client) UpdateCredentialResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// UpdateSecret modifies the specified secret for use with external data sources in the specified database. This is
// deprecated and will be removed in the next release. Please use UpdateCredential instead.
//
// accountName is the Azure Data Lake Analytics account upon which to execute catalog operations. databaseName is
// the name of the database containing the secret. secretName is the name of the secret. parameters is the
// parameters required to modify the secret (name and password)
func (client Client) UpdateSecret(ctx context.Context, accountName string, databaseName string, secretName string, parameters DataLakeAnalyticsCatalogSecretCreateOrUpdateParameters) (result autorest.Response, err error) {
	req, err := client.UpdateSecretPreparer(ctx, accountName, databaseName, secretName, parameters)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "UpdateSecret", nil, "Failure preparing request")
		return
	}

	resp, err := client.UpdateSecretSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "catalog.Client", "UpdateSecret", resp, "Failure sending request")
		return
	}

	result, err = client.UpdateSecretResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "catalog.Client", "UpdateSecret", resp, "Failure responding to request")
	}

	return
}

// UpdateSecretPreparer prepares the UpdateSecret request.
func (client Client) UpdateSecretPreparer(ctx context.Context, accountName string, databaseName string, secretName string, parameters DataLakeAnalyticsCatalogSecretCreateOrUpdateParameters) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":          accountName,
		"adlaCatalogDnsSuffix": client.AdlaCatalogDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"databaseName": autorest.Encode("path", databaseName),
		"secretName":   autorest.Encode("path", secretName),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
	}

	preparer := autorest.CreatePreparer(
		autorest.AsJSON(),
		autorest.AsPatch(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlaCatalogDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/catalog/usql/databases/{databaseName}/secrets/{secretName}", pathParameters),
		autorest.WithJSON(parameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare((&http.Request{}).WithContext(ctx))
}

// UpdateSecretSender sends the UpdateSecret request. The method will close the
// http.Response Body if it receives an error.
func (client Client) UpdateSecretSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client, req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// UpdateSecretResponder handles the response to the UpdateSecret request. The method always
// closes the http.Response Body.
func (client Client) UpdateSecretResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}
