package errorpage

import (
	"html/template"
)

// ErrorPageTemplateExample is a basic template for customizing the error page.
const ErrorPageTemplateExample = `<!DOCTYPE html>
<!--

This template can be modified and used to customize the error page. To replace
the error page, set master configuration option oauthConfig.templates.error to
the path of the template file.

oauthConfig:
  templates:
    error: templates/error-template.html

The Error field contains an error message, which is human readable, and subject to change.
Default error messages are intentionally generic to avoid leaking information about authentication errors.

The ErrorCode field contains a programmatic error code, which may be (but is not limited to):
- mapping_claim_error
- mapping_lookup_error
- authentication_error
- grant_error
-->
<html>
  <head>
    <title>Error</title>
    <style type="text/css">
      body {
        font-family: "Open Sans", Helvetica, Arial, sans-serif;
        font-size: 14px;
        margin: 15px;
      }
    </style>
  </head>
  <body>

    <div>
		<!-- example of handling a particular error code in a special way -->
		{{ if eq .ErrorCode "mapping_claim_error" }}
			Could not create your user. Contact your administrator to resolve this issue.
		{{ else }}
			{{ .Error }}
		{{ end }}
		</div>

  </body>
</html>
`

var defaultErrorPageTemplate = template.Must(template.New("defaultErrorPageTemplate").Parse(defaultErrorPageTemplateString))

const defaultErrorPageTemplateString = `<!DOCTYPE html>
<!--[if IE 8]><html class="ie8 login-pf"><![endif]-->
<!--[if IE 9]><html class="ie9 login-pf"><![endif]-->
<!--[if gt IE 9]><!-->
<html class="login-pf">
<!--<![endif]-->
  <head>
    <title>Error - OpenShift Container Platform</title>
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="shortcut icon" href="data:image/ico;base64,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">
      <style>
        /* Standalone login -- OpenShift Container Platform edition */
@font-face {
  font-family: 'Open Sans';
  src: url(data:application/x-font-woff;charset=utf-8;base64,d09GRgABAAAAAFigABMAAAAAlYwAAQAAAAAAAAAAAAAAAAAAAAAAAAAAAABGRlRNAAABqAAAABwAAAAcauKfMUdERUYAAAHEAAAAHQAAAB4AJwD2R1BPUwAAAeQAAASjAAAJni1yF0JHU1VCAAAGiAAAAIEAAACooGOIoU9TLzIAAAcMAAAAYAAAAGCgqpiQY21hcAAAB2wAAAGiAAACCs3ywEljdnQgAAAJEAAAADAAAAA8KcYGO2ZwZ20AAAlAAAAE+gAACZGLC3pBZ2FzcAAADjwAAAAIAAAACAAAABBnbHlmAAAORAAAQTcAAG9g4Tc27mhlYWQAAE98AAAAMwAAADYHI01+aGhlYQAAT7AAAAAgAAAAJA2dBVRobXR4AABP0AAAAkUAAAPA/YtZ22xvY2EAAFIYAAAB2AAAAeK6PZ9ObWF4cAAAU/AAAAAgAAAAIAMbAgduYW1lAABUEAAAAfwAAARyUBqcRXBvc3QAAFYMAAAB+gAAAvpj5wT6cHJlcAAAWAgAAACQAAAAkPNEIux3ZWJmAABYmAAAAAYAAAAGxDNUvgAAAAEAAAAA0Mj48wAAAADJNTGLAAAAANDkdLJ42mNgZGBg4AFiMSBmYmAEwvdAzALmMQAADeMBHgAAAHjarZZNTJRHGMf/uyzuFm2RtmnTj2hjKKE0tikxAbboiQCljdUF7Npiaz9MDxoTSWPSkHhAV9NDE9NYasYPGtRFUfZgEAl+tUEuHnodAoVTjxNOpgdjuv3NwKJ2K22T5skv8zLvM8/Hf+YdVhFJZerQZ4o1Nb/XoRc//7p7j6q+7N61W7V7Pv1qrzYpho/yeXnff/Mc2b2re68S/ikQUzSMCUUS3cFzp+7oTuRopC9yF+5F09EsTEXnotmS1dF0yQEYif0Sux+7H82Wzq/4LXI0/ly8Op6CL3jaD/7v6vhP8VQimUjG9yeSxLv3wIiWhQVLP2zEDVY6X3IgxClY9aOW2AlJT3SqdJ5K74aq+wJvqTK/T3V6TQ2QhEY9q6Z8Ts35jFqgFdryE9oCWyHF3+2MHYydjNsgDb3EOQiHIAOH4Qj0E28A3zPEPAvnIAuDcB4u8G4ILsIlGIYRuAKjcBXGYByukec63ICbcJu5SeJHtF5jel5VeaMaqIUNUEf++rxVA35JaIRvmD8G30Mf/ADHwcAJfE/CKTgN/fhPMD/JGCFajhylxCyDKt7XwPpIGfks+WzI14BXEhZyWXJZcllyWXJZcllyFWLbEHuadbPwjMpZWQGVIdoE0RzRnN7m70bGjdDL80E4BBk4DEdCREc0pxnWz8GqpRoL9S1Xj6/F69jDunJqqoB1nAdfyeMyzuAzBy+hSheqdBVlrIN6ampgTIYeJpat4gS+J+EUnIZ+/BdUmkClLlTq0pMq/+N3VUAle+OVWVDFUKOhRkONhhoNNRrN4DcHzaGr1UHfQmf7iutlvokczbxrgVZogy1E2gopntsZOxg7GbcRK824nbUfwkfQBTvI87gvYrn+B3h/hvxn4RxkYRDOwwXeDcFFuATDMAJXYBSuwhiMwzVqug434CbcWtzh27yz1DYFhd1biTIWVSyKeB0dVTuqdlTtqNpRtT9VFm92EG+Dt1nUMIeGDg0dGjo0dOhn0c+in0U/i34O/Rz6OfSz6OfQz6KfQz+Hfj5rjqw5subImiNrjqw5tHJo5dDKoZVDK4dWDq0cWlm0smhl0cqilUUri1YWrSxaWbSyaGXRyqKVRSuLVhatLFpZtLJo5dDKoZVDK4dODp386TZ0bLTxL99DpujUNOHVDC3QCm3MPbgvzeJ9aRbvy1y4L3eE7ypD1xm6ztB1hq4zdJ35hxNi6NrQtaFrQ9eGrg1dG7o2dG3o2tC1oWtD14auDV0bujZ0bejaFN2lC6fDLJ2KVUX7utxeeM1i3AKOW8DxpTq+VJ6XZoq/DxfOZMGTtWhbBtMwC36mh5keZnqY6dHTj5wqf5I6gh7/bbf9zq4hdorYqb89qw9H/j/Ol884Ta5ZeGIpc+GmXxd6ToVb23v4m9sradHN62PRx/LLYy0rS8OvnJXc0+WqUIkqWbtCb+hNdqtWG/QU99cm3jRx272gVr2jl/UutkabsbXaona9ok6sUh9gr2q7uLP1MVajXn2r1/UdVqdjOq56Gf3I6R/QIBGHNKw2XcY2a0Sjep//uGPUO46165Z+5tcXp4iok1haVr8SfQ775E+Ohly2AHjaY2BkYGDgYohiyGBgcXHzCWGQSq4symFQSS9KzWbQy0ksyWOwYGABqmH4/x9IYGMJMDD5+vsoMAgE+fsCSbAoyFTGnMz0RAYOEAuMWcB6GIEijAx6YJoFaLMQgxSDAsM7BmYGTwZ/hrdg2ofhDQMTkPcaSPoAVTIyeAIAomcaGQAAAAADBD4BkAAFAAQFmgUzAAABHwWaBTMAAAPRAGYB8QgCAgsGBgMFBAICBOAAAu9AACBbAAAAKAAAAAAxQVNDAEAADfsEBmb+ZgAAB3MCGCAAAZ8AAAAABEgFtgAAACAAA3jaY2BgYGaAYBkGRiDJwMgC5DGC+SwML4C0GYMCkCUGZPEy1DH8ZzRkDGasYDrGdIvpjgKXgoiClIKcgpKCmoK+gpWCi0K8QonCGkUlJSHVP79Z/v8HmQjUp8CwAKgvCK6PQUFAQUJBBqrPEk0fI1Af4/+v/x//P/R/4v/C/77/GP6+/fvmwckHRx4cfHDgwd4Hux5serDywYIHbQ+KHljfP3bv+q13rK8g7icHMLIxwDUzMgEJJnQFwCBiYWVj5+Dk4ubh5eMXEBQSFhEVE5eQlJKWkZWTV1BUUlZRVVPX0NTS1tHV0zcwNDI2MTUzt7C0sraxtbN3cHRydnF1c/fw9PL28fXzDwgMCg4JDQuPiIyKjomNi09IZGhr7+yePGPe4kVLli1dvnL1qjVr16/bsHHz1i3bdmzfs3vvPoailNTM+xULC7JflGUxdMxiKGZgSC8Huy6nhmHFrsbkPBA7t/ZBUlPr9MNHrt+4c/fmrZ0MB48+ef7o8es3DJW37zG09DT3dvVPmNg3dRrDlDlzZx86dqKQgeF4FVAjAOp1mFcAAHjaY2BAA2sYekCYdRsDA+tPFg8Ghn8iHEl/17Ke/f8GyI/5/wbCZ3BhFQQAXyERInjanVVpd9NGFJW8JI6T0CULBXUZM3Gg0ciELRgwaSrFdiFdHAitBF2kLHTlOx/7Wb/mKbTn9CM/rfeOl4SWntM2J0fvzpurt1y9GYtjRKVPA3GNOlTyciCV1cdS6T6JG7rh5bGSwSBuyFbiKWkTtZNEyWw3O5RLXM52lawTrJPxchCrpyrPMyX1QZzCo7hXJ9og2ki9NEkSTxw/SbQ4g/goSQIpGYU4lWaGEqrRIJaqDmVKh16jkYibBlI2GvWow6K6HyruHM+6pbUGYKRylSNcsV5t5rtxOvCyB0msE+xtPYyx4bH6UapAKkamI//YKTlRGgZSxVKHWomjw0x+3UcyqawFMmUUKyp1D8Tt7qfbtojpodPxdVGrNFPVzXVG0WyPjkcdRHnINk4n5abOtocv10xRrXbFzbYDmTFwKSUz0X0SAXSYSJ2rB1jVsQqkbtQfFWefjwMkktkoVXkK7VFvILNmZy8upt3tZEXmj/TzQObMzm6883Do9BrwL1j/vCmcuehRXMzNRUgfSt1PxImk1AyLGT7qeIi7DBHKzUFcuFAGnyLMoSvSzqw1NF4bY2+4z1dKTetJ0EYfxfdT6HciWeE4CxqtR+JsHruua+U+g1qq3b3YkTkdqhRxf5+fd51ZJwzztJiv+vLM9y6g+TdAPOMH8qYpXNq3TFGifdsUZdoFU1RoF6Eq7ZIppmiXTTFNe9YUNdp3TDFDe85Izf+Xuc8j9zm84yE37bvITfsectO+j9y0HyA3rUJu2gZy015AblqN3LQrRnXsCDQN0s6nKoKgaWT1w7itrDUCWTXS9KWJybuIIeurEx111tYqfxT/1YkvHMiliZ7uslxcE3dp3bbw4el2X91aM+qGrcY3jpSH8TDS49CEzvJvDv+2N3W7WHOXUJVBD6hgUgAGKGsHEpjW2U4grdfs4ssfgHEZ4jnLTdVSfZ4xNH0vz/u6j5MT73s83TjLLdddWkSWdYPcmD38W4pMdf2jvKWV6uSIdeVkW7WGMaTCi6LrK0l5jrZ24xclVVbei9Jq+XwS8mTXcENoy9Y9DHaEKU15iIfXVClKD7WUo+wQh7cUZR5wyoMLWobEuA51D2prxOmhehgbCyGGobS9ELBIKV0V37TKd/Eeq2va6HjiivB0IzmJiE9xlf0oeKqro350B21es26pYUqV6uk+41Ps67Z9VFYaqePsxS3VwTXNukZOxfQT+ZpY3RsOWvdADxUfTdBIVc0xujHKGI1lTfmbgC7Gym8YrVpsv4f7qZO0ilV3EZN9c+IenHa3X2W/lnPLyLr/2qC3jVzxcyTmt0WBf+dA7JasgnpnMhBjATkLGsPYwuQOw3UML+vwf0xO/78NC4vkWe1onM1TH66RjCq5y5bHXW6yy4YetTmqdtLYR2hsaXhijh0ejoWWGByQrX/wf4x7wF1ckAA4NHIZJqI2Xaineri6x2psG86VRIBdc+w4HYAegEvQN8eu9XwCYD33yLkLcJ8cgh1yCD4lh+Azcm4BfE4OwRfkEAzIIdgl5w7AA3IIHpJDsEcOwSNyNgG+JIfgK3IIYnIIEnJuAzwmh+AJOQRfk0PwjZGrE5m/5UI2gL6z6CZQaqcGizYWmZFrE/Y+F5Z9YBHZhxaRemTk+oT6lAtL/d4iUn+wiNQfjdyYUH/iwlJ/tojUXywi9ZnxpXYk5ZXBc97RwZ/uYa1oAAAAAQAB//8AD3japX0JYJNF9vjMfFeSJmnOpgc90jQNpRRo04NyNbTlkENKW5ACi9yWoiK3yCICcsklh+VWRKxYWEQshyyieCDIKrKoyKLLT/FYV5dV1/WAZvp/M9+XNC2g/n5/oLRN5pt3zJt3zXsTRFApQmS8NAQJSEEdnseoY7cDitjmXznPy9JH3Q4IBH5EzwvsZYm9fECRExu7HcDsdb/VbfW6re5SkkLT8CZaLQ25tqdUfBvBlGhT0xW8XDoE85pRUiABXsODEcbGEkSIUIEEwSmUpqVaLaI9E3sEv5Drz4lxOmRPajruPcl/9pP7uxQFCnNL8XrRc61hWe/iQJ8iPu9ioY7s5/MqyBNIIZhNLAmiAHOjUlFESFRERZZggGCVLZlY8Ahu+MJFbSdlkMyM6gzpUPBbYmFfbD4/PPAjzJeAklHvQInRQHTRFpOgIJ0y3CwTJAkEE4SrorBebywRMSEmAjxLTkpsA88kxMfFugBvuzX8JxZAup0A0s6/8tz8yy/wLyeGX4X3i3ECfbtyeSU9V76sjF7DyaX0a5xZvqIcZ1cuqcS6xs9xx2J6TlhE9y6g5Xgf+1qAK+fjBjqAfc2ne3ElsAOIXNS0XDTKNpSC0lEWmhSIdWBRyGznTUtsEx9nNhiIqGcUCMX997ctGwYLIQpEEEk1kI4RwSPYJINhOcyoNCGQEn5XEJEwMDxGENBgBqxX1SFrrNPqkByZ2CErTk9earovLwn7rR1wXm5+QZ7fGeNS0n3WJKLkwrd87IhxWc1YNP7l8IL7/lpScbHq7afPPLPgyJ7cx7Zs39avvuqhi8GPh08ZNxGfWPa86x+XPcnbvB3xkZ57ly3abTvUIPVa1DWK3p5z59wJfava07lJgjJgZAZeZPkDoC2h6qZv5CzpDNIjJ3ID9dlob//9LqCyLQiHhBRpJBMZEZERIHlChUFPBMFRgkRRrtBhWXbKpQn998fB+Patxxv4avOn0A3PBDr95nCdzqQ+g/gjVVUBa4cOHbI7ZNvT+J/U1KjYTLsjxp9jtXhSZYkLPnAQNgFmr+bn5aa3el2PPfj7fhW7dlX0w29t3rBy62Pr1m7Ddf0qK8vKKiv74TObN6ze/Ni61U9Q2vj+eiFTJPX1uBKX767/7Kurl698cbXx0p5nn/nTnqef3nPlq6t/v/LF10LKtX58J01t+kY6L72NooCPeWh6wJIeJ8CSd8ryRZslLJBilUupjNwK2L9AoCCIFRIWRVOJjDHWfha5CLUYNBiFxkgVSJKcUimwwmTMzenYIbOdO9noNDnbKrBLQU56YJVoe04BNhOnI8YLEtWBqNSDalBwD1zgJwr2+MyY6YnH+96xYMJdQ6umbP3uCdp/ysj2W+mLKxqGdE97/bmdR5dtxxs7l7h2ly7HmZ+/OOuH2gv/Etf3mjes//yKgWNGX9++Be8urZrYc+byawtPTbxzbE1h7e5nHpt88A90To9nxtFPN9CPD9SMfI/tMcx0Di7inIoNOLkmI5irG74DBaugqhmuYlTtoj5XTo8RNzxnQrZANEgH20TIiEtj7MSaabfYCvwy0GlzedJJ+dZ1Ox9du2HFjvVbSDbW43f2naA5P3xL81+qxyfZXN1hLmN4rpA2Repc2EIUT74tL5f4/DE2Yty6bseKDWsf3ckmo7/QLruP4TPf/oDfOfEczYa5hpL5oll2gGbODlhMxiiDXgcaE7Y+NqGi/vvTy4YdAmQRUwkN7IcxVYcR+9VMooFQr0uyK1HYZ/cWgHJcm4lXx9NFP+/dv2P/93RpIl6aKTvozCmHkunRUbiG1o7CvZMPTcErAG41uiJmiK+BnHlBmCREpIEiRqC5mYIF6SCD4SXSi6tRi+zMxKBHrR4r6E+rn6zC2+j4FXQi3rJCcD1Ch+L6R/Belc9F9Gd8D7qKdCgt4AaEMS4mDG88kK0T4zpGvWCkDukcsFZel8zZVYBHRcePzZuX0NN01TWe/jStGueMZGMr8UVSRKbC2iYG4hFbuIGhVUe4V/Oi2/Pczkr8Nb64aRPHg9s89D3Qp61RBTzrxGDnYI0KQnIMorupe2GXnsWF/pKakl69Snr2LlLpcMDWuRSSMy5iJfC6KVLOmBYgl4KX65hg8e0LSnpi0zdiFt+/Lm5rBYFDdjBbiyrgYScqZUqH2VoL8QA2Fps/x4b5/1b+ipj17x+/+fHqD1d/avykdlfdY4/V7aolH9PF9BE8H0/DD+Jp9EG6jp6gH2Mf7gp/vfQyx/kYIHAG0DCg+IBLp4gC4z5qRj3WArS7PdbcAjNWfNhPzuzUOXPfG4YXrRRtC2c4O+ydhjNhnvFgh72gy+NQu4DPZTYBDWDbQffAXM36hO8gmNWV1pZ5Dhk4D/cgqtpQfCp/QUE43aK3sQgv2H9f59UPjnhq/LC3rr7zz20f0FfIt2vwogObHq2YubzboKm7zx9YQb99l76pA/ijgIcJAN+HugUKU91gBWE/EMEM0E2w6mIxrAABXTgCFBpXZQ6m70Pc9aa5U70ZTGSZWUzCTofo1kxjitXi9uSFVbri64EBT+Hq9kf319O/0//OODHizguj8Vw66tF1e06tf3B0/T2Vw79e+P434qiVB5J0MQ3rzn3iaf94x2ycgQ1rNi6Z/EBu7/v6DH4N1j0TeFYjHQfe21BOoKOMkYiLFZmIhGGM2Jbitp2bJhMujYqKskXZHFYbbC8d4Orhmwust98N6+MBDSvWXHwiWEgO7b9Ilxl0ndrRAlxG9+OytcLHjRn4izUNo4uCs1SdCPxKhHWPRyWBQJyDCMiuB37pMPM1ABNYOiSMAAy4CeBuX2g7YMR8JpvFFKVIKB7HK6BVckSnA3lSOceAVRZ3quKzMzcqn2R9ifX0Mv1pYe93J+x/jS6/84mhBeRC8LB3ujDv8zevUDpoR5a/bjvOSSwgezfT21zcT5wJ+HWE9YxBaag4UCTBWskYZKoYEVmSiQQuDpKJII8IraKjRMGAbQVg6xRLY11JbVxpsWlpbpsnVecAbY3cOS4nrCAR/JqgeUCHactqhhVPwjPxINx3as+B477+2Wi89+obV3557wr9EX+9evu6tcNrq8rWk6n4ObzHviaOXqIn9179y2f0Oh5y6oVn19b1W9j7rgPVTA5hTTOBrzLTZRLXZc2+sxByP+Ft2WoVYWdh0I5OjBPJqMYrwtvBeilx8+JrZ1XfEPwjMYPzIBV1QHmBnIwUqywSAReDVRdQBShf5IhcmTQPRp4OaR0S4kxRKAbHyGxluO1lIq36Il7u6gkhLwVsMU6VnSH7Tfqe+ueS9Qdq6cf/bMQ5j9z/9exnNj5Wt+3Vx5bgLvNWz3pizey10pmju+4+cNuQP889dPHtY9dX3n7wvideul53/5KVD4zZ2CewVbjr/vEjHy7u9sjICbMRX8saoIPpBhfyamspAOp8LSVZInI1sEPAsjAivIItNmdcbHJirDfOm5Zq87hhLTFIly8P8LfZPdyTyMtFsKI2lQ5/DiOkAxa9weoZpWUTv/1vlLHg0LRXP0NN7z52+X7qWLPt0fUjNg8rXy/0bqxzrImHfemvuOMf736GdZvpJdzpyK5Hn+73UO9JByaG4yBxPPetVN3MtZmrJMI+eDxcN0daCCvEQ61sxay5kRZD+PPSparhINz+LQYYOtADGYF01QKK3AKiIWGIAsiNNdpo4GZQijCDQhhWs0F0zgZoXbo320XybwavN9/71eh7sOMXmRC+APq+baYX9LBTj8WMxiLhROP3gvl7XDcWX36MrqJHAL/Z+IQYJ1xRY8wA01qayQd/nAxkAeFg5mYKvW4M35iJxfA1W9jZOErYKQxfvpzetXw5ugEPWQA87HqGCK4Wohu/A0SK8HLcG898jLrH0uGAR0LTFaEQ5CgBYqXCQL4Z+GQCRAhsB8LUu1AdWhTNw1V1AUJeT3Iii/bSMmRwtlikE3JWQ1YHjJsZJ+LIMNbdd+yyPovmDqod0/Wld15+zzfgwQk9D4TD2vyZGyunzyifOMWbvXTcsT23TRl379Bpd7rphYhYd05Tb/mI1AA+eRFaEbA5sEHoirGhLZZRBlZkEfzyRPDLOyBsEA1YrIYVAM0vi2gEsESRlTtA/eoqIBgB9Wsw8Ngk2gA+evvQA0C0AQuGEbd+kPnr0T26FRZ4PV4v30EQuGAH3za+0O4BNyLGJTgdXCOQNE+qSJxMXxY4ZU8KAmcjzZ0j2jC8n4AZ0+Qjw3ZUD50VpWv72MTaZ795pXRvr7jFI6ZtoP9+/jI9tA8X445//fSVH+hjdMoHeCVGF/Cgw9d/fO2szdx3yML15OLqbxZWD75j7Nv7/4Ka4mJou5gDF/YexJb1L9JnP6Fn6ZGhiyvxGjwRi7j28kH6At1FcSGWHA2MpyxwtEjHQBLNqFMgK4pRDvYKBA++QEiIsYSF/Mxc6nTgMpp1ZhirgEeq2DLdLJWA/XospPtkRRBHHwoeathLiteQIjqh3u2JydiLz9GO0rFrpaQGvzF03ujptKuqi0+DZB8HvW4GLZbCdLFOAV8XM3OEJZFIPMhmCsvIAkcuhIBCbGxsSmxyapq7XYoCRsidIjJl7BStDtHjtru5Bs7EcVj9yXMabwf/UFy7Ag+kP12lpB7HNjy+/5VrOPv5Fw7/WTq079jCZ+MMhfTS6x8JpVOXzL0nuDb48fJ1yx5S99M80LNnub3ID/gd4H05dSAWenDEhGJmK0LRreaROQmYcrvVaAAzDsZCAmORCqsNulRb+Fxbmj/HpaQLXb+jX2HTL5tf33SZvkSffBYXffjF3r51kp++TL+in9A3Cx4rxMvwpE9x5ZHKdbezdQJ+ScOAXzpk5XpTM1LGkhCfwAt0g9fFfVrYDlZ3ChJA6DxWfwo8SWfTNXQyfgUPwQ80AITPfzqLO+Ec8hWtpQukQ3QJfQYn4dTr94HzyWkHeMLPAC8KtQ14Yb1ESRCrGFQtayVJId8zHJ4woBCg8C/h58ZzQsfgXDIquJMslg5tpBm1wS8i5tWj1EAymxe1nDc8pUVyMD9ZmxFPrw9NB5MFv9qorY/UneuvLgFwpwUSDWsUh0WQIwi/mZ8giUiqCml8R1iSwMbY0jxejxpUWWHbRqwO36JstwIVA/B3M759nDbQ1YfxsC//+VbxycP0v/Q97Maxm9bSFwkNFnrT8Qo8/jN8x8GhtZX0VfoF/ZC+48GvqrRKyZyHKYFEoJT5o1XM1TCWsDxdJO8YpeB9shSiX0qub6T19QKpJ/uDZRBAryP3htYET4+Iu9k0fFs0x0Pgu1pP17NQCOGm0XQ+ruYxsi+QxsJKonoJ4PfBbHewzc1E1qAHTWfCJjE6U1Adgfw8mMYhG0fl9xk47q76V+j8hLWO+6fCtMOOnwvLh/gVpy050CZKJ7C8Dy5mDpmxBKvE2ax2FpS4PVjhpGE/Fr86Ffx6c3092Xg62EDeWBZ8DcjLJO8HF0fKnKRlSLmfTsqbvTwJSdzLY1S6nafrmWRd+2yL9qwMCCI7ah/IYP42GSxyZFj0N1inSEKIU3Zk5+lJu1UfA/6iJ8bJ3H5Azm9lYZkHC2XUojNvBx6d1eHztN6so0bJsuW9xlHSoev9ROCtsG/n8Wvfh/ggsVyIhcszeGKD1QUGUUPl2io7GDB1lTVwACydFMBSF73a+InOsv0VIVFHzOQJ8cyHDY0nAEyXBOwtEvqo/h6T8+O/N55FblV+U1BkPCsdp9vp60zZ4NEQcXTFY69PD373n59/+v4/QYhrn6T3ghRX4wl4Ob2P7qAX6Ns4B7eD2Dabqvlrtt7VfN/atJ0LYoRY9Mf3ViikSnO7rak8HSq6eTiVgv2aVhar6Tn6xd56PJekBaM2f/rW8dPHRePf/x0EeoMJ655cu5rTSndxWqNhV0PkC6smMI8fo8khUNwXCXvoEJHw9KuFJcKwcnPaP30Hj/yJXim4Jf1f0pXFtBb3I7/CBZUHZ4EHRmA4xJd65nYXIxHiKCQKLLrUVLK27LAu9mgTjDaCDgNriTk/RKdFAky9IbaAZsNrgXWY/g2vOk0fp2e/ati956WPyejgDunQO2fp3ycGp5DR69asWfsQsIH5/gR0XhrKCrSLBZ0XJ4M5YoGRUByh6Jq9fU+mKhhgIVNC4SS48z4etqgKj/n5SdiVBLaf/uNb2rhy+HvV9Xu7r1n3l+foub8dzju4Z+mmzouXf/EnvPjEhyW70tsvmD5gTHnubaeefPZU2YYBM+4aMGZwdvkxvh9swKPhwCOF5T9g9TBLYpEIxa6qBJbEcrrJinqaLfal2VLyxo2qXjkKsh4Hz1uZzmQpOPWIJEQN/G9FVo/dIzF+OthSc12lKi0pjv6V/gx/L9efevXFU9Khxtuv0U9xSqOwr7H34dffOCIcYTDgj/haOEfDwoNitmQhGQYcBZjdD6vkxzGu/ALReDj43e7gDw24rEtqWhc10dR4+1Nbdj7JcQZdpQyA+eIYzk6zxDIjxSxd0qwLY2KsLtWiYeC9zGTVb2dzayDsfumN3fSiMxfHdKKf7qZzG77sFJOQh+UGbO/ktud90SCc73nS8fD2Rj+AnnN8y+HnhDmN87e+tvovAted4JuJtc36WLqZPrbZVObrMfwDrw30ce0J8Azua8Bf0K7H8GR870HalSwKziON5GjwJVISHBDi11zue7QJxMlgaVpzzGazAsfYhCDi2E+64yVHaOxeGneUXCKXGhcET5OOwlI1BwxzFXE9An6FotokLaulCUpzVstmtYWyWjyqd4tFjXZBF7QILzf+LCQtFjdtXnx9kio3dfQYmcLlLiEQy7STAKFeyK0gIcHDmntCptACfJpexm56TL626pqbz+ECBfTxr+cb7bBwR4mnLvg/oXQjPOenx3BjGDYfOyRMEA7Bdrl55tbtB6CXAXjBIenyql9keD6b+ESPdAJiDZAficujwG31ZGYEUWR+w87EUo+z8YFx+ONNdCU9QHzClsaJ5Ksgy/Pgpkb6hrClqS/g4Ao4EE/6onDO18qSj4yX8MhY4XH6xiL2DMTjFwWb7IY1yTqkFwmLltUTjihV8WJsxhAm6eEXXINY/vsFB5/JA7Kc5wfVIdju7Xmg7ZmUae3Fi7mvFDl7HcwDfTmq6RvhbXE47Is0tChgsIG+tGISPj9pi1hshWR29qYgoowIxZiuEj2WZVyhY46cE/MI7ZZDdViSTNoDqk2oCjgT4hFKSYpPS0gD4HFpHk+q1QCbLxShWbwFPFzVko9OqzcnvzuLWJ2OGKHcrSt/6oGnXiT2o1MeWP4n/+ATY15/iZq3Pl938rl7tt112+6teIBFLl0wt3J++5x9rwQdM+s3j1OUe6YPHwXY7Qf9PFN2gJeRjHardHpZal+RSBXga+aYogodrIyrBCkKrmBKkhHJDxoCGbceLAgm9QmkcSU8tsUwUeRJzhZjq6oC8U6IYBLiHMnOZJvVzE85IE6xY7uBJbVUUwkhrMI8NYfiz3dp2R9mpci+D//93QeTn+9m9Myu0+lm/KW+dkv95tpacTi9SL+Hv+8PKl8lO+iS+RN2rXjtyy/fvHzug7+q+3I6yMAKcaQaQ7HQyWEACXDCpgd7pQUxLjWIYSEVqHgWQ5mN4RhK0mKoGJfSAfx0ruohniITr9LrWP/fQY9n+QsW5tADT+1Y/ui9DuzFRmzH7VNdq2IS6dC3Puy6vpD7L4CHWADrYoN12XnQhBUSEvI0pi1Aq6kcB6ZpCSkdFympxeq0vcVYxGRQFyGDCRFDm0eJojT4hqGwNhaHPSHOnuxIBj8xzW3ROzO9Hjl0LuHyd8SwJh4uo7mqtY4Rtuinf37h399+cHm2SRHrltLH6zdvrV+3dcv6Z3A6joa/7XcOGoiP//LN7Bff8Xz15pWzf/0gzAcbrIcdxbPcoQU8CKuqgUGWBJAlBWtBkrYqsroqTHpiXY54Z3y0KcqgSo6ueXVsidjNfIsCJ8RhqbLiBqGZ/x/6JZY+eufboEl6cffzfxq2fdvD282k+0oHbosVrMed6Xd/n3TiVL8N6W7h872btj8D8pII6tMsJyMHGh6INmJRMClEAhcLSezIPkE9shd55G3mjhbnaLRmiBICcVqsOuKG96oO2z1poYM0sFOevII8i1tzgAB98jj9qu711/GYO2Zmji4dNRy02puNhcKb/bt1xxs8i5PnPdIH5vZRh5gN/GuHClBP9HLAQjBRdKkG0EttMHiCmlBlgJpCWAEHWacTKkCpCoKL4cFPwE1MXrhMWAmTLFaIkBV6QA/8jy7hj6FbPxTIDAPASJawfONzzaNBwiBcDPTo1iXf3zELUG+X4UvL8xnAyXGGMtk+JlVdeXabHdWwnJ5awxDyE8Pn8Jk4VeZH9Xm5wDgx+0yb9h+9kp1Z02/4qwdeph/Rf1z46qEZ7QoDvYZM/vDk0F7UWrvy3Ol7N7059cHhC2f858eZD4p9J8V6pvZ58hVd5yFZmbVrDr28Y934dfH2srxuw9t5dt/d8JrjOqoaOW9yVa+7hW7TZ33z04OqLtkPvkMp7OEYVKVuRxaGsANTMAvEzGy1ZhNkiWji0Dwi/Cb3CkIjqgImeCcGxaTarXaPAjbCqm4xq6YHNQMhPP7Awj89UVenM2QfnHH6NHljycPHPgi+BjovY0jnQSNefjeYp8ZqOwHRidJlABsdmd/mLnm05k6keUJesF+IyHHjoXV1DZ3bte3SpW27zmJfnFGYl9+5c0EBm7dpLXXweY0oFnUO5EVE89pu4NEgl3VUCeEAuMmgSV1OazQP743YKEdn2iPCe0sk6Mw7u/Yquf22ZvDUEbfUMeQOsfG6hR5T/hDCRF2DRFgDCxoSiJKwiPQ6AnFDsbogLr4tAeHoElYBgcu1BEUCewcJNTe+UxUwwosWZLHabXYZPDTmErNTXUVbAVyZOQyPfYOW4kun6bx5u3bpSHb3cXg2zQquIPLddKLsaHyzYLomH3go4Aa7VEUnirkuGtMT+G8h17LqBat2EOtnbIeH1OflM7C3PWi9uouTFSzB6xKuZtpErNDJhG0yg15Qswmwd5NhmCc8DLG32Vi1OsYVMZRbawVJSjVSMPybeOuxrDwG0PQgcFtSrSCXdpZljpDMZhF1MQFNcYXEVPbeN2vrhrr7Zm9bW7c0QddxTw3Gg3TZR2cffZGcXrTowIvBbez7n98PnhD71pYNPzp0/Mt/BdHV9hbQ7kBDVNoTInYOqWC5lOib7isudDfsPHVfOZCD499qX7kid9XeHQxT/5GpJ0+xXXX0A47a4CqOl2qvxgJeTO4jch+ulrkPr5b7AFEmecweociyIXEs/f7q+s/+iI1Xv8DRjS/tfuqpZ599+qk64qU/0POPYPIncBoy6Tv0+l8/unT+3MUPmB8Hen4m54cbzQr5cSJoXBFzd0uoYHmeaOYDqJ6XsySkmjNC49j5Anfh+AG568ax4JXFgCuUmOB0x7gtZrXyRBSQA7wkZlvBH4uJ4Bhz0phxjfDLmE9mrFvq0gXq7/3bv767uruWbKlf/eSTjkHlo4fS7nJu7fAy+gH9D/PRhCtHz3i/fPOLt96+pOqq6UBjAedtciSNChKVKl68xUhkjhCTAPDdCOAty1ZZo1EdpxlYzercOBZoNJswSkrUSgNtpmRzsk5mGgloBKdGdcX5kWnY4ynQcnEyub1uk7Tt2XVbN8977+q3H3wyRx+7qC7KNH32gfPeL966cvbsxRW4HY4CketQX/vLX/D743s/E7IVgg9os6CNKmUGVq0iYBDk4gT2i6D9AqbECG/HIfgFiaN55gT8IO5K87yjVhMXzwZgVgt444hAUvhNLhNshISbBzAXDyPV6QZlbMEWfpKs5V5CC0x23dbFVZq7+KW65XZd973iSOOW6A+fDDaIfc9MnoHU3I8wHWj6tdyP66a5H7m5lCA33acVkbVI/QjTP3vnbw8MOli5cOWUJzcvKPrb8eef7fr0kln3Z41f/dpynLm5rteWth0qhgRG9Cgccnf/JVv7Li3t17N9j855fR5luCU3fUN2S73BlBYFuhkxEQWMCDtzZaiJQo0EIsFLH0CLsKi0MlT7YAepjjZHGUAkwOFXVJFwspR8HPbkFWG/k0Xsjhhia18emzCpHX11+/beo3EP+uqomSZlvsmKB5GVZb3+SRcE546r4Wu/C3RGodi3hS5jTELVXD1xfaWqNL54ya3fhdfDKs0p3qjLcCjfBFIaCimteADosifqlsfq/QdnnHpT7BssBB/hfRK4fmT94KHHz5G3USiXQgA3fibAtL+6ewaLsNVM7Ega3omy262SLdPNU0T+fJvdj/EOOvrAN2VmXdTMvxygo2H2WZ+V5uGBpNP1I6HcheyBeduw/Dejb7AsEqab2R5mCXASyn+3QW3sdge4O3aW/xZAx8gKO8WLx0wIC+wcaA8MQMWOp+25urZJZ1+h7++rmaLTRWXbTje83tmhEz0v76XnyKKu5567Mzhf7EvH0bL+hQfzyMzgir0z02rJRxwtwMsH9Oo4vcmBNnqIh1h+oRhzwrXEvs1ms4KUqikpnpvy4CN00UvYjVP+TBfhtcfo2/StYySbuOhIvCv4VfAsPkZL1fkJ6DAZ5neyk3SnBUQuGgsiSygxLSxWaywGJ6kStohVKPV47HaWiYugk60gd4yKMDnw7SC7zlTxxQGa0fvPywb0Kyjdc1t3YPeaD+70/0T+eD3lxa3WxcZXtmm5LOEegK1nVQYsj4WLFfBqeoZyP9G/mswS7mn8nAwNniVfBg+QP0wVhi5Y0HgUhWobT0iH1LWMjyOsQoWdxyNWbsUGYHYgzcXTKJVanfDXBlGNFzSoE9x1ZmnzerDv+X4nqBeH4gQrc2nm1OPvPnj/whkfHvri8mXjxJFkJanfgjtWV60iI0fjnK17V8gn6MULPqPvAhBwlVrI3Ja+FEJmJqtWzH0prQgHftN8KXZ+gq+uWiU7fmpU16Yv0MFq4RIZHYkQXvLzXQS6k1VCQZgm3hEqMDAilQ4Hi85gN+X68piDwChiGtIhKxpFBcRw+cqR87MWPvDg28enzLh3KinyXcC+E/KKfZvpO2NGkFVV1fTc5j1A3ciJOGPa/Q6tNoKUiRkgAy5Wz8aUNUS+iBRHYxwQsFDEmTsIBIatIgRValE+EfqwbAQvzHJhFz/RBUyY3+zi2pQd7IKV7oBxed78/PGPlk6tKh7bMf+h/Imr+z7Yt/9wcrokf+M9bdLbJAQKa+91p6TE8hoYugIvF0fxXgA1YwhLOiTUCOAioUYAEFGP3RGueL7E+gDmdOtWVOjvTfZevywNWNonUNQ3wJ8/DrbhEj8LHRWwWgygfxOsEDhDUK9GztmaS8di45HAfAQu64iIg5sEdqYskJobhmixc1qa1WW1Minjde7MzfLm+Z1gOzQrwtIywplJ6xefOYFXLu2/Ojt78T17dz7xzKrV3+bLp95JwdZruLHH3l2C27XSf/7ie2e6qrWqzO+UToOM9AmUqiaaYAeWRAH2EpHZXhaquEnTqdVZoRyKBLaDuRTMfmh2NREn6iPtKhgOVh4A3/Lyu4Mx8YP8kPrj9d23rJk7D9fR4T37CQnXr7998uT/SKcrFw/840p6cf7H45e137Kq4w+X5+NuB9Szr7lYFl3iDuAtr4aM0rP+ChRvMRJALs5lt4qsyM8Ami0QayOYCRMQcUfoZN4llKa1bZ/OFjS3wFfgYqa3wKWAL6S4FB8zzYqvIL0gwmE90m/pqKXVC+dPXH7nkn79Fo1cPnH+kvFLRy7ut2D7jBmP75g2Yzs5/8DklSMX33bb4pGPTJ216C54sw/8vGzyH+/b8cTUaTt3cZsMvGVnojEsYkQ8r6zHWAfIYjJcwYKM+am+Qc3kc5626N2AoIPl9Bkbw5XH7C/2k7547j46Gsv0NdydvraLvo67wReiEw6Qy+RCI51fP5/+gI3wTWA1JiRCnyioq6pRnKF4zMybYAar+jkhYGeCodZDhV+tOhjK4Pt5XMg1DdM1jT9TU+NPqj87t+lnoVZORNmoOyoL2I0wfdf8tDYmQRZYmbAgaAVCzpCLZC4JuU3cEeCAIa4ZEfFq1cG0jt62fO3CuRA1Fam6rDx9oqhhdIvjNQILmoy1ksAVQ2/vV3n2jVFPjMkbu3TAjBlztxytLe+3+Z/v/e2h218tX7Kq093TVy8pXvfwM9nL179YOkTIGLrc23bKkDkrEn2LfAldA90qC0o3Th6+KmPwupWbe27wZg3o06FLl8zc4dPGDpzU3V42peK+Qvt4xuc8ySxUS+f5WXFMwM6Sy6iCGUEHDp8RRwT+3sj8Q16eN72gIN2bh+fleb0FBV5vnjQ1t0OH3Jzs7BztOzvdGN/0jVzK62LSUT66PdC/VW0MEnRYPR3GrJwHjJaihMoQtWDIybZvbk77dmmpbeI1DQvun55rWFWZOLVqKtVTxbw8Q7Q5HUT0pKYBd22iPyfNFiqwEubNWzp/0dB5nSb0fvXdj19+aFaXuxs3nsKj3mJfr9Kd756lO1+btA9n7d2H2z23j17Yv49+8Jzo2bdl986sPzrafPfhuR+7zfLTI/wZuvOtN2jdu2fx8JN/ouf/tA9n7NceY3LWW3CQqdKbQH8yeliVZbMF5AuMCdNgoG8TIl+ATVasJazahLJ1SFJAXUi8CENN3zAB9CgwmBWtVDePQ61GQSiBUFKb+FiWtQJOpOpgRX1aX0wRZkk5rQjP6nX4c/ixBnHEjlx555QtT8/cML46c9rihctp2X2nx953l+CuHDNu4sRJsuhb7B/dedIc2u3Y+IZsUSxkdJajmUK9cBzJ4L/oDhp14MO3zbRjlx4r2Kt+K8dDZ7NjtNm0Hq/Aw2bST3DyTLqb+PDGXnQnfbIP3pjQ/COzw1gHdtgFRKUF3Px4iSl8TAYyZ5B5FIR3FIQqPfiZeZ6YsbzxlOAKfocXP9LUFOpHsHjQv9nPdhmxBpOb9SawXhws2oQEVmQZlludJrmlgZ5Gtt+Lmf7WCURXjdQ8Du8CA4+gCun1PPnKW5iUCpBjp1Kam5Oe7nVbrc60FENsZri+QQI9YPcIvOohJ+a35JWcv7tszNA/1OADQwZe3X126QYs1W29fvk3JZZcqwj077msbG4SnYnH0seFuQvpG78ps7jpmgx+srz+N84FRfn6D6JRdszn+nS+MJJcBX6x3AvzU1qaNYcQSrzYc0NFnyCFkXZs/sGFCw+Vr6tceHD6zMFl06eWV0wX1y88dHBR5ZrKhkXl06ZWVE6dBlPBmoLEs14YSyr6dzW2pYNnhKwK+gqh4AxWlxjZF8Mq1kkVrIoyWIcVxaSUmkwmi8liU62W3p3Jz1T7kVXgV8SyKhITDnVPmLl/rh0PhzopcGmM2j8BLCho3T8hR2M3WRXc9U1VubusKHdsr+JFmx5eX7V+Hz5C+lV/MXTMgPzSXuk5w6YsvK9s4yNPM366SSGH7UHdA12SYEsnYtaQSYQogAjWFkvFfAVgXzc3T2g7PJrw5gmPT2ueCFkWnm/PxDc0TpDp5aX7j+zeNGP93PNfzpx917iykuL7OvcqWjlq2Xbxy7K7XB13PrSm89zSnWsnDyrtXprpGZ6VP7vVmS9wlPkGoTNf7eCZd0Dqed4BxrMzX95axZrOQDUrHry805SUt9rvC0wRv++xv6ez5/FclneBeKwUYqIYND58vKZG16w+ZjCLjyLj73BCqe3NhrUIxMP5JIPVboXtZ3frXJnY0hyIMz6FyquG0j1P1C2LN0z5vHaYwVBXh+fRay+9x4PxZ0ZW7qCvyLlqXDCVNoqjwS+JRmUHWUME1jLZsSF8mO2chFhPHzvzY45yYsu3RHhvEKssHMzCh15Vh8B759WLvJ/EY+Wq2ZLvd4ujp70/vUflh2f+9g+SRRvl/b+UCX7btetYpCoum8gAvFwYzmx3wMF12SAtPkK4D2+askc2TfXorBbCk/PFvXoV8+J3jLzUgRvgKRvTr1YzeD5E0noDMKnhpwO8hqYUbAercglLF/dUCvxKDG7YXOvPyuo6aPCge4bdW9BAL4+p0dfo2uZn5lkPTPcBjAGkH/4kVD8DnjevB4no5wuHg6Eup/I1i+WY/KUl0qHGDaRkxPCYrDtHgzYpBnt2HOwo81VU/QL/7giXtwlhhyWi7VOKUC/k+J7adc/u3rjp6SAdUT1p5MhJd40Up+w6fPTJpw4e2vkA/Jk7axZMWQZ2bHdrO6bHPj0uwJL6TdhNd8/EyfSTmXgYndn8MzHSSX3wnXhULzopoflHVWf1Bzt1TjrEdFZ/ZPGhz8gihCzp4Icy3fWFkBg8jCoCZdHmKEHBSVgA3oo6oThKT3SswlonDufFicSgVjkpChosqaU6CQkYJ6QlpLEABzxyCHFYRyK3ZUawZQC7A8C+pMLuoMIuUmF/ymDPEG4H2AMD/RhsmRXiSzaMFKE4CQjuCYyQBFkaqULXNUMXVegYt/W5kxPiIqDqmQVlcMHGym7pNIMbz+DN/EWFfhmpNN8DcD/SDmqTQfUB2SIzrwVGgqOiQFA6Yp1RBgeJvavc4t0qdQJ2LgHyhEZEm4jRgHWyUTeca0yzQlrozBIUFSUPZpl2E9MlhTd5DuRQ4VWZrfVti2erAsm5uV5vbvfc7v6c7E4dstpntsvwtvW2tXOl3NYCtoUgP50tVEL8nojasqoGM3i54OmpfRMCcxyq1ZB+WLhKhSX2fF53MjySmNaONU2wqIF39bIsgkuV7x6koAfujiNCfnJ7/zFzusc9/EBZ7diub5w8/qEnUJU/oWfDnK49igtYFsD/wIbKmv4DO4+fmt5p6eij9b0nVZV3HDbjD8k4c2mv4kCfAFszXuOvTGdrBoohHaXxtSppehWXa91qkmzE4JMUg9Mp1yCJSDW6Fi0AitYCoIf4ETSsRR+tj4ZAUme12vSJmbeGMQstVmE4LWC0NRiSLNUgmci/AiMu1mH/vTBOIg/A6BMoTXBZhBZAFMyg6FtA0UVASUpsBccQCadjKzinQNOXs/1s5D0SIFfALR246HrFRCSil0YYW0CK0iABtwCWNy01JTEhLjbGYfFZfREwzZEwva1g1qACgFkcKFJhKljSAU2SPMLQApQ+gqj0tJSk+FiHLQJEFIBohiG3gvEmKkQL+F0PsF1Bj8tIIbIShsAbkSt0+PfCQaQJzJm8AOxDNLKjgkAueGyyTpKrwO/Gio4dn2klSprW0cpl7bZQ3sECbjVrJhA8ONxQwEo25AVBW5Be2U+OCqeDx0hi8AopvV6Jj8/iTSZajwH+KzkItM4DP+SsdIbROo/Rms5pfYCwbMTKQBwrQIrRA2QDlnFGepIgkUQgXwLr7wDNw6ycLOPRSkSXhyyTSsTPtbTcq1XkfguMRFiu/q2hVQEH4Xq1TbzDZooCC+AiLl18JuDK695Brlk8c5qvTFtthx6G1e8a6BxjhRWBpVFkBXYOlmt+tSHDobFRl6rp7FvNPwsVqfPbTESR2dIrMLWCld+Y3/E75z/ZdB3m7xHoGuswCWEA4LP/NgQNhN7TzB/vDfPXIBvMDxIGm4IFozKrC/yVaT3utDSPOrHBo+HO0pyrVZ0isZnbq5ynf4SZswLtzHot+06EGtD8pIa3JYhq7QLGUQbNPkoh+/gCbIBuGi9eQM0zzmpqp85okLQ8uJpsFVrNaDbpdeEZm5qa3oT5JkXoPHW2k/TPMFt2oIPFKAnh6UQcnk8KzWe3hWeUE5ppJp1VXRNBcw19F+aEaElkLiNr/xvBp5JDUyXEsXp2vcInU/hkBHUEndKX18S7WJZfYjlbVlxXFapLUTd5qEPJavXAVyr33FmXEkvK8k4lrdXHnyL15b1Kc7dswQfwYPzHg8E3v8Rz6eLjZKzarUSW15KNdDndRQLBxlpqCMugVAQ6u1lGsjWtvR6o6hUojsJ62D96EBGDzkhkwcBbnW/WrmG3+dLTUpMTE+JjY2wZ9gzevmE1pXA9zXsx1PXgeiVHlRc0R12P1ETClagkgo0DPtXctFnDYQPP1iOnaOtx8zlnYYM6Z1IckXijgQTTSVi6xZzO3zHnSdQAc+YGstNS4oTwpDK+9azqtIo7Mzxnx1ZznkJbYM6egR6wBYFuRRqBdLKBiFgnjmg5Z6U6Z3xch6zMDGCxOzkxrlN8JwYgKqkF3t5WMGrQ+wCDd/aIYARBU4+4VRcM++NhqgmD72ERjeImiC/kF0TWl8tzVprjL+wdTXfgUaPp43TXBDyK7hiPx4ibxsGvO8bhO+mT4/BoPHoC3Q57dF7Ta9Jx6UewZgkQ04O+tEaDUkhOcsXYdDIRFAkUg1BswTggswONmyGW3q5tqD1HTffzFjN+DCBYfDiGXz/hYt2gWqvO1xsm3H/3Adascm77mPtrXmm4Nzh6+v7//Nzom0C21+wOde2MXzVoy0l8D2tZGbW0bMNb9DFs2dI4oJz1rdBrW4TnbmP9O8BX3leh6aWjfH901iS3N3A2L5DTxtVKcuXm5gtt4VS/DNuwzWNPU1S/5VbzzkJn1Xlj7a2k92bztklwxfy+eU+ieTBvYSA/Kd7eWoKVm8zsTomYWxc5t/eGuWvQJpib3xwRkjLdDR0oGGf4PO7EBJdTmxScU9W3EudG6OluGheyYcZOgaxodiKniKzzjpXsCexCqWqthYJFYeoxm8Nhs9mk5Eyt12iO1muUHvCAFEqiAFKvOmWRJ3ewj0Cj8nSR2nPlBy0qNrddzaHn6CdH6usv4Dgc03ht96dvHX/rL4LlylV6QjrUhM4G/7lm12OP8JqZpm/EbXIy6sSy2rxmxoYxKw9Wa2bUXwS1Zoa9HQNYqGkRtehEUEuFE9ToLT7iXV6O3lyXwoJ91p8ikhGt36qqOuSxdUzrwHvIFPUeCVeoRpWfr/AKloim8xReEceiJTFuzbDOCX1uXznx1ReOTS7a1vdCxb3zR/fq0z+wbD79pu6jv7/zifj90um9S9wp7Qr9d26fsGNPry2+jof6T+5dPreyqCavcHhe2ZDL1weIBw78ebuqj3gPg+xm6zqKrSu7NEBGJasReioQ3bEd0entEMPatNOlOB616vSiXidW39CzoNebS5obF6J4Dbgh3OOQj/Q6fQ3SibqaW3c7GHi3Q1SLboe4NgkYt8/0pUPM7m3jxfE4nnU82IzJmb9Gw6xMTkNmOtHrbqQBCAAybqRBp/t1GnR6QB/o/7/Q0LFDu4z/HQ0nP0Rod8CSnZUu3EhEvgH/PiqMHJWoMBUFnAp4+lfJiOJkGFuQkcDIyPXfhBBTcmYzHdLHreg49RJCLwUMIQpCHTLdw9CNBgvRiUbdiAhhiqow4agoRwQh0RwZc5iQnv/L582cpOgWJHl96Yyorl065/uzO3Zo3y69u697K+KsyZnNtJ1vRVvNp0zObqAttzVn2Y2CugoIq3WRKJk4MsYwSZ1/32NGTompBSWJjI5uXfJzO3Vo17YVBeaWcra4FQ1v/oTQnhtpaEZGjwxResMIcDV0YlQEl29JR7ff/+j/jRawGECLzOoFM1Au6oqWBBzs/MFqIAinJYFRTsayxO7MyGJZ+ih2eCtKSBxhZOlIUsHO49TuKrlCCd30lwFbG+kHRbFqaBhbdeuxVQFnfl5h57yu+V392Wker9sOblmqKZalsItwZPeVi5e2JmK3zR2q/PNFnCTxA0ystpSM+pkemTH70ady+705duHTGXl77n3lH0GA22XE9sqhj02gl+cNfmPJUy/umzxs7e7Hj+4SXpqzIoooD+GOT76gUzu2MvLuuHPYaPrfv0+mMz2+Denur+bX1G+8s+rZLeMU3T0kp+7xbbu5HEynDta3xORgOpOD3mr+APdW8wcGTLAzCiQhBssCzx+gUP7AyfMHApFlYXREh5Mso0pFrf1tmT+AkURg8vzrQ1vkD6JNkfkD3tOj6kaOa1/VRh1EaHUg2ptCZAXMNrHxnqvQBY8yBOEyrg51/UCMxrrNWrb+6DR7zrpZwpkH7YFbjOUVzQSnuhPbxMU6E2ISWFUz4OogDsOv4Drrdo6ruw1R5BtxBUQB3WZcZflXcW3OYvwuXL1pKcn/G1xPgi5YH7D4PG2EG5HN0uGbYWtohYE+hG27yJSI9sQtBlcF2jB022XcDOGoeE1/cZxVHRyBcw1EyUMDFSFsm7u9IB5S1P4DBxi0lnANKlwGNKcT6P20lKTWYI2MT01/A+dxnMon7vu+quVQWNql7yEthaLyJxbAAhK4Wm0Uae6JSQg4tSSLekwVer3qIEsFiUlsPZ5FiOQAHOa3PxsBaVZTCoekpVZCkPg9r6HeoJaQ1OQLFlpBcnBIjI9NLI8z5AaaTlJWXT7gcCjtooKKE/EtYcWEMjOtgB3iOS7w9QHWNwArWV2zCFg1lN0ymRpIZhkK9bi0ZRcRT2apAT9hd5aKY2U3v8spO9DByIoqilnFvcPO6qsNOMCKcQRMwhfrsGv6PClJaamszIK1TiB+75erIN0nQYDqK4jh4alNYD62WvEsjqXXf/o7/QorH306Xlf4Nv3utq9Gjuq1beLVAWc3Pb27YRt97rmdzz1J/PRL+lds+vQLLM8RP3h5692Le2TPvK3/I5Nnr6HT6D/W19NNzxw+zeWV952AP8TWdD+nfrDqEZkRWqOyt1O418SgM4GyZJkbRdFXMMvvKIloTzGG21NyWz4itnqkZc+KMdyzkgiRIs5s1zbd405OahMfk+XKAnPrZHXfaWbVx5mu8jm8v8pVXfseQqMOeZJ4CB26HRZ+hgCVHWibS27at8JKz29IFLUYUXWYpYq8LK1zS9izhjLYyfE8zA7Bhp9FtR/pV2DfkFBqDdv5W7BPwiqNO+x1xwvNwD3sAqffhJ5yk8xTS/ABvUNFQHFH6Laxqu8cgcOp7xA7YYpIOkmSUsEUm+MWwOOZG+vzelJTkuKzE7IZjHD+SYVxvhWMmgTE6svV+/RkzCvZbt6JxP54ef6J1TI0ihnibt4LmMByIRD0srLnEQhMafONb4Oaq40RYkcqZqNO5hcOKNqFby0uwmv+uVo4EdzXNT+3S1d/bvfQd7LqkUfoP7p1L+pSGCgiP2g/cH9wetMxcay4tEUei4TyWALrJGeGLJzHatmd3TqP5QnnsbRyVsnB81eylMvzWfm862rLfZXTRi3AxqtbZlTOHDO/8aU8fK7/tKfqyEY/7dhn6lPPqj1YvWbevmgTRqwNa8DsAYs3/bJnIlmS98F7u8YH5/kvsXXhPR2a7t/F9cQwdee9jtD9TPrZAUnoXoGQcxMqKWHOja659aMy3PoR6dhog3WtmkAqQ00gVi1NFVIIeqa7b4nXrKkMr4QYfrASwivkGoTwYq6B7mYtKZFOzC3wqojAKzkpPu534nXShtC8w+7EGKEZsYxmpyUSM/3NONbCYdFG62/FMjvGaZ4WyBlCcRbHD/ZZa/xq3AjNVNHyheePcFIMN2NX5q+NvIFhToxVbyY+NoxXVAgv3k+i6jnCsBqpriZwbVrA0sZFZCnGCmrGwiq6NPalI1lCksyLNnnPCdf4NzSeJAQ82tE3gqW9VXsKU3sxrEfFqvBaA96HyHueU9E8tTSpAwRaMquBq1bvfUBaoxlvrzI3X/tQEoLbPvwAy7opYujCCBS+gyJiOPAn1oVQcqIrNTaV1ZhaHda0VIODVX6FrlTyeVwxoQ5pr9aQmGLNTcfDiutrLnz73Qd/f8Ao6urqZNx39yaypR533CA0VA2i79H/st3+ZOrAIpqnQ7RD7vCEI6czvnwT779wLoL/qv6N4H8NxDI9Al3DrG/u79Hx9kqmmB03spzrYsZLvScz1B8ssrtR1PvHQ55Ui1ZgeF+7c0v+UnobOVAyCgYMSWDTEmGXEC3AY9d1y4okVxl0hN1LJXDvjIkfqtCu20lQ794vaDWSN4PznyPOwNVnSpBeH/m4miYJPY51uuZr7H7jWTA1v/UYPxNlTzXfQ1EVsMU428Q7k2OS+SFpmtsS5crMwHLoSjxX83F86GZDa7r8ZRBNHElfpE/gkTgwcbhgDr5AfMGLZGBj6S+0CeOf7rvjDgdejmtwNV7iUk/pxe30LL3IbthyizOT1Dv3lWRxOEpC7VA2q7IwY72SQMCpacMrqpEeDKderIrSEXafhppxAPdBAqINhNum9pkYdeyQmd0+m2Xo4/idykaDLKIknGSMzpTUtELoNpHWWYfIhmGvmm6QaumleT3OLb9Mf8Dy14ve7tHllQdPXw36dLjfqCfvGLbpunvTM09t3vp03WNin/lrjSTlYcfXM2bjbKzDetx+9rT75tCfPp1EZ3t8G3wppPD8pffPffz+hx/uevzxXaF6DHFuRM2HemZQ0/Sdeg6rHhfwy4lZuZh2p0U47Q/uv81mU50jjI3iRWEf+P9mlH04SlGvaFIlyMhLocMVnQZe0cm678dUNTgcdlYEyGo6k7DL6WHXeHqwcf5EV13MPfNmD1u8rGK6+O2yhzLaLV7oKli0JJf3ckwBWFnyYl635w4kNQcW4GQM0pwFAfW5SfWekhtuFsoXsuZOnHD/nPET/nh3sd9f3K0wt6d0YOzsmWPHTp89qnP37p3hi9dFgk93UfhW+higAVeQCCEA7P07wh3KDtZG6nSwC7T1OlhuF3axK7RTOxBWJuthC51EWJmV34wFR2b5oNtSc/zmsabpIzoMvb1vSnan6HGmqeIlb5a3W/c5y+Fb1x5zlre89x61vNr+/+O9cgzr9Dvv0i8XRobv0ofnlN/9nNL8XJxwApfyu6mSAglYu+usuVQT/leQwj4FwO7LZx/AoHiNMWM7zy32CidihmMjSDC62Tzhe7ibP/lBmwfza5RtBTiP3Z38QO8U6VAMuzQZJ37IZF77bAIm80XILiP2uQBpATe5xacUwE96rHeyAn7CcCANHIdoJgms2YRtD/5hN2oZkVG7pTZaF20yMozs7CNvfPl2PyDl1ijEtdGdH5ib1NUUvNBM6wWsm04vj4mgORKeRYOnItniI3Y4PIvOosJzyM0ccIc/VyHECuIzx47uPC++2NTMlJgR9McHxuPcP6j2L47+QhpU/sQxbfAxQtc7szNJA+HQOVzGpAiygU+sIktv1VvNJlaL5WSlHhjtwj+QUcKV31lPvGvq3ZNmTK+ZPI18M+fBP86dvWARw2dT0xVpH/qe+2mbeIWpFbGKTyuxcW1lFb0wZ3zAJbLSKlzBLwtWL2bUyhFYn+xMkszrb+UXJIzDPSQRzSORDSNc3srgmcTIZ7Rje5J48+JceIbOgUVb8zs+L2JN45w1wtLmz4twkH6wzseQAax9p0CWGYsC++wAUgwuU/izA1itCDOe/GMDHFHqpwawZnCP05OnfWpAXm6BWYjGpGHeWnov3rRiHv1elmKTkqJ2C4633ppASoJvv7akv9GXlWX9kfU+gg+8XDqD0tEfAsPTMIrywhJZwNOw8sy8gck+06xVOiwoaq+eEdY+qgLxYxuWbVc/TwcjlrBgKVdXDEsOs3sk2OeXoHScbgJVmMI+gUlxMvOtXtIE640ET57fl1cA/9uQei1xAj0xCNPgDxD4naU9vWfqt21et+df9HKHuk2EbNrlw2n/Ov10nzopj96zoDCQtaDhjeLanuzqygXt2pctgFA+bcyKocBPL/lcWsz3TSyLzC38cxj0rLcWOFmtXgoJrhQvkjQAorAG5ervbDvpYnVABrDXxgoCLcZYXkgPas2DBY+LeZ0JvLa8wCPgqXPPkeTzZNhpnTz7z6+faZgv686Tz8ldNTXBzaTjAlpOzgXPkY7BjnjX8uB5tQ94Fe8/+j16FOJrFk2H+/b7iOwGKGQF/0fsc/1F9sX3bCK810ZOZns2ke2Q2/jOGEDY7YBzAzbYkgJ4MyJ44XoFnBlRu8jK13yRFWJnWIgdYanXVjF/NjpUeZWgXsCsBwfgV8dVBUxpqR52R0GaVuGm4Wa8AbdBHLdd6tlPtAkE3qwDV0vRmtfYK0rEK1o9diJHWMGtrt4KoanOlsQL8AZFvK/wGwjCxCTf7IYuCEJCI6pYh4dGhC41U6VBPh+mwYduI/tRBJ+le4CWjYGYVBMxCx4ziRYTAfXoNgpp5nUWS6oLFXqOO4qONlcgs9lRwsp3GBKmKMLQMCqhBCIoCgEeiWY8/63RVYG4wgJ/tlosnp7mZs6zir4lvAbya7fEfxDH/7TKPTsQIQqeRLNOBM9XIep68FdNLV/V1sSjEmZin+VlDmMYhVuQo53xGbgOGdRyEH8SD24m3auSzlao5UhkMjWPY3eK3YRis0e7P7z5j3JAwlKv0aXZCP0//LQtWgB42mNgZGBgYJScVffl+eJ4fpuvDPIcDCBw4UnJJhj9r/KfAPs69mIgl4OBCSQKALKBDuoAeNpjYGRgYC/++4KBgYPhX+W/avZ1DEARFPABAJjjBv942m2TT2RcURTGv3fvfX9UFlVDpFExIrIIjTFmEWMMFWlpFzEqqxoVo6ZDjDGiIp4uahZZRoissoiodvcI1VZkUzFmUTVilGpXXUSJqKouRuT1OzczNY0sfr53z73nvnPPd686wWwAwCQAJYxjS2fQcKeQNht44W2i7H5GzTlEQxVRIDlTwQLnys4f5NUGHqokttRPJBh7QvZJiRTJFGmQ5d64TCp2fRL53viZqK5i1E9hxb0OuNNouUMI3Q5apk6SHB9xfIyWypLx+LH5wfgkWv4MWl5AsghNu6e/OFdCxSzhBvPemw+AX8ao2UZgVnnWdZ5jBy9Z8zA1bRaQ0pvxmdl21vi/ojlGpD+hTq2bEHX1BrfMIib5z0h52FFevG7S9jvya4gkbjp2fSQ5epb5bZ7zCGOc2zUK8GYwbFLcI4DSByjogH0sO6fUe3L+fu/5fUCkN6tkTNbw/KusLeO9Qkl1MKe7KNgc9l5iBnFXL+G5jTWRIkl7lt+I3Bxq0m+njQnGH2jgDvPnvRzuk9vkJnuftn2/Au8sPhcvrA8D0AeX7Kls3JRvt4npvg+XkTsgKl4MYr34zv267Jv0/Qq8byhaL8L/oQdf2P/X1D1yYg5R++fDZeSeiYoXg9AL6xnVermI0F/jPlLXvjPEHlapgX7H+1MH+qo473wluQtwSg2pTzkn76GHAQp8WwXnEUYs8l4+YkTQOaKw683RG+aqKu9kFfPOtfMV2ZteJcxb5L0MJmz9d6Um3kPiL1/A2vEXhDHf4gAAAHjaY2Bg0IHCCIYGhgeMcUxMTJOY1jFdYfrFbMacxNzFvIz5GPMjFgUWF5YWlnusMqw5rCdY37EFsW1ge8QuxW7EHsdexn6Oo4JjCycbpw9nC+cGzmtcalx+XGlcU7j2cd3hluD24u7hPsDDxxPAs4DnAM8nXineGN4u3g28d3j/8UnxWfEl8FXxLeNX4F/B/0agQOCAoIDgOSEFIR+hCUL3hP4J1wifEGERKRN5IuonOkf0kZiBWIzYCrEb4mLiTuIt4svE30gYAWGMxA5JLsk0yTWS96QSpCZJC0m3SW+QviT9TiZHpk1mj8wTWSXZEtkFsj/k7OQS5CbJbZN7JS8mHyI/QUFIIUthjsI5RSZFG8U8xUWKT5SslLKUZimdUvqhrKDsoVykPEP5joqAipVKikqfygGVZ6pcqjmqM1SPqH5Sk1FzUatT51DPUj+jYaAxT5NBM02LRStKa4M2m3aC9iztczosOlY6MTo7dAV0/XR7dM/pMeip6eXpndGX0E/R32LAYOBhcMDgnSGL4R4jM6MYo0lGp4zZcEARYyVjA2Mf4wzjHuMNxudM2EwcTNJMOkxWAeERk3smv0x+mZqZLjL9YMZjJmWWYXbKXMt8ifkGAHL7iWoAAQAAAPAAQgAFAD4ABQACAHoAhwBuAAABNAD9AAQAAXjanVO7ThtRED3rBQIKIJQCoYhiRUUBy4IUFCEUiYdBQRaRAEFDs6yNMfgB67USqPMFfAMN/8AHAJFSpaGh4gP4BM7MjgFjp0HWXJ87d86587gL4BPu4cLp6gMQ01Ls0B8bzmAQvw27mMe54S6M4a/hbozg0XAPhpxewx9w4YwY7sW4c2X4I746D4b7sZsZNTxA/MvwILYy/wxfY9gdN3yDwP1m+JYJVw3/QX8T37n47J5hGSUUaQntDAXk4dFC7kOiCDUc45R1StQBvR4uaTMIME2bNDSNCXpXGV1jXJk6HpaIY7JlDVW/hip8/KCvQORhk/4q6tjgvogGeSFjF+iJNCLPNWbcJK2d5WGRnBJZkrNkE3SMalXfVs26ZSM8X7lNZpPXSamkq/Ql0Zokv4qqHtFXw35bD0KtwtOoU/7vqTfWjEQt0WzSrpf0tkg90v10f8jMY43Nc42e+1hn3u2d6txzmVtC7xym+PupP5/nrezIuL6iCiPfy0tY67FWVdBOFxmbdt1XzQq7k9NqClpJWn/jVR0J46RTC9QJGZfuWjny4t5Oc4Y3BP/N+0XL15yLPC23aNbpyeE7+5jFOief1Rcumjs83eOE5Z7E3k2ALWYtma3ppNPvQs5mebe8qnRtfi9fdK4yzwZvWnnW2sSJvuRY30L5CWKgs5p42m3QVWzTcRDA8e9tXdt17i64Q/tvu254u624uzPYKjC20VFg2CC4BkICTxDsBQiuQR+A4BacBJ5xeABeoWt/vHEvn9wld7k7ogjHHw8e/hc/QKIkmmh0xKDHgJFYTMQRTwKJJJFMCqmkkU4GmWSRTQ655JFPAYUUUUwrWtOGtrSjPR3oSCc604WudKM7PTBjQcOKDTslOCiljJ70ojd96Es/+uPERTkVVOJmAAMZxGCGMJRhDGcEIxnFaMYwlnGMZwITmcRkpjCVaUxnBjOpEh0HWcNarrKLD6xjG5vZw2EOSQybeMtqdopeDGxlNxu4wXsxspcj/OInvznAMe5ym+PMYjbbqeY+NdzhHo95wEMe8TH0vWc84Skn8IZ+toOXPOcFPj7zlY3Mwc9c5lFLHfuoZz4NBGgkyAIWsohPLGYJTSxlOcu4yH6aWcFKVvGFb1ziFSc5xWVe8443EismiZN4SZBESZJkSZFUSZN0yZBMTnOG81zgJmc5xy3Wc1SyuMZ1rki25EguW/gueZIvBVIoRVKs99Y2NfgshmCd32w2V0R0mpUqd2lKq7KsRS3UoLQoNaVVaVPalSVKh7JU+W+eM6JFzbVYTB6/Nxioqa5q9EVKmjuiXemw6SqDgfpwYneXt+h2RfYJqSmtSpsxfK6mWf8CBq2luAAAS7gAyFJYsQEBjlm5CAAIAGMgsAEjRLADI3CwF0UgIEu4AA5RS7AGU1pYsDQbsChZYGYgilVYsAIlYbABRWMjYrACI0SyCwEGKrIMBgYqshQGBipZsgQoCUVSRLIMCAcqsQYBRLEkAYhRWLBAiFixBgNEsSYBiFFYuAQAiFixBgFEWVlZWbgB/4WwBI2xBQBEAAFUvsQyAAA=) format('woff');
  font-weight: 400;
  font-style: normal;
}
@font-face {
  font-family: 'Open Sans';
  src: url(data:application/x-font-woff;charset=utf-8;base64,d09GRgABAAAAAFeoABMAAAAAlkQAAQAAAAAAAAAAAAAAAAAAAAAAAAAAAABGRlRNAAABqAAAABwAAAAcavCZq0dERUYAAAHEAAAAHQAAAB4AJwD1R1BPUwAAAeQAAASjAAAJni1yF0JHU1VCAAAGiAAAAIEAAACooGKInk9TLzIAAAcMAAAAYAAAAGCh3ZrDY21hcAAAB2wAAAGcAAACAv1rbL5jdnQgAAAJCAAAADIAAAA8K3MG4GZwZ20AAAk8AAAE+gAACZGLC3pBZ2FzcAAADjgAAAAIAAAACAAAABBnbHlmAAAOQAAAQG0AAHBIDuDVH2hlYWQAAE6wAAAANAAAADYHgk2EaGhlYQAATuQAAAAgAAAAJA37BfVobXR4AABPBAAAAjgAAAO8MaBM1GxvY2EAAFE8AAAB1QAAAeB9N5qybWF4cAAAUxQAAAAgAAAAIAMhAjxuYW1lAABTNAAAAeQAAARWRvKTBXBvc3QAAFUYAAAB+AAAAvgEbWOAcHJlcAAAVxAAAACQAAAAkPNEIux3ZWJmAABXoAAAAAYAAAAGxDVUvgAAAAEAAAAA0MoNVwAAAADJQhegAAAAANDkdLN42mNgZGBg4AFiMSBmYmAEwndAzALmMQAADdgBHQAAAHjarZZNTJRHGMf/uyzuFm2RtmnTj2hjKKE0tikxAbboiQCljdUF7Npiaz9MDxoTSWPSkHhAV9NDE9NYasYPGtRFUfZgEAl+tUEuHnodAoVTjxNOpgdjuv3NwKJ2K22T5skv8zLvM8/Hf+YdVhFJZerQZ4o1Nb/XoRc//7p7j6q+7N61W7V7Pv1qrzYpho/yeXnff/Mc2b2re68S/ikQUzSMCUUS3cFzp+7oTuRopC9yF+5F09EsTEXnotmS1dF0yQEYif0Sux+7H82Wzq/4LXI0/ly8Op6CL3jaD/7v6vhP8VQimUjG9yeSxLv3wIiWhQVLP2zEDVY6X3IgxClY9aOW2AlJT3SqdJ5K74aq+wJvqTK/T3V6TQ2QhEY9q6Z8Ts35jFqgFdryE9oCWyHF3+2MHYydjNsgDb3EOQiHIAOH4Qj0E28A3zPEPAvnIAuDcB4u8G4ILsIlGIYRuAKjcBXGYByukec63ICbcJu5SeJHtF5jel5VeaMaqIUNUEf++rxVA35JaIRvmD8G30Mf/ADHwcAJfE/CKTgN/fhPMD/JGCFajhylxCyDKt7XwPpIGfks+WzI14BXEhZyWXJZcllyWXJZcllyFWLbEHuadbPwjMpZWQGVIdoE0RzRnN7m70bGjdDL80E4BBk4DEdCREc0pxnWz8GqpRoL9S1Xj6/F69jDunJqqoB1nAdfyeMyzuAzBy+hSheqdBVlrIN6ampgTIYeJpat4gS+J+EUnIZ+/BdUmkClLlTq0pMq/+N3VUAle+OVWVDFUKOhRkONhhoNNRrN4DcHzaGr1UHfQmf7iutlvokczbxrgVZogy1E2gopntsZOxg7GbcRK824nbUfwkfQBTvI87gvYrn+B3h/hvxn4RxkYRDOwwXeDcFFuATDMAJXYBSuwhiMwzVqug434CbcWtzh27yz1DYFhd1biTIWVSyKeB0dVTuqdlTtqNpRtT9VFm92EG+Dt1nUMIeGDg0dGjo0dOhn0c+in0U/i34O/Rz6OfSz6OfQz6KfQz+Hfj5rjqw5subImiNrjqw5tHJo5dDKoZVDK4dWDq0cWlm0smhl0cqilUUri1YWrSxaWbSyaGXRyqKVRSuLVhatLFpZtLJo5dDKoZVDK4dODp386TZ0bLTxL99DpujUNOHVDC3QCm3MPbgvzeJ9aRbvy1y4L3eE7ypD1xm6ztB1hq4zdJ35hxNi6NrQtaFrQ9eGrg1dG7o2dG3o2tC1oWtD14auDV0bujZ0bejaFN2lC6fDLJ2KVUX7utxeeM1i3AKOW8DxpTq+VJ6XZoq/DxfOZMGTtWhbBtMwC36mh5keZnqY6dHTj5wqf5I6gh7/bbf9zq4hdorYqb89qw9H/j/Ol884Ta5ZeGIpc+GmXxd6ToVb23v4m9sradHN62PRx/LLYy0rS8OvnJXc0+WqUIkqWbtCb+hNdqtWG/QU99cm3jRx272gVr2jl/UutkabsbXaona9ok6sUh9gr2q7uLP1MVajXn2r1/UdVqdjOq56Gf3I6R/QIBGHNKw2XcY2a0Sjep//uGPUO46165Z+5tcXp4iok1haVr8SfQ775E+Ohly2AHjaY2BkYGDgYohiyGBgcXHzCWGQSq4symFQSS9KzWbQy0ksyWOwYGABqmH4/x9IYGMJMDD5+vsoMAgE+fsCSbAoyFTGnMz0RAYOEAuMWcB6GIEijAx6YJoFaLMQgxSDAsNbBmYGTwZ/hjdg2ofhNQMTkPcKSPoAVTIyeAIAohEaFQAAAAADBHsCvAAFAAQFmgUzAAABHwWaBTMAAAPRAGYB/AgCAgsIBgMFBAICBOAAAu9AACBbAAAAKAAAAAAxQVNDACAADfsEBmb+ZgAAB3MCFCAAAZ8AAAAABF4FtgAAACAAA3jaY2BgYGaAYBkGRgYQ+APkMYL5LAwPgLQJgwKQJQJk8TLUMfxnNGQMZqxgOsZ0i+mOApeCiIKUgpyCkoKagr6ClYKLQrxCicIaRSXVP79Z/v8Hm8cL1L8AqCsIrotBQUBBQkEGqssSRRcjUBfj/6//H/8/9H/i/8L/vv8Y/r79++bByQdHHhx8cODB3ge7Hmx6sPLBggdtD4oeWN8/dust60uoy0kGjGwMcK2MTECCCV0BMGhYWNnYOTi5uHl4+fgFBIWERUTFxCUkpaRlZOXkFRSVlFVU1dQ1NLW0dXT19A0MjYxNTM3MLSytrG1s7ewdHJ2cXVzd3D08vbx9fP38AwKDgkNCw8IjIqOiY2Lj4hMSGdraO7snz5i3eNGSZUuXr1y9as3a9es2bNy8dcu2Hdv37N67j6EoJTXzfsXCguwXZVkMHbMYihkY0svBrsupYVixqzE5D8TOrX2Q1NQ6/fCR6zfu3L15ayfDwaNPnj96/Oo1Q+XtewwtPc29Xf0TJvZNncYwZc7c2YeOnShkYDheBdQIAJpJlyF42mNgQAWM5gxfQZh1GwMDmwhLHAPDPxGO3r8NrGf/v2GTZyn+/wbCZ3BhFQQANckPeAAAeNqdVWl300YUlbwkjpPQJQsFdRkzcaDRyIQtGDBpKsV2IV0cCK0EXaQsdOU7H/tZv+YptOf0Iz+t946XhJae0zYnR+/Om6u3XL0Zi2NEpU8DcY06VPJyIJXVx1LpPokbuuHlsZLBIG7IVuIpaRO1k0TJbDc7lEtcznaVrBOsk/FyEKunKs8zJfVBnMKjuFcn2iDaSL00SRJPHD9JtDiD+ChJAikZhTiVZoYSqtEglqoOZUqHXqORiJsGUjYa9ajDorofKu4cz7qltQZgpHKVI1yxXm3mu3E68LIHSawT7G09jLHhsfpRqkAqRqYj/9gpOVEaBlLFUodaiaPDTH7dRzKprAUyZRQrKnUPxO3up9u2iOmh0/F1Uas0U9XNdUbRbI+ORx1Eecg2Tiflps62hy/XTFGtdsXNtgOZMXApJTPRfRIBdJhInasHWNWxCqRu1B8VZ5+PAySS2ShVeQrtUW8gs2ZnLy6m3e1kReaP9PNA5szObrzzcOj0GvAvWP+8KZy56FFczM1FSB9K3U/EiaTUDIsZPup4iLsMEcrNQVy4UAafIsyhK9LOrDU0Xhtjb7jPV0pN60nQRh/F91PodyJZ4TgLGq1H4mweu65r5T6DWqrdvdiROR2qFHF/n593nVknDPO0mK/68sz3LqD5N0A84wfypilc2rdMUaJ92xRl2gVTVGgXoSrtkimmaJdNMU171hQ12ndMMUN7zkjN/5e5zyP3ObzjITftu8hN+x5y076P3LQfIDetQm7aBnLTXkBuWo3ctCtGdewINA3SzqcqgqBpZPXDuK2sNQJZNdL0pYnJu4gh66sTHXXW1ip/FP/ViS8cyKWJnu6yXFwTd2ndtvDh6XZf3Voz6oatxjeOlIfxMNLj0ITO8m8O/7Y3dbtYc5dQlUEPqGBSAAYoawcSmNbZTiCt1+ziyx+AcRniOctN1VJ9njE0fS/P+7qPkxPvezzdOMst111aRJZ1g9yYPfxbikx1/aO8pZXq5Ih15WRbtYYxpMKLousrSXmOtnbjFyVVVt6L0mr5fBLyZNdwQ2jL1j0MdoQpTXmIh9dUKUoPtZSj7BCHtxRlHnDKgwtahsS4DnUPamvE6aF6GBsLIYahtL0QsEgpXRXftMp38R6ra9roeOKK8HQjOYmIT3GV/Sh4qqujfnQHbV6zbqlhSpXq6T7jU+zrtn1UVhqp4+zFLdXBNc26Rk7F9BP5mljdGw5a90APFR9N0EhVzTG6McoYjWVN+ZuALsbKbxitWmy/h/upk7SKVXcRk31z4h6cdrdfZb+Wc8vIuv/aoLeNXPFzJOa3RYF/50DslqyCemcyEGMBOQsaw9jC5A7DdQwv6/B/TE7/vw0Li+RZ7WiczVMfrpGMKrnLlsddbrLLhh61Oap20thHaGxpeGKOHR6OhZYYHJCtf/B/jHvAXVyQADg0chkmojZdqKd6uLrHamwbzpVEgF1z7DgdgB6AS9A3x671fAJgPffIuQtwnxyCHXIIPiWH4DNybgF8Tg7BF+QQDMgh2CXnDsADcggekkOwRw7BI3I2Ab4kh+ArcghicggScm4DPCaH4Ak5BF+TQ/CNkasTmb/lQjaAvrPoJlBqpwaLNhaZkWsT9j4Xln1gEdmHFpF6ZOT6hPqUC0v93iJSf7CI1B+N3JhQf+LCUn+2iNRfLCL1mfGldiTllcFz3tHBn+5hrWgAAAABAAH//wAPeNrNfWtglMXV8Jx5bnvLJnvL5kqy2d1sQgiQLEkM1yWJiAECSQAJiSlCwBBAQEQEjIiISBURFUS0liIgUkqRIqJFLSAg3ijl9aO+SJVavIFgLSpCMnxn5tlNNgGqfb/3xycuWXbnOWfOmTPnNudMCCVlhNAGZRSRiEa6vwikR9/tmpzydf6LqvJR3+0SxbfkRYl/rPCPt2tqakvf7cA/D9o8Nr/H5imj6cwHq1mjMurib8vk9wiCJLMvn4Kjyk6EG0u6hJLxM6gkAJZSQqlUTSTJJZX5vLY42ZEDXskDvQqD+fEup+rNyIS154Mwlq0bO7q6tq5qZB2cko5c/HDk6DFVw2trOOzF0kapRMDWiDeUToEDVyRZQvikTJYJkTVZUxUcINnUuByQEAO+4Jns/dn0NfxL2dn6DY3jLw4vFx9gyg6STNLIoFCpxUQNsXExkkYM2lirSokiUaAEasxgNFpKZaA0hiLf0rqkpuAzyUmJCW6cusPW9l9CDmgeF6J0ePmrwFOEL0dQCvKXSwkWeaXPewBlx4Z9VLF36LHys+Dq0QKeYceHHag4VvFNa+qbPd6Uhn7xPmuCVfz1/hdH4Ek2mb+OfPEF8lIiYy4vlctUO0knmaQb6Rfq7QRZyunq96WmJCXGGE1UNvBZSyVElqgk00akEwiFWv50JfLfSspsiU6bU3HmgFPVXN6CzICtC7htge5Q0KuwqCDoindr/DOq9MoMuArBGe8uUOWyzw4vPb/rpu/GlR7Y8Ok7S0+9Uv/M+n0bhrGjZWUPstv6lS2EQ7/e43jvkFIJhpwSFQqTKl5esuKPzqdWmqq+ClnZh0Nuu//Wrr3TfnTT17sVdzntwAkRhQy4fFb9XnmHGImTJJAMXJNtQ7YljhgTysKPNGrU6rhYqYTWohRJ1SaFSpKzlKiqXG0AWY6Ry5KHbEvB8d06jzeBoqDY8afIFc+Eev7kcBzInzFUE4PBZSirqQm5XK5uXQN+T1pqsivBleDw+jIyzJyN8cH8gl7eDFURclwQ58mP94MXHFf7Avrk5o8bl58Lf9y+ZcMOqHnhJbqt5YNvpJzFnT+XyYLmlovzmz/7/OtP4NDXf71Yruxsofpnpz7/+mP8jAiZqLp8RiXIw1SUiTzSFHL0MFFJ8XlT3fEKSgGgKJToTPUQRaHVuBWRPkki1YBbKIaTGn5LkDNpHcdUkvAQqEbmuKCs5uVAZkLXjDjVlZMNAaTLx8VGCI++lTUoLPKoVPNz8ovACtArE3ngcrr7g0rYkllv/PM8+/vcx4eUfbnv1Y9/uQZSbugF/X33jGj508KJ90xkO3qXwq2Di0uH/WJ03czFn7yxdM/I0b+6efWrv1sxZ38NOz171xJ2ecKi0ZP6QXm3cfSBgn6hPmOaet7MeQFcR8AzQkckhFxC+1AQ6kFwSrJJuloQKkHXBvpzVWw3nYvPxRB7KBYJFXRboCzeQW05jjh7UVClLqfd7c2kVU8/dnHZo48/eOHxNTQPjPD+1j0s//x3rOiVzXCAw+qHsBraYEU0INFhQRzVvIX2gl40EIy304anH7uw5IlHl13kwNiPrPemXXDo+/Pw/p7fszyENYCOllNVJ7GSnFBWjMVMQUFgJSajARWdSgZwfFM4iVbgJFqJ1SFI9LsVh2aGgMNfhGpsWzbM9rAtH3/6zNIzJ9jWAEzLVp3s0caWBHZsLhSzQ3MgO+liI8wQvBhDPpWL5f3ETPyhDNQdCh2GKlYGogCpQcFQKolCleuFyuNiAJ4Cjw2tgstj88Jx1gyLjsMi1nycNhyHB9i842yBzuMB7AK8Q84SlaSGkvDfSAfhOnxYmN9ArrfbkUV+VfLaizzwTq+7Pxrog4Rjb7PTYDojYPSDTXQ0XYdryWHwB4dFVpnA9e2L7CjwuPrRRNh08aJ4TtglKEaawmsSlmZfBiIsirJCsyPmZ3mb5eHPWxFyWUSuhEiVErFh2lEawQNSWevH7DT1KDv5bkXdW375jDxYeQ/xuoU9lCSB2cntIanGh12kzOfzZXB7GEc9GcQWZ/fkE4jDvVNoi+MbRx58iV1qZZcvgtwKUmvBzbdNHXfLlGn19DhbyB6Fu2EWLILp7F72CPvXl2fABDGnT4s5z8WpVeAsTCQp5DZosqRzvG3qCXFIezbYBoAaCwEIShXsHUkZ1mXTSjjGpIqHVg9LWPIqzBGwKtBWjhA6JjfUFbc0TY6hEqBukSQOsrPCIGWJmVncyGdDAfSnBUIDaIH+NJjfBVxOK8SCyyOPaKEw/bmGbrdUjlp706apj2xqXPqXO25YuXs3bT4Gs55feFufMaMrBh+sG5rdsOOOiS++uuVFq5gL8rQE55JJBoT6ZoCsIN9lVHRUisGZWFAQlBJcFAoSWg1ZVqpRozm5am9juNfj82VzyeUWkE9I9mRwK4g8x6nmQIF4I6Zc6MmXZffGVS/uZO+zf5zdNerdhqce27Rr+szNv/rz4JW1y98C16egydOX/smnxv92xdHTw0HLKWycdevor2umbuzZ54NHd3Ffw4v8mynWwkEKQvkqKmVcC+68KHKjplIZWQkybi9JElYoBsrMZrPD7HDZ7LjLDDhXL/pdngLArebhGteLilaeuY39pfVRuhBSt7EMk2Tw92LnoQc7Aj2OSdtaJl/of8ZRVcGm6DLcgHwrwDkkk9JQKNFJJeIwItMMwP0FnBGRKJFqkVlyJbJNuGyRbQKE+zq22BizppBkSNZic4DvGBLmW3pBLw/nlgOnKBUV0kn/dYn9hX1+ftXwv9RDMjtecG/W/CIpsfX7ZG8/aePZw9+xC8PB3LXgixMuSwn9kl1kJzUr59NgnOMI5SD6Aj5SEhqg4KKpgIJWQqikSlRtxIVVFUmtjSyns1RDyy1X44xdcllSYnqXRF+Sz+dxeDMMTtTcxJPvdqH0WUEL6ruc4C6nilhpYZbiB8N0uK35hpqbfn2owRgz4b/f/Bu78NW6fy6icROaJjTUL26m02A7bIr9wTlu9+82f//hV+zcKkh/Y/H8KfPnVc5ZL3hbJPbHDtRtvpBH0XVbm+8rRVxH/Fq12WRbThCX0eOCT6m/tYec2HpM3gby3ktW3Z5zHpQrh1BjZJDuXFqyPXZVxu1WogC32qiJiTN6dXxeIN7uvu4pSTGoZ8Ct8tWRuSkWsh1lon0SN8qc9IDqEMSjUcZ/0fK/ANz9y/XL2WeffcPOLn68+TaQHXc1zrp95oIP/j78lmETx1c0KIfeWDfj99ePeeP2Hcff/WPz3vLhO6b8au+l3aPHTagsnV0ynr5bWdb3F/ndxw24fgRfyxJBx0GSSPx8LVX0rThn+FqqkkqlRmSHBKpU27aCHXZpcpInLcmf7Pdl6GsJcVzaCnD+/aHIawXuUfUiuKR2v6BEp0suZ0+w5feVjqx7+q0mg6X36tvf+AjMn637132t58ZNHd9Q/0CzNIiNYKOtF1y1+35bP/S7/z4NttXs4z3339109/wRYj3DcYw8K0pvCxXnLo22HbrejrYetqtZEv7z5sp2iyI9xN9U1oxBPNwmcjwqsZPsUCYIyZG5MwZkVBtWCWXHFms2cvlxKHE5frcwkVJ09LQ7VpjLHhFsZ9262aTHI9h02/6OXKwIQfwD2oOsHD8qaJcR5OKWNOlk60i65R3YuRy++YbtY5/j/MbAOvQFiB4nhnJQykHERLgdJTqMB3WVKJRUuv7K8IubYMDXGOkkhy35j6NvcPw4uWIeqoTzcBj5RGAM3dI6kj8Ar0Ii9PuGxS1n5TgP9fIpaS3qfC5LxaHCGOSTBSdCcUtQrutRmMILI6xQRB8Q4svokoJPJXqzVXS++D4QAu/NQNUe9letkAodAlFn1bTJpeNGr/rTtI8vvv+vhofHBdmx9qi0fNwjI4bU9x5YNv5484ENt61tuKG8b1+2OeIuUDLp8iT1IO7fXuieVYWGO8Ak9QHZFEDvNws0VS7hmh43cSOyHdW+KpNa5IOmajeh7hXxBtJgMolAJNZU1r/vdYV+n9+PSs3uNWOM6RSyH/CqbTvALTmF9HfHL2Tq4kqvsMiletKJrZfdhxbMDnzAAOAaTz1Yt2l6+e2JcVOff/FdsPx19OFS99DQ0Af+9czb7P/8Gh2BhCY2//+wS+x+dtOHsAKUj2Ds7hayZWy9KSYYan6IfvPouQdv6L3wg1eOAvW4mfuhPz75mx/u28gOvsfOsA975P6pFh6Fhh/g8VM72A62+eiC5SfMzyBfeMC3WNmNkhRDeoZyzZx6tDsoOPgai7JkKeUhN7d9BgMhhhhDDI7V0MvU7DkeDOU94DECRdUlyUVNraeb2HYqw3yqtrLH/QbPM1DH1iu7L5bRcfDh3b57mYaCgcZXPo82LxYtSjrXpQaNKogWVRAoMlVqRRzEff22GAfRJyYmpiemZfg8OekaGhFPulCmuqfg9bS5CYn6O88uOAhFkDh/0kPz2ccXWk9A4Y57Zs5f/Nzb985jLcrOF/cs2mQzpW1e9tYn0qyKsSNvbN3PFo2fuJPvgzmoI4+gXMeTwlDQge6UU0ORMKBnJZVwPR+JRsMulouiKbbHmY0oNPEQr6Ciz8D19uQX2VSvvtrBfLfWHQOA3/8Jpq85vuNvbA/buhGKjnxwrKFqo3yI/XiauYexliHoNDb9A256+daWQG8idB7ySpmNvDKEczdhA2MpjfAI3TqP16P7qgQdR5QviPN4bcF0ZTabyO5m4+EdmAQPs9fZuA2L4U9o2J9k9yo72QNsAxxrGYz08vWgiMNMskJ+XB8ZXSAeU0SyRIoScSDbwguOyOXRXzJt2SmVt56Bc8xGnQiZvcHYEh2uxBCukWRgBItwSUe4bSDjFGcURPiSHZYGtZ4GVDQc2pJWfU2UAK5JEukdKopB7WbFdUlAPxNlU5a5XVdkotREtLMzOjq2+7w+rx4U2XCb4pIID0TDjUeFjSqyeeintGzZj/exl9hzsBJuPXnk1rW/O/TtvldvaWCnpYJWU3c/LIapMB4eHnthBPv2H2cvOSFPp1GZJHiXHkpFCrkPWcNdAksphSt5FvmjTGLvtL7G3oZCWgoFdE7rUgx899F+Qg8jTDgbFS9zUGILtMc1PN+3Cwp5SCN09+VGViWeiSGBkA8jQ0p1q45+mtDsuJG5mJq4kMZAjByb40AvUdhnBBVH0yZXjCiZNAYKv2JV8UCdC5YoYy9uukzYeRKZkzIC4VtIWijFYpCojL58CXeiLKXhNIXD5tCjC9AEkUUYeykj2Fq2cBEndB48DGMYpY31rd8jsRO2wD9b50ZgywMQthLOTgpnm1a1e2gKUXQPTcBFummlsvNS+WXSNjeV5zedpEeoG3IIJQMaue9MK2UxQYOmSBEGOonTyVfEYTPG53jAa3dxoCCcP/QUHF6QvGyt0YF2eiOMVdEN3cg+lixGtkSez9bNbh2DmNfJ9RfL6Y68e8F5aUk7fw4Jncbjc1kEDo08c1XZQRgcHLEuDO2oKS6Bcojd2Xq/QLuEztdiwS1XsePNrcjzS69A6p10h64P+D74XMSurmvGrhmR2JV4dFlHlYBinm6Lw3hA+ZytZtvwz2qYDJX4Z+KlD159BWax5a/spifYSjYPHoBp+Gch6o4nvrsAX8PZH9t0kbxZ6CKH2NH4maQnAYVpFzopRirz2Ly+dJ7KjKhkBUM5ESily3B20vJH5ixYRnezD9k3i1EujmJw7JLUmVOnNb595mLrBWXnKUEnmg1OZyz6CV1DAVw+iXvvQKYgpvbINrLFkWSPSJ/GUQ3Fzns1ul9+EcaxWey8+5rEM/ZcJZsDg6/KAV0XDxb0u7mtdPC0D84JjRblqiecpmzLShuNRrfR7fbYfdxgRRkowQluvIjbA3C24fGHl6+qR1Zshq7Q5ZH7oLyBbWLPSbnjJzeOaZ3beljZ+eGJhYeKmeNRmsdZUI/2yY26MMDj/iTUhckq1d14jPvbFWC71+7vpgsEDzvaQuruEOhOMfrgrorGnRT017uAuwvIbvYp+2rHE++NmjSt583LH3hgBGhf3H54ZsPsp8rH1GdWP/3ealh14B9jIL2ksGJYTmn/sgF3rLl1798K8/7ZM7OqJLtfcfn4/Xye2SgvPH+n8VwHriDQSuFQtCl+XV3oGpGeYW+zWfIIfO0Ayhg+vxVlvR6ft3PdypOm+pFFhCr8207sXodPseeAzanKuOpcmelRh02pX8HO/pF9xt6Cggd/tRoVbEvFuvP3geeStLVl4QvP/nqz1KzvXa5hgkKH4zyN3N0v4ds1EvPjPCXE4BF/kEOFRdJFKGaxbCuLg2JwuPNUc0hPLrWsGfZeVt1nOkxtN8JM5XNPiFV46qOE50Pa9WWC25agWz4pkwYwFua+lIMHf0UOjwMRiTfqYPbrA+YCWe0Zsx916UOQ98e8eDXH8QH4P7B2NZu62V+FPAktP3QrPBdsRTve8vcT64rPlUjBlveK/l5Z8eVgqUs7naMFnajDzQblKjrcbrfri2Lkbh3/H5TR7Fm2Bf/sgFOsN4yGPhgUVLACmt16jH5L32z9llpbc8LwZSb2R0ooUUVL1ZmTdrsNOcmB4jKBhy6APISUzN5lyTBCqqNjWxa1HqQ9pLBOzQvnZ4zcL9F0myZRkdoKC1J7astus0dSWyKK98gzL/2JvtY6Wk5qLaOHD0lfA3mrxS7gLme76Swhl8mhBLQX+P+oNreERgRTwEE/is5iKfCPTz5hu9WLRy9uwect+ETJT+cdkUAPO936cTjvCCSIeN1teMXYUW3EQASvO2jzFiDu4Mcfwz9Yykxl5NEfVTFvAzXI/ZTXMS5BmVKEnErCzvNckp5Oj+QyHGjQvEYwwM5H4dw59P2+oAZpZ0s5XdTaLPyFFrZbqrg8GOfgDjk7Z3xtPAnJM8hSRctWqYrtfpg/A0PlD6VlqgfXF+VH44cYfHnD+K0Cv4EYnDzXjcqlwBPvdtF9A86XwegBW+QPu+0vcU18Cz1YMhr1Vx95tjhbag6Z7CDJNqC4svqBSAC3DmgE+DGZqlG1NhI7YqCvaVBt4D4f6vxkHvpqikaUq4w0gKLE6OPDRyQhGyHpXZITEW2CN8OXYTM6czyBSM4C4zCeXAwHoDbAANQl4jdpUh/DsOcW/eaVHy/t3frASxP3nD75DTty5+L7nmhasPqWIbs2b3/BqOZtrnx/4ptvtbqpKstjxi6cN5Hbi7VI5w7ViVoqjdyp0+ZHX4Yn72twktZSI6gqqTYg98OkcSUpKMvm46SrjkMb25GuJCfGKMmJjjRnmi0uxmLm5xAKsYPdxFNOdlxM1PpejVtgpxYsdGdg1BTJzEjWA598fnD/DFfwS8izWKbMmNlIp9xRP32GPIu9w/7FTrM/L5+vOtmq61dvPP/QWs+OZ363YcMGlJG6y2ekE/Is1MMYI2kYGtkNuIIOPfccDlLcepDCQyZU10DirBYTup8ucGGMpKSHvXBuemxxOJ0idM7ppNbvIQWMe9eMHXx3/7Nnx6waUv6kk/aDVMgdeibVjzp3V498dqmHD/mLc5Bnhvn7xyHb4sSJIwWcjdZIFNwzyjh+uNrGLXeYkbKB6sksfU1yo59RFXTb/u1Dobyo8bKqyJ3Hi1PK9iXDZ2o6LVKsNcYStUhKBrXF2TFGDGZy++UNeNUAsqMtF0rp/pOnDh2dbnJD/ld9YhqbZk9TpjdPmnm7E/IhFnCvr28eD5N+PLNiw78efC6yRjp/xos8VjLP9SVgcJ0IGGnzEx5ZAblGjQSz7lJxwIhU8WjWoAGJd9ptVouWbEiWJfRwVAOPalXNg1Gs8K2C+SQVPMKVKHLhuk2/5TYDbKG3F7Jv2Z8h8cLXYGjtoTx638Qd40fslNbMnzlzfksV+jc2HqCzb88+cd9jXbufyQqE/QRptpqGnBJZJFmy4kxxbopUooECA5GhYpbxwovmJ11KbNiiOLz+yHkX51xBkTjxFKGdivOTTOwCO7xp0+7Dz86vqKsY2BsM0tyWJdLcx6uq3tjW45PUYX0H6fldlTnlacirbFJIQmSzLhkOAH5yrxnSjZRoSaCCXJIsPpU7fVqjP5CNup2g7qojBoNUbZSpYC2AWk1U1VUaOcW1URSj3MhYIwawSA9/glxzfA2PBEP9+xQX5HfvhrPMDvh9vTJN6Je4NJEi6wEoKn3EAYDsERpNP/9tOwV2Q2TT85V0hP/hzZCn/b7npY+Lcntsen3XHvYKe/+rH+6Z12NQ+aAxt5490WOhnQXmTlv/6m2znh05c3r1qNEjNm6S65/OHXLzjkOS4utW8uxTb/71uccnPpjqrA2GRmVnbrr95bds8iV5wOCxFQN6DpeG1TY11b4t7NdatOebcL+6SLdQthVjdyjRtw6AOA8SW4dUhUMlj9PmCJ9o2/ixjwiWbM5Y0I8M5E3s8I57b2eHIU/T4ib9bf+7dNm3W/e1fovaap9/6dj1/3WA6+DHEfHHiNOABpLHS2Gn2MpVUyz3JmO4wfd60PA6egXzRbKRy9LjGzdWV2+E/hy+smDlyhuGX3LL9SIOutzMnAJmDIkn14UKoqLtsKBaOWAhomQkuuXExrVfvNMWGwm/VRF+F0TCb6cqRXAPmj1y0I3DboC892eIGTBn4jeO0SPlzZeyt+/RpvCJtPHye3EG3SuUJ2OwadAoyDz1qucgwkqJkNio5IGVWG0Ou11F56ZId+70SF2D/YvRkx3DyuAY+pivL+JUx9Fxm2EkS25dAvsns/Wqs7WckQhuOIK4pfC5vDXi0YfPLj1whANQnfpYrQT3lZcsHLLNhlskFbeLhBZ/HDraGDBpvHgkttSAbr0oddD3UZfwIGi8+qhQRtsAsY58lFLNjyzcbYNQ6cZ40hwZtgyMub1xpgQhSJlhSQq6g/2Bq1z+dzz/RP9KXsoOV940u4kd/jLblrv5tks1ybm/v+31fez9yptmTKfL5s7dsr/1W7l++bCb1leM3nesNcA/W7u1Tb6RVieZrtOaSHRSRWjI8wCxkTxAmM4kEqHiyhGhLiSaxPZNEh5QU/OyTll4i3SiTN8ho+v5FkFSgn+4fd/bfKq/PSCmX1l99CAJ24jV4kwlMbw/IloIvw1nN/3hfEJEf/DiGX7uk0kLuAmPt8Pj0+bPmzJ1/rypksI+YZd/8/39GChJ+IPmbdz8wvMbN254np1j7y4DwzawQ8+H2UXhH6HO3YG4HdH+UTiFxu1nhCNoWsOmtDSiPLMj48LLr6oRDdJ5LJpelxNISpIzzZVm1b0jVaLEAQ5hegPe+DDXUK3YbU6qShEHCXkqqftnxMWxwxdyrFP+cuCTyXtPCAfpvYnO5Q/Hsz5q+YqN7H32zR/YhV9Kjwv/CKp024u0zRR8TW/3TcLSoOH3Gm1fc+6tcm4bufnl9U02td03icjAz3oIfRPUqkr0+LCtDK/qlc9w3yTGAqRLql4757BZ0mPSDSoxgxkZ5I84IhhWu8N+Srw7WCSyZfzEj06/Y6oyff6xKfKBT04dbHq2ryEZw/f3Y2Ly1h1fvNG/4xm2ddPGcxjoxeHK9xoxcvmPO+GDflMqR7bpEmkO8imOLBMUv0S5AitJxp8S/kS7atE3El/ucUJ9o1ssHGGhu9s3UljtXTniyo1EqhVoH4AciANiFZ4zquc4iBNntW6pw46S4vpnu/0Fjz3HDn+eG9vrRXmWkf3dtGpJ60G5fnf9LBL2ibcgLX6el0kEiSahTpbUDnkZd4eahxy/vrH4qVGggHtTbXmZwmDYN+bnwV2otOXc2Xlbhoz6oHJjt6ljF84t+vy/3n6jbuRjQ5fc9Piieb1h6JYdnvSWrMJxvtzizMK6OTetfG7MR77uN2b37VNQd5fO6x44vyJlKGIeEOprBlHfQNF6ET49WWpSUCxEmUFsKfcMYWSkzsAuXFbBHXTeNS4VRS5x0oUu1wAIunjmHv3UOSNm3L7x5ecf2zRmHxSzgzd+7P00+MorNHnhpDNnT7WeGthfn8ca1Dnr0Jg6yV36zkhCIcBpjOOJs1iR1hupRoI8fX2T+QgUiMarDeGpSEmRMA7Uv1X12I+PCcdIaAr01K/DiwZBiwr82vNFHpu0Kzsub8es/W9BHtebkLe8YtTRA/TD1llcbVLrpXVteRR5Gc7fzCteebYZF7iRa+XYUhlEChRHmInZ4bAp9hw9qRMsLHJ4ADaz0TDgrz6jqmQdgwFstFzfumju1HGLabMADjhHou5H2KkkL9SdJ1gpkRoVXm0hwSRVppw+burDFj2VpDrjebmrnWezzaJ8V0c3AHgSSeKiyzHLRaPv0N5lu9he6LkwLV2V0+J+CSMWyRgxgmrsbnoActjrsATOtjwk1zPb4q+GbhhFE1q/iB99y80pgy91g3N8gkBMSPsqQXtaKMWoSkSmuE1BEB9O2NvtdhtKNboXycLJMKOfsZithrEfHoex7AloZhtOn2EbaR/qZc9AQ+uJ1v2wgC0M8xb1JudtPN9DcSBjfCPJesRCQA5zmftZI3F9bVKZ1+tw8CSarZ3JfGF132oAQMXRZJNqTj8MN7DAwH/+ZvjQooGVczPsyPeHqmbcWktnX3L8fqvt25gJDUWRmhFpM+K/Rs4p9lo5J36u7pE2tyyjKkuUSpiBmnbQPV8eaK3U6Upju+kHyk60dOj7Oh00UmiHwXEtkqbIyk0RFWGRymwptnibnYc1qHoL8gv7AS9iwDjLZUPbFJ8KXBu7UHwhbVHzPfftPT9hiPqPr8obzu8F601158ZUGaD/U80npUFl7IPdGRbvi0b2Qdkg6e/Na3ge5yDrQdepVpTbwJBtftxdZt0jBozOcDeZI6WWVtw5f9BzQUGbFw4eP65aL3jaajMlA9KTIXz5GKrw0imkA736WpFOIjdFqst5Ss1tS7U5kB6Fn5MX9IcCm6hrzgygjQ0U8PozGze5LlS41FD12vMThtw561O1vOH51yvnL25elDvj9oK/07IbIMc8eorNkrEbcgaVSScLZt3G9ppGjjxXdYuoX6ABuVhqRv3WN1SMqoDLDqElVoAQKo8BaBAlOpzwokmJc12fHpVu4Ke3FhOG43pmgse5BSJiihfRrVDOmVBS1Fx461M3LhyycHzBPYUTny65a+wiura08PNp6elFoeLPpyX5r9PPSWaz5XAUZYjX2+uZPjQBoyLsiKWRYntHEKVVr0IWEdqm80G2EeojxSK0viVP8bcVigBZjnozTnkHo/lHQjaTgqIZb0YF4SZUlng1MVelabyynMC48DEup1Ivm+XBZIzC9WlPHJauD6ON1xyHVjMyJHLOExkqBqD7aXN6fbZ4XFW/D/2CcJW6fpIQp3NOc3lpkNUdObp942Ke0Jl7x9dV9zcs+OX2bX3gIob2KWkbnzUOPJ3qe/637JW0XNbPuJvLFm6LJlHXeEOojCcYeVGjA90ang+gKlcIUo0wowa96kqWRVIAZwWEezHcXoWteSqkGoU1j6QE0FY5NV601x/6oSgGkfV09759N/9qVsVImMOWPknfv/TFrcO2bz+lvFP1Waimes/x5ZsrWMuFbrc3rDi+79CH+hrPJEz2yJtxLQI8a2ExGxUJXUdE7+YHfKg3SkyoGkMJdgoDBB/hpsiRfaxU5svqlilCzqJAkZub+iI3D+M1t8artwNaoCizKMrd3l248PEFtU2TaxesWlhQ0PzYgl9MnDWyecWCwsPTRlROnzm8Yjo9OukXCx5fUFC0cPnCmtum1DWvaA4G569ovrlx2PTbKkbcNgP5qiJfl+KeFVErEfljDJwM3LTQsRr3VMTJvwnaxaBDL4U5QeS1gy6vi788BZx9+AIPHYxgitnCfd9//z07+cMPP7zOlkBfntFvdW3/5fYTJ/Avehr5FqV7NNJb1z6uNn0jGlIqdcWeHLKLDStS5G3q6KVIch4Vki2slLhaajnCFrQcwQea0MdZoGaTXHIdt6AmfLYwLyPRLKkSTp3qlbFiFeJLIwoXQxxfD19ArEdhUWaU7yWK8XiFiSja4ysT5aQBrpBblO1ZYe7gfos/3rl5T1XZsorK8gm3P7emuf+Ac4feeazywKC9/rEjXv7lx/fdUTFxsb9A8g9flDVy5aJnR76eHuzeM688N/TCtBersyaXP/H74Yezi2cHegYzyn/1UGlj9+Kasvo8K4/3KbjlC9J+lGde+xYfcvBEManm1tDaVvMWXS7ljy6dcg8JlQwZOjA0BB6v7DtwWEWob6XSPHBQeb++Nw4aMGjg4AH9Bg/kObCGy2fUMtx3LvRgC0hFaIiofTEgFqOofSGSgfuyhCt43IaoETQtUiIYDsBcfAsG83KyvZ7kRK5VdZfRKLRqRCtEFB7uSRDFMbLd5aS8JJIWOe1yMN/HS/F9vGxKWnBL0zvF9xffdf+7n5w4uPKZ8pktzW9B/dv8tZet+/Nhtm7fI+shZd16SH5uPfvHuvXs1HPSt08sYl90SX2j+OKJj76r2NSbfSmeYeve3s82/vkwjD3Ah0U9pu/pHvQbuks5iBz2kPuHbMtG2bTGoczEovEHkY5Mjv4ANwuPT/iwFL67FFBRN2q465V24+fkAYhXw8G8QKWxfRzpNAqDEEI8aSlJiN7t8/i8BlzRQPvphDicCCcrRBcHOgWaSvPScu4un9284ZUJCyofLCn59YSlK1lZ9/RhNRNW0JbbeodmN02bZpRn91uSnv/IIjbog0xf9UCfahL0VpF6eY0so16IIYaXLAb0/7NyHOA2ggZ+/UcVzDrOvgT3cbYMFfbtH/H3H7FHaD84OIetZ+vnwP6k9rei0uQC2uABSJzeC0CmoPCCqErk52sUrtePU/1eVMEeufh4y0Xp5AWYevzy5UgfQZyXnOXvHSrhjSBX6ykA3vog26VkXvzYJreWsOS+/BJfHUCLaMa16c5zcyi4Bm7r0IYotW25A6NRJFqdPPmqVaM0uzRuIF34VM/2p64crSiRJEnUc/8pIh6UBPMyM3Hzemw2h4kXEWq8SQ01G+9JiuRqf2qLyIuXf8229zgFhpdeYAvSSoqvuz61ddxPb5KlmwewGVDFtkpPPMy+LS4ZWMg+/qltApcvqk5ZVU/8xLmhrF46L1tU5yEhZ3VSosT9FitJEv5QR2voRGvoy/QK7cvlW+9FQLGPNoB1dWvr635z881rb15x6I2akpKxtQNDtfJM/um6upt/M/bgitDYuv4D6us4PnSW5GKFoq6M5XUa0X0yRFGpUoNLoVUaQNNitDKLxRJribXrVVoYRHVonOFefVTzDGuWTtJ1xzGamX+cLWAfi8JZjKtpOX1d1AtmhzItKO1maGvDCJ8uR7oxoCzeG9BPlwuuKHqNRUP6euvcTyYV98yqnDD/8deWPlu/cDKMo+WbjjfU5GdmjVm9cPHsUatm/0L3wfvQYrpTOYQy3z/UJx1VTBq6RrzEQNInIXouwr54e8tFuxvq93m83izRcqHXUarhwCmq3yLcI8L1Dl2zub5x26vL73ti/vIxFTPqqivzgz1GFk/s/+StC9bJJ5cXxThvGz73oUGvj51SULC2V3EmznhZj/73Xnl2rMA1zo5Fn1RBYZEHQx0Nhm4ZAKN7XbpB/jD5rYmuXu/l83wRxoeb0LdOIEtesnPPpURPtXbhBcdUGqeiaojVk5eiDIdvVpsSSSPwKkTuVDRea1jI22mE+A7lSdQuh0fhzjXaXDz5mqHFR9Ki+sGdXgWcJmpa4MiY+ntuh7zPc2K73jWmf2O6ovC0OByZO/d3b/JswqpRo7p3G/Ur9ie1XKznTNYiL1edKLUYe/IOC34+IiyuiHsnRwqtuMnP0KsmgWsLr03vZCwMeuTlv9m4rv/Iv77z31/QOtaijvhxmxS0X7wEcjh3P5tmw1GMkcxi//KtOzxcpE3gBtF45ehcLi9ikaPtccjl48wJ60T1jS/kiYtBz4pGegiANolTDyIOPZy+DFT1SrTjFCgKoi8F6558tqIsuTZx8MaKPcnlYy7ee7trk2btX524pl/CRJFLwz01+Io+qfa+PdEnhdTrfVL+IB18VlKSe1ZMg9w3aDm8PuMXvq79FpwW+qcY7fpJtOvcd9L1D/5/U1vpnNTmQEXFYkqU8qEnX/71U6/84bnVrzLvwMGDBwwYPHigXPfbPfuff2Hv/o0NEyc2NEyYcC17aoSAEYow1hE/5DVs2XGIZ18dh1lMbX9PvazPHKiDujmsOKn9LU4f7WJ3QtTHlJ1xGeRsdxKXSeb/hcQFyD8oalebRj6Tnml9mVSGKowGRdKgCxAtHmSDVGI2UgO69NQgjxWMM4UrqDSNVip6qiY52WJJzkjO4EETevoYNwllmGCzpOa0495xTdx3SN8j7vLQDRy3CimgqMmoanFjanIJ8kFBb6dOIDdEIQ/Xx1ksmf601KSENqRGjhRx4nTVbcohjnMsxzk7TeCU4sL0voo4/zZkWxJ3vDyo9tKjWs3MxRY0L3lgsKjorfGvtWt9XaODKOAq0gKkNjaGWkxgUC2GsUJdWjXaQWGWErNZreRnDTE81V58ledQa2miaL2zsu3wbE0orbAwECjsV9ivoFcwv2eP7rndcgJdA10dQiNnx3lyhNz2Y3OlY6jreIxZGArGoNdtxc0l+isk3hjWqKcWxkTKXmJ55jHTl96Ft8L7RXOF3kuh13sUuSPuZBEPfqPFnR6qmTqt+Kaa4TvGTRr/Q9NHl25Z/osCyGpPRZSPf2RYxZgBxeW9u5647vo9L0x9djIGEH1gdEQvXL6s9xJos/i6oXLIJD6xXhgbQBUZHLo+3k5VxYICiYpCUZUm5AZVmwwdeg20cK9BbKwRQ9NYd6zb6TBajVb0qww2m92UmnNNPHeSIzoeZ9z/BI/D/jPxHCCjEc/Q0I3JLpukqGFEqqI2mXgbA8cXjcnQEVNKbEpCR5rMUbjGdcL1lsA1OlRtEX0ZMbi/NGU4MZiQJINaa+mACf11DSrNYXxxAp/Xk9YlJcntcthjM+Myo7Baw3tNxzu6E94mgbckNEDHq4FiQIyKWmvqiDGMi2Pye9NSee6jI2VROEwdcKjk4Owx4oYHvQ5AJRpVtTb4NaL0wQA/E4voY9H8ov7awU9f80UdGaofg6LWoGcBmoGfKaIeqNS7OTWNVLadtOYSo6zIxnGdHyBXHY9+u8MeyYmEz5VFH0ykIQLQFdD8LU8y1yq2ny6XV7W+R62t39LgpVSYt0B0xISbJNC3q9D17Bz0a44o73AezeE8yhTrMI+m4tcPhxKd6E269EhchSx/F0mhqcg4Bb0fkyAV40oVxmlR3SmqSkcScc4XbuSz8TsWskQeCNTGnxpaE3JSCGRmpKckOWw8Owpu6jYkcVkV/SK4L3ictkusaJa+0y+/jFIzMNQfd6Cm8kXV+L7TQGv6t10lMTExrhiXS2eoEZXfv8FxJ6nTcdit/wMczg44fo84nGEcv4/CceDyd4ijLDQw0RErqRoi0VStyWigKqg/iSUpJskRJsWUocu/TsvoK2hpuvwj4ikK9cLNxYNtlP9/C513SHvDoL26fb7Mi1y0KB3VTV8JNg8h54d6WHhVHu+skmiTgo6k1CRyvLJeEqIoAEqMEmMyinBaTdbn+zzC/CLMl+ejoN55uVaHalJ/DlSjIQIV58nPeZUr5nmAfYYQ0bbZzKiReYU//t3Ej7kI1WEqUTDtit0amaumz/VyK871t1H6S4fbJOB2DQVk7pryVsRaAU3VoQEkJ7ocsTFGrR0Wz/+Ee60sGF10C2Uros1ZI1JNpJdI91+q9WNjDNq9Np8nzuDOcQRteqcFagKvnwdPtiAvHU7vsTTcfCUasZbK8p//8tzXXy+CvVIVbWAPsN/xXiw6ctkX59gaxrbosqIcQjvQLit5YUtwVLdukd4S9DgsKDFoBYjRDJpk1GqvbDex2/xeT3qXlET07mxZ9izRfmKLSdfXWfSS6GsidE5+2F7P1qWyS1L4VEeRm1R0YJWmq3abuN3uNHdafAbKppaWc224d8IgHW5Kwn8A1/WTcA+QfQi3d6jIm5ooCVuiyEoTb98H+ZqQfW6fQ0zZ0IEX4zrBfkvARm2DOxRNk6LKtUQz4pw10aR4JegEd052INOXkZaanOTuntCdozB3CfvSOo7RnXA0CRwob+ir83JH+RqgfTxK4TNGrQJkIVjkOHkNxjTqH9Au8DijyChydhhwwBb2wWrIhbynMODMXckOs6Or6FHIXcM+gJxV+M8j+C0OQW0z5/JG5XNVRquZTDJIn9B1dlss7u20Lu549Jl54xRPY8QBhFQgAyKVvc629kOcWGbXrEi7kX5yoRf28hMNcLpFe6vSKwDxwXS992j/Q2PXjT3AG3AOPDZy7Zi9lz64Di6OnPzKbvpZkFmGNmI0JPqQBi6Z+uBeWMQbcQY1Ny7ayx544w56PvfH73bMbrUFkJ+iPySsp7aKvXJdWIJzdIlIdrdLmiYkTW3vIhmpz97pRHJSnCmJCeAAh9fhNwh/7Fqw7wSrDjvR9fNhJ7h/FuwDZA3C5rmcpPh2STbokqxdHbrH6Ultn7sxCv7oK+A3Cfiov9slzXBFVw1A1yzen50Q3wGm6O2I0t99w9Z4oa6/Y81UljSZdxuKGkSUGlnPWoguDWHLUPGaTCa7ye602+1ql5wOPWU8d5spsi+RzCnXcrV6m4d4VqRJM0S2KtxO5grmy21pUnkz+5RdfIp3Gm5nYGkta1jxyIIH75boqQvsuLLz1FnWd/K8KY2i5gj506imkQIyWU8LmVwA1ClSGMmRf/D7LcJVvfE4C+EWxevFOxKEK8GTor4QZfntpT01IUuv/B7dc7IT3fY4fqjstMp6JY/rimqf6A54UXDUi7eHyye7hspDXQtKHpsc/PLTCY/2LdgQ2pReVX7P+EGDhvVf0HDHfbi5/UDPQ6b8YeiWob17do1P7pVVO35exZatqRnf+vLnZwf7dh08pzLUlFc0onvfysnjLq2WZx369EVcT9GDoXr4eo7m61kmfPLS5wjZGIrNCVCjwYFhs12cmOlcKDCD0SAbjHLjFf0ZBoO1tFOThqmtOKc4+rmO3RodnzNd2bKR6PMB+Lr5umVn8aMRSIRE3rpht6TlXJOGO6cJGrL9/7/RkBW4Jg3pnWg48MNl8ttQXM+umZLB2JmIQn4pnlHGCV1JhdF4BRXmNip6d3iwIxkdHzRfSUayICPPl5fbeTFi0sL2TafF2omWt0yE/DFkilAR6fLp33nusWAwmDDQNZmspTFgNJqridls6UyNtY2aks40/CQA65VUefxI1XWFvYJ5Pbp1zQr4+vj7dKAtriNtcifamlRCfnUlbcHOtPHbCg3V6LJdKWSWNoKKOhN09acsVxExgN7X9crv3i0r84qVaZv79k5zP5hNyIYr517UNncjMZmNplp0MQyy2VB7jW6s9vn3aZv/Tz75H9KAVgFpUJk8GyOnfLRmK0POBCeVZAdO3A+amilRosnhowC/GSj6axxjjIlynGiQZLfIj5BqsET6eLoSNK2G4ThaTK/m3wyuCWX0CgIpLAhe1+u6nqjVfRnpaTyvYo8Tty9lQZaVFxNnBqLbySJnO25+fBJpl+HdMu3dZZDBW2qUvKUnB4/f/+xv9n37zzdqhj28YcJrX346q+R3d50A0pqrPHrfm7uG72odOPPuux+eeBf1fmiDrbTJOXHy7JrXNvD2s5HXlwx7e+Ki0ED29ZdP3PfYmDOBLJokyzfVLpw3Eb45M+NuXYbrmJP3bHE5qONyMEjPMcAiPcfAL7hwGFESnKBKAV+qpJCUcI7BInIMElVVaVxUd5eqkpGaXmvdMceAI6mkNv7UUJFjyPR70pIT9WusIJ7G6zkG0buk63Yx18G6fXqBkJtDY33pGPsngiInYXgnlZhQAjBOlxojfU2IzSpKnaObm4x6c1NqKgbykOpP9Wd4RF4sxpBiTEFXTqOaOUnsl6vjvnOWwJ3R5X8Dtyf9P8J9wEDILaH6bG8a4mpHbjZQRI0TaMeuabxdsCN2UwfsXVO7Zl6Fdks0frkT/iY74VnICOb2DjK+TdRq3G1ItKkTWrOOliMN5uXm8Es6r4IUY/in0Sn7SKdZ+JZ7wzkMfjLVPZQTzmGgc8kvUeVKlYpixdhw24nCK5ViXDabEs43rkDH8i8Ij/u/K6Ig3nn5eh1iOH/xExCdAiLO7wGEd/qK+R1g7+LovFD3SO5CkSn+QJCNka6mKHh2xc7TQmo4Frz8KcJcofM5CmaTgJkRSuNRvH402QGQTySA1HTddxa9G6pXnA/0CHVz86KUkuSkRH4NJSpEE4SIKPCmtVHRmi/bl8ujtcxAUbyI0eyuOOLNIOLSIndRJkZwkbtB+ZkBsBGDJz56+8mGzc88wf52+Tv23yCfeG+QobT5swZqnzdl7vS7ZsxvUtwFgW0Dhz60pnE5W/kF+4IdBPvpL8E2Xl4355FnWhun3rf0sYcfWY9ro/eAWcXarBWUV+peAtrM/br+7hnuk7AAv+lVN39mlDNjNeF3/l6zmSYvUub/8x8MFfzUM9dovQkluOO7ZmX6vZ4uKUmJ8bnuXNGLo6dWIvtoddQ+rtL1128Ijx1Rh/DYkd92hHEe7l7e5HvV1pvk5GRvstft15Mg14DL4zDeMo+Q01P+M8jxYcj/Zs4HLITfaRzwpIqYlCdWFAnB89aSa8POSs5yiGnzBEsbbGsn2G8h7OtDqERlOaJFjOgaiBIZy9VhpyT3yM1B1vOTjeT8lHx/OMPShkPuhKPJIva8fp8dRr7XYgjuLb8/nF+RyBhyQi6WD+o1vPyOKl7cTHhuhqhRt60Nb68iJiQpwWm3WgyquARA0y9bU65xEd0Y6WTrovKSkhvLBw4cUl5SOpj/pA3Hj58bOqTixkFVFZI8ZFjlDYMqK/T+1LrLO+TV8gqRp/H+7DyN+3+Up8kvlFfzNq57ppauKVvNO7nuv6Node9HWi45oF/OgHlT4T0LeyezcN4UmsfbunInTm54Bpy8s6tX1dSRT7NL9YPhuHvD85V9WX7sC0K2RM9HWCevEft+TJtF374zNVEk8/US5iwjL2xTefOS3hYiTGpU44ehY29IdofxvEnkmg+EcqPHipaRqLGGq/SNuOLjAeK7xHdJTsJPXLx/xG4K75er08R340bCqUp2/39PVVJiNFX/Zp0O5BGy6+WMlARxIKKTlG3i/gfS2E4Tdz+ipmjsSFPXjg8Ioq71RKh7h8E6Ve2DjVehyi2o8sZ706JXyxxFl3wFXU09CFmvk5MZJiLap4mamakjLYEIBT81nN9SoU//ypGmq60NQLeu/BA/KaETDaI3JsoPqdP90n6E18njFlIVG8gQx4vASjRQFX6fQWOkc0Zo6yvaZxISEtIT0t28i8YmUuFtfZmiT9tDhoQGO0EhVpEPQ1ESt1eMjbqVgrRdXsEpKNXBuuMJ6ZIS73F7EIjL5rD5MkzOHOBxUOQ+Cr1GyhYL7qh2SziSv27ygU8+PvhBo1XT9N7uL8Mdl9J7laPEnRSocZb2r2d9DITNvWtwRXvXJWnjkdyJR00fCRsWYU97P5FBdI5y23AV5vh4XSLnjNGb09b3LPo69L7n2Kv3PeMIcU+XNkJ5j8QTLwRDJg8a5HS+SXHruFF6MvhRs6aoNSjk/D4r3jrljASdHbya3E4jxd1O4n34aLrzQ0kiiP85D+nXyIuygXAYjo+n4+O92h5Hf6j9Pr2feBZt5L97jLQ9hZ5VW9yP/4XsCe4uKW5vgjjbtPsy4sxuZGf7TX3x7vaj9YhfWmTrDtqIliczH8lle9g2qIb+/e73Sg1tR+27Nm7Zsn7gvEzvYqMZfgmTYQosshr1Y3fpnUv/Ov+jyy495IgXtvWMNl+eRdJIDsknTPdCY60YoKTgsqVSKkq/9V9pgB8bOnwc+USNfFJTowPwEiP6C0a5BqM0fmrIl9gtehNxtUxtN5CkiDWODBWM00eSaz4Uuo6PV689nkfb4eHi/LjtUWR2l9xuQHr26Jafm88T/UkJLqct1mJCBZUGaRZ+LwvvQIm0dvPeFJ65cLXd9BHdBe7XbwRRNrJ/rlx2cd2FxZAMdMZbNzx67jzEtcZrKxdPebWh8rVLk9raw7+qv9sIe+ls59NPPwyGF1EEcocPe4bfB2JYsKxrd564uD7cMb531tzpzZH6EZlFnfH2DZ9m2PUzXv3YQVzEzC8uleTa6DuneMhkt9s1UVMF5FX5QzlN9eBewSjLgnEgryuUKBWV6pEb+2NIjMMp23iQpGpeXo/aBdwuaeSvhlU8+c5rpTAx7anB8oc3V1f+Ivm1Gb3+MChZ6E0YKR+TytXtos7Qo/dP6cEXGoDhYUdMIjdcpdpQi4q4pPJHZsx4cMnM2x59trBr18KePbsWKgen3n/vbVMX3j85NxjMze3VS6/lPCwfky2qFbHlhrqK+w7DEaMCkbvoePeu0x4XG2M2qLLE76HW9AY3ryi1pnxRgzxn1Z3CobTimJXm5XdmDx9W7hPvV9zB3yslBYUr1nsC6b3ED0+nO/9Jx2v9/x++q4Id0la65mf9HoEqqQ52fPZZ5DntZz+ntT+XKO2B9eI+ri6hZAjf+9b+ayZ4SRHR+HVWjkChvShItYA1fmLB8RKvtCe+Fhwz2XdfdYCj6nDaKlXbf8sFvyKalzbrV0TDKmv+PcevT1N2xu9nn0HWe/8rv4eB8nlIcwQ9sRyKgYgqa9EqJGqXkCa70+7gsqfT4/GHf8IZnTCM3NMi76RjYRqjaG3HwWtcEQe/ZQD0S96gUtLptTvsdo5Dn6kSnvB6QTIkwkn9TYKyVic+wgNx77b0rOBBDO+sDsOWxMXbw9rI4JzQMfg7YoiwBP7ekTfuNhbhs+NhK10rZ//M+ufxTzy0dNWTD/5yJT2z5rn1T63euJEL0+zLp9Tx4vdWuDEWSwq5eb0KwhAXJet3kOMWV10dbiH3X+N9243k9IaxN/F3o+ro4M6/76Ky08/wfiH1SIss7uvGLd/WuxPVtBPdqHP1Z8L1yXTt1YuR8Rk2h+eqfuq+Oy/QFS0fnJNywtfd4eJZaDn9XtlNTMTO+zCsIPPLEND7UpX2363Aa1Z0TWU2m+1mu/jVCvy+TK/LW+ApgKD41QpF/LdT0O+bz7KFsPBMM1upGgtiF9CTTz65hg5sfW/96PFJU91vIH38fvtFyh4SEHliIGY/LnEc2msbOsRyiQnAKPodawwgaXq/owXlK3JWpCj61Usq6k2eYeBn6+54pz1Wvx1CQR0agEBM+DIsl0dzcd8kXr/rOYMEvAXBzIIi/LuQhO989s571g2Ps3qMyCewrUGwLr//jhe/Y29mL71Tpncs9kPZv3avH3/9XfKhZQ9s8/bwbP9X8WXSB0ynt7sDBdu+hcSqL9K8ul8eR08p68T+TiDNIin/UpzQFMn8p9x2W0a6UZb4pUSV6G/y83OzyaDI/DZc/d/cgwjwKipRR6Tfy3mtkTUhh8FgSDAgE3Bh7LzsMc6SINoOUJnyu3LdvBEzWfRjFnklaN60n/Y6SAds0OTnfli7+ZtNsrqVnqJ9V65sfZPapiMfjjAKrLUHrJ7beq79PoVxMm9ZJjwmkMddWstfwvbzO0QvqGnc9mfzWvAbhe0fKiXiQ88M2daN39LFBSsWNFkTPXoGfmVYsvhU7fRpjf5AtrinSQV+oxh6TZG3+uViPLDkQbNwHpBR7ZePER69icD6KiNrQjHeDJs3Q1xIxsvzwnP/6oq5DxdzX9Jh7sqVc9c6fRqee6qYjwadbkNTI/P1iBj5qjemqZGZmniqkk+Tp7r0eRqsHecZFyA38nlyXqsYIpPXdPRun5XKkj+WxsipmnB3uxiomLP4xnLlN+F59+Q5PqnaYqSClTExbe8N3AnDOcaaKZ+lNXy5RXIoV3/ECOEnLNXEYrFea3xNKLH4uqLCgmDP7t1yeHrQFqEyLlwrKegk16RzuKBzYyc6lavSGXPlN2E6vfqkYzrSZhaiYmkjLYDk8AOTdvquPrIm5NK7AzoQZPXmRPZN+D9tuwLK9ePK8gj5v5waKg8AAAB42mNgZGBgYJSctaDw2Ml4fpuvDPIcDCBw4UnJZhj9r/yfCAcfezEDIwMHAxNIFACaiw30eNpjYGRgYC/++4aBgWPBv/J/lRx8DEARFPAeAJzSBv942m2TT2gTQRjF386/DSVIDgUJUoqISJAapEgRCYHgIQQpEkoQCaVIkCh4CCKhlB56iCAiofRWoSxB1JN6Kmvp2SIeRERE4q0HL8GDiIeiWd+32ULQHn682W/mm519b0cNcDkFwEwCSqjgnt5Dx57FjFnHNf8Cig6oqpPoqG3qNgqmjqLMqSqKagMFVWbPPI6xViHLZD7hFGmQMplLtCTrpVf2OER/gPNncNN2ALuE0GbRtgOEZoU0+PwObTeNUD0Voqa9ynoHof8QoVsji1zvEi1zrolF00POpfHCzgL+Lvet8TuHpIfz3KfLM6eps6aElK5Ev03fu2I+oWYzCMwU6tS62UFdZ5Hju5wtIVAtbKhWtGp+xePA7yOQuvkZrw+kR/cQ6APqMvKc2zSPAPcFkybAhIz1N8zpM5g2TW+PWo29TLznuEuk1iIuXrOP2zzbcfccDZ1B3gySHnovNYPoQN/hWcXHFPLkknwLfQhsAS3x23sS9Vmv6xO4KP1+GucSrtP7Quz7EfhbVGYR5zDipSgzeEPvnlED8odZ5Q9z+Beeay0eM4txJAvJzL6if/T9KPwadWqUwzjM4DH9X6feJ/ux/0kO/yH/2Gh+cxzJIs6aGmf5Fm3/I9fLP9LHjtnyFqjv9Wv6cJe5JapWAO8rKYzAd+oq9Rbn5D4kGN4b3q2qt4usoBZQ1F1kBXOaY4Ub7jOzYa/6wbtFvIlhW/ZmVmm5K3aIjFniOR8gJ6SCETw3/gLn8tkeeNpjYGDQgcIohiaGO4wujM+Ycpg6mNYxXWHmY7ZijmGuYJ7BvIX5E4sESxzLDpZfrD6sK1jPsRmxTWA7wvaHXYTdgT2CfR2HFUcJxxFOLk4bzgzOPs5bXCxcclx+XDVcc7jucQtxB3C3cV/gEeHJ4Ong2cPzjNeAN4G3g3cL7xXeD3wCfGZ8AXwlfC/4w/hnCGgI1AjsEywQXCV4RUhMKEQoS+iBsJVwhfAzkSqRB6IGoiWiV8SYxILEmsQOib0TNxEPE18g/kT8iYSARIvENUkDySLJRVJ8Ug5SB6QtpJOkG6TnyATJ5MgskbkjKyGbJztJjkHOSC5Erk1ujdwDeR55L/kWBQ6FBIUpCicU/ilaKeYozlP8pxSi1Ka0Q+mNspSyi3Ke8izlAypMKhYqKSoTVPapfFAVU7VSnaF6TPWLmoqan1qV2hZ1M/UZ6r80EjTOaepo9mkZaLVpHdFm0nbSrtFeon1G+5uOhE6RzjNdIV0H3TzdBbrH9Pj08vRu6DvprzMQMHAxWGRwweCZ4R6jHqNNRg+MJYzDcMAU4yLjBuM5xjuM7xj/M1EwCTJpMNlkcsVUAgiNTH1MM4BwkZmAWY1Zl9kLcxvzDRYSFioAsiOMgwAAAAABAAAA7wBBAAUAPgAFAAIAegCHAG4AAAE7ATMABAABeNqdVMsuBFEQPT3tGY+IhYhY9MLCwrQ2CRE77xBhQdjYtJ4xhnnQ0wgrC0ufYeM/RNjaSXyC+Aan6t4ZxmAhnbp9blWduvW43QD68AAXTksngJhisIN+7gxOoQPXFrtYwo3FLRjGs8WtGMS7xW0Yctosbset41ncgVHn3uIuTDtvFndjNzVicQ/xlcW92Eq9WvyIATew+AmBu2Twi4sh9wwLKCBPSSiXyCELjxJyHxJFqOAYF6xHvA6o9XBHySDABCVt0QTGqF2md4V+RcbxME8cky1rqPErKMPHBnU5Ig+b1JdRxRwtRZ46SxypLcs1pkea0uzv1RmCThnZYMko+NN/W+NWbS7C8JVV49QY6Trjp2gFXaU/idYm2Zb4jnFEXQX7Tb0ItSZPvS743lNtzDWv0RLNy3S/oKdFqpEpmP0h64zVN8s1qvezygqa+/Zz72V+CbUzGOdzro9PeyM7slxfUYme/+UlrPVYq8ppz/P0Nf33NWaJ3VnTanJaian/9EsdCf2kU7OME9LP7Bo5cvO+zzXDE4Jf8/6M5WvOeVqLDTGr1KxhhX1cxDonv6g3XWLu0LrHCcs5ib1BAbaYtWS2qpM234fYpni23C9ZM/XvZpJ/ghp/Eyd6g2Odf/ED/fivJHjabdBVbNNxEMDx721d23XuLrhD+2+7bni7rbi7M9gqMLbRUWDYILgGQgJPEOwFCK5BH4DgFpwEnnF4AF6ha3+8cS+f3CV3uTuiCMcfDx7+F99BoiSaaHTEoMeAkVhMxBFPAokkkUwKqaSRTgaZZJFNDrnkkU8BhRRRTCta04a2tKM9HehIJzrTha50ozs9MGNBw4oNOyU4KKWMnvSiN33oSz/648RFORVU4mYAAxnEYIYwlGEMZwQjGcVoxjCWcYxnAhOZxGSmMJVpTGcGM6kSHQdZw1qusosPrGMbm9nDYQ5JDJt4y2p2il4MbGU3G7jBezGylyP84ie/OcAx7nKb48xiNtup5j413OEej3nAQx7xMfS9ZzzhKSfw8oMdvOQ5L/Dxma9sZA5+5jKPWurYRz3zaSBAI0EWsJBFfGIxS2hiKctZxkX208wKVrKKL3zjEq84ySku85p3vJFYMUmcxEuCJEqSJEuKpEqapEuGZHKaM5znAjc5yzlusZ6jksU1rnNFsiVHctkieZIvBVIoRVKs99Y2NfgshmCd32w2V0R0mpUqd2lKq7KsRS3UoLQoNaVVaVPalSVKh7JU+W+eM6JFzbVYTB6/Nxioqa5q9EVKmjuiXemw6SqDgfpwYneXt+h2RfYJqSmtSpsxfK6maX8BUcqkx0u4AMhSWLEBAY5ZuQgACABjILABI0SwAyNwsBdFICBLuAAOUUuwBlNaWLA0G7AoWWBmIIpVWLACJWGwAUVjI2KwAiNEsgsBBiqyDAYGKrIUBgYqWbIEKAlFUkSyDAgHKrEGAUSxJAGIUViwQIhYsQYDRLEmAYhRWLgEAIhYsQYBRFlZWVm4Af+FsASNsQUARAABVL7ENAAA) format('woff');
  font-weight: 700;
  font-style: normal;
}
.alert {
  padding: 7px;
  margin-bottom: 20px;
  border: 1px solid transparent;
  border-radius: 1px;
}
.alert h4 {
  margin-top: 0;
  color: inherit;
}
.alert .alert-link {
  font-weight: 500;
}
.alert > p,
.alert > ul {
  margin-bottom: 0;
}
.alert > p + p {
  margin-top: 5px;
}
.alert-dismissable {
  padding-right: 27px;
}
.alert-dismissable .close {
  position: relative;
  top: -2px;
  right: -21px;
  color: inherit;
}
.alert-success {
  background-color: #ffffff;
  border-color: #5cb75c;
  color: #333333;
}
.alert-success hr {
  border-top-color: #4cad4c;
}
.alert-success .alert-link {
  color: #1a1a1a;
}
.alert-info {
  background-color: #ffffff;
  border-color: #cccccc;
  color: #333333;
}
.alert-info hr {
  border-top-color: #bfbfbf;
}
.alert-info .alert-link {
  color: #1a1a1a;
}
.alert-warning {
  background-color: #ffffff;
  border-color: #eb7720;
  color: #333333;
}
.alert-warning hr {
  border-top-color: #de6a14;
}
.alert-warning .alert-link {
  color: #1a1a1a;
}
.alert-danger {
  background-color: #ffffff;
  border-color: #c90813;
  color: #333333;
}
.alert-danger hr {
  border-top-color: #b00711;
}
.alert-danger .alert-link {
  color: #1a1a1a;
}
.btn {
  display: inline-block;
  margin-bottom: 0;
  font-weight: 600;
  text-align: center;
  vertical-align: middle;
  cursor: pointer;
  background-image: none;
  border: 1px solid transparent;
  white-space: nowrap;
  padding: 2px 6px;
  font-size: 12px;
  line-height: 1.66666667;
  border-radius: 1px;
  -webkit-user-select: none;
  -moz-user-select: none;
  -ms-user-select: none;
  user-select: none;
}
.btn:focus,
.btn:active:focus,
.btn.active:focus {
  outline: thin dotted;
  outline: 5px auto -webkit-focus-ring-color;
  outline-offset: -2px;
}
.btn:hover,
.btn:focus {
  color: #4d5258;
  text-decoration: none;
}
.btn:active,
.btn.active {
  outline: 0;
  background-image: none;
  -webkit-box-shadow: inset 0 3px 5px rgba(0, 0, 0, 0.125);
  box-shadow: inset 0 3px 5px rgba(0, 0, 0, 0.125);
}
.btn.disabled,
.btn[disabled],
fieldset[disabled] .btn {
  cursor: not-allowed;
  pointer-events: none;
  opacity: 0.65;
  filter: alpha(opacity=65);
  -webkit-box-shadow: none;
  box-shadow: none;
}
.btn-default {
  color: #4d5258;
  background-color: #eeeeee;
  border-color: #b7b7b7;
}
.btn-default:hover,
.btn-default:focus,
.btn-default:active,
.btn-default.active,
.open .dropdown-toggle.btn-default {
  color: #4d5258;
  background-color: #dadada;
  border-color: #989898;
}
.btn-default:active,
.btn-default.active,
.open .dropdown-toggle.btn-default {
  background-image: none;
}
.btn-default.disabled,
.btn-default[disabled],
fieldset[disabled] .btn-default,
.btn-default.disabled:hover,
.btn-default[disabled]:hover,
fieldset[disabled] .btn-default:hover,
.btn-default.disabled:focus,
.btn-default[disabled]:focus,
fieldset[disabled] .btn-default:focus,
.btn-default.disabled:active,
.btn-default[disabled]:active,
fieldset[disabled] .btn-default:active,
.btn-default.disabled.active,
.btn-default[disabled].active,
fieldset[disabled] .btn-default.active {
  background-color: #eeeeee;
  border-color: #b7b7b7;
}
.btn-default .badge {
  color: #eeeeee;
  background-color: #4d5258;
}
.btn-primary {
  color: #ffffff;
  background-color: #189ad1;
  border-color: #267da1;
}
.btn-primary:hover,
.btn-primary:focus,
.btn-primary:active,
.btn-primary.active,
.open .dropdown-toggle.btn-primary {
  color: #ffffff;
  background-color: #147fac;
  border-color: #1a576f;
}
.btn-primary:active,
.btn-primary.active,
.open .dropdown-toggle.btn-primary {
  background-image: none;
}
.btn-primary.disabled,
.btn-primary[disabled],
fieldset[disabled] .btn-primary,
.btn-primary.disabled:hover,
.btn-primary[disabled]:hover,
fieldset[disabled] .btn-primary:hover,
.btn-primary.disabled:focus,
.btn-primary[disabled]:focus,
fieldset[disabled] .btn-primary:focus,
.btn-primary.disabled:active,
.btn-primary[disabled]:active,
fieldset[disabled] .btn-primary:active,
.btn-primary.disabled.active,
.btn-primary[disabled].active,
fieldset[disabled] .btn-primary.active {
  background-color: #189ad1;
  border-color: #267da1;
}
.btn-primary .badge {
  color: #189ad1;
  background-color: #ffffff;
}
.btn-success {
  color: #ffffff;
  background-color: #5cb75c;
  border-color: #4cad4c;
}
.btn-success:hover,
.btn-success:focus,
.btn-success:active,
.btn-success.active,
.open .dropdown-toggle.btn-success {
  color: #ffffff;
  background-color: #48a248;
  border-color: #3a833a;
}
.btn-success:active,
.btn-success.active,
.open .dropdown-toggle.btn-success {
  background-image: none;
}
.btn-success.disabled,
.btn-success[disabled],
fieldset[disabled] .btn-success,
.btn-success.disabled:hover,
.btn-success[disabled]:hover,
fieldset[disabled] .btn-success:hover,
.btn-success.disabled:focus,
.btn-success[disabled]:focus,
fieldset[disabled] .btn-success:focus,
.btn-success.disabled:active,
.btn-success[disabled]:active,
fieldset[disabled] .btn-success:active,
.btn-success.disabled.active,
.btn-success[disabled].active,
fieldset[disabled] .btn-success.active {
  background-color: #5cb75c;
  border-color: #4cad4c;
}
.btn-success .badge {
  color: #5cb75c;
  background-color: #ffffff;
}
.btn-info {
  color: #ffffff;
  background-color: #27799c;
  border-color: #226988;
}
.btn-info:hover,
.btn-info:focus,
.btn-info:active,
.btn-info.active,
.open .dropdown-toggle.btn-info {
  color: #ffffff;
  background-color: #1f607b;
  border-color: #164357;
}
.btn-info:active,
.btn-info.active,
.open .dropdown-toggle.btn-info {
  background-image: none;
}
.btn-info.disabled,
.btn-info[disabled],
fieldset[disabled] .btn-info,
.btn-info.disabled:hover,
.btn-info[disabled]:hover,
fieldset[disabled] .btn-info:hover,
.btn-info.disabled:focus,
.btn-info[disabled]:focus,
fieldset[disabled] .btn-info:focus,
.btn-info.disabled:active,
.btn-info[disabled]:active,
fieldset[disabled] .btn-info:active,
.btn-info.disabled.active,
.btn-info[disabled].active,
fieldset[disabled] .btn-info.active {
  background-color: #27799c;
  border-color: #226988;
}
.btn-info .badge {
  color: #27799c;
  background-color: #ffffff;
}
.btn-warning {
  color: #ffffff;
  background-color: #eb7720;
  border-color: #de6a14;
}
.btn-warning:hover,
.btn-warning:focus,
.btn-warning:active,
.btn-warning.active,
.open .dropdown-toggle.btn-warning {
  color: #ffffff;
  background-color: #d06413;
  border-color: #a54f0f;
}
.btn-warning:active,
.btn-warning.active,
.open .dropdown-toggle.btn-warning {
  background-image: none;
}
.btn-warning.disabled,
.btn-warning[disabled],
fieldset[disabled] .btn-warning,
.btn-warning.disabled:hover,
.btn-warning[disabled]:hover,
fieldset[disabled] .btn-warning:hover,
.btn-warning.disabled:focus,
.btn-warning[disabled]:focus,
fieldset[disabled] .btn-warning:focus,
.btn-warning.disabled:active,
.btn-warning[disabled]:active,
fieldset[disabled] .btn-warning:active,
.btn-warning.disabled.active,
.btn-warning[disabled].active,
fieldset[disabled] .btn-warning.active {
  background-color: #eb7720;
  border-color: #de6a14;
}
.btn-warning .badge {
  color: #eb7720;
  background-color: #ffffff;
}
.btn-danger {
  color: #ffffff;
  background-color: #ab070f;
  border-color: #781919;
}
.btn-danger:hover,
.btn-danger:focus,
.btn-danger:active,
.btn-danger.active,
.open .dropdown-toggle.btn-danger {
  color: #ffffff;
  background-color: #84050c;
  border-color: #450e0e;
}
.btn-danger:active,
.btn-danger.active,
.open .dropdown-toggle.btn-danger {
  background-image: none;
}
.btn-danger.disabled,
.btn-danger[disabled],
fieldset[disabled] .btn-danger,
.btn-danger.disabled:hover,
.btn-danger[disabled]:hover,
fieldset[disabled] .btn-danger:hover,
.btn-danger.disabled:focus,
.btn-danger[disabled]:focus,
fieldset[disabled] .btn-danger:focus,
.btn-danger.disabled:active,
.btn-danger[disabled]:active,
fieldset[disabled] .btn-danger:active,
.btn-danger.disabled.active,
.btn-danger[disabled].active,
fieldset[disabled] .btn-danger.active {
  background-color: #ab070f;
  border-color: #781919;
}
.btn-danger .badge {
  color: #ab070f;
  background-color: #ffffff;
}
.btn-link {
  color: #0099d3;
  font-weight: normal;
  cursor: pointer;
  border-radius: 0;
}
.btn-link,
.btn-link:active,
.btn-link[disabled],
fieldset[disabled] .btn-link {
  background-color: transparent;
  -webkit-box-shadow: none;
  box-shadow: none;
}
.btn-link,
.btn-link:hover,
.btn-link:focus,
.btn-link:active {
  border-color: transparent;
}
.btn-link:hover,
.btn-link:focus {
  color: #00618a;
  text-decoration: underline;
  background-color: transparent;
}
.btn-link[disabled]:hover,
fieldset[disabled] .btn-link:hover,
.btn-link[disabled]:focus,
fieldset[disabled] .btn-link:focus {
  color: #999999;
  text-decoration: none;
}
.btn-lg {
  padding: 6px 10px;
  font-size: 14px;
  line-height: 1.33;
  border-radius: 1px;
}
.btn-sm {
  padding: 2px 6px;
  font-size: 11px;
  line-height: 1.5;
  border-radius: 1px;
}
.btn-xs {
  padding: 1px 5px;
  font-size: 11px;
  line-height: 1.5;
  border-radius: 1px;
}
.btn-block {
  display: block;
  width: 100%;
  padding-left: 0;
  padding-right: 0;
}
.btn-block + .btn-block {
  margin-top: 5px;
}
.fade {
  opacity: 0;
  -webkit-transition: opacity 0.15s linear;
  transition: opacity 0.15s linear;
}
.fade.in {
  opacity: 1;
}
.collapse {
  display: none;
}
.collapse.in {
  display: block;
}
.collapsing {
  position: relative;
  height: 0;
  overflow: hidden;
  -webkit-transition: height 0.35s ease;
  transition: height 0.35s ease;
}
.container {
  margin-right: auto;
  margin-left: auto;
  padding-left: 20px;
  padding-right: 20px;
}
@media (min-width: 768px) {
  .container {
    width: 760px;
  }
}
@media (min-width: 992px) {
  .container {
    width: 980px;
  }
}
@media (min-width: 1200px) {
  .container {
    width: 1180px;
  }
}
.container-fluid {
  margin-right: auto;
  margin-left: auto;
  padding-left: 20px;
  padding-right: 20px;
}
.row {
  margin-left: -20px;
  margin-right: -20px;
}
.col-xs-1, .col-sm-1, .col-md-1, .col-lg-1, .col-xs-2, .col-sm-2, .col-md-2, .col-lg-2, .col-xs-3, .col-sm-3, .col-md-3, .col-lg-3, .col-xs-4, .col-sm-4, .col-md-4, .col-lg-4, .col-xs-5, .col-sm-5, .col-md-5, .col-lg-5, .col-xs-6, .col-sm-6, .col-md-6, .col-lg-6, .col-xs-7, .col-sm-7, .col-md-7, .col-lg-7, .col-xs-8, .col-sm-8, .col-md-8, .col-lg-8, .col-xs-9, .col-sm-9, .col-md-9, .col-lg-9, .col-xs-10, .col-sm-10, .col-md-10, .col-lg-10, .col-xs-11, .col-sm-11, .col-md-11, .col-lg-11, .col-xs-12, .col-sm-12, .col-md-12, .col-lg-12 {
  position: relative;
  min-height: 1px;
  padding-left: 20px;
  padding-right: 20px;
}
.col-xs-1, .col-xs-2, .col-xs-3, .col-xs-4, .col-xs-5, .col-xs-6, .col-xs-7, .col-xs-8, .col-xs-9, .col-xs-10, .col-xs-11, .col-xs-12 {
  float: left;
}
.col-xs-12 {
  width: 100%;
}
.col-xs-11 {
  width: 91.66666666666666%;
}
.col-xs-10 {
  width: 83.33333333333334%;
}
.col-xs-9 {
  width: 75%;
}
.col-xs-8 {
  width: 66.66666666666666%;
}
.col-xs-7 {
  width: 58.333333333333336%;
}
.col-xs-6 {
  width: 50%;
}
.col-xs-5 {
  width: 41.66666666666667%;
}
.col-xs-4 {
  width: 33.33333333333333%;
}
.col-xs-3 {
  width: 25%;
}
.col-xs-2 {
  width: 16.666666666666664%;
}
.col-xs-1 {
  width: 8.333333333333332%;
}
.col-xs-pull-12 {
  right: 100%;
}
.col-xs-pull-11 {
  right: 91.66666666666666%;
}
.col-xs-pull-10 {
  right: 83.33333333333334%;
}
.col-xs-pull-9 {
  right: 75%;
}
.col-xs-pull-8 {
  right: 66.66666666666666%;
}
.col-xs-pull-7 {
  right: 58.333333333333336%;
}
.col-xs-pull-6 {
  right: 50%;
}
.col-xs-pull-5 {
  right: 41.66666666666667%;
}
.col-xs-pull-4 {
  right: 33.33333333333333%;
}
.col-xs-pull-3 {
  right: 25%;
}
.col-xs-pull-2 {
  right: 16.666666666666664%;
}
.col-xs-pull-1 {
  right: 8.333333333333332%;
}
.col-xs-pull-0 {
  right: 0%;
}
.col-xs-push-12 {
  left: 100%;
}
.col-xs-push-11 {
  left: 91.66666666666666%;
}
.col-xs-push-10 {
  left: 83.33333333333334%;
}
.col-xs-push-9 {
  left: 75%;
}
.col-xs-push-8 {
  left: 66.66666666666666%;
}
.col-xs-push-7 {
  left: 58.333333333333336%;
}
.col-xs-push-6 {
  left: 50%;
}
.col-xs-push-5 {
  left: 41.66666666666667%;
}
.col-xs-push-4 {
  left: 33.33333333333333%;
}
.col-xs-push-3 {
  left: 25%;
}
.col-xs-push-2 {
  left: 16.666666666666664%;
}
.col-xs-push-1 {
  left: 8.333333333333332%;
}
.col-xs-push-0 {
  left: 0%;
}
.col-xs-offset-12 {
  margin-left: 100%;
}
.col-xs-offset-11 {
  margin-left: 91.66666666666666%;
}
.col-xs-offset-10 {
  margin-left: 83.33333333333334%;
}
.col-xs-offset-9 {
  margin-left: 75%;
}
.col-xs-offset-8 {
  margin-left: 66.66666666666666%;
}
.col-xs-offset-7 {
  margin-left: 58.333333333333336%;
}
.col-xs-offset-6 {
  margin-left: 50%;
}
.col-xs-offset-5 {
  margin-left: 41.66666666666667%;
}
.col-xs-offset-4 {
  margin-left: 33.33333333333333%;
}
.col-xs-offset-3 {
  margin-left: 25%;
}
.col-xs-offset-2 {
  margin-left: 16.666666666666664%;
}
.col-xs-offset-1 {
  margin-left: 8.333333333333332%;
}
.col-xs-offset-0 {
  margin-left: 0%;
}
@media (min-width: 768px) {
  .col-sm-1, .col-sm-2, .col-sm-3, .col-sm-4, .col-sm-5, .col-sm-6, .col-sm-7, .col-sm-8, .col-sm-9, .col-sm-10, .col-sm-11, .col-sm-12 {
    float: left;
  }
  .col-sm-12 {
    width: 100%;
  }
  .col-sm-11 {
    width: 91.66666666666666%;
  }
  .col-sm-10 {
    width: 83.33333333333334%;
  }
  .col-sm-9 {
    width: 75%;
  }
  .col-sm-8 {
    width: 66.66666666666666%;
  }
  .col-sm-7 {
    width: 58.333333333333336%;
  }
  .col-sm-6 {
    width: 50%;
  }
  .col-sm-5 {
    width: 41.66666666666667%;
  }
  .col-sm-4 {
    width: 33.33333333333333%;
  }
  .col-sm-3 {
    width: 25%;
  }
  .col-sm-2 {
    width: 16.666666666666664%;
  }
  .col-sm-1 {
    width: 8.333333333333332%;
  }
  .col-sm-pull-12 {
    right: 100%;
  }
  .col-sm-pull-11 {
    right: 91.66666666666666%;
  }
  .col-sm-pull-10 {
    right: 83.33333333333334%;
  }
  .col-sm-pull-9 {
    right: 75%;
  }
  .col-sm-pull-8 {
    right: 66.66666666666666%;
  }
  .col-sm-pull-7 {
    right: 58.333333333333336%;
  }
  .col-sm-pull-6 {
    right: 50%;
  }
  .col-sm-pull-5 {
    right: 41.66666666666667%;
  }
  .col-sm-pull-4 {
    right: 33.33333333333333%;
  }
  .col-sm-pull-3 {
    right: 25%;
  }
  .col-sm-pull-2 {
    right: 16.666666666666664%;
  }
  .col-sm-pull-1 {
    right: 8.333333333333332%;
  }
  .col-sm-pull-0 {
    right: 0%;
  }
  .col-sm-push-12 {
    left: 100%;
  }
  .col-sm-push-11 {
    left: 91.66666666666666%;
  }
  .col-sm-push-10 {
    left: 83.33333333333334%;
  }
  .col-sm-push-9 {
    left: 75%;
  }
  .col-sm-push-8 {
    left: 66.66666666666666%;
  }
  .col-sm-push-7 {
    left: 58.333333333333336%;
  }
  .col-sm-push-6 {
    left: 50%;
  }
  .col-sm-push-5 {
    left: 41.66666666666667%;
  }
  .col-sm-push-4 {
    left: 33.33333333333333%;
  }
  .col-sm-push-3 {
    left: 25%;
  }
  .col-sm-push-2 {
    left: 16.666666666666664%;
  }
  .col-sm-push-1 {
    left: 8.333333333333332%;
  }
  .col-sm-push-0 {
    left: 0%;
  }
  .col-sm-offset-12 {
    margin-left: 100%;
  }
  .col-sm-offset-11 {
    margin-left: 91.66666666666666%;
  }
  .col-sm-offset-10 {
    margin-left: 83.33333333333334%;
  }
  .col-sm-offset-9 {
    margin-left: 75%;
  }
  .col-sm-offset-8 {
    margin-left: 66.66666666666666%;
  }
  .col-sm-offset-7 {
    margin-left: 58.333333333333336%;
  }
  .col-sm-offset-6 {
    margin-left: 50%;
  }
  .col-sm-offset-5 {
    margin-left: 41.66666666666667%;
  }
  .col-sm-offset-4 {
    margin-left: 33.33333333333333%;
  }
  .col-sm-offset-3 {
    margin-left: 25%;
  }
  .col-sm-offset-2 {
    margin-left: 16.666666666666664%;
  }
  .col-sm-offset-1 {
    margin-left: 8.333333333333332%;
  }
  .col-sm-offset-0 {
    margin-left: 0%;
  }
}
@media (min-width: 992px) {
  .col-md-1, .col-md-2, .col-md-3, .col-md-4, .col-md-5, .col-md-6, .col-md-7, .col-md-8, .col-md-9, .col-md-10, .col-md-11, .col-md-12 {
    float: left;
  }
  .col-md-12 {
    width: 100%;
  }
  .col-md-11 {
    width: 91.66666666666666%;
  }
  .col-md-10 {
    width: 83.33333333333334%;
  }
  .col-md-9 {
    width: 75%;
  }
  .col-md-8 {
    width: 66.66666666666666%;
  }
  .col-md-7 {
    width: 58.333333333333336%;
  }
  .col-md-6 {
    width: 50%;
  }
  .col-md-5 {
    width: 41.66666666666667%;
  }
  .col-md-4 {
    width: 33.33333333333333%;
  }
  .col-md-3 {
    width: 25%;
  }
  .col-md-2 {
    width: 16.666666666666664%;
  }
  .col-md-1 {
    width: 8.333333333333332%;
  }
  .col-md-pull-12 {
    right: 100%;
  }
  .col-md-pull-11 {
    right: 91.66666666666666%;
  }
  .col-md-pull-10 {
    right: 83.33333333333334%;
  }
  .col-md-pull-9 {
    right: 75%;
  }
  .col-md-pull-8 {
    right: 66.66666666666666%;
  }
  .col-md-pull-7 {
    right: 58.333333333333336%;
  }
  .col-md-pull-6 {
    right: 50%;
  }
  .col-md-pull-5 {
    right: 41.66666666666667%;
  }
  .col-md-pull-4 {
    right: 33.33333333333333%;
  }
  .col-md-pull-3 {
    right: 25%;
  }
  .col-md-pull-2 {
    right: 16.666666666666664%;
  }
  .col-md-pull-1 {
    right: 8.333333333333332%;
  }
  .col-md-pull-0 {
    right: 0%;
  }
  .col-md-push-12 {
    left: 100%;
  }
  .col-md-push-11 {
    left: 91.66666666666666%;
  }
  .col-md-push-10 {
    left: 83.33333333333334%;
  }
  .col-md-push-9 {
    left: 75%;
  }
  .col-md-push-8 {
    left: 66.66666666666666%;
  }
  .col-md-push-7 {
    left: 58.333333333333336%;
  }
  .col-md-push-6 {
    left: 50%;
  }
  .col-md-push-5 {
    left: 41.66666666666667%;
  }
  .col-md-push-4 {
    left: 33.33333333333333%;
  }
  .col-md-push-3 {
    left: 25%;
  }
  .col-md-push-2 {
    left: 16.666666666666664%;
  }
  .col-md-push-1 {
    left: 8.333333333333332%;
  }
  .col-md-push-0 {
    left: 0%;
  }
  .col-md-offset-12 {
    margin-left: 100%;
  }
  .col-md-offset-11 {
    margin-left: 91.66666666666666%;
  }
  .col-md-offset-10 {
    margin-left: 83.33333333333334%;
  }
  .col-md-offset-9 {
    margin-left: 75%;
  }
  .col-md-offset-8 {
    margin-left: 66.66666666666666%;
  }
  .col-md-offset-7 {
    margin-left: 58.333333333333336%;
  }
  .col-md-offset-6 {
    margin-left: 50%;
  }
  .col-md-offset-5 {
    margin-left: 41.66666666666667%;
  }
  .col-md-offset-4 {
    margin-left: 33.33333333333333%;
  }
  .col-md-offset-3 {
    margin-left: 25%;
  }
  .col-md-offset-2 {
    margin-left: 16.666666666666664%;
  }
  .col-md-offset-1 {
    margin-left: 8.333333333333332%;
  }
  .col-md-offset-0 {
    margin-left: 0%;
  }
}
@media (min-width: 1200px) {
  .col-lg-1, .col-lg-2, .col-lg-3, .col-lg-4, .col-lg-5, .col-lg-6, .col-lg-7, .col-lg-8, .col-lg-9, .col-lg-10, .col-lg-11, .col-lg-12 {
    float: left;
  }
  .col-lg-12 {
    width: 100%;
  }
  .col-lg-11 {
    width: 91.66666666666666%;
  }
  .col-lg-10 {
    width: 83.33333333333334%;
  }
  .col-lg-9 {
    width: 75%;
  }
  .col-lg-8 {
    width: 66.66666666666666%;
  }
  .col-lg-7 {
    width: 58.333333333333336%;
  }
  .col-lg-6 {
    width: 50%;
  }
  .col-lg-5 {
    width: 41.66666666666667%;
  }
  .col-lg-4 {
    width: 33.33333333333333%;
  }
  .col-lg-3 {
    width: 25%;
  }
  .col-lg-2 {
    width: 16.666666666666664%;
  }
  .col-lg-1 {
    width: 8.333333333333332%;
  }
  .col-lg-pull-12 {
    right: 100%;
  }
  .col-lg-pull-11 {
    right: 91.66666666666666%;
  }
  .col-lg-pull-10 {
    right: 83.33333333333334%;
  }
  .col-lg-pull-9 {
    right: 75%;
  }
  .col-lg-pull-8 {
    right: 66.66666666666666%;
  }
  .col-lg-pull-7 {
    right: 58.333333333333336%;
  }
  .col-lg-pull-6 {
    right: 50%;
  }
  .col-lg-pull-5 {
    right: 41.66666666666667%;
  }
  .col-lg-pull-4 {
    right: 33.33333333333333%;
  }
  .col-lg-pull-3 {
    right: 25%;
  }
  .col-lg-pull-2 {
    right: 16.666666666666664%;
  }
  .col-lg-pull-1 {
    right: 8.333333333333332%;
  }
  .col-lg-pull-0 {
    right: 0%;
  }
  .col-lg-push-12 {
    left: 100%;
  }
  .col-lg-push-11 {
    left: 91.66666666666666%;
  }
  .col-lg-push-10 {
    left: 83.33333333333334%;
  }
  .col-lg-push-9 {
    left: 75%;
  }
  .col-lg-push-8 {
    left: 66.66666666666666%;
  }
  .col-lg-push-7 {
    left: 58.333333333333336%;
  }
  .col-lg-push-6 {
    left: 50%;
  }
  .col-lg-push-5 {
    left: 41.66666666666667%;
  }
  .col-lg-push-4 {
    left: 33.33333333333333%;
  }
  .col-lg-push-3 {
    left: 25%;
  }
  .col-lg-push-2 {
    left: 16.666666666666664%;
  }
  .col-lg-push-1 {
    left: 8.333333333333332%;
  }
  .col-lg-push-0 {
    left: 0%;
  }
  .col-lg-offset-12 {
    margin-left: 100%;
  }
  .col-lg-offset-11 {
    margin-left: 91.66666666666666%;
  }
  .col-lg-offset-10 {
    margin-left: 83.33333333333334%;
  }
  .col-lg-offset-9 {
    margin-left: 75%;
  }
  .col-lg-offset-8 {
    margin-left: 66.66666666666666%;
  }
  .col-lg-offset-7 {
    margin-left: 58.333333333333336%;
  }
  .col-lg-offset-6 {
    margin-left: 50%;
  }
  .col-lg-offset-5 {
    margin-left: 41.66666666666667%;
  }
  .col-lg-offset-4 {
    margin-left: 33.33333333333333%;
  }
  .col-lg-offset-3 {
    margin-left: 25%;
  }
  .col-lg-offset-2 {
    margin-left: 16.666666666666664%;
  }
  .col-lg-offset-1 {
    margin-left: 8.333333333333332%;
  }
  .col-lg-offset-0 {
    margin-left: 0%;
  }
}
/*! normalize.css v3.0.0 | MIT License | git.io/normalize */
html {
  font-family: sans-serif;
  -ms-text-size-adjust: 100%;
  -webkit-text-size-adjust: 100%;
}
body {
  margin: 0;
}
article,
aside,
details,
figcaption,
figure,
footer,
header,
hgroup,
main,
nav,
section,
summary {
  display: block;
}
audio,
canvas,
progress,
video {
  display: inline-block;
  vertical-align: baseline;
}
audio:not([controls]) {
  display: none;
  height: 0;
}
[hidden],
template {
  display: none;
}
a {
  background: transparent;
}
a:active,
a:hover {
  outline: 0;
}
abbr[title] {
  border-bottom: 1px dotted;
}
b,
strong {
  font-weight: bold;
}
dfn {
  font-style: italic;
}
h1 {
  font-size: 2em;
  margin: 0.67em 0;
}
mark {
  background: #ff0;
  color: #000;
}
small {
  font-size: 80%;
}
sub,
sup {
  font-size: 75%;
  line-height: 0;
  position: relative;
  vertical-align: baseline;
}
sup {
  top: -0.5em;
}
sub {
  bottom: -0.25em;
}
img {
  border: 0;
}
svg:not(:root) {
  overflow: hidden;
}
figure {
  margin: 1em 40px;
}
hr {
  -moz-box-sizing: content-box;
  box-sizing: content-box;
  height: 0;
}
pre {
  overflow: auto;
}
code,
kbd,
pre,
samp {
  font-family: monospace, monospace;
  font-size: 1em;
}
button,
input,
optgroup,
select,
textarea {
  color: inherit;
  font: inherit;
  margin: 0;
}
button {
  overflow: visible;
}
button,
select {
  text-transform: none;
}
button,
html input[type="button"],
input[type="reset"],
input[type="submit"] {
  -webkit-appearance: button;
  cursor: pointer;
}
button[disabled],
html input[disabled] {
  cursor: default;
}
button::-moz-focus-inner,
input::-moz-focus-inner {
  border: 0;
  padding: 0;
}
input {
  line-height: normal;
}
input[type="checkbox"],
input[type="radio"] {
  box-sizing: border-box;
  padding: 0;
}
input[type="number"]::-webkit-inner-spin-button,
input[type="number"]::-webkit-outer-spin-button {
  height: auto;
}
input[type="search"] {
  -webkit-appearance: textfield;
  -moz-box-sizing: content-box;
  -webkit-box-sizing: content-box;
  box-sizing: content-box;
}
input[type="search"]::-webkit-search-cancel-button,
input[type="search"]::-webkit-search-decoration {
  -webkit-appearance: none;
}
fieldset {
  border: 1px solid #c0c0c0;
  margin: 0 2px;
  padding: 0.35em 0.625em 0.75em;
}
legend {
  border: 0;
  padding: 0;
}
textarea {
  overflow: auto;
}
optgroup {
  font-weight: bold;
}
table {
  border-collapse: collapse;
  border-spacing: 0;
}
td,
th {
  padding: 0;
}
@-ms-viewport {
  width: device-width;
}
.visible-xs,
.visible-sm,
.visible-md,
.visible-lg {
  display: none !important;
}
@media (max-width: 767px) {
  .visible-xs {
    display: block !important;
  }
  table.visible-xs {
    display: table;
  }
  tr.visible-xs {
    display: table-row !important;
  }
  th.visible-xs,
  td.visible-xs {
    display: table-cell !important;
  }
}
@media (min-width: 768px) and (max-width: 991px) {
  .visible-sm {
    display: block !important;
  }
  table.visible-sm {
    display: table;
  }
  tr.visible-sm {
    display: table-row !important;
  }
  th.visible-sm,
  td.visible-sm {
    display: table-cell !important;
  }
}
@media (min-width: 992px) and (max-width: 1199px) {
  .visible-md {
    display: block !important;
  }
  table.visible-md {
    display: table;
  }
  tr.visible-md {
    display: table-row !important;
  }
  th.visible-md,
  td.visible-md {
    display: table-cell !important;
  }
}
@media (min-width: 1200px) {
  .visible-lg {
    display: block !important;
  }
  table.visible-lg {
    display: table;
  }
  tr.visible-lg {
    display: table-row !important;
  }
  th.visible-lg,
  td.visible-lg {
    display: table-cell !important;
  }
}
@media (max-width: 767px) {
  .hidden-xs {
    display: none !important;
  }
}
@media (min-width: 768px) and (max-width: 991px) {
  .hidden-sm {
    display: none !important;
  }
}
@media (min-width: 992px) and (max-width: 1199px) {
  .hidden-md {
    display: none !important;
  }
}
@media (min-width: 1200px) {
  .hidden-lg {
    display: none !important;
  }
}
.visible-print {
  display: none !important;
}
@media print {
  .visible-print {
    display: block !important;
  }
  table.visible-print {
    display: table;
  }
  tr.visible-print {
    display: table-row !important;
  }
  th.visible-print,
  td.visible-print {
    display: table-cell !important;
  }
}
@media print {
  .hidden-print {
    display: none !important;
  }
}
* {
  -webkit-box-sizing: border-box;
  -moz-box-sizing: border-box;
  box-sizing: border-box;
}
*:before,
*:after {
  -webkit-box-sizing: border-box;
  -moz-box-sizing: border-box;
  box-sizing: border-box;
}
html {
  font-size: 62.5%;
  -webkit-tap-highlight-color: rgba(0, 0, 0, 0);
}
body {
  font-family: "Open Sans", Helvetica, Arial, sans-serif;
  font-size: 12px;
  line-height: 1.66666667;
  color: #333333;
  background-color: #ffffff;
}
input,
button,
select,
textarea {
  font-family: inherit;
  font-size: inherit;
  line-height: inherit;
}
a {
  color: #0099d3;
  text-decoration: none;
}
a:hover,
a:focus {
  color: #00618a;
  text-decoration: underline;
}
a:focus {
  outline: thin dotted;
  outline: 5px auto -webkit-focus-ring-color;
  outline-offset: -2px;
}
figure {
  margin: 0;
}
img {
  vertical-align: middle;
}
.img-responsive {
  display: block;
  max-width: 100%;
  height: auto;
}
.img-rounded {
  border-radius: 1px;
}
.img-thumbnail {
  padding: 4px;
  line-height: 1.66666667;
  background-color: #ffffff;
  border: 1px solid #dddddd;
  border-radius: 1px;
  -webkit-transition: all 0.2s ease-in-out;
  transition: all 0.2s ease-in-out;
  display: inline-block;
  max-width: 100%;
  height: auto;
}
.img-circle {
  border-radius: 50%;
}
hr {
  margin-top: 20px;
  margin-bottom: 20px;
  border: 0;
  border-top: 1px solid #eeeeee;
}
.sr-only {
  position: absolute;
  width: 1px;
  height: 1px;
  margin: -1px;
  padding: 0;
  overflow: hidden;
  clip: rect(0, 0, 0, 0);
  border: 0;
}
.clearfix:before,
.clearfix:after,
.form-horizontal .form-group:before,
.form-horizontal .form-group:after,
.container:before,
.container:after,
.container-fluid:before,
.container-fluid:after,
.row:before,
.row:after {
  content: " ";
  display: table;
}
.clearfix:after,
.form-horizontal .form-group:after,
.container:after,
.container-fluid:after,
.row:after {
  clear: both;
}
.center-block {
  display: block;
  margin-left: auto;
  margin-right: auto;
}
.pull-right {
  float: right !important;
}
.pull-left {
  float: left !important;
}
.hide {
  display: none !important;
}
.show {
  display: block !important;
}
.invisible {
  visibility: hidden;
}
.text-hide {
  font: 0/0 a;
  color: transparent;
  text-shadow: none;
  background-color: transparent;
  border: 0;
}
.hidden {
  display: none !important;
  visibility: hidden !important;
}
.affix {
  position: fixed;
}
/* PatternFly specific */
/* Bootstrap overrides */
/* PatternFly-specific variables based on Bootstrap overides */
.alert {
  border-width: 2px;
  padding-left: 34px;
  position: relative;
}
.alert .alert-link {
  color: #0099d3;
}
.alert .alert-link:hover {
  color: #00618a;
}
.alert > .pficon,
.alert > .pficon-layered {
  font-size: 20px;
  position: absolute;
  left: 7px;
  top: 7px;
}
.alert .pficon-info {
  color: #72767b;
}
.alert-dismissable .close {
  right: -16px;
  top: 1px;
}
/* Bootstrap overrides */
/* PatternFly-specific */
.login-pf {
  height: 100%;
}
.login-pf #brand {
  position: relative;
  top: -70px;
}
.login-pf #brand img {
  display: block;
  height: 18px;
  margin: 0 auto;
  max-width: 100%;
}
@media (min-width: 768px) {
  .login-pf #brand img {
    margin: 0;
    text-align: left;
  }
}
.login-pf #badge {
  display: block;
  margin: 20px auto 70px;
  position: relative;
  text-align: center;
}
@media (min-width: 768px) {
  .login-pf #badge {
    float: right;
    margin-right: 64px;
    margin-top: 50px;
  }
}
.login-pf body {
  background: #1a1a1a url("../img/bg-login.png") repeat-x 50% 0;
  background-size: auto;
}
@media (min-width: 768px) {
  .login-pf body {
    background-size: 100% auto;
  }
}
.login-pf .container {
  background-color: transparent;
  clear: right;
  color: #fff;
  padding-bottom: 40px;
  padding-top: 20px;
  width: auto;
}
@media (min-width: 768px) {
  .login-pf .container {
    bottom: 13%;
    padding-left: 80px;
    position: absolute;
    width: 100%;
  }
}
.login-pf .container [class^='alert'] {
  background: transparent;
  color: #fff;
}
.login-pf .container .details p:first-child {
  border-top: 1px solid #474747;
  padding-top: 25px;
  margin-top: 25px;
}
@media (min-width: 768px) {
  .login-pf .container .details {
    border-left: 1px solid #474747;
    padding-left: 40px;
  }
  .login-pf .container .details p:first-child {
    border-top: 0;
    padding-top: 0;
    margin-top: 0;
  }
}
.login-pf .container .details p {
  margin-bottom: 2px;
}
.login-pf .container .form-horizontal .control-label {
  font-size: 13px;
  font-weight: 400;
  text-align: left;
}
.login-pf .container .form-horizontal .form-group:last-child,
.login-pf .container .form-horizontal .form-group:last-child .help-block:last-child {
  margin-bottom: 0;
}
.login-pf .container .help-block {
  color: #fff;
}
@media (min-width: 768px) {
  .login-pf .container .login {
    padding-right: 40px;
  }
}
.login-pf .container .submit {
  text-align: right;
}
.ie8.login-pf #badge {
  background: url('../img/logo.png') no-repeat;
  height: 44px;
  width: 137px;
}
.ie8.login-pf #badge img {
  width: 0;
}
.ie8.login-pf #brand {
  background: url('../img/brand-lg.png') no-repeat center;
  background-size: cover auto;
}
@media (min-width: 768px) {
  .ie8.login-pf #brand {
    background-position: 0 0;
  }
}
.ie8.login-pf #brand img {
  width: 0;
}
/* Bootstrap overrides */
/* RCUE-specific */
/* components/patternfly/less/login.less minus the font @import */
@font-face {
  font-family: 'PatternFlyIcons-webfont';
  src: url(data:application/font-woff;charset=utf-8;base64,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) format('woff');
  font-weight: normal;
  font-style: normal;
}
[class*="-exclamation"] {
  color: #fff;
}
[class^="pficon-"],
[class*=" pficon-"] {
  display: inline-block;
  font-family: 'PatternFlyIcons-webfont';
  font-style: normal;
  font-variant: normal;
  font-weight: normal;
  line-height: 1;
  speak: none;
  text-transform: none;
  /* Better Font Rendering =========== */
  -webkit-font-smoothing: antialiased;
  -moz-osx-font-smoothing: grayscale;
}
.pficon-layered {
  position: relative;
}
.pficon-layered .pficon:first-child {
  position: absolute;
  z-index: 1;
}
.pficon-layered .pficon:first-child + .pficon {
  position: relative;
  z-index: 2;
}
.pficon-warning-exclamation:before {
  content: "\e60d";
}
.pficon-screen:before {
  content: "\e600";
}
.pficon-save:before {
  content: "\e601";
}
.pficon-ok:before {
  color: #57a81c;
  content: "\e602";
}
.pficon-messages:before {
  content: "\e603";
}
.pficon-info:before {
  content: "\e604";
}
.pficon-help:before {
  content: "\e605";
}
.pficon-folder-open:before {
  content: "\e606";
}
.pficon-folder-close:before {
  content: "\e607";
}
.pficon-error-exclamation:before {
  content: "\e608";
}
.pficon-error-octagon:before {
  color: #c90813;
  content: "\e609";
}
.pficon-edit:before {
  content: "\e60a";
}
.pficon-close:before {
  content: "\e60b";
}
.pficon-warning-triangle:before {
  color: #eb7720;
  content: "\e60c";
}
.pficon-user:before {
  content: "\e60e";
}
.pficon-users:before {
  content: "\e60f";
}
.pficon-settings:before {
  content: "\e610";
}
.pficon-delete:before {
  content: "\e611";
}
.pficon-print:before {
  content: "\e612";
}
.pficon-refresh:before {
  content: "\e613";
}
.pficon-running:before {
  content: "\e614";
}
.pficon-import:before {
  content: "\e615";
}
.pficon-export:before {
  content: "\e616";
}
.pficon-history:before {
  content: "\e617";
}
.pficon-home:before {
  content: "\e618";
}
.pficon-remove:before {
  content: "\e619";
}
.pficon-add:before {
  content: "\e61a";
}
/* RCUE overrides */
.login-pf {
  background-color: #1a1a1a;
}
@media (min-width: 768px) {
  .login-pf {
    background-image: url("../img/bg-login-2.png");
    background-position: 100% 100%;
    background-repeat: no-repeat;
    background-size: 30%;
  }
}
@media (min-width: 992px) {
  .login-pf {
    background-size: auto;
  }
}
.login-pf #badge {
  margin-bottom: 50px;
}
.login-pf body {
  background: transparent;
}
@media (min-width: 768px) {
  .login-pf body {
    background-image: url("../img/bg-login.png");
    background-repeat: no-repeat;
    background-size: 30%;
    height: 100%;
  }
}
@media (min-width: 992px) {
  .login-pf body {
    background-size: auto;
  }
}
.login-pf #brand {
  top: -30px;
}
@media (min-width: 768px) {
  .login-pf #brand + .alert {
    margin-top: -20px;
  }
}
.login-pf .container {
  padding-top: 0;
}
@media (min-width: 992px) {
  .login-pf .container {
    bottom: 20%;
    padding-right: 120px;
  }
}
@media (max-width: 767px) {
  .login-pf #badge {
    margin-bottom: 20px;
  }
  .login-pf #brand {
    display: none;
  }
  .login {
    text-align: center;
  }
}
@media (min-width: 768px) {
  .login-pf {
    background-image: none;
  }
  .login-pf body {
    background-image: none;
  }
}
      </style>
  </head>
  <body>
    <span id="badge">
      <img src="data:image/svg+xml;base64,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" alt="OpenShift Container Platform logo" />
    </span>
    <div class="container">
      <div class="row">
        <div class="col-sm-12">
          <div id="brand">
            <img src="data:image/svg+xml;base64,PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iVVRGLTgiIHN0YW5kYWxvbmU9Im5vIj8+PHN2ZyB3aWR0aD0iMzIwcHgiIGhlaWdodD0iMTRweCIgdmlld0JveD0iMCAwIDMyMCAxNCIgdmVyc2lvbj0iMS4xIiB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHhtbG5zOnhsaW5rPSJodHRwOi8vd3d3LnczLm9yZy8xOTk5L3hsaW5rIj4gICAgICAgIDx0aXRsZT5Hcm91cDwvdGl0bGU+ICAgIDxkZXNjPkNyZWF0ZWQgd2l0aCBTa2V0Y2guPC9kZXNjPiAgICA8ZGVmcz48L2RlZnM+ICAgIDxnIGlkPSJQYWdlLTEiIHN0cm9rZT0ibm9uZSIgc3Ryb2tlLXdpZHRoPSIxIiBmaWxsPSJub25lIiBmaWxsLXJ1bGU9ImV2ZW5vZGQiPiAgICAgICAgPGcgaWQ9Ikdyb3VwIiBmaWxsPSIjRkZGRkZGIj4gICAgICAgICAgICA8cGF0aCBkPSJNMTIuNjY2OTkyMiw2LjkxNTUyNzM0IEMxMi42NjY5OTIyLDguOTU3NTI5NzQgMTIuMTY2MTgzNCwxMC41MDk3NjAzIDExLjE2NDU1MDgsMTEuNTcyMjY1NiBDMTAuMTYyOTE4MiwxMi42MzQ3NzA5IDguNjk2NDYxNDgsMTMuMTY2MDE1NiA2Ljc2NTEzNjcyLDEzLjE2NjAxNTYgQzQuODYxNDgxMzcsMTMuMTY2MDE1NiAzLjQwMTk0MTkzLDEyLjYzMjAwNCAyLjM4NjQ3NDYxLDExLjU2Mzk2NDggQzEuMzcxMDA3MjksMTAuNDk1OTI1NiAwLjg2MzI4MTI1LDguOTQwOTI4MTggMC44NjMyODEyNSw2Ljg5ODkyNTc4IEMwLjg2MzI4MTI1LDQuODc5MDU4OTEgMS4zNjgyNDAzOSwzLjMzNjUxMjQ5IDIuMzc4MTczODMsMi4yNzEyNDAyMyBDMy4zODgxMDcyNiwxLjIwNTk2Nzk4IDQuODU1OTQ3NCwwLjY3MzMzOTg0NCA2Ljc4MTczODI4LDAuNjczMzM5ODQ0IEM4LjcxMzA2MzA0LDAuNjczMzM5ODQ0IDEwLjE3Njc1MjgsMS4yMDE4MTc2MyAxMS4xNzI4NTE2LDIuMjU4Nzg5MDYgQzEyLjE2ODk1MDMsMy4zMTU3NjA0OSAxMi42NjY5OTIyLDQuODY3OTkxMDYgMTIuNjY2OTkyMiw2LjkxNTUyNzM0IEwxMi42NjY5OTIyLDYuOTE1NTI3MzQgWiBNNC4zMTY0MDYyNSw2LjkxNTUyNzM0IEM0LjMxNjQwNjI1LDkuMjYxODkzMjQgNS4xMzI2NDE1OCwxMC40MzUwNTg2IDYuNzY1MTM2NzIsMTAuNDM1MDU4NiBDNy41OTUyMTg5OSwxMC40MzUwNTg2IDguMjEwODU0MTEsMTAuMTUwMDY4IDguNjEyMDYwNTUsOS41ODAwNzgxMiBDOS4wMTMyNjY5OCw5LjAxMDA4ODMgOS4yMTM4NjcxOSw4LjEyMTkxMzU4IDkuMjEzODY3MTksNi45MTU1MjczNCBDOS4yMTM4NjcxOSw1LjcwMzYwNzIyIDkuMDEwNTAwMDgsNC44MDg1MTUyNiA4LjYwMzc1OTc3LDQuMjMwMjI0NjEgQzguMTk3MDE5NDUsMy42NTE5MzM5NiA3LjU4OTY4NTAzLDMuMzYyNzkyOTcgNi43ODE3MzgyOCwzLjM2Mjc5Mjk3IEM1LjEzODE3NTM4LDMuMzYyNzkyOTcgNC4zMTY0MDYyNSw0LjU0NzAyNTkyIDQuMzE2NDA2MjUsNi45MTU1MjczNCBMNC4zMTY0MDYyNSw2LjkxNTUyNzM0IFogTTI3LjY2NTUyNzMsNC43MjQxMjEwOSBDMjcuNjY1NTI3Myw2LjA3OTkyMjE0IDI3LjI2NTcxMDQsNy4xMjcxOTM1NyAyNi40NjYwNjQ1LDcuODY1OTY2OCBDMjUuNjY2NDE4NSw4LjYwNDc0MDAyIDI0LjUzMDYwNjMsOC45NzQxMjEwOSAyMy4wNTg1OTM4LDguOTc0MTIxMDkgTDIyLjEzNzIwNyw4Ljk3NDEyMTA5IEwyMi4xMzcyMDcsMTMgTDE4Ljg1ODM5ODQsMTMgTDE4Ljg1ODM5ODQsMC44NjQyNTc4MTIgTDIzLjA1ODU5MzgsMC44NjQyNTc4MTIgQzI0LjU5MTQ3OSwwLjg2NDI1NzgxMiAyNS43NDI1MDkyLDEuMTk5MDUyNjQgMjYuNTExNzE4OCwxLjg2ODY1MjM0IEMyNy4yODA5MjgzLDIuNTM4MjUyMDUgMjcuNjY1NTI3MywzLjQ5MDA2NTQ0IDI3LjY2NTUyNzMsNC43MjQxMjEwOSBMMjcuNjY1NTI3Myw0LjcyNDEyMTA5IFogTTIyLjEzNzIwNyw2LjMwMTI2OTUzIEwyMi43MzQ4NjMzLDYuMzAxMjY5NTMgQzIzLjIyNzM3ODgsNi4zMDEyNjk1MyAyMy42MTg4OTUsNi4xNjI5MjQ1NiAyMy45MDk0MjM4LDUuODg2MjMwNDcgQzI0LjE5OTk1MjYsNS42MDk1MzYzOCAyNC4zNDUyMTQ4LDUuMjI3NzA0MjYgMjQuMzQ1MjE0OCw0Ljc0MDcyMjY2IEMyNC4zNDUyMTQ4LDMuOTIxNzA4MTQgMjMuODkxNDQzMywzLjUxMjIwNzAzIDIyLjk4Mzg4NjcsMy41MTIyMDcwMyBMMjIuMTM3MjA3LDMuNTEyMjA3MDMgTDIyLjEzNzIwNyw2LjMwMTI2OTUzIFogTTQwLjgwNDY4NzUsMTMgTDMzLjU5OTYwOTQsMTMgTDMzLjU5OTYwOTQsMC44NjQyNTc4MTIgTDQwLjgwNDY4NzUsMC44NjQyNTc4MTIgTDQwLjgwNDY4NzUsMy40OTU2MDU0NyBMMzYuODc4NDE4LDMuNDk1NjA1NDcgTDM2Ljg3ODQxOCw1LjQwNDc4NTE2IEw0MC41MTQxNjAyLDUuNDA0Nzg1MTYgTDQwLjUxNDE2MDIsOC4wMzYxMzI4MSBMMzYuODc4NDE4LDguMDM2MTMyODEgTDM2Ljg3ODQxOCwxMC4zMjcxNDg0IEw0MC44MDQ2ODc1LDEwLjMyNzE0ODQgTDQwLjgwNDY4NzUsMTMgWiBNNTguNDg0Mzc1LDEzIEw1NC4xODQ1NzAzLDEzIEw0OS43NTE5NTMxLDQuNDUwMTk1MzEgTDQ5LjY3NzI0NjEsNC40NTAxOTUzMSBDNDkuNzgyMzg5OCw1Ljc5NDkyODYgNDkuODM0OTYwOSw2LjgyMTQ0ODI4IDQ5LjgzNDk2MDksNy41Mjk3ODUxNiBMNDkuODM0OTYwOSwxMyBMNDYuOTI5Njg3NSwxMyBMNDYuOTI5Njg3NSwwLjg2NDI1NzgxMiBMNTEuMjEyODkwNiwwLjg2NDI1NzgxMiBMNTUuNjI4OTA2Miw5LjI5Nzg1MTU2IEw1NS42Nzg3MTA5LDkuMjk3ODUxNTYgQzU1LjYwMTIzNjYsOC4wNzQ4NjM2OCA1NS41NjI1LDcuMDkyNjE0MzggNTUuNTYyNSw2LjM1MTA3NDIyIEw1NS41NjI1LDAuODY0MjU3ODEyIEw1OC40ODQzNzUsMC44NjQyNTc4MTIgTDU4LjQ4NDM3NSwxMyBaIE03My4wNDI5Njg4LDkuMzE0NDUzMTIgQzczLjA0Mjk2ODgsMTAuMDY3MDYxMSA3Mi44NTIwNTI3LDEwLjczNTI2NzMgNzIuNDcwMjE0OCwxMS4zMTkwOTE4IEM3Mi4wODgzNzcsMTEuOTAyOTE2MyA3MS41Mzc3NjQsMTIuMzU2Njg3OCA3MC44MTgzNTk0LDEyLjY4MDQxOTkgQzcwLjA5ODk1NDcsMTMuMDA0MTUyIDY5LjI1NTA1MDQsMTMuMTY2MDE1NiA2OC4yODY2MjExLDEzLjE2NjAxNTYgQzY3LjQ3ODY3NDMsMTMuMTY2MDE1NiA2Ni44MDA3ODQsMTMuMTA5Mjk0MiA2Ni4yNTI5Mjk3LDEyLjk5NTg0OTYgQzY1LjcwNTA3NTQsMTIuODgyNDA1IDY1LjEzNTA5NDEsMTIuNjg0NTcxNyA2NC41NDI5Njg4LDEyLjQwMjM0MzggTDY0LjU0Mjk2ODgsOS40ODA0Njg3NSBDNjUuMTY4Mjk3NCw5LjgwMTQzMzkgNjUuODE4NTE4OCwxMC4wNTE4MzgzIDY2LjQ5MzY1MjMsMTAuMjMxNjg5NSBDNjcuMTY4Nzg1OSwxMC40MTE1NDA2IDY3Ljc4ODU3MTQsMTAuNTAxNDY0OCA2OC4zNTMwMjczLDEwLjUwMTQ2NDggQzY4Ljg0MDAwODksMTAuNTAxNDY0OCA2OS4xOTY5MzksMTAuNDE3MDc0NCA2OS40MjM4MjgxLDEwLjI0ODI5MSBDNjkuNjUwNzE3MywxMC4wNzk1MDc2IDY5Ljc2NDE2MDIsOS44NjIzMDYwMiA2OS43NjQxNjAyLDkuNTk2Njc5NjkgQzY5Ljc2NDE2MDIsOS40MzA2NjMyMyA2OS43MTg1MDYzLDkuMjg1NDAxMDEgNjkuNjI3MTk3Myw5LjE2MDg4ODY3IEM2OS41MzU4ODgyLDkuMDM2Mzc2MzMgNjkuMzg5MjQyNSw4LjkxMDQ4MjQxIDY5LjE4NzI1NTksOC43ODMyMDMxMiBDNjguOTg1MjY5Miw4LjY1NTkyMzg0IDY4LjQ0NzEwNzIsOC4zOTU4MzUzIDY3LjU3Mjc1MzksOC4wMDI5Mjk2OSBDNjYuNzgxNDA4OCw3LjY0MzIyNzM3IDY2LjE4NzkwODksNy4yOTQ1OTgwNCA2NS43OTIyMzYzLDYuOTU3MDMxMjUgQzY1LjM5NjU2MzgsNi42MTk0NjQ0NiA2NS4xMDMyNzI0LDYuMjMyMDk4NTQgNjQuOTEyMzUzNSw1Ljc5NDkyMTg4IEM2NC43MjE0MzQ2LDUuMzU3NzQ1MjEgNjQuNjI1OTc2Niw0Ljg0MDMzNTAyIDY0LjYyNTk3NjYsNC4yNDI2NzU3OCBDNjQuNjI1OTc2NiwzLjEyNDgzMTY1IDY1LjAzMjcxMDgsMi4yNTMyNTgzMyA2NS44NDYxOTE0LDEuNjI3OTI5NjkgQzY2LjY1OTY3MiwxLjAwMjYwMTA0IDY3Ljc3NzQ5OTQsMC42ODk5NDE0MDYgNjkuMTk5NzA3LDAuNjg5OTQxNDA2IEM3MC40NTU4OTgyLDAuNjg5OTQxNDA2IDcxLjczNjk3MjYsMC45ODA0NjU4NDUgNzMuMDQyOTY4OCwxLjU2MTUyMzQ0IEw3Mi4wMzg1NzQyLDQuMDkzMjYxNzIgQzcwLjkwNDEyODQsMy41NzMwNzY4MyA2OS45MjQ2NDYsMy4zMTI5ODgyOCA2OS4xMDAwOTc3LDMuMzEyOTg4MjggQzY4LjY3Mzk4ODgsMy4zMTI5ODgyOCA2OC4zNjQwOTYsMy4zODc2OTQ1NyA2OC4xNzA0MTAyLDMuNTM3MTA5MzggQzY3Ljk3NjcyNDMsMy42ODY1MjQxOCA2Ny44Nzk4ODI4LDMuODcxOTA2NDUgNjcuODc5ODgyOCw0LjA5MzI2MTcyIEM2Ny44Nzk4ODI4LDQuMzMxMjE4NjQgNjguMDAzMDA5OCw0LjU0NDI2OTg5IDY4LjI0OTI2NzYsNC43MzI0MjE4OCBDNjguNDk1NTI1Myw0LjkyMDU3Mzg2IDY5LjE2MzczMTUsNS4yNjM2NjkzOCA3MC4yNTM5MDYyLDUuNzYxNzE4NzUgQzcxLjI5OTgwOTksNi4yMzIwOTg3MSA3Mi4wMjYxMjEsNi43MzcwNTc4NSA3Mi40MzI4NjEzLDcuMjc2NjExMzMgQzcyLjgzOTYwMTYsNy44MTYxNjQ4MSA3My4wNDI5Njg4LDguNDk1NDM4NjEgNzMuMDQyOTY4OCw5LjMxNDQ1MzEyIEw3My4wNDI5Njg4LDkuMzE0NDUzMTIgWiBNODkuMzE5ODI0MiwxMyBMODYuMDI0NDE0MSwxMyBMODYuMDI0NDE0MSw4LjA1MjczNDM4IEw4Mi4yMTQzNTU1LDguMDUyNzM0MzggTDgyLjIxNDM1NTUsMTMgTDc4LjkxODk0NTMsMTMgTDc4LjkxODk0NTMsMC44NjQyNTc4MTIgTDgyLjIxNDM1NTUsMC44NjQyNTc4MTIgTDgyLjIxNDM1NTUsNS4zNjMyODEyNSBMODYuMDI0NDE0MSw1LjM2MzI4MTI1IEw4Ni4wMjQ0MTQxLDAuODY0MjU3ODEyIEw4OS4zMTk4MjQyLDAuODY0MjU3ODEyIEw4OS4zMTk4MjQyLDEzIFogTTk1Ljk0Mjg3MTEsMTMgTDk1Ljk0Mjg3MTEsMC44NjQyNTc4MTIgTDk5LjIzODI4MTIsMC44NjQyNTc4MTIgTDk5LjIzODI4MTIsMTMgTDk1Ljk0Mjg3MTEsMTMgWiBNMTA5LjA5MDMzMiwxMyBMMTA1Ljg2MTMyOCwxMyBMMTA1Ljg2MTMyOCwwLjg2NDI1NzgxMiBMMTEzLjAzMzIwMywwLjg2NDI1NzgxMiBMMTEzLjAzMzIwMywzLjQ5NTYwNTQ3IEwxMDkuMDkwMzMyLDMuNDk1NjA1NDcgTDEwOS4wOTAzMzIsNS44MTE1MjM0NCBMMTEyLjcyNjA3NCw1LjgxMTUyMzQ0IEwxMTIuNzI2MDc0LDguNDQyODcxMDkgTDEwOS4wOTAzMzIsOC40NDI4NzEwOSBMMTA5LjA5MDMzMiwxMyBaIE0xMjQuMzc5Mzk1LDEzIEwxMjEuMTAwNTg2LDEzIEwxMjEuMTAwNTg2LDMuNTQ1NDEwMTYgTDExOC4xMzcyMDcsMy41NDU0MTAxNiBMMTE4LjEzNzIwNywwLjg2NDI1NzgxMiBMMTI3LjMzNDQ3MywwLjg2NDI1NzgxMiBMMTI3LjMzNDQ3MywzLjU0NTQxMDE2IEwxMjQuMzc5Mzk1LDMuNTQ1NDEwMTYgTDEyNC4zNzkzOTUsMTMgWiIgaWQ9Ik9QRU5TSElGVCI+PC9wYXRoPiAgICAgICAgICAgIDxwYXRoIGQ9Ik0xNDEuODY0NzQ2LDEuOTUxNjYwMTYgQzE0MC41MzEwODEsMS45NTE2NjAxNiAxMzkuNDc4Mjc1LDIuMzk1NzQ3NTEgMTM4LjcwNjI5OSwzLjI4MzkzNTU1IEMxMzcuOTM0MzIyLDQuMTcyMTIzNTggMTM3LjU0ODM0LDUuMzg4MTc1ODcgMTM3LjU0ODM0LDYuOTMyMTI4OTEgQzEzNy41NDgzNCw4LjUyMDM1Mjk5IDEzNy45MjA0ODgsOS43NDc0NzI4OCAxMzguNjY0Nzk1LDEwLjYxMzUyNTQgQzEzOS40MDkxMDIsMTEuNDc5NTc3OSAxNDAuNDcwMjA4LDExLjkxMjU5NzcgMTQxLjg0ODE0NSwxMS45MTI1OTc3IEMxNDIuNjk0ODI4LDExLjkxMjU5NzcgMTQzLjY2MDQ3NiwxMS43NjA0MTgyIDE0NC43NDUxMTcsMTEuNDU2MDU0NyBMMTQ0Ljc0NTExNywxMi42OTI4NzExIEMxNDMuOTAzOTY3LDEzLjAwODMwMjQgMTQyLjg2NjM4LDEzLjE2NjAxNTYgMTQxLjYzMjMyNCwxMy4xNjYwMTU2IEMxMzkuODQ0ODgsMTMuMTY2MDE1NiAxMzguNDY1NTgxLDEyLjYyMzcwMzMgMTM3LjQ5NDM4NSwxMS41MzkwNjI1IEMxMzYuNTIzMTg5LDEwLjQ1NDQyMTcgMTM2LjAzNzU5OCw4LjkxMzI1ODY5IDEzNi4wMzc1OTgsNi45MTU1MjczNCBDMTM2LjAzNzU5OCw1LjY2NDg3MDA1IDEzNi4yNzE0MDEsNC41NjkxNzc4OCAxMzYuNzM5MDE0LDMuNjI4NDE3OTcgQzEzNy4yMDY2MjcsMi42ODc2NTgwNiAxMzcuODgxNzUsMS45NjI3MzA0MSAxMzguNzY0NDA0LDEuNDUzNjEzMjggQzEzOS42NDcwNTgsMC45NDQ0OTYxNTIgMTQwLjY4NjAyOSwwLjY4OTk0MTQwNiAxNDEuODgxMzQ4LDAuNjg5OTQxNDA2IEMxNDMuMTU0MTQsMC42ODk5NDE0MDYgMTQ0LjI2NjQzNCwwLjkyMjM2MDk1NyAxNDUuMjE4MjYyLDEuMzg3MjA3MDMgTDE0NC42MjA2MDUsMi41OTkxMjEwOSBDMTQzLjcwMTk4MSwyLjE2NzQ3ODMxIDE0Mi43ODMzNywxLjk1MTY2MDE2IDE0MS44NjQ3NDYsMS45NTE2NjAxNiBMMTQxLjg2NDc0NiwxLjk1MTY2MDE2IFogTTE1Ny45MjY3NTgsNi45MTU1MjczNCBDMTU3LjkyNjc1OCw4Ljg1NzkxOTg3IDE1Ny40MzU2MzMsMTAuMzg1MjQ4MyAxNTYuNDUzMzY5LDExLjQ5NzU1ODYgQzE1NS40NzExMDUsMTIuNjA5ODY4OCAxNTQuMTA1NjQsMTMuMTY2MDE1NiAxNTIuMzU2OTM0LDEzLjE2NjAxNTYgQzE1MC41Njk0OSwxMy4xNjYwMTU2IDE0OS4xOTAxOSwxMi42MTk1NTMgMTQ4LjIxODk5NCwxMS41MjY2MTEzIEMxNDcuMjQ3Nzk4LDEwLjQzMzY2OTcgMTQ2Ljc2MjIwNyw4Ljg5MTEyMzI0IDE0Ni43NjIyMDcsNi44OTg5MjU3OCBDMTQ2Ljc2MjIwNyw0LjkyMzMyOTk3IDE0Ny4yNDkxODEsMy4zOTE4NTExNCAxNDguMjIzMTQ1LDIuMzA0NDQzMzYgQzE0OS4xOTcxMDgsMS4yMTcwMzU1OCAxNTAuNTgwNTU3LDAuNjczMzM5ODQ0IDE1Mi4zNzM1MzUsMC42NzMzMzk4NDQgQzE1NC4xMTY3MDgsMC42NzMzMzk4NDQgMTU1LjQ3ODAyMiwxLjIyNjcxOTczIDE1Ni40NTc1MiwyLjMzMzQ5NjA5IEMxNTcuNDM3MDE3LDMuNDQwMjcyNDYgMTU3LjkyNjc1OCw0Ljk2NzYwMDk0IDE1Ny45MjY3NTgsNi45MTU1MjczNCBMMTU3LjkyNjc1OCw2LjkxNTUyNzM0IFogTTE0OC4yNTYzNDgsNi45MTU1MjczNCBDMTQ4LjI1NjM0OCw4LjU1OTA5MDI1IDE0OC42MDYzNiw5LjgwNTU3ODQ0IDE0OS4zMDYzOTYsMTAuNjU1MDI5MyBDMTUwLjAwNjQzMywxMS41MDQ0ODAyIDE1MS4wMjMyNjgsMTEuOTI5MTk5MiAxNTIuMzU2OTM0LDExLjkyOTE5OTIgQzE1My43MDE2NjcsMTEuOTI5MTk5MiAxNTQuNzE3MTE5LDExLjUwNTg2MzYgMTU1LjQwMzMyLDEwLjY1OTE3OTcgQzE1Ni4wODk1MjIsOS44MTI0OTU3NyAxNTYuNDMyNjE3LDguNTY0NjI0MTMgMTU2LjQzMjYxNyw2LjkxNTUyNzM0IEMxNTYuNDMyNjE3LDUuMjgzMDMyMiAxNTYuMDkwOTA1LDQuMDQ0ODQ0NzEgMTU1LjQwNzQ3MSwzLjIwMDkyNzczIEMxNTQuNzI0MDM2LDIuMzU3MDEwNzUgMTUzLjcxMjczNSwxLjkzNTA1ODU5IDE1Mi4zNzM1MzUsMS45MzUwNTg1OSBDMTUxLjAyODgwMiwxLjkzNTA1ODU5IDE1MC4wMDY0MzMsMi4zNTk3Nzc2NSAxNDkuMzA2Mzk2LDMuMjA5MjI4NTIgQzE0OC42MDYzNiw0LjA1ODY3OTM4IDE0OC4yNTYzNDgsNS4yOTQwOTk5NyAxNDguMjU2MzQ4LDYuOTE1NTI3MzQgTDE0OC4yNTYzNDgsNi45MTU1MjczNCBaIE0xNzAuMTEyMzA1LDEzIEwxNjguNTAxOTUzLDEzIEwxNjEuODY5NjI5LDIuODE0OTQxNDEgTDE2MS44MDMyMjMsMi44MTQ5NDE0MSBDMTYxLjg5MTc2NSw0LjAxMDI1OTg4IDE2MS45MzYwMzUsNS4xMDU5NTIwNSAxNjEuOTM2MDM1LDYuMTAyMDUwNzggTDE2MS45MzYwMzUsMTMgTDE2MC42MzI4MTIsMTMgTDE2MC42MzI4MTIsMC44NjQyNTc4MTIgTDE2Mi4yMjY1NjIsMC44NjQyNTc4MTIgTDE2OC44NDIyODUsMTEuMDA3ODEyNSBMMTY4LjkwODY5MSwxMS4wMDc4MTI1IEMxNjguODk3NjI0LDEwLjg1ODM5NzcgMTY4Ljg3MjcyMiwxMC4zNzgzNDA2IDE2OC44MzM5ODQsOS41Njc2MjY5NSBDMTY4Ljc5NTI0Nyw4Ljc1NjkxMzI2IDE2OC43ODE0MTMsOC4xNzcyNDc4NCAxNjguNzkyNDgsNy44Mjg2MTMyOCBMMTY4Ljc5MjQ4LDAuODY0MjU3ODEyIEwxNzAuMTEyMzA1LDAuODY0MjU3ODEyIEwxNzAuMTEyMzA1LDEzIFogTTE3Ny4xODQ1NywxMyBMMTc1Ljc3MzQzOCwxMyBMMTc1Ljc3MzQzOCwyLjExNzY3NTc4IEwxNzEuOTMwMTc2LDIuMTE3Njc1NzggTDE3MS45MzAxNzYsMC44NjQyNTc4MTIgTDE4MS4wMjc4MzIsMC44NjQyNTc4MTIgTDE4MS4wMjc4MzIsMi4xMTc2NzU3OCBMMTc3LjE4NDU3LDIuMTE3Njc1NzggTDE3Ny4xODQ1NywxMyBaIE0xOTAuNDgyNDIyLDEzIEwxODguOTcxNjgsOS4xNDAxMzY3MiBMMTg0LjEwNzQyMiw5LjE0MDEzNjcyIEwxODIuNjEzMjgxLDEzIEwxODEuMTg1NTQ3LDEzIEwxODUuOTgzMzk4LDAuODE0NDUzMTI1IEwxODcuMTcwNDEsMC44MTQ0NTMxMjUgTDE5MS45NDMzNTksMTMgTDE5MC40ODI0MjIsMTMgWiBNMTg4LjUzMTczOCw3Ljg3MDExNzE5IEwxODcuMTIwNjA1LDQuMTA5ODYzMjggQzE4Ni45Mzc5ODcsMy42MzM5NDk0NCAxODYuNzQ5ODM4LDMuMDUwMTMzNjcgMTg2LjU1NjE1MiwyLjM1ODM5ODQ0IEMxODYuNDM0NDA3LDIuODg5NjUxMDkgMTg2LjI2MDA5MiwzLjQ3MzQ2Njg3IDE4Ni4wMzMyMDMsNC4xMDk4NjMyOCBMMTg0LjYwNTQ2OSw3Ljg3MDExNzE5IEwxODguNTMxNzM4LDcuODcwMTE3MTkgWiBNMTkzLjYxMTgxNiwxMyBMMTkzLjYxMTgxNiwwLjg2NDI1NzgxMiBMMTk1LjAyMjk0OSwwLjg2NDI1NzgxMiBMMTk1LjAyMjk0OSwxMyBMMTkzLjYxMTgxNiwxMyBaIE0yMDcuODMxMDU1LDEzIEwyMDYuMjIwNzAzLDEzIEwxOTkuNTg4Mzc5LDIuODE0OTQxNDEgTDE5OS41MjE5NzMsMi44MTQ5NDE0MSBDMTk5LjYxMDUxNSw0LjAxMDI1OTg4IDE5OS42NTQ3ODUsNS4xMDU5NTIwNSAxOTkuNjU0Nzg1LDYuMTAyMDUwNzggTDE5OS42NTQ3ODUsMTMgTDE5OC4zNTE1NjIsMTMgTDE5OC4zNTE1NjIsMC44NjQyNTc4MTIgTDE5OS45NDUzMTIsMC44NjQyNTc4MTIgTDIwNi41NjEwMzUsMTEuMDA3ODEyNSBMMjA2LjYyNzQ0MSwxMS4wMDc4MTI1IEMyMDYuNjE2Mzc0LDEwLjg1ODM5NzcgMjA2LjU5MTQ3MiwxMC4zNzgzNDA2IDIwNi41NTI3MzQsOS41Njc2MjY5NSBDMjA2LjUxMzk5Nyw4Ljc1NjkxMzI2IDIwNi41MDAxNjMsOC4xNzcyNDc4NCAyMDYuNTExMjMsNy44Mjg2MTMyOCBMMjA2LjUxMTIzLDAuODY0MjU3ODEyIEwyMDcuODMxMDU1LDAuODY0MjU3ODEyIEwyMDcuODMxMDU1LDEzIFogTTIxNy45MzMxMDUsMTMgTDIxMS4xNjc5NjksMTMgTDIxMS4xNjc5NjksMC44NjQyNTc4MTIgTDIxNy45MzMxMDUsMC44NjQyNTc4MTIgTDIxNy45MzMxMDUsMi4xMTc2NzU3OCBMMjEyLjU3OTEwMiwyLjExNzY3NTc4IEwyMTIuNTc5MTAyLDYuMDI3MzQzNzUgTDIxNy42MDkzNzUsNi4wMjczNDM3NSBMMjE3LjYwOTM3NSw3LjI3MjQ2MDk0IEwyMTIuNTc5MTAyLDcuMjcyNDYwOTQgTDIxMi41NzkxMDIsMTEuNzM4MjgxMiBMMjE3LjkzMzEwNSwxMS43MzgyODEyIEwyMTcuOTMzMTA1LDEzIFogTTIyMi4wMzM2OTEsNy45NTMxMjUgTDIyMi4wMzM2OTEsMTMgTDIyMC42MjI1NTksMTMgTDIyMC42MjI1NTksMC44NjQyNTc4MTIgTDIyMy45NTExNzIsMC44NjQyNTc4MTIgQzIyNS40Mzk3ODYsMC44NjQyNTc4MTIgMjI2LjUzOTYyOSwxLjE0OTI0ODQ1IDIyNy4yNTA3MzIsMS43MTkyMzgyOCBDMjI3Ljk2MTgzNiwyLjI4OTIyODExIDIyOC4zMTczODMsMy4xNDY5NjY5MyAyMjguMzE3MzgzLDQuMjkyNDgwNDcgQzIyOC4zMTczODMsNS44OTczMDYyIDIyNy41MDM5MTQsNi45ODE5MzA3NyAyMjUuODc2OTUzLDcuNTQ2Mzg2NzIgTDIyOS4xNzIzNjMsMTMgTDIyNy41MDM5MDYsMTMgTDIyNC41NjU0Myw3Ljk1MzEyNSBMMjIyLjAzMzY5MSw3Ljk1MzEyNSBaIE0yMjIuMDMzNjkxLDYuNzQxMjEwOTQgTDIyMy45Njc3NzMsNi43NDEyMTA5NCBDMjI0Ljk2Mzg3Miw2Ljc0MTIxMDk0IDIyNS42OTQzMzQsNi41NDMzNzc2MyAyMjYuMTU5MTgsNi4xNDc3MDUwOCBDMjI2LjYyNDAyNiw1Ljc1MjAzMjUzIDIyNi44NTY0NDUsNS4xNTg1MzI2IDIyNi44NTY0NDUsNC4zNjcxODc1IEMyMjYuODU2NDQ1LDMuNTY0Nzc0NjMgMjI2LjYxOTg3NSwyLjk4NjQ5MjY2IDIyNi4xNDY3MjksMi42MzIzMjQyMiBDMjI1LjY3MzU4MiwyLjI3ODE1NTc4IDIyNC45MTQwNjgsMi4xMDEwNzQyMiAyMjMuODY4MTY0LDIuMTAxMDc0MjIgTDIyMi4wMzM2OTEsMi4xMDEwNzQyMiBMMjIyLjAzMzY5MSw2Ljc0MTIxMDk0IFogTTI0My4yNDIxODgsNC40MDAzOTA2MiBDMjQzLjI0MjE4OCw1LjYyODkxMjM5IDI0Mi44MjMwMDIsNi41NzM4MDg1NCAyNDEuOTg0NjE5LDcuMjM1MTA3NDIgQzI0MS4xNDYyMzYsNy44OTY0MDYzIDIzOS45NDY3ODUsOC4yMjcwNTA3OCAyMzguMzg2MjMsOC4yMjcwNTA3OCBMMjM2Ljk1ODQ5Niw4LjIyNzA1MDc4IEwyMzYuOTU4NDk2LDEzIEwyMzUuNTQ3MzYzLDEzIEwyMzUuNTQ3MzYzLDAuODY0MjU3ODEyIEwyMzguNjkzMzU5LDAuODY0MjU3ODEyIEMyNDEuNzI1OTI3LDAuODY0MjU3ODEyIDI0My4yNDIxODgsMi4wNDI5NTY5NiAyNDMuMjQyMTg4LDQuNDAwMzkwNjIgTDI0My4yNDIxODgsNC40MDAzOTA2MiBaIE0yMzYuOTU4NDk2LDcuMDE1MTM2NzIgTDIzOC4yMjg1MTYsNy4wMTUxMzY3MiBDMjM5LjQ3OTE3Myw3LjAxNTEzNjcyIDI0MC4zODM5NDksNi44MTMxNTMwNiAyNDAuOTQyODcxLDYuNDA5MTc5NjkgQzI0MS41MDE3OTMsNi4wMDUyMDYzMSAyNDEuNzgxMjUsNS4zNTc3NTE4NSAyNDEuNzgxMjUsNC40NjY3OTY4OCBDMjQxLjc4MTI1LDMuNjY0Mzg0MDEgMjQxLjUxODM5NSwzLjA2NjczMzc0IDI0MC45OTI2NzYsMi42NzM4MjgxMiBDMjQwLjQ2Njk1NywyLjI4MDkyMjUxIDIzOS42NDc5NTUsMi4wODQ0NzI2NiAyMzguNTM1NjQ1LDIuMDg0NDcyNjYgTDIzNi45NTg0OTYsMi4wODQ0NzI2NiBMMjM2Ljk1ODQ5Niw3LjAxNTEzNjcyIFogTTI0NS43ODIyMjcsMTMgTDI0NS43ODIyMjcsMC44NjQyNTc4MTIgTDI0Ny4xOTMzNTksMC44NjQyNTc4MTIgTDI0Ny4xOTMzNTksMTEuNzIxNjc5NyBMMjUyLjU0NzM2MywxMS43MjE2Nzk3IEwyNTIuNTQ3MzYzLDEzIEwyNDUuNzgyMjI3LDEzIFogTTI2Mi4yMzQzNzUsMTMgTDI2MC43MjM2MzMsOS4xNDAxMzY3MiBMMjU1Ljg1OTM3NSw5LjE0MDEzNjcyIEwyNTQuMzY1MjM0LDEzIEwyNTIuOTM3NSwxMyBMMjU3LjczNTM1MiwwLjgxNDQ1MzEyNSBMMjU4LjkyMjM2MywwLjgxNDQ1MzEyNSBMMjYzLjY5NTMxMiwxMyBMMjYyLjIzNDM3NSwxMyBaIE0yNjAuMjgzNjkxLDcuODcwMTE3MTkgTDI1OC44NzI1NTksNC4xMDk4NjMyOCBDMjU4LjY4OTk0LDMuNjMzOTQ5NDQgMjU4LjUwMTc5MSwzLjA1MDEzMzY3IDI1OC4zMDgxMDUsMi4zNTgzOTg0NCBDMjU4LjE4NjM2LDIuODg5NjUxMDkgMjU4LjAxMjA0NSwzLjQ3MzQ2Njg3IDI1Ny43ODUxNTYsNC4xMDk4NjMyOCBMMjU2LjM1NzQyMiw3Ljg3MDExNzE5IEwyNjAuMjgzNjkxLDcuODcwMTE3MTkgWiBNMjY5LjA5OTEyMSwxMyBMMjY3LjY4Nzk4OCwxMyBMMjY3LjY4Nzk4OCwyLjExNzY3NTc4IEwyNjMuODQ0NzI3LDIuMTE3Njc1NzggTDI2My44NDQ3MjcsMC44NjQyNTc4MTIgTDI3Mi45NDIzODMsMC44NjQyNTc4MTIgTDI3Mi45NDIzODMsMi4xMTc2NzU3OCBMMjY5LjA5OTEyMSwyLjExNzY3NTc4IEwyNjkuMDk5MTIxLDEzIFogTTI3Ni4xNzk2ODgsMTMgTDI3NC43Njg1NTUsMTMgTDI3NC43Njg1NTUsMC44NjQyNTc4MTIgTDI4MS41MzM2OTEsMC44NjQyNTc4MTIgTDI4MS41MzM2OTEsMi4xMTc2NzU3OCBMMjc2LjE3OTY4OCwyLjExNzY3NTc4IEwyNzYuMTc5Njg4LDYuNTU4NTkzNzUgTDI4MS4yMDk5NjEsNi41NTg1OTM3NSBMMjgxLjIwOTk2MSw3LjgxMjAxMTcyIEwyNzYuMTc5Njg4LDcuODEyMDExNzIgTDI3Ni4xNzk2ODgsMTMgWiBNMjk0LjA3NjE3Miw2LjkxNTUyNzM0IEMyOTQuMDc2MTcyLDguODU3OTE5ODcgMjkzLjU4NTA0NywxMC4zODUyNDgzIDI5Mi42MDI3ODMsMTEuNDk3NTU4NiBDMjkxLjYyMDUxOSwxMi42MDk4Njg4IDI5MC4yNTUwNTQsMTMuMTY2MDE1NiAyODguNTA2MzQ4LDEzLjE2NjAxNTYgQzI4Ni43MTg5MDQsMTMuMTY2MDE1NiAyODUuMzM5NjA0LDEyLjYxOTU1MyAyODQuMzY4NDA4LDExLjUyNjYxMTMgQzI4My4zOTcyMTIsMTAuNDMzNjY5NyAyODIuOTExNjIxLDguODkxMTIzMjQgMjgyLjkxMTYyMSw2Ljg5ODkyNTc4IEMyODIuOTExNjIxLDQuOTIzMzI5OTcgMjgzLjM5ODU5NSwzLjM5MTg1MTE0IDI4NC4zNzI1NTksMi4zMDQ0NDMzNiBDMjg1LjM0NjUyMiwxLjIxNzAzNTU4IDI4Ni43Mjk5NzIsMC42NzMzMzk4NDQgMjg4LjUyMjk0OSwwLjY3MzMzOTg0NCBDMjkwLjI2NjEyMiwwLjY3MzMzOTg0NCAyOTEuNjI3NDM3LDEuMjI2NzE5NzMgMjkyLjYwNjkzNCwyLjMzMzQ5NjA5IEMyOTMuNTg2NDMxLDMuNDQwMjcyNDYgMjk0LjA3NjE3Miw0Ljk2NzYwMDk0IDI5NC4wNzYxNzIsNi45MTU1MjczNCBMMjk0LjA3NjE3Miw2LjkxNTUyNzM0IFogTTI4NC40MDU3NjIsNi45MTU1MjczNCBDMjg0LjQwNTc2Miw4LjU1OTA5MDI1IDI4NC43NTU3NzQsOS44MDU1Nzg0NCAyODUuNDU1ODExLDEwLjY1NTAyOTMgQzI4Ni4xNTU4NDcsMTEuNTA0NDgwMiAyODcuMTcyNjgyLDExLjkyOTE5OTIgMjg4LjUwNjM0OCwxMS45MjkxOTkyIEMyODkuODUxMDgxLDExLjkyOTE5OTIgMjkwLjg2NjUzMywxMS41MDU4NjM2IDI5MS41NTI3MzQsMTAuNjU5MTc5NyBDMjkyLjIzODkzNiw5LjgxMjQ5NTc3IDI5Mi41ODIwMzEsOC41NjQ2MjQxMyAyOTIuNTgyMDMxLDYuOTE1NTI3MzQgQzI5Mi41ODIwMzEsNS4yODMwMzIyIDI5Mi4yNDAzMTksNC4wNDQ4NDQ3MSAyOTEuNTU2ODg1LDMuMjAwOTI3NzMgQzI5MC44NzM0NSwyLjM1NzAxMDc1IDI4OS44NjIxNDksMS45MzUwNTg1OSAyODguNTIyOTQ5LDEuOTM1MDU4NTkgQzI4Ny4xNzgyMTYsMS45MzUwNTg1OSAyODYuMTU1ODQ3LDIuMzU5Nzc3NjUgMjg1LjQ1NTgxMSwzLjIwOTIyODUyIEMyODQuNzU1Nzc0LDQuMDU4Njc5MzggMjg0LjQwNTc2Miw1LjI5NDA5OTk3IDI4NC40MDU3NjIsNi45MTU1MjczNCBMMjg0LjQwNTc2Miw2LjkxNTUyNzM0IFogTTI5OC4xOTMzNTksNy45NTMxMjUgTDI5OC4xOTMzNTksMTMgTDI5Ni43ODIyMjcsMTMgTDI5Ni43ODIyMjcsMC44NjQyNTc4MTIgTDMwMC4xMTA4NCwwLjg2NDI1NzgxMiBDMzAxLjU5OTQ1NCwwLjg2NDI1NzgxMiAzMDIuNjk5Mjk3LDEuMTQ5MjQ4NDUgMzAzLjQxMDQsMS43MTkyMzgyOCBDMzA0LjEyMTUwNCwyLjI4OTIyODExIDMwNC40NzcwNTEsMy4xNDY5NjY5MyAzMDQuNDc3MDUxLDQuMjkyNDgwNDcgQzMwNC40NzcwNTEsNS44OTczMDYyIDMwMy42NjM1ODIsNi45ODE5MzA3NyAzMDIuMDM2NjIxLDcuNTQ2Mzg2NzIgTDMwNS4zMzIwMzEsMTMgTDMwMy42NjM1NzQsMTMgTDMwMC43MjUwOTgsNy45NTMxMjUgTDI5OC4xOTMzNTksNy45NTMxMjUgWiBNMjk4LjE5MzM1OSw2Ljc0MTIxMDk0IEwzMDAuMTI3NDQxLDYuNzQxMjEwOTQgQzMwMS4xMjM1NCw2Ljc0MTIxMDk0IDMwMS44NTQwMDIsNi41NDMzNzc2MyAzMDIuMzE4ODQ4LDYuMTQ3NzA1MDggQzMwMi43ODM2OTQsNS43NTIwMzI1MyAzMDMuMDE2MTEzLDUuMTU4NTMyNiAzMDMuMDE2MTEzLDQuMzY3MTg3NSBDMzAzLjAxNjExMywzLjU2NDc3NDYzIDMwMi43Nzk1NDMsMi45ODY0OTI2NiAzMDIuMzA2Mzk2LDIuNjMyMzI0MjIgQzMwMS44MzMyNSwyLjI3ODE1NTc4IDMwMS4wNzM3MzYsMi4xMDEwNzQyMiAzMDAuMDI3ODMyLDIuMTAxMDc0MjIgTDI5OC4xOTMzNTksMi4xMDEwNzQyMiBMMjk4LjE5MzM1OSw2Ljc0MTIxMDk0IFogTTMxMi42NjE2MjEsMTMgTDMwOC41NDQ0MzQsMi4yNDIxODc1IEwzMDguNDc4MDI3LDIuMjQyMTg3NSBDMzA4LjU1NTUwMiwzLjA5NDQwNTMgMzA4LjU5NDIzOCw0LjEwNzA5MDQ5IDMwOC41OTQyMzgsNS4yODAyNzM0NCBMMzA4LjU5NDIzOCwxMyBMMzA3LjI5MTAxNiwxMyBMMzA3LjI5MTAxNiwwLjg2NDI1NzgxMiBMMzA5LjQxNjAxNiwwLjg2NDI1NzgxMiBMMzEzLjI1OTI3NywxMC44NzUgTDMxMy4zMjU2ODQsMTAuODc1IEwzMTcuMjAyMTQ4LDAuODY0MjU3ODEyIEwzMTkuMzEwNTQ3LDAuODY0MjU3ODEyIEwzMTkuMzEwNTQ3LDEzIEwzMTcuODk5NDE0LDEzIEwzMTcuODk5NDE0LDUuMTgwNjY0MDYgQzMxNy44OTk0MTQsNC4yODQxNzUyMSAzMTcuOTM4MTUxLDMuMzEwMjI2NjEgMzE4LjAxNTYyNSwyLjI1ODc4OTA2IEwzMTcuOTQ5MjE5LDIuMjU4Nzg5MDYgTDMxMy43OTg4MjgsMTMgTDMxMi42NjE2MjEsMTMgWiIgaWQ9IkNPTlRBSU5FUi1QTEFURk9STSI+PC9wYXRoPiAgICAgICAgPC9nPiAgICA8L2c+PC9zdmc+" alt="OpenShift Container Platform">
          </div><!--/#brand-->
          <div class="alert alert-danger">
            <span class="pficon-layered">
              <span class="pficon pficon-error-octagon"></span>
              <span class="pficon pficon-error-exclamation"></span>
            </span>
            {{ .Error }}
          </div>
        </div><!--/.col-*-->
      </div><!--/.row-->
    </div><!--/.container-->
  </body>
</html>
`
