# Syslog protocol

roughly conforms to the murky shade of grey known as http://www.faqs.org/rfcs/rfc3164.html

## Examples

### Manipulate packets manually

    require 'syslog_protocol'
    
    p = SyslogProtocol::Packet.new
    p.hostname = "space_station"
    p.facility = "kern"
    p.severity = "warn"
    p.tag = "test"
    p.content = "flight control broken"
    p.to_s
    # => "<4>Aug  1 14:01:17 space_station flight control broken"
    p.pri
    # => 4
    p.facility
    # => 0
    p.facility_name
    # => "kern"
    p.severity_name
    # => "warn"
    p.warn?
    # => true
    p.info?
    # => false


### Use a Logger to generate packets

    require 'syslog_protocol'
    
    logger = SyslogProtocol::Logger.new("space_station", "uucp")
    logger.debug("looking for uucp on board the space station")
    # => "<67>Aug  1 14:02:29 space_station looking for uucp on board the space station"
    logger.emerg("omg we cant find uucp on the space station")
    # => "<64>Aug  1 14:03:56 space_station omg we cant find uucp on the space station"


### Parse packets

    require 'syslog_protocol'
    
    p = SyslogProtocol.parse("<34>Oct 11 22:14:15 space_station space is really getting to me")
    p.facility
    # => 4
    p.severity_name
    # => "crit"
    p.time
    # => Sun Oct 11 22:14:15 -0700 2009
    p.content
    # => "space is really getting to me"


### It yells at you for trying to abuse the protocol

    p = SyslogProtocol::Packet.new
    p.facility = 34534534
    # => ArgumentError: Facility must be within 0-23
    p.hostname = "my host"
    # => ArgumentError: Hostname may not contain spaces
    p.hostname = "h\000stname"
    # => ArgumentError: Hostname may only contain ASCII characters 33-126
    # ...etc.
    # It will also unintelligently truncate messages > 1024 bytes so beware.


## Caveats

Syslog is a terrible and loosely defined protocol. Many devices and programs do not
conform to it and so their packets may not be parsed correctly by this interpretation,
nor may the packets generated by this necessarily be recognized by other devices or programs ;)

This is probably wrong and buggy, and i know the code is ugly, thanks.

Good luck.

## TODO

* Update to more closely map to the ruby `syslog` API where possible
