/*
 * Copyright (C) 2016 Red Hat, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.syndesis.connector.mongo;

import java.util.ArrayList;
import java.util.List;

import com.mongodb.client.model.Filters;
import com.mongodb.client.result.UpdateResult;
import io.syndesis.common.model.integration.Step;
import org.assertj.core.api.Assertions;
import org.bson.Document;
import org.junit.Test;

public class MongoDBConnectorSaveTest extends MongoDBConnectorProducerTestSupport {

    private final static String COLLECTION = "upsertCollection";

    @Override
    public String getCollectionName() {
        return COLLECTION;
    }

    @Override
    protected List<Step> createSteps() {
        String filter = null;
        switch (getTestMethodName()) {
            case "mongoSaveNewTest":
                filter = "{\"_id\": :#id,\"test\":\":#someText\"}";
                break;
            case "mongoSaveNewAutogeneratedIDTest":
                filter = "{\"test\": \":#someText\"}";
                break;
            case "mongoSaveExistingTest":
                filter = "{\"_id\": :#id,\"test\":\":#someText\", \"newField\": :#isNew}";
                break;
        }
        return fromDirectToMongo("start", "io.syndesis.connector:connector-mongodb-upsert", DATABASE, COLLECTION, null,
            filter);
    }

    @Test
    public void mongoSaveNewTest() {
        // When
        // Given
        String saveArguments = "{\"id\":11,\"someText\":\"new\"}";
        UpdateResult result = template.requestBody("direct:start", saveArguments, UpdateResult.class);
        // Then
        Assertions.assertThat(result.getUpsertedId().asNumber().longValue()).isEqualTo(11L);
        List<Document> docsFound = collection.find(Filters.eq("_id", 11)).into(new ArrayList<>());
        Assertions.assertThat(docsFound).hasSize(1);
        Assertions.assertThat(docsFound.get(0).getString("test")).isEqualTo("new");
    }

    @Test
    public void mongoSaveNewAutogeneratedIDTest() {
        // When
        // Given
        String saveArguments = "{\"someText\":\"new\"}";
        UpdateResult result = template.requestBody("direct:start", saveArguments, UpdateResult.class);
        // Then
        List<Document> docsFound = collection.find(Filters.eq("_id",
            result.getUpsertedId().asObjectId())).into(new ArrayList<>());
        Assertions.assertThat(docsFound).hasSize(1);
        Assertions.assertThat(docsFound.get(0).getString("test")).isEqualTo("new");
        Assertions.assertThat(result.getUpsertedId().asObjectId().getValue()).isEqualTo(docsFound.get(0).getObjectId("_id"));
    }

    @Test
    public void mongoSaveExistingTest() {
        // When
        String saveArguments = "{\"_id\":32,\"test\":\"new\"}";
        collection.insertOne(Document.parse(saveArguments));
        // Given
        String saveArguments2 = "{\"id\":32,\"someText\":\"save\",\"isNew\":true}";
        UpdateResult result = template.requestBody("direct:start", saveArguments2, UpdateResult.class);
        // Then
        Assertions.assertThat(result.getModifiedCount()).isEqualTo(1L);
        List<Document> docsFound = collection.find(Filters.eq("_id", 32)).into(new ArrayList<>());
        Assertions.assertThat(docsFound).hasSize(1);
        Assertions.assertThat(docsFound.get(0).getString("test")).isEqualTo("save");
        Assertions.assertThat(docsFound.get(0).getBoolean("newField")).isEqualTo(true);
    }

}
