#!/bin/bash

set -o pipefail
set -eu

display_usage() {
    cat <<EOT
Build script to start the docker build of dv-operator.

Usage: build-image.sh [options] -t  <teiid-spring-boot-version> -v <teiid-version>

where:
-t|--teiidsb    is the version of teiid-spring-boot that PNC built
-v|--teiid      is the version of teiid that PNC built

with options:
-d, --dry-run   Run without committing and changes or running in OSBS.
    --scratch   When running the build, do a scratch build (only applicable if NOT running in dry-run mode)
--help          This help message
EOT
}

cleanup() {
    echo "Perform cleaup"

#    rm -f *.gz
    rm -rf teiid-operator  2> /dev/null || true

    if [ -d "teiid-operator" ]
    then
        echo "Could not delete teiid-operator directory."
        echo "Please run the following command:"
        echo "sudo rm -rf teiid-operator"
    fi
}

update_config_yaml() {
    local teiidSB=$1
    local teiid=$2
    local CONFIG_FILE=./build/conf/config.yaml
    
    
    local tsbversion=`cat ./build/conf/config.yaml | grep 'teiidSpringBootVersion:' | cut -d ' ' -f 2`    
    local tversion=`cat ./build/conf/config.yaml | grep 'teiidVersion:' | cut -d ' ' -f 2`
    
    echo "Updating config.yaml setting teiidSpringBoot version $teiidSB"

    sed -i "s/$tsbversion/$teiidSB/1" $CONFIG_FILE

    echo "Updating config.yaml setting teiid version $teiid"

    sed -i "s/$tversion/$teiid/1" $CONFIG_FILE

    echo "Adding updated config.yaml with teiidSpringBoot version $teiidSB and teiid version $teiid"
    git add ./build/conf/config.yaml
}

update_source_repos() {
    echo "Updating source repos"

    local dvsha=$1
    local dryRun=$2

    sed -i "/teiid-operator /{s/ [^ ]*/ $dvsha/1}" source-repos

    if [ "$dryRun" == "false" ]
    then
         git add source-repos
    fi
}

checkout_sources() {
    echo "Checkout sources"
    local dryRun=$1
    
    local scmURL=`cat source-repos | grep teiid-operator | cut -d ' ' -f 1`
    
    local cloneOpts=""

    if [[ $TEIIDOP_TAG != "master" ]] ; then
        cloneOpts="--branch $TEIIDOP_TAG"
    fi    
    
    if ! git clone $scmURL $cloneOpts
    then
        echo "ERROR: Failed to clone $scmURL"
        return 1
    fi    
    
    pushd teiid-operator
    
#    git checkout $TEIIDOP_TAG
    DV_SHA=`git rev-parse HEAD` 

    if ! git archive --format=tar.gz -o ../teiid-operator-$DV_SHA.tar.gz --prefix=teiid-operator-$DV_SHA/ $DV_SHA
    then
        echo "ERROR: Failed to archive Teiid Operator $teiidTag with sha $DV_SHA"
        popd
        return 1
    fi

    popd

#     if ! rhpkg upload teiid-operator-$TEIIDOP_TAG.tar.gz
#     then
#         echo "Error uploading teiid-operator-$TEIIDOP_TAG.tar.gz to lookaside cache"
#         exit 1
#     fi
     
    return 0
}

get_sha() {
    pushd teiid-operator
    
    DV_SHA=`git rev-parse HEAD`
    
    popd
}

update_dockerfile() {
    echo "Update dockerfile"
    local sha=$1
    local dryRun=$2

    sed -i "/^ENV/,/^$/s/\(.*\)DV_OPERATOR_TAG\([ =]\)[0-9a-zA-Z\.-]\+\(.*\)/\1DV_OPERATOR_TAG\2$sha\3/" Dockerfile

    if [ "$dryRun" == "false" ]
    then
        echo "Adding new Dockerfile"
         git add Dockerfile
    fi
}

osbs_build() {
    local scratchBuild=$1

    git status

    num_files=$(git status --porcelain  | { egrep '^\s?[MADRC]' || true; } | wc -l)
    if ((num_files > 0)) ; then
        echo "Committing $num_files"
        git commit -m "Updated for build of dv-operator"
        git push
    else
        echo "There are no files to be committed. Skipping commit + push"
    fi

    if [ "$scratchBuild" == "false" ]
    then
        echo "Starting OSBS build"
        rhpkg container-build
    else
        local branch=$(git rev-parse --abbrev-ref HEAD)
        local build_options=""

        # If we are building on a private branch, then we need to use the correct target
        if [[ $branch == *"private"* ]] ; then
            # Remove the private part of the branch name: from private-opiske-fuse-7.4-openshift-rhel-7
            # to fuse-7.4-openshift-rhel-7 and we add the containers candidate to the options
            local target="${branch#*-*-}-containers-candidate"

            build_options="${build_options} --target ${target}"
            echo "Using target ${target} for the private container build"
        fi

        echo "Starting OSBS scratch build"
        rhpkg container-build --scratch ${build_options} 
    fi
}

main() {
# TEIIDOP_TAG is the branch to use, for which the DV_SHA will be derived 
    TEIIDOP_TAG=7.6-0.1.x
    TEIIDSPRINGBOOT_VERSION=
    TEIID_VERSION=
    DRY_RUN=false
    SCRATCH=false
    DV_SHA=

    # Parse command line arguments
    while [ $# -gt 0 ]
    do
        arg="$1"

        case $arg in
          -h|--help)
            display_usage
            exit 0
            ;;
          -d|--dry-run)
            DRY_RUN=true
            ;;
          --scratch)
            SCRATCH=true
            ;;
          -t|--teiidsb)
            shift
            TEIIDSPRINGBOOT_VERSION="$1"
            ;;   
          -v|--teiid)
            shift
            TEIID_VERSION="$1"
            ;;   
          *)
            echo "Unknonwn argument: $1"
            display_usage
            exit 1
            ;;            
        esac
        shift
    done
    
    if [ -z "$TEIIDSPRINGBOOT_VERSION" ]
    then
        echo "ERROR: -t Teiid Spring Boot version wasn't specified."
        exit 1
    fi
    
    if [ -z "$TEIID_VERSION" ]
    then
        echo "ERROR: -t Teiid version wasn't specified."
        exit 1
    fi    
    
#---- Steps
#   checkout from repo
#   get the SHA from the branch version/tag branch
#   update SOURCE_REPO file with DV_SHA (this is what OSBS needs)
#   update DOCKER with DV_SHA
#   trigger OSBS build
   
    if [ ! -d teiid-operator ]
    then
        if ! checkout_sources $DRY_RUN
        then
            echo "Could not checkout teiid sources using tag $TEIIDOP_TAG."
            exit 1
        fi
    else
       get_sha
       echo "teiid-operator already exist, will not checkout again"
    fi
    
    update_source_repos $DV_SHA $DRY_RUN
        
    update_dockerfile $DV_SHA $DRY_RUN
    
    update_config_yaml $TEIIDSPRINGBOOT_VERSION $TEIID_VERSION

    if [ "$DRY_RUN" == "false" ]
    then
        osbs_build $SCRATCH
    fi

    cleanup
}

main $*
