# Register Schemes

The manager process generated by kubebuilder only knows about the resources
we defined. It does not know about the resources defined by the common Cluster
API code. Therefore replace `cmd/manager/main.go` with this code:

```go
/*
Copyright 2018 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package main

import (
        "flag"
        "fmt"
        "os"

        "sigs.k8s.io/cluster-api-provider-solas/pkg/apis"
        "sigs.k8s.io/cluster-api-provider-solas/pkg/cloud/solas/actuators/cluster"
        "sigs.k8s.io/cluster-api-provider-solas/pkg/cloud/solas/actuators/machine"
        clusterapis "sigs.k8s.io/cluster-api/pkg/apis"
        "sigs.k8s.io/cluster-api/pkg/apis/cluster/common"
        "sigs.k8s.io/cluster-api/pkg/client/clientset_generated/clientset"
        capicluster "sigs.k8s.io/cluster-api/pkg/controller/cluster"
        capimachine "sigs.k8s.io/cluster-api/pkg/controller/machine"
        "sigs.k8s.io/controller-runtime/pkg/client/config"
        "sigs.k8s.io/controller-runtime/pkg/manager"
        logf "sigs.k8s.io/controller-runtime/pkg/runtime/log"
        "sigs.k8s.io/controller-runtime/pkg/runtime/signals"
)

func main() {
        cfg := config.GetConfigOrDie()
        if cfg == nil {
                panic(fmt.Errorf("GetConfigOrDie didn't die"))
        }

        metricsAddr := flag.String("metrics-addr", ":8080", "The address the metric endpoint binds to.")
        flag.Parse()

        log := logf.Log.WithName("solas-controller-manager")
        logf.SetLogger(logf.ZapLogger(false))
        entryLog := log.WithName("entrypoint")

        // Setup a Manager
        mgr, err := manager.New(cfg, manager.Options{MetricsBindAddress: *metricsAddr})
        if err != nil {
                entryLog.Error(err, "unable to set up overall controller manager")
                os.Exit(1)
        }

        cs, err := clientset.NewForConfig(cfg)
        if err != nil {
                panic(err)
        }

        clusterActuator, err := cluster.NewActuator(cluster.ActuatorParams{
                ClustersGetter: cs.ClusterV1alpha1(),
        })
        if err != nil {
                panic(err)
        }

        machineActuator, err := machine.NewActuator(machine.ActuatorParams{
                MachinesGetter: cs.ClusterV1alpha1(),
        })
        if err != nil {
                panic(err)
        }

        // Register our cluster deployer (the interface is in clusterctl and we define the Deployer interface on the actuator)
        common.RegisterClusterProvisioner("solas", clusterActuator)

        if err := apis.AddToScheme(mgr.GetScheme()); err != nil {
                panic(err)
        }

        if err := clusterapis.AddToScheme(mgr.GetScheme()); err != nil {
                panic(err)
        }

        capimachine.AddWithActuator(mgr, machineActuator)

        capicluster.AddWithActuator(mgr, clusterActuator)

        if err := mgr.Start(signals.SetupSignalHandler()); err != nil {
                entryLog.Error(err, "unable to run manager")
                os.Exit(1)
        }
}
```
