#  Copyright 2011 Red Hat 
#  
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#  
#  http://www.apache.org/licenses/LICENSE-2.0
#  
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.

function Log ($message) {
    if ($logFile) {
        $date = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
        "$date $message" >> $logFile
    }
}


function Say ($message) {
    Log $message
    Write-Host $message
}


# This function aborts the migration procedure showing the given message to the
# user:
function Abort ($message) {
    if ($message) {
        Say $message
    }
    Write-Host "The migration process is aborted."
    Write-Host "Check the log file ""$logFile"" for details."
    exit 1
}


# This function stops and disables the given service:
function Disable-Service ($name) {
    Set-Service -Name $name -StartupType Disabled
    Stop-Service -Name $name
    if ((Get-Service $name).Status -ne "Stopped") {
        Abort "Cannot stop service ""$name""."
    }
}

# This function is required to correctly count results from the API that may be
# empty, a single object or an array of objects:
function Count-Elements ($object) {
    if (-not $object) {
        0
    }
    elseif ($object -is [System.Array]) {
        $object.Count
    }
    else {
        1
    }
}

function Backup-Database ($dbName) {
    # Check that the backup file doesn't exist:
    $backupFile = "$backupDir\sql\$dbName.bak"
    if (Test-Path $backupFile) {
        Abort "The backup file ""$backupFile"" for database ""$dbName"" already exists."
    }
   
    # Connect to the database: 
    $connection = New-Object System.Data.SQLClient.SQLConnection
    $connection.ConnectionString = $sqlString
    try {
        $connection.Open()
    }
    catch  {
        Say "Cannot connect to the SQL Server: $sqlName"
        Say "Possible issues: "
        Say "- Wrong sql server name"
        Say "- Program using exclusive the database (SQL Management Studio?)"
        Abort
    }
   
    # Send the backup command: 
    $command = New-Object System.Data.SQLClient.SQLCommand
    $command.Connection = $connection
    $command.CommandText = "backup database $dbName to disk='$backupFile'" 
    try {
        $Command.ExecuteNonQuery() | Out-Null
    }
    catch {
        Abort "Cannot create the database backup. Please verify the arguments provided."
    }
    finally {
        if ($connection) {
            $connection.Close()
        }
    }
}


function Copy-Database ($dbName) {
    try {
        # Open the connection:
        $connection = New-Object System.Data.SQLClient.SQLConnection
        $connection.ConnectionString = $sqlString
        $connection.Open()

        # Create the command:
        $command = New-Object System.Data.SQLClient.SQLCommand
        $command.Connection = $connection

        # Build the SQL statement used to restore the backup as a new database:
        $backupFile = "$backupDir\sql\$dbName.bak"
        $listSql = "restore filelistonly from disk='$backupFile'"
        Log "SQL statement to find logical to physical name mapping is ""$listSql""."
        $restoreSql = "restore database ${dbName}_migration from disk='$backupFile' with replace"
        try {
            $command.CommandText = $listSql
            $reader = $command.ExecuteReader()
            while ($reader.Read()) {
                $logical = $reader["LogicalName"]
                $physical = $reader["PhysicalName"]
                Log "Physical name for logical name ""$logical"" is ""$physical""."
                $dot = $physical.LastIndexOf('.')
                if ($dot -eq -1) {
                    Abort("Can't find extension in physical name ""$physical"" for database ""$dbName""")
                }
                $base = $physical.Substring(0, $dot)
                $extension = $physical.Substring($dot)
                $move = "${base}_migration${extension}"
                Log "Move name is ""$move""."
                $restoreSql += ", move '$logical' to '$move'"
            }
        }
        catch {
            Log $_
            Abort "Can't get logical to physical name mapping for database ""$dbName""."
        }
        finally {
            if ($reader) {
                $reader.Close()
            }
        }

        # Run SQL statement used to perform the copy:
        Log "SQL statement to copy the database is ""$restoreSql""."
        try {
            $command.CommandText = $restoreSql
            $count = $command.ExecuteNonQuery()
        }
        catch {
            Log $_
            Abort "Can't execute the SQL statement to copy the database."
        }

    }
    catch {
        Log $_
        Abort "Can't copy database ""$dbName""."
    }
    finally {
        if ($connection) {
            $connection.Close()
        }
    }
}


# A simple function to prompt the user for input using a default value that
# will be displayed between square brackets:
function Read-Text ($prompt, $default) {
    if ($default) {
        $value = Read-Host -Prompt "$prompt [$default]"
    }
    else {
        $value = Read-Host -Prompt $prompt
    }
    if (-not $value) {
        $value = $default
    }
    $value
}


# A simple function to read a password without echoing it:
function Read-Passwd ($prompt) {
    $passwd = Read-Host -Prompt $prompt -AsSecureString
    $credentials = New-Object System.Management.Automation.PsCredential("None", $passwd)
    $credentials.GetNetworkCredential().Password
}


function Create-Directory ($directory) {
    New-Item $directory -Type directory -ErrorAction SilentlyContinue -ErrorVariable message | Out-Null
    if (!$?) {
        Abort $message
    }
}


# Prepare the log file:
$date = Get-Date -Format "yyyy_MM_dd_HH_mm_ss"
$logFile = "rhevm-migrate-$date.log"

# Request the address, host name and password to connect to the database and
# make sure they are correct:
Say @"

In order to perform backup and to access the database we need to know the host
name or IP address of the database server, the name of the instance, the name
of the administrator user and its password. Please confirm or change the values
below:

"@

$sqlServer = "localhost"
$sqlInstance = ""
$sqlName = ""
$sqlUser = "sa"
$sqlDatabase = "rhevm"
$sqlString = ""

while ($true) {
    # Request the parameters to the user:
    $sqlServer = Read-Text "Database server" $sqlServer
    $sqlInstance = Read-Text "Database instance" $sqlInstance
    if (-not $sqlInstance) {
        $sqlName = "$sqlServer"
    }
    else {
        $sqlName = "$sqlServer\$sqlInstance"
    }
    $sqlDatabase = Read-Text "Database name" $sqlDatabase
    $sqlUser = Read-Text "Database user" "sa"
    $sqlPasswd = Read-Passwd "Database password"

    # Build the connection string:
    $sqlString = "Data Source=$sqlName;Initial Catalog=$sqlDatabase;User Id=$sqlUser;Password=$sqlPasswd"
   
    # Try to connect: 
    $connection = New-Object System.Data.SQLClient.SQLConnection
    $connection.ConnectionString = $sqlString
    try {
        $connection.Open()
        break
    }
    catch  {
        Say "Cannot connect to the SQL Server: $sqlName"
        Say "Possible issues: "
        Say "- Wrong server name"
        Say "- Another program is using the database (SQL Management Studio?)"
    }
    finally {
        if ($connection) {
             $connection.Close()
        }
    }
}

# Find the OS architecture as we need it in order to build
# correctly the registry paths (for 64 bits we need to look into Wow6432Node):
$os = Get-WmiObject Win32_OperatingSystem
$registryBranch = "HKLM:\SOFTWARE"
if ($os.OSArchitecture -eq "64-bit") {
    $registryBranch = "$registryBranch\Wow6432Node"
}

# Request the user name and password used to connect to the server using the
# API and make sure they are correct:
Say @"

In order to perform the migration process we need to know the user, the
password and the domain of the administrator user that was provided during the
initial installation. Please confirm or change the values below:

"@

$registryPath = "$registryBranch\RedHat\RHEVM Service\"
$registryKey = Get-ItemProperty -Path $registryPath -Name User -ErrorAction SilentlyContinue
if ($registryKey) {
    $adminUser = $registryKey.User
}
$registryKey = Get-ItemProperty -Path $registryPath -Name Domain -ErrorAction SilentlyContinue
if ($registryKey) {
    $adminDomain = $registryKey.Domain
}

while ($true) {
    # Request the parameters:
    $adminUser = Read-Text "Admin user" $adminUser
    $adminDomain = Read-Text "Admin domain" $adminDomain
    $adminPasswd = Read-Passwd "Admin password"

    # Try to log in:
    try {
        Login-User $adminUser $adminPasswd $adminDomain
        break
    }
    catch {
        Say "Can't login with the given user name, password and domain."
        Say "Possible issues:"
        Say "- Wrong parameters user name, password or domain."
        Say "- You are not running this script by RHEV Manager Scripting Library"
        Say "To execute: Start -> All Programs -> Red Hat -> RHEV Manager Scripting Library"
    }
    finally {
        Logout-User
    }
}


# Try to find the installation directory and ask the user for confirmation or
# maybe change:
Say @"

In order to make backup copies of some important files we need to know the
install locations of the engine and the documents directory of the web server.
Please confirm or change the values below:

"@

$registryPath = "$registryBranch\RedHat\SetupInfo\"
$registryKey = Get-ItemProperty -Path $registryPath -Name InstallLocation -ErrorAction SilentlyContinue
if ($registryKey) {
    $installDir = $registryKey.InstallLocation
}
while ($true) {
    $installDir = Read-Text "Install directory" $installDir
    if (-not (Test-Path $installDir)) {
        Say "The installation directory ""$installDir"" doesn't exist."
    }
    else {
        break
    }
}

$wwwDir = "C:\inetpub\wwwroot"
if (-not (Test-Path $wwwDir)) {
    $wwwDir = ""
}
while ($true) {
    $wwwDir = Read-Text "Web root" $wwwDir
    if (-not (Test-Path $wwwDir)) {
        Say "The web root ""$wwwDir"" doesn't exist."
    }
    else {
        break
    }
}


# Ask the user where to keep the backup copies:
Say @"

The backup of the database and some other important files that need to be
copied to the Linux machine will be stored in a temporary directory that will
be created for you. Please provide the full path for that temporary directory
below:

"@

$backupDir = ""
while ($true) {
    $backupDir = Read-Text "Backup directory" $backupDir
    if (Test-Path $backupDir) {
        Say "The directory ""$backupDir"" already exists."
    }
    else {
        break
    }
}


# Give the user a summary of the parameters and give the opportunity to cancel
# the migration now:
Say @"

We are ready to start the migration process.

Please note that once the migration process starts your existing installation
will become non operational. Your hosts and virtual machines will continue
running without interruption, but you will not be able to manage them. If you
want to roll back the migration you will have to manually enable the services.

Please review the settings below and confirm that you want to proceed:

Database server: $sqlServer
Database instance: $sqlInstance 
Database name: $sqlDatabase
Database user: $sqlUser
Admin user: $adminUser
Admin domain: $adminDomain
Install directory: $installDir
Web root: $wwwDir
Backup directory: $backupDir

"@

while ($true) {
    $response = Read-Text "Continue" "no"
    if ($response -eq "yes") {
        break
    }
    elseif ($response -eq "no") {
        Abort
    }
    else {
        Say "Please type ""yes"" or ""no""."
    }
}
Say


# Make sure that all the files required to perform the upgrade of the
# database are available:
Say "Checking required files ..."
$upgradeDir = "set-tables-to-rhev3"
$upgradeFiles = @(
    "insert_predefined_roles.sql",
    "Cleanup.sql",
    "DropConstrains.sql",
    "CreateConstrains.sql",
    "functions.sql",
    "upgradeToEAP4_6.cmd",
    "upgradeToEAP4_6.sql",
    "common_sp.sql",
    "INT2UUIDUpgrade.sql",
    "refreshStoredProcedures.cmd",
    "create_views.sql"
)
$missingUpgradeFiles = $false
foreach ($upgradeFile in $upgradeFiles) {
    $upgradePath = "$upgradeDir\$upgradeFile"
    if (-not (Test-Path $upgradePath)) {
        Say "The upgrade file ""$upgradePath"" is missing."
        $missingUpgradeFiles = $true
    }
}
if ($missingUpgradeFiles) {
    Abort "Some of the required files are missing. Aborting."
}


# For the next few tasks we need to be logged in:
Say "Log in ..."
try {
    Login-User $adminUser $adminPasswd $adminDomain
}
catch {
    Abort "Can't login with the given user name, password and domain."
}


# Make sure that there are no 2.1 data centers as those can't be
# migrated directly to version 3.0:
Say "Making sure that there are no 2.1 data centers ..."
$badDataCenters = $false
foreach ($dataCenter in Select-DataCenter) {
    $name = $dataCenter.Name
    $majorVersion = $dataCenter.CompatibilityVersion.Major
    $minorVersion = $dataCenter.CompatibilityVersion.Minor
    $fullVersion = "$majorVersion.$minorVersion"
    if ($fullVersion -eq "2.1") {
        $badDataCenters = $true
        Say "The data center ""$name"" is using version 2.1."
    }
}
if ($badDataCenters) {
    Abort "Some of the data centers are using version 2.1, please upgrade them to 2.2 before migrating to version 3."
}


# Make sure that there are no time lease pools as they are no longer
# supported in version 3.0:
Say "Making sure that there are no time lease pools ..."
$badPools = $False
foreach ($pool in Select-VmPool) {
    $name = $pool.Name
    if ($pool.PoolType -eq "TimeLease") {
        $badPools = $True
        Say "Pool ""$name"" is of type time lease."
    }
}
if ($badPools) {
    Abort "Pools of type time lease are not supported in RHEV 3, please remove them before migrating."
}


# Find the name of the default cluster (the one with id 0):
Say "Finding the name of the default cluster ..."
$defaultCluster = Select-Cluster | Where-Object { $_.ClusterID -eq 0 }
if (!$defaultCluster) {
    Abort "Can't find the default cluster."
}
$defaultClusterName = $defaultCluster.Name


# Gather counts of relevant objects:
Say "Counting relevant objects ..."
$numberOfUsers = Count-Elements $(Select-User)
$numberOfVms = Count-Elements $(Select-Vm)
$numberOfDataCenters = Count-Elements $(Select-DataCenter)
$numberOfHosts = Count-Elements $(Select-Host)
$numberOfStorageDomains = Count-Elements $(Select-StorageDomain)
$numberOfClusters = Count-Elements $(Select-Cluster)
$numberOfTemplates = Count-Elements $(Select-Template)
$version = Get-Version


# Get the names and addresses of the hosts:
Say "Getting host names and addresses ..."
$hostsList = ""
$hostsArray = Select-Host
if ($hostsArray -isnot [array]) {
    $hostsElement = $hostsArray
    $hostsArray = @($hostsElement)
}
foreach ($hostElement in $hostsArray) {
   $hostInfo = $hostElement.Name + ":" + $hostElement.Address
   if ($hostsList) {
       $hostsList = $hostsList + ", " + $hostInfo
   }
   else {
       $hostsList = $hostInfo
   }
}


# We are done with the API:
Say "Log out ..."
Logout-User


# Now we are done with the API we can stop and disable the services, so they
# will not be a problem while copying files or making backups:
Say "Disabling services ..."
Disable-Service "RHEVManager"
Disable-Service "RHEVMHistoryService"
Disable-Service "RHEVMNetConsole"
Disable-Service "RHEVMNotificationService"


# Create the backup directory and its subdirectories:    
Say "Creating backup directories ..."
Create-Directory $backupDir
Create-Directory $backupDir\inetpub
Create-Directory $backupDir\service
Create-Directory $backupDir\service\sysprep
Create-Directory $backupDir\sql
Create-Directory $backupDir\certificates
Create-Directory $backupDir\certificates\TrustedRootCertificatesAuthorities
Create-Directory $backupDir\certificates\TrustedPublisher
Create-Directory $backupDir\certificates\Personal


# Backup the databases:
Say "Backing up databases ..."
Backup-Database "rhevm"
Backup-Database "rhevm_history"


# Create the copy of the database that we will upgrade (we aren't touching the
# original database in order to simplify the rollback procedure, should it be
# required):
Say "Copying databases ..."
Copy-Database "rhevm"


# Copy files:
Say "Copying files ..."
Copy-Item $installDir\Service\ca $backupDir\service -recurse
Copy-Item $installDir\Service\rhevm.pfx $backupDir\service
Copy-Item $installDir\Service\rhevm.ssh.key $backupDir\service
Copy-Item $installDir\Service\RhevManager.exe.config $backupDir\service
Copy-Item $installDir\Service\RhevmHistoryService.exe.config $backupDir\service
Copy-Item $installDir\Service\RHEVMNotificationService.exe.config $backupDir\service
Copy-Item $installDir\Service\FieldsInVDCConfig.xml $backupDir\service


# Copy sysprep files:
Say "Copying sysprep files ..."
Copy-Item $installDir\Service\sysprep\* $backupDir\service\sysprep -recurse


# Copy web server files:
Say "Copying web server files ..."
Copy-Item $wwwDir\ca.crt $backupDir\inetpub
Copy-Item $wwwDir\rhevm.ssh.key.txt $backupDir\inetpub


# Export certificates:
Say "Exporting certificates ..."

# We need this in order to export X509 certificates:
$type = [System.Security.Cryptography.X509Certificates.X509ContentType]::Cert

# LocalMachine, Trusted Root Certification Authorities:
$certs = (dir cert:\localmachine\root) | where { $_.Subject -like "*CN=RHEVM CA*" }
if ($certs -eq $null) {
    Abort "Cannot locate certificate from LocalMachine\Trusted Root Certificate Authorities."
}
$index = 0
foreach ($cert in $certs) {
    $bytes = $cert.Export($type)
    $file = $backupDir + "\certificates\TrustedRootCertificatesAuthorities\rhev_CA_Trusted_Authorities" + $index + ".cer"
    [System.IO.File]::WriteAllBytes($file, $bytes)
    $index++
}

# LocalMachine, Trusted Publisher:
$certs = (dir cert:\localmachine\trustedpublisher) | where { $_.Subject -like "*Red Hat*" }
if ($certs -eq $null) {
    Abort "Cannot locate certificate from Local Machine\Trusted Publisher."
}
$index = 0
foreach ($cert in $certs) {
    $bytes = $cert.Export($type)
    $file = $backupDir + "\certificates\TrustedPublisher\rhev_CA_Trusted_Publisher" + $index + ".cer"
    [System.IO.File]::WriteAllBytes($file, $bytes)
    $index++
}

# LocalMachine, Personal Certs:
$certs = (dir cert:\localmachine\my)
if ($certs -eq $null) {
    Abort "Cannot locate certificate from Local Machine\Personal."
}
$index = 0
foreach ($cert in $certs) {
    $bytes = $cert.Export($type)
    $file = $backupDir + "\certificates\Personal\cert_personal" + $index + ".cer"
    [System.IO.File]::WriteAllBytes($file, $bytes)
    $index++
}


# Call the script that performs the upgrade of the database:
Say "Upgrading the database ..."
cmd.exe /c $upgradeDir\upgradeToEAP4_6.cmd $sqlName ${sqlDatabase}_migration $sqlUser $sqlPasswd >> $logFile
if (-not $lastexitcode -eq 0) {
    Abort "The upgrade of the database failed."
}


# Write the gathered results to the file that we will read in the
# Linux part of the migration tool:
Say "Dump the results ..."
Out-File -FilePath $backupdir\results.ini -Encoding "ASCII" -InputObject @"
[global]
version=$version

[admin]
user=$adminUser
passwd=$adminPasswd
domain=$adminDomain

[sql]
user=$sqlUser
passwd=$sqlPasswd

[cluster]
default=$defaultClusterName
count=$numberOfClusters

[user]
count=$numberOfUsers

[vm]
count=$numberOfVms

[dataCenter]
count=$numberOfDataCenters

[host]
count=$numberOfHosts
list=$hostsList

[storageDomain]
count=$numberOfStorageDomains

[template]
count=$numberOfTemplates
"@


# Done, give the user some pointers about the next steps:
Say @"

The part of the migration process that runs on Windows finished.

Please copy all the files under the backup directory to the Linux machine and
run the "rhevm-migration" script to complete the migration process.

Note that these files may contain confidential information, like user names and
passwors, so you should handle them carefully.

The log is available in the file "$logFile".
"@


# Bye:
exit 0
