#!/usr/bin/python
#
#  Copyright 2011 Red Hat 
#  
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#  
#  http://www.apache.org/licenses/LICENSE-2.0
#  
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.


import ConfigParser
import datetime
import getpass
import glob
import optparse
import os
import psycopg2
import ldap
import libxml2
import logging
import re
import rpm
import shutil
import socket
import stat
import subprocess
import sys
import tempfile
import traceback
import uuid


# The log file:
logFile = None


# Values for the command line options and arguments:
config = None
options = None
args = None


# We will keep a connection to the PostgreSQL database open for
# the duration of the migration:
connection = None


# Directories for the Windows and Linux CAs:
windowsCaDir = None
linuxCaDir = '/etc/pki/rhevm'


# Details of the key store file:
ksFile = os.path.join(linuxCaDir, '.keystore')
ksPass = 'mypass'
ksAlias = 'rhevm'

# The location of the files copied from Windws:
windowsFilesDir = None


# The results gathered in the Windows machine:
windowsResults = None


# Details for the connection to the databases:
sqlHost = None
sqlPort = None
sqlName = None
sqlUser = None
sqlPasswd = None
pgHost = None
pgPort = None
pgName = None
pgUser = None
pgPasswd = None


# The domain passwords:
domainNames = set([])
domainUsers = dict()
domainPasswords = dict()


# Contents of the options tables:
windowsOptions = None
linuxOptions = None


# The SSH key files:
sshPem = None
sshKey = None


# The list of hosts:
hostsList = []
hostsFailed = []


# The HTTP host name and port numbers:
httpHost = None
httpPort = None
httpsPort = None


# The list of packages that we must make sure are installed before
# trying to do the installation:
# XXX: Is this really required? This script should go in a package
# itself and that package should have these dependencies, so this
# check shouldn't be needed.
REQUIRED_PACKAGES = [
    'rhevm',
    'rhevm-migration-tool',
    'jbossas',
    'postgresql',
    'postgresql-server',
    'openssl',
    'java-1.6.0-sun'
]


# The list of files that must be collected in the previous
# installation:
REQUIRED_FILES = [
    # Certificate authority files:
    'service/ca/ca.pem',
    'service/ca/private/ca.pem',
    'service/ca/certs/rhevm.cer',
    'service/ca/keys/rhevm.pem',
    'service/ca/database.txt',
    'service/ca/database.txt.attr',
    'service/ca/serial.txt',
]


# The list of options that should be copied from the Windows
# installation to the Linux installation wihtout changes (we
# should include here at least all the options that are modifiable
# by the user with the 2.2 configuration tool):
ASIS_OPTIONS = [
   # Sysprep (note that the paths *do* need translation, that is
   # why they are not here):
   'LocalAdminPassword', # XXX: Is this encrypted?
   'DefaultWorkgroup',
   'VirtualMachineDomainName',
   'ProductKey',
   'ProductKey2003',
   'ProductKey2003x64',
   'ProductKey2008',
   'ProductKey2008x64',
   'ProductKey2008R2',
   'ProductKeyWindow7',
   'ProductKeyWindow7x64',
   
   # Storage pool manager:
   'FreeSpaceLow',
   'FreeSpaceCriticalLow',
   'StoragePoolRefreshTimeInSeconds',
   'SPMFailOverAttempts',
   'SpmCommandFailOverRetries',
   'AsyncTaskPollingRate',
   'StorageDomainFalureTimeoutInMinutes',
   'MaxNumberOfHostsInStoragePool',

   # Hosts:
   'vdsTimeout',
   'ServerRebootTimeout',
   'VdsRefreshRate',
   'MaxVdsMemOverCommit',
   'MaxVdsMemOverCommitForServers',
   'EnableUSBAsDefault',
   'UseSecureConnectionWithServers',
   'VdsRecoveryTimeoutInMintues', # XXX: This name has typo, shold be *minutes*, not *mintues*!
   'VDSAttemptsToResetCount',
   'TimeoutToResetVdsInSeconds',
   'BlockMigrationOnSwapUsagePercentage',
   'WaitForVdsInitInSec',

   # Security:
   'SSHInactivityTimoutSeconds',
   'SSLEnabled',
   'SpiceSecureChannels',
   'CertificatePassword',

   # Load balancing:
   'EnableVdsLoadBalancing',
   'VdsLoadBalancingeIntervalInMinutes',
   'HighUtilizationForEvenlyDistribute',
   'HighUtilizationForPowerSave',
   'LowUtilizationForPowerSave',
   'UtilizationThresholdInPercent',
   'CpuOverCommitDurationMinutes',
   'SpmVCpuConsumption',

   # Miscellaneous:
   'SearchResultsLimit',
   'LogXmlRpcData',
   'MacPoolRanges',
   'MaxMacsCountInPool',
   'ValidNumOfMonitors',
   'VmGracefulShutdownMessage',
   'SpiceReleaseCursorKeys',
   'SpiceToggleFullScreenKeys',
   'SpiceUsbAutoShare',
   'RDPLoginWithFQN',
   'VncKeyboardLayout',
   'MaxDiskSize',
   'AsyncTaskZombieTaskLifeInMinutes',
   'EnableSpiceRootCertificateValidation',
   'MaxNumOfVmCpus',
   'MaxNumOfCpuPerSocket',
   'AuditLogCleanupTime',
   'AuditLogAgingThreashold',
   'NumberOfFailedRunsOnVds',
   'TimeToReduceFailedRunOnVdsInMinutes',
   'MaxRerunVmOnVdsCount',
   'NumberVmRefreshesBeforeSave',
   'DisableFenceAtStartupInSec',

   # The organization name also has to be copied, otherwise the
   # spice console will not accept the digital certificates used by
   # the hosts:
   'OrganizationName',
]


# Options containing paths of sysprep files:
SYSPREP_OPTIONS = [
    'SysPrepXPPath',
    'SysPrep2K3Path',
    'SysPrep2K8Path',
    'SysPrep2K8x64Path',
    'SysPrep2K8R2Path',
    'SysPrepWindows7Path',
    'SysPrepWindows7x64Path',
]


# The snippet of Java code used to encrypt:
ENCRYPT_SNIPPET="""
import java.io.FileInputStream;
import java.security.KeyStore;
import java.security.PublicKey;
import java.security.cert.Certificate;
import javax.crypto.Cipher;
import sun.misc.BASE64Encoder;

// Load the key:
KeyStore keystore = KeyStore.getInstance("JKS");
keystore.load(new FileInputStream(file), password.toCharArray());
Certificate certificate = keystore.getCertificate(alias);
PublicKey key = certificate.getPublicKey();

// Encrypt:
byte[] clearBytes = text.getBytes("UTF-8");
Cipher cipher = Cipher.getInstance("RSA");
cipher.init(Cipher.ENCRYPT_MODE, key);
byte[] encryptedBytes = cipher.doFinal(clearBytes);

// Encode:
BASE64Encoder encoder = new BASE64Encoder();
String encryptedText = encoder.encode(encryptedBytes).replaceAll("\\\\s", "");

// Print:
System.out.print(encryptedText);
"""


# The snippet of Java code used to decrypt:
DECRYPT_SNIPPET="""
import java.io.FileInputStream;
import java.security.Key;
import java.security.KeyStore;
import javax.crypto.Cipher;
import sun.misc.BASE64Decoder;

// Load the key:
KeyStore keystore = KeyStore.getInstance("JKS");
keystore.load(new FileInputStream(file), password.toCharArray());
Key key = keystore.getKey(alias, password.toCharArray());

// Decode:
BASE64Decoder decoder = new BASE64Decoder();
byte[] encryptedBytes = decoder.decodeBuffer(text);

// Decrypt:
Cipher cipher = Cipher.getInstance("RSA");
cipher.init(Cipher.DECRYPT_MODE, key);
byte[] clearBytes = cipher.doFinal(encryptedBytes);
String clearText = new String(clearBytes, "UTF-8");

// Print:
System.out.print(clearText);
"""


# The Java script used to decrypt the database password stored in the
# login-config.xml file:
DECRYPT_PG_PASSWD_SNIPPET="""
import java.math.BigInteger;
import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;

// Create the encryption key spec:
byte[] keyBytes = "jaas is the way".getBytes();
SecretKeySpec keySpec = new SecretKeySpec(keyBytes, "Blowfish");

// Load the encrypted password:
BigInteger passwdValue = new BigInteger(passwd, 16);
byte[] encryptedPasswdBytes = passwdValue.toByteArray();

// Decrypt the password:
Cipher cipher = Cipher.getInstance("Blowfish");
cipher.init(Cipher.DECRYPT_MODE, keySpec);
byte[] clearPasswdBytes = cipher.doFinal(encryptedPasswdBytes);

// Print:
System.out.print(new String(clearPasswdBytes));
"""


def abort(message):
    logging.error(message)

    logging.error('\n'
        '\033[1;31mThe migration is aborted, see the log file for details.\033[0;39m\n'
        '\n'
        'You can rollback to your previous installation manually enabling\n'
        'and starting the services in the Windows machine.\n')

    sys.exit(1) 


def formatList(list):
    result = ''
    if list:
        index = 0
        length = len(list)
        for element in list:
            if index > 0:
                if index < length - 1:
                    result += ', '
                else:
                    result += ' and '
            result += '"' + element + '"'
            index += 1
    return result


def parseCommandLine():
    # Create the command line parser:
    usage = "Usage: %prog [OPTION]..."
    parser = optparse.OptionParser(usage)
    parser.description = \
        'Perform the second step of the migration from 2.2 to 3.'

    # Help option:
    helpOption = parser.get_option('--help')
    helpOption.help = \
        'Show this help message and exit.'

    # Debug option:
    parser.add_option(
        '-d',
        '--debug', 
        action='store_true', 
        dest='debug', 
        default=False, 
	help=
            'Enable debug mode so that all debug messages will be '
            'sent to the console in addition to the log file.'
    )

    # Option to migrate the users only:
    parser.add_option(
        '-u',
        '--users-only', 
        action='store_true', 
        dest='usersOnly', 
        default=False, 
	help=
            'Migrate only the users. This is useful when some users '
            'weren\'t migrated correctly and you want to try again '
            'without performing the complete migration.'
    )

    # Option to disable the check of domains:
    parser.add_option(
        '--no-check-domains', 
        action='store_false', 
        dest='checkDomains', 
        default=True, 
	help=
            'Don\'t check that there is at least one domain configured '
            'before starting the migration.'
    )

    # Option to disable the check of hosts:
    parser.add_option(
        '--no-migrate-hosts', 
        action='store_false', 
        dest='migrateHosts', 
        default=True, 
	help=
            'Don\'t verify that hosts can be reached with SSH and don\'t try to '
            'perform any migration tasks on them. You will need to do those '
            'migration tasks manually.'
    )

    # Parse the command line, saving the results to the
    # corresponding global variables for later use:
    global options
    global args
    (options, args) = parser.parse_args()


def askQuestion(prompt, default=None, config=None, section=None, key=None, passwd=False):
    # First try to get the default value from the config:
    if not default and config and section and key:
        try:
            default = config.get(section, key)
        except ConfigParser.NoOptionError:
            logging.debug('No default answer for question "%s" is provided in section "%s".' % (section, key))

    # If there is a default value then update the prompt so that it is
    # contained there:
    ps = None
    if default:
        if passwd:
            ps = "%s [********]: " % prompt
        else:
            ps = "%s [%s]: " % (prompt, default)
    else:
        ps = "%s: " % prompt

    # Ask for the value:
    if passwd:
        answer = getpass.getpass(ps)
    else:
        answer = raw_input(ps)
    if not answer:
        answer = default

    return answer


def askWindowsFilesDir():
    # Ask the user for the location of the files copyed from the Windows
    # machine:
    logging.info('\n'
        'We need to know the location of the directory containing the files\n'
        'generated by the part of the process that you ran previously in the\n'
        'Windows machine. Please provide the directory below:\n')

    global windowsFilesDir
    while True:
        windowsFilesDir = raw_input("Windows files dir: ")
        if os.path.exists(windowsFilesDir):
            break
        logging.info('The directory "%s" doesn\'t exist.' % windowsFilesDir)
    logging.debug('Directory containing files copied from Windows is "%s".' % windowsFilesDir)

    # We need now to read the results file generated in Windows as it contains
    # answers to several questions that user already provided:
    windowsResultsFile = os.path.join(windowsFilesDir, 'results.ini')
    if not os.path.exists(windowsResultsFile):
        abort('Can\'t find the results file "%s" generated in Windows.' % windowsResultsFile)
    global windowsResults
    windowsResults = ConfigParser.RawConfigParser()
    windowsResults.read(windowsResultsFile)
    logging.debug('Loaded Windows results file "%s".' % windowsResultsFile)


def askSQLServerDetails():
    # Globals for the SQL Server database:
    global sqlHost
    global sqlPort
    global sqlName
    global sqlUser
    global sqlPasswd

    # Ask the user for the details needed to connect to the SQL Server database:
    logging.info('\n'
        'We need to know the details needed to connect to the SQL Server\n'
        'database. Make sure that the database server is configured to\n'
        'allow TCP connections and that your firewalls are not blocking the\n'
        'connections. Please confirm or change the details below:\n')

    sqlHost = askQuestion('SQL Server host', config=windowsResults, section='sql', key='host')
    logging.debug('SQL server host is "%s".' % sqlHost)

    sqlPort = askQuestion('SQL Server port', config=windowsResults, section='sql', key='port', default=1433)
    logging.debug('SQL server port is "%s".' % sqlPort)

    sqlName = askQuestion('SQL Server database', config=windowsResults, section='sql', key='database', default='rhevm_migration')
    logging.debug('SQL server database is "%s".' % sqlName)

    sqlUser = askQuestion('SQL Server user', config=windowsResults, section='sql', key='user', default='sa')
    logging.debug('SQL server user is "%s".' % sqlUser)

    sqlPasswd = askQuestion('SQL Server password', config=windowsResults, section='sql', key='passwd', passwd=True)
    logging.debug('SQL server password is "%s".' % '********')


def askPostgreSQLDetails():
    # Globals for the PostgreSQL database:
    global pgHost
    global pgPort
    global pgName
    global pgUser
    global pgPasswd

    # Try to extract the PostgreSQL encrypted password from the
    # login-config.xml file:
    loginConfigFile = '/usr/share/jbossas/server/rhevm-slimmed/conf/login-config.xml'
    loginConfigDoc = libxml2.parseFile(loginConfigFile)
    xpathContext = loginConfigDoc.xpathNewContext()
    xpathResults = xpathContext.xpathEval(
        "/policy"
        "/application-policy[@name='EncryptDBPassword']"
        "/authentication"
        "/login-module[@code='org.jboss.resource.security.SecureIdentityLoginModule']"
        "/module-option[@name='password']"
        "/text()")
    encryptedPgPasswd = None
    if xpathResults:
        encryptedPgPasswd = str(xpathResults[0])
        logging.debug('Found encrypted PostgreSQL database password "%s" inside file "%s".' % (encryptedPgPasswd, loginConfigFile))
    else:
        logging.debug('Can\'t find the encrypted PostgreSQL database password inside file "%s".' % loginConfigFile)
    xpathContext.xpathFreeContext()
    loginConfigDoc.freeDoc()

    # Try to decrypt the PostgreSQL encrypted password:
    (result, pgPasswd, _) = evalJava(DECRYPT_PG_PASSWD_SNIPPET, log=False, passwd=encryptedPgPasswd)
    if result != 0:
        logging.debug('Failed to decrypt PostgreSQL database password')
        pgPasswd = None

    # Ask the user for the details needed to connect to the PostgreSQL database:
    logging.info('\n'
        'We need to know the details to connect to the PostgreSQL\n'
        'database. Make sure that the database is started and confirm or\n'
        'change the details below:\n')
    
    pgHost = askQuestion('PostgreSQL host', default='localhost')
    logging.debug('PostgreSQL host is "%s".' % pgHost)
    
    pgPort = askQuestion('PostgreSQL port', default=5432)
    logging.debug('PostgreSQL port is "%s".' % pgPort)

    pgName = askQuestion('PostgreSQL database', default='rhevm')
    logging.debug('PostgreSQL database is "%s".' % pgName)

    pgUser = askQuestion('PostgreSQL user', default='postgres')
    logging.debug('PostgreSQL user is "%s".' % pgUser)

    pgPasswd = askQuestion('PostgreSQL password', default=pgPasswd, passwd=True)
    logging.debug('PostgreSQL password is "%s".' % '********')


def askContinue():
    # We are ready to start, ask the user for confirmation:
    logging.info('\n'
        'We are ready to start the migration process. This process will\n'
        'wipe all existing data including configuration settings,\n'
        'certificates and database. Please confirm below that you want to\n'
        'proceed:\n')

    while True:
        result = askQuestion('Are you sure you want to proceed?', default='no')
        if result != 'yes' and result != 'no':
            logging.info('Please type "yes" or "no".')
        else:
            break
    if result == 'no':
        abort('Aborted by request of the user.')


def configureLogging():
    # Create the file handler where we will send all the messages:
    global logFile
    logFile = '/var/log/rhevm/rhevm-migration-%s.log' % datetime.datetime.now().strftime('%Y_%m_%d_%H_%M_%S')
    fileHandler = logging.FileHandler(filename=logFile, mode='w')
    fileHandler.setLevel(logging.DEBUG)
    fileFormatter = logging.Formatter('%(asctime)s %(levelname)s %(message)s')
    fileHandler.setFormatter(fileFormatter)
    logging.root.addHandler(fileHandler)

    # Create the console handler where we will send only relevant
    # messages, unless the user selects the debug mode:
    consoleHandler = logging.StreamHandler(strm=sys.stdout)
    consoleHandler.setLevel(logging.DEBUG if options.debug else logging.INFO)
    consoleFormatter = logging.Formatter('%(message)s')
    consoleHandler.setFormatter(consoleFormatter)
    logging.root.addHandler(consoleHandler)

    # Set the default log level to info so that the user will see
    # all those messages in the console:
    logging.root.setLevel(logging.DEBUG)

    # Tell the user what is the version where is the log file:
    logging.info('The log of the migration process is available in "%s".' % logFile) 


def checkPackage(name):
    installed = False
    transaction = rpm.TransactionSet()
    try:
        match = transaction.dbMatch('name', name)
        for header in match:
            logging.debug('Package "%(name)s" is installed, version is "%(version)s-%(release)s".' % header)
            installed = True
            break
    finally:
        transaction.closeDB()
    if not installed:
        logging.debug('Package "%s" isn\'t installed.' % name)
    return installed


def findPaths(dir, pattern='.*'):
    dirs = [dir]
    paths = []
    matcher = re.compile(pattern)
    while dirs:
        current = dirs.pop()
        names = os.listdir(current)
        for name in names:
            path = os.path.join(current, name)
            if os.path.isdir(path):
                dirs.append(path)
            elif os.path.isfile(path):
                relative = os.path.relpath(path, dir)
                if matcher.match(relative):
                    paths.append(relative)
    logging.debug('Files %s in directory "%s" match pattern "%s".' % (formatList(paths), dir, pattern))
    return paths


def replaceFile(srcDir, dstDir, name):
    srcPath = os.path.join(srcDir, name)
    dstPath = os.path.join(dstDir, name)
    logging.debug('Replacing file "%s" with "%s".' % (dstPath, srcPath))
    shutil.copyfile(srcPath, dstPath)

def copyFile(src, dst):
    shutil.copy(src, dst)
    logging.debug('Copyed file "%s" to file "%s".' % (src, dst))

def copyFiles(src, dst, pattern):
    # Get the paths of the files that match the given pattern and
    # copy them to the destination directory:
    paths = findPaths(src, pattern)
    for path in paths:
        srcPath = os.path.join(src, path)
        dstPath = os.path.join(dst, path)
        shutil.copy(srcPath, dstPath)
        logging.debug('Copyed file "%s" from directory "%s" to directory "%s".' % (path, src, dst))


def removeFiles(dir, pattern):
    # Get the paths of the files that match the given pattern and
    # remove them:
    paths = findPaths(dir, pattern)
    for path in paths:
        fullPath = os.path.join(dir, path)
        os.remove(fullPath)
        logging.debug('Removed file "%s" from directory "%s".' % (path, dir)) 


def removeFile(path, name=None):
    # If the second argument is not provided then we assume that
    # the first argument is the full path name of the file to
    # remove:
    if name:
       path = os.path.join(path, name)
    if not os.path.exists(path):
        logging.debug('Trying to remove unexistent file "%s".' % path)
    else:
        os.remove(path)
        logging.debug('Removed file "%s".' % path)


def runCommand(args, log=True, mask=[], **kwargs):
    # Make sure that we mask the information that should not
    # appear in the log:
    command = ' '.join(args)
    for masked in mask:
        command = command.replace(masked, '********')
    logging.debug('Running command "%s".' % command)

    # Run the command:
    if 'stdout' not in kwargs:
        kwargs['stdout'] = subprocess.PIPE
    if 'stderr' not in kwargs:
        kwargs['stderr'] = subprocess.PIPE
    process = subprocess.Popen(args, **kwargs)
    (output, errors) = process.communicate()
    if output:
        if log:
            for masked in mask:
                output = output.replace(masked, '********')
            logging.debug('Standard output follows:\n%s' % output)
        else:
            logging.debug('Standard output is suppressed.')
    else:
        logging.debug('Standard output is empty.')
    if errors:
        if log:
            for masked in mask:
                errors = errors.replace(masked, '********')
            logging.debug('Standard errors follows:\n%s' % errors)
        else:
            logging.debug('Standard errors is suppressed.')
    else:
        logging.debug('Standard errors is empty.')
    result = process.wait()
    logging.debug('Exit code is %d.' % result)

    return result


def runRemoteCommand(user, host, key, args, **kwargs):
    ssh = [
        'ssh',
        '-i', key,
        '-o', 'PasswordAuthentication no',
        '-o', 'StrictHostKeyChecking no',
        '-o', 'UserKnownHostsFile /dev/null',
        '%s@%s' % (user, host)
    ]
    return runCommand(ssh + args, **kwargs)


def evalCommand(args, log=True, mask=[]):
    # Make sure that we mask the information that should not
    # appear in the log:
    command = ' '.join(args)
    for masked in mask:
        command = command.replace(masked, '********')
    logging.debug('Evaluating command "%s".' % command)

    # Run the command:
    process = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    (output, errors) = process.communicate()
    if output:
        if log:
            for masked in mask:
                output = output.replace(masked, '********')
            logging.debug('Standard output follows:\n%s' % output)
        else:
            logging.debug('Standard output is suppressed.')
    else:
        logging.debug('Standard output is empty.')
    if errors:
        if log:
            for masked in mask:
                errors = errors.replace(masked, '********')
            logging.debug('Standard errors follows:\n%s' % errors)
        else:
            logging.debug('Standard errors is suppressed.')
    else:
        logging.debug('Standard errors is empty.')
    result = process.wait()
    logging.debug('Exit code is %d.' % result)

    return (result, output, errors)


def formatJavaString(text):
    escaped = '"'
    for character in text:
        if character == '"':
            escaped += '\\'
            escaped += character
        elif character == '\n':
            escaped += '\\'
            escaped += 'n'
        else:
            escaped += character
    escaped += '"'
    return escaped

       
def evalJava(code, log=True, mask=[], **kwargs):
    # Create a temporary file to contain the code to run:
    (sourceFd, sourceFileName) = tempfile.mkstemp(prefix="snippet", suffix='.java')

    # Compute the name of the class from the name of the file:
    className = os.path.basename(sourceFileName).replace('.java', '')

    # Generate the source:
    with os.fdopen(sourceFd, 'w') as sourceFile:
        lines = code.split('\n')
        imports = []
        body = []
        for line in lines:
            if line.startswith("import "):
                imports.append(line)
            else:
                body.append(line)
        for line in imports:
            sourceFile.write(line)
            sourceFile.write('\n')
        sourceFile.write('class %s {\n' % className)
        sourceFile.write('public static void main (String[] args) throws Exception {\n')
        for key, value in kwargs.items():
            sourceFile.write('final String %s = %s;\n' % (key, formatJavaString(value)))
        for line in body:
            sourceFile.write(line)
            sourceFile.write('\n')
        sourceFile.write('}\n')
        sourceFile.write('}\n')

    # Send the generated source to the log:
    logging.debug('Content of generated source file "%s" follows:' % sourceFileName)
    result = runCommand(['cat', sourceFileName], mask=mask)
    if result != 0:
        abort('Can\'t read the generated source file "%s".' % sourceFileName)

    # Compile the generated source code:
    result = runCommand(['javac', sourceFileName], mask=mask)
    if result != 0:
        abort('Error compiling Java source from file "%s".' % sourceFileName)

    # Remove the generated source: 
    removeFile(sourceFileName)

    # Execute the generated code:
    classFileName = sourceFileName.replace('.java', '.class')
    (result, output, errors) = evalCommand(['java', '-cp', os.path.dirname(classFileName), className], log=log, mask=mask)
    if result != 0:
        logging.error('Error running Java class from file "%s".' % classFileName)

    # Remove the generated class file: 
    removeFile(classFileName)

    return (result, output, errors)


def checkPackages():
    # Check the packages that are required before starting this
    # step of the migration:
    logging.info('Checking required packages ...')
    missing = []
    for package in REQUIRED_PACKAGES:
       if not checkPackage(package):
           missing.append(package)
    if missing:
       if len(missing) == 1:
           abort('Required package %s isn\'t installed.' % formatList(missing))
       else:
           abort('Required packages %s aren\'t installed.' % formatList(missing))


def checkSQLServer():
    # Check that the SQL Server database is available through the network (we
    # can't test real connection to the database, but at least we can check
    # that we can open a TCP connection):
    logging.info('Checking connection to the SQL Server database at host "%s" and port %s ...' % (sqlHost, sqlPort))
    sock = None
    try:
        sock = socket.create_connection((sqlHost, sqlPort))
    except Exception:
        logging.debug(traceback.format_exc())
        abort('The SQL Server database at host "%s" and port %s is not reachable.' % (sqlHost, sqlPort))
    finally:
        if sock:
            sock.close()


def checkPostgreSQL():
    # Check that the PostgreSQL database is up and running:
    logging.info('Checking the status of the PostgreSQL service ...')
    status = runCommand(["/sbin/service", "postgresql", "status"])
    if status != 0:
        abort('The PostgreSQL database service is not running.')

    # Open the connection to the database:
    logging.info('Checking connection to the PostgreSQL database at host "%s" and port %s ...' % (pgHost, pgPort))
    connectionString = "host='%s' port='%s' dbname='%s' user='%s' password='%s'" % (pgHost, pgPort, pgName, pgUser, pgPasswd)
    global connection
    try:
        connection = psycopg2.connect(connectionString)
    except Exception:
        logging.debug(traceback.format_exc())
        abort('Can\'t connect to the PostgreSQL database using connection string "%s".' % connectionString)


def checkSetup():
    # Check that manager has been installed:
    # XXX: Any other way to check this better?
    logging.info('Checking the installation of the application ...')
    instanceDir = '/usr/share/jbossas/server/rhevm-slimmed'
    if not os.path.isdir(instanceDir):
        abort('The installation of the manager is not complete (the directory "%s" doesn\'t exist). Did you run the "rhevm-setup" script?' % instanceDir)


def checkServer():
    # Ideally we should use "service jbossas status", but unfortunatelly that
    # doesn't return a correct code when the server is stopped, so whe have to
    # run it, get the output and look for the "stopped" word:
    # XXX: Fix this if the application server script is itself fixed.
    logging.info('Checking that the application server is stopped ...')
    result = runCommand(['/sbin/service jbossas status | grep stopped'], shell=True)
    if result != 0:
        abort('Please stop the application server (running "service jbossas stop") before starting the migration.')


def checkFiles():
    # Check that the directory containing files collected in the other machine
    # exists:
    logging.info('Checking files copied from Windows ...')
    if not os.path.exists(windowsFilesDir) or not os.path.isdir(windowsFilesDir):
        abort('Directory "%s" doesn\'t exist. Please copy the files from Windows before starting the migration.' % windowsFilesDir)

    # Check that all required files are available:
    missing = []
    for name in REQUIRED_FILES:
        path = os.path.join(windowsFilesDir, name)
        if not os.path.exists(path) or not os.path.isfile(path):
            missing.append(name)
    if missing:
        if len(missing) == 1:
            abort('Required file %s is missing from directory "%s".' % (formatList(missing), windowsFilesDir))
        else:
            abort('Required files %s are missing from directory "%s".' % (formatList(missing), windowsFilesDir))


def checkDomains():
    # Do nothing if the check is disabled in the command line:
    if not options.checkDomains:
        return

    # Check that at least one domain has been configured:
    logging.info('Checking domains ...')
    (result, output, _) = evalCommand(['/usr/bin/rhevm-manage-domains', '-action=list'])
    if result != 0:
        abort('The execution of the tool that checks domains failed.')
    lines = output.split('\n')
    if lines:
        for line in lines:
            if line.startswith('Domain:'):
                return
    abort('Please configure at least one domain (running "rhevm-manage-domains") before starting the migration.')


def backupPostgreSQL():
    logging.info('Making a backup of the PostgreSQL database ...')
    backupFile = '/var/log/rhevm/rhevm-migration-%s.sql' % datetime.datetime.now().strftime('%Y_%m_%d_%H_%M_%S')
    result = runCommand([
        '/usr/bin/pg_dump', 
            '--host', pgHost, 
            '--user', pgUser, 
            '--file', backupFile, 
            pgName
    ])
    if result != 0:
        abort('Backup of the PostgreSQL database failed.')
    else:
        logging.info('The backup of the PostgreSQL database is available in "%s".' % backupFile)


def generateContext(file):
    fd = None
    try:
        # Open the file:
        logging.debug('Generating context file "%s".' % file)
        fd = open(file, 'w')

        # Write the options for the SQL Server database:
        sqlUrl = 'jdbc:sqlserver://%s:%s;databaseName=%s' % (sqlHost, sqlPort, sqlName)
        logging.debug('SQL Server URL is "%s".' % sqlUrl)
        fd.write('rhevm_2_2_ClassName=com.microsoft.sqlserver.jdbc.SQLServerDriver\n')
        fd.write('rhevm_2_2_Schema=dbo\n');
        fd.write('rhevm_2_2_Login=%s\n' % sqlUser)
        fd.write('rhevm_2_2_Password=%s\n' % sqlPasswd)
        fd.write('rhevm_2_2_JdbcUrl=%s\n' % sqlUrl)
        fd.write('rhevm_2_2_MappingFile=id_MSSQL\n')

        # Write the properties for the PostgreSQL database:
        pgUrl = 'jdbc:postgresql://%s:%s/%s?stringtype=unspecified' % (pgHost, pgPort, pgName)
        logging.debug('PostgreSQL URL is "%s".' % pgUrl)
        fd.write('rhevm_3_0_ClassName=org.postgresql.Driver\n')
        fd.write('rhevm_3_0_Schema=%s\n' % pgName)
        fd.write('rhevm_3_0_Login=%s\n' % pgUser)
        fd.write('rhevm_3_0_Password=%s\n' % pgPasswd)
        fd.write('rhevm_3_0_JdbcUrl=%s\n' % pgUrl)
        fd.write('rhevm_3_0_MappingFile=postgres_id\n')
    finally:
        if fd != None:
            fd.close()

    # Dump the content of the file to the log:
    logging.debug('Content of the context file "%s" follows:' % file)
    result = runCommand(args=['cat', file], mask=[sqlPasswd, pgPasswd])
    if result != 0:
        abort('Failed to read the generated context file "%s".' % file)


def loadOptionsTable(table):
    # Load the options from the database:
    cursor = None
    rows = None
    try:
        cursor = connection.cursor()
        cursor.execute('select option_name, option_value from %s' % table)
        rows = cursor.fetchall()
        connection.commit()
    except Exception:
        logging.debug(traceback.format_exc())
        connection.rollback()
        abort('Can\'t load options from table "%s".' % table)
    finally:
        if cursor:
            cursor.close()

    # Build a dictionary containing the options:
    result = dict()
    for row in rows:
        result[row[0]] = row[1] 

    return result


def loadOptions():
    logging.info('Loading options ...')
    global windowsOptions
    windowsOptions = loadOptionsTable('vdc_options_old_2_2')
    global linuxOptions
    linuxOptions = loadOptionsTable('vdc_options')


def migrateDatabase():
    # Remove the copy of the "vdc_options" table, otherwise the
    # ETL script will complain and fail when we try to run more
    # more than once:
    logging.info('Removing old backup of the options table ...')
    cursor = None
    try:
        cursor = connection.cursor()
        cursor.execute("drop table if exists vdc_options_old_2_2")
        cursor.execute("drop sequence if exists vdc_options_old_2_2_seq")
        connection.commit()
    except Exception:
        logging.debug(traceback.format_exc())
        connection.rollback()
        abort('Can\'t remove old options table.')
    finally:
        if cursor:
            cursor.close()

    # Generate the context files:
    logging.info('Generating context files for the ETL script ...')
    etlContextFile = '/etc/rhevm/rhevm-migration/Default.properties'
    generateContext(etlContextFile)

    # The exit code returned by the ETL script is not really
    # useful as it is always 0 (at least in my experience) so we
    # have to check the status looking for exceptions in the log
    # file, and in order to do that we need to remove it before
    # starting:
    etlLogFile = '/var/log/rhevm/migration-etl.log'
    removeFile(etlLogFile)

    # Run the ETL script:
    logging.info('Running the ETL script ...')
    result = runCommand(['/usr/share/rhevm-migration-tool/migration_etl.sh'])
    if result != 0:
        abort('The ETL script failed with code %d. See the log file "%s" for details.' % (result, etlLogFile))

    # Now we need to look for lines with the "Exception" text
    # inside the ETL log file:
    with open(etlLogFile) as etlLogFd:
        for etlLogLine in etlLogFd:
            if etlLogLine.find("Exception") != -1:
                abort('The ETL script log file "%s" indicates exceptions.' % etlLogFile)

    # Calculate the path for the database upgrade script:
    upgradeDir = '/usr/share/rhevm/dbscripts'
    upgradeScript = os.path.join(upgradeDir, 'upgrade.sh')
    if not os.path.exists(upgradeScript):
        abort('The database upgrade script "%s" doesn\'t exist.' % upgradeScript)

    # Remove the database upgrade log file to get rid of messages
    # generated by previous executions:
    upgradeLogFile = upgradeScript + '.log'
    if os.path.exists(upgradeLogFile):
        removeFile(upgradeLogFile)

    # Run the upgrade scripts on the PostgreSQL database:
    logging.info('Upgrading the database ...')
    result = runCommand(cwd=upgradeDir, args=[upgradeScript, '-u', 'rhevm'])
    if result != 0:
        abort('The upgrade of the database failed with code "%d". See the log file "%s" for details.' % (result, upgradeLogFile))


def migrateOptions():
    # Load the Windows and Linux options:
    loadOptions()

    # Copy the options that don't need any modification:
    logging.info('Copying options that don\'t need translation ...')
    for optionName in ASIS_OPTIONS:
        optionValue = windowsOptions.get(optionName)
        if not optionValue:
            logging.debug('Option "%s" doesn\'t appear in the Windows table.' % optionName)
            continue
        logging.debug('Copying option "%s", value is "%s".' % (optionName, optionValue))
        cursor = None
        try:
            cursor = connection.cursor()
            cursor.execute('update vdc_options set option_value = %s where option_name = %s', (optionValue, optionName))
            connection.commit()
        except Exception:
            logging.debug(traceback.format_exc())
            connection.rollback()
            abort('Can\'t update option "%s".' % optionName)
        finally:
            if cursor:
                cursor.close()

    # Update the option that indicates the id of the default cluster:
    logging.info('Updating default cluster ...')
    defaultClusterOption = 'PowerClientAutoRegistrationDefaultVdsGroupID'
    defaultClusterName = None
    try:
        defaultClusterName = windowsResults.get('cluster', 'default')
    except ConfigParser.NoOptionError:
        abort('Can\'t find the default cluster name in the Windows results.')
    logging.debug('Default cluster name is "%s".' % defaultClusterName)
    cursor = None
    try:
        cursor = connection.cursor()
        cursor.execute('update vdc_options set option_value = (select vds_group_id from vds_groups where name = %s) where option_name = %s', (defaultClusterName, defaultClusterOption))
        connection.commit()
    except Exception:
        logging.debug(traceback.format_exc())
        connection.rollback()
        abort('Can\'t update default cluster.')
    finally:
        if cursor:
            cursor.close()

     # XXX: Handle options that require translation.
     #oVirtISOsRepositoryPath
     #BootstrapInstallerFileName


def migrateFiles():
    # Sysprep:
    logging.info('Copying sysprep files ...')
    for optionName in SYSPREP_OPTIONS:
        windowsValue = windowsOptions.get(optionName)
        if not windowsValue:
            logging.error('Can\'t find sysprep option "%s" in the Windows table.' % optionName)
            continue
        windowsPath = os.path.join(windowsFilesDir, 'service', windowsValue.replace('\\', '/'))
        if not os.path.exists(windowsPath):
            logging.error('Can\'t find sysprep file "%s". Full path is "%s". You will have to migrate it manually.' % (windowsValue, windowsPath))
            continue
        linuxPath = linuxOptions.get(optionName)
        if not linuxPath:
            logging.error('Can\'t find sysprep option "%s" in the Linux table.' % optionName)
            continue
        logging.debug('Copying sysprep file "%s" to "%s".' % (windowsPath, linuxPath))
        shutil.copy(windowsPath, linuxPath)


def findLdapURIs(domain):
    uris = []
    (result, output, _) = evalCommand(['dig', '-t', 'SRV', '+short', '_ldap._tcp.%s' % domain])
    if result != 0:
        abort('Can\'t find LDAP SRV records for domain "%s".' % domain)
    records = output.split('\n')
    if not records:
        abort('Can\'t find LDAP SRV records for domain "%s".' % domain)
    for record in records:
        if record == '':
            continue
        logging.debug('Found LDAP SRV record "%s" for domain "%s".' % (record, domain))
        fields = record.split()
        priority = int(fields[0])
        weight = int(fields[1])
        port = int(fields[2])
        host = fields[3][:-1]
        #logging.debug('LDAP SRV priority is %d.' % priority)
        #logging.debug('LDAP SRV weight is %d.' % weight)
        #logging.debug('LDAP SRV port is %d.' % port)
        #logging.debug('LDAP SRV host is %s.' % host)
        uri = 'ldap://%s:%d' % (host, port)
        logging.debug('LDAP server URI is "%s".' % uri)
        uris.append(uri)
    return uris


def runIPAQuery(uri, base, scope=ldap.SCOPE_SUBTREE, filter='(objectclass=*)', attrlist=None):
    logging.debug('Running IPA query "%s" on server "%s" with base "%s".' % (filter, uri, base))
    entries = []
    try:
       connection = ldap.initialize(uri)
       connection.simple_bind_s()
       connection.set_option(ldap.OPT_REFERRALS, 0)
       results = connection.search_s(base, scope, filter, attrlist)
       if results:
           for (_, attributes) in results:
               if isinstance(attributes, dict):
                   entries.append(attributes)
    except Exception:
        logging.debug('Can\'t run IPA query.')
        logging.debug(traceback.format_exc())
    finally:
       if connection:
           connection.unbind_s()
    return entries


def runADQuery(uri, user, password, base, scope=ldap.SCOPE_SUBTREE, filter='(objectclass=*)', attrlist=None):
    logging.debug('Running AD query "%s" on server "%s" with base "%s" and user "%s".' % (filter, uri, base, user))
    entries = []
    try:
       connection = ldap.initialize(uri)
       connection.simple_bind_s(user, password)
       connection.set_option(ldap.OPT_REFERRALS, 0)
       results = connection.search_s(base, scope, filter, attrlist)
       if results:
           for (_, attributes) in results:
               if isinstance(attributes, dict):
                   entries.append(attributes)
    except Exception:
        logging.debug('Can\'t run IPA query.')
        logging.debug(traceback.format_exc())
    finally:
       if connection:
           connection.unbind_s()
    return entries


def runLdapQuery(uri, user, password, base, scope=ldap.SCOPE_SUBTREE, filter='(objectclass=*)', attrlist=None):
    # First try with IPA, then with AD:
    entries = runIPAQuery(uri, base, scope, filter, attrlist)
    if not entries:
        entries = runADQuery(uri, user, password, base, scope, filter, attrlist)
    return entries


def findUserDirectoryId(domainName, userName):
    # Some debug info:
    logging.debug('Finding id for user "%s" in domain "%s" ...' % (userName, domainName))

    # Get the user and password for the domain:
    domainUser = domainUsers[domainName]
    domainPassword = domainPasswords[domainName]

    # Try to find the LDAP server for that domain:
    ldapUris = findLdapURIs(domainName)
    if not ldapUris:
        logging.debug('Can\'t find any LDAP server for domain "%s".' % domainName)
        return None

    # If more than one URI is avaible then use the first one:
    ldapUri = ldapUris[0]
    logging.debug('Selected LDAP URI is "%s", others will be ignored.' % ldapUri)
      
    # Find the default naming context:
    entries = runLdapQuery(ldapUri, domainUser, domainPassword, '', ldap.SCOPE_BASE, '(objectclass=*)', ['namingContexts'])
    if not entries:
        logging.debug('Can\'t find the naming contexts.')
        return None
    entry = entries[0]
    namingContexts = entry['namingContexts']
    namingContext = namingContexts[0]
    logging.debug('Selected naming context is "%s", others will be ignored.' % namingContext)

    # Find the LDAP entry for the user:
    ldapQuery = '(|(&(objectclass=ipaobject)(uid=%s))(&(objectclass=user)(sAMAccountName=%s)))' % (userName, userName)
    ldapAttributes = ['objectGUID', 'ipaUniqueID']
    entries = runLdapQuery(ldapUri, domainUser, domainPassword, namingContext, ldap.SCOPE_SUBTREE, ldapQuery, ldapAttributes)
    if not entries:
        logging.debug('Can\'t find entry for user "%s".' % userName)
        return None
    entry = entries[0]

    # Get the id of the user in the domain:
    directoryId = None
    if 'ipaUniqueID' in entry:
        directoryId = entry['ipaUniqueID'][0]
    elif 'objectGUID' in entry:
        directoryId = uuid.UUID(bytes_le=entry['objectGUID'][0])

    return directoryId

    
def findGroupDirectoryId(domainName, groupName):
    # Some debug info:
    logging.debug('Finding id for group "%s" in domain "%s" ...' % (groupName, domainName))

    # Get the group and password for the domain:
    domainUser = domainUsers[domainName]
    domainPassword = domainPasswords[domainName]

    # Try to find the LDAP server for that domain:
    ldapUris = findLdapURIs(domainName)
    if not ldapUris:
        logging.debug('Can\'t find any LDAP server for domain "%s".' % domainName)
        return None

    # If more than one URI is avaible then use the first one:
    ldapUri = ldapUris[0]
    logging.debug('Selected LDAP URI is "%s", others will be ignored.' % ldapUri)
      
    # Find the default naming context:
    entries = runLdapQuery(ldapUri, domainUser, domainPassword, '', ldap.SCOPE_BASE, '(objectclass=*)', ['namingContexts'])
    if not entries:
        logging.debug('Can\'t find the naming contexts.')
        return None
    entry = entries[0]
    namingContexts = entry['namingContexts']
    namingContext = namingContexts[0]
    logging.debug('Selected naming context is "%s", others will be ignored.' % namingContext)

    # Find the LDAP entry for the group:
    ldapQuery = '(|(&(objectclass=ipausergroup)(cn=%s))(&(objectclass=group)(sAMAccountName=%s)))' % (groupName, groupName)
    ldapAttributes = ['objectGUID', 'ipaUniqueID']
    entries = runLdapQuery(ldapUri, domainUser, domainPassword, namingContext, ldap.SCOPE_SUBTREE, ldapQuery, ldapAttributes)
    if not entries:
        logging.debug('Can\'t find entry for group "%s".' % groupName)
        return None
    entry = entries[0]

    # Get the id of the group in the domain:
    directoryId = None
    if 'ipaUniqueID' in entry:
        directoryId = entry['ipaUniqueID'][0]
    elif 'objectGUID' in entry:
        directoryId = uuid.UUID(bytes_le=entry['objectGUID'][0])
        
    return directoryId


def migrateUser(userName, databaseId):
    # Tell the user that we are migrating the user:
    logging.debug('Migrating user "%s" ...' % userName)

    # Try to find the user in one of the domains:
    logging.debug('Finding directory id for user "%s" ...' % userName)
    directoryId = None
    for domainName in domainNames:
        if domainName == 'internal':
            continue
        directoryId = findUserDirectoryId(domainName, userName)
        if directoryId:
            logging.debug('Found directory id "%s" for user "%s" in domain "%s", others will be ignored.' % (directoryId, userName, domainName))
            break
    if not directoryId:
        logging.debug('Can\'t find user "%s" in any domain, the user will not be migrated.' % userName)
        return False

    # Skip the user if the domain id is equal to the database id:
    if directoryId == databaseId:
        logging.debug('Domain id "%s" for user "%s" is equal to database id "%s", the user is already migrated.' % (directoryId, userName, databaseId))
        return True

    # Check if there is already an user in the database with the id found in
    # the directory. This can happen if that user has been added to the
    # database before the migration. Typical for the user automatically added
    # by the domain management tool:
    logging.debug('Checking users with database id equal to directory id "%s" ...' % directoryId)
    cursor = None
    userExists = False
    try:
        cursor = connection.cursor()
        query = 'select user_id from users where user_id = \'%s\'' % directoryId
        logging.debug('Query to check users with id "%s" is "%s".' % (directoryId, query))
        cursor.execute(query)
        rows = cursor.fetchall()
        if rows:
            userExists = True
        connection.commit()
    except Exception:
        logging.debug(traceback.format_exc())
        connection.rollback()
        abort('Can\'t check users with database id equal to directory id "%s".' % directoryId)
    finally:
        if cursor:
            cursor.close()

    # If there is already an user in the database with the same directory id
    # then we need to delete from the dtabase that user and its permissions:
    if userExists:
        logging.debug('Deleting old user and permissions ...')
        cursor = None
        try:
            cursor = connection.cursor()
            update = 'delete from permissions where ad_element_id = \'%s\'' % directoryId
            logging.debug('Statement to delete permissions is "%s".' % update)
            cursor.execute(update)
            update = 'delete from users where user_id = \'%s\'' % directoryId
            logging.debug('Statement to delete user is "%s".' % update)
            cursor.execute(update)
            connection.commit()
        except Exception:
            logging.debug(traceback.format_exc())
            connection.rollback()
            abort('Can\'t delete user and permissions with id "%s".' % databaseId)
        finally:
            if cursor:
                cursor.close()

    # Replace the database id with the directory id in the users and permissions
    # tables:
    logging.debug('Updating the user and the permissions ...')
    cursor = None
    try:
        cursor = connection.cursor()
        update = 'update users set user_id = \'%s\', domain = \'%s\', username = \'%s@%s\' where user_id = \'%s\'' % (directoryId, domainName, userName, domainName, databaseId)
        logging.debug('Update for users table is "%s".' % update)
        cursor.execute(update)
        update = 'update permissions set ad_element_id = \'%s\' where ad_element_id = \'%s\'' % (directoryId, databaseId)
        logging.debug('Update for permissions table is "%s".' % update)
        cursor.execute(update)
        connection.commit()
    except Exception:
        logging.debug(traceback.format_exc())
        connection.rollback()
        abort('Can\'t update ids for user "%s".' % userName)
        return False
    finally:
        if cursor:
            cursor.close()

    # Everything went okay:
    return True


def migrateGroup(groupName, databaseId):
    # Tell the user that we are migrating the group:
    logging.debug('Migrating group "%s" ...' % groupName)

    # Try to find the group in one of the domains:
    logging.debug('Finding directory id for group "%s" ...' % groupName)
    directoryId = None
    for domainName in domainNames:
        if domainName == 'internal':
            continue
        directoryId = findGroupDirectoryId(domainName, groupName)
        if directoryId:
            logging.debug('Found directory id "%s" for group "%s" in domain "%s", others will be ignored.' % (directoryId, groupName, domainName))
            break
    if not directoryId:
        logging.debug('Can\'t find group "%s" in any domain, the group will not be migrated.' % groupName)
        return False

    # Skip the group if the domain id is equal to the database id:
    if directoryId == databaseId:
        logging.debug('Domain id "%s" for group "%s" is equal to database id "%s", the group is already migrated.' % (directoryId, groupName, databaseId))
        return True

    # Replace the database id with the directory id in the groups and permissions
    # tables:
    logging.debug('No group with directory id "%s" exists in the database, will update the group and the permissions.' % directoryId)
    logging.debug('Database id is "%s" and directory id is "%s" ...' % (databaseId, directoryId))
    cursor = None
    try:
        cursor = connection.cursor()
        update = 'update ad_groups set id = \'%s\', domain = \'%s\', name = \'%s\' where id = \'%s\'' % (directoryId, domainName, groupName, databaseId)
        logging.debug('Update for groups table is "%s".' % update)
        cursor.execute(update)
        update = 'update permissions set ad_element_id = \'%s\' where ad_element_id = \'%s\'' % (directoryId, databaseId)
        logging.debug('Update for permissions table is "%s".' % update)
        cursor.execute(update)
        connection.commit()
    except Exception:
        logging.debug(traceback.format_exc())
        connection.rollback()
        abort('Can\'t update ids for group "%s".' % groupName)
        return False
    finally:
        if cursor:
            cursor.close()

    # Everything went okay:
    return True


def encryptPasswords():
    # Encrypt the builtin admin@internal password:
    logging.info('Encrypting admin@internal password ...')
    adminPassword = domainPasswords['internal']
    (result, encryptedAdminPassword, _) = evalJava(ENCRYPT_SNIPPET, file=ksFile, alias=ksAlias, password=ksPass, mask=[ksPass, adminPassword], text=adminPassword)
    if result != 0:
        abort('Failed to encrypt admin@internal password.')
    cursor = None
    try:
        cursor = connection.cursor()
        cursor.execute('update vdc_options set option_value = %s where option_name = \'AdminPassword\'', (encryptedAdminPassword,))
        connection.commit()
    except Exception:
        logging.debug(traceback.format_exc())
        connection.rollback()
        abort('Can\'t update encrypted admin@internal password.')
    finally:
        if cursor:
            cursor.close()

    # Encrypt the domain passwords:
    logging.debug('Encrypting domain passwords ...')
    domainPasswordMap = ''
    for domainName in domainNames:
        if domainName == 'internal':
            continue
        logging.info('Encrypting password for domain "%s" ...' % domainName)
        domainPassword = domainPasswords[domainName]
        (result, encryptedDomainPassword, _) = evalJava(ENCRYPT_SNIPPET, file=ksFile, alias=ksAlias, password=ksPass, mask=[ksPass, domainPassword], text=domainPassword)
        if result != 0:
            abort('Failed to encrypt password for domain "%s".', domainName)
        domainPasswordPair = "%s:%s" % (domainName, encryptedDomainPassword)
        if domainPasswordMap != '':
            domainPasswordMap += ','
        domainPasswordMap += domainPasswordPair
    logging.debug('Domain passwords map is "%s".' % domainPasswordMap)

    # Save the encrypted domain passwords:
    logging.debug('Saving domain passwords map ...')
    cursor = None
    try:
        cursor = connection.cursor()
        cursor.execute('update vdc_options set option_value = %s where option_name = \'AdUserPassword\'', (domainPasswordMap,))
        connection.commit()
    except Exception:
        logging.debug(traceback.format_exc())
        connection.rollback()
        abort('Can\'t update domain password map.')
    finally:
        if cursor:
            cursor.close()


def migrateUsersAndGroups():
    # Check the type of authentication that was used in the original
    # installation and skip the rest or the process if it isn't local:
    logging.info('Checking the authentication method ...')
    authMethod = windowsOptions['AuthenticationMethod']
    logging.debug('Authentication method is "%s".' % authMethod)
    if authMethod != 'Windows':
        return

    migrateUsers()
    migrateGroups()


def migrateUsers():
    # Get the list of users:
    logging.info('Loading the list of users ...')
    localDomain = windowsOptions['DomainName']
    logging.debug('Local domain is "%s".' % localDomain)
    cursor = None
    rows = None
    try:
        cursor = connection.cursor()
        query = 'select username, user_id from users where domain = \'%s\'' % localDomain
        logging.debug('Query to load users is "%s".' % query)
        cursor.execute(query)
        rows = cursor.fetchall()
        connection.commit()
    except Exception:
        logging.debug(traceback.format_exc())
        connection.rollback()
        abort('Can\'t get the list of users.')
    finally:
        if cursor:
            cursor.close()

    # Build a dictionary containing the user name as key and the unique id as
    # value:
    usersMap = dict()
    for row in rows:
        userName = row[0]
        atIndex = userName.find('@')
        if atIndex != -1:
            userName = userName[:atIndex]
        userName = userName.lower()
        userId = row[1]
        logging.debug('Loaded user with name "%s" and database id "%s".' % (userName, userId))
        usersMap[userName] = userId

    # For each user try to find it in one of the authentication directories:
    logging.info('Migrating users ...')
    failedUsers = set([])
    for userName, userId in usersMap.items():
        if not migrateUser(userName, userId):
            failedUsers.add(userName)

    # Show the list of users that haven't been migrated:
    if failedUsers:
        logging.info('\n'
            'The following user(s) have been migrated, but they were not\n'
            'located in any of the domains that you have configured:\n')

        for failedUser in failedUsers:
            logging.info('  %s' % failedUser)

        logging.info('\n'
            'While the user entries have been migrated, they will not be able\n'
            'to log in to the administration or user portals. In order to\n'
            'solve this problem please create these users in one of your\n'
            'domains and then run the "rhevm-migration" tool again with the\n'
            '"-u" option. The "-u" option will only try to locate users in the\n'
            'domain to update their identities.\n')

    # Warn that the "Administrator" user will not be able to log-in if using
    # active directory:
    if "administrator" in usersMap:
        logging.info(
            'The migration tool detected that you have migrated the\n'
            '\'Administrator\' user. Please pay attention that in case the\n'
            'target directory being used is Active Directory, by default the\n'
            'domain administrator account does not have a Universal Principal\n'
            'Name (UPN) and cannot be used to login into RHEV-M.\n')


def migrateGroups():
    # Get the list of groups:
    logging.info('Loading the list of groups ...')
    localDomain = windowsOptions['DomainName']
    logging.debug('Local domain is "%s".' % localDomain)
    cursor = None
    rows = None
    try:
        cursor = connection.cursor()
        query = 'select name, id from ad_groups where domain = \'%s\'' % localDomain
        logging.debug('Query to load groups is "%s".' % query)
        cursor.execute(query)
        rows = cursor.fetchall()
        connection.commit()
    except Exception:
        logging.debug(traceback.format_exc())
        connection.rollback()
        abort('Can\'t get the list of groups.')
    finally:
        if cursor:
            cursor.close()

    # Build a dictionary containing the user name as key and the unique id as
    # value:
    groupsMap = dict()
    for row in rows:
        groupName = row[0]
        groupId = row[1]
        logging.debug('Loaded group with name "%s" and database id "%s".' % (groupName, groupId))
        groupsMap[groupName] = groupId

    # For each group try to find it in one of the authentication directories:
    logging.info('Migrating groups ...')
    failedGroups = set([])
    for groupName, groupId in groupsMap.items():
        if not migrateGroup(groupName, groupId):
            failedGroups.add(groupName)

    # Show the list of groups that haven't been migrated:
    if failedGroups:
        logging.info('\n'
            'The following group(s) have been migrated, but they were not\n'
            'located in any of the domains that you have configured:\n')

        for failedGroup in failedGroups:
            logging.info('  %s' % failedGroup)

        logging.info('\n'
            'While the group entries have been migrated, the permissions will\n'
            'not work correctly. In order to solve this problem please create\n'
            'these groups in one of your domains and then run the\n'
            '"rhevm-migration" tool again with the "-u" option. The "-u"\n'
            'option will only try to locate groups in the domain to\n'
            'update their identities.\n')


def decryptPasswords():
    # Decrypt the builtin administrator password before replacing
    # the keystore:
    logging.info('Decrypting admin@internal password ...')
    encryptedAdminPasswd = linuxOptions['AdminPassword']
    logging.debug('Encrypted admin@internal password is "%s".' % encryptedAdminPasswd)
    global domainNames
    global domainUsers
    global domainPasswords
    (result, adminPassword, _) = evalJava(DECRYPT_SNIPPET, log=False, file=ksFile, alias=ksAlias, password=ksPass, mask=[ksPass], text=encryptedAdminPasswd)
    if result != 0:
        abort('Failed to decrypt admin@internal password.')
    domainNames.add('internal')
    domainUsers['internal'] = 'admin'
    domainPasswords['internal'] = adminPassword
    
    # Get the list of domains users:
    logging.debug('Loading domain users  ...')
    domainUserMap = linuxOptions['AdUserName']
    domainUserPairs = domainUserMap.split(',')
    for domainUserPair in domainUserPairs:
        colonIndex = domainUserPair.find(':')
        if colonIndex != -1:
            domainName = domainUserPair[0:colonIndex].strip()
            logging.debug('Domain name is "%s".' % domainName)
            userName = domainUserPair[colonIndex + 1:].strip()
            logging.debug('User name is "%s".' % userName)
            domainNames.add(domainName)
            domainUsers[domainName] = userName
        else:
            logging.debug('Can\'t find separator ":" in domain user pair "%s".' % domainUserPair)

    # Decrypt the passwords of the domains:
    domainPasswordMap = linuxOptions['AdUserPassword']
    domainPasswordPairs = domainPasswordMap.split(',')
    for domainPasswordPair in domainPasswordPairs:
        colonIndex = domainPasswordPair.find(':')
        if colonIndex != -1:
            domainName = domainPasswordPair[0:colonIndex].strip()
            logging.info('Decrypting password for domain "%s" ...' % domainName)
            encryptedDomainPassword = domainPasswordPair[colonIndex + 1:].strip()
            logging.debug('Domain encrypted password is "%s".' % encryptedDomainPassword)
            (result, domainPassword, _) = evalJava(DECRYPT_SNIPPET, log=False, file=ksFile, alias=ksAlias, password=ksPass, mask=[ksPass], text=encryptedDomainPassword)
            if result != 0:
                abort('Failed to decrypt password for domain "%s".' % domainName)
            domainNames.add(domainName)
            domainPasswords[domainName] = domainPassword
        else:
            logging.debug('Can\'t find separator ":" in domain password pair "%s".' % domainPasswordPair)


def extractSubjectAttributes(certificate):
    # Extract the subject string:
    (result, subject, _) = evalCommand([
       '/usr/bin/openssl', 'x509',
       '-in', certificate,
       '-noout',
       '-subject'
    ])
    if result != 0:
        abort('Can\'t get subject from certificate "%s".' % certificate)
    subject = subject.replace('subject=', '').strip()
    logging.debug('Subject from the certificate "%s" is "%s".' % (certificate, subject))

    # Extract the attributes from the certificate string:
    attributes = dict()
    components = subject.split('/')
    for component in components:
        if not component:
            continue
        separator = component.find('=')
        if separator == -1:
            logging.debug('Can\'t find separator "=" inside subject component "%s".' % component)
            continue
        name = component[:separator]
        value = component[separator + 1:]
        logging.debug('Value for attribute "%s" is "%s".' % (name, value))
        attributes[name] = value

    # Make sure that we have C, O and CN:
    if not 'C' in attributes:
        abort('The subject "%s" extracted from certificate "%s" doesn\'t contain the C attribute.' % (subject, certificate))
    if not 'O' in attributes:
        abort('The subject "%s" extracted from certificate "%s" doesn\'t contain the O attribute.' % (subject, certificate))
    if not 'CN' in attributes:
        abort('The subject "%s" extracted from certificate "%s" doesn\'t contain the CN attribute.' % (subject, certificate))

    return attributes


def migrateCA():
    # The base directories for the source Windows and Linux
    # certificate authorities:
    logging.info('Replacing CA ...')
    global windowsCaDir
    windowsCaDir = os.path.join(windowsFilesDir, "service/ca")

    # Replace the CA public and private keys:
    logging.debug('Replacing CA private keys ...')
    replaceFile(windowsCaDir, linuxCaDir, 'ca.pem')
    replaceFile(windowsCaDir, linuxCaDir, 'private/ca.pem')

    # Generate the DER version of the CA certificate (this is not available in
    # the files copied from the old installation):
    logging.debug('Generating DER version of the CA certificate ...')
    caKeyFile = os.path.join(linuxCaDir, 'private/ca.pem')
    caPemFile = os.path.join(linuxCaDir, 'ca.pem')
    caDerFile = os.path.join(linuxCaDir, 'certs/ca.der')
    result = runCommand(cwd=linuxCaDir, args=[
        '/usr/bin/openssl', 'x509', 
            '-in', caPemFile, 
            '-outform', 'DER', 
            '-out', caDerFile,
    ])
    if result != 0:
        abort('Can\'t create DER version of the CA certificate.')

    # Extract the attributes and the organization from the CA
    # certificate copied from Windows:
    logging.debug('Extracting CA certificate attributes ...')
    caAttributes = extractSubjectAttributes(caPemFile)

    # Generate the cacert.conf file from the template:
    logging.debug('Regenerating cacert.conf from template ...')
    cacertTemplateFile = os.path.join(linuxCaDir, 'cacert.template')
    cacertConfFile = os.path.join(linuxCaDir, 'cacert.conf')
    copyFile(cacertTemplateFile, cacertConfFile)
    with open(cacertConfFile, 'a') as cacertConfFd:
        cacertConfFd.write("C = %s\n" % caAttributes['C'])
        cacertConfFd.write("O = %s\n" % caAttributes['O'])
        cacertConfFd.write("CN = %s\n" % caAttributes['CN'])

    # Remove the keystore file and create it again with the CA certificate
    # inside:
    logging.debug('Replacing keystore file ...')
    removeFile(ksFile)
    result = runCommand(cwd=linuxCaDir, args=[
        '/usr/bin/keytool',
            '-keystore', ksFile,
            '-storepass', ksPass,
            '-importcert',
            '-trustcacerts',
            '-noprompt',
            '-alias', 'cacert',
            '-file', caDerFile,
    ])
    if result != 0:
        abort('Can\'t import CA certificate into keystore.')

    # Replace all the files corresponding to the certificates
    # database:
    logging.debug('Replacing certificate database ...')
    replaceFile(windowsCaDir, linuxCaDir, 'database.txt')
    replaceFile(windowsCaDir, linuxCaDir, 'database.txt.old')
    replaceFile(windowsCaDir, linuxCaDir, 'database.txt.attr')
    replaceFile(windowsCaDir, linuxCaDir, 'database.txt.attr.old')
    replaceFile(windowsCaDir, linuxCaDir, "serial.txt")
    replaceFile(windowsCaDir, linuxCaDir, "serial.txt.old")

    # Remove all the certificates that have been generated during
    # the installation in the Linux machine and replace them with
    # those from the Windows machine:
    logging.debug('Replacing certificates ...')
    removeFiles(linuxCaDir, 'certs/[0-9]+\.pem')
    copyFiles(windowsCaDir, linuxCaDir, 'certs/[0-9]+\.pem')

    # Remove the requests generated during the Linux installation
    # and copy those from the Windows installation:
    logging.debug('Replacing certificate requests ...')
    removeFiles(linuxCaDir, 'certs/[0-9]+\.pem')
    removeFiles(linuxCaDir, 'requests/.*')
    copyFiles(windowsCaDir, linuxCaDir, 'requests/.*')

    # Get the subject from the manager certificate generated during the Linux
    # installation:
    managerAttributes = extractSubjectAttributes(os.path.join(linuxCaDir, 'certs/rhevm.cer'))
    csrSubject = "/C=%s/O=%s/CN=%s" % (caAttributes['C'], caAttributes['O'], managerAttributes['CN'])
    logging.debug('The subject for the manager certificate is "%s".' % csrSubject)

    # Remove the certificates for the manager generated during the
    # installation in the Linux machine:
    logging.debug('Removing manager certificate ...')
    removeFile(linuxCaDir, 'certs/rhevm.der')

    # Generate a CSR for the manager from the old private key. This is needed
    # in order to preserve the key pair, otherwise the SSH public key will not
    # match the public keys stored in the hosts and re-installation will break:
    logging.debug('Generating manager certificate request ...')
    result = runCommand(mask=[ksPass], args=[
        '/usr/bin/openssl', 'req',
            '-new',
            '-subj', csrSubject,
            '-key', os.path.join(windowsCaDir, 'keys/rhevm.pem'),
            '-out', os.path.join(linuxCaDir, 'requests/rhevm.req'),
            '-passin', 'pass:%s' % ksPass
    ])
    if result != 0:
        abort('Can\'t create manager certificate signing request.')

    # Sign the certificate request starting one day in the past:
    logging.debug('Signing manager certificate ...')
    certDate = datetime.date.today() - datetime.timedelta(days=1)
    result = runCommand(cwd=linuxCaDir, mask=[ksPass], args=[
        './SignReq.sh',
            'rhevm.req',
            'rhevm.cer',
            '1800', 
            linuxCaDir,
            certDate.strftime('%y%m%d%H%M%S+0000'),
            ksPass
    ])
    if result != 0:
        abort('Can\'t sign manager certificate request.')

    # Generate the DER version of the manager certificate:
    logging.debug('Generating DER version of the manager certificate ...')
    result = runCommand(cwd=linuxCaDir, args=[
        '/usr/bin/openssl', 'x509', 
            '-in', 'certs/rhevm.cer',
            '-out', 'certs/rhevm.der',
            '-outform', 'DER', 
    ])
    if result != 0:
        abort('Can\'t create DER version of the manager certificate.')

    # Export the manager private key and certificate to a PKCS12 file:
    logging.debug('Exporting manager private key and certificate as PKCS12 ...')
    result = runCommand(mask=[ksPass], args=[
        '/usr/bin/openssl', 'pkcs12',
            '-export',
            '-name', ksAlias,
            '-inkey', os.path.join(windowsCaDir, 'keys/rhevm.pem'),
            '-in', os.path.join(linuxCaDir, 'certs/rhevm.cer'),
            '-passin', 'pass:%s' % ksPass,
            '-out', os.path.join(linuxCaDir, 'certs/rhevm.pfx'),
            '-passout', 'pass:%s' % ksPass
    ])
    if result != 0:
        abort('Can\'t create PKCS12 file containing the manager private key and certificate.')

    # Import the PKCS12 file into the keystore:
    logging.debug('Importing manager private key and certificate into keystore ...')
    result = runCommand(mask=[ksPass], args=[
        '/usr/bin/keytool',
            '-keystore', ksFile,
            '-importkeystore',
            '-srckeystore', os.path.join(linuxCaDir, 'certs/rhevm.pfx'),
            '-srcalias', ksAlias,
            '-srcstoretype', 'PKCS12',
            '-srcstorepass', ksPass,
            '-destalias', ksAlias,
            '-deststorepass', ksPass
    ])
    if result != 0:
        abort('Can\'t import the PKCS12 file into the keystore.')

    # Export the manager public key: 
    logging.debug('Importing manager public key ...')
    result = runCommand(cwd=linuxCaDir, mask=[ksPass], args=[
        './exportK2SSH.sh',
            ksFile,
            ksAlias,
            'rhevm.pub',
            ksPass
    ])
    if result != 0:
        abort('Can\'t export manager public key.')

    # Convert the RHEV-M public key into an SSH key:
    logging.debug('Exporting manager SSH key ...')
    managerSshFile = os.path.join(linuxCaDir, 'keys/rhevm.ssh.key.txt')
    with open(managerSshFile, 'w') as managerSshFd:
        result = runCommand(cwd=linuxCaDir, stdout=managerSshFd, args=[
            './pubkey2ssh',
                'keys/rhevm.pub',
                ksAlias
        ])
    if result != 0:
        abort('Can\'t generate manager SSH key.')

    # Copy the CA certificate and the SSH key of the manager to the root web
    # application:
    logging.debug('Copying CA certificate and SSH key to web root ...')
    webRootDir = '/var/lib/jbossas/server/rhevm-slimmed/deploy/ROOT.war'
    copyFile(os.path.join(linuxCaDir, 'keys/rhevm.ssh.key.txt'), os.path.join(webRootDir, 'rhevm.ssh.key.txt'))
    copyFile(os.path.join(linuxCaDir, 'certs/ca.der'), os.path.join(webRootDir, 'ca.crt'))


def loadHosts():
    # In order to check the hosts we need to connect to them using SSH, and in
    # order to do that the need the decrypt the SSH private key from the
    # Windows machine:
    logging.info('Decrypting SSH key ...')
    global sshPem
    sshPem = os.path.join(windowsFilesDir, 'service/ca/keys/rhevm.pem')
    global sshKey
    sshKey = os.path.join(windowsFilesDir, 'service/ca/keys/rhevm.key')
    result = runCommand(mask=[ksPass], args=[
        '/usr/bin/openssl', 'rsa',
            '-in', sshPem,
            '-out', sshKey,
            '-passin', 'pass:%s' % ksPass
    ])
    if result != 0:
        abort('Can\'t decrypt SSH private key.')

    # Change the permissions of the private key file so that SSH will not
    # complain:
    os.chmod(sshKey, stat.S_IRUSR)

    # Get the list of hosts from the Windows results (it is a comman separated
    # list where each element is the host name followed by a colon and the host
    # address):
    logging.info('Loading hosts list ...')
    hostsInfo = windowsResults.get('host', 'list')
    logging.debug('Hosts info is "%s".' % hostsInfo)

    # For each host get its name and its address:
    hostsInfo = re.split('\s*,\s*', hostsInfo)
    global hostsList
    for hostInfo in hostsInfo:
        logging.debug('Host info is "%s".' % hostInfo)
        colonIndex = hostInfo.find(':')
        if colonIndex != -1:
            hostName = hostInfo[:colonIndex].strip()
            hostAddress = hostInfo[colonIndex + 1:].strip()
        else:
            hostName = hostInfo
            hostAddress = hostInfo
        hostsList.append((hostName, hostAddress))


def checkHosts():
    # Do nothing if the check of hosts is disabled in the command line:
    if not options.migrateHosts:
        return

    # Figure out the host name and port number that should be writen to the
    # vdsm registration configuration file:
    logging.debug('Finding port numbers ...')
    serverFile = '/usr/share/jbossas/server/rhevm-slimmed/deploy/jbossweb.sar/server.xml'
    serverDoc = libxml2.parseFile(serverFile)
    serverContext = serverDoc.xpathNewContext()
    global httpPort
    httpPort = serverContext.xpathEval(
        "string("
            "/Server"
            "/Service[@name='jboss.web']"
            "/Connector[@protocol='HTTP/1.1' and not(@SSLEnabled)]"
            "/@port"
        ")")
    if not httpPort:
        abort('Can\'t find the HTTP port number inside file "%s".' % serverFile)
    logging.debug('HTTP port is "%s".' % httpPort)
    global httpsPort
    httpsPort = serverContext.xpathEval(
        "string("
            "/Server"
            "/Service[@name='jboss.web']"
            "/Connector[@protocol='HTTP/1.1' and @SSLEnabled='true']"
            "/@port"
        ")")
    if not httpsPort:
        abort('Can\'t find the HTTPS port number inside file "%s".' % serverFile)
    logging.debug('HTTPS port is "%s".' % httpsPort)
    serverContext.xpathFreeContext()
    serverDoc.freeDoc()

    # Find the fully qualified host name of the manager:
    global httpHost
    httpHost = socket.gethostname()
    logging.debug('HTTP host is "%s".' % httpHost)

    # Check each host:
    logging.info('Checking hosts ...')
    global hostsFailed
    for hostName, hostAddress in hostsList:
        if not checkHost(hostName, hostAddress):
            hostsFailed.append(hostName)

    # If any host check failed let the user know and request
    # confirmation to continue:
    if hostsFailed:
        logging.info('\n'
            'The following hosts(s) can\'t be reached using SSH:\n')

        for hostFailed in hostsFailed:
            logging.info('  %s' % hostFailed)

        logging.info('\n'
            'This means that the migration tool will not be able to connect to\n'
            'the host(s) and update the VDSM registration configuration file\n'
            '/etc/vdsm-reg/vdsm-reg.conf. This is not a serious problem and the\n'
            'host(s) will continue working.\n'
            '\n'
            'You can continue with the rest of the migration process and update\n'
            'the file later, in particular you will need to update the\n'
            'following configuration parameters:\n')
        logging.info(' vdc_host_name=%s' % httpHost)
        logging.info(' vdc_host_port=%s' % httpsPort)

        while True:
            result = askQuestion('\n'
                'Do you want to continue with the migration without updating the\n'
                'configuration of the unreachable host(s)?', default='yes')
            if result != 'yes' and result != 'no':
                logging.info('Please type "yes" or "no".')
            else:
                break
        if result == 'no':
            abort('Aborted by request of the user.')


def checkHost(hostName, hostAddress):
    # Check that we can connect to the host with SSH using the manager key:
    logging.info('Checking host "%s" ...' % hostName)
    result = runRemoteCommand(user='root', host=hostAddress, key=sshKey, args=['hostname'], log=True)
    if result != 0:
        logging.debug('SSH connection to host "%s" using address "%s" failed.')
        return False
    else:
        return True


def migrateHosts():
    # Do nothing if the check of hosts is disabled in the command line:
    if not options.migrateHosts:
        return

    # Perform the host migration tasks:
    logging.info('Migrating hosts ...')
    for hostName, hostAddress in hostsList:
        if not hostName in hostsFailed:
            migrateHost(hostName, hostAddress)
        else:
            logging.debug('Host "%s" will not be migrated because it didn\'t pass the checks.' % hostName)


def migrateHost(hostName, hostAddress):
    # Give some feedback to the user:
    logging.info('Migrating host "%s" ...' % hostName)

    # Run the script to update the VDSM registration configuration file:
    logging.debug('Fixing the registration configuration file ...')
    vdsmRegConf = '/etc/vdsm-reg/vdsm-reg.conf'
    result = runRemoteCommand(user='root', host=hostAddress, key=sshKey, log=True, args=[
        'if', '[', '-f', vdsmRegConf, '];', 'then', 
            'sed', '-i', '-c',
            '-e', 's/^vdc_host_name=.*$/vdc_host_name=%s/' % httpHost,
            '-e', 's/^vdc_host_port=.*$/vdc_host_port=%s/' % httpsPort,
            vdsmRegConf, ';',
        'fi'
    ])
    if result != 0:
        abort('Failed to fix the registration configuration file for host "%s".')

    # XXX: If this task gets longer it would be better to create a new
    # "vds-migration.py" script and then copy and execute it in the host.
        
def main():
    try:

        # Parse the command line and prepare the logging
        # configuration:
        parseCommandLine()
        configureLogging()

        # Some things we can check before asking any question to the user:
        checkPackages()
        checkSetup()
        checkServer()
        checkDomains()

        # Perform the migration tasks:
        if options.usersOnly:
            askPostgreSQLDetails()
            checkPostgreSQL()
            askContinue()
            backupPostgreSQL()
            loadOptions()
            decryptPasswords()
            migrateUsersAndGroups()
        else:
            askWindowsFilesDir()
            askSQLServerDetails()
            askPostgreSQLDetails()
            checkSQLServer()
            checkPostgreSQL()
            checkFiles()
            loadHosts()
            checkHosts()
            askContinue()
            backupPostgreSQL()
            migrateDatabase()
            loadOptions()
            migrateOptions()
            migrateFiles()
            decryptPasswords()
            migrateCA()
            encryptPasswords()
            migrateUsersAndGroups()
            migrateHosts()

        # If we are here then the process finished correctly, tell
        # the user:
        logging.info(
            '\n'
            '\033[1;32mThe migration process finished successfully.\033[0;39m\n'
            '\n'
            'You can now start the application server (running "service jbossas\n'
            'start") and then connect to the administration tool to check that\n'
            'all your Data Centers, Clusters, Hosts, Virtual Machines,\n'
            'Templates, etc, have been correctly migrated.\n' 
            '\n'
            'The RHEV-H ISO images haven\'t been copied automatically, you will\n'
            'need to transfer them manually from the Windows machine to the\n'
            '"/usr/share/rhev-hypervisor" directory in the Linux machine.\n'
            '\n'
            'Remember that if something has not been migrated correctly you can\n'
            'rollback to your previous installation stoping the application\n'
            'server (running "service jbossas stop") and then restarting the\n'
            'services in the Windows machine.\n')

    finally:

        # Remember to close the database connection if it is still
        # open:
        if connection:
             connection.close()


if __name__ == "__main__":
    try:
        main()
    except SystemExit:
        raise
    except BaseException as exception:
        logging.error(traceback.format_exc())
        sys.exit(1)
