using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Controls;
using System.Windows;
using System.Collections;
using System.ComponentModel;
using System.Linq;
using System.Windows.Controls.Primitives;
using System.Windows.Data;
using System.Windows.Media;
using System.Windows.Input;
using System.Globalization;

namespace UI.WPFClient
{
	public class SortListViewEventArgs : RoutedEventArgs
	{
		public string SortPropertyName { get; private set; }
		public ListSortDirection SortDirection { get; private set; }


		public SortListViewEventArgs(RoutedEvent routedEvent, string sortPropertyName, ListSortDirection sortDirection)
			: base(routedEvent)
		{
			SortPropertyName = sortPropertyName;
			SortDirection = sortDirection;
		}
	}


	public class QListView : System.Windows.Controls.ListView
	{
		protected override DependencyObject GetContainerForItemOverride()
		{
			return new QListViewItem();
		}

		protected override void OnInitialized(EventArgs e)
		{
			//Add the event handler to the GridViewColumnHeader. This strongly ties this ListView to a GridView.
			AddHandler(ButtonBase.ClickEvent, new RoutedEventHandler(GridViewColumnHeaderClickedHandler));


			GridView gridView = View as GridView;
			if (gridView != null)
			{
				QGridViewColumn column = gridView.Columns.OfType<QGridViewColumn>().SingleOrDefault(a => a.IsDefaultSortColumn);
				if (column != null)
				{
					Sort(column);
					SelectedIndex = 0;
				}
			}


			base.OnInitialized(e);
		}

		#region Sorting Related Members

		public bool UseExternalSort { get; set; }

		public static RoutedEvent SortExternalEvent =
			EventManager.RegisterRoutedEvent("SortExternal", RoutingStrategy.Bubble, typeof(RoutedEventHandler), typeof(QListView));

		public event RoutedEventHandler SortExternal
		{
			add { AddHandler(SortExternalEvent, value); }
			remove { RemoveHandler(SortExternalEvent, value); }
		}


		private void Sort(QGridViewColumn column)
		{
			//Ensure that the column header is the correct type and a sort property has been set.
			if (!String.IsNullOrEmpty(column.SortPropertyName))
			{
				ListSortDirection direction;
				bool newSortColumn = false;

				//Determine if this is a new sort, or a switch in sort direction.
				QGridViewColumn lastSortedOnColumn = FindLastSortedColumn();
				if (lastSortedOnColumn == null
					|| String.IsNullOrEmpty(lastSortedOnColumn.SortPropertyName)
					|| !String.Equals(column.SortPropertyName, lastSortedOnColumn.SortPropertyName, StringComparison.InvariantCultureIgnoreCase))
				{
					newSortColumn = true;
					direction = ListSortDirection.Ascending;
				}
				else
				{
					direction = lastSortedOnColumn.SortDirection == ListSortDirection.Ascending ? ListSortDirection.Descending : ListSortDirection.Ascending;
				}

				//Sort the data.
				if (UseExternalSort)
				{
					ExternalSort(column.SortPropertyName, direction);
				}
				else
				{
					InternalSort(column.SortPropertyName, direction);
				}

				column.SortDirection = direction;
				column.IsSorted = true;


				//Make previously sorted header to be unsorted.
				if (newSortColumn && lastSortedOnColumn != null)
				{
					lastSortedOnColumn.IsSorted = false;
				}
			}
		}

		private void InternalSort(string sortBy, ListSortDirection direction)
		{
			ICollectionView dataView = CollectionViewSource.GetDefaultView(ItemsSource);
			if (dataView != null)
			{
				dataView.SortDescriptions.Clear();
				SortDescription sd = new SortDescription(sortBy, direction);
				dataView.SortDescriptions.Add(sd);
				dataView.Refresh();
			}
		}

		private void ExternalSort(string sortBy, ListSortDirection direction)
		{
			SortListViewEventArgs args = new SortListViewEventArgs(SortExternalEvent, sortBy, direction);
			RaiseEvent(args);
		}

		#endregion

		#region Grouping Related Members

		private QGridViewColumn lastGroupedColumn;
		private int lastGroupedColumnIndex;


		public string GroupPropertyName
		{
			get { return (string)GetValue(GroupPropertyNameProperty); }
			set { SetValue(GroupPropertyNameProperty, value); }
		}

		public static readonly DependencyProperty GroupPropertyNameProperty =
			DependencyProperty.Register("GroupPropertyName", typeof(string), typeof(QListView), new UIPropertyMetadata(null, new PropertyChangedCallback(GroupPropertyNamePropertyChanged)));

		private static void GroupPropertyNamePropertyChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
		{
			(d as QListView).GroupPropertyNameChanged();
		}


		private void GroupPropertyNameChanged()
		{
			//Find a column corresponding to specified group property name.
			GridView gridView = View as GridView;
			if (gridView != null)
			{
				//Show previously grouped column.
				if (lastGroupedColumn != null)
				{
					gridView.Columns.Insert(lastGroupedColumnIndex, lastGroupedColumn);
				}

				QGridViewColumn column = gridView.Columns.OfType<QGridViewColumn>().SingleOrDefault(a => String.Equals(a.GroupPropertyName, GroupPropertyName, StringComparison.InvariantCultureIgnoreCase));
				int columnIndex = gridView.Columns.IndexOf(column);
				if (column != null)
				{
					//Hide grouped column.
					gridView.Columns.Remove(column);
				}

				lastGroupedColumn = column;
				lastGroupedColumnIndex = columnIndex;

				//Empty parameter will ungroup column.
				InternalGroup(column != null ? column.GroupPropertyName : null);
			}
		}

		private void InternalGroup(string groupBy)
		{
			ICollectionView dataView = CollectionViewSource.GetDefaultView(ItemsSource);
			if (dataView != null)
			{
				dataView.GroupDescriptions.Clear();

				//When groupBy is null or empty string then ungroup column.
				if (!String.IsNullOrEmpty(groupBy))
				{
					PropertyGroupDescription gd = new PropertyGroupDescription(groupBy);
					dataView.GroupDescriptions.Add(gd);
				}

				dataView.Refresh();
			}
		}

		#endregion

		#region Private Members

		private void GridViewColumnHeaderClickedHandler(object sender, RoutedEventArgs e)
		{
			GridViewColumnHeader headerClicked = e.OriginalSource as GridViewColumnHeader;
			if (headerClicked != null && headerClicked.Role != GridViewColumnHeaderRole.Padding)
			{
				QGridViewColumn column = (headerClicked.Column) as QGridViewColumn;
				if (column != null)
				{
					Sort(column);
				}
			}
		}

		private QGridViewColumn FindLastSortedColumn()
		{
			GridView gridView = View as GridView;
			if (gridView != null)
			{
				return gridView.Columns.OfType<QGridViewColumn>().SingleOrDefault(a => a.IsSorted);
			}

			return null;
		}

		#endregion
	}


	public class QListViewItem : ListViewItem
	{
		#region Events

		public static readonly RoutedEvent IsExpandedChangedEvent = EventManager.RegisterRoutedEvent("IsExpandedChanged", RoutingStrategy.Bubble, typeof(RoutedEventHandler), typeof(QListViewItem));
		public static void AddIsExpandedChangedHandler(DependencyObject d, RoutedEventHandler handler)
		{
			UIElement element = d as UIElement;
			if (element != null)
			{
				element.AddHandler(IsExpandedChangedEvent, handler);
			}
		}
		public static void RemoveIsExpandedChangedHandler(DependencyObject d, RoutedEventHandler handler)
		{
			UIElement element = d as UIElement;
			if (element != null)
			{
				element.RemoveHandler(IsExpandedChangedEvent, handler);
			}
		}

		#endregion

		#region Properties

		public bool IsExpanded
		{
			get { return (bool)GetValue(IsExpandedProperty); }
			set { SetValue(IsExpandedProperty, value); }
		}
		public static readonly DependencyProperty IsExpandedProperty =
			DependencyProperty.Register("IsExpanded", typeof(bool), typeof(QListViewItem), new UIPropertyMetadata(false, IsExpandedChanged));

		private static void IsExpandedChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
		{
			QListViewItem instance = (QListViewItem)d;
			instance.IsExpandedChanged();
		}


		public DataTemplate ExpandBodyTemplate
		{
			get { return (DataTemplate)GetValue(ExpandBodyTemplateProperty); }
			set { SetValue(ExpandBodyTemplateProperty, value); }
		}
		public static readonly DependencyProperty ExpandBodyTemplateProperty =
			DependencyProperty.Register("ExpandBodyTemplate", typeof(DataTemplate), typeof(QListViewItem), new UIPropertyMetadata(null));

		#endregion

		public QListViewItem()
		{
			InputBindings.Add(new InputBinding(Commands.EditCommand, new KeyGesture(Key.Enter)));
			InputBindings.Add(new InputBinding(Commands.EditCommand, new MouseGesture(MouseAction.LeftDoubleClick)));
		}

		private void IsExpandedChanged()
		{
			RaiseEvent(new RoutedEventArgs(IsExpandedChangedEvent));
		}
	}
}
