using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Text;
using org.ovirt.engine.ui.uicommon.dataprovider;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.Interfaces;
using VdcCommon.VdcQueries;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using System.ComponentModel;
using System.Collections;

namespace org.ovirt.engine.ui.uicommon.models.vms
{
	public class VmDiskListModel : SearchableListModel
	{
		#region Commands

		public UICommand NewCommand { get; private set; }
		public UICommand EditCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }

		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		#endregion

		public VmDiskListModel()
		{
			Title = "Virtual Disks";

			NewCommand = new UICommand("New", this);
			EditCommand = new UICommand("Edit", this);
			RemoveCommand = new UICommand("Remove", this);

			UpdateActionAvailability();
		}

		protected override void OnEntityChanged()
		{
			base.OnEntityChanged();

			if (Entity != null)
			{
				SearchCommand.Execute();
			}

			UpdateActionAvailability();
		}

		protected override void SyncSearch()
		{
			VM vm = (VM)Entity;

			base.SyncSearch(VdcQueryType.GetAllDisksByVmId, new GetAllDisksByVmIdParameters(vm.vm_guid));
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			VM vm = (VM)Entity;

			AsyncResult = Frontend.RegisterQuery(VdcQueryType.GetAllDisksByVmId, new GetAllDisksByVmIdParameters(vm.vm_guid));
			Items = AsyncResult.Data;
		}

		private void New()
		{
			VM vm = (VM)Entity;

			if (Window != null)
			{
				return;
			}

			DiskModel model = new DiskModel();
			Window = model;
			model.Title = "New Virtual Disk";
			model.HashName = "new_virtual_disk";
			model.IsNew = true;
			AsyncQuery _asyncQuery1 = new AsyncQuery();
			_asyncQuery1.Model = this;
			/*START_DELEGATE*/_asyncQuery1.asyncCallback = delegate(Object model1, Object result1)
											{
												VmDiskListModel vmDiskListModel = (VmDiskListModel)model1;
												DiskModel diskModel = (DiskModel)vmDiskListModel.Window;
												List<DiskImage> disks = Items != null ? Linq.Cast<DiskImage>(Items) : new List<DiskImage>();
												bool hasDisks = disks.Count > 0;
												List<storage_domains> storageDomains = new List<storage_domains>();
												foreach (storage_domains a in (List<storage_domains>)(valueObjectEnumerableList)result1)
												{
													if (a.storage_domain_type != StorageDomainType.ISO
														&& a.storage_domain_type != StorageDomainType.ImportExport
														&& a.status == StorageDomainStatus.Active)
													{
														storageDomains.Add(a);
													}
												}

												diskModel.StorageDomain.Items = storageDomains;
												diskModel.StorageDomain.IsAvailable = !hasDisks;

												if (hasDisks)
												{
													// the StorageDomain value should be the one that all other Disks are on
													// (although this field is not-available, we use its value in the 'OnSave' method):
													AsyncQuery _asyncQuery2 = new AsyncQuery();
													_asyncQuery2.Model = model1;
													/*START_DELEGATE*/_asyncQuery2.asyncCallback = delegate(Object model2, Object result2)
													{
														VmDiskListModel vmDiskListModel2 = (VmDiskListModel)model2;
														DiskModel diskModel2 = (DiskModel)vmDiskListModel2.Window;
														List<storage_domains> storageDomains2 = (List<storage_domains>)diskModel2.StorageDomain.Items;
														storage_domains storage2 = (storage_domains)result2;
														vmDiskListModel2.StepA(storage2 != null && Linq.IsSDItemExistInList(storageDomains2, storage2.id) ? storage2 : null);
													};//END_DELEGATE
													AsyncDataProvider.GetStorageDomainById(_asyncQuery2, disks[0].storage_id.Value);
												}
												else // first disk -> just choose the first from the list of available storage-domains:
												{
													vmDiskListModel.StepA(Linq.FirstOrDefault(storageDomains));
												}
											};//END_DELEGATE
			AsyncDataProvider.GetStorageDomainList(_asyncQuery1, vm.storage_pool_id);
		}

		private void Edit()
		{
			DiskImage disk = (DiskImage)SelectedItem;

			if (Window != null)
			{
				return;
			}

			DiskModel model = new DiskModel();
			Window = model;
			model.Title = "Edit Virtual Disk";
			model.HashName = "edit_virtual_disk";
			model.StorageDomain.IsAvailable = false;
			model.Size.Entity = disk.SizeInGigabytes;
			model.Size.IsChangable = false;
			AsyncQuery _asyncQuery1 = new AsyncQuery();
			_asyncQuery1.Model = this;

			/*START_DELEGATE*/_asyncQuery1.asyncCallback = delegate(Object model1, Object result1)
				{
					VmDiskListModel vmDiskListModel = (VmDiskListModel)model1;
					DiskModel diskModel = (DiskModel)vmDiskListModel.Window;
					VM vm = (VM)vmDiskListModel.Entity;
					DiskImage disk2 = (DiskImage)vmDiskListModel.SelectedItem;
					storage_domains storage = (storage_domains)result1;

					diskModel.StorageDomain.SelectedItem = storage;

					DiskImageBase preset =
						new DiskImageBase
						{
							disk_type = disk2.disk_type
						};
					diskModel.Preset.Items = new List<DiskImageBase> { preset };
					diskModel.Preset.SelectedItem = preset;
					diskModel.Preset.IsChangable = false;

					diskModel.VolumeType.SelectedItem = disk2.volume_type;
					diskModel.VolumeType.IsChangable = false;

					diskModel.VolumeFormat = disk2.volume_format;

					List<DiskInterface> interfaces = DataProvider.GetDiskInterfaceList(vm.vm_os, vm.vds_group_compatibility_version);
					if (!interfaces.Contains(disk2.disk_interface))
					{
						interfaces.Add(disk2.disk_interface);
					}
					diskModel.Interface.Items = interfaces;
					diskModel.Interface.SelectedItem = disk2.disk_interface;
					diskModel.Interface.IsChangable = false;

					diskModel.WipeAfterDelete.Entity = disk2.wipe_after_delete;
					if (diskModel.StorageDomain != null && diskModel.StorageDomain.SelectedItem != null)
					{
						vmDiskListModel.UpdateWipeAfterDelete(storage.storage_type, diskModel.WipeAfterDelete, false);
					}


					List<DiskImage> disks = vmDiskListModel.Items != null ? Linq.Cast<DiskImage>(vmDiskListModel.Items) : new List<DiskImage>();

					DiskImage bootableDisk = null;
					foreach (DiskImage a in disks)
					{
						if (a.boot)
						{
							bootableDisk = a;
							break;
						}
					}
					if (bootableDisk != null && !bootableDisk.Id.Equals(disk2.Id))
					{
						diskModel.IsBootable.IsChangable = false;
						diskModel.IsBootable.ChangeProhibitionReasons.Add("There can be only one bootable disk defined.");
					}
					diskModel.IsBootable.Entity = disk2.boot;


					diskModel.Commands.Add(
						new UICommand("OnSave", vmDiskListModel)
						{
							Title = "OK",
							IsDefault = true
						});
					diskModel.Commands.Add(
						new UICommand("Cancel", vmDiskListModel)
						{
							Title = "Cancel",
							IsCancel = true
						});

				};//END_DELEGATE
			AsyncDataProvider.GetStorageDomainById(_asyncQuery1, disk.storage_id.Value);
		}

		private void Remove()
		{
			if (Window != null)
			{
				return;
			}

			bool hasSystemDiskWarning = false;
			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Remove Disk(s)";
			model.HashName = "remove_disk";
			model.Message = "Disk(s)";

			List<string> items = new List<string>();
			foreach (object item in SelectedItems)
			{
				DiskImage a = (DiskImage)item;
				if (a.disk_type == DiskType.System)
				{
					items.Add(string.Format("Disk {0} (System Disk)", a.internal_drive_mapping));
					if (!hasSystemDiskWarning)
					{
						model.Note = "Note that removing a system disk would make the VM unbootable.";
						hasSystemDiskWarning = true;
					}
				}
				else
				{
					items.Add(string.Format("Disk {0}", a.internal_drive_mapping));
				}
			}
			model.Items = items;

			model.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnSystemRemoveConfirm()
		{
			//var systemDisks = SelectedItems.Cast<DiskImage>().Where(a => a.disk_type == DiskType.System);
			//if (systemDisks.Count() > 0)
			//{
			//    SystemDiskRemoveConfirmModel = new ConfirmModel();
			//    SystemDiskRemoveConfirmModel.View = new ConfirmationView();
			//    SystemDiskRemoveConfirmModel.IsOpen = true;
			//    SystemDiskRemoveConfirmModel.Header = "Remove System Disk(s)";
			//    SystemDiskRemoveConfirmModel.ConfirmMsg = "Are you sure you want to remove the following System Disk(s)?";
			//    SystemDiskRemoveConfirmModel.EntityNames = systemDisks.Select(a => string.Format("Disk {0}", a.internal_drive_mapping));
			//    SystemDiskRemoveConfirmModel.Commands =
			//        new ArrayList
			//    {
			//        new
			//        {
			//            Command = new DelegateCommand(OnRemove),
			//            Text = "OK",
			//            IsDefault = true
			//        },
			//        new
			//        {
			//            Command = new DelegateCommand(Cancel),
			//            Text = "Cancel"
			//        }
			//    };
			//}
			//else
			//{
			//    OnRemove();
			//}
		}

		private void OnRemove()
		{
			VM vm = (VM)Entity;

			//TODO: Confirm system disk removal.

			//List<Guid> images = SelectedItems.Cast<DiskImage>().Select(a =>(Guid) a.image_guid).ToList();

			List<Guid> images = new List<Guid>();
			foreach (object item in SelectedItems)
			{
				DiskImage a = (DiskImage)item;
				images.Add(a.Id);
			}

			Frontend.RunAction(VdcActionType.RemoveDisksFromVm,
				new RemoveDisksFromVmParameters(vm.vm_guid, images),
				result =>
				{
				},
				null
			);

			Cancel();
		}

		private void OnSave()
		{
			VM vm = (VM)Entity;
			DiskModel model = (DiskModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			//Save changes.
			storage_domains storageDomain = (storage_domains)model.StorageDomain.SelectedItem;

			DiskImage disk = model.IsNew ? new DiskImage() : (DiskImage)SelectedItem;
			disk.SizeInGigabytes = Convert.ToInt32(model.Size.Entity.ToString());
			disk.disk_type = ((DiskImageBase)model.Preset.SelectedItem).disk_type;
			disk.disk_interface = (DiskInterface)model.Interface.SelectedItem;
			disk.volume_type = (VolumeType)model.VolumeType.SelectedItem;
			disk.volume_format = model.VolumeFormat;
			disk.wipe_after_delete = (bool)model.WipeAfterDelete.Entity;
			disk.boot = (bool)model.IsBootable.Entity;

			//NOTE: Since we doesn't support partial snapshots in GUI, propagate errors flag always must be set false.
			//disk.propagate_errors = model.PropagateErrors.ValueAsBoolean() ? PropagateErrors.On : PropagateErrors.Off;
			disk.propagate_errors = PropagateErrors.Off;


			model.StartProgress(null);

			if (model.IsNew)
			{
				Frontend.RunAction(VdcActionType.AddDiskToVm,
					new AddDiskToVmParameters(vm.vm_guid, disk)
					{
						StorageDomainId = storageDomain.id
					},
					result =>
					{
						VmDiskListModel localModel = (VmDiskListModel)result.State;

						localModel.PostOnSaveInternal(result.ReturnValue);
					},
					this
				);
			}
			else
			{
				Frontend.RunAction(VdcActionType.UpdateVmDisk,
					new UpdateVmDiskParameters(vm.vm_guid, disk.Id, disk),
					result =>
					{
						VmDiskListModel localModel = (VmDiskListModel)result.State;

						localModel.PostOnSaveInternal(result.ReturnValue);
					},
					this
				);
			}
		}

		public void PostOnSaveInternal(VdcReturnValueBase returnValue)
		{
			DiskModel model = (DiskModel)Window;

			model.StopProgress();

			if (returnValue != null && returnValue.Succeeded)
			{
				Cancel();
			}
		}

		private void Cancel()
		{
			Window = null;
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		protected override void EntityPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.EntityPropertyChanged(sender, e);

			if (e.PropertyName == "status")
			{
				UpdateActionAvailability();
			}
		}

		private void UpdateActionAvailability()
		{
			VM vm = (VM)Entity;
			bool isDown = vm != null && vm.status == VMStatus.Down;

			NewCommand.IsExecutionAllowed = isDown;

			EditCommand.IsExecutionAllowed = SelectedItem != null
				&& SelectedItems != null
				&& SelectedItems.Count == 1
				&& isDown;

			RemoveCommand.IsExecutionAllowed = SelectedItems != null
				&& SelectedItems.Count > 0
				&& isDown;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == NewCommand)
			{
				New();
			}
			else if (command == EditCommand)
			{
				Edit();
			}
			else if (command == RemoveCommand)
			{
				Remove();
			}
			else if (command.Name == "OnSave")
			{
				OnSave();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
			else if (command.Name == "OnRemove")
			{
				OnRemove();
			}
		}

		public void StepA(storage_domains storage)
		{
			DiskModel model = (DiskModel)Window;
			VM vm = (VM)Entity;

			model.StorageDomain.SelectedItem = storage;

			if (storage != null)
			{
				UpdateWipeAfterDelete(storage.storage_type, model.WipeAfterDelete, true);
			}

			AsyncQuery _asyncQuery1 = new AsyncQuery();
			_asyncQuery1.Model = this;
			/*START_DELEGATE*/_asyncQuery1.asyncCallback = delegate(Object model1, Object result1)
											{
												VmDiskListModel vmDiskListModel1 = (VmDiskListModel)model1;
												DiskModel vmModel = (DiskModel)vmDiskListModel1.Window;
												VM vm1 = (VM)vmDiskListModel1.Entity;

												List<DiskImage> disks = vmDiskListModel1.Items != null ? Linq.Cast<DiskImage>(vmDiskListModel1.Items) : new List<DiskImage>();
												bool hasDisks = disks.Count > 0;
												storage_domains storage1 = (storage_domains)vmModel.StorageDomain.SelectedItem;

												List<DiskImageBase> presets = (List<DiskImageBase>)result1;
												vmModel.Preset.Items = presets;
												vmModel.Preset.SelectedItem = null;

												foreach (DiskImageBase a in presets)
												{
													if ((hasDisks && a.disk_type == DiskType.Data) ||
														(!hasDisks && a.disk_type == DiskType.System))
													{
														vmModel.Preset.SelectedItem = a;
														break;
													}
												}


												vmModel.Interface.Items = DataProvider.GetDiskInterfaceList(vm1.vm_os, vm1.vds_group_compatibility_version);
												vmModel.Interface.SelectedItem = DataProvider.GetDefaultDiskInterface(vm1.vm_os, disks);


												bool hasBootableDisk = false;
												foreach (DiskImage a in disks)
												{
													if (a.boot)
													{
														hasBootableDisk = true;
														break;
													}
												}

												vmModel.IsBootable.Entity = !hasBootableDisk;
												if (hasBootableDisk)
												{
													vmModel.IsBootable.IsChangable = false;
													vmModel.IsBootable.ChangeProhibitionReasons.Add("There can be only one bootable disk defined.");
												}

												List<UICommand> commands = new List<UICommand>();

												if (storage1 == null)
												{
													string cantCreateMessage = "There is no active Storage Domain to create the Disk in. Please activate a Storage Domain.";
													if (hasDisks)
													{
														cantCreateMessage = "Error in retrieving the relevant Storage Domain.";
														//if (storage.storage_name != null)
														//{
														//    cantCreateMessage = string.Format("'{0}' Storage Domain is not active. Please activate it.", storage.storage_name);
														//}
													}

													vmModel.Message = cantCreateMessage;

													vmModel.Commands.Add(
														new UICommand("Cancel", vmDiskListModel1)
														{
															Title = "Close",
															IsDefault = true,
															IsCancel = true
														});
												}
												else
												{
													vmModel.Commands.Add(
														new UICommand("OnSave", vmDiskListModel1)
														{
															Title = "OK",
															IsDefault = true
														});

													vmModel.Commands.Add(
														new UICommand("Cancel", vmDiskListModel1)
														{
															Title = "Cancel",
															IsCancel = true
														});
												}

											};//END_DELEGATE
			AsyncDataProvider.GetDiskPresetList(_asyncQuery1,
				vm.vm_type,
				model.StorageDomain.SelectedItem == null ? StorageType.UNKNOWN : storage.storage_type);
		}

		private void UpdateWipeAfterDelete(StorageType storageType, EntityModel wipeAfterDeleteModel, bool isNew)
		{
			if (storageType == StorageType.NFS || storageType == StorageType.LOCALFS)
			{
				wipeAfterDeleteModel.IsChangable = false;
			}
			else
			{
				wipeAfterDeleteModel.IsChangable = true;
				if (isNew)
				{
					AsyncQuery _asyncQuery = new AsyncQuery();
					_asyncQuery.Model = Window;
					/*START_DELEGATE*/_asyncQuery.asyncCallback = delegate(Object model, Object result)
												  {
													  DiskModel diskModel = (DiskModel)model;
													  diskModel.WipeAfterDelete.Entity = result;
												  };//END_DELEGATE
					AsyncDataProvider.GetSANWipeAfterDelete(_asyncQuery);
				}
			}
		}
	}
}
