﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using org.ovirt.engine.ui.uicommon.models.users;
using VdcCommon.BusinessEntities;
using System.Collections.ObjectModel;
using VdcFrontend;
using VdcCommon.Interfaces;
using VdcCommon.Users;

namespace org.ovirt.engine.ui.uicommon.models.configure
{
	public class SystemPermissionListModel : SearchableListModel
	{
		#region Commands

		public UICommand AddCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }

		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		#endregion

		public SystemPermissionListModel()
		{
			Title = "System Permission";

			AddCommand = new UICommand("Add", this);
			RemoveCommand = new UICommand("Remove", this);

			SearchCommand.Execute();

			UpdateActionAvailability();
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			AsyncResult = Frontend.RegisterQuery(VdcQueryType.GetSystemPermissions, new VdcQueryParametersBase());
			Items = AsyncResult.Data;
		}

		private void UpdateActionAvailability()
		{
			RemoveCommand.IsExecutionAllowed = SelectedItems != null && SelectedItems.Count > 0;
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		private void Add()
		{
			if (Window != null)
			{
				return;
			}

			AdElementListModel model = new AdElementListModel();
			Window = model;
			model.Title = "Add System Permission to User";
			model.HashName = "add_system_permission_to_user";
			//model.Role.IsAvailable = true;
			//model.ExcludeItems = Items;

			model.Commands.Add(
				new UICommand("OnAttach", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnAttach()
		{
			AdElementListModel model = (AdElementListModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (model.SelectedItems == null)
			{
				Cancel();
				return;
			}

			List<DbUser> items = new List<DbUser>();
			foreach (object item in model.Items)
			{
				EntityModel entityModel = (EntityModel)item;
				if (entityModel.IsSelected)
				{
					items.Add((DbUser)entityModel.Entity);
				}
			}

			roles role = (roles)model.Role.SelectedItem;

			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (DbUser user in items)
			{
				permissions perm =
					new permissions
					{
						ad_element_id = user.user_id,
						role_id = role.Id
					};

				if (user.IsGroup)
				{
					list.Add(
						new PermissionsOperationsParametes
						{
							Permission = perm,
							AdGroup = new ad_groups(user.user_id, user.name, user.domain)
						});
				}
				else
				{
					list.Add(
						new PermissionsOperationsParametes
						{
							Permission = perm,
							VdcUser = new VdcUser(user.user_id, user.username, user.domain)
						});
				}
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.AddSystemPermission, list,
				result =>
				{
					AdElementListModel localModel = (AdElementListModel)result.State;

					localModel.StopProgress();
					Cancel();
				},
				model
			);
		}

		private void OnSave()
		{
		}

		public void Cancel()
		{
			Window = null;
		}

		private void Remove()
		{
			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Remove System Permission(s)";
			model.HashName = "remove_system_permission";
			model.Message = "System Permission(s):";

			List<string> list = new List<string>();
			foreach (object item in SelectedItems)
			{
				permissions permission = (permissions)item;
				list.Add("User: " + permission.OwnerName + " with Role: " + permission.RoleName);
			}
			model.Items = list;

			model.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnRemove()
		{
			if (SelectedItems != null && SelectedItems.Count > 0)
			{
				ConfirmationModel model = (ConfirmationModel)Window;

				if (model.Progress != null)
				{
					return;
				}

				List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
				foreach (object perm in SelectedItems)
				{
					list.Add(new PermissionsOperationsParametes { Permission = (permissions)perm });
				}


				model.StartProgress(null);

				Frontend.RunMultipleAction(VdcActionType.RemovePermission, list,
					result =>
					{
						ConfirmationModel localModel = (ConfirmationModel)result.State;

						localModel.StopProgress();
						Cancel();
					},
					model
				);
			}
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == AddCommand)
			{
				Add();
			}
			else if (command.Name == "OnSave")
			{
				OnSave();
			}
			else if (command.Name == "OnAttach")
			{
				OnAttach();
			}
			else if (command == RemoveCommand)
			{
				Remove();
			}
			else if (command.Name == "OnRemove")
			{
				OnRemove();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
		}
	}
}
