﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.BusinessEntities;

namespace org.ovirt.engine.ui.uicommon.models
{
	public class SystemTreeModel : SearchableListModel
	{
		#region Events

		public static EventDefinition ResetRequestedEventDefinition;
		public Event ResetRequestedEvent { get; private set; }

		#endregion

		#region Commands

		public UICommand ResetCommand { get; private set; }
		public UICommand ExpandAllCommand { get; private set; }
		public UICommand CollapseAllCommand { get; private set; }

		#endregion

		#region Properties

		public new List<SystemTreeItemModel> Items
		{
			get { return (List<SystemTreeItemModel>)base.Items; }
			set { base.Items = value; }
		}

		#endregion


		static SystemTreeModel()
		{
			ResetRequestedEventDefinition = new EventDefinition("ResetRequested", typeof(SystemTreeModel));
		}

		public SystemTreeModel()
		{
			ResetRequestedEvent = new Event(ResetRequestedEventDefinition);

			ResetCommand = new UICommand("Reset", this);
			ExpandAllCommand = new UICommand("ExpandAll", this);
			CollapseAllCommand = new UICommand("CollapseAll", this);

			Items = new List<SystemTreeItemModel>();
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();
			SyncSearch();
		}

		protected override void SyncSearch()
		{
			base.SyncSearch();


			//Build maps.
			List<storage_pool> dataCenters = DataProvider.GetDataCenterList();

			//Cluster by data center map.
			List<VDSGroup> clusters = DataProvider.GetClusterList();
			var clusterMap = new Dictionary<Guid, List<VDSGroup>>();
			foreach (VDSGroup cluster in clusters)
			{
				if (cluster.storage_pool_id != null)
				{
					Guid key = cluster.storage_pool_id.Value;
					if (!clusterMap.ContainsKey(key))
					{
						clusterMap.Add(key, new List<VDSGroup>());
					}
					List<VDSGroup> list = clusterMap[key];
					list.Add(cluster);
				}
			}

			//Host by cluster map.
			List<VDS> hosts = DataProvider.GetHostList();
			var hostMap = new Dictionary<Guid, List<VDS>>();
			foreach (VDS host in hosts)
			{
				Guid key = host.vds_group_id;
				if (!hostMap.ContainsKey(key))
				{
					hostMap.Add(key, new List<VDS>());
				}
				List<VDS> list = hostMap[key];
				list.Add(host);
			}


			//Build tree items.
			SystemTreeItemModel systemItem = new SystemTreeItemModel();
			systemItem.Type = SystemTreeItemType.System;
			systemItem.IsSelected = true;
			systemItem.Title = "System";

			foreach (storage_pool dataCenter in dataCenters)
			{
				SystemTreeItemModel dataCenterItem = new SystemTreeItemModel();
				dataCenterItem.Type = SystemTreeItemType.DataCenter;
				dataCenterItem.Title = dataCenter.name;
				dataCenterItem.Entity = dataCenter;
				systemItem.Children.Add(dataCenterItem);

				SystemTreeItemModel storagesItem = new SystemTreeItemModel();
				storagesItem.Type = SystemTreeItemType.Storages;
				storagesItem.Title = "Storage";
				storagesItem.Parent = dataCenterItem;
				storagesItem.Entity = dataCenter;
				dataCenterItem.Children.Add(storagesItem);

				var storages = DataProvider.GetStorageDomainList(dataCenter.Id);
				if (storages.Count > 0)
				{
					foreach (storage_domains storage in storages)
					{
						SystemTreeItemModel storageItem = new SystemTreeItemModel();
						storageItem.Type = SystemTreeItemType.Storage;
						storageItem.Title = storage.storage_name;
						storageItem.Parent = dataCenterItem;
						storageItem.Entity = storage;
						storagesItem.Children.Add(storageItem);
					}
				}

				SystemTreeItemModel templatesItem = new SystemTreeItemModel();
				templatesItem.Type = SystemTreeItemType.Templates;
				templatesItem.Title = "Templates";
				templatesItem.Parent = dataCenterItem;
				templatesItem.Entity = dataCenter;
				dataCenterItem.Children.Add(templatesItem);


				SystemTreeItemModel clustersItem = new SystemTreeItemModel();
				clustersItem.Type = SystemTreeItemType.Clusters;
				clustersItem.Title = "Clusters";
				clustersItem.Parent = dataCenterItem;
				clustersItem.Entity = dataCenter;
				dataCenterItem.Children.Add(clustersItem);

				if (clusterMap.ContainsKey(dataCenter.Id))
				{
					foreach (VDSGroup cluster in clusterMap[dataCenter.Id])
					{
						SystemTreeItemModel clusterItem = new SystemTreeItemModel();
						clusterItem.Type = SystemTreeItemType.Cluster;
						clusterItem.Title = cluster.name;
						clusterItem.Parent = dataCenterItem;
						clusterItem.Entity = cluster;
						clustersItem.Children.Add(clusterItem);


						SystemTreeItemModel hostsItem = new SystemTreeItemModel();
						hostsItem.Type = SystemTreeItemType.Hosts;
						hostsItem.Title = "Hosts";
						hostsItem.Parent = clusterItem;
						hostsItem.Entity = cluster;
						clusterItem.Children.Add(hostsItem);

						if (hostMap.ContainsKey(cluster.ID))
						{
							foreach (VDS host in hostMap[cluster.ID])
							{
								SystemTreeItemModel hostItem = new SystemTreeItemModel();
								hostItem.Type = SystemTreeItemType.Host;
								hostItem.Title = host.vds_name;
								hostItem.Parent = clusterItem;
								hostItem.Entity = host;
								hostsItem.Children.Add(hostItem);
							}
						}


						SystemTreeItemModel vmsItem = new SystemTreeItemModel();
						vmsItem.Type = SystemTreeItemType.VMs;
						vmsItem.Title = "VMs";
						vmsItem.Parent = clusterItem;
						vmsItem.Entity = cluster;
						clusterItem.Children.Add(vmsItem);
					}
				}
			}


			Items = new List<SystemTreeItemModel> { systemItem };
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == ResetCommand)
			{
				Reset();
			}
			else if (command == ExpandAllCommand)
			{
				ExpandAll();
			}
			else if (command == CollapseAllCommand)
			{
				CollapseAll();
			}
		}

		private void CollapseAll()
		{
			SetIsExpandedRecursively(false, Items[0]);
		}

		private void ExpandAll()
		{
			SetIsExpandedRecursively(true, Items[0]);
		}

		private void SetIsExpandedRecursively(bool value, SystemTreeItemModel root)
		{
			root.IsExpanded = value;

			foreach (SystemTreeItemModel model in root.Children)
			{
				SetIsExpandedRecursively(value, model);
			}
		}

		private void Reset()
		{
			ResetRequestedEvent.raise(this, EventArgs.Empty);
		}
	}



	public enum SystemTreeItemType
	{
		System,
		DataCenter,
		Storages,
		Storage,
		Templates,
		Clusters,
		Cluster,
		VMs,
		Hosts,
		Host
	}


	public class SystemTreeItemModel : EntityModel
	{
		private SystemTreeItemType type;
		public SystemTreeItemType Type
		{
			get { return type; }
			set
			{
				if (type != value)
				{
					type = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Type"));
				}
			}
		}

		public IList<SystemTreeItemModel> Children { get; set; }

		public SystemTreeItemModel Parent { get; set; }

		private bool isExpanded;
		public bool IsExpanded
		{
			get { return isExpanded; }
			set
			{
				if (isExpanded != value)
				{
					isExpanded = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsExpanded"));
				}
			}
		}


		public SystemTreeItemModel()
		{
			Children = new ObservableCollection<SystemTreeItemModel>();
		}


		public static SystemTreeItemModel FindAncestor(SystemTreeItemType type, SystemTreeItemModel root)
		{
			if (root.Type != type)
			{
				if (root.Parent != null)
				{
					return FindAncestor(type, root.Parent);
				}

				return null;
			}

			return root;
		}
	}
}
