using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using org.ovirt.engine.ui.uicommon.models.monitor;
using org.ovirt.engine.ui.uicompat;
using org.ovirt.engine.ui.uicommon.models.autocomplete;
using org.ovirt.engine.ui.uicommon.models.bookmarks;
using org.ovirt.engine.ui.uicommon.models.clusters;
using org.ovirt.engine.ui.uicommon.models.common;
using org.ovirt.engine.ui.uicommon.models.configure;
using org.ovirt.engine.ui.uicommon.models.datacenters;
using org.ovirt.engine.ui.uicommon.models.events;
using org.ovirt.engine.ui.uicommon.models.hosts;
using org.ovirt.engine.ui.uicommon.models.pools;
using org.ovirt.engine.ui.uicommon.models.storage;
using org.ovirt.engine.ui.uicommon.models.tags;
using org.ovirt.engine.ui.uicommon.models.templates;
using org.ovirt.engine.ui.uicommon.models.users;
using org.ovirt.engine.ui.uicommon.models.vms;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using System.Text.RegularExpressions;
using org.ovirt.engine.ui.uicommon.models.configure.roles_ui;
using VdcCommon.Interfaces;
using VdcCommon.Users;

namespace org.ovirt.engine.ui.uicommon.models
{
	public class CommonModel : ListModel
	{
		#region Events

		public static EventDefinition SignedOutEventDefinition;
		public Event SignedOutEvent { get; private set; }

		#endregion

		#region Commands

		public UICommand SearchCommand { get; private set; }
		public UICommand ConfigureCommand { get; private set; }
		public UICommand AboutCommand { get; private set; }
		public UICommand SignOutCommand { get; private set; }
		public UICommand ClearSearchStringCommand { get; private set; }

		#endregion

		#region Properties

		public new IList<SearchableListModel> Items
		{
			get { return (IList<SearchableListModel>)base.Items; }
			set { base.Items = value; }
		}

		public new SearchableListModel SelectedItem
		{
			get { return (SearchableListModel)base.SelectedItem; }
			set { base.SelectedItem = value; }
		}


		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		public BookmarkListModel BookmarkList { get; private set; }
		public TagListModel TagList { get; private set; }
		public SystemTreeModel SystemTree { get; private set; }
		public SearchableListModel EventList { get; private set; }
		public AlertListModel AlertList { get; private set; }
		public SearchSuggestModel AutoCompleteModel { get; private set; }


		private string searchStringPrefix;
		public string SearchStringPrefix
		{
			get { return searchStringPrefix; }
			set
			{
				if (searchStringPrefix != value)
				{
					searchStringPrefix = value;
					SearchStringPrefixChanged();
					OnPropertyChanged(new PropertyChangedEventArgs("SearchStringPrefix"));
				}
			}
		}

		private bool hasSearchStringPrefix;
		public bool HasSearchStringPrefix
		{
			get { return hasSearchStringPrefix; }
			private set
			{
				if (hasSearchStringPrefix != value)
				{
					hasSearchStringPrefix = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HasSearchStringPrefix"));
				}
			}
		}

		private string searchString;
		public string SearchString
		{
			get { return searchString; }
			set
			{
				if (searchString != value)
				{
					searchString = value;
					SearchStringChanged();
					OnPropertyChanged(new PropertyChangedEventArgs("SearchString"));
				}
			}
		}

		private VdcUser loggedInUser;
		public VdcUser LoggedInUser
		{
			get { return loggedInUser; }
			set
			{
				if (loggedInUser != value)
				{
					loggedInUser = value;
					OnPropertyChanged(new PropertyChangedEventArgs("LoggedInUser"));
				}
			}
		}

		public IList<AuditLog> Events { get; set; }

		private AuditLog lastEvent;
		public AuditLog LastEvent
		{
			get { return lastEvent; }
			set
			{
				if (lastEvent != value)
				{
					lastEvent = value;
					OnPropertyChanged(new PropertyChangedEventArgs("LastEvent"));
				}
			}
		}

		private AuditLog lastAlert;
		public AuditLog LastAlert
		{
			get { return lastAlert; }
			set
			{
				if (lastAlert != value)
				{
					lastAlert = value;
					OnPropertyChanged(new PropertyChangedEventArgs("LastAlert"));
				}
			}
		}

		private bool hasSelectedTags;
		public bool HasSelectedTags
		{
			get { return hasSelectedTags; }
			set
			{
				if (hasSelectedTags != value)
				{
					hasSelectedTags = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HasSelectedTags"));
				}
			}
		}

		private string marketplaceUrl;
		public string MarketplaceUrl
		{
			get { return marketplaceUrl; }
			set
			{
				if (marketplaceUrl != value)
				{
					marketplaceUrl = value;
					OnPropertyChanged(new PropertyChangedEventArgs("MarketplaceUrl"));
				}
			}
		}

		#endregion

		static CommonModel()
		{
			SignedOutEventDefinition = new EventDefinition("SingedOut", typeof(CommonModel));
		}

		public CommonModel()
		{
			SignedOutEvent = new Event(SignedOutEventDefinition);

			SearchCommand = new UICommand("Search", this) { IsDefault = true};
			AboutCommand = new UICommand("About", this);
			SignOutCommand = new UICommand("SignOut", this);
			ConfigureCommand = new UICommand("Configure", this);
			ClearSearchStringCommand = new UICommand("ClearSearchString", this);

			AutoCompleteModel = new SearchSuggestModel();

			BookmarkList = new BookmarkListModel();
			BookmarkList.NavigatedEvent.addListener(this);

			TagList = new TagListModel();
			TagList.SelectedItemsChangedEvent.addListener(this);

			SystemTree = new SystemTreeModel();
			SystemTree.SelectedItemChangedEvent.addListener(this);
			SystemTree.SearchCommand.Execute();

			EventList = new EventListModel();
			EventList.SearchCommand.Execute();

			AlertList = new AlertListModel();
			AlertList.SearchCommand.Execute();

			InitItems();

			LoggedInUser = Frontend.LoggedInUser;
			MarketplaceUrl = DataProvider.GetMarketplaceUrl();
		}

		private void UpdateHasSelectedTags()
		{
			List<TagModel> selectedTags = TagList.SelectedItems != null
					? Linq.Cast<TagModel>(TagList.SelectedItems)
					: new List<TagModel>();

			HasSelectedTags = SelectedItem != null && selectedTags.Count > 0;
		}

		void TagListModel_SelectedItemsChanged(object sender, EventArgs e)
		{
			//Reset system tree to the root item.
			SystemTree.SelectedItemChangedEvent.removeListener(this);
			SystemTree.ResetCommand.Execute();
			SystemTree.SelectedItemChangedEvent.addListener(this);

			UpdateHasSelectedTags();


			dataCenterList.IsAvailable = !HasSelectedTags;
			clusterList.IsAvailable = !HasSelectedTags;
			hostList.IsAvailable = true;
			storageList.IsAvailable = !HasSelectedTags;
			vmList.IsAvailable = true;

			if (poolList != null)
			{
				poolList.IsAvailable = !HasSelectedTags;
			}

			templateList.IsAvailable = !HasSelectedTags;
			userList.IsAvailable = true;
			eventList.IsAvailable = !HasSelectedTags;
			//JUICOMMENT_BEGIN
			monitor.IsAvailable = !HasSelectedTags;
			//JUICOMMENT_END

			//Switch the selected item as neccessary.
			ListModel oldSelectedItem = SelectedItem;
			if (HasSelectedTags && oldSelectedItem != hostList && oldSelectedItem != vmList && oldSelectedItem != userList)
			{
				SelectedItem = vmList;
			}
			else
			{
				string prefix;
				string search;
				SplitSearchString(SelectedItem.DefaultSearchString, out prefix, out search);

				SearchStringPrefix = prefix;
				SearchString = search;

				SearchCommand.Execute();
				SearchStringChanged();
			}
		}

		private void BookmarkListModel_Navigated(object sender, BookmarkEventArgs e)
		{
			//Reset tags tree to the root item.
			TagList.SelectedItemsChangedEvent.removeListener(this);
			TagList.ResetCommand.Execute();
			TagList.SelectedItemsChangedEvent.addListener(this);

			//Reset system tree to the root item.
			SystemTree.SelectedItemChangedEvent.removeListener(this);
			SystemTree.ResetCommand.Execute();
			SystemTree.SelectedItemChangedEvent.addListener(this);

			foreach (SearchableListModel item in Items)
			{
				item.IsAvailable = true;
			}

			SearchStringPrefix = null;
			SearchString = e.Bookmark.bookmark_value;
			SearchCommand.Execute();
		}

		public string EffectiveSearchString
		{
			get { return SearchStringPrefix + SearchString; }
		}

		private void SystemTree_ItemChanged(object sender, EventArgs args)
		{
			//Reset tags tree to the root item.
			TagList.SelectedItemsChangedEvent.removeListener(this);
			TagList.ResetCommand.Execute();
			UpdateHasSelectedTags();
			TagList.SelectedItemsChangedEvent.addListener(this);


			SystemTreeItemModel model = (SystemTreeItemModel)SystemTree.SelectedItem;
			if (model == null)
			{
				return;
			}


			//Update items availability depending on system tree selection.
			dataCenterList.IsAvailable = model.Type == SystemTreeItemType.DataCenter
				|| model.Type == SystemTreeItemType.Storage
				|| model.Type == SystemTreeItemType.System;

			clusterList.IsAvailable = model.Type == SystemTreeItemType.DataCenter
				|| model.Type == SystemTreeItemType.Clusters
				|| model.Type == SystemTreeItemType.Cluster
				|| model.Type == SystemTreeItemType.Storage
				|| model.Type == SystemTreeItemType.System;

			hostList.IsAvailable = model.Type == SystemTreeItemType.DataCenter
				|| model.Type == SystemTreeItemType.Cluster
				|| model.Type == SystemTreeItemType.Hosts
				|| model.Type == SystemTreeItemType.Host
				|| model.Type == SystemTreeItemType.Storage
				|| model.Type == SystemTreeItemType.System;

			storageList.IsAvailable = model.Type == SystemTreeItemType.DataCenter
				|| model.Type == SystemTreeItemType.Cluster
				|| model.Type == SystemTreeItemType.Host
				|| model.Type == SystemTreeItemType.Storages
				|| model.Type == SystemTreeItemType.Storage
				|| model.Type == SystemTreeItemType.System;


			bool isDataStorage = false;
			if (model.Type == SystemTreeItemType.Storage)
			{
				storage_domains storage = (storage_domains)model.Entity;
				isDataStorage = storage.storage_domain_type == StorageDomainType.Data || storage.storage_domain_type == StorageDomainType.Master;
			}

			vmList.IsAvailable = model.Type == SystemTreeItemType.DataCenter
				|| model.Type == SystemTreeItemType.Cluster
				|| model.Type == SystemTreeItemType.Host
				|| isDataStorage
				|| model.Type == SystemTreeItemType.VMs
				|| model.Type == SystemTreeItemType.System;

			if (poolList != null)
			{
				poolList.IsAvailable = model.Type == SystemTreeItemType.System;
			}

			templateList.IsAvailable = model.Type == SystemTreeItemType.DataCenter
				|| model.Type == SystemTreeItemType.Cluster
				|| model.Type == SystemTreeItemType.Host
				|| isDataStorage
				|| model.Type == SystemTreeItemType.Templates
				|| model.Type == SystemTreeItemType.System;

			userList.IsAvailable = model.Type == SystemTreeItemType.System;
			//JUICOMMENT_BEGIN
			monitor.IsAvailable = model.Type == SystemTreeItemType.System;
			//JUICOMMENT_END
			eventList.IsAvailable = model.Type == SystemTreeItemType.DataCenter
				|| model.Type == SystemTreeItemType.Cluster
				|| model.Type == SystemTreeItemType.Host
				|| model.Type == SystemTreeItemType.Storage
				|| model.Type == SystemTreeItemType.System;

			//Select a default item depending on system tree selection.
			ListModel oldSelectedItem = SelectedItem;

			switch (model.Type)
			{
				case SystemTreeItemType.DataCenter:
					SelectedItem = dataCenterList;
					break;
				case SystemTreeItemType.Clusters:
				case SystemTreeItemType.Cluster:
					SelectedItem = clusterList;
					break;
				case SystemTreeItemType.Hosts:
				case SystemTreeItemType.Host:
					SelectedItem = hostList;
					break;
				case SystemTreeItemType.Storages:
				case SystemTreeItemType.Storage:
					SelectedItem = storageList;
					break;
				case SystemTreeItemType.Templates:
					SelectedItem = templateList;
					break;
				case SystemTreeItemType.VMs:
					SelectedItem = vmList;
					break;
				default:
					SelectedItem = DefaultItem;
					break;
			}


			//Update search string if selected item was not changed. If it is,
			//search string will be updated in OnSelectedItemChanged method.
			if (SelectedItem == oldSelectedItem)
			{
				string prefix;
				string search;
				SplitSearchString(SelectedItem.DefaultSearchString, out prefix, out search);

				SearchStringPrefix = prefix;
				SearchString = search;

				SearchCommand.Execute();


				if (SelectedItem is ISupportSystemTreeContext)
				{
					ISupportSystemTreeContext treeContext = (ISupportSystemTreeContext)SelectedItem;
					treeContext.SystemTreeSelectedItem = (SystemTreeItemModel)SystemTree.SelectedItem;
				}
			}
		}

		private void SearchStringChanged()
		{
			BookmarkList.SearchString = EffectiveSearchString;
		}

		private void SearchStringPrefixChanged()
		{
			HasSearchStringPrefix = !String.IsNullOrEmpty(SearchStringPrefix);
			AutoCompleteModel.Prefix = SearchStringPrefix;
		}

		private void ClearSearchString()
		{
			SearchString = HasSearchStringPrefix ? null : SelectedItem.DefaultSearchString;
			SearchCommand.Execute();
		}

		public void About()
		{
			if (Window != null)
			{
				return;
			}

			AboutModel model = new AboutModel();
			Window = model;
			model.ShowOnlyVersion = false;
			model.Title = "About RHEV Manager";
			model.HashName = "about_rhev_manager";

			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Close",
					IsDefault = true,
					IsCancel = true
				});
		}

		public void SignOut()
		{
			//Stop search on all list models.
			foreach (SearchableListModel listModel in Items)
			{
				listModel.EnsureAsyncSearchStopped();
			}

			EventList.EnsureAsyncSearchStopped();
			AlertList.EnsureAsyncSearchStopped();

			if (Frontend.IsUserLoggedIn)
			{
				Frontend.Logoff(new LogoutUserParameters(Frontend.LoggedInUser.UserId));

				LoggedInUser = null;
				SignedOutEvent.raise(this, EventArgs.Empty);
			}
		}

		public void Configure()
		{
			if (Window != null)
			{
				return;
			}

			EntityModel model = new EntityModel();
			Window = model;
			model.Title = "Configure";
			model.HashName = "configure";
			model.Entity =
				new Model[]
				{
					new RoleListModel(),
                    new SystemPermissionListModel()
				};


			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Close",
					IsDefault = true,
					IsCancel = true
				});
		}

		public void Cancel()
		{
			Window = null;
		}

		private SearchableListModel dataCenterList;
		private SearchableListModel clusterList;
		private SearchableListModel hostList;
		private SearchableListModel storageList;
		private SearchableListModel vmList;
		private SearchableListModel poolList;
		private SearchableListModel templateList;
		private SearchableListModel userList;
		private SearchableListModel eventList;
		private SearchableListModel monitor;

		private void InitItems()
		{
			ObservableCollection<SearchableListModel> list = new ObservableCollection<SearchableListModel>();

			dataCenterList = new DataCenterListModel();
			list.Add(dataCenterList);
			clusterList = new ClusterListModel();
			list.Add(clusterList);
			hostList = new HostListModel();
			list.Add(hostList);
			storageList = new StorageListModel();
			list.Add(storageList);
			vmList = new VmListModel();
			list.Add(vmList);

			if (DataProvider.IsLicenseHasDesktops())
			{
				poolList = new PoolListModel();
				list.Add(poolList);
			}

			templateList = new TemplateListModel();
			list.Add(templateList);
			userList = new UserListModel();
			list.Add(userList);
			//JUICOMMENT_BEGIN
			monitor = new MonitorModel();
			list.Add(monitor);
			//JUICOMMENT_END
			eventList = new EventListModel();
			list.Add(eventList);

			Items = list;


			//Activate the default list model.
			SelectedItem = DefaultItem;
		}

		private SearchableListModel DefaultItem
		{
			get { return vmList; }
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(SelectedItemsChangedEventDefinition) && sender == TagList)
			{
				TagListModel_SelectedItemsChanged(sender, args);
			}
			else if (ev.Equals(BookmarkListModel.NavigatedEventDefinition) && sender == BookmarkList)
			{
				BookmarkListModel_Navigated(sender, (BookmarkEventArgs)args);
			}
			else if (ev.Equals(SelectedItemChangedEventDefinition) && sender == SystemTree)
			{
				SystemTree_ItemChanged(sender, args);
			}
		}

		private bool executingSearch;

		protected override void OnSelectedItemChanging(object newValue, object oldValue)
		{
			base.OnSelectedItemChanging(newValue, oldValue);

			SearchableListModel oldModel = (SearchableListModel)oldValue;
			
			if (oldValue != null)
			{
				//clear the IsEmpty flag, that in the next search the flag will be initialized.
				oldModel.IsEmpty = false;

				oldModel.Items = null;

				ListWithDetailsModel listWithDetails = oldValue as ListWithDetailsModel;
				if (listWithDetails != null)
				{
					listWithDetails.ActiveDetailModel = null;
				}

				oldModel.EnsureAsyncSearchStopped();
			}
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();

			if (!executingSearch && SelectedItem != null)
			{
				//Split search string as necessary.
				string prefix;
				string search;
				SplitSearchString(SelectedItem.DefaultSearchString, out prefix, out search);

				SearchStringPrefix = prefix;
				SearchString = search;

				SelectedItem.SearchString = EffectiveSearchString;
				SelectedItem.SearchCommand.Execute();

				if (SelectedItem is ISupportSystemTreeContext)
				{
					ISupportSystemTreeContext treeContext = (ISupportSystemTreeContext)SelectedItem;
					treeContext.SystemTreeSelectedItem = (SystemTreeItemModel)SystemTree.SelectedItem;
				}
			}

			UpdateHasSelectedTags();
		}

		public virtual void Search()
		{
			executingSearch = true;

			//Prevent from entering an empty search string.
			if (String.IsNullOrEmpty(EffectiveSearchString) && SelectedItem != null)
			{
				SearchString = SelectedItem.DefaultSearchString;
			}

			//Determine a list model matching the search string.
			SearchableListModel model = null;
			foreach (SearchableListModel a in Items)
			{
				if (a.IsSearchStringMatch(EffectiveSearchString))
				{
					model = a;
					break;
				}
			}

			if (model == null)
			{
				return;
			}

			//Transfer a search string to the model.
			model.SearchString = EffectiveSearchString;

			//Change active list model as neccesary.
			SelectedItem = model;

			//Propagate search command to a concrete list model.
			SelectedItem.SearchCommand.Execute();

			executingSearch = false;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == SearchCommand)
			{
				Search();
			}
			else if (command == AboutCommand)
			{
				About();
			}
			else if (command == SignOutCommand)
			{
				SignOut();
			}
			else if (command == ConfigureCommand)
			{
				Configure();
			}
			else if (command == ClearSearchStringCommand)
			{
				ClearSearchString();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
		}

		/// <summary>
		/// Splits a search string into two component,
		/// the prefix and a search string itself.
		/// </summary>
		private void SplitSearchString(string source, out string prefix, out string search)
		{
			List<TagModel> tags = (List<TagModel>)TagList.SelectedItems;
			SystemTreeItemModel model = (SystemTreeItemModel)SystemTree.SelectedItem;

			prefix = null;

			//Split for tags.
			if (tags != null && tags.Count > 0)
			{
				Regex regex = new Regex(@"tag\s*=\s*(?:[\w-]+)(?:\sor\s)?", RegexOptions.IgnoreCase);

				string[] array = source.Split(':');
				string entityClause = array[0];
				string searchClause = array[1];

				string tagsClause = string.Empty;
				foreach (TagModel tag in tags)
				{
					tagsClause += ("tag=" + tag.Name.Entity);
					if (tag != tags[tags.Count - 1])
					{
						tagsClause += " or ";
					}
				}

				prefix = String.Format("{0}: {1} ", entityClause, tagsClause);
				
				search = regex.Replace(searchClause, String.Empty).Trim();
			}
			//Split for system tree.
			else if (model != null && model.Type != SystemTreeItemType.System)
			{
				AutoCompleteModel.Filter = new[] { "or", "and" };

				switch (model.Type)
				{
					case SystemTreeItemType.DataCenter:
						{
							if (dataCenterList.IsSearchStringMatch(source))
							{
								prefix = String.Format("DataCenter: name = {0}", model.Title);
							}
							else if (clusterList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Cluster: datacenter.name = {0}", model.Title);
							}
							else if (hostList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Host: datacenter = {0}", model.Title);
							}
							else if (storageList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Storage: datacenter = {0}", model.Title);
							}
							else if (vmList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Vms: datacenter = {0}", model.Title);
							}
							else if (templateList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Template: datacenter = {0}", model.Title);
							}
							else if (eventList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Events: event_datacenter = {0}", model.Title);
							}
						}
						break;
					case SystemTreeItemType.Clusters:
						{
							if (clusterList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Cluster: datacenter.name = {0}", model.Parent.Title);
							}
						}
						break;
					case SystemTreeItemType.Cluster:
						{
							if (clusterList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Cluster: name = {0}", model.Title);
							}
							else if (hostList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Host: cluster = {0}", model.Title);
							}
							else if (storageList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Storage: cluster.name = {0}", model.Title);
							}
							else if (vmList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Vms: cluster = {0}", model.Title);
							}
							else if (templateList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Template: cluster = {0}", model.Title);
							}
							else if (eventList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Events: cluster = {0}", model.Title);
							}
						}
						break;
					case SystemTreeItemType.Hosts:
						{
							if (hostList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Host: cluster = {0}", model.Parent.Title);
							}
						}
						break;
					case SystemTreeItemType.Host:
						{
							if (hostList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Host: name = {0}", model.Title);
							}
							else if (storageList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Storage: host.name = {0}", model.Title);
							}
							else if (vmList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Vms: Hosts.name = {0}", model.Title);
							}
							else if (templateList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Template: Hosts.name = {0}", model.Title);
							}
							else if (eventList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Events: host.name = {0}", model.Title);
							}
						}
						break;
					case SystemTreeItemType.Storages:
						{
							if (storageList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Storage: datacenter = {0}", model.Parent.Title);
							}
						}
						break;
					case SystemTreeItemType.Storage:
						{
							if (dataCenterList.IsSearchStringMatch(source))
							{
								prefix = String.Format("DataCenter: storage.name = {0}", model.Title);
							}
							else if (clusterList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Cluster: storage.name = {0}", model.Title);
							}
							else if (hostList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Host: storage.name = {0}", model.Title);
							}
							else if (storageList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Storage: name = {0}", model.Title);
							}
							else if (vmList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Vms: storage.name = {0}", model.Title);
							}
							else if (templateList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Templates: storage.name = {0}", model.Title);
							}
							else if (eventList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Events: event_storage = {0}", model.Title);
							}
						}
						break;
					case SystemTreeItemType.Templates:
						{
							if (templateList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Template: datacenter = {0}", model.Parent.Title);
							}
						}
						break;
					case SystemTreeItemType.VMs:
						{
							if (vmList.IsSearchStringMatch(source))
							{
								prefix = String.Format("Vms: cluster = {0}", model.Parent.Title);
							}
						}
						break;
				}

				prefix = prefix + " ";
				search = null;
			}
			else
			{
				search = source;
				AutoCompleteModel.Filter = null;
			}
		}
	}
}
