using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using org.ovirt.engine.ui.uicommon.models.storage;
using org.ovirt.engine.ui.uicommon.models.vms;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.BusinessEntities;

namespace org.ovirt.engine.ui.uicommon
{
	public static class Linq
	{
		public class AuditLogComparer : IComparer<AuditLog>
		{
			public int Compare(AuditLog x, AuditLog y)
			{
				long xid = x.audit_log_id;
				long yid = y.audit_log_id;

				return xid.CompareTo(yid);
			}
		}


		public class VmAndPoolByNameComparer : IComparer
		{
			public int Compare(object x, object y)
			{
				return GetValue(x).CompareTo(GetValue(y));
			}

			private string GetValue(object obj)
			{
				if (obj is VM)
				{
					return ((VM)obj).vm_name;
				}
				if (obj is vm_pools)
				{
					return ((vm_pools)obj).vm_pool_name;
				}

				throw new NotImplementedException();
			}
		}


		/// <summary>
		/// Checks if host belongs to any of clusters from list.
		/// </summary>
		/// <param name="clusters"></param>
		/// <param name="host"></param>
		/// <returns></returns>
		public class VdsGroupByNameComparer : IComparer<VDSGroup>
		{
			#region IComparer<VDSGroup> Members

			public int Compare(VDSGroup x, VDSGroup y)
			{
				return string.Compare(x.name, y.name);
			}

			#endregion
		}

		public class VmTemplateByNameComparer : IComparer<VmTemplate>
		{
			#region IComparer<VDSGroup> Members

			public int Compare(VmTemplate x, VmTemplate y)
			{
				return string.Compare(x.name, y.name);
			}

			#endregion
		}

		public class DiskImageByLastModifiedComparer : IComparer<DiskImage>
		{
			#region IComparer<DiskImage> Members

			public int Compare(DiskImage x, DiskImage y)
			{
				if (x.lastModified.before(y.lastModified))
					return -1;

				if (x.lastModified.after(y.lastModified))
					return 1;

				return 0;
			}

			#endregion
		}

		public class DiskImageByLastModifiedTimeOfDayComparer : IComparer<DiskImage>
		{
			#region IComparer<DiskImage> Members

			public int Compare(DiskImage x, DiskImage y)
			{
				return DateTimeUtils.getTimeOfDay(x.lastModified).CompareTo(DateTimeUtils.getTimeOfDay(y.lastModified));
			}

			#endregion
		}

		public class SnapshotModelDateComparer : IComparer<SnapshotModel>
		{
			#region IComparer<SnapshotModel> Members

			public int Compare(SnapshotModel x, SnapshotModel y)
			{
				if (x.Date == null)
				{
					return 1;
				}
				if (y.Date == null)
				{
					return -1;
				}
				return ((DateTime)x.Date).CompareTo(y.Date);
			}

			#endregion
		}

		public class StorageDomainByNameComparer : IComparer<storage_domains>
		{
			#region IComparer<DiskImage> Members

			public int Compare(storage_domains x, storage_domains y)
			{
				return string.Compare(x.storage_name, y.storage_name);
			}

			#endregion
		}

		public class DiskByInternalDriveMappingComparer : IComparer<DiskImage>
		{
			#region IComparer<DiskImage> Members

			public int Compare(DiskImage x, DiskImage y)
			{
				return string.Compare(x.internal_drive_mapping, y.internal_drive_mapping);
			}

			#endregion
		}

		public class NetworkByNameComparer : IComparer<network>
		{
			public int Compare(network x, network y)
			{
				return String.Compare(x.name, y.name);
			}
		}

		public class CaseInsensitiveComparer : IComparer<string>
		{
			#region IComparer<VDSGroup> Members

			public int Compare(string str1, string str2)
			{
				return string.Compare(str1.ToLower(), str2.ToLower());
			}

			#endregion
		}


		public static bool IsHostBelongsToAnyOfClusters(List<VDSGroup> clusters, VDS host)
		{
			foreach (VDSGroup cluster in clusters)
			{
				if (cluster.ID.Equals(host.vds_group_id))
				{
					return true;
				}
			}

			return false;
		}
		/// <summary>
		/// Checks if Any StorageDomain Is Matser And Active
		/// </summary>
		/// <param name="sdl"></param>
		/// <returns></returns>
		public static bool IsAnyStorageDomainIsMatserAndActive(IList<storage_domains> sdl)
		{
			foreach (storage_domains a in sdl)
				if (a.storage_domain_type == StorageDomainType.Master && a.status.HasValue && a.status.Value == StorageDomainStatus.Active)
					return true;
			return false;
		}
		/// <summary>
		/// Finds min Version by clusters list.
		/// </summary>
		/// <param name="source">IList to look in</param>
		/// <returns>Version MinVersion</returns>
		public static Version GetMinVersionByClusters(IList<VDSGroup> source)
		{
			Version minVersion = source != null && source.Count > 0 ? source[0].compatibility_version : null;

			if (minVersion != null)
				foreach (VDSGroup cluster in source)
				{
					minVersion = cluster.compatibility_version.compareTo(minVersion) < 0 ? (Version)cluster.compatibility_version : minVersion;
				}

			return minVersion;
		}
		/// <summary>
		/// Check if storage_domains item with specified guid exist in List
		/// </summary>
		/// <param name="items"></param>
		/// <param name="id"></param>
		/// <returns></returns>
		public static bool IsSDItemExistInList(List<storage_domains> items, Guid id)
		{
			foreach (storage_domains b in items)
				if (b.id.Equals(id)) return true;
			return false;
		}
		/// <summary>
		/// Check if VDSGroup item with specified id exist in List
		/// </summary>
		/// <param name="items"></param>
		/// <param name="id"></param>
		/// <returns></returns>
		public static bool IsClusterItemExistInList(IList<VDSGroup> items, Guid id)
		{
			foreach (VDSGroup a in items)
				if (a.ID == id) return true;
			return false;
		}

		public static NetworkInterface FindInterfaceByName(List<NetworkInterface> items, string name)
		{
			foreach (NetworkInterface i in items)
			{
				if (i.Name == name)
				{
					return i;
				}
			}
			return null;
		}

		public static List<NetworkInterface> VdsNetworkInterfaceListToBase(List<VdsNetworkInterface> items)
		{
			List<NetworkInterface> networkInterfaces = new List<NetworkInterface>();
			foreach (VdsNetworkInterface item in items)
			{
				networkInterfaces.Add(item);
			}

			return networkInterfaces;
		}

		public static List<NetworkInterface> VmNetworkInterfaceListToBase(List<VmNetworkInterface> items)
		{
			List<NetworkInterface> networkInterfaces = new List<NetworkInterface>();
			foreach (VmNetworkInterface item in items)
			{
				networkInterfaces.Add(item);
			}

			return networkInterfaces;
		}

		public static NetworkInterface FindInterfaceByNetworkName(List<NetworkInterface> items, string name)
		{
			foreach (NetworkInterface i in items)
			{
				if (i.NetworkName == name)
				{
					return i;
				}
			}
			return null;
		}

		public static VdsNetworkInterface FindInterfaceByIsBond(List<VdsNetworkInterface> items)
		{
			foreach (VdsNetworkInterface i in items)
			{
				if (i.Bonded.HasValue && i.Bonded.Value)
				{
					return i;
				}
			}
			return null;
		}

		public static NetworkInterface FindInterfaceNetworkNameNotEmpty(List<NetworkInterface> items)
		{
			foreach (NetworkInterface i in items)
			{
				if (!string.IsNullOrEmpty(i.NetworkName))
				{
					return i;
				}
			}
			return null;
		}

		public static List<NetworkInterface> FindAllInterfaceNetworkNameNotEmpty(List<NetworkInterface> items)
		{
			List<NetworkInterface> ret = new List<NetworkInterface>();
			foreach (NetworkInterface i in items)
			{
				if (!string.IsNullOrEmpty(i.NetworkName))
				{
					ret.Add(i);
				}
			}
			return ret;
		}

		public static List<VdsNetworkInterface> FindAllInterfaceBondNameIsEmpty(List<VdsNetworkInterface> items)
		{
			List<VdsNetworkInterface> ret = new List<VdsNetworkInterface>();
			foreach (VdsNetworkInterface i in items)
			{
				if (string.IsNullOrEmpty(i.BondName))
				{
					ret.Add(i);
				}
			}
			return ret;
		}

		public static List<VdsNetworkInterface> FindAllInterfaceVlanIdIsEmpty(List<VdsNetworkInterface> items)
		{
			List<VdsNetworkInterface> ret = new List<VdsNetworkInterface>();
			foreach (VdsNetworkInterface i in items)
			{
				if (!i.VlanId.HasValue)
				{
					ret.Add(i);
				}
			}
			return ret;
		}

		public static network FindNetworkByName(List<network> items, string name)
		{
			foreach (network n in items)
			{
				if (n.name == name)
				{
					return n;
				}
			}
			return null;
		}

		public static List<VDS> FindAllVDSByPmEnabled(List<VDS> items)
		{
			List<VDS> ret = new List<VDS>();
			foreach (VDS i in items)
			{
				if (i.pm_enabled)
				{
					ret.Add(i);
				}
			}
			return ret;
		}

		public static SanTargetModel FindSanTargetByNotIsConnected(List<SanTargetModel> items)
		{
			foreach (SanTargetModel i in items)
			{
				if (!i.IsLoggedIn)
				{
					return i;
				}
			}
			return null;
		}

		public static List<storage_domains> FindAllStorageDomainsBySharedStatus(List<storage_domains> items, StorageDomainSharedStatus status)
		{
			List<storage_domains> ret = new List<storage_domains>();
			foreach (storage_domains i in items)
			{
				if (i.storage_domain_shared_status == status)
				{
					ret.Add(i);
				}
			}
			return ret;
		}

		public static VdcReturnValueBase FindVdcReturnValueByDescription(List<VdcReturnValueBase> items, string description)
		{
			foreach (VdcReturnValueBase i in items)
			{
				if (i.Description == description)
				{
					return i;
				}
			}
			return null;
		}

		/// <summary>
		/// Determines if all elements of source satisfy a condition.
		/// </summary>
		//public static bool All<TSource>(IEnumerable<TSource> source, Func<TSource, bool> predicate)
		//{
		//    foreach (TSource item in source)
		//        if (!predicate(item))
		//            return false;

		//    return true;
		//}

		/// <summary>
		/// Find min TSource by TKey.
		/// </summary>
		/// <typeparam name="TSource">Source type.</typeparam>
		/// <typeparam name="TKey">Min TSource to be found according to this type.</typeparam>
		/// <param name="source">IEnumerable to iterate for min TSource.</param>
		/// <param name="selector">Param by which to search for min TSource.</param>
		/// <returns>min(IEnumerable<TSource>)</returns>
		//public static TSource Min<TSource, TKey>(IEnumerable<TSource> source, Func<TSource, TKey> selector)
		//{
		//    using (IEnumerator<TSource> sIterator = source.GetEnumerator())
		//    {
		//        if (!sIterator.MoveNext()) return default(TSource);

		//        IComparer<TKey> comparer = Comparer<TKey>.Default;
		//        TKey minKey = selector(sIterator.Current);
		//        TSource minVal = sIterator.Current;

		//        while (sIterator.MoveNext())
		//        {
		//            TSource minCandidate = sIterator.Current;
		//            TKey minCandidateKeyValue = selector(minCandidate);
		//            if (comparer.Compare(minCandidateKeyValue, minKey) < 0)
		//            {
		//                minKey = minCandidateKeyValue;
		//                minVal = minCandidate;
		//            }
		//        }
		//        return minVal;
		//    }
		//}

		/// <summary>
		/// Produces the set difference of two sequences by using the default equality
		//  comparer to compare values.
		/// </summary>
		/// <typeparam name="TSource"></typeparam>
		/// <param name="first">An System.Collections.Generic.IEnumerable<T> whose elements that are not 
		/// also in second will be returned.
		/// </param>
		/// <param name="second">An System.Collections.Generic.IEnumerable<T> whose elements that also occur 
		/// in the first sequence will cause those elements to be removed from the returned sequence.
		/// </param>
		/// <returns>A sequence that contains the set difference of the elements of two sequences.
		/// </returns>
		public static List<TSource> Except<TSource>(List<TSource> first, List<TSource> second)
		{
			List<TSource> newIEnumerable = new List<TSource>();

			if (first != null && second != null)
			{
				foreach (TSource t in first)
				{
					if (!second.Contains(t))
					{
						newIEnumerable.Add(t);
					}
				}
			}

			return second == null ? first : newIEnumerable;
		}
		/// <summary>
		/// OrderBy
		/// </summary>
		/// <typeparam name="TSource">TSource of IEnumerable to sort.</typeparam>
		/// <typeparam name="TKey">TKey of IEnumerable to sort by.</typeparam>
		/// <param name="source">IEnumerable to sort.</param>
		/// <param name="keySelector">Key of IEnumerable to sort by.</param>
		/// <returns></returns>
		//public static IEnumerable<TSource> OrderBy<TSource, TKey>(IEnumerable<TSource> source, Func<TSource, TKey> keySelector)
		//{
		//    SortedList<TKey, TSource> sl = new SortedList<TKey, TSource>();

		//    foreach (TSource item in source)
		//    {
		//        sl.Add(keySelector(item), item);
		//    }

		//    return Cast<TSource>(sl.Values);
		//}

		public static int Count(IEnumerable source)
		{
			int result = 0;
			foreach (object item in source)
			{
				result++;
			}

			return result;
		}

		public static TSource FirstOrDefault<TSource>(IEnumerable<TSource> source)
		{
			foreach (TSource item in source)
			{
				return item;
			}

			return default(TSource);
		}

		public static IEnumerable<TSource> Where<TSource>(IEnumerable<TSource> source, IPredicate<TSource> predicate)
		{
			List<TSource> list = new List<TSource>();

			foreach (TSource item in source)
			{
				if (predicate.Match(item))
				{
					list.Add(item);
				}
			}

			return list;
		}

		public static Version SelectHighestVersion(List<Version> versions)
		{
			Version retVersion = (Version)FirstOrDefault(versions);
			foreach (Version version in versions)
			{
				if (version.CompareTo(retVersion) > 0)
				{
					retVersion = version;
				}
			}
			return retVersion;
		}

		public static TSource FirstOrDefault<TSource>(IEnumerable<TSource> source, IPredicate<TSource> predicate)
		{
			foreach (TSource item in source)
			{
				if (predicate.Match(item))
				{
					return item;
				}
			}

			return default(TSource);
		}

		/// <summary>
		/// Returns a new instance of list containing all items of the provided source.
		/// </summary>
		public static List<TSource> ToList<TSource>(IEnumerable<TSource> source)
		{
			List<TSource> list = new List<TSource>();
			foreach (TSource item in source)
			{
				list.Add(item);
			}

			return list;
		}

		//public static TSource First<TSource>(IEnumerable<TSource> source, Func<TSource, bool> predicate)
		//{
		//    foreach (TSource item in source)
		//    {
		//        if (predicate(item))
		//        {
		//            return item;
		//        }
		//    }

		//    throw new InvalidOperationException();
		//}

		//public static bool Any<TSource>(IEnumerable<TSource> source, Func<TSource, bool> predicate)
		//{
		//    foreach (TSource item in source)
		//    {
		//        if (predicate(item))
		//        {
		//            return true;
		//        }
		//    }

		//    return false;
		//}

		public static List<TSource> Distinct<TSource>(List<TSource> source, IEqualityComparer<TSource> comparer)
		{
			List<TSource> list = new List<TSource>();
			foreach (TSource a in source)
			{
				bool found = false;
				foreach (TSource b in list)
				{
					if (comparer.Equals(a, b))
					{
						found = true;
						break;
					}
				}

				if (!found)
				{
					list.Add(a);
				}
			}

			return list;
		}



		//public static IDictionary<TKey, IList<TSource>> GroupBy<TSource, TKey>(IEnumerable<TSource> source, Func<TSource, TKey> keySelector)
		//{
		//    IDictionary<TKey, IList<TSource>> bag = new Dictionary<TKey, IList<TSource>>();
		//    foreach (TSource item in source)
		//    {
		//        TKey key = keySelector(item);
		//        if (!bag.ContainsKey(key))
		//        {
		//            bag.Add(key, new List<TSource>());
		//        }

		//        IList<TSource> list = bag[key];
		//        list.Add(item);
		//    }

		//    return bag;
		//}

		public static List<TResult> Cast<TResult>(IEnumerable source)
		{
			List<TResult> list = new List<TResult>();
			foreach (object a in source)
			{
				TResult item = (TResult)a;
				list.Add(item);
			}

			return list;
		}

		public static List<T> OrderByDescending<T>(IList<T> source) where T : IComparable<T>
		{
			List<T> list = new List<T>();

			List<T> sorted = new List<T>(source);
			sorted.Sort();

			for (int i = sorted.Count; i > 0; i--)
			{
				list.Add(sorted[i - 1]);
			}

			return list;
		}

		public static List<T> OrderByDescending<T>(IList<T> source, IComparer<T> comparer)
		{
			List<T> list = new List<T>();

			List<T> sorted = new List<T>(source);
			sorted.Sort(comparer);

			for (int i = sorted.Count; i > 0; i--)
			{
				list.Add(sorted[i - 1]);
			}

			return list;
		}

		/// <summary>
		/// Sorts a not typed list. Allows to do a sort on a list
		/// containing elements of different types.
		/// </summary>
		public static void Sort(IList source, IComparer comparer)
		{
			int pos = 0;
			while (pos < source.Count)
			{
				if (pos == 0 || comparer.Compare(source[pos], source[pos - 1]) >= 1)
				{
					pos++;
				}
				else
				{
					object temp = source[pos];
					source[pos] = source[pos - 1];
					source[pos - 1] = temp;
					pos--;
				}
			}
		}

		public static IList Concat(params IList[] lists)
		{
			IList result = new List<object>();
			foreach (IList list in lists)
			{
				foreach (object item in list)
				{
					result.Add(item);
				}
			}

			return result;
		}

		public sealed class TimeZonePredicate : IPredicate<KeyValuePair<string, string>>
		{
			private readonly string timeZone;

			public TimeZonePredicate(string timeZone)
			{
				this.timeZone = timeZone;
			}

			public bool Match(KeyValuePair<string, string> source)
			{
				return source.Key == timeZone;
			}
		}

		public sealed class ServerCpuPredicate : IPredicate<ServerCpu>
		{
			private readonly string cpuName;

			public ServerCpuPredicate(string cpuName)
			{
				this.cpuName = cpuName;
			}

			public bool Match(ServerCpu source)
			{
				return source.CpuName == cpuName;
			}
		}

		public sealed class VersionPredicate : IPredicate<Version>
		{
			private readonly Version version;

			public VersionPredicate(Version version)
			{
				this.version = version;
			}

			public bool Match(Version source)
			{
				return source.Equals(version);
			}
		}


		public sealed class DataCenterPredicate : IPredicate<storage_pool>
		{
			private readonly Guid id;

			public DataCenterPredicate(Guid id)
			{
				this.id = id;
			}

			public bool Match(storage_pool source)
			{
				return id.Equals(source.Id);
			}
		}

		public sealed class DataCenterStatusPredicate : IPredicate<storage_pool>
		{
			private readonly StoragePoolStatus status;

			public DataCenterStatusPredicate(StoragePoolStatus status)
			{
				this.status = status;
			}

			public bool Match(storage_pool source)
			{
				return source.status == status;
			}
		}

		public sealed class ClusterPredicate : IPredicate<VDSGroup>
		{
			private readonly Guid id;

			public ClusterPredicate(Guid id)
			{
				this.id = id;
			}

			public bool Match(VDSGroup source)
			{
				return id.Equals(source.ID);
			}
		}

		public sealed class HostPredicate : IPredicate<VDS>
		{
			private readonly Guid id;

			public HostPredicate(Guid id)
			{
				this.id = id;
			}

			public bool Match(VDS source)
			{
				return source.vds_id.Equals(id);
			}
		}

		public sealed class VmPredicate : IPredicate<VM>
		{
			private readonly Guid id;

			public VmPredicate(Guid id)
			{
				this.id = id;
			}

			public bool Match(VM source)
			{
				return source.vm_guid.Equals(id);
			}
		}

		public sealed class HostStatusPredicate : IPredicate<VDS>
		{
			private readonly VDSStatus status;

			public HostStatusPredicate(VDSStatus status)
			{
				this.status = status;
			}

			public bool Match(VDS source)
			{
				return source.status.Equals(status);
			}
		}

		public sealed class TemplatePredicate : IPredicate<VmTemplate>
		{
			private readonly Guid id;

			public TemplatePredicate(Guid id)
			{
				this.id = id;
			}

			public bool Match(VmTemplate source)
			{
				return source.Id.Equals(id);
			}
		}

		public sealed class StoragePredicate : IPredicate<storage_domains>
		{
			private readonly Guid id;

			public StoragePredicate(Guid id)
			{
				this.id = id;
			}

			public bool Match(storage_domains source)
			{
				return id.Equals(source.id);
			}
		}

		public sealed class LunPredicate : IPredicate<LunModel>
		{
			private readonly LunModel lun;

			public LunPredicate(LunModel lun)
			{
				this.lun = lun;
			}

			public bool Match(LunModel source)
			{
				return source.LunId == lun.LunId;
			}
		}

		public sealed class TargetPredicate : IPredicate<SanTargetModel>
		{
			private readonly SanTargetModel target;

			public TargetPredicate(SanTargetModel target)
			{
				this.target = target;
			}

			public bool Match(SanTargetModel source)
			{
				return source.Name == target.Name && source.Address == target.Address && source.Port == target.Port;
			}
		}


		public interface IPredicate<TSource>
		{
			bool Match(TSource source);
		}
	}
}
