﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;

namespace UI.WPFClient
{
	public class ProportionalPanel : Panel
	{
		public static double GetProportion(DependencyObject obj)
		{
			return (double)obj.GetValue(ProportionProperty);
		}

		public static void SetProportion(DependencyObject obj, double value)
		{
			obj.SetValue(ProportionProperty, value);
		}

		public static readonly DependencyProperty ProportionProperty =
			DependencyProperty.RegisterAttached("Proportion", typeof(double), typeof(ProportionalPanel),
			new FrameworkPropertyMetadata(1.0, FrameworkPropertyMetadataOptions.AffectsArrange |
				FrameworkPropertyMetadataOptions.AffectsMeasure |
				FrameworkPropertyMetadataOptions.AffectsRender | FrameworkPropertyMetadataOptions.AffectsParentArrange | FrameworkPropertyMetadataOptions.AffectsParentMeasure));


		public Orientation Orientation
		{
			get { return (Orientation)GetValue(OrientationProperty); }
			set { SetValue(OrientationProperty, value); }
		}
		public static readonly DependencyProperty OrientationProperty =
			DependencyProperty.Register("Orientation", typeof(Orientation), typeof(ProportionalPanel),
			new FrameworkPropertyMetadata(Orientation.Vertical,
				FrameworkPropertyMetadataOptions.AffectsArrange | FrameworkPropertyMetadataOptions.AffectsMeasure | FrameworkPropertyMetadataOptions.AffectsRender));


		protected override Size ArrangeOverride(Size arrangeSize)
		{
			double childWidth = 0.0;
			double childHeight = 0.0;
			double lastChildX = 0.0;
			double lastChildY = 0.0;
			double totalProportion = 0.0;

			foreach (UIElement child in Children)
			{
				totalProportion += GetProportion(child);
			}
			foreach (UIElement child in Children)
			{
				if (Orientation == Orientation.Horizontal)
				{
					childWidth = arrangeSize.Width * (GetProportion(child) / totalProportion);

					var element = child as FrameworkElement;
					if (element != null && childWidth > element.MaxWidth)
					{
						childWidth = element.MaxWidth;
					}

					childHeight = arrangeSize.Height;
				}
				if (Orientation == Orientation.Vertical)
				{
					childWidth = arrangeSize.Width;

					childHeight = arrangeSize.Height * (GetProportion(child) / totalProportion);

					var element = child as FrameworkElement;
					if (element != null && childHeight > element.MaxHeight)
					{
						childHeight = element.MaxHeight;
					}
				}
				if (Double.IsNaN(childWidth))
				{
					childWidth = 0.0;
				}
				if (Double.IsNaN(childHeight))
				{
					childHeight = 0.0;
				}
				child.Arrange(new Rect(lastChildX, lastChildY, childWidth, childHeight));

				if (Orientation == Orientation.Horizontal)
				{
					lastChildX += childWidth;
				}
				if (Orientation == Orientation.Vertical)
				{
					lastChildY += childHeight;
				}
			}
			return arrangeSize;
		}

		protected override Size MeasureOverride(Size constraint)
		{
			double childWidth = 0.0;
			double childHeight = 0.0;
			double maxChildHeight = 0.0;
			double maxChildWidth = 0.0;
			double totalProportion = 0.0;

			foreach (UIElement child in Children)
			{
				totalProportion += GetProportion(child);
			}
			foreach (UIElement child in Children)
			{
				if (Orientation == Orientation.Horizontal)
				{
					childWidth = constraint.Width * (GetProportion(child) / totalProportion);
					childHeight = constraint.Height;
				}
				if (Orientation == Orientation.Vertical)
				{
					childWidth = constraint.Width;
					childHeight = constraint.Height * (GetProportion(child) / totalProportion);
				}
				if (Double.IsNaN(childWidth))
				{
					childWidth = 0.0;
				}
				if (Double.IsNaN(childHeight))
				{
					childHeight = 0.0;
				}

				child.Measure(new Size(childWidth, childHeight));
				maxChildHeight = Math.Max(child.DesiredSize.Height, maxChildHeight);
				maxChildWidth = Math.Max(child.DesiredSize.Width, maxChildWidth);
			}

			Size controlSize = new Size();
			if (Orientation == Orientation.Horizontal)
			{
				controlSize = new Size(constraint.Width, maxChildHeight);
			}
			if (Orientation == Orientation.Vertical)
			{
				controlSize = new Size(maxChildWidth, constraint.Height);
			}

			return controlSize;
		}
	}
}
