using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Text;
using org.ovirt.engine.ui.uicommon.models.configure;
using org.ovirt.engine.ui.uicompat;
using VdcCommon;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.VdcQueries;
using System.Collections;
using System.ComponentModel;
using System.Globalization;
using org.ovirt.engine.ui.uicompat;
using org.ovirt.engine.ui.uicommon.dataprovider;

namespace org.ovirt.engine.ui.uicommon.models.pools
{
	public class PoolListModel : ListWithDetailsModel
	{
		#region Commands

		public UICommand NewCommand { get; private set; }
		public UICommand EditCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }

		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		protected object[] SelectedKeys
		{
			get
			{
				//              return SelectedItems == null ? new object[0] : SelectedItems.Cast<vm_pools>().Select(a => a.vm_pool_id).Cast<object>().ToArray(); }
				if (SelectedItems == null)
				{
					return new object[0];
				}
				else
				{
					object[] keys = new object[SelectedItems.Count];
					for (int i = 0; i < SelectedItems.Count; i++)
					{
						keys[i] = ((vm_pools)SelectedItems[i]).vm_pool_id;
					}
					return keys;
				}
			}
		}
		#endregion


		public PoolListModel()
		{
			Title = "Pools";

			DefaultSearchString = "Pools:";
			SearchString = DefaultSearchString;

			NewCommand = new UICommand("New", this);
			EditCommand = new UICommand("Edit", this);
			RemoveCommand = new UICommand("Remove", this);

			UpdateActionAvailability();

			SearchNextPageCommand.IsAvailable = true;
			SearchPreviousPageCommand.IsAvailable = true;
		}

		protected override void InitDetailModels()
		{
			base.InitDetailModels();

			ObservableCollection<EntityModel> list = new ObservableCollection<EntityModel>();
			list.Add(new PoolGeneralModel());
			list.Add(new PoolVmListModel());
			list.Add(new PermissionListModel());
			DetailModels = list;
		}

		public override bool IsSearchStringMatch(string searchString)
		{
			return searchString.Trim().ToLower().StartsWith("pool");
		}

		protected override void SyncSearch()
		{
			base.SyncSearch(VdcQueryType.Search, new SearchParameters(SearchString, SearchType.VmPools)
			{
				MaxCount = SearchPageSize
			});
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			AsyncResult = Frontend.RegisterSearch(SearchString, SearchType.VmPools, SearchPageSize);
			Items = AsyncResult.Data;
		}

		public override void Search()
		{
			base.Search();
		}

		public void New()
		{
			if (Window != null)
			{
				return;
			}

			PoolModel model = new PoolModel();
			Window = model;
			model.Title = "New Pool";
			model.HashName = "new_pool";
			model.IsNew = true;
			model.VmType = VmType.Desktop;
			//			model.DataCenter.Value = model.DataCenter.Options.Cast<storage_pool>().FirstOrDefault();
			model.DataCenter.SelectedItem = Linq.FirstOrDefault(Linq.Cast<storage_pool>(model.DataCenter.Items));

			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void Edit()
		{
			vm_pools pool = (vm_pools)SelectedItem;

			if (Window != null)
			{
				return;
			}

			VM vm = Linq.FirstOrDefault(DataProvider.GetVmList(pool.vm_pool_name));

			PoolModel model = new PoolModel();
			Window = model;
			model.Title = "Edit Pool";
			model.HashName = "edit_pool";
			model.VmType = VmType.Desktop;
			model.Name.Entity = pool.vm_pool_name;
			model.Description.Entity = pool.vm_pool_description;
			model.AssignedVms = pool.vm_assigned_count;

			//model.PoolType.Value = model.PoolType.Options
			//		.Cast<EntityModel>()
			//		.FirstOrDefault(a => (VmPoolType)a.Entity == pool.vm_pool_type);
			foreach (object item in model.PoolType.Items)
			{
				EntityModel a = (EntityModel)item;
				if ((VmPoolType)a.Entity == pool.vm_pool_type)
				{
					model.PoolType.SelectedItem = a;
					break;
				}
			}

			string cdImage = null;

			if (vm != null)
			{
				//model.DataCenter.Value = model.DataCenter.Options
				//	.Cast<storage_pool>()
				//	.FirstOrDefault(a => a.id == vm.storage_pool_id);
				model.DataCenter.SelectedItem = null;
				foreach (object item in model.DataCenter.Items)
				{
					storage_pool a = (storage_pool)item;
					if (a.Id.Equals(vm.storage_pool_id))
					{
						model.DataCenter.SelectedItem = a;
						break;
					}
				}

				model.DataCenter.IsChangable = false;
				model.Template.IsChangable = false;

				if (model.DataCenter.SelectedItem == null)
				{
					List<storage_pool> list = new List<storage_pool>() { DataProvider.GetDataCenterById(vm.storage_pool_id) };
					model.DataCenter.Items = list;
					model.DataCenter.SelectedItem = list[0];
				}

				List<VmTemplate> templates = new List<VmTemplate>();
				VmTemplate basedOnTemplate = DataProvider.GetTemplateByID(vm.vmt_guid);
				if (basedOnTemplate != null)
				{
					templates.Add(basedOnTemplate);
				}

				model.Template.Items = templates;
				model.Template.SelectedItem = basedOnTemplate;

				//model.DefaultHost.Value = model.DefaultHost.Options
				//    .Cast<VDS>()
				//    .FirstOrDefault(a => a.vds_id == (vm.dedicated_vm_for_vds.HasValue ? vm.dedicated_vm_for_vds : -1));
				model.DefaultHost.SelectedItem = null;
				VDS host = null;
				foreach (object item in model.DefaultHost.Items)
				{
					VDS a = (VDS)item;
					if (a.vds_id.Equals(vm.dedicated_vm_for_vds ?? Guid.Empty))
					{
						host = a;
						break;
					}
				}

				if (host == null)
				{
					model.IsAutoAssign.Entity = true;
				}
				else
				{
					model.DefaultHost.SelectedItem = host;
					model.IsAutoAssign.Entity = false;
				}

				if (vm.MigrationSupport == MigrationSupport.PINNED_TO_HOST)
				{
					model.RunVMOnSpecificHost.Entity = true;
				}
				else
				{
					if (vm.MigrationSupport == MigrationSupport.IMPLICITLY_NON_MIGRATABLE)
					{
						model.DontMigrateVM.Entity = true;
					}
				}

				model.MemSize.Entity = vm.vm_mem_size_mb;
				model.MinAllocatedMemory.Entity = vm.MinAllocatedMem;
				model.OSType.SelectedItem = vm.vm_os;
				model.Domain.SelectedItem = vm.vm_domain;

				if (!string.IsNullOrEmpty(vm.time_zone))
				{
					model.TimeZone.SelectedItem = Linq.FirstOrDefault((IEnumerable<KeyValuePair<string, string>>)model.TimeZone.Items, new Linq.TimeZonePredicate(vm.time_zone));
				}

				//model.DisplayProtocol.Value = model.DisplayProtocol.Options
				//	.Cast<EntityModel>()
				//	.FirstOrDefault(a => (DisplayType)a.Entity == vm.default_display_type);
				EntityModel displayType = null;
				foreach (object item in model.DisplayProtocol.Items)
				{
					EntityModel a = (EntityModel)item;
					DisplayType dt = (DisplayType)a.Entity;
					if (dt == vm.default_display_type)
					{
						displayType = a;
						break;
					}
				}
				model.DisplayProtocol.SelectedItem = displayType;

				model.UsbPolicy.SelectedItem = vm.usb_policy;
				model.NumOfMonitors.SelectedItem = vm.num_of_monitors;
				model.NumOfSockets.Entity = vm.num_of_sockets;
				model.TotalCPUCores.Entity = vm.num_of_cpus;
				model.BootSequence = vm.default_boot_sequence;

				model.Kernel_path.Entity = vm.kernel_url;
				model.Kernel_parameters.Entity = vm.kernel_params;
				model.Initrd_path.Entity = vm.initrd_url;

				//feature for filling storage domain in case of datacenter list empty
				List<DiskImage> disks = DataProvider.GetVmDiskList(vm.vm_guid);
				Guid? storageId = disks[0].storage_id;
				if (disks.Count > 0 && storageId.HasValue)
				{
					storage_domains storage = DataProvider.GetStorageDomainById(disks[0].storage_id.Value);
					model.StorageDomain.Items = new List<storage_domains> { storage };
					model.StorageDomain.SelectedItem = storage;
				}
				model.StorageDomain.IsChangable = false;

				cdImage = vm.iso_path;
			}
			else
			{
				//model.DataCenter.Value = model.DataCenter.Options.Cast<storage_pool>().FirstOrDefault();
				model.DataCenter.SelectedItem = Linq.FirstOrDefault(Linq.Cast<storage_pool>(model.DataCenter.Items));
			}

			// make sure that Clusters list won't be null:
			List<VDSGroup> clusters = new List<VDSGroup>();
			if (model.Cluster.Items == null)
			{
				VDSGroup poolCluster = DataProvider.GetClusterById(pool.vds_group_id);
				if (poolCluster != null)
				{
					clusters.Add(poolCluster);
				}

				model.Cluster.Items = clusters;
			}

			//model.Cluster.Value = model.Cluster.Options
			//         .Cast<VDSGroup>()
			//         .FirstOrDefault(a => a.ID == pool.vds_group_id);
			model.Cluster.SelectedItem = null;
			foreach (object item in model.Cluster.Items)
			{
				VDSGroup a = (VDSGroup)item;
				if (a.ID.Equals(pool.vds_group_id))
				{
					model.Cluster.SelectedItem = a;
					break;
				}
			}

			model.Cluster.IsChangable = vm == null;


			bool hasCd = !String.IsNullOrEmpty(cdImage);
			model.CdImage.IsChangable = hasCd;
			if (hasCd)
			{
				model.CdImage.SelectedItem = cdImage;
			}


			model.Provisioning.IsChangable = false;
			model.StorageDomain.IsChangable = false;


			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void Remove()
		{
			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Remove Pool(s)";
			model.HashName = "remove_pool";
			model.Message = "Pool(s)";

			List<string> list = new List<string>();
			foreach (vm_pools item in Linq.Cast<vm_pools>(SelectedItems))
			{
				list.Add(item.vm_pool_name);
			}
			model.Items = list;


			model.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnRemove()
		{
			ConfirmationModel model = (ConfirmationModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				vm_pools pool = (vm_pools)item;
				list.Add(new VmPoolParametersBase(pool.vm_pool_id));
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.RemoveVmPool, list,
				result =>
				{
					ConfirmationModel localModel = (ConfirmationModel)result.State;

					localModel.StopProgress();
					Cancel();
				},
				model
			);
		}

		public void OnSave()
		{
			PoolModel model = (PoolModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (!model.IsNew && SelectedItem == null)
			{
				Cancel();
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			vm_pools pool = model.IsNew ? new vm_pools() : (vm_pools)Cloner.Clone(SelectedItem);

			string name = (string)model.Name.Entity;

			//Check name unicitate.
			if (!DataProvider.IsPoolNameUnique(name) && String.Compare(name, pool.vm_pool_name, true) != 0)
			{
				model.Name.IsValid = false;
				model.Name.InvalidityReasons.Add("Name must be unique.");
				model.IsGeneralTabValid = false;
				return;
			}


			//Save changes.
			pool.vm_pool_name = (string)model.Name.Entity;
			pool.vm_pool_description = (string)model.Description.Entity;
			pool.vds_group_id = ((VDSGroup)model.Cluster.SelectedItem).ID;

			EntityModel poolTypeSelectedItem = (EntityModel)model.PoolType.SelectedItem;
			pool.vm_pool_type = (VmPoolType)poolTypeSelectedItem.Entity;

			Guid? default_host;
			VDS defaultHost = (VDS)model.DefaultHost.SelectedItem;
			if ((bool)model.IsAutoAssign.Entity)
			{
				default_host = null;
			}
			else
			{
				default_host = defaultHost.vds_id;
			}

			MigrationSupport migrationSupport = MigrationSupport.MIGRATABLE;
			if ((bool)model.RunVMOnSpecificHost.Entity)
				migrationSupport = MigrationSupport.PINNED_TO_HOST;
			else if ((bool)model.DontMigrateVM.Entity)
				migrationSupport = MigrationSupport.IMPLICITLY_NON_MIGRATABLE;

			VM desktop =
				new VM
					{
						vmt_guid = ((VmTemplate)model.Template.SelectedItem).Id,
						vm_name = name,
						vm_os = (VmOsType)model.OSType.SelectedItem,
						num_of_monitors = (int)model.NumOfMonitors.SelectedItem,
						vm_domain = model.Domain.IsAvailable ? (string)model.Domain.SelectedItem : String.Empty,
						vm_mem_size_mb = (int)model.MemSize.Entity,
						MinAllocatedMem = (int)model.MinAllocatedMemory.Entity,
						vds_group_id = ((VDSGroup)model.Cluster.SelectedItem).ID,
						time_zone = (model.TimeZone.IsAvailable && model.TimeZone.SelectedItem != null) ? ((KeyValuePair<string, string>)model.TimeZone.SelectedItem).Key : String.Empty,
						num_of_sockets = (int)model.NumOfSockets.Entity,
						cpu_per_socket = (int)model.TotalCPUCores.Entity / (int)model.NumOfSockets.Entity,
						usb_policy = (UsbPolicy)model.UsbPolicy.SelectedItem,
						is_auto_suspend = false,
						is_stateless = false,
						default_boot_sequence = model.BootSequence,
						iso_path = model.CdImage.IsChangable ? (string)model.CdImage.SelectedItem : String.Empty,
						dedicated_vm_for_vds = default_host,
						kernel_url = (string)model.Kernel_path.Entity,
						kernel_params = (string)model.Kernel_parameters.Entity,
						initrd_url = (string)model.Initrd_path.Entity,
						MigrationSupport = migrationSupport
					};

			EntityModel displayProtocolSelectedItem = (EntityModel)model.DisplayProtocol.SelectedItem;
			desktop.default_display_type = (DisplayType)displayProtocolSelectedItem.Entity;


			AddVmPoolWithVmsParameters param =
				new AddVmPoolWithVmsParameters(pool, desktop, model.IsAddVMMode ? Convert.ToInt32(model.NumOfDesktops.Entity.ToString()) : 0, 0)
				{
					StorageDomainId = ((storage_domains)model.StorageDomain.SelectedItem).id
				};


			model.StartProgress(null);

			if (model.IsNew)
			{
				Frontend.RunMultipleAction(VdcActionType.AddVmPoolWithVms, new List<VdcActionParametersBase> { param },
				result =>
				{
					PoolListModel poolListModel = (PoolListModel)result.State;
					poolListModel.Cancel();
					poolListModel.StopProgress();
				}, this);
			}
			else
			{
				Frontend.RunMultipleAction(VdcActionType.UpdateVmPoolWithVms,
					new List<VdcActionParametersBase> { param },
				result =>
				{
					PoolListModel poolListModel = (PoolListModel)result.State;
					poolListModel.Cancel();
					poolListModel.StopProgress();
				}, this);
			}
		}

		public void Cancel()
		{
			Window = null;
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.SelectedItemPropertyChanged(sender, e);
			UpdateActionAvailability();
		}

		private void UpdateActionAvailability()
		{
			EditCommand.IsExecutionAllowed = SelectedItem != null
				&& SelectedItems != null
				&& SelectedItems.Count == 1;

			RemoveCommand.IsExecutionAllowed = SelectedItems != null
				&& SelectedItems.Count > 0
				&& VdcActionUtils.CanExecute(SelectedItems, typeof(vm_pools), VdcActionType.RemoveVmPool);
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == NewCommand)
			{
				New();
			}
			if (command == EditCommand)
			{
				Edit();
			}
			if (command == RemoveCommand)
			{
				Remove();
			}
			if (command.Name == "Cancel")
			{
				Cancel();
			}
			if (command.Name == "OnSave")
			{
				OnSave();
			}
			if (command.Name == "OnRemove")
			{
				OnRemove();
			}
		}
	}
}
