using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Text;
using System.Windows;
using org.ovirt.engine.ui.uicommon.models.common;
using org.ovirt.engine.ui.uicompat;
using VdcCommon;
using VdcCommon.VdcQueries;
using VdcFrontend;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using System.Collections.ObjectModel;
using System.Collections;

namespace org.ovirt.engine.ui.uicommon.models.configure.roles_ui
{
	public class RoleListModel : ListWithDetailsModel
	{
		private const string COPY_OF = "Copy_of_";

		private enum CommandType
		{
			New, Edit, Clone
		}

		#region Commands

		public UICommand NewCommand { get; private set; }
		public UICommand EditCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }
		public UICommand CloneCommand { get; private set; }
		public UICommand SearchAllRolesCommand { get; private set; }
		public UICommand SearchAdminRolesCommand { get; private set; }
		public UICommand SearchUserRolesCommand { get; private set; }

		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		#endregion

		private CommandType commandType;
		private List<ActionGroup> attachedActions;
		public List<ActionGroup> detachActionGroup;
		public List<ActionGroup> attachActionGroup;
		public roles role;
		public RoleType? ItemsFilter { get; set; }

		public RoleListModel()
		{
			Title = "Roles";

			NewCommand = new UICommand("New", this);
			EditCommand = new UICommand("Edit", this);
			RemoveCommand = new UICommand("Remove", this);
			CloneCommand = new UICommand("Clone", this);
			SearchAllRolesCommand = new UICommand("SearchAllRoles", this);
			SearchAdminRolesCommand = new UICommand("SearchAdminRoles", this);
			SearchUserRolesCommand = new UICommand("SearchUserRoles", this);

			SearchCommand.Execute();

			UpdateActionAvailability();
		}

		protected override void InitDetailModels()
		{
			base.InitDetailModels();

			ObservableCollection<EntityModel> list = new ObservableCollection<EntityModel>();
			list.Add(new RolePermissionListModel());

			DetailModels = list;
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			AsyncQuery _asyncQuery = new AsyncQuery();
			_asyncQuery.Model = this;
			/*START_DELEGATE*/_asyncQuery.asyncCallback = delegate(Object model, Object ReturnValue)
				{
					RoleListModel roleListModel = (RoleListModel)model;
					List<roles> filteredList = new List<roles>();
					foreach (roles item in (List<roles>)((VdcQueryReturnValue)ReturnValue).ReturnValue)
					{
						if (!roleListModel.ItemsFilter.HasValue || roleListModel.ItemsFilter == item.Type)
						{
							filteredList.Add(item);
						}
					}

					roleListModel.Items = filteredList;
				};//END_DELEGATE


			Frontend.RunQuery(VdcQueryType.GetAllRoles,
				new MultilevelAdministrationsQueriesParameters(),
				_asyncQuery);
		}

		private void SearchAllRoles()
		{
			ItemsFilter = null;
			SearchCommand.Execute();
		}

		private void SearchUserRoles()
		{
			ItemsFilter = RoleType.USER;
			SearchCommand.Execute();
		}

		private void SearchAdminRoles()
		{
			ItemsFilter = RoleType.ADMIN;
			SearchCommand.Execute();
		}

		public void Remove()
		{
			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Remove Role(s)";
			model.HashName = "remove_role";
			model.Message = "Role(s):";

			List<string> list = new List<string>();
			foreach (roles role in Linq.Cast<roles>(SelectedItems))
			{
				list.Add(role.name);
			}
			model.Items = list;


			model.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnRemove()
		{
			foreach (object item in SelectedItems)
			{
				roles role = (roles)item;
				Frontend.RunAction(VdcActionType.RemoveRole, new RolesParameterBase(role.Id));
			}

			Cancel();

			//Execute search to keep list updated.
			SearchCommand.Execute();
		}

		public void Edit()
		{
			commandType = CommandType.Edit;
			roles role = (roles)SelectedItem;
			InitRoleDialog(role);
		}

		public void New()
		{
			commandType = CommandType.New;
			roles role = new roles();
			InitRoleDialog(role);
		}


		public void CloneRole()
		{
			commandType = CommandType.Clone;
			roles role = (roles)SelectedItem;
			InitRoleDialog(role);
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);


			if (Window != null && sender == ((RoleModel)Window).IsAdminRole)
			{
				RoleModel model = (RoleModel)Window;
				roles role = (roles)SelectedItem;
				List<ActionGroup> attachedActions = (commandType == CommandType.New) ? new List<ActionGroup>() : DataProvider.GetRoleActionGroupsByRoleId(role.Id);

				model.PermissionGroupModels = null;

				model.PermissionGroupModels = RoleTreeView.GetRoleTreeView((model.IsNew ? false : role.is_readonly), (bool)model.IsAdminRole.Entity);
				foreach (SelectionTreeNodeModel sm in model.PermissionGroupModels)
				{
					foreach (SelectionTreeNodeModel smChild in sm.Children)
					{
						smChild.Parent = sm;
						smChild.IsSelectedNotificationPrevent = false;

						foreach (SelectionTreeNodeModel smGrandChild in smChild.Children)
						{
							smGrandChild.Parent = smChild;
							smGrandChild.IsSelectedNotificationPrevent = false;

							if (attachedActions.Contains((ActionGroup)Enum.Parse(typeof(ActionGroup), smGrandChild.Title)))
							{
								smGrandChild.IsSelectedNullable = true;
								smGrandChild.UpdateParentSelection();
							}

							if (smChild.Children[0].Equals(smGrandChild))
							{
								smGrandChild.UpdateParentSelection();
							}
						}
					}
				}
			}
		}

		private void InitRoleDialog(roles role)
		{
			if (Window != null)
			{
				return;
			}

			RoleModel model = new RoleModel();
			Window = model;
			model.IsNew = commandType != CommandType.Edit;
			if (commandType == CommandType.New)
			{
				role.Type = RoleType.USER;
			}
			model.IsAdminRole.EntityChangedEvent.addListener(this);
			model.IsAdminRole.Entity = role.Type == RoleType.ADMIN;
			model.Name.Entity = role.name;
			if (commandType == CommandType.Clone)
			{
				model.Name.Entity = COPY_OF + role.name;
			}
			model.Description.Entity = role.description;
			if (commandType == CommandType.Edit)
			{
				model.Name.IsChangable = !role.is_readonly;
				model.Description.IsChangable = !role.is_readonly;
			}
			string title = null;
			string hashName = null;
			switch (commandType)
			{
				case CommandType.New:
					title = "New Role";
					hashName = "new_role";
					break;
				case CommandType.Edit:
					title = "Edit Role";
					hashName = "edit_role";
					model.IsAdminRole.IsChangable = false;
					break;
				case CommandType.Clone:
					title = "Copy Role";
					hashName = "copy_role";
					model.IsAdminRole.IsChangable = false;
					break;

			}

			model.Title = title;
			model.HashName = hashName;
			if (!role.is_readonly || commandType == CommandType.Clone)
			{
				model.Commands.Add(
					new UICommand("OnSave", this)
					{
						Title = "OK",
						IsDefault = true
					});
				model.Commands.Add(
					new UICommand("OnReset", this)
					{
						Title = "Reset",
					});
			}

			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = !role.is_readonly ? "Cancel" : "Close",
					IsCancel = true,
					IsDefault = role.is_readonly
				});
		}

		public void OnReset()
		{
			roles role = (roles)SelectedItem;
			RoleModel model = (RoleModel)Window;

			List<ActionGroup> attachedActions = commandType == CommandType.New
				? new List<ActionGroup>()
				: DataProvider.GetRoleActionGroupsByRoleId(role.Id);

			foreach (SelectionTreeNodeModel sm in model.PermissionGroupModels)
			{
				foreach (SelectionTreeNodeModel smChild in sm.Children)
				{
					foreach (SelectionTreeNodeModel smGrandChild in smChild.Children)
					{
						smGrandChild.IsSelectedNullable = attachedActions.Contains((ActionGroup)Enum.Parse(typeof(ActionGroup), smGrandChild.Title));
					}
				}
			}
		}

		public void OnSave()
		{
			RoleModel model = (RoleModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			role = commandType != CommandType.Edit ? new roles() : (roles)SelectedItem;
			role.Type = ((bool)model.IsAdminRole.Entity ? RoleType.ADMIN : RoleType.USER);
			
			if (!model.Validate())
			{
				return;
			}

			//Check name unicitate.
			string name = (string)model.Name.Entity;

			if (!DataProvider.IsRoleNameUnique(name) && String.Compare(name, role.name, true) != 0)
			{
				model.Name.IsValid = false;
				model.Name.InvalidityReasons.Add("Name must be unique.");
				return;
			}


			role.name = (string)model.Name.Entity;
			role.description = (string)model.Description.Entity;

			List<ActionGroup> actions = new List<ActionGroup>();
			Dictionary<ActionGroup, ActionGroup> actionDistinctSet = new Dictionary<ActionGroup, ActionGroup>();
			foreach (SelectionTreeNodeModel sm in model.PermissionGroupModels)
			{
				foreach (SelectionTreeNodeModel smChild in sm.Children)
				{
					if (smChild.IsSelectedNullable == null || smChild.IsSelectedNullable.Value)
					{
						foreach (SelectionTreeNodeModel smGrandChild in smChild.Children)
						{
							if (smGrandChild.IsSelectedNullable.Value)
							{
								ActionGroup actionGroup = (ActionGroup)Enum.Parse(typeof(ActionGroup), smGrandChild.Title);
								if(actionDistinctSet.ContainsKey(actionGroup))
									continue;
								actionDistinctSet.Add(actionGroup, actionGroup);
								actions.Add(actionGroup);
							}
						}
					}
				}
			}

			VdcReturnValueBase returnValue;


			model.StartProgress(null);

			if (commandType != CommandType.Edit)
			{
				//Add a new role.
				Frontend.RunAction(VdcActionType.AddRoleWithActionGroups,
					new RoleWithActionGroupsParameters
					{
						Role = role,
						ActionGroups = actions
					},
					result =>
					{
						RoleListModel localModel = (RoleListModel)result.State;

						localModel.PostOnSaveNew(result.ReturnValue);
					},
					this
				);
			}
			else
			{
				attachedActions = DataProvider.GetRoleActionGroupsByRoleId(role.Id);
				detachActionGroup = Linq.Except(attachedActions, actions);
				attachActionGroup = Linq.Except(actions, attachedActions);

				Frontend.RunAction(VdcActionType.UpdateRole,
					new RolesOperationsParameters(role),
					result =>
						{
							RoleListModel roleListModel = (RoleListModel) result.State;
							VdcReturnValueBase retVal = result.ReturnValue;
							if (retVal != null && retVal.Succeeded)
							{
								if (roleListModel.detachActionGroup.Count > 0)
								{
									retVal = Frontend.RunAction(VdcActionType.DetachActionGroupsFromRole,
		                                               new ActionGroupsToRoleParameter
		                                               	{
		                                               		ActionGroups = roleListModel.detachActionGroup,
		                                               		RoleId = roleListModel.role.Id
		                                               	});
								}
								if (roleListModel.attachActionGroup.Count > 0)
								{
									retVal = Frontend.RunAction(VdcActionType.AttachActionGroupsToRole,
													new ActionGroupsToRoleParameter
				                            			{
				                            				ActionGroups = roleListModel.attachActionGroup,
				                            				RoleId = roleListModel.role.Id
				                            			});
								}

								roleListModel.Window.StopProgress();
								roleListModel.Cancel();
							}
							else
							{
								roleListModel.Window.StopProgress();
							}
						},
					this
				);
			}
		}

		public void PostOnSaveNew(VdcReturnValueBase returnValue)
		{
			RoleModel model = (RoleModel)Window;

			model.StopProgress();

			if (returnValue != null && returnValue.Succeeded)
			{
				Cancel();
				SearchCommand.Execute();
			}
		}
           
		public void Cancel()
		{
			Window = null;
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		private void UpdateActionAvailability()
		{
			bool temp = SelectedItems != null && SelectedItems.Count == 1;

			CloneCommand.IsExecutionAllowed = temp;
			EditCommand.IsExecutionAllowed = temp;
			RemoveCommand.IsExecutionAllowed = SelectedItems != null && SelectedItems.Count > 0 && !IsAnyRoleReadOnly(SelectedItems);
		}

		private bool IsAnyRoleReadOnly(IList roles)
		{
			foreach (object item in roles)
			{
				roles r = (roles)item;
				if (r.is_readonly)
				{
					return true;
				}
			}
			return false;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == NewCommand)
			{
				New();
			}
			else if (command == EditCommand)
			{
				Edit();
			}
			else if (command == RemoveCommand)
			{
				Remove();
			}
			else if (command == SearchAllRolesCommand)
			{
				SearchAllRoles();
			}
			else if (command == SearchAdminRolesCommand)
			{
				SearchAdminRoles();
			}
			else if (command == SearchUserRolesCommand)
			{
				SearchUserRoles();
			}
			else if (command.Name == "OnSave")
			{
				OnSave();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
			else if (command.Name == "OnRemove")
			{
				OnRemove();
			}
			else if (command.Name == "OnReset")
			{
				OnReset();
			}
			else if (command.Name == "Clone")
			{
				CloneRole();
			}
		}
	}
}
