using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using VdcUtils;

namespace SearchBackend
{
    public enum SyntaxObjectType
    {
        BEGIN                   = 0,
        SEARCH_OBJECT           = 1,
        COLON                   = 2,
        CROSS_REF_OBJ           = 3,
        DOT                     = 4,
        CONDITION_FIELD         = 5,
        CONDITION_RELATION      = 6,
        CONDITION_VALUE         = 7,
        OR                      = 8,
        AND                     = 9,
        SORTBY                  = 10,
        SORT_FIELD              = 11,
        SORT_DIRECTION          = 12,
        PAGE                    = 13,
        PAGE_VALUE              = 14,
        END                     = 15

    }
 
    public enum PagingType
    {
        Range,
        Offset
    }

    public class SyntaxChecker : ISyntaxChecker
    {
        private readonly SearchObjectAutoCompleter mSearchObjectAC;
        private readonly BaseAutoCompleter mColonAC;
        private readonly BaseAutoCompleter mPluralAC;
        private readonly BaseAutoCompleter mSortbyAC;
        private readonly BaseAutoCompleter mPageAC;
        private readonly BaseAutoCompleter mAndAC;
        private readonly BaseAutoCompleter mOrAC;
        private readonly BaseAutoCompleter mDotAC;
        private readonly BaseAutoCompleter mSortDirectionAC;
        private readonly Dictionary<SyntaxObjectType, SyntaxObjectType[]> mStateMap;

        private readonly Regex mFirstDQRegexp;
        private readonly Regex mNonSpaceRegexp;
        private readonly List<char> mDisAllowedChars;
        private readonly int mSearchReasultsLimit;


        internal SyntaxChecker(int searchReasultsLimit, bool hasDesktop)
        {
            
            mSearchReasultsLimit = searchReasultsLimit;
        
            mSearchObjectAC         = new SearchObjectAutoCompleter(hasDesktop);
            mColonAC                = new BaseAutoCompleter(":");
            mPluralAC               = new BaseAutoCompleter("S");
            mSortbyAC               = new BaseAutoCompleter("SORTBY");
            mPageAC                 = new BaseAutoCompleter("PAGE");
            mSortDirectionAC        = new BaseAutoCompleter(new[] { "ASC", "DESC" });
            mAndAC                  = new BaseAutoCompleter("AND");
            mOrAC                   = new BaseAutoCompleter("OR");
            mDotAC                  = new BaseAutoCompleter(".");
            mDisAllowedChars        = new List<char> {'\'', ';'};

            mFirstDQRegexp =  new Regex("^\\s*\"$");
            mNonSpaceRegexp = new Regex("^\\S+$");

            mStateMap = new Dictionary<SyntaxObjectType, SyntaxObjectType[]>();
            mStateMap.Add(SyntaxObjectType.BEGIN, new []{SyntaxObjectType.SEARCH_OBJECT});
            mStateMap.Add(SyntaxObjectType.SEARCH_OBJECT, new[] { SyntaxObjectType.COLON });
            SyntaxObjectType[] afterColon = { SyntaxObjectType.CROSS_REF_OBJ, SyntaxObjectType.CONDITION_FIELD, SyntaxObjectType.SORTBY, SyntaxObjectType.PAGE ,SyntaxObjectType.CONDITION_VALUE, SyntaxObjectType.END };
            mStateMap.Add(SyntaxObjectType.COLON, afterColon);
            
            SyntaxObjectType[] afterCrossRefObj = { SyntaxObjectType.DOT, SyntaxObjectType.CONDITION_RELATION };
            mStateMap.Add(SyntaxObjectType.CROSS_REF_OBJ, afterCrossRefObj);
            mStateMap.Add(SyntaxObjectType.DOT, new[] { SyntaxObjectType.CONDITION_FIELD });

            mStateMap.Add(SyntaxObjectType.CONDITION_FIELD, new[] { SyntaxObjectType.CONDITION_RELATION });
            mStateMap.Add(SyntaxObjectType.CONDITION_RELATION, new[] { SyntaxObjectType.CONDITION_VALUE });
            SyntaxObjectType[] afterConditionValue = {SyntaxObjectType.OR, SyntaxObjectType.AND, SyntaxObjectType.CROSS_REF_OBJ, SyntaxObjectType.CONDITION_FIELD, SyntaxObjectType.SORTBY, SyntaxObjectType.PAGE, SyntaxObjectType.CONDITION_VALUE };
            mStateMap.Add(SyntaxObjectType.CONDITION_VALUE, afterConditionValue);

            SyntaxObjectType[] AndOrArray = { SyntaxObjectType.CROSS_REF_OBJ, SyntaxObjectType.CONDITION_FIELD, SyntaxObjectType.CONDITION_VALUE};
            mStateMap.Add(SyntaxObjectType.AND, AndOrArray);
            mStateMap.Add(SyntaxObjectType.OR, AndOrArray);

            mStateMap.Add(SyntaxObjectType.SORTBY, new [] { SyntaxObjectType.SORT_FIELD });
            mStateMap.Add(SyntaxObjectType.SORT_FIELD, new[] { SyntaxObjectType.SORT_DIRECTION });
            mStateMap.Add(SyntaxObjectType.SORT_DIRECTION, new[] { SyntaxObjectType.PAGE });

            mStateMap.Add(SyntaxObjectType.PAGE, new[] { SyntaxObjectType.PAGE_VALUE });
            mStateMap.Add(SyntaxObjectType.PAGE_VALUE, new[] { SyntaxObjectType.END });
            
		}

        private enum ValueParseResult
        {
            Err,
            Normal,
            FreeText
        }
        private ValueParseResult handleValuePhrase(bool final, string searchText, int idx, ref int startPos, SyntaxContainer container)
        {
            bool addObjFlag = false;
            ValueParseResult retval = ValueParseResult.Normal;
            IConditionFieldAutoCompleter curConditionFieldAC;
            char curChar = searchText[idx];
            string strRealObj = searchText.Substring(startPos, idx - startPos + 1);

            bool betweenDoubleQuotes = searchText.Substring(startPos, idx - startPos).Contains("\"");
            if (curChar == '"')
            {
                betweenDoubleQuotes = (!betweenDoubleQuotes);
                if (betweenDoubleQuotes)
                {
                    if (!mFirstDQRegexp.IsMatch(strRealObj))
                    {
                        container.setErr(SyntaxError.INVALID_CONDITION_VALUE, startPos, idx + 1);
                        return 0;
                    }
                }
                else
                {
                    strRealObj = strRealObj.Trim(new char[] { '\"' });
                    addObjFlag = true;
                }
            }
            //Doing this condition to identify whether this is the last searchObject and no space is predicted !!
            if (final)
            {
                if (((curChar == ' ') || (idx + 1 == searchText.Length)) && (betweenDoubleQuotes == false) && (addObjFlag == false))
                {
                    strRealObj = strRealObj.Trim();
                    if (mNonSpaceRegexp.IsMatch(strRealObj))
                    {
                        addObjFlag = true;
                    }
                    else
                    {
                        startPos = idx + 1;
                    }
                }
            }
            else
            {
                if ((curChar == ' ') && (betweenDoubleQuotes == false) && (addObjFlag == false))
                {
                    strRealObj = strRealObj.Trim();
                    if (mNonSpaceRegexp.IsMatch(strRealObj))
                    {
                        addObjFlag = true;
                    }
                    else
                    {
                        startPos = idx + 1;
                    }
                }
            }
            if (addObjFlag)
            {
                string curRefObj = container.getPreviousSyntaxObject(3, SyntaxObjectType.CROSS_REF_OBJ);
                string curConditionField = container.getPreviousSyntaxObject(1, SyntaxObjectType.CONDITION_FIELD);
                curConditionFieldAC = mSearchObjectAC.getFieldAutoCompleter(curRefObj);
                if (curConditionFieldAC == null)
                {
                    container.setErr(SyntaxError.CANT_GET_CONDITION_FIELD_AC, startPos, idx);
                    return ValueParseResult.Err;
                }
                if ((curConditionField != string.Empty) && (!curConditionFieldAC.validateFieldValue(curConditionField, strRealObj)))
                {
                    container.setErr(SyntaxError.INVALID_CONDITION_VALUE, startPos, idx);
                    return ValueParseResult.Err;
                }
                container.addSyntaxObject(SyntaxObjectType.CONDITION_VALUE, strRealObj, startPos, idx + 1);
                retval = ValueParseResult.FreeText;
                startPos = idx + 1;
                container.valid = true;
            }
            return retval;
        }

        public SyntaxContainer analyzeSyntaxState(string searchText, bool final)
        {
            SyntaxContainer retval = new SyntaxContainer(searchText);
            IConditionFieldAutoCompleter curConditionFieldAC = null;
            IAutoCompleter curConditionRelationAC = null;
            List<string> freeTextObjSearched = new List<string>();
            char[] searchCharArr = searchText.ToCharArray();
            int curStartPos = 0;


            string tryNextObj = string.Empty;
            bool keepValid;
            for (int idx = 0; idx < searchCharArr.Length; idx++)
            {
                SyntaxObjectType curState = retval.getState();
                char curChar = searchCharArr[idx];
                if (mDisAllowedChars.Contains(curChar))
                {
                    retval.setErr(SyntaxError.INVALID_CHARECTER, curStartPos, idx + 1);
                    return retval;
                }
                if ((curChar == ' ') && 
                    (curState != SyntaxObjectType.CONDITION_RELATION) && 
                    (curState != SyntaxObjectType.COLON)&&
                    (curState != SyntaxObjectType.CONDITION_VALUE)&&
                    (curState != SyntaxObjectType.OR)&&
                    (curState != SyntaxObjectType.AND))
                {
                    curStartPos += 1;
                    continue;
                }
                string strRealObj = searchText.Substring(curStartPos, idx - curStartPos + 1);
                string nextObject = strRealObj.ToUpper();
                switch (curState)
                {   
                    #region BEGIN
                    case SyntaxObjectType.BEGIN:
                        //we have found a search-object
                        if (!mSearchObjectAC.validate(nextObject))
                        {
                            if (!mSearchObjectAC.validateCompletion(nextObject))
                            {
                                retval.setErr(SyntaxError.INVALID_SEARCH_OBJECT, curStartPos, idx - curStartPos + 1);
                                return retval;
                            }
                        }
                        else
                        {
                            if (searchCharArr.Length >= idx + 2) // Check that this maybe a plural
                            {
                                // Validate that the next character is an 's'
                                if (mPluralAC.validate(searchText.Substring(idx + 1, 1)))
                                {
                                    // Then just move things along.
                                    idx++;
                                    StringBuilder sb = new StringBuilder(nextObject);
                                    sb.Append('S');
                                    nextObject = sb.ToString();
                                }
                            }
                            retval.addSyntaxObject(SyntaxObjectType.SEARCH_OBJECT, nextObject, curStartPos, idx + 1);
                            retval.valid = true;
                            curStartPos = idx + 1;
                        }     
                        break;
                    #endregion
                    #region SEARCH_OBJECT
                    case SyntaxObjectType.SEARCH_OBJECT:
                        
                        if (!mColonAC.validate(nextObject))
                        {
                            if (!mColonAC.validateCompletion(nextObject))
                            {
                                retval.setErr(SyntaxError.COLON_NOT_NEXT_TO_SEARCH_OBJECT, curStartPos, idx + 1);
                                return retval;
                            }
                        }
                        else
                        {
                            retval.addSyntaxObject(SyntaxObjectType.COLON, nextObject, idx, idx + 1);
                            curStartPos = idx + 1;
                            retval.valid = true;
                        }
                        break;
                    #endregion
                    #region CROSS_REF_OBJ
                    case SyntaxObjectType.CROSS_REF_OBJ:
                        string curRefObj = retval.getPreviousSyntaxObject(0, SyntaxObjectType.CROSS_REF_OBJ);
                        curConditionRelationAC = mSearchObjectAC.getObjectRelationshipAutoCompleter(curRefObj);
                        if (idx + 1 < searchCharArr.Length)
                        {
                            tryNextObj = searchText.Substring(curStartPos, idx - curStartPos + 2).ToUpper();
                        }
                        if (curConditionRelationAC == null)
                        {
                            retval.setErr(SyntaxError.CONDITION_CANT_CREATE_RRELATIONS_AC, curStartPos, idx + 1);
                            return retval;
                        }
                        if (mDotAC.validate(nextObject))
                        {
                            retval.addSyntaxObject(SyntaxObjectType.DOT, nextObject, curStartPos, idx + 1);
                            curStartPos = idx + 1;
                        }
                        else if ((tryNextObj != string.Empty) && (curConditionRelationAC.validate(tryNextObj)))
                        {
                            break; //i.e. the relation object has another charecter
                        }
                        else if (curConditionRelationAC.validate(nextObject))
                        {
                            retval.addSyntaxObject(SyntaxObjectType.CONDITION_RELATION, nextObject, curStartPos, idx + 1);
                            curStartPos = idx + 1;

                        }
                        else if ((!curConditionRelationAC.validateCompletion(nextObject)) &&
                            (!mDotAC.validateCompletion(nextObject)))
                        {
                            retval.setErr(SyntaxError.INVALID_POST_CROSS_REF_OBJ, curStartPos, idx + 1);
                            return retval;
                        }
                        tryNextObj = string.Empty;
                        break;
                    #endregion
                    #region DOT
                    case SyntaxObjectType.DOT:
                        curRefObj = retval.getPreviousSyntaxObject(1, SyntaxObjectType.CROSS_REF_OBJ);
                        curConditionFieldAC = mSearchObjectAC.getFieldAutoCompleter(curRefObj);
                        if (curConditionFieldAC == null)
                        {

                            retval.setErr(SyntaxError.CANT_GET_CONDITION_FIELD_AC, curStartPos, idx);
                            return retval;
                        }
                        if (!curConditionFieldAC.validate(nextObject))
                        {
                            if (!curConditionFieldAC.validateCompletion(nextObject))
                            {
                                retval.setErr(SyntaxError.INVALID_CONDITION_FILED, curStartPos, idx + 1);
                                return retval;
                            }
                        }
                        else
                        {
                            retval.addSyntaxObject(SyntaxObjectType.CONDITION_FIELD, nextObject, curStartPos, idx + 1);
                            curStartPos = idx + 1;
                        }
                        break;
                    #endregion
                    #region OR AND
                    case SyntaxObjectType.AND:
                    case SyntaxObjectType.OR:
                        keepValid = false;
                        curConditionFieldAC = mSearchObjectAC.getFieldAutoCompleter(retval.getSearchObjectStr());
                        if (curConditionFieldAC.validate(nextObject))
                        {
                            retval.addSyntaxObject(SyntaxObjectType.CONDITION_FIELD, nextObject, curStartPos, idx + 1);
                            curStartPos = idx + 1;

                        }
                        else if (mSearchObjectAC.isCrossReferece(nextObject, retval.First.Value.Body))
                        {
                            if (searchCharArr.Length >= idx + 2) // Check that this maybe a plural
                            {
                                // Validate that the next character is an 's'
                                if (mPluralAC.validate(searchText.Substring(idx + 1, 1)))
                                {
                                    // Then just move things along.
                                    idx++;
                                    StringBuilder sb = new StringBuilder(nextObject);
                                    sb.Append('S');
                                    nextObject = sb.ToString();
                                }
                            }
                            retval.addSyntaxObject(SyntaxObjectType.CROSS_REF_OBJ, nextObject, curStartPos, idx + 1);
                            curStartPos = idx + 1;
                        }
                        else
                        {
                            ValueParseResult ans = handleValuePhrase(final, searchText, idx, ref curStartPos, retval);
                            if (ans != ValueParseResult.Err)
                            {
                                if (ans == ValueParseResult.FreeText)
                                {
                                    curRefObj = retval.getSearchObjectStr();
                                    if (freeTextObjSearched.Contains(curRefObj))
                                    {
                                        retval.setErr(SyntaxError.FREE_TEXT_ALLOWED_ONCE_PER_OBJ, curStartPos, idx + 1);
                                        return retval;
                                    }
                                    freeTextObjSearched.Add(curRefObj);
                                    retval.valid = true;
                                    keepValid = true;
                                }
                            }
                            else if ((!curConditionFieldAC.validateCompletion(nextObject)) &&
                                (!mSearchObjectAC.validateCompletion(nextObject)))
                            {
                                retval.setErr(SyntaxError.INVALID_POST_OR_AND_PHRASE, curStartPos, idx + 1);
                                return retval;
                            }
                        }
                        if (keepValid == false)
                        {
                            retval.valid = false;
                        }
                        break;
                    #endregion
                    #region COLON
                    case SyntaxObjectType.COLON:
                        keepValid = false;
                        curConditionFieldAC = mSearchObjectAC.getFieldAutoCompleter(retval.getSearchObjectStr());
                        if (curConditionFieldAC.validate(nextObject))
                        {
                            retval.addSyntaxObject(SyntaxObjectType.CONDITION_FIELD, nextObject, curStartPos, idx + 1);
                            curStartPos = idx + 1;

                        }
                        else if (mSortbyAC.validate(nextObject))
                        {
                            retval.addSyntaxObject(SyntaxObjectType.SORTBY, nextObject, curStartPos, idx + 1);
                            curStartPos = idx + 1;
                        }
                        else if (mPageAC.validate(nextObject))
                        {
                            retval.addSyntaxObject(SyntaxObjectType.PAGE, nextObject, curStartPos, idx + 1);
                            curStartPos = idx + 1;
                        }
                        else if (mSearchObjectAC.isCrossReferece(nextObject, retval.First.Value.Body))
                        {
                            if (searchCharArr.Length >= idx + 2) // Check that this maybe a plural
                            {
                                // Validate that the next character is an 's'
                                if (mPluralAC.validate(searchText.Substring(idx + 1, 1)))
                                {
                                    // Then just move things along.
                                    idx++;
                                    StringBuilder sb = new StringBuilder(nextObject);
                                    sb.Append('S');
                                    nextObject = sb.ToString();
                                }
                            }
                            retval.addSyntaxObject(SyntaxObjectType.CROSS_REF_OBJ, nextObject, curStartPos, idx + 1);
                            curStartPos = idx + 1;
                        }
                        else
                        {
                            ValueParseResult ans = handleValuePhrase(final, searchText, idx, ref curStartPos, retval);
                            if (ans != ValueParseResult.Err)
                            {
                                if (ans == ValueParseResult.FreeText)
                                {
                                    freeTextObjSearched.Add(retval.getSearchObjectStr());
                                }
                                keepValid = true;
                            }
                            else if ((!curConditionFieldAC.validateCompletion(nextObject)) &&
                           (!mSortbyAC.validateCompletion(nextObject)) &&
                           (!mSearchObjectAC.validateCompletion(nextObject)))
                            {
                                retval.setErr(SyntaxError.INVALID_POST_COLON_PHRASE, curStartPos, idx + 1);
                                return retval;
                            }
                        }
                        if (keepValid == false)
                        {
                            retval.valid = false;
                        }
                        break;
                    #endregion
                    #region CONDITION_VALUE
                    case SyntaxObjectType.CONDITION_VALUE:
                        nextObject = nextObject.Trim();
                        if (nextObject.Length > 0)
                        {
                            keepValid = false;
                            curRefObj = retval.getSearchObjectStr();
                            curConditionFieldAC = mSearchObjectAC.getFieldAutoCompleter(curRefObj);
                            if (curConditionFieldAC.validate(nextObject))
                            {
                                retval.addSyntaxObject(SyntaxObjectType.CONDITION_FIELD, nextObject, curStartPos, idx + 1);
                                curStartPos = idx + 1;

                            }
                            else if (mSortbyAC.validate(nextObject))
                            {
                                retval.addSyntaxObject(SyntaxObjectType.SORTBY, nextObject, curStartPos, idx + 1);
                                curStartPos = idx + 1;
                            }
                            else if (mPageAC.validate(nextObject))
                            {
                                retval.addSyntaxObject(SyntaxObjectType.PAGE, nextObject, curStartPos, idx + 1);
                                curStartPos = idx + 1;
                            }
                            else if (mSearchObjectAC.isCrossReferece(nextObject, retval.First.Value.Body))
                            {
                                if (searchCharArr.Length >= idx + 2) // Check that this maybe a plural
                                {
                                    // Validate that the next character is an 's'
                                    if (mPluralAC.validate(searchText.Substring(idx + 1, 1)))
                                    {
                                        // Then just move things along.
                                        idx++;
                                        StringBuilder sb = new StringBuilder(nextObject);
                                        sb.Append('S');
                                        nextObject = sb.ToString();
                                    }
                                }
                                retval.addSyntaxObject(SyntaxObjectType.CROSS_REF_OBJ, nextObject, curStartPos, idx + 1);
                                curStartPos = idx + 1;
                            }
                            else if (mAndAC.validate(nextObject))
                            {
                                retval.addSyntaxObject(SyntaxObjectType.AND, nextObject, curStartPos, idx + 1);
                                curStartPos = idx + 1;
                            }
                            else if (mOrAC.validate(nextObject))
                            {
                                retval.addSyntaxObject(SyntaxObjectType.OR, nextObject, curStartPos, idx + 1);
                                curStartPos = idx + 1;
                            }


                            else if ((!curConditionFieldAC.validateCompletion(nextObject)) &&
                                (!mSortbyAC.validateCompletion(nextObject)) &&
                                (!mSearchObjectAC.validateCompletion(nextObject)) &&
                                (!mAndAC.validateCompletion(nextObject)) &&
                                (!mOrAC.validateCompletion(nextObject)))
                            {
                                ValueParseResult ans = handleValuePhrase(final, searchText, idx, ref curStartPos, retval);
                                if (ans != ValueParseResult.Err)
                                {
                                    if (ans == ValueParseResult.FreeText)
                                    {
                                        if (freeTextObjSearched.Contains(curRefObj))
                                        {
                                            retval.setErr(SyntaxError.FREE_TEXT_ALLOWED_ONCE_PER_OBJ, curStartPos, idx + 1);
                                            return retval;
                                        }
                                        freeTextObjSearched.Add(curRefObj);
                                        retval.valid = true;
                                        keepValid = true;
                                    }
                                }
                                else
                                {
                                    retval.setErr(SyntaxError.INVALID_POST_CONDITION_VALUE_PHRASE, curStartPos, idx + 1);
                                    return retval;
                                }
                            }
                            if (keepValid == false)
                            {
                                retval.valid = false;
                            }
                        }
                        break;
                    #endregion
                    #region CONDITION_FIELD
                    case SyntaxObjectType.CONDITION_FIELD:
                        curRefObj = retval.getPreviousSyntaxObject(2, SyntaxObjectType.CROSS_REF_OBJ);
                        string curConditionField = retval.getPreviousSyntaxObject(0, SyntaxObjectType.CONDITION_FIELD);
                        curConditionRelationAC = mSearchObjectAC.getFieldRelationshipAutoCompleter(curRefObj, curConditionField);
                        if (curConditionRelationAC == null)
                        {
                            retval.setErr(SyntaxError.CONDITION_CANT_CREATE_RRELATIONS_AC, curStartPos, idx + 1);
                            return retval;
                        }
                        if (idx + 1 < searchCharArr.Length)
                        {
                            tryNextObj = searchText.Substring(curStartPos, idx - curStartPos + 2).ToUpper();
                            if (curConditionRelationAC.validate(tryNextObj))
                            {
                                break;
                            }
                        }
                        if (!curConditionRelationAC.validate(nextObject))
                        {
                            if (!curConditionRelationAC.validateCompletion(nextObject))
                            {
                                retval.setErr(SyntaxError.INVALID_CONDITION_RELATION, curStartPos, idx + 1);
                                return retval;
                            }  
                        }
                        else
                        {
                            retval.addSyntaxObject(SyntaxObjectType.CONDITION_RELATION, nextObject, curStartPos, idx + 1);
                        }
                        curStartPos = idx + 1;
                        retval.valid = false;
                        tryNextObj = string.Empty;

                        break;
                    #endregion
                    #region CONDITION_RELATION
                    case SyntaxObjectType.CONDITION_RELATION:
                        {
                            ValueParseResult ans = handleValuePhrase(final, searchText, idx, ref curStartPos, retval);
                            if (ans == ValueParseResult.Err)
                            {
                                return retval;
                            }
                            if (ans == ValueParseResult.FreeText)
                            {
                                if (retval.getPreviousSyntaxObjectType(2) == SyntaxObjectType.CROSS_REF_OBJ)
                                {
                                    curRefObj =
                                        retval.getObjSingularName(retval.getPreviousSyntaxObject(2,
                                                                                                 SyntaxObjectType.
                                                                                                     CROSS_REF_OBJ));
                                    if (freeTextObjSearched.Contains(curRefObj))
                                    {
                                        retval.setErr(SyntaxError.FREE_TEXT_ALLOWED_ONCE_PER_OBJ, curStartPos, idx + 1);
                                        return retval;
                                    }
                                    freeTextObjSearched.Add(curRefObj);
                                }
                            }
                        }
                        break;
                    #endregion
                    #region SORTBY
                    case SyntaxObjectType.SORTBY:
                        curConditionFieldAC = mSearchObjectAC.getFieldAutoCompleter(retval.getSearchObjectStr());
                        if (!curConditionFieldAC.validate(nextObject))
                        {
                            if (!curConditionFieldAC.validateCompletion(nextObject))
                            {
                                retval.setErr(SyntaxError.INVALID_SORT_FIELD, curStartPos, idx + 1);
                                return retval;
                            }
                        }
                        else
                        {
                            retval.addSyntaxObject(SyntaxObjectType.SORT_FIELD, nextObject, curStartPos, idx + 1);
                            curStartPos = idx + 1;
                            retval.valid = true; 
                        }
                        break;
                    #endregion
                    #region PAGE
                    case SyntaxObjectType.PAGE:
                        int pageNumber;
                        if(!int.TryParse(nextObject,out pageNumber))
                        {
                            retval.setErr(SyntaxError.INVALID_CHARECTER, curStartPos, idx + 1);
                            return retval;
                        }
                        else
                        {
                            string s = string.Empty;
                            int pos=idx;
                            // parsing the whole page number (can be more than one char)
                            while (pos < searchText.Length - 1 && Char.IsNumber(nextObject,0))
                            {
                                s += nextObject;
                                pos++;
                                strRealObj = searchText.Substring(pos, 1);
                                nextObject = strRealObj.ToUpper();                             
                            }
                            s += nextObject;
                            retval.addSyntaxObject(SyntaxObjectType.PAGE_VALUE, s, curStartPos, idx + s.Length);
                            // update index position
                            idx = pos+1;
                            retval.valid = true;
                        }
                        break;
                    #endregion
                    #region SORT_FIELD
                    case SyntaxObjectType.SORT_FIELD:
                        if (! mSortDirectionAC.validate(nextObject))
                        {
                            if (!mSortDirectionAC.validateCompletion(nextObject))
                            {
                                retval.setErr(SyntaxError.INVALID_SORT_DIRECTION, curStartPos, idx + 1);
                                return retval;
                            }
                        }
                        else
                        {
                            retval.addSyntaxObject(SyntaxObjectType.SORT_DIRECTION, nextObject, curStartPos, idx + 1);
                            curStartPos = idx + 1;
                            retval.valid = true;
                        }
                        break;
                    #endregion
                    #region PAGE_VALUE
                    case SyntaxObjectType.PAGE_VALUE:
                        if (curChar != ' ')
                        {
                            retval.setErr(SyntaxError.NOTHING_COMES_AFTER_PAGE_VALUE, curStartPos, idx + 1);
                                return retval;
                        }
                        break;
                    #endregion
                    #region SORT_DIRECTION
                    case SyntaxObjectType.SORT_DIRECTION:
                        if (!mPageAC.validate(nextObject))
                        {
                            if(!mPageAC.validateCompletion(nextObject))
                            {
                                retval.setErr(SyntaxError.INVALID_PAGE_FEILD, curStartPos, idx);
                                return retval;
                            }
                        }
                        else
                        {
                            retval.addSyntaxObject(SyntaxObjectType.PAGE, nextObject, curStartPos, idx + 1);
                            curStartPos = idx + 1;
                            retval.valid = true;
                        }
                        break;
                    #endregion
                    default:
                        retval.setErr(SyntaxError.UNIDENTIFIED_STATE, curStartPos, idx);
                        return retval;
                }
            }
            return retval;
            
        }

        public SyntaxContainer getCompletion(string searchText)
        {
            SyntaxContainer retval = analyzeSyntaxState(searchText, false);
            if (retval.Error == SyntaxError.NO_ERROR)
            {
                IConditionFieldAutoCompleter conditionFieldAC;
                IAutoCompleter conditionRelationAC;
                IConditionValueAutoCompleter conditionValueAC;
                int lastIdx = retval.getLastHandledIndex();
                string curPartialWord = "";
                if (lastIdx < searchText.Length)
                {
                    curPartialWord = searchText.Substring(lastIdx, searchText.Length - lastIdx);
                    curPartialWord = curPartialWord.Trim();
                }
                SyntaxObjectType curState = retval.getState();
                for (int idx = 0; idx < mStateMap[curState].Length; idx++)
                {
                    switch (mStateMap[curState][idx])
                    {
                        case SyntaxObjectType.SEARCH_OBJECT:
                            retval.addToACList(mSearchObjectAC.getCompletion(curPartialWord));
                            break;
                        case SyntaxObjectType.CROSS_REF_OBJ:
                            IAutoCompleter crossRefAC = mSearchObjectAC.getCrossRefAutoCompleter(retval.First.Value.Body);
                            if (crossRefAC != null)
                            {
                                retval.addToACList(crossRefAC.getCompletion(curPartialWord));
                            }
                            break;
                        case SyntaxObjectType.DOT:
                            retval.addToACList(mDotAC.getCompletion(curPartialWord));
                            break;
                        case SyntaxObjectType.COLON:
                            retval.addToACList(mColonAC.getCompletion(curPartialWord));
                            break;
                        case SyntaxObjectType.AND:
                            retval.addToACList(mAndAC.getCompletion(curPartialWord));
                            break;
                        case SyntaxObjectType.OR:
                            retval.addToACList(mOrAC.getCompletion(curPartialWord));
                            break;
                        case SyntaxObjectType.CONDITION_FIELD:
                            string relObj = retval.getPreviousSyntaxObject(1, SyntaxObjectType.CROSS_REF_OBJ);
                            conditionFieldAC = mSearchObjectAC.getFieldAutoCompleter(relObj);
                            if (conditionFieldAC != null)
                            {
                                retval.addToACList(conditionFieldAC.getCompletion(curPartialWord));
                            }
                            break;
                        case SyntaxObjectType.CONDITION_RELATION:
                            {
                                if (curState == SyntaxObjectType.CONDITION_FIELD)
                                {
                                    relObj = retval.getPreviousSyntaxObject(2, SyntaxObjectType.CROSS_REF_OBJ);
                                    string fldName = retval.getPreviousSyntaxObject(0, SyntaxObjectType.CONDITION_FIELD);
                                    conditionRelationAC = mSearchObjectAC.getFieldRelationshipAutoCompleter(relObj,
                                                                                                            fldName);
                                }
                                else // curState == SyntaxObjectType.CROSS_REF_OBJ
                                {
                                    relObj = retval.getPreviousSyntaxObject(0, SyntaxObjectType.CROSS_REF_OBJ);
                                    conditionRelationAC = mSearchObjectAC.getObjectRelationshipAutoCompleter(relObj);

                                }
                                if (conditionRelationAC != null)
                                {
                                    retval.addToACList(conditionRelationAC.getCompletion(curPartialWord));
                                }
                            }
                            break;
                        case SyntaxObjectType.CONDITION_VALUE:
                            {
                                relObj = retval.getPreviousSyntaxObject(3, SyntaxObjectType.CROSS_REF_OBJ);
                                string fldName = retval.getPreviousSyntaxObject(1, SyntaxObjectType.CONDITION_FIELD);
                                conditionValueAC = mSearchObjectAC.getFieldValueAutoCompleter(relObj, fldName);
                                if (conditionValueAC != null)
                                {
                                    retval.addToACList(conditionValueAC.getCompletion(curPartialWord));
                                }
                            }
                            break;
                        case SyntaxObjectType.SORTBY:
                            retval.addToACList(mSortbyAC.getCompletion(curPartialWord));
                            break;
                        case SyntaxObjectType.PAGE:
                            retval.addToACList(mPageAC.getCompletion(curPartialWord));
                            break;
                        case SyntaxObjectType.SORT_FIELD:
                            conditionFieldAC = mSearchObjectAC.getFieldAutoCompleter(retval.getSearchObjectStr());
                            if (conditionFieldAC != null)
                            {
                                retval.addToACList(conditionFieldAC.getCompletion(curPartialWord));
                            }
                            break;
                        case SyntaxObjectType.SORT_DIRECTION:
                            retval.addToACList(mSortDirectionAC.getCompletion(curPartialWord));
                            break;
                    }
                }
            }
            return retval;
        }

        //public string generateQueryFromSyntaxContainer(SyntaxContainer syntax)
        //{
        //    string retval = string.Empty;
        //    if (syntax.valid)
        //    {
        //        retval = generateSqlFromSyntaxContainer(syntax);        
        //    }
        //    return retval;
        //}


        private string generateFromStatement(SyntaxContainer syntax)
        {
            LinkedList<string> innerJoins = new LinkedList<string>();
            List<string> refObjList = syntax.getCrossRefObjList();
            string searchObjStr = syntax.getSearchObjectStr();
            if (refObjList.Count > 0)
            {
                switch (searchObjStr)
                {
                    case SearchObjects.TEMPLATE_OBJ_NAME:
                        innerJoins.AddFirst(mSearchObjectAC.getInnerJoin(SearchObjects.TEMPLATE_OBJ_NAME, SearchObjects.VM_OBJ_NAME));
                        if (refObjList.Contains(SearchObjects.VM_OBJ_NAME))
                        {
                            refObjList.Remove(SearchObjects.VM_OBJ_NAME);
                        }
                        if (refObjList.Contains(SearchObjects.VDC_USER_OBJ_NAME))
                        {
                            innerJoins.AddLast(mSearchObjectAC.getInnerJoin(SearchObjects.VM_OBJ_NAME, SearchObjects.VDC_USER_OBJ_NAME)); 
                            refObjList.Remove(SearchObjects.VDC_USER_OBJ_NAME);
                        }
                        if (refObjList.Contains(SearchObjects.VDS_OBJ_NAME))
                        {
                            innerJoins.AddLast(mSearchObjectAC.getInnerJoin(SearchObjects.VM_OBJ_NAME, SearchObjects.VDS_OBJ_NAME));
                            refObjList.Remove(SearchObjects.VDS_OBJ_NAME);
                        }
                        if (refObjList.Contains(SearchObjects.AUDIT_OBJ_NAME))
                        {
                            innerJoins.AddLast(mSearchObjectAC.getInnerJoin(SearchObjects.VM_OBJ_NAME, SearchObjects.AUDIT_OBJ_NAME));
                            refObjList.Remove(SearchObjects.AUDIT_OBJ_NAME);
                        }
                        break;
                    case SearchObjects.VDS_OBJ_NAME:
                        if ((refObjList.Contains(SearchObjects.VDC_USER_OBJ_NAME)) || (refObjList.Contains(SearchObjects.TEMPLATE_OBJ_NAME)))
                        {
                            innerJoins.AddFirst(mSearchObjectAC.getInnerJoin(SearchObjects.VDS_OBJ_NAME, SearchObjects.VM_OBJ_NAME));
                            if (refObjList.Contains(SearchObjects.VM_OBJ_NAME))
                            {
                                refObjList.Remove(SearchObjects.VM_OBJ_NAME);
                            }
                        }
                        if (refObjList.Contains(SearchObjects.VDC_USER_OBJ_NAME))
                        {
                            innerJoins.AddLast(mSearchObjectAC.getInnerJoin(SearchObjects.VM_OBJ_NAME, SearchObjects.VDC_USER_OBJ_NAME));
                            refObjList.Remove(SearchObjects.VDC_USER_OBJ_NAME);
                        }
                        if (refObjList.Contains(SearchObjects.TEMPLATE_OBJ_NAME))
                        {
                            innerJoins.AddLast(mSearchObjectAC.getInnerJoin(SearchObjects.VM_OBJ_NAME, SearchObjects.TEMPLATE_OBJ_NAME));
                            refObjList.Remove(SearchObjects.TEMPLATE_OBJ_NAME);
                        }
                        break;
                    case SearchObjects.VDC_USER_OBJ_NAME:
                        if ((refObjList.Contains(SearchObjects.VDS_OBJ_NAME)) || (refObjList.Contains(SearchObjects.TEMPLATE_OBJ_NAME)))
                        {
                            innerJoins.AddFirst(mSearchObjectAC.getInnerJoin(SearchObjects.VDC_USER_OBJ_NAME, SearchObjects.VM_OBJ_NAME));
                            if (refObjList.Contains(SearchObjects.VM_OBJ_NAME))
                            {
                                refObjList.Remove(SearchObjects.VM_OBJ_NAME);
                            }
                        }
                        if (refObjList.Contains(SearchObjects.VDS_OBJ_NAME))
                        {
                            innerJoins.AddLast(mSearchObjectAC.getInnerJoin(SearchObjects.VM_OBJ_NAME, SearchObjects.VDS_OBJ_NAME));
                            refObjList.Remove(SearchObjects.VDS_OBJ_NAME);
                        }
                        if (refObjList.Contains(SearchObjects.TEMPLATE_OBJ_NAME))
                        {
                            innerJoins.AddLast(mSearchObjectAC.getInnerJoin(SearchObjects.VM_OBJ_NAME, SearchObjects.TEMPLATE_OBJ_NAME));
                            refObjList.Remove(SearchObjects.TEMPLATE_OBJ_NAME);
                        }
                        break;
                    case SearchObjects.AUDIT_OBJ_NAME:
                        if (refObjList.Contains(SearchObjects.TEMPLATE_OBJ_NAME))
                        {
                            innerJoins.AddFirst(mSearchObjectAC.getInnerJoin(SearchObjects.AUDIT_OBJ_NAME, SearchObjects.VM_OBJ_NAME));
                            innerJoins.AddLast(mSearchObjectAC.getInnerJoin(SearchObjects.VM_OBJ_NAME, SearchObjects.TEMPLATE_OBJ_NAME));
                            refObjList.Remove(SearchObjects.TEMPLATE_OBJ_NAME);
                            if (refObjList.Contains(SearchObjects.VM_OBJ_NAME))
                            {
                                refObjList.Remove(SearchObjects.VM_OBJ_NAME);
                            }
                        }
                       
                        break;
                }
            }
            foreach (string cro in refObjList)
            {
                innerJoins.AddLast(mSearchObjectAC.getInnerJoin(searchObjStr, cro));
            }
            innerJoins.AddFirst(mSearchObjectAC.getRelatedTableName(searchObjStr));
            LinkedListNode<string> part = innerJoins.First;
            StringBuilder sb = new StringBuilder();
            while (part != null)
            {
                sb.Append(" ");
                sb.Append(part.Value);
                sb.Append(" ");
                part = part.Next;
            }
            return sb.ToString();

        }


        //private string generateSqlFromSyntaxContainer(SyntaxContainer syntax)
        //{
        //    string retval = string.Empty;
        //    if (syntax.valid)
        //    {
        //        LinkedListNode<SyntaxObject> curObj = syntax.First;
        //        IConditionFieldAutoCompleter conditionFieldAC;
        //        LinkedList<string> queryBuilder = new LinkedList<string>();
        //        LinkedList<string> whereBuilder = new LinkedList<string>();
        //        LinkedListNode<string> queryPart;
        //        string searchObjStr = syntax.getSearchObjectStr();
        //        string sortByPhrase = string.Empty;
        //        string fromStatement = string.Empty;
        //        string pageNumber = string.Empty;

        //        while (curObj != null)
        //        {
        //            switch (curObj.Value.Type)
        //            {
        //                case SyntaxObjectType.SEARCH_OBJECT:
        //                    fromStatement = generateFromStatement(syntax);
        //                    break;  
        //                case SyntaxObjectType.OR:
        //                case SyntaxObjectType.AND:
        //                    whereBuilder.AddLast(new LinkedListNode<string>(curObj.Value.Body));
        //                    break;
        //                case SyntaxObjectType.CONDITION_VALUE:
        //                    whereBuilder.AddLast(generateConditionStatment(curObj, searchObjStr));
        //                    break;
        //                case SyntaxObjectType.SORTBY:
        //                    break;
        //                case SyntaxObjectType.PAGE_VALUE:
        //                    pageNumber = curObj.Value.Body;
        //                    break;
        //                case SyntaxObjectType.SORT_FIELD:
        //                    conditionFieldAC = mSearchObjectAC.getFieldAutoCompleter(searchObjStr);
        //                    sortByPhrase = string.Format(" ORDER BY {0}", conditionFieldAC.getDbFieldName(curObj.Value.Body));
        //                    break;
        //                case SyntaxObjectType.SORT_DIRECTION:
        //                    sortByPhrase = string.Format("{0} {1}", sortByPhrase, curObj.Value.Body);
        //                    break;
        //                default:
        //                    break;
        //            }

        //            curObj = curObj.Next;
        //        }
                
        //        //implying precedence rules
        //        string[] lookFor = { "AND" , "OR" };
        //        for (int idx = 0; idx < lookFor.Length; idx++)
        //        {
        //            bool found = true;
        //            while (found)
        //            {
        //                found = false;
        //                queryPart = whereBuilder.First;
        //                while (queryPart != null)
        //                {
        //                    if (queryPart.Value == lookFor[idx])
        //                    {
        //                        queryPart.Previous.Value = string.Format("( {0} {1} {2} )",
        //                            queryPart.Previous.Value, queryPart.Value, queryPart.Next.Value);
        //                        whereBuilder.Remove(queryPart.Next);
        //                        whereBuilder.Remove(queryPart);
        //                        found = true;
        //                        break;
        //                    }
        //                    queryPart = queryPart.Next;
        //                }
        //            }
        //        }
        //        // adding WHERE if required and All implicit AND
        //        StringBuilder wherePhrase = new StringBuilder();
        //        if (whereBuilder.Count > 0)
        //        {
        //            wherePhrase.Append(" WHERE ");
        //            queryPart = whereBuilder.First;
        //            while (queryPart != null)
        //            {
        //                wherePhrase.Append(queryPart.Value);
        //                if (queryPart.Next != null)
        //                {
        //                    wherePhrase.Append(" AND ");
        //                }
        //                queryPart = queryPart.Next;
        //            }
        //        }

        //        //adding the sorting part if required
        //        if (sortByPhrase == string.Empty)
        //        {
        //            sortByPhrase = string.Format(" ORDER BY {0}",  mSearchObjectAC.getDefaultSort(searchObjStr));                    
        //        }
        //        //adding the paging phrase
        //        string pagePhrase = getPagePhrase(syntax, pageNumber);

        //        string primeryKey = mSearchObjectAC.getPrimeryKeyName(searchObjStr);
        //        string tableName = mSearchObjectAC.getRelatedTableName(searchObjStr);
        //        string tableNameWithOutTags = mSearchObjectAC.getRelatedTableNameWithOutTags(searchObjStr);

        //        string innerQuery = string.Format("SELECT {0}.{1} FROM {2} {3}", tableName, primeryKey, fromStatement, wherePhrase);
        //        string inQuery = string.Format("SELECT * FROM {0} WHERE ( {1} IN ({2})", tableNameWithOutTags, primeryKey, innerQuery);
        //        retval = string.Format(SyntaxCheckerFactory.ConfigDBSearchTemplate, sortByPhrase, inQuery, pagePhrase);
        //        QLogger.Instance.DebugFormat("Search: {0}", retval); 
        //    }
        //    return retval;
        //}

        //private string getPagePhrase(SyntaxContainer syntax, string pageNumber)
        //{
        //    string result = string.Empty;
        //    int page;
        //    if (!int.TryParse(pageNumber, out page))
        //    {
        //        page = 1;
        //    }
        //    string pagingTypeStr = SyntaxCheckerFactory.ConfigDBPagingType;
        //    if (Enum.IsDefined(typeof(PagingType), pagingTypeStr))
        //    {
        //        PagingType pagingType = (PagingType)Enum.Parse(typeof(PagingType), pagingTypeStr, true);
        //        string pagingSyntax = SyntaxCheckerFactory.ConfigDBPagingSyntax;
        //        switch (pagingType)
        //        {
        //            case PagingType.Range:
        //                result = string.Format(pagingSyntax, (page - 1) * syntax.MaxCount + 1, page * syntax.MaxCount);   
        //                break;
        //            case PagingType.Offset:
        //                result = string.Format(pagingSyntax, (page - 1) * syntax.MaxCount + 1, syntax.MaxCount);   
        //                break;
        //        }                
        //    }
        //    else
        //    {
        //        QLogger.Instance.Error(string.Format("Unknown paging type {0}", pagingTypeStr));
        //    }

        //    return result;
            
        //}

        private enum ConditionType
        {
            None,
            FreeText,
            FreeTextSpecificObj,
            ConditionWithDefaultObj,
            ConditionwithSpesificObj
        }
        private string generateConditionStatment(LinkedListNode<SyntaxObject> curObj, string searchObjStr)
        {
            IConditionFieldAutoCompleter conditionFieldAC;
            IConditionValueAutoCompleter conditionValueAC = null;
            string customizedValue = string.Format("'{0}'", curObj.Value.Body);
            string customizedRelation;
            string fieldName = string.Empty;
            string objName;
            ConditionType conditionType; 
            if (curObj.Previous.Value.Type != SyntaxObjectType.CONDITION_RELATION)
            {
                //free text of default search object
                customizedRelation = "=";
                objName = searchObjStr;
                conditionFieldAC = mSearchObjectAC.getFieldAutoCompleter(searchObjStr);
                conditionType = ConditionType.FreeText;
            }
            else
            {
                customizedRelation = curObj.Previous.Value.Body;
                if (curObj.Previous.Previous.Value.Type == SyntaxObjectType.CROSS_REF_OBJ)
                {   //free text search for some object
                    objName = curObj.Previous.Previous.Value.Body;
                    conditionFieldAC = mSearchObjectAC.getFieldAutoCompleter(objName);
                    conditionType = ConditionType.FreeTextSpecificObj;
                }
                else //if (curObj.Previous.Previous.Value.Type == SyntaxObjectType.CONDITION_FIELD) 
                {
                    fieldName = curObj.Previous.Previous.Value.Body;
                    if (curObj.Previous.Previous.Previous.Value.Type != SyntaxObjectType.DOT)
                    {
                        //standard condition with default AC (search obj)
                        objName = searchObjStr;
                        conditionFieldAC = mSearchObjectAC.getFieldAutoCompleter(searchObjStr);
                        conditionType = ConditionType.ConditionWithDefaultObj;
                    }
                    else
                    {
                        //standard condition with specific AC
                        objName = curObj.Previous.Previous.Previous.Previous.Value.Body;
                        conditionFieldAC = mSearchObjectAC.getFieldAutoCompleter(objName);
                        conditionType = ConditionType.ConditionwithSpesificObj;
                    }
                }
                conditionValueAC = conditionFieldAC.getFieldValueAutoCompleter(fieldName);
            }

            BaseConditionFieldAutoCompleter conditionAsBase = conditionFieldAC as BaseConditionFieldAutoCompleter;
            Type curType;
            if (conditionAsBase != null &&
                conditionAsBase.TypeDictionary.TryGetValue(fieldName, out curType) &&
                curType == typeof(string) && 
                !string.IsNullOrEmpty(customizedValue) &&
                customizedValue != "''" &&
                customizedValue != "'*'")
            {
                customizedValue = string.Format("N{0}", customizedValue);
            }

            if (conditionValueAC != null)
            {
                customizedValue = string.Format("'{0}'", conditionValueAC.convertFieldEnumValueToActualValue(curObj.Value.Body));
            }
            else if (customizedValue.Contains("*"))
            {
                customizedValue = customizedValue.Replace('*', '%');

                if (customizedRelation == "=")
                {
                    customizedRelation = "LIKE";
                }
                else if (customizedRelation == "!=")
                {
                    customizedRelation = "NOT LIKE";
                }
            }
            string condition = string.Empty;
            string tableName = mSearchObjectAC.getRelatedTableName(objName);
            switch (conditionType)
            {
                case ConditionType.FreeText:
                case ConditionType.FreeTextSpecificObj:
                    condition = conditionFieldAC.buildFreeTextConditionSql(tableName, customizedRelation,
                                                                           customizedValue);
                    break;
                case ConditionType.ConditionWithDefaultObj:
                case ConditionType.ConditionwithSpesificObj:
                    condition = conditionFieldAC.buildConditionSql(fieldName, customizedValue, customizedRelation, tableName);
                    break;
            }
            return condition;
        }
    }
}
