﻿using System;
using System.Collections.Generic;
using VdcCommon.BusinessEntities;
using VdcUtils;

namespace VdcCommon
{
    public enum EventNotificationEntity
	{
        UNKNOWN = 0,
		Host = 1,
		Vm = 2,
        Storage = 3,
        Rhevm =4
	}


    public static class VdcEventNotificationUtils
    {
        private static readonly IDictionary<EventNotificationEntity, HashSet<AuditLogType>> _eventNotificationTypeMap =
            new Dictionary<EventNotificationEntity, HashSet<AuditLogType>>();


        /// <summary>
        /// Initializes the <see cref="VdcEventNotificationUtils"/> class.
        /// </summary>
        static VdcEventNotificationUtils()
        {
            // VDC
            AddEventNotificationEntry(EventNotificationEntity.Rhevm, AuditLogType.VDC_STOP);
            // VDS
            AddEventNotificationEntry(EventNotificationEntity.Host, AuditLogType.VDS_FAILURE);
            AddEventNotificationEntry(EventNotificationEntity.Host, AuditLogType.USER_VDS_MAINTENANCE);
            AddEventNotificationEntry(EventNotificationEntity.Host, AuditLogType.USER_VDS_MAINTENANCE_MIGRATION_FAILED);
            AddEventNotificationEntry(EventNotificationEntity.Host, AuditLogType.VDS_ACTIVATE_FAILED);
            AddEventNotificationEntry(EventNotificationEntity.Host, AuditLogType.VDS_RECOVER_FAILED);
            AddEventNotificationEntry(EventNotificationEntity.Host, AuditLogType.VDS_APPROVE_FAILED);
            AddEventNotificationEntry(EventNotificationEntity.Host, AuditLogType.VDS_INSTALL_FAILED);
            //VM
            AddEventNotificationEntry(EventNotificationEntity.Vm, AuditLogType.VM_FAILURE);
            AddEventNotificationEntry(EventNotificationEntity.Vm, AuditLogType.VM_MIGRATION_START);
            AddEventNotificationEntry(EventNotificationEntity.Vm, AuditLogType.VM_MIGRATION_FAILED);
            AddEventNotificationEntry(EventNotificationEntity.Vm, AuditLogType.VM_NOT_RESPONDING);
            //IRS
            AddEventNotificationEntry(EventNotificationEntity.Storage, AuditLogType.VDS_SLOW_STORAGE_RESPONSE_TIME);
            AddEventNotificationEntry(EventNotificationEntity.Storage, AuditLogType.IRS_FAILURE);
            AddEventNotificationEntry(EventNotificationEntity.Storage, AuditLogType.IRS_DISK_SPACE_LOW);
            AddEventNotificationEntry(EventNotificationEntity.Storage, AuditLogType.IRS_DISK_SPACE_LOW_ERROR);
        }

        /// <summary>
        /// Gets all notification events.
        /// </summary>
        /// <returns></returns>
        public static IDictionary<EventNotificationEntity, HashSet<AuditLogType>> GetNotificationEvents()
        {
            return _eventNotificationTypeMap;
        }

        /// <summary>
        /// Gets notification events by type.
        /// </summary>
        /// <param name="type">The type.</param>
        /// <returns></returns>
        public static IDictionary<EventNotificationEntity, HashSet<AuditLogType>> GetNotificationEventsByType(EventNotificationEntity type)
        {
            IDictionary<EventNotificationEntity, HashSet<AuditLogType>> map = new Dictionary<EventNotificationEntity, HashSet<AuditLogType>>();
            if (_eventNotificationTypeMap.ContainsKey(type))
            {
                map.Add(type, _eventNotificationTypeMap[type]);
            }
            return map;
        }

        /// <summary>
        /// Gets the event entity.
        /// </summary>
        /// <param name="auditLogType">Type of the audit log.</param>
        /// <returns></returns>
        public static EventNotificationEntity GetEventEntity(AuditLogType auditLogType)
        {
            EventNotificationEntity entity = EventNotificationEntity.UNKNOWN;
            string prefix = string.Empty;
            try
            {
                prefix = GetPrefix(Enum.GetName(typeof(AuditLogType), auditLogType));
                entity =
                    (EventNotificationEntity)Enum.Parse(typeof(EventNotificationEntity), prefix, false);
            }
            catch (Exception e)
            {
                QLogger.Instance.WarnFormat("{0}\nGetEventEntity: Unsupported AuditLogType prefix {1}",
                                            e.Message, prefix);
            }
            return entity;
        }

        /// <summary>
        /// Gets the prefix of a string until the first UNDERLINE character.
        /// </summary>
        /// <param name="s">The s.</param>
        /// <returns></returns>
        private static string GetPrefix(string s)
        {
            const char UNDERLINE = '_';
            int i = s.IndexOf(UNDERLINE);
            if (i > 0)
            {
                s = s.Substring(0, i - 1);
            }
            return s;
        }

        /// <summary>
        /// Adds an event notification entry.
        /// </summary>
        /// <param name="entity">The entity.</param>
        /// <param name="auditLogType">Type of the audit log.</param>
        private static void AddEventNotificationEntry(EventNotificationEntity entity, AuditLogType auditLogType)
        {
            HashSet<AuditLogType> entry;
            if (!_eventNotificationTypeMap.ContainsKey(entity))
            {
                entry = _eventNotificationTypeMap[entity] = new HashSet<AuditLogType>();
            }
            else
            {
                entry = _eventNotificationTypeMap[entity];
            }
            if (!entry.Contains(auditLogType))
            {
                entry.Add(auditLogType);
            }
            _eventNotificationTypeMap[entity] = entry;
        }

    }
}
