﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;
using System.Windows.Input;
using System.Windows.Interactivity;

namespace UI.WPFClient
{
	public class SuggestBehavior : Behavior<TextBox>
	{
		#region Properties

		public static bool GetIsSuggestionOpen(DependencyObject obj)
		{
			return (bool)obj.GetValue(IsSuggestionOpenProperty);
		}

		public static void SetIsSuggestionOpen(DependencyObject obj, bool value)
		{
			obj.SetValue(IsSuggestionOpenProperty, value);
		}

		public static readonly DependencyProperty IsSuggestionOpenProperty =
			DependencyProperty.RegisterAttached("IsSuggestionOpen", typeof(bool), typeof(SuggestBehavior));


		public string SearchString
		{
			get { return (string)GetValue(SearchStringProperty); }
			set { SetValue(SearchStringProperty, value); }
		}
		public static readonly DependencyProperty SearchStringProperty =
			DependencyProperty.Register("SearchString", typeof(string), typeof(SuggestBehavior));


		public Popup Popup
		{
			get { return (Popup)GetValue(PopupProperty); }
			set { SetValue(PopupProperty, value); }
		}

		public static readonly DependencyProperty PopupProperty =
			DependencyProperty.Register("Popup", typeof(Popup), typeof(SuggestBehavior));


		public Selector Selector
		{
			get { return (Selector)GetValue(ItemsControlProperty); }
			set { SetValue(ItemsControlProperty, value); }
		}
		public static readonly DependencyProperty ItemsControlProperty =
			DependencyProperty.Register("Selector", typeof(Selector), typeof(SuggestBehavior));

		#endregion


		protected override void OnAttached()
		{
			base.OnAttached();

			AssociatedObject.GotKeyboardFocus += AssociatedObject_GotKeyboardFocus;
			AssociatedObject.LostFocus += AssociatedObject_LostFocus;
			AssociatedObject.TextChanged += AssociatedObject_TextChanged;
			AssociatedObject.PreviewKeyDown += AssociatedObject_PreviewKeyDown;
			AssociatedObject.MouseDoubleClick += AssociatedObject_MouseDoubleClick;

			Popup.PlacementTarget = AssociatedObject;
			Popup.Focusable = false;
			Popup.Opened += Popup_Opened;
			Popup.Closed += Popup_Closed;
			Popup.PreviewKeyDown += Popup_PreviewKeyDown;
			Popup.GotKeyboardFocus += Popup_GotKeyboardFocus;
			Popup.LostKeyboardFocus += Popup_LostKeyboardFocus;

			Selector.KeyDown += Selector_KeyDown;
			Selector.MouseDoubleClick += Selector_MouseDoubleClick;

			Popup.Child = Selector;
		}

		void Selector_MouseDoubleClick(object sender, MouseButtonEventArgs e)
		{
			if (e.ChangedButton == MouseButton.Left)
			{
				CommitSelectedItem();
				Selector.SelectedIndex = -1;
				AssociatedObject.Focus();
				AssociatedObject.Text = SearchString;
				AssociatedObject.CaretIndex = AssociatedObject.Text.Length;
				CloseSuggestion();
			}
		}

		void Selector_KeyDown(object sender, KeyEventArgs e)
		{
			switch (e.Key)
			{
				case Key.Enter:
					CommitSelectedItem();
					CloseSuggestion();
					Selector.SelectedIndex = -1;
					AssociatedObject.Focus();
					AssociatedObject.Text = SearchString;
					AssociatedObject.CaretIndex = AssociatedObject.Text.Length;
					break;
			}
		}

		void Popup_Closed(object sender, EventArgs e)
		{
			SetIsSuggestionOpen(AssociatedObject, false);
		}

		void Popup_Opened(object sender, EventArgs e)
		{
			SetIsSuggestionOpen(AssociatedObject, true);
		}

		void Popup_PreviewKeyDown(object sender, KeyEventArgs e)
		{
			switch (e.Key)
			{
				case Key.Escape:
					CloseSuggestion();
					Selector.SelectedIndex = -1;
					AssociatedObject.Focus();
					break;

				case Key.Up:
					if (Selector.SelectedIndex == 0)
					{
						Selector.SelectedIndex = -1;
						AssociatedObject.Focus();
					}
					break;
			}
		}

		void Popup_GotKeyboardFocus(object sender, KeyboardFocusChangedEventArgs e)
		{
			Popup.StaysOpen = false;
		}

		void Popup_LostKeyboardFocus(object sender, KeyboardFocusChangedEventArgs e)
		{
			if (AssociatedObject.IsKeyboardFocusWithin)
			{
				Selector.SelectedIndex = -1;
				OpenSuggestion();
			}
		}

		void AssociatedObject_GotKeyboardFocus(object sender, KeyboardFocusChangedEventArgs e)
		{
			if (!GetIsSuggestionOpen(AssociatedObject))
			{
				OpenSuggestion();
				SearchString = AssociatedObject.Text;
			}
		}

		void AssociatedObject_LostFocus(object sender, RoutedEventArgs e)
		{
			if (!Popup.IsKeyboardFocusWithin)
			{
				CloseSuggestion();
			}
		}

		void AssociatedObject_TextChanged(object sender, TextChangedEventArgs e)
		{
			if (!GetIsSuggestionOpen(AssociatedObject)
				&& AssociatedObject.IsKeyboardFocused
				&& keyPresOccured)
			{
				OpenSuggestion();
			}

			SearchString = AssociatedObject.Text;

			keyPresOccured = false;
		}

		private bool keyPresOccured;

		void AssociatedObject_PreviewKeyDown(object sender, KeyEventArgs e)
		{
			switch (e.Key)
			{
				case Key.Escape:
					CloseSuggestion();
					break;

				case Key.Enter:
					CommitText();
					CloseSuggestion();
					break;

				case Key.Down:
					if (!GetIsSuggestionOpen(AssociatedObject))
					{
						OpenSuggestion();
					}
					Selector.Focus();
					Selector.SelectedIndex = 0;
					break;

				default:
					keyPresOccured = true;
					break;
			}
		}

		void AssociatedObject_MouseDoubleClick(object sender, MouseButtonEventArgs e)
		{
			if (e.ChangedButton == MouseButton.Left && !GetIsSuggestionOpen(AssociatedObject))
			{
				OpenSuggestion();
			}
		}

		private void CommitText()
		{
			var be = AssociatedObject.GetBindingExpression(TextBox.TextProperty);
			if (be != null)
			{
				be.UpdateSource();
			}
		}

		private void CommitSelectedItem()
		{
			var be = Selector.GetBindingExpression(Selector.SelectedItemProperty);
			if (be != null)
			{
				be.UpdateSource();
			}
		}

		private void OpenSuggestion()
		{
			Popup.StaysOpen = true;
			Popup.IsOpen = true;
		}

		private void CloseSuggestion()
		{
			Popup.IsOpen = false;
		}
	}
}
