﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;

namespace UI.WPFClient
{
	public static class HttpUtility
	{
		public static string UrlPathEncode(string str)
		{
			if (String.IsNullOrEmpty(str))
			{
				return str;
			}

			// recurse in case there is a query string 
			int i = str.IndexOf('?');
			if (i >= 0)
				return UrlPathEncode(str.Substring(0, i)) + str.Substring(i);

			// encode DBCS characters and spaces only
			return UrlEncodeSpaces(UrlEncodeNonAscii(str, Encoding.UTF8));
		}

		//  Helper to encode the non-ASCII url characters only
		private static String UrlEncodeNonAscii(string str, Encoding e)
		{
			if (String.IsNullOrEmpty(str))
				return str;
			if (e == null)
				e = Encoding.UTF8;
			byte[] bytes = e.GetBytes(str);
			byte[] encodedBytes = UrlEncodeNonAscii(bytes, 0, bytes.Length, false /* alwaysCreateNewReturnValue */);
			return Encoding.ASCII.GetString(encodedBytes);
		}

		private static byte[] UrlEncodeNonAscii(byte[] bytes, int offset, int count, bool alwaysCreateNewReturnValue)
		{
			if (!ValidateUrlEncodingParameters(bytes, offset, count))
			{
				return null;
			}

			int cNonAscii = 0;

			// count them first
			for (int i = 0; i < count; i++)
			{
				if (IsNonAsciiByte(bytes[offset + i]))
					cNonAscii++;
			}

			// nothing to expand?
			if (!alwaysCreateNewReturnValue && cNonAscii == 0)
				return bytes;

			// expand not 'safe' characters into %XX, spaces to +s 
			byte[] expandedBytes = new byte[count + cNonAscii * 2];
			int pos = 0;

			for (int i = 0; i < count; i++)
			{
				byte b = bytes[offset + i];

				if (IsNonAsciiByte(b))
				{
					expandedBytes[pos++] = (byte)'%';
					expandedBytes[pos++] = (byte)IntToHex((b >> 4) & 0xf);
					expandedBytes[pos++] = (byte)IntToHex(b & 0x0f);
				}
				else
				{
					expandedBytes[pos++] = b;
				}
			}

			return expandedBytes;
		}

		private static bool ValidateUrlEncodingParameters(byte[] bytes, int offset, int count)
		{
			if (bytes == null && count == 0)
				return false;
			if (bytes == null)
			{
				throw new ArgumentNullException("bytes");
			}
			if (offset < 0 || offset > bytes.Length)
			{
				throw new ArgumentOutOfRangeException("offset");
			}
			if (count < 0 || offset + count > bytes.Length)
			{
				throw new ArgumentOutOfRangeException("count");
			}

			return true;
		}

		private static bool IsNonAsciiByte(byte b)
		{
			return (b >= 0x7F || b < 0x20);
		}

		private static char IntToHex(int n)
		{
			Debug.Assert(n < 0x10);

			if (n <= 9)
				return (char)(n + (int)'0');
			else
				return (char)(n - 10 + (int)'a');
		}

		private static String UrlEncodeSpaces(string str)
		{
			if (str != null && str.IndexOf(' ') >= 0)
				str = str.Replace(" ", "%20");
			return str;
		}
	}
}
