using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Text;
using System.Collections.ObjectModel;
using System.Collections.Specialized;

using VdcCommon.BusinessEntities;
using VdcFrontend;
using org.ovirt.engine.ui.uicompat;

namespace org.ovirt.engine.ui.uicommon.models.vms
{
	public class VmGuideModel : GuideModel
	{
		#region Action Constants
		public readonly string VmConfigureNetworkInterfacesAction = "Configure Network Interfaces";
		public readonly string VmAddAnotherNetworkInterfaceAction = "Add another Network Interface";
		public readonly string VmConfigureVirtualDisksAction = "Configure Virtual Disks";
		public readonly string VmAddAnotherVirtualDiskAction = "Add another Virtual Disk";
		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		public new VM Entity
		{
			get { return (VM)base.Entity; }
			set { base.Entity = value; }
		}

		#endregion

		protected override void OnEntityChanged()
		{
			base.OnEntityChanged();
			UpdateOptions();
		}

		private void UpdateOptions()
		{
			CompulsoryActions.Clear();
			OptionalActions.Clear();

			if (Entity != null)
			{
				//Add NIC action.
				UICommand addNicAction = new UICommand("AddNetwork", this);

				List<VmNetworkInterface> nics = DataProvider.GetVmNicList(Entity.vm_guid);
				if (nics.Count == 0)
				{
					addNicAction.Title = VmConfigureNetworkInterfacesAction;
					CompulsoryActions.Add(addNicAction);
				}
				else
				{
					addNicAction.Title = VmAddAnotherNetworkInterfaceAction;
					OptionalActions.Add(addNicAction);
				}


				//Add disk action.
				UICommand addDiskAction = new UICommand("AddDisk", this);

				List<DiskImage> disks = DataProvider.GetVmDiskList(Entity.vm_guid);
				if (disks.Count == 0)
				{
					addDiskAction.Title = VmConfigureVirtualDisksAction;
					CompulsoryActions.Add(addDiskAction);
				}
				else
				{
					//                    if (!(Entity.vm_os == VmOsType.WindowsXP && disks.Count(a => a.disk_interface == DiskInterface.IDE) > 2))
					int ideDiskCount = 0;
					foreach (DiskImage a in disks)
					{
						if (a.disk_interface == DiskInterface.IDE)
						{
							ideDiskCount++;
						}

					}
					if (!(Entity.vm_os == VmOsType.WindowsXP && ideDiskCount > 2))
					{
						addDiskAction.Title = VmAddAnotherVirtualDiskAction;
						OptionalActions.Add(addDiskAction);
					}
				}
			}
		}

		public void AddNetwork()
		{
			if (Entity != null)
			{
				List<VmNetworkInterface> nics = DataProvider.GetVmNicList(Entity.vm_guid);
				int nicCount = nics.Count;
				string newNicName = DataProvider.GetNewNicName(nics);

				//var networks = DataProvider.GetNetworkList(Entity.vds_group_id)
				//    .Where(a => a.Status == NetworkStatus.Operational)
				//    .ToList();
				List<network> networks = new List<network>();
				foreach (network a in DataProvider.GetClusterNetworkList(Entity.vds_group_id))
				{
					if (a.Status == NetworkStatus.Operational)
					{
						networks.Add(a);
					}
				}

				VmInterfaceModel model = new VmInterfaceModel();
				Window = model;
				model.Title = "New Network Interface";
				model.HashName  = "new_network_interface_vms_guide";
				model.IsNew = true;
				model.Network.Items = networks;
				model.Network.SelectedItem = networks.Count > 0 ? networks[0] : null;
				model.NicType.Items = DataProvider.GetNicTypeList(Entity.vm_os, false);
				model.NicType.SelectedItem = DataProvider.GetDefaultNicType(Entity.vm_os);
				model.Name.Entity = newNicName;
				model.MAC.IsChangable = false;


				model.Commands.Add(
					new UICommand("OnAddNetwork", this)
					{
						Title = "OK",
						IsDefault = true
					});
				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
		}

		private void OnAddNetwork()
		{
			if (Entity != null)
			{
				VmInterfaceModel model = (VmInterfaceModel)Window;

				if (model.Progress != null)
				{
					return;
				}

				if (!model.Validate())
				{
					return;
				}

				//Save changes.
				int? _type;
				if(model.NicType.SelectedItem == null)
				{
					_type = null;
				}
				else
				{
					_type = (int?) (VmInterfaceType) model.NicType.SelectedItem;
				}

				VmNetworkInterface vmNetworkInterface = new VmNetworkInterface();
				vmNetworkInterface.Name = (string)model.Name.Entity;
				vmNetworkInterface.NetworkName = ((network)model.Network.SelectedItem).name;
				vmNetworkInterface.Type = _type;
				vmNetworkInterface.MacAddress = model.MAC.IsChangable ? (model.MAC.Entity == null ? null : ((string)(model.MAC.Entity)).ToLower()) : String.Empty;
				
				AddVmInterfaceParameters parameters = new AddVmInterfaceParameters(Entity.vm_guid, vmNetworkInterface);

				model.StartProgress(null);

				Frontend.RunAction(VdcActionType.AddVmInterface,
					parameters,
					result =>
					{
						VmGuideModel vmGuideModel = (VmGuideModel)result.State;
						vmGuideModel.Window.StopProgress();
						VdcReturnValueBase returnValueBase = result.ReturnValue;
						if (returnValueBase != null && returnValueBase.Succeeded)
						{
							vmGuideModel.Cancel();
							vmGuideModel.PostAction();
						}
					},
					this
				);
			}
			else
			{
				Cancel();
			}
		}

		public void AddDisk()
		{
			if (Entity != null)
			{
				List<DiskImage> disks = DataProvider.GetVmDiskList(Entity.vm_guid);
				bool hasDisks = disks.Count > 0;

				DiskModel model = new DiskModel();
				Window = model;
				model.Title = "New Virtual Disk";
				model.HashName = "new_virtual_disk";
				model.IsNew = true;

				//var storageDomains = DataProvider.GetStorageDomainList(Entity.storage_pool_id)
				//    .Where(a => a.storage_domain_type != StorageDomainType.ISO && a.storage_domain_type != StorageDomainType.ImportExport
				//    && a.status == StorageDomainStatus.Active);

				List<storage_domains> storageDomains = new List<storage_domains>();
				foreach (storage_domains a in DataProvider.GetStorageDomainList(Entity.storage_pool_id))
				{
					if (a.storage_domain_type != StorageDomainType.ISO && a.storage_domain_type != StorageDomainType.ImportExport
					&& a.status == StorageDomainStatus.Active)
					{
						storageDomains.Add(a);
					}
				}
				model.StorageDomain.Items = storageDomains;

				storage_domains storage = null;
				bool storage_available = false;

				if (hasDisks)
				{
					// the StorageDomain value should be the one that all other Disks are on
					// (although this field is not-available, we use its value in the 'OnSave' method):
					storage = DataProvider.GetStorageDomainByDiskList(disks);
					if (storage != null && Linq.IsSDItemExistInList(storageDomains, storage.id))
					{
						storage_available = true;
					}
				}
				else // first disk -> just choose the first from the list of available storage-domains:
				{
					storage = Linq.FirstOrDefault<storage_domains>(storageDomains);
					storage_available = true;
				}

				model.StorageDomain.SelectedItem = storage;
				model.StorageDomain.IsAvailable = !hasDisks;

				if (model.StorageDomain != null && model.StorageDomain.SelectedItem != null)
					DataProvider.GetWipeAfterDeleteDefaultsByStorageType
				   (
					   ((storage_domains)model.StorageDomain.SelectedItem).storage_type,
					   model.WipeAfterDelete,
					   true
				   );

				List<DiskImageBase> presets = DataProvider.GetDiskPresetList(
					Entity.vm_type,
					model.StorageDomain.SelectedItem == null ? StorageType.UNKNOWN : storage.storage_type);

				model.Preset.Items = presets;
				//model.Preset.SelectedItem = hasDisks
				//    ? presets.FirstOrDefault(a => a.disk_type == DiskType.Data)
				//    : presets.FirstOrDefault(a => a.disk_type == DiskType.System);

				model.Preset.SelectedItem = null;
				foreach (DiskImageBase a in presets)
				{
					if ((hasDisks && a.disk_type == DiskType.Data) ||
						(!hasDisks && a.disk_type == DiskType.System)
						)
					{
						model.Preset.SelectedItem = a;
						break;
					}
				}

				model.Interface.Items = DataProvider.GetDiskInterfaceList
				(
					Entity.vm_os,
					DataProvider.GetClusterById(Entity.vds_group_id).compatibility_version
				);
				model.Interface.SelectedItem = DataProvider.GetDefaultDiskInterface(Entity.vm_os, disks);

				//			bool hasBootableDisk = disks.Any(a => a.boot);
				bool hasBootableDisk = false;
				foreach (DiskImage a in disks)
				{
					if (a.boot)
					{
						hasBootableDisk = true;
						break;
					}
				}
				model.IsBootable.Entity = !hasBootableDisk;
				if (hasBootableDisk)
				{
					model.IsBootable.IsChangable = false;
					model.IsBootable.ChangeProhibitionReasons.Add("There can be only one bootable disk defined.");
				}


				if (storage == null || storage_available == false)
				{
					string cantCreateMessage = "There is no active Storage Domain to create the Disk in. Please activate a Storage Domain.";
					if (hasDisks)
					{
						cantCreateMessage = "Error in retrieving the relevant Storage Domain.";
						if (storage != null && storage.storage_name != null)
						{
							cantCreateMessage = string.Format("'{0}' Storage Domain is not active. Please activate it.", storage.storage_name);
						}
					}

					model.Message = cantCreateMessage;

					model.Commands.Add(
						new UICommand("Cancel", this)
						{
							Title = "Close",
							IsDefault = true,
							IsCancel = true
						});
				}
				else
				{
					model.Commands.Add(
						new UICommand("OnAddDisk", this)
						{
							Title = "OK",
							IsDefault = true
						});

					model.Commands.Add(
						new UICommand("Cancel", this)
						{
							Title = "Cancel",
							IsCancel = true
						});
				}
			}
		}

		public void OnAddDisk()
		{
			if (Entity != null)
			{
				DiskModel model = (DiskModel)Window;

				if (model.Progress != null)
				{
					return;
				}

				if (!model.Validate())
				{
					return;
				}

				//Save changes.
				storage_domains storageDomain = (storage_domains)model.StorageDomain.SelectedItem;

				DiskImage disk =
					new DiskImage
					{
						SizeInGigabytes = Convert.ToInt32(model.Size.Entity.ToString()),
						disk_type = ((DiskImageBase)model.Preset.SelectedItem).disk_type,
						disk_interface = (DiskInterface)model.Interface.SelectedItem,
						volume_type = (VolumeType)model.VolumeType.SelectedItem,
						volume_format = model.VolumeFormat,
						wipe_after_delete = (bool)model.WipeAfterDelete.Entity,
						boot = (bool)model.IsBootable.Entity,
						propagate_errors = PropagateErrors.Off
					};

				model.StartProgress(null);

				Frontend.RunAction(VdcActionType.AddDiskToVm,
					new AddDiskToVmParameters(Entity.vm_guid, disk)
					{
						StorageDomainId = storageDomain.id
					},
					result =>
					{
						VmGuideModel vmGuideModel = (VmGuideModel)result.State;
						vmGuideModel.Window.StopProgress();
						VdcReturnValueBase returnValueBase = result.ReturnValue;
						if (returnValueBase != null && returnValueBase.Succeeded)
						{
							vmGuideModel.Cancel();
							vmGuideModel.PostAction();
						}
					},
					this
				);
			}
			else
			{
				Cancel();
			}
		}

		public void PostAction()
		{
			UpdateOptions();
		}

		public void Cancel()
		{
			Window = null;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command.Name == "AddNetwork")
			{
				AddNetwork();
			}
			if (command.Name == "AddDisk")
			{
				AddDisk();
			}
			if (command.Name == "OnAddNetwork")
			{
				OnAddNetwork();
			}
			if (command.Name == "OnAddDisk")
			{
				OnAddDisk();
			}
			if (command.Name == "Cancel")
			{
				Cancel();
			}
		}
	}
}
