﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using org.ovirt.engine.ui.uicommon.dataprovider;
using org.ovirt.engine.ui.uicommon.validation;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.BusinessEntities;
using VdcFrontend;

namespace org.ovirt.engine.ui.uicommon.models.vms
{
	public abstract class IVmModelBehavior
	{
		public UnitVmModel Model { get; set; }
		public SystemTreeItemModel SystemTreeSelectedItem { get; set; }

		public virtual void Initialize(SystemTreeItemModel systemTreeSelectedItem)
		{
			this.SystemTreeSelectedItem = systemTreeSelectedItem;
		}
		public abstract void DataCenter_SelectedItemChanged();
		public abstract void Template_SelectedItemChanged();
		public abstract void Cluster_SelectedItemChanged();
		public abstract void DefaultHost_SelectedItemChanged();
		public abstract void Provisioning_SelectedItemChanged();
		public abstract void UpdateMinAllocatedMemory();

		public virtual bool Validate()
		{
			return true;
		}

		protected void UpdateCdImage()
		{
			storage_pool dataCenter = (storage_pool)Model.DataCenter.SelectedItem;
			if (dataCenter == null)
			{
				return;
			}

			AsyncDataProvider.GetIsoDomainByDataCenterId(new AsyncQuery(this,
				(target, returnValue) =>
				{
					IVmModelBehavior behavior = (IVmModelBehavior)target;
					storage_domains storageDomain = (storage_domains)returnValue;

					if (storageDomain != null)
					{
						behavior.PostUpdateCdImage(storageDomain.id);
					}
					else
					{
						behavior.Model.CdImage.Items = new List<string>();
					}
				}, Model.Hash),
				dataCenter.Id
			);
		}

		public void PostUpdateCdImage(Guid storageDomainId)
		{
			AsyncDataProvider.GetIrsImageList(new AsyncQuery(Model,
				(target, returnValue) =>
				{
					UnitVmModel model = (UnitVmModel)target;
					List<string> images = (List<string>)returnValue;

					//If there was some cd image selected before, try select it again.
					string oldCdImage = (string)model.CdImage.SelectedItem;

					model.CdImage.Items = images;
					model.CdImage.SelectedItem = oldCdImage ?? Linq.FirstOrDefault(images);
				}, Model.Hash),
				storageDomainId, false
			);
		}


		private IEnumerable<KeyValuePair<string, string>> cachedTimeZones;

		protected void UpdateTimeZone()
		{
			if (cachedTimeZones == null)
			{
				AsyncDataProvider.GetTimeZoneList(new AsyncQuery(this,
					(target, returnValue) =>
					{
						IVmModelBehavior behavior = (IVmModelBehavior)target;
						cachedTimeZones = ((Dictionary<string, string>)(valueObjectEnumerableMap)returnValue).entrySet();

						behavior.PostUpdateTimeZone();
					}, Model.Hash)
				);
			}
			else
			{
				PostUpdateTimeZone();
			}
		}

		public void PostUpdateTimeZone()
		{
			//If there was some time zone selected before, try select it again.
			string oldTimeZoneKey = ((KeyValuePair<string, string>)Model.TimeZone.SelectedItem).Key;

			Model.TimeZone.Items = cachedTimeZones;
			if (oldTimeZoneKey != null)
			{
				Model.TimeZone.SelectedItem = Linq.FirstOrDefault(cachedTimeZones, new Linq.TimeZonePredicate(oldTimeZoneKey));
			}
			else
			{
				Model.TimeZone.SelectedItem = null;
			}
		}

		private string cachedDefaultTimeZoneKey;

		protected void UpdateDefaultTimeZone()
		{
			if (cachedDefaultTimeZoneKey == null)
			{
				AsyncDataProvider.GetDefaultTimeZone(new AsyncQuery(this,
					(target, returnValue) =>
					{
						IVmModelBehavior behavior = (IVmModelBehavior)target;
						cachedDefaultTimeZoneKey = (string)returnValue;

						behavior.PostUpdateDefaultTimeZone();
					}, Model.Hash)
				);
			}
			else
			{
				PostUpdateDefaultTimeZone();
			}
		}

		public void PostUpdateDefaultTimeZone()
		{
			//Patch! Create key-value pair with a right key.
			Model.TimeZone.SelectedItem = new KeyValuePair<string, string>(cachedDefaultTimeZoneKey, String.Empty);

			UpdateTimeZone();
		}

		protected void UpdateDomain()
		{
			AsyncDataProvider.GetDomainList(new AsyncQuery(this,
				(target, returnValue) =>
				{
					IVmModelBehavior behavior = (IVmModelBehavior)target;
					IList<string> domains = (IList<string>)returnValue;

					// Get last selected domain
					string oldDomain = (string)behavior.Model.Domain.SelectedItem;

					if (oldDomain != null && !oldDomain.Equals(String.Empty) && !domains.Contains(oldDomain))
					{
						domains.Insert(0, oldDomain);	
					}

					behavior.Model.Domain.Items = domains;
					behavior.Model.Domain.SelectedItem = oldDomain ?? Linq.FirstOrDefault(domains);
				}, Model.Hash), true);
		}


		protected void InitPriority(int priority)
		{
			AsyncDataProvider.GetRoundedPriority(new AsyncQuery(Model,
				(target, returnValue) =>
				{
					UnitVmModel model = (UnitVmModel)target;
					int value = (int)returnValue;

					//Patch!
					model.Priority.SelectedItem = new EntityModel { Entity = value };

					UpdatePriority();
				}, Model.Hash),
				priority
			);
		}

		private int? cachedMaxPrority;

		protected void UpdatePriority()
		{
			if (cachedMaxPrority == null)
			{
				AsyncDataProvider.GetMaxVmPriority(new AsyncQuery(this,
					(target, returnValue) =>
					{
						IVmModelBehavior behavior = (IVmModelBehavior)target;
						cachedMaxPrority = (int)returnValue;

						behavior.PostUpdatePriority();
					}, Model.Hash)
				);
			}
			else
			{
				PostUpdatePriority();
			}
		}

		private void PostUpdatePriority()
		{
			List<EntityModel> items = new List<EntityModel>();
			items.Add(
				new EntityModel
				{
					Title = "Low",
					Entity = 1
				});
			items.Add(
				new EntityModel
				{
					Title = "Medium",
					Entity = cachedMaxPrority / 2
				});
			items.Add(
				new EntityModel
				{
					Title = "High",
					Entity = cachedMaxPrority
				});


			//If there was some priority selected before, try select it again.
			EntityModel oldPriority = (EntityModel)Model.Priority.SelectedItem;

			Model.Priority.Items = items;

			if (oldPriority != null)
			{
				foreach (EntityModel item in items)
				{
					int val1 = (int) item.Entity;
					int val2 = (int) oldPriority.Entity;
					if (val1.Equals(val2))
					{
						Model.Priority.SelectedItem = item;
						break;
					}
				}
			}
			else
			{
				Model.Priority.SelectedItem = Linq.FirstOrDefault(items);
			}
		}

		protected virtual void ChangeDefualtHost()
		{
			
		}

		protected void UpdateDefaultHost()
		{
			VDSGroup cluster = (VDSGroup)Model.Cluster.SelectedItem;

			if (cluster == null)
			{
				Model.DefaultHost.Items = new List<VDS>();
				Model.DefaultHost.SelectedItem = null;

				return;
			}

			AsyncDataProvider.GetHostListByCluster(new AsyncQuery(Model,
				(target, returnValue) =>
				{
					UnitVmModel model = (UnitVmModel)target;
					List<VDS> hosts = (List<VDS>)returnValue;
					
					//If there was some host selected before, try select it again.
					VDS oldDefaultHost = (VDS)model.DefaultHost.SelectedItem;
					
					if(model.Behavior.SystemTreeSelectedItem != null && model.Behavior.SystemTreeSelectedItem.Type == SystemTreeItemType.Host)
					{
						VDS host = (VDS) model.Behavior.SystemTreeSelectedItem.Entity;
						foreach (VDS vds in hosts)
						{
							if(host.vds_id.Equals(vds.vds_id))
							{
								model.DefaultHost.Items = new List<VDS> {vds};
								model.DefaultHost.SelectedItem = vds;
								model.DefaultHost.IsChangable = false;
								model.DefaultHost.Info = "Cannot choose other Host in tree context";
								break;
							}
						}
					}
					else
					{
						model.DefaultHost.Items = hosts;
						model.DefaultHost.SelectedItem =
						oldDefaultHost != null
							? Linq.FirstOrDefault(hosts, new Linq.HostPredicate(oldDefaultHost.vds_id))
							: Linq.FirstOrDefault(hosts);
					}
					ChangeDefualtHost();
				}, Model.Hash),
				cluster.name
			);
		}

		protected void UpdateIsCustomPropertiesAvailable()
		{
			VDSGroup cluster = (VDSGroup)Model.Cluster.SelectedItem;

			if (cluster != null)
			{
				AsyncDataProvider.IsCustomPropertiesAvailable(new AsyncQuery(Model,
					(target, returnValue) =>
					{
						UnitVmModel model = (UnitVmModel)target;

						model.IsCustomPropertiesAvailable = (bool)returnValue;
					}, Model.Hash),
					cluster.compatibility_version.ToString()
				);
			}
		}


		public int maxCpus = 0;
		public int maxCpusPerSocket = 0;

		protected void UpdateNumOfSockets()
		{
			VDSGroup cluster = (VDSGroup)Model.Cluster.SelectedItem;
			if (cluster == null)
			{
				return;
			}


			string version = cluster.compatibility_version.ToString();

			AsyncDataProvider.GetMaxNumOfVmSockets(new AsyncQuery(new object[] { this, Model },
				(target, returnValue) =>
				{
					object[] array = (object[])target;
					IVmModelBehavior behavior = (IVmModelBehavior)array[0];
					UnitVmModel model = (UnitVmModel)array[1];

					model.NumOfSockets.Max = (int)returnValue;
					model.NumOfSockets.Min = 1;
					model.NumOfSockets.Interval = 1;
					model.NumOfSockets.IsAllValuesSet = true;

					if (model.NumOfSockets.Entity == null)
					{
						model.NumOfSockets.Entity = 1;
					}

					behavior.PostUpdateNumOfSockets();
				}, Model.Hash),
				version
			);
		}

		public void PostUpdateNumOfSockets()
		{
			VDSGroup cluster = (VDSGroup)Model.Cluster.SelectedItem;
			string version = cluster.compatibility_version.ToString();

			AsyncDataProvider.GetMaxNumOfVmCpus(new AsyncQuery(this,
				(target, returnValue) =>
				{
					IVmModelBehavior behavior = (IVmModelBehavior)target;

					behavior.maxCpus = (int)returnValue;
					behavior.PostUpdateNumOfSockets2();
				}, Model.Hash),
				version
			);
		}

		public void PostUpdateNumOfSockets2()
		{
			VDSGroup cluster = (VDSGroup)Model.Cluster.SelectedItem;
			string version = cluster.compatibility_version.ToString();

			AsyncDataProvider.GetMaxNumOfCPUsPerSocket(new AsyncQuery(this,
				(target, returnValue) =>
				{
					IVmModelBehavior behavior = (IVmModelBehavior)target;

					behavior.maxCpusPerSocket = (int)returnValue;
					behavior.UpdateTotalCpus();
				}, Model.Hash),
				version
			);
		}

		public void UpdateTotalCpus()
		{
			int numOfSockets = Convert.ToInt32(Model.NumOfSockets.Entity.ToString());

			int totalCPUCores = Model.TotalCPUCores.Entity != null
				? Convert.ToInt32(Model.TotalCPUCores.Entity.ToString())
				: 0;

			int realMaxCpus = maxCpus < numOfSockets * maxCpusPerSocket ? maxCpus : numOfSockets * maxCpusPerSocket;

			if (maxCpus == 0 || maxCpusPerSocket == 0)
			{
				return;
			}

			Model.TotalCPUCores.Max = realMaxCpus - (realMaxCpus % numOfSockets);
			Model.TotalCPUCores.Min = numOfSockets;
			Model.TotalCPUCores.Interval = numOfSockets;

			//update value if needed
			//if the slider in the range but not on tick update it to lowest tick
			if ((totalCPUCores % numOfSockets != 0) && totalCPUCores < Model.TotalCPUCores.Max && totalCPUCores > Model.TotalCPUCores.Min)
			{
				Model.TotalCPUCores.Entity = totalCPUCores - (totalCPUCores % numOfSockets);
			}
			//if the value is lower than range update it to min
			else if (totalCPUCores < Model.TotalCPUCores.Min)
			{
				Model.TotalCPUCores.Entity = Convert.ToInt32(Model.TotalCPUCores.Min);
			}
			//if the value is higher than range update it to max
			else if (totalCPUCores > Model.TotalCPUCores.Max)
			{
				Model.TotalCPUCores.Entity = Convert.ToInt32(Model.TotalCPUCores.Max);
			}

			Model.TotalCPUCores.IsAllValuesSet = true;
		}

		
	}


	public class UnitVmModel : Model
	{
		#region Const

		public const int WINDOWS_VM_NAME_MAX_LIMIT = 15;
		public const int NON_WINDOWS_VM_NAME_MAX_LIMIT = 64;

		#endregion

		#region Properties

		public virtual bool IsNew { get; set; }
		public VmType VmType { get; set; }
		
		public String Hash { get; set; }

		private bool isBlankTemplate;
		public bool IsBlankTemplate
		{
			get { return isBlankTemplate; }
			set
			{
				if (isBlankTemplate != value)
				{
					isBlankTemplate = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsBlankTemplate"));
				}
			}
		}

		bool isWindowsOS;
		public bool IsWindowsOS
		{
			get { return isWindowsOS; }
			set
			{
				if (isWindowsOS != value)
				{
					isWindowsOS = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsWindowsOS"));
				}
			}
		}


		bool isLinux_Unassign_UnknownOS;
		public bool IsLinux_Unassign_UnknownOS
		{
			get { return isLinux_Unassign_UnknownOS; }
			set
			{
				if (isLinux_Unassign_UnknownOS != value)
				{
					isLinux_Unassign_UnknownOS = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsLinux_Unassign_UnknownOS"));
				}
			}
		}

		private string cpuNotification;
		public string CPUNotification
		{
			get { return cpuNotification; }
			set
			{
				if (cpuNotification != value)
				{
					cpuNotification = value;
					OnPropertyChanged(new PropertyChangedEventArgs("CPUNotification"));
				}
			}
		}

		public bool isCPUsAmountValid;
		public bool IsCPUsAmountValid
		{
			get { return isCPUsAmountValid; }
			set
			{
				if (isCPUsAmountValid != value)
				{
					isCPUsAmountValid = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsCPUsAmountValid"));
				}
			}
		}

		bool isGeneralTabValid;
		public bool IsGeneralTabValid
		{
			get { return isGeneralTabValid; }
			set
			{
				if (isGeneralTabValid != value)
				{
					isGeneralTabValid = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsGeneralTabValid"));
				}
			}
		}

		bool isFirstRunTabValid;
		public bool IsFirstRunTabValid
		{
			get { return isFirstRunTabValid; }
			set
			{
				if (isFirstRunTabValid != value)
				{
					isFirstRunTabValid = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsFirstRunTabValid"));
				}
			}
		}

		bool isDisplayTabValid;
		public bool IsDisplayTabValid
		{
			get { return isDisplayTabValid; }
			set
			{
				if (isDisplayTabValid != value)
				{
					isDisplayTabValid = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsDisplayTabValid"));
				}
			}
		}

		bool isAllocationTabValid;
		public bool IsAllocationTabValid
		{
			get { return isAllocationTabValid; }
			set
			{
				if (isAllocationTabValid != value)
				{
					isAllocationTabValid = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsAllocationTabValid"));
				}
			}
		}

		bool isHostTabValid;
		public bool IsHostTabValid
		{
			get { return isHostTabValid; }
			set
			{
				if (isHostTabValid != value)
				{
					isHostTabValid = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsHostTabValid"));
				}
			}
		}

		bool isBootSequenceTabValid;
		public bool IsBootSequenceTabValid
		{
			get { return isBootSequenceTabValid; }
			set
			{
				if (isBootSequenceTabValid != value)
				{
					isBootSequenceTabValid = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsBootSequenceTabValid"));
				}
			}
		}

		bool isCustomPropertiesTabValid;
		public bool IsCustomPropertiesTabValid
		{
			get { return isCustomPropertiesTabValid; }
			set
			{
				if (isCustomPropertiesTabValid != value)
				{
					isCustomPropertiesTabValid = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsCustomPropertiesTabValid"));
				}
			}
		}

		public ListModel DataCenter { get; private set; }
		public ListModel StorageDomain { get; private set; }
		public ListModel Template { get; private set; }
		public EntityModel Name { get; private set; }
		public ListModel OSType { get; private set; }
		public ListModel NumOfMonitors { get; private set; }
		public EntityModel Description { get; private set; }
		public ListModel Domain { get; private set; }
		public EntityModel MemSize { get; private set; }
		public EntityModel MinAllocatedMemory { get; private set; }
		public ListModel Cluster { get; private set; }
		public ListModel UsbPolicy { get; private set; }
		public ListModel TimeZone { get; private set; }

		public RangeEntityModel NumOfSockets { get; private set; }
		public RangeEntityModel TotalCPUCores { get; private set; }

		public ListModel DefaultHost { get; private set; }
		public EntityModel IsStateless { get; private set; }
		public ListModel DisplayProtocol { get; private set; }
		public ListModel Provisioning { get; private set; }
		public ListModel Priority { get; private set; }
		public EntityModel IsHighlyAvailable { get; private set; }
		public ListModel FirstBootDevice { get; private set; }
		public ListModel SecondBootDevice { get; private set; }
		public ListModel CdImage { get; private set; }

		public EntityModel Initrd_path { get; private set; }
		public EntityModel Kernel_path { get; private set; }
		public EntityModel Kernel_parameters { get; private set; }

		public EntityModel CustomProperties { get; private set; }
		public List<string> CustomPropertiesKeysList { get; set; }

		public EntityModel IsAutoAssign { get; set; }
		public EntityModel RunVMOnSpecificHost { get; set; }
		public EntityModel DontMigrateVM { get; set; }

		public EntityModel IsTemplatePublic { get; private set; }

		public bool IsFirstRun { get; set; }

		private IList<DiskModel> disks;
		public IList<DiskModel> Disks
		{
			get { return disks; }
			set
			{
				if (disks != value)
				{
					disks = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Disks"));
				}
			}
		}

		private bool isDisksAvailable;
		public bool IsDisksAvailable
		{
			get { return isDisksAvailable; }
			set
			{
				if (isDisksAvailable != value)
				{
					isDisksAvailable = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsDisksAvailable"));
				}
			}
		}

		private bool isCustomPropertiesAvailable;
		public bool IsCustomPropertiesAvailable
		{
			get { return isCustomPropertiesAvailable; }
			set
			{
				if (isCustomPropertiesAvailable != value)
				{
					isCustomPropertiesAvailable = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsCustomPropertiesAvailable"));

					if (value == false)
					{
						CustomProperties.Entity = String.Empty;
					}
				}
			}
		}

		private bool isHostAvailable;
		public bool IsHostAvailable
		{
			get { return isHostAvailable; }
			set
			{
				if (isHostAvailable != value)
				{
					isHostAvailable = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsHostAvailable"));
				}
			}
		}

		private bool? isDatacenterAvailable;
		public bool? IsDatacenterAvailable
		{
			get { return isDatacenterAvailable; }
			set
			{
				if (isDatacenterAvailable == null && value == null)
					return;
				if (isDatacenterAvailable == null || isDatacenterAvailable != value)
				{
					isDatacenterAvailable = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsDatacenterAvailable"));
				}
			}
		}

		#endregion

		public IVmModelBehavior Behavior
		{
			get { return behavior; }
			private set
			{
				
			}
		}
		private readonly IVmModelBehavior behavior;

		private int _minMemSize = 1;
		public int _MinMemSize { get { return _minMemSize; } set { _minMemSize = value; } }

		private int _maxMemSize32 = 20480;
		public int _MaxMemSize32 { get { return _maxMemSize32; } set { _maxMemSize32 = value; } }

		private int _maxMemSize64 = 262144;
		public int _MaxMemSize64 { get { return _maxMemSize64; } set { _maxMemSize64 = value; } }


		public UnitVmModel(IVmModelBehavior behavior)
		{
			Frontend.QueryStartedEvent.addListener(this);
			Frontend.QueryCompleteEvent.addListener(this);

			Frontend.Subscribe(
				new VdcQueryType[]
				{
					VdcQueryType.GetStorageDomainsByStoragePoolId,
					VdcQueryType.GetAllIsoImagesList,
					VdcQueryType.GetTimeZones,
					VdcQueryType.GetDefualtTimeZone,
					VdcQueryType.GetDomainList,
					VdcQueryType.GetConfigurationValue,
					VdcQueryType.GetVdsGroupsByStoragePoolId,
					VdcQueryType.GetVmTemplatesByStoragePoolId,
					VdcQueryType.GetVmTemplatesDisks,
					VdcQueryType.GetStorageDomainsByVmTemplateId,
					VdcQueryType.GetStorageDomainById,
					VdcQueryType.GetDataCentersWithPermittedActionOnClusters,
					VdcQueryType.GetClustersWithPermittedAction,
					VdcQueryType.GetVmTemplatesWithPermittedAction,
					VdcQueryType.GetVdsGroupById,
					VdcQueryType.GetStoragePoolById,
					VdcQueryType.GetAllDisksByVmId,
					VdcQueryType.GetVmTemplate,
					VdcQueryType.Search
				});


			this.behavior = behavior;
			this.behavior.Model = this;

			StorageDomain = new ListModel();
			Name = new EntityModel();
			NumOfMonitors = new ListModel();
			Description = new EntityModel();
			Domain = new ListModel();
			MinAllocatedMemory = new EntityModel();
			UsbPolicy = new ListModel();
			IsStateless = new EntityModel();
			CdImage = new ListModel();
			IsHighlyAvailable = new EntityModel();
			DontMigrateVM = new EntityModel();
			IsTemplatePublic = new EntityModel();
			Kernel_parameters = new EntityModel();
			Kernel_path = new EntityModel();
			Initrd_path = new EntityModel();
			CustomProperties = new EntityModel();
			DisplayProtocol = new ListModel();
			SecondBootDevice = new ListModel();
			Priority = new ListModel();
			TotalCPUCores = new RangeEntityModel();

			DataCenter = new ListModel();
			DataCenter.SelectedItemChangedEvent.addListener(this);

			Template = new ListModel();
			Template.SelectedItemChangedEvent.addListener(this);

			Cluster = new ListModel();
			Cluster.SelectedItemChangedEvent.addListener(this);

			TimeZone = new ListModel();
			TimeZone.SelectedItemChangedEvent.addListener(this);

			DefaultHost = new ListModel();
			DefaultHost.SelectedItemChangedEvent.addListener(this);

			OSType = new ListModel();
			OSType.SelectedItemChangedEvent.addListener(this);

			FirstBootDevice = new ListModel();
			FirstBootDevice.SelectedItemChangedEvent.addListener(this);

			Provisioning = new ListModel();
			Provisioning.SelectedItemChangedEvent.addListener(this);

			MemSize = new EntityModel();
			MemSize.EntityChangedEvent.addListener(this);

			NumOfSockets = new RangeEntityModel();
			NumOfSockets.EntityChangedEvent.addListener(this);

			RunVMOnSpecificHost = new EntityModel();
			RunVMOnSpecificHost.EntityChangedEvent.addListener(this);

			IsAutoAssign = new EntityModel();
			IsAutoAssign.EntityChangedEvent.addListener(this);

			IsGeneralTabValid = IsFirstRunTabValid = IsDisplayTabValid = IsAllocationTabValid = IsBootSequenceTabValid = IsCustomPropertiesTabValid = IsHostTabValid = true;
		}

		public void Initialize(SystemTreeItemModel SystemTreeSelectedItem)
		{
			base.Initialize();

			Hash = HashName + DateTime.Now;

			MemSize.Entity = 256;
			MinAllocatedMemory.Entity = 256;
			IsStateless.Entity = false;
			IsHighlyAvailable.Entity = false;
			DontMigrateVM.Entity = false;
			IsAutoAssign.Entity = true;
			IsTemplatePublic.Entity = false;

			RunVMOnSpecificHost.Entity = false;
			RunVMOnSpecificHost.IsChangable = false;

			CdImage.IsChangable = false;

			InitUsbPolicy();
			InitOSType();
			InitDisplayProtocol();
			InitFirstBootDevice();
			InitProvisioning();
			InitNumOfMonitors();
			InitMinimalVmMemSize();
			InitMaximalVmMemSize32OS();

			behavior.Initialize(SystemTreeSelectedItem);
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(Frontend.QueryStartedEventDefinition) && Frontend.CurrentContext == Hash)
			{
				Frontend_QueryStarted();
			}
			else if (ev.Equals(Frontend.QueryCompleteEventDefinition) && Frontend.CurrentContext == Hash)
			{
				Frontend_QueryComplete();
			}
			else if (ev.Equals(ListModel.SelectedItemChangedEventDefinition))
			{
				if (sender == DataCenter)
				{
					DataCenter_SelectedItemChanged(sender, args);
				}
				else if (sender == Template)
				{
					Template_SelectedItemChanged(sender, args);
				}
				else if (sender == Cluster)
				{
					Cluster_SelectedItemChanged(sender, args);
				}
				else if (sender == TimeZone)
				{
					TimeZone_SelectedItemChanged(sender, args);
				}
				else if (sender == DefaultHost)
				{
					DefaultHost_SelectedItemChanged(sender, args);
				}
				else if (sender == OSType)
				{
					OSType_SelectedItemChanged(sender, args);
				}
				else if (sender == FirstBootDevice)
				{
					FirstBootDevice_SelectedItemChanged(sender, args);
				}
				else if (sender == Provisioning)
				{
					Provisioning_SelectedItemChanged(sender, args);
				}
				else if (sender == DisplayProtocol)
				{
					DisplayProtocol_SelectedItemChanged(sender, args);
				}
			}
			else if (ev.Equals(EntityModel.EntityChangedEventDefinition))
			{
				if (sender == MemSize)
				{
					MemSize_EntityChanged(sender, args);
				}
				else if (sender == NumOfSockets)
				{
					NumOfSockets_EntityChanged(sender, args);
				}
				else if (sender == RunVMOnSpecificHost)
				{
					RunVMOnSpecificHost_EntityChanged(sender, args);
				}
				else if (sender == IsAutoAssign)
				{
					IsAutoAssign_EntityChanged(sender, args);
				}
			}
		}

		private int queryCounter;

		private void Frontend_QueryStarted()
		{
			queryCounter++;
			if (Progress == null)
			{
				StartProgress(null);
			}
		}

		private void Frontend_QueryComplete()
		{
			queryCounter--;
			if (queryCounter == 0)
			{
				StopProgress();
			}
		}

		protected void InitNumOfMonitors()
		{
			if (VmType == VmType.Desktop)
			{
				AsyncDataProvider.GetNumOfMonitorList(new AsyncQuery(this,
					(target, returnValue) =>
					{
						UnitVmModel model = (UnitVmModel)target;

						int? oldNumOfMonitors = null;
						if (model.NumOfMonitors.SelectedItem != null)
						{
							oldNumOfMonitors = (int)model.NumOfMonitors.SelectedItem;
						}

						List<int> numOfMonitors = (List<int>)returnValue;
						model.NumOfMonitors.Items = numOfMonitors;

						if (oldNumOfMonitors != null)
						{
							model.NumOfMonitors.SelectedItem = oldNumOfMonitors;
						}
					}, Hash)
				);
			}
			else
			{
				NumOfMonitors.Items = new List<int> { 1 };
				NumOfMonitors.SelectedItem = 1;
			}
		}

		private void InitOSType()
		{
			OSType.Items = DataProvider.GetOSList();
			OSType.SelectedItem = VmOsType.Unassigned;
		}

		private void InitUsbPolicy()
		{
			UsbPolicy.Items = DataProvider.GetUsbPolicyList();
		}

		private void InitMinimalVmMemSize()
		{
			AsyncDataProvider.GetMinimalVmMemSize(new AsyncQuery(this,
					(target, returnValue) =>
					{
						UnitVmModel vmModel = (UnitVmModel)target;
						vmModel._MinMemSize = (int)returnValue;
					}, Hash)
				);
		}

		private void InitMaximalVmMemSize32OS()
		{
			AsyncDataProvider.GetMaximalVmMemSize32OS(new AsyncQuery(this,
					(target, returnValue) =>
					{
						UnitVmModel vmModel = (UnitVmModel)target;
						vmModel._MaxMemSize32 = (int)returnValue;
					}, Hash)
				);
		}

		private void UpdateMaximalVmMemSize()
		{
			VDSGroup cluster = (VDSGroup)Cluster.SelectedItem;

			if (cluster != null)
			{
				AsyncDataProvider.GetMaximalVmMemSize64OS(new AsyncQuery(this,
						(target, returnValue) =>
						{
							UnitVmModel vmModel = (UnitVmModel)target;
							vmModel._MaxMemSize64 = (int)returnValue;
						}, Hash),
						cluster.compatibility_version.ToString()
					);
			}
		}

		private void InitDisplayProtocol()
		{
			List<EntityModel> displayProtocolOptions = new List<EntityModel>();

			EntityModel spiceProtocol = new EntityModel();
			spiceProtocol.Title = "Spice";
			spiceProtocol.Entity = DisplayType.qxl;

			EntityModel vncProtocol = new EntityModel();
			vncProtocol.Title = "VNC";
			vncProtocol.Entity = DisplayType.vnc;

			displayProtocolOptions.Add(spiceProtocol);
			displayProtocolOptions.Add(vncProtocol);
			DisplayProtocol.Items = displayProtocolOptions;

			DisplayProtocol.SelectedItemChangedEvent.addListener(this);
		}

		private void InitFirstBootDevice()
		{
			EntityModel hardDiskOption =
				new EntityModel
				{
					Title = "Hard Disk",
					Entity = BootSequence.C
				};

			List<EntityModel> firstBootDeviceItems = new List<EntityModel>();
			firstBootDeviceItems.Add(hardDiskOption);
			firstBootDeviceItems.Add(
				new EntityModel
				{
					Title = "CD-ROM",
					Entity = BootSequence.D
				});
			firstBootDeviceItems.Add(
				new EntityModel
				{
					Title = "Network (PXE)",
					Entity = BootSequence.N
				});
			FirstBootDevice.Items = firstBootDeviceItems;
			FirstBootDevice.SelectedItem = hardDiskOption;
		}

		private void InitProvisioning()
		{
			List<EntityModel> provisioningItems = new List<EntityModel>();
			provisioningItems.Add(
				new EntityModel
				{
					Title = "Thin",
					Entity = false
				});
			provisioningItems.Add(
				new EntityModel
				{
					Title = "Clone",
					Entity = true
				});
			Provisioning.Items = provisioningItems;
		}

		private void DataCenter_SelectedItemChanged(object sender, EventArgs args)
		{
			behavior.DataCenter_SelectedItemChanged();
		}

		private void Template_SelectedItemChanged(object sender, EventArgs args)
		{
			behavior.Template_SelectedItemChanged();
		}

		private void Cluster_SelectedItemChanged(object sender, EventArgs args)
		{
			behavior.Cluster_SelectedItemChanged();

			UpdateMaximalVmMemSize();
		}

		private void TimeZone_SelectedItemChanged(object sender, EventArgs args)
		{
		}

		private void DefaultHost_SelectedItemChanged(object sender, EventArgs args)
		{
			behavior.DefaultHost_SelectedItemChanged();
		}

		private void OSType_SelectedItemChanged(object sender, EventArgs args)
		{
			VmOsType osType = (VmOsType)OSType.SelectedItem;

			IsWindowsOS = DataProvider.IsWindowsOsType(osType);
			IsLinux_Unassign_UnknownOS = DataProvider.IsLinuxOsType(osType) || osType == VmOsType.Unassigned || osType == VmOsType.Other;

			Initrd_path.IsChangable = IsLinux_Unassign_UnknownOS;
			Initrd_path.IsAvailable = IsLinux_Unassign_UnknownOS;

			Kernel_path.IsChangable = IsLinux_Unassign_UnknownOS;
			Kernel_path.IsAvailable = IsLinux_Unassign_UnknownOS;

			Kernel_parameters.IsChangable = IsLinux_Unassign_UnknownOS;
			Kernel_parameters.IsAvailable = IsLinux_Unassign_UnknownOS;

			Domain.IsChangable = IsWindowsOS;
			Domain.IsAvailable = IsWindowsOS;

			TimeZone.IsChangable = IsWindowsOS;
			TimeZone.IsAvailable = IsWindowsOS;

			UpdateNumOfMonitors();
		}

		private void FirstBootDevice_SelectedItemChanged(object sender, EventArgs args)
		{
			EntityModel entityModel = (EntityModel)FirstBootDevice.SelectedItem;
			BootSequence firstDevice = (BootSequence)entityModel.Entity;

			List<EntityModel> list = new List<EntityModel>();
			foreach (object item in FirstBootDevice.Items)
			{
				EntityModel a = (EntityModel)item;
				if ((BootSequence)a.Entity != firstDevice)
				{
					list.Add(a);
				}
			}

			EntityModel noneOption = new EntityModel { Title = "[None]" };

			list.Insert(0, noneOption);

			SecondBootDevice.Items = list;
			SecondBootDevice.SelectedItem = noneOption;
		}

		private void Provisioning_SelectedItemChanged(object sender, EventArgs args)
		{
			behavior.Provisioning_SelectedItemChanged();
		}

		private void DisplayProtocol_SelectedItemChanged(object sender, EventArgs args)
		{
			EntityModel entityModel = (EntityModel)DisplayProtocol.SelectedItem;
			if (entityModel == null)
			{
				return;
			}
			DisplayType type = (DisplayType)entityModel.Entity;

			if (type == DisplayType.vnc)
			{
				UsbPolicy.SelectedItem = VdcCommon.BusinessEntities.UsbPolicy.Disabled;
			}

			UsbPolicy.IsChangable = type == DisplayType.qxl;

			UpdateNumOfMonitors();
		}

		private void MemSize_EntityChanged(object sender, EventArgs args)
		{
			behavior.UpdateMinAllocatedMemory();
		}

		private void NumOfSockets_EntityChanged(object sender, EventArgs args)
		{
			behavior.UpdateTotalCpus();
		}

		private void RunVMOnSpecificHost_EntityChanged(object sender, EventArgs args)
		{
			if ((bool)RunVMOnSpecificHost.Entity == true)
			{
				DontMigrateVM.Entity = true;
				DontMigrateVM.IsChangable = false;
			}
			else
			{
				DontMigrateVM.IsChangable = true;
			}
		}

		private void IsAutoAssign_EntityChanged(object sender, EventArgs args)
		{
			if ((bool)IsAutoAssign.Entity == true)
			{
				RunVMOnSpecificHost.Entity = false;
				RunVMOnSpecificHost.IsChangable = false;
			}
			else
			{
				RunVMOnSpecificHost.IsChangable = true;
			}
		}

		private void UpdateNumOfMonitors()
		{
			bool isLinux = false;
			bool isVnc = false;

			if (OSType.SelectedItem != null)
			{
				VmOsType osType = (VmOsType)OSType.SelectedItem;
				isLinux = DataProvider.IsLinuxOsType(osType);
			}

			if (DisplayProtocol.SelectedItem != null)
			{
				DisplayType displayType = (DisplayType)((EntityModel)DisplayProtocol.SelectedItem).Entity;
				isVnc = displayType == DisplayType.vnc;
			}

			if (isVnc)
			{
				NumOfMonitors.SelectedItem = 1;
			}

			NumOfMonitors.IsChangable = !isLinux && !isVnc;
		}

		public BootSequence BootSequence
		{
			get
			{
				EntityModel firstSelectedItem = (EntityModel)FirstBootDevice.SelectedItem;
				EntityModel secondSelectedItem = (EntityModel)SecondBootDevice.SelectedItem;

				string firstSelectedString = firstSelectedItem.Entity == null ? string.Empty : firstSelectedItem.Entity.ToString();
				string secondSelectedString = secondSelectedItem.Entity == null ? string.Empty : secondSelectedItem.Entity.ToString();

				return (BootSequence)Enum.Parse(typeof(BootSequence), firstSelectedString + secondSelectedString);
			}
			set
			{
				List<BootSequence> items = new List<BootSequence>();
				foreach (char a in value.ToString().ToCharArray())
				{
					items.Add((BootSequence)Enum.Parse(typeof(BootSequence), a.ToString()));
				}

				object firstBootDevice = null;
				foreach (object item in FirstBootDevice.Items)
				{
					EntityModel a = (EntityModel)item;
					if ((BootSequence)a.Entity == Linq.FirstOrDefault(items))
					{
						firstBootDevice = a;
					}
				}
				FirstBootDevice.SelectedItem = firstBootDevice;

				List<EntityModel> secondDeviceOptions = Linq.Cast<EntityModel>(SecondBootDevice.Items);

				if (items.Count > 1)
				{
					BootSequence last = items[items.Count - 1];
					foreach (EntityModel a in secondDeviceOptions)
					{
						if (a.Entity != null && (BootSequence)a.Entity == last)
						{
							SecondBootDevice.SelectedItem = a;
							break;
						}
					}
				}
				else
				{
					foreach (EntityModel a in secondDeviceOptions)
					{
						if (a.Entity == null)
						{
							SecondBootDevice.SelectedItem = a;
							break;
						}
					}
				}
			}
		}

		public void SetDataCenter(UnitVmModel model, List<storage_pool> list)
		{
			if (model.Behavior.SystemTreeSelectedItem != null && model.Behavior.SystemTreeSelectedItem.Type != SystemTreeItemType.System)
			{
				switch (model.Behavior.SystemTreeSelectedItem.Type)
				{
					case SystemTreeItemType.DataCenter:
						storage_pool selectDataCenter = (storage_pool)model.Behavior.SystemTreeSelectedItem.Entity;
						model.DataCenter.Items = new List<storage_pool> { selectDataCenter };
						model.DataCenter.SelectedItem = selectDataCenter;
						model.DataCenter.IsChangable = false;
						model.DataCenter.Info = "Cannot choose Data Center in tree context";
						break;
					case SystemTreeItemType.Cluster:
					case SystemTreeItemType.VMs:
						VDSGroup cluster = (VDSGroup)model.Behavior.SystemTreeSelectedItem.Entity;
						foreach (storage_pool dc in list)
						{
							if (dc.Id.Equals(cluster.storage_pool_id))
							{
								model.DataCenter.Items = new List<storage_pool> { dc };
								model.DataCenter.SelectedItem = dc;
								break;
							}
						}
						model.DataCenter.IsChangable = false;
						model.DataCenter.Info = "Cannot choose Data Center in tree context";
						break;
					case SystemTreeItemType.Host:
						VDS host = (VDS)model.Behavior.SystemTreeSelectedItem.Entity;
						foreach (storage_pool dc in list)
						{
							if (dc.Id.Equals(host.storage_pool_id))
							{
								model.DataCenter.Items = new List<storage_pool> { dc };
								model.DataCenter.SelectedItem = dc;
								model.DataCenter.IsChangable = false;
								model.DataCenter.Info = "Cannot choose Data Center in tree context";
								break;
							}
						}
						break;
					case SystemTreeItemType.Storage:
						storage_domains storage = (storage_domains)model.Behavior.SystemTreeSelectedItem.Entity;
						foreach (storage_pool dc in list)
						{
							if (dc.Id.Equals(storage.storage_pool_id))
							{
								model.DataCenter.Items = new List<storage_pool> { dc };
								model.DataCenter.SelectedItem = dc;
								model.DataCenter.IsChangable = false;
								model.DataCenter.Info = "Cannot choose Data Center in tree context";
								break;
							}
						}
						break;
					default:
						break;
				}
			}
			else
			{
				model.DataCenter.Items = list;
				model.DataCenter.SelectedItem = Linq.FirstOrDefault(list);
			}
		}

		public void SetClusters(UnitVmModel model, List<VDSGroup> clusters, Guid? clusterGuid)
		{
			IVmModelBehavior behavior = model.Behavior;
			if (behavior.SystemTreeSelectedItem != null && behavior.SystemTreeSelectedItem.Type != SystemTreeItemType.System)
			{
				switch (model.Behavior.SystemTreeSelectedItem.Type)
				{
					case SystemTreeItemType.Cluster:
					case SystemTreeItemType.VMs:
						VDSGroup cluster = (VDSGroup)behavior.SystemTreeSelectedItem.Entity;
						model.Cluster.Items = new List<VDSGroup>() { cluster };
						model.Cluster.SelectedItem = cluster;
						model.Cluster.IsChangable = false;
						model.Cluster.Info = "Cannot choose Cluster in tree context";
						break;
					case SystemTreeItemType.Host:
						VDS host = (VDS)behavior.SystemTreeSelectedItem.Entity;
						foreach (VDSGroup iterCluster in clusters)
						{
							if (iterCluster.ID.Equals(host.vds_group_id))
							{
								model.Cluster.Items = new List<VDSGroup>() { iterCluster };
								model.Cluster.SelectedItem = iterCluster;
								model.Cluster.IsChangable = false;
								model.Cluster.Info = "Cannot choose Cluster in tree context";
								break;
							}
						}
						break;
					default:
						model.Cluster.Items = clusters;
						if (clusterGuid == null)
						{
							model.Cluster.SelectedItem = Linq.FirstOrDefault(clusters);
						}
						else
						{
							model.Cluster.SelectedItem = Linq.FirstOrDefault(clusters, new Linq.ClusterPredicate((Guid)clusterGuid));
						}
						break;
				}
			}
			else
			{
				
				model.Cluster.Items = clusters;
				if (clusterGuid == null)
				{
					model.Cluster.SelectedItem = Linq.FirstOrDefault(clusters);
				}
				else
				{
					model.Cluster.SelectedItem = Linq.FirstOrDefault(clusters, new Linq.ClusterPredicate((Guid)clusterGuid));
				}
			}
		}

		public bool Validate()
		{
			DataCenter.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });
			Cluster.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });
			MemSize.ValidateEntity(new IValidation[] { new ByteSizeValidation() });
			MinAllocatedMemory.ValidateEntity(new IValidation[] { new ByteSizeValidation() });
			OSType.ValidateSelectedItem(new[] { new NotEmptyValidation() });

			Description.ValidateEntity(new IValidation[] { new AsciiOrNoneValidation() });
			if (OSType.IsValid)
			{
				VmOsType osType = (VmOsType)OSType.SelectedItem;

				string nameExpr;
				string nameMsg;
				if (DataProvider.IsWindowsOsType(osType))
				{
					nameExpr = @"^[0-9a-zA-Z-_]{1," + WINDOWS_VM_NAME_MAX_LIMIT + "}$";
					nameMsg = "Name must contain only alphanumeric characters. Maximum length: " + WINDOWS_VM_NAME_MAX_LIMIT + ".";
				}
				else
				{
					nameExpr = @"^[-\w]{1," + NON_WINDOWS_VM_NAME_MAX_LIMIT + "}$";
					nameMsg = "Name cannot contain blanks or special characters. Maximum length: " + NON_WINDOWS_VM_NAME_MAX_LIMIT +
							  ".";
				}

				Name.ValidateEntity(
					new IValidation[]
					{
						new NotEmptyValidation(),
						new RegexValidation
						{
							Expression = nameExpr,
							Message = nameMsg
						}
					});


				bool is64OsType = (osType == VmOsType.Other || osType == VmOsType.OtherLinux || DataProvider.Is64bitOsType(osType));
				int maxMemSize = is64OsType ? _MaxMemSize64 : _MaxMemSize32;

				ValidateMemorySize(MemSize, maxMemSize, _minMemSize);
				if(!(this.Behavior is TemplateVmModelBehavior))
				{
					// Minimum 'Physical Memory Guaranteed' is 1MB
					ValidateMemorySize(MinAllocatedMemory, (int)MemSize.Entity, 1);
				}
			}

			if ((bool)IsAutoAssign.Entity == false)
			{
				DefaultHost.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });
			}
			else
			{
				DefaultHost.IsValid = true;
			}

			VmTemplate template = (VmTemplate)Template.SelectedItem;
			storage_domains storageDomain = (storage_domains)StorageDomain.SelectedItem;

			Template.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });

			StorageDomain.IsValid = true;
			if (template != null && !template.Id.Equals(Guid.Empty) && storageDomain == null)
			{
				StorageDomain.IsValid = false;
				StorageDomain.InvalidityReasons.Add("Storage Domain must be specified.");
			}


			CdImage.IsValid = true;
			if (CdImage.IsChangable)
			{
				CdImage.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });
			}

			Kernel_path.IsValid = true;
			Kernel_parameters.IsValid = true;
			Initrd_path.IsValid = true;
			if (Kernel_path.Entity == null)
			{
				Kernel_path.Entity = "";
			}
			if (Kernel_parameters.Entity == null)
			{
				Kernel_parameters.Entity = "";
			}
			if (Initrd_path.Entity == null)
			{
				Initrd_path.Entity = "";
			}

			if (isLinux_Unassign_UnknownOS && ((((string)Kernel_parameters.Entity).Length > 0 || ((string)Initrd_path.Entity).Length > 0) && ((string)Kernel_path.Entity).Length == 0))
			{
				int count = 0;
				string msg = "When ";
				if (((string)Kernel_parameters.Entity).Length > 0)
				{
					Kernel_parameters.IsValid = false;
					msg += "a kernel parameter argument ";
					count++;
				}
				if (((string)Initrd_path.Entity).Length > 0)
				{
					Initrd_path.IsValid = false;
					if (count == 1)
					{
						msg += "or ";
					}
					msg += "an initrd path ";
				}
				msg += "is used, kernel path must be non-empty";

				Kernel_path.IsValid = false;
				Initrd_path.InvalidityReasons.Add(msg);
				Kernel_parameters.InvalidityReasons.Add(msg);
				Kernel_path.InvalidityReasons.Add(msg);
			}

			CustomProperties.ValidateEntity(new IValidation[] { new CustomPropertyValidation(CustomPropertiesKeysList) });


			IsGeneralTabValid =
				IsFirstRunTabValid =
				IsDisplayTabValid =
				IsAllocationTabValid =
				IsBootSequenceTabValid = true;


			IsGeneralTabValid =
				Name.IsValid
				&& Description.IsValid
				&& DataCenter.IsValid
				&& Template.IsValid
				&& Cluster.IsValid
				&& MemSize.IsValid
				&& MinAllocatedMemory.IsValid;

			IsFirstRunTabValid = Domain.IsValid && TimeZone.IsValid;
			IsDisplayTabValid = UsbPolicy.IsValid && NumOfMonitors.IsValid;
			IsHostTabValid = DefaultHost.IsValid;
			IsAllocationTabValid = StorageDomain.IsValid && MinAllocatedMemory.IsValid;
			IsBootSequenceTabValid = CdImage.IsValid && Kernel_path.IsValid;
			IsCustomPropertiesTabValid = CustomProperties.IsValid;

			return Name.IsValid
				   && Description.IsValid
				   && DataCenter.IsValid
				   && StorageDomain.IsValid
				   && Template.IsValid
				   && Cluster.IsValid
				   && DefaultHost.IsValid
				   && MemSize.IsValid
				   && MinAllocatedMemory.IsValid
				   && NumOfMonitors.IsValid
				   && Domain.IsValid
				   && UsbPolicy.IsValid
				   && TimeZone.IsValid
				   && OSType.IsValid
				   && CdImage.IsValid
				   && Kernel_path.IsValid
				   && CustomProperties.IsValid
				   && behavior.Validate();
		}

		private void ValidateMemorySize(EntityModel memorySizeEntityModel, int maxMemSize, int minMemSize)
		{
			bool isValid = false;

			int memSize = (int)memorySizeEntityModel.Entity;

			if (memSize == 0)
			{
				memorySizeEntityModel.InvalidityReasons.Add("Memory size is between " + minMemSize + " MB and " + maxMemSize + " MB");
			}
			else if (memSize > maxMemSize)
			{
				memorySizeEntityModel.InvalidityReasons.Add("Maximum memory size is " + maxMemSize + " MB.");
			}
			else if (memSize < minMemSize)
			{
				memorySizeEntityModel.InvalidityReasons.Add("Minimum memory size is " + minMemSize + " MB.");
			}
			else
			{
				isValid = true;
			}

			memorySizeEntityModel.IsValid = isValid;
		}
	}



	public class NewVmModelBehavior : IVmModelBehavior
	{
		public override void Initialize(SystemTreeItemModel systemTreeSelectedItem)
		{
			base.Initialize(systemTreeSelectedItem);
			AsyncDataProvider.GetDataCenterList(new AsyncQuery(Model,
				(target, returnValue) =>
					{
						UnitVmModel model = (UnitVmModel) target;

						//Show only data centers with status up.
						List<storage_pool> list = new List<storage_pool>();
						foreach (storage_pool a in (List<storage_pool>) (valueObjectEnumerableList) returnValue)
						{
							if (a.status == StoragePoolStatus.Up)
							{
								list.Add(a);
							}
						}
						model.SetDataCenter(model, list);
					}, Model.Hash));
		}

		public override void DataCenter_SelectedItemChanged()
		{
			storage_pool dataCenter = (storage_pool)Model.DataCenter.SelectedItem;

			Model.IsHostAvailable = dataCenter.storage_pool_type != StorageType.LOCALFS;

			AsyncDataProvider.GetClusterList(new AsyncQuery(new object[] { this, Model },
				(target, returnValue) =>
				{
					object[] array = (object[])target;
					NewVmModelBehavior behavior = (NewVmModelBehavior)array[0];
					UnitVmModel model = (UnitVmModel)array[1];

					List<VDSGroup> clusters = (List<VDSGroup>)returnValue;

					model.SetClusters(model, clusters, null);
					behavior.InitTemplate();
					behavior.InitCdImage();
				}, Model.Hash),
				dataCenter.Id
			);
		}

		public override void Template_SelectedItemChanged()
		{
			VmTemplate template = (VmTemplate)Model.Template.SelectedItem;

			if (template != null)
			{
				//Copy VM parameters from template.
				Model.OSType.SelectedItem = template.os;
				Model.NumOfSockets.Entity = template.num_of_sockets;
				Model.TotalCPUCores.Entity = template.num_of_cpus;
				Model.NumOfMonitors.SelectedItem = template.num_of_monitors;
				Model.Domain.SelectedItem = template.domain;
				Model.MemSize.Entity = template.mem_size_mb;
				Model.UsbPolicy.SelectedItem = template.usb_policy;
				Model.BootSequence = template.default_boot_sequence;
				Model.IsHighlyAvailable.Entity = template.auto_startup;


				Model.CdImage.IsChangable = !String.IsNullOrEmpty(template.iso_path);
				if (Model.CdImage.IsChangable)
				{
					Model.CdImage.SelectedItem = template.iso_path;
				}


				if (!String.IsNullOrEmpty(template.time_zone))
				{
					//Patch! Create key-value pair with a right key.
					Model.TimeZone.SelectedItem = new KeyValuePair<string, string>(template.time_zone, String.Empty);

					UpdateTimeZone();
				}
				else
				{
					UpdateDefaultTimeZone();
				}

				// Update domain list
				UpdateDomain();

				List<VDSGroup> clusters = (List<VDSGroup>)Model.Cluster.Items;
				VDSGroup selectCluster = (VDSGroup)Linq.FirstOrDefault(clusters, new Linq.ClusterPredicate(template.vds_group_id));

				Model.Cluster.SelectedItem = selectCluster ?? Linq.FirstOrDefault(clusters);


				// Update display protocol selected item
				EntityModel displayProtocol = null;
				bool isFirst = true;
				foreach (object item in Model.DisplayProtocol.Items)
				{
					EntityModel a = (EntityModel)item;
					if (isFirst)
					{
						displayProtocol = a;
						isFirst = false;
					}
					DisplayType dt = (DisplayType)a.Entity;
					if (dt == template.default_display_type)
					{
						displayProtocol = a;
						break;
					}
				}
				Model.DisplayProtocol.SelectedItem = displayProtocol;


				//By default, take kernel params from template.
				Model.Kernel_path.Entity = template.kernel_url;
				Model.Kernel_parameters.Entity = template.kernel_params;
				Model.Initrd_path.Entity = template.initrd_url;


				if (!template.Id.Equals(Guid.Empty))
				{
					Model.StorageDomain.IsChangable = true;
					Model.Provisioning.IsChangable = true;

					Model.IsBlankTemplate = false;
					InitDisks();
				}
				else
				{
					Model.StorageDomain.IsChangable = false;
					Model.Provisioning.IsChangable = false;

					Model.IsBlankTemplate = true;
					Model.IsDisksAvailable = false;
					Model.Disks = null;
				}


				InitPriority(template.priority);
				InitStorageDomains();
				UpdateMinAllocatedMemory();
			}
		}

		public override void Cluster_SelectedItemChanged()
		{
			UpdateDefaultHost();
			UpdateIsCustomPropertiesAvailable();
			UpdateMinAllocatedMemory();
			UpdateNumOfSockets();
		}

		public override void DefaultHost_SelectedItemChanged()
		{
			UpdateCdImage();
		}

		public override void Provisioning_SelectedItemChanged()
		{
			UpdateIsDisksAvailable();
			InitStorageDomains();
		}

		public override void UpdateMinAllocatedMemory()
		{
			VDSGroup cluster = (VDSGroup)Model.Cluster.SelectedItem;
			if (cluster == null)
			{
				return;
			}

			double overCommitFactor = 100.0 / cluster.max_vds_memory_over_commit;
			Model.MinAllocatedMemory.Entity = (int)((int)Model.MemSize.Entity * overCommitFactor);
		}

		private void InitTemplate()
		{
			storage_pool dataCenter = (storage_pool)Model.DataCenter.SelectedItem;

			//Filter according to system tree selection.
			if (SystemTreeSelectedItem != null && SystemTreeSelectedItem.Type == SystemTreeItemType.Storage)
			{
				storage_domains storage = (storage_domains)SystemTreeSelectedItem.Entity;

				AsyncDataProvider.GetTemplateListByDataCenter(new AsyncQuery(new object[] { this, storage },
					(target1, returnValue1) =>
					{
						object[] array1 = (object[])target1;
						NewVmModelBehavior behavior1 = (NewVmModelBehavior)array1[0];
						storage_domains storage1 = (storage_domains)array1[1];

						AsyncDataProvider.GetTemplateListByStorage(new AsyncQuery(new object[] { behavior1, returnValue1 },
							(target2, returnValue2) =>
							{
								object[] array2 = (object[])target2;
								NewVmModelBehavior behavior2 = (NewVmModelBehavior)array2[0];
								List<VmTemplate> templatesByDataCenter = (List<VmTemplate>)array2[1];

								List<VmTemplate> templatesByStorage = (List<VmTemplate>)returnValue2;

								VmTemplate blankTemplate = Linq.FirstOrDefault(templatesByDataCenter, new Linq.TemplatePredicate(Guid.Empty));
								if (blankTemplate != null)
								{
									templatesByStorage.Insert(0, blankTemplate);
								}

								behavior2.PostInitTemplate((List<VmTemplate>)returnValue2);
							}),
							storage1.id
						);
					}, Model.Hash),
					dataCenter.Id
				);
			}
			else
			{
				AsyncDataProvider.GetTemplateListByDataCenter(new AsyncQuery(this,
					(target, returnValue) =>
					{
						NewVmModelBehavior behavior = (NewVmModelBehavior)target;

						behavior.PostInitTemplate((List<VmTemplate>)returnValue);
					}, Model.Hash),
					dataCenter.Id
				);
			}
		}

		private void PostInitTemplate(List<VmTemplate> templates)
		{
			//If there was some template selected before, try select it again.
			VmTemplate oldTemplate = (VmTemplate)Model.Template.SelectedItem;

			Model.Template.Items = templates;

			Model.Template.SelectedItem = Linq.FirstOrDefault(templates,
				oldTemplate != null
					? new Linq.TemplatePredicate(oldTemplate.Id)
					: new Linq.TemplatePredicate(Guid.Empty));

			UpdateIsDisksAvailable();
		}

		public void InitCdImage()
		{
			UpdateCdImage();
		}

		private void InitDisks()
		{
			VmTemplate template = (VmTemplate)Model.Template.SelectedItem;

			AsyncDataProvider.GetTemplateDiskList(new AsyncQuery(Model,
				(target, returnValue) =>
				{
					UnitVmModel model = (UnitVmModel)target;

					List<DiskImage> disks = (List<DiskImage>)(valueObjectEnumerableList)returnValue;
					disks.Sort(new Linq.DiskByInternalDriveMappingComparer());

					List<DiskModel> list = new List<DiskModel>();
					foreach (DiskImage a in disks)
					{
						DiskModel diskModel = new DiskModel();
						diskModel.IsNew = true;
						diskModel.Name = a.internal_drive_mapping;
						diskModel.Size =
							new EntityModel
							{
								Entity = a.SizeInGigabytes
							};
						diskModel.VolumeType =
							new ListModel
							{
								//BZ#716869: enable only preallocate in case its selected
								Items = (a.volume_type == VolumeType.Preallocated ? new List<VolumeType> {VolumeType.Preallocated} : DataProvider.GetVolumeTypeList()),
								SelectedItem = a.volume_type
							};
						list.Add(diskModel);
					}

					model.Disks = list;

					UpdateIsDisksAvailable();
				}, Model.Hash),
				template.Id
			);
		}

		public void UpdateIsDisksAvailable()
		{
			bool provisioning = (bool)((EntityModel)Model.Provisioning.SelectedItem).Entity;

			Model.IsDisksAvailable = provisioning && Model.Disks != null;
		}

		private void InitStorageDomains()
		{
			VmTemplate template = (VmTemplate)Model.Template.SelectedItem;

			if (template != null && !template.Id.Equals(Guid.Empty))
			{
				bool provisioning = (bool)((EntityModel)Model.Provisioning.SelectedItem).Entity;

				if (!provisioning)
				{
					AsyncDataProvider.GetStorageDomainListByTemplate(new AsyncQuery(this,
						(target, returnValue) =>
						{
							NewVmModelBehavior behavior = (NewVmModelBehavior)target;

							List<storage_domains> storageDomains = (List<storage_domains>)(valueObjectEnumerableList)returnValue;
							behavior.PostInitStorageDomains(storageDomains);
						}, Model.Hash),
						template.Id
					);
				}
				else
				{
					AsyncDataProvider.GetStorageDomainListByTemplate(new AsyncQuery(this,
						(target, returnValue) =>
						{
							List<storage_domains> storageDomains = (List<storage_domains>)(valueObjectEnumerableList)returnValue;

							bool isStorageDomainActive = false;
							foreach (storage_domains storageDomain in storageDomains)
							{
								isStorageDomainActive = storageDomain.status == StorageDomainStatus.Active;

								if (!isStorageDomainActive)
								{
									break;
								}
							}

							if (isStorageDomainActive)
							{
								NewVmModelBehavior behavior = (NewVmModelBehavior)target;
								storage_pool dataCenter = (storage_pool)behavior.Model.DataCenter.SelectedItem;
								behavior.InitStorageDomains(dataCenter);
							}
						}, Model.Hash),
						template.Id
					);
				}
			}
			else
			{
				Model.StorageDomain.Items = new List<storage_domains>();
				Model.StorageDomain.SelectedItem = null;
				Model.StorageDomain.IsChangable = false;
			}
		}

		public void PostInitStorageDomains(List<storage_domains> storageDomains)
		{
			// filter only the Active storage domains (Active regarding the relevant storage pool).
			List<storage_domains> list = new List<storage_domains>();
			foreach (storage_domains a in storageDomains)
			{
				if (a.status.HasValue && a.status.Value == StorageDomainStatus.Active)
				{
					list.Add(a);
				}
			}

			//Filter according to system tree selection.
			if (SystemTreeSelectedItem != null && SystemTreeSelectedItem.Type == SystemTreeItemType.Storage)
			{
				storage_domains selectStorage = (storage_domains)SystemTreeSelectedItem.Entity;
				storage_domains sd = Linq.FirstOrDefault(list, new Linq.StoragePredicate(selectStorage.id));

				Model.StorageDomain.Items = new List<storage_domains> { sd };
				Model.StorageDomain.SelectedItem = sd;
				Model.StorageDomain.IsChangable = false;
			}
			else
			{
				Model.StorageDomain.Items = list;
				Model.StorageDomain.SelectedItem = Linq.FirstOrDefault(list);
				Model.StorageDomain.IsChangable = true;
			}
		}

		public void InitStorageDomains(storage_pool dataCenter)
		{
			AsyncDataProvider.GetStorageDomainList(new AsyncQuery(this,
				(target, returnValue) =>
				{
					NewVmModelBehavior behavior = (NewVmModelBehavior)target;

					List<storage_domains> storageDomains = new List<storage_domains>();
					foreach (storage_domains a in (List<storage_domains>)(valueObjectEnumerableList)returnValue)
					{
						if (a.storage_domain_type == StorageDomainType.Data || a.storage_domain_type == StorageDomainType.Master)
						{
							storageDomains.Add(a);
						}
					}

					behavior.PostInitStorageDomains(storageDomains);
				}, Model.Hash),
				dataCenter.Id
			);
		}
	}


	public class ExistingVmModelBehavior : IVmModelBehavior
	{
		protected readonly VM vm;

		public ExistingVmModelBehavior(VM vm)
		{
			this.vm = vm;
		}

		public override void Initialize(SystemTreeItemModel systemTreeSelectedItem)
		{
			base.Initialize(systemTreeSelectedItem);
			AsyncDataProvider.GetDataCenterById(new AsyncQuery(Model,
				(target, returnValue) =>
				{
					UnitVmModel model = (UnitVmModel)target;
					if (returnValue != null)
					{
						storage_pool dataCenter = (storage_pool) (valueObjectSingle) returnValue;
						List<storage_pool> list = new List<storage_pool>() {dataCenter};
						model.SetDataCenter(model, list);
						model.DataCenter.IsChangable = false;
					}
					else
					{
						ExistingVmModelBehavior behavior = (ExistingVmModelBehavior) model.Behavior;
						VM currentVm = behavior.vm;
						VDSGroup cluster = new VDSGroup() { ID = currentVm.vds_group_id, name = currentVm.vds_group_name };
						model.Cluster.Items = new List<VDSGroup> {cluster};
						model.Cluster.SelectedItem = cluster;
						behavior.InitTemplate();
						behavior.InitCdImage();
					}
				}, Model.Hash),
				vm.storage_pool_id
			);
		}

		public override void DataCenter_SelectedItemChanged()
		{
			storage_pool dataCenter = (storage_pool)Model.DataCenter.SelectedItem;

			Model.IsHostAvailable = dataCenter.storage_pool_type != StorageType.LOCALFS;

			AsyncDataProvider.GetClusterList(new AsyncQuery(new object[] { this, Model },
				(target, returnValue) =>
				{
					object[] array = (object[])target;
					ExistingVmModelBehavior behavior = (ExistingVmModelBehavior)array[0];
					UnitVmModel model = (UnitVmModel)array[1];
					VM vm = ((ExistingVmModelBehavior) array[0]).vm;

					List<VDSGroup> clusters = (List<VDSGroup>)returnValue;
					model.SetClusters(model, clusters, vm.vds_group_id.Value);


					behavior.InitTemplate();
					behavior.InitCdImage();
				}, Model.Hash),
				dataCenter.Id
			);
		}

		public override void Template_SelectedItemChanged()
		{
			//This method will be called even if a VM created from Blank template.

			//Update model state according to VM properties.
			Model.Name.Entity = vm.vm_name;
			Model.Description.Entity = vm.vm_description;
			Model.MemSize.Entity = vm.vm_mem_size_mb;
			Model.MinAllocatedMemory.Entity = vm.MinAllocatedMem;
			Model.OSType.SelectedItem = vm.vm_os;
			Model.Domain.SelectedItem = vm.vm_domain;
			Model.UsbPolicy.SelectedItem = vm.usb_policy;
			Model.NumOfMonitors.SelectedItem = vm.num_of_monitors;
			Model.BootSequence = vm.default_boot_sequence;
			Model.IsHighlyAvailable.Entity = vm.auto_startup;

			Model.NumOfSockets.Entity = vm.num_of_sockets;
			Model.NumOfSockets.IsChangable = !vm.isStatusUp();

			Model.TotalCPUCores.Entity = vm.num_of_cpus;
			Model.TotalCPUCores.IsChangable = !vm.isStatusUp();

			Model.IsStateless.Entity = vm.is_stateless;
			Model.IsStateless.IsAvailable = vm.VmPoolId == null;

			Model.Kernel_parameters.Entity = vm.kernel_params;
			Model.Kernel_path.Entity = vm.kernel_url;
			Model.Initrd_path.Entity = vm.initrd_url;

			Model.CustomProperties.Entity = vm.CustomProperties;

			if (vm.is_initialized)
			{
				Model.TimeZone.IsChangable = false;
				Model.TimeZone.ChangeProhibitionReasons.Add("Time Zone cannot be change since the Virtual Machine was booted at the first time.");
			}

			Model.TimeZone.SelectedItem = new KeyValuePair<string, string>(vm.time_zone, String.Empty);
			UpdateTimeZone();

			// Update domain list
			UpdateDomain();

			switch (vm.MigrationSupport)
			{
				case MigrationSupport.PINNED_TO_HOST:
					Model.RunVMOnSpecificHost.Entity = true;
					break;
				case MigrationSupport.IMPLICITLY_NON_MIGRATABLE:
					Model.DontMigrateVM.Entity = true;
					break;
			}

			//Storage domain and provisioning are not available for an existing VM.
			Model.StorageDomain.IsChangable = false;
			Model.Provisioning.IsAvailable = false;

			//If a VM has at least one disk, present its storage domain.
			AsyncDataProvider.GetVmDiskList(new AsyncQuery(this,
				(target, returnValue) =>
				{
					ExistingVmModelBehavior behavior = (ExistingVmModelBehavior)target;

					List<DiskImage> disks = new List<DiskImage>();
					IEnumerable disksEnumerable = (IEnumerable)returnValue;
					IEnumerator disksIterator = disksEnumerable.GetEnumerator();
					while (disksIterator.MoveNext())
					{
						disks.Add((DiskImage)disksIterator.Current);
					}

					if (disks.Count > 0)
					{
						behavior.InitStorageDomains(disks[0].storage_id);
					}
				}, Model.Hash),
				vm.vm_guid
			);


			//Select display protocol.
			foreach (object item in Model.DisplayProtocol.Items)
			{
				EntityModel model = (EntityModel)item;
				DisplayType displayType = (DisplayType)model.Entity;

				if (displayType == vm.default_display_type)
				{
					Model.DisplayProtocol.SelectedItem = item;
					break;
				}
			}


			InitPriority(vm.priority);
		}

		public override void Cluster_SelectedItemChanged()
		{
			UpdateDefaultHost();
			UpdateIsCustomPropertiesAvailable();
			UpdateNumOfSockets();
		}
		
		protected override void ChangeDefualtHost()
		{
			base.ChangeDefualtHost();
			
			if (vm.dedicated_vm_for_vds != null)
			{
				Guid vdsId = vm.dedicated_vm_for_vds.Value;
				if (Model.DefaultHost.Items != null)
				{
					Model.DefaultHost.SelectedItem = Linq.FirstOrDefault((List<VDS>)Model.DefaultHost.Items,
																		 new Linq.HostPredicate(vdsId));
				}
				Model.IsAutoAssign.Entity = false;
			}
			else
			{
				Model.IsAutoAssign.Entity = true;
			}
		}

		public override void DefaultHost_SelectedItemChanged()
		{
			UpdateCdImage();
		}

		public override void Provisioning_SelectedItemChanged()
		{
		}

		public override void UpdateMinAllocatedMemory()
		{
			VDSGroup cluster = (VDSGroup)Model.Cluster.SelectedItem;

			if (cluster == null)
			{
				return;
			}

			if ((int)Model.MemSize.Entity < vm.vm_mem_size_mb)
			{
				double overCommitFactor = 100.0 / cluster.max_vds_memory_over_commit;
				Model.MinAllocatedMemory.Entity = (int)((int)Model.MemSize.Entity * overCommitFactor);
			}
			else
			{
				Model.MinAllocatedMemory.Entity = vm.MinAllocatedMem;
			}
		}

		public void InitTemplate()
		{
			AsyncDataProvider.GetTemplateById(new AsyncQuery(Model,
				(target, returnValue) =>
				{
					UnitVmModel model = (UnitVmModel)target;
					VmTemplate template = (VmTemplate)(valueObjectSingle)returnValue;

					model.Template.Items = new List<VmTemplate> { template };
					model.Template.SelectedItem = template;
					model.Template.IsChangable = false;
				}, Model.Hash),
				vm.vmt_guid
			);
		}

		public void InitCdImage()
		{
			Model.CdImage.SelectedItem = vm.iso_path;
			Model.CdImage.IsChangable = !String.IsNullOrEmpty(vm.iso_path);

			UpdateCdImage();
		}

		public void InitStorageDomains(Guid? storageDomainId)
		{
			if (storageDomainId == null)
			{
				return;
			}

			AsyncDataProvider.GetStorageDomainById(new AsyncQuery(Model,
				(target, returnValue) =>
				{
					UnitVmModel model = (UnitVmModel)target;
					storage_domains storageDomain = (storage_domains)returnValue;

					model.StorageDomain.Items = new List<storage_domains> { storageDomain };
					model.StorageDomain.SelectedItem = storageDomain;
				}, Model.Hash),
				storageDomainId.Value
			);
		}
	}

	public class NewTemplateVmModelBehavior : IVmModelBehavior
	{
		private readonly VM vm;

		public NewTemplateVmModelBehavior(VM vm)
		{
			this.vm = vm;
		}

		public override void Initialize(SystemTreeItemModel systemTreeSelectedItem)
		{
			base.Initialize(systemTreeSelectedItem);
			Model.Template.IsChangable = false;

			AsyncDataProvider.GetDataCenterById(new AsyncQuery(this,
				(target, returnValue) =>
				{
					NewTemplateVmModelBehavior behavior = (NewTemplateVmModelBehavior)target;

					storage_pool dataCenter = (storage_pool)(valueObjectSingle)returnValue;

					if (dataCenter == null)
					{
						DisableNewTemplateModel("Data Center is not accessible.");
					}
					else
					{
						behavior.Model.DataCenter.Items = new List<storage_pool> { dataCenter };
						behavior.Model.DataCenter.SelectedItem = dataCenter;
						behavior.Model.DataCenter.IsChangable = false;
					}
				}, Model.Hash),
				vm.storage_pool_id
			);
		}
		
		public override void DataCenter_SelectedItemChanged()
		{
			storage_pool dataCenter = (storage_pool)Model.DataCenter.SelectedItem;

			Model.IsHostAvailable = dataCenter.storage_pool_type != StorageType.LOCALFS;

			AsyncDataProvider.GetClusterList(new AsyncQuery(new object[] { this, Model },
				(target, returnValue) =>
				{
					object[] array = (object[])target;
					NewTemplateVmModelBehavior behavior = (NewTemplateVmModelBehavior)array[0];
					UnitVmModel model = (UnitVmModel)array[1];

					List<VDSGroup> clusters = (List<VDSGroup>)returnValue;

					model.SetClusters(model, clusters, vm.vds_group_id.Value);
					behavior.InitTemplate();
				}, Model.Hash),
				dataCenter.Id
			);

			//If a VM has at least one disk, present its storage domain.
			AsyncDataProvider.GetVmDiskList(new AsyncQuery(this,
				(target, returnValue) =>
				{
					NewTemplateVmModelBehavior behavior = (NewTemplateVmModelBehavior)target;

					List<DiskImage> disks = new List<DiskImage>();
					IEnumerable disksEnumerable = (IEnumerable)returnValue;
					IEnumerator disksIterator = disksEnumerable.GetEnumerator();
					while (disksIterator.MoveNext())
					{
						disks.Add((DiskImage)disksIterator.Current);
					}

					if (disks.Count == 0)
					{
						behavior.DisableNewTemplateModel("Cannot create Template. VM has no disks.");
					}
					else
					{
						behavior.InitStorageDomains(disks[0].storage_id);
					}
				}, Model.Hash),
				vm.vm_guid
			);
		}

		public override void Template_SelectedItemChanged()
		{
		}

		public override void Cluster_SelectedItemChanged()
		{
		}

		public override void DefaultHost_SelectedItemChanged()
		{
		}

		public override void Provisioning_SelectedItemChanged()
		{
		}

		public override void UpdateMinAllocatedMemory()
		{
		}

		private void InitTemplate()
		{
			//Update model state according to VM properties.
			Model.MemSize.Entity = this.vm.vm_mem_size_mb;
			Model.OSType.SelectedItem = this.vm.vm_os;
			Model.Domain.SelectedItem = this.vm.vm_domain;
			Model.UsbPolicy.SelectedItem = this.vm.usb_policy;
			Model.NumOfMonitors.SelectedItem = this.vm.num_of_monitors;
			Model.BootSequence = this.vm.default_boot_sequence;
			Model.NumOfSockets.Entity = this.vm.num_of_sockets;
			Model.TotalCPUCores.Entity = this.vm.num_of_cpus;
			Model.IsStateless.Entity = this.vm.is_stateless;

			if (!String.IsNullOrEmpty(this.vm.time_zone))
			{
				Model.TimeZone.SelectedItem = new KeyValuePair<string, string>(this.vm.time_zone, String.Empty);
				UpdateTimeZone();
			}
			else
			{
				UpdateDefaultTimeZone();
			}

			// Update domain list
			UpdateDomain();

			Model.StorageDomain.IsChangable = true;
			Model.Provisioning.IsAvailable = false;

			//Select display protocol.
			foreach (object item in Model.DisplayProtocol.Items)
			{
				EntityModel model = (EntityModel)item;
				DisplayType displayType = (DisplayType)model.Entity;

				if (displayType == this.vm.default_display_type)
				{
					Model.DisplayProtocol.SelectedItem = item;
					break;
				}
			}

			InitPriority(this.vm.priority);
		}

		public void InitStorageDomains(Guid? storageDomainId)
		{
			if (storageDomainId == null)
			{
				return;
			}

			AsyncDataProvider.GetStorageDomainById(new AsyncQuery(this,
				(target, returnValue) =>
				{
					NewTemplateVmModelBehavior behavior = (NewTemplateVmModelBehavior)target;
					storage_domains currentStorageDomain = (storage_domains)returnValue;

					behavior.PostInitStorageDomains((storage_pool)behavior.Model.DataCenter.SelectedItem, currentStorageDomain);
				}, Model.Hash),
				storageDomainId.Value
			);
		}

		public void PostInitStorageDomains(storage_pool dataCenter, storage_domains storage)
		{
			AsyncDataProvider.GetStorageDomainList(new AsyncQuery(new object[] { this, storage },
				(target, returnValue) =>
				{
					object[] array = (object[])target;
					NewTemplateVmModelBehavior behavior = (NewTemplateVmModelBehavior)array[0];
					storage_domains currentStorageDomain = (storage_domains)array[1];
					storage_domains vmStorageDomain = null;
					List<storage_domains> activeStorageDomainList = new List<storage_domains>();

					// Filter non-active storage domains
					foreach (storage_domains storageDomain in (List<storage_domains>)(valueObjectEnumerableList)returnValue)
					{
						if (storageDomain.status == StorageDomainStatus.Active &&
							(storageDomain.storage_domain_type == StorageDomainType.Data || storageDomain.storage_domain_type == StorageDomainType.Master))
						{
							if (currentStorageDomain.id.Equals(storageDomain.id))
							{
								vmStorageDomain = storageDomain;
							}
							activeStorageDomainList.Add(storageDomain);
						}
					}

					if (activeStorageDomainList.Count > 0)
					{
						// If current storage domain is active, set it as selected item
						if (vmStorageDomain != null)
						{
							if (SystemTreeSelectedItem != null && SystemTreeSelectedItem.Type == SystemTreeItemType.Storage)
							{
								storage_domains selectStorage = (storage_domains)SystemTreeSelectedItem.Entity;

								storage_domains s = Linq.FirstOrDefault(activeStorageDomainList, new Linq.StoragePredicate(selectStorage.id));
								behavior.Model.StorageDomain.Items = new List<storage_domains> { s };
								behavior.Model.StorageDomain.IsChangable = false;
								behavior.Model.StorageDomain.SelectedItem = s;
							}
							else
							{
								behavior.Model.StorageDomain.Items = activeStorageDomainList;
								behavior.Model.StorageDomain.IsChangable = true;
								behavior.Model.StorageDomain.SelectedItem = vmStorageDomain;
							}
						}
						else
						{
							behavior.DisableNewTemplateModel("VM's Storage Domain (" + currentStorageDomain.storage_name + ") is not accessible.");
							return;
						}
					}
				}, Model.Hash),
				dataCenter.Id
			);
		}

		private void DisableNewTemplateModel(String errMessage)
		{
			Model.IsValid = false;
			Model.Message = errMessage;
			Model.Name.IsChangable = false;
			Model.Description.IsChangable = false;
			Model.Cluster.IsChangable = false;
			Model.StorageDomain.IsChangable = false;
			Model.IsTemplatePublic.IsChangable = false;
		}

		public override bool Validate()
		{
			Model.StorageDomain.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });

			return base.Validate()
			       && Model.StorageDomain.IsValid;
		}
	}

	public class TemplateVmModelBehavior : IVmModelBehavior
	{
		private readonly VmTemplate template;

		public TemplateVmModelBehavior(VmTemplate template)
		{
			this.template = template;
		}

		public override void Initialize(SystemTreeItemModel systemTreeSelectedItem)
		{
			base.Initialize(systemTreeSelectedItem);
			Model.Template.IsChangable = false;
			Model.Provisioning.IsChangable = false;
			Model.StorageDomain.IsChangable = false;

			if (template.storage_pool_id != null && !template.storage_pool_id.Value.Equals(Guid.Empty))
			{
				AsyncDataProvider.GetDataCenterById(new AsyncQuery(Model,
					(target, returnValue) =>
					{
						UnitVmModel model = (UnitVmModel)target;
						storage_pool dataCenter = (storage_pool)(valueObjectSingle)returnValue;
						model.SetDataCenter(model, new List<storage_pool> { dataCenter });
						model.DataCenter.IsChangable = false;
					}, Model.Hash),
					template.storage_pool_id.Value
				);
			}
		}

		public override void DataCenter_SelectedItemChanged()
		{
			storage_pool dataCenter = (storage_pool)Model.DataCenter.SelectedItem;

			Model.IsHostAvailable = dataCenter.storage_pool_type != StorageType.LOCALFS;

			AsyncDataProvider.GetClusterList(new AsyncQuery(new object[] { this, Model },
				(target, returnValue) =>
				{
					object[] array = (object[])target;
					TemplateVmModelBehavior behavior = (TemplateVmModelBehavior)array[0];
					UnitVmModel model = (UnitVmModel)array[1];
					VmTemplate vmTemplate = ((TemplateVmModelBehavior)(array[0])).template;
					List<VDSGroup> clusters = (List<VDSGroup>)returnValue;

					model.SetClusters(model, clusters, vmTemplate.vds_group_id.Value);

					behavior.InitTemplate();
					behavior.InitCdImage();
				}, Model.Hash),
				dataCenter.Id
			);
		}

		public override void Template_SelectedItemChanged()
		{
			//Leave this method empty. Not relevant for template.
		}

		public override void Cluster_SelectedItemChanged()
		{
			UpdateDefaultHost();
			UpdateNumOfSockets();
		}

		public override void DefaultHost_SelectedItemChanged()
		{
			UpdateCdImage();
		}

		public override void Provisioning_SelectedItemChanged()
		{
		}

		public override void UpdateMinAllocatedMemory()
		{
		}

		private void InitTemplate()
		{
			//Update model state according to VM properties.
			Model.Name.Entity = this.template.name;
			Model.Description.Entity = this.template.description;
			Model.MemSize.Entity = this.template.mem_size_mb;
			Model.OSType.SelectedItem = this.template.os;
			Model.Domain.SelectedItem = this.template.domain;
			Model.UsbPolicy.SelectedItem = this.template.usb_policy;
			Model.NumOfMonitors.SelectedItem = this.template.num_of_monitors;
			Model.BootSequence = this.template.default_boot_sequence;
			Model.IsHighlyAvailable.Entity = this.template.auto_startup;
			Model.NumOfSockets.Entity = this.template.num_of_sockets;
			Model.TotalCPUCores.Entity = this.template.num_of_cpus;
			Model.IsStateless.Entity = this.template.is_stateless;

			Model.Kernel_parameters.Entity = this.template.kernel_params;
			Model.Kernel_path.Entity = this.template.kernel_url;
			Model.Initrd_path.Entity = this.template.initrd_url;

			if (!String.IsNullOrEmpty(template.time_zone))
			{
				//Patch! Create key-value pair with a right key.
				Model.TimeZone.SelectedItem = new KeyValuePair<string, string>(template.time_zone, String.Empty);

				UpdateTimeZone();
			}
			else
			{
				UpdateDefaultTimeZone();
			}

			// Update domain list
			UpdateDomain();

			//Storage domain and provisioning are not available for an existing VM.
			Model.StorageDomain.IsChangable = false;
			Model.Provisioning.IsAvailable = false;


			//Select display protocol.
			foreach (object item in Model.DisplayProtocol.Items)
			{
				EntityModel model = (EntityModel)item;
				DisplayType displayType = (DisplayType)model.Entity;

				if (displayType == this.template.default_display_type)
				{
					Model.DisplayProtocol.SelectedItem = item;
					break;
				}
			}


			InitPriority(this.template.priority);
		}

		private void InitCdImage()
		{
			Model.CdImage.SelectedItem = template.iso_path;
			Model.CdImage.IsChangable = !String.IsNullOrEmpty(template.iso_path);

			UpdateCdImage();
		}
	}

	public class UserPortalNewVmModelBehavior : NewVmModelBehavior, IFrontendMultipleQueryAsyncCallback
	{
		private const ActionGroup CREATE_VM = ActionGroup.CREATE_VM;
		
		public override void Initialize(SystemTreeItemModel systemTreeSelectedItem)
		{
			// Get datacenters with permitted create action
			AsyncDataProvider.GetDataCentersWithPermittedActionOnClusters(new AsyncQuery(Model,
				(target, returnValue) =>
				{
					UnitVmModel model = (UnitVmModel)target;

					//Show only data centers with status up.
					List<storage_pool> list = new List<storage_pool>();
					foreach (storage_pool a in (List<storage_pool>)(valueObjectEnumerableList)returnValue)
					{
						if (a.status == StoragePoolStatus.Up)
						{
							list.Add(a);
						}
					}
					model.IsDatacenterAvailable = list.Count > 0;
					model.SetDataCenter(model, list);
				}, Model.Hash), CREATE_VM);
		}

		public override void DataCenter_SelectedItemChanged()
		{
			storage_pool dataCenter = (storage_pool)Model.DataCenter.SelectedItem;
			Model.IsHostAvailable = dataCenter.storage_pool_type != StorageType.LOCALFS;
			
			List<VdcQueryType> queryTypeList = new List<VdcQueryType>();
			queryTypeList.Add(VdcQueryType.GetClustersWithPermittedAction);
			queryTypeList.Add(VdcQueryType.GetVmTemplatesWithPermittedAction);

			GetEntitiesWithPermittedActionParameters getEntitiesWithPermittedActionParameters = 
				new GetEntitiesWithPermittedActionParameters{ ActionGroup = CREATE_VM };

			List<VdcQueryParametersBase> parametersList = new List<VdcQueryParametersBase>
          	{
          		getEntitiesWithPermittedActionParameters,
          		getEntitiesWithPermittedActionParameters
          	};

			// Get clusters and templates
			Frontend.RunMultipleQueries(queryTypeList, parametersList, this, Model.Hash);
		}
		
		public void Executed(FrontendMultipleQueryAsyncResult result)
		{
			IList<VdcQueryReturnValue> returnValueList = result.ReturnValues;
			List<VDSGroup> clusters = (List<VDSGroup>)returnValueList[0].ReturnValue;
			List<VmTemplate> templates = (List<VmTemplate>)returnValueList[1].ReturnValue;

			InitClusters(clusters);
			InitTemplates(templates);
			InitCdImage();
		}

		private void InitClusters(List<VDSGroup> clusters)
		{
			// Filter clusters list (include only clusters that belong to the selected datacenter)
			List<VDSGroup> filteredList = new List<VDSGroup>();
			storage_pool selectedDataCenter = (storage_pool)Model.DataCenter.SelectedItem;

			foreach (VDSGroup cluster in clusters)
			{
				if (cluster.storage_pool_id != null && selectedDataCenter.Id.Equals(cluster.storage_pool_id))
				{
					filteredList.Add(cluster);
				}
			}

			filteredList.Sort(new Linq.VdsGroupByNameComparer());
			Model.SetClusters(Model, filteredList, null);
		}

		private void InitTemplates(List<VmTemplate> templates)
		{
			// Filter templates list (include only templates that belong to the selected datacenter)
			List<VmTemplate> templatesList = new List<VmTemplate>();
			VmTemplate blankTemplate = new VmTemplate();
			storage_pool selectedDataCenter = (storage_pool)Model.DataCenter.SelectedItem;
			Guid selectedDataCenterId = selectedDataCenter.Id.Value;

			foreach (VmTemplate template in templates)
			{
				Guid datacenterId = template.storage_pool_id == null ? Guid.Empty : template.storage_pool_id.Value;
				
				if (template.Id.Equals(Guid.Empty))
				{
					blankTemplate = template;
				}
				else if (datacenterId != selectedDataCenterId)
				{
					continue;
				}
				else if (template.status == VmTemplateStatus.OK)
				{
					templatesList.Add(template);
				}
			}

			// Sort list and position "Blank" template as first
			templatesList.Sort(new Linq.VmTemplateByNameComparer());
			if (templates.Contains(blankTemplate))
			{
				templatesList.Insert(0, blankTemplate);
			}
			
			//If there was some template selected before, try select it again.
			VmTemplate oldTemplate = (VmTemplate)Model.Template.SelectedItem;

			Model.Template.Items = templatesList;

			Model.Template.SelectedItem = Linq.FirstOrDefault(templatesList,
				oldTemplate != null
					? new Linq.TemplatePredicate(oldTemplate.Id)
					: new Linq.TemplatePredicate(Guid.Empty));

			UpdateIsDisksAvailable();
		}
	}

	public class UserPortalExistingVmModelBehavior : ExistingVmModelBehavior
	{
		private const ActionGroup EDIT_VM_PROPERTIES = ActionGroup.EDIT_VM_PROPERTIES;

		public UserPortalExistingVmModelBehavior(VM vm) : base(vm)
		{
		}

		public override void DataCenter_SelectedItemChanged()
		{
			// Get clusters with permitted edit action
			AsyncDataProvider.GetClustersWithPermittedAction(new AsyncQuery(new object[] { this, Model },
				(target, returnValue) =>
				{
					object[] array = (object[])target;
					ExistingVmModelBehavior behavior = (ExistingVmModelBehavior)array[0];
					UnitVmModel model = (UnitVmModel)array[1];

					List<VDSGroup> clusters = (List<VDSGroup>)returnValue;
					InitClusters(clusters, model);

					behavior.InitTemplate();
					behavior.InitCdImage();
				}, Model.Hash),
				EDIT_VM_PROPERTIES
			);
		}

		private void InitClusters(List<VDSGroup> clusters, UnitVmModel model)
		{
			// Filter clusters list (include only clusters that belong to the selected datacenter)
			List<VDSGroup> filteredList = new List<VDSGroup>();
			storage_pool selectedDataCenter = (storage_pool) Model.DataCenter.SelectedItem;
			bool listContainsVmCluster = false;

			foreach (VDSGroup cluster in clusters)
			{
				if (cluster.storage_pool_id != null && selectedDataCenter.Id.Equals(cluster.storage_pool_id))
				{
					filteredList.Add(cluster);

					if (cluster.ID == vm.vds_group_id.Value)
					{
						listContainsVmCluster = true;
					}
				}
			}

			if (!listContainsVmCluster)
			{
				// Add VM's cluster if not contained in the cluster list 
				AddVmCluster(filteredList);
			}
			else
			{
				filteredList.Sort(new Linq.VdsGroupByNameComparer());
				model.SetClusters(model, filteredList, vm.vds_group_id.Value);	
			}
		}

		private void AddVmCluster(List<VDSGroup> clusters)
		{
			AsyncDataProvider.GetClusterById(new AsyncQuery(new object[] { Model, clusters },
				(target, returnValue) =>
				{
					object[] array = (object[])target;
					UnitVmModel model = (UnitVmModel)array[0];
					List<VDSGroup> clusterList = (List<VDSGroup>)array[1];

					VDSGroup cluster = (VDSGroup)returnValue;
					if (cluster != null)
					{
						clusterList.Add(cluster);
					}

					clusterList.Sort(new Linq.VdsGroupByNameComparer());
					model.SetClusters(model, clusterList, vm.vds_group_id.Value);
				}, Model.Hash),
				vm.vds_group_id
			);
		}
 	}
}
