using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Text;
using System.Threading;
using System.Windows.Threading;
using org.ovirt.engine.ui.uicommon.models.common;
using org.ovirt.engine.ui.uicommon.models.configure;
using org.ovirt.engine.ui.uicompat;
using VdcCommon;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.VdcQueries;
using System.Collections;
using System.ComponentModel;

namespace org.ovirt.engine.ui.uicommon.models.storage
{
	public class StorageListModel : ListWithDetailsModel, ITaskTarget, ISupportSystemTreeContext
	{
		#region Commands

		public UICommand NewDomainCommand { get; private set; }
		public UICommand ImportDomainCommand { get; private set; }
		public UICommand EditCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }
		public UICommand DestroyCommand { get; private set; }

		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		protected object[] SelectedKeys
		{
			//			get { return SelectedItems == null ? new object[0] : SelectedItems.Cast<storage_domains>().Select(a => a.id).Cast<object>().ToArray(); }
			get
			{
				if (SelectedItems == null)
				{
					return new object[0];
				}
				else
				{
					List<object> items = new List<object>();
					foreach (object item in SelectedItems)
					{
						storage_domains i = (storage_domains)item;
						items.Add(i.id);
					}
					return items.ToArray();
				}
			}
		}

		#endregion


		public StorageListModel()
		{
			Title = "Storage";

			DefaultSearchString = "Storage:";
			SearchString = DefaultSearchString;

			NewDomainCommand = new UICommand("NewDomain", this);
			ImportDomainCommand = new UICommand("ImportDomain", this);
			EditCommand = new UICommand("Edit", this);
			RemoveCommand = new UICommand("Remove", this);
			DestroyCommand = new UICommand("Destroy", this);

			UpdateActionAvailability();

			SearchNextPageCommand.IsAvailable = true;
			SearchPreviousPageCommand.IsAvailable = true;
		}

		private EntityModel vmBackupModel;
		private EntityModel templateBackupModel;
		private ListModel vmListModel;
		private ListModel templateListModel;
		private ListModel isoListModel;

		protected override void InitDetailModels()
		{
			base.InitDetailModels();


			vmBackupModel = new VmBackupModel();
			vmBackupModel.IsAvailable = false;

			templateBackupModel = new TemplateBackupModel();
			templateBackupModel.IsAvailable = false;

			vmListModel = new StorageVmListModel();
			vmListModel.IsAvailable = false;

			templateListModel = new StorageTemplateListModel();
			templateListModel.IsAvailable = false;

			isoListModel = new StorageIsoListModel();
			isoListModel.IsAvailable = false;

			ObservableCollection<EntityModel> list = new ObservableCollection<EntityModel>();
			list.Add(new StorageGeneralModel());
			list.Add(new StorageDataCenterListModel());
			list.Add(vmBackupModel);
			list.Add(templateBackupModel);
			list.Add(vmListModel);
			list.Add(templateListModel);
			list.Add(isoListModel);
			list.Add(new StorageEventListModel());
			list.Add(new PermissionListModel());
			DetailModels = list;
		}

		public override bool IsSearchStringMatch(string searchString)
		{
			return searchString.Trim().ToLower().StartsWith("storage");
		}

		protected override void SyncSearch()
		{
			base.SyncSearch(VdcQueryType.Search, new SearchParameters(SearchString, SearchType.StorageDomain)
			{
				MaxCount = SearchPageSize
			});
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			AsyncResult = Frontend.RegisterSearch(SearchString, SearchType.StorageDomain, SearchPageSize);
			Items = AsyncResult.Data;
		}

		private void NewDomain()
		{
			if (Window != null)
			{
				return;
			}

			StorageModel model = new StorageModel(new NewEditStorageModelBehavior());
			Window = model;
			model.Title = "New Domain";
			model.HashName = "new_domain";
			model.SystemTreeSelectedItem = SystemTreeSelectedItem;

			List<IStorageModel> items = new List<IStorageModel>();
			// putting all Data domains at the beginning on purpose (so when choosing the
			// first selectable storage type/function, it will be a Data one, if relevant).
			items.Add(new NfsStorageModel { Role = StorageDomainType.Data });
			items.Add(new IscsiStorageModel { Role = StorageDomainType.Data, IsGrouppedByTarget = true });
			items.Add(new FcpStorageModel { Role = StorageDomainType.Data });
			items.Add(new LocalStorageModel {Role = StorageDomainType.Data});

			items.Add(new NfsStorageModel { Role = StorageDomainType.ISO });

			items.Add(new NfsStorageModel { Role = StorageDomainType.ImportExport });
			items.Add(new IscsiStorageModel { Role = StorageDomainType.ImportExport, IsGrouppedByTarget = true });
			items.Add(new FcpStorageModel { Role = StorageDomainType.ImportExport });

			model.Items = items;

			model.Initialize();


			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void Edit()
		{
			storage_domains storage = (storage_domains)SelectedItem;

			if (Window != null)
			{
				return;
			}

			StorageModel model = new StorageModel(new NewEditStorageModelBehavior());
			Window = model;
			model.Title = "Edit Domain";
			model.HashName = "edit_domain";
			model.SystemTreeSelectedItem = SystemTreeSelectedItem;
			model.Storage = storage;
			model.Name.Entity = storage.storage_name;
			model.OriginalName = storage.storage_name;

			model.DataCenter.IsAvailable = false;
			model.Format.IsAvailable = false;


			IStorageModel item = null;
			switch (storage.storage_type)
			{
				case StorageType.NFS:
					item = PrepareNfsStorageForEdit(storage);
					break;

				case StorageType.FCP:
					item = PrepareFcpStorageForEdit(storage);
					break;

				case StorageType.ISCSI:
					item = PrepareIscsiStorageForEdit(storage);
					break;

				case StorageType.LOCALFS:
					item = PrepareLocalStorageForEdit(storage);
					break;
			}

			model.Items = new List<IStorageModel> { item };
			model.SelectedItem = item;

			model.Initialize();


			if (SystemTreeSelectedItem != null && SystemTreeSelectedItem.Type != SystemTreeItemType.System)
			{
				switch (SystemTreeSelectedItem.Type)
				{
					case SystemTreeItemType.Storage:
						{
							model.Name.IsChangable = false;
							model.Name.Info = "Cannot edit Storage Domains's Name in this tree context";
						}
						break;
				}
			}




			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private IStorageModel PrepareNfsStorageForEdit(storage_domains storage)
		{
			NfsStorageModel model = new NfsStorageModel();
			model.Role = storage.storage_domain_type;

			storage_server_connections connection = DataProvider.GetStorageConnectionById(storage.storage);
			model.Path.Entity = connection.connection;
			model.Path.IsAvailable = false;

			return model;
		}

		private IStorageModel PrepareLocalStorageForEdit(storage_domains storage)
		{
			LocalStorageModel model = new LocalStorageModel();
			model.Role = storage.storage_domain_type;

			storage_server_connections connection = DataProvider.GetStorageConnectionById(storage.storage);
			model.Path.Entity = connection.connection;
			model.Path.IsAvailable = false;

			return model;
		}

		private IStorageModel PrepareIscsiStorageForEdit(storage_domains storage)
		{
			IscsiStorageModel model = new IscsiStorageModel();
			model.Role = storage.storage_domain_type;

			PrepareSanStorageForEdit(model);

			return model;
		}

		private IStorageModel PrepareFcpStorageForEdit(storage_domains storage)
		{
			FcpStorageModel model = new FcpStorageModel();
			model.Role = storage.storage_domain_type;

			PrepareSanStorageForEdit(model);

			return model;
		}

		private void PrepareSanStorageForEdit(SanStorageModel model)
		{
			storage_domains storage = (storage_domains)SelectedItem;

			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetLunsByVgId,
				new GetLunsByVgIdParameters
				{
					VgId = storage.storage
				});

			if (returnValue != null && returnValue.Succeeded)
			{
				model.ApplyData((List<LUNs>)returnValue.ReturnValue, true);
			}
		}

		private void ImportDomain()
		{
			if (Window != null)
			{
				return;
			}

			StorageModel model = new StorageModel(new ImportStorageModelBehavior());
			Window = model;
			model.Title = "Import Pre-Configured Domain";
			model.HashName = "import_pre-configured_domain";
			model.SystemTreeSelectedItem = SystemTreeSelectedItem;
			model.Name.IsAvailable = false;
			model.Format.IsAvailable = false;

			List<IStorageModel> items = new List<IStorageModel>();
			items.Add(new NfsStorageModel { Role = StorageDomainType.ISO });
			items.Add(new NfsStorageModel { Role = StorageDomainType.ImportExport });

			items.Add(new IscsiImportStorageModel { Role = StorageDomainType.ImportExport });

			items.Add(new FcpImportStorageModel { Role = StorageDomainType.ImportExport });

			model.Items = items;

			model.Initialize();


			model.Commands.Add(
				new UICommand("OnImport", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnImport()
		{
			StorageModel model = (StorageModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			model.StartProgress("Importing Storage Domain...");

			VDS host = (VDS)model.Host.SelectedItem;

			//Save changes.
			if (model.SelectedItem is NfsStorageModel)
			{
				NfsStorageModel nfsModel = (NfsStorageModel)model.SelectedItem;
				nfsModel.Message = null;

				Task.Create(this,
					new List<object>
					{
						"ImportNfs",
						host.vds_id,
						nfsModel.Path.Entity,
						nfsModel.Role

					}).Run();
			}
			else
			{
				Task.Create(this,
					new List<object>
					{
						"ImportSan",
						host.vds_id,
					}).Run();
			}
		}

		private storage_server_connections nfsConnection;

		private void CleanNfsConnection(Guid hostId)
		{
			if (nfsConnection != null)
			{
				VdcReturnValueBase ret1 = Frontend.RunAction(VdcActionType.RemoveStorageServerConnection,
					new StorageServerConnectionParametersBase(nfsConnection, hostId));

				if (ret1 != null && ret1.Succeeded)
				{
					nfsConnection = null;
				}
			}
		}

		private void Remove()
		{
			storage_domains storage = (storage_domains)SelectedItem;

			if (Window != null)
			{
				return;
			}

			RemoveStorageModel model = new RemoveStorageModel();
			Window = model;
			model.Title = "Remove Storage(s)";
			model.HashName = "remove_storage";

			List<VDS> hosts = DataProvider.GetUpHostList();

			model.HostList.Items = hosts;
			model.HostList.SelectedItem = Linq.FirstOrDefault(hosts);

			model.Format.IsAvailable = storage.storage_domain_type == StorageDomainType.ISO ||
										storage.storage_domain_type == StorageDomainType.ImportExport;


			if (hosts.Count == 0)
			{
				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Close",
						IsDefault = true,
						IsCancel = true
					});
			}
			else
			{
				model.Commands.Add(
					new UICommand("OnRemove", this)
					{
						Title = "OK",
						IsDefault = true
					});
				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
		}

		private void OnRemove()
		{
			if (SelectedItem != null)
			{
				storage_domains storage = (storage_domains)SelectedItem;
				RemoveStorageModel model = (RemoveStorageModel)Window;

				if (!model.Validate())
				{
					return;
				}

				VDS host = (VDS)model.HostList.SelectedItem;

				Frontend.RunActionAsyncroniousely(VdcActionType.RemoveStorageDomain,
					new RemoveStorageDomainParameters(storage.id)
					{
						VdsId = host.vds_id,
						DoFormat = (storage.storage_domain_type == StorageDomainType.Data || storage.storage_domain_type == StorageDomainType.Master) ? true : (bool)model.Format.Entity
					});
			}

			Cancel();
		}

		private void Destroy()
		{
			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Destroy Storage Domain";
			model.HashName = "destroy_storage_domain";
			List<string> items = new List<string>();
			items.Add(((storage_domains)SelectedItem).storage_name);
			model.Items = items;

			model.Latch.IsAvailable = true;
			model.Latch.IsChangable = true;


			model.Commands.Add(
				new UICommand("OnDestroy", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnDestroy()
		{
			ConfirmationModel model = (ConfirmationModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			storage_domains storageDomain = (storage_domains)SelectedItem;


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.ForceRemoveStorageDomain,
				new List<VdcActionParametersBase>
				{
					new StorageDomainParametersBase(storageDomain.id)
				},
				result =>
				{
					ConfirmationModel localModel = (ConfirmationModel)result.State;

					localModel.StopProgress();
					Cancel();
				},
				model
			);
		}

		private void OnSave()
		{
			StorageModel model = (StorageModel)Window;

			if (!model.Validate())
			{
				return;
			}


			if (model.SelectedItem is NfsStorageModel)
			{
				SaveNfsStorage();
			}
			else if (model.SelectedItem is LocalStorageModel)
			{
				SaveLocalStorage();
			}
			else
			{
				SaveSanStorage();
			}
		}

		private void SaveLocalStorage()
		{
			if (Window.Progress != null)
			{
				return;
			}

			Window.StartProgress(null);

			Task.Create(this, new List<object> { "SaveLocal" }).Run();
		}

		private void SaveNfsStorage()
		{
			if (Window.Progress != null)
			{
				return;
			}

			Window.StartProgress(null);

			Task.Create(this, new List<object> { "SaveNfs" }).Run();
		}

		private void SaveSanStorage()
		{
			if (Window.Progress != null)
			{
				return;
			}

			Window.StartProgress(null);

			Task.Create(this, new List<object> { "SaveSan" }).Run();
		}

		private void Cancel()
		{
			Window = null;
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();
		}

		protected override void ItemsCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
		{
			base.ItemsCollectionChanged(sender, e);

			//Try to select an item corresponding to the system tree selection.
			if (SystemTreeSelectedItem != null && SystemTreeSelectedItem.Type == SystemTreeItemType.Storage)
			{
				storage_domains storage = (storage_domains)SystemTreeSelectedItem.Entity;

				SelectedItem = Linq.FirstOrDefault(Linq.Cast<storage_domains>(Items), new Linq.StoragePredicate(storage.id));
			}
		}

		protected override void UpdateDetailsAvailability()
		{
			if (SelectedItem != null)
			{
				storage_domains storage = (storage_domains)SelectedItem;
				bool isBackupStorage = storage.storage_domain_type == StorageDomainType.ImportExport;
				bool isDataStorage = storage.storage_domain_type == StorageDomainType.Data || storage.storage_domain_type == StorageDomainType.Master;
				bool isIsoStorage = storage.storage_domain_type == StorageDomainType.ISO;

				vmBackupModel.IsAvailable = isBackupStorage;
				templateBackupModel.IsAvailable = isBackupStorage;

				vmListModel.IsAvailable = isDataStorage;
				templateListModel.IsAvailable = isDataStorage;

				isoListModel.IsAvailable = isIsoStorage;
			}
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.SelectedItemPropertyChanged(sender, e);

			if (e.PropertyName == "storage_domain_shared_status")
			{
				UpdateActionAvailability();
			}
		}

		private void UpdateActionAvailability()
		{
			List<storage_domains> items = SelectedItems != null ? Linq.Cast<storage_domains>(SelectedItems) : new List<storage_domains>();

			storage_domains item = (storage_domains)SelectedItem;

			NewDomainCommand.IsAvailable = true;

			EditCommand.IsExecutionAllowed = item != null
				&& items.Count == 1
				&& (item.storage_domain_shared_status == StorageDomainSharedStatus.Active || item.storage_domain_shared_status == StorageDomainSharedStatus.Mixed);

			RemoveCommand.IsExecutionAllowed = items.Count == 1
				&& Linq.FindAllStorageDomainsBySharedStatus(items, StorageDomainSharedStatus.Unattached).Count == items.Count;

			DestroyCommand.IsExecutionAllowed = item != null && items.Count == 1;


			//System tree dependent actions.
			bool isAvailable = !(SystemTreeSelectedItem != null && SystemTreeSelectedItem.Type == SystemTreeItemType.Storage);

			NewDomainCommand.IsAvailable = isAvailable;
			RemoveCommand.IsAvailable = isAvailable;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == NewDomainCommand)
			{
				NewDomain();
			}
			else if (command == ImportDomainCommand)
			{
				ImportDomain();
			}
			else if (command == EditCommand)
			{
				Edit();
			}
			else if (command == RemoveCommand)
			{
				Remove();
			}
			else if (command == DestroyCommand)
			{
				Destroy();
			}
			else if (command.Name == "OnSave")
			{
				OnSave();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
			else if (command.Name == "OnImport")
			{
				OnImport();
			}
			else if (command.Name == "OnRemove")
			{
				OnRemove();
			}
			else if (command.Name == "OnDestroy")
			{
				OnDestroy();
			}
		}

		private void SaveNfsStorage(TaskContext context)
		{
			storage_domains selectedItem = (storage_domains)SelectedItem;
			StorageModel model = (StorageModel)Window;
			NfsStorageModel nfsModel = (NfsStorageModel)model.SelectedItem;
			VDS host = (VDS)model.Host.SelectedItem;
			bool isNew = model.Storage == null;


			string path = (string)nfsModel.Path.Entity;
			string storageName;
			if (isNew && DataProvider.IsDomainAlreadyExist(path, out storageName))
			{
				context.InvokeUIThread(this,
				new List<object>
				{
					"Finish",
					false,
					nfsModel,
					"Create Operation failed. Domain " + storageName + " already exists in the system."
				});

				return;
			}


			//Create storage connection.
			storage_server_connections connection =
				new storage_server_connections
				{
					connection = path,
					storage_type = nfsModel.Type
				};

			storage_domain_static storageDomain =
				isNew
				? new storage_domain_static()
				: (storage_domain_static)Cloner.Clone(selectedItem.StorageStaticData);

			storageDomain.storage_type =
				isNew
				? nfsModel.Type
				: storageDomain.storage_type;

			storageDomain.storage_domain_type =
				isNew
				? nfsModel.Role
				: storageDomain.storage_domain_type;

			storageDomain.storage_name = (string)model.Name.Entity;
			storageDomain.StorageFormat = (StorageFormatType)model.Format.SelectedItem;


			VdcReturnValueBase returnValue;

			if (isNew)
			{
				bool attach = false;

				returnValue = Frontend.RunAction(VdcActionType.AddStorageServerConnection,
					new StorageServerConnectionParametersBase(
						connection,
						host.vds_id
					)
				);

				if (returnValue != null && returnValue.Succeeded)
				{
					storageDomain.storage = (string)returnValue.ActionReturnValue;

					//Add storage domain.
					returnValue = Frontend.RunAction(VdcActionType.AddNFSStorageDomain,
						new StorageDomainManagementParameter(storageDomain)
						{
							VdsId = host.vds_id
						}
					);

					attach = returnValue != null && returnValue.Succeeded;
				}

				//Clean up connection.
				Frontend.RunAction(VdcActionType.RemoveStorageServerConnection,
					new StorageServerConnectionParametersBase(connection, host.vds_id));

				if (attach)
				{
					//Attach storage to data center as neccessary.
					storage_pool dataCenter = (storage_pool)model.DataCenter.SelectedItem;
					if (!dataCenter.Id.Equals(StorageModel.UnassignedDataCenterId))
					{
						nguid storageId = (nguid)returnValue.ActionReturnValue;
						AttachStorageToDataCenter((Guid)storageId, dataCenter.Id);
					}
				}
			}
			else
			{
				returnValue = Frontend.RunAction(VdcActionType.UpdateStorageDomain,
					new StorageDomainManagementParameter(storageDomain));
			}


			context.InvokeUIThread(this,
				new List<object>
				{
					"Finish",
					returnValue != null && returnValue.Succeeded,
					nfsModel,
					null
				});
		}

		private void SaveLocalStorage(TaskContext context)
		{
			storage_domains selectedItem = (storage_domains)SelectedItem;
			StorageModel model = (StorageModel)Window;
			LocalStorageModel localModel = (LocalStorageModel)model.SelectedItem;
			VDS host = (VDS)model.Host.SelectedItem;
			bool isNew = model.Storage == null;


			string path = (string)localModel.Path.Entity;
			string storageName;
            if (isNew && DataProvider.IsDomainAlreadyExist(host.storage_pool_id, path, out storageName))
			{
				context.InvokeUIThread(this,
				new List<object>
				{
					"Finish",
					false,
					localModel,
					"Create Operation failed. Domain " + storageName + " already exists in the system."
				});

				return;
			}


			//Create storage connection.
			storage_server_connections connection =
				new storage_server_connections
				{
					connection = path,
					storage_type = localModel.Type
				};

			storage_domain_static storageDomain =
				isNew
				? new storage_domain_static()
				: (storage_domain_static)Cloner.Clone(selectedItem.StorageStaticData);

			storageDomain.storage_type =
				isNew
				? localModel.Type
				: storageDomain.storage_type;

			storageDomain.storage_domain_type =
				isNew
				? localModel.Role
				: storageDomain.storage_domain_type;

			storageDomain.storage_name = (string)model.Name.Entity;


			VdcReturnValueBase returnValue;
			bool removeConnection = false;

			if (isNew)
			{
				returnValue = Frontend.RunAction(VdcActionType.AddStorageServerConnection,
					new StorageServerConnectionParametersBase(
						connection,
						host.vds_id
					)
				);

				if (returnValue != null && returnValue.Succeeded)
				{
					storageDomain.storage = (string)returnValue.ActionReturnValue;

					//Add storage domain.
					returnValue = Frontend.RunAction(VdcActionType.AddLocalStorageDomain,
						new StorageDomainManagementParameter(storageDomain)
						{
							VdsId = host.vds_id
						}
					);

					if (returnValue == null || !returnValue.Succeeded)
					{
						removeConnection = true;
					}
				}

				//Clean up connection in case of storage creation failure.
				if (removeConnection)
				{
					Frontend.RunAction(VdcActionType.RemoveStorageServerConnection,
						new StorageServerConnectionParametersBase(connection, host.vds_id));
				}
			}
			else
			{
				returnValue = Frontend.RunAction(VdcActionType.UpdateStorageDomain,
					new StorageDomainManagementParameter(storageDomain));
			}


			context.InvokeUIThread(this,
				new List<object>
				{
					"Finish",
					returnValue != null && returnValue.Succeeded,
					localModel,
					null
				});
		}

		private void SaveSanStorage(TaskContext context)
		{
			StorageModel model = (StorageModel)Window;
			SanStorageModel sanModel = (SanStorageModel)model.SelectedItem;
			storage_domains storage = (storage_domains)SelectedItem;
			bool isNew = model.Storage == null;

			VDS host = (VDS)model.Host.SelectedItem;


			List<string> lunIds = new List<string>();
			foreach (LunModel lun in sanModel.AddedLuns)
			{
				lunIds.Add(lun.LunId);
			}


			storage_domain_static storageDomain =
				isNew
				? new storage_domain_static()
				: (storage_domain_static)Cloner.Clone(storage.StorageStaticData);

			storageDomain.storage_type =
				isNew
				? sanModel.Type
				: storageDomain.storage_type;

			storageDomain.storage_domain_type =
				isNew
				? sanModel.Role
				: storageDomain.storage_domain_type;

			storageDomain.StorageFormat =
				isNew
				? (StorageFormatType)sanModel.Container.Format.SelectedItem
				: storageDomain.StorageFormat;

			storageDomain.storage_name = (string)model.Name.Entity;

			VdcReturnValueBase returnValue;

			if (isNew)
			{
				returnValue = Frontend.RunAction(VdcActionType.AddSANStorageDomain,
					new AddSANStorageDomainParameters(storageDomain)
					{
						VdsId = host.vds_id,
						LunIds = lunIds
					});

				if (returnValue != null && returnValue.Succeeded)
				{
					//Attach storage to data center as neccessary.
					storage_pool dataCenter = (storage_pool)model.DataCenter.SelectedItem;
					if (!dataCenter.Id.Equals(StorageModel.UnassignedDataCenterId))
					{
						nguid storageId = (nguid)returnValue.ActionReturnValue;
						AttachStorageToDataCenter((Guid)storageId, dataCenter.Id);
					}
				}
			}
			else
			{
				returnValue = Frontend.RunAction(VdcActionType.UpdateStorageDomain,
					new StorageDomainManagementParameter(storageDomain));


				//Extend storage as neccessary.
				if (returnValue != null && returnValue.Succeeded && lunIds.Count > 0)
				{
					returnValue = Frontend.RunAction(VdcActionType.ExtendSANStorageDomain,
						new ExtendSANStorageDomainParameters(storage.id, lunIds));
				}
			}


			context.InvokeUIThread(this,
				new List<object>
				{
					"Finish",
					returnValue != null && returnValue.Succeeded,
					sanModel,
					null
				});
		}

		private void AttachStorageToDataCenter(Guid storageId, Guid dataCenterId)
		{
			Frontend.RunActionAsyncroniousely(VdcActionType.AttachStorageDomainToPool,
				new StorageDomainPoolParametersBase(storageId, dataCenterId));
		}

		private void ImportNfsStorage(TaskContext context)
		{
			List<object> data = (List<object>)context.State;

			StorageModel model = (StorageModel)Window;
			NfsStorageModel storageModel = (NfsStorageModel)model.SelectedItem;

			Guid hostId = (guid)data[1];
			string path = (string)data[2];
			StorageDomainType domainType = (StorageDomainType)data[3];

			bool success = true;
			string message = null;

			CleanNfsConnection(hostId);

			try
			{
				string storageName;
				if (DataProvider.IsDomainAlreadyExist(path, out storageName))
				{
					throw new Exception("Import Operation failed. Domain " + storageName + " already exists in the system.");
				}

				nfsConnection =
					new storage_server_connections
					{
						connection = path,
						storage_type = StorageType.NFS
					};

				VdcReturnValueBase ret2 = Frontend.RunAction(VdcActionType.ConnectStorageToVds,
						new StorageServerConnectionParametersBase(
							nfsConnection,
							hostId
						)
					);

				if (ret2 == null || !ret2.Succeeded)
				{
					throw new Exception("Failed to retrieve existing storage domain information.");
				}

				VdcQueryReturnValue ret3 = Frontend.RunQuery(VdcQueryType.GetExistingStorageDomainList,
					new GetExistingStorageDomainListParameters(hostId, StorageType.NFS, domainType, path));

				if (ret3 == null || !ret3.Succeeded)
				{
					throw new Exception("Failed to retrieve existing storage domain information.");
				}

				List<storage_domains> domains = ((List<storage_domains>)ret3.ReturnValue);
				if (domains.Count == 0)
				{
					throw new Exception("There is no storage domain under the specified path. Please check path.");
				}

				storage_domains sdToAdd = Linq.FirstOrDefault(domains);
				storage_domain_static sdsToAdd = sdToAdd == null ? null : sdToAdd.StorageStaticData;

				VdcReturnValueBase ret4 = Frontend.RunAction(VdcActionType.AddExistingNFSStorageDomain,
					new StorageDomainManagementParameter(sdsToAdd)
					{
						VdsId = hostId
					});

				if (ret4 == null || !ret4.Succeeded)
				{
					throw new Exception();
				}


				//Attach storage to data center as neccessary.
				storage_pool dataCenter = (storage_pool)model.DataCenter.SelectedItem;
				if (!dataCenter.Id.Equals(StorageModel.UnassignedDataCenterId))
				{
					AttachStorageToDataCenter(sdToAdd.id, dataCenter.Id);
				}
			}
			catch (Exception ex)
			{
				success = false;
				message = ex.Message;
			}
			finally
			{
				CleanNfsConnection(hostId);
			}

			context.InvokeUIThread(this,
				new List<object>
				{
					"Finish",
					success,
					storageModel,
					message
				});
		}

		private void ImportSanStorage(TaskContext context)
		{
			List<object> data = (List<object>)context.State;

			StorageModel model = (StorageModel)Window;
			ImportSanStorageModel storageModel = (ImportSanStorageModel)model.SelectedItem;
			storage_domains storage = (storage_domains)storageModel.SelectedItem;

			Guid hostId = (guid)data[1];

			VdcReturnValueBase returnValue = Frontend.RunAction(VdcActionType.AddExistingSANStorageDomain,
				new AddSANStorageDomainParameters(storage == null ? null : storage.StorageStaticData)
				{
					VdsId = hostId
				});


			if (returnValue != null && returnValue.Succeeded)
			{
				//Attach storage to data center as neccessary.
				storage_pool dataCenter = (storage_pool)model.DataCenter.SelectedItem;
				if (!dataCenter.Id.Equals(StorageModel.UnassignedDataCenterId))
				{
					AttachStorageToDataCenter(storage.id, dataCenter.Id);
				}
			}


			context.InvokeUIThread(this,
				new List<object>
				{
					"Finish",
					returnValue != null && returnValue.Succeeded,
					storageModel,
					null
				});
		}

		public void run(TaskContext context)
		{
			List<object> data = (List<object>)context.State;
			string key = (string)data[0];

			switch (key)
			{
				case "SaveNfs":
					SaveNfsStorage(context);
					break;

				case "SaveLocal":
					SaveLocalStorage(context);
					break;

				case "SaveSan":
					SaveSanStorage(context);
					break;

				case "ImportNfs":
					ImportNfsStorage(context);
					break;

				case "ImportSan":
					ImportSanStorage(context);
					break;

				case "Finish":
					Window.StopProgress();

					if ((bool)data[1])
					{
						Cancel();
					}
					else
					{
						((Model)data[2]).Message = (string)data[3];
					}
					break;
			}
		}


		private SystemTreeItemModel systemTreeSelectedItem;
		public SystemTreeItemModel SystemTreeSelectedItem
		{
			get { return systemTreeSelectedItem; }
			set
			{
				if (systemTreeSelectedItem != value)
				{
					systemTreeSelectedItem = value;
					OnSystemTreeSelectedItemChanged();
				}
			}
		}

		private void OnSystemTreeSelectedItemChanged()
		{
			UpdateActionAvailability();
		}
	}
}
