using System;
using System.Collections.Generic;
using System.Text;
using org.ovirt.engine.ui.uicommon.validation;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.BusinessEntities;


namespace org.ovirt.engine.ui.uicommon.models.datacenters
{
	public class DataCenterModel : Model
	{
		#region Properties

		public Guid? DataCenterId { get; set; }
		public bool IsNew { get; set; }
		public string OriginalName { get; set; }

		public EntityModel Name { get; set; }
		public EntityModel Description { get; set; }
		public ListModel StorageTypeList { get; set; }
		public ListModel Version { get; set; }

		#endregion

		public DataCenterModel()
		{
			Name = new EntityModel();
			Description = new EntityModel();
			Version = new ListModel();

			StorageTypeList = new ListModel();
			StorageTypeList.SelectedItemChangedEvent.addListener(this);
			StorageTypeList.Items = DataProvider.GetStoragePoolTypeList();
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(ListModel.SelectedItemChangedEventDefinition) && sender == StorageTypeList)
			{
				StorageType_SelectedItemChanged();
			}
		}

		private void StorageType_SelectedItemChanged()
		{
			StorageType type = (StorageType)StorageTypeList.SelectedItem;

			//Rebuild version items.
			List<Version> list = new List<Version>();
			foreach (Version item in DataProvider.GetDataCenterVersions(DataCenterId))
			{
				if (DataProvider.IsVersionMatchStorageType(item, type))
				{
					list.Add(item);
				}
			}
			
			if(type == StorageType.LOCALFS)
			{
				List<Version> tempList = new List<Version>();
				foreach (Version version in list)
				{
					Version version3_0 = new Version(3, 0);
					if(version.CompareTo(version3_0) >= 0)
					{
						tempList.Add(version);
					}
				}
				list = tempList;
			}

			Version selectedVersion = null;
			if (Version.SelectedItem != null)
			{
				selectedVersion = (Version)Version.SelectedItem;
				bool hasSelectedVersion = false;
				foreach (Version version in list)
				{
					if (selectedVersion.Equals(version))
					{
						selectedVersion = version;
						hasSelectedVersion = true;
						break;
					}
				}
				if (!hasSelectedVersion)
				{
					selectedVersion = null;
				}
			}

			Version.Items = list;
			
			if (selectedVersion == null)
			{
				Version.SelectedItem = Linq.SelectHighestVersion(list);
			}
			else
			{
				Version.SelectedItem = selectedVersion;
			}
		}

		internal bool Validate()
		{
			int nameMaxLength = DataProvider.GetDataCenterMaxNameLength();
			string nameRegex = string.Format(@"^[A-Za-z0-9_-]{{1,{0}}}$", nameMaxLength);
			string nameMessage = string.Format("Name can contain only 'A-Z', 'a-z', '0-9', '_' or '-' characters, max length: {0}", nameMaxLength);

			Name.ValidateEntity(
				new IValidation[]
				{
					new NotEmptyValidation(),
					new RegexValidation {Expression = nameRegex, Message = nameMessage}
				});
			StorageTypeList.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });

			Version.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });

			string name = (string)Name.Entity;

			if (String.Compare(name, OriginalName, true) != 0 && !DataProvider.IsDataCenterNameUnique(name))
			{
				Name.IsValid = false;
				Name.InvalidityReasons.Add("Name must be unique.");
			}


			return Name.IsValid
				   && Description.IsValid
				   && StorageTypeList.IsValid
				   && Version.IsValid;
		}
	}
}
