using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Text;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.VdcQueries;
using System.Collections;

namespace org.ovirt.engine.ui.uicommon.models.clusters
{
	public class ClusterNetworkListModel : SearchableListModel
	{
		#region Commands

		public UICommand NewNetworkCommand { get; private set; }
		public UICommand ManageCommand { get; private set; }
		public UICommand SetAsDisplayCommand { get; private set; }

		#endregion

		#region Properties

		public new VDSGroup Entity
		{
			get { return base.Entity as VDSGroup; }
			set { base.Entity = value; }
		}

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		#endregion

		public ClusterNetworkListModel()
		{
			Title = "Logical Networks";

			ManageCommand = new UICommand("Manage", this);
			SetAsDisplayCommand = new UICommand("SetAsDisplay", this);
			NewNetworkCommand = new UICommand("New", this);

			UpdateActionAvailability();
		}

		protected override void OnEntityChanged()
		{
			base.OnEntityChanged();
			SearchCommand.Execute();
		}

		public override void Search()
		{
			if (Entity != null)
			{
				base.Search();
			}
		}

		protected override void SyncSearch()
		{
			base.SyncSearch();

			AsyncQuery _asyncQuery = new AsyncQuery();
			_asyncQuery.Model = this;
			/*START_DELEGATE*/_asyncQuery.asyncCallback = delegate(Object model, Object ReturnValue)
			{
				SearchableListModel searchableListModel = (SearchableListModel)model;
				searchableListModel.Items = (List<network>)((VdcQueryReturnValue)ReturnValue).ReturnValue;
			};//END_DELEGATE

			Frontend.RunQuery(VdcQueryType.GetAllNetworksByClusterId,
				new VdsGroupQueryParamenters(Entity.ID),
				_asyncQuery);
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			AsyncResult = Frontend.RegisterQuery(VdcQueryType.GetAllNetworksByClusterId, new VdsGroupQueryParamenters(Entity.ID));
			Items = AsyncResult.Data;
		}

		public void SetAsDisplay()
		{
			network network = (network)SelectedItem;

			Frontend.RunAction(VdcActionType.UpdateDisplayToVdsGroup,
				new DisplayNetworkToVdsGroupParameters(Entity, network, true));
		}

		public void Manage()
		{
			if (Window != null)
			{
				return;
			}

			ListModel model = new ListModel();
			Window = model;
			model.Title = "Assign/Detach Networks";
			model.HashName = "assign_networks";

			//var items = DataProvider.GetNetworkList(Entity.storage_pool_id.GetValueOrDefault())
			//	.Where(a => !a.name.Equals("rhevm", StringComparison.CurrentCulture))
			//	.Select(a => new EntityModel() { Entity = a })
			//	.ToList();
			Guid storagePoolId = (Entity.storage_pool_id != null) ? Entity.storage_pool_id.Value : Guid.Empty;

			List<network> networkList = DataProvider.GetNetworkList(storagePoolId);
			networkList.Sort(new Linq.NetworkByNameComparer());

			List<EntityModel> items = new List<EntityModel>();
			foreach (network a in networkList)
			{
				if (!a.name.Equals("rhevm"))
				{
					items.Add(new EntityModel { Entity = a });
				}
			}
			model.Items = items;

			bool noItems = items.Count == 0;

			List<network> networks = Linq.Cast<network>(Items);
			foreach (EntityModel item in items)
			{
				network net = (network)item.Entity;
				//				Selector.SetIsSelected(item, networks.Any(a => a.id == network.id));
				bool value = false;
				foreach (network a in networks)
				{
					if (a.Id.Equals(net.Id))
					{
						value = true;
						break;
					}
				}
				item.IsSelected = value;
			}


			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsDefault = noItems,
					IsCancel = true
				});

			if (!noItems)
			{
				model.Commands.Insert(0,
					new UICommand("OnManage", this)
					{
						Title = "OK",
						IsDefault = true
					});
			}
		}

		public void OnManage()
		{
			ListModel model = (ListModel)Window;

			List<EntityModel> items = Linq.Cast<EntityModel>(model.Items);
			List<network> networks = Linq.Cast<network>(Items);

			if (Entity == null)
			{
				Cancel();
				return;
			}
			
			List<VdcActionParametersBase> prms1 = new List<VdcActionParametersBase>();
			foreach (EntityModel a in items)
			{
				bool value = false;
				foreach (network b in networks)
				{
					if (b.Id.Equals(((network)a.Entity).Id))
					{
						value = true;
						break;
					}
				}
				if (a.IsSelected && !value)
				{
					prms1.Add(new AttachNetworkToVdsGroupParameter(Entity, (network)a.Entity));
				}
			}

			// Call the command only if necessary (i.e. only if there are paramters):
			if (prms1.Count > 0)
			{
				Frontend.RunMultipleAction(VdcActionType.AttachNetworkToVdsGroup, prms1);
			}


			List<VdcActionParametersBase> prms2 = new List<VdcActionParametersBase>();
			foreach (EntityModel a in items)
			{
				if (!a.IsSelected)
				{
					prms2.Add(new AttachNetworkToVdsGroupParameter(Entity, (network)a.Entity));
				}
			}

			// Call the command only if necessary (i.e. only if there are paramters):
			if (prms2.Count > 0)
			{
				Frontend.RunMultipleAction(VdcActionType.DetachNetworkToVdsGroup, prms2);
			}

			Cancel();
		}

		public void Cancel()
		{
			Window = null;
		}

		protected override void EntityChanging(object newValue, object oldValue)
		{
			VDSGroup vdsGroup = (VDSGroup)newValue;
			NewNetworkCommand.IsExecutionAllowed = vdsGroup != null
				&& vdsGroup.storage_pool_id != null;
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		private void UpdateActionAvailability()
		{
			network network = (network)SelectedItem;

			//CanRemove = SelectedItems != null && SelectedItems.Count > 0;
			SetAsDisplayCommand.IsExecutionAllowed = SelectedItems != null
				&& SelectedItems.Count == 1
				&& network != null
				&& !network.is_display.GetValueOrDefault()
				&& network.Status != NetworkStatus.NonOperational;
		}

		public void New()
		{
			if (Window != null)
			{
				return;
			}

			ClusterNetworkModel model = new ClusterNetworkModel();
			Window = model;
			model.Title = "New Logical Network";
			model.HashName = "new_logical_network";
			model.IsNew = true;
			if (Entity.storage_pool_id != null)
			{
				storage_pool dataCenter = DataProvider.GetDataCenterById(Entity.storage_pool_id.Value);
				if (dataCenter != null)
				{
					model.DataCenterName = dataCenter.name;
				}
			}
			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnSave()
		{
			ClusterNetworkModel model = (ClusterNetworkModel)Window;
			network network = new network();

			if (Entity == null)
			{
				Cancel();
				return;
			}

			if (!model.Validate() || Entity.storage_pool_id == null)
			{
				return;
			}

			network.storage_pool_id = Entity.storage_pool_id;
			network.name = (string)model.Name.Entity;
			network.stp = model.IsStpEnabled;
			network.description = (string)model.Description.Entity;
			network.vlan_id = null;
			if (model.HasVLanTag)
			{
				network.vlan_id = Convert.ToInt32(model.VLanTag.Entity.ToString());
			}
			VdcReturnValueBase returnValue = Frontend.RunAction(VdcActionType.AddNetwork,
					new AddNetworkStoragePoolParameters(network.storage_pool_id.Value, network));

			if (returnValue != null && returnValue.Succeeded)
			{
				Frontend.RunAction(VdcActionType.AttachNetworkToVdsGroup, new AttachNetworkToVdsGroupParameter(Entity, new network() { Id = (guid)returnValue.ActionReturnValue, name = network.name }));
				Cancel();
			}
		}


		#region ICommandTarget Members

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == ManageCommand)
			{
				Manage();
			}
			else if (command == SetAsDisplayCommand)
			{
				SetAsDisplay();
			}

			else if (command.Name == "OnManage")
			{
				OnManage();
			}
			else if (command.Name == "New")
			{
				New();
			}
			else if (command.Name == "OnSave")
			{
				OnSave();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
		}
		
		#endregion
	}
}
