﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using VdcCommon.BusinessEntities;
using System.Xml;

namespace VdcCommon.Ovf
{
    public abstract class OvfReader : IOvfBuilder
    {        
        protected List<DiskImage> _images;
        protected XmlDocument _document;
        protected XmlNamespaceManager _xmlNS;
        private const int BYTES_IN_GB = 1024 * 1024 * 1024;

        public OvfReader(XmlDocument document, List<DiskImage> images)
        {
            _images = images;
            _document = document;

            _xmlNS = new XmlNamespaceManager(_document.NameTable);
            _xmlNS.AddNamespace("ovf", "http://schemas.dmtf.org/ovf/envelope/1/");
            _xmlNS.AddNamespace("rasd", "http://schemas.dmtf.org/wbem/wscim/1/cim-schema/2/CIM_ResourceAllocationSettingData");
            _xmlNS.AddNamespace("vssd", "http://schemas.dmtf.org/wbem/wscim/1/cim-schema/2/CIM_VirtualSystemSettingData");
            _xmlNS.AddNamespace("xsi", "http://www.w3.org/2001/XMLSchema-instance");
        }

        #region IOvfBuilder Members
        public void BuildReference()
        {
            XmlNodeList list = _document.SelectNodes("//*/File", _xmlNS);
            foreach (XmlNode node in list)
            {
                DiskImage image = new DiskImage();
                image.image_guid = new Guid(node.Attributes["ovf:id"].Value);
                image.image_group_id = OvfParser.GetImageGrupIdFromImageFile(node.Attributes["ovf:href"].Value);
                // Default values:
                image.active = true;
                image.imageStatus = ImageStatus.OK;
                image.description = node.Attributes["ovf:description"].Value;
                _images.Add(image);
            }
        }

        public void BuildNetwork()
        {
        }

        protected long GigabyteToBytes(long gb)
        {
            return gb * BYTES_IN_GB;
        } 
        public void BuildDisk()
        {
            XmlNodeList list = _document.SelectNodes("//*/Section/Disk");
            foreach (XmlNode node in list)
            {
                Guid guid = new Guid(node.Attributes["ovf:diskId"].Value);
                DiskImage image = _images.FirstOrDefault(img => img.image_guid == guid);

                if (node.Attributes["ovf:vm_snapshot_id"] != null)
                {
                    image.vm_snapshot_id = new Guid(node.Attributes["ovf:vm_snapshot_id"].Value);
                }

                if (node.Attributes["ovf:size"] != null && !string.IsNullOrEmpty(node.Attributes["ovf:size"].Value))
                {
                    image.size = GigabyteToBytes(Int64.Parse(node.Attributes["ovf:size"].Value));
                }
                if (node.Attributes["ovf:actual_size"] != null && !string.IsNullOrEmpty(node.Attributes["ovf:actual_size"].Value))
                {
                    image.actual_size = GigabyteToBytes(Int64.Parse(node.Attributes["ovf:actual_size"].Value));
                }
                if (node.Attributes["ovf:volume-format"] != null)
                {
                    if (!string.IsNullOrEmpty(node.Attributes["ovf:volume-format"].Value))
                    {
                        image.volume_format = (VolumeFormat)Enum.Parse(typeof(VolumeFormat), node.Attributes["ovf:volume-format"].Value);
                    }
                    else
                    {
                        image.volume_format = VolumeFormat.Unassigned;
                    }
                }
                if (node.Attributes["ovf:volume-type"] != null)
                {
                    if (!string.IsNullOrEmpty(node.Attributes["ovf:volume-type"].Value))
                    {
                        image.volume_type = (VolumeType)Enum.Parse(typeof(VolumeType), node.Attributes["ovf:volume-type"].Value);
                    }
                    else
                    {
                        image.volume_type = VolumeType.Unassigned;
                    }
                }
                if (node.Attributes["ovf:disk-interface"] != null)
                {
                    if (!string.IsNullOrEmpty(node.Attributes["ovf:disk-interface"].Value))
                    {
                        image.disk_interface = (DiskInterface)Enum.Parse(typeof(DiskInterface), node.Attributes["ovf:disk-interface"].Value);
                    }
                }
                if (node.Attributes["ovf:disk-type"] != null)
                {
                    if (!string.IsNullOrEmpty(node.Attributes["ovf:disk-type"].Value))
                    {
                        image.disk_type = (DiskType)Enum.Parse(typeof(DiskType), node.Attributes["ovf:disk-type"].Value);
                    }
                    else
                    {
                        image.disk_type = DiskType.Unassigned;
                    }
                } 
                if (node.Attributes["ovf:boot"] != null)
                {
                    if (!string.IsNullOrEmpty(node.Attributes["ovf:boot"].Value))
                    {
                        image.boot = Convert.ToBoolean(node.Attributes["ovf:boot"].Value);
                    }
                } 
                if (node.Attributes["ovf:wipe-after-delete"] != null)
                {
                    if (!string.IsNullOrEmpty(node.Attributes["ovf:wipe-after-delete"].Value))
                    {
                        image.wipe_after_delete = Convert.ToBoolean(node.Attributes["ovf:wipe-after-delete"].Value);
                    }
                } 
            }
        }

        public void BuildVirtualSystem()
        {
            ReadGeneralData();
        }
        #endregion        

        protected abstract void ReadOsSection(XmlNode section);

        protected abstract void ReadHardwareSection(XmlNode section);

        protected abstract void ReadGeneralData();
    }
} 
