/*
 * JBoss, the OpenSource EJB server
 * 
 * Distributable under LGPL license. See terms of license at gnu.org.
 */
package javax.resource.spi.work;


/**
 * Interface used to associate the resource adapter with objects that implement
 * this interface.
 */
public interface WorkManager
{
   /** Unknown/unspecified start delay */
   static final long UNKNOWN = -1l;
   /** No start delay */
   static final long IMMEDIATE = 0l;
   /** Indefinite start delay */
   static final long INDEFINITE = Long.MAX_VALUE;
   
   /**
    * Executes the work, the call blocks until the work completes
    *
    * @param work the work
    * @throws WorkException a generic error
    * @throws WorkRejectedException if the work is rejected
    * @throws WorkCompletedException if the work completes with an exception 
    */
   void doWork(Work work) throws WorkException;
   
   /**
    * Executes the work, the call blocks until the work completes
    *
    * @param work the work
    * @param startTimeout the wait before execution
    * @param ctx the execution context
    * @param listener the work listener
    * @throws WorkException a generic error
    * @throws WorkRejectedException if the work is rejected
    * @throws WorkCompletedException if the work completes with an exception 
    */
   void doWork(Work work, long startTimeout, ExecutionContext ctx, WorkListener listener) throws WorkException;
   
   /**
    * Executes the work, the call blocks until the work starts
    *
    * @param work the work
    * @return the time elapsed until the work starts
    * @throws WorkException a generic error
    * @throws WorkRejectedException if the work is rejected
    */
   long startWork(Work work) throws WorkException;
   
   
   /**
    * Executes the work, the call blocks until the work starts
    *
    * @param work the work
    * @param startTimeout the wait before execution
    * @param ctx the execution context
    * @param listener the work listener
    * @return the time elapsed until the work starts
    * @throws WorkException a generic error
    * @throws WorkRejectedException if the work is rejected
    */
   long startWork(Work work, long startTimeout, ExecutionContext ctx, WorkListener listener) throws WorkException;
   
   /**
    * Executes the work, the call returns immediatley
    *
    * @param work the work
    * @throws WorkException a generic error
    * @throws WorkRejectedException if the work is rejected
    */
   void scheduleWork(Work work) throws WorkException;
   
   /**
    * Executes the work, the call returns immediately
    *
    * @param work the work
    * @param startTimeout the wait before execution
    * @param ctx the execution context
    * @param listener the work listener
    * @throws WorkException a generic error
    * @throws WorkRejectedException if the work is rejected
    */
   void scheduleWork(Work work, long startTimeout, ExecutionContext ctx, WorkListener listener) throws WorkException;
}