/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.gd.jpdl.ui.command;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.gef.EditPart;
import org.eclipse.gef.commands.Command;
import org.jbpm.gd.common.notation.Edge;
import org.jbpm.gd.common.notation.Node;
import org.jbpm.gd.common.notation.AbstractNotationElement;
import org.jbpm.gd.common.notation.NotationElementFactory;
import org.jbpm.gd.jpdl.model.AbstractJpdlElement;
import org.jbpm.gd.jpdl.model.AbstractNode;
import org.jbpm.gd.jpdl.model.ActionElement;
import org.jbpm.gd.jpdl.model.Event;
import org.jbpm.gd.jpdl.model.ProcessDefinition;
import org.jbpm.gd.jpdl.model.StartState;
import org.jbpm.gd.jpdl.model.Swimlane;
import org.jbpm.gd.jpdl.model.Task;
import org.jbpm.gd.jpdl.model.TaskNode;
import org.jbpm.gd.jpdl.model.Transition;


public class TreeItemDeleteCommand extends Command {

	private EditPart host;
	private AbstractJpdlElement parent;
	private AbstractJpdlElement child;
	private AbstractNotationElement notationElement;
	private ArrayList childCommands;

	public void setHost(EditPart host) {
		this.host = host;
		this.parent = (AbstractJpdlElement)host.getParent().getModel();
		this.child = (AbstractJpdlElement)host.getModel();		
	}
	
	private void constructChildCommands() {
		childCommands = new ArrayList(); 
		if (notationElement instanceof Node)  {
			List edges = ((Node)notationElement).getArrivingEdges();
			for (int i = 0; i < edges.size(); i++) {
				TreeItemDeleteCommand command = new TreeItemDeleteCommand();
				Edge edge = (Edge)edges.get(i);
				command.setHost((EditPart)host.getViewer().getEditPartRegistry().get(edge.getSemanticElement()));
				childCommands.add(command);
			}
		}
		List children = host.getChildren();
		for (int i = 0; i < children.size(); i++) {
			TreeItemDeleteCommand command = new TreeItemDeleteCommand();
			command.setHost((EditPart)children.get(i));
			childCommands.add(command);
		}
	}
	
	private void executeChildCommands() {
		for (int i = 0; i < childCommands.size(); i++) {
			((TreeItemDeleteCommand)childCommands.get(i)).execute();
		}
	}
	
	public void execute() {	
		notationElement = 
			NotationElementFactory.INSTANCE.getRegisteredNotationElementFor(child);
		if (childCommands == null) {
			constructChildCommands();
		}
		executeChildCommands();
		if (parent instanceof ProcessDefinition) {
			deleteChild((ProcessDefinition)parent);
		} else if (parent instanceof StartState) {
			deleteChild((StartState)parent);
		} else if (parent instanceof TaskNode) {
			deleteChild((TaskNode)parent);
		} else if (parent instanceof AbstractNode) {
			deleteChild((AbstractNode)parent);
		} else if (parent instanceof Event) {
			deleteChild((Event)parent);
		}
	}
	
	private void deleteChild(ProcessDefinition processDefinition) {
		if (child instanceof Task) {
			processDefinition.removeTask((Task)child);
		} else if (child instanceof StartState) {
			processDefinition.removeStartState((StartState)child);
		} else if (child instanceof AbstractNode) {
			processDefinition.removeNodeElement((AbstractNode)child);
		} else if (child instanceof Event) {
			processDefinition.removeEvent((Event)child);
		} else if (child instanceof Swimlane) {
			processDefinition.removeSwimlane((Swimlane)child);
		}
	}
	
	private void deleteChild(StartState startState) {
		if (child instanceof Task) {
			startState.setTask(null);
		} else {
			deleteChild((AbstractNode)startState);
		}
	}
	
	private void deleteChild(TaskNode taskNode) {
		if (child instanceof Task) {
			taskNode.removeTask((Task)child);
		} else {
			deleteChild((AbstractNode)taskNode);
		}
	}
	
	private void deleteChild(AbstractNode abstractNode) {
		if (child instanceof Transition) {
			abstractNode.removeTransition((Transition)child);
		} else if (child instanceof Event) {
			abstractNode.removeEvent((Event)child);
		}
	}
	
	private void deleteChild(Event event) {
		if (child instanceof ActionElement) {
			event.removeActionElement((ActionElement)child);
		}
	}
	
	private void undoChildCommands() {
		for (int i = 0; i < childCommands.size(); i++) {
			((TreeItemDeleteCommand)childCommands.get(i)).undo();
		}
	}
	
	public void undo() {
		if (notationElement != null) {
			notationElement.register();
		}
		if (parent instanceof ProcessDefinition) {
			addChild((ProcessDefinition)parent);
		} else if (parent instanceof StartState) {
			addChild((StartState)parent);
		} else if (parent instanceof TaskNode) {
			addChild((TaskNode)parent);
		} else if (parent instanceof AbstractNode) {
			addChild((AbstractNode)parent);
		} else if (parent instanceof Event) {
			addChild((Event)parent);
		}
		undoChildCommands();
	}

	private void addChild(ProcessDefinition processDefinition) {
		if (child instanceof Task) {
			processDefinition.addTask((Task)child);
		} else if (child instanceof StartState) {
			processDefinition.addStartState((StartState)child);
		} else if (child instanceof AbstractNode) {
			processDefinition.addNodeElement((AbstractNode)child);
		} else if (child instanceof Event) {
			processDefinition.addEvent((Event)child);
		} else if (child instanceof Swimlane) {
			processDefinition.addSwimlane((Swimlane)child);
		}
	}
		
	private void addChild(StartState startState) {
		if (child instanceof Task) {
			startState.setTask((Task)child);
		} else {
			addChild((AbstractNode)startState);
		}
	}
	
	private void addChild(TaskNode taskNode) {
		if (child instanceof Task) {
			taskNode.addTask((Task)child);
		}
	}
	
	private void addChild(AbstractNode abstractNode) {
		if (child instanceof Transition) {
			abstractNode.addTransition((Transition)child);
		} else if (child instanceof Event) {
			abstractNode.addEvent((Event)child);
		} 
	}
	
	private void addChild(Event event) {
		if (child instanceof ActionElement) {
			event.addActionElement((ActionElement)child);
		}
	}
		
}
