/*
 * JBoss, a division of Red Hat
 * Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ide.eclipse.archives.core.project;

import java.util.Comparator;
import java.util.Map;
import java.util.TreeSet;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.jboss.ide.eclipse.archives.core.ArchivesCore;
import org.jboss.ide.eclipse.archives.core.build.ArchiveBuildDelegate;
import org.jboss.ide.eclipse.archives.core.model.ArchivesModel;
import org.jboss.ide.eclipse.archives.core.model.IArchive;
import org.jboss.ide.eclipse.archives.core.model.IArchiveFileSet;
import org.jboss.ide.eclipse.archives.core.model.IArchiveModel;
import org.jboss.ide.eclipse.archives.core.model.IArchiveModelRootNode;
import org.jboss.ide.eclipse.archives.core.model.IArchiveNode;
import org.jboss.ide.eclipse.archives.core.model.IArchiveNodeVisitor;
import org.jboss.ide.eclipse.archives.core.util.internal.TrueZipUtil;

/**
 * This builder is responsible for building the archives
 * It delegates to a delegate. 
 * @author Rob Stryker (rob.stryker@redhat.com)
 *
 */
public class ArchivesBuilder extends IncrementalProjectBuilder {

	public static final String BUILDER_ID = "org.jboss.ide.eclipse.archives.core.archivesBuilder";
	
	/**
	 * Build. 
	 * @see IncrementalProjectBuilder#build(int, Map, IProgressMonitor)
	 * @see IProject#build(int, String, Map, IProgressMonitor) 
	 */
	protected IProject[] build(int kind, Map args, IProgressMonitor monitor) throws CoreException {
		
		// if we're not to build, get out of here
		if( !ArchivesCore.getInstance().getPreferenceManager().isBuilderEnabled(getProject().getLocation())) 
			return new IProject[]{};
		
		IProject[] interestingProjects = getInterestingProjectsInternal();

		final TreeSet addedChanged = createDefaultTreeSet();
		final TreeSet removed = createDefaultTreeSet();

		ArchiveBuildDelegate delegate = new ArchiveBuildDelegate();
		if (kind == IncrementalProjectBuilder.INCREMENTAL_BUILD || kind == IncrementalProjectBuilder.AUTO_BUILD) {
			fillDeltas(interestingProjects, addedChanged, removed);
			delegate.projectIncrementalBuild(addedChanged, removed);
			
		} else if (kind == IncrementalProjectBuilder.FULL_BUILD){
			// build each package fully
			IProject p = getProject();
			delegate.fullProjectBuild(p.getLocation());
		}
		
		return interestingProjects;
	}
	
	/**
	 * Delete all archives that were created or represented by this
	 * project's archive model. 
	 */
	protected void clean(IProgressMonitor monitor) throws CoreException {
		IPath p = getProject().getLocation();
		IArchiveModelRootNode root = ArchivesModel.instance().getRoot(p);
		if(root!=null) {
			IArchiveNode[] nodes = root.getChildren(IArchiveNode.TYPE_ARCHIVE);
			for( int i = 0; i < nodes.length; i++ ) {
				IPath path = ((IArchive)nodes[i]).getArchiveFilePath();
				TrueZipUtil.deleteAll(path);
			}
		}
	}

	/**
	 * Browse through the deltas and fill the treesets with
	 * affected resources. 
	 * @param projects The interesting projects
	 * @param addedChanged A collection of resources that have been added or changed
	 * @param removed A collection of resources that have been removed.
	 */
	protected void fillDeltas(IProject[] projects, final TreeSet addedChanged, final TreeSet removed) {
		for( int i = 0; i < projects.length; i++ ) {
			final IProject proj = projects[i];
			IResourceDelta delta = getDelta(proj);
			if( delta == null ) continue;
			try {
				delta.accept(new IResourceDeltaVisitor () { 
					public boolean visit(IResourceDelta delta) throws CoreException {
						if (delta.getResource().getType() == IResource.FILE)  {
							if( (delta.getKind() & IResourceDelta.ADDED) > 0  
									|| (delta.getKind() & IResourceDelta.CHANGED) > 0) {
								
								// ignore the packages project. that will it's own build call, 
								// or will handle the change in some other way
								if( !delta.getResource().equals(proj.findMember(IArchiveModel.DEFAULT_PACKAGES_FILE))) 
									addedChanged.add(delta.getResource().getLocation());
							} else if( (delta.getKind() & IResourceDelta.REMOVED ) > 0 ) {
								removed.add(delta.getResource().getLocation());
							}
						} 
						return true;
					}
				});
			} catch( CoreException ce) {
			}
		}
	}
	
	/**
	 * Get any projects that the current project may depend on 
	 * (regarding it's archive model, if any filesets match files in 
	 * another project).
	 * @return The list of projects that matter
	 */
	protected IProject[] getInterestingProjectsInternal() {
		final TreeSet set = createDefaultTreeSet();
		set.add(getProject());
		
		final IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
		final int count = workspaceRoot.getLocation().segmentCount();

		IArchiveModelRootNode root = ArchivesModel.instance().getRoot(getProject().getLocation());
		if(root!=null) {
			root.accept(new IArchiveNodeVisitor () {
				public boolean visit (IArchiveNode node) {
					if (node.getNodeType() == IArchiveNode.TYPE_ARCHIVE_FILESET) {
						IArchiveFileSet fileset = (IArchiveFileSet)node;
						IPath p = fileset.getGlobalSourcePath();
						if( p != null ) {
							if( workspaceRoot.getLocation().isPrefixOf(p)) {
								IProject proj = workspaceRoot.getProject(p.segment(count));
								set.add(proj);
							}
						}
					}
					return true;
				}
			});
			return (IProject[]) set.toArray(new IProject[set.size()]);
		} else {
			return new IProject[0];
		}
	}
	
	/** 
	 * Default treeset with default comparator
	 * @return
	 */
	protected TreeSet createDefaultTreeSet() {
		return new TreeSet(new Comparator () {
			public int compare(Object o1, Object o2) {
				if (o1.equals(o2)) return 0;
				else return -1;
			}
		});		
	}
	
}