;; Various helper functions

(defmacro call-with-i18n-file (file mode encoding . thunk)
  `(let ((p #f))
     (dynamic-wind
      (lambda () (set! p (open-file ,file ,mode ,encoding)))
      (lambda () (,@thunk p))
      (lambda () (close-port p)))))

(defmacro call-with-junet-input-file (file . thunk)
  `(call-with-i18n-file ,file OPEN_READ *junet* ,@thunk))

(defmacro call-with-junet-output-file (file . thunk)
  `(call-with-i18n-file ,file OPEN_WRITE *junet* ,@thunk))

(defmacro call-with-iso-8859-1-input-file (file . thunk)
  `(call-with-i18n-file ,file OPEN_READ *iso-8859-1* ,@thunk))

(defmacro call-with-iso-8859-1-output-file (file . thunk)
  `(call-with-i18n-file ,file OPEN_WRITE *iso-8859-1* ,@thunk))

(defmacro call-with-euc-japan-input-file (file . thunk)
  `(call-with-i18n-file ,file OPEN_READ *euc-japan* ,@thunk))

(defmacro call-with-euc-japan-output-file (file . thunk)
  `(call-with-i18n-file ,file OPEN_WRITE *euc-japan* ,@thunk))

(defmacro call-with-sjis-input-file (file . thunk)
  `(call-with-i18n-file ,file OPEN_READ *sjis* ,@thunk))

(defmacro call-with-sjis-output-file (file . thunk)
  `(call-with-i18n-file ,file OPEN_WRITE *sjis* ,@thunk))


;; Given a junet encoded text file, return a string
;; containing the contents of that file.

(define (junet-file->string file)
  (call-with-junet-input-file file
    (lambda (p)
      (let ((s ""))
	(letrec ((read-loop 
		  (lambda (s) 
		    (let ((c (read-char p)))
		      (if (not (eof-object? c))
			  (read-loop (string-append s (string c)))
			  s)))))
	  (read-loop s))))))

;; Given a latin1 encoded text file, return a string
;; containing the contents of that file.

(define (latin1-file->string file)
  (call-with-iso-8859-1-input-file file
    (lambda (p)
      (let ((s ""))
        (letrec ((read-loop
                  (lambda (s)
                    (let ((c (read-char p)))
                      (if (not (eof-object? c))
                          (read-loop (string-append s (string c)))
                          s)))))
          (read-loop s))))))

;; Load a junet encoded text FILE into a string. 
;; Write and read this string into different files with
;; different encodings. 
;; Compare the end result string with the initial string.

(define (japanese-file-test file)
  (let ((n (junet-file->string file)))

    (call-with-junet-output-file "tmp.junet"
      (lambda (p)
	(write n p)))

    (call-with-euc-japan-output-file "tmp.euc-japan"
      (lambda (p)
	(write
	 (call-with-junet-input-file "tmp.junet"
				     (lambda (p)
				       (read p)))
	 p)))

    (call-with-sjis-output-file "tmp.sjis"
      (lambda (p)
	(write
	 (call-with-euc-japan-input-file "tmp.euc-japan"
					 (lambda (p)
					   (read p)))
	 p)))

    (if (string=? n (call-with-sjis-input-file "tmp.sjis"
					       (lambda (p)
						 (read p))))
	'pass
	'fail)))

;; Load a latin1 encoded text FILE into a string. 
;; Write and read this string to and from a file.
;; Compare the end result string with the initial string.

(define (latin1-file-test file)
  (let ((n (latin1-file->string file))) 

    (call-with-iso-8859-1-output-file "tmp.iso-8859-1"
      (lambda (p)
	(write n p)))

    (if (string=? n (call-with-iso-8859-1-input-file "tmp.iso-8859-1"
						     (lambda (p)
						       (read p))))
	'pass
	'fail)))
						    