/* ========================================================================
 * Copyright (c) 2005-2021 The OPC Foundation, Inc. All rights reserved.
 *
 * OPC Foundation MIT License 1.00
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * The complete license agreement can be found here:
 * http://opcfoundation.org/License/MIT/1.00/
 * ======================================================================*/

using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using System.Runtime.Serialization;

namespace Opc.Ua
{
    #region Enumeration Enumeration
    #if (!OPCUA_EXCLUDE_Enumeration)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum Enumeration
    {
    }

    #region EnumerationCollection Class
    /// <summary>
    /// A collection of Enumeration objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfEnumeration", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "Enumeration")]
    #if !NET_STANDARD
    public partial class EnumerationCollection : List<Enumeration>, ICloneable
    #else
    public partial class EnumerationCollection : List<Enumeration>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public EnumerationCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public EnumerationCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public EnumerationCollection(IEnumerable<Enumeration> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator EnumerationCollection(Enumeration[] values)
        {
            if (values != null)
            {
                return new EnumerationCollection(values);
            }

            return new EnumerationCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator Enumeration[](EnumerationCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (EnumerationCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EnumerationCollection clone = new EnumerationCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((Enumeration)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region NamingRuleType Enumeration
    #if (!OPCUA_EXCLUDE_NamingRuleType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum NamingRuleType
    {
        /// <remarks />
        [EnumMember(Value = "Mandatory_1")]
        Mandatory = 1,

        /// <remarks />
        [EnumMember(Value = "Optional_2")]
        Optional = 2,

        /// <remarks />
        [EnumMember(Value = "Constraint_3")]
        Constraint = 3,
    }

    #region NamingRuleTypeCollection Class
    /// <summary>
    /// A collection of NamingRuleType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfNamingRuleType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "NamingRuleType")]
    #if !NET_STANDARD
    public partial class NamingRuleTypeCollection : List<NamingRuleType>, ICloneable
    #else
    public partial class NamingRuleTypeCollection : List<NamingRuleType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public NamingRuleTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public NamingRuleTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public NamingRuleTypeCollection(IEnumerable<NamingRuleType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator NamingRuleTypeCollection(NamingRuleType[] values)
        {
            if (values != null)
            {
                return new NamingRuleTypeCollection(values);
            }

            return new NamingRuleTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator NamingRuleType[](NamingRuleTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (NamingRuleTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            NamingRuleTypeCollection clone = new NamingRuleTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((NamingRuleType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region Union Class
    #if (!OPCUA_EXCLUDE_Union)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class Union : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public Union()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.Union; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.Union_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.Union_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            Union value = encodeable as Union;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (Union)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            Union clone = (Union)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region UnionCollection Class
    /// <summary>
    /// A collection of Union objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUnion", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "Union")]
    #if !NET_STANDARD
    public partial class UnionCollection : List<Union>, ICloneable
    #else
    public partial class UnionCollection : List<Union>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public UnionCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public UnionCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public UnionCollection(IEnumerable<Union> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator UnionCollection(Union[] values)
        {
            if (values != null)
            {
                return new UnionCollection(values);
            }

            return new UnionCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator Union[](UnionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (UnionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UnionCollection clone = new UnionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((Union)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region KeyValuePair Class
    #if (!OPCUA_EXCLUDE_KeyValuePair)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class KeyValuePair : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public KeyValuePair()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_key = null;
            m_value = Variant.Null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Key", IsRequired = false, Order = 1)]
        public QualifiedName Key
        {
            get { return m_key;  }
            set { m_key = value; }
        }

        /// <remarks />
        [DataMember(Name = "Value", IsRequired = false, Order = 2)]
        public Variant Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.KeyValuePair; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.KeyValuePair_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.KeyValuePair_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteQualifiedName("Key", Key);
            encoder.WriteVariant("Value", Value);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Key = decoder.ReadQualifiedName("Key");
            Value = decoder.ReadVariant("Value");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            KeyValuePair value = encodeable as KeyValuePair;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_key, value.m_key)) return false;
            if (!Utils.IsEqual(m_value, value.m_value)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (KeyValuePair)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            KeyValuePair clone = (KeyValuePair)base.MemberwiseClone();

            clone.m_key = (QualifiedName)Utils.Clone(this.m_key);
            clone.m_value = (Variant)Utils.Clone(this.m_value);

            return clone;
        }
        #endregion

        #region Private Fields
        private QualifiedName m_key;
        private Variant m_value;
        #endregion
    }

    #region KeyValuePairCollection Class
    /// <summary>
    /// A collection of KeyValuePair objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfKeyValuePair", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "KeyValuePair")]
    #if !NET_STANDARD
    public partial class KeyValuePairCollection : List<KeyValuePair>, ICloneable
    #else
    public partial class KeyValuePairCollection : List<KeyValuePair>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public KeyValuePairCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public KeyValuePairCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public KeyValuePairCollection(IEnumerable<KeyValuePair> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator KeyValuePairCollection(KeyValuePair[] values)
        {
            if (values != null)
            {
                return new KeyValuePairCollection(values);
            }

            return new KeyValuePairCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator KeyValuePair[](KeyValuePairCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (KeyValuePairCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            KeyValuePairCollection clone = new KeyValuePairCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((KeyValuePair)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region AdditionalParametersType Class
    #if (!OPCUA_EXCLUDE_AdditionalParametersType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class AdditionalParametersType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public AdditionalParametersType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_parameters = new KeyValuePairCollection();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "Parameters", IsRequired = false, Order = 1)]
        public KeyValuePairCollection Parameters
        {
            get
            {
                return m_parameters;
            }

            set
            {
                m_parameters = value;

                if (value == null)
                {
                    m_parameters = new KeyValuePairCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.AdditionalParametersType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.AdditionalParametersType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.AdditionalParametersType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeableArray("Parameters", Parameters.ToArray(), typeof(KeyValuePair));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Parameters = (KeyValuePairCollection)decoder.ReadEncodeableArray("Parameters", typeof(KeyValuePair));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            AdditionalParametersType value = encodeable as AdditionalParametersType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_parameters, value.m_parameters)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (AdditionalParametersType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            AdditionalParametersType clone = (AdditionalParametersType)base.MemberwiseClone();

            clone.m_parameters = (KeyValuePairCollection)Utils.Clone(this.m_parameters);

            return clone;
        }
        #endregion

        #region Private Fields
        private KeyValuePairCollection m_parameters;
        #endregion
    }
    #endif
    #endregion

    #region EphemeralKeyType Class
    #if (!OPCUA_EXCLUDE_EphemeralKeyType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class EphemeralKeyType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public EphemeralKeyType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_publicKey = null;
            m_signature = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "PublicKey", IsRequired = false, Order = 1)]
        public byte[] PublicKey
        {
            get { return m_publicKey;  }
            set { m_publicKey = value; }
        }

        /// <remarks />
        [DataMember(Name = "Signature", IsRequired = false, Order = 2)]
        public byte[] Signature
        {
            get { return m_signature;  }
            set { m_signature = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.EphemeralKeyType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.EphemeralKeyType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.EphemeralKeyType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteByteString("PublicKey", PublicKey);
            encoder.WriteByteString("Signature", Signature);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            PublicKey = decoder.ReadByteString("PublicKey");
            Signature = decoder.ReadByteString("Signature");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            EphemeralKeyType value = encodeable as EphemeralKeyType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_publicKey, value.m_publicKey)) return false;
            if (!Utils.IsEqual(m_signature, value.m_signature)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (EphemeralKeyType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EphemeralKeyType clone = (EphemeralKeyType)base.MemberwiseClone();

            clone.m_publicKey = (byte[])Utils.Clone(this.m_publicKey);
            clone.m_signature = (byte[])Utils.Clone(this.m_signature);

            return clone;
        }
        #endregion

        #region Private Fields
        private byte[] m_publicKey;
        private byte[] m_signature;
        #endregion
    }
    #endif
    #endregion

    #region EndpointType Class
    #if (!OPCUA_EXCLUDE_EndpointType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class EndpointType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public EndpointType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_endpointUrl = null;
            m_securityMode = MessageSecurityMode.Invalid;
            m_securityPolicyUri = null;
            m_transportProfileUri = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "EndpointUrl", IsRequired = false, Order = 1)]
        public string EndpointUrl
        {
            get { return m_endpointUrl;  }
            set { m_endpointUrl = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityMode", IsRequired = false, Order = 2)]
        public MessageSecurityMode SecurityMode
        {
            get { return m_securityMode;  }
            set { m_securityMode = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityPolicyUri", IsRequired = false, Order = 3)]
        public string SecurityPolicyUri
        {
            get { return m_securityPolicyUri;  }
            set { m_securityPolicyUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "TransportProfileUri", IsRequired = false, Order = 4)]
        public string TransportProfileUri
        {
            get { return m_transportProfileUri;  }
            set { m_transportProfileUri = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.EndpointType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.EndpointType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.EndpointType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("EndpointUrl", EndpointUrl);
            encoder.WriteEnumerated("SecurityMode", SecurityMode);
            encoder.WriteString("SecurityPolicyUri", SecurityPolicyUri);
            encoder.WriteString("TransportProfileUri", TransportProfileUri);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            EndpointUrl = decoder.ReadString("EndpointUrl");
            SecurityMode = (MessageSecurityMode)decoder.ReadEnumerated("SecurityMode", typeof(MessageSecurityMode));
            SecurityPolicyUri = decoder.ReadString("SecurityPolicyUri");
            TransportProfileUri = decoder.ReadString("TransportProfileUri");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            EndpointType value = encodeable as EndpointType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_endpointUrl, value.m_endpointUrl)) return false;
            if (!Utils.IsEqual(m_securityMode, value.m_securityMode)) return false;
            if (!Utils.IsEqual(m_securityPolicyUri, value.m_securityPolicyUri)) return false;
            if (!Utils.IsEqual(m_transportProfileUri, value.m_transportProfileUri)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (EndpointType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EndpointType clone = (EndpointType)base.MemberwiseClone();

            clone.m_endpointUrl = (string)Utils.Clone(this.m_endpointUrl);
            clone.m_securityMode = (MessageSecurityMode)Utils.Clone(this.m_securityMode);
            clone.m_securityPolicyUri = (string)Utils.Clone(this.m_securityPolicyUri);
            clone.m_transportProfileUri = (string)Utils.Clone(this.m_transportProfileUri);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_endpointUrl;
        private MessageSecurityMode m_securityMode;
        private string m_securityPolicyUri;
        private string m_transportProfileUri;
        #endregion
    }

    #region EndpointTypeCollection Class
    /// <summary>
    /// A collection of EndpointType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfEndpointType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "EndpointType")]
    #if !NET_STANDARD
    public partial class EndpointTypeCollection : List<EndpointType>, ICloneable
    #else
    public partial class EndpointTypeCollection : List<EndpointType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public EndpointTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public EndpointTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public EndpointTypeCollection(IEnumerable<EndpointType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator EndpointTypeCollection(EndpointType[] values)
        {
            if (values != null)
            {
                return new EndpointTypeCollection(values);
            }

            return new EndpointTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator EndpointType[](EndpointTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (EndpointTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EndpointTypeCollection clone = new EndpointTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((EndpointType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region RationalNumber Class
    #if (!OPCUA_EXCLUDE_RationalNumber)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class RationalNumber : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public RationalNumber()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_numerator = (int)0;
            m_denominator = (uint)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Numerator", IsRequired = false, Order = 1)]
        public int Numerator
        {
            get { return m_numerator;  }
            set { m_numerator = value; }
        }

        /// <remarks />
        [DataMember(Name = "Denominator", IsRequired = false, Order = 2)]
        public uint Denominator
        {
            get { return m_denominator;  }
            set { m_denominator = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.RationalNumber; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.RationalNumber_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.RationalNumber_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteInt32("Numerator", Numerator);
            encoder.WriteUInt32("Denominator", Denominator);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Numerator = decoder.ReadInt32("Numerator");
            Denominator = decoder.ReadUInt32("Denominator");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            RationalNumber value = encodeable as RationalNumber;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_numerator, value.m_numerator)) return false;
            if (!Utils.IsEqual(m_denominator, value.m_denominator)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (RationalNumber)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            RationalNumber clone = (RationalNumber)base.MemberwiseClone();

            clone.m_numerator = (int)Utils.Clone(this.m_numerator);
            clone.m_denominator = (uint)Utils.Clone(this.m_denominator);

            return clone;
        }
        #endregion

        #region Private Fields
        private int m_numerator;
        private uint m_denominator;
        #endregion
    }

    #region RationalNumberCollection Class
    /// <summary>
    /// A collection of RationalNumber objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfRationalNumber", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "RationalNumber")]
    #if !NET_STANDARD
    public partial class RationalNumberCollection : List<RationalNumber>, ICloneable
    #else
    public partial class RationalNumberCollection : List<RationalNumber>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public RationalNumberCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public RationalNumberCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public RationalNumberCollection(IEnumerable<RationalNumber> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator RationalNumberCollection(RationalNumber[] values)
        {
            if (values != null)
            {
                return new RationalNumberCollection(values);
            }

            return new RationalNumberCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator RationalNumber[](RationalNumberCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (RationalNumberCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            RationalNumberCollection clone = new RationalNumberCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((RationalNumber)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region Vector Class
    #if (!OPCUA_EXCLUDE_Vector)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class Vector : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public Vector()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.Vector; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.Vector_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.Vector_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            Vector value = encodeable as Vector;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (Vector)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            Vector clone = (Vector)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region VectorCollection Class
    /// <summary>
    /// A collection of Vector objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfVector", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "Vector")]
    #if !NET_STANDARD
    public partial class VectorCollection : List<Vector>, ICloneable
    #else
    public partial class VectorCollection : List<Vector>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public VectorCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public VectorCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public VectorCollection(IEnumerable<Vector> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator VectorCollection(Vector[] values)
        {
            if (values != null)
            {
                return new VectorCollection(values);
            }

            return new VectorCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator Vector[](VectorCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (VectorCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            VectorCollection clone = new VectorCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((Vector)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ThreeDVector Class
    #if (!OPCUA_EXCLUDE_ThreeDVector)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ThreeDVector : Vector
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ThreeDVector()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_x = (double)0;
            m_y = (double)0;
            m_z = (double)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "X", IsRequired = false, Order = 1)]
        public double X
        {
            get { return m_x;  }
            set { m_x = value; }
        }

        /// <remarks />
        [DataMember(Name = "Y", IsRequired = false, Order = 2)]
        public double Y
        {
            get { return m_y;  }
            set { m_y = value; }
        }

        /// <remarks />
        [DataMember(Name = "Z", IsRequired = false, Order = 3)]
        public double Z
        {
            get { return m_z;  }
            set { m_z = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ThreeDVector; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ThreeDVector_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ThreeDVector_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteDouble("X", X);
            encoder.WriteDouble("Y", Y);
            encoder.WriteDouble("Z", Z);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            X = decoder.ReadDouble("X");
            Y = decoder.ReadDouble("Y");
            Z = decoder.ReadDouble("Z");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ThreeDVector value = encodeable as ThreeDVector;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_x, value.m_x)) return false;
            if (!Utils.IsEqual(m_y, value.m_y)) return false;
            if (!Utils.IsEqual(m_z, value.m_z)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ThreeDVector)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ThreeDVector clone = (ThreeDVector)base.MemberwiseClone();

            clone.m_x = (double)Utils.Clone(this.m_x);
            clone.m_y = (double)Utils.Clone(this.m_y);
            clone.m_z = (double)Utils.Clone(this.m_z);

            return clone;
        }
        #endregion

        #region Private Fields
        private double m_x;
        private double m_y;
        private double m_z;
        #endregion
    }

    #region ThreeDVectorCollection Class
    /// <summary>
    /// A collection of ThreeDVector objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfThreeDVector", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ThreeDVector")]
    #if !NET_STANDARD
    public partial class ThreeDVectorCollection : List<ThreeDVector>, ICloneable
    #else
    public partial class ThreeDVectorCollection : List<ThreeDVector>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public ThreeDVectorCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public ThreeDVectorCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public ThreeDVectorCollection(IEnumerable<ThreeDVector> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator ThreeDVectorCollection(ThreeDVector[] values)
        {
            if (values != null)
            {
                return new ThreeDVectorCollection(values);
            }

            return new ThreeDVectorCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator ThreeDVector[](ThreeDVectorCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (ThreeDVectorCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ThreeDVectorCollection clone = new ThreeDVectorCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ThreeDVector)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region CartesianCoordinates Class
    #if (!OPCUA_EXCLUDE_CartesianCoordinates)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class CartesianCoordinates : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public CartesianCoordinates()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.CartesianCoordinates; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.CartesianCoordinates_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.CartesianCoordinates_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            CartesianCoordinates value = encodeable as CartesianCoordinates;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (CartesianCoordinates)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            CartesianCoordinates clone = (CartesianCoordinates)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region CartesianCoordinatesCollection Class
    /// <summary>
    /// A collection of CartesianCoordinates objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfCartesianCoordinates", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "CartesianCoordinates")]
    #if !NET_STANDARD
    public partial class CartesianCoordinatesCollection : List<CartesianCoordinates>, ICloneable
    #else
    public partial class CartesianCoordinatesCollection : List<CartesianCoordinates>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public CartesianCoordinatesCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public CartesianCoordinatesCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public CartesianCoordinatesCollection(IEnumerable<CartesianCoordinates> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator CartesianCoordinatesCollection(CartesianCoordinates[] values)
        {
            if (values != null)
            {
                return new CartesianCoordinatesCollection(values);
            }

            return new CartesianCoordinatesCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator CartesianCoordinates[](CartesianCoordinatesCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (CartesianCoordinatesCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            CartesianCoordinatesCollection clone = new CartesianCoordinatesCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((CartesianCoordinates)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ThreeDCartesianCoordinates Class
    #if (!OPCUA_EXCLUDE_ThreeDCartesianCoordinates)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ThreeDCartesianCoordinates : CartesianCoordinates
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ThreeDCartesianCoordinates()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_x = (double)0;
            m_y = (double)0;
            m_z = (double)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "X", IsRequired = false, Order = 1)]
        public double X
        {
            get { return m_x;  }
            set { m_x = value; }
        }

        /// <remarks />
        [DataMember(Name = "Y", IsRequired = false, Order = 2)]
        public double Y
        {
            get { return m_y;  }
            set { m_y = value; }
        }

        /// <remarks />
        [DataMember(Name = "Z", IsRequired = false, Order = 3)]
        public double Z
        {
            get { return m_z;  }
            set { m_z = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ThreeDCartesianCoordinates; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ThreeDCartesianCoordinates_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ThreeDCartesianCoordinates_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteDouble("X", X);
            encoder.WriteDouble("Y", Y);
            encoder.WriteDouble("Z", Z);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            X = decoder.ReadDouble("X");
            Y = decoder.ReadDouble("Y");
            Z = decoder.ReadDouble("Z");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ThreeDCartesianCoordinates value = encodeable as ThreeDCartesianCoordinates;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_x, value.m_x)) return false;
            if (!Utils.IsEqual(m_y, value.m_y)) return false;
            if (!Utils.IsEqual(m_z, value.m_z)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ThreeDCartesianCoordinates)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ThreeDCartesianCoordinates clone = (ThreeDCartesianCoordinates)base.MemberwiseClone();

            clone.m_x = (double)Utils.Clone(this.m_x);
            clone.m_y = (double)Utils.Clone(this.m_y);
            clone.m_z = (double)Utils.Clone(this.m_z);

            return clone;
        }
        #endregion

        #region Private Fields
        private double m_x;
        private double m_y;
        private double m_z;
        #endregion
    }

    #region ThreeDCartesianCoordinatesCollection Class
    /// <summary>
    /// A collection of ThreeDCartesianCoordinates objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfThreeDCartesianCoordinates", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ThreeDCartesianCoordinates")]
    #if !NET_STANDARD
    public partial class ThreeDCartesianCoordinatesCollection : List<ThreeDCartesianCoordinates>, ICloneable
    #else
    public partial class ThreeDCartesianCoordinatesCollection : List<ThreeDCartesianCoordinates>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public ThreeDCartesianCoordinatesCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public ThreeDCartesianCoordinatesCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public ThreeDCartesianCoordinatesCollection(IEnumerable<ThreeDCartesianCoordinates> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator ThreeDCartesianCoordinatesCollection(ThreeDCartesianCoordinates[] values)
        {
            if (values != null)
            {
                return new ThreeDCartesianCoordinatesCollection(values);
            }

            return new ThreeDCartesianCoordinatesCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator ThreeDCartesianCoordinates[](ThreeDCartesianCoordinatesCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (ThreeDCartesianCoordinatesCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ThreeDCartesianCoordinatesCollection clone = new ThreeDCartesianCoordinatesCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ThreeDCartesianCoordinates)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region Orientation Class
    #if (!OPCUA_EXCLUDE_Orientation)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class Orientation : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public Orientation()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.Orientation; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.Orientation_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.Orientation_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            Orientation value = encodeable as Orientation;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (Orientation)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            Orientation clone = (Orientation)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region OrientationCollection Class
    /// <summary>
    /// A collection of Orientation objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfOrientation", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "Orientation")]
    #if !NET_STANDARD
    public partial class OrientationCollection : List<Orientation>, ICloneable
    #else
    public partial class OrientationCollection : List<Orientation>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public OrientationCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public OrientationCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public OrientationCollection(IEnumerable<Orientation> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator OrientationCollection(Orientation[] values)
        {
            if (values != null)
            {
                return new OrientationCollection(values);
            }

            return new OrientationCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator Orientation[](OrientationCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (OrientationCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            OrientationCollection clone = new OrientationCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((Orientation)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ThreeDOrientation Class
    #if (!OPCUA_EXCLUDE_ThreeDOrientation)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ThreeDOrientation : Orientation
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ThreeDOrientation()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_a = (double)0;
            m_b = (double)0;
            m_c = (double)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "A", IsRequired = false, Order = 1)]
        public double A
        {
            get { return m_a;  }
            set { m_a = value; }
        }

        /// <remarks />
        [DataMember(Name = "B", IsRequired = false, Order = 2)]
        public double B
        {
            get { return m_b;  }
            set { m_b = value; }
        }

        /// <remarks />
        [DataMember(Name = "C", IsRequired = false, Order = 3)]
        public double C
        {
            get { return m_c;  }
            set { m_c = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ThreeDOrientation; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ThreeDOrientation_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ThreeDOrientation_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteDouble("A", A);
            encoder.WriteDouble("B", B);
            encoder.WriteDouble("C", C);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            A = decoder.ReadDouble("A");
            B = decoder.ReadDouble("B");
            C = decoder.ReadDouble("C");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ThreeDOrientation value = encodeable as ThreeDOrientation;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_a, value.m_a)) return false;
            if (!Utils.IsEqual(m_b, value.m_b)) return false;
            if (!Utils.IsEqual(m_c, value.m_c)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ThreeDOrientation)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ThreeDOrientation clone = (ThreeDOrientation)base.MemberwiseClone();

            clone.m_a = (double)Utils.Clone(this.m_a);
            clone.m_b = (double)Utils.Clone(this.m_b);
            clone.m_c = (double)Utils.Clone(this.m_c);

            return clone;
        }
        #endregion

        #region Private Fields
        private double m_a;
        private double m_b;
        private double m_c;
        #endregion
    }

    #region ThreeDOrientationCollection Class
    /// <summary>
    /// A collection of ThreeDOrientation objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfThreeDOrientation", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ThreeDOrientation")]
    #if !NET_STANDARD
    public partial class ThreeDOrientationCollection : List<ThreeDOrientation>, ICloneable
    #else
    public partial class ThreeDOrientationCollection : List<ThreeDOrientation>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public ThreeDOrientationCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public ThreeDOrientationCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public ThreeDOrientationCollection(IEnumerable<ThreeDOrientation> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator ThreeDOrientationCollection(ThreeDOrientation[] values)
        {
            if (values != null)
            {
                return new ThreeDOrientationCollection(values);
            }

            return new ThreeDOrientationCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator ThreeDOrientation[](ThreeDOrientationCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (ThreeDOrientationCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ThreeDOrientationCollection clone = new ThreeDOrientationCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ThreeDOrientation)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region Frame Class
    #if (!OPCUA_EXCLUDE_Frame)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class Frame : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public Frame()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.Frame; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.Frame_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.Frame_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            Frame value = encodeable as Frame;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (Frame)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            Frame clone = (Frame)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region FrameCollection Class
    /// <summary>
    /// A collection of Frame objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfFrame", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "Frame")]
    #if !NET_STANDARD
    public partial class FrameCollection : List<Frame>, ICloneable
    #else
    public partial class FrameCollection : List<Frame>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public FrameCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public FrameCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public FrameCollection(IEnumerable<Frame> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator FrameCollection(Frame[] values)
        {
            if (values != null)
            {
                return new FrameCollection(values);
            }

            return new FrameCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator Frame[](FrameCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (FrameCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            FrameCollection clone = new FrameCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((Frame)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ThreeDFrame Class
    #if (!OPCUA_EXCLUDE_ThreeDFrame)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ThreeDFrame : Frame
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ThreeDFrame()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_cartesianCoordinates = new ThreeDCartesianCoordinates();
            m_orientation = new ThreeDOrientation();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "CartesianCoordinates", IsRequired = false, Order = 1)]
        public ThreeDCartesianCoordinates CartesianCoordinates
        {
            get
            {
                return m_cartesianCoordinates;
            }

            set
            {
                m_cartesianCoordinates = value;

                if (value == null)
                {
                    m_cartesianCoordinates = new ThreeDCartesianCoordinates();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "Orientation", IsRequired = false, Order = 2)]
        public ThreeDOrientation Orientation
        {
            get
            {
                return m_orientation;
            }

            set
            {
                m_orientation = value;

                if (value == null)
                {
                    m_orientation = new ThreeDOrientation();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ThreeDFrame; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ThreeDFrame_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ThreeDFrame_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("CartesianCoordinates", CartesianCoordinates, typeof(ThreeDCartesianCoordinates));
            encoder.WriteEncodeable("Orientation", Orientation, typeof(ThreeDOrientation));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            CartesianCoordinates = (ThreeDCartesianCoordinates)decoder.ReadEncodeable("CartesianCoordinates", typeof(ThreeDCartesianCoordinates));
            Orientation = (ThreeDOrientation)decoder.ReadEncodeable("Orientation", typeof(ThreeDOrientation));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ThreeDFrame value = encodeable as ThreeDFrame;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_cartesianCoordinates, value.m_cartesianCoordinates)) return false;
            if (!Utils.IsEqual(m_orientation, value.m_orientation)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ThreeDFrame)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ThreeDFrame clone = (ThreeDFrame)base.MemberwiseClone();

            clone.m_cartesianCoordinates = (ThreeDCartesianCoordinates)Utils.Clone(this.m_cartesianCoordinates);
            clone.m_orientation = (ThreeDOrientation)Utils.Clone(this.m_orientation);

            return clone;
        }
        #endregion

        #region Private Fields
        private ThreeDCartesianCoordinates m_cartesianCoordinates;
        private ThreeDOrientation m_orientation;
        #endregion
    }

    #region ThreeDFrameCollection Class
    /// <summary>
    /// A collection of ThreeDFrame objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfThreeDFrame", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ThreeDFrame")]
    #if !NET_STANDARD
    public partial class ThreeDFrameCollection : List<ThreeDFrame>, ICloneable
    #else
    public partial class ThreeDFrameCollection : List<ThreeDFrame>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public ThreeDFrameCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public ThreeDFrameCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public ThreeDFrameCollection(IEnumerable<ThreeDFrame> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator ThreeDFrameCollection(ThreeDFrame[] values)
        {
            if (values != null)
            {
                return new ThreeDFrameCollection(values);
            }

            return new ThreeDFrameCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator ThreeDFrame[](ThreeDFrameCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (ThreeDFrameCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ThreeDFrameCollection clone = new ThreeDFrameCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ThreeDFrame)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region OpenFileMode Enumeration
    #if (!OPCUA_EXCLUDE_OpenFileMode)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum OpenFileMode
    {
        /// <remarks />
        [EnumMember(Value = "Read_1")]
        Read = 1,

        /// <remarks />
        [EnumMember(Value = "Write_2")]
        Write = 2,

        /// <remarks />
        [EnumMember(Value = "EraseExisting_4")]
        EraseExisting = 4,

        /// <remarks />
        [EnumMember(Value = "Append_8")]
        Append = 8,
    }

    #region OpenFileModeCollection Class
    /// <summary>
    /// A collection of OpenFileMode objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfOpenFileMode", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "OpenFileMode")]
    #if !NET_STANDARD
    public partial class OpenFileModeCollection : List<OpenFileMode>, ICloneable
    #else
    public partial class OpenFileModeCollection : List<OpenFileMode>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public OpenFileModeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public OpenFileModeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public OpenFileModeCollection(IEnumerable<OpenFileMode> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator OpenFileModeCollection(OpenFileMode[] values)
        {
            if (values != null)
            {
                return new OpenFileModeCollection(values);
            }

            return new OpenFileModeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator OpenFileMode[](OpenFileModeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (OpenFileModeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            OpenFileModeCollection clone = new OpenFileModeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((OpenFileMode)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region IdentityCriteriaType Enumeration
    #if (!OPCUA_EXCLUDE_IdentityCriteriaType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum IdentityCriteriaType
    {
        /// <remarks />
        [EnumMember(Value = "UserName_1")]
        UserName = 1,

        /// <remarks />
        [EnumMember(Value = "Thumbprint_2")]
        Thumbprint = 2,

        /// <remarks />
        [EnumMember(Value = "Role_3")]
        Role = 3,

        /// <remarks />
        [EnumMember(Value = "GroupId_4")]
        GroupId = 4,

        /// <remarks />
        [EnumMember(Value = "Anonymous_5")]
        Anonymous = 5,

        /// <remarks />
        [EnumMember(Value = "AuthenticatedUser_6")]
        AuthenticatedUser = 6,

        /// <remarks />
        [EnumMember(Value = "Application_7")]
        Application = 7,
    }

    #region IdentityCriteriaTypeCollection Class
    /// <summary>
    /// A collection of IdentityCriteriaType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfIdentityCriteriaType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "IdentityCriteriaType")]
    #if !NET_STANDARD
    public partial class IdentityCriteriaTypeCollection : List<IdentityCriteriaType>, ICloneable
    #else
    public partial class IdentityCriteriaTypeCollection : List<IdentityCriteriaType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public IdentityCriteriaTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public IdentityCriteriaTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public IdentityCriteriaTypeCollection(IEnumerable<IdentityCriteriaType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator IdentityCriteriaTypeCollection(IdentityCriteriaType[] values)
        {
            if (values != null)
            {
                return new IdentityCriteriaTypeCollection(values);
            }

            return new IdentityCriteriaTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator IdentityCriteriaType[](IdentityCriteriaTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (IdentityCriteriaTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            IdentityCriteriaTypeCollection clone = new IdentityCriteriaTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((IdentityCriteriaType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region IdentityMappingRuleType Class
    #if (!OPCUA_EXCLUDE_IdentityMappingRuleType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class IdentityMappingRuleType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public IdentityMappingRuleType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_criteriaType = IdentityCriteriaType.UserName;
            m_criteria = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "CriteriaType", IsRequired = false, Order = 1)]
        public IdentityCriteriaType CriteriaType
        {
            get { return m_criteriaType;  }
            set { m_criteriaType = value; }
        }

        /// <remarks />
        [DataMember(Name = "Criteria", IsRequired = false, Order = 2)]
        public string Criteria
        {
            get { return m_criteria;  }
            set { m_criteria = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.IdentityMappingRuleType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.IdentityMappingRuleType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.IdentityMappingRuleType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEnumerated("CriteriaType", CriteriaType);
            encoder.WriteString("Criteria", Criteria);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            CriteriaType = (IdentityCriteriaType)decoder.ReadEnumerated("CriteriaType", typeof(IdentityCriteriaType));
            Criteria = decoder.ReadString("Criteria");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            IdentityMappingRuleType value = encodeable as IdentityMappingRuleType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_criteriaType, value.m_criteriaType)) return false;
            if (!Utils.IsEqual(m_criteria, value.m_criteria)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (IdentityMappingRuleType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            IdentityMappingRuleType clone = (IdentityMappingRuleType)base.MemberwiseClone();

            clone.m_criteriaType = (IdentityCriteriaType)Utils.Clone(this.m_criteriaType);
            clone.m_criteria = (string)Utils.Clone(this.m_criteria);

            return clone;
        }
        #endregion

        #region Private Fields
        private IdentityCriteriaType m_criteriaType;
        private string m_criteria;
        #endregion
    }

    #region IdentityMappingRuleTypeCollection Class
    /// <summary>
    /// A collection of IdentityMappingRuleType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfIdentityMappingRuleType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "IdentityMappingRuleType")]
    #if !NET_STANDARD
    public partial class IdentityMappingRuleTypeCollection : List<IdentityMappingRuleType>, ICloneable
    #else
    public partial class IdentityMappingRuleTypeCollection : List<IdentityMappingRuleType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public IdentityMappingRuleTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public IdentityMappingRuleTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public IdentityMappingRuleTypeCollection(IEnumerable<IdentityMappingRuleType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator IdentityMappingRuleTypeCollection(IdentityMappingRuleType[] values)
        {
            if (values != null)
            {
                return new IdentityMappingRuleTypeCollection(values);
            }

            return new IdentityMappingRuleTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator IdentityMappingRuleType[](IdentityMappingRuleTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (IdentityMappingRuleTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            IdentityMappingRuleTypeCollection clone = new IdentityMappingRuleTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((IdentityMappingRuleType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region CurrencyUnitType Class
    #if (!OPCUA_EXCLUDE_CurrencyUnitType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class CurrencyUnitType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public CurrencyUnitType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_numericCode = (short)0;
            m_exponent = (sbyte)0;
            m_alphabeticCode = null;
            m_currency = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "NumericCode", IsRequired = false, Order = 1)]
        public short NumericCode
        {
            get { return m_numericCode;  }
            set { m_numericCode = value; }
        }

        /// <remarks />
        [DataMember(Name = "Exponent", IsRequired = false, Order = 2)]
        public sbyte Exponent
        {
            get { return m_exponent;  }
            set { m_exponent = value; }
        }

        /// <remarks />
        [DataMember(Name = "AlphabeticCode", IsRequired = false, Order = 3)]
        public string AlphabeticCode
        {
            get { return m_alphabeticCode;  }
            set { m_alphabeticCode = value; }
        }

        /// <remarks />
        [DataMember(Name = "Currency", IsRequired = false, Order = 4)]
        public LocalizedText Currency
        {
            get { return m_currency;  }
            set { m_currency = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.CurrencyUnitType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.CurrencyUnitType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.CurrencyUnitType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteInt16("NumericCode", NumericCode);
            encoder.WriteSByte("Exponent", Exponent);
            encoder.WriteString("AlphabeticCode", AlphabeticCode);
            encoder.WriteLocalizedText("Currency", Currency);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            NumericCode = decoder.ReadInt16("NumericCode");
            Exponent = decoder.ReadSByte("Exponent");
            AlphabeticCode = decoder.ReadString("AlphabeticCode");
            Currency = decoder.ReadLocalizedText("Currency");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            CurrencyUnitType value = encodeable as CurrencyUnitType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_numericCode, value.m_numericCode)) return false;
            if (!Utils.IsEqual(m_exponent, value.m_exponent)) return false;
            if (!Utils.IsEqual(m_alphabeticCode, value.m_alphabeticCode)) return false;
            if (!Utils.IsEqual(m_currency, value.m_currency)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (CurrencyUnitType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            CurrencyUnitType clone = (CurrencyUnitType)base.MemberwiseClone();

            clone.m_numericCode = (short)Utils.Clone(this.m_numericCode);
            clone.m_exponent = (sbyte)Utils.Clone(this.m_exponent);
            clone.m_alphabeticCode = (string)Utils.Clone(this.m_alphabeticCode);
            clone.m_currency = (LocalizedText)Utils.Clone(this.m_currency);

            return clone;
        }
        #endregion

        #region Private Fields
        private short m_numericCode;
        private sbyte m_exponent;
        private string m_alphabeticCode;
        private LocalizedText m_currency;
        #endregion
    }

    #region CurrencyUnitTypeCollection Class
    /// <summary>
    /// A collection of CurrencyUnitType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfCurrencyUnitType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "CurrencyUnitType")]
    #if !NET_STANDARD
    public partial class CurrencyUnitTypeCollection : List<CurrencyUnitType>, ICloneable
    #else
    public partial class CurrencyUnitTypeCollection : List<CurrencyUnitType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public CurrencyUnitTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public CurrencyUnitTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public CurrencyUnitTypeCollection(IEnumerable<CurrencyUnitType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator CurrencyUnitTypeCollection(CurrencyUnitType[] values)
        {
            if (values != null)
            {
                return new CurrencyUnitTypeCollection(values);
            }

            return new CurrencyUnitTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator CurrencyUnitType[](CurrencyUnitTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (CurrencyUnitTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            CurrencyUnitTypeCollection clone = new CurrencyUnitTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((CurrencyUnitType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region TrustListMasks Enumeration
    #if (!OPCUA_EXCLUDE_TrustListMasks)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum TrustListMasks
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "TrustedCertificates_1")]
        TrustedCertificates = 1,

        /// <remarks />
        [EnumMember(Value = "TrustedCrls_2")]
        TrustedCrls = 2,

        /// <remarks />
        [EnumMember(Value = "IssuerCertificates_4")]
        IssuerCertificates = 4,

        /// <remarks />
        [EnumMember(Value = "IssuerCrls_8")]
        IssuerCrls = 8,

        /// <remarks />
        [EnumMember(Value = "All_15")]
        All = 15,
    }
    #endif
    #endregion

    #region TrustListDataType Class
    #if (!OPCUA_EXCLUDE_TrustListDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class TrustListDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public TrustListDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_specifiedLists = (uint)0;
            m_trustedCertificates = new ByteStringCollection();
            m_trustedCrls = new ByteStringCollection();
            m_issuerCertificates = new ByteStringCollection();
            m_issuerCrls = new ByteStringCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "SpecifiedLists", IsRequired = false, Order = 1)]
        public uint SpecifiedLists
        {
            get { return m_specifiedLists;  }
            set { m_specifiedLists = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "TrustedCertificates", IsRequired = false, Order = 2)]
        public ByteStringCollection TrustedCertificates
        {
            get
            {
                return m_trustedCertificates;
            }

            set
            {
                m_trustedCertificates = value;

                if (value == null)
                {
                    m_trustedCertificates = new ByteStringCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "TrustedCrls", IsRequired = false, Order = 3)]
        public ByteStringCollection TrustedCrls
        {
            get
            {
                return m_trustedCrls;
            }

            set
            {
                m_trustedCrls = value;

                if (value == null)
                {
                    m_trustedCrls = new ByteStringCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "IssuerCertificates", IsRequired = false, Order = 4)]
        public ByteStringCollection IssuerCertificates
        {
            get
            {
                return m_issuerCertificates;
            }

            set
            {
                m_issuerCertificates = value;

                if (value == null)
                {
                    m_issuerCertificates = new ByteStringCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "IssuerCrls", IsRequired = false, Order = 5)]
        public ByteStringCollection IssuerCrls
        {
            get
            {
                return m_issuerCrls;
            }

            set
            {
                m_issuerCrls = value;

                if (value == null)
                {
                    m_issuerCrls = new ByteStringCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.TrustListDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.TrustListDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.TrustListDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("SpecifiedLists", SpecifiedLists);
            encoder.WriteByteStringArray("TrustedCertificates", TrustedCertificates);
            encoder.WriteByteStringArray("TrustedCrls", TrustedCrls);
            encoder.WriteByteStringArray("IssuerCertificates", IssuerCertificates);
            encoder.WriteByteStringArray("IssuerCrls", IssuerCrls);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            SpecifiedLists = decoder.ReadUInt32("SpecifiedLists");
            TrustedCertificates = decoder.ReadByteStringArray("TrustedCertificates");
            TrustedCrls = decoder.ReadByteStringArray("TrustedCrls");
            IssuerCertificates = decoder.ReadByteStringArray("IssuerCertificates");
            IssuerCrls = decoder.ReadByteStringArray("IssuerCrls");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            TrustListDataType value = encodeable as TrustListDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_specifiedLists, value.m_specifiedLists)) return false;
            if (!Utils.IsEqual(m_trustedCertificates, value.m_trustedCertificates)) return false;
            if (!Utils.IsEqual(m_trustedCrls, value.m_trustedCrls)) return false;
            if (!Utils.IsEqual(m_issuerCertificates, value.m_issuerCertificates)) return false;
            if (!Utils.IsEqual(m_issuerCrls, value.m_issuerCrls)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (TrustListDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TrustListDataType clone = (TrustListDataType)base.MemberwiseClone();

            clone.m_specifiedLists = (uint)Utils.Clone(this.m_specifiedLists);
            clone.m_trustedCertificates = (ByteStringCollection)Utils.Clone(this.m_trustedCertificates);
            clone.m_trustedCrls = (ByteStringCollection)Utils.Clone(this.m_trustedCrls);
            clone.m_issuerCertificates = (ByteStringCollection)Utils.Clone(this.m_issuerCertificates);
            clone.m_issuerCrls = (ByteStringCollection)Utils.Clone(this.m_issuerCrls);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_specifiedLists;
        private ByteStringCollection m_trustedCertificates;
        private ByteStringCollection m_trustedCrls;
        private ByteStringCollection m_issuerCertificates;
        private ByteStringCollection m_issuerCrls;
        #endregion
    }

    #region TrustListDataTypeCollection Class
    /// <summary>
    /// A collection of TrustListDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfTrustListDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "TrustListDataType")]
    #if !NET_STANDARD
    public partial class TrustListDataTypeCollection : List<TrustListDataType>, ICloneable
    #else
    public partial class TrustListDataTypeCollection : List<TrustListDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public TrustListDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public TrustListDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public TrustListDataTypeCollection(IEnumerable<TrustListDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator TrustListDataTypeCollection(TrustListDataType[] values)
        {
            if (values != null)
            {
                return new TrustListDataTypeCollection(values);
            }

            return new TrustListDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator TrustListDataType[](TrustListDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (TrustListDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TrustListDataTypeCollection clone = new TrustListDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((TrustListDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DecimalDataType Class
    #if (!OPCUA_EXCLUDE_DecimalDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DecimalDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public DecimalDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_scale = (short)0;
            m_value = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Scale", IsRequired = false, Order = 1)]
        public short Scale
        {
            get { return m_scale;  }
            set { m_scale = value; }
        }

        /// <remarks />
        [DataMember(Name = "Value", IsRequired = false, Order = 2)]
        public byte[] Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.DecimalDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.DecimalDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.DecimalDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteInt16("Scale", Scale);
            encoder.WriteByteString("Value", Value);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Scale = decoder.ReadInt16("Scale");
            Value = decoder.ReadByteString("Value");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DecimalDataType value = encodeable as DecimalDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_scale, value.m_scale)) return false;
            if (!Utils.IsEqual(m_value, value.m_value)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DecimalDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DecimalDataType clone = (DecimalDataType)base.MemberwiseClone();

            clone.m_scale = (short)Utils.Clone(this.m_scale);
            clone.m_value = (byte[])Utils.Clone(this.m_value);

            return clone;
        }
        #endregion

        #region Private Fields
        private short m_scale;
        private byte[] m_value;
        #endregion
    }
    #endif
    #endregion

    #region DataTypeSchemaHeader Class
    #if (!OPCUA_EXCLUDE_DataTypeSchemaHeader)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataTypeSchemaHeader : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public DataTypeSchemaHeader()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_namespaces = new StringCollection();
            m_structureDataTypes = new StructureDescriptionCollection();
            m_enumDataTypes = new EnumDescriptionCollection();
            m_simpleDataTypes = new SimpleTypeDescriptionCollection();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "Namespaces", IsRequired = false, Order = 1)]
        public StringCollection Namespaces
        {
            get
            {
                return m_namespaces;
            }

            set
            {
                m_namespaces = value;

                if (value == null)
                {
                    m_namespaces = new StringCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "StructureDataTypes", IsRequired = false, Order = 2)]
        public StructureDescriptionCollection StructureDataTypes
        {
            get
            {
                return m_structureDataTypes;
            }

            set
            {
                m_structureDataTypes = value;

                if (value == null)
                {
                    m_structureDataTypes = new StructureDescriptionCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "EnumDataTypes", IsRequired = false, Order = 3)]
        public EnumDescriptionCollection EnumDataTypes
        {
            get
            {
                return m_enumDataTypes;
            }

            set
            {
                m_enumDataTypes = value;

                if (value == null)
                {
                    m_enumDataTypes = new EnumDescriptionCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "SimpleDataTypes", IsRequired = false, Order = 4)]
        public SimpleTypeDescriptionCollection SimpleDataTypes
        {
            get
            {
                return m_simpleDataTypes;
            }

            set
            {
                m_simpleDataTypes = value;

                if (value == null)
                {
                    m_simpleDataTypes = new SimpleTypeDescriptionCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.DataTypeSchemaHeader; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.DataTypeSchemaHeader_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.DataTypeSchemaHeader_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteStringArray("Namespaces", Namespaces);
            encoder.WriteEncodeableArray("StructureDataTypes", StructureDataTypes.ToArray(), typeof(StructureDescription));
            encoder.WriteEncodeableArray("EnumDataTypes", EnumDataTypes.ToArray(), typeof(EnumDescription));
            encoder.WriteEncodeableArray("SimpleDataTypes", SimpleDataTypes.ToArray(), typeof(SimpleTypeDescription));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Namespaces = decoder.ReadStringArray("Namespaces");
            StructureDataTypes = (StructureDescriptionCollection)decoder.ReadEncodeableArray("StructureDataTypes", typeof(StructureDescription));
            EnumDataTypes = (EnumDescriptionCollection)decoder.ReadEncodeableArray("EnumDataTypes", typeof(EnumDescription));
            SimpleDataTypes = (SimpleTypeDescriptionCollection)decoder.ReadEncodeableArray("SimpleDataTypes", typeof(SimpleTypeDescription));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataTypeSchemaHeader value = encodeable as DataTypeSchemaHeader;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_namespaces, value.m_namespaces)) return false;
            if (!Utils.IsEqual(m_structureDataTypes, value.m_structureDataTypes)) return false;
            if (!Utils.IsEqual(m_enumDataTypes, value.m_enumDataTypes)) return false;
            if (!Utils.IsEqual(m_simpleDataTypes, value.m_simpleDataTypes)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DataTypeSchemaHeader)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataTypeSchemaHeader clone = (DataTypeSchemaHeader)base.MemberwiseClone();

            clone.m_namespaces = (StringCollection)Utils.Clone(this.m_namespaces);
            clone.m_structureDataTypes = (StructureDescriptionCollection)Utils.Clone(this.m_structureDataTypes);
            clone.m_enumDataTypes = (EnumDescriptionCollection)Utils.Clone(this.m_enumDataTypes);
            clone.m_simpleDataTypes = (SimpleTypeDescriptionCollection)Utils.Clone(this.m_simpleDataTypes);

            return clone;
        }
        #endregion

        #region Private Fields
        private StringCollection m_namespaces;
        private StructureDescriptionCollection m_structureDataTypes;
        private EnumDescriptionCollection m_enumDataTypes;
        private SimpleTypeDescriptionCollection m_simpleDataTypes;
        #endregion
    }

    #region DataTypeSchemaHeaderCollection Class
    /// <summary>
    /// A collection of DataTypeSchemaHeader objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataTypeSchemaHeader", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataTypeSchemaHeader")]
    #if !NET_STANDARD
    public partial class DataTypeSchemaHeaderCollection : List<DataTypeSchemaHeader>, ICloneable
    #else
    public partial class DataTypeSchemaHeaderCollection : List<DataTypeSchemaHeader>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public DataTypeSchemaHeaderCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public DataTypeSchemaHeaderCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public DataTypeSchemaHeaderCollection(IEnumerable<DataTypeSchemaHeader> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator DataTypeSchemaHeaderCollection(DataTypeSchemaHeader[] values)
        {
            if (values != null)
            {
                return new DataTypeSchemaHeaderCollection(values);
            }

            return new DataTypeSchemaHeaderCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator DataTypeSchemaHeader[](DataTypeSchemaHeaderCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (DataTypeSchemaHeaderCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataTypeSchemaHeaderCollection clone = new DataTypeSchemaHeaderCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataTypeSchemaHeader)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataTypeDescription Class
    #if (!OPCUA_EXCLUDE_DataTypeDescription)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataTypeDescription : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public DataTypeDescription()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_dataTypeId = null;
            m_name = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "DataTypeId", IsRequired = false, Order = 1)]
        public NodeId DataTypeId
        {
            get { return m_dataTypeId;  }
            set { m_dataTypeId = value; }
        }

        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 2)]
        public QualifiedName Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.DataTypeDescription; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.DataTypeDescription_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.DataTypeDescription_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("DataTypeId", DataTypeId);
            encoder.WriteQualifiedName("Name", Name);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            DataTypeId = decoder.ReadNodeId("DataTypeId");
            Name = decoder.ReadQualifiedName("Name");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataTypeDescription value = encodeable as DataTypeDescription;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_dataTypeId, value.m_dataTypeId)) return false;
            if (!Utils.IsEqual(m_name, value.m_name)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DataTypeDescription)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataTypeDescription clone = (DataTypeDescription)base.MemberwiseClone();

            clone.m_dataTypeId = (NodeId)Utils.Clone(this.m_dataTypeId);
            clone.m_name = (QualifiedName)Utils.Clone(this.m_name);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_dataTypeId;
        private QualifiedName m_name;
        #endregion
    }

    #region DataTypeDescriptionCollection Class
    /// <summary>
    /// A collection of DataTypeDescription objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataTypeDescription", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataTypeDescription")]
    #if !NET_STANDARD
    public partial class DataTypeDescriptionCollection : List<DataTypeDescription>, ICloneable
    #else
    public partial class DataTypeDescriptionCollection : List<DataTypeDescription>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public DataTypeDescriptionCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public DataTypeDescriptionCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public DataTypeDescriptionCollection(IEnumerable<DataTypeDescription> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator DataTypeDescriptionCollection(DataTypeDescription[] values)
        {
            if (values != null)
            {
                return new DataTypeDescriptionCollection(values);
            }

            return new DataTypeDescriptionCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator DataTypeDescription[](DataTypeDescriptionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (DataTypeDescriptionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataTypeDescriptionCollection clone = new DataTypeDescriptionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataTypeDescription)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region StructureDescription Class
    #if (!OPCUA_EXCLUDE_StructureDescription)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class StructureDescription : DataTypeDescription
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public StructureDescription()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_structureDefinition = new StructureDefinition();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "StructureDefinition", IsRequired = false, Order = 1)]
        public StructureDefinition StructureDefinition
        {
            get
            {
                return m_structureDefinition;
            }

            set
            {
                m_structureDefinition = value;

                if (value == null)
                {
                    m_structureDefinition = new StructureDefinition();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.StructureDescription; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.StructureDescription_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.StructureDescription_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("StructureDefinition", StructureDefinition, typeof(StructureDefinition));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            StructureDefinition = (StructureDefinition)decoder.ReadEncodeable("StructureDefinition", typeof(StructureDefinition));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            StructureDescription value = encodeable as StructureDescription;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_structureDefinition, value.m_structureDefinition)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (StructureDescription)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            StructureDescription clone = (StructureDescription)base.MemberwiseClone();

            clone.m_structureDefinition = (StructureDefinition)Utils.Clone(this.m_structureDefinition);

            return clone;
        }
        #endregion

        #region Private Fields
        private StructureDefinition m_structureDefinition;
        #endregion
    }

    #region StructureDescriptionCollection Class
    /// <summary>
    /// A collection of StructureDescription objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfStructureDescription", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "StructureDescription")]
    #if !NET_STANDARD
    public partial class StructureDescriptionCollection : List<StructureDescription>, ICloneable
    #else
    public partial class StructureDescriptionCollection : List<StructureDescription>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public StructureDescriptionCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public StructureDescriptionCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public StructureDescriptionCollection(IEnumerable<StructureDescription> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator StructureDescriptionCollection(StructureDescription[] values)
        {
            if (values != null)
            {
                return new StructureDescriptionCollection(values);
            }

            return new StructureDescriptionCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator StructureDescription[](StructureDescriptionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (StructureDescriptionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            StructureDescriptionCollection clone = new StructureDescriptionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((StructureDescription)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region EnumDescription Class
    #if (!OPCUA_EXCLUDE_EnumDescription)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class EnumDescription : DataTypeDescription
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public EnumDescription()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_enumDefinition = new EnumDefinition();
            m_builtInType = (byte)0;
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "EnumDefinition", IsRequired = false, Order = 1)]
        public EnumDefinition EnumDefinition
        {
            get
            {
                return m_enumDefinition;
            }

            set
            {
                m_enumDefinition = value;

                if (value == null)
                {
                    m_enumDefinition = new EnumDefinition();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "BuiltInType", IsRequired = false, Order = 2)]
        public byte BuiltInType
        {
            get { return m_builtInType;  }
            set { m_builtInType = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.EnumDescription; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.EnumDescription_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.EnumDescription_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("EnumDefinition", EnumDefinition, typeof(EnumDefinition));
            encoder.WriteByte("BuiltInType", BuiltInType);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            EnumDefinition = (EnumDefinition)decoder.ReadEncodeable("EnumDefinition", typeof(EnumDefinition));
            BuiltInType = decoder.ReadByte("BuiltInType");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            EnumDescription value = encodeable as EnumDescription;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_enumDefinition, value.m_enumDefinition)) return false;
            if (!Utils.IsEqual(m_builtInType, value.m_builtInType)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (EnumDescription)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EnumDescription clone = (EnumDescription)base.MemberwiseClone();

            clone.m_enumDefinition = (EnumDefinition)Utils.Clone(this.m_enumDefinition);
            clone.m_builtInType = (byte)Utils.Clone(this.m_builtInType);

            return clone;
        }
        #endregion

        #region Private Fields
        private EnumDefinition m_enumDefinition;
        private byte m_builtInType;
        #endregion
    }

    #region EnumDescriptionCollection Class
    /// <summary>
    /// A collection of EnumDescription objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfEnumDescription", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "EnumDescription")]
    #if !NET_STANDARD
    public partial class EnumDescriptionCollection : List<EnumDescription>, ICloneable
    #else
    public partial class EnumDescriptionCollection : List<EnumDescription>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public EnumDescriptionCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public EnumDescriptionCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public EnumDescriptionCollection(IEnumerable<EnumDescription> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator EnumDescriptionCollection(EnumDescription[] values)
        {
            if (values != null)
            {
                return new EnumDescriptionCollection(values);
            }

            return new EnumDescriptionCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator EnumDescription[](EnumDescriptionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (EnumDescriptionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EnumDescriptionCollection clone = new EnumDescriptionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((EnumDescription)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region SimpleTypeDescription Class
    #if (!OPCUA_EXCLUDE_SimpleTypeDescription)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class SimpleTypeDescription : DataTypeDescription
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public SimpleTypeDescription()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_baseDataType = null;
            m_builtInType = (byte)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "BaseDataType", IsRequired = false, Order = 1)]
        public NodeId BaseDataType
        {
            get { return m_baseDataType;  }
            set { m_baseDataType = value; }
        }

        /// <remarks />
        [DataMember(Name = "BuiltInType", IsRequired = false, Order = 2)]
        public byte BuiltInType
        {
            get { return m_builtInType;  }
            set { m_builtInType = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.SimpleTypeDescription; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.SimpleTypeDescription_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.SimpleTypeDescription_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("BaseDataType", BaseDataType);
            encoder.WriteByte("BuiltInType", BuiltInType);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            BaseDataType = decoder.ReadNodeId("BaseDataType");
            BuiltInType = decoder.ReadByte("BuiltInType");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            SimpleTypeDescription value = encodeable as SimpleTypeDescription;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_baseDataType, value.m_baseDataType)) return false;
            if (!Utils.IsEqual(m_builtInType, value.m_builtInType)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (SimpleTypeDescription)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SimpleTypeDescription clone = (SimpleTypeDescription)base.MemberwiseClone();

            clone.m_baseDataType = (NodeId)Utils.Clone(this.m_baseDataType);
            clone.m_builtInType = (byte)Utils.Clone(this.m_builtInType);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_baseDataType;
        private byte m_builtInType;
        #endregion
    }

    #region SimpleTypeDescriptionCollection Class
    /// <summary>
    /// A collection of SimpleTypeDescription objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfSimpleTypeDescription", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "SimpleTypeDescription")]
    #if !NET_STANDARD
    public partial class SimpleTypeDescriptionCollection : List<SimpleTypeDescription>, ICloneable
    #else
    public partial class SimpleTypeDescriptionCollection : List<SimpleTypeDescription>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public SimpleTypeDescriptionCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public SimpleTypeDescriptionCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public SimpleTypeDescriptionCollection(IEnumerable<SimpleTypeDescription> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator SimpleTypeDescriptionCollection(SimpleTypeDescription[] values)
        {
            if (values != null)
            {
                return new SimpleTypeDescriptionCollection(values);
            }

            return new SimpleTypeDescriptionCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator SimpleTypeDescription[](SimpleTypeDescriptionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (SimpleTypeDescriptionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SimpleTypeDescriptionCollection clone = new SimpleTypeDescriptionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((SimpleTypeDescription)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region UABinaryFileDataType Class
    #if (!OPCUA_EXCLUDE_UABinaryFileDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class UABinaryFileDataType : DataTypeSchemaHeader
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public UABinaryFileDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_schemaLocation = null;
            m_fileHeader = new KeyValuePairCollection();
            m_body = Variant.Null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "SchemaLocation", IsRequired = false, Order = 1)]
        public string SchemaLocation
        {
            get { return m_schemaLocation;  }
            set { m_schemaLocation = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "FileHeader", IsRequired = false, Order = 2)]
        public KeyValuePairCollection FileHeader
        {
            get
            {
                return m_fileHeader;
            }

            set
            {
                m_fileHeader = value;

                if (value == null)
                {
                    m_fileHeader = new KeyValuePairCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "Body", IsRequired = false, Order = 3)]
        public Variant Body
        {
            get { return m_body;  }
            set { m_body = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.UABinaryFileDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.UABinaryFileDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.UABinaryFileDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("SchemaLocation", SchemaLocation);
            encoder.WriteEncodeableArray("FileHeader", FileHeader.ToArray(), typeof(KeyValuePair));
            encoder.WriteVariant("Body", Body);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            SchemaLocation = decoder.ReadString("SchemaLocation");
            FileHeader = (KeyValuePairCollection)decoder.ReadEncodeableArray("FileHeader", typeof(KeyValuePair));
            Body = decoder.ReadVariant("Body");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            UABinaryFileDataType value = encodeable as UABinaryFileDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_schemaLocation, value.m_schemaLocation)) return false;
            if (!Utils.IsEqual(m_fileHeader, value.m_fileHeader)) return false;
            if (!Utils.IsEqual(m_body, value.m_body)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (UABinaryFileDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UABinaryFileDataType clone = (UABinaryFileDataType)base.MemberwiseClone();

            clone.m_schemaLocation = (string)Utils.Clone(this.m_schemaLocation);
            clone.m_fileHeader = (KeyValuePairCollection)Utils.Clone(this.m_fileHeader);
            clone.m_body = (Variant)Utils.Clone(this.m_body);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_schemaLocation;
        private KeyValuePairCollection m_fileHeader;
        private Variant m_body;
        #endregion
    }

    #region UABinaryFileDataTypeCollection Class
    /// <summary>
    /// A collection of UABinaryFileDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUABinaryFileDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "UABinaryFileDataType")]
    #if !NET_STANDARD
    public partial class UABinaryFileDataTypeCollection : List<UABinaryFileDataType>, ICloneable
    #else
    public partial class UABinaryFileDataTypeCollection : List<UABinaryFileDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public UABinaryFileDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public UABinaryFileDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public UABinaryFileDataTypeCollection(IEnumerable<UABinaryFileDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator UABinaryFileDataTypeCollection(UABinaryFileDataType[] values)
        {
            if (values != null)
            {
                return new UABinaryFileDataTypeCollection(values);
            }

            return new UABinaryFileDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator UABinaryFileDataType[](UABinaryFileDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (UABinaryFileDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UABinaryFileDataTypeCollection clone = new UABinaryFileDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((UABinaryFileDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PubSubState Enumeration
    #if (!OPCUA_EXCLUDE_PubSubState)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum PubSubState
    {
        /// <remarks />
        [EnumMember(Value = "Disabled_0")]
        Disabled = 0,

        /// <remarks />
        [EnumMember(Value = "Paused_1")]
        Paused = 1,

        /// <remarks />
        [EnumMember(Value = "Operational_2")]
        Operational = 2,

        /// <remarks />
        [EnumMember(Value = "Error_3")]
        Error = 3,

        /// <remarks />
        [EnumMember(Value = "PreOperational_4")]
        PreOperational = 4,
    }

    #region PubSubStateCollection Class
    /// <summary>
    /// A collection of PubSubState objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPubSubState", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PubSubState")]
    #if !NET_STANDARD
    public partial class PubSubStateCollection : List<PubSubState>, ICloneable
    #else
    public partial class PubSubStateCollection : List<PubSubState>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public PubSubStateCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public PubSubStateCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public PubSubStateCollection(IEnumerable<PubSubState> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator PubSubStateCollection(PubSubState[] values)
        {
            if (values != null)
            {
                return new PubSubStateCollection(values);
            }

            return new PubSubStateCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator PubSubState[](PubSubStateCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (PubSubStateCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubStateCollection clone = new PubSubStateCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PubSubState)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetMetaDataType Class
    #if (!OPCUA_EXCLUDE_DataSetMetaDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataSetMetaDataType : DataTypeSchemaHeader
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public DataSetMetaDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_name = null;
            m_description = null;
            m_fields = new FieldMetaDataCollection();
            m_dataSetClassId = Uuid.Empty;
            m_configurationVersion = new ConfigurationVersionDataType();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "Description", IsRequired = false, Order = 2)]
        public LocalizedText Description
        {
            get { return m_description;  }
            set { m_description = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "Fields", IsRequired = false, Order = 3)]
        public FieldMetaDataCollection Fields
        {
            get
            {
                return m_fields;
            }

            set
            {
                m_fields = value;

                if (value == null)
                {
                    m_fields = new FieldMetaDataCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "DataSetClassId", IsRequired = false, Order = 4)]
        public Uuid DataSetClassId
        {
            get { return m_dataSetClassId;  }
            set { m_dataSetClassId = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ConfigurationVersion", IsRequired = false, Order = 5)]
        public ConfigurationVersionDataType ConfigurationVersion
        {
            get
            {
                return m_configurationVersion;
            }

            set
            {
                m_configurationVersion = value;

                if (value == null)
                {
                    m_configurationVersion = new ConfigurationVersionDataType();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.DataSetMetaDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.DataSetMetaDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.DataSetMetaDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteLocalizedText("Description", Description);
            encoder.WriteEncodeableArray("Fields", Fields.ToArray(), typeof(FieldMetaData));
            encoder.WriteGuid("DataSetClassId", DataSetClassId);
            encoder.WriteEncodeable("ConfigurationVersion", ConfigurationVersion, typeof(ConfigurationVersionDataType));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            Description = decoder.ReadLocalizedText("Description");
            Fields = (FieldMetaDataCollection)decoder.ReadEncodeableArray("Fields", typeof(FieldMetaData));
            DataSetClassId = decoder.ReadGuid("DataSetClassId");
            ConfigurationVersion = (ConfigurationVersionDataType)decoder.ReadEncodeable("ConfigurationVersion", typeof(ConfigurationVersionDataType));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataSetMetaDataType value = encodeable as DataSetMetaDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_description, value.m_description)) return false;
            if (!Utils.IsEqual(m_fields, value.m_fields)) return false;
            if (!Utils.IsEqual(m_dataSetClassId, value.m_dataSetClassId)) return false;
            if (!Utils.IsEqual(m_configurationVersion, value.m_configurationVersion)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (DataSetMetaDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetMetaDataType clone = (DataSetMetaDataType)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_description = (LocalizedText)Utils.Clone(this.m_description);
            clone.m_fields = (FieldMetaDataCollection)Utils.Clone(this.m_fields);
            clone.m_dataSetClassId = (Uuid)Utils.Clone(this.m_dataSetClassId);
            clone.m_configurationVersion = (ConfigurationVersionDataType)Utils.Clone(this.m_configurationVersion);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private LocalizedText m_description;
        private FieldMetaDataCollection m_fields;
        private Uuid m_dataSetClassId;
        private ConfigurationVersionDataType m_configurationVersion;
        #endregion
    }

    #region DataSetMetaDataTypeCollection Class
    /// <summary>
    /// A collection of DataSetMetaDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataSetMetaDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataSetMetaDataType")]
    #if !NET_STANDARD
    public partial class DataSetMetaDataTypeCollection : List<DataSetMetaDataType>, ICloneable
    #else
    public partial class DataSetMetaDataTypeCollection : List<DataSetMetaDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public DataSetMetaDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public DataSetMetaDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public DataSetMetaDataTypeCollection(IEnumerable<DataSetMetaDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator DataSetMetaDataTypeCollection(DataSetMetaDataType[] values)
        {
            if (values != null)
            {
                return new DataSetMetaDataTypeCollection(values);
            }

            return new DataSetMetaDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator DataSetMetaDataType[](DataSetMetaDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (DataSetMetaDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetMetaDataTypeCollection clone = new DataSetMetaDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataSetMetaDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region FieldMetaData Class
    #if (!OPCUA_EXCLUDE_FieldMetaData)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class FieldMetaData : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public FieldMetaData()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_name = null;
            m_description = null;
            m_fieldFlags = 0;
            m_builtInType = (byte)0;
            m_dataType = null;
            m_valueRank = (int)0;
            m_arrayDimensions = new UInt32Collection();
            m_maxStringLength = (uint)0;
            m_dataSetFieldId = Uuid.Empty;
            m_properties = new KeyValuePairCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "Description", IsRequired = false, Order = 2)]
        public LocalizedText Description
        {
            get { return m_description;  }
            set { m_description = value; }
        }

        /// <remarks />
        [DataMember(Name = "FieldFlags", IsRequired = false, Order = 3)]
        public ushort FieldFlags
        {
            get { return m_fieldFlags;  }
            set { m_fieldFlags = value; }
        }

        /// <remarks />
        [DataMember(Name = "BuiltInType", IsRequired = false, Order = 4)]
        public byte BuiltInType
        {
            get { return m_builtInType;  }
            set { m_builtInType = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataType", IsRequired = false, Order = 5)]
        public NodeId DataType
        {
            get { return m_dataType;  }
            set { m_dataType = value; }
        }

        /// <remarks />
        [DataMember(Name = "ValueRank", IsRequired = false, Order = 6)]
        public int ValueRank
        {
            get { return m_valueRank;  }
            set { m_valueRank = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ArrayDimensions", IsRequired = false, Order = 7)]
        public UInt32Collection ArrayDimensions
        {
            get
            {
                return m_arrayDimensions;
            }

            set
            {
                m_arrayDimensions = value;

                if (value == null)
                {
                    m_arrayDimensions = new UInt32Collection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "MaxStringLength", IsRequired = false, Order = 8)]
        public uint MaxStringLength
        {
            get { return m_maxStringLength;  }
            set { m_maxStringLength = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetFieldId", IsRequired = false, Order = 9)]
        public Uuid DataSetFieldId
        {
            get { return m_dataSetFieldId;  }
            set { m_dataSetFieldId = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "Properties", IsRequired = false, Order = 10)]
        public KeyValuePairCollection Properties
        {
            get
            {
                return m_properties;
            }

            set
            {
                m_properties = value;

                if (value == null)
                {
                    m_properties = new KeyValuePairCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.FieldMetaData; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.FieldMetaData_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.FieldMetaData_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteLocalizedText("Description", Description);
            encoder.WriteUInt16("FieldFlags", FieldFlags);
            encoder.WriteByte("BuiltInType", BuiltInType);
            encoder.WriteNodeId("DataType", DataType);
            encoder.WriteInt32("ValueRank", ValueRank);
            encoder.WriteUInt32Array("ArrayDimensions", ArrayDimensions);
            encoder.WriteUInt32("MaxStringLength", MaxStringLength);
            encoder.WriteGuid("DataSetFieldId", DataSetFieldId);
            encoder.WriteEncodeableArray("Properties", Properties.ToArray(), typeof(KeyValuePair));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            Description = decoder.ReadLocalizedText("Description");
            FieldFlags = decoder.ReadUInt16("FieldFlags");
            BuiltInType = decoder.ReadByte("BuiltInType");
            DataType = decoder.ReadNodeId("DataType");
            ValueRank = decoder.ReadInt32("ValueRank");
            ArrayDimensions = decoder.ReadUInt32Array("ArrayDimensions");
            MaxStringLength = decoder.ReadUInt32("MaxStringLength");
            DataSetFieldId = decoder.ReadGuid("DataSetFieldId");
            Properties = (KeyValuePairCollection)decoder.ReadEncodeableArray("Properties", typeof(KeyValuePair));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            FieldMetaData value = encodeable as FieldMetaData;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_description, value.m_description)) return false;
            if (!Utils.IsEqual(m_fieldFlags, value.m_fieldFlags)) return false;
            if (!Utils.IsEqual(m_builtInType, value.m_builtInType)) return false;
            if (!Utils.IsEqual(m_dataType, value.m_dataType)) return false;
            if (!Utils.IsEqual(m_valueRank, value.m_valueRank)) return false;
            if (!Utils.IsEqual(m_arrayDimensions, value.m_arrayDimensions)) return false;
            if (!Utils.IsEqual(m_maxStringLength, value.m_maxStringLength)) return false;
            if (!Utils.IsEqual(m_dataSetFieldId, value.m_dataSetFieldId)) return false;
            if (!Utils.IsEqual(m_properties, value.m_properties)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (FieldMetaData)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            FieldMetaData clone = (FieldMetaData)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_description = (LocalizedText)Utils.Clone(this.m_description);
            clone.m_fieldFlags = (ushort)Utils.Clone(this.m_fieldFlags);
            clone.m_builtInType = (byte)Utils.Clone(this.m_builtInType);
            clone.m_dataType = (NodeId)Utils.Clone(this.m_dataType);
            clone.m_valueRank = (int)Utils.Clone(this.m_valueRank);
            clone.m_arrayDimensions = (UInt32Collection)Utils.Clone(this.m_arrayDimensions);
            clone.m_maxStringLength = (uint)Utils.Clone(this.m_maxStringLength);
            clone.m_dataSetFieldId = (Uuid)Utils.Clone(this.m_dataSetFieldId);
            clone.m_properties = (KeyValuePairCollection)Utils.Clone(this.m_properties);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private LocalizedText m_description;
        private ushort m_fieldFlags;
        private byte m_builtInType;
        private NodeId m_dataType;
        private int m_valueRank;
        private UInt32Collection m_arrayDimensions;
        private uint m_maxStringLength;
        private Uuid m_dataSetFieldId;
        private KeyValuePairCollection m_properties;
        #endregion
    }

    #region FieldMetaDataCollection Class
    /// <summary>
    /// A collection of FieldMetaData objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfFieldMetaData", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "FieldMetaData")]
    #if !NET_STANDARD
    public partial class FieldMetaDataCollection : List<FieldMetaData>, ICloneable
    #else
    public partial class FieldMetaDataCollection : List<FieldMetaData>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public FieldMetaDataCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public FieldMetaDataCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public FieldMetaDataCollection(IEnumerable<FieldMetaData> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator FieldMetaDataCollection(FieldMetaData[] values)
        {
            if (values != null)
            {
                return new FieldMetaDataCollection(values);
            }

            return new FieldMetaDataCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator FieldMetaData[](FieldMetaDataCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (FieldMetaDataCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            FieldMetaDataCollection clone = new FieldMetaDataCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((FieldMetaData)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetFieldFlags Enumeration
    #if (!OPCUA_EXCLUDE_DataSetFieldFlags)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum DataSetFieldFlags : UInt16
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "PromotedField_1")]
        PromotedField = 1,
    }
    #endif
    #endregion

    #region ConfigurationVersionDataType Class
    #if (!OPCUA_EXCLUDE_ConfigurationVersionDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ConfigurationVersionDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ConfigurationVersionDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_majorVersion = (uint)0;
            m_minorVersion = (uint)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "MajorVersion", IsRequired = false, Order = 1)]
        public uint MajorVersion
        {
            get { return m_majorVersion;  }
            set { m_majorVersion = value; }
        }

        /// <remarks />
        [DataMember(Name = "MinorVersion", IsRequired = false, Order = 2)]
        public uint MinorVersion
        {
            get { return m_minorVersion;  }
            set { m_minorVersion = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ConfigurationVersionDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ConfigurationVersionDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ConfigurationVersionDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("MajorVersion", MajorVersion);
            encoder.WriteUInt32("MinorVersion", MinorVersion);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            MajorVersion = decoder.ReadUInt32("MajorVersion");
            MinorVersion = decoder.ReadUInt32("MinorVersion");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ConfigurationVersionDataType value = encodeable as ConfigurationVersionDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_majorVersion, value.m_majorVersion)) return false;
            if (!Utils.IsEqual(m_minorVersion, value.m_minorVersion)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ConfigurationVersionDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ConfigurationVersionDataType clone = (ConfigurationVersionDataType)base.MemberwiseClone();

            clone.m_majorVersion = (uint)Utils.Clone(this.m_majorVersion);
            clone.m_minorVersion = (uint)Utils.Clone(this.m_minorVersion);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_majorVersion;
        private uint m_minorVersion;
        #endregion
    }

    #region ConfigurationVersionDataTypeCollection Class
    /// <summary>
    /// A collection of ConfigurationVersionDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfConfigurationVersionDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ConfigurationVersionDataType")]
    #if !NET_STANDARD
    public partial class ConfigurationVersionDataTypeCollection : List<ConfigurationVersionDataType>, ICloneable
    #else
    public partial class ConfigurationVersionDataTypeCollection : List<ConfigurationVersionDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public ConfigurationVersionDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public ConfigurationVersionDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public ConfigurationVersionDataTypeCollection(IEnumerable<ConfigurationVersionDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator ConfigurationVersionDataTypeCollection(ConfigurationVersionDataType[] values)
        {
            if (values != null)
            {
                return new ConfigurationVersionDataTypeCollection(values);
            }

            return new ConfigurationVersionDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator ConfigurationVersionDataType[](ConfigurationVersionDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (ConfigurationVersionDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ConfigurationVersionDataTypeCollection clone = new ConfigurationVersionDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ConfigurationVersionDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PublishedDataSetDataType Class
    #if (!OPCUA_EXCLUDE_PublishedDataSetDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PublishedDataSetDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public PublishedDataSetDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_name = null;
            m_dataSetFolder = new StringCollection();
            m_dataSetMetaData = new DataSetMetaDataType();
            m_extensionFields = new KeyValuePairCollection();
            m_dataSetSource = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "DataSetFolder", IsRequired = false, Order = 2)]
        public StringCollection DataSetFolder
        {
            get
            {
                return m_dataSetFolder;
            }

            set
            {
                m_dataSetFolder = value;

                if (value == null)
                {
                    m_dataSetFolder = new StringCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "DataSetMetaData", IsRequired = false, Order = 3)]
        public DataSetMetaDataType DataSetMetaData
        {
            get
            {
                return m_dataSetMetaData;
            }

            set
            {
                m_dataSetMetaData = value;

                if (value == null)
                {
                    m_dataSetMetaData = new DataSetMetaDataType();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ExtensionFields", IsRequired = false, Order = 4)]
        public KeyValuePairCollection ExtensionFields
        {
            get
            {
                return m_extensionFields;
            }

            set
            {
                m_extensionFields = value;

                if (value == null)
                {
                    m_extensionFields = new KeyValuePairCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "DataSetSource", IsRequired = false, Order = 5)]
        public ExtensionObject DataSetSource
        {
            get { return m_dataSetSource;  }
            set { m_dataSetSource = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.PublishedDataSetDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.PublishedDataSetDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.PublishedDataSetDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteStringArray("DataSetFolder", DataSetFolder);
            encoder.WriteEncodeable("DataSetMetaData", DataSetMetaData, typeof(DataSetMetaDataType));
            encoder.WriteEncodeableArray("ExtensionFields", ExtensionFields.ToArray(), typeof(KeyValuePair));
            encoder.WriteExtensionObject("DataSetSource", DataSetSource);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            DataSetFolder = decoder.ReadStringArray("DataSetFolder");
            DataSetMetaData = (DataSetMetaDataType)decoder.ReadEncodeable("DataSetMetaData", typeof(DataSetMetaDataType));
            ExtensionFields = (KeyValuePairCollection)decoder.ReadEncodeableArray("ExtensionFields", typeof(KeyValuePair));
            DataSetSource = decoder.ReadExtensionObject("DataSetSource");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PublishedDataSetDataType value = encodeable as PublishedDataSetDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_dataSetFolder, value.m_dataSetFolder)) return false;
            if (!Utils.IsEqual(m_dataSetMetaData, value.m_dataSetMetaData)) return false;
            if (!Utils.IsEqual(m_extensionFields, value.m_extensionFields)) return false;
            if (!Utils.IsEqual(m_dataSetSource, value.m_dataSetSource)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (PublishedDataSetDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedDataSetDataType clone = (PublishedDataSetDataType)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_dataSetFolder = (StringCollection)Utils.Clone(this.m_dataSetFolder);
            clone.m_dataSetMetaData = (DataSetMetaDataType)Utils.Clone(this.m_dataSetMetaData);
            clone.m_extensionFields = (KeyValuePairCollection)Utils.Clone(this.m_extensionFields);
            clone.m_dataSetSource = (ExtensionObject)Utils.Clone(this.m_dataSetSource);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private StringCollection m_dataSetFolder;
        private DataSetMetaDataType m_dataSetMetaData;
        private KeyValuePairCollection m_extensionFields;
        private ExtensionObject m_dataSetSource;
        #endregion
    }

    #region PublishedDataSetDataTypeCollection Class
    /// <summary>
    /// A collection of PublishedDataSetDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPublishedDataSetDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PublishedDataSetDataType")]
    #if !NET_STANDARD
    public partial class PublishedDataSetDataTypeCollection : List<PublishedDataSetDataType>, ICloneable
    #else
    public partial class PublishedDataSetDataTypeCollection : List<PublishedDataSetDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public PublishedDataSetDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public PublishedDataSetDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public PublishedDataSetDataTypeCollection(IEnumerable<PublishedDataSetDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator PublishedDataSetDataTypeCollection(PublishedDataSetDataType[] values)
        {
            if (values != null)
            {
                return new PublishedDataSetDataTypeCollection(values);
            }

            return new PublishedDataSetDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator PublishedDataSetDataType[](PublishedDataSetDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (PublishedDataSetDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedDataSetDataTypeCollection clone = new PublishedDataSetDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PublishedDataSetDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PublishedDataSetSourceDataType Class
    #if (!OPCUA_EXCLUDE_PublishedDataSetSourceDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PublishedDataSetSourceDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public PublishedDataSetSourceDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.PublishedDataSetSourceDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.PublishedDataSetSourceDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.PublishedDataSetSourceDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PublishedDataSetSourceDataType value = encodeable as PublishedDataSetSourceDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (PublishedDataSetSourceDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedDataSetSourceDataType clone = (PublishedDataSetSourceDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region PublishedDataSetSourceDataTypeCollection Class
    /// <summary>
    /// A collection of PublishedDataSetSourceDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPublishedDataSetSourceDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PublishedDataSetSourceDataType")]
    #if !NET_STANDARD
    public partial class PublishedDataSetSourceDataTypeCollection : List<PublishedDataSetSourceDataType>, ICloneable
    #else
    public partial class PublishedDataSetSourceDataTypeCollection : List<PublishedDataSetSourceDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public PublishedDataSetSourceDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public PublishedDataSetSourceDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public PublishedDataSetSourceDataTypeCollection(IEnumerable<PublishedDataSetSourceDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator PublishedDataSetSourceDataTypeCollection(PublishedDataSetSourceDataType[] values)
        {
            if (values != null)
            {
                return new PublishedDataSetSourceDataTypeCollection(values);
            }

            return new PublishedDataSetSourceDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator PublishedDataSetSourceDataType[](PublishedDataSetSourceDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (PublishedDataSetSourceDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedDataSetSourceDataTypeCollection clone = new PublishedDataSetSourceDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PublishedDataSetSourceDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PublishedVariableDataType Class
    #if (!OPCUA_EXCLUDE_PublishedVariableDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PublishedVariableDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public PublishedVariableDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_publishedVariable = null;
            m_attributeId = (uint)0;
            m_samplingIntervalHint = (double)0;
            m_deadbandType = (uint)0;
            m_deadbandValue = (double)0;
            m_indexRange = null;
            m_substituteValue = Variant.Null;
            m_metaDataProperties = new QualifiedNameCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "PublishedVariable", IsRequired = false, Order = 1)]
        public NodeId PublishedVariable
        {
            get { return m_publishedVariable;  }
            set { m_publishedVariable = value; }
        }

        /// <remarks />
        [DataMember(Name = "AttributeId", IsRequired = false, Order = 2)]
        public uint AttributeId
        {
            get { return m_attributeId;  }
            set { m_attributeId = value; }
        }

        /// <remarks />
        [DataMember(Name = "SamplingIntervalHint", IsRequired = false, Order = 3)]
        public double SamplingIntervalHint
        {
            get { return m_samplingIntervalHint;  }
            set { m_samplingIntervalHint = value; }
        }

        /// <remarks />
        [DataMember(Name = "DeadbandType", IsRequired = false, Order = 4)]
        public uint DeadbandType
        {
            get { return m_deadbandType;  }
            set { m_deadbandType = value; }
        }

        /// <remarks />
        [DataMember(Name = "DeadbandValue", IsRequired = false, Order = 5)]
        public double DeadbandValue
        {
            get { return m_deadbandValue;  }
            set { m_deadbandValue = value; }
        }

        /// <remarks />
        [DataMember(Name = "IndexRange", IsRequired = false, Order = 6)]
        public string IndexRange
        {
            get { return m_indexRange;  }
            set { m_indexRange = value; }
        }

        /// <remarks />
        [DataMember(Name = "SubstituteValue", IsRequired = false, Order = 7)]
        public Variant SubstituteValue
        {
            get { return m_substituteValue;  }
            set { m_substituteValue = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "MetaDataProperties", IsRequired = false, Order = 8)]
        public QualifiedNameCollection MetaDataProperties
        {
            get
            {
                return m_metaDataProperties;
            }

            set
            {
                m_metaDataProperties = value;

                if (value == null)
                {
                    m_metaDataProperties = new QualifiedNameCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.PublishedVariableDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.PublishedVariableDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.PublishedVariableDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("PublishedVariable", PublishedVariable);
            encoder.WriteUInt32("AttributeId", AttributeId);
            encoder.WriteDouble("SamplingIntervalHint", SamplingIntervalHint);
            encoder.WriteUInt32("DeadbandType", DeadbandType);
            encoder.WriteDouble("DeadbandValue", DeadbandValue);
            encoder.WriteString("IndexRange", IndexRange);
            encoder.WriteVariant("SubstituteValue", SubstituteValue);
            encoder.WriteQualifiedNameArray("MetaDataProperties", MetaDataProperties);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            PublishedVariable = decoder.ReadNodeId("PublishedVariable");
            AttributeId = decoder.ReadUInt32("AttributeId");
            SamplingIntervalHint = decoder.ReadDouble("SamplingIntervalHint");
            DeadbandType = decoder.ReadUInt32("DeadbandType");
            DeadbandValue = decoder.ReadDouble("DeadbandValue");
            IndexRange = decoder.ReadString("IndexRange");
            SubstituteValue = decoder.ReadVariant("SubstituteValue");
            MetaDataProperties = decoder.ReadQualifiedNameArray("MetaDataProperties");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PublishedVariableDataType value = encodeable as PublishedVariableDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_publishedVariable, value.m_publishedVariable)) return false;
            if (!Utils.IsEqual(m_attributeId, value.m_attributeId)) return false;
            if (!Utils.IsEqual(m_samplingIntervalHint, value.m_samplingIntervalHint)) return false;
            if (!Utils.IsEqual(m_deadbandType, value.m_deadbandType)) return false;
            if (!Utils.IsEqual(m_deadbandValue, value.m_deadbandValue)) return false;
            if (!Utils.IsEqual(m_indexRange, value.m_indexRange)) return false;
            if (!Utils.IsEqual(m_substituteValue, value.m_substituteValue)) return false;
            if (!Utils.IsEqual(m_metaDataProperties, value.m_metaDataProperties)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (PublishedVariableDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedVariableDataType clone = (PublishedVariableDataType)base.MemberwiseClone();

            clone.m_publishedVariable = (NodeId)Utils.Clone(this.m_publishedVariable);
            clone.m_attributeId = (uint)Utils.Clone(this.m_attributeId);
            clone.m_samplingIntervalHint = (double)Utils.Clone(this.m_samplingIntervalHint);
            clone.m_deadbandType = (uint)Utils.Clone(this.m_deadbandType);
            clone.m_deadbandValue = (double)Utils.Clone(this.m_deadbandValue);
            clone.m_indexRange = (string)Utils.Clone(this.m_indexRange);
            clone.m_substituteValue = (Variant)Utils.Clone(this.m_substituteValue);
            clone.m_metaDataProperties = (QualifiedNameCollection)Utils.Clone(this.m_metaDataProperties);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_publishedVariable;
        private uint m_attributeId;
        private double m_samplingIntervalHint;
        private uint m_deadbandType;
        private double m_deadbandValue;
        private string m_indexRange;
        private Variant m_substituteValue;
        private QualifiedNameCollection m_metaDataProperties;
        #endregion
    }

    #region PublishedVariableDataTypeCollection Class
    /// <summary>
    /// A collection of PublishedVariableDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPublishedVariableDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PublishedVariableDataType")]
    #if !NET_STANDARD
    public partial class PublishedVariableDataTypeCollection : List<PublishedVariableDataType>, ICloneable
    #else
    public partial class PublishedVariableDataTypeCollection : List<PublishedVariableDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public PublishedVariableDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public PublishedVariableDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public PublishedVariableDataTypeCollection(IEnumerable<PublishedVariableDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator PublishedVariableDataTypeCollection(PublishedVariableDataType[] values)
        {
            if (values != null)
            {
                return new PublishedVariableDataTypeCollection(values);
            }

            return new PublishedVariableDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator PublishedVariableDataType[](PublishedVariableDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (PublishedVariableDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedVariableDataTypeCollection clone = new PublishedVariableDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PublishedVariableDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PublishedDataItemsDataType Class
    #if (!OPCUA_EXCLUDE_PublishedDataItemsDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PublishedDataItemsDataType : PublishedDataSetSourceDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public PublishedDataItemsDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_publishedData = new PublishedVariableDataTypeCollection();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "PublishedData", IsRequired = false, Order = 1)]
        public PublishedVariableDataTypeCollection PublishedData
        {
            get
            {
                return m_publishedData;
            }

            set
            {
                m_publishedData = value;

                if (value == null)
                {
                    m_publishedData = new PublishedVariableDataTypeCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.PublishedDataItemsDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.PublishedDataItemsDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.PublishedDataItemsDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeableArray("PublishedData", PublishedData.ToArray(), typeof(PublishedVariableDataType));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            PublishedData = (PublishedVariableDataTypeCollection)decoder.ReadEncodeableArray("PublishedData", typeof(PublishedVariableDataType));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PublishedDataItemsDataType value = encodeable as PublishedDataItemsDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_publishedData, value.m_publishedData)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (PublishedDataItemsDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedDataItemsDataType clone = (PublishedDataItemsDataType)base.MemberwiseClone();

            clone.m_publishedData = (PublishedVariableDataTypeCollection)Utils.Clone(this.m_publishedData);

            return clone;
        }
        #endregion

        #region Private Fields
        private PublishedVariableDataTypeCollection m_publishedData;
        #endregion
    }

    #region PublishedDataItemsDataTypeCollection Class
    /// <summary>
    /// A collection of PublishedDataItemsDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPublishedDataItemsDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PublishedDataItemsDataType")]
    #if !NET_STANDARD
    public partial class PublishedDataItemsDataTypeCollection : List<PublishedDataItemsDataType>, ICloneable
    #else
    public partial class PublishedDataItemsDataTypeCollection : List<PublishedDataItemsDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public PublishedDataItemsDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public PublishedDataItemsDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public PublishedDataItemsDataTypeCollection(IEnumerable<PublishedDataItemsDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator PublishedDataItemsDataTypeCollection(PublishedDataItemsDataType[] values)
        {
            if (values != null)
            {
                return new PublishedDataItemsDataTypeCollection(values);
            }

            return new PublishedDataItemsDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator PublishedDataItemsDataType[](PublishedDataItemsDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (PublishedDataItemsDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedDataItemsDataTypeCollection clone = new PublishedDataItemsDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PublishedDataItemsDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PublishedEventsDataType Class
    #if (!OPCUA_EXCLUDE_PublishedEventsDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PublishedEventsDataType : PublishedDataSetSourceDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public PublishedEventsDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_eventNotifier = null;
            m_selectedFields = new SimpleAttributeOperandCollection();
            m_filter = new ContentFilter();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "EventNotifier", IsRequired = false, Order = 1)]
        public NodeId EventNotifier
        {
            get { return m_eventNotifier;  }
            set { m_eventNotifier = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "SelectedFields", IsRequired = false, Order = 2)]
        public SimpleAttributeOperandCollection SelectedFields
        {
            get
            {
                return m_selectedFields;
            }

            set
            {
                m_selectedFields = value;

                if (value == null)
                {
                    m_selectedFields = new SimpleAttributeOperandCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "Filter", IsRequired = false, Order = 3)]
        public ContentFilter Filter
        {
            get
            {
                return m_filter;
            }

            set
            {
                m_filter = value;

                if (value == null)
                {
                    m_filter = new ContentFilter();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.PublishedEventsDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.PublishedEventsDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.PublishedEventsDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("EventNotifier", EventNotifier);
            encoder.WriteEncodeableArray("SelectedFields", SelectedFields.ToArray(), typeof(SimpleAttributeOperand));
            encoder.WriteEncodeable("Filter", Filter, typeof(ContentFilter));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            EventNotifier = decoder.ReadNodeId("EventNotifier");
            SelectedFields = (SimpleAttributeOperandCollection)decoder.ReadEncodeableArray("SelectedFields", typeof(SimpleAttributeOperand));
            Filter = (ContentFilter)decoder.ReadEncodeable("Filter", typeof(ContentFilter));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PublishedEventsDataType value = encodeable as PublishedEventsDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_eventNotifier, value.m_eventNotifier)) return false;
            if (!Utils.IsEqual(m_selectedFields, value.m_selectedFields)) return false;
            if (!Utils.IsEqual(m_filter, value.m_filter)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (PublishedEventsDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedEventsDataType clone = (PublishedEventsDataType)base.MemberwiseClone();

            clone.m_eventNotifier = (NodeId)Utils.Clone(this.m_eventNotifier);
            clone.m_selectedFields = (SimpleAttributeOperandCollection)Utils.Clone(this.m_selectedFields);
            clone.m_filter = (ContentFilter)Utils.Clone(this.m_filter);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_eventNotifier;
        private SimpleAttributeOperandCollection m_selectedFields;
        private ContentFilter m_filter;
        #endregion
    }

    #region PublishedEventsDataTypeCollection Class
    /// <summary>
    /// A collection of PublishedEventsDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPublishedEventsDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PublishedEventsDataType")]
    #if !NET_STANDARD
    public partial class PublishedEventsDataTypeCollection : List<PublishedEventsDataType>, ICloneable
    #else
    public partial class PublishedEventsDataTypeCollection : List<PublishedEventsDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public PublishedEventsDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public PublishedEventsDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public PublishedEventsDataTypeCollection(IEnumerable<PublishedEventsDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator PublishedEventsDataTypeCollection(PublishedEventsDataType[] values)
        {
            if (values != null)
            {
                return new PublishedEventsDataTypeCollection(values);
            }

            return new PublishedEventsDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator PublishedEventsDataType[](PublishedEventsDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (PublishedEventsDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedEventsDataTypeCollection clone = new PublishedEventsDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PublishedEventsDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetFieldContentMask Enumeration
    #if (!OPCUA_EXCLUDE_DataSetFieldContentMask)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum DataSetFieldContentMask : UInt32
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "StatusCode_1")]
        StatusCode = 1,

        /// <remarks />
        [EnumMember(Value = "SourceTimestamp_2")]
        SourceTimestamp = 2,

        /// <remarks />
        [EnumMember(Value = "ServerTimestamp_4")]
        ServerTimestamp = 4,

        /// <remarks />
        [EnumMember(Value = "SourcePicoSeconds_8")]
        SourcePicoSeconds = 8,

        /// <remarks />
        [EnumMember(Value = "ServerPicoSeconds_16")]
        ServerPicoSeconds = 16,

        /// <remarks />
        [EnumMember(Value = "RawData_32")]
        RawData = 32,
    }

    #region DataSetFieldContentMaskCollection Class
    /// <summary>
    /// A collection of DataSetFieldContentMask objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataSetFieldContentMask", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataSetFieldContentMask")]
    #if !NET_STANDARD
    public partial class DataSetFieldContentMaskCollection : List<DataSetFieldContentMask>, ICloneable
    #else
    public partial class DataSetFieldContentMaskCollection : List<DataSetFieldContentMask>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public DataSetFieldContentMaskCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public DataSetFieldContentMaskCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public DataSetFieldContentMaskCollection(IEnumerable<DataSetFieldContentMask> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator DataSetFieldContentMaskCollection(DataSetFieldContentMask[] values)
        {
            if (values != null)
            {
                return new DataSetFieldContentMaskCollection(values);
            }

            return new DataSetFieldContentMaskCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator DataSetFieldContentMask[](DataSetFieldContentMaskCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (DataSetFieldContentMaskCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetFieldContentMaskCollection clone = new DataSetFieldContentMaskCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataSetFieldContentMask)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetWriterDataType Class
    #if (!OPCUA_EXCLUDE_DataSetWriterDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataSetWriterDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public DataSetWriterDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_name = null;
            m_enabled = true;
            m_dataSetWriterId = (ushort)0;
            m_dataSetFieldContentMask = 0;
            m_keyFrameCount = (uint)0;
            m_dataSetName = null;
            m_dataSetWriterProperties = new KeyValuePairCollection();
            m_transportSettings = null;
            m_messageSettings = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "Enabled", IsRequired = false, Order = 2)]
        public bool Enabled
        {
            get { return m_enabled;  }
            set { m_enabled = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetWriterId", IsRequired = false, Order = 3)]
        public ushort DataSetWriterId
        {
            get { return m_dataSetWriterId;  }
            set { m_dataSetWriterId = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetFieldContentMask", IsRequired = false, Order = 4)]
        public uint DataSetFieldContentMask
        {
            get { return m_dataSetFieldContentMask;  }
            set { m_dataSetFieldContentMask = value; }
        }

        /// <remarks />
        [DataMember(Name = "KeyFrameCount", IsRequired = false, Order = 5)]
        public uint KeyFrameCount
        {
            get { return m_keyFrameCount;  }
            set { m_keyFrameCount = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetName", IsRequired = false, Order = 6)]
        public string DataSetName
        {
            get { return m_dataSetName;  }
            set { m_dataSetName = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "DataSetWriterProperties", IsRequired = false, Order = 7)]
        public KeyValuePairCollection DataSetWriterProperties
        {
            get
            {
                return m_dataSetWriterProperties;
            }

            set
            {
                m_dataSetWriterProperties = value;

                if (value == null)
                {
                    m_dataSetWriterProperties = new KeyValuePairCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "TransportSettings", IsRequired = false, Order = 8)]
        public ExtensionObject TransportSettings
        {
            get { return m_transportSettings;  }
            set { m_transportSettings = value; }
        }

        /// <remarks />
        [DataMember(Name = "MessageSettings", IsRequired = false, Order = 9)]
        public ExtensionObject MessageSettings
        {
            get { return m_messageSettings;  }
            set { m_messageSettings = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.DataSetWriterDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.DataSetWriterDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.DataSetWriterDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteBoolean("Enabled", Enabled);
            encoder.WriteUInt16("DataSetWriterId", DataSetWriterId);
            encoder.WriteUInt32("DataSetFieldContentMask", DataSetFieldContentMask);
            encoder.WriteUInt32("KeyFrameCount", KeyFrameCount);
            encoder.WriteString("DataSetName", DataSetName);
            encoder.WriteEncodeableArray("DataSetWriterProperties", DataSetWriterProperties.ToArray(), typeof(KeyValuePair));
            encoder.WriteExtensionObject("TransportSettings", TransportSettings);
            encoder.WriteExtensionObject("MessageSettings", MessageSettings);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            Enabled = decoder.ReadBoolean("Enabled");
            DataSetWriterId = decoder.ReadUInt16("DataSetWriterId");
            DataSetFieldContentMask = decoder.ReadUInt32("DataSetFieldContentMask");
            KeyFrameCount = decoder.ReadUInt32("KeyFrameCount");
            DataSetName = decoder.ReadString("DataSetName");
            DataSetWriterProperties = (KeyValuePairCollection)decoder.ReadEncodeableArray("DataSetWriterProperties", typeof(KeyValuePair));
            TransportSettings = decoder.ReadExtensionObject("TransportSettings");
            MessageSettings = decoder.ReadExtensionObject("MessageSettings");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataSetWriterDataType value = encodeable as DataSetWriterDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_enabled, value.m_enabled)) return false;
            if (!Utils.IsEqual(m_dataSetWriterId, value.m_dataSetWriterId)) return false;
            if (!Utils.IsEqual(m_dataSetFieldContentMask, value.m_dataSetFieldContentMask)) return false;
            if (!Utils.IsEqual(m_keyFrameCount, value.m_keyFrameCount)) return false;
            if (!Utils.IsEqual(m_dataSetName, value.m_dataSetName)) return false;
            if (!Utils.IsEqual(m_dataSetWriterProperties, value.m_dataSetWriterProperties)) return false;
            if (!Utils.IsEqual(m_transportSettings, value.m_transportSettings)) return false;
            if (!Utils.IsEqual(m_messageSettings, value.m_messageSettings)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DataSetWriterDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetWriterDataType clone = (DataSetWriterDataType)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_enabled = (bool)Utils.Clone(this.m_enabled);
            clone.m_dataSetWriterId = (ushort)Utils.Clone(this.m_dataSetWriterId);
            clone.m_dataSetFieldContentMask = (uint)Utils.Clone(this.m_dataSetFieldContentMask);
            clone.m_keyFrameCount = (uint)Utils.Clone(this.m_keyFrameCount);
            clone.m_dataSetName = (string)Utils.Clone(this.m_dataSetName);
            clone.m_dataSetWriterProperties = (KeyValuePairCollection)Utils.Clone(this.m_dataSetWriterProperties);
            clone.m_transportSettings = (ExtensionObject)Utils.Clone(this.m_transportSettings);
            clone.m_messageSettings = (ExtensionObject)Utils.Clone(this.m_messageSettings);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private bool m_enabled;
        private ushort m_dataSetWriterId;
        private uint m_dataSetFieldContentMask;
        private uint m_keyFrameCount;
        private string m_dataSetName;
        private KeyValuePairCollection m_dataSetWriterProperties;
        private ExtensionObject m_transportSettings;
        private ExtensionObject m_messageSettings;
        #endregion
    }

    #region DataSetWriterDataTypeCollection Class
    /// <summary>
    /// A collection of DataSetWriterDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataSetWriterDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataSetWriterDataType")]
    #if !NET_STANDARD
    public partial class DataSetWriterDataTypeCollection : List<DataSetWriterDataType>, ICloneable
    #else
    public partial class DataSetWriterDataTypeCollection : List<DataSetWriterDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public DataSetWriterDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public DataSetWriterDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public DataSetWriterDataTypeCollection(IEnumerable<DataSetWriterDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator DataSetWriterDataTypeCollection(DataSetWriterDataType[] values)
        {
            if (values != null)
            {
                return new DataSetWriterDataTypeCollection(values);
            }

            return new DataSetWriterDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator DataSetWriterDataType[](DataSetWriterDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (DataSetWriterDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetWriterDataTypeCollection clone = new DataSetWriterDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataSetWriterDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetWriterTransportDataType Class
    #if (!OPCUA_EXCLUDE_DataSetWriterTransportDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataSetWriterTransportDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public DataSetWriterTransportDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.DataSetWriterTransportDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.DataSetWriterTransportDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.DataSetWriterTransportDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataSetWriterTransportDataType value = encodeable as DataSetWriterTransportDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DataSetWriterTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetWriterTransportDataType clone = (DataSetWriterTransportDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region DataSetWriterTransportDataTypeCollection Class
    /// <summary>
    /// A collection of DataSetWriterTransportDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataSetWriterTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataSetWriterTransportDataType")]
    #if !NET_STANDARD
    public partial class DataSetWriterTransportDataTypeCollection : List<DataSetWriterTransportDataType>, ICloneable
    #else
    public partial class DataSetWriterTransportDataTypeCollection : List<DataSetWriterTransportDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public DataSetWriterTransportDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public DataSetWriterTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public DataSetWriterTransportDataTypeCollection(IEnumerable<DataSetWriterTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator DataSetWriterTransportDataTypeCollection(DataSetWriterTransportDataType[] values)
        {
            if (values != null)
            {
                return new DataSetWriterTransportDataTypeCollection(values);
            }

            return new DataSetWriterTransportDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator DataSetWriterTransportDataType[](DataSetWriterTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (DataSetWriterTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetWriterTransportDataTypeCollection clone = new DataSetWriterTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataSetWriterTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetWriterMessageDataType Class
    #if (!OPCUA_EXCLUDE_DataSetWriterMessageDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataSetWriterMessageDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public DataSetWriterMessageDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.DataSetWriterMessageDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.DataSetWriterMessageDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.DataSetWriterMessageDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataSetWriterMessageDataType value = encodeable as DataSetWriterMessageDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DataSetWriterMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetWriterMessageDataType clone = (DataSetWriterMessageDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region DataSetWriterMessageDataTypeCollection Class
    /// <summary>
    /// A collection of DataSetWriterMessageDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataSetWriterMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataSetWriterMessageDataType")]
    #if !NET_STANDARD
    public partial class DataSetWriterMessageDataTypeCollection : List<DataSetWriterMessageDataType>, ICloneable
    #else
    public partial class DataSetWriterMessageDataTypeCollection : List<DataSetWriterMessageDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public DataSetWriterMessageDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public DataSetWriterMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public DataSetWriterMessageDataTypeCollection(IEnumerable<DataSetWriterMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator DataSetWriterMessageDataTypeCollection(DataSetWriterMessageDataType[] values)
        {
            if (values != null)
            {
                return new DataSetWriterMessageDataTypeCollection(values);
            }

            return new DataSetWriterMessageDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator DataSetWriterMessageDataType[](DataSetWriterMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (DataSetWriterMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetWriterMessageDataTypeCollection clone = new DataSetWriterMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataSetWriterMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PubSubGroupDataType Class
    #if (!OPCUA_EXCLUDE_PubSubGroupDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PubSubGroupDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public PubSubGroupDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_name = null;
            m_enabled = true;
            m_securityMode = MessageSecurityMode.Invalid;
            m_securityGroupId = null;
            m_securityKeyServices = new EndpointDescriptionCollection();
            m_maxNetworkMessageSize = (uint)0;
            m_groupProperties = new KeyValuePairCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "Enabled", IsRequired = false, Order = 2)]
        public bool Enabled
        {
            get { return m_enabled;  }
            set { m_enabled = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityMode", IsRequired = false, Order = 3)]
        public MessageSecurityMode SecurityMode
        {
            get { return m_securityMode;  }
            set { m_securityMode = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityGroupId", IsRequired = false, Order = 4)]
        public string SecurityGroupId
        {
            get { return m_securityGroupId;  }
            set { m_securityGroupId = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "SecurityKeyServices", IsRequired = false, Order = 5)]
        public EndpointDescriptionCollection SecurityKeyServices
        {
            get
            {
                return m_securityKeyServices;
            }

            set
            {
                m_securityKeyServices = value;

                if (value == null)
                {
                    m_securityKeyServices = new EndpointDescriptionCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "MaxNetworkMessageSize", IsRequired = false, Order = 6)]
        public uint MaxNetworkMessageSize
        {
            get { return m_maxNetworkMessageSize;  }
            set { m_maxNetworkMessageSize = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "GroupProperties", IsRequired = false, Order = 7)]
        public KeyValuePairCollection GroupProperties
        {
            get
            {
                return m_groupProperties;
            }

            set
            {
                m_groupProperties = value;

                if (value == null)
                {
                    m_groupProperties = new KeyValuePairCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.PubSubGroupDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.PubSubGroupDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.PubSubGroupDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteBoolean("Enabled", Enabled);
            encoder.WriteEnumerated("SecurityMode", SecurityMode);
            encoder.WriteString("SecurityGroupId", SecurityGroupId);
            encoder.WriteEncodeableArray("SecurityKeyServices", SecurityKeyServices.ToArray(), typeof(EndpointDescription));
            encoder.WriteUInt32("MaxNetworkMessageSize", MaxNetworkMessageSize);
            encoder.WriteEncodeableArray("GroupProperties", GroupProperties.ToArray(), typeof(KeyValuePair));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            Enabled = decoder.ReadBoolean("Enabled");
            SecurityMode = (MessageSecurityMode)decoder.ReadEnumerated("SecurityMode", typeof(MessageSecurityMode));
            SecurityGroupId = decoder.ReadString("SecurityGroupId");
            SecurityKeyServices = (EndpointDescriptionCollection)decoder.ReadEncodeableArray("SecurityKeyServices", typeof(EndpointDescription));
            MaxNetworkMessageSize = decoder.ReadUInt32("MaxNetworkMessageSize");
            GroupProperties = (KeyValuePairCollection)decoder.ReadEncodeableArray("GroupProperties", typeof(KeyValuePair));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PubSubGroupDataType value = encodeable as PubSubGroupDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_enabled, value.m_enabled)) return false;
            if (!Utils.IsEqual(m_securityMode, value.m_securityMode)) return false;
            if (!Utils.IsEqual(m_securityGroupId, value.m_securityGroupId)) return false;
            if (!Utils.IsEqual(m_securityKeyServices, value.m_securityKeyServices)) return false;
            if (!Utils.IsEqual(m_maxNetworkMessageSize, value.m_maxNetworkMessageSize)) return false;
            if (!Utils.IsEqual(m_groupProperties, value.m_groupProperties)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (PubSubGroupDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubGroupDataType clone = (PubSubGroupDataType)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_enabled = (bool)Utils.Clone(this.m_enabled);
            clone.m_securityMode = (MessageSecurityMode)Utils.Clone(this.m_securityMode);
            clone.m_securityGroupId = (string)Utils.Clone(this.m_securityGroupId);
            clone.m_securityKeyServices = (EndpointDescriptionCollection)Utils.Clone(this.m_securityKeyServices);
            clone.m_maxNetworkMessageSize = (uint)Utils.Clone(this.m_maxNetworkMessageSize);
            clone.m_groupProperties = (KeyValuePairCollection)Utils.Clone(this.m_groupProperties);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private bool m_enabled;
        private MessageSecurityMode m_securityMode;
        private string m_securityGroupId;
        private EndpointDescriptionCollection m_securityKeyServices;
        private uint m_maxNetworkMessageSize;
        private KeyValuePairCollection m_groupProperties;
        #endregion
    }

    #region PubSubGroupDataTypeCollection Class
    /// <summary>
    /// A collection of PubSubGroupDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPubSubGroupDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PubSubGroupDataType")]
    #if !NET_STANDARD
    public partial class PubSubGroupDataTypeCollection : List<PubSubGroupDataType>, ICloneable
    #else
    public partial class PubSubGroupDataTypeCollection : List<PubSubGroupDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public PubSubGroupDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public PubSubGroupDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public PubSubGroupDataTypeCollection(IEnumerable<PubSubGroupDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator PubSubGroupDataTypeCollection(PubSubGroupDataType[] values)
        {
            if (values != null)
            {
                return new PubSubGroupDataTypeCollection(values);
            }

            return new PubSubGroupDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator PubSubGroupDataType[](PubSubGroupDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (PubSubGroupDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubGroupDataTypeCollection clone = new PubSubGroupDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PubSubGroupDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region WriterGroupDataType Class
    #if (!OPCUA_EXCLUDE_WriterGroupDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class WriterGroupDataType : PubSubGroupDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public WriterGroupDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_writerGroupId = (ushort)0;
            m_publishingInterval = (double)0;
            m_keepAliveTime = (double)0;
            m_priority = (byte)0;
            m_localeIds = new StringCollection();
            m_headerLayoutUri = null;
            m_transportSettings = null;
            m_messageSettings = null;
            m_dataSetWriters = new DataSetWriterDataTypeCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "WriterGroupId", IsRequired = false, Order = 1)]
        public ushort WriterGroupId
        {
            get { return m_writerGroupId;  }
            set { m_writerGroupId = value; }
        }

        /// <remarks />
        [DataMember(Name = "PublishingInterval", IsRequired = false, Order = 2)]
        public double PublishingInterval
        {
            get { return m_publishingInterval;  }
            set { m_publishingInterval = value; }
        }

        /// <remarks />
        [DataMember(Name = "KeepAliveTime", IsRequired = false, Order = 3)]
        public double KeepAliveTime
        {
            get { return m_keepAliveTime;  }
            set { m_keepAliveTime = value; }
        }

        /// <remarks />
        [DataMember(Name = "Priority", IsRequired = false, Order = 4)]
        public byte Priority
        {
            get { return m_priority;  }
            set { m_priority = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "LocaleIds", IsRequired = false, Order = 5)]
        public StringCollection LocaleIds
        {
            get
            {
                return m_localeIds;
            }

            set
            {
                m_localeIds = value;

                if (value == null)
                {
                    m_localeIds = new StringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "HeaderLayoutUri", IsRequired = false, Order = 6)]
        public string HeaderLayoutUri
        {
            get { return m_headerLayoutUri;  }
            set { m_headerLayoutUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "TransportSettings", IsRequired = false, Order = 7)]
        public ExtensionObject TransportSettings
        {
            get { return m_transportSettings;  }
            set { m_transportSettings = value; }
        }

        /// <remarks />
        [DataMember(Name = "MessageSettings", IsRequired = false, Order = 8)]
        public ExtensionObject MessageSettings
        {
            get { return m_messageSettings;  }
            set { m_messageSettings = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "DataSetWriters", IsRequired = false, Order = 9)]
        public DataSetWriterDataTypeCollection DataSetWriters
        {
            get
            {
                return m_dataSetWriters;
            }

            set
            {
                m_dataSetWriters = value;

                if (value == null)
                {
                    m_dataSetWriters = new DataSetWriterDataTypeCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.WriterGroupDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.WriterGroupDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.WriterGroupDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt16("WriterGroupId", WriterGroupId);
            encoder.WriteDouble("PublishingInterval", PublishingInterval);
            encoder.WriteDouble("KeepAliveTime", KeepAliveTime);
            encoder.WriteByte("Priority", Priority);
            encoder.WriteStringArray("LocaleIds", LocaleIds);
            encoder.WriteString("HeaderLayoutUri", HeaderLayoutUri);
            encoder.WriteExtensionObject("TransportSettings", TransportSettings);
            encoder.WriteExtensionObject("MessageSettings", MessageSettings);
            encoder.WriteEncodeableArray("DataSetWriters", DataSetWriters.ToArray(), typeof(DataSetWriterDataType));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            WriterGroupId = decoder.ReadUInt16("WriterGroupId");
            PublishingInterval = decoder.ReadDouble("PublishingInterval");
            KeepAliveTime = decoder.ReadDouble("KeepAliveTime");
            Priority = decoder.ReadByte("Priority");
            LocaleIds = decoder.ReadStringArray("LocaleIds");
            HeaderLayoutUri = decoder.ReadString("HeaderLayoutUri");
            TransportSettings = decoder.ReadExtensionObject("TransportSettings");
            MessageSettings = decoder.ReadExtensionObject("MessageSettings");
            DataSetWriters = (DataSetWriterDataTypeCollection)decoder.ReadEncodeableArray("DataSetWriters", typeof(DataSetWriterDataType));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            WriterGroupDataType value = encodeable as WriterGroupDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_writerGroupId, value.m_writerGroupId)) return false;
            if (!Utils.IsEqual(m_publishingInterval, value.m_publishingInterval)) return false;
            if (!Utils.IsEqual(m_keepAliveTime, value.m_keepAliveTime)) return false;
            if (!Utils.IsEqual(m_priority, value.m_priority)) return false;
            if (!Utils.IsEqual(m_localeIds, value.m_localeIds)) return false;
            if (!Utils.IsEqual(m_headerLayoutUri, value.m_headerLayoutUri)) return false;
            if (!Utils.IsEqual(m_transportSettings, value.m_transportSettings)) return false;
            if (!Utils.IsEqual(m_messageSettings, value.m_messageSettings)) return false;
            if (!Utils.IsEqual(m_dataSetWriters, value.m_dataSetWriters)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (WriterGroupDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            WriterGroupDataType clone = (WriterGroupDataType)base.MemberwiseClone();

            clone.m_writerGroupId = (ushort)Utils.Clone(this.m_writerGroupId);
            clone.m_publishingInterval = (double)Utils.Clone(this.m_publishingInterval);
            clone.m_keepAliveTime = (double)Utils.Clone(this.m_keepAliveTime);
            clone.m_priority = (byte)Utils.Clone(this.m_priority);
            clone.m_localeIds = (StringCollection)Utils.Clone(this.m_localeIds);
            clone.m_headerLayoutUri = (string)Utils.Clone(this.m_headerLayoutUri);
            clone.m_transportSettings = (ExtensionObject)Utils.Clone(this.m_transportSettings);
            clone.m_messageSettings = (ExtensionObject)Utils.Clone(this.m_messageSettings);
            clone.m_dataSetWriters = (DataSetWriterDataTypeCollection)Utils.Clone(this.m_dataSetWriters);

            return clone;
        }
        #endregion

        #region Private Fields
        private ushort m_writerGroupId;
        private double m_publishingInterval;
        private double m_keepAliveTime;
        private byte m_priority;
        private StringCollection m_localeIds;
        private string m_headerLayoutUri;
        private ExtensionObject m_transportSettings;
        private ExtensionObject m_messageSettings;
        private DataSetWriterDataTypeCollection m_dataSetWriters;
        #endregion
    }

    #region WriterGroupDataTypeCollection Class
    /// <summary>
    /// A collection of WriterGroupDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfWriterGroupDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "WriterGroupDataType")]
    #if !NET_STANDARD
    public partial class WriterGroupDataTypeCollection : List<WriterGroupDataType>, ICloneable
    #else
    public partial class WriterGroupDataTypeCollection : List<WriterGroupDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public WriterGroupDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public WriterGroupDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public WriterGroupDataTypeCollection(IEnumerable<WriterGroupDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator WriterGroupDataTypeCollection(WriterGroupDataType[] values)
        {
            if (values != null)
            {
                return new WriterGroupDataTypeCollection(values);
            }

            return new WriterGroupDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator WriterGroupDataType[](WriterGroupDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (WriterGroupDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            WriterGroupDataTypeCollection clone = new WriterGroupDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((WriterGroupDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region WriterGroupTransportDataType Class
    #if (!OPCUA_EXCLUDE_WriterGroupTransportDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class WriterGroupTransportDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public WriterGroupTransportDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.WriterGroupTransportDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.WriterGroupTransportDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.WriterGroupTransportDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            WriterGroupTransportDataType value = encodeable as WriterGroupTransportDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (WriterGroupTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            WriterGroupTransportDataType clone = (WriterGroupTransportDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region WriterGroupTransportDataTypeCollection Class
    /// <summary>
    /// A collection of WriterGroupTransportDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfWriterGroupTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "WriterGroupTransportDataType")]
    #if !NET_STANDARD
    public partial class WriterGroupTransportDataTypeCollection : List<WriterGroupTransportDataType>, ICloneable
    #else
    public partial class WriterGroupTransportDataTypeCollection : List<WriterGroupTransportDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public WriterGroupTransportDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public WriterGroupTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public WriterGroupTransportDataTypeCollection(IEnumerable<WriterGroupTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator WriterGroupTransportDataTypeCollection(WriterGroupTransportDataType[] values)
        {
            if (values != null)
            {
                return new WriterGroupTransportDataTypeCollection(values);
            }

            return new WriterGroupTransportDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator WriterGroupTransportDataType[](WriterGroupTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (WriterGroupTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            WriterGroupTransportDataTypeCollection clone = new WriterGroupTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((WriterGroupTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region WriterGroupMessageDataType Class
    #if (!OPCUA_EXCLUDE_WriterGroupMessageDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class WriterGroupMessageDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public WriterGroupMessageDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.WriterGroupMessageDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.WriterGroupMessageDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.WriterGroupMessageDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            WriterGroupMessageDataType value = encodeable as WriterGroupMessageDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (WriterGroupMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            WriterGroupMessageDataType clone = (WriterGroupMessageDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region WriterGroupMessageDataTypeCollection Class
    /// <summary>
    /// A collection of WriterGroupMessageDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfWriterGroupMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "WriterGroupMessageDataType")]
    #if !NET_STANDARD
    public partial class WriterGroupMessageDataTypeCollection : List<WriterGroupMessageDataType>, ICloneable
    #else
    public partial class WriterGroupMessageDataTypeCollection : List<WriterGroupMessageDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public WriterGroupMessageDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public WriterGroupMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public WriterGroupMessageDataTypeCollection(IEnumerable<WriterGroupMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator WriterGroupMessageDataTypeCollection(WriterGroupMessageDataType[] values)
        {
            if (values != null)
            {
                return new WriterGroupMessageDataTypeCollection(values);
            }

            return new WriterGroupMessageDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator WriterGroupMessageDataType[](WriterGroupMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (WriterGroupMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            WriterGroupMessageDataTypeCollection clone = new WriterGroupMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((WriterGroupMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PubSubConnectionDataType Class
    #if (!OPCUA_EXCLUDE_PubSubConnectionDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PubSubConnectionDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public PubSubConnectionDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_name = null;
            m_enabled = true;
            m_publisherId = Variant.Null;
            m_transportProfileUri = null;
            m_address = null;
            m_connectionProperties = new KeyValuePairCollection();
            m_transportSettings = null;
            m_writerGroups = new WriterGroupDataTypeCollection();
            m_readerGroups = new ReaderGroupDataTypeCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "Enabled", IsRequired = false, Order = 2)]
        public bool Enabled
        {
            get { return m_enabled;  }
            set { m_enabled = value; }
        }

        /// <remarks />
        [DataMember(Name = "PublisherId", IsRequired = false, Order = 3)]
        public Variant PublisherId
        {
            get { return m_publisherId;  }
            set { m_publisherId = value; }
        }

        /// <remarks />
        [DataMember(Name = "TransportProfileUri", IsRequired = false, Order = 4)]
        public string TransportProfileUri
        {
            get { return m_transportProfileUri;  }
            set { m_transportProfileUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "Address", IsRequired = false, Order = 5)]
        public ExtensionObject Address
        {
            get { return m_address;  }
            set { m_address = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ConnectionProperties", IsRequired = false, Order = 6)]
        public KeyValuePairCollection ConnectionProperties
        {
            get
            {
                return m_connectionProperties;
            }

            set
            {
                m_connectionProperties = value;

                if (value == null)
                {
                    m_connectionProperties = new KeyValuePairCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "TransportSettings", IsRequired = false, Order = 7)]
        public ExtensionObject TransportSettings
        {
            get { return m_transportSettings;  }
            set { m_transportSettings = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "WriterGroups", IsRequired = false, Order = 8)]
        public WriterGroupDataTypeCollection WriterGroups
        {
            get
            {
                return m_writerGroups;
            }

            set
            {
                m_writerGroups = value;

                if (value == null)
                {
                    m_writerGroups = new WriterGroupDataTypeCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ReaderGroups", IsRequired = false, Order = 9)]
        public ReaderGroupDataTypeCollection ReaderGroups
        {
            get
            {
                return m_readerGroups;
            }

            set
            {
                m_readerGroups = value;

                if (value == null)
                {
                    m_readerGroups = new ReaderGroupDataTypeCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.PubSubConnectionDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.PubSubConnectionDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.PubSubConnectionDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteBoolean("Enabled", Enabled);
            encoder.WriteVariant("PublisherId", PublisherId);
            encoder.WriteString("TransportProfileUri", TransportProfileUri);
            encoder.WriteExtensionObject("Address", Address);
            encoder.WriteEncodeableArray("ConnectionProperties", ConnectionProperties.ToArray(), typeof(KeyValuePair));
            encoder.WriteExtensionObject("TransportSettings", TransportSettings);
            encoder.WriteEncodeableArray("WriterGroups", WriterGroups.ToArray(), typeof(WriterGroupDataType));
            encoder.WriteEncodeableArray("ReaderGroups", ReaderGroups.ToArray(), typeof(ReaderGroupDataType));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            Enabled = decoder.ReadBoolean("Enabled");
            PublisherId = decoder.ReadVariant("PublisherId");
            TransportProfileUri = decoder.ReadString("TransportProfileUri");
            Address = decoder.ReadExtensionObject("Address");
            ConnectionProperties = (KeyValuePairCollection)decoder.ReadEncodeableArray("ConnectionProperties", typeof(KeyValuePair));
            TransportSettings = decoder.ReadExtensionObject("TransportSettings");
            WriterGroups = (WriterGroupDataTypeCollection)decoder.ReadEncodeableArray("WriterGroups", typeof(WriterGroupDataType));
            ReaderGroups = (ReaderGroupDataTypeCollection)decoder.ReadEncodeableArray("ReaderGroups", typeof(ReaderGroupDataType));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PubSubConnectionDataType value = encodeable as PubSubConnectionDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_enabled, value.m_enabled)) return false;
            if (!Utils.IsEqual(m_publisherId, value.m_publisherId)) return false;
            if (!Utils.IsEqual(m_transportProfileUri, value.m_transportProfileUri)) return false;
            if (!Utils.IsEqual(m_address, value.m_address)) return false;
            if (!Utils.IsEqual(m_connectionProperties, value.m_connectionProperties)) return false;
            if (!Utils.IsEqual(m_transportSettings, value.m_transportSettings)) return false;
            if (!Utils.IsEqual(m_writerGroups, value.m_writerGroups)) return false;
            if (!Utils.IsEqual(m_readerGroups, value.m_readerGroups)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (PubSubConnectionDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubConnectionDataType clone = (PubSubConnectionDataType)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_enabled = (bool)Utils.Clone(this.m_enabled);
            clone.m_publisherId = (Variant)Utils.Clone(this.m_publisherId);
            clone.m_transportProfileUri = (string)Utils.Clone(this.m_transportProfileUri);
            clone.m_address = (ExtensionObject)Utils.Clone(this.m_address);
            clone.m_connectionProperties = (KeyValuePairCollection)Utils.Clone(this.m_connectionProperties);
            clone.m_transportSettings = (ExtensionObject)Utils.Clone(this.m_transportSettings);
            clone.m_writerGroups = (WriterGroupDataTypeCollection)Utils.Clone(this.m_writerGroups);
            clone.m_readerGroups = (ReaderGroupDataTypeCollection)Utils.Clone(this.m_readerGroups);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private bool m_enabled;
        private Variant m_publisherId;
        private string m_transportProfileUri;
        private ExtensionObject m_address;
        private KeyValuePairCollection m_connectionProperties;
        private ExtensionObject m_transportSettings;
        private WriterGroupDataTypeCollection m_writerGroups;
        private ReaderGroupDataTypeCollection m_readerGroups;
        #endregion
    }

    #region PubSubConnectionDataTypeCollection Class
    /// <summary>
    /// A collection of PubSubConnectionDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPubSubConnectionDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PubSubConnectionDataType")]
    #if !NET_STANDARD
    public partial class PubSubConnectionDataTypeCollection : List<PubSubConnectionDataType>, ICloneable
    #else
    public partial class PubSubConnectionDataTypeCollection : List<PubSubConnectionDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public PubSubConnectionDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public PubSubConnectionDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public PubSubConnectionDataTypeCollection(IEnumerable<PubSubConnectionDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator PubSubConnectionDataTypeCollection(PubSubConnectionDataType[] values)
        {
            if (values != null)
            {
                return new PubSubConnectionDataTypeCollection(values);
            }

            return new PubSubConnectionDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator PubSubConnectionDataType[](PubSubConnectionDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (PubSubConnectionDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubConnectionDataTypeCollection clone = new PubSubConnectionDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PubSubConnectionDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ConnectionTransportDataType Class
    #if (!OPCUA_EXCLUDE_ConnectionTransportDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ConnectionTransportDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ConnectionTransportDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ConnectionTransportDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ConnectionTransportDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ConnectionTransportDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ConnectionTransportDataType value = encodeable as ConnectionTransportDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ConnectionTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ConnectionTransportDataType clone = (ConnectionTransportDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region ConnectionTransportDataTypeCollection Class
    /// <summary>
    /// A collection of ConnectionTransportDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfConnectionTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ConnectionTransportDataType")]
    #if !NET_STANDARD
    public partial class ConnectionTransportDataTypeCollection : List<ConnectionTransportDataType>, ICloneable
    #else
    public partial class ConnectionTransportDataTypeCollection : List<ConnectionTransportDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public ConnectionTransportDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public ConnectionTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public ConnectionTransportDataTypeCollection(IEnumerable<ConnectionTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator ConnectionTransportDataTypeCollection(ConnectionTransportDataType[] values)
        {
            if (values != null)
            {
                return new ConnectionTransportDataTypeCollection(values);
            }

            return new ConnectionTransportDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator ConnectionTransportDataType[](ConnectionTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (ConnectionTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ConnectionTransportDataTypeCollection clone = new ConnectionTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ConnectionTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region NetworkAddressDataType Class
    #if (!OPCUA_EXCLUDE_NetworkAddressDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class NetworkAddressDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public NetworkAddressDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_networkInterface = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "NetworkInterface", IsRequired = false, Order = 1)]
        public string NetworkInterface
        {
            get { return m_networkInterface;  }
            set { m_networkInterface = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.NetworkAddressDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.NetworkAddressDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.NetworkAddressDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("NetworkInterface", NetworkInterface);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            NetworkInterface = decoder.ReadString("NetworkInterface");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            NetworkAddressDataType value = encodeable as NetworkAddressDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_networkInterface, value.m_networkInterface)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (NetworkAddressDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            NetworkAddressDataType clone = (NetworkAddressDataType)base.MemberwiseClone();

            clone.m_networkInterface = (string)Utils.Clone(this.m_networkInterface);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_networkInterface;
        #endregion
    }

    #region NetworkAddressDataTypeCollection Class
    /// <summary>
    /// A collection of NetworkAddressDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfNetworkAddressDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "NetworkAddressDataType")]
    #if !NET_STANDARD
    public partial class NetworkAddressDataTypeCollection : List<NetworkAddressDataType>, ICloneable
    #else
    public partial class NetworkAddressDataTypeCollection : List<NetworkAddressDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public NetworkAddressDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public NetworkAddressDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public NetworkAddressDataTypeCollection(IEnumerable<NetworkAddressDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator NetworkAddressDataTypeCollection(NetworkAddressDataType[] values)
        {
            if (values != null)
            {
                return new NetworkAddressDataTypeCollection(values);
            }

            return new NetworkAddressDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator NetworkAddressDataType[](NetworkAddressDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (NetworkAddressDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            NetworkAddressDataTypeCollection clone = new NetworkAddressDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((NetworkAddressDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region NetworkAddressUrlDataType Class
    #if (!OPCUA_EXCLUDE_NetworkAddressUrlDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class NetworkAddressUrlDataType : NetworkAddressDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public NetworkAddressUrlDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_url = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Url", IsRequired = false, Order = 1)]
        public string Url
        {
            get { return m_url;  }
            set { m_url = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.NetworkAddressUrlDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.NetworkAddressUrlDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.NetworkAddressUrlDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Url", Url);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Url = decoder.ReadString("Url");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            NetworkAddressUrlDataType value = encodeable as NetworkAddressUrlDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_url, value.m_url)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (NetworkAddressUrlDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            NetworkAddressUrlDataType clone = (NetworkAddressUrlDataType)base.MemberwiseClone();

            clone.m_url = (string)Utils.Clone(this.m_url);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_url;
        #endregion
    }

    #region NetworkAddressUrlDataTypeCollection Class
    /// <summary>
    /// A collection of NetworkAddressUrlDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfNetworkAddressUrlDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "NetworkAddressUrlDataType")]
    #if !NET_STANDARD
    public partial class NetworkAddressUrlDataTypeCollection : List<NetworkAddressUrlDataType>, ICloneable
    #else
    public partial class NetworkAddressUrlDataTypeCollection : List<NetworkAddressUrlDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public NetworkAddressUrlDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public NetworkAddressUrlDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public NetworkAddressUrlDataTypeCollection(IEnumerable<NetworkAddressUrlDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator NetworkAddressUrlDataTypeCollection(NetworkAddressUrlDataType[] values)
        {
            if (values != null)
            {
                return new NetworkAddressUrlDataTypeCollection(values);
            }

            return new NetworkAddressUrlDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator NetworkAddressUrlDataType[](NetworkAddressUrlDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (NetworkAddressUrlDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            NetworkAddressUrlDataTypeCollection clone = new NetworkAddressUrlDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((NetworkAddressUrlDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ReaderGroupDataType Class
    #if (!OPCUA_EXCLUDE_ReaderGroupDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ReaderGroupDataType : PubSubGroupDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ReaderGroupDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_transportSettings = null;
            m_messageSettings = null;
            m_dataSetReaders = new DataSetReaderDataTypeCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "TransportSettings", IsRequired = false, Order = 1)]
        public ExtensionObject TransportSettings
        {
            get { return m_transportSettings;  }
            set { m_transportSettings = value; }
        }

        /// <remarks />
        [DataMember(Name = "MessageSettings", IsRequired = false, Order = 2)]
        public ExtensionObject MessageSettings
        {
            get { return m_messageSettings;  }
            set { m_messageSettings = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "DataSetReaders", IsRequired = false, Order = 3)]
        public DataSetReaderDataTypeCollection DataSetReaders
        {
            get
            {
                return m_dataSetReaders;
            }

            set
            {
                m_dataSetReaders = value;

                if (value == null)
                {
                    m_dataSetReaders = new DataSetReaderDataTypeCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ReaderGroupDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ReaderGroupDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ReaderGroupDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteExtensionObject("TransportSettings", TransportSettings);
            encoder.WriteExtensionObject("MessageSettings", MessageSettings);
            encoder.WriteEncodeableArray("DataSetReaders", DataSetReaders.ToArray(), typeof(DataSetReaderDataType));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            TransportSettings = decoder.ReadExtensionObject("TransportSettings");
            MessageSettings = decoder.ReadExtensionObject("MessageSettings");
            DataSetReaders = (DataSetReaderDataTypeCollection)decoder.ReadEncodeableArray("DataSetReaders", typeof(DataSetReaderDataType));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ReaderGroupDataType value = encodeable as ReaderGroupDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_transportSettings, value.m_transportSettings)) return false;
            if (!Utils.IsEqual(m_messageSettings, value.m_messageSettings)) return false;
            if (!Utils.IsEqual(m_dataSetReaders, value.m_dataSetReaders)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ReaderGroupDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReaderGroupDataType clone = (ReaderGroupDataType)base.MemberwiseClone();

            clone.m_transportSettings = (ExtensionObject)Utils.Clone(this.m_transportSettings);
            clone.m_messageSettings = (ExtensionObject)Utils.Clone(this.m_messageSettings);
            clone.m_dataSetReaders = (DataSetReaderDataTypeCollection)Utils.Clone(this.m_dataSetReaders);

            return clone;
        }
        #endregion

        #region Private Fields
        private ExtensionObject m_transportSettings;
        private ExtensionObject m_messageSettings;
        private DataSetReaderDataTypeCollection m_dataSetReaders;
        #endregion
    }

    #region ReaderGroupDataTypeCollection Class
    /// <summary>
    /// A collection of ReaderGroupDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfReaderGroupDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ReaderGroupDataType")]
    #if !NET_STANDARD
    public partial class ReaderGroupDataTypeCollection : List<ReaderGroupDataType>, ICloneable
    #else
    public partial class ReaderGroupDataTypeCollection : List<ReaderGroupDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public ReaderGroupDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public ReaderGroupDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public ReaderGroupDataTypeCollection(IEnumerable<ReaderGroupDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator ReaderGroupDataTypeCollection(ReaderGroupDataType[] values)
        {
            if (values != null)
            {
                return new ReaderGroupDataTypeCollection(values);
            }

            return new ReaderGroupDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator ReaderGroupDataType[](ReaderGroupDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (ReaderGroupDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReaderGroupDataTypeCollection clone = new ReaderGroupDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ReaderGroupDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ReaderGroupTransportDataType Class
    #if (!OPCUA_EXCLUDE_ReaderGroupTransportDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ReaderGroupTransportDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ReaderGroupTransportDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ReaderGroupTransportDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ReaderGroupTransportDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ReaderGroupTransportDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ReaderGroupTransportDataType value = encodeable as ReaderGroupTransportDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ReaderGroupTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReaderGroupTransportDataType clone = (ReaderGroupTransportDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region ReaderGroupTransportDataTypeCollection Class
    /// <summary>
    /// A collection of ReaderGroupTransportDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfReaderGroupTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ReaderGroupTransportDataType")]
    #if !NET_STANDARD
    public partial class ReaderGroupTransportDataTypeCollection : List<ReaderGroupTransportDataType>, ICloneable
    #else
    public partial class ReaderGroupTransportDataTypeCollection : List<ReaderGroupTransportDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public ReaderGroupTransportDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public ReaderGroupTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public ReaderGroupTransportDataTypeCollection(IEnumerable<ReaderGroupTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator ReaderGroupTransportDataTypeCollection(ReaderGroupTransportDataType[] values)
        {
            if (values != null)
            {
                return new ReaderGroupTransportDataTypeCollection(values);
            }

            return new ReaderGroupTransportDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator ReaderGroupTransportDataType[](ReaderGroupTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (ReaderGroupTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReaderGroupTransportDataTypeCollection clone = new ReaderGroupTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ReaderGroupTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ReaderGroupMessageDataType Class
    #if (!OPCUA_EXCLUDE_ReaderGroupMessageDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ReaderGroupMessageDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ReaderGroupMessageDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ReaderGroupMessageDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ReaderGroupMessageDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ReaderGroupMessageDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ReaderGroupMessageDataType value = encodeable as ReaderGroupMessageDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ReaderGroupMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReaderGroupMessageDataType clone = (ReaderGroupMessageDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region ReaderGroupMessageDataTypeCollection Class
    /// <summary>
    /// A collection of ReaderGroupMessageDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfReaderGroupMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ReaderGroupMessageDataType")]
    #if !NET_STANDARD
    public partial class ReaderGroupMessageDataTypeCollection : List<ReaderGroupMessageDataType>, ICloneable
    #else
    public partial class ReaderGroupMessageDataTypeCollection : List<ReaderGroupMessageDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public ReaderGroupMessageDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public ReaderGroupMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public ReaderGroupMessageDataTypeCollection(IEnumerable<ReaderGroupMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator ReaderGroupMessageDataTypeCollection(ReaderGroupMessageDataType[] values)
        {
            if (values != null)
            {
                return new ReaderGroupMessageDataTypeCollection(values);
            }

            return new ReaderGroupMessageDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator ReaderGroupMessageDataType[](ReaderGroupMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (ReaderGroupMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReaderGroupMessageDataTypeCollection clone = new ReaderGroupMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ReaderGroupMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetReaderDataType Class
    #if (!OPCUA_EXCLUDE_DataSetReaderDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataSetReaderDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public DataSetReaderDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_name = null;
            m_enabled = true;
            m_publisherId = Variant.Null;
            m_writerGroupId = (ushort)0;
            m_dataSetWriterId = (ushort)0;
            m_dataSetMetaData = new DataSetMetaDataType();
            m_dataSetFieldContentMask = 0;
            m_messageReceiveTimeout = (double)0;
            m_keyFrameCount = (uint)0;
            m_headerLayoutUri = null;
            m_securityMode = MessageSecurityMode.Invalid;
            m_securityGroupId = null;
            m_securityKeyServices = new EndpointDescriptionCollection();
            m_dataSetReaderProperties = new KeyValuePairCollection();
            m_transportSettings = null;
            m_messageSettings = null;
            m_subscribedDataSet = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "Enabled", IsRequired = false, Order = 2)]
        public bool Enabled
        {
            get { return m_enabled;  }
            set { m_enabled = value; }
        }

        /// <remarks />
        [DataMember(Name = "PublisherId", IsRequired = false, Order = 3)]
        public Variant PublisherId
        {
            get { return m_publisherId;  }
            set { m_publisherId = value; }
        }

        /// <remarks />
        [DataMember(Name = "WriterGroupId", IsRequired = false, Order = 4)]
        public ushort WriterGroupId
        {
            get { return m_writerGroupId;  }
            set { m_writerGroupId = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetWriterId", IsRequired = false, Order = 5)]
        public ushort DataSetWriterId
        {
            get { return m_dataSetWriterId;  }
            set { m_dataSetWriterId = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "DataSetMetaData", IsRequired = false, Order = 6)]
        public DataSetMetaDataType DataSetMetaData
        {
            get
            {
                return m_dataSetMetaData;
            }

            set
            {
                m_dataSetMetaData = value;

                if (value == null)
                {
                    m_dataSetMetaData = new DataSetMetaDataType();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "DataSetFieldContentMask", IsRequired = false, Order = 7)]
        public uint DataSetFieldContentMask
        {
            get { return m_dataSetFieldContentMask;  }
            set { m_dataSetFieldContentMask = value; }
        }

        /// <remarks />
        [DataMember(Name = "MessageReceiveTimeout", IsRequired = false, Order = 8)]
        public double MessageReceiveTimeout
        {
            get { return m_messageReceiveTimeout;  }
            set { m_messageReceiveTimeout = value; }
        }

        /// <remarks />
        [DataMember(Name = "KeyFrameCount", IsRequired = false, Order = 9)]
        public uint KeyFrameCount
        {
            get { return m_keyFrameCount;  }
            set { m_keyFrameCount = value; }
        }

        /// <remarks />
        [DataMember(Name = "HeaderLayoutUri", IsRequired = false, Order = 10)]
        public string HeaderLayoutUri
        {
            get { return m_headerLayoutUri;  }
            set { m_headerLayoutUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityMode", IsRequired = false, Order = 11)]
        public MessageSecurityMode SecurityMode
        {
            get { return m_securityMode;  }
            set { m_securityMode = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityGroupId", IsRequired = false, Order = 12)]
        public string SecurityGroupId
        {
            get { return m_securityGroupId;  }
            set { m_securityGroupId = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "SecurityKeyServices", IsRequired = false, Order = 13)]
        public EndpointDescriptionCollection SecurityKeyServices
        {
            get
            {
                return m_securityKeyServices;
            }

            set
            {
                m_securityKeyServices = value;

                if (value == null)
                {
                    m_securityKeyServices = new EndpointDescriptionCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "DataSetReaderProperties", IsRequired = false, Order = 14)]
        public KeyValuePairCollection DataSetReaderProperties
        {
            get
            {
                return m_dataSetReaderProperties;
            }

            set
            {
                m_dataSetReaderProperties = value;

                if (value == null)
                {
                    m_dataSetReaderProperties = new KeyValuePairCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "TransportSettings", IsRequired = false, Order = 15)]
        public ExtensionObject TransportSettings
        {
            get { return m_transportSettings;  }
            set { m_transportSettings = value; }
        }

        /// <remarks />
        [DataMember(Name = "MessageSettings", IsRequired = false, Order = 16)]
        public ExtensionObject MessageSettings
        {
            get { return m_messageSettings;  }
            set { m_messageSettings = value; }
        }

        /// <remarks />
        [DataMember(Name = "SubscribedDataSet", IsRequired = false, Order = 17)]
        public ExtensionObject SubscribedDataSet
        {
            get { return m_subscribedDataSet;  }
            set { m_subscribedDataSet = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.DataSetReaderDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.DataSetReaderDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.DataSetReaderDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteBoolean("Enabled", Enabled);
            encoder.WriteVariant("PublisherId", PublisherId);
            encoder.WriteUInt16("WriterGroupId", WriterGroupId);
            encoder.WriteUInt16("DataSetWriterId", DataSetWriterId);
            encoder.WriteEncodeable("DataSetMetaData", DataSetMetaData, typeof(DataSetMetaDataType));
            encoder.WriteUInt32("DataSetFieldContentMask", DataSetFieldContentMask);
            encoder.WriteDouble("MessageReceiveTimeout", MessageReceiveTimeout);
            encoder.WriteUInt32("KeyFrameCount", KeyFrameCount);
            encoder.WriteString("HeaderLayoutUri", HeaderLayoutUri);
            encoder.WriteEnumerated("SecurityMode", SecurityMode);
            encoder.WriteString("SecurityGroupId", SecurityGroupId);
            encoder.WriteEncodeableArray("SecurityKeyServices", SecurityKeyServices.ToArray(), typeof(EndpointDescription));
            encoder.WriteEncodeableArray("DataSetReaderProperties", DataSetReaderProperties.ToArray(), typeof(KeyValuePair));
            encoder.WriteExtensionObject("TransportSettings", TransportSettings);
            encoder.WriteExtensionObject("MessageSettings", MessageSettings);
            encoder.WriteExtensionObject("SubscribedDataSet", SubscribedDataSet);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            Enabled = decoder.ReadBoolean("Enabled");
            PublisherId = decoder.ReadVariant("PublisherId");
            WriterGroupId = decoder.ReadUInt16("WriterGroupId");
            DataSetWriterId = decoder.ReadUInt16("DataSetWriterId");
            DataSetMetaData = (DataSetMetaDataType)decoder.ReadEncodeable("DataSetMetaData", typeof(DataSetMetaDataType));
            DataSetFieldContentMask = decoder.ReadUInt32("DataSetFieldContentMask");
            MessageReceiveTimeout = decoder.ReadDouble("MessageReceiveTimeout");
            KeyFrameCount = decoder.ReadUInt32("KeyFrameCount");
            HeaderLayoutUri = decoder.ReadString("HeaderLayoutUri");
            SecurityMode = (MessageSecurityMode)decoder.ReadEnumerated("SecurityMode", typeof(MessageSecurityMode));
            SecurityGroupId = decoder.ReadString("SecurityGroupId");
            SecurityKeyServices = (EndpointDescriptionCollection)decoder.ReadEncodeableArray("SecurityKeyServices", typeof(EndpointDescription));
            DataSetReaderProperties = (KeyValuePairCollection)decoder.ReadEncodeableArray("DataSetReaderProperties", typeof(KeyValuePair));
            TransportSettings = decoder.ReadExtensionObject("TransportSettings");
            MessageSettings = decoder.ReadExtensionObject("MessageSettings");
            SubscribedDataSet = decoder.ReadExtensionObject("SubscribedDataSet");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataSetReaderDataType value = encodeable as DataSetReaderDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_enabled, value.m_enabled)) return false;
            if (!Utils.IsEqual(m_publisherId, value.m_publisherId)) return false;
            if (!Utils.IsEqual(m_writerGroupId, value.m_writerGroupId)) return false;
            if (!Utils.IsEqual(m_dataSetWriterId, value.m_dataSetWriterId)) return false;
            if (!Utils.IsEqual(m_dataSetMetaData, value.m_dataSetMetaData)) return false;
            if (!Utils.IsEqual(m_dataSetFieldContentMask, value.m_dataSetFieldContentMask)) return false;
            if (!Utils.IsEqual(m_messageReceiveTimeout, value.m_messageReceiveTimeout)) return false;
            if (!Utils.IsEqual(m_keyFrameCount, value.m_keyFrameCount)) return false;
            if (!Utils.IsEqual(m_headerLayoutUri, value.m_headerLayoutUri)) return false;
            if (!Utils.IsEqual(m_securityMode, value.m_securityMode)) return false;
            if (!Utils.IsEqual(m_securityGroupId, value.m_securityGroupId)) return false;
            if (!Utils.IsEqual(m_securityKeyServices, value.m_securityKeyServices)) return false;
            if (!Utils.IsEqual(m_dataSetReaderProperties, value.m_dataSetReaderProperties)) return false;
            if (!Utils.IsEqual(m_transportSettings, value.m_transportSettings)) return false;
            if (!Utils.IsEqual(m_messageSettings, value.m_messageSettings)) return false;
            if (!Utils.IsEqual(m_subscribedDataSet, value.m_subscribedDataSet)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DataSetReaderDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetReaderDataType clone = (DataSetReaderDataType)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_enabled = (bool)Utils.Clone(this.m_enabled);
            clone.m_publisherId = (Variant)Utils.Clone(this.m_publisherId);
            clone.m_writerGroupId = (ushort)Utils.Clone(this.m_writerGroupId);
            clone.m_dataSetWriterId = (ushort)Utils.Clone(this.m_dataSetWriterId);
            clone.m_dataSetMetaData = (DataSetMetaDataType)Utils.Clone(this.m_dataSetMetaData);
            clone.m_dataSetFieldContentMask = (uint)Utils.Clone(this.m_dataSetFieldContentMask);
            clone.m_messageReceiveTimeout = (double)Utils.Clone(this.m_messageReceiveTimeout);
            clone.m_keyFrameCount = (uint)Utils.Clone(this.m_keyFrameCount);
            clone.m_headerLayoutUri = (string)Utils.Clone(this.m_headerLayoutUri);
            clone.m_securityMode = (MessageSecurityMode)Utils.Clone(this.m_securityMode);
            clone.m_securityGroupId = (string)Utils.Clone(this.m_securityGroupId);
            clone.m_securityKeyServices = (EndpointDescriptionCollection)Utils.Clone(this.m_securityKeyServices);
            clone.m_dataSetReaderProperties = (KeyValuePairCollection)Utils.Clone(this.m_dataSetReaderProperties);
            clone.m_transportSettings = (ExtensionObject)Utils.Clone(this.m_transportSettings);
            clone.m_messageSettings = (ExtensionObject)Utils.Clone(this.m_messageSettings);
            clone.m_subscribedDataSet = (ExtensionObject)Utils.Clone(this.m_subscribedDataSet);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private bool m_enabled;
        private Variant m_publisherId;
        private ushort m_writerGroupId;
        private ushort m_dataSetWriterId;
        private DataSetMetaDataType m_dataSetMetaData;
        private uint m_dataSetFieldContentMask;
        private double m_messageReceiveTimeout;
        private uint m_keyFrameCount;
        private string m_headerLayoutUri;
        private MessageSecurityMode m_securityMode;
        private string m_securityGroupId;
        private EndpointDescriptionCollection m_securityKeyServices;
        private KeyValuePairCollection m_dataSetReaderProperties;
        private ExtensionObject m_transportSettings;
        private ExtensionObject m_messageSettings;
        private ExtensionObject m_subscribedDataSet;
        #endregion
    }

    #region DataSetReaderDataTypeCollection Class
    /// <summary>
    /// A collection of DataSetReaderDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataSetReaderDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataSetReaderDataType")]
    #if !NET_STANDARD
    public partial class DataSetReaderDataTypeCollection : List<DataSetReaderDataType>, ICloneable
    #else
    public partial class DataSetReaderDataTypeCollection : List<DataSetReaderDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public DataSetReaderDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public DataSetReaderDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public DataSetReaderDataTypeCollection(IEnumerable<DataSetReaderDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator DataSetReaderDataTypeCollection(DataSetReaderDataType[] values)
        {
            if (values != null)
            {
                return new DataSetReaderDataTypeCollection(values);
            }

            return new DataSetReaderDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator DataSetReaderDataType[](DataSetReaderDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (DataSetReaderDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetReaderDataTypeCollection clone = new DataSetReaderDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataSetReaderDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetReaderTransportDataType Class
    #if (!OPCUA_EXCLUDE_DataSetReaderTransportDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataSetReaderTransportDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public DataSetReaderTransportDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.DataSetReaderTransportDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.DataSetReaderTransportDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.DataSetReaderTransportDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataSetReaderTransportDataType value = encodeable as DataSetReaderTransportDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DataSetReaderTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetReaderTransportDataType clone = (DataSetReaderTransportDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region DataSetReaderTransportDataTypeCollection Class
    /// <summary>
    /// A collection of DataSetReaderTransportDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataSetReaderTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataSetReaderTransportDataType")]
    #if !NET_STANDARD
    public partial class DataSetReaderTransportDataTypeCollection : List<DataSetReaderTransportDataType>, ICloneable
    #else
    public partial class DataSetReaderTransportDataTypeCollection : List<DataSetReaderTransportDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public DataSetReaderTransportDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public DataSetReaderTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public DataSetReaderTransportDataTypeCollection(IEnumerable<DataSetReaderTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator DataSetReaderTransportDataTypeCollection(DataSetReaderTransportDataType[] values)
        {
            if (values != null)
            {
                return new DataSetReaderTransportDataTypeCollection(values);
            }

            return new DataSetReaderTransportDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator DataSetReaderTransportDataType[](DataSetReaderTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (DataSetReaderTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetReaderTransportDataTypeCollection clone = new DataSetReaderTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataSetReaderTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetReaderMessageDataType Class
    #if (!OPCUA_EXCLUDE_DataSetReaderMessageDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataSetReaderMessageDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public DataSetReaderMessageDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.DataSetReaderMessageDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.DataSetReaderMessageDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.DataSetReaderMessageDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataSetReaderMessageDataType value = encodeable as DataSetReaderMessageDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DataSetReaderMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetReaderMessageDataType clone = (DataSetReaderMessageDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region DataSetReaderMessageDataTypeCollection Class
    /// <summary>
    /// A collection of DataSetReaderMessageDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataSetReaderMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataSetReaderMessageDataType")]
    #if !NET_STANDARD
    public partial class DataSetReaderMessageDataTypeCollection : List<DataSetReaderMessageDataType>, ICloneable
    #else
    public partial class DataSetReaderMessageDataTypeCollection : List<DataSetReaderMessageDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public DataSetReaderMessageDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public DataSetReaderMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public DataSetReaderMessageDataTypeCollection(IEnumerable<DataSetReaderMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator DataSetReaderMessageDataTypeCollection(DataSetReaderMessageDataType[] values)
        {
            if (values != null)
            {
                return new DataSetReaderMessageDataTypeCollection(values);
            }

            return new DataSetReaderMessageDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator DataSetReaderMessageDataType[](DataSetReaderMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (DataSetReaderMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetReaderMessageDataTypeCollection clone = new DataSetReaderMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataSetReaderMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region SubscribedDataSetDataType Class
    #if (!OPCUA_EXCLUDE_SubscribedDataSetDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class SubscribedDataSetDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public SubscribedDataSetDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.SubscribedDataSetDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.SubscribedDataSetDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.SubscribedDataSetDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            SubscribedDataSetDataType value = encodeable as SubscribedDataSetDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (SubscribedDataSetDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SubscribedDataSetDataType clone = (SubscribedDataSetDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region SubscribedDataSetDataTypeCollection Class
    /// <summary>
    /// A collection of SubscribedDataSetDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfSubscribedDataSetDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "SubscribedDataSetDataType")]
    #if !NET_STANDARD
    public partial class SubscribedDataSetDataTypeCollection : List<SubscribedDataSetDataType>, ICloneable
    #else
    public partial class SubscribedDataSetDataTypeCollection : List<SubscribedDataSetDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public SubscribedDataSetDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public SubscribedDataSetDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public SubscribedDataSetDataTypeCollection(IEnumerable<SubscribedDataSetDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator SubscribedDataSetDataTypeCollection(SubscribedDataSetDataType[] values)
        {
            if (values != null)
            {
                return new SubscribedDataSetDataTypeCollection(values);
            }

            return new SubscribedDataSetDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator SubscribedDataSetDataType[](SubscribedDataSetDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (SubscribedDataSetDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SubscribedDataSetDataTypeCollection clone = new SubscribedDataSetDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((SubscribedDataSetDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region TargetVariablesDataType Class
    #if (!OPCUA_EXCLUDE_TargetVariablesDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class TargetVariablesDataType : SubscribedDataSetDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public TargetVariablesDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_targetVariables = new FieldTargetDataTypeCollection();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "TargetVariables", IsRequired = false, Order = 1)]
        public FieldTargetDataTypeCollection TargetVariables
        {
            get
            {
                return m_targetVariables;
            }

            set
            {
                m_targetVariables = value;

                if (value == null)
                {
                    m_targetVariables = new FieldTargetDataTypeCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.TargetVariablesDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.TargetVariablesDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.TargetVariablesDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeableArray("TargetVariables", TargetVariables.ToArray(), typeof(FieldTargetDataType));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            TargetVariables = (FieldTargetDataTypeCollection)decoder.ReadEncodeableArray("TargetVariables", typeof(FieldTargetDataType));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            TargetVariablesDataType value = encodeable as TargetVariablesDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_targetVariables, value.m_targetVariables)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (TargetVariablesDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TargetVariablesDataType clone = (TargetVariablesDataType)base.MemberwiseClone();

            clone.m_targetVariables = (FieldTargetDataTypeCollection)Utils.Clone(this.m_targetVariables);

            return clone;
        }
        #endregion

        #region Private Fields
        private FieldTargetDataTypeCollection m_targetVariables;
        #endregion
    }

    #region TargetVariablesDataTypeCollection Class
    /// <summary>
    /// A collection of TargetVariablesDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfTargetVariablesDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "TargetVariablesDataType")]
    #if !NET_STANDARD
    public partial class TargetVariablesDataTypeCollection : List<TargetVariablesDataType>, ICloneable
    #else
    public partial class TargetVariablesDataTypeCollection : List<TargetVariablesDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public TargetVariablesDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public TargetVariablesDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public TargetVariablesDataTypeCollection(IEnumerable<TargetVariablesDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator TargetVariablesDataTypeCollection(TargetVariablesDataType[] values)
        {
            if (values != null)
            {
                return new TargetVariablesDataTypeCollection(values);
            }

            return new TargetVariablesDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator TargetVariablesDataType[](TargetVariablesDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (TargetVariablesDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TargetVariablesDataTypeCollection clone = new TargetVariablesDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((TargetVariablesDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region FieldTargetDataType Class
    #if (!OPCUA_EXCLUDE_FieldTargetDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class FieldTargetDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public FieldTargetDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_dataSetFieldId = Uuid.Empty;
            m_receiverIndexRange = null;
            m_targetNodeId = null;
            m_attributeId = (uint)0;
            m_writeIndexRange = null;
            m_overrideValueHandling = OverrideValueHandling.Disabled;
            m_overrideValue = Variant.Null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "DataSetFieldId", IsRequired = false, Order = 1)]
        public Uuid DataSetFieldId
        {
            get { return m_dataSetFieldId;  }
            set { m_dataSetFieldId = value; }
        }

        /// <remarks />
        [DataMember(Name = "ReceiverIndexRange", IsRequired = false, Order = 2)]
        public string ReceiverIndexRange
        {
            get { return m_receiverIndexRange;  }
            set { m_receiverIndexRange = value; }
        }

        /// <remarks />
        [DataMember(Name = "TargetNodeId", IsRequired = false, Order = 3)]
        public NodeId TargetNodeId
        {
            get { return m_targetNodeId;  }
            set { m_targetNodeId = value; }
        }

        /// <remarks />
        [DataMember(Name = "AttributeId", IsRequired = false, Order = 4)]
        public uint AttributeId
        {
            get { return m_attributeId;  }
            set { m_attributeId = value; }
        }

        /// <remarks />
        [DataMember(Name = "WriteIndexRange", IsRequired = false, Order = 5)]
        public string WriteIndexRange
        {
            get { return m_writeIndexRange;  }
            set { m_writeIndexRange = value; }
        }

        /// <remarks />
        [DataMember(Name = "OverrideValueHandling", IsRequired = false, Order = 6)]
        public OverrideValueHandling OverrideValueHandling
        {
            get { return m_overrideValueHandling;  }
            set { m_overrideValueHandling = value; }
        }

        /// <remarks />
        [DataMember(Name = "OverrideValue", IsRequired = false, Order = 7)]
        public Variant OverrideValue
        {
            get { return m_overrideValue;  }
            set { m_overrideValue = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.FieldTargetDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.FieldTargetDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.FieldTargetDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteGuid("DataSetFieldId", DataSetFieldId);
            encoder.WriteString("ReceiverIndexRange", ReceiverIndexRange);
            encoder.WriteNodeId("TargetNodeId", TargetNodeId);
            encoder.WriteUInt32("AttributeId", AttributeId);
            encoder.WriteString("WriteIndexRange", WriteIndexRange);
            encoder.WriteEnumerated("OverrideValueHandling", OverrideValueHandling);
            encoder.WriteVariant("OverrideValue", OverrideValue);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            DataSetFieldId = decoder.ReadGuid("DataSetFieldId");
            ReceiverIndexRange = decoder.ReadString("ReceiverIndexRange");
            TargetNodeId = decoder.ReadNodeId("TargetNodeId");
            AttributeId = decoder.ReadUInt32("AttributeId");
            WriteIndexRange = decoder.ReadString("WriteIndexRange");
            OverrideValueHandling = (OverrideValueHandling)decoder.ReadEnumerated("OverrideValueHandling", typeof(OverrideValueHandling));
            OverrideValue = decoder.ReadVariant("OverrideValue");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            FieldTargetDataType value = encodeable as FieldTargetDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_dataSetFieldId, value.m_dataSetFieldId)) return false;
            if (!Utils.IsEqual(m_receiverIndexRange, value.m_receiverIndexRange)) return false;
            if (!Utils.IsEqual(m_targetNodeId, value.m_targetNodeId)) return false;
            if (!Utils.IsEqual(m_attributeId, value.m_attributeId)) return false;
            if (!Utils.IsEqual(m_writeIndexRange, value.m_writeIndexRange)) return false;
            if (!Utils.IsEqual(m_overrideValueHandling, value.m_overrideValueHandling)) return false;
            if (!Utils.IsEqual(m_overrideValue, value.m_overrideValue)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (FieldTargetDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            FieldTargetDataType clone = (FieldTargetDataType)base.MemberwiseClone();

            clone.m_dataSetFieldId = (Uuid)Utils.Clone(this.m_dataSetFieldId);
            clone.m_receiverIndexRange = (string)Utils.Clone(this.m_receiverIndexRange);
            clone.m_targetNodeId = (NodeId)Utils.Clone(this.m_targetNodeId);
            clone.m_attributeId = (uint)Utils.Clone(this.m_attributeId);
            clone.m_writeIndexRange = (string)Utils.Clone(this.m_writeIndexRange);
            clone.m_overrideValueHandling = (OverrideValueHandling)Utils.Clone(this.m_overrideValueHandling);
            clone.m_overrideValue = (Variant)Utils.Clone(this.m_overrideValue);

            return clone;
        }
        #endregion

        #region Private Fields
        private Uuid m_dataSetFieldId;
        private string m_receiverIndexRange;
        private NodeId m_targetNodeId;
        private uint m_attributeId;
        private string m_writeIndexRange;
        private OverrideValueHandling m_overrideValueHandling;
        private Variant m_overrideValue;
        #endregion
    }

    #region FieldTargetDataTypeCollection Class
    /// <summary>
    /// A collection of FieldTargetDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfFieldTargetDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "FieldTargetDataType")]
    #if !NET_STANDARD
    public partial class FieldTargetDataTypeCollection : List<FieldTargetDataType>, ICloneable
    #else
    public partial class FieldTargetDataTypeCollection : List<FieldTargetDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public FieldTargetDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public FieldTargetDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public FieldTargetDataTypeCollection(IEnumerable<FieldTargetDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator FieldTargetDataTypeCollection(FieldTargetDataType[] values)
        {
            if (values != null)
            {
                return new FieldTargetDataTypeCollection(values);
            }

            return new FieldTargetDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator FieldTargetDataType[](FieldTargetDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (FieldTargetDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            FieldTargetDataTypeCollection clone = new FieldTargetDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((FieldTargetDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region OverrideValueHandling Enumeration
    #if (!OPCUA_EXCLUDE_OverrideValueHandling)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum OverrideValueHandling
    {
        /// <remarks />
        [EnumMember(Value = "Disabled_0")]
        Disabled = 0,

        /// <remarks />
        [EnumMember(Value = "LastUsableValue_1")]
        LastUsableValue = 1,

        /// <remarks />
        [EnumMember(Value = "OverrideValue_2")]
        OverrideValue = 2,
    }

    #region OverrideValueHandlingCollection Class
    /// <summary>
    /// A collection of OverrideValueHandling objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfOverrideValueHandling", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "OverrideValueHandling")]
    #if !NET_STANDARD
    public partial class OverrideValueHandlingCollection : List<OverrideValueHandling>, ICloneable
    #else
    public partial class OverrideValueHandlingCollection : List<OverrideValueHandling>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public OverrideValueHandlingCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public OverrideValueHandlingCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public OverrideValueHandlingCollection(IEnumerable<OverrideValueHandling> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator OverrideValueHandlingCollection(OverrideValueHandling[] values)
        {
            if (values != null)
            {
                return new OverrideValueHandlingCollection(values);
            }

            return new OverrideValueHandlingCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator OverrideValueHandling[](OverrideValueHandlingCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (OverrideValueHandlingCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            OverrideValueHandlingCollection clone = new OverrideValueHandlingCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((OverrideValueHandling)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region SubscribedDataSetMirrorDataType Class
    #if (!OPCUA_EXCLUDE_SubscribedDataSetMirrorDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class SubscribedDataSetMirrorDataType : SubscribedDataSetDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public SubscribedDataSetMirrorDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_parentNodeName = null;
            m_rolePermissions = new RolePermissionTypeCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ParentNodeName", IsRequired = false, Order = 1)]
        public string ParentNodeName
        {
            get { return m_parentNodeName;  }
            set { m_parentNodeName = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "RolePermissions", IsRequired = false, Order = 2)]
        public RolePermissionTypeCollection RolePermissions
        {
            get
            {
                return m_rolePermissions;
            }

            set
            {
                m_rolePermissions = value;

                if (value == null)
                {
                    m_rolePermissions = new RolePermissionTypeCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.SubscribedDataSetMirrorDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.SubscribedDataSetMirrorDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.SubscribedDataSetMirrorDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("ParentNodeName", ParentNodeName);
            encoder.WriteEncodeableArray("RolePermissions", RolePermissions.ToArray(), typeof(RolePermissionType));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ParentNodeName = decoder.ReadString("ParentNodeName");
            RolePermissions = (RolePermissionTypeCollection)decoder.ReadEncodeableArray("RolePermissions", typeof(RolePermissionType));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            SubscribedDataSetMirrorDataType value = encodeable as SubscribedDataSetMirrorDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_parentNodeName, value.m_parentNodeName)) return false;
            if (!Utils.IsEqual(m_rolePermissions, value.m_rolePermissions)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (SubscribedDataSetMirrorDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SubscribedDataSetMirrorDataType clone = (SubscribedDataSetMirrorDataType)base.MemberwiseClone();

            clone.m_parentNodeName = (string)Utils.Clone(this.m_parentNodeName);
            clone.m_rolePermissions = (RolePermissionTypeCollection)Utils.Clone(this.m_rolePermissions);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_parentNodeName;
        private RolePermissionTypeCollection m_rolePermissions;
        #endregion
    }

    #region SubscribedDataSetMirrorDataTypeCollection Class
    /// <summary>
    /// A collection of SubscribedDataSetMirrorDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfSubscribedDataSetMirrorDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "SubscribedDataSetMirrorDataType")]
    #if !NET_STANDARD
    public partial class SubscribedDataSetMirrorDataTypeCollection : List<SubscribedDataSetMirrorDataType>, ICloneable
    #else
    public partial class SubscribedDataSetMirrorDataTypeCollection : List<SubscribedDataSetMirrorDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public SubscribedDataSetMirrorDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public SubscribedDataSetMirrorDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public SubscribedDataSetMirrorDataTypeCollection(IEnumerable<SubscribedDataSetMirrorDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator SubscribedDataSetMirrorDataTypeCollection(SubscribedDataSetMirrorDataType[] values)
        {
            if (values != null)
            {
                return new SubscribedDataSetMirrorDataTypeCollection(values);
            }

            return new SubscribedDataSetMirrorDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator SubscribedDataSetMirrorDataType[](SubscribedDataSetMirrorDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (SubscribedDataSetMirrorDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SubscribedDataSetMirrorDataTypeCollection clone = new SubscribedDataSetMirrorDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((SubscribedDataSetMirrorDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PubSubConfigurationDataType Class
    #if (!OPCUA_EXCLUDE_PubSubConfigurationDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PubSubConfigurationDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public PubSubConfigurationDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_publishedDataSets = new PublishedDataSetDataTypeCollection();
            m_connections = new PubSubConnectionDataTypeCollection();
            m_enabled = true;
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "PublishedDataSets", IsRequired = false, Order = 1)]
        public PublishedDataSetDataTypeCollection PublishedDataSets
        {
            get
            {
                return m_publishedDataSets;
            }

            set
            {
                m_publishedDataSets = value;

                if (value == null)
                {
                    m_publishedDataSets = new PublishedDataSetDataTypeCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "Connections", IsRequired = false, Order = 2)]
        public PubSubConnectionDataTypeCollection Connections
        {
            get
            {
                return m_connections;
            }

            set
            {
                m_connections = value;

                if (value == null)
                {
                    m_connections = new PubSubConnectionDataTypeCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "Enabled", IsRequired = false, Order = 3)]
        public bool Enabled
        {
            get { return m_enabled;  }
            set { m_enabled = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.PubSubConfigurationDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.PubSubConfigurationDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.PubSubConfigurationDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeableArray("PublishedDataSets", PublishedDataSets.ToArray(), typeof(PublishedDataSetDataType));
            encoder.WriteEncodeableArray("Connections", Connections.ToArray(), typeof(PubSubConnectionDataType));
            encoder.WriteBoolean("Enabled", Enabled);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            PublishedDataSets = (PublishedDataSetDataTypeCollection)decoder.ReadEncodeableArray("PublishedDataSets", typeof(PublishedDataSetDataType));
            Connections = (PubSubConnectionDataTypeCollection)decoder.ReadEncodeableArray("Connections", typeof(PubSubConnectionDataType));
            Enabled = decoder.ReadBoolean("Enabled");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PubSubConfigurationDataType value = encodeable as PubSubConfigurationDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_publishedDataSets, value.m_publishedDataSets)) return false;
            if (!Utils.IsEqual(m_connections, value.m_connections)) return false;
            if (!Utils.IsEqual(m_enabled, value.m_enabled)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (PubSubConfigurationDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubConfigurationDataType clone = (PubSubConfigurationDataType)base.MemberwiseClone();

            clone.m_publishedDataSets = (PublishedDataSetDataTypeCollection)Utils.Clone(this.m_publishedDataSets);
            clone.m_connections = (PubSubConnectionDataTypeCollection)Utils.Clone(this.m_connections);
            clone.m_enabled = (bool)Utils.Clone(this.m_enabled);

            return clone;
        }
        #endregion

        #region Private Fields
        private PublishedDataSetDataTypeCollection m_publishedDataSets;
        private PubSubConnectionDataTypeCollection m_connections;
        private bool m_enabled;
        #endregion
    }

    #region PubSubConfigurationDataTypeCollection Class
    /// <summary>
    /// A collection of PubSubConfigurationDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPubSubConfigurationDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PubSubConfigurationDataType")]
    #if !NET_STANDARD
    public partial class PubSubConfigurationDataTypeCollection : List<PubSubConfigurationDataType>, ICloneable
    #else
    public partial class PubSubConfigurationDataTypeCollection : List<PubSubConfigurationDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public PubSubConfigurationDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public PubSubConfigurationDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public PubSubConfigurationDataTypeCollection(IEnumerable<PubSubConfigurationDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator PubSubConfigurationDataTypeCollection(PubSubConfigurationDataType[] values)
        {
            if (values != null)
            {
                return new PubSubConfigurationDataTypeCollection(values);
            }

            return new PubSubConfigurationDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator PubSubConfigurationDataType[](PubSubConfigurationDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (PubSubConfigurationDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubConfigurationDataTypeCollection clone = new PubSubConfigurationDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PubSubConfigurationDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetOrderingType Enumeration
    #if (!OPCUA_EXCLUDE_DataSetOrderingType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum DataSetOrderingType
    {
        /// <remarks />
        [EnumMember(Value = "Undefined_0")]
        Undefined = 0,

        /// <remarks />
        [EnumMember(Value = "AscendingWriterId_1")]
        AscendingWriterId = 1,

        /// <remarks />
        [EnumMember(Value = "AscendingWriterIdSingle_2")]
        AscendingWriterIdSingle = 2,
    }

    #region DataSetOrderingTypeCollection Class
    /// <summary>
    /// A collection of DataSetOrderingType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataSetOrderingType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataSetOrderingType")]
    #if !NET_STANDARD
    public partial class DataSetOrderingTypeCollection : List<DataSetOrderingType>, ICloneable
    #else
    public partial class DataSetOrderingTypeCollection : List<DataSetOrderingType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public DataSetOrderingTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public DataSetOrderingTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public DataSetOrderingTypeCollection(IEnumerable<DataSetOrderingType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator DataSetOrderingTypeCollection(DataSetOrderingType[] values)
        {
            if (values != null)
            {
                return new DataSetOrderingTypeCollection(values);
            }

            return new DataSetOrderingTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator DataSetOrderingType[](DataSetOrderingTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (DataSetOrderingTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetOrderingTypeCollection clone = new DataSetOrderingTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataSetOrderingType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region UadpNetworkMessageContentMask Enumeration
    #if (!OPCUA_EXCLUDE_UadpNetworkMessageContentMask)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum UadpNetworkMessageContentMask : UInt32
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "PublisherId_1")]
        PublisherId = 1,

        /// <remarks />
        [EnumMember(Value = "GroupHeader_2")]
        GroupHeader = 2,

        /// <remarks />
        [EnumMember(Value = "WriterGroupId_4")]
        WriterGroupId = 4,

        /// <remarks />
        [EnumMember(Value = "GroupVersion_8")]
        GroupVersion = 8,

        /// <remarks />
        [EnumMember(Value = "NetworkMessageNumber_16")]
        NetworkMessageNumber = 16,

        /// <remarks />
        [EnumMember(Value = "SequenceNumber_32")]
        SequenceNumber = 32,

        /// <remarks />
        [EnumMember(Value = "PayloadHeader_64")]
        PayloadHeader = 64,

        /// <remarks />
        [EnumMember(Value = "Timestamp_128")]
        Timestamp = 128,

        /// <remarks />
        [EnumMember(Value = "PicoSeconds_256")]
        PicoSeconds = 256,

        /// <remarks />
        [EnumMember(Value = "DataSetClassId_512")]
        DataSetClassId = 512,

        /// <remarks />
        [EnumMember(Value = "PromotedFields_1024")]
        PromotedFields = 1024,
    }

    #region UadpNetworkMessageContentMaskCollection Class
    /// <summary>
    /// A collection of UadpNetworkMessageContentMask objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUadpNetworkMessageContentMask", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "UadpNetworkMessageContentMask")]
    #if !NET_STANDARD
    public partial class UadpNetworkMessageContentMaskCollection : List<UadpNetworkMessageContentMask>, ICloneable
    #else
    public partial class UadpNetworkMessageContentMaskCollection : List<UadpNetworkMessageContentMask>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public UadpNetworkMessageContentMaskCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public UadpNetworkMessageContentMaskCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public UadpNetworkMessageContentMaskCollection(IEnumerable<UadpNetworkMessageContentMask> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator UadpNetworkMessageContentMaskCollection(UadpNetworkMessageContentMask[] values)
        {
            if (values != null)
            {
                return new UadpNetworkMessageContentMaskCollection(values);
            }

            return new UadpNetworkMessageContentMaskCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator UadpNetworkMessageContentMask[](UadpNetworkMessageContentMaskCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (UadpNetworkMessageContentMaskCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UadpNetworkMessageContentMaskCollection clone = new UadpNetworkMessageContentMaskCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((UadpNetworkMessageContentMask)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region UadpWriterGroupMessageDataType Class
    #if (!OPCUA_EXCLUDE_UadpWriterGroupMessageDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class UadpWriterGroupMessageDataType : WriterGroupMessageDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public UadpWriterGroupMessageDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_groupVersion = (uint)0;
            m_dataSetOrdering = DataSetOrderingType.Undefined;
            m_networkMessageContentMask = 0;
            m_samplingOffset = (double)0;
            m_publishingOffset = new DoubleCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "GroupVersion", IsRequired = false, Order = 1)]
        public uint GroupVersion
        {
            get { return m_groupVersion;  }
            set { m_groupVersion = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetOrdering", IsRequired = false, Order = 2)]
        public DataSetOrderingType DataSetOrdering
        {
            get { return m_dataSetOrdering;  }
            set { m_dataSetOrdering = value; }
        }

        /// <remarks />
        [DataMember(Name = "NetworkMessageContentMask", IsRequired = false, Order = 3)]
        public uint NetworkMessageContentMask
        {
            get { return m_networkMessageContentMask;  }
            set { m_networkMessageContentMask = value; }
        }

        /// <remarks />
        [DataMember(Name = "SamplingOffset", IsRequired = false, Order = 4)]
        public double SamplingOffset
        {
            get { return m_samplingOffset;  }
            set { m_samplingOffset = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "PublishingOffset", IsRequired = false, Order = 5)]
        public DoubleCollection PublishingOffset
        {
            get
            {
                return m_publishingOffset;
            }

            set
            {
                m_publishingOffset = value;

                if (value == null)
                {
                    m_publishingOffset = new DoubleCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.UadpWriterGroupMessageDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.UadpWriterGroupMessageDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.UadpWriterGroupMessageDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("GroupVersion", GroupVersion);
            encoder.WriteEnumerated("DataSetOrdering", DataSetOrdering);
            encoder.WriteUInt32("NetworkMessageContentMask", NetworkMessageContentMask);
            encoder.WriteDouble("SamplingOffset", SamplingOffset);
            encoder.WriteDoubleArray("PublishingOffset", PublishingOffset);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            GroupVersion = decoder.ReadUInt32("GroupVersion");
            DataSetOrdering = (DataSetOrderingType)decoder.ReadEnumerated("DataSetOrdering", typeof(DataSetOrderingType));
            NetworkMessageContentMask = decoder.ReadUInt32("NetworkMessageContentMask");
            SamplingOffset = decoder.ReadDouble("SamplingOffset");
            PublishingOffset = decoder.ReadDoubleArray("PublishingOffset");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            UadpWriterGroupMessageDataType value = encodeable as UadpWriterGroupMessageDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_groupVersion, value.m_groupVersion)) return false;
            if (!Utils.IsEqual(m_dataSetOrdering, value.m_dataSetOrdering)) return false;
            if (!Utils.IsEqual(m_networkMessageContentMask, value.m_networkMessageContentMask)) return false;
            if (!Utils.IsEqual(m_samplingOffset, value.m_samplingOffset)) return false;
            if (!Utils.IsEqual(m_publishingOffset, value.m_publishingOffset)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (UadpWriterGroupMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UadpWriterGroupMessageDataType clone = (UadpWriterGroupMessageDataType)base.MemberwiseClone();

            clone.m_groupVersion = (uint)Utils.Clone(this.m_groupVersion);
            clone.m_dataSetOrdering = (DataSetOrderingType)Utils.Clone(this.m_dataSetOrdering);
            clone.m_networkMessageContentMask = (uint)Utils.Clone(this.m_networkMessageContentMask);
            clone.m_samplingOffset = (double)Utils.Clone(this.m_samplingOffset);
            clone.m_publishingOffset = (DoubleCollection)Utils.Clone(this.m_publishingOffset);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_groupVersion;
        private DataSetOrderingType m_dataSetOrdering;
        private uint m_networkMessageContentMask;
        private double m_samplingOffset;
        private DoubleCollection m_publishingOffset;
        #endregion
    }

    #region UadpWriterGroupMessageDataTypeCollection Class
    /// <summary>
    /// A collection of UadpWriterGroupMessageDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUadpWriterGroupMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "UadpWriterGroupMessageDataType")]
    #if !NET_STANDARD
    public partial class UadpWriterGroupMessageDataTypeCollection : List<UadpWriterGroupMessageDataType>, ICloneable
    #else
    public partial class UadpWriterGroupMessageDataTypeCollection : List<UadpWriterGroupMessageDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public UadpWriterGroupMessageDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public UadpWriterGroupMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public UadpWriterGroupMessageDataTypeCollection(IEnumerable<UadpWriterGroupMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator UadpWriterGroupMessageDataTypeCollection(UadpWriterGroupMessageDataType[] values)
        {
            if (values != null)
            {
                return new UadpWriterGroupMessageDataTypeCollection(values);
            }

            return new UadpWriterGroupMessageDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator UadpWriterGroupMessageDataType[](UadpWriterGroupMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (UadpWriterGroupMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UadpWriterGroupMessageDataTypeCollection clone = new UadpWriterGroupMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((UadpWriterGroupMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region UadpDataSetMessageContentMask Enumeration
    #if (!OPCUA_EXCLUDE_UadpDataSetMessageContentMask)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum UadpDataSetMessageContentMask : UInt32
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "Timestamp_1")]
        Timestamp = 1,

        /// <remarks />
        [EnumMember(Value = "PicoSeconds_2")]
        PicoSeconds = 2,

        /// <remarks />
        [EnumMember(Value = "Status_4")]
        Status = 4,

        /// <remarks />
        [EnumMember(Value = "MajorVersion_8")]
        MajorVersion = 8,

        /// <remarks />
        [EnumMember(Value = "MinorVersion_16")]
        MinorVersion = 16,

        /// <remarks />
        [EnumMember(Value = "SequenceNumber_32")]
        SequenceNumber = 32,
    }

    #region UadpDataSetMessageContentMaskCollection Class
    /// <summary>
    /// A collection of UadpDataSetMessageContentMask objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUadpDataSetMessageContentMask", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "UadpDataSetMessageContentMask")]
    #if !NET_STANDARD
    public partial class UadpDataSetMessageContentMaskCollection : List<UadpDataSetMessageContentMask>, ICloneable
    #else
    public partial class UadpDataSetMessageContentMaskCollection : List<UadpDataSetMessageContentMask>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public UadpDataSetMessageContentMaskCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public UadpDataSetMessageContentMaskCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public UadpDataSetMessageContentMaskCollection(IEnumerable<UadpDataSetMessageContentMask> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator UadpDataSetMessageContentMaskCollection(UadpDataSetMessageContentMask[] values)
        {
            if (values != null)
            {
                return new UadpDataSetMessageContentMaskCollection(values);
            }

            return new UadpDataSetMessageContentMaskCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator UadpDataSetMessageContentMask[](UadpDataSetMessageContentMaskCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (UadpDataSetMessageContentMaskCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UadpDataSetMessageContentMaskCollection clone = new UadpDataSetMessageContentMaskCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((UadpDataSetMessageContentMask)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region UadpDataSetWriterMessageDataType Class
    #if (!OPCUA_EXCLUDE_UadpDataSetWriterMessageDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class UadpDataSetWriterMessageDataType : DataSetWriterMessageDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public UadpDataSetWriterMessageDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_dataSetMessageContentMask = 0;
            m_configuredSize = (ushort)0;
            m_networkMessageNumber = (ushort)0;
            m_dataSetOffset = (ushort)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "DataSetMessageContentMask", IsRequired = false, Order = 1)]
        public uint DataSetMessageContentMask
        {
            get { return m_dataSetMessageContentMask;  }
            set { m_dataSetMessageContentMask = value; }
        }

        /// <remarks />
        [DataMember(Name = "ConfiguredSize", IsRequired = false, Order = 2)]
        public ushort ConfiguredSize
        {
            get { return m_configuredSize;  }
            set { m_configuredSize = value; }
        }

        /// <remarks />
        [DataMember(Name = "NetworkMessageNumber", IsRequired = false, Order = 3)]
        public ushort NetworkMessageNumber
        {
            get { return m_networkMessageNumber;  }
            set { m_networkMessageNumber = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetOffset", IsRequired = false, Order = 4)]
        public ushort DataSetOffset
        {
            get { return m_dataSetOffset;  }
            set { m_dataSetOffset = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.UadpDataSetWriterMessageDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.UadpDataSetWriterMessageDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.UadpDataSetWriterMessageDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("DataSetMessageContentMask", DataSetMessageContentMask);
            encoder.WriteUInt16("ConfiguredSize", ConfiguredSize);
            encoder.WriteUInt16("NetworkMessageNumber", NetworkMessageNumber);
            encoder.WriteUInt16("DataSetOffset", DataSetOffset);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            DataSetMessageContentMask = decoder.ReadUInt32("DataSetMessageContentMask");
            ConfiguredSize = decoder.ReadUInt16("ConfiguredSize");
            NetworkMessageNumber = decoder.ReadUInt16("NetworkMessageNumber");
            DataSetOffset = decoder.ReadUInt16("DataSetOffset");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            UadpDataSetWriterMessageDataType value = encodeable as UadpDataSetWriterMessageDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_dataSetMessageContentMask, value.m_dataSetMessageContentMask)) return false;
            if (!Utils.IsEqual(m_configuredSize, value.m_configuredSize)) return false;
            if (!Utils.IsEqual(m_networkMessageNumber, value.m_networkMessageNumber)) return false;
            if (!Utils.IsEqual(m_dataSetOffset, value.m_dataSetOffset)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (UadpDataSetWriterMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UadpDataSetWriterMessageDataType clone = (UadpDataSetWriterMessageDataType)base.MemberwiseClone();

            clone.m_dataSetMessageContentMask = (uint)Utils.Clone(this.m_dataSetMessageContentMask);
            clone.m_configuredSize = (ushort)Utils.Clone(this.m_configuredSize);
            clone.m_networkMessageNumber = (ushort)Utils.Clone(this.m_networkMessageNumber);
            clone.m_dataSetOffset = (ushort)Utils.Clone(this.m_dataSetOffset);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_dataSetMessageContentMask;
        private ushort m_configuredSize;
        private ushort m_networkMessageNumber;
        private ushort m_dataSetOffset;
        #endregion
    }

    #region UadpDataSetWriterMessageDataTypeCollection Class
    /// <summary>
    /// A collection of UadpDataSetWriterMessageDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUadpDataSetWriterMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "UadpDataSetWriterMessageDataType")]
    #if !NET_STANDARD
    public partial class UadpDataSetWriterMessageDataTypeCollection : List<UadpDataSetWriterMessageDataType>, ICloneable
    #else
    public partial class UadpDataSetWriterMessageDataTypeCollection : List<UadpDataSetWriterMessageDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public UadpDataSetWriterMessageDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public UadpDataSetWriterMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public UadpDataSetWriterMessageDataTypeCollection(IEnumerable<UadpDataSetWriterMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator UadpDataSetWriterMessageDataTypeCollection(UadpDataSetWriterMessageDataType[] values)
        {
            if (values != null)
            {
                return new UadpDataSetWriterMessageDataTypeCollection(values);
            }

            return new UadpDataSetWriterMessageDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator UadpDataSetWriterMessageDataType[](UadpDataSetWriterMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (UadpDataSetWriterMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UadpDataSetWriterMessageDataTypeCollection clone = new UadpDataSetWriterMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((UadpDataSetWriterMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region UadpDataSetReaderMessageDataType Class
    #if (!OPCUA_EXCLUDE_UadpDataSetReaderMessageDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class UadpDataSetReaderMessageDataType : DataSetReaderMessageDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public UadpDataSetReaderMessageDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_groupVersion = (uint)0;
            m_networkMessageNumber = (ushort)0;
            m_dataSetOffset = (ushort)0;
            m_dataSetClassId = Uuid.Empty;
            m_networkMessageContentMask = 0;
            m_dataSetMessageContentMask = 0;
            m_publishingInterval = (double)0;
            m_receiveOffset = (double)0;
            m_processingOffset = (double)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "GroupVersion", IsRequired = false, Order = 1)]
        public uint GroupVersion
        {
            get { return m_groupVersion;  }
            set { m_groupVersion = value; }
        }

        /// <remarks />
        [DataMember(Name = "NetworkMessageNumber", IsRequired = false, Order = 2)]
        public ushort NetworkMessageNumber
        {
            get { return m_networkMessageNumber;  }
            set { m_networkMessageNumber = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetOffset", IsRequired = false, Order = 3)]
        public ushort DataSetOffset
        {
            get { return m_dataSetOffset;  }
            set { m_dataSetOffset = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetClassId", IsRequired = false, Order = 4)]
        public Uuid DataSetClassId
        {
            get { return m_dataSetClassId;  }
            set { m_dataSetClassId = value; }
        }

        /// <remarks />
        [DataMember(Name = "NetworkMessageContentMask", IsRequired = false, Order = 5)]
        public uint NetworkMessageContentMask
        {
            get { return m_networkMessageContentMask;  }
            set { m_networkMessageContentMask = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetMessageContentMask", IsRequired = false, Order = 6)]
        public uint DataSetMessageContentMask
        {
            get { return m_dataSetMessageContentMask;  }
            set { m_dataSetMessageContentMask = value; }
        }

        /// <remarks />
        [DataMember(Name = "PublishingInterval", IsRequired = false, Order = 7)]
        public double PublishingInterval
        {
            get { return m_publishingInterval;  }
            set { m_publishingInterval = value; }
        }

        /// <remarks />
        [DataMember(Name = "ReceiveOffset", IsRequired = false, Order = 8)]
        public double ReceiveOffset
        {
            get { return m_receiveOffset;  }
            set { m_receiveOffset = value; }
        }

        /// <remarks />
        [DataMember(Name = "ProcessingOffset", IsRequired = false, Order = 9)]
        public double ProcessingOffset
        {
            get { return m_processingOffset;  }
            set { m_processingOffset = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.UadpDataSetReaderMessageDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.UadpDataSetReaderMessageDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.UadpDataSetReaderMessageDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("GroupVersion", GroupVersion);
            encoder.WriteUInt16("NetworkMessageNumber", NetworkMessageNumber);
            encoder.WriteUInt16("DataSetOffset", DataSetOffset);
            encoder.WriteGuid("DataSetClassId", DataSetClassId);
            encoder.WriteUInt32("NetworkMessageContentMask", NetworkMessageContentMask);
            encoder.WriteUInt32("DataSetMessageContentMask", DataSetMessageContentMask);
            encoder.WriteDouble("PublishingInterval", PublishingInterval);
            encoder.WriteDouble("ReceiveOffset", ReceiveOffset);
            encoder.WriteDouble("ProcessingOffset", ProcessingOffset);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            GroupVersion = decoder.ReadUInt32("GroupVersion");
            NetworkMessageNumber = decoder.ReadUInt16("NetworkMessageNumber");
            DataSetOffset = decoder.ReadUInt16("DataSetOffset");
            DataSetClassId = decoder.ReadGuid("DataSetClassId");
            NetworkMessageContentMask = decoder.ReadUInt32("NetworkMessageContentMask");
            DataSetMessageContentMask = decoder.ReadUInt32("DataSetMessageContentMask");
            PublishingInterval = decoder.ReadDouble("PublishingInterval");
            ReceiveOffset = decoder.ReadDouble("ReceiveOffset");
            ProcessingOffset = decoder.ReadDouble("ProcessingOffset");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            UadpDataSetReaderMessageDataType value = encodeable as UadpDataSetReaderMessageDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_groupVersion, value.m_groupVersion)) return false;
            if (!Utils.IsEqual(m_networkMessageNumber, value.m_networkMessageNumber)) return false;
            if (!Utils.IsEqual(m_dataSetOffset, value.m_dataSetOffset)) return false;
            if (!Utils.IsEqual(m_dataSetClassId, value.m_dataSetClassId)) return false;
            if (!Utils.IsEqual(m_networkMessageContentMask, value.m_networkMessageContentMask)) return false;
            if (!Utils.IsEqual(m_dataSetMessageContentMask, value.m_dataSetMessageContentMask)) return false;
            if (!Utils.IsEqual(m_publishingInterval, value.m_publishingInterval)) return false;
            if (!Utils.IsEqual(m_receiveOffset, value.m_receiveOffset)) return false;
            if (!Utils.IsEqual(m_processingOffset, value.m_processingOffset)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (UadpDataSetReaderMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UadpDataSetReaderMessageDataType clone = (UadpDataSetReaderMessageDataType)base.MemberwiseClone();

            clone.m_groupVersion = (uint)Utils.Clone(this.m_groupVersion);
            clone.m_networkMessageNumber = (ushort)Utils.Clone(this.m_networkMessageNumber);
            clone.m_dataSetOffset = (ushort)Utils.Clone(this.m_dataSetOffset);
            clone.m_dataSetClassId = (Uuid)Utils.Clone(this.m_dataSetClassId);
            clone.m_networkMessageContentMask = (uint)Utils.Clone(this.m_networkMessageContentMask);
            clone.m_dataSetMessageContentMask = (uint)Utils.Clone(this.m_dataSetMessageContentMask);
            clone.m_publishingInterval = (double)Utils.Clone(this.m_publishingInterval);
            clone.m_receiveOffset = (double)Utils.Clone(this.m_receiveOffset);
            clone.m_processingOffset = (double)Utils.Clone(this.m_processingOffset);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_groupVersion;
        private ushort m_networkMessageNumber;
        private ushort m_dataSetOffset;
        private Uuid m_dataSetClassId;
        private uint m_networkMessageContentMask;
        private uint m_dataSetMessageContentMask;
        private double m_publishingInterval;
        private double m_receiveOffset;
        private double m_processingOffset;
        #endregion
    }

    #region UadpDataSetReaderMessageDataTypeCollection Class
    /// <summary>
    /// A collection of UadpDataSetReaderMessageDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUadpDataSetReaderMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "UadpDataSetReaderMessageDataType")]
    #if !NET_STANDARD
    public partial class UadpDataSetReaderMessageDataTypeCollection : List<UadpDataSetReaderMessageDataType>, ICloneable
    #else
    public partial class UadpDataSetReaderMessageDataTypeCollection : List<UadpDataSetReaderMessageDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public UadpDataSetReaderMessageDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public UadpDataSetReaderMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public UadpDataSetReaderMessageDataTypeCollection(IEnumerable<UadpDataSetReaderMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator UadpDataSetReaderMessageDataTypeCollection(UadpDataSetReaderMessageDataType[] values)
        {
            if (values != null)
            {
                return new UadpDataSetReaderMessageDataTypeCollection(values);
            }

            return new UadpDataSetReaderMessageDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator UadpDataSetReaderMessageDataType[](UadpDataSetReaderMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (UadpDataSetReaderMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UadpDataSetReaderMessageDataTypeCollection clone = new UadpDataSetReaderMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((UadpDataSetReaderMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region JsonNetworkMessageContentMask Enumeration
    #if (!OPCUA_EXCLUDE_JsonNetworkMessageContentMask)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum JsonNetworkMessageContentMask : UInt32
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "NetworkMessageHeader_1")]
        NetworkMessageHeader = 1,

        /// <remarks />
        [EnumMember(Value = "DataSetMessageHeader_2")]
        DataSetMessageHeader = 2,

        /// <remarks />
        [EnumMember(Value = "SingleDataSetMessage_4")]
        SingleDataSetMessage = 4,

        /// <remarks />
        [EnumMember(Value = "PublisherId_8")]
        PublisherId = 8,

        /// <remarks />
        [EnumMember(Value = "DataSetClassId_16")]
        DataSetClassId = 16,

        /// <remarks />
        [EnumMember(Value = "ReplyTo_32")]
        ReplyTo = 32,
    }

    #region JsonNetworkMessageContentMaskCollection Class
    /// <summary>
    /// A collection of JsonNetworkMessageContentMask objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfJsonNetworkMessageContentMask", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "JsonNetworkMessageContentMask")]
    #if !NET_STANDARD
    public partial class JsonNetworkMessageContentMaskCollection : List<JsonNetworkMessageContentMask>, ICloneable
    #else
    public partial class JsonNetworkMessageContentMaskCollection : List<JsonNetworkMessageContentMask>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public JsonNetworkMessageContentMaskCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public JsonNetworkMessageContentMaskCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public JsonNetworkMessageContentMaskCollection(IEnumerable<JsonNetworkMessageContentMask> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator JsonNetworkMessageContentMaskCollection(JsonNetworkMessageContentMask[] values)
        {
            if (values != null)
            {
                return new JsonNetworkMessageContentMaskCollection(values);
            }

            return new JsonNetworkMessageContentMaskCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator JsonNetworkMessageContentMask[](JsonNetworkMessageContentMaskCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (JsonNetworkMessageContentMaskCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            JsonNetworkMessageContentMaskCollection clone = new JsonNetworkMessageContentMaskCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((JsonNetworkMessageContentMask)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region JsonWriterGroupMessageDataType Class
    #if (!OPCUA_EXCLUDE_JsonWriterGroupMessageDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class JsonWriterGroupMessageDataType : WriterGroupMessageDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public JsonWriterGroupMessageDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_networkMessageContentMask = 0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "NetworkMessageContentMask", IsRequired = false, Order = 1)]
        public uint NetworkMessageContentMask
        {
            get { return m_networkMessageContentMask;  }
            set { m_networkMessageContentMask = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.JsonWriterGroupMessageDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.JsonWriterGroupMessageDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.JsonWriterGroupMessageDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("NetworkMessageContentMask", NetworkMessageContentMask);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            NetworkMessageContentMask = decoder.ReadUInt32("NetworkMessageContentMask");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            JsonWriterGroupMessageDataType value = encodeable as JsonWriterGroupMessageDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_networkMessageContentMask, value.m_networkMessageContentMask)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (JsonWriterGroupMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            JsonWriterGroupMessageDataType clone = (JsonWriterGroupMessageDataType)base.MemberwiseClone();

            clone.m_networkMessageContentMask = (uint)Utils.Clone(this.m_networkMessageContentMask);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_networkMessageContentMask;
        #endregion
    }

    #region JsonWriterGroupMessageDataTypeCollection Class
    /// <summary>
    /// A collection of JsonWriterGroupMessageDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfJsonWriterGroupMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "JsonWriterGroupMessageDataType")]
    #if !NET_STANDARD
    public partial class JsonWriterGroupMessageDataTypeCollection : List<JsonWriterGroupMessageDataType>, ICloneable
    #else
    public partial class JsonWriterGroupMessageDataTypeCollection : List<JsonWriterGroupMessageDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public JsonWriterGroupMessageDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public JsonWriterGroupMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public JsonWriterGroupMessageDataTypeCollection(IEnumerable<JsonWriterGroupMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator JsonWriterGroupMessageDataTypeCollection(JsonWriterGroupMessageDataType[] values)
        {
            if (values != null)
            {
                return new JsonWriterGroupMessageDataTypeCollection(values);
            }

            return new JsonWriterGroupMessageDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator JsonWriterGroupMessageDataType[](JsonWriterGroupMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (JsonWriterGroupMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            JsonWriterGroupMessageDataTypeCollection clone = new JsonWriterGroupMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((JsonWriterGroupMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region JsonDataSetMessageContentMask Enumeration
    #if (!OPCUA_EXCLUDE_JsonDataSetMessageContentMask)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum JsonDataSetMessageContentMask : UInt32
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "DataSetWriterId_1")]
        DataSetWriterId = 1,

        /// <remarks />
        [EnumMember(Value = "MetaDataVersion_2")]
        MetaDataVersion = 2,

        /// <remarks />
        [EnumMember(Value = "SequenceNumber_4")]
        SequenceNumber = 4,

        /// <remarks />
        [EnumMember(Value = "Timestamp_8")]
        Timestamp = 8,

        /// <remarks />
        [EnumMember(Value = "Status_16")]
        Status = 16,

        /// <remarks />
        [EnumMember(Value = "MessageType_32")]
        MessageType = 32,
    }

    #region JsonDataSetMessageContentMaskCollection Class
    /// <summary>
    /// A collection of JsonDataSetMessageContentMask objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfJsonDataSetMessageContentMask", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "JsonDataSetMessageContentMask")]
    #if !NET_STANDARD
    public partial class JsonDataSetMessageContentMaskCollection : List<JsonDataSetMessageContentMask>, ICloneable
    #else
    public partial class JsonDataSetMessageContentMaskCollection : List<JsonDataSetMessageContentMask>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public JsonDataSetMessageContentMaskCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public JsonDataSetMessageContentMaskCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public JsonDataSetMessageContentMaskCollection(IEnumerable<JsonDataSetMessageContentMask> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator JsonDataSetMessageContentMaskCollection(JsonDataSetMessageContentMask[] values)
        {
            if (values != null)
            {
                return new JsonDataSetMessageContentMaskCollection(values);
            }

            return new JsonDataSetMessageContentMaskCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator JsonDataSetMessageContentMask[](JsonDataSetMessageContentMaskCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (JsonDataSetMessageContentMaskCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            JsonDataSetMessageContentMaskCollection clone = new JsonDataSetMessageContentMaskCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((JsonDataSetMessageContentMask)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region JsonDataSetWriterMessageDataType Class
    #if (!OPCUA_EXCLUDE_JsonDataSetWriterMessageDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class JsonDataSetWriterMessageDataType : DataSetWriterMessageDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public JsonDataSetWriterMessageDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_dataSetMessageContentMask = 0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "DataSetMessageContentMask", IsRequired = false, Order = 1)]
        public uint DataSetMessageContentMask
        {
            get { return m_dataSetMessageContentMask;  }
            set { m_dataSetMessageContentMask = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.JsonDataSetWriterMessageDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.JsonDataSetWriterMessageDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.JsonDataSetWriterMessageDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("DataSetMessageContentMask", DataSetMessageContentMask);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            DataSetMessageContentMask = decoder.ReadUInt32("DataSetMessageContentMask");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            JsonDataSetWriterMessageDataType value = encodeable as JsonDataSetWriterMessageDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_dataSetMessageContentMask, value.m_dataSetMessageContentMask)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (JsonDataSetWriterMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            JsonDataSetWriterMessageDataType clone = (JsonDataSetWriterMessageDataType)base.MemberwiseClone();

            clone.m_dataSetMessageContentMask = (uint)Utils.Clone(this.m_dataSetMessageContentMask);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_dataSetMessageContentMask;
        #endregion
    }

    #region JsonDataSetWriterMessageDataTypeCollection Class
    /// <summary>
    /// A collection of JsonDataSetWriterMessageDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfJsonDataSetWriterMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "JsonDataSetWriterMessageDataType")]
    #if !NET_STANDARD
    public partial class JsonDataSetWriterMessageDataTypeCollection : List<JsonDataSetWriterMessageDataType>, ICloneable
    #else
    public partial class JsonDataSetWriterMessageDataTypeCollection : List<JsonDataSetWriterMessageDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public JsonDataSetWriterMessageDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public JsonDataSetWriterMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public JsonDataSetWriterMessageDataTypeCollection(IEnumerable<JsonDataSetWriterMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator JsonDataSetWriterMessageDataTypeCollection(JsonDataSetWriterMessageDataType[] values)
        {
            if (values != null)
            {
                return new JsonDataSetWriterMessageDataTypeCollection(values);
            }

            return new JsonDataSetWriterMessageDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator JsonDataSetWriterMessageDataType[](JsonDataSetWriterMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (JsonDataSetWriterMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            JsonDataSetWriterMessageDataTypeCollection clone = new JsonDataSetWriterMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((JsonDataSetWriterMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region JsonDataSetReaderMessageDataType Class
    #if (!OPCUA_EXCLUDE_JsonDataSetReaderMessageDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class JsonDataSetReaderMessageDataType : DataSetReaderMessageDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public JsonDataSetReaderMessageDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_networkMessageContentMask = 0;
            m_dataSetMessageContentMask = 0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "NetworkMessageContentMask", IsRequired = false, Order = 1)]
        public uint NetworkMessageContentMask
        {
            get { return m_networkMessageContentMask;  }
            set { m_networkMessageContentMask = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetMessageContentMask", IsRequired = false, Order = 2)]
        public uint DataSetMessageContentMask
        {
            get { return m_dataSetMessageContentMask;  }
            set { m_dataSetMessageContentMask = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.JsonDataSetReaderMessageDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.JsonDataSetReaderMessageDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.JsonDataSetReaderMessageDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("NetworkMessageContentMask", NetworkMessageContentMask);
            encoder.WriteUInt32("DataSetMessageContentMask", DataSetMessageContentMask);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            NetworkMessageContentMask = decoder.ReadUInt32("NetworkMessageContentMask");
            DataSetMessageContentMask = decoder.ReadUInt32("DataSetMessageContentMask");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            JsonDataSetReaderMessageDataType value = encodeable as JsonDataSetReaderMessageDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_networkMessageContentMask, value.m_networkMessageContentMask)) return false;
            if (!Utils.IsEqual(m_dataSetMessageContentMask, value.m_dataSetMessageContentMask)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (JsonDataSetReaderMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            JsonDataSetReaderMessageDataType clone = (JsonDataSetReaderMessageDataType)base.MemberwiseClone();

            clone.m_networkMessageContentMask = (uint)Utils.Clone(this.m_networkMessageContentMask);
            clone.m_dataSetMessageContentMask = (uint)Utils.Clone(this.m_dataSetMessageContentMask);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_networkMessageContentMask;
        private uint m_dataSetMessageContentMask;
        #endregion
    }

    #region JsonDataSetReaderMessageDataTypeCollection Class
    /// <summary>
    /// A collection of JsonDataSetReaderMessageDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfJsonDataSetReaderMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "JsonDataSetReaderMessageDataType")]
    #if !NET_STANDARD
    public partial class JsonDataSetReaderMessageDataTypeCollection : List<JsonDataSetReaderMessageDataType>, ICloneable
    #else
    public partial class JsonDataSetReaderMessageDataTypeCollection : List<JsonDataSetReaderMessageDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public JsonDataSetReaderMessageDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public JsonDataSetReaderMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public JsonDataSetReaderMessageDataTypeCollection(IEnumerable<JsonDataSetReaderMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator JsonDataSetReaderMessageDataTypeCollection(JsonDataSetReaderMessageDataType[] values)
        {
            if (values != null)
            {
                return new JsonDataSetReaderMessageDataTypeCollection(values);
            }

            return new JsonDataSetReaderMessageDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator JsonDataSetReaderMessageDataType[](JsonDataSetReaderMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (JsonDataSetReaderMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            JsonDataSetReaderMessageDataTypeCollection clone = new JsonDataSetReaderMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((JsonDataSetReaderMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DatagramConnectionTransportDataType Class
    #if (!OPCUA_EXCLUDE_DatagramConnectionTransportDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DatagramConnectionTransportDataType : ConnectionTransportDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public DatagramConnectionTransportDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_discoveryAddress = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "DiscoveryAddress", IsRequired = false, Order = 1)]
        public ExtensionObject DiscoveryAddress
        {
            get { return m_discoveryAddress;  }
            set { m_discoveryAddress = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.DatagramConnectionTransportDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.DatagramConnectionTransportDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.DatagramConnectionTransportDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteExtensionObject("DiscoveryAddress", DiscoveryAddress);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            DiscoveryAddress = decoder.ReadExtensionObject("DiscoveryAddress");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DatagramConnectionTransportDataType value = encodeable as DatagramConnectionTransportDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_discoveryAddress, value.m_discoveryAddress)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (DatagramConnectionTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DatagramConnectionTransportDataType clone = (DatagramConnectionTransportDataType)base.MemberwiseClone();

            clone.m_discoveryAddress = (ExtensionObject)Utils.Clone(this.m_discoveryAddress);

            return clone;
        }
        #endregion

        #region Private Fields
        private ExtensionObject m_discoveryAddress;
        #endregion
    }

    #region DatagramConnectionTransportDataTypeCollection Class
    /// <summary>
    /// A collection of DatagramConnectionTransportDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDatagramConnectionTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DatagramConnectionTransportDataType")]
    #if !NET_STANDARD
    public partial class DatagramConnectionTransportDataTypeCollection : List<DatagramConnectionTransportDataType>, ICloneable
    #else
    public partial class DatagramConnectionTransportDataTypeCollection : List<DatagramConnectionTransportDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public DatagramConnectionTransportDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public DatagramConnectionTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public DatagramConnectionTransportDataTypeCollection(IEnumerable<DatagramConnectionTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator DatagramConnectionTransportDataTypeCollection(DatagramConnectionTransportDataType[] values)
        {
            if (values != null)
            {
                return new DatagramConnectionTransportDataTypeCollection(values);
            }

            return new DatagramConnectionTransportDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator DatagramConnectionTransportDataType[](DatagramConnectionTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (DatagramConnectionTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DatagramConnectionTransportDataTypeCollection clone = new DatagramConnectionTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DatagramConnectionTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DatagramWriterGroupTransportDataType Class
    #if (!OPCUA_EXCLUDE_DatagramWriterGroupTransportDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DatagramWriterGroupTransportDataType : WriterGroupTransportDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public DatagramWriterGroupTransportDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_messageRepeatCount = (byte)0;
            m_messageRepeatDelay = (double)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "MessageRepeatCount", IsRequired = false, Order = 1)]
        public byte MessageRepeatCount
        {
            get { return m_messageRepeatCount;  }
            set { m_messageRepeatCount = value; }
        }

        /// <remarks />
        [DataMember(Name = "MessageRepeatDelay", IsRequired = false, Order = 2)]
        public double MessageRepeatDelay
        {
            get { return m_messageRepeatDelay;  }
            set { m_messageRepeatDelay = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.DatagramWriterGroupTransportDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.DatagramWriterGroupTransportDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.DatagramWriterGroupTransportDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteByte("MessageRepeatCount", MessageRepeatCount);
            encoder.WriteDouble("MessageRepeatDelay", MessageRepeatDelay);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            MessageRepeatCount = decoder.ReadByte("MessageRepeatCount");
            MessageRepeatDelay = decoder.ReadDouble("MessageRepeatDelay");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DatagramWriterGroupTransportDataType value = encodeable as DatagramWriterGroupTransportDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_messageRepeatCount, value.m_messageRepeatCount)) return false;
            if (!Utils.IsEqual(m_messageRepeatDelay, value.m_messageRepeatDelay)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (DatagramWriterGroupTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DatagramWriterGroupTransportDataType clone = (DatagramWriterGroupTransportDataType)base.MemberwiseClone();

            clone.m_messageRepeatCount = (byte)Utils.Clone(this.m_messageRepeatCount);
            clone.m_messageRepeatDelay = (double)Utils.Clone(this.m_messageRepeatDelay);

            return clone;
        }
        #endregion

        #region Private Fields
        private byte m_messageRepeatCount;
        private double m_messageRepeatDelay;
        #endregion
    }

    #region DatagramWriterGroupTransportDataTypeCollection Class
    /// <summary>
    /// A collection of DatagramWriterGroupTransportDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDatagramWriterGroupTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DatagramWriterGroupTransportDataType")]
    #if !NET_STANDARD
    public partial class DatagramWriterGroupTransportDataTypeCollection : List<DatagramWriterGroupTransportDataType>, ICloneable
    #else
    public partial class DatagramWriterGroupTransportDataTypeCollection : List<DatagramWriterGroupTransportDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public DatagramWriterGroupTransportDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public DatagramWriterGroupTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public DatagramWriterGroupTransportDataTypeCollection(IEnumerable<DatagramWriterGroupTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator DatagramWriterGroupTransportDataTypeCollection(DatagramWriterGroupTransportDataType[] values)
        {
            if (values != null)
            {
                return new DatagramWriterGroupTransportDataTypeCollection(values);
            }

            return new DatagramWriterGroupTransportDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator DatagramWriterGroupTransportDataType[](DatagramWriterGroupTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (DatagramWriterGroupTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DatagramWriterGroupTransportDataTypeCollection clone = new DatagramWriterGroupTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DatagramWriterGroupTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region BrokerConnectionTransportDataType Class
    #if (!OPCUA_EXCLUDE_BrokerConnectionTransportDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class BrokerConnectionTransportDataType : ConnectionTransportDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public BrokerConnectionTransportDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_resourceUri = null;
            m_authenticationProfileUri = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ResourceUri", IsRequired = false, Order = 1)]
        public string ResourceUri
        {
            get { return m_resourceUri;  }
            set { m_resourceUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "AuthenticationProfileUri", IsRequired = false, Order = 2)]
        public string AuthenticationProfileUri
        {
            get { return m_authenticationProfileUri;  }
            set { m_authenticationProfileUri = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.BrokerConnectionTransportDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.BrokerConnectionTransportDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.BrokerConnectionTransportDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("ResourceUri", ResourceUri);
            encoder.WriteString("AuthenticationProfileUri", AuthenticationProfileUri);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ResourceUri = decoder.ReadString("ResourceUri");
            AuthenticationProfileUri = decoder.ReadString("AuthenticationProfileUri");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            BrokerConnectionTransportDataType value = encodeable as BrokerConnectionTransportDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_resourceUri, value.m_resourceUri)) return false;
            if (!Utils.IsEqual(m_authenticationProfileUri, value.m_authenticationProfileUri)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (BrokerConnectionTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            BrokerConnectionTransportDataType clone = (BrokerConnectionTransportDataType)base.MemberwiseClone();

            clone.m_resourceUri = (string)Utils.Clone(this.m_resourceUri);
            clone.m_authenticationProfileUri = (string)Utils.Clone(this.m_authenticationProfileUri);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_resourceUri;
        private string m_authenticationProfileUri;
        #endregion
    }

    #region BrokerConnectionTransportDataTypeCollection Class
    /// <summary>
    /// A collection of BrokerConnectionTransportDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfBrokerConnectionTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "BrokerConnectionTransportDataType")]
    #if !NET_STANDARD
    public partial class BrokerConnectionTransportDataTypeCollection : List<BrokerConnectionTransportDataType>, ICloneable
    #else
    public partial class BrokerConnectionTransportDataTypeCollection : List<BrokerConnectionTransportDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public BrokerConnectionTransportDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public BrokerConnectionTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public BrokerConnectionTransportDataTypeCollection(IEnumerable<BrokerConnectionTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator BrokerConnectionTransportDataTypeCollection(BrokerConnectionTransportDataType[] values)
        {
            if (values != null)
            {
                return new BrokerConnectionTransportDataTypeCollection(values);
            }

            return new BrokerConnectionTransportDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator BrokerConnectionTransportDataType[](BrokerConnectionTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (BrokerConnectionTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            BrokerConnectionTransportDataTypeCollection clone = new BrokerConnectionTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((BrokerConnectionTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region BrokerTransportQualityOfService Enumeration
    #if (!OPCUA_EXCLUDE_BrokerTransportQualityOfService)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum BrokerTransportQualityOfService
    {
        /// <remarks />
        [EnumMember(Value = "NotSpecified_0")]
        NotSpecified = 0,

        /// <remarks />
        [EnumMember(Value = "BestEffort_1")]
        BestEffort = 1,

        /// <remarks />
        [EnumMember(Value = "AtLeastOnce_2")]
        AtLeastOnce = 2,

        /// <remarks />
        [EnumMember(Value = "AtMostOnce_3")]
        AtMostOnce = 3,

        /// <remarks />
        [EnumMember(Value = "ExactlyOnce_4")]
        ExactlyOnce = 4,
    }

    #region BrokerTransportQualityOfServiceCollection Class
    /// <summary>
    /// A collection of BrokerTransportQualityOfService objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfBrokerTransportQualityOfService", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "BrokerTransportQualityOfService")]
    #if !NET_STANDARD
    public partial class BrokerTransportQualityOfServiceCollection : List<BrokerTransportQualityOfService>, ICloneable
    #else
    public partial class BrokerTransportQualityOfServiceCollection : List<BrokerTransportQualityOfService>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public BrokerTransportQualityOfServiceCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public BrokerTransportQualityOfServiceCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public BrokerTransportQualityOfServiceCollection(IEnumerable<BrokerTransportQualityOfService> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator BrokerTransportQualityOfServiceCollection(BrokerTransportQualityOfService[] values)
        {
            if (values != null)
            {
                return new BrokerTransportQualityOfServiceCollection(values);
            }

            return new BrokerTransportQualityOfServiceCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator BrokerTransportQualityOfService[](BrokerTransportQualityOfServiceCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (BrokerTransportQualityOfServiceCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            BrokerTransportQualityOfServiceCollection clone = new BrokerTransportQualityOfServiceCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((BrokerTransportQualityOfService)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region BrokerWriterGroupTransportDataType Class
    #if (!OPCUA_EXCLUDE_BrokerWriterGroupTransportDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class BrokerWriterGroupTransportDataType : WriterGroupTransportDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public BrokerWriterGroupTransportDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_queueName = null;
            m_resourceUri = null;
            m_authenticationProfileUri = null;
            m_requestedDeliveryGuarantee = BrokerTransportQualityOfService.NotSpecified;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "QueueName", IsRequired = false, Order = 1)]
        public string QueueName
        {
            get { return m_queueName;  }
            set { m_queueName = value; }
        }

        /// <remarks />
        [DataMember(Name = "ResourceUri", IsRequired = false, Order = 2)]
        public string ResourceUri
        {
            get { return m_resourceUri;  }
            set { m_resourceUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "AuthenticationProfileUri", IsRequired = false, Order = 3)]
        public string AuthenticationProfileUri
        {
            get { return m_authenticationProfileUri;  }
            set { m_authenticationProfileUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "RequestedDeliveryGuarantee", IsRequired = false, Order = 4)]
        public BrokerTransportQualityOfService RequestedDeliveryGuarantee
        {
            get { return m_requestedDeliveryGuarantee;  }
            set { m_requestedDeliveryGuarantee = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.BrokerWriterGroupTransportDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.BrokerWriterGroupTransportDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.BrokerWriterGroupTransportDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("QueueName", QueueName);
            encoder.WriteString("ResourceUri", ResourceUri);
            encoder.WriteString("AuthenticationProfileUri", AuthenticationProfileUri);
            encoder.WriteEnumerated("RequestedDeliveryGuarantee", RequestedDeliveryGuarantee);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            QueueName = decoder.ReadString("QueueName");
            ResourceUri = decoder.ReadString("ResourceUri");
            AuthenticationProfileUri = decoder.ReadString("AuthenticationProfileUri");
            RequestedDeliveryGuarantee = (BrokerTransportQualityOfService)decoder.ReadEnumerated("RequestedDeliveryGuarantee", typeof(BrokerTransportQualityOfService));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            BrokerWriterGroupTransportDataType value = encodeable as BrokerWriterGroupTransportDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_queueName, value.m_queueName)) return false;
            if (!Utils.IsEqual(m_resourceUri, value.m_resourceUri)) return false;
            if (!Utils.IsEqual(m_authenticationProfileUri, value.m_authenticationProfileUri)) return false;
            if (!Utils.IsEqual(m_requestedDeliveryGuarantee, value.m_requestedDeliveryGuarantee)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (BrokerWriterGroupTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            BrokerWriterGroupTransportDataType clone = (BrokerWriterGroupTransportDataType)base.MemberwiseClone();

            clone.m_queueName = (string)Utils.Clone(this.m_queueName);
            clone.m_resourceUri = (string)Utils.Clone(this.m_resourceUri);
            clone.m_authenticationProfileUri = (string)Utils.Clone(this.m_authenticationProfileUri);
            clone.m_requestedDeliveryGuarantee = (BrokerTransportQualityOfService)Utils.Clone(this.m_requestedDeliveryGuarantee);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_queueName;
        private string m_resourceUri;
        private string m_authenticationProfileUri;
        private BrokerTransportQualityOfService m_requestedDeliveryGuarantee;
        #endregion
    }

    #region BrokerWriterGroupTransportDataTypeCollection Class
    /// <summary>
    /// A collection of BrokerWriterGroupTransportDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfBrokerWriterGroupTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "BrokerWriterGroupTransportDataType")]
    #if !NET_STANDARD
    public partial class BrokerWriterGroupTransportDataTypeCollection : List<BrokerWriterGroupTransportDataType>, ICloneable
    #else
    public partial class BrokerWriterGroupTransportDataTypeCollection : List<BrokerWriterGroupTransportDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public BrokerWriterGroupTransportDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public BrokerWriterGroupTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public BrokerWriterGroupTransportDataTypeCollection(IEnumerable<BrokerWriterGroupTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator BrokerWriterGroupTransportDataTypeCollection(BrokerWriterGroupTransportDataType[] values)
        {
            if (values != null)
            {
                return new BrokerWriterGroupTransportDataTypeCollection(values);
            }

            return new BrokerWriterGroupTransportDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator BrokerWriterGroupTransportDataType[](BrokerWriterGroupTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (BrokerWriterGroupTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            BrokerWriterGroupTransportDataTypeCollection clone = new BrokerWriterGroupTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((BrokerWriterGroupTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region BrokerDataSetWriterTransportDataType Class
    #if (!OPCUA_EXCLUDE_BrokerDataSetWriterTransportDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class BrokerDataSetWriterTransportDataType : DataSetWriterTransportDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public BrokerDataSetWriterTransportDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_queueName = null;
            m_resourceUri = null;
            m_authenticationProfileUri = null;
            m_requestedDeliveryGuarantee = BrokerTransportQualityOfService.NotSpecified;
            m_metaDataQueueName = null;
            m_metaDataUpdateTime = (double)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "QueueName", IsRequired = false, Order = 1)]
        public string QueueName
        {
            get { return m_queueName;  }
            set { m_queueName = value; }
        }

        /// <remarks />
        [DataMember(Name = "ResourceUri", IsRequired = false, Order = 2)]
        public string ResourceUri
        {
            get { return m_resourceUri;  }
            set { m_resourceUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "AuthenticationProfileUri", IsRequired = false, Order = 3)]
        public string AuthenticationProfileUri
        {
            get { return m_authenticationProfileUri;  }
            set { m_authenticationProfileUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "RequestedDeliveryGuarantee", IsRequired = false, Order = 4)]
        public BrokerTransportQualityOfService RequestedDeliveryGuarantee
        {
            get { return m_requestedDeliveryGuarantee;  }
            set { m_requestedDeliveryGuarantee = value; }
        }

        /// <remarks />
        [DataMember(Name = "MetaDataQueueName", IsRequired = false, Order = 5)]
        public string MetaDataQueueName
        {
            get { return m_metaDataQueueName;  }
            set { m_metaDataQueueName = value; }
        }

        /// <remarks />
        [DataMember(Name = "MetaDataUpdateTime", IsRequired = false, Order = 6)]
        public double MetaDataUpdateTime
        {
            get { return m_metaDataUpdateTime;  }
            set { m_metaDataUpdateTime = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.BrokerDataSetWriterTransportDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.BrokerDataSetWriterTransportDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.BrokerDataSetWriterTransportDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("QueueName", QueueName);
            encoder.WriteString("ResourceUri", ResourceUri);
            encoder.WriteString("AuthenticationProfileUri", AuthenticationProfileUri);
            encoder.WriteEnumerated("RequestedDeliveryGuarantee", RequestedDeliveryGuarantee);
            encoder.WriteString("MetaDataQueueName", MetaDataQueueName);
            encoder.WriteDouble("MetaDataUpdateTime", MetaDataUpdateTime);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            QueueName = decoder.ReadString("QueueName");
            ResourceUri = decoder.ReadString("ResourceUri");
            AuthenticationProfileUri = decoder.ReadString("AuthenticationProfileUri");
            RequestedDeliveryGuarantee = (BrokerTransportQualityOfService)decoder.ReadEnumerated("RequestedDeliveryGuarantee", typeof(BrokerTransportQualityOfService));
            MetaDataQueueName = decoder.ReadString("MetaDataQueueName");
            MetaDataUpdateTime = decoder.ReadDouble("MetaDataUpdateTime");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            BrokerDataSetWriterTransportDataType value = encodeable as BrokerDataSetWriterTransportDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_queueName, value.m_queueName)) return false;
            if (!Utils.IsEqual(m_resourceUri, value.m_resourceUri)) return false;
            if (!Utils.IsEqual(m_authenticationProfileUri, value.m_authenticationProfileUri)) return false;
            if (!Utils.IsEqual(m_requestedDeliveryGuarantee, value.m_requestedDeliveryGuarantee)) return false;
            if (!Utils.IsEqual(m_metaDataQueueName, value.m_metaDataQueueName)) return false;
            if (!Utils.IsEqual(m_metaDataUpdateTime, value.m_metaDataUpdateTime)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (BrokerDataSetWriterTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            BrokerDataSetWriterTransportDataType clone = (BrokerDataSetWriterTransportDataType)base.MemberwiseClone();

            clone.m_queueName = (string)Utils.Clone(this.m_queueName);
            clone.m_resourceUri = (string)Utils.Clone(this.m_resourceUri);
            clone.m_authenticationProfileUri = (string)Utils.Clone(this.m_authenticationProfileUri);
            clone.m_requestedDeliveryGuarantee = (BrokerTransportQualityOfService)Utils.Clone(this.m_requestedDeliveryGuarantee);
            clone.m_metaDataQueueName = (string)Utils.Clone(this.m_metaDataQueueName);
            clone.m_metaDataUpdateTime = (double)Utils.Clone(this.m_metaDataUpdateTime);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_queueName;
        private string m_resourceUri;
        private string m_authenticationProfileUri;
        private BrokerTransportQualityOfService m_requestedDeliveryGuarantee;
        private string m_metaDataQueueName;
        private double m_metaDataUpdateTime;
        #endregion
    }

    #region BrokerDataSetWriterTransportDataTypeCollection Class
    /// <summary>
    /// A collection of BrokerDataSetWriterTransportDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfBrokerDataSetWriterTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "BrokerDataSetWriterTransportDataType")]
    #if !NET_STANDARD
    public partial class BrokerDataSetWriterTransportDataTypeCollection : List<BrokerDataSetWriterTransportDataType>, ICloneable
    #else
    public partial class BrokerDataSetWriterTransportDataTypeCollection : List<BrokerDataSetWriterTransportDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public BrokerDataSetWriterTransportDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public BrokerDataSetWriterTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public BrokerDataSetWriterTransportDataTypeCollection(IEnumerable<BrokerDataSetWriterTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator BrokerDataSetWriterTransportDataTypeCollection(BrokerDataSetWriterTransportDataType[] values)
        {
            if (values != null)
            {
                return new BrokerDataSetWriterTransportDataTypeCollection(values);
            }

            return new BrokerDataSetWriterTransportDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator BrokerDataSetWriterTransportDataType[](BrokerDataSetWriterTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (BrokerDataSetWriterTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            BrokerDataSetWriterTransportDataTypeCollection clone = new BrokerDataSetWriterTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((BrokerDataSetWriterTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region BrokerDataSetReaderTransportDataType Class
    #if (!OPCUA_EXCLUDE_BrokerDataSetReaderTransportDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class BrokerDataSetReaderTransportDataType : DataSetReaderTransportDataType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public BrokerDataSetReaderTransportDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_queueName = null;
            m_resourceUri = null;
            m_authenticationProfileUri = null;
            m_requestedDeliveryGuarantee = BrokerTransportQualityOfService.NotSpecified;
            m_metaDataQueueName = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "QueueName", IsRequired = false, Order = 1)]
        public string QueueName
        {
            get { return m_queueName;  }
            set { m_queueName = value; }
        }

        /// <remarks />
        [DataMember(Name = "ResourceUri", IsRequired = false, Order = 2)]
        public string ResourceUri
        {
            get { return m_resourceUri;  }
            set { m_resourceUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "AuthenticationProfileUri", IsRequired = false, Order = 3)]
        public string AuthenticationProfileUri
        {
            get { return m_authenticationProfileUri;  }
            set { m_authenticationProfileUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "RequestedDeliveryGuarantee", IsRequired = false, Order = 4)]
        public BrokerTransportQualityOfService RequestedDeliveryGuarantee
        {
            get { return m_requestedDeliveryGuarantee;  }
            set { m_requestedDeliveryGuarantee = value; }
        }

        /// <remarks />
        [DataMember(Name = "MetaDataQueueName", IsRequired = false, Order = 5)]
        public string MetaDataQueueName
        {
            get { return m_metaDataQueueName;  }
            set { m_metaDataQueueName = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.BrokerDataSetReaderTransportDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.BrokerDataSetReaderTransportDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.BrokerDataSetReaderTransportDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("QueueName", QueueName);
            encoder.WriteString("ResourceUri", ResourceUri);
            encoder.WriteString("AuthenticationProfileUri", AuthenticationProfileUri);
            encoder.WriteEnumerated("RequestedDeliveryGuarantee", RequestedDeliveryGuarantee);
            encoder.WriteString("MetaDataQueueName", MetaDataQueueName);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            QueueName = decoder.ReadString("QueueName");
            ResourceUri = decoder.ReadString("ResourceUri");
            AuthenticationProfileUri = decoder.ReadString("AuthenticationProfileUri");
            RequestedDeliveryGuarantee = (BrokerTransportQualityOfService)decoder.ReadEnumerated("RequestedDeliveryGuarantee", typeof(BrokerTransportQualityOfService));
            MetaDataQueueName = decoder.ReadString("MetaDataQueueName");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            BrokerDataSetReaderTransportDataType value = encodeable as BrokerDataSetReaderTransportDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_queueName, value.m_queueName)) return false;
            if (!Utils.IsEqual(m_resourceUri, value.m_resourceUri)) return false;
            if (!Utils.IsEqual(m_authenticationProfileUri, value.m_authenticationProfileUri)) return false;
            if (!Utils.IsEqual(m_requestedDeliveryGuarantee, value.m_requestedDeliveryGuarantee)) return false;
            if (!Utils.IsEqual(m_metaDataQueueName, value.m_metaDataQueueName)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (BrokerDataSetReaderTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            BrokerDataSetReaderTransportDataType clone = (BrokerDataSetReaderTransportDataType)base.MemberwiseClone();

            clone.m_queueName = (string)Utils.Clone(this.m_queueName);
            clone.m_resourceUri = (string)Utils.Clone(this.m_resourceUri);
            clone.m_authenticationProfileUri = (string)Utils.Clone(this.m_authenticationProfileUri);
            clone.m_requestedDeliveryGuarantee = (BrokerTransportQualityOfService)Utils.Clone(this.m_requestedDeliveryGuarantee);
            clone.m_metaDataQueueName = (string)Utils.Clone(this.m_metaDataQueueName);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_queueName;
        private string m_resourceUri;
        private string m_authenticationProfileUri;
        private BrokerTransportQualityOfService m_requestedDeliveryGuarantee;
        private string m_metaDataQueueName;
        #endregion
    }

    #region BrokerDataSetReaderTransportDataTypeCollection Class
    /// <summary>
    /// A collection of BrokerDataSetReaderTransportDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfBrokerDataSetReaderTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "BrokerDataSetReaderTransportDataType")]
    #if !NET_STANDARD
    public partial class BrokerDataSetReaderTransportDataTypeCollection : List<BrokerDataSetReaderTransportDataType>, ICloneable
    #else
    public partial class BrokerDataSetReaderTransportDataTypeCollection : List<BrokerDataSetReaderTransportDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public BrokerDataSetReaderTransportDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public BrokerDataSetReaderTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public BrokerDataSetReaderTransportDataTypeCollection(IEnumerable<BrokerDataSetReaderTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator BrokerDataSetReaderTransportDataTypeCollection(BrokerDataSetReaderTransportDataType[] values)
        {
            if (values != null)
            {
                return new BrokerDataSetReaderTransportDataTypeCollection(values);
            }

            return new BrokerDataSetReaderTransportDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator BrokerDataSetReaderTransportDataType[](BrokerDataSetReaderTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (BrokerDataSetReaderTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            BrokerDataSetReaderTransportDataTypeCollection clone = new BrokerDataSetReaderTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((BrokerDataSetReaderTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DiagnosticsLevel Enumeration
    #if (!OPCUA_EXCLUDE_DiagnosticsLevel)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum DiagnosticsLevel
    {
        /// <remarks />
        [EnumMember(Value = "Basic_0")]
        Basic = 0,

        /// <remarks />
        [EnumMember(Value = "Advanced_1")]
        Advanced = 1,

        /// <remarks />
        [EnumMember(Value = "Info_2")]
        Info = 2,

        /// <remarks />
        [EnumMember(Value = "Log_3")]
        Log = 3,

        /// <remarks />
        [EnumMember(Value = "Debug_4")]
        Debug = 4,
    }

    #region DiagnosticsLevelCollection Class
    /// <summary>
    /// A collection of DiagnosticsLevel objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDiagnosticsLevel", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DiagnosticsLevel")]
    #if !NET_STANDARD
    public partial class DiagnosticsLevelCollection : List<DiagnosticsLevel>, ICloneable
    #else
    public partial class DiagnosticsLevelCollection : List<DiagnosticsLevel>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public DiagnosticsLevelCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public DiagnosticsLevelCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public DiagnosticsLevelCollection(IEnumerable<DiagnosticsLevel> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator DiagnosticsLevelCollection(DiagnosticsLevel[] values)
        {
            if (values != null)
            {
                return new DiagnosticsLevelCollection(values);
            }

            return new DiagnosticsLevelCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator DiagnosticsLevel[](DiagnosticsLevelCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (DiagnosticsLevelCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DiagnosticsLevelCollection clone = new DiagnosticsLevelCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DiagnosticsLevel)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PubSubDiagnosticsCounterClassification Enumeration
    #if (!OPCUA_EXCLUDE_PubSubDiagnosticsCounterClassification)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum PubSubDiagnosticsCounterClassification
    {
        /// <remarks />
        [EnumMember(Value = "Information_0")]
        Information = 0,

        /// <remarks />
        [EnumMember(Value = "Error_1")]
        Error = 1,
    }

    #region PubSubDiagnosticsCounterClassificationCollection Class
    /// <summary>
    /// A collection of PubSubDiagnosticsCounterClassification objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPubSubDiagnosticsCounterClassification", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PubSubDiagnosticsCounterClassification")]
    #if !NET_STANDARD
    public partial class PubSubDiagnosticsCounterClassificationCollection : List<PubSubDiagnosticsCounterClassification>, ICloneable
    #else
    public partial class PubSubDiagnosticsCounterClassificationCollection : List<PubSubDiagnosticsCounterClassification>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public PubSubDiagnosticsCounterClassificationCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public PubSubDiagnosticsCounterClassificationCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public PubSubDiagnosticsCounterClassificationCollection(IEnumerable<PubSubDiagnosticsCounterClassification> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator PubSubDiagnosticsCounterClassificationCollection(PubSubDiagnosticsCounterClassification[] values)
        {
            if (values != null)
            {
                return new PubSubDiagnosticsCounterClassificationCollection(values);
            }

            return new PubSubDiagnosticsCounterClassificationCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator PubSubDiagnosticsCounterClassification[](PubSubDiagnosticsCounterClassificationCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (PubSubDiagnosticsCounterClassificationCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubDiagnosticsCounterClassificationCollection clone = new PubSubDiagnosticsCounterClassificationCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PubSubDiagnosticsCounterClassification)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region AliasNameDataType Class
    #if (!OPCUA_EXCLUDE_AliasNameDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class AliasNameDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public AliasNameDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_aliasName = null;
            m_referencedNodes = new ExpandedNodeIdCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "AliasName", IsRequired = false, Order = 1)]
        public QualifiedName AliasName
        {
            get { return m_aliasName;  }
            set { m_aliasName = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ReferencedNodes", IsRequired = false, Order = 2)]
        public ExpandedNodeIdCollection ReferencedNodes
        {
            get
            {
                return m_referencedNodes;
            }

            set
            {
                m_referencedNodes = value;

                if (value == null)
                {
                    m_referencedNodes = new ExpandedNodeIdCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.AliasNameDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.AliasNameDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.AliasNameDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteQualifiedName("AliasName", AliasName);
            encoder.WriteExpandedNodeIdArray("ReferencedNodes", ReferencedNodes);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            AliasName = decoder.ReadQualifiedName("AliasName");
            ReferencedNodes = decoder.ReadExpandedNodeIdArray("ReferencedNodes");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            AliasNameDataType value = encodeable as AliasNameDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_aliasName, value.m_aliasName)) return false;
            if (!Utils.IsEqual(m_referencedNodes, value.m_referencedNodes)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (AliasNameDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            AliasNameDataType clone = (AliasNameDataType)base.MemberwiseClone();

            clone.m_aliasName = (QualifiedName)Utils.Clone(this.m_aliasName);
            clone.m_referencedNodes = (ExpandedNodeIdCollection)Utils.Clone(this.m_referencedNodes);

            return clone;
        }
        #endregion

        #region Private Fields
        private QualifiedName m_aliasName;
        private ExpandedNodeIdCollection m_referencedNodes;
        #endregion
    }

    #region AliasNameDataTypeCollection Class
    /// <summary>
    /// A collection of AliasNameDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfAliasNameDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "AliasNameDataType")]
    #if !NET_STANDARD
    public partial class AliasNameDataTypeCollection : List<AliasNameDataType>, ICloneable
    #else
    public partial class AliasNameDataTypeCollection : List<AliasNameDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public AliasNameDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public AliasNameDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public AliasNameDataTypeCollection(IEnumerable<AliasNameDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator AliasNameDataTypeCollection(AliasNameDataType[] values)
        {
            if (values != null)
            {
                return new AliasNameDataTypeCollection(values);
            }

            return new AliasNameDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator AliasNameDataType[](AliasNameDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (AliasNameDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            AliasNameDataTypeCollection clone = new AliasNameDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((AliasNameDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region Duplex Enumeration
    #if (!OPCUA_EXCLUDE_Duplex)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum Duplex
    {
        /// <remarks />
        [EnumMember(Value = "Full_0")]
        Full = 0,

        /// <remarks />
        [EnumMember(Value = "Half_1")]
        Half = 1,

        /// <remarks />
        [EnumMember(Value = "Unknown_2")]
        Unknown = 2,
    }

    #region DuplexCollection Class
    /// <summary>
    /// A collection of Duplex objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDuplex", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "Duplex")]
    #if !NET_STANDARD
    public partial class DuplexCollection : List<Duplex>, ICloneable
    #else
    public partial class DuplexCollection : List<Duplex>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public DuplexCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public DuplexCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public DuplexCollection(IEnumerable<Duplex> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator DuplexCollection(Duplex[] values)
        {
            if (values != null)
            {
                return new DuplexCollection(values);
            }

            return new DuplexCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator Duplex[](DuplexCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (DuplexCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DuplexCollection clone = new DuplexCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((Duplex)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region InterfaceAdminStatus Enumeration
    #if (!OPCUA_EXCLUDE_InterfaceAdminStatus)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum InterfaceAdminStatus
    {
        /// <remarks />
        [EnumMember(Value = "Up_0")]
        Up = 0,

        /// <remarks />
        [EnumMember(Value = "Down_1")]
        Down = 1,

        /// <remarks />
        [EnumMember(Value = "Testing_2")]
        Testing = 2,
    }

    #region InterfaceAdminStatusCollection Class
    /// <summary>
    /// A collection of InterfaceAdminStatus objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfInterfaceAdminStatus", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "InterfaceAdminStatus")]
    #if !NET_STANDARD
    public partial class InterfaceAdminStatusCollection : List<InterfaceAdminStatus>, ICloneable
    #else
    public partial class InterfaceAdminStatusCollection : List<InterfaceAdminStatus>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public InterfaceAdminStatusCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public InterfaceAdminStatusCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public InterfaceAdminStatusCollection(IEnumerable<InterfaceAdminStatus> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator InterfaceAdminStatusCollection(InterfaceAdminStatus[] values)
        {
            if (values != null)
            {
                return new InterfaceAdminStatusCollection(values);
            }

            return new InterfaceAdminStatusCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator InterfaceAdminStatus[](InterfaceAdminStatusCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (InterfaceAdminStatusCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            InterfaceAdminStatusCollection clone = new InterfaceAdminStatusCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((InterfaceAdminStatus)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region InterfaceOperStatus Enumeration
    #if (!OPCUA_EXCLUDE_InterfaceOperStatus)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum InterfaceOperStatus
    {
        /// <remarks />
        [EnumMember(Value = "Up_0")]
        Up = 0,

        /// <remarks />
        [EnumMember(Value = "Down_1")]
        Down = 1,

        /// <remarks />
        [EnumMember(Value = "Testing_2")]
        Testing = 2,

        /// <remarks />
        [EnumMember(Value = "Unknown_3")]
        Unknown = 3,

        /// <remarks />
        [EnumMember(Value = "Dormant_4")]
        Dormant = 4,

        /// <remarks />
        [EnumMember(Value = "NotPresent_5")]
        NotPresent = 5,

        /// <remarks />
        [EnumMember(Value = "LowerLayerDown_6")]
        LowerLayerDown = 6,
    }

    #region InterfaceOperStatusCollection Class
    /// <summary>
    /// A collection of InterfaceOperStatus objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfInterfaceOperStatus", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "InterfaceOperStatus")]
    #if !NET_STANDARD
    public partial class InterfaceOperStatusCollection : List<InterfaceOperStatus>, ICloneable
    #else
    public partial class InterfaceOperStatusCollection : List<InterfaceOperStatus>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public InterfaceOperStatusCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public InterfaceOperStatusCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public InterfaceOperStatusCollection(IEnumerable<InterfaceOperStatus> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator InterfaceOperStatusCollection(InterfaceOperStatus[] values)
        {
            if (values != null)
            {
                return new InterfaceOperStatusCollection(values);
            }

            return new InterfaceOperStatusCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator InterfaceOperStatus[](InterfaceOperStatusCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (InterfaceOperStatusCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            InterfaceOperStatusCollection clone = new InterfaceOperStatusCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((InterfaceOperStatus)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region NegotiationStatus Enumeration
    #if (!OPCUA_EXCLUDE_NegotiationStatus)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum NegotiationStatus
    {
        /// <remarks />
        [EnumMember(Value = "InProgress_0")]
        InProgress = 0,

        /// <remarks />
        [EnumMember(Value = "Complete_1")]
        Complete = 1,

        /// <remarks />
        [EnumMember(Value = "Failed_2")]
        Failed = 2,

        /// <remarks />
        [EnumMember(Value = "Unknown_3")]
        Unknown = 3,

        /// <remarks />
        [EnumMember(Value = "NoNegotiation_4")]
        NoNegotiation = 4,
    }

    #region NegotiationStatusCollection Class
    /// <summary>
    /// A collection of NegotiationStatus objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfNegotiationStatus", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "NegotiationStatus")]
    #if !NET_STANDARD
    public partial class NegotiationStatusCollection : List<NegotiationStatus>, ICloneable
    #else
    public partial class NegotiationStatusCollection : List<NegotiationStatus>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public NegotiationStatusCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public NegotiationStatusCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public NegotiationStatusCollection(IEnumerable<NegotiationStatus> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator NegotiationStatusCollection(NegotiationStatus[] values)
        {
            if (values != null)
            {
                return new NegotiationStatusCollection(values);
            }

            return new NegotiationStatusCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator NegotiationStatus[](NegotiationStatusCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (NegotiationStatusCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            NegotiationStatusCollection clone = new NegotiationStatusCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((NegotiationStatus)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region TsnFailureCode Enumeration
    #if (!OPCUA_EXCLUDE_TsnFailureCode)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum TsnFailureCode
    {
        /// <remarks />
        [EnumMember(Value = "NoFailure_0")]
        NoFailure = 0,

        /// <remarks />
        [EnumMember(Value = "InsufficientBandwidth_1")]
        InsufficientBandwidth = 1,

        /// <remarks />
        [EnumMember(Value = "InsufficientResources_2")]
        InsufficientResources = 2,

        /// <remarks />
        [EnumMember(Value = "InsufficientTrafficClassBandwidth_3")]
        InsufficientTrafficClassBandwidth = 3,

        /// <remarks />
        [EnumMember(Value = "StreamIdInUse_4")]
        StreamIdInUse = 4,

        /// <remarks />
        [EnumMember(Value = "StreamDestinationAddressInUse_5")]
        StreamDestinationAddressInUse = 5,

        /// <remarks />
        [EnumMember(Value = "StreamPreemptedByHigherRank_6")]
        StreamPreemptedByHigherRank = 6,

        /// <remarks />
        [EnumMember(Value = "LatencyHasChanged_7")]
        LatencyHasChanged = 7,

        /// <remarks />
        [EnumMember(Value = "EgressPortNotAvbCapable_8")]
        EgressPortNotAvbCapable = 8,

        /// <remarks />
        [EnumMember(Value = "UseDifferentDestinationAddress_9")]
        UseDifferentDestinationAddress = 9,

        /// <remarks />
        [EnumMember(Value = "OutOfMsrpResources_10")]
        OutOfMsrpResources = 10,

        /// <remarks />
        [EnumMember(Value = "OutOfMmrpResources_11")]
        OutOfMmrpResources = 11,

        /// <remarks />
        [EnumMember(Value = "CannotStoreDestinationAddress_12")]
        CannotStoreDestinationAddress = 12,

        /// <remarks />
        [EnumMember(Value = "PriorityIsNotAnSrcClass_13")]
        PriorityIsNotAnSrcClass = 13,

        /// <remarks />
        [EnumMember(Value = "MaxFrameSizeTooLarge_14")]
        MaxFrameSizeTooLarge = 14,

        /// <remarks />
        [EnumMember(Value = "MaxFanInPortsLimitReached_15")]
        MaxFanInPortsLimitReached = 15,

        /// <remarks />
        [EnumMember(Value = "FirstValueChangedForStreamId_16")]
        FirstValueChangedForStreamId = 16,

        /// <remarks />
        [EnumMember(Value = "VlanBlockedOnEgress_17")]
        VlanBlockedOnEgress = 17,

        /// <remarks />
        [EnumMember(Value = "VlanTaggingDisabledOnEgress_18")]
        VlanTaggingDisabledOnEgress = 18,

        /// <remarks />
        [EnumMember(Value = "SrClassPriorityMismatch_19")]
        SrClassPriorityMismatch = 19,

        /// <remarks />
        [EnumMember(Value = "FeatureNotPropagated_20")]
        FeatureNotPropagated = 20,

        /// <remarks />
        [EnumMember(Value = "MaxLatencyExceeded_21")]
        MaxLatencyExceeded = 21,

        /// <remarks />
        [EnumMember(Value = "BridgeDoesNotProvideNetworkId_22")]
        BridgeDoesNotProvideNetworkId = 22,

        /// <remarks />
        [EnumMember(Value = "StreamTransformNotSupported_23")]
        StreamTransformNotSupported = 23,

        /// <remarks />
        [EnumMember(Value = "StreamIdTypeNotSupported_24")]
        StreamIdTypeNotSupported = 24,

        /// <remarks />
        [EnumMember(Value = "FeatureNotSupported_25")]
        FeatureNotSupported = 25,
    }

    #region TsnFailureCodeCollection Class
    /// <summary>
    /// A collection of TsnFailureCode objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfTsnFailureCode", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "TsnFailureCode")]
    #if !NET_STANDARD
    public partial class TsnFailureCodeCollection : List<TsnFailureCode>, ICloneable
    #else
    public partial class TsnFailureCodeCollection : List<TsnFailureCode>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public TsnFailureCodeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public TsnFailureCodeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public TsnFailureCodeCollection(IEnumerable<TsnFailureCode> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator TsnFailureCodeCollection(TsnFailureCode[] values)
        {
            if (values != null)
            {
                return new TsnFailureCodeCollection(values);
            }

            return new TsnFailureCodeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator TsnFailureCode[](TsnFailureCodeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (TsnFailureCodeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TsnFailureCodeCollection clone = new TsnFailureCodeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((TsnFailureCode)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region TsnStreamState Enumeration
    #if (!OPCUA_EXCLUDE_TsnStreamState)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum TsnStreamState
    {
        /// <remarks />
        [EnumMember(Value = "Disabled_0")]
        Disabled = 0,

        /// <remarks />
        [EnumMember(Value = "Configuring_1")]
        Configuring = 1,

        /// <remarks />
        [EnumMember(Value = "Ready_2")]
        Ready = 2,

        /// <remarks />
        [EnumMember(Value = "Operational_3")]
        Operational = 3,

        /// <remarks />
        [EnumMember(Value = "Error_4")]
        Error = 4,
    }

    #region TsnStreamStateCollection Class
    /// <summary>
    /// A collection of TsnStreamState objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfTsnStreamState", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "TsnStreamState")]
    #if !NET_STANDARD
    public partial class TsnStreamStateCollection : List<TsnStreamState>, ICloneable
    #else
    public partial class TsnStreamStateCollection : List<TsnStreamState>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public TsnStreamStateCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public TsnStreamStateCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public TsnStreamStateCollection(IEnumerable<TsnStreamState> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator TsnStreamStateCollection(TsnStreamState[] values)
        {
            if (values != null)
            {
                return new TsnStreamStateCollection(values);
            }

            return new TsnStreamStateCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator TsnStreamState[](TsnStreamStateCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (TsnStreamStateCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TsnStreamStateCollection clone = new TsnStreamStateCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((TsnStreamState)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region TsnTalkerStatus Enumeration
    #if (!OPCUA_EXCLUDE_TsnTalkerStatus)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum TsnTalkerStatus
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "Ready_1")]
        Ready = 1,

        /// <remarks />
        [EnumMember(Value = "Failed_2")]
        Failed = 2,
    }

    #region TsnTalkerStatusCollection Class
    /// <summary>
    /// A collection of TsnTalkerStatus objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfTsnTalkerStatus", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "TsnTalkerStatus")]
    #if !NET_STANDARD
    public partial class TsnTalkerStatusCollection : List<TsnTalkerStatus>, ICloneable
    #else
    public partial class TsnTalkerStatusCollection : List<TsnTalkerStatus>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public TsnTalkerStatusCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public TsnTalkerStatusCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public TsnTalkerStatusCollection(IEnumerable<TsnTalkerStatus> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator TsnTalkerStatusCollection(TsnTalkerStatus[] values)
        {
            if (values != null)
            {
                return new TsnTalkerStatusCollection(values);
            }

            return new TsnTalkerStatusCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator TsnTalkerStatus[](TsnTalkerStatusCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (TsnTalkerStatusCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TsnTalkerStatusCollection clone = new TsnTalkerStatusCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((TsnTalkerStatus)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region TsnListenerStatus Enumeration
    #if (!OPCUA_EXCLUDE_TsnListenerStatus)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum TsnListenerStatus
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "Ready_1")]
        Ready = 1,

        /// <remarks />
        [EnumMember(Value = "PartialFailed_2")]
        PartialFailed = 2,

        /// <remarks />
        [EnumMember(Value = "Failed_3")]
        Failed = 3,
    }

    #region TsnListenerStatusCollection Class
    /// <summary>
    /// A collection of TsnListenerStatus objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfTsnListenerStatus", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "TsnListenerStatus")]
    #if !NET_STANDARD
    public partial class TsnListenerStatusCollection : List<TsnListenerStatus>, ICloneable
    #else
    public partial class TsnListenerStatusCollection : List<TsnListenerStatus>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public TsnListenerStatusCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public TsnListenerStatusCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public TsnListenerStatusCollection(IEnumerable<TsnListenerStatus> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator TsnListenerStatusCollection(TsnListenerStatus[] values)
        {
            if (values != null)
            {
                return new TsnListenerStatusCollection(values);
            }

            return new TsnListenerStatusCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator TsnListenerStatus[](TsnListenerStatusCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (TsnListenerStatusCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TsnListenerStatusCollection clone = new TsnListenerStatusCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((TsnListenerStatus)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region UnsignedRationalNumber Class
    #if (!OPCUA_EXCLUDE_UnsignedRationalNumber)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class UnsignedRationalNumber : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public UnsignedRationalNumber()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_numerator = (uint)0;
            m_denominator = (uint)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Numerator", IsRequired = false, Order = 1)]
        public uint Numerator
        {
            get { return m_numerator;  }
            set { m_numerator = value; }
        }

        /// <remarks />
        [DataMember(Name = "Denominator", IsRequired = false, Order = 2)]
        public uint Denominator
        {
            get { return m_denominator;  }
            set { m_denominator = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.UnsignedRationalNumber; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.UnsignedRationalNumber_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.UnsignedRationalNumber_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("Numerator", Numerator);
            encoder.WriteUInt32("Denominator", Denominator);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Numerator = decoder.ReadUInt32("Numerator");
            Denominator = decoder.ReadUInt32("Denominator");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            UnsignedRationalNumber value = encodeable as UnsignedRationalNumber;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_numerator, value.m_numerator)) return false;
            if (!Utils.IsEqual(m_denominator, value.m_denominator)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (UnsignedRationalNumber)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UnsignedRationalNumber clone = (UnsignedRationalNumber)base.MemberwiseClone();

            clone.m_numerator = (uint)Utils.Clone(this.m_numerator);
            clone.m_denominator = (uint)Utils.Clone(this.m_denominator);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_numerator;
        private uint m_denominator;
        #endregion
    }

    #region UnsignedRationalNumberCollection Class
    /// <summary>
    /// A collection of UnsignedRationalNumber objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUnsignedRationalNumber", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "UnsignedRationalNumber")]
    #if !NET_STANDARD
    public partial class UnsignedRationalNumberCollection : List<UnsignedRationalNumber>, ICloneable
    #else
    public partial class UnsignedRationalNumberCollection : List<UnsignedRationalNumber>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public UnsignedRationalNumberCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public UnsignedRationalNumberCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public UnsignedRationalNumberCollection(IEnumerable<UnsignedRationalNumber> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator UnsignedRationalNumberCollection(UnsignedRationalNumber[] values)
        {
            if (values != null)
            {
                return new UnsignedRationalNumberCollection(values);
            }

            return new UnsignedRationalNumberCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator UnsignedRationalNumber[](UnsignedRationalNumberCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (UnsignedRationalNumberCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UnsignedRationalNumberCollection clone = new UnsignedRationalNumberCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((UnsignedRationalNumber)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region IdType Enumeration
    #if (!OPCUA_EXCLUDE_IdType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum IdType
    {
        /// <remarks />
        [EnumMember(Value = "Numeric_0")]
        Numeric = 0,

        /// <remarks />
        [EnumMember(Value = "String_1")]
        String = 1,

        /// <remarks />
        [EnumMember(Value = "Guid_2")]
        Guid = 2,

        /// <remarks />
        [EnumMember(Value = "Opaque_3")]
        Opaque = 3,
    }

    #region IdTypeCollection Class
    /// <summary>
    /// A collection of IdType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfIdType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "IdType")]
    #if !NET_STANDARD
    public partial class IdTypeCollection : List<IdType>, ICloneable
    #else
    public partial class IdTypeCollection : List<IdType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public IdTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public IdTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public IdTypeCollection(IEnumerable<IdType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator IdTypeCollection(IdType[] values)
        {
            if (values != null)
            {
                return new IdTypeCollection(values);
            }

            return new IdTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator IdType[](IdTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (IdTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            IdTypeCollection clone = new IdTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((IdType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region NodeClass Enumeration
    #if (!OPCUA_EXCLUDE_NodeClass)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum NodeClass
    {
        /// <remarks />
        [EnumMember(Value = "Unspecified_0")]
        Unspecified = 0,

        /// <remarks />
        [EnumMember(Value = "Object_1")]
        Object = 1,

        /// <remarks />
        [EnumMember(Value = "Variable_2")]
        Variable = 2,

        /// <remarks />
        [EnumMember(Value = "Method_4")]
        Method = 4,

        /// <remarks />
        [EnumMember(Value = "ObjectType_8")]
        ObjectType = 8,

        /// <remarks />
        [EnumMember(Value = "VariableType_16")]
        VariableType = 16,

        /// <remarks />
        [EnumMember(Value = "ReferenceType_32")]
        ReferenceType = 32,

        /// <remarks />
        [EnumMember(Value = "DataType_64")]
        DataType = 64,

        /// <remarks />
        [EnumMember(Value = "View_128")]
        View = 128,
    }
    #endif
    #endregion

    #region PermissionType Enumeration
    #if (!OPCUA_EXCLUDE_PermissionType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum PermissionType : UInt32
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "Browse_1")]
        Browse = 1,

        /// <remarks />
        [EnumMember(Value = "ReadRolePermissions_2")]
        ReadRolePermissions = 2,

        /// <remarks />
        [EnumMember(Value = "WriteAttribute_4")]
        WriteAttribute = 4,

        /// <remarks />
        [EnumMember(Value = "WriteRolePermissions_8")]
        WriteRolePermissions = 8,

        /// <remarks />
        [EnumMember(Value = "WriteHistorizing_16")]
        WriteHistorizing = 16,

        /// <remarks />
        [EnumMember(Value = "Read_32")]
        Read = 32,

        /// <remarks />
        [EnumMember(Value = "Write_64")]
        Write = 64,

        /// <remarks />
        [EnumMember(Value = "ReadHistory_128")]
        ReadHistory = 128,

        /// <remarks />
        [EnumMember(Value = "InsertHistory_256")]
        InsertHistory = 256,

        /// <remarks />
        [EnumMember(Value = "ModifyHistory_512")]
        ModifyHistory = 512,

        /// <remarks />
        [EnumMember(Value = "DeleteHistory_1024")]
        DeleteHistory = 1024,

        /// <remarks />
        [EnumMember(Value = "ReceiveEvents_2048")]
        ReceiveEvents = 2048,

        /// <remarks />
        [EnumMember(Value = "Call_4096")]
        Call = 4096,

        /// <remarks />
        [EnumMember(Value = "AddReference_8192")]
        AddReference = 8192,

        /// <remarks />
        [EnumMember(Value = "RemoveReference_16384")]
        RemoveReference = 16384,

        /// <remarks />
        [EnumMember(Value = "DeleteNode_32768")]
        DeleteNode = 32768,

        /// <remarks />
        [EnumMember(Value = "AddNode_65536")]
        AddNode = 65536,
    }
    #endif
    #endregion

    #region AccessLevelType Enumeration
    #if (!OPCUA_EXCLUDE_AccessLevelType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum AccessLevelType : Byte
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "CurrentRead_1")]
        CurrentRead = 1,

        /// <remarks />
        [EnumMember(Value = "CurrentWrite_2")]
        CurrentWrite = 2,

        /// <remarks />
        [EnumMember(Value = "HistoryRead_4")]
        HistoryRead = 4,

        /// <remarks />
        [EnumMember(Value = "HistoryWrite_8")]
        HistoryWrite = 8,

        /// <remarks />
        [EnumMember(Value = "SemanticChange_16")]
        SemanticChange = 16,

        /// <remarks />
        [EnumMember(Value = "StatusWrite_32")]
        StatusWrite = 32,

        /// <remarks />
        [EnumMember(Value = "TimestampWrite_64")]
        TimestampWrite = 64,
    }
    #endif
    #endregion

    #region AccessLevelExType Enumeration
    #if (!OPCUA_EXCLUDE_AccessLevelExType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum AccessLevelExType : UInt32
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "CurrentRead_1")]
        CurrentRead = 1,

        /// <remarks />
        [EnumMember(Value = "CurrentWrite_2")]
        CurrentWrite = 2,

        /// <remarks />
        [EnumMember(Value = "HistoryRead_4")]
        HistoryRead = 4,

        /// <remarks />
        [EnumMember(Value = "HistoryWrite_8")]
        HistoryWrite = 8,

        /// <remarks />
        [EnumMember(Value = "SemanticChange_16")]
        SemanticChange = 16,

        /// <remarks />
        [EnumMember(Value = "StatusWrite_32")]
        StatusWrite = 32,

        /// <remarks />
        [EnumMember(Value = "TimestampWrite_64")]
        TimestampWrite = 64,

        /// <remarks />
        [EnumMember(Value = "NonatomicRead_256")]
        NonatomicRead = 256,

        /// <remarks />
        [EnumMember(Value = "NonatomicWrite_512")]
        NonatomicWrite = 512,

        /// <remarks />
        [EnumMember(Value = "WriteFullArrayOnly_1024")]
        WriteFullArrayOnly = 1024,

        /// <remarks />
        [EnumMember(Value = "NoSubDataTypes_2048")]
        NoSubDataTypes = 2048,
    }
    #endif
    #endregion

    #region EventNotifierType Enumeration
    #if (!OPCUA_EXCLUDE_EventNotifierType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum EventNotifierType : Byte
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "SubscribeToEvents_1")]
        SubscribeToEvents = 1,

        /// <remarks />
        [EnumMember(Value = "HistoryRead_4")]
        HistoryRead = 4,

        /// <remarks />
        [EnumMember(Value = "HistoryWrite_8")]
        HistoryWrite = 8,
    }
    #endif
    #endregion

    #region AccessRestrictionType Enumeration
    #if (!OPCUA_EXCLUDE_AccessRestrictionType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum AccessRestrictionType : UInt16
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "SigningRequired_1")]
        SigningRequired = 1,

        /// <remarks />
        [EnumMember(Value = "EncryptionRequired_2")]
        EncryptionRequired = 2,

        /// <remarks />
        [EnumMember(Value = "SessionRequired_4")]
        SessionRequired = 4,

        /// <remarks />
        [EnumMember(Value = "ApplyRestrictionsToBrowse_8")]
        ApplyRestrictionsToBrowse = 8,
    }
    #endif
    #endregion

    #region RolePermissionType Class
    #if (!OPCUA_EXCLUDE_RolePermissionType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class RolePermissionType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public RolePermissionType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_roleId = null;
            m_permissions = 0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "RoleId", IsRequired = false, Order = 1)]
        public NodeId RoleId
        {
            get { return m_roleId;  }
            set { m_roleId = value; }
        }

        /// <remarks />
        [DataMember(Name = "Permissions", IsRequired = false, Order = 2)]
        public uint Permissions
        {
            get { return m_permissions;  }
            set { m_permissions = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.RolePermissionType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.RolePermissionType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.RolePermissionType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("RoleId", RoleId);
            encoder.WriteUInt32("Permissions", Permissions);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            RoleId = decoder.ReadNodeId("RoleId");
            Permissions = decoder.ReadUInt32("Permissions");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            RolePermissionType value = encodeable as RolePermissionType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_roleId, value.m_roleId)) return false;
            if (!Utils.IsEqual(m_permissions, value.m_permissions)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (RolePermissionType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            RolePermissionType clone = (RolePermissionType)base.MemberwiseClone();

            clone.m_roleId = (NodeId)Utils.Clone(this.m_roleId);
            clone.m_permissions = (uint)Utils.Clone(this.m_permissions);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_roleId;
        private uint m_permissions;
        #endregion
    }

    #region RolePermissionTypeCollection Class
    /// <summary>
    /// A collection of RolePermissionType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfRolePermissionType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "RolePermissionType")]
    #if !NET_STANDARD
    public partial class RolePermissionTypeCollection : List<RolePermissionType>, ICloneable
    #else
    public partial class RolePermissionTypeCollection : List<RolePermissionType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public RolePermissionTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public RolePermissionTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public RolePermissionTypeCollection(IEnumerable<RolePermissionType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator RolePermissionTypeCollection(RolePermissionType[] values)
        {
            if (values != null)
            {
                return new RolePermissionTypeCollection(values);
            }

            return new RolePermissionTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator RolePermissionType[](RolePermissionTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (RolePermissionTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            RolePermissionTypeCollection clone = new RolePermissionTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((RolePermissionType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataTypeDefinition Class
    #if (!OPCUA_EXCLUDE_DataTypeDefinition)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataTypeDefinition : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public DataTypeDefinition()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.DataTypeDefinition; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.DataTypeDefinition_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.DataTypeDefinition_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataTypeDefinition value = encodeable as DataTypeDefinition;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DataTypeDefinition)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataTypeDefinition clone = (DataTypeDefinition)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region DataTypeDefinitionCollection Class
    /// <summary>
    /// A collection of DataTypeDefinition objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataTypeDefinition", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataTypeDefinition")]
    #if !NET_STANDARD
    public partial class DataTypeDefinitionCollection : List<DataTypeDefinition>, ICloneable
    #else
    public partial class DataTypeDefinitionCollection : List<DataTypeDefinition>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public DataTypeDefinitionCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public DataTypeDefinitionCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public DataTypeDefinitionCollection(IEnumerable<DataTypeDefinition> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator DataTypeDefinitionCollection(DataTypeDefinition[] values)
        {
            if (values != null)
            {
                return new DataTypeDefinitionCollection(values);
            }

            return new DataTypeDefinitionCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator DataTypeDefinition[](DataTypeDefinitionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (DataTypeDefinitionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataTypeDefinitionCollection clone = new DataTypeDefinitionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataTypeDefinition)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region StructureType Enumeration
    #if (!OPCUA_EXCLUDE_StructureType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum StructureType
    {
        /// <remarks />
        [EnumMember(Value = "Structure_0")]
        Structure = 0,

        /// <remarks />
        [EnumMember(Value = "StructureWithOptionalFields_1")]
        StructureWithOptionalFields = 1,

        /// <remarks />
        [EnumMember(Value = "Union_2")]
        Union = 2,

        /// <remarks />
        [EnumMember(Value = "StructureWithSubtypedValues_3")]
        StructureWithSubtypedValues = 3,

        /// <remarks />
        [EnumMember(Value = "UnionWithSubtypedValues_4")]
        UnionWithSubtypedValues = 4,
    }
    #endif
    #endregion

    #region StructureField Class
    #if (!OPCUA_EXCLUDE_StructureField)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class StructureField : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public StructureField()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_name = null;
            m_description = null;
            m_dataType = null;
            m_valueRank = (int)0;
            m_arrayDimensions = new UInt32Collection();
            m_maxStringLength = (uint)0;
            m_isOptional = false;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "Description", IsRequired = false, Order = 2)]
        public LocalizedText Description
        {
            get { return m_description;  }
            set { m_description = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataType", IsRequired = false, Order = 3)]
        public NodeId DataType
        {
            get { return m_dataType;  }
            set { m_dataType = value; }
        }

        /// <remarks />
        [DataMember(Name = "ValueRank", IsRequired = false, Order = 4)]
        public int ValueRank
        {
            get { return m_valueRank;  }
            set { m_valueRank = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ArrayDimensions", IsRequired = false, Order = 5)]
        public UInt32Collection ArrayDimensions
        {
            get
            {
                return m_arrayDimensions;
            }

            set
            {
                m_arrayDimensions = value;

                if (value == null)
                {
                    m_arrayDimensions = new UInt32Collection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "MaxStringLength", IsRequired = false, Order = 6)]
        public uint MaxStringLength
        {
            get { return m_maxStringLength;  }
            set { m_maxStringLength = value; }
        }

        /// <remarks />
        [DataMember(Name = "IsOptional", IsRequired = false, Order = 7)]
        public bool IsOptional
        {
            get { return m_isOptional;  }
            set { m_isOptional = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.StructureField; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.StructureField_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.StructureField_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteLocalizedText("Description", Description);
            encoder.WriteNodeId("DataType", DataType);
            encoder.WriteInt32("ValueRank", ValueRank);
            encoder.WriteUInt32Array("ArrayDimensions", ArrayDimensions);
            encoder.WriteUInt32("MaxStringLength", MaxStringLength);
            encoder.WriteBoolean("IsOptional", IsOptional);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            Description = decoder.ReadLocalizedText("Description");
            DataType = decoder.ReadNodeId("DataType");
            ValueRank = decoder.ReadInt32("ValueRank");
            ArrayDimensions = decoder.ReadUInt32Array("ArrayDimensions");
            MaxStringLength = decoder.ReadUInt32("MaxStringLength");
            IsOptional = decoder.ReadBoolean("IsOptional");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            StructureField value = encodeable as StructureField;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_description, value.m_description)) return false;
            if (!Utils.IsEqual(m_dataType, value.m_dataType)) return false;
            if (!Utils.IsEqual(m_valueRank, value.m_valueRank)) return false;
            if (!Utils.IsEqual(m_arrayDimensions, value.m_arrayDimensions)) return false;
            if (!Utils.IsEqual(m_maxStringLength, value.m_maxStringLength)) return false;
            if (!Utils.IsEqual(m_isOptional, value.m_isOptional)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (StructureField)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            StructureField clone = (StructureField)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_description = (LocalizedText)Utils.Clone(this.m_description);
            clone.m_dataType = (NodeId)Utils.Clone(this.m_dataType);
            clone.m_valueRank = (int)Utils.Clone(this.m_valueRank);
            clone.m_arrayDimensions = (UInt32Collection)Utils.Clone(this.m_arrayDimensions);
            clone.m_maxStringLength = (uint)Utils.Clone(this.m_maxStringLength);
            clone.m_isOptional = (bool)Utils.Clone(this.m_isOptional);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private LocalizedText m_description;
        private NodeId m_dataType;
        private int m_valueRank;
        private UInt32Collection m_arrayDimensions;
        private uint m_maxStringLength;
        private bool m_isOptional;
        #endregion
    }

    #region StructureFieldCollection Class
    /// <summary>
    /// A collection of StructureField objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfStructureField", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "StructureField")]
    #if !NET_STANDARD
    public partial class StructureFieldCollection : List<StructureField>, ICloneable
    #else
    public partial class StructureFieldCollection : List<StructureField>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public StructureFieldCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public StructureFieldCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public StructureFieldCollection(IEnumerable<StructureField> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator StructureFieldCollection(StructureField[] values)
        {
            if (values != null)
            {
                return new StructureFieldCollection(values);
            }

            return new StructureFieldCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator StructureField[](StructureFieldCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (StructureFieldCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            StructureFieldCollection clone = new StructureFieldCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((StructureField)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region StructureDefinition Class
    #if (!OPCUA_EXCLUDE_StructureDefinition)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class StructureDefinition : DataTypeDefinition
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public StructureDefinition()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_defaultEncodingId = null;
            m_baseDataType = null;
            m_structureType = StructureType.Structure;
            m_fields = new StructureFieldCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "DefaultEncodingId", IsRequired = false, Order = 1)]
        public NodeId DefaultEncodingId
        {
            get { return m_defaultEncodingId;  }
            set { m_defaultEncodingId = value; }
        }

        /// <remarks />
        [DataMember(Name = "BaseDataType", IsRequired = false, Order = 2)]
        public NodeId BaseDataType
        {
            get { return m_baseDataType;  }
            set { m_baseDataType = value; }
        }

        /// <remarks />
        [DataMember(Name = "StructureType", IsRequired = false, Order = 3)]
        public StructureType StructureType
        {
            get { return m_structureType;  }
            set { m_structureType = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "Fields", IsRequired = false, Order = 4)]
        public StructureFieldCollection Fields
        {
            get
            {
                return m_fields;
            }

            set
            {
                m_fields = value;

                if (value == null)
                {
                    m_fields = new StructureFieldCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.StructureDefinition; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.StructureDefinition_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.StructureDefinition_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("DefaultEncodingId", DefaultEncodingId);
            encoder.WriteNodeId("BaseDataType", BaseDataType);
            encoder.WriteEnumerated("StructureType", StructureType);
            encoder.WriteEncodeableArray("Fields", Fields.ToArray(), typeof(StructureField));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            DefaultEncodingId = decoder.ReadNodeId("DefaultEncodingId");
            BaseDataType = decoder.ReadNodeId("BaseDataType");
            StructureType = (StructureType)decoder.ReadEnumerated("StructureType", typeof(StructureType));
            Fields = (StructureFieldCollection)decoder.ReadEncodeableArray("Fields", typeof(StructureField));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            StructureDefinition value = encodeable as StructureDefinition;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_defaultEncodingId, value.m_defaultEncodingId)) return false;
            if (!Utils.IsEqual(m_baseDataType, value.m_baseDataType)) return false;
            if (!Utils.IsEqual(m_structureType, value.m_structureType)) return false;
            if (!Utils.IsEqual(m_fields, value.m_fields)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (StructureDefinition)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            StructureDefinition clone = (StructureDefinition)base.MemberwiseClone();

            clone.m_defaultEncodingId = (NodeId)Utils.Clone(this.m_defaultEncodingId);
            clone.m_baseDataType = (NodeId)Utils.Clone(this.m_baseDataType);
            clone.m_structureType = (StructureType)Utils.Clone(this.m_structureType);
            clone.m_fields = (StructureFieldCollection)Utils.Clone(this.m_fields);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_defaultEncodingId;
        private NodeId m_baseDataType;
        private StructureType m_structureType;
        private StructureFieldCollection m_fields;
        #endregion
    }

    #region StructureDefinitionCollection Class
    /// <summary>
    /// A collection of StructureDefinition objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfStructureDefinition", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "StructureDefinition")]
    #if !NET_STANDARD
    public partial class StructureDefinitionCollection : List<StructureDefinition>, ICloneable
    #else
    public partial class StructureDefinitionCollection : List<StructureDefinition>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public StructureDefinitionCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public StructureDefinitionCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public StructureDefinitionCollection(IEnumerable<StructureDefinition> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator StructureDefinitionCollection(StructureDefinition[] values)
        {
            if (values != null)
            {
                return new StructureDefinitionCollection(values);
            }

            return new StructureDefinitionCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator StructureDefinition[](StructureDefinitionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (StructureDefinitionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            StructureDefinitionCollection clone = new StructureDefinitionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((StructureDefinition)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region EnumDefinition Class
    #if (!OPCUA_EXCLUDE_EnumDefinition)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class EnumDefinition : DataTypeDefinition
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public EnumDefinition()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_fields = new EnumFieldCollection();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "Fields", IsRequired = false, Order = 1)]
        public EnumFieldCollection Fields
        {
            get
            {
                return m_fields;
            }

            set
            {
                m_fields = value;

                if (value == null)
                {
                    m_fields = new EnumFieldCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.EnumDefinition; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.EnumDefinition_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.EnumDefinition_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeableArray("Fields", Fields.ToArray(), typeof(EnumField));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Fields = (EnumFieldCollection)decoder.ReadEncodeableArray("Fields", typeof(EnumField));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            EnumDefinition value = encodeable as EnumDefinition;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_fields, value.m_fields)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (EnumDefinition)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EnumDefinition clone = (EnumDefinition)base.MemberwiseClone();

            clone.m_fields = (EnumFieldCollection)Utils.Clone(this.m_fields);

            return clone;
        }
        #endregion

        #region Private Fields
        private EnumFieldCollection m_fields;
        #endregion
    }

    #region EnumDefinitionCollection Class
    /// <summary>
    /// A collection of EnumDefinition objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfEnumDefinition", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "EnumDefinition")]
    #if !NET_STANDARD
    public partial class EnumDefinitionCollection : List<EnumDefinition>, ICloneable
    #else
    public partial class EnumDefinitionCollection : List<EnumDefinition>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public EnumDefinitionCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public EnumDefinitionCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public EnumDefinitionCollection(IEnumerable<EnumDefinition> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator EnumDefinitionCollection(EnumDefinition[] values)
        {
            if (values != null)
            {
                return new EnumDefinitionCollection(values);
            }

            return new EnumDefinitionCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator EnumDefinition[](EnumDefinitionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (EnumDefinitionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EnumDefinitionCollection clone = new EnumDefinitionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((EnumDefinition)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region Node Class
    #if (!OPCUA_EXCLUDE_Node)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class Node : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public Node()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_nodeId = null;
            m_nodeClass = NodeClass.Unspecified;
            m_browseName = null;
            m_displayName = null;
            m_description = null;
            m_writeMask = (uint)0;
            m_userWriteMask = (uint)0;
            m_rolePermissions = new RolePermissionTypeCollection();
            m_userRolePermissions = new RolePermissionTypeCollection();
            m_accessRestrictions = (ushort)0;
            m_references = new ReferenceNodeCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "NodeId", IsRequired = false, Order = 1)]
        public NodeId NodeId
        {
            get { return m_nodeId;  }
            set { m_nodeId = value; }
        }

        /// <remarks />
        [DataMember(Name = "NodeClass", IsRequired = false, Order = 2)]
        public NodeClass NodeClass
        {
            get { return m_nodeClass;  }
            set { m_nodeClass = value; }
        }

        /// <remarks />
        [DataMember(Name = "BrowseName", IsRequired = false, Order = 3)]
        public QualifiedName BrowseName
        {
            get { return m_browseName;  }
            set { m_browseName = value; }
        }

        /// <remarks />
        [DataMember(Name = "DisplayName", IsRequired = false, Order = 4)]
        public LocalizedText DisplayName
        {
            get { return m_displayName;  }
            set { m_displayName = value; }
        }

        /// <remarks />
        [DataMember(Name = "Description", IsRequired = false, Order = 5)]
        public LocalizedText Description
        {
            get { return m_description;  }
            set { m_description = value; }
        }

        /// <remarks />
        [DataMember(Name = "WriteMask", IsRequired = false, Order = 6)]
        public uint WriteMask
        {
            get { return m_writeMask;  }
            set { m_writeMask = value; }
        }

        /// <remarks />
        [DataMember(Name = "UserWriteMask", IsRequired = false, Order = 7)]
        public uint UserWriteMask
        {
            get { return m_userWriteMask;  }
            set { m_userWriteMask = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "RolePermissions", IsRequired = false, Order = 8)]
        public RolePermissionTypeCollection RolePermissions
        {
            get
            {
                return m_rolePermissions;
            }

            set
            {
                m_rolePermissions = value;

                if (value == null)
                {
                    m_rolePermissions = new RolePermissionTypeCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "UserRolePermissions", IsRequired = false, Order = 9)]
        public RolePermissionTypeCollection UserRolePermissions
        {
            get
            {
                return m_userRolePermissions;
            }

            set
            {
                m_userRolePermissions = value;

                if (value == null)
                {
                    m_userRolePermissions = new RolePermissionTypeCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "AccessRestrictions", IsRequired = false, Order = 10)]
        public ushort AccessRestrictions
        {
            get { return m_accessRestrictions;  }
            set { m_accessRestrictions = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "References", IsRequired = false, Order = 11)]
        public ReferenceNodeCollection References
        {
            get
            {
                return m_references;
            }

            set
            {
                m_references = value;

                if (value == null)
                {
                    m_references = new ReferenceNodeCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.Node; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.Node_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.Node_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("NodeId", NodeId);
            encoder.WriteEnumerated("NodeClass", NodeClass);
            encoder.WriteQualifiedName("BrowseName", BrowseName);
            encoder.WriteLocalizedText("DisplayName", DisplayName);
            encoder.WriteLocalizedText("Description", Description);
            encoder.WriteUInt32("WriteMask", WriteMask);
            encoder.WriteUInt32("UserWriteMask", UserWriteMask);
            encoder.WriteEncodeableArray("RolePermissions", RolePermissions.ToArray(), typeof(RolePermissionType));
            encoder.WriteEncodeableArray("UserRolePermissions", UserRolePermissions.ToArray(), typeof(RolePermissionType));
            encoder.WriteUInt16("AccessRestrictions", AccessRestrictions);
            encoder.WriteEncodeableArray("References", References.ToArray(), typeof(ReferenceNode));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            NodeId = decoder.ReadNodeId("NodeId");
            NodeClass = (NodeClass)decoder.ReadEnumerated("NodeClass", typeof(NodeClass));
            BrowseName = decoder.ReadQualifiedName("BrowseName");
            DisplayName = decoder.ReadLocalizedText("DisplayName");
            Description = decoder.ReadLocalizedText("Description");
            WriteMask = decoder.ReadUInt32("WriteMask");
            UserWriteMask = decoder.ReadUInt32("UserWriteMask");
            RolePermissions = (RolePermissionTypeCollection)decoder.ReadEncodeableArray("RolePermissions", typeof(RolePermissionType));
            UserRolePermissions = (RolePermissionTypeCollection)decoder.ReadEncodeableArray("UserRolePermissions", typeof(RolePermissionType));
            AccessRestrictions = decoder.ReadUInt16("AccessRestrictions");
            References = (ReferenceNodeCollection)decoder.ReadEncodeableArray("References", typeof(ReferenceNode));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            Node value = encodeable as Node;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_nodeId, value.m_nodeId)) return false;
            if (!Utils.IsEqual(m_nodeClass, value.m_nodeClass)) return false;
            if (!Utils.IsEqual(m_browseName, value.m_browseName)) return false;
            if (!Utils.IsEqual(m_displayName, value.m_displayName)) return false;
            if (!Utils.IsEqual(m_description, value.m_description)) return false;
            if (!Utils.IsEqual(m_writeMask, value.m_writeMask)) return false;
            if (!Utils.IsEqual(m_userWriteMask, value.m_userWriteMask)) return false;
            if (!Utils.IsEqual(m_rolePermissions, value.m_rolePermissions)) return false;
            if (!Utils.IsEqual(m_userRolePermissions, value.m_userRolePermissions)) return false;
            if (!Utils.IsEqual(m_accessRestrictions, value.m_accessRestrictions)) return false;
            if (!Utils.IsEqual(m_references, value.m_references)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (Node)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            Node clone = (Node)base.MemberwiseClone();

            clone.m_nodeId = (NodeId)Utils.Clone(this.m_nodeId);
            clone.m_nodeClass = (NodeClass)Utils.Clone(this.m_nodeClass);
            clone.m_browseName = (QualifiedName)Utils.Clone(this.m_browseName);
            clone.m_displayName = (LocalizedText)Utils.Clone(this.m_displayName);
            clone.m_description = (LocalizedText)Utils.Clone(this.m_description);
            clone.m_writeMask = (uint)Utils.Clone(this.m_writeMask);
            clone.m_userWriteMask = (uint)Utils.Clone(this.m_userWriteMask);
            clone.m_rolePermissions = (RolePermissionTypeCollection)Utils.Clone(this.m_rolePermissions);
            clone.m_userRolePermissions = (RolePermissionTypeCollection)Utils.Clone(this.m_userRolePermissions);
            clone.m_accessRestrictions = (ushort)Utils.Clone(this.m_accessRestrictions);
            clone.m_references = (ReferenceNodeCollection)Utils.Clone(this.m_references);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_nodeId;
        private NodeClass m_nodeClass;
        private QualifiedName m_browseName;
        private LocalizedText m_displayName;
        private LocalizedText m_description;
        private uint m_writeMask;
        private uint m_userWriteMask;
        private RolePermissionTypeCollection m_rolePermissions;
        private RolePermissionTypeCollection m_userRolePermissions;
        private ushort m_accessRestrictions;
        private ReferenceNodeCollection m_references;
        #endregion
    }

    #region NodeCollection Class
    /// <summary>
    /// A collection of Node objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfNode", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "Node")]
    #if !NET_STANDARD
    public partial class NodeCollection : List<Node>, ICloneable
    #else
    public partial class NodeCollection : List<Node>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public NodeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public NodeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public NodeCollection(IEnumerable<Node> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator NodeCollection(Node[] values)
        {
            if (values != null)
            {
                return new NodeCollection(values);
            }

            return new NodeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator Node[](NodeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (NodeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            NodeCollection clone = new NodeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((Node)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region InstanceNode Class
    #if (!OPCUA_EXCLUDE_InstanceNode)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class InstanceNode : Node
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public InstanceNode()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.InstanceNode; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.InstanceNode_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.InstanceNode_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            InstanceNode value = encodeable as InstanceNode;

            if (value == null)
            {
                return false;
            }


            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (InstanceNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            InstanceNode clone = (InstanceNode)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region TypeNode Class
    #if (!OPCUA_EXCLUDE_TypeNode)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class TypeNode : Node
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public TypeNode()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.TypeNode; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.TypeNode_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.TypeNode_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            TypeNode value = encodeable as TypeNode;

            if (value == null)
            {
                return false;
            }


            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (TypeNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TypeNode clone = (TypeNode)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region ObjectNode Class
    #if (!OPCUA_EXCLUDE_ObjectNode)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ObjectNode : InstanceNode
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ObjectNode()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_eventNotifier = (byte)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "EventNotifier", IsRequired = false, Order = 1)]
        public byte EventNotifier
        {
            get { return m_eventNotifier;  }
            set { m_eventNotifier = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ObjectNode; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ObjectNode_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ObjectNode_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteByte("EventNotifier", EventNotifier);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            EventNotifier = decoder.ReadByte("EventNotifier");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ObjectNode value = encodeable as ObjectNode;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_eventNotifier, value.m_eventNotifier)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ObjectNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ObjectNode clone = (ObjectNode)base.MemberwiseClone();

            clone.m_eventNotifier = (byte)Utils.Clone(this.m_eventNotifier);

            return clone;
        }
        #endregion

        #region Private Fields
        private byte m_eventNotifier;
        #endregion
    }
    #endif
    #endregion

    #region ObjectTypeNode Class
    #if (!OPCUA_EXCLUDE_ObjectTypeNode)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ObjectTypeNode : TypeNode
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ObjectTypeNode()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_isAbstract = true;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "IsAbstract", IsRequired = false, Order = 1)]
        public bool IsAbstract
        {
            get { return m_isAbstract;  }
            set { m_isAbstract = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ObjectTypeNode; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ObjectTypeNode_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ObjectTypeNode_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteBoolean("IsAbstract", IsAbstract);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            IsAbstract = decoder.ReadBoolean("IsAbstract");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ObjectTypeNode value = encodeable as ObjectTypeNode;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_isAbstract, value.m_isAbstract)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ObjectTypeNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ObjectTypeNode clone = (ObjectTypeNode)base.MemberwiseClone();

            clone.m_isAbstract = (bool)Utils.Clone(this.m_isAbstract);

            return clone;
        }
        #endregion

        #region Private Fields
        private bool m_isAbstract;
        #endregion
    }
    #endif
    #endregion

    #region VariableNode Class
    #if (!OPCUA_EXCLUDE_VariableNode)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class VariableNode : InstanceNode
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public VariableNode()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_value = Variant.Null;
            m_dataType = null;
            m_valueRank = (int)0;
            m_arrayDimensions = new UInt32Collection();
            m_accessLevel = (byte)0;
            m_userAccessLevel = (byte)0;
            m_minimumSamplingInterval = (double)0;
            m_historizing = true;
            m_accessLevelEx = (uint)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Value", IsRequired = false, Order = 1)]
        public Variant Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataType", IsRequired = false, Order = 2)]
        public NodeId DataType
        {
            get { return m_dataType;  }
            set { m_dataType = value; }
        }

        /// <remarks />
        [DataMember(Name = "ValueRank", IsRequired = false, Order = 3)]
        public int ValueRank
        {
            get { return m_valueRank;  }
            set { m_valueRank = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ArrayDimensions", IsRequired = false, Order = 4)]
        public UInt32Collection ArrayDimensions
        {
            get
            {
                return m_arrayDimensions;
            }

            set
            {
                m_arrayDimensions = value;

                if (value == null)
                {
                    m_arrayDimensions = new UInt32Collection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "AccessLevel", IsRequired = false, Order = 5)]
        public byte AccessLevel
        {
            get { return m_accessLevel;  }
            set { m_accessLevel = value; }
        }

        /// <remarks />
        [DataMember(Name = "UserAccessLevel", IsRequired = false, Order = 6)]
        public byte UserAccessLevel
        {
            get { return m_userAccessLevel;  }
            set { m_userAccessLevel = value; }
        }

        /// <remarks />
        [DataMember(Name = "MinimumSamplingInterval", IsRequired = false, Order = 7)]
        public double MinimumSamplingInterval
        {
            get { return m_minimumSamplingInterval;  }
            set { m_minimumSamplingInterval = value; }
        }

        /// <remarks />
        [DataMember(Name = "Historizing", IsRequired = false, Order = 8)]
        public bool Historizing
        {
            get { return m_historizing;  }
            set { m_historizing = value; }
        }

        /// <remarks />
        [DataMember(Name = "AccessLevelEx", IsRequired = false, Order = 9)]
        public uint AccessLevelEx
        {
            get { return m_accessLevelEx;  }
            set { m_accessLevelEx = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.VariableNode; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.VariableNode_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.VariableNode_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteVariant("Value", Value);
            encoder.WriteNodeId("DataType", DataType);
            encoder.WriteInt32("ValueRank", ValueRank);
            encoder.WriteUInt32Array("ArrayDimensions", ArrayDimensions);
            encoder.WriteByte("AccessLevel", AccessLevel);
            encoder.WriteByte("UserAccessLevel", UserAccessLevel);
            encoder.WriteDouble("MinimumSamplingInterval", MinimumSamplingInterval);
            encoder.WriteBoolean("Historizing", Historizing);
            encoder.WriteUInt32("AccessLevelEx", AccessLevelEx);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Value = decoder.ReadVariant("Value");
            DataType = decoder.ReadNodeId("DataType");
            ValueRank = decoder.ReadInt32("ValueRank");
            ArrayDimensions = decoder.ReadUInt32Array("ArrayDimensions");
            AccessLevel = decoder.ReadByte("AccessLevel");
            UserAccessLevel = decoder.ReadByte("UserAccessLevel");
            MinimumSamplingInterval = decoder.ReadDouble("MinimumSamplingInterval");
            Historizing = decoder.ReadBoolean("Historizing");
            AccessLevelEx = decoder.ReadUInt32("AccessLevelEx");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            VariableNode value = encodeable as VariableNode;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_value, value.m_value)) return false;
            if (!Utils.IsEqual(m_dataType, value.m_dataType)) return false;
            if (!Utils.IsEqual(m_valueRank, value.m_valueRank)) return false;
            if (!Utils.IsEqual(m_arrayDimensions, value.m_arrayDimensions)) return false;
            if (!Utils.IsEqual(m_accessLevel, value.m_accessLevel)) return false;
            if (!Utils.IsEqual(m_userAccessLevel, value.m_userAccessLevel)) return false;
            if (!Utils.IsEqual(m_minimumSamplingInterval, value.m_minimumSamplingInterval)) return false;
            if (!Utils.IsEqual(m_historizing, value.m_historizing)) return false;
            if (!Utils.IsEqual(m_accessLevelEx, value.m_accessLevelEx)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (VariableNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            VariableNode clone = (VariableNode)base.MemberwiseClone();

            clone.m_value = (Variant)Utils.Clone(this.m_value);
            clone.m_dataType = (NodeId)Utils.Clone(this.m_dataType);
            clone.m_valueRank = (int)Utils.Clone(this.m_valueRank);
            clone.m_arrayDimensions = (UInt32Collection)Utils.Clone(this.m_arrayDimensions);
            clone.m_accessLevel = (byte)Utils.Clone(this.m_accessLevel);
            clone.m_userAccessLevel = (byte)Utils.Clone(this.m_userAccessLevel);
            clone.m_minimumSamplingInterval = (double)Utils.Clone(this.m_minimumSamplingInterval);
            clone.m_historizing = (bool)Utils.Clone(this.m_historizing);
            clone.m_accessLevelEx = (uint)Utils.Clone(this.m_accessLevelEx);

            return clone;
        }
        #endregion

        #region Private Fields
        private Variant m_value;
        private NodeId m_dataType;
        private int m_valueRank;
        private UInt32Collection m_arrayDimensions;
        private byte m_accessLevel;
        private byte m_userAccessLevel;
        private double m_minimumSamplingInterval;
        private bool m_historizing;
        private uint m_accessLevelEx;
        #endregion
    }
    #endif
    #endregion

    #region VariableTypeNode Class
    #if (!OPCUA_EXCLUDE_VariableTypeNode)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class VariableTypeNode : TypeNode
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public VariableTypeNode()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_value = Variant.Null;
            m_dataType = null;
            m_valueRank = (int)0;
            m_arrayDimensions = new UInt32Collection();
            m_isAbstract = true;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Value", IsRequired = false, Order = 1)]
        public Variant Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataType", IsRequired = false, Order = 2)]
        public NodeId DataType
        {
            get { return m_dataType;  }
            set { m_dataType = value; }
        }

        /// <remarks />
        [DataMember(Name = "ValueRank", IsRequired = false, Order = 3)]
        public int ValueRank
        {
            get { return m_valueRank;  }
            set { m_valueRank = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ArrayDimensions", IsRequired = false, Order = 4)]
        public UInt32Collection ArrayDimensions
        {
            get
            {
                return m_arrayDimensions;
            }

            set
            {
                m_arrayDimensions = value;

                if (value == null)
                {
                    m_arrayDimensions = new UInt32Collection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "IsAbstract", IsRequired = false, Order = 5)]
        public bool IsAbstract
        {
            get { return m_isAbstract;  }
            set { m_isAbstract = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.VariableTypeNode; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.VariableTypeNode_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.VariableTypeNode_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteVariant("Value", Value);
            encoder.WriteNodeId("DataType", DataType);
            encoder.WriteInt32("ValueRank", ValueRank);
            encoder.WriteUInt32Array("ArrayDimensions", ArrayDimensions);
            encoder.WriteBoolean("IsAbstract", IsAbstract);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Value = decoder.ReadVariant("Value");
            DataType = decoder.ReadNodeId("DataType");
            ValueRank = decoder.ReadInt32("ValueRank");
            ArrayDimensions = decoder.ReadUInt32Array("ArrayDimensions");
            IsAbstract = decoder.ReadBoolean("IsAbstract");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            VariableTypeNode value = encodeable as VariableTypeNode;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_value, value.m_value)) return false;
            if (!Utils.IsEqual(m_dataType, value.m_dataType)) return false;
            if (!Utils.IsEqual(m_valueRank, value.m_valueRank)) return false;
            if (!Utils.IsEqual(m_arrayDimensions, value.m_arrayDimensions)) return false;
            if (!Utils.IsEqual(m_isAbstract, value.m_isAbstract)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (VariableTypeNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            VariableTypeNode clone = (VariableTypeNode)base.MemberwiseClone();

            clone.m_value = (Variant)Utils.Clone(this.m_value);
            clone.m_dataType = (NodeId)Utils.Clone(this.m_dataType);
            clone.m_valueRank = (int)Utils.Clone(this.m_valueRank);
            clone.m_arrayDimensions = (UInt32Collection)Utils.Clone(this.m_arrayDimensions);
            clone.m_isAbstract = (bool)Utils.Clone(this.m_isAbstract);

            return clone;
        }
        #endregion

        #region Private Fields
        private Variant m_value;
        private NodeId m_dataType;
        private int m_valueRank;
        private UInt32Collection m_arrayDimensions;
        private bool m_isAbstract;
        #endregion
    }
    #endif
    #endregion

    #region ReferenceTypeNode Class
    #if (!OPCUA_EXCLUDE_ReferenceTypeNode)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ReferenceTypeNode : TypeNode
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ReferenceTypeNode()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_isAbstract = true;
            m_symmetric = true;
            m_inverseName = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "IsAbstract", IsRequired = false, Order = 1)]
        public bool IsAbstract
        {
            get { return m_isAbstract;  }
            set { m_isAbstract = value; }
        }

        /// <remarks />
        [DataMember(Name = "Symmetric", IsRequired = false, Order = 2)]
        public bool Symmetric
        {
            get { return m_symmetric;  }
            set { m_symmetric = value; }
        }

        /// <remarks />
        [DataMember(Name = "InverseName", IsRequired = false, Order = 3)]
        public LocalizedText InverseName
        {
            get { return m_inverseName;  }
            set { m_inverseName = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ReferenceTypeNode; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ReferenceTypeNode_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ReferenceTypeNode_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteBoolean("IsAbstract", IsAbstract);
            encoder.WriteBoolean("Symmetric", Symmetric);
            encoder.WriteLocalizedText("InverseName", InverseName);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            IsAbstract = decoder.ReadBoolean("IsAbstract");
            Symmetric = decoder.ReadBoolean("Symmetric");
            InverseName = decoder.ReadLocalizedText("InverseName");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ReferenceTypeNode value = encodeable as ReferenceTypeNode;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_isAbstract, value.m_isAbstract)) return false;
            if (!Utils.IsEqual(m_symmetric, value.m_symmetric)) return false;
            if (!Utils.IsEqual(m_inverseName, value.m_inverseName)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ReferenceTypeNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReferenceTypeNode clone = (ReferenceTypeNode)base.MemberwiseClone();

            clone.m_isAbstract = (bool)Utils.Clone(this.m_isAbstract);
            clone.m_symmetric = (bool)Utils.Clone(this.m_symmetric);
            clone.m_inverseName = (LocalizedText)Utils.Clone(this.m_inverseName);

            return clone;
        }
        #endregion

        #region Private Fields
        private bool m_isAbstract;
        private bool m_symmetric;
        private LocalizedText m_inverseName;
        #endregion
    }
    #endif
    #endregion

    #region MethodNode Class
    #if (!OPCUA_EXCLUDE_MethodNode)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class MethodNode : InstanceNode
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public MethodNode()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_executable = true;
            m_userExecutable = true;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Executable", IsRequired = false, Order = 1)]
        public bool Executable
        {
            get { return m_executable;  }
            set { m_executable = value; }
        }

        /// <remarks />
        [DataMember(Name = "UserExecutable", IsRequired = false, Order = 2)]
        public bool UserExecutable
        {
            get { return m_userExecutable;  }
            set { m_userExecutable = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.MethodNode; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.MethodNode_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.MethodNode_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteBoolean("Executable", Executable);
            encoder.WriteBoolean("UserExecutable", UserExecutable);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Executable = decoder.ReadBoolean("Executable");
            UserExecutable = decoder.ReadBoolean("UserExecutable");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            MethodNode value = encodeable as MethodNode;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_executable, value.m_executable)) return false;
            if (!Utils.IsEqual(m_userExecutable, value.m_userExecutable)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (MethodNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            MethodNode clone = (MethodNode)base.MemberwiseClone();

            clone.m_executable = (bool)Utils.Clone(this.m_executable);
            clone.m_userExecutable = (bool)Utils.Clone(this.m_userExecutable);

            return clone;
        }
        #endregion

        #region Private Fields
        private bool m_executable;
        private bool m_userExecutable;
        #endregion
    }
    #endif
    #endregion

    #region ViewNode Class
    #if (!OPCUA_EXCLUDE_ViewNode)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ViewNode : InstanceNode
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ViewNode()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_containsNoLoops = true;
            m_eventNotifier = (byte)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ContainsNoLoops", IsRequired = false, Order = 1)]
        public bool ContainsNoLoops
        {
            get { return m_containsNoLoops;  }
            set { m_containsNoLoops = value; }
        }

        /// <remarks />
        [DataMember(Name = "EventNotifier", IsRequired = false, Order = 2)]
        public byte EventNotifier
        {
            get { return m_eventNotifier;  }
            set { m_eventNotifier = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ViewNode; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ViewNode_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ViewNode_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteBoolean("ContainsNoLoops", ContainsNoLoops);
            encoder.WriteByte("EventNotifier", EventNotifier);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ContainsNoLoops = decoder.ReadBoolean("ContainsNoLoops");
            EventNotifier = decoder.ReadByte("EventNotifier");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ViewNode value = encodeable as ViewNode;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_containsNoLoops, value.m_containsNoLoops)) return false;
            if (!Utils.IsEqual(m_eventNotifier, value.m_eventNotifier)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ViewNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ViewNode clone = (ViewNode)base.MemberwiseClone();

            clone.m_containsNoLoops = (bool)Utils.Clone(this.m_containsNoLoops);
            clone.m_eventNotifier = (byte)Utils.Clone(this.m_eventNotifier);

            return clone;
        }
        #endregion

        #region Private Fields
        private bool m_containsNoLoops;
        private byte m_eventNotifier;
        #endregion
    }
    #endif
    #endregion

    #region DataTypeNode Class
    #if (!OPCUA_EXCLUDE_DataTypeNode)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataTypeNode : TypeNode
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public DataTypeNode()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_isAbstract = true;
            m_dataTypeDefinition = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "IsAbstract", IsRequired = false, Order = 1)]
        public bool IsAbstract
        {
            get { return m_isAbstract;  }
            set { m_isAbstract = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataTypeDefinition", IsRequired = false, Order = 2)]
        public ExtensionObject DataTypeDefinition
        {
            get { return m_dataTypeDefinition;  }
            set { m_dataTypeDefinition = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.DataTypeNode; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.DataTypeNode_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.DataTypeNode_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteBoolean("IsAbstract", IsAbstract);
            encoder.WriteExtensionObject("DataTypeDefinition", DataTypeDefinition);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            IsAbstract = decoder.ReadBoolean("IsAbstract");
            DataTypeDefinition = decoder.ReadExtensionObject("DataTypeDefinition");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataTypeNode value = encodeable as DataTypeNode;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_isAbstract, value.m_isAbstract)) return false;
            if (!Utils.IsEqual(m_dataTypeDefinition, value.m_dataTypeDefinition)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (DataTypeNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataTypeNode clone = (DataTypeNode)base.MemberwiseClone();

            clone.m_isAbstract = (bool)Utils.Clone(this.m_isAbstract);
            clone.m_dataTypeDefinition = (ExtensionObject)Utils.Clone(this.m_dataTypeDefinition);

            return clone;
        }
        #endregion

        #region Private Fields
        private bool m_isAbstract;
        private ExtensionObject m_dataTypeDefinition;
        #endregion
    }
    #endif
    #endregion

    #region ReferenceNode Class
    #if (!OPCUA_EXCLUDE_ReferenceNode)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ReferenceNode : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ReferenceNode()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_referenceTypeId = null;
            m_isInverse = true;
            m_targetId = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ReferenceTypeId", IsRequired = false, Order = 1)]
        public NodeId ReferenceTypeId
        {
            get { return m_referenceTypeId;  }
            set { m_referenceTypeId = value; }
        }

        /// <remarks />
        [DataMember(Name = "IsInverse", IsRequired = false, Order = 2)]
        public bool IsInverse
        {
            get { return m_isInverse;  }
            set { m_isInverse = value; }
        }

        /// <remarks />
        [DataMember(Name = "TargetId", IsRequired = false, Order = 3)]
        public ExpandedNodeId TargetId
        {
            get { return m_targetId;  }
            set { m_targetId = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ReferenceNode; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ReferenceNode_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ReferenceNode_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("ReferenceTypeId", ReferenceTypeId);
            encoder.WriteBoolean("IsInverse", IsInverse);
            encoder.WriteExpandedNodeId("TargetId", TargetId);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ReferenceTypeId = decoder.ReadNodeId("ReferenceTypeId");
            IsInverse = decoder.ReadBoolean("IsInverse");
            TargetId = decoder.ReadExpandedNodeId("TargetId");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ReferenceNode value = encodeable as ReferenceNode;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_referenceTypeId, value.m_referenceTypeId)) return false;
            if (!Utils.IsEqual(m_isInverse, value.m_isInverse)) return false;
            if (!Utils.IsEqual(m_targetId, value.m_targetId)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ReferenceNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReferenceNode clone = (ReferenceNode)base.MemberwiseClone();

            clone.m_referenceTypeId = (NodeId)Utils.Clone(this.m_referenceTypeId);
            clone.m_isInverse = (bool)Utils.Clone(this.m_isInverse);
            clone.m_targetId = (ExpandedNodeId)Utils.Clone(this.m_targetId);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_referenceTypeId;
        private bool m_isInverse;
        private ExpandedNodeId m_targetId;
        #endregion
    }

    #region ReferenceNodeCollection Class
    /// <summary>
    /// A collection of ReferenceNode objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfReferenceNode", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ReferenceNode")]
    #if !NET_STANDARD
    public partial class ReferenceNodeCollection : List<ReferenceNode>, ICloneable
    #else
    public partial class ReferenceNodeCollection : List<ReferenceNode>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public ReferenceNodeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public ReferenceNodeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public ReferenceNodeCollection(IEnumerable<ReferenceNode> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator ReferenceNodeCollection(ReferenceNode[] values)
        {
            if (values != null)
            {
                return new ReferenceNodeCollection(values);
            }

            return new ReferenceNodeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator ReferenceNode[](ReferenceNodeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (ReferenceNodeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReferenceNodeCollection clone = new ReferenceNodeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ReferenceNode)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region Argument Class
    #if (!OPCUA_EXCLUDE_Argument)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class Argument : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public Argument()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_name = null;
            m_dataType = null;
            m_valueRank = (int)0;
            m_arrayDimensions = new UInt32Collection();
            m_description = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataType", IsRequired = false, Order = 2)]
        public NodeId DataType
        {
            get { return m_dataType;  }
            set { m_dataType = value; }
        }

        /// <remarks />
        [DataMember(Name = "ValueRank", IsRequired = false, Order = 3)]
        public int ValueRank
        {
            get { return m_valueRank;  }
            set { m_valueRank = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ArrayDimensions", IsRequired = false, Order = 4)]
        public UInt32Collection ArrayDimensions
        {
            get
            {
                return m_arrayDimensions;
            }

            set
            {
                m_arrayDimensions = value;

                if (value == null)
                {
                    m_arrayDimensions = new UInt32Collection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "Description", IsRequired = false, Order = 5)]
        public LocalizedText Description
        {
            get { return m_description;  }
            set { m_description = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.Argument; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.Argument_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.Argument_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteNodeId("DataType", DataType);
            encoder.WriteInt32("ValueRank", ValueRank);
            encoder.WriteUInt32Array("ArrayDimensions", ArrayDimensions);
            encoder.WriteLocalizedText("Description", Description);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            DataType = decoder.ReadNodeId("DataType");
            ValueRank = decoder.ReadInt32("ValueRank");
            ArrayDimensions = decoder.ReadUInt32Array("ArrayDimensions");
            Description = decoder.ReadLocalizedText("Description");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            Argument value = encodeable as Argument;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_dataType, value.m_dataType)) return false;
            if (!Utils.IsEqual(m_valueRank, value.m_valueRank)) return false;
            if (!Utils.IsEqual(m_arrayDimensions, value.m_arrayDimensions)) return false;
            if (!Utils.IsEqual(m_description, value.m_description)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (Argument)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            Argument clone = (Argument)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_dataType = (NodeId)Utils.Clone(this.m_dataType);
            clone.m_valueRank = (int)Utils.Clone(this.m_valueRank);
            clone.m_arrayDimensions = (UInt32Collection)Utils.Clone(this.m_arrayDimensions);
            clone.m_description = (LocalizedText)Utils.Clone(this.m_description);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private NodeId m_dataType;
        private int m_valueRank;
        private UInt32Collection m_arrayDimensions;
        private LocalizedText m_description;
        #endregion
    }

    #region ArgumentCollection Class
    /// <summary>
    /// A collection of Argument objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfArgument", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "Argument")]
    #if !NET_STANDARD
    public partial class ArgumentCollection : List<Argument>, ICloneable
    #else
    public partial class ArgumentCollection : List<Argument>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public ArgumentCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public ArgumentCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public ArgumentCollection(IEnumerable<Argument> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator ArgumentCollection(Argument[] values)
        {
            if (values != null)
            {
                return new ArgumentCollection(values);
            }

            return new ArgumentCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator Argument[](ArgumentCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (ArgumentCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ArgumentCollection clone = new ArgumentCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((Argument)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region EnumValueType Class
    #if (!OPCUA_EXCLUDE_EnumValueType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class EnumValueType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public EnumValueType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_value = (long)0;
            m_displayName = null;
            m_description = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Value", IsRequired = false, Order = 1)]
        public long Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }

        /// <remarks />
        [DataMember(Name = "DisplayName", IsRequired = false, Order = 2)]
        public LocalizedText DisplayName
        {
            get { return m_displayName;  }
            set { m_displayName = value; }
        }

        /// <remarks />
        [DataMember(Name = "Description", IsRequired = false, Order = 3)]
        public LocalizedText Description
        {
            get { return m_description;  }
            set { m_description = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.EnumValueType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.EnumValueType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.EnumValueType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteInt64("Value", Value);
            encoder.WriteLocalizedText("DisplayName", DisplayName);
            encoder.WriteLocalizedText("Description", Description);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Value = decoder.ReadInt64("Value");
            DisplayName = decoder.ReadLocalizedText("DisplayName");
            Description = decoder.ReadLocalizedText("Description");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            EnumValueType value = encodeable as EnumValueType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_value, value.m_value)) return false;
            if (!Utils.IsEqual(m_displayName, value.m_displayName)) return false;
            if (!Utils.IsEqual(m_description, value.m_description)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (EnumValueType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EnumValueType clone = (EnumValueType)base.MemberwiseClone();

            clone.m_value = (long)Utils.Clone(this.m_value);
            clone.m_displayName = (LocalizedText)Utils.Clone(this.m_displayName);
            clone.m_description = (LocalizedText)Utils.Clone(this.m_description);

            return clone;
        }
        #endregion

        #region Private Fields
        private long m_value;
        private LocalizedText m_displayName;
        private LocalizedText m_description;
        #endregion
    }

    #region EnumValueTypeCollection Class
    /// <summary>
    /// A collection of EnumValueType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfEnumValueType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "EnumValueType")]
    #if !NET_STANDARD
    public partial class EnumValueTypeCollection : List<EnumValueType>, ICloneable
    #else
    public partial class EnumValueTypeCollection : List<EnumValueType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public EnumValueTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public EnumValueTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public EnumValueTypeCollection(IEnumerable<EnumValueType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator EnumValueTypeCollection(EnumValueType[] values)
        {
            if (values != null)
            {
                return new EnumValueTypeCollection(values);
            }

            return new EnumValueTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator EnumValueType[](EnumValueTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (EnumValueTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EnumValueTypeCollection clone = new EnumValueTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((EnumValueType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region EnumField Class
    #if (!OPCUA_EXCLUDE_EnumField)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class EnumField : EnumValueType
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public EnumField()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_name = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.EnumField; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.EnumField_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.EnumField_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            EnumField value = encodeable as EnumField;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_name, value.m_name)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (EnumField)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EnumField clone = (EnumField)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        #endregion
    }

    #region EnumFieldCollection Class
    /// <summary>
    /// A collection of EnumField objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfEnumField", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "EnumField")]
    #if !NET_STANDARD
    public partial class EnumFieldCollection : List<EnumField>, ICloneable
    #else
    public partial class EnumFieldCollection : List<EnumField>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public EnumFieldCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public EnumFieldCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public EnumFieldCollection(IEnumerable<EnumField> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator EnumFieldCollection(EnumField[] values)
        {
            if (values != null)
            {
                return new EnumFieldCollection(values);
            }

            return new EnumFieldCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator EnumField[](EnumFieldCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (EnumFieldCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EnumFieldCollection clone = new EnumFieldCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((EnumField)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region OptionSet Class
    #if (!OPCUA_EXCLUDE_OptionSet)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class OptionSet : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public OptionSet()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_value = null;
            m_validBits = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Value", IsRequired = false, Order = 1)]
        public byte[] Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }

        /// <remarks />
        [DataMember(Name = "ValidBits", IsRequired = false, Order = 2)]
        public byte[] ValidBits
        {
            get { return m_validBits;  }
            set { m_validBits = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.OptionSet; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.OptionSet_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.OptionSet_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteByteString("Value", Value);
            encoder.WriteByteString("ValidBits", ValidBits);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Value = decoder.ReadByteString("Value");
            ValidBits = decoder.ReadByteString("ValidBits");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            OptionSet value = encodeable as OptionSet;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_value, value.m_value)) return false;
            if (!Utils.IsEqual(m_validBits, value.m_validBits)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (OptionSet)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            OptionSet clone = (OptionSet)base.MemberwiseClone();

            clone.m_value = (byte[])Utils.Clone(this.m_value);
            clone.m_validBits = (byte[])Utils.Clone(this.m_validBits);

            return clone;
        }
        #endregion

        #region Private Fields
        private byte[] m_value;
        private byte[] m_validBits;
        #endregion
    }

    #region OptionSetCollection Class
    /// <summary>
    /// A collection of OptionSet objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfOptionSet", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "OptionSet")]
    #if !NET_STANDARD
    public partial class OptionSetCollection : List<OptionSet>, ICloneable
    #else
    public partial class OptionSetCollection : List<OptionSet>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public OptionSetCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public OptionSetCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public OptionSetCollection(IEnumerable<OptionSet> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator OptionSetCollection(OptionSet[] values)
        {
            if (values != null)
            {
                return new OptionSetCollection(values);
            }

            return new OptionSetCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator OptionSet[](OptionSetCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (OptionSetCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            OptionSetCollection clone = new OptionSetCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((OptionSet)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region TimeZoneDataType Class
    #if (!OPCUA_EXCLUDE_TimeZoneDataType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class TimeZoneDataType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public TimeZoneDataType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_offset = (short)0;
            m_daylightSavingInOffset = true;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Offset", IsRequired = false, Order = 1)]
        public short Offset
        {
            get { return m_offset;  }
            set { m_offset = value; }
        }

        /// <remarks />
        [DataMember(Name = "DaylightSavingInOffset", IsRequired = false, Order = 2)]
        public bool DaylightSavingInOffset
        {
            get { return m_daylightSavingInOffset;  }
            set { m_daylightSavingInOffset = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.TimeZoneDataType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.TimeZoneDataType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.TimeZoneDataType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteInt16("Offset", Offset);
            encoder.WriteBoolean("DaylightSavingInOffset", DaylightSavingInOffset);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Offset = decoder.ReadInt16("Offset");
            DaylightSavingInOffset = decoder.ReadBoolean("DaylightSavingInOffset");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            TimeZoneDataType value = encodeable as TimeZoneDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_offset, value.m_offset)) return false;
            if (!Utils.IsEqual(m_daylightSavingInOffset, value.m_daylightSavingInOffset)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (TimeZoneDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TimeZoneDataType clone = (TimeZoneDataType)base.MemberwiseClone();

            clone.m_offset = (short)Utils.Clone(this.m_offset);
            clone.m_daylightSavingInOffset = (bool)Utils.Clone(this.m_daylightSavingInOffset);

            return clone;
        }
        #endregion

        #region Private Fields
        private short m_offset;
        private bool m_daylightSavingInOffset;
        #endregion
    }

    #region TimeZoneDataTypeCollection Class
    /// <summary>
    /// A collection of TimeZoneDataType objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfTimeZoneDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "TimeZoneDataType")]
    #if !NET_STANDARD
    public partial class TimeZoneDataTypeCollection : List<TimeZoneDataType>, ICloneable
    #else
    public partial class TimeZoneDataTypeCollection : List<TimeZoneDataType>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public TimeZoneDataTypeCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public TimeZoneDataTypeCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public TimeZoneDataTypeCollection(IEnumerable<TimeZoneDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator TimeZoneDataTypeCollection(TimeZoneDataType[] values)
        {
            if (values != null)
            {
                return new TimeZoneDataTypeCollection(values);
            }

            return new TimeZoneDataTypeCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator TimeZoneDataType[](TimeZoneDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (TimeZoneDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TimeZoneDataTypeCollection clone = new TimeZoneDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((TimeZoneDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ApplicationType Enumeration
    #if (!OPCUA_EXCLUDE_ApplicationType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum ApplicationType
    {
        /// <remarks />
        [EnumMember(Value = "Server_0")]
        Server = 0,

        /// <remarks />
        [EnumMember(Value = "Client_1")]
        Client = 1,

        /// <remarks />
        [EnumMember(Value = "ClientAndServer_2")]
        ClientAndServer = 2,

        /// <remarks />
        [EnumMember(Value = "DiscoveryServer_3")]
        DiscoveryServer = 3,
    }
    #endif
    #endregion

    #region ApplicationDescription Class
    #if (!OPCUA_EXCLUDE_ApplicationDescription)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ApplicationDescription : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ApplicationDescription()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_applicationUri = null;
            m_productUri = null;
            m_applicationName = null;
            m_applicationType = ApplicationType.Server;
            m_gatewayServerUri = null;
            m_discoveryProfileUri = null;
            m_discoveryUrls = new StringCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ApplicationUri", IsRequired = false, Order = 1)]
        public string ApplicationUri
        {
            get { return m_applicationUri;  }
            set { m_applicationUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "ProductUri", IsRequired = false, Order = 2)]
        public string ProductUri
        {
            get { return m_productUri;  }
            set { m_productUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "ApplicationName", IsRequired = false, Order = 3)]
        public LocalizedText ApplicationName
        {
            get { return m_applicationName;  }
            set { m_applicationName = value; }
        }

        /// <remarks />
        [DataMember(Name = "ApplicationType", IsRequired = false, Order = 4)]
        public ApplicationType ApplicationType
        {
            get { return m_applicationType;  }
            set { m_applicationType = value; }
        }

        /// <remarks />
        [DataMember(Name = "GatewayServerUri", IsRequired = false, Order = 5)]
        public string GatewayServerUri
        {
            get { return m_gatewayServerUri;  }
            set { m_gatewayServerUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "DiscoveryProfileUri", IsRequired = false, Order = 6)]
        public string DiscoveryProfileUri
        {
            get { return m_discoveryProfileUri;  }
            set { m_discoveryProfileUri = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "DiscoveryUrls", IsRequired = false, Order = 7)]
        public StringCollection DiscoveryUrls
        {
            get
            {
                return m_discoveryUrls;
            }

            set
            {
                m_discoveryUrls = value;

                if (value == null)
                {
                    m_discoveryUrls = new StringCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ApplicationDescription; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ApplicationDescription_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ApplicationDescription_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("ApplicationUri", ApplicationUri);
            encoder.WriteString("ProductUri", ProductUri);
            encoder.WriteLocalizedText("ApplicationName", ApplicationName);
            encoder.WriteEnumerated("ApplicationType", ApplicationType);
            encoder.WriteString("GatewayServerUri", GatewayServerUri);
            encoder.WriteString("DiscoveryProfileUri", DiscoveryProfileUri);
            encoder.WriteStringArray("DiscoveryUrls", DiscoveryUrls);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ApplicationUri = decoder.ReadString("ApplicationUri");
            ProductUri = decoder.ReadString("ProductUri");
            ApplicationName = decoder.ReadLocalizedText("ApplicationName");
            ApplicationType = (ApplicationType)decoder.ReadEnumerated("ApplicationType", typeof(ApplicationType));
            GatewayServerUri = decoder.ReadString("GatewayServerUri");
            DiscoveryProfileUri = decoder.ReadString("DiscoveryProfileUri");
            DiscoveryUrls = decoder.ReadStringArray("DiscoveryUrls");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ApplicationDescription value = encodeable as ApplicationDescription;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_applicationUri, value.m_applicationUri)) return false;
            if (!Utils.IsEqual(m_productUri, value.m_productUri)) return false;
            if (!Utils.IsEqual(m_applicationName, value.m_applicationName)) return false;
            if (!Utils.IsEqual(m_applicationType, value.m_applicationType)) return false;
            if (!Utils.IsEqual(m_gatewayServerUri, value.m_gatewayServerUri)) return false;
            if (!Utils.IsEqual(m_discoveryProfileUri, value.m_discoveryProfileUri)) return false;
            if (!Utils.IsEqual(m_discoveryUrls, value.m_discoveryUrls)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ApplicationDescription)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ApplicationDescription clone = (ApplicationDescription)base.MemberwiseClone();

            clone.m_applicationUri = (string)Utils.Clone(this.m_applicationUri);
            clone.m_productUri = (string)Utils.Clone(this.m_productUri);
            clone.m_applicationName = (LocalizedText)Utils.Clone(this.m_applicationName);
            clone.m_applicationType = (ApplicationType)Utils.Clone(this.m_applicationType);
            clone.m_gatewayServerUri = (string)Utils.Clone(this.m_gatewayServerUri);
            clone.m_discoveryProfileUri = (string)Utils.Clone(this.m_discoveryProfileUri);
            clone.m_discoveryUrls = (StringCollection)Utils.Clone(this.m_discoveryUrls);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_applicationUri;
        private string m_productUri;
        private LocalizedText m_applicationName;
        private ApplicationType m_applicationType;
        private string m_gatewayServerUri;
        private string m_discoveryProfileUri;
        private StringCollection m_discoveryUrls;
        #endregion
    }

    #region ApplicationDescriptionCollection Class
    /// <summary>
    /// A collection of ApplicationDescription objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfApplicationDescription", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ApplicationDescription")]
    #if !NET_STANDARD
    public partial class ApplicationDescriptionCollection : List<ApplicationDescription>, ICloneable
    #else
    public partial class ApplicationDescriptionCollection : List<ApplicationDescription>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public ApplicationDescriptionCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public ApplicationDescriptionCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public ApplicationDescriptionCollection(IEnumerable<ApplicationDescription> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator ApplicationDescriptionCollection(ApplicationDescription[] values)
        {
            if (values != null)
            {
                return new ApplicationDescriptionCollection(values);
            }

            return new ApplicationDescriptionCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator ApplicationDescription[](ApplicationDescriptionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (ApplicationDescriptionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ApplicationDescriptionCollection clone = new ApplicationDescriptionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ApplicationDescription)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region RequestHeader Class
    #if (!OPCUA_EXCLUDE_RequestHeader)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class RequestHeader : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public RequestHeader()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_authenticationToken = null;
            m_timestamp = DateTime.MinValue;
            m_requestHandle = (uint)0;
            m_returnDiagnostics = (uint)0;
            m_auditEntryId = null;
            m_timeoutHint = (uint)0;
            m_additionalHeader = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "AuthenticationToken", IsRequired = false, Order = 1)]
        public NodeId AuthenticationToken
        {
            get { return m_authenticationToken;  }
            set { m_authenticationToken = value; }
        }

        /// <remarks />
        [DataMember(Name = "Timestamp", IsRequired = false, Order = 2)]
        public DateTime Timestamp
        {
            get { return m_timestamp;  }
            set { m_timestamp = value; }
        }

        /// <remarks />
        [DataMember(Name = "RequestHandle", IsRequired = false, Order = 3)]
        public uint RequestHandle
        {
            get { return m_requestHandle;  }
            set { m_requestHandle = value; }
        }

        /// <remarks />
        [DataMember(Name = "ReturnDiagnostics", IsRequired = false, Order = 4)]
        public uint ReturnDiagnostics
        {
            get { return m_returnDiagnostics;  }
            set { m_returnDiagnostics = value; }
        }

        /// <remarks />
        [DataMember(Name = "AuditEntryId", IsRequired = false, Order = 5)]
        public string AuditEntryId
        {
            get { return m_auditEntryId;  }
            set { m_auditEntryId = value; }
        }

        /// <remarks />
        [DataMember(Name = "TimeoutHint", IsRequired = false, Order = 6)]
        public uint TimeoutHint
        {
            get { return m_timeoutHint;  }
            set { m_timeoutHint = value; }
        }

        /// <remarks />
        [DataMember(Name = "AdditionalHeader", IsRequired = false, Order = 7)]
        public ExtensionObject AdditionalHeader
        {
            get { return m_additionalHeader;  }
            set { m_additionalHeader = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.RequestHeader; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.RequestHeader_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.RequestHeader_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("AuthenticationToken", AuthenticationToken);
            encoder.WriteDateTime("Timestamp", Timestamp);
            encoder.WriteUInt32("RequestHandle", RequestHandle);
            encoder.WriteUInt32("ReturnDiagnostics", ReturnDiagnostics);
            encoder.WriteString("AuditEntryId", AuditEntryId);
            encoder.WriteUInt32("TimeoutHint", TimeoutHint);
            encoder.WriteExtensionObject("AdditionalHeader", AdditionalHeader);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            AuthenticationToken = decoder.ReadNodeId("AuthenticationToken");
            Timestamp = decoder.ReadDateTime("Timestamp");
            RequestHandle = decoder.ReadUInt32("RequestHandle");
            ReturnDiagnostics = decoder.ReadUInt32("ReturnDiagnostics");
            AuditEntryId = decoder.ReadString("AuditEntryId");
            TimeoutHint = decoder.ReadUInt32("TimeoutHint");
            AdditionalHeader = decoder.ReadExtensionObject("AdditionalHeader");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            RequestHeader value = encodeable as RequestHeader;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_authenticationToken, value.m_authenticationToken)) return false;
            if (!Utils.IsEqual(m_timestamp, value.m_timestamp)) return false;
            if (!Utils.IsEqual(m_requestHandle, value.m_requestHandle)) return false;
            if (!Utils.IsEqual(m_returnDiagnostics, value.m_returnDiagnostics)) return false;
            if (!Utils.IsEqual(m_auditEntryId, value.m_auditEntryId)) return false;
            if (!Utils.IsEqual(m_timeoutHint, value.m_timeoutHint)) return false;
            if (!Utils.IsEqual(m_additionalHeader, value.m_additionalHeader)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (RequestHeader)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            RequestHeader clone = (RequestHeader)base.MemberwiseClone();

            clone.m_authenticationToken = (NodeId)Utils.Clone(this.m_authenticationToken);
            clone.m_timestamp = (DateTime)Utils.Clone(this.m_timestamp);
            clone.m_requestHandle = (uint)Utils.Clone(this.m_requestHandle);
            clone.m_returnDiagnostics = (uint)Utils.Clone(this.m_returnDiagnostics);
            clone.m_auditEntryId = (string)Utils.Clone(this.m_auditEntryId);
            clone.m_timeoutHint = (uint)Utils.Clone(this.m_timeoutHint);
            clone.m_additionalHeader = (ExtensionObject)Utils.Clone(this.m_additionalHeader);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_authenticationToken;
        private DateTime m_timestamp;
        private uint m_requestHandle;
        private uint m_returnDiagnostics;
        private string m_auditEntryId;
        private uint m_timeoutHint;
        private ExtensionObject m_additionalHeader;
        #endregion
    }
    #endif
    #endregion

    #region ResponseHeader Class
    #if (!OPCUA_EXCLUDE_ResponseHeader)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ResponseHeader : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ResponseHeader()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_timestamp = DateTime.MinValue;
            m_requestHandle = (uint)0;
            m_serviceResult = StatusCodes.Good;
            m_serviceDiagnostics = null;
            m_stringTable = new StringCollection();
            m_additionalHeader = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Timestamp", IsRequired = false, Order = 1)]
        public DateTime Timestamp
        {
            get { return m_timestamp;  }
            set { m_timestamp = value; }
        }

        /// <remarks />
        [DataMember(Name = "RequestHandle", IsRequired = false, Order = 2)]
        public uint RequestHandle
        {
            get { return m_requestHandle;  }
            set { m_requestHandle = value; }
        }

        /// <remarks />
        [DataMember(Name = "ServiceResult", IsRequired = false, Order = 3)]
        public StatusCode ServiceResult
        {
            get { return m_serviceResult;  }
            set { m_serviceResult = value; }
        }

        /// <remarks />
        [DataMember(Name = "ServiceDiagnostics", IsRequired = false, Order = 4)]
        public DiagnosticInfo ServiceDiagnostics
        {
            get { return m_serviceDiagnostics;  }
            set { m_serviceDiagnostics = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "StringTable", IsRequired = false, Order = 5)]
        public StringCollection StringTable
        {
            get
            {
                return m_stringTable;
            }

            set
            {
                m_stringTable = value;

                if (value == null)
                {
                    m_stringTable = new StringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "AdditionalHeader", IsRequired = false, Order = 6)]
        public ExtensionObject AdditionalHeader
        {
            get { return m_additionalHeader;  }
            set { m_additionalHeader = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ResponseHeader; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ResponseHeader_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ResponseHeader_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteDateTime("Timestamp", Timestamp);
            encoder.WriteUInt32("RequestHandle", RequestHandle);
            encoder.WriteStatusCode("ServiceResult", ServiceResult);
            encoder.WriteDiagnosticInfo("ServiceDiagnostics", ServiceDiagnostics);
            encoder.WriteStringArray("StringTable", StringTable);
            encoder.WriteExtensionObject("AdditionalHeader", AdditionalHeader);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Timestamp = decoder.ReadDateTime("Timestamp");
            RequestHandle = decoder.ReadUInt32("RequestHandle");
            ServiceResult = decoder.ReadStatusCode("ServiceResult");
            ServiceDiagnostics = decoder.ReadDiagnosticInfo("ServiceDiagnostics");
            StringTable = decoder.ReadStringArray("StringTable");
            AdditionalHeader = decoder.ReadExtensionObject("AdditionalHeader");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ResponseHeader value = encodeable as ResponseHeader;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_timestamp, value.m_timestamp)) return false;
            if (!Utils.IsEqual(m_requestHandle, value.m_requestHandle)) return false;
            if (!Utils.IsEqual(m_serviceResult, value.m_serviceResult)) return false;
            if (!Utils.IsEqual(m_serviceDiagnostics, value.m_serviceDiagnostics)) return false;
            if (!Utils.IsEqual(m_stringTable, value.m_stringTable)) return false;
            if (!Utils.IsEqual(m_additionalHeader, value.m_additionalHeader)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ResponseHeader)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ResponseHeader clone = (ResponseHeader)base.MemberwiseClone();

            clone.m_timestamp = (DateTime)Utils.Clone(this.m_timestamp);
            clone.m_requestHandle = (uint)Utils.Clone(this.m_requestHandle);
            clone.m_serviceResult = (StatusCode)Utils.Clone(this.m_serviceResult);
            clone.m_serviceDiagnostics = (DiagnosticInfo)Utils.Clone(this.m_serviceDiagnostics);
            clone.m_stringTable = (StringCollection)Utils.Clone(this.m_stringTable);
            clone.m_additionalHeader = (ExtensionObject)Utils.Clone(this.m_additionalHeader);

            return clone;
        }
        #endregion

        #region Private Fields
        private DateTime m_timestamp;
        private uint m_requestHandle;
        private StatusCode m_serviceResult;
        private DiagnosticInfo m_serviceDiagnostics;
        private StringCollection m_stringTable;
        private ExtensionObject m_additionalHeader;
        #endregion
    }
    #endif
    #endregion

    #region ServiceFault Class
    #if (!OPCUA_EXCLUDE_ServiceFault)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ServiceFault : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ServiceFault()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_responseHeader = new ResponseHeader();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ResponseHeader", IsRequired = false, Order = 1)]
        public ResponseHeader ResponseHeader
        {
            get
            {
                return m_responseHeader;
            }

            set
            {
                m_responseHeader = value;

                if (value == null)
                {
                    m_responseHeader = new ResponseHeader();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ServiceFault; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ServiceFault_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ServiceFault_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("ResponseHeader", ResponseHeader, typeof(ResponseHeader));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ResponseHeader = (ResponseHeader)decoder.ReadEncodeable("ResponseHeader", typeof(ResponseHeader));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ServiceFault value = encodeable as ServiceFault;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_responseHeader, value.m_responseHeader)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ServiceFault)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ServiceFault clone = (ServiceFault)base.MemberwiseClone();

            clone.m_responseHeader = (ResponseHeader)Utils.Clone(this.m_responseHeader);

            return clone;
        }
        #endregion

        #region Private Fields
        private ResponseHeader m_responseHeader;
        #endregion
    }
    #endif
    #endregion

    #region SessionlessInvokeRequestType Class
    #if (!OPCUA_EXCLUDE_SessionlessInvokeRequestType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class SessionlessInvokeRequestType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public SessionlessInvokeRequestType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_urisVersion = (uint)0;
            m_namespaceUris = new StringCollection();
            m_serverUris = new StringCollection();
            m_localeIds = new StringCollection();
            m_serviceId = (uint)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "UrisVersion", IsRequired = false, Order = 1)]
        public uint UrisVersion
        {
            get { return m_urisVersion;  }
            set { m_urisVersion = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "NamespaceUris", IsRequired = false, Order = 2)]
        public StringCollection NamespaceUris
        {
            get
            {
                return m_namespaceUris;
            }

            set
            {
                m_namespaceUris = value;

                if (value == null)
                {
                    m_namespaceUris = new StringCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ServerUris", IsRequired = false, Order = 3)]
        public StringCollection ServerUris
        {
            get
            {
                return m_serverUris;
            }

            set
            {
                m_serverUris = value;

                if (value == null)
                {
                    m_serverUris = new StringCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "LocaleIds", IsRequired = false, Order = 4)]
        public StringCollection LocaleIds
        {
            get
            {
                return m_localeIds;
            }

            set
            {
                m_localeIds = value;

                if (value == null)
                {
                    m_localeIds = new StringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "ServiceId", IsRequired = false, Order = 5)]
        public uint ServiceId
        {
            get { return m_serviceId;  }
            set { m_serviceId = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.SessionlessInvokeRequestType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.SessionlessInvokeRequestType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.SessionlessInvokeRequestType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("UrisVersion", UrisVersion);
            encoder.WriteStringArray("NamespaceUris", NamespaceUris);
            encoder.WriteStringArray("ServerUris", ServerUris);
            encoder.WriteStringArray("LocaleIds", LocaleIds);
            encoder.WriteUInt32("ServiceId", ServiceId);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            UrisVersion = decoder.ReadUInt32("UrisVersion");
            NamespaceUris = decoder.ReadStringArray("NamespaceUris");
            ServerUris = decoder.ReadStringArray("ServerUris");
            LocaleIds = decoder.ReadStringArray("LocaleIds");
            ServiceId = decoder.ReadUInt32("ServiceId");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            SessionlessInvokeRequestType value = encodeable as SessionlessInvokeRequestType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_urisVersion, value.m_urisVersion)) return false;
            if (!Utils.IsEqual(m_namespaceUris, value.m_namespaceUris)) return false;
            if (!Utils.IsEqual(m_serverUris, value.m_serverUris)) return false;
            if (!Utils.IsEqual(m_localeIds, value.m_localeIds)) return false;
            if (!Utils.IsEqual(m_serviceId, value.m_serviceId)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (SessionlessInvokeRequestType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SessionlessInvokeRequestType clone = (SessionlessInvokeRequestType)base.MemberwiseClone();

            clone.m_urisVersion = (uint)Utils.Clone(this.m_urisVersion);
            clone.m_namespaceUris = (StringCollection)Utils.Clone(this.m_namespaceUris);
            clone.m_serverUris = (StringCollection)Utils.Clone(this.m_serverUris);
            clone.m_localeIds = (StringCollection)Utils.Clone(this.m_localeIds);
            clone.m_serviceId = (uint)Utils.Clone(this.m_serviceId);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_urisVersion;
        private StringCollection m_namespaceUris;
        private StringCollection m_serverUris;
        private StringCollection m_localeIds;
        private uint m_serviceId;
        #endregion
    }
    #endif
    #endregion

    #region SessionlessInvokeResponseType Class
    #if (!OPCUA_EXCLUDE_SessionlessInvokeResponseType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class SessionlessInvokeResponseType : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public SessionlessInvokeResponseType()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_namespaceUris = new StringCollection();
            m_serverUris = new StringCollection();
            m_serviceId = (uint)0;
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "NamespaceUris", IsRequired = false, Order = 1)]
        public StringCollection NamespaceUris
        {
            get
            {
                return m_namespaceUris;
            }

            set
            {
                m_namespaceUris = value;

                if (value == null)
                {
                    m_namespaceUris = new StringCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ServerUris", IsRequired = false, Order = 2)]
        public StringCollection ServerUris
        {
            get
            {
                return m_serverUris;
            }

            set
            {
                m_serverUris = value;

                if (value == null)
                {
                    m_serverUris = new StringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "ServiceId", IsRequired = false, Order = 3)]
        public uint ServiceId
        {
            get { return m_serviceId;  }
            set { m_serviceId = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.SessionlessInvokeResponseType; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.SessionlessInvokeResponseType_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.SessionlessInvokeResponseType_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteStringArray("NamespaceUris", NamespaceUris);
            encoder.WriteStringArray("ServerUris", ServerUris);
            encoder.WriteUInt32("ServiceId", ServiceId);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            NamespaceUris = decoder.ReadStringArray("NamespaceUris");
            ServerUris = decoder.ReadStringArray("ServerUris");
            ServiceId = decoder.ReadUInt32("ServiceId");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            SessionlessInvokeResponseType value = encodeable as SessionlessInvokeResponseType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_namespaceUris, value.m_namespaceUris)) return false;
            if (!Utils.IsEqual(m_serverUris, value.m_serverUris)) return false;
            if (!Utils.IsEqual(m_serviceId, value.m_serviceId)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (SessionlessInvokeResponseType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SessionlessInvokeResponseType clone = (SessionlessInvokeResponseType)base.MemberwiseClone();

            clone.m_namespaceUris = (StringCollection)Utils.Clone(this.m_namespaceUris);
            clone.m_serverUris = (StringCollection)Utils.Clone(this.m_serverUris);
            clone.m_serviceId = (uint)Utils.Clone(this.m_serviceId);

            return clone;
        }
        #endregion

        #region Private Fields
        private StringCollection m_namespaceUris;
        private StringCollection m_serverUris;
        private uint m_serviceId;
        #endregion
    }
    #endif
    #endregion

    #region FindServersRequest Class
    #if (!OPCUA_EXCLUDE_FindServersRequest)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class FindServersRequest : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public FindServersRequest()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_requestHeader = new RequestHeader();
            m_endpointUrl = null;
            m_localeIds = new StringCollection();
            m_serverUris = new StringCollection();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "RequestHeader", IsRequired = false, Order = 1)]
        public RequestHeader RequestHeader
        {
            get
            {
                return m_requestHeader;
            }

            set
            {
                m_requestHeader = value;

                if (value == null)
                {
                    m_requestHeader = new RequestHeader();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "EndpointUrl", IsRequired = false, Order = 2)]
        public string EndpointUrl
        {
            get { return m_endpointUrl;  }
            set { m_endpointUrl = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "LocaleIds", IsRequired = false, Order = 3)]
        public StringCollection LocaleIds
        {
            get
            {
                return m_localeIds;
            }

            set
            {
                m_localeIds = value;

                if (value == null)
                {
                    m_localeIds = new StringCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ServerUris", IsRequired = false, Order = 4)]
        public StringCollection ServerUris
        {
            get
            {
                return m_serverUris;
            }

            set
            {
                m_serverUris = value;

                if (value == null)
                {
                    m_serverUris = new StringCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.FindServersRequest; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.FindServersRequest_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.FindServersRequest_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("RequestHeader", RequestHeader, typeof(RequestHeader));
            encoder.WriteString("EndpointUrl", EndpointUrl);
            encoder.WriteStringArray("LocaleIds", LocaleIds);
            encoder.WriteStringArray("ServerUris", ServerUris);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            RequestHeader = (RequestHeader)decoder.ReadEncodeable("RequestHeader", typeof(RequestHeader));
            EndpointUrl = decoder.ReadString("EndpointUrl");
            LocaleIds = decoder.ReadStringArray("LocaleIds");
            ServerUris = decoder.ReadStringArray("ServerUris");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            FindServersRequest value = encodeable as FindServersRequest;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_requestHeader, value.m_requestHeader)) return false;
            if (!Utils.IsEqual(m_endpointUrl, value.m_endpointUrl)) return false;
            if (!Utils.IsEqual(m_localeIds, value.m_localeIds)) return false;
            if (!Utils.IsEqual(m_serverUris, value.m_serverUris)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (FindServersRequest)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            FindServersRequest clone = (FindServersRequest)base.MemberwiseClone();

            clone.m_requestHeader = (RequestHeader)Utils.Clone(this.m_requestHeader);
            clone.m_endpointUrl = (string)Utils.Clone(this.m_endpointUrl);
            clone.m_localeIds = (StringCollection)Utils.Clone(this.m_localeIds);
            clone.m_serverUris = (StringCollection)Utils.Clone(this.m_serverUris);

            return clone;
        }
        #endregion

        #region Private Fields
        private RequestHeader m_requestHeader;
        private string m_endpointUrl;
        private StringCollection m_localeIds;
        private StringCollection m_serverUris;
        #endregion
    }
    #endif
    #endregion

    #region FindServersResponse Class
    #if (!OPCUA_EXCLUDE_FindServersResponse)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class FindServersResponse : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public FindServersResponse()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_responseHeader = new ResponseHeader();
            m_servers = new ApplicationDescriptionCollection();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ResponseHeader", IsRequired = false, Order = 1)]
        public ResponseHeader ResponseHeader
        {
            get
            {
                return m_responseHeader;
            }

            set
            {
                m_responseHeader = value;

                if (value == null)
                {
                    m_responseHeader = new ResponseHeader();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "Servers", IsRequired = false, Order = 2)]
        public ApplicationDescriptionCollection Servers
        {
            get
            {
                return m_servers;
            }

            set
            {
                m_servers = value;

                if (value == null)
                {
                    m_servers = new ApplicationDescriptionCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.FindServersResponse; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.FindServersResponse_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.FindServersResponse_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("ResponseHeader", ResponseHeader, typeof(ResponseHeader));
            encoder.WriteEncodeableArray("Servers", Servers.ToArray(), typeof(ApplicationDescription));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ResponseHeader = (ResponseHeader)decoder.ReadEncodeable("ResponseHeader", typeof(ResponseHeader));
            Servers = (ApplicationDescriptionCollection)decoder.ReadEncodeableArray("Servers", typeof(ApplicationDescription));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            FindServersResponse value = encodeable as FindServersResponse;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_responseHeader, value.m_responseHeader)) return false;
            if (!Utils.IsEqual(m_servers, value.m_servers)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (FindServersResponse)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            FindServersResponse clone = (FindServersResponse)base.MemberwiseClone();

            clone.m_responseHeader = (ResponseHeader)Utils.Clone(this.m_responseHeader);
            clone.m_servers = (ApplicationDescriptionCollection)Utils.Clone(this.m_servers);

            return clone;
        }
        #endregion

        #region Private Fields
        private ResponseHeader m_responseHeader;
        private ApplicationDescriptionCollection m_servers;
        #endregion
    }
    #endif
    #endregion

    #region ServerOnNetwork Class
    #if (!OPCUA_EXCLUDE_ServerOnNetwork)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ServerOnNetwork : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ServerOnNetwork()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_recordId = (uint)0;
            m_serverName = null;
            m_discoveryUrl = null;
            m_serverCapabilities = new StringCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "RecordId", IsRequired = false, Order = 1)]
        public uint RecordId
        {
            get { return m_recordId;  }
            set { m_recordId = value; }
        }

        /// <remarks />
        [DataMember(Name = "ServerName", IsRequired = false, Order = 2)]
        public string ServerName
        {
            get { return m_serverName;  }
            set { m_serverName = value; }
        }

        /// <remarks />
        [DataMember(Name = "DiscoveryUrl", IsRequired = false, Order = 3)]
        public string DiscoveryUrl
        {
            get { return m_discoveryUrl;  }
            set { m_discoveryUrl = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ServerCapabilities", IsRequired = false, Order = 4)]
        public StringCollection ServerCapabilities
        {
            get
            {
                return m_serverCapabilities;
            }

            set
            {
                m_serverCapabilities = value;

                if (value == null)
                {
                    m_serverCapabilities = new StringCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ServerOnNetwork; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ServerOnNetwork_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ServerOnNetwork_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("RecordId", RecordId);
            encoder.WriteString("ServerName", ServerName);
            encoder.WriteString("DiscoveryUrl", DiscoveryUrl);
            encoder.WriteStringArray("ServerCapabilities", ServerCapabilities);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            RecordId = decoder.ReadUInt32("RecordId");
            ServerName = decoder.ReadString("ServerName");
            DiscoveryUrl = decoder.ReadString("DiscoveryUrl");
            ServerCapabilities = decoder.ReadStringArray("ServerCapabilities");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ServerOnNetwork value = encodeable as ServerOnNetwork;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_recordId, value.m_recordId)) return false;
            if (!Utils.IsEqual(m_serverName, value.m_serverName)) return false;
            if (!Utils.IsEqual(m_discoveryUrl, value.m_discoveryUrl)) return false;
            if (!Utils.IsEqual(m_serverCapabilities, value.m_serverCapabilities)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ServerOnNetwork)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ServerOnNetwork clone = (ServerOnNetwork)base.MemberwiseClone();

            clone.m_recordId = (uint)Utils.Clone(this.m_recordId);
            clone.m_serverName = (string)Utils.Clone(this.m_serverName);
            clone.m_discoveryUrl = (string)Utils.Clone(this.m_discoveryUrl);
            clone.m_serverCapabilities = (StringCollection)Utils.Clone(this.m_serverCapabilities);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_recordId;
        private string m_serverName;
        private string m_discoveryUrl;
        private StringCollection m_serverCapabilities;
        #endregion
    }

    #region ServerOnNetworkCollection Class
    /// <summary>
    /// A collection of ServerOnNetwork objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfServerOnNetwork", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ServerOnNetwork")]
    #if !NET_STANDARD
    public partial class ServerOnNetworkCollection : List<ServerOnNetwork>, ICloneable
    #else
    public partial class ServerOnNetworkCollection : List<ServerOnNetwork>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public ServerOnNetworkCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public ServerOnNetworkCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public ServerOnNetworkCollection(IEnumerable<ServerOnNetwork> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator ServerOnNetworkCollection(ServerOnNetwork[] values)
        {
            if (values != null)
            {
                return new ServerOnNetworkCollection(values);
            }

            return new ServerOnNetworkCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator ServerOnNetwork[](ServerOnNetworkCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (ServerOnNetworkCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ServerOnNetworkCollection clone = new ServerOnNetworkCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ServerOnNetwork)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region FindServersOnNetworkRequest Class
    #if (!OPCUA_EXCLUDE_FindServersOnNetworkRequest)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class FindServersOnNetworkRequest : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public FindServersOnNetworkRequest()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_requestHeader = new RequestHeader();
            m_startingRecordId = (uint)0;
            m_maxRecordsToReturn = (uint)0;
            m_serverCapabilityFilter = new StringCollection();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "RequestHeader", IsRequired = false, Order = 1)]
        public RequestHeader RequestHeader
        {
            get
            {
                return m_requestHeader;
            }

            set
            {
                m_requestHeader = value;

                if (value == null)
                {
                    m_requestHeader = new RequestHeader();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "StartingRecordId", IsRequired = false, Order = 2)]
        public uint StartingRecordId
        {
            get { return m_startingRecordId;  }
            set { m_startingRecordId = value; }
        }

        /// <remarks />
        [DataMember(Name = "MaxRecordsToReturn", IsRequired = false, Order = 3)]
        public uint MaxRecordsToReturn
        {
            get { return m_maxRecordsToReturn;  }
            set { m_maxRecordsToReturn = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ServerCapabilityFilter", IsRequired = false, Order = 4)]
        public StringCollection ServerCapabilityFilter
        {
            get
            {
                return m_serverCapabilityFilter;
            }

            set
            {
                m_serverCapabilityFilter = value;

                if (value == null)
                {
                    m_serverCapabilityFilter = new StringCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.FindServersOnNetworkRequest; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.FindServersOnNetworkRequest_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.FindServersOnNetworkRequest_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("RequestHeader", RequestHeader, typeof(RequestHeader));
            encoder.WriteUInt32("StartingRecordId", StartingRecordId);
            encoder.WriteUInt32("MaxRecordsToReturn", MaxRecordsToReturn);
            encoder.WriteStringArray("ServerCapabilityFilter", ServerCapabilityFilter);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            RequestHeader = (RequestHeader)decoder.ReadEncodeable("RequestHeader", typeof(RequestHeader));
            StartingRecordId = decoder.ReadUInt32("StartingRecordId");
            MaxRecordsToReturn = decoder.ReadUInt32("MaxRecordsToReturn");
            ServerCapabilityFilter = decoder.ReadStringArray("ServerCapabilityFilter");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            FindServersOnNetworkRequest value = encodeable as FindServersOnNetworkRequest;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_requestHeader, value.m_requestHeader)) return false;
            if (!Utils.IsEqual(m_startingRecordId, value.m_startingRecordId)) return false;
            if (!Utils.IsEqual(m_maxRecordsToReturn, value.m_maxRecordsToReturn)) return false;
            if (!Utils.IsEqual(m_serverCapabilityFilter, value.m_serverCapabilityFilter)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (FindServersOnNetworkRequest)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            FindServersOnNetworkRequest clone = (FindServersOnNetworkRequest)base.MemberwiseClone();

            clone.m_requestHeader = (RequestHeader)Utils.Clone(this.m_requestHeader);
            clone.m_startingRecordId = (uint)Utils.Clone(this.m_startingRecordId);
            clone.m_maxRecordsToReturn = (uint)Utils.Clone(this.m_maxRecordsToReturn);
            clone.m_serverCapabilityFilter = (StringCollection)Utils.Clone(this.m_serverCapabilityFilter);

            return clone;
        }
        #endregion

        #region Private Fields
        private RequestHeader m_requestHeader;
        private uint m_startingRecordId;
        private uint m_maxRecordsToReturn;
        private StringCollection m_serverCapabilityFilter;
        #endregion
    }
    #endif
    #endregion

    #region FindServersOnNetworkResponse Class
    #if (!OPCUA_EXCLUDE_FindServersOnNetworkResponse)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class FindServersOnNetworkResponse : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public FindServersOnNetworkResponse()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_responseHeader = new ResponseHeader();
            m_lastCounterResetTime = DateTime.MinValue;
            m_servers = new ServerOnNetworkCollection();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ResponseHeader", IsRequired = false, Order = 1)]
        public ResponseHeader ResponseHeader
        {
            get
            {
                return m_responseHeader;
            }

            set
            {
                m_responseHeader = value;

                if (value == null)
                {
                    m_responseHeader = new ResponseHeader();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "LastCounterResetTime", IsRequired = false, Order = 2)]
        public DateTime LastCounterResetTime
        {
            get { return m_lastCounterResetTime;  }
            set { m_lastCounterResetTime = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "Servers", IsRequired = false, Order = 3)]
        public ServerOnNetworkCollection Servers
        {
            get
            {
                return m_servers;
            }

            set
            {
                m_servers = value;

                if (value == null)
                {
                    m_servers = new ServerOnNetworkCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.FindServersOnNetworkResponse; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.FindServersOnNetworkResponse_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.FindServersOnNetworkResponse_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("ResponseHeader", ResponseHeader, typeof(ResponseHeader));
            encoder.WriteDateTime("LastCounterResetTime", LastCounterResetTime);
            encoder.WriteEncodeableArray("Servers", Servers.ToArray(), typeof(ServerOnNetwork));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ResponseHeader = (ResponseHeader)decoder.ReadEncodeable("ResponseHeader", typeof(ResponseHeader));
            LastCounterResetTime = decoder.ReadDateTime("LastCounterResetTime");
            Servers = (ServerOnNetworkCollection)decoder.ReadEncodeableArray("Servers", typeof(ServerOnNetwork));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            FindServersOnNetworkResponse value = encodeable as FindServersOnNetworkResponse;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_responseHeader, value.m_responseHeader)) return false;
            if (!Utils.IsEqual(m_lastCounterResetTime, value.m_lastCounterResetTime)) return false;
            if (!Utils.IsEqual(m_servers, value.m_servers)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (FindServersOnNetworkResponse)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            FindServersOnNetworkResponse clone = (FindServersOnNetworkResponse)base.MemberwiseClone();

            clone.m_responseHeader = (ResponseHeader)Utils.Clone(this.m_responseHeader);
            clone.m_lastCounterResetTime = (DateTime)Utils.Clone(this.m_lastCounterResetTime);
            clone.m_servers = (ServerOnNetworkCollection)Utils.Clone(this.m_servers);

            return clone;
        }
        #endregion

        #region Private Fields
        private ResponseHeader m_responseHeader;
        private DateTime m_lastCounterResetTime;
        private ServerOnNetworkCollection m_servers;
        #endregion
    }
    #endif
    #endregion

    #region MessageSecurityMode Enumeration
    #if (!OPCUA_EXCLUDE_MessageSecurityMode)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum MessageSecurityMode
    {
        /// <remarks />
        [EnumMember(Value = "Invalid_0")]
        Invalid = 0,

        /// <remarks />
        [EnumMember(Value = "None_1")]
        None = 1,

        /// <remarks />
        [EnumMember(Value = "Sign_2")]
        Sign = 2,

        /// <remarks />
        [EnumMember(Value = "SignAndEncrypt_3")]
        SignAndEncrypt = 3,
    }
    #endif
    #endregion

    #region UserTokenType Enumeration
    #if (!OPCUA_EXCLUDE_UserTokenType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum UserTokenType
    {
        /// <remarks />
        [EnumMember(Value = "Anonymous_0")]
        Anonymous = 0,

        /// <remarks />
        [EnumMember(Value = "UserName_1")]
        UserName = 1,

        /// <remarks />
        [EnumMember(Value = "Certificate_2")]
        Certificate = 2,

        /// <remarks />
        [EnumMember(Value = "IssuedToken_3")]
        IssuedToken = 3,
    }
    #endif
    #endregion

    #region UserTokenPolicy Class
    #if (!OPCUA_EXCLUDE_UserTokenPolicy)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class UserTokenPolicy : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public UserTokenPolicy()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_policyId = null;
            m_tokenType = UserTokenType.Anonymous;
            m_issuedTokenType = null;
            m_issuerEndpointUrl = null;
            m_securityPolicyUri = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "PolicyId", IsRequired = false, Order = 1)]
        public string PolicyId
        {
            get { return m_policyId;  }
            set { m_policyId = value; }
        }

        /// <remarks />
        [DataMember(Name = "TokenType", IsRequired = false, Order = 2)]
        public UserTokenType TokenType
        {
            get { return m_tokenType;  }
            set { m_tokenType = value; }
        }

        /// <remarks />
        [DataMember(Name = "IssuedTokenType", IsRequired = false, Order = 3)]
        public string IssuedTokenType
        {
            get { return m_issuedTokenType;  }
            set { m_issuedTokenType = value; }
        }

        /// <remarks />
        [DataMember(Name = "IssuerEndpointUrl", IsRequired = false, Order = 4)]
        public string IssuerEndpointUrl
        {
            get { return m_issuerEndpointUrl;  }
            set { m_issuerEndpointUrl = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityPolicyUri", IsRequired = false, Order = 5)]
        public string SecurityPolicyUri
        {
            get { return m_securityPolicyUri;  }
            set { m_securityPolicyUri = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.UserTokenPolicy; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.UserTokenPolicy_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.UserTokenPolicy_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("PolicyId", PolicyId);
            encoder.WriteEnumerated("TokenType", TokenType);
            encoder.WriteString("IssuedTokenType", IssuedTokenType);
            encoder.WriteString("IssuerEndpointUrl", IssuerEndpointUrl);
            encoder.WriteString("SecurityPolicyUri", SecurityPolicyUri);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            PolicyId = decoder.ReadString("PolicyId");
            TokenType = (UserTokenType)decoder.ReadEnumerated("TokenType", typeof(UserTokenType));
            IssuedTokenType = decoder.ReadString("IssuedTokenType");
            IssuerEndpointUrl = decoder.ReadString("IssuerEndpointUrl");
            SecurityPolicyUri = decoder.ReadString("SecurityPolicyUri");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            UserTokenPolicy value = encodeable as UserTokenPolicy;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_policyId, value.m_policyId)) return false;
            if (!Utils.IsEqual(m_tokenType, value.m_tokenType)) return false;
            if (!Utils.IsEqual(m_issuedTokenType, value.m_issuedTokenType)) return false;
            if (!Utils.IsEqual(m_issuerEndpointUrl, value.m_issuerEndpointUrl)) return false;
            if (!Utils.IsEqual(m_securityPolicyUri, value.m_securityPolicyUri)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (UserTokenPolicy)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UserTokenPolicy clone = (UserTokenPolicy)base.MemberwiseClone();

            clone.m_policyId = (string)Utils.Clone(this.m_policyId);
            clone.m_tokenType = (UserTokenType)Utils.Clone(this.m_tokenType);
            clone.m_issuedTokenType = (string)Utils.Clone(this.m_issuedTokenType);
            clone.m_issuerEndpointUrl = (string)Utils.Clone(this.m_issuerEndpointUrl);
            clone.m_securityPolicyUri = (string)Utils.Clone(this.m_securityPolicyUri);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_policyId;
        private UserTokenType m_tokenType;
        private string m_issuedTokenType;
        private string m_issuerEndpointUrl;
        private string m_securityPolicyUri;
        #endregion
    }

    #region UserTokenPolicyCollection Class
    /// <summary>
    /// A collection of UserTokenPolicy objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUserTokenPolicy", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "UserTokenPolicy")]
    #if !NET_STANDARD
    public partial class UserTokenPolicyCollection : List<UserTokenPolicy>, ICloneable
    #else
    public partial class UserTokenPolicyCollection : List<UserTokenPolicy>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public UserTokenPolicyCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public UserTokenPolicyCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public UserTokenPolicyCollection(IEnumerable<UserTokenPolicy> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator UserTokenPolicyCollection(UserTokenPolicy[] values)
        {
            if (values != null)
            {
                return new UserTokenPolicyCollection(values);
            }

            return new UserTokenPolicyCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator UserTokenPolicy[](UserTokenPolicyCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (UserTokenPolicyCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UserTokenPolicyCollection clone = new UserTokenPolicyCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((UserTokenPolicy)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region EndpointDescription Class
    #if (!OPCUA_EXCLUDE_EndpointDescription)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class EndpointDescription : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public EndpointDescription()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_endpointUrl = null;
            m_server = new ApplicationDescription();
            m_serverCertificate = null;
            m_securityMode = MessageSecurityMode.Invalid;
            m_securityPolicyUri = null;
            m_userIdentityTokens = new UserTokenPolicyCollection();
            m_transportProfileUri = null;
            m_securityLevel = (byte)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "EndpointUrl", IsRequired = false, Order = 1)]
        public string EndpointUrl
        {
            get { return m_endpointUrl;  }
            set { m_endpointUrl = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "Server", IsRequired = false, Order = 2)]
        public ApplicationDescription Server
        {
            get
            {
                return m_server;
            }

            set
            {
                m_server = value;

                if (value == null)
                {
                    m_server = new ApplicationDescription();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "ServerCertificate", IsRequired = false, Order = 3)]
        public byte[] ServerCertificate
        {
            get { return m_serverCertificate;  }
            set { m_serverCertificate = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityMode", IsRequired = false, Order = 4)]
        public MessageSecurityMode SecurityMode
        {
            get { return m_securityMode;  }
            set { m_securityMode = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityPolicyUri", IsRequired = false, Order = 5)]
        public string SecurityPolicyUri
        {
            get { return m_securityPolicyUri;  }
            set { m_securityPolicyUri = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "UserIdentityTokens", IsRequired = false, Order = 6)]
        public UserTokenPolicyCollection UserIdentityTokens
        {
            get
            {
                return m_userIdentityTokens;
            }

            set
            {
                m_userIdentityTokens = value;

                if (value == null)
                {
                    m_userIdentityTokens = new UserTokenPolicyCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "TransportProfileUri", IsRequired = false, Order = 7)]
        public string TransportProfileUri
        {
            get { return m_transportProfileUri;  }
            set { m_transportProfileUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityLevel", IsRequired = false, Order = 8)]
        public byte SecurityLevel
        {
            get { return m_securityLevel;  }
            set { m_securityLevel = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.EndpointDescription; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.EndpointDescription_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.EndpointDescription_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("EndpointUrl", EndpointUrl);
            encoder.WriteEncodeable("Server", Server, typeof(ApplicationDescription));
            encoder.WriteByteString("ServerCertificate", ServerCertificate);
            encoder.WriteEnumerated("SecurityMode", SecurityMode);
            encoder.WriteString("SecurityPolicyUri", SecurityPolicyUri);
            encoder.WriteEncodeableArray("UserIdentityTokens", UserIdentityTokens.ToArray(), typeof(UserTokenPolicy));
            encoder.WriteString("TransportProfileUri", TransportProfileUri);
            encoder.WriteByte("SecurityLevel", SecurityLevel);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            EndpointUrl = decoder.ReadString("EndpointUrl");
            Server = (ApplicationDescription)decoder.ReadEncodeable("Server", typeof(ApplicationDescription));
            ServerCertificate = decoder.ReadByteString("ServerCertificate");
            SecurityMode = (MessageSecurityMode)decoder.ReadEnumerated("SecurityMode", typeof(MessageSecurityMode));
            SecurityPolicyUri = decoder.ReadString("SecurityPolicyUri");
            UserIdentityTokens = (UserTokenPolicyCollection)decoder.ReadEncodeableArray("UserIdentityTokens", typeof(UserTokenPolicy));
            TransportProfileUri = decoder.ReadString("TransportProfileUri");
            SecurityLevel = decoder.ReadByte("SecurityLevel");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            EndpointDescription value = encodeable as EndpointDescription;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_endpointUrl, value.m_endpointUrl)) return false;
            if (!Utils.IsEqual(m_server, value.m_server)) return false;
            if (!Utils.IsEqual(m_serverCertificate, value.m_serverCertificate)) return false;
            if (!Utils.IsEqual(m_securityMode, value.m_securityMode)) return false;
            if (!Utils.IsEqual(m_securityPolicyUri, value.m_securityPolicyUri)) return false;
            if (!Utils.IsEqual(m_userIdentityTokens, value.m_userIdentityTokens)) return false;
            if (!Utils.IsEqual(m_transportProfileUri, value.m_transportProfileUri)) return false;
            if (!Utils.IsEqual(m_securityLevel, value.m_securityLevel)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (EndpointDescription)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EndpointDescription clone = (EndpointDescription)base.MemberwiseClone();

            clone.m_endpointUrl = (string)Utils.Clone(this.m_endpointUrl);
            clone.m_server = (ApplicationDescription)Utils.Clone(this.m_server);
            clone.m_serverCertificate = (byte[])Utils.Clone(this.m_serverCertificate);
            clone.m_securityMode = (MessageSecurityMode)Utils.Clone(this.m_securityMode);
            clone.m_securityPolicyUri = (string)Utils.Clone(this.m_securityPolicyUri);
            clone.m_userIdentityTokens = (UserTokenPolicyCollection)Utils.Clone(this.m_userIdentityTokens);
            clone.m_transportProfileUri = (string)Utils.Clone(this.m_transportProfileUri);
            clone.m_securityLevel = (byte)Utils.Clone(this.m_securityLevel);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_endpointUrl;
        private ApplicationDescription m_server;
        private byte[] m_serverCertificate;
        private MessageSecurityMode m_securityMode;
        private string m_securityPolicyUri;
        private UserTokenPolicyCollection m_userIdentityTokens;
        private string m_transportProfileUri;
        private byte m_securityLevel;
        #endregion
    }

    #region EndpointDescriptionCollection Class
    /// <summary>
    /// A collection of EndpointDescription objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfEndpointDescription", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "EndpointDescription")]
    #if !NET_STANDARD
    public partial class EndpointDescriptionCollection : List<EndpointDescription>, ICloneable
    #else
    public partial class EndpointDescriptionCollection : List<EndpointDescription>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public EndpointDescriptionCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public EndpointDescriptionCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public EndpointDescriptionCollection(IEnumerable<EndpointDescription> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator EndpointDescriptionCollection(EndpointDescription[] values)
        {
            if (values != null)
            {
                return new EndpointDescriptionCollection(values);
            }

            return new EndpointDescriptionCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator EndpointDescription[](EndpointDescriptionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (EndpointDescriptionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EndpointDescriptionCollection clone = new EndpointDescriptionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((EndpointDescription)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region GetEndpointsRequest Class
    #if (!OPCUA_EXCLUDE_GetEndpointsRequest)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class GetEndpointsRequest : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public GetEndpointsRequest()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_requestHeader = new RequestHeader();
            m_endpointUrl = null;
            m_localeIds = new StringCollection();
            m_profileUris = new StringCollection();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "RequestHeader", IsRequired = false, Order = 1)]
        public RequestHeader RequestHeader
        {
            get
            {
                return m_requestHeader;
            }

            set
            {
                m_requestHeader = value;

                if (value == null)
                {
                    m_requestHeader = new RequestHeader();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "EndpointUrl", IsRequired = false, Order = 2)]
        public string EndpointUrl
        {
            get { return m_endpointUrl;  }
            set { m_endpointUrl = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "LocaleIds", IsRequired = false, Order = 3)]
        public StringCollection LocaleIds
        {
            get
            {
                return m_localeIds;
            }

            set
            {
                m_localeIds = value;

                if (value == null)
                {
                    m_localeIds = new StringCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ProfileUris", IsRequired = false, Order = 4)]
        public StringCollection ProfileUris
        {
            get
            {
                return m_profileUris;
            }

            set
            {
                m_profileUris = value;

                if (value == null)
                {
                    m_profileUris = new StringCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.GetEndpointsRequest; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.GetEndpointsRequest_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.GetEndpointsRequest_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("RequestHeader", RequestHeader, typeof(RequestHeader));
            encoder.WriteString("EndpointUrl", EndpointUrl);
            encoder.WriteStringArray("LocaleIds", LocaleIds);
            encoder.WriteStringArray("ProfileUris", ProfileUris);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            RequestHeader = (RequestHeader)decoder.ReadEncodeable("RequestHeader", typeof(RequestHeader));
            EndpointUrl = decoder.ReadString("EndpointUrl");
            LocaleIds = decoder.ReadStringArray("LocaleIds");
            ProfileUris = decoder.ReadStringArray("ProfileUris");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            GetEndpointsRequest value = encodeable as GetEndpointsRequest;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_requestHeader, value.m_requestHeader)) return false;
            if (!Utils.IsEqual(m_endpointUrl, value.m_endpointUrl)) return false;
            if (!Utils.IsEqual(m_localeIds, value.m_localeIds)) return false;
            if (!Utils.IsEqual(m_profileUris, value.m_profileUris)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (GetEndpointsRequest)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            GetEndpointsRequest clone = (GetEndpointsRequest)base.MemberwiseClone();

            clone.m_requestHeader = (RequestHeader)Utils.Clone(this.m_requestHeader);
            clone.m_endpointUrl = (string)Utils.Clone(this.m_endpointUrl);
            clone.m_localeIds = (StringCollection)Utils.Clone(this.m_localeIds);
            clone.m_profileUris = (StringCollection)Utils.Clone(this.m_profileUris);

            return clone;
        }
        #endregion

        #region Private Fields
        private RequestHeader m_requestHeader;
        private string m_endpointUrl;
        private StringCollection m_localeIds;
        private StringCollection m_profileUris;
        #endregion
    }
    #endif
    #endregion

    #region GetEndpointsResponse Class
    #if (!OPCUA_EXCLUDE_GetEndpointsResponse)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class GetEndpointsResponse : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public GetEndpointsResponse()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_responseHeader = new ResponseHeader();
            m_endpoints = new EndpointDescriptionCollection();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ResponseHeader", IsRequired = false, Order = 1)]
        public ResponseHeader ResponseHeader
        {
            get
            {
                return m_responseHeader;
            }

            set
            {
                m_responseHeader = value;

                if (value == null)
                {
                    m_responseHeader = new ResponseHeader();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "Endpoints", IsRequired = false, Order = 2)]
        public EndpointDescriptionCollection Endpoints
        {
            get
            {
                return m_endpoints;
            }

            set
            {
                m_endpoints = value;

                if (value == null)
                {
                    m_endpoints = new EndpointDescriptionCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.GetEndpointsResponse; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.GetEndpointsResponse_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.GetEndpointsResponse_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("ResponseHeader", ResponseHeader, typeof(ResponseHeader));
            encoder.WriteEncodeableArray("Endpoints", Endpoints.ToArray(), typeof(EndpointDescription));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ResponseHeader = (ResponseHeader)decoder.ReadEncodeable("ResponseHeader", typeof(ResponseHeader));
            Endpoints = (EndpointDescriptionCollection)decoder.ReadEncodeableArray("Endpoints", typeof(EndpointDescription));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            GetEndpointsResponse value = encodeable as GetEndpointsResponse;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_responseHeader, value.m_responseHeader)) return false;
            if (!Utils.IsEqual(m_endpoints, value.m_endpoints)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (GetEndpointsResponse)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            GetEndpointsResponse clone = (GetEndpointsResponse)base.MemberwiseClone();

            clone.m_responseHeader = (ResponseHeader)Utils.Clone(this.m_responseHeader);
            clone.m_endpoints = (EndpointDescriptionCollection)Utils.Clone(this.m_endpoints);

            return clone;
        }
        #endregion

        #region Private Fields
        private ResponseHeader m_responseHeader;
        private EndpointDescriptionCollection m_endpoints;
        #endregion
    }
    #endif
    #endregion

    #region RegisteredServer Class
    #if (!OPCUA_EXCLUDE_RegisteredServer)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class RegisteredServer : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public RegisteredServer()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_serverUri = null;
            m_productUri = null;
            m_serverNames = new LocalizedTextCollection();
            m_serverType = ApplicationType.Server;
            m_gatewayServerUri = null;
            m_discoveryUrls = new StringCollection();
            m_semaphoreFilePath = null;
            m_isOnline = true;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ServerUri", IsRequired = false, Order = 1)]
        public string ServerUri
        {
            get { return m_serverUri;  }
            set { m_serverUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "ProductUri", IsRequired = false, Order = 2)]
        public string ProductUri
        {
            get { return m_productUri;  }
            set { m_productUri = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ServerNames", IsRequired = false, Order = 3)]
        public LocalizedTextCollection ServerNames
        {
            get
            {
                return m_serverNames;
            }

            set
            {
                m_serverNames = value;

                if (value == null)
                {
                    m_serverNames = new LocalizedTextCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "ServerType", IsRequired = false, Order = 4)]
        public ApplicationType ServerType
        {
            get { return m_serverType;  }
            set { m_serverType = value; }
        }

        /// <remarks />
        [DataMember(Name = "GatewayServerUri", IsRequired = false, Order = 5)]
        public string GatewayServerUri
        {
            get { return m_gatewayServerUri;  }
            set { m_gatewayServerUri = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "DiscoveryUrls", IsRequired = false, Order = 6)]
        public StringCollection DiscoveryUrls
        {
            get
            {
                return m_discoveryUrls;
            }

            set
            {
                m_discoveryUrls = value;

                if (value == null)
                {
                    m_discoveryUrls = new StringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "SemaphoreFilePath", IsRequired = false, Order = 7)]
        public string SemaphoreFilePath
        {
            get { return m_semaphoreFilePath;  }
            set { m_semaphoreFilePath = value; }
        }

        /// <remarks />
        [DataMember(Name = "IsOnline", IsRequired = false, Order = 8)]
        public bool IsOnline
        {
            get { return m_isOnline;  }
            set { m_isOnline = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.RegisteredServer; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.RegisteredServer_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.RegisteredServer_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("ServerUri", ServerUri);
            encoder.WriteString("ProductUri", ProductUri);
            encoder.WriteLocalizedTextArray("ServerNames", ServerNames);
            encoder.WriteEnumerated("ServerType", ServerType);
            encoder.WriteString("GatewayServerUri", GatewayServerUri);
            encoder.WriteStringArray("DiscoveryUrls", DiscoveryUrls);
            encoder.WriteString("SemaphoreFilePath", SemaphoreFilePath);
            encoder.WriteBoolean("IsOnline", IsOnline);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ServerUri = decoder.ReadString("ServerUri");
            ProductUri = decoder.ReadString("ProductUri");
            ServerNames = decoder.ReadLocalizedTextArray("ServerNames");
            ServerType = (ApplicationType)decoder.ReadEnumerated("ServerType", typeof(ApplicationType));
            GatewayServerUri = decoder.ReadString("GatewayServerUri");
            DiscoveryUrls = decoder.ReadStringArray("DiscoveryUrls");
            SemaphoreFilePath = decoder.ReadString("SemaphoreFilePath");
            IsOnline = decoder.ReadBoolean("IsOnline");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            RegisteredServer value = encodeable as RegisteredServer;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_serverUri, value.m_serverUri)) return false;
            if (!Utils.IsEqual(m_productUri, value.m_productUri)) return false;
            if (!Utils.IsEqual(m_serverNames, value.m_serverNames)) return false;
            if (!Utils.IsEqual(m_serverType, value.m_serverType)) return false;
            if (!Utils.IsEqual(m_gatewayServerUri, value.m_gatewayServerUri)) return false;
            if (!Utils.IsEqual(m_discoveryUrls, value.m_discoveryUrls)) return false;
            if (!Utils.IsEqual(m_semaphoreFilePath, value.m_semaphoreFilePath)) return false;
            if (!Utils.IsEqual(m_isOnline, value.m_isOnline)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (RegisteredServer)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            RegisteredServer clone = (RegisteredServer)base.MemberwiseClone();

            clone.m_serverUri = (string)Utils.Clone(this.m_serverUri);
            clone.m_productUri = (string)Utils.Clone(this.m_productUri);
            clone.m_serverNames = (LocalizedTextCollection)Utils.Clone(this.m_serverNames);
            clone.m_serverType = (ApplicationType)Utils.Clone(this.m_serverType);
            clone.m_gatewayServerUri = (string)Utils.Clone(this.m_gatewayServerUri);
            clone.m_discoveryUrls = (StringCollection)Utils.Clone(this.m_discoveryUrls);
            clone.m_semaphoreFilePath = (string)Utils.Clone(this.m_semaphoreFilePath);
            clone.m_isOnline = (bool)Utils.Clone(this.m_isOnline);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_serverUri;
        private string m_productUri;
        private LocalizedTextCollection m_serverNames;
        private ApplicationType m_serverType;
        private string m_gatewayServerUri;
        private StringCollection m_discoveryUrls;
        private string m_semaphoreFilePath;
        private bool m_isOnline;
        #endregion
    }

    #region RegisteredServerCollection Class
    /// <summary>
    /// A collection of RegisteredServer objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfRegisteredServer", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "RegisteredServer")]
    #if !NET_STANDARD
    public partial class RegisteredServerCollection : List<RegisteredServer>, ICloneable
    #else
    public partial class RegisteredServerCollection : List<RegisteredServer>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public RegisteredServerCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public RegisteredServerCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public RegisteredServerCollection(IEnumerable<RegisteredServer> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator RegisteredServerCollection(RegisteredServer[] values)
        {
            if (values != null)
            {
                return new RegisteredServerCollection(values);
            }

            return new RegisteredServerCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator RegisteredServer[](RegisteredServerCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (RegisteredServerCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            RegisteredServerCollection clone = new RegisteredServerCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((RegisteredServer)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region RegisterServerRequest Class
    #if (!OPCUA_EXCLUDE_RegisterServerRequest)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class RegisterServerRequest : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public RegisterServerRequest()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_requestHeader = new RequestHeader();
            m_server = new RegisteredServer();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "RequestHeader", IsRequired = false, Order = 1)]
        public RequestHeader RequestHeader
        {
            get
            {
                return m_requestHeader;
            }

            set
            {
                m_requestHeader = value;

                if (value == null)
                {
                    m_requestHeader = new RequestHeader();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "Server", IsRequired = false, Order = 2)]
        public RegisteredServer Server
        {
            get
            {
                return m_server;
            }

            set
            {
                m_server = value;

                if (value == null)
                {
                    m_server = new RegisteredServer();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.RegisterServerRequest; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.RegisterServerRequest_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.RegisterServerRequest_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("RequestHeader", RequestHeader, typeof(RequestHeader));
            encoder.WriteEncodeable("Server", Server, typeof(RegisteredServer));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            RequestHeader = (RequestHeader)decoder.ReadEncodeable("RequestHeader", typeof(RequestHeader));
            Server = (RegisteredServer)decoder.ReadEncodeable("Server", typeof(RegisteredServer));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            RegisterServerRequest value = encodeable as RegisterServerRequest;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_requestHeader, value.m_requestHeader)) return false;
            if (!Utils.IsEqual(m_server, value.m_server)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (RegisterServerRequest)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            RegisterServerRequest clone = (RegisterServerRequest)base.MemberwiseClone();

            clone.m_requestHeader = (RequestHeader)Utils.Clone(this.m_requestHeader);
            clone.m_server = (RegisteredServer)Utils.Clone(this.m_server);

            return clone;
        }
        #endregion

        #region Private Fields
        private RequestHeader m_requestHeader;
        private RegisteredServer m_server;
        #endregion
    }
    #endif
    #endregion

    #region RegisterServerResponse Class
    #if (!OPCUA_EXCLUDE_RegisterServerResponse)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class RegisterServerResponse : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public RegisterServerResponse()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_responseHeader = new ResponseHeader();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ResponseHeader", IsRequired = false, Order = 1)]
        public ResponseHeader ResponseHeader
        {
            get
            {
                return m_responseHeader;
            }

            set
            {
                m_responseHeader = value;

                if (value == null)
                {
                    m_responseHeader = new ResponseHeader();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.RegisterServerResponse; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.RegisterServerResponse_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.RegisterServerResponse_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("ResponseHeader", ResponseHeader, typeof(ResponseHeader));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ResponseHeader = (ResponseHeader)decoder.ReadEncodeable("ResponseHeader", typeof(ResponseHeader));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            RegisterServerResponse value = encodeable as RegisterServerResponse;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_responseHeader, value.m_responseHeader)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (RegisterServerResponse)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            RegisterServerResponse clone = (RegisterServerResponse)base.MemberwiseClone();

            clone.m_responseHeader = (ResponseHeader)Utils.Clone(this.m_responseHeader);

            return clone;
        }
        #endregion

        #region Private Fields
        private ResponseHeader m_responseHeader;
        #endregion
    }
    #endif
    #endregion

    #region DiscoveryConfiguration Class
    #if (!OPCUA_EXCLUDE_DiscoveryConfiguration)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DiscoveryConfiguration : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public DiscoveryConfiguration()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.DiscoveryConfiguration; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.DiscoveryConfiguration_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.DiscoveryConfiguration_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DiscoveryConfiguration value = encodeable as DiscoveryConfiguration;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DiscoveryConfiguration)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DiscoveryConfiguration clone = (DiscoveryConfiguration)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region MdnsDiscoveryConfiguration Class
    #if (!OPCUA_EXCLUDE_MdnsDiscoveryConfiguration)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class MdnsDiscoveryConfiguration : DiscoveryConfiguration
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public MdnsDiscoveryConfiguration()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_mdnsServerName = null;
            m_serverCapabilities = new StringCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "MdnsServerName", IsRequired = false, Order = 1)]
        public string MdnsServerName
        {
            get { return m_mdnsServerName;  }
            set { m_mdnsServerName = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ServerCapabilities", IsRequired = false, Order = 2)]
        public StringCollection ServerCapabilities
        {
            get
            {
                return m_serverCapabilities;
            }

            set
            {
                m_serverCapabilities = value;

                if (value == null)
                {
                    m_serverCapabilities = new StringCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId
        {
            get { return DataTypeIds.MdnsDiscoveryConfiguration; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.MdnsDiscoveryConfiguration_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.MdnsDiscoveryConfiguration_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("MdnsServerName", MdnsServerName);
            encoder.WriteStringArray("ServerCapabilities", ServerCapabilities);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            MdnsServerName = decoder.ReadString("MdnsServerName");
            ServerCapabilities = decoder.ReadStringArray("ServerCapabilities");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            MdnsDiscoveryConfiguration value = encodeable as MdnsDiscoveryConfiguration;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_mdnsServerName, value.m_mdnsServerName)) return false;
            if (!Utils.IsEqual(m_serverCapabilities, value.m_serverCapabilities)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (MdnsDiscoveryConfiguration)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            MdnsDiscoveryConfiguration clone = (MdnsDiscoveryConfiguration)base.MemberwiseClone();

            clone.m_mdnsServerName = (string)Utils.Clone(this.m_mdnsServerName);
            clone.m_serverCapabilities = (StringCollection)Utils.Clone(this.m_serverCapabilities);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_mdnsServerName;
        private StringCollection m_serverCapabilities;
        #endregion
    }
    #endif
    #endregion

    #region RegisterServer2Request Class
    #if (!OPCUA_EXCLUDE_RegisterServer2Request)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class RegisterServer2Request : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public RegisterServer2Request()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_requestHeader = new RequestHeader();
            m_server = new RegisteredServer();
            m_discoveryConfiguration = new ExtensionObjectCollection();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "RequestHeader", IsRequired = false, Order = 1)]
        public RequestHeader RequestHeader
        {
            get
            {
                return m_requestHeader;
            }

            set
            {
                m_requestHeader = value;

                if (value == null)
                {
                    m_requestHeader = new RequestHeader();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "Server", IsRequired = false, Order = 2)]
        public RegisteredServer Server
        {
            get
            {
                return m_server;
            }

            set
            {
                m_server = value;

                if (value == null)
                {
                    m_server = new RegisteredServer();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "DiscoveryConfiguration", IsRequired = false, Order = 3)]
        public ExtensionObjectCollection DiscoveryConfiguration
        {
            get
            {
                return m_discoveryConfiguration;
            }

            set
            {
                m_discoveryConfiguration = value;

                if (value == null)
                {
                    m_discoveryConfiguration = new ExtensionObjectCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.RegisterServer2Request; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.RegisterServer2Request_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.RegisterServer2Request_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("RequestHeader", RequestHeader, typeof(RequestHeader));
            encoder.WriteEncodeable("Server", Server, typeof(RegisteredServer));
            encoder.WriteExtensionObjectArray("DiscoveryConfiguration", DiscoveryConfiguration);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            RequestHeader = (RequestHeader)decoder.ReadEncodeable("RequestHeader", typeof(RequestHeader));
            Server = (RegisteredServer)decoder.ReadEncodeable("Server", typeof(RegisteredServer));
            DiscoveryConfiguration = decoder.ReadExtensionObjectArray("DiscoveryConfiguration");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            RegisterServer2Request value = encodeable as RegisterServer2Request;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_requestHeader, value.m_requestHeader)) return false;
            if (!Utils.IsEqual(m_server, value.m_server)) return false;
            if (!Utils.IsEqual(m_discoveryConfiguration, value.m_discoveryConfiguration)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (RegisterServer2Request)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            RegisterServer2Request clone = (RegisterServer2Request)base.MemberwiseClone();

            clone.m_requestHeader = (RequestHeader)Utils.Clone(this.m_requestHeader);
            clone.m_server = (RegisteredServer)Utils.Clone(this.m_server);
            clone.m_discoveryConfiguration = (ExtensionObjectCollection)Utils.Clone(this.m_discoveryConfiguration);

            return clone;
        }
        #endregion

        #region Private Fields
        private RequestHeader m_requestHeader;
        private RegisteredServer m_server;
        private ExtensionObjectCollection m_discoveryConfiguration;
        #endregion
    }
    #endif
    #endregion

    #region RegisterServer2Response Class
    #if (!OPCUA_EXCLUDE_RegisterServer2Response)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class RegisterServer2Response : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public RegisterServer2Response()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_responseHeader = new ResponseHeader();
            m_configurationResults = new StatusCodeCollection();
            m_diagnosticInfos = new DiagnosticInfoCollection();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ResponseHeader", IsRequired = false, Order = 1)]
        public ResponseHeader ResponseHeader
        {
            get
            {
                return m_responseHeader;
            }

            set
            {
                m_responseHeader = value;

                if (value == null)
                {
                    m_responseHeader = new ResponseHeader();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ConfigurationResults", IsRequired = false, Order = 2)]
        public StatusCodeCollection ConfigurationResults
        {
            get
            {
                return m_configurationResults;
            }

            set
            {
                m_configurationResults = value;

                if (value == null)
                {
                    m_configurationResults = new StatusCodeCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "DiagnosticInfos", IsRequired = false, Order = 3)]
        public DiagnosticInfoCollection DiagnosticInfos
        {
            get
            {
                return m_diagnosticInfos;
            }

            set
            {
                m_diagnosticInfos = value;

                if (value == null)
                {
                    m_diagnosticInfos = new DiagnosticInfoCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.RegisterServer2Response; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.RegisterServer2Response_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.RegisterServer2Response_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("ResponseHeader", ResponseHeader, typeof(ResponseHeader));
            encoder.WriteStatusCodeArray("ConfigurationResults", ConfigurationResults);
            encoder.WriteDiagnosticInfoArray("DiagnosticInfos", DiagnosticInfos);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ResponseHeader = (ResponseHeader)decoder.ReadEncodeable("ResponseHeader", typeof(ResponseHeader));
            ConfigurationResults = decoder.ReadStatusCodeArray("ConfigurationResults");
            DiagnosticInfos = decoder.ReadDiagnosticInfoArray("DiagnosticInfos");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            RegisterServer2Response value = encodeable as RegisterServer2Response;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_responseHeader, value.m_responseHeader)) return false;
            if (!Utils.IsEqual(m_configurationResults, value.m_configurationResults)) return false;
            if (!Utils.IsEqual(m_diagnosticInfos, value.m_diagnosticInfos)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (RegisterServer2Response)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            RegisterServer2Response clone = (RegisterServer2Response)base.MemberwiseClone();

            clone.m_responseHeader = (ResponseHeader)Utils.Clone(this.m_responseHeader);
            clone.m_configurationResults = (StatusCodeCollection)Utils.Clone(this.m_configurationResults);
            clone.m_diagnosticInfos = (DiagnosticInfoCollection)Utils.Clone(this.m_diagnosticInfos);

            return clone;
        }
        #endregion

        #region Private Fields
        private ResponseHeader m_responseHeader;
        private StatusCodeCollection m_configurationResults;
        private DiagnosticInfoCollection m_diagnosticInfos;
        #endregion
    }
    #endif
    #endregion

    #region SecurityTokenRequestType Enumeration
    #if (!OPCUA_EXCLUDE_SecurityTokenRequestType)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum SecurityTokenRequestType
    {
        /// <remarks />
        [EnumMember(Value = "Issue_0")]
        Issue = 0,

        /// <remarks />
        [EnumMember(Value = "Renew_1")]
        Renew = 1,
    }
    #endif
    #endregion

    #region ChannelSecurityToken Class
    #if (!OPCUA_EXCLUDE_ChannelSecurityToken)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ChannelSecurityToken : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public ChannelSecurityToken()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_channelId = (uint)0;
            m_tokenId = (uint)0;
            m_createdAt = DateTime.MinValue;
            m_revisedLifetime = (uint)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ChannelId", IsRequired = false, Order = 1)]
        public uint ChannelId
        {
            get { return m_channelId;  }
            set { m_channelId = value; }
        }

        /// <remarks />
        [DataMember(Name = "TokenId", IsRequired = false, Order = 2)]
        public uint TokenId
        {
            get { return m_tokenId;  }
            set { m_tokenId = value; }
        }

        /// <remarks />
        [DataMember(Name = "CreatedAt", IsRequired = false, Order = 3)]
        public DateTime CreatedAt
        {
            get { return m_createdAt;  }
            set { m_createdAt = value; }
        }

        /// <remarks />
        [DataMember(Name = "RevisedLifetime", IsRequired = false, Order = 4)]
        public uint RevisedLifetime
        {
            get { return m_revisedLifetime;  }
            set { m_revisedLifetime = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.ChannelSecurityToken; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.ChannelSecurityToken_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.ChannelSecurityToken_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("ChannelId", ChannelId);
            encoder.WriteUInt32("TokenId", TokenId);
            encoder.WriteDateTime("CreatedAt", CreatedAt);
            encoder.WriteUInt32("RevisedLifetime", RevisedLifetime);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ChannelId = decoder.ReadUInt32("ChannelId");
            TokenId = decoder.ReadUInt32("TokenId");
            CreatedAt = decoder.ReadDateTime("CreatedAt");
            RevisedLifetime = decoder.ReadUInt32("RevisedLifetime");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ChannelSecurityToken value = encodeable as ChannelSecurityToken;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_channelId, value.m_channelId)) return false;
            if (!Utils.IsEqual(m_tokenId, value.m_tokenId)) return false;
            if (!Utils.IsEqual(m_createdAt, value.m_createdAt)) return false;
            if (!Utils.IsEqual(m_revisedLifetime, value.m_revisedLifetime)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ChannelSecurityToken)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ChannelSecurityToken clone = (ChannelSecurityToken)base.MemberwiseClone();

            clone.m_channelId = (uint)Utils.Clone(this.m_channelId);
            clone.m_tokenId = (uint)Utils.Clone(this.m_tokenId);
            clone.m_createdAt = (DateTime)Utils.Clone(this.m_createdAt);
            clone.m_revisedLifetime = (uint)Utils.Clone(this.m_revisedLifetime);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_channelId;
        private uint m_tokenId;
        private DateTime m_createdAt;
        private uint m_revisedLifetime;
        #endregion
    }
    #endif
    #endregion

    #region OpenSecureChannelRequest Class
    #if (!OPCUA_EXCLUDE_OpenSecureChannelRequest)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class OpenSecureChannelRequest : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public OpenSecureChannelRequest()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_requestHeader = new RequestHeader();
            m_clientProtocolVersion = (uint)0;
            m_requestType = SecurityTokenRequestType.Issue;
            m_securityMode = MessageSecurityMode.Invalid;
            m_clientNonce = null;
            m_requestedLifetime = (uint)0;
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "RequestHeader", IsRequired = false, Order = 1)]
        public RequestHeader RequestHeader
        {
            get
            {
                return m_requestHeader;
            }

            set
            {
                m_requestHeader = value;

                if (value == null)
                {
                    m_requestHeader = new RequestHeader();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "ClientProtocolVersion", IsRequired = false, Order = 2)]
        public uint ClientProtocolVersion
        {
            get { return m_clientProtocolVersion;  }
            set { m_clientProtocolVersion = value; }
        }

        /// <remarks />
        [DataMember(Name = "RequestType", IsRequired = false, Order = 3)]
        public SecurityTokenRequestType RequestType
        {
            get { return m_requestType;  }
            set { m_requestType = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityMode", IsRequired = false, Order = 4)]
        public MessageSecurityMode SecurityMode
        {
            get { return m_securityMode;  }
            set { m_securityMode = value; }
        }

        /// <remarks />
        [DataMember(Name = "ClientNonce", IsRequired = false, Order = 5)]
        public byte[] ClientNonce
        {
            get { return m_clientNonce;  }
            set { m_clientNonce = value; }
        }

        /// <remarks />
        [DataMember(Name = "RequestedLifetime", IsRequired = false, Order = 6)]
        public uint RequestedLifetime
        {
            get { return m_requestedLifetime;  }
            set { m_requestedLifetime = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.OpenSecureChannelRequest; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.OpenSecureChannelRequest_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.OpenSecureChannelRequest_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("RequestHeader", RequestHeader, typeof(RequestHeader));
            encoder.WriteUInt32("ClientProtocolVersion", ClientProtocolVersion);
            encoder.WriteEnumerated("RequestType", RequestType);
            encoder.WriteEnumerated("SecurityMode", SecurityMode);
            encoder.WriteByteString("ClientNonce", ClientNonce);
            encoder.WriteUInt32("RequestedLifetime", RequestedLifetime);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            RequestHeader = (RequestHeader)decoder.ReadEncodeable("RequestHeader", typeof(RequestHeader));
            ClientProtocolVersion = decoder.ReadUInt32("ClientProtocolVersion");
            RequestType = (SecurityTokenRequestType)decoder.ReadEnumerated("RequestType", typeof(SecurityTokenRequestType));
            SecurityMode = (MessageSecurityMode)decoder.ReadEnumerated("SecurityMode", typeof(MessageSecurityMode));
            ClientNonce = decoder.ReadByteString("ClientNonce");
            RequestedLifetime = decoder.ReadUInt32("RequestedLifetime");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            OpenSecureChannelRequest value = encodeable as OpenSecureChannelRequest;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_requestHeader, value.m_requestHeader)) return false;
            if (!Utils.IsEqual(m_clientProtocolVersion, value.m_clientProtocolVersion)) return false;
            if (!Utils.IsEqual(m_requestType, value.m_requestType)) return false;
            if (!Utils.IsEqual(m_securityMode, value.m_securityMode)) return false;
            if (!Utils.IsEqual(m_clientNonce, value.m_clientNonce)) return false;
            if (!Utils.IsEqual(m_requestedLifetime, value.m_requestedLifetime)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (OpenSecureChannelRequest)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            OpenSecureChannelRequest clone = (OpenSecureChannelRequest)base.MemberwiseClone();

            clone.m_requestHeader = (RequestHeader)Utils.Clone(this.m_requestHeader);
            clone.m_clientProtocolVersion = (uint)Utils.Clone(this.m_clientProtocolVersion);
            clone.m_requestType = (SecurityTokenRequestType)Utils.Clone(this.m_requestType);
            clone.m_securityMode = (MessageSecurityMode)Utils.Clone(this.m_securityMode);
            clone.m_clientNonce = (byte[])Utils.Clone(this.m_clientNonce);
            clone.m_requestedLifetime = (uint)Utils.Clone(this.m_requestedLifetime);

            return clone;
        }
        #endregion

        #region Private Fields
        private RequestHeader m_requestHeader;
        private uint m_clientProtocolVersion;
        private SecurityTokenRequestType m_requestType;
        private MessageSecurityMode m_securityMode;
        private byte[] m_clientNonce;
        private uint m_requestedLifetime;
        #endregion
    }
    #endif
    #endregion

    #region OpenSecureChannelResponse Class
    #if (!OPCUA_EXCLUDE_OpenSecureChannelResponse)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class OpenSecureChannelResponse : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public OpenSecureChannelResponse()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_responseHeader = new ResponseHeader();
            m_serverProtocolVersion = (uint)0;
            m_securityToken = new ChannelSecurityToken();
            m_serverNonce = null;
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ResponseHeader", IsRequired = false, Order = 1)]
        public ResponseHeader ResponseHeader
        {
            get
            {
                return m_responseHeader;
            }

            set
            {
                m_responseHeader = value;

                if (value == null)
                {
                    m_responseHeader = new ResponseHeader();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "ServerProtocolVersion", IsRequired = false, Order = 2)]
        public uint ServerProtocolVersion
        {
            get { return m_serverProtocolVersion;  }
            set { m_serverProtocolVersion = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "SecurityToken", IsRequired = false, Order = 3)]
        public ChannelSecurityToken SecurityToken
        {
            get
            {
                return m_securityToken;
            }

            set
            {
                m_securityToken = value;

                if (value == null)
                {
                    m_securityToken = new ChannelSecurityToken();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "ServerNonce", IsRequired = false, Order = 4)]
        public byte[] ServerNonce
        {
            get { return m_serverNonce;  }
            set { m_serverNonce = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.OpenSecureChannelResponse; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.OpenSecureChannelResponse_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.OpenSecureChannelResponse_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("ResponseHeader", ResponseHeader, typeof(ResponseHeader));
            encoder.WriteUInt32("ServerProtocolVersion", ServerProtocolVersion);
            encoder.WriteEncodeable("SecurityToken", SecurityToken, typeof(ChannelSecurityToken));
            encoder.WriteByteString("ServerNonce", ServerNonce);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ResponseHeader = (ResponseHeader)decoder.ReadEncodeable("ResponseHeader", typeof(ResponseHeader));
            ServerProtocolVersion = decoder.ReadUInt32("ServerProtocolVersion");
            SecurityToken = (ChannelSecurityToken)decoder.ReadEncodeable("SecurityToken", typeof(ChannelSecurityToken));
            ServerNonce = decoder.ReadByteString("ServerNonce");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            OpenSecureChannelResponse value = encodeable as OpenSecureChannelResponse;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_responseHeader, value.m_responseHeader)) return false;
            if (!Utils.IsEqual(m_serverProtocolVersion, value.m_serverProtocolVersion)) return false;
            if (!Utils.IsEqual(m_securityToken, value.m_securityToken)) return false;
            if (!Utils.IsEqual(m_serverNonce, value.m_serverNonce)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (OpenSecureChannelResponse)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            OpenSecureChannelResponse clone = (OpenSecureChannelResponse)base.MemberwiseClone();

            clone.m_responseHeader = (ResponseHeader)Utils.Clone(this.m_responseHeader);
            clone.m_serverProtocolVersion = (uint)Utils.Clone(this.m_serverProtocolVersion);
            clone.m_securityToken = (ChannelSecurityToken)Utils.Clone(this.m_securityToken);
            clone.m_serverNonce = (byte[])Utils.Clone(this.m_serverNonce);

            return clone;
        }
        #endregion

        #region Private Fields
        private ResponseHeader m_responseHeader;
        private uint m_serverProtocolVersion;
        private ChannelSecurityToken m_securityToken;
        private byte[] m_serverNonce;
        #endregion
    }
    #endif
    #endregion

    #region CloseSecureChannelRequest Class
    #if (!OPCUA_EXCLUDE_CloseSecureChannelRequest)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class CloseSecureChannelRequest : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public CloseSecureChannelRequest()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_requestHeader = new RequestHeader();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "RequestHeader", IsRequired = false, Order = 1)]
        public RequestHeader RequestHeader
        {
            get
            {
                return m_requestHeader;
            }

            set
            {
                m_requestHeader = value;

                if (value == null)
                {
                    m_requestHeader = new RequestHeader();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.CloseSecureChannelRequest; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.CloseSecureChannelRequest_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.CloseSecureChannelRequest_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("RequestHeader", RequestHeader, typeof(RequestHeader));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            RequestHeader = (RequestHeader)decoder.ReadEncodeable("RequestHeader", typeof(RequestHeader));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            CloseSecureChannelRequest value = encodeable as CloseSecureChannelRequest;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_requestHeader, value.m_requestHeader)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (CloseSecureChannelRequest)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            CloseSecureChannelRequest clone = (CloseSecureChannelRequest)base.MemberwiseClone();

            clone.m_requestHeader = (RequestHeader)Utils.Clone(this.m_requestHeader);

            return clone;
        }
        #endregion

        #region Private Fields
        private RequestHeader m_requestHeader;
        #endregion
    }
    #endif
    #endregion

    #region CloseSecureChannelResponse Class
    #if (!OPCUA_EXCLUDE_CloseSecureChannelResponse)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class CloseSecureChannelResponse : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public CloseSecureChannelResponse()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_responseHeader = new ResponseHeader();
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ResponseHeader", IsRequired = false, Order = 1)]
        public ResponseHeader ResponseHeader
        {
            get
            {
                return m_responseHeader;
            }

            set
            {
                m_responseHeader = value;

                if (value == null)
                {
                    m_responseHeader = new ResponseHeader();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.CloseSecureChannelResponse; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.CloseSecureChannelResponse_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.CloseSecureChannelResponse_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("ResponseHeader", ResponseHeader, typeof(ResponseHeader));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ResponseHeader = (ResponseHeader)decoder.ReadEncodeable("ResponseHeader", typeof(ResponseHeader));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            CloseSecureChannelResponse value = encodeable as CloseSecureChannelResponse;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_responseHeader, value.m_responseHeader)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (CloseSecureChannelResponse)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            CloseSecureChannelResponse clone = (CloseSecureChannelResponse)base.MemberwiseClone();

            clone.m_responseHeader = (ResponseHeader)Utils.Clone(this.m_responseHeader);

            return clone;
        }
        #endregion

        #region Private Fields
        private ResponseHeader m_responseHeader;
        #endregion
    }
    #endif
    #endregion

    #region SignedSoftwareCertificate Class
    #if (!OPCUA_EXCLUDE_SignedSoftwareCertificate)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class SignedSoftwareCertificate : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public SignedSoftwareCertificate()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_certificateData = null;
            m_signature = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "CertificateData", IsRequired = false, Order = 1)]
        public byte[] CertificateData
        {
            get { return m_certificateData;  }
            set { m_certificateData = value; }
        }

        /// <remarks />
        [DataMember(Name = "Signature", IsRequired = false, Order = 2)]
        public byte[] Signature
        {
            get { return m_signature;  }
            set { m_signature = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.SignedSoftwareCertificate; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.SignedSoftwareCertificate_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.SignedSoftwareCertificate_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteByteString("CertificateData", CertificateData);
            encoder.WriteByteString("Signature", Signature);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            CertificateData = decoder.ReadByteString("CertificateData");
            Signature = decoder.ReadByteString("Signature");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            SignedSoftwareCertificate value = encodeable as SignedSoftwareCertificate;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_certificateData, value.m_certificateData)) return false;
            if (!Utils.IsEqual(m_signature, value.m_signature)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (SignedSoftwareCertificate)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SignedSoftwareCertificate clone = (SignedSoftwareCertificate)base.MemberwiseClone();

            clone.m_certificateData = (byte[])Utils.Clone(this.m_certificateData);
            clone.m_signature = (byte[])Utils.Clone(this.m_signature);

            return clone;
        }
        #endregion

        #region Private Fields
        private byte[] m_certificateData;
        private byte[] m_signature;
        #endregion
    }

    #region SignedSoftwareCertificateCollection Class
    /// <summary>
    /// A collection of SignedSoftwareCertificate objects.
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfSignedSoftwareCertificate", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "SignedSoftwareCertificate")]
    #if !NET_STANDARD
    public partial class SignedSoftwareCertificateCollection : List<SignedSoftwareCertificate>, ICloneable
    #else
    public partial class SignedSoftwareCertificateCollection : List<SignedSoftwareCertificate>
    #endif
    {
        #region Constructors
        /// <summary>
        /// Initializes the collection with default values.
        /// </summary>
        public SignedSoftwareCertificateCollection() {}

        /// <summary>
        /// Initializes the collection with an initial capacity.
        /// </summary>
        public SignedSoftwareCertificateCollection(int capacity) : base(capacity) {}

        /// <summary>
        /// Initializes the collection with another collection.
        /// </summary>
        public SignedSoftwareCertificateCollection(IEnumerable<SignedSoftwareCertificate> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <summary>
        /// Converts an array to a collection.
        /// </summary>
        public static implicit operator SignedSoftwareCertificateCollection(SignedSoftwareCertificate[] values)
        {
            if (values != null)
            {
                return new SignedSoftwareCertificateCollection(values);
            }

            return new SignedSoftwareCertificateCollection();
        }

        /// <summary>
        /// Converts a collection to an array.
        /// </summary>
        public static explicit operator SignedSoftwareCertificate[](SignedSoftwareCertificateCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <summary>
        /// Creates a deep copy of the collection.
        /// </summary>
        public object Clone()
        {
            return (SignedSoftwareCertificateCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SignedSoftwareCertificateCollection clone = new SignedSoftwareCertificateCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((SignedSoftwareCertificate)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region SignatureData Class
    #if (!OPCUA_EXCLUDE_SignatureData)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class SignatureData : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public SignatureData()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_algorithm = null;
            m_signature = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Algorithm", IsRequired = false, Order = 1)]
        public string Algorithm
        {
            get { return m_algorithm;  }
            set { m_algorithm = value; }
        }

        /// <remarks />
        [DataMember(Name = "Signature", IsRequired = false, Order = 2)]
        public byte[] Signature
        {
            get { return m_signature;  }
            set { m_signature = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.SignatureData; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.SignatureData_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.SignatureData_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Algorithm", Algorithm);
            encoder.WriteByteString("Signature", Signature);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Algorithm = decoder.ReadString("Algorithm");
            Signature = decoder.ReadByteString("Signature");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            SignatureData value = encodeable as SignatureData;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_algorithm, value.m_algorithm)) return false;
            if (!Utils.IsEqual(m_signature, value.m_signature)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (SignatureData)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SignatureData clone = (SignatureData)base.MemberwiseClone();

            clone.m_algorithm = (string)Utils.Clone(this.m_algorithm);
            clone.m_signature = (byte[])Utils.Clone(this.m_signature);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_algorithm;
        private byte[] m_signature;
        #endregion
    }
    #endif
    #endregion

    #region CreateSessionRequest Class
    #if (!OPCUA_EXCLUDE_CreateSessionRequest)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class CreateSessionRequest : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public CreateSessionRequest()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_requestHeader = new RequestHeader();
            m_clientDescription = new ApplicationDescription();
            m_serverUri = null;
            m_endpointUrl = null;
            m_sessionName = null;
            m_clientNonce = null;
            m_clientCertificate = null;
            m_requestedSessionTimeout = (double)0;
            m_maxResponseMessageSize = (uint)0;
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "RequestHeader", IsRequired = false, Order = 1)]
        public RequestHeader RequestHeader
        {
            get
            {
                return m_requestHeader;
            }

            set
            {
                m_requestHeader = value;

                if (value == null)
                {
                    m_requestHeader = new RequestHeader();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ClientDescription", IsRequired = false, Order = 2)]
        public ApplicationDescription ClientDescription
        {
            get
            {
                return m_clientDescription;
            }

            set
            {
                m_clientDescription = value;

                if (value == null)
                {
                    m_clientDescription = new ApplicationDescription();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "ServerUri", IsRequired = false, Order = 3)]
        public string ServerUri
        {
            get { return m_serverUri;  }
            set { m_serverUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "EndpointUrl", IsRequired = false, Order = 4)]
        public string EndpointUrl
        {
            get { return m_endpointUrl;  }
            set { m_endpointUrl = value; }
        }

        /// <remarks />
        [DataMember(Name = "SessionName", IsRequired = false, Order = 5)]
        public string SessionName
        {
            get { return m_sessionName;  }
            set { m_sessionName = value; }
        }

        /// <remarks />
        [DataMember(Name = "ClientNonce", IsRequired = false, Order = 6)]
        public byte[] ClientNonce
        {
            get { return m_clientNonce;  }
            set { m_clientNonce = value; }
        }

        /// <remarks />
        [DataMember(Name = "ClientCertificate", IsRequired = false, Order = 7)]
        public byte[] ClientCertificate
        {
            get { return m_clientCertificate;  }
            set { m_clientCertificate = value; }
        }

        /// <remarks />
        [DataMember(Name = "RequestedSessionTimeout", IsRequired = false, Order = 8)]
        public double RequestedSessionTimeout
        {
            get { return m_requestedSessionTimeout;  }
            set { m_requestedSessionTimeout = value; }
        }

        /// <remarks />
        [DataMember(Name = "MaxResponseMessageSize", IsRequired = false, Order = 9)]
        public uint MaxResponseMessageSize
        {
            get { return m_maxResponseMessageSize;  }
            set { m_maxResponseMessageSize = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.CreateSessionRequest; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.CreateSessionRequest_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.CreateSessionRequest_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("RequestHeader", RequestHeader, typeof(RequestHeader));
            encoder.WriteEncodeable("ClientDescription", ClientDescription, typeof(ApplicationDescription));
            encoder.WriteString("ServerUri", ServerUri);
            encoder.WriteString("EndpointUrl", EndpointUrl);
            encoder.WriteString("SessionName", SessionName);
            encoder.WriteByteString("ClientNonce", ClientNonce);
            encoder.WriteByteString("ClientCertificate", ClientCertificate);
            encoder.WriteDouble("RequestedSessionTimeout", RequestedSessionTimeout);
            encoder.WriteUInt32("MaxResponseMessageSize", MaxResponseMessageSize);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            RequestHeader = (RequestHeader)decoder.ReadEncodeable("RequestHeader", typeof(RequestHeader));
            ClientDescription = (ApplicationDescription)decoder.ReadEncodeable("ClientDescription", typeof(ApplicationDescription));
            ServerUri = decoder.ReadString("ServerUri");
            EndpointUrl = decoder.ReadString("EndpointUrl");
            SessionName = decoder.ReadString("SessionName");
            ClientNonce = decoder.ReadByteString("ClientNonce");
            ClientCertificate = decoder.ReadByteString("ClientCertificate");
            RequestedSessionTimeout = decoder.ReadDouble("RequestedSessionTimeout");
            MaxResponseMessageSize = decoder.ReadUInt32("MaxResponseMessageSize");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            CreateSessionRequest value = encodeable as CreateSessionRequest;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_requestHeader, value.m_requestHeader)) return false;
            if (!Utils.IsEqual(m_clientDescription, value.m_clientDescription)) return false;
            if (!Utils.IsEqual(m_serverUri, value.m_serverUri)) return false;
            if (!Utils.IsEqual(m_endpointUrl, value.m_endpointUrl)) return false;
            if (!Utils.IsEqual(m_sessionName, value.m_sessionName)) return false;
            if (!Utils.IsEqual(m_clientNonce, value.m_clientNonce)) return false;
            if (!Utils.IsEqual(m_clientCertificate, value.m_clientCertificate)) return false;
            if (!Utils.IsEqual(m_requestedSessionTimeout, value.m_requestedSessionTimeout)) return false;
            if (!Utils.IsEqual(m_maxResponseMessageSize, value.m_maxResponseMessageSize)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (CreateSessionRequest)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            CreateSessionRequest clone = (CreateSessionRequest)base.MemberwiseClone();

            clone.m_requestHeader = (RequestHeader)Utils.Clone(this.m_requestHeader);
            clone.m_clientDescription = (ApplicationDescription)Utils.Clone(this.m_clientDescription);
            clone.m_serverUri = (string)Utils.Clone(this.m_serverUri);
            clone.m_endpointUrl = (string)Utils.Clone(this.m_endpointUrl);
            clone.m_sessionName = (string)Utils.Clone(this.m_sessionName);
            clone.m_clientNonce = (byte[])Utils.Clone(this.m_clientNonce);
            clone.m_clientCertificate = (byte[])Utils.Clone(this.m_clientCertificate);
            clone.m_requestedSessionTimeout = (double)Utils.Clone(this.m_requestedSessionTimeout);
            clone.m_maxResponseMessageSize = (uint)Utils.Clone(this.m_maxResponseMessageSize);

            return clone;
        }
        #endregion

        #region Private Fields
        private RequestHeader m_requestHeader;
        private ApplicationDescription m_clientDescription;
        private string m_serverUri;
        private string m_endpointUrl;
        private string m_sessionName;
        private byte[] m_clientNonce;
        private byte[] m_clientCertificate;
        private double m_requestedSessionTimeout;
        private uint m_maxResponseMessageSize;
        #endregion
    }
    #endif
    #endregion

    #region CreateSessionResponse Class
    #if (!OPCUA_EXCLUDE_CreateSessionResponse)
    /// <summary>
    /// 
    /// </summary>
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class CreateSessionResponse : IEncodeable
    {
        #region Constructors
        /// <summary>
        /// The default constructor.
        /// </summary>
        public CreateSessionResponse()
        {
            Initialize();
        }

        /// <summary>
        /// Called by the .NET framework during deserialization.
        /// </summary>
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        /// <summary>
        /// Sets private members to default values.
        /// </summary>
        private void Initialize()
        {
            m_responseHeader = new ResponseHeader();
            m_sessionId = null;
            m_authenticationToken = null;
            m_revisedSessionTimeout = (double)0;
            m_serverNonce = null;
            m_serverCertificate = null;
            m_serverEndpoints = new EndpointDescriptionCollection();
            m_serverSoftwareCertificates = new SignedSoftwareCertificateCollection();
            m_serverSignature = new SignatureData();
            m_maxRequestMessageSize = (uint)0;
        }
        #endregion

        #region Public Properties
        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ResponseHeader", IsRequired = false, Order = 1)]
        public ResponseHeader ResponseHeader
        {
            get
            {
                return m_responseHeader;
            }

            set
            {
                m_responseHeader = value;

                if (value == null)
                {
                    m_responseHeader = new ResponseHeader();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "SessionId", IsRequired = false, Order = 2)]
        public NodeId SessionId
        {
            get { return m_sessionId;  }
            set { m_sessionId = value; }
        }

        /// <remarks />
        [DataMember(Name = "AuthenticationToken", IsRequired = false, Order = 3)]
        public NodeId AuthenticationToken
        {
            get { return m_authenticationToken;  }
            set { m_authenticationToken = value; }
        }

        /// <remarks />
        [DataMember(Name = "RevisedSessionTimeout", IsRequired = false, Order = 4)]
        public double RevisedSessionTimeout
        {
            get { return m_revisedSessionTimeout;  }
            set { m_revisedSessionTimeout = value; }
        }

        /// <remarks />
        [DataMember(Name = "ServerNonce", IsRequired = false, Order = 5)]
        public byte[] ServerNonce
        {
            get { return m_serverNonce;  }
            set { m_serverNonce = value; }
        }

        /// <remarks />
        [DataMember(Name = "ServerCertificate", IsRequired = false, Order = 6)]
        public byte[] ServerCertificate
        {
            get { return m_serverCertificate;  }
            set { m_serverCertificate = value; }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ServerEndpoints", IsRequired = false, Order = 7)]
        public EndpointDescriptionCollection ServerEndpoints
        {
            get
            {
                return m_serverEndpoints;
            }

            set
            {
                m_serverEndpoints = value;

                if (value == null)
                {
                    m_serverEndpoints = new EndpointDescriptionCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ServerSoftwareCertificates", IsRequired = false, Order = 8)]
        public SignedSoftwareCertificateCollection ServerSoftwareCertificates
        {
            get
            {
                return m_serverSoftwareCertificates;
            }

            set
            {
                m_serverSoftwareCertificates = value;

                if (value == null)
                {
                    m_serverSoftwareCertificates = new SignedSoftwareCertificateCollection();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        [DataMember(Name = "ServerSignature", IsRequired = false, Order = 9)]
        public SignatureData ServerSignature
        {
            get
            {
                return m_serverSignature;
            }

            set
            {
                m_serverSignature = value;

                if (value == null)
                {
                    m_serverSignature = new SignatureData();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "MaxRequestMessageSize", IsRequired = false, Order = 10)]
        public uint MaxRequestMessageSize
        {
            get { return m_maxRequestMessageSize;  }
            set { m_maxRequestMessageSize = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId
        {
            get { return DataTypeIds.CreateSessionResponse; }
        }

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId
        {
            get { return ObjectIds.CreateSessionResponse_Encoding_DefaultBinary; }
        }

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId
        {
            get { return ObjectIds.CreateSessionResponse_Encoding_DefaultXml; }
        }

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("ResponseHeader", ResponseHeader, typeof(ResponseHeader));
            encoder.WriteNodeId("SessionId", SessionId);
            encoder.WriteNodeId("AuthenticationToken", AuthenticationToken);
            encoder.WriteDouble("RevisedSessionTimeout", RevisedSessionTimeout);
            encoder.WriteByteString("ServerNonce", ServerNonce);
            encoder.WriteByteString("ServerCertificate", ServerCertificate);
            encoder.WriteEncodeableArray("ServerEndpoints", ServerEndpoints.ToArray(), typeof(EndpointDescription));
            encoder.WriteEncodeableArray("ServerSoftwareCertificates", ServerSoftwareCertificates.ToArray(), typeof(SignedSoftwareCertificate));
            encoder.WriteEncodeable("ServerSignature", ServerSignature, typeof(SignatureData));
            encoder.WriteUInt32("MaxRequestMessageSize", MaxRequestMessageSize);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ResponseHeader = (ResponseHeader)decoder.ReadEncodeable("ResponseHeader", typeof(ResponseHeader));
            SessionId = decoder.ReadNodeId("SessionId");
            AuthenticationToken = decoder.ReadNodeId("AuthenticationToken");
            RevisedSessionTimeout = decoder.ReadDouble("RevisedSessionTimeout");
            ServerNonce = decoder.ReadByteString("ServerNonce");
            ServerCertificate = decoder.ReadByteString("ServerCertificate");
            ServerEndpoints = (EndpointDescriptionCollection)decoder.ReadEncodeableArray("ServerEndpoints", typeof(EndpointDescription));
            ServerSoftwareCertificates = (SignedSoftwareCertificateCollection)decoder.ReadEncodeableArray("ServerSoftwareCertificates", typeof(SignedSoftwareCertificate));
            ServerSignature = (SignatureData)decoder.ReadEncodeable("ServerSignature", typeof(SignatureData));
            MaxRequestMessageSize = decoder.ReadUInt32("MaxRequestMessageSize");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            CreateSessionResponse value = encodeable as CreateSessionResponse;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_responseHeader, value.m_responseHeader)) return false;
            if (!Utils.IsEqual(m_sessionId, value.m_sessionId)) return false;
            if (!Utils.IsEqual(m_authenticationToken, value.m_authenticationToken)) return false;
          