/*  This file is part of the KDE libraries
    Copyright (C) 1998 Jrgen Hochwald (juergen.hochwald@privat.kkf.net

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this library; see the file COPYING.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.*/



#include <stdio.h>
#include <math.h>
#include "kwr.h"
#include "kwr.moc"
#include "kapp.h"

// #define PI_2  1.570796327
float AR1[8] = {0, 0.4, 0.3, 1, 0.8, 1, 0.3, 0.4};
float AW1[8] = {0-90, -45-90, -20-90,-15-90, 0-90,   15-90,20-90,  45-90};
float AR2[3] = {0, 1,   0.7};
float AW2a[3] = {0-90, 20-90,  0-90};
float AW2b[3] = {0-90, -20-90,  0-90};

static float ResArr[4] = {90, 45, 22.5, 11.25};
static const char* DirText[33]={"N","N_E","NNE","NE_N","NE","NE_E","ENE","E_N",
                          "E","E_S","ESE","SE_E","SE","SE_S","SSE","S_E",
                          "S","S_W","SSW","SW_S","SW","SW_W","WSW","W_S",
                          "W","W_N","WNW","NW_W","NW","NW_N","NNW","N_W",
                          ""};


void Pol(float x, float y, float *r ,float *w)
// Umwandlung in Polarkoordinaten
{
   *r = sqrt(x*x+y*y);
   if (x==0.0000) { /* x etwa Null*/
      *r = (y<0) ? -y : y;
      if (y>0.0) *w = M_PI_2; else *w = -M_PI_2;
      return;
   }
   *w = atan(y/x);
   if (x<0.0)
   if (y>0.0)
      *w += M_PI;
   else
      *w += M_PI;
}

void Rect(float r, float w, float *x, float *y)
// Umwandlung in rechtwinklige Koordinaten
{
   *x = r*cos(w);
   *y = r*sin(w);
}

KWindRose::KWindRose( QWidget* parent, const char* name,WFlags f ,bool allowLines)
:QFrame (parent, name, f, allowLines)
{
   //debug("Constructor KWindRose");

   setFrameStyle( Panel | Sunken );
   setLineWidth( 1 );
   setFocusPolicy(QWidget::StrongFocus);

   Paint = new QPainter();
   Pnts3 = new QPointArray(3);
   Pnts8 = new QPointArray(8);

   PlusBtn = new QPushButton("+", this);
   PlusBtn->setAutoRepeat(TRUE);
   PlusBtn->setFocusPolicy(QWidget::NoFocus);
   MinusBtn = new QPushButton("-", this);
   MinusBtn->setAutoRepeat(TRUE);
   MinusBtn->setFocusPolicy(QWidget::NoFocus);
   connect(PlusBtn, SIGNAL(clicked()), SLOT(PlusBtnClick()));
   connect(MinusBtn, SIGNAL(clicked()), SLOT(MinusBtnClick()));

   ColL = qRgb(0,0,0);
   Col1 = qRgb(128,128,255);
   Col2 = qRgb(192,255,255);
   Col3 = qRgb(16,16,16);
   Col4 = qRgb(64,64,64);
   Direction = 0; // = Nord
   Width = 0.2;
   Details=2;  // 8 Spitzen
   ArrowType = ARROW1;
   Editable = TRUE;
   Resolution = 1.0;
   Valid = KWR_VALID;
}

KWindRose::~KWindRose(void) {
   //debug("Destructor KWindRose");
   delete Paint;
   delete Pnts3;
   delete Pnts8;
}

void KWindRose::paintEvent(QPaintEvent * e) {
float r,w,x,y,cx,cy,x1,y1,x2,y2;
float Size,Dir;  // Gre und Richtung von dem Pfeil
int i,s;

   QBrush Brush1(Col1);
   QBrush Brush2(Col2);
   QBrush Brush3(Col3);
   QBrush Brush4(Col4);
   
   QFrame::paintEvent(e);

   Paint->begin(this);

   // Mitte der Anzeige
   cx = 0.5*width();
   cy = 0.5*height();
   
   if (width() > height())
      r = height();
   else
      r = width();
      
   r = (0.5*r) - 2*frameWidth();// - 5;
   w = 0.0;

   {
      QPen P(ColL,0,NoPen);
      Paint->setPen(P);
   }

   //debug("Details = %d",Details);
   // ganz feine Teilung (12.125)
   if (Details>=4)
   for (i=1; i<32; i+=2) {
      w = i/16.0*M_PI;
      Rect(0.7*r,w, &x,&y);
      Rect(Width*0.7*r,(w+M_PI_2), &x1,&y1);
      Rect(Width*0.7*r,(w-M_PI_2), &x2,&y2);
      
      // helle Seite zeichnen
      Pnts3->setPoint(0,(int)(cx+x), (int)(cy+y));
      Pnts3->setPoint(1,(int)(cx+x1),(int)(cy+y1));
      Pnts3->setPoint(2,(int)(cx),   (int)(cy));
      Paint->setBrush(Brush1);
      Paint->drawPolygon(*Pnts3);

      // dunkle Seite zeichnen
      Pnts3->setPoint(0,(int)(cx+x), (int)(cy+y));
      Pnts3->setPoint(1,(int)(cx),   (int)(cy));
      Pnts3->setPoint(2,(int)(cx+x2),(int)(cy+y2));
      Paint->setBrush(Brush2);
      Paint->drawPolygon(*Pnts3);
   }
   // 22.5
   if (Details>=3)
   for (i=2; i<32; i+=4) {
      w = i/16.0*M_PI;
      Rect(0.8*r,w, &x,&y);
      Rect(Width*0.8*r,(w+M_PI_2), &x1,&y1);
      Rect(Width*0.8*r,(w-M_PI_2), &x2,&y2);
      
      // helle Seite zeichnen
      Pnts3->setPoint(0,(int)(cx+x), (int)(cy+y));
      Pnts3->setPoint(1,(int)(cx+x1),(int)(cy+y1));
      Pnts3->setPoint(2,(int)(cx),   (int)(cy));
      Paint->setBrush(Brush1);
      Paint->drawPolygon(*Pnts3);

      // dunkle Seite zeichnen
      Pnts3->setPoint(0,(int)(cx+x), (int)(cy+y));
      Pnts3->setPoint(1,(int)(cx),   (int)(cy));
      Pnts3->setPoint(2,(int)(cx+x2),(int)(cy+y2));
      Paint->setBrush(Brush2);
      Paint->drawPolygon(*Pnts3);
   }
   // 45 
   if (Details>=2)
   for (i=4; i<32; i+=8) {
      w = i/16.0*M_PI;
      Rect(0.9*r,w, &x,&y);
      Rect(Width*0.9*r,(w+M_PI_2), &x1,&y1);
      Rect(Width*0.9*r,(w-M_PI_2), &x2,&y2);
      
      // helle Seite zeichnen
      Pnts3->setPoint(0,(int)(cx+x), (int)(cy+y));
      Pnts3->setPoint(1,(int)(cx+x1),(int)(cy+y1));
      Pnts3->setPoint(2,(int)(cx),   (int)(cy));
      Paint->setBrush(Brush1);
      Paint->drawPolygon(*Pnts3);

      // dunkle Seite zeichnen
      Pnts3->setPoint(0,(int)(cx+x), (int)(cy+y));
      Pnts3->setPoint(1,(int)(cx),   (int)(cy));
      Pnts3->setPoint(2,(int)(cx+x2),(int)(cy+y2));
      Paint->setBrush(Brush2);
      Paint->drawPolygon(*Pnts3);
   }
   // 90 
   if (Details>=1)
   for (i=0; i<32; i+=8) {
      w = i/16.0*M_PI;
      Rect(r,w, &x,&y);
      Rect(Width*r,(w+M_PI_2), &x1,&y1);
      Rect(Width*r,(w-M_PI_2), &x2,&y2);
      
      // helle Seite zeichnen
      Pnts3->setPoint(0,(int)(cx+x), (int)(cy+y));
      Pnts3->setPoint(1,(int)(cx+x1),(int)(cy+y1));
      Pnts3->setPoint(2,(int)(cx),   (int)(cy));
      Paint->setBrush(Brush1);
      Paint->drawPolygon(*Pnts3);

      // dunkle Seite zeichnen
      Pnts3->setPoint(0,(int)(cx+x), (int)(cy+y));
      Pnts3->setPoint(1,(int)(cx),   (int)(cy));
      Pnts3->setPoint(2,(int)(cx+x2),(int)(cy+y2));
      Paint->setBrush(Brush2);
      Paint->drawPolygon(*Pnts3);
   }

   // Mitte ausmalen (wenn Valid=Invalid, dann in rot)
   if (Valid==KWR_INVALID) {
      QBrush B(qRgb(255,0,0));
      s = (int)(Width*1.3*r);  // 1.3 durch Ausprobieren
     
      Paint->setBrush(B);
      Paint->drawEllipse((int)(cx-s),(int)(cy-s),  s+s,  s+s);
   } else {
      for (i=0; i<4; i++) {
         s = (int)(Width*1.3*r);  // 1.3 durch Ausprobieren
         if (i%2)
            Paint->setBrush(Brush1);
         else
            Paint->setBrush(Brush2);
         
         Paint->drawPie((int)(cx-s),(int)(cy-s),  s+s,  s+s,  i*1440, 1440);
      }
   }

   // Pfeil malen (nur, wenn Valid=KWR_VALID ist)
   if (Valid==KWR_VALID) {   
      Size = r; //r-10;
      Dir = Direction/180.0*M_PI;
      if (ArrowType==ARROW1) {
         Pnts8->setPoint(0,(int)cx,(int)cy);
         for (i=1; i<8; i++) {
            Rect(AR1[i]*Size, AW1[i]*2*M_PI/360+Dir, &x,&y);
            Pnts8->setPoint(i,(int)(cx+x),(int)(cy+y));
         }
         Paint->setBrush(Brush3);
         Paint->drawPolygon(*Pnts8);
      } else {
         // rechte Hlfte
         Pnts3->setPoint(0,(int)cx,(int)cy);
         for (i=1; i<3; i++) {
            Rect(AR2[i]*Size, AW2a[i]*2*M_PI/360+Dir, &x,&y);
            Pnts3->setPoint(i,(int)(cx+x),(int)(cy+y));
         }
         Paint->setBrush(Brush3);
         Paint->drawPolygon(*Pnts3);

         // linke Seite (Punkt 1 mit negativen Winkel)
         for (i=1; i<3; i++) {
            Rect(AR2[i]*Size, AW2b[i]*2*M_PI/360+Dir, &x,&y);
            Pnts3->setPoint(i,(int)(cx+x),(int)(cy+y));
         }
         Paint->setBrush(Brush4);
         Paint->drawPolygon(*Pnts3);
      }
   }

   Paint->end();
}

void KWindRose::resizeEvent ( QResizeEvent * rev) {
int Size;
   if (height()>width())
      Size=width();
   else
      Size=height();
   Size=Size / 20;
   if (Size<10) Size=10;
   //debug("resizeEvent");
   MinusBtn->setGeometry(0,height()-frameWidth()-Size, Size,Size);
   PlusBtn->setGeometry(width()-frameWidth()-Size,height()-frameWidth()-Size,  Size,Size);
}

void KWindRose::mousePressEvent(QMouseEvent * qme) {
   if (qme->button() == LeftButton)
      CalcDir(qme);
}

void KWindRose::CalcDir(QMouseEvent * qme) {
// neue Richtung berechnen und setzen
int x,y;
float r,w;
int s;

   if (!Editable) return;  // nicht editierbar

   // Gre der Mitte berechnen
   if (width() > height())
      r = height();
   else
      r = width();
   r = (0.5*r) - 2*frameWidth();// - 5;
   s = (int)(Width*1.3*r);  // 1.3 durch Ausprobieren

   x = qme->x() - width()/2;
   y = qme->y() - height()/2;
   Pol(x,y,&r,&w);
   w = w*180.0/M_PI + 90.0;
   //debug("x=%d  y=%d",x,y);
   //debug("r=%f  w=%f",r,w);
   //Direction = w;
   if (Resolution>0)
      w = w-fmod(w+0.5*Resolution,Resolution)+0.5*Resolution;
   w = fmod(w,360.0);
   
   //debug("r=%f  s=%d",r,s);
   if (r<s)
     Valid=KWR_NODIR;
   else
     Valid=KWR_VALID;

   if (Direction!=w) {
      //debug("dir=%f  w=%f",Direction,w);
      Direction = w;
      update();
      emit valueChanged(Direction);
   }
// directionToText(Direction, 3);
}

void KWindRose::mouseMoveEvent(QMouseEvent * qme) {
   if (qme->state() == LeftButton)
      CalcDir(qme);
}

void KWindRose::setColors(QColor c1, QColor c2, QColor c3, QColor c4) {
   Col1 = c1;
   Col2 = c2;
   Col3 = c3;
   Col4 = c4;
   update();
}

void KWindRose::setRoseWidth(float w) {
   Width=w;
   // Bereich checken
   if (Width<0.03) Width=0.03;
   if (Width>0.4) Width=0.4;
   update();
}

void KWindRose::setDetails(int d) {
   //debug("setDetails %d",d);
   if ( (d>=1) && (d<=4) ) {
      Details=d;
      update();
   }
}

void KWindRose::setColor(int Num, QColor Col) {
   //debug("KWindRose::setColor");
   switch (Num) {
      case 1 : Col1=Col; break;
      case 2 : Col2=Col; break;
      case 3 : Col3=Col; break;
      case 4 : Col4=Col; break;
      default: debug ("Invalid color index %d",Num);
   }
   update();
}

int KWindRose::getDetails(void) {
   return Details;
}

void KWindRose::setDirection(float Angle) {
   Direction = fmod(Angle,360);
   Valid=KWR_VALID;
   update();
}

float KWindRose::getDirection(void) {
   return Direction;
}

QColor KWindRose::getColor(int Num) {
   if (Num==1) return Col1; else
   if (Num==2) return Col2; else
   if (Num==3) return Col3; else
   if (Num==4) return Col4; else
   return qRgb(0,0,0);
}

void KWindRose::setArrowType(int T) {
   if (T==ARROW1) ArrowType=ARROW1; else
   if (T==ARROW2) ArrowType=ARROW2; else
   ArrowType=ARROW1;
}

void KWindRose::setEditable(int E) {
// Editierbar-Bit setzen

   Editable = (E!=0);
   if (E) {
      PlusBtn->show();
      MinusBtn->show();
   } else {
      PlusBtn->hide();
      MinusBtn->hide();
   }
}

void KWindRose::setResolution(float Res) {
// Auflsung sezten

   Resolution=Res;
   if (Resolution>0) {
      Direction = Direction-fmod(Direction,Resolution);
      update();
   }
}

void KWindRose::PlusBtnClick() {
   //debug("Plus");
   if (Valid!=KWR_VALID) {
     Valid=KWR_VALID;
     Direction=0.0;
   }
   Direction += (Resolution>0)?Resolution:1;
   if (Direction>=360.0) Direction-=360.0;
   update();
   emit valueChanged(Direction);
}

void KWindRose::MinusBtnClick() {
   //debug("minus");
   if (Valid!=KWR_VALID) {
     Valid=KWR_VALID;
     Direction=0.0;
   }
   Direction-=(Resolution>0)?Resolution:1;
   if (Direction<0.0) Direction+=360.0;
   update();
   emit valueChanged(Direction);
}
      
void KWindRose::focusInEvent ( QFocusEvent * ) {
   //debug("focusin");
}
void KWindRose::focusOutEvent ( QFocusEvent * ) {
   //debug("focusout");
}
void KWindRose::keyPressEvent (QKeyEvent *kev) {
int OldVal;
   //debug("keypress");
   if (!Editable) return;

   OldVal=Valid;
   if (Valid!=KWR_VALID) {
     Valid=KWR_VALID;
     Direction=0.0;
   }

   switch (kev->key()) {
      case Key_Minus:
      case Key_Left : 
      case Key_Up   :
      case Key_Prior: MinusBtnClick();
                      return;

      case Key_Plus :
      case Key_Right: 
      case Key_Down :
      case Key_Next : PlusBtnClick();
                      return;  // return, damit "valueChanged" nicht zweimal ausgelst wird
                               // Ist schon in den BtnClick-Funktionen enthalten

      case Key_8    : Direction = 0; 
                      update();
                      break;
      case Key_9    : Direction = 45; 
                      update();
                      break;
      case Key_6    : Direction = 90; 
                      update();
                      break;
      case Key_3    : Direction = 135; 
                      update();
                      break;
      case Key_2    : Direction = 180; 
                      update();
                      break;
      case Key_1    : Direction = 225; 
                      update();
                      break;
      case Key_4    : Direction = 270; 
                      update();
                      break;
      case Key_7    : Direction = 315; 
                      update();
                      break;
      case Key_5    : Direction = 0; 
                      Valid=KWR_NODIR;
                      update();
                      break;
      case Key_Space:
      case Key_0    : 
                      Valid=KWR_INVALID;
                      update();
                      break;
      default       : kev->ignore();
                      Valid=OldVal;
   }

   emit valueChanged(Direction);

}

const char* KWindRose::getDirText(int Item) {
   if ( (Item<0) || (Item>=33) )
      return NULL;
   else {
      if (Item==32)
        return "";
      else
        return i18n(DirText[Item]);
   }
}

int KWindRose::directionToIndex(float Dir, int Res) {
int d;
float R,D;

   Res--;
   if ((Res<0)||(Res>=4)) Res=2;  //out of range
   R=ResArr[Res];
   D = Dir-fmod(Dir+0.5*R,R)+0.5*R;
   D = fmod(D,360.0);
   d = (int)(D/ResArr[Res]);
   d = d << (3-Res);
   //debug("directionToIndex: d=%2d   %s",d,DirText[d]);
   return d;
}

const char* KWindRose::directionToText(float Dir, int Res) {
/*
int d;
float R,D;

   if ((Res<0)||(Res>=4)) Res=2;  //out of range
   R=ResArr[Res];
   D = Dir-fmod(Dir+0.5*R,R)+0.5*R;
   D = fmod(D,360.0);
   d = (int)(D/ResArr[Res]);
   d = d << (3-Res);
   debug("directionToText: d=%2d   %s",d,DirText[d]);
*/
   int i=directionToIndex(Dir,Res);
   if (i>=32)
     return "";
   else
     return i18n(DirText[directionToIndex(Dir,Res)]);
}

const char* KWindRose::directionToText(int Res) {
   if (Valid!=KWR_VALID) return "";
   return directionToText(Direction, Res);
}

const char* KWindRose::directionToText(void) {
   if (Valid!=KWR_VALID) return "";
   return directionToText(Direction, Details-1);
}

int KWindRose::indexToDirection(int Index) {
// Index in Richtung umwandeln

   if (Index<0) return FALSE;
   if (Index>=33) return FALSE;  // Out of range
   if (Index==32) {
     Valid=KWR_NODIR;
   } else {
     Direction=Index*11.25;
     Valid=KWR_VALID;
   }
   update();
   return TRUE;
   
}

int KWindRose::textToDirection(const char* Text) {
// Text in Richtung umwandeln
int Len,Lauf;
char S[10];

   Len=strlen(Text);
   //if (Len<=0) return FALSE;
   if (Len>4) return FALSE;  // Lngenfehler

   strcpy(S,Text);
   // Vorformattieren
   for (Lauf=0; Lauf<Len; Lauf++) {
      /*switch (S[Lauf]) {
         case ' ':
         case 'N':
         case 'W':
         case 'S':
         case 'O': break;
         case 'n':
         case 'w':
         case 's':
         case 'o': S[Lauf]-=0x20; break;
         default : S[Lauf]='_';
      }*/
      if (S[Lauf]>='a' && S[Lauf]<='z')
        S[Lauf] -= 32;
      //strcpy(S,i18n(S));
   }
   //debug(">%s<",S);
   // Vergleichen
   if ( (strlen(S)==0)||(strcmp(S," ")==0) ) {  // keine Richtungsangabe
     Direction=0.0;
     Valid=KWR_NODIR;
     update();
     return TRUE;
   }
   for (Lauf=0; Lauf<33; Lauf++) {
//debug("%s   %s",S,DirText[Lauf]);
      if (strcmp(S, i18n(DirText[Lauf]))==0) {
         // gefunden
         Direction = 11.25*Lauf;
         Valid=KWR_VALID;
         update();
         return TRUE;
      }
   }
   
   // ungltige Richtungsangabe
   Direction=0.0;
   Valid=KWR_INVALID;
   update();
   return FALSE;
}

int KWindRose::getValid(void) {
  return Valid;
}

void KWindRose::setValid(int Vld) {
  Valid=Vld;
  update();
}
