#include <fstream.h>
#include <unistd.h>

#include "kdbm.h"

#define WORDS_FILE  "/usr/dict/words"
#define DICT_FILE   "words.db"

const int MAX_WORDS = 50000;

char* dict[MAX_WORDS];
char* shuffled_dict[MAX_WORDS];
int num_words = 0;
int max_words = MAX_WORDS;

int
get_random_index()
{
  return rand() % num_words;
}

void
load_dict()
{
  cout << "Loading dictionary: ";

  ifstream in;

  in.open(WORDS_FILE);

  char buf[40];

  while (!in.eof() && num_words < max_words) {
    in >> buf;
    dict[num_words] = strdup(buf);
    shuffled_dict[num_words] = dict[num_words];
    num_words++;

    if ((num_words % 50) == 0) {
      cout << '.';
      cout.flush();
    }
  }
  
  in.close();

  cout << endl;
}

void
shuffle_dict()
{
  for (int i = 0; i < num_words; i++) {
    int index = get_random_index();
    char* tmp = shuffled_dict[index];
    shuffled_dict[index] = shuffled_dict[i];
    shuffled_dict[i] = tmp;
  }
}

void
store_dict()
{

  cout << "Creating database: ";

  KDbm<int> db;

  db.setFastMode(true);
  db.create(DICT_FILE);

  for (int i = 0; i < num_words; i++) {
    if (db.exists(shuffled_dict[i])) {
      cerr << "fatal error: " << shuffled_dict[i] << "already exists" << endl;
      exit(-1);
    }

    db.insert(shuffled_dict[i], i);

    if ((i % 50) == 0) {
      cout << '.';
      cout.flush();
    }
  }

  db.close();

  cout << endl;
}

void
check_dict()
{
  cout << "Checking database: ";

  KDbm<int> db;

  db.open(DICT_FILE);

  assert(db.isOpen);

  for (int i = 0; i < num_words; i++) {
    int val = *db[dict[i]];

    if ((i % 50) == 0)
      cout << '.';

    if (!db.found()) {
      cerr << "fatal error: " << dict[i] << " not found. Please report bug." << endl;
      exit(-1);
    }
    
    if (strcmp(shuffled_dict[val], dict[i]) != 0) {
      cerr << "fatal error: retrieved item (";
      cerr << shuffled_dict[val] << ") does not match with ";
      cerr << dict[i] << ". Please report bug." << endl;
      exit(-1);
    }
  }

  db.close();

  cout << endl << "CHECK OK -- TEST PASSED" << endl;
}

int
main(int argc, char* argv[])
{
  if (argc == 2)
    max_words = atoi(argv[1]);

  if (max_words > MAX_WORDS)
    max_words = MAX_WORDS;

  ifstream in;
  in.open(WORDS_FILE);

  if (!in.is_open()) {
    cerr << "Please install " WORDS_FILE " to run the test." << endl;
    cerr << "Cannot run test." << endl;
    exit(-1);
  }
  else
    in.close();

  cout << "WARNING: this test may run for several minutes..." << endl;
  cout << "Testing with " << max_words << " words..." << endl;
  cout << "(to run the test with a different number of words, run as: " << argv[0] << " <num_words>)" << endl;

  load_dict();
  shuffle_dict();
  store_dict();
  check_dict();

  unlink(DICT_FILE);
}
