#!/usr/bin/env python

"""
A Web interface to a calendar event.

Copyright (C) 2014, 2015, 2016 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from imiptools.data import get_uri, get_verbose_address, uri_dict, uri_items, \
                           uri_parts, uri_values
from imiptools.dates import format_datetime, to_timezone
from imiptools.mail import Messenger
from imipweb.data import EventPeriod, event_period_from_period, FormPeriod, PeriodError
from imipweb.resource import DateTimeFormUtilities, FormUtilities, ResourceClientForObject

# Fake gettext method for strings to be translated later.

_ = lambda s: s

class EventPageFragment(ResourceClientForObject, DateTimeFormUtilities, FormUtilities):

    "A resource presenting the details of an event."

    def __init__(self, resource=None):
        ResourceClientForObject.__init__(self, resource)

    # Various property values and labels.

    property_items = [
        ("SUMMARY",     _("Summary")),
        ("DTSTART",     _("Start")),
        ("DTEND",       _("End")),
        ("ORGANIZER",   _("Organiser")),
        ("ATTENDEE",    _("Attendee")),
        ]

    partstat_items = [
        ("NEEDS-ACTION", _("Not confirmed")),
        ("ACCEPTED",    _("Attending")),
        ("TENTATIVE",   _("Tentatively attending")),
        ("DECLINED",    _("Not attending")),
        ("DELEGATED",   _("Delegated")),
        (None,          _("Not indicated")),
        ]

    def can_remove_recurrence(self, recurrence):

        """
        Return whether the 'recurrence' can be removed from the current object
        without notification.
        """

        return (self.can_edit_recurrence(recurrence) or not self.is_organiser()) and \
               recurrence.origin != "RRULE"

    def can_edit_recurrence(self, recurrence):

        "Return whether 'recurrence' can be edited."

        return self.recurrence_is_new(recurrence) or not self.obj.is_shared()

    def recurrence_is_new(self, recurrence):

        "Return whether 'recurrence' is new to the current object."

        return recurrence not in self.get_stored_recurrences()

    def can_remove_attendee(self, attendee):

        """
        Return whether 'attendee' can be removed from the current object without
        notification.
        """

        return self.can_edit_attendee(attendee) or attendee == self.user and self.is_organiser()

    def can_edit_attendee(self, attendee):

        "Return whether 'attendee' can be edited by an organiser."

        return self.attendee_is_new(attendee) or not self.obj.is_shared()

    def attendee_is_new(self, attendee):

        "Return whether 'attendee' is new to the current object."

        return attendee not in uri_values(self.get_stored_attendees())

    # Access to stored object information.

    def get_stored_attendees(self):
        return [get_verbose_address(value, attr) for value, attr in self.obj.get_items("ATTENDEE") or []]

    def get_stored_main_period(self):

        "Return the main event period for the current object."

        (dtstart, dtstart_attr), (dtend, dtend_attr) = self.obj.get_main_period_items()
        return EventPeriod(dtstart, dtend, self.get_tzid(), None, dtstart_attr, dtend_attr)

    def get_stored_recurrences(self):

        "Return recurrences computed using the current object."

        recurrenceids = self._get_recurrences(self.uid)
        recurrences = []
        for period in self.get_periods(self.obj):
            period = event_period_from_period(period)
            period.replaced = period.is_replaced(recurrenceids)
            if period.origin != "DTSTART":
                recurrences.append(period)
        return recurrences

    # Access to current object information.

    def get_current_main_period(self):
        return self.get_stored_main_period()

    def get_current_recurrences(self):
        return self.get_stored_recurrences()

    def get_current_attendees(self):
        return self.get_stored_attendees()

    # Page fragment methods.

    def show_request_controls(self):

        "Show form controls for a request."

        _ = self.get_translator()

        page = self.page
        args = self.env.get_args()

        attendees = uri_values(self.get_current_attendees())
        is_attendee = self.user in attendees

        if not self.obj.is_shared():
            page.p(_("This event has not been shared."))

        # Show appropriate options depending on the role of the user.

        if is_attendee and not self.is_organiser():
            page.p(_("An action is required for this request:"))

            page.p()
            self.control("reply", "submit", _("Send reply"))
            page.add(" ")
            self.control("discard", "submit", _("Discard event"))
            page.add(" ")
            self.control("ignore", "submit", _("Return to the calendar"), class_="ignore")
            page.p.close()

        if self.is_organiser():
            page.p(_("As organiser, you can perform the following:"))

            page.p()
            self.control("create", "submit", _("Update event"))
            page.add(" ")

            if self._get_counters(self.uid, self.recurrenceid):
                self.control("uncounter", "submit", _("Ignore counter-proposals"))
                page.add(" ")

            if self.obj.is_shared() and not self._is_request():
                self.control("cancel", "submit", _("Cancel event"))
            else:
                self.control("discard", "submit", _("Discard event"))

            page.add(" ")
            self.control("ignore", "submit", _("Return to the calendar"), class_="ignore")
            page.add(" ")
            self.control("save", "submit", _("Save without sending"))
            page.p.close()

    def show_object_on_page(self, errors=None):

        """
        Show the calendar object on the current page. If 'errors' is given, show
        a suitable message for the different errors provided.
        """

        _ = self.get_translator()

        page = self.page
        page.form(method="POST")
        self.validator()

        # Add a hidden control to help determine whether editing has already begun.

        self.control("editing", "hidden", "true")

        args = self.env.get_args()

        # Obtain basic event information, generating any necessary editing controls.

        attendees = self.get_current_attendees()
        period = self.get_current_main_period()
        stored_period = self.get_stored_main_period()
        self.show_object_datetime_controls(period)

        # Obtain any separate recurrences for this event.

        recurrenceids = self._get_recurrences(self.uid)
        replaced = not self.recurrenceid and period.is_replaced(recurrenceids)
        excluded = period == stored_period and period not in self.get_periods(self.obj)

        # Provide a summary of the object.

        page.table(class_="object", cellspacing=5, cellpadding=5)
        page.thead()
        page.tr()
        page.th(_("Event"), class_="mainheading", colspan=3)
        page.tr.close()
        page.thead.close()
        page.tbody()

        for name, label in self.property_items:
            field = name.lower()

            items = uri_items(self.obj.get_items(name) or [])
            rowspan = len(items)

            # Adjust rowspan for add button rows.
            # Skip properties without items apart from attendee (where items
            # may be added) and the end datetime (which might be described by a
            # duration property).

            if name in "ATTENDEE":
                rowspan = len(attendees) + 1
            elif name == "DTEND":
                rowspan = 2
            elif not items:
                continue

            page.tr()
            page.th(_(label), class_="objectheading %s%s" % (field, errors and field in errors and " error" or ""), rowspan=rowspan)

            # Handle datetimes specially.

            if name in ("DTSTART", "DTEND"):
                if not replaced and not excluded:

                    # Obtain the datetime.

                    is_start = name == "DTSTART"

                    # Where no end datetime exists, use the start datetime as the
                    # basis of any potential datetime specified if dt-control is
                    # set.

                    self.show_datetime_controls(is_start and period.get_form_start() or period.get_form_end(), is_start)

                elif name == "DTSTART":

                    # Replaced occurrences link to their replacements.

                    if replaced:
                        page.td(class_="objectvalue %s replaced" % field, rowspan=2, colspan=2)
                        page.a(_("First occurrence replaced by a separate event"), href=self.link_to(self.uid, replaced))
                        page.td.close()

                    # NOTE: Should provide a way of editing recurrences when the
                    # NOTE: first occurrence is excluded, plus a way of
                    # NOTE: reinstating the occurrence.

                    elif excluded:
                        page.td(class_="objectvalue %s excluded" % field, rowspan=2, colspan=2)
                        page.add(_("First occurrence excluded"))
                        page.td.close()

                page.tr.close()

                # After the end datetime, show a control to add recurrences.

                if name == "DTEND":
                    page.tr()
                    page.td(colspan=2)
                    self.control("recur-add", "submit", "add", id="recur-add", class_="add")
                    page.label(_("Add a recurrence"), for_="recur-add", class_="add")
                    page.td.close()
                    page.tr.close()

            # Handle the summary specially.

            elif name == "SUMMARY":
                value = args.get("summary", [self.obj.get_value(name)])[0]

                page.td(class_="objectvalue summary", colspan=2)
                if self.is_organiser():
                    self.control("summary", "text", value, size=80)
                else:
                    page.add(value)
                page.td.close()
                page.tr.close()

            # Handle attendees specially.

            elif name == "ATTENDEE":
                attendee_map = dict(items)
                first = True

                for i, value in enumerate(attendees):
                    if not first:
                        page.tr()
                    else:
                        first = False

                    # Obtain details of attendees to supply attributes.

                    self.show_attendee(i, value, attendee_map.get(get_uri(value)))
                    page.tr.close()

                # Allow more attendees to be specified.

                if not first:
                    page.tr()

                page.td(colspan=2)
                self.control("add", "submit", "add", id="add", class_="add")
                page.label(_("Add attendee"), for_="add", class_="add")
                page.td.close()
                page.tr.close()

            # Handle potentially many values of other kinds.

            else:
                first = True

                for i, (value, attr) in enumerate(items):
                    if not first:
                        page.tr()
                    else:
                        first = False

                    page.td(class_="objectvalue %s" % field, colspan=2)
                    if name == "ORGANIZER":
                        page.add(get_verbose_address(value, attr))
                    else:
                        page.add(value)
                    page.td.close()
                    page.tr.close()

        page.tbody.close()
        page.table.close()

        self.show_recurrences(errors)
        self.show_counters()
        self.show_conflicting_events()
        self.show_request_controls()

        page.form.close()

    def show_attendee(self, i, attendee, attendee_attr):

        """
        For the current object, show the attendee in position 'i' with the given
        'attendee' value, having 'attendee_attr' as any stored attributes.
        """

        _ = self.get_translator()

        page = self.page
        args = self.env.get_args()

        attendee_uri = get_uri(attendee)
        partstat = attendee_attr and attendee_attr.get("PARTSTAT")

        page.td(class_="objectvalue")

        # Show a form control as organiser for new attendees.

        if self.can_edit_attendee(attendee_uri):
            self.control("attendee", "value", attendee, size="40")
        else:
            self.control("attendee", "hidden", attendee)
            page.add(attendee)
        page.add(" ")

        # Show participation status, editable for the current user.

        partstat_items = [(key, _(partstat_label)) for (key, partstat_label) in self.partstat_items]

        if attendee_uri == self.user:
            self.menu("partstat", partstat, partstat_items, class_="partstat")

        # Allow the participation indicator to act as a submit
        # button in order to refresh the page and show a control for
        # the current user, if indicated.

        elif self.is_organiser() and self.attendee_is_new(attendee_uri):
            self.control("partstat-refresh", "submit", "refresh", id="partstat-%d" % i, class_="refresh")
            page.label(dict(partstat_items).get(partstat, ""), for_="partstat-%s" % i, class_="partstat")

        # Otherwise, just show a label with the participation status.

        else:
            page.span(dict(partstat_items).get(partstat, ""), class_="partstat")

        page.td.close()
        page.td()

        # Permit organisers to remove attendees.

        if self.can_remove_attendee(attendee_uri) or self.is_organiser():

            # Permit the removal of newly-added attendees.

            remove_type = self.can_remove_attendee(attendee_uri) and "submit" or "checkbox"
            self.control("remove", remove_type, str(i), str(i) in args.get("remove", []), id="remove-%d" % i, class_="remove")

            page.label(_("Remove"), for_="remove-%d" % i, class_="remove")
            page.label(for_="remove-%d" % i, class_="removed")
            page.add(_("(Uninvited)"))
            page.span(_("Re-invite"), class_="action")
            page.label.close()

        page.td.close()

    def show_recurrences(self, errors=None):

        """
        Show recurrences for the current object. If 'errors' is given, show a
        suitable message for the different errors provided.
        """

        _ = self.get_translator()

        page = self.page

        # Obtain any parent object if this object is a specific recurrence.

        if self.recurrenceid:
            parent = self.get_stored_object(self.uid, None)
            if not parent:
                return

            page.p()
            page.a(_("This event modifies a recurring event."), href=self.link_to(self.uid))
            page.p.close()

        # Obtain the periods associated with the event.

        recurrences = self.get_current_recurrences()

        if len(recurrences) < 1:
            return

        page.p(_("This event occurs on the following occasions within the next %d days:") % self.get_window_size())

        # Show each recurrence in a separate table.

        for index, period in enumerate(recurrences):
            self.show_recurrence(index, period, self.recurrenceid, errors)

    def show_recurrence(self, index, period, recurrenceid, errors=None):

        """
        Show recurrence controls for a recurrence provided by the current object
        with the given 'index' position in the list of periods, the given
        'period' details, where a 'recurrenceid' indicates any specific
        recurrence.

        If 'errors' is given, show a suitable message for the different errors
        provided.
        """

        _ = self.get_translator()

        page = self.page
        args = self.env.get_args()

        # Isolate the controls from neighbouring tables.

        page.div()

        self.show_object_datetime_controls(period, index)

        page.table(cellspacing=5, cellpadding=5, class_="recurrence")
        page.caption(period.origin == "RRULE" and _("Occurrence from rule") or _("Occurrence"))
        page.tbody()

        page.tr()
        error = errors and ("dtstart", index) in errors and " error" or ""
        page.th("Start", class_="objectheading start%s" % error)
        self.show_recurrence_controls(index, period, recurrenceid, True)
        page.tr.close()
        page.tr()
        error = errors and ("dtend", index) in errors and " error" or ""
        page.th("End", class_="objectheading end%s" % error)
        self.show_recurrence_controls(index, period, recurrenceid, False)
        page.tr.close()

        # Permit the removal of recurrences.

        if not period.replaced:
            page.tr()
            page.th("")
            page.td()

            # Attendees can instantly remove recurrences and thus produce a
            # counter-proposal. Organisers may need to unschedule recurrences
            # instead.

            remove_type = self.can_remove_recurrence(period) and "submit" or "checkbox"

            self.control("recur-remove", remove_type, str(index),
                str(index) in args.get("recur-remove", []),
                id="recur-remove-%d" % index, class_="remove")

            page.label(_("Remove"), for_="recur-remove-%d" % index, class_="remove")
            page.label(for_="recur-remove-%d" % index, class_="removed")
            page.add(_("(Removed)"))
            page.span(_("Re-add"), class_="action")
            page.label.close()

            page.td.close()
            page.tr.close()

        page.tbody.close()
        page.table.close()

        page.div.close()

    def show_counters(self):

        "Show any counter-proposals for the current object."

        _ = self.get_translator()

        page = self.page
        query = self.env.get_query()
        counter = query.get("counter", [None])[0]

        attendees = self._get_counters(self.uid, self.recurrenceid)
        tzid = self.get_tzid()

        if not attendees:
            return

        attendees = self.get_verbose_attendees(attendees)
        current_attendees = [uri for (name, uri) in uri_parts(self.get_current_attendees())]
        current_periods = set(self.get_periods(self.obj))

        # Get suggestions. Attendees are aggregated and reference the existing
        # attendees suggesting them. Periods are referenced by each existing
        # attendee.

        suggested_attendees = {}
        suggested_periods = {}

        for i, attendee in enumerate(attendees):
            attendee_uri = get_uri(attendee)
            obj = self.get_stored_object(self.uid, self.recurrenceid, "counters", attendee_uri)

            # Get suggested attendees.

            for suggested_uri, suggested_attr in uri_dict(obj.get_value_map("ATTENDEE")).items():
                if suggested_uri == attendee_uri or suggested_uri in current_attendees:
                    continue
                suggested = get_verbose_address(suggested_uri, suggested_attr)

                if not suggested_attendees.has_key(suggested):
                    suggested_attendees[suggested] = []
                suggested_attendees[suggested].append(attendee)

            # Get suggested periods.

            periods = self.get_periods(obj)
            if current_periods.symmetric_difference(periods):
                suggested_periods[attendee] = periods

        # Present the suggested attendees.

        if suggested_attendees:
            page.p(_("The following attendees have been suggested for this event:"))

            page.table(cellspacing=5, cellpadding=5, class_="counters")
            page.thead()
            page.tr()
            page.th(_("Attendee"))
            page.th(_("Suggested by..."))
            page.tr.close()
            page.thead.close()
            page.tbody()

            suggested_attendees = list(suggested_attendees.items())
            suggested_attendees.sort()

            for i, (suggested, attendees) in enumerate(suggested_attendees):
                page.tr()
                page.td(suggested)
                page.td(", ".join(attendees))
                page.td()
                self.control("suggested-attendee", "hidden", suggested)
                self.control("add-suggested-attendee-%d" % i, "submit", "Add")
                page.td.close()
                page.tr.close()

            page.tbody.close()
            page.table.close()

        # Present the suggested periods.

        if suggested_periods:
            page.p(_("The following periods have been suggested for this event:"))

            page.table(cellspacing=5, cellpadding=5, class_="counters")
            page.thead()
            page.tr()
            page.th(_("Periods"), colspan=2)
            page.th(_("Suggested by..."), rowspan=2)
            page.tr.close()
            page.tr()
            page.th(_("Start"))
            page.th(_("End"))
            page.tr.close()
            page.thead.close()
            page.tbody()

            recurrenceids = self._get_recurrences(self.uid)

            suggested_periods = list(suggested_periods.items())
            suggested_periods.sort()

            for attendee, periods in suggested_periods:
                first = True
                for p in periods:
                    replaced = not self.recurrenceid and p.is_replaced(recurrenceids)
                    identifier = "%s-%s" % (format_datetime(p.get_start_point()), format_datetime(p.get_end_point()))
                    css = identifier == counter and "selected" or ""
                    
                    page.tr(class_=css)

                    start = self.format_datetime(to_timezone(p.get_start(), tzid), "long")
                    end = self.format_datetime(to_timezone(p.get_end(), tzid), "long")

                    # Show each period.

                    css = replaced and "replaced" or ""
                    page.td(start, class_=css)
                    page.td(end, class_=css)

                    # Show attendees and controls alongside the first period in each
                    # attendee's collection.

                    if first:
                        page.td(attendee, rowspan=len(periods))
                        page.td(rowspan=len(periods))
                        self.control("accept-%d" % i, "submit", "Accept")
                        self.control("decline-%d" % i, "submit", "Decline")
                        self.control("counter", "hidden", attendee)
                        page.td.close()

                    page.tr.close()
                    first = False

            page.tbody.close()
            page.table.close()

    def show_conflicting_events(self):

        "Show conflicting events for the current object."

        _ = self.get_translator()

        page = self.page
        recurrenceids = self._get_active_recurrences(self.uid)

        # Obtain the user's timezone.

        tzid = self.get_tzid()
        periods = self.get_periods(self.obj)

        # Indicate whether there are conflicting events.

        conflicts = set()
        attendee_map = uri_dict(self.obj.get_value_map("ATTENDEE"))

        for name, participant in uri_parts(self.get_current_attendees()):
            if participant == self.user:
                freebusy = self.store.get_freebusy(participant)
            elif participant:
                freebusy = self.store.get_freebusy_for_other(self.user, participant)
            else:
                continue

            if not freebusy:
                continue

            # Obtain any time zone details from the suggested event.

            _dtstart, attr = self.obj.get_item("DTSTART")
            tzid = attr.get("TZID", tzid)

            # Show any conflicts with periods of actual attendance.

            participant_attr = attendee_map.get(participant)
            partstat = participant_attr and participant_attr.get("PARTSTAT")
            recurrences = self.obj.get_recurrence_start_points(recurrenceids, tzid)

            for p in freebusy.have_conflict(periods, True):
                if not self.recurrenceid and p.is_replaced(recurrences):
                    continue

                if ( # Unidentified or different event
                     (p.uid != self.uid or self.recurrenceid and p.recurrenceid and p.recurrenceid != self.recurrenceid) and
                     # Different period or unclear participation with the same period
                     (p not in periods or not partstat in ("ACCEPTED", "TENTATIVE")) and
                     # Participant not limited to organising
                     p.transp != "ORG"
                   ):

                    conflicts.add(p)

        conflicts = list(conflicts)
        conflicts.sort()

        # Show any conflicts with periods of actual attendance.

        if conflicts:
            page.p(_("This event conflicts with others:"))

            page.table(cellspacing=5, cellpadding=5, class_="conflicts")
            page.thead()
            page.tr()
            page.th(_("Event"))
            page.th(_("Start"))
            page.th(_("End"))
            page.tr.close()
            page.thead.close()
            page.tbody()

            for p in conflicts:

                # Provide details of any conflicting event.

                start = self.format_datetime(to_timezone(p.get_start(), tzid), "long")
                end = self.format_datetime(to_timezone(p.get_end(), tzid), "long")

                page.tr()

                # Show the event summary for the conflicting event.

                page.td()
                if p.summary:
                    page.a(p.summary, href=self.link_to(p.uid, p.recurrenceid))
                else:
                    page.add(_("(Unspecified event)"))
                page.td.close()

                page.td(start)
                page.td(end)

                page.tr.close()

            page.tbody.close()
            page.table.close()

class EventPage(EventPageFragment):

    "A request handler for the event page."

    def __init__(self, resource=None, messenger=None):
        ResourceClientForObject.__init__(self, resource, messenger or Messenger())

    def link_to(self, uid=None, recurrenceid=None):
        args = self.env.get_query()
        d = {}
        for name in ("start", "end"):
            if args.get(name):
                d[name] = args[name][0]
        return ResourceClientForObject.link_to(self, uid, recurrenceid, d)

    # Request logic methods.

    def is_initial_load(self):

        "Return whether the event is being loaded and shown for the first time."

        return not self.env.get_args().has_key("editing")

    def handle_request(self):

        """
        Handle actions involving the current object, returning an error if one
        occurred, or None if the request was successfully handled.
        """

        # Handle a submitted form.

        args = self.env.get_args()

        # Get the possible actions.

        reply = args.has_key("reply")
        discard = args.has_key("discard")
        create = args.has_key("create")
        cancel = args.has_key("cancel")
        ignore = args.has_key("ignore")
        save = args.has_key("save")
        uncounter = args.has_key("uncounter")
        accept = self.prefixed_args("accept-", int)
        decline = self.prefixed_args("decline-", int)

        have_action = reply or discard or create or cancel or ignore or save or accept or decline or uncounter

        if not have_action:
            return ["action"]

        # Check the validation token.

        if not self.check_validation_token():
            return ["token"]

        # If ignoring the object, return to the calendar.

        if ignore:
            self.redirect(self.link_to())
            return None

        # Update the object.

        single_user = False
        changed = False

        if reply or create or cancel or save:

            # Update time periods (main and recurring).

            try:
                period = self.handle_main_period()
            except PeriodError, exc:
                return exc.args

            try:
                periods = self.handle_recurrence_periods()
            except PeriodError, exc:
                return exc.args

            # Set the periods in the object, first obtaining removed and
            # modified period information.
            # NOTE: Currently, rules are not updated.

            to_unschedule, to_exclude = self.get_removed_periods(periods)
            periods = set(periods)
            active_periods = [p for p in periods if not p.replaced]

            changed = self.obj.set_period(period) or changed
            changed = self.obj.set_periods(periods) or changed

            # Add and remove exceptions.

            changed = self.obj.update_exceptions(to_exclude, active_periods) or changed

            # Assert periods restored after cancellation.

            changed = self.revert_cancellations(active_periods) or changed

            # Organiser-only changes...

            if self.is_organiser():

                # Update summary.

                if args.has_key("summary"):
                    self.obj["SUMMARY"] = [(args["summary"][0], {})]

            # Obtain any new participants and those to be removed.

            attendees = self.get_attendees_from_page()
            removed = [attendees[int(i)] for i in args.get("remove", [])]
            added, to_cancel = self.update_attendees(attendees, removed)
            single_user = not attendees or uri_values(attendees) == [self.user]
            changed = added or changed

            # Update attendee participation for the current user.

            if args.has_key("partstat"):
                self.update_participation(args["partstat"][0])

        # Process any action.

        invite = not save and create and not single_user
        save = save or create and single_user

        handled = True

        if reply or invite or cancel:

            # Process the object and remove it from the list of requests.

            if reply and self.process_received_request(changed):
                if self.has_indicated_attendance():
                    self.remove_request()

            elif self.is_organiser() and (invite or cancel):

                # Invitation, uninvitation and unscheduling...

                if self.process_created_request(
                    invite and "REQUEST" or "CANCEL", to_cancel, to_unschedule):

                    self.remove_request()

        # Save single user events.

        elif save:
            self.store.set_event(self.user, self.uid, self.recurrenceid, node=self.obj.to_node())
            self.update_event_in_freebusy()
            self.remove_request()

        # Remove the request and the object.

        elif discard:
            self.remove_event_from_freebusy()
            self.remove_event()
            self.remove_request()

        # Update counter-proposal records synchronously instead of assuming
        # that the outgoing handler will have done so before the form is
        # refreshed.

        # Accept a counter-proposal and decline all others, sending a new
        # request to all attendees.

        elif accept:

            # Take the first accepted proposal, although there should be only
            # one anyway.

            for i in accept:
                attendee_uri = get_uri(args.get("counter", [])[i])
                obj = self.get_stored_object(self.uid, self.recurrenceid, "counters", attendee_uri)
                self.obj.set_periods(self.get_periods(obj))
                self.obj.set_rule(obj.get_item("RRULE"))
                self.obj.set_exceptions(obj.get_items("EXDATE"))
                break

            # Remove counter-proposals and issue a new invitation.

            attendees = uri_values(args.get("counter", []))
            self.remove_counters(attendees)
            self.process_created_request("REQUEST")

        # Decline a counter-proposal individually.

        elif decline:
            for i in decline:
                attendee_uri = get_uri(args.get("counter", [])[i])
                self.process_declined_counter(attendee_uri)
                self.remove_counter(attendee_uri)

            # Redirect to the event.

            self.redirect(self.env.get_url())
            handled = False

        # Remove counter-proposals without acknowledging them.

        elif uncounter:
            self.store.remove_counters(self.user, self.uid, self.recurrenceid)
            self.remove_request()

            # Redirect to the event.

            self.redirect(self.env.get_url())
            handled = False

        else:
            handled = False

        # Upon handling an action, redirect to the main page.

        if handled:
            self.redirect(self.link_to())

        return None

    def handle_main_period(self):

        "Return period details for the main start/end period in an event."

        return self.get_main_period_from_page().as_event_period()

    def handle_recurrence_periods(self):

        "Return period details for the recurrences specified for an event."

        return [p.as_event_period(i) for i, p in enumerate(self.get_recurrences_from_page())]

    # Access to form-originating object information.

    def get_main_period_from_page(self):

        "Return the main period defined in the event form."

        args = self.env.get_args()

        dtend_enabled = args.get("dtend-control", [None])[0]
        dttimes_enabled = args.get("dttimes-control", [None])[0]
        start = self.get_date_control_values("dtstart")
        end = self.get_date_control_values("dtend")

        period = FormPeriod(start, end, dtend_enabled, dttimes_enabled, self.get_tzid(), "DTSTART")

        # Handle absent main period details.

        if not period.get_start():
            return self.get_stored_main_period()
        else:
            return period

    def get_recurrences_from_page(self):

        "Return the recurrences defined in the event form."

        args = self.env.get_args()

        all_dtend_enabled = args.get("dtend-control-recur", [])
        all_dttimes_enabled = args.get("dttimes-control-recur", [])
        all_starts = self.get_date_control_values("dtstart-recur", multiple=True)
        all_ends = self.get_date_control_values("dtend-recur", multiple=True, tzid_name="dtstart-recur")
        all_origins = args.get("recur-origin", [])
        all_replaced = args.get("recur-replaced", [])

        periods = []

        for index, (start, end, origin) in \
            enumerate(map(None, all_starts, all_ends, all_origins)):

            dtend_enabled = str(index) in all_dtend_enabled
            dttimes_enabled = str(index) in all_dttimes_enabled
            replaced = str(index) in all_replaced

            period = FormPeriod(start, end, dtend_enabled, dttimes_enabled, self.get_tzid(), origin, replaced)
            periods.append(period)

        return periods

    def set_recurrences_in_page(self, recurrences):

        "Set the recurrences defined in the event form."

        args = self.env.get_args()

        args["dtend-control-recur"] = []
        args["dttimes-control-recur"] = []
        args["recur-origin"] = []
        args["recur-replaced"] = []

        all_starts = []
        all_ends = []

        for index, period in enumerate(recurrences):
            if period.end_enabled:
                args["dtend-control-recur"].append(str(index))
            if period.times_enabled:
                args["dttimes-control-recur"].append(str(index))
            if period.replaced:
                args["recur-replaced"].append(str(index))
            args["recur-origin"].append(period.origin or "")

            all_starts.append(period.get_form_start())
            all_ends.append(period.get_form_end())

        self.set_date_control_values("dtstart-recur", all_starts)
        self.set_date_control_values("dtend-recur", all_ends, tzid_name="dtstart-recur")

    def get_removed_periods(self, periods):

        """
        Return those from the recurrence 'periods' to remove upon updating an
        event along with those to exclude in a tuple of the form (unscheduled,
        excluded).
        """

        args = self.env.get_args()
        to_unschedule = []
        to_exclude = []

        for i in args.get("recur-remove", []):
            try:
                period = periods[int(i)]
            except (IndexError, ValueError):
                continue

            if not self.can_edit_recurrence(period) and self.is_organiser():
                to_unschedule.append(period)
            else:
                to_exclude.append(period)

        return to_unschedule, to_exclude

    def get_attendees_from_page(self):

        """
        Return attendees from the request, using any stored attributes to obtain
        verbose details.
        """

        return self.get_verbose_attendees(self.env.get_args().get("attendee", []))

    def get_verbose_attendees(self, attendees):

        """
        Use any stored attributes to obtain verbose details for the given
        'attendees'.
        """

        attendee_map = self.obj.get_value_map("ATTENDEE")
        return [get_verbose_address(value, attendee_map.get(value)) for value in attendees]

    def update_attendees_from_page(self):

        "Add or remove attendees. This does not affect the stored object."

        args = self.env.get_args()

        attendees = self.get_attendees_from_page()

        if args.has_key("add"):
            attendees.append("")

        # Add attendees suggested in counter-proposals.

        add_suggested = self.prefixed_args("add-suggested-attendee-", int)

        if add_suggested:
            for i in add_suggested:
                try:
                    suggested = args["suggested-attendee"][i]
                except (IndexError, KeyError):
                    continue
                if suggested not in attendees:
                    attendees.append(suggested)

        # Only actually remove attendees if the event is unsent, if the attendee
        # is new, or if it is the current user being removed.

        if args.has_key("remove"):
            still_to_remove = []
            correction = 0

            for i in args["remove"]:
                try:
                    i = int(i) - correction
                    attendee = attendees[i]
                except (IndexError, ValueError):
                    continue

                if self.can_remove_attendee(get_uri(attendee)):
                    del attendees[i]
                    correction += 1
                else:
                    still_to_remove.append(str(i))

            args["remove"] = still_to_remove

        args["attendee"] = attendees
        return attendees

    def update_recurrences_from_page(self):

        "Add or remove recurrences. This does not affect the stored object."

        args = self.env.get_args()

        recurrences = self.get_recurrences_from_page()

        if args.has_key("recur-add"):
            period = self.get_current_main_period().as_form_period()
            period.origin = "RDATE"
            recurrences.append(period)

        # Only actually remove recurrences if the event is unsent, or if the
        # recurrence is new, but only for explicit recurrences.

        if args.has_key("recur-remove"):
            still_to_remove = []
            correction = 0

            for i in args["recur-remove"]:
                try:
                    i = int(i) - correction
                    recurrence = recurrences[i]
                except (IndexError, ValueError):
                    continue

                if self.can_remove_recurrence(recurrence):
                    del recurrences[i]
                    correction += 1
                else:
                    still_to_remove.append(str(i))

            args["recur-remove"] = still_to_remove

        self.set_recurrences_in_page(recurrences)
        return recurrences

    # Access to current object information.

    def get_current_main_period(self):

        """
        Return the currently active main period for the current object depending
        on whether editing has begun or whether the object has just been loaded.
        """

        if self.is_initial_load():
            return self.get_stored_main_period()
        else:
            return self.get_main_period_from_page()

    def get_current_recurrences(self):

        """
        Return recurrences for the current object using the original object
        details where no editing is in progress, using form data otherwise.
        """

        if self.is_initial_load():
            return self.get_stored_recurrences()
        else:
            return self.get_recurrences_from_page()

    def update_current_recurrences(self):

        "Return an updated collection of recurrences for the current object."

        if self.is_initial_load():
            return self.get_stored_recurrences()
        else:
            return self.update_recurrences_from_page()

    def get_current_attendees(self):

        """
        Return attendees for the current object depending on whether the object
        has been edited or instead provides such information from its stored
        form.
        """

        if self.is_initial_load():
            return self.get_stored_attendees()
        else:
            return self.get_attendees_from_page()

    def update_current_attendees(self):

        "Return an updated collection of attendees for the current object."

        if self.is_initial_load():
            return self.get_stored_attendees()
        else:
            return self.update_attendees_from_page()

    # Full page output methods.

    def show(self, path_info):

        "Show an object request using the given 'path_info' for the current user."

        uid, recurrenceid = self.get_identifiers(path_info)
        obj = self.get_stored_object(uid, recurrenceid)
        self.set_object(obj)

        if not obj:
            return False

        errors = self.handle_request()

        if not errors:
            return True

        self.update_current_attendees()
        self.update_current_recurrences()

        _ = self.get_translator()

        self.new_page(title=_("Event"))
        self.show_object_on_page(errors)

        return True

# vim: tabstop=4 expandtab shiftwidth=4
