/*
 * Copyright (c) 2024
 *      Tim Woodall. All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * SPDX short identifier: BSD-2-Clause
 */

#include "faketape-lib.h"

#include <string.h>		// strerror
#include <cstring>		// std::memset
#include <zlib.h>
#include <algorithm>		// find_if
#include <fcntl.h>		// O_ACCMODE
#include <sys/stat.h>		// stat

#define USE_ZLIB_COMPRESSION 1

uint32_t FakeTape::crc32(const BlockTable* bt) {
	static constexpr uint32_t crcTable[256] = {
		0x00000000, 0xedb88320, 0x36c98560, 0xdb710640, 0x6d930ac0, 0x802b89e0, 0x5b5a8fa0, 0xb6e20c80,
		0xdb261580, 0x369e96a0, 0xedef90e0, 0x005713c0, 0xb6b51f40, 0x5b0d9c60, 0x807c9a20, 0x6dc41900,
		0x5bf4a820, 0xb64c2b00, 0x6d3d2d40, 0x8085ae60, 0x3667a2e0, 0xdbdf21c0, 0x00ae2780, 0xed16a4a0,
		0x80d2bda0, 0x6d6a3e80, 0xb61b38c0, 0x5ba3bbe0, 0xed41b760, 0x00f93440, 0xdb883200, 0x3630b120,
		0xb7e95040, 0x5a51d360, 0x8120d520, 0x6c985600, 0xda7a5a80, 0x37c2d9a0, 0xecb3dfe0, 0x010b5cc0,
		0x6ccf45c0, 0x8177c6e0, 0x5a06c0a0, 0xb7be4380, 0x015c4f00, 0xece4cc20, 0x3795ca60, 0xda2d4940,
		0xec1df860, 0x01a57b40, 0xdad47d00, 0x376cfe20, 0x818ef2a0, 0x6c367180, 0xb74777c0, 0x5afff4e0,
		0x373bede0, 0xda836ec0, 0x01f26880, 0xec4aeba0, 0x5aa8e720, 0xb7106400, 0x6c616240, 0x81d9e160,
		0x826a23a0, 0x6fd2a080, 0xb4a3a6c0, 0x591b25e0, 0xeff92960, 0x0241aa40, 0xd930ac00, 0x34882f20,
		0x594c3620, 0xb4f4b500, 0x6f85b340, 0x823d3060, 0x34df3ce0, 0xd967bfc0, 0x0216b980, 0xefae3aa0,
		0xd99e8b80, 0x342608a0, 0xef570ee0, 0x02ef8dc0, 0xb40d8140, 0x59b50260, 0x82c40420, 0x6f7c8700,
		0x02b89e00, 0xef001d20, 0x34711b60, 0xd9c99840, 0x6f2b94c0, 0x829317e0, 0x59e211a0, 0xb45a9280,
		0x358373e0, 0xd83bf0c0, 0x034af680, 0xeef275a0, 0x58107920, 0xb5a8fa00, 0x6ed9fc40, 0x83617f60,
		0xeea56660, 0x031de540, 0xd86ce300, 0x35d46020, 0x83366ca0, 0x6e8eef80, 0xb5ffe9c0, 0x58476ae0,
		0x6e77dbc0, 0x83cf58e0, 0x58be5ea0, 0xb506dd80, 0x03e4d100, 0xee5c5220, 0x352d5460, 0xd895d740,
		0xb551ce40, 0x58e94d60, 0x83984b20, 0x6e20c800, 0xd8c2c480, 0x357a47a0, 0xee0b41e0, 0x03b3c2c0,
		0xe96cc460, 0x04d44740, 0xdfa54100, 0x321dc220, 0x84ffcea0, 0x69474d80, 0xb2364bc0, 0x5f8ec8e0,
		0x324ad1e0, 0xdff252c0, 0x04835480, 0xe93bd7a0, 0x5fd9db20, 0xb2615800, 0x69105e40, 0x84a8dd60,
		0xb2986c40, 0x5f20ef60, 0x8451e920, 0x69e96a00, 0xdf0b6680, 0x32b3e5a0, 0xe9c2e3e0, 0x047a60c0,
		0x69be79c0, 0x8406fae0, 0x5f77fca0, 0xb2cf7f80, 0x042d7300, 0xe995f020, 0x32e4f660, 0xdf5c7540,
		0x5e859420, 0xb33d1700, 0x684c1140, 0x85f49260, 0x33169ee0, 0xdeae1dc0, 0x05df1b80, 0xe86798a0,
		0x85a381a0, 0x681b0280, 0xb36a04c0, 0x5ed287e0, 0xe8308b60, 0x05880840, 0xdef90e00, 0x33418d20,
		0x05713c00, 0xe8c9bf20, 0x33b8b960, 0xde003a40, 0x68e236c0, 0x855ab5e0, 0x5e2bb3a0, 0xb3933080,
		0xde572980, 0x33efaaa0, 0xe89eace0, 0x05262fc0, 0xb3c42340, 0x5e7ca060, 0x850da620, 0x68b52500,
		0x6b06e7c0, 0x86be64e0, 0x5dcf62a0, 0xb077e180, 0x0695ed00, 0xeb2d6e20, 0x305c6860, 0xdde4eb40,
		0xb020f240, 0x5d987160, 0x86e97720, 0x6b51f400, 0xddb3f880, 0x300b7ba0, 0xeb7a7de0, 0x06c2fec0,
		0x30f24fe0, 0xdd4accc0, 0x063bca80, 0xeb8349a0, 0x5d614520, 0xb0d9c600, 0x6ba8c040, 0x86104360,
		0xebd45a60, 0x066cd940, 0xdd1ddf00, 0x30a55c20, 0x864750a0, 0x6bffd380, 0xb08ed5c0, 0x5d3656e0,
		0xdcefb780, 0x315734a0, 0xea2632e0, 0x079eb1c0, 0xb17cbd40, 0x5cc43e60, 0x87b53820, 0x6a0dbb00,
		0x07c9a200, 0xea712120, 0x31002760, 0xdcb8a440, 0x6a5aa8c0, 0x87e22be0, 0x5c932da0, 0xb12bae80,
		0x871b1fa0, 0x6aa39c80, 0xb1d29ac0, 0x5c6a19e0, 0xea881560, 0x07309640, 0xdc419000, 0x31f91320,
		0x5c3d0a20, 0xb1858900, 0x6af48f40, 0x874c0c60, 0x31ae00e0, 0xdc1683c0, 0x07678580, 0xeadf06a0,
	};

	auto data = reinterpret_cast<const uint8_t*>(bt);
	uint32_t crc = 0xffffffffu;

	for (size_t i = 0; i < sizeof(*bt); ++i) {
		const uint32_t lookupIndex = (crc ^ data[i]) & 0xff;
		crc = (crc >> 8) ^ crcTable[lookupIndex];  // CRCTable is an array of 256 32-bit constants
	}

	// Finalize the CRC-32 value by inverting all the bits
	crc ^= 0xFFFFFFFFu;
	return crc;
}

#define returnError(x) do { DEBUG("DONE " << x); return Response::ERROR; } while(0)

bool FakeTape::readBlockTable() {
	TRACE("ENTER");
	if (fd == -1)
		return false;

	int rl = read(fd, &cbt, sizeof cbt);
	if (rl != sizeof cbt)
		return false;

	uint32_t read_crc = cbt.crc;
	cbt.crc = 0;
	cbt.crc = crc32(&cbt);
	if (cbt.crc != read_crc)
		return false;
	cbt_index = 0;
	TRACE("DONE");
	return true;
}

/* Seek to the blocktable for the current block */
bool FakeTape::seekBlockTable() {
	TRACE("ENTER");
	if (fd == -1)
		return false;

	off_t seek = -sizeof cbt;
	for (uint32_t i=0; i < cbt_index; ++i)
		seek -= cbt.disksize(i);

	if(lseek(fd, seek, SEEK_CUR) < 0)
		return false;
	TRACE("DONE");
	return true;
}

/* This replaces the existing block table that we're using */
bool FakeTape::writeBlockTable() {
	TRACE("ENTER");
	if (fd == -1 || !writing)
		return false;

	off_t curpos = lseek(fd, 0, SEEK_CUR);
	if (curpos < 0 )
		return false;

	if(!seekBlockTable())
		return false;

	BlockTable bt = cbt;
	bt.crc = 0;
	bt.crc = crc32(&bt);

	if(!readBlockTable())
		return false;
	cbt_index = bt.count;

	if(lseek(fd, -sizeof cbt, SEEK_CUR) < 0)
		return false;

	if (write(fd, &bt, sizeof bt) != sizeof bt)
		return false;

	if(lseek(fd, curpos, SEEK_SET) < 0)
		return false;

	cbt = bt;
	TRACE("DONE");
	return true;
}

/* This writes a new block table */
bool FakeTape::writeNewBlockTable() {
	TRACE("ENTER");
	if (fd == -1 || !writing)
		return false;

	BlockTable bt;
	if (cbt.count == cbt.maxblocks()) {
		bt.prevCount = cbt.count;
		bt.prevSize = sizeof bt;
		for (uint32_t i=0; i < cbt.maxblocks(); ++i) {
			bt.prevSize += cbt.disksize(i);
			if(cbt.size(i) == 0)
				bt.prevSize |= 1ULL<<63;
		}
	} else if (cbt.count != 0 || cbt.prevSize != 0)
		return false;
	bt.crc = crc32(&bt);

	if (write(fd, &bt, sizeof bt) != sizeof bt)
		return false;
	TRACE("wrote " << sizeof bt << " bytes to tape for new BlockTable ");
	cbt = bt;
	cbt_index = 0;
	off_t curpos = lseek(fd, 0, SEEK_CUR);
	if (curpos < 0 )
		return false;
	if(ftruncate(fd, curpos) < 0)
		return false;
	TRACE("DONE");
	return true;
};

FakeTape::Response FakeTape::advanceOneBlock() {
	TRACE("ENTER");
	if (fd == -1)
		returnError("tape not open");

	if (writing)
		returnError("tape must be open for reading to seek");

	if (cbt_index == cbt.count)
		returnError("seeking past end of tape");
	uint32_t datasize = cbt.size(cbt_index);
	off_t disksize = cbt.disksize(cbt_index++);
	if (lseek(fd, disksize, SEEK_CUR) < 0)
		return Response::FATAL;
	if (cbt_index == cbt.maxblocks())
		if(!readBlockTable())
			return Response::FATAL;
	block++;
	fileblock++;
	if (datasize == 0) {
		filenum++;
		fileblock=0;
	}
	TRACE("DONE");
	return Response::OK;
}

FakeTape::Response FakeTape::reverseOneBlock() {
	TRACE("ENTER");
	if (fd == -1)
		returnError("tape not open");

	if (writing)
		returnError("tape must be open for reading to seek");

	if (cbt_index == 0) {
		if (cbt.prevSize == 0)
			returnError("seeking past start of tape");
		off_t prevSize = cbt.prevSize & (~0ULL>>1);
		if (lseek(fd, -prevSize-sizeof cbt, SEEK_CUR) < 0)
			return Response::FATAL;
		if(!readBlockTable())
			return Response::FATAL;
		if (lseek(fd, prevSize - sizeof cbt, SEEK_CUR) < 0)
			return Response::FATAL;
		cbt_index = cbt.maxblocks();
	}
	uint32_t datasize = cbt.size(--cbt_index);
	off_t disksize = cbt.disksize(cbt_index);
	if (lseek(fd, -disksize, SEEK_CUR) < 0)
		return Response::FATAL;
	block--;
	if (datasize == 0) {
		if (read(fd, &fileblock, disksize) != disksize)
			return Response::FATAL;
		if (lseek(fd, -disksize, SEEK_CUR) < 0)
			return Response::FATAL;
		filenum--;
	} else
		fileblock--;

	TRACE("DONE");
	return Response::OK;
}

FakeTape::Response FakeTape::blockSeekFwd(size_t offset) {
	TRACE("ENTER offset=" << offset);
	if (fd == -1)
		returnError("tape not open");

	if (writing)
		returnError("tape must be open for reading to seek");

	bool handleEOF = false;
	while (offset) {
		if (cbt_index || handleEOF) {
			if (Response r = advanceOneBlock(); r != Response::OK)
				return r;
			else
				offset--;
			handleEOF = false;
		} else if (offset >= cbt.maxblocks()) {
			if (cbt.count != cbt.maxblocks())
				returnError("seek past end of tape");
			off_t bs = 0;
			for (uint32_t i=0; i < cbt.maxblocks(); ++i) {
				if (!cbt.size(i))
					handleEOF = true;
				else
					bs += cbt.disksize(i);
			}
			if(!handleEOF) {
				if (lseek(fd, bs, SEEK_CUR) < 0)
					return Response::FATAL;
				offset -= cbt.maxblocks();
				block += cbt.maxblocks();
				fileblock += cbt.maxblocks();
				if(!readBlockTable())
					return Response::FATAL;
			}
		} else
			handleEOF = true;
	}
	TRACE("DONE");
	return Response::OK;
}

FakeTape::Response FakeTape::blockSeekBwd(size_t offset) {
	TRACE("ENTER offset=" << offset);
	if (fd == -1)
		returnError("tape not open");

	if (writing)
		returnError("tape must be open for reading to seek");

	bool handleEOF = false;
	while(offset) {
		if (cbt_index || handleEOF) {
			if (Response r = reverseOneBlock(); r != Response::OK)
				return r;
			else
				offset--;
		} else if (cbt.prevCount && offset >= cbt.prevCount) {
			if (cbt.prevSize == 0)
				returnError("seek past beginning of tape");
			if (cbt.prevSize & (1ULL<<63)) {
				handleEOF = true;
				continue;
			}
			off_t seek = -cbt.prevSize - sizeof cbt;
			if (lseek(fd, seek, SEEK_CUR) < 0)
				return Response::FATAL;
			if(!readBlockTable())
				return Response::FATAL;

			offset -= cbt.maxblocks();
			block -= cbt.maxblocks();
		} else
			handleEOF = true;
	}
	TRACE("DONE");
	return Response::OK;
}

FakeTape::Response FakeTape::closefd(const char* wmsg) {
	TRACE("ENTER");

	if(wmsg)
		DEBUG(wmsg << " errno=" << errno << ":" << strerror(errno));

	if (fd != -1)
		::close(fd);
	fd = -1;

	for(size_t i = 0; i< maxWorkers; ++i) {
		context[i].in.clear();
		context[i].out.clear();
	}

	TRACE("DONE");
	return Response::FATAL;
}

std::unique_ptr<FakeTapeTimer> writeDataImpl_time;
std::unique_ptr<FakeTapeTimer> writeDataImpl_wait_time;

FakeTape::Response FakeTape::writeDataImpl(const uint8_t* data, size_t size, bool async) {
	CalcTime ct(*writeDataImpl_time);
	TRACE("ENTER " << (async?"ASYNC":"SYNC") << " size=" << size);

	if (fd == -1 || !writing)
		returnError("tape not open");

	if (size > max_block_size) {
		errno = EIO;
		returnError("block too big");
	}

	/* inbufidx, writebufidx | inbufidx jobin |
	 *    0            0     |     1     0    |
	 *    1            0     |     2     1    |
	 *    2            0     |     3     2    |
	 *    3            0     |     4     3    |
	 *    4            0     |     5     4    |
	 *    5            0     |     6     5    |
	 *    6            0     |     7     6    |
	 *    7            0     |     8     7    |
	 *    8            0     | -------------- | wait for writebufidx to change
	 *    8            1     |     9     0    |
	 */
	// We do not need a lock here unless we're waiting on all of the workers.
	if (inBufIdx == writeBufIdx + maxWorkers) {
		CalcTime ct(*writeDataImpl_wait_time);
		std::unique_lock lk(filemutex);
		block_written.wait(lk, [this]() { return inBufIdx < writeBufIdx + maxWorkers; } );
	}

	if (join() != Response::OK)
		/* We hold the write lock here so we can safely close */
		return closefd("Fatal error during async writes");

	size_t idx = inBufIdx++;
	worker_context& ctxt = context[idx % maxWorkers];

	if (ctxt.in.size() < size) {
		ctxt.in.resize(size);
		ctxt.out.resize(compressBound(size));
	}
	ctxt.size = size;
	memcpy(ctxt.in.data(), data, size);
	written = true;

	ctxt.t = std::thread(&FakeTape::worker, this, idx);
	Response r = Response::OK;
	if (!async) {
		ctxt.t.join();
		joinIdx++;
		r = ctxt.response;
	}

	TRACE("DONE");
	return r;
}

FakeTape::Response FakeTape::join() {
	TRACE("ENTER");
	Response r = Response::OK;
	while (joinIdx < writeBufIdx) {
		worker_context& jctxt = context[joinIdx++%maxWorkers];
		jctxt.t.join();
		if(r == Response::OK)
			r = jctxt.response;
		else if (jctxt.response == Response::FATAL)
			r = Response::FATAL;
	}
	TRACE("DONE");
	return r;
}

void FakeTape::worker(size_t idx) {
	TRACE("ENTER idx=" << idx);
	std::unique_lock lk(filemutex);

	bool zeroblock = false;	/* This flags an all zero buffer */
	worker_context& ctxt = context[idx%maxWorkers];
	size_t size = ctxt.size;
	uint8_t* indata = ctxt.in.data();
	uLongf cmprlen = size;

	if (size == 0) {
		/* End of file marker */
		zeroblock = false;
		ctxt.block_written.wait(lk, [this, &idx]() { return writeBufIdx == idx; });
		TRACE("GOT LOCK writeBufIdx=" << writeBufIdx << " idx=" << idx);

		if(write(fd, &fileblock, sizeof fileblock) != sizeof(uint64_t)) {
			ctxt.response = closefd("write of fileblock failed in writeData");
			return;
		}
		TRACE("wrote " << sizeof(uint64_t) << " bytes to tape for fileblock " << fileblock);
		filenum++;
		fileblock = 0;
	} else {
		zeroblock = size>1 && std::find_if(indata, indata+size, [](uint8_t b){ return b!=0; }) == indata+size;
		if(!zeroblock) {
#if USE_ZLIB_COMPRESSION
			cmprlen = ctxt.out.size();
			uint8_t* outdata = ctxt.out.data();
			int status = compress2(outdata, &cmprlen, indata, size, 2);
			switch (status) {
				case Z_OK: break;
				case Z_MEM_ERROR:
					DEBUG("Z_MEM_ERROR from compress");
					ctxt.response = Response::FATAL;
					return;
				case Z_BUF_ERROR:
					DEBUG("Z_BUF_ERROR from compress");
					ctxt.response = Response::FATAL;
					return;
				case Z_STREAM_ERROR:
					DEBUG("Z_STREAM_ERROR from compress");
					ctxt.response = Response::FATAL;
					return;
				default:
					DEBUG("Unexpected return " << status << " from compress2");
					ctxt.response = Response::FATAL;
					return;
			}
			ctxt.block_written.wait(lk, [this, &idx]() { return writeBufIdx == idx; });
			if (write(fd, outdata, cmprlen) != (ssize_t)cmprlen) {
				ctxt.response = closefd("write failed in worker");
				return;
			}
			TRACE("wrote " << cmprlen << " bytes to tape for fileblock " << fileblock);
#else
			ctxt.block_written.wait(lk, [this, &idx]() { return writeBufIdx == idx; });
			if (write(fd, indata, cmprlen) != (ssize_t)cmprlen) {
				ctxt.response = closefd("write failed in writeData");
				return;
			}
			TRACE("wrote " << cmprlen << " bytes to tape for fileblock " << fileblock);
#endif
		} else {
			ctxt.block_written.wait(lk, [this, &idx]() { return writeBufIdx == idx; });
		}
		fileblock++;
	}

	cbt.setsize(cbt_index++, cmprlen, zeroblock);
	cbt.count = cbt_index;
	if (cbt.count == cbt.maxblocks()) {
		if (!writeBlockTable()) {
			ctxt.response = closefd("writeBlockTable failed");
			return;
		}
		if (!writeNewBlockTable()) {
			ctxt.response = closefd("writeNewBlockTable failed");
			return;
		}
	}
	block++;
	writeBufIdx++;
	lk.unlock();
	// Tell the main thread that there's space available.
	block_written.notify_one();
	// Tell the next block to start writing.
	context[(idx + 1) % maxWorkers].block_written.notify_one();
	ctxt.response = Response::OK;
	TRACE("DONE idx=" << idx);
}


/* public functions below here */
FakeTape::Response FakeTape::open(const std::string& filename, int flags) {
	TRACE("ENTER filename=" << filename << " flags=" << flags);

	errno = 0;
	if (fd != -1)
		returnError("tape already open");
	if (flags & ~(O_ACCMODE|O_CREAT)) {
		errno = EINVAL;
		returnError("invalid flags");
	}
	bool open_for_writing = flags != O_RDONLY;
	if (flags & O_CREAT && !open_for_writing) {
		errno = EINVAL;
		returnError("invalid flags");
	}
	writing = false;
	written = false;

	// We always open for RDWR due to having to save the tape position when we close
	flags &= O_CREAT;
	flags |= O_RDWR;

	cbt = BlockTable{};
	cbt_index = 0;
	block = 0;
	filenum = 0;
	fileblock = 0;

	struct stat s;
	if(stat(filename.c_str(), &s) == -1) {
		if(!(flags&O_CREAT))
			returnError("tape does not exist");
	} else if(!S_ISREG(s.st_mode))
		returnError("not a valid tape");

	fd = ::open(filename.c_str(), flags, 0600);
	if (fd == -1)
		returnError("Can't open tape");

	if(fstat(fd, &s) == -1) {
		return closefd("stat on fd failed");
	}

	for(size_t i = 0; i< maxWorkers; ++i) {
		context[i].in.resize(max_block_size);
		context[i].out.resize(compressBound(max_block_size));
	}

	/*
	 * If the file didn't exist or was empty we'll create a new tape. If
	 * the file contained data then it must be a valid taper
	 */
	if(s.st_size == 0) {
		writing = open_for_writing;

		if(!writing)
			return closefd("Empty file but not opened for writing. Bug?");

		if(!writeNewBlockTable())
			return closefd("Can't write initial block");
	} else {
		BlockTable bt;
		if(read(fd, &bt, sizeof bt) != sizeof bt)
			return closefd("Can't read initial BlockTable");

		uint32_t read_crc = bt.crc;
		bt.crc = 0;
		if(read_crc != crc32(&bt))
			return closefd("Initial crc corrupt");

		off_t blockTablePos;
		uint32_t offset;
		offset = bt.recdata.firstblock.cbt_index;
		block = bt.recdata.firstblock.blockno;
		filenum = bt.recdata.firstblock.filenum;
		fileblock = bt.recdata.firstblock.fileblock;
		blockTablePos = bt.recdata.firstblock.blockTablePos;
		if(lseek(fd, blockTablePos, SEEK_SET) != blockTablePos)
			return closefd("Corrupted tape. Cannot seek to restored position");
		if (!readBlockTable())
			return closefd("Corrupted tape. Cannot seek to restored position");
		if(blockSeekFwd(offset) != Response::OK)
			return closefd("Corrupted tape. Cannot seek to restored position");
		writing = open_for_writing;
		if(writing) {
			cbt.count = cbt_index;
			for(uint32_t i=cbt_index; i<cbt.maxblocks(); ++i)
				cbt.setsize(i, 0, false);
		}
	}
	TRACE("DONE");
	writeDataImpl_time = std::make_unique<FakeTapeTimer>("writeDataImpl_time");
	writeDataImpl_wait_time = std::make_unique<FakeTapeTimer>("writeDataImpl_wait_time");
	return Response::OK;
}

void FakeTape::close() {
	TRACE("ENTER");

	std::unique_lock lk(filemutex);
	block_written.wait(lk, [this]() { return inBufIdx == writeBufIdx; } );

	/* All the writers have finished */
	if (join() != Response::OK)
		closefd("Fatal error during async writes");

	const char* close_with_err = "Error while closing tape";
	if (fd != -1) {
		lk.unlock();
		BlockTable bt;
		off_t blockpos;
		uint32_t save_cbt_index;
		if (written == true) {
			if (writeData(nullptr, 0) != Response::OK)
				goto close;
			if (sync() != Response::OK)
				goto close;
			if (!writeBlockTable())
				goto close;
			blockpos = lseek(fd, 0, SEEK_CUR);
			if (blockpos == -1)
				goto close;
			if (ftruncate(fd, blockpos) == -1)
				goto close;
		}
		save_cbt_index = cbt_index;
		writing = false;
		if (blockSeekBwd(cbt_index) != Response::OK)
			goto close;
		blockpos = lseek(fd, 0, SEEK_CUR) - sizeof bt;
		if (blockpos == -1)
			goto close;
		if(lseek(fd, 0, SEEK_SET) != 0)
			goto close;
		if(read(fd, &bt, sizeof bt) != sizeof bt)
			goto close;
		if(lseek(fd, 0, SEEK_SET) != 0)
			goto close;
		bt.recdata.firstblock.cbt_index = save_cbt_index;
		bt.recdata.firstblock.blockno = block;
		bt.recdata.firstblock.filenum = filenum;
		bt.recdata.firstblock.fileblock = fileblock;
		bt.recdata.firstblock.blockTablePos = blockpos;
		bt.crc = 0;
		bt.crc = crc32(&bt);
		if(write(fd, &bt, sizeof bt) == sizeof bt)
			close_with_err = nullptr;
close:
		closefd(close_with_err);
	}
	fd = -1;

	TRACE("DONE");
	writeDataImpl_time.reset();
	writeDataImpl_wait_time.reset();
}

FakeTape::Response FakeTape::rewind() {
	TRACE("ENTER");

	if (fd == -1)
		returnError("tape not open");

	if(writing)
		returnError("tape must be open for reading to seek");

	if(lseek(fd, 0, SEEK_SET) != 0)
		return closefd("Corrupted tape. Cannot seek to start of tape");
	block = 0;
	fileblock = 0;
	filenum = 0;

	if (!readBlockTable())
		return closefd("Corrupted tape. Cannot read start of tape");

	TRACE("DONE");
	return Response::OK;
}

FakeTape::Response FakeTape::erase() {
	TRACE("ENTER");

	if (fd == -1)
		returnError("tape not open");

	if(!writing || written)
		returnError("tape not open for writing or has been modified");

	if(lseek(fd, 0, SEEK_SET) != 0)
		return closefd("Corrupted tape. Cannot seek to start of tape");

	block = 0;
	fileblock = 0;
	filenum = 0;

	if (!readBlockTable())
		return closefd("Corrupted tape. Cannot read start of tape");

	if (ftruncate(fd, sizeof(BlockTable)) == -1)
		return closefd("truncate failed");
	TRACE("DONE");
	return Response::OK;
}

FakeTape::Response FakeTape::blockSeek(size_t offset) {
	TRACE("ENTER offset=" << offset);
	if (fd == -1)
		returnError("tape not open");

	if (writing)
		returnError("tape must be open for reading to seek");

	Response status;
	if (offset == block)
		status = Response::OK;
	else if (offset > block)
		status = blockSeekFwd(offset - block);
	else
		status = blockSeekBwd(block - offset);
	if (status == Response::FATAL)
		closefd("corrupted tape while seeking");
	return status;
}

FakeTape::Response FakeTape::writeData(const uint8_t* data, size_t size) {
	return writeDataImpl(data, size, false);
}

FakeTape::Response FakeTape::writeDataAsync(const uint8_t* data, size_t size) {
	return writeDataImpl(data, size, true);
}

FakeTape::Response FakeTape::sync() {
	TRACE("ENTER");
	std::unique_lock lk(filemutex);
	block_written.wait(lk, [this]() { return inBufIdx == writeBufIdx; } );
	Response r = Response::OK;
	while (joinIdx < writeBufIdx) {
		worker_context& jctxt = context[joinIdx++%maxWorkers];
		jctxt.t.join();
		if(r == Response::OK)
			r = jctxt.response;
		else if (jctxt.response == Response::FATAL)
			r = Response::FATAL;
	}
	TRACE("DONE");
	return r;
}

FakeTape::Response FakeTape::readData(uint8_t* data, size_t maxsize, size_t* readsize) {
	TRACE("ENTER maxsize=" << maxsize);
	if (fd == -1 || writing)
		returnError("tape not open for reading");

	if (maxsize > max_block_size) {
		errno = EIO;
		returnError("Block too big");
	}

	if (cbt_index == cbt.count) {
		*readsize = 0;
		return Response::OK;
	}

	uint32_t disksize = cbt.disksize(cbt_index);
	bool zeroblock = cbt.zeroblock(cbt_index);
	uint32_t blocksize = cbt.size(cbt_index++);

	if(zeroblock) {
		if (blocksize < maxsize)
			maxsize = blocksize;
		std::memset(data, 0, maxsize);
		if (lseek(fd, disksize, SEEK_CUR) < 0) {
			return closefd("Error seeking to end of block");
		}
	} else if (blocksize == 0) {
		maxsize = 0;
	} else {
#if USE_ZLIB_COMPRESSION
		static std::vector<uint8_t> cmprbuf;
		if (cmprbuf.size() < disksize)
			cmprbuf.resize(disksize);
		if (read(fd, cmprbuf.data(), disksize) != (ssize_t)disksize)
			return closefd("Error reading block from tape");
		TRACE("read " << disksize << " bytes from tape for fileblock " << fileblock);
		uLongf decmprlen = maxsize;
		int status = uncompress(data, &decmprlen, cmprbuf.data(), disksize);
		switch (status) {
			case Z_OK: break;
			case Z_MEM_ERROR: return closefd("Memory error while decompressing");
			case Z_BUF_ERROR: break;
			case Z_DATA_ERROR: return closefd("Corrupt compressed data while decompressing");
			default:
					  DEBUG("Unexpected return " << status << " from uncompress");
					  return closefd("Unexpected return from uncompress");
		}
		TRACE("Decompressed to " << decmprlen << " bytes");
		if (decmprlen < maxsize)
			maxsize = decmprlen;
		/* In the case where there is not enough room, uncompress()
		 * will fill the output buffer with the uncompressed data up to
		 * that point.
		 * */
#else
		if (blocksize < maxsize)
			maxsize = blocksize;
		if (read(fd, data, maxsize) != (ssize_t)maxsize)
			return closefd("Error reading block from tape");
		if (lseek(fd, disksize - maxsize, SEEK_CUR) < 0)
			return closefd("Error seeking to end of block");
#endif
	}
	*readsize = maxsize;
	if (cbt_index == cbt.maxblocks())
		if(!readBlockTable()) {
			return Response::FATAL;
		}
	block++;
	fileblock++;
	if (blocksize == 0) {
		filenum++;
		fileblock=0;
		if (lseek(fd, disksize, SEEK_CUR) < 0)
			return closefd("Error seeking past fileblock");
	}
	TRACE("DONE");
	return Response::OK;
}

FakeTape::Response FakeTape::fsr(size_t offset) {
	TRACE("ENTER offset=" << offset);
	if (fd == -1)
		returnError("tape not open");

	if (writing)
		returnError("tape must be open for reading to seek");

	return blockSeekFwd(offset);
}

FakeTape::Response FakeTape::bsr(size_t offset) {
	TRACE("ENTER offset=" << offset);
	if (fd == -1)
		returnError("tape not open");

	if (writing)
		returnError("tape must be open for reading to seek");

	return blockSeekBwd(offset);
}

size_t FakeTape::max_block_size = 16*1024*1024;
bool FakeTape::trace_ = 0;
std::unique_ptr<std::ostream> FakeTape::debug;

/* vim: set sw=8 sts=8 ts=8 noexpandtab: */
