/*
+  DB Mixer   
   ========   
   Description: 
    interface controller for an external mixer to work with the DBMix system. 
 
    Copyright (c) 2000, 2001 Simon Mark Werner 
 
    DBMix Author: Robert Michael S Dean 
    exmixer Author: Simon Mark Werner 
    Version: 1.0 
 
 
   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version. 
 
   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of 
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
   GNU General Public License for more details. 
 
   You should have received a copy of the GNU General Public License 
   along with this program; if not, write to the Free Software 
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 
 */ 
 
#include <sys/types.h> 
#include <sys/stat.h> 
#include <unistd.h> 
#include <string.h> 
#include <stdio.h> 
#include <stdlib.h> 
 
#include <gdk/gdkkeysyms.h> 
#include <gtk/gtk.h> 
 
/* required for js device */ 
#include <linux/joystick.h> 
#include <sys/ioctl.h> 
#include <sys/time.h> 
#include <fcntl.h> 
 
#include <dbchannel.h>
#include "save_prefs.h" 
#include "exmixer.h" 
#include "exmixer_prefs.h" 
#include <dbdebug.h>
 
typedef struct channel_widget_id_s { 
    int dev_id; 
    int type_id; 
} channel_widget_id; 
 
extern dbfsd_data * sysdata; 
 
GtkWidget *dev_frame_container; 
GtkWidget *External_Mixer_Settings; 
 
channel_widget_id but_ch_wid[MAX_BUT * MAX_DEV]; 
channel_widget_id axes_ch_wid[MAX_AXES * MAX_DEV]; 
 
int but_num, axes_num; 
int edit_prefs;    /* Flag to see if the preferences window is here */ 
int device_combo_changed_escape; 
 
GtkWidget* create_External_Mixer_Settings (void); 
void frame_widget( int ); 
void set_all_combos( int ); 
 
 
/* Create/Redisplay the widgets in each device frame */ 
void create_frame_widgets( void ) 
{ 
    int dev_id; 
    
    Debug("create_frame_widgets()..."); 
    
    /* Create the individual Device frames */ 
    for (dev_id=0; dev_id < MAX_DEV; dev_id++) 
    { 
       /* Only create the widgets when D_DEV_FRAME has not been created */ 
       if (D_DEV_FRAME == NULL) 
       { 
           /* Create the frame */ 
           frame_widget( dev_id ); 
           
           gtk_widget_show( D_DEV_FRAME ); 
           gtk_box_pack_start (GTK_BOX( dev_frame_container ), D_DEV_FRAME, TRUE, TRUE, 0); 
           gtk_container_set_border_width (GTK_CONTAINER (D_DEV_FRAME), 10); 
       } 
       else 
       { 
           set_all_combos( dev_id ); 
           gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(D_CB_ENABLED), D_ENABLED); 
       } 
    } /* dev_id */ 
    
    edit_prefs = TRUE; 
} 
 
 
 
void exit_prefs (GtkObject *object) 
{ 
    int dev_id; 
    
    edit_prefs = FALSE; 
    
    for (dev_id=0; dev_id<MAX_DEV; dev_id++) 
       print_js_settings( dev_id ); 
 
    gtk_widget_destroy(External_Mixer_Settings); 
} 
 
 
void enable_js (GtkToggleButton *t_button, gpointer *d_id) 
{ 
    gint dev_id = GPOINTER_TO_INT(d_id); 
    gint em_enabled = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(t_button)); 
 
    if (em_enabled) 
    { 
        exmixer_start_device( dev_id ); 
    } 
    else 
    { 
        exmixer_disable_device( dev_id ); 
    } 
 
    create_frame_widgets(); 
} 
 
 
void device_combo_changed (GtkEditable *editable, gpointer *d_id) 
{ 
    gint dev_id; 
    gchar *label; 
    
    label = gtk_entry_get_text( GTK_ENTRY(editable) ); 
 
    /* Avoid infinite recursion */ 
    if (device_combo_changed_escape) 
    { 
       device_combo_changed_escape = FALSE; 
       return; 
    } 
    
    /* Check to see we don't use the same device twice */ 
    for (dev_id=0; dev_id<MAX_DEV; dev_id++) 
    { 
       if (dev_id == GPOINTER_TO_INT(d_id)) continue; 
       if (strcmp(D_FILENAME, label) == 0) 
       { 
           gdk_beep(); 
           Debug("Device %s already in use.", D_FILENAME); 
           /* Avoid infinite recursion */ 
           device_combo_changed_escape = TRUE; 
           dev_id = GPOINTER_TO_INT(d_id); 
           gtk_entry_set_text(GTK_ENTRY(editable), D_FILENAME); 
           return; 
       } 
    } 
    
    dev_id = GPOINTER_TO_INT(d_id); 
    sprintf(D_FILENAME, label); 
    exmixer_start_device( dev_id ); 
    
    create_frame_widgets(); 
} 
 
 
void chan_combo_changed (GtkEditable *editable, gint *chan_assignment) 
{ 
    gint ch_id; 
    gchar *label = gtk_entry_get_text( GTK_ENTRY(editable)); 
    
    if (strcmp(label, " ")) 
       ch_id = atoi( label ) - 1; 
    else 
       ch_id = 0; 
    *chan_assignment = ch_id; 
//    Debug("Chan_assignment label = \"%s\", ch_id = %d", label, ch_id); 
} 
 
 
void axes_assign_combo_changed (GtkEditable *editable, gpointer num) 
{ 
    int a_num = GPOINTER_TO_INT( num ); 
    int dev_id = axes_ch_wid[a_num].dev_id; 
    int axis_id = axes_ch_wid[a_num].type_id; 
    gchar *label; 
    int is_sensitive; /* weather or not to shade the channel combo boxes */ 
    
    label = gtk_entry_get_text( GTK_ENTRY(editable) ); 
 
      if ( strcmp( label, "Pitch") == 0) 
      { 
          A_ASSIGN = EM_A_PITCH; 
          is_sensitive = TRUE; 
      } 
    else if ( strcmp( label, "Volume") == 0) 
    { 
       A_ASSIGN = EM_A_VOLUME; is_sensitive = TRUE; 
    } 
    else if ( strcmp( label, "Main Volume") == 0) 
    { 
       A_ASSIGN = EM_A_MAIN_VOLUME; is_sensitive = FALSE; 
    } 
    else if ( strcmp( label, "Cue Volume") == 0) 
    { 
       A_ASSIGN = EM_A_CUE_VOLUME; is_sensitive = FALSE; 
    } 
    else if ( strcmp( label, "Cross Fade") == 0) 
    { 
       A_ASSIGN = EM_A_CROSS_FADE; is_sensitive = FALSE; 
    } 
    else if ( strcmp( label, "Fade Speed") == 0) 
    { 
       A_ASSIGN = EM_A_FADE_SPEED; is_sensitive = FALSE; 
    } 
    else if ( strcmp( label, "Balance") == 0) 
    { 
       A_ASSIGN = EM_A_BALANCE; is_sensitive = FALSE; 
    } 
    else 
    { 
       A_ASSIGN = EM_A_NULL;  is_sensitive = FALSE; 
    } 
    
    gtk_widget_set_sensitive( GTK_WIDGET( A_CHAN_CMB ), is_sensitive ); 
    
    /* Set the label correctly */ 
    if (is_sensitive) 
    { 
       sprintf(label, " %d ", A_CHAN + 1); 
       gtk_entry_set_text( GTK_ENTRY( A_CHAN_ENT ), label); 
    } 
    else 
       gtk_entry_set_text( GTK_ENTRY( A_CHAN_ENT ), " "); 
    
//    Debug("dev_id = %d, axis_id = %d, is_sensitive = %d\n", dev_id, axis_id, is_sensitive); 
} 
 
void button_assign_combo_changed(GtkEditable *editable, gpointer num) 
{ 
    int b_num = GPOINTER_TO_INT( num ); 
    int dev_id = but_ch_wid[b_num].dev_id; 
    int but_id = but_ch_wid[b_num].type_id; 
    gchar *label; 
    int is_sensitive; 
    
    label = gtk_entry_get_text( GTK_ENTRY(editable) ); 
    
    if ( strcmp( label, "Play") == 0) 
    { 
       B_ASSIGN = EM_B_PLAY; 
       is_sensitive = TRUE; 
    } 
    else if ( strcmp( label, "Pause") == 0) 
    { 
       B_ASSIGN = EM_B_PAUSE; 
       is_sensitive = TRUE; 
    } 
    else if ( strcmp( label, "Stop") == 0) 
    { 
       B_ASSIGN = EM_B_STOP; 
       is_sensitive = TRUE; 
    } 
    else if ( strcmp( label, "Cue") == 0) 
    { 
       B_ASSIGN = EM_B_CUE; 
       is_sensitive = TRUE; 
    } 
    else if ( strcmp( label, "Fast Forward") == 0) 
    { 
       B_ASSIGN = EM_B_FAST_FORWARD; 
       is_sensitive = TRUE; 
    } 
    else if ( strcmp( label, "Rewind") == 0) 
    { 
       B_ASSIGN = EM_B_REWIND; 
       is_sensitive = TRUE; 
    } 
    else if ( strcmp( label, "Punch Left") == 0) 
    { 
       B_ASSIGN = EM_B_PUNCH_LEFT; 
       is_sensitive = FALSE; 
    } 
    else if ( strcmp( label, "Punch Right") == 0) 
    { 
       B_ASSIGN = EM_B_PUNCH_RIGHT; 
       is_sensitive = FALSE; 
    } 
    else if ( strcmp( label, "Mute") == 0) 
    { 
       B_ASSIGN = EM_B_MUTE; 
       is_sensitive = TRUE; 
    } 
    else if ( strcmp( label, "Eject") == 0) 
    { 
       B_ASSIGN = EM_B_EJECT; 
       is_sensitive = TRUE; 
    } 
    else if ( strcmp( label, "Next Track") == 0) 
    { 
       B_ASSIGN = EM_B_NEXT; 
       is_sensitive = TRUE; 
    } 
    else if ( strcmp( label, "Previous Track") == 0) 
    { 
       B_ASSIGN = EM_B_PREV; 
       is_sensitive = TRUE; 
    } 
    else 
    { 
       B_ASSIGN = EM_B_NULL; 
       is_sensitive = FALSE; 
    } 
 
    gtk_widget_set_sensitive( GTK_WIDGET( B_CHAN_CMB ), is_sensitive ); 
    
    /* Set the label correctly */ 
    if (is_sensitive) 
    { 
       sprintf(label, " %d ", B_CHAN + 1); 
       gtk_entry_set_text( GTK_ENTRY( B_CHAN_ENT ), label); 
    } 
    else 
       gtk_entry_set_text( GTK_ENTRY( B_CHAN_ENT ), " "); 
    
//    Debug("dev_id = %d, but_id = %d, is_sensitive = %d", dev_id, but_id, is_sensitive); 
 
} 
 
/* Label and (un)shade the slider combos */ 
void set_axis_assign_combos( int dev_id, int axis_id ) 
{ 
    gchar *assign_label = (gchar *)malloc(STR_LEN); 
    gchar *chan_label = (gchar *)malloc(STR_LEN); 
    int is_sensitive = TRUE, ch_sens; 
    
    if (D_FD != -1) 
    { 
       /* Set to the saved settings */ 
       switch (A_ASSIGN) 
       { 
       case EM_A_PITCH: sprintf(assign_label, "Pitch"); ch_sens = TRUE; break; 
       case EM_A_VOLUME: sprintf(assign_label, "Volume"); ch_sens = TRUE; break; 
       case EM_A_CUE_VOLUME: sprintf(assign_label, "Cue Volume"); ch_sens = FALSE; break; 
       case EM_A_MAIN_VOLUME: sprintf(assign_label, "Main Volume"); ch_sens = FALSE; break; 
       case EM_A_CROSS_FADE: sprintf(assign_label, "Cross Fade"); ch_sens = FALSE; break; 
       case EM_A_FADE_SPEED: sprintf(assign_label, "Fade Speed"); ch_sens = FALSE; break; 
       case EM_A_BALANCE: sprintf(assign_label, "Balance"); ch_sens = FALSE; break; 
       case EM_A_NULL: 
       default: 
           sprintf(assign_label, " "); 
           ch_sens = FALSE; 
       } 
        
       if (axis_id < N_AXES) 
           is_sensitive = TRUE; 
       else 
           is_sensitive = FALSE; 
    } 
    else 
    { 
       sprintf(assign_label, " "); 
       ch_sens = FALSE; 
       is_sensitive = FALSE; 
    } 
    
    gtk_entry_set_text(GTK_ENTRY (A_ASSIGN_ENT), assign_label); 
    gtk_widget_set_sensitive( GTK_WIDGET(A_ASSIGN_CMB), is_sensitive); 
    
    if (ch_sens) 
    { 
       sprintf(chan_label, " %d ", A_CHAN + 1); 
    } 
    else 
       sprintf(chan_label, " "); 
        
    
    gtk_entry_set_text(GTK_ENTRY (A_CHAN_ENT), chan_label); 
    gtk_widget_set_sensitive( GTK_WIDGET(A_CHAN_CMB), ch_sens); 
    
    free(assign_label); 
    free(chan_label); 
} 
 
/* Label and (un)shade the button combos */ 
void set_button_assign_combos( int dev_id, int but_id ) 
{ 
    gchar *assign_label = (gchar *)malloc(STR_LEN); 
    gchar *chan_label = (gchar *)malloc(STR_LEN); 
    int is_sensitive = TRUE, ch_sens; 
    
    /* Set the assignment combo */ 
    if (D_FD != -1) 
    { 
       switch (B_ASSIGN) 
       { 
       case EM_B_PLAY: sprintf(assign_label, "Play"); ch_sens = TRUE; break; 
       case EM_B_PAUSE: sprintf(assign_label, "Pause"); ch_sens = TRUE; break; 
       case EM_B_STOP: sprintf(assign_label, "Stop"); ch_sens = TRUE; break; 
       case EM_B_CUE: sprintf(assign_label, "Cue"); ch_sens = TRUE; break; 
       case EM_B_FAST_FORWARD: sprintf(assign_label, "Fast Forward"); ch_sens = TRUE; break; 
       case EM_B_REWIND: sprintf(assign_label, "Rewind"); ch_sens = TRUE; break; 
       case EM_B_MUTE: sprintf(assign_label, "Mute"); ch_sens = TRUE; break; 
       case EM_B_EJECT: sprintf(assign_label, "Eject"); ch_sens = TRUE; break; 
       case EM_B_NEXT: sprintf(assign_label, "Next Track"); ch_sens = TRUE; break; 
       case EM_B_PREV: sprintf(assign_label, "Previous Track"); ch_sens = TRUE; break; 
       case EM_B_PUNCH_LEFT: sprintf(assign_label, "Punch Left"); ch_sens = FALSE; break; 
       case EM_B_PUNCH_RIGHT: sprintf(assign_label, "Punch Right"); ch_sens = FALSE; break; 
       case EM_B_NULL: 
       default: 
           sprintf(assign_label, " "); 
           ch_sens = FALSE; 
       } 
        
       if (but_id < N_BUT) 
           is_sensitive = TRUE; 
       else 
           is_sensitive = FALSE; 
    } 
    else 
    { 
       sprintf(assign_label, " "); 
       ch_sens = FALSE; 
       is_sensitive = FALSE; 
    } 
/*     Debug("dev_id = %d, but_id = %d, is_sensitive = %d, assign_label \"%s\"", dev_id, but_id, is_sensitive, assign_label);  */
    
    gtk_entry_set_text(GTK_ENTRY (B_ASSIGN_ENT), assign_label); 
    gtk_widget_set_sensitive( GTK_WIDGET(B_ASSIGN_CMB), is_sensitive); 
    
    if (ch_sens) 
       sprintf(chan_label, " %d ", B_CHAN + 1); 
    else 
       sprintf(chan_label, " "); 
        
    gtk_entry_set_text(GTK_ENTRY (B_CHAN_ENT), chan_label); 
    gtk_widget_set_sensitive( GTK_WIDGET(B_CHAN_CMB), ch_sens); 
    
    free(assign_label); 
    free(chan_label); 
}    
 
 
/* Set the combo box to their defaults and shade them if required */ 
void set_all_combos( int dev_id ) 
{ 
    int axis_id, but_id; 
    
    for (axis_id=0; axis_id < MAX_AXES; axis_id++) 
    { 
       set_axis_assign_combos( dev_id, axis_id ); 
    } 
    
    for (but_id=0; but_id < MAX_BUT; but_id++) 
    { 
       set_button_assign_combos( dev_id, but_id ); 
    } 
} 
 
/* create each Device frame */ 
void frame_widget( int dev_id ) 
{ 
    GList *device_combo_items = NULL; 
    GList *axes_combo_items = NULL; 
    GList *button_combo_items = NULL; 
    GList *channel_combo_items = NULL; 
    
    GtkWidget *g_label; 
    
    gint i, axis_id, but_id, ch_id = 0; 
    gchar *label = (gchar *)malloc(25); 
    gchar *ch_label[sysdata->num_channels]; 
    gchar *dev_label[4]; 
 
    /* Frame and Label */ 
    sprintf( label, "Device %d", dev_id ); 
    D_DEV_FRAME = gtk_frame_new ( label ); 
    gtk_container_set_border_width (GTK_CONTAINER (D_DEV_FRAME), 10); 
    
    /* Main table inside frame */ 
    i = MAX_BUT + MAX_AXES + 2; 
    D_TABLE = gtk_table_new (i, 3, FALSE); 
    gtk_widget_show (D_TABLE); 
    gtk_container_add (GTK_CONTAINER (D_DEV_FRAME), D_TABLE); 
    
    /* Enabled button */ 
    D_CB_ENABLED = gtk_check_button_new_with_label ("Enabled"); 
    gtk_widget_show (D_CB_ENABLED); 
    gtk_table_attach (GTK_TABLE (D_TABLE), D_CB_ENABLED, 0, 1, 0, 1, 
 (GtkAttachOptions) (0), 
 (GtkAttachOptions) (0), 0, 0); 
    gtk_container_set_border_width (GTK_CONTAINER (D_CB_ENABLED), 4); 
    
    /* Device label */ 
    g_label = gtk_label_new ("Device"); 
    gtk_widget_show (g_label); 
    gtk_table_attach (GTK_TABLE (D_TABLE), g_label, 1, 2, 0, 1, 
 (GtkAttachOptions) (GTK_EXPAND), 
 (GtkAttachOptions) (GTK_EXPAND), 0, 0); 
    gtk_misc_set_padding (GTK_MISC (g_label), 0, 20); 
    
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(D_CB_ENABLED), D_ENABLED); 
    gtk_signal_connect (GTK_OBJECT (D_CB_ENABLED), "toggled", 
                  GTK_SIGNAL_FUNC (enable_js), GINT_TO_POINTER(dev_id)); 
    
    /* Device entry field */ 
    /* avialable device combo box selection*/ 
    for (i=0; i < 4; i++) 
    { 
       /* Create "1", "2", "3" ... */ 
       dev_label[i] = (gchar *)malloc(30); 
       sprintf(dev_label[i], "/dev/js%d", i); 
       device_combo_items = g_list_append (device_combo_items, dev_label[i]); 
    } 
 
    /* Device combo box */ 
    D_COMBO = gtk_combo_new (); 
    gtk_widget_show (D_COMBO); 
    gtk_table_attach (GTK_TABLE (D_TABLE), D_COMBO, 2, 3, 0, 1, 
 (GtkAttachOptions) (GTK_EXPAND), 
 (GtkAttachOptions) (GTK_EXPAND), 8, 0); 
 
    gtk_widget_set_usize (D_COMBO, 143, -2); 
    gtk_combo_set_popdown_strings (GTK_COMBO (D_COMBO), device_combo_items); 
 
    /* the entry box for the device combo */ 
    D_ENTRY = GTK_COMBO (D_COMBO)->entry; 
    gtk_widget_show (D_ENTRY); 
    
    sprintf(label, D_FILENAME); 
    gtk_entry_set_text(GTK_ENTRY (D_ENTRY), label); 
    gtk_entry_set_editable (GTK_ENTRY (D_ENTRY), FALSE); 
 
    gtk_signal_connect (GTK_OBJECT (D_ENTRY), "changed", 
                  GTK_SIGNAL_FUNC (device_combo_changed), GINT_TO_POINTER(dev_id)); 
    
    /* Assignment and channel labels */ 
    g_label = gtk_label_new ("Assignment"); 
    gtk_widget_show (g_label); 
    gtk_table_attach (GTK_TABLE (D_TABLE), g_label, 1, 2, 1, 2, 
 (GtkAttachOptions) (0), 
 (GtkAttachOptions) (0), 0, 0); 
 
    g_label = gtk_label_new ("Channel"); 
    gtk_widget_show (g_label); 
    gtk_table_attach (GTK_TABLE (D_TABLE), g_label, 2, 3, 1, 2, 
 (GtkAttachOptions) (0), 
 (GtkAttachOptions) (0), 0, 0); 
        
    /* Combo box selections */ 
    for (ch_id=0; ch_id < sysdata->num_channels; ch_id++) 
    { 
       /* Create "1", "2", "3" ... */ 
       ch_label[ch_id] = (gchar *)malloc(4); 
       sprintf(ch_label[ch_id], " %d ", ch_id + 1); 
       channel_combo_items = g_list_append (channel_combo_items, ch_label[ch_id]); 
    } 
        
    /* axes assignments */ 
    axes_combo_items = g_list_append (axes_combo_items, " "); 
    axes_combo_items = g_list_append (axes_combo_items, "Pitch"); 
    axes_combo_items = g_list_append (axes_combo_items, "Volume"); 
    axes_combo_items = g_list_append (axes_combo_items, "Main Volume"); 
    axes_combo_items = g_list_append (axes_combo_items, "Cue Volume"); 
    axes_combo_items = g_list_append (axes_combo_items, "Cross Fade"); 
    axes_combo_items = g_list_append (axes_combo_items, "Fade Speed"); 
    axes_combo_items = g_list_append (axes_combo_items, "Balance"); 
 
    /* button combo items */ 
    button_combo_items = g_list_append (button_combo_items, " "); 
    button_combo_items = g_list_append (button_combo_items, "Play"); 
    button_combo_items = g_list_append (button_combo_items, "Pause"); 
    button_combo_items = g_list_append (button_combo_items, "Stop"); 
    button_combo_items = g_list_append (button_combo_items, "Mute"); 
    button_combo_items = g_list_append (button_combo_items, "Cue"); 
    button_combo_items = g_list_append (button_combo_items, "Fast Forward"); 
    button_combo_items = g_list_append (button_combo_items, "Rewind"); 
    button_combo_items = g_list_append (button_combo_items, "Next Track"); 
    button_combo_items = g_list_append (button_combo_items, "Previous Track"); 
    button_combo_items = g_list_append (button_combo_items, "Eject"); 
    button_combo_items = g_list_append (button_combo_items, "Punch Left"); 
    button_combo_items = g_list_append (button_combo_items, "Punch Right"); 
    
    /* Create the axis rows */ 
    for (axis_id=0; axis_id < MAX_AXES; axis_id++) 
    { 
       /* The Axis label */ 
       sprintf(label, "Slider %d", axis_id + 1); 
        
       g_label = gtk_label_new ( label ); 
       gtk_widget_show (g_label); 
       gtk_table_attach (GTK_TABLE (D_TABLE), g_label, 
                  0, 1, axis_id + 2, axis_id + 3, 
 (GtkAttachOptions) (GTK_EXPAND), 
 (GtkAttachOptions) (GTK_EXPAND), 0, 0); 
 
        /* the axis assignment combo */ 
       A_ASSIGN_CMB = gtk_combo_new (); 
       gtk_widget_show (A_ASSIGN_CMB); 
       gtk_table_attach (GTK_TABLE (D_TABLE), A_ASSIGN_CMB, 
                  1, 2, axis_id + 2, axis_id + 3, 
 (GtkAttachOptions) (GTK_EXPAND), 
 (GtkAttachOptions) (GTK_EXPAND), 0, 2); 
       gtk_widget_set_usize (A_ASSIGN_CMB, 143, -2); 
       gtk_combo_set_popdown_strings (GTK_COMBO (A_ASSIGN_CMB), axes_combo_items); 
        
       /* the entry box for the axis assignment combo */ 
       A_ASSIGN_ENT = GTK_COMBO (A_ASSIGN_CMB)->entry; 
       gtk_widget_show (A_ASSIGN_ENT); 
       gtk_entry_set_editable (GTK_ENTRY (A_ASSIGN_ENT), FALSE); 
 
        /* Channel combo Box */ 
       A_CHAN_CMB = gtk_combo_new (); 
       gtk_widget_show (A_CHAN_CMB); 
 
       gtk_table_attach (GTK_TABLE (D_TABLE), A_CHAN_CMB, 
                  2, 3, axis_id + 2, axis_id + 3, 
 (GtkAttachOptions) (GTK_EXPAND), 
 (GtkAttachOptions) 0, 0, 0); 
 
       gtk_widget_set_usize (A_CHAN_CMB, 64, -2); 
       gtk_combo_set_popdown_strings (GTK_COMBO (A_CHAN_CMB), channel_combo_items); 
 
       /* the entry box for the channel */ 
       A_CHAN_ENT = GTK_COMBO (A_CHAN_CMB)->entry; 
       gtk_widget_show (A_CHAN_ENT); 
       gtk_entry_set_editable (GTK_ENTRY (A_CHAN_ENT), FALSE); 
 
          /* Signal handlers for combo boxes */ 
       set_axis_assign_combos( dev_id, axis_id ); 
          
          gtk_signal_connect (GTK_OBJECT (A_CHAN_ENT), "changed", 
              GTK_SIGNAL_FUNC (chan_combo_changed), &A_CHAN); 
                  
       axes_ch_wid[axes_num].dev_id = dev_id; 
       axes_ch_wid[axes_num].type_id = axis_id; 
       gtk_signal_connect (GTK_OBJECT (A_ASSIGN_ENT), "changed", 
                  GTK_SIGNAL_FUNC (axes_assign_combo_changed), GINT_TO_POINTER(axes_num)); 
       axes_num++; 
    } 
 
    for (but_id=0; but_id < MAX_BUT; but_id++) 
    { 
       /* Button Labels */ 
       sprintf(label, "Button %d", but_id + 1); 
 
       g_label = gtk_label_new (label); 
       gtk_widget_show (g_label); 
       gtk_table_attach (GTK_TABLE (D_TABLE), g_label, 
                  0, 1, but_id + MAX_AXES + 2, but_id + MAX_AXES + 3, 
 (GtkAttachOptions) (GTK_EXPAND), 
 (GtkAttachOptions) (GTK_EXPAND), 0, 0); 
 
        /* Button assignment combo */ 
       B_ASSIGN_CMB = gtk_combo_new (); 
       gtk_widget_show (B_ASSIGN_CMB); 
       gtk_table_attach (GTK_TABLE (D_TABLE), B_ASSIGN_CMB, 
                  1, 2, but_id + MAX_AXES + 2, but_id + MAX_AXES + 3, 
 (GtkAttachOptions) (GTK_EXPAND), 
 (GtkAttachOptions) (GTK_EXPAND), 0, 2); 
       gtk_widget_set_usize (B_ASSIGN_CMB, 143, -2); 
       gtk_combo_set_popdown_strings (GTK_COMBO (B_ASSIGN_CMB), button_combo_items); 
        
       /* the entry box for the combo */ 
       B_ASSIGN_ENT = GTK_COMBO (B_ASSIGN_CMB)->entry; 
       gtk_widget_ref (B_ASSIGN_ENT); 
       gtk_widget_show (B_ASSIGN_ENT); 
       gtk_entry_set_editable (GTK_ENTRY (B_ASSIGN_ENT), FALSE); 
        
        /* Button channel combo Box */ 
       B_CHAN_CMB = gtk_combo_new (); 
       gtk_widget_show (B_CHAN_CMB); 
       gtk_table_attach (GTK_TABLE (D_TABLE), B_CHAN_CMB, 
                  2, 3, but_id + MAX_AXES + 2, but_id + MAX_AXES + 3, 
 (GtkAttachOptions) (GTK_EXPAND), 
 (GtkAttachOptions) 0, 0, 0); 
 
       gtk_widget_set_usize (B_CHAN_CMB, 64, -2); 
       gtk_combo_set_popdown_strings (GTK_COMBO (B_CHAN_CMB), channel_combo_items); 
 
       /* Button channel combo label */ 
       B_CHAN_ENT = GTK_COMBO (B_CHAN_CMB)->entry; 
       gtk_widget_show (B_CHAN_ENT); 
       gtk_entry_set_editable (GTK_ENTRY (B_CHAN_ENT), FALSE); 
    
          /* Signal handlers for combo boxes */ 
       set_button_assign_combos( dev_id, but_id ); 
        
       gtk_signal_connect (GTK_OBJECT (B_CHAN_ENT), "changed", 
                  GTK_SIGNAL_FUNC (chan_combo_changed), &B_CHAN); 
    
       but_ch_wid[but_num].dev_id = dev_id; 
       but_ch_wid[but_num].type_id = but_id; 
       gtk_signal_connect (GTK_OBJECT (B_ASSIGN_ENT), "changed", 
                  GTK_SIGNAL_FUNC (button_assign_combo_changed), GINT_TO_POINTER(but_num)); 
       but_num++; 
        
    } 
    
    /* Clean up */ 
    g_list_free (device_combo_items); 
    g_list_free (channel_combo_items); 
    g_list_free (axes_combo_items); 
    g_list_free (button_combo_items); 
    free(label); 
    for (i=0; i<4; i++) free(dev_label[i]); 
    for (i=0; i<sysdata->num_channels; i++) free(ch_label[i]); 
    
} 
 
 
GtkWidget* create_External_Mixer_Settings (void) 
{ 
    GtkWidget *Wid_External_Mixer_Settings; 
    GtkWidget *dialog_vbox1; 
    GtkWidget *upper_dialog_area; 
 
    GtkWidget *lower_dialog_area; 
    GtkWidget *bottom_button_box; 
    GtkWidget *b_close; 
 
    Wid_External_Mixer_Settings = gtk_dialog_new (); 
    gtk_window_set_title (GTK_WINDOW (Wid_External_Mixer_Settings), "Exmixer settings"); 
    gtk_window_set_policy (GTK_WINDOW (Wid_External_Mixer_Settings), TRUE, TRUE, FALSE); 
 
    dialog_vbox1 = GTK_DIALOG (Wid_External_Mixer_Settings)->vbox; 
    gtk_object_set_data (GTK_OBJECT (Wid_External_Mixer_Settings), "dialog_vbox1", dialog_vbox1); 
    gtk_widget_show (dialog_vbox1); 
 
    upper_dialog_area = gtk_vbox_new (FALSE, 0); 
    gtk_widget_show (upper_dialog_area); 
    gtk_box_pack_start (GTK_BOX (dialog_vbox1), upper_dialog_area, TRUE, TRUE, 0); 
 
    dev_frame_container = gtk_hbox_new (FALSE, 0); 
    gtk_widget_show (dev_frame_container); 
    gtk_box_pack_start (GTK_BOX (upper_dialog_area), dev_frame_container, TRUE, TRUE, 0); 
 
    create_frame_widgets(); 
 
    lower_dialog_area = GTK_DIALOG (Wid_External_Mixer_Settings)->action_area; 
    gtk_object_set_data (GTK_OBJECT (Wid_External_Mixer_Settings), "lower_dialog_area", lower_dialog_area); 
    gtk_widget_show (lower_dialog_area); 
    gtk_container_set_border_width (GTK_CONTAINER (lower_dialog_area), 10); 
 
    bottom_button_box = gtk_hbox_new (FALSE, 0); 
    gtk_widget_show (bottom_button_box); 
    gtk_box_pack_start (GTK_BOX (lower_dialog_area), bottom_button_box, TRUE, TRUE, 0); 
 
    b_close = gtk_button_new_with_label("  Ok  "); 
    gtk_widget_show (b_close); 
    gtk_box_pack_start (GTK_BOX (bottom_button_box), b_close, TRUE, FALSE, 0); 
 
    gtk_signal_connect (GTK_OBJECT (Wid_External_Mixer_Settings), "destroy", 
   GTK_SIGNAL_FUNC (exit_prefs), NULL); 
    
    gtk_signal_connect (GTK_OBJECT (b_close), "clicked", 
   GTK_SIGNAL_FUNC (exit_prefs), NULL); 
 
    return Wid_External_Mixer_Settings; 
} 
 
 
/* Main part of exmixer prefrences dialog */ 
void start_exmixer_config (GtkWidget *w) 
{ 
    int dev_id; 
 
    /* Initializations */ 
    for (dev_id=0; dev_id < MAX_DEV; dev_id++) 
       D_DEV_FRAME = NULL; 
        
    but_num = 0; 
    axes_num = 0; 
    edit_prefs = FALSE; 
    device_combo_changed_escape = FALSE; 
    
    /* Create the window */           
    External_Mixer_Settings = create_External_Mixer_Settings (); 
    gtk_widget_show (External_Mixer_Settings); 
} 
