/* SPDX-License-Identifier: MIT */
/**
	@file		main.cpp
	@brief		Basic Functionality Tests for the AJA Anc Library.
	@copyright	(C) 2013-2022 AJA Video Systems, Inc. All rights reserved.
**/
#define DOCTEST_CONFIG_IMPLEMENT_WITH_MAIN
#include "doctest.h"
#include "ajantv2/includes/ntv2endian.h"
#include "ajantv2/includes/ntv2utils.h"
#include "ajabase/common/common.h"
#include "ajabase/common/options_popt.h"
#include "ajabase/common/performance.h"
#include "ajabase/system/debug.h"
#include "ancillarydata_cea608_line21.h"
#include "ancillarydata_cea608_vanc.h"
#include "ancillarydata_cea708.h"
#include "ancillarydata_hdr_hlg.h"
#include "ancillarydata_timecode_atc.h"
#include "ancillarylist.h"

#ifdef AJANTV2_PROPRIETARY
// includes from proprietary libajacc library
#include "ntv2smpteancdata.h"
#include "ntv2captionlogging.h"
#endif

#include <iomanip>
#include <utility>	//	std::rel_ops
#include <random>	//	std::uniform_int_distribution
#if defined(AJA_LINUX)
	#include <string.h>
#endif

using namespace std;
using namespace std::rel_ops;

#define	LOGMYERROR(__x__)	AJA_sREPORT(AJA_DebugUnit_Testing, AJA_DebugSeverity_Error,		__x__)
#define	LOGMYWARN(__x__)	AJA_sREPORT(AJA_DebugUnit_Testing, AJA_DebugSeverity_Warning,	__x__)
#define	LOGMYNOTE(__x__)	AJA_sREPORT(AJA_DebugUnit_Testing, AJA_DebugSeverity_Notice,	__x__)
#define	LOGMYINFO(__x__)	AJA_sREPORT(AJA_DebugUnit_Testing, AJA_DebugSeverity_Info,		__x__)
#define	LOGMYDEBUG(__x__)	AJA_sREPORT(AJA_DebugUnit_Testing, AJA_DebugSeverity_Debug,		__x__)
#if defined(_DEBUG)
	#define	DBG_CHECK_EQ(__x__, __y__)		CHECK_EQ((__x__), (__y__))
#else
	#define	DBG_CHECK_EQ(__x__, __y__)	
#endif

static int	gIsVerbose(0);	//	Verbose output?
static NTV2Buffer	gGumpBuffers[NTV2_MAX_NUM_VIDEO_FORMATS];
static NTV2Buffer gRTPBuffers[NTV2_MAX_NUM_VIDEO_FORMATS];
static NTV2Buffer gVanc8Buffers[NTV2_MAX_NUM_VIDEO_FORMATS];
static NTV2Buffer gVanc10Buffers[NTV2_MAX_NUM_VIDEO_FORMATS];


static const AJA_FrameRate	sNTV2Rate2AJARate[] = {	AJA_FrameRate_Unknown	//	NTV2_FRAMERATE_UNKNOWN	= 0,
													,AJA_FrameRate_6000		//	NTV2_FRAMERATE_6000		= 1,
													,AJA_FrameRate_5994		//	NTV2_FRAMERATE_5994		= 2,
													,AJA_FrameRate_3000		//	NTV2_FRAMERATE_3000		= 3,
													,AJA_FrameRate_2997		//	NTV2_FRAMERATE_2997		= 4,
													,AJA_FrameRate_2500		//	NTV2_FRAMERATE_2500		= 5,
													,AJA_FrameRate_2400		//	NTV2_FRAMERATE_2400		= 6,
													,AJA_FrameRate_2398		//	NTV2_FRAMERATE_2398		= 7,
													,AJA_FrameRate_5000		//	NTV2_FRAMERATE_5000		= 8,
													,AJA_FrameRate_4800		//	NTV2_FRAMERATE_4800		= 9,
													,AJA_FrameRate_4795		//	NTV2_FRAMERATE_4795		= 10,
													,AJA_FrameRate_12000	//	NTV2_FRAMERATE_12000	= 11,
													,AJA_FrameRate_11988	//	NTV2_FRAMERATE_11988	= 12,
													,AJA_FrameRate_1500		//	NTV2_FRAMERATE_1500		= 13,
													,AJA_FrameRate_1498		//	NTV2_FRAMERATE_1498		= 14,
#if !defined(NTV2_DEPRECATE_16_0)
													,AJA_FrameRate_1900		//	NTV2_FRAMERATE_1900		= 15,	// Formerly 09 in older SDKs
													,AJA_FrameRate_1898		//	NTV2_FRAMERATE_1898		= 16, 	// Formerly 10 in older SDKs
													,AJA_FrameRate_1800		//	NTV2_FRAMERATE_1800		= 17,	// Formerly 11 in older SDKs
													,AJA_FrameRate_1798		//	NTV2_FRAMERATE_1798		= 18,	// Formerly 12 in older SDKs
#endif	//	!defined(NTV2_DEPRECATE_16_0)
										};
//	This sequence of 10-bit YUV component values contains two SD ancillary data packets,
//	as they would appear in a NTV2_FBF_10BIT_YCBCR frame buffer.
//		DID=0x45 SDID=0x01 DC=216 CS=0xD2
//		DID=0x45 SDID=0x02 DC=2   CS=0x62
//	It's used in BFT_YUVComponentsTo10BitYUVPackedBuffer,  BFT_SMPTEAncData
static const vector<uint16_t> SD10BitYUVComponents = {	//	Cb		Y		Cr		Cb		Y		Cr		Cb		Y		Cr		Cb		Y		Cr			Cb		Y		Cr		Cb		Y		Cr		Cb		Y		Cr		Cb		Y		Cr
															0x000,	0x3FF,	0x3FF,	0x145,	0x101,	0x2D8,	0x120,	0x221,	0x222,	0x123,	0x224,	0x125,		0x126,	0x227,	0x228,	0x129,	0x12A,	0x22B,	0x12C,	0x22D,	0x22E,	0x12F,	0x230,	0x131,
															0x132,	0x233,	0x134,	0x235,	0x236,	0x137,	0x138,	0x239,	0x23A,	0x13B,	0x23C,	0x13D,		0x13E,	0x23F,	0x140,	0x241,	0x242,	0x143,	0x244,	0x145,	0x146,	0x247,	0x248,	0x149,
															0x14A,	0x24B,	0x14C,	0x24D,	0x24E,	0x14F,	0x250,	0x151,	0x152,	0x253,	0x154,	0x255,		0x256,	0x157,	0x158,	0x259,	0x25A,	0x15B,	0x25C,	0x15D,	0x15E,	0x25F,	0x260,	0x161,
															0x162,	0x263,	0x164,	0x265,	0x266,	0x167,	0x168,	0x269,	0x26A,	0x16B,	0x26C,	0x16D,		0x16E,	0x26F,	0x170,	0x271,	0x272,	0x173,	0x274,	0x175,	0x176,	0x277,	0x278,	0x179,
															0x17A,	0x27B,	0x17C,	0x27D,	0x27E,	0x17F,	0x180,	0x281,	0x282,	0x183,	0x284,	0x185,		0x186,	0x287,	0x288,	0x189,	0x18A,	0x28B,	0x18C,	0x28D,	0x28E,	0x18F,	0x290,	0x191,
															0x192,	0x293,	0x194,	0x295,	0x296,	0x197,	0x198,	0x299,	0x29A,	0x19B,	0x29C,	0x19D,		0x19E,	0x29F,	0x2A0,	0x1A1,	0x1A2,	0x2A3,	0x1A4,	0x2A5,	0x2A6,	0x1A7,	0x1A8,	0x2A9,
															0x2AA,	0x1AB,	0x2AC,	0x1AD,	0x1AE,	0x2AF,	0x1B0,	0x2B1,	0x2B2,	0x1B3,	0x2B4,	0x1B5,		0x1B6,	0x2B7,	0x2B8,	0x1B9,	0x1BA,	0x2BB,	0x1BC,	0x2BD,	0x2BE,	0x1BF,	0x2C0,	0x1C1,
															0x1C2,	0x2C3,	0x1C4,	0x2C5,	0x2C6,	0x1C7,	0x1C8,	0x2C9,	0x2CA,	0x1CB,	0x2CC,	0x1CD,		0x1CE,	0x2CF,	0x1D0,	0x2D1,	0x2D2,	0x1D3,	0x2D4,	0x1D5,	0x1D6,	0x2D7,	0x2D8,	0x1D9,
															0x1DA,	0x2DB,	0x1DC,	0x2DD,	0x2DE,	0x1DF,	0x1E0,	0x2E1,	0x2E2,	0x1E3,	0x2E4,	0x1E5,		0x1E6,	0x2E7,	0x2E8,	0x1E9,	0x1EA,	0x2EB,	0x1EC,	0x2ED,	0x2EE,	0x1EF,	0x2F0,	0x1F1,
															0x1F2,	0x2F3,	0x1F4,	0x2F5,	0x2F6,	0x1F7,	0x2D2,	0x000,	0x3FF,	0x3FF,	0x145,	0x102,		0x102,	0x20F,	0x20A,	0x162,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,
															0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,		0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040,	0x200,	0x040	};
static const vector<uint16_t>	YUVLine (SD10BitYUVComponents);

static inline uint32_t ENDIAN_32HtoN(const uint32_t inValue)	{return NTV2EndianSwap32HtoB(inValue);}


		TEST_CASE("BFT_AncEnums")
		{
			AJADebug::Open();
			//	AJAAncDataType
			for (unsigned ordinal(0);  ordinal < AJAAncDataType_Size;  ordinal++)
			{
				const AJAAncDataType	ancType = AJAAncDataType(ordinal);
				//cerr << DEC(ordinal) << " " << xHEX0N(uint16_t(ordinal),2) << "\t" << ::AJAAncDataTypeToString(ancType,false) << "\t" << ::AJAAncDataTypeToString(ancType,true) << endl;
				CHECK(IS_VALID_AJAAncDataType(ancType));
			}

			//	AJAAncDataLink
			for (unsigned ordinal(0);  ordinal < AJAAncDataLink_Size;  ordinal++)
			{
				const AJAAncDataLink	link = AJAAncDataLink(ordinal);
				cerr << DEC(ordinal) << " " << xHEX0N(uint16_t(ordinal),2) << "\t" << ::AJAAncDataLinkToString(link,false) << "\t" << ::AJAAncDataLinkToString(link,true) << endl;
				if (link == AJAAncDataLink_Unknown)
					CHECK_FALSE(IS_VALID_AJAAncDataLink(link));
				else
					CHECK(IS_VALID_AJAAncDataLink(link));
			}

			//	AJAAncDataStream
			for (unsigned ordinal(0);  ordinal < AJAAncDataStream_Size;  ordinal++)
			{
				const AJAAncDataStream	stream = AJAAncDataStream(ordinal);
				cerr << DEC(ordinal) << " " << xHEX0N(uint16_t(ordinal),2) << "\t" << ::AJAAncDataStreamToString(stream,false) << "\t" << ::AJAAncDataStreamToString(stream,true) << endl;
				if (stream == AJAAncDataStream_Unknown)
					CHECK_FALSE(IS_VALID_AJAAncDataStream(stream));
				else
					CHECK(IS_VALID_AJAAncDataStream(stream));
			}

			//	AJAAncDataChannel
			for (unsigned ordinal(0);  ordinal < AJAAncDataChannel_Size;  ordinal++)
			{
				const AJAAncDataChannel	chan = AJAAncDataChannel(ordinal);
				cerr << DEC(ordinal) << " " << xHEX0N(uint16_t(ordinal),2) << "\t" << ::AJAAncDataChannelToString(chan,false) << "\t" << ::AJAAncDataChannelToString(chan,true) << endl;
				if (chan == AJAAncDataChannel_Unknown)
					CHECK_FALSE(IS_VALID_AJAAncDataChannel(chan));
				else
					CHECK(IS_VALID_AJAAncDataChannel(chan));
			}

			//	AJAAncDataSpace
			for (unsigned ordinal(0);  ordinal < AJAAncDataSpace_Size;  ordinal++)
			{
				const AJAAncDataSpace	space = AJAAncDataSpace(ordinal);
				cerr << DEC(ordinal) << " " << xHEX0N(uint16_t(ordinal),2) << "\t" << ::AJAAncDataSpaceToString(space,false) << "\t" << ::AJAAncDataSpaceToString(space,true) << endl;
				if (space == AJAAncDataSpace_Unknown)
					CHECK_FALSE(IS_VALID_AJAAncDataSpace(space));
				else
					CHECK(IS_VALID_AJAAncDataSpace(space));
			}
		}	//	TEST_CASE("BFT_AncEnums")

		TEST_CASE("BFT_DataLocCompare")
		{
			string info;
			AJAAncDataLoc a, b;
			a.SetDataLink(AJAAncDataLink_A)
				.SetDataStream(AJAAncDataStream_1)
				.SetDataChannel(AJAAncDataChannel_C)
				.SetLineNumber(9)
				.SetHorizontalOffset(200);
			b = a;
			CHECK(AJA_SUCCESS(a.Compare(b)));
			info = a.CompareWithInfo(b);
			CHECK(info.empty());

			b.SetDataLink(AJAAncDataLink_B);
			CHECK(AJA_FAILURE(a.Compare(b)));
			info = a.CompareWithInfo(b);
			CHECK_FALSE(info.empty());	if (gIsVerbose)	cerr << "BFT_DataLocCompare " << __LINE__ << ": " << info << endl;
			CHECK_EQ(info, "Link A != B");

			b = a;
			b.SetDataStream(AJAAncDataStream_2);
			CHECK(AJA_FAILURE(a.Compare(b)));
			info = a.CompareWithInfo(b);
			CHECK_FALSE(info.empty());	if (gIsVerbose) cerr << "BFT_DataLocCompare " << __LINE__ << ": " << info << endl;
			CHECK_EQ(info, "DS1 != DS2");

			b = a;
			b.SetDataChannel(AJAAncDataChannel_Y);
			CHECK(AJA_FAILURE(a.Compare(b)));
			info = a.CompareWithInfo(b);
			CHECK_FALSE(info.empty());	if (gIsVerbose) cerr << "BFT_DataLocCompare " << __LINE__ << ": " << info << endl;
			CHECK_EQ(info, "Data Channel C != Y");

			b = a;
			b.SetLineNumber(10);
			CHECK(AJA_FAILURE(a.Compare(b)));
			info = a.CompareWithInfo(b);
			CHECK_FALSE(info.empty());	if (gIsVerbose) cerr << "BFT_DataLocCompare " << __LINE__ << ": " << info << endl;
			CHECK_EQ(info, "Line 9 != 10");

			b = a;
			b.SetHorizontalOffset(400);
			CHECK(AJA_FAILURE(a.Compare(b)));
			info = a.CompareWithInfo(b);
			CHECK_FALSE(info.empty());	if (gIsVerbose) cerr << "BFT_DataLocCompare " << __LINE__ << ": " << info << endl;
			CHECK_EQ(info, "HOffset 200 != 400");

			//	Multiple diffs
			b = a;
			b.SetDataLink(AJAAncDataLink_B)
				.SetDataStream(AJAAncDataStream_2)
				.SetDataChannel(AJAAncDataChannel_Y)
				.SetLineNumber(10)
				.SetHorizontalOffset(400);
			CHECK(AJA_FAILURE(a.Compare(b)));
			info = a.CompareWithInfo(b);
			CHECK_FALSE(info.empty());	if (gIsVerbose) cerr << "BFT_DataLocCompare " << __LINE__ << ": " << info << endl;
			CHECK(info.find("A != B") > 0);
			CHECK(info.find("DS1 != DS2") > 0);
			CHECK(info.find("C != Y") > 0);
			CHECK(info.find("9 != 10") > 0);
			CHECK(info.find("200 != 400") > 0);
		}	//	TEST_CASE("BFT_DataLocCompare")

		TEST_CASE("BFT_DataLocation")
		{
			typedef	std::set<AJAAncDataLoc>	AncLocationSet;
			//typedef	AncLocationSet::const_iterator		AncLocationSetConstIter;
			AncLocationSet	ancLocations;
			static const uint16_t			lines[]		=	{9,		16,		220,	285,	1910,	2320};
			static const uint16_t			hOffsets[]	=	{0,	AJAAncDataHorizOffset_Anywhere,	AJAAncDataHorizOffset_AnyVanc, AJAAncDataHorizOffset_AnyHanc,	127,	898,	1321};
			static const AJAAncDataChannel	channels[]	=	{AJAAncDataChannel_C,		AJAAncDataChannel_Y};
			static const AJAAncDataStream	streams[]	=	{AJAAncDataStream_1,		AJAAncDataStream_2,	AJAAncDataStream_3,	AJAAncDataStream_4};
			static const AJAAncDataLink		links[]		=	{AJAAncDataLink_A,				AJAAncDataLink_B};
			AJAAncDataLoc	nullLoc;
			AJAAncDataLoc	aFromSets, toSetAllAtOnce, toBeSet;
			string info;

			CHECK_FALSE(nullLoc.IsValid());	//	Invalid, because default constructor makes everything "unknown"
			CHECK(nullLoc == aFromSets);

			for (unsigned lineNdx(0);  lineNdx < sizeof(lines)/sizeof(lines[0]);  lineNdx++)
			{
				const uint16_t lineNum(lines[lineNdx]);
				toBeSet.SetLineNumber(lineNum);
				for (unsigned chanNdx(0);  chanNdx < sizeof(channels)/sizeof(channels[0]);  chanNdx++)
				{
					toBeSet.SetDataChannel(channels[chanNdx]);
					for (unsigned streamNdx(0);  streamNdx < sizeof(streams)/sizeof(streams[0]);  streamNdx++)
					{
						toBeSet.SetDataStream(streams[streamNdx]);
						for (unsigned hOffNdx(0);  hOffNdx < sizeof(hOffsets)/sizeof(hOffsets[0]);  hOffNdx++)
						{
							const uint16_t hOffset(hOffsets[hOffNdx]);
							toBeSet.SetHorizontalOffset(hOffset);
							for (unsigned linkNdx(0);  linkNdx < sizeof(links)/sizeof(links[0]);  linkNdx++)
							{
								const AJAAncDataLoc	bFromConstructor(links[linkNdx], channels[chanNdx], AJAAncDataSpace_Unknown, lineNum, hOffset, streams[streamNdx]);
								toBeSet.SetDataLink(links[linkNdx]);
								toSetAllAtOnce.SetDataLink(links[linkNdx]).SetDataChannel(channels[chanNdx]).SetLineNumber(lineNum).SetHorizontalOffset(hOffset).SetDataStream(streams[streamNdx]);
								aFromSets.SetLineNumber(lineNum).SetHorizontalOffset(hOffsets[hOffNdx]).SetDataChannel(channels[chanNdx]).SetDataStream(streams[streamNdx]).SetDataLink(links[linkNdx]);
								CHECK(bFromConstructor == toBeSet);
								CHECK(aFromSets == bFromConstructor);
								CHECK(aFromSets == toSetAllAtOnce);
								CHECK(aFromSets.IsValid());
								CHECK_EQ(aFromSets.GetDataLink(), links[linkNdx]);
								CHECK_EQ(bFromConstructor.GetDataStream(), streams[streamNdx]);
								CHECK_EQ(toBeSet.GetDataChannel(), channels[chanNdx]);
								CHECK_EQ(aFromSets.GetLineNumber(), lineNum);
								CHECK_EQ(aFromSets.GetHorizontalOffset(), hOffset);
								aFromSets.Reset();
								CHECK_FALSE(aFromSets == bFromConstructor);
								CHECK_FALSE(aFromSets.IsValid());
								ancLocations.insert(bFromConstructor);
							}	//	for each AJAAncDataLink
						}	//	for each horizOffset
					}	//	for each AJAAncDataStream
				}	//	for each AJAAncDataChannel
			}	//	for each line number
			//cerr << endl << endl;		for (AncLocationSetConstIter it(ancLocations.begin());  it != ancLocations.end();  ++it)	cerr << *it << endl;	cerr << endl << endl;

			SUBCASE("SimpleOrdering")
			{	//	Ordering Tests
				AJAAncDataLoc	a, b;
				CHECK_EQ(a, b);	//	AJAAncDataLoc doesn't have != -- requires rel_ops
				//	Check Individual Component Magnitude Comparison
				//	Line Number (highest)
				a.Reset();  a.SetLineNumber(100);  b = a;  b.SetLineNumber(99);		//	B precedes A
				CHECK(b < a);
				CHECK_FALSE(b == a);
				info = a.CompareWithInfo(b);	CHECK_EQ(info.find("Line 100 != 99"), 0);
				//	Horizontal Offset
				a.Reset();  a.SetHorizontalOffset(1023);  b = a;  b.SetHorizontalOffset(1022);	//	B precedes A
				CHECK(b < a);
				CHECK_FALSE(b == a);
				info = a.CompareWithInfo(b);	CHECK_EQ(info.find("HOffset 1023 != 1022"), 0);
				//	Horizontal Offset -- HANC anywhere versus VANC anywhere:
				a.Reset();  a.SetHorizontalOffset(AJAAncDataHorizOffset_AnyHanc);  b = a;  b.SetHorizontalOffset(AJAAncDataHorizOffset_AnyVanc);	//	B precedes A
				CHECK(b < a);
				CHECK_FALSE(b == a);
				info = a.CompareWithInfo(b);	CHECK_EQ(info.find("HOffset 4094 != 4093"), 0);
				//	Data Channel
				a.Reset();  a.SetDataChannel(AJAAncDataChannel_Y);  b = a;  b.SetDataChannel(AJAAncDataChannel_C);	//	B precedes A
				CHECK(b < a);
				CHECK_FALSE(b == a);
				info = a.CompareWithInfo(b);	CHECK_EQ(info.find("Data Channel Y != C"), 0);
				//	Data Stream
				a.Reset();  a.SetDataStream(AJAAncDataStream_3);  b = a;  b.SetDataStream(AJAAncDataStream_2);	//	B precedes A
				CHECK(b < a);
				CHECK_FALSE(b == a);
				info = a.CompareWithInfo(b);	CHECK_EQ(info.find("DS3 != DS2"), 0);
				//	Data Link
				a.Reset();  a.SetDataLink(AJAAncDataLink_B);  b = a;  b.SetDataLink(AJAAncDataLink_A);	//	B precedes A
				CHECK(b < a);
				CHECK_FALSE(b == a);
				info = a.CompareWithInfo(b);	CHECK_EQ(info.find("Link B != A"), 0);
			}

			SUBCASE("ComboOrdering")
			{	//	Combo Tests
				AJAAncDataLoc	a, b;
				a.SetDataLink(AJAAncDataLink_B).SetDataStream(AJAAncDataStream_3).SetDataChannel(AJAAncDataChannel_Y);
				a.SetHorizontalOffset(1000).SetLineNumber(12);
				b = a;  b.SetDataLink(AJAAncDataLink_A);
				CHECK(b < a);
				CHECK_FALSE(b == a);
				b = a;  b.SetDataStream(AJAAncDataStream_2);
				CHECK(b < a);
				CHECK_FALSE(b == a);
				b = a;  b.SetDataChannel(AJAAncDataChannel_C);
				CHECK(b < a);
				CHECK_FALSE(b == a);
				b = a;  b.SetHorizontalOffset(a.GetHorizontalOffset()-1);
				CHECK(b < a);
				CHECK_FALSE(b == a);
				b = a;  b.SetLineNumber(a.GetLineNumber()-1);
				CHECK(b < a);
				CHECK_FALSE(b == a);
			}
		}	//	TEST_CASE("BFT_DataLocation")


		TEST_CASE("BFT_AncDataCompare")
		{
			const string a("This is a test"), b("This is A test");
			const NTV2Buffer A(a.c_str(), a.length()), B(b.c_str(), b.length());
			string info;
			AJAAncillaryData pkt;
			pkt.SetDID(0x05);	pkt.SetSID(0x07);
			pkt.SetDataCoding(AJAAncDataCoding_Digital);
			pkt.SetLocationVideoLink(AJAAncDataLink_A);
			pkt.SetLocationHorizOffset(AJAAncDataHorizOffset_AnyVanc);
			pkt.SetLocationDataChannel(AJAAncDataChannel_Y);
			pkt.SetLocationLineNumber(9);
			pkt.SetPayloadData (A, static_cast<uint32_t>(size_t(A)));
			AJAAncillaryData pktA(pkt);
			CHECK(AJA_SUCCESS(pkt.Compare(pktA, /*ignoreLoc*/false, /*ignoreChkSum*/false)));
			info =   pkt.CompareWithInfo (pktA, /*ignoreLoc*/false, /*ignoreChkSum*/false);
			CHECK(info.empty());

			//	Validate "ignoreLoc"
			pktA.SetLocationDataChannel(AJAAncDataChannel_C);	//	pktA location differs from pkt
			CHECK(AJA_FAILURE(pkt.Compare(pktA, /*ignoreLoc*/false, /*ignoreChkSum*/false)));
			info =   pkt.CompareWithInfo (pktA, /*ignoreLoc*/false, /*ignoreChkSum*/false);
			CHECK_FALSE(info.empty());	if (gIsVerbose) cerr << "BFT_AncDataCompare " << __LINE__ << ": ignoreLoc=N ignoreCS=N " << info << endl;
			CHECK(AJA_FAILURE(pkt.Compare(pktA, /*ignoreLoc*/false, /*ignoreChkSum*/true)));
			info =   pkt.CompareWithInfo (pktA, /*ignoreLoc*/false, /*ignoreChkSum*/true);
			CHECK_FALSE(info.empty());	if (gIsVerbose) cerr << "BFT_AncDataCompare " << __LINE__ << ": ignoreLoc=N ignoreCS=Y " << info << endl;
			CHECK(AJA_SUCCESS(pkt.Compare(pktA, /*ignoreLoc*/true,  /*ignoreChkSum*/false)));
			info =   pkt.CompareWithInfo (pktA, /*ignoreLoc*/true,  /*ignoreChkSum*/false);
			CHECK(info.empty());
			CHECK(AJA_SUCCESS(pkt.Compare(pktA, /*ignoreLoc*/true,  /*ignoreChkSum*/true)));
			info =   pkt.CompareWithInfo (pktA, /*ignoreLoc*/true,  /*ignoreChkSum*/true);
			CHECK(info.empty());

			//	Validate "ignoreChkSum"
			pkt.SetChecksum(pkt.Calculate8BitChecksum());
			pktA = pkt;  pktA.SetPayloadData(B, static_cast<uint32_t>(size_t(B)));	//	pkt & pktA payloads now differ
			pkt.SetChecksum(pktA.Calculate8BitChecksum());	//	force bad pkt CS
			pktA.SetPayloadData(A, static_cast<uint32_t>(size_t(A)));				//	pkt & pktA payloads now match, but CS differs
			CHECK(AJA_FAILURE(pkt.Compare(pktA, /*ignoreLoc*/false, /*ignoreChkSum*/false)));
			info =   pkt.CompareWithInfo (pktA, /*ignoreLoc*/false, /*ignoreChkSum*/false);
			CHECK_FALSE(info.empty());	if (gIsVerbose) cerr << "BFT_AncDataCompare " << __LINE__ << ": ignoreLoc=N ignoreCS=N " << info << endl;
			CHECK(AJA_SUCCESS(pkt.Compare(pktA, /*ignoreLoc*/false, /*ignoreChkSum*/true)));
			info =   pkt.CompareWithInfo (pktA, /*ignoreLoc*/false, /*ignoreChkSum*/true);
			CHECK(info.empty());
			CHECK(AJA_FAILURE(pkt.Compare(pktA, /*ignoreLoc*/true,  /*ignoreChkSum*/false)));
			info =   pkt.CompareWithInfo (pktA, /*ignoreLoc*/true,  /*ignoreChkSum*/false);
			CHECK_FALSE(info.empty());	if (gIsVerbose) cerr << "BFT_AncDataCompare " << __LINE__ << ": ignoreLoc=Y ignoreCS=N " << info << endl;
			CHECK(AJA_SUCCESS(pkt.Compare(pktA, /*ignoreLoc*/true,  /*ignoreChkSum*/true)));
			info =   pkt.CompareWithInfo (pktA, /*ignoreLoc*/true,  /*ignoreChkSum*/true);
			CHECK(info.empty());

			//	Validate both "ignoreLoc" & "ignoreChkSum"
			pktA.SetLocationDataChannel(AJAAncDataChannel_C);	//	pktA location differs from pkt
			CHECK(AJA_FAILURE(pkt.Compare(pktA, /*ignoreLoc*/false, /*ignoreChkSum*/false)));
			info =   pkt.CompareWithInfo (pktA, /*ignoreLoc*/false, /*ignoreChkSum*/false);
			CHECK_FALSE(info.empty());	if (gIsVerbose) cerr << "BFT_AncDataCompare " << __LINE__ << ": ignoreLoc=N ignoreCS=N " << info << endl;
			CHECK(AJA_FAILURE(pkt.Compare(pktA, /*ignoreLoc*/false, /*ignoreChkSum*/true)));
			info =   pkt.CompareWithInfo (pktA, /*ignoreLoc*/false, /*ignoreChkSum*/true);
			CHECK_FALSE(info.empty());	if (gIsVerbose) cerr << "BFT_AncDataCompare " << __LINE__ << ": ignoreLoc=N ignoreCS=Y " << info << endl;
			CHECK(AJA_FAILURE(pkt.Compare(pktA, /*ignoreLoc*/true,  /*ignoreChkSum*/false)));
			info =   pkt.CompareWithInfo (pktA, /*ignoreLoc*/true,  /*ignoreChkSum*/false);
			CHECK_FALSE(info.empty());	if (gIsVerbose) cerr << "BFT_AncDataCompare " << __LINE__ << ": ignoreLoc=Y ignoreCS=N " << info << endl;
			CHECK(AJA_SUCCESS(pkt.Compare(pktA, /*ignoreLoc*/true,  /*ignoreChkSum*/true)));
			info =   pkt.CompareWithInfo (pktA, /*ignoreLoc*/true,  /*ignoreChkSum*/true);
			CHECK(info.empty());
		}	//	TEST_CASE("BFT_AncDataCompare")


		TEST_CASE("BFT_AncListCompare")
		{
			const string a("This is a test"), b("This is A test");
			const NTV2Buffer A(a.c_str(), a.length()), B(b.c_str(), b.length());
			string info;
			NTV2StringList nfo;
			AJAAncillaryData pkt, pktA;
			pkt.SetDID(0x05);	pkt.SetSID(0x07);
			pkt.SetDataCoding(AJAAncDataCoding_Digital);
			pkt.SetLocationVideoLink(AJAAncDataLink_A);
			pkt.SetLocationHorizOffset(AJAAncDataHorizOffset_AnyVanc);
			pkt.SetLocationDataChannel(AJAAncDataChannel_Y);
			pkt.SetLocationLineNumber(9);
			pkt.SetPayloadData (A, static_cast<uint32_t>(size_t(A)));
			pkt.SetChecksum(123);	//	arbitrary CS

			//	Make two lists with same 4 packets:  1st normal;  2nd uses LinkB;  3rd uses C chl;  4th uses line 10
			AJAAncillaryList pktsA, pktsB;
			pktsA.AddAncillaryData(pkt);
			pktA = pkt;		pktA.GetDataLocation().SetDataLink(AJAAncDataLink_B);
			pktsA.AddAncillaryData(pktA);
			pktA = pkt;		pktA.GetDataLocation().SetDataChannel(AJAAncDataChannel_C);
			pktsA.AddAncillaryData(pktA);
			pktA = pkt;		pktA.GetDataLocation().SetLineNumber(10);
			pktsA.AddAncillaryData(pktA);
			pktsB = pktsA;

			//	Test 1:		1 diff:		2nd packet in list B has location difference:   uses LinkA instead of LinkB
			pktsB.GetAncillaryDataAtIndex(1)->GetDataLocation().SetDataLink(AJAAncDataLink_A);
			CHECK(AJA_FAILURE(pktsA.Compare (pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/false)));
//			info =    pktsA.CompareWithInfo (pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/false);		cerr << __LINE__ << ":\t" << info << endl;
			pktsA.CompareWithInfo (nfo,      pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/false);		cerr << __LINE__ << ":\t" << aja::join(nfo,"\n\t") << endl;

			CHECK(AJA_FAILURE(pktsA.Compare (pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/true)));
//			info =    pktsA.CompareWithInfo (pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/true);		cerr << __LINE__ << ":\t" << info << endl;
			pktsA.CompareWithInfo (nfo,      pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/true);		cerr << __LINE__ << ":\t" << aja::join(nfo,"\n\t") << endl;

			CHECK(AJA_SUCCESS(pktsA.Compare (pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/false)));
//			info =    pktsA.CompareWithInfo (pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/false);
			CHECK(pktsA.CompareWithInfo (nfo,pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/false));

			CHECK(AJA_SUCCESS(pktsA.Compare (pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/true)));
//			info =    pktsA.CompareWithInfo (pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/true);
			CHECK(pktsA.CompareWithInfo (nfo,pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/true));


			//	Test 2:		1 diff:		3rd packet in list B has checksum difference
			pktsB = pktsA;
			pktsB.GetAncillaryDataAtIndex(2)->SetChecksum(100);
			CHECK(AJA_FAILURE(pktsA.Compare (pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/false)));
//			info =    pktsA.CompareWithInfo (pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/false);		cerr << __LINE__ << ":\t" << info << endl;
			pktsA.CompareWithInfo (nfo,      pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/false);		cerr << __LINE__ << ":\t" << aja::join(nfo,"\n\t") << endl;

			CHECK(AJA_SUCCESS(pktsA.Compare (pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/true)));
//			info =    pktsA.CompareWithInfo (pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/true);
			CHECK(pktsA.CompareWithInfo (nfo,pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/true));

			CHECK(AJA_FAILURE(pktsA.Compare (pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/false)));
//			info =    pktsA.CompareWithInfo (pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/false);		cerr << __LINE__ << ":\t" << info << endl;
			pktsA.CompareWithInfo (nfo,      pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/false);		cerr << __LINE__ << ":\t" << aja::join(nfo,"\n\t") << endl;

			CHECK(AJA_SUCCESS(pktsA.Compare (pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/true)));
//			info =    pktsA.CompareWithInfo (pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/true);
			CHECK(pktsA.CompareWithInfo (nfo,pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/true));


			//	Test 3:		2 diffs:	Combo of tests 1 & 2
			pktsB.GetAncillaryDataAtIndex(1)->GetDataLocation().SetDataLink(AJAAncDataLink_A);
			CHECK(AJA_FAILURE(pktsA.Compare (pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/false)));
//			info =    pktsA.CompareWithInfo (pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/false);		cerr << __LINE__ << ":\t" << info << endl;
			pktsA.CompareWithInfo (nfo,      pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/false);		cerr << __LINE__ << ":\t" << aja::join(nfo,"\n\t") << endl;

			CHECK(AJA_FAILURE(pktsA.Compare (pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/true)));
//			info =    pktsA.CompareWithInfo (pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/true);		cerr << __LINE__ << ":\t" << info << endl;
			pktsA.CompareWithInfo (nfo,      pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/true);		cerr << __LINE__ << ":\t" << aja::join(nfo,"\n\t") << endl;

			CHECK(AJA_FAILURE(pktsA.Compare (pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/false)));
//			info =    pktsA.CompareWithInfo (pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/false);		cerr << __LINE__ << ":\t" << info << endl;
			pktsA.CompareWithInfo (nfo,      pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/false);		cerr << __LINE__ << ":\t" << aja::join(nfo,"\n\t") << endl;

			CHECK(AJA_SUCCESS(pktsA.Compare (pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/true)));
//			info =    pktsA.CompareWithInfo (pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/true);
			CHECK(pktsA.CompareWithInfo (nfo,pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/true));


			//	Test 4:		3 diffs:	Combo of test 3 + different packet data in 4th packet of list B
			pktsB.GetAncillaryDataAtIndex(3)->SetPayloadData (B, static_cast<uint32_t>(size_t(B)));
			pktsB.GetAncillaryDataAtIndex(3)->GetDataLocation().SetLineNumber(11);
			pktsB.GetAncillaryDataAtIndex(3)->SetSID(0xFE);
			CHECK(AJA_FAILURE(pktsA.Compare (pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/false)));
//			info =    pktsA.CompareWithInfo (pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/false);		cerr << __LINE__ << ":\t" << info << endl;
			pktsA.CompareWithInfo (nfo,      pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/false);		cerr << __LINE__ << ":\t" << aja::join(nfo,"\n\t") << endl;

			CHECK(AJA_FAILURE(pktsA.Compare (pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/true)));
//			info =    pktsA.CompareWithInfo (pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/true);		cerr << __LINE__ << ":\t" << info << endl;
			pktsA.CompareWithInfo (nfo,      pktsB, /*ignoreLoc*/false, /*ignoreChkSum*/true);		cerr << __LINE__ << ":\t" << aja::join(nfo,"\n\t") << endl;

			CHECK(AJA_FAILURE(pktsA.Compare (pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/false)));
//			info =    pktsA.CompareWithInfo (pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/false);		cerr << __LINE__ << ":\t" << info << endl;
			pktsA.CompareWithInfo (nfo,      pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/false);		cerr << __LINE__ << ":\t" << aja::join(nfo,"\n\t") << endl;

			CHECK(AJA_FAILURE(pktsA.Compare (pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/true)));
//			info =    pktsA.CompareWithInfo (pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/true);
			pktsA.CompareWithInfo (nfo,      pktsB, /*ignoreLoc*/true,  /*ignoreChkSum*/true);		cerr << __LINE__ << ":\t" << aja::join(nfo,"\n\t") << endl;
		}	//	TEST_CASE("BFT_AncListCompare")


		TEST_CASE("BFT_AnalogGUMP")
		{
			static const uint8_t GumpBuffer[] = {0xFF, 0xB0, 0x0B, 0x60, 0x60, 0x10, 0x20, 0x00, 0x10, 0x00, 0x50, 0x00, 0x20, 0x30, 0x20, 0x00, 0x40, 0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0xFF, 0xE0, 0x15, 0x00, 0x00, 0xFF, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x0F, 0x10, 0x0F, 0x11, 0x16, 0x1D, 0x26, 0x30, 0x3D, 0x49, 0x56, 0x62, 0x6C, 0x73, 0x7A, 0x7C, 0x7C, 0x7A, 0x73, 0x6C, 0x62, 0x56, 0x49, 
			0x3D, 0x30, 0x26, 0x1D, 0x16, 0x11, 0x0F, 0x11, 0x16, 0x1D, 0x26, 0x31, 0x3C, 0x4A, 0x55, 0x62, 0x6C, 0x74, 0x7A, 0x7D, 0x7D, 0x7A, 0x73, 0x6C, 0x62, 0x55, 0x4A, 0x3D, 0x30, 0x26, 0x1D, 0x15, 0x11, 0x10, 0x11, 0x15, 0x1D, 0x26, 0x31, 0x3C, 0x4A, 0x56, 0x62, 0x6C, 0x74, 0x7A, 0x7C, 0x7D, 0x7A, 0x73, 0x6C, 0x62, 0x56, 0x4A, 0x3C, 0x31, 0x26, 0x1D, 0x16, 0x10, 0x10, 0x11, 0x16, 0x1D, 
			0x26, 0x31, 0x3C, 0x4A, 0x55, 0x61, 0x6C, 0x74, 0x7A, 0x7D, 0x7C, 0x7A, 0x73, 0x6C, 0x62, 0x56, 0x4A, 0x3D, 0x31, 0x25, 0x1D, 0x16, 0x11, 0x0F, 0x11, 0x16, 0x1D, 0x26, 0x31, 0x3D, 0x49, 0x56, 0x62, 0x6C, 0x73, 0x7A, 0x7D, 0x7D, 0x7A, 0x74, 0x6C, 0x61, 0x56, 0x49, 0x3D, 0x31, 0x26, 0x1D, 0x15, 0x11, 0x10, 0x10, 0x16, 0x1D, 0x25, 0x31, 0x3D, 0x49, 0x56, 0x62, 0x6C, 0x74, 0x79, 0x7D, 
			0x7D, 0x79, 0x73, 0x6B, 0x62, 0x56, 0x4A, 0x3D, 0x31, 0x26, 0x1C, 0x16, 0x11, 0x0F, 0x11, 0x15, 0x1D, 0x25, 0x31, 0x3D, 0x49, 0x56, 0x61, 0x6C, 0x74, 0x7A, 0x7D, 0x7D, 0x7A, 0x74, 0x6C, 0x62, 0x55, 0x4A, 0x3D, 0x30, 0x26, 0x1D, 0x16, 0x11, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 
			0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x0F, 0x10, 0x10, 0x0F, 0x10, 0x24, 0x46, 0x6A, 0x7E, 0x7E, 0x7E, 0x7D, 0x00, 0xFF, 0xE0, 0x15, 0x00, 0x00, 0xFF, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7D, 0x6A, 0x46, 0x24, 0x0F, 0x10, 0x0F, 0x0F, 0x10, 
			0x0F, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x0F, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x24, 0x46, 0x6A, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7D, 0x7E, 0x7E, 0x7E, 
			0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7D, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7D, 0x7D, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7D, 0x7E, 0x7E, 0x7D, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7D, 0x7D, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7D, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x6A, 
			0x46, 0x23, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x0F, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x0F, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00, 0xFF, 0xE0, 0x15, 0x00, 0x00, 0xFF, 0x0F, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x0F, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 
			0x10, 0x10, 0x10, 0x24, 0x46, 0x6A, 0x7D, 0x7E, 0x7E, 0x7E, 0x7E, 0x7D, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7D, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7D, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7D, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 
			0x7E, 0x7D, 0x7E, 0x7E, 0x7D, 0x7D, 0x7E, 0x7E, 0x7E, 0x7E, 0x7D, 0x7D, 0x7D, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7D, 0x6A, 0x47, 0x24, 0x10, 0x0F, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00, 0xFF, 0xE0, 0x15, 0x00, 0x00, 0xFF, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00, 0xFF, 0xE0, 0x15, 0x00, 0x00, 0xFF, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00, 0xFF, 0xE0, 0x15, 0x00, 0x00, 0xFF, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00, 0xFF, 0xE0, 0x15, 0x00, 0x00, 
			0xFF, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x00, 0xFF, 0xE0, 0x15, 0x00, 0x00, 0x87, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0xFF, 0xE2, 0x1C, 0x00, 0x00, 0xFF, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x15, 0x1D, 0x25, 0x31, 0x3D, 0x49, 0x56, 0x62, 0x6C, 0x73, 0x7A, 0x7C, 0x7C, 0x7A, 0x73, 0x6C, 0x62, 0x55, 0x49, 0x3E, 0x31, 0x26, 0x1D, 0x15, 0x11, 0x0F, 0x11, 0x16, 0x1C, 0x25, 0x31, 0x3D, 0x4A, 0x56, 0x61, 0x6B, 0x74, 0x79, 0x7D, 0x7D, 0x7A, 0x74, 
			0x6C, 0x61, 0x56, 0x4A, 0x3D, 0x31, 0x26, 0x1D, 0x16, 0x11, 0x0F, 0x11, 0x16, 0x1C, 0x26, 0x30, 0x3D, 0x4A, 0x56, 0x62, 0x6C, 0x73, 0x7A, 0x7D, 0x7E, 0x7A, 0x73, 0x6C, 0x61, 0x55, 0x49, 0x3D, 0x30, 0x26, 0x1D, 0x15, 0x11, 0x0F, 0x11, 0x16, 0x1C, 0x25, 0x31, 0x3D, 0x4A, 0x56, 0x62, 0x6B, 0x73, 0x79, 0x7D, 0x7D, 0x7A, 0x74, 0x6C, 0x62, 0x56, 0x49, 0x3C, 0x31, 0x26, 0x1D, 0x15, 0x11, 
			0x10, 0x11, 0x15, 0x1D, 0x26, 0x30, 0x3D, 0x49, 0x56, 0x62, 0x6C, 0x73, 0x7A, 0x7C, 0x7D, 0x7A, 0x74, 0x6C, 0x61, 0x56, 0x4A, 0x3D, 0x30, 0x27, 0x1D, 0x16, 0x11, 0x0F, 0x11, 0x15, 0x1D, 0x26, 0x31, 0x3C, 0x4A, 0x56, 0x62, 0x6C, 0x74, 0x79, 0x7C, 0x7D, 0x7A, 0x74, 0x6B, 0x62, 0x55, 0x4A, 0x3C, 0x31, 0x26, 0x1D, 0x16, 0x10, 0x0F, 0x11, 0x16, 0x1D, 0x26, 0x31, 0x3D, 0x4A, 0x56, 0x61, 
			0x6C, 0x74, 0x7A, 0x7D, 0x7C, 0x7A, 0x74, 0x6C, 0x62, 0x56, 0x49, 0x3D, 0x30, 0x26, 0x1D, 0x16, 0x11, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x0F, 0x0F, 0x10, 0x10, 0x10, 0x0F, 0x0F, 0x10, 0x10, 0x24, 0x46, 
			0x6B, 0x7D, 0x7E, 0x7D, 0x7E, 0x00, 0xFF, 0xE2, 0x1C, 0x00, 0x00, 0xFF, 0x7E, 0x7E, 0x7D, 0x7D, 0x7E, 0x7E, 0x7D, 0x7E, 0x7E, 0x7D, 0x7E, 0x7D, 0x7D, 0x7E, 0x7E, 0x7D, 0x7E, 0x7E, 0x7D, 0x7E, 0x7D, 0x69, 0x47, 0x23, 0x11, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x0F, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x0F, 
			0x0F, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x0F, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x0F, 0x0F, 0x0F, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x24, 0x46, 0x6A, 0x7D, 0x7E, 0x7D, 0x7D, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7D, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7D, 0x7E, 0x7D, 0x6A, 0x46, 0x24, 0x10, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00, 0xFF, 0xE2, 0x1C, 0x00, 0x00, 0xFF, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 
			0x0F, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x0F, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x0F, 0x10, 0x10, 0x0F, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x0F, 0x0F, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 
			0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x0F, 0x0F, 0x10, 0x10, 0x0F, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x24, 0x47, 0x6A, 
			0x7D, 0x7E, 0x7D, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7D, 0x7D, 0x7E, 0x7D, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7D, 0x7F, 0x7D, 0x6A, 0x47, 0x24, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0xFF, 0xE2, 0x1C, 0x00, 0x00, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0xFF, 0xE2, 0x1C, 0x00, 0x00, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0xFF, 0xE2, 0x1C, 0x00, 0x00, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0xFF, 0xE2, 0x1C, 0x00, 0x00, 0xFF, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0xFF, 0xE2, 0x1C, 0x00, 0x00, 0x87, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 
			0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00	};
			AJAAncillaryData::ResetInstanceCounts();
			{
				AJAAncillaryList pkts;
				CHECK(AJA_SUCCESS(pkts.AddReceivedAncillaryData (NTV2Buffer(GumpBuffer, sizeof(GumpBuffer)))));
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 2);	//	2 alive
				cerr << pkts << endl << "C" << DEC(AJAAncillaryData::GetNumConstructed()) << " D" << DEC(AJAAncillaryData::GetNumDestructed()) << endl;
				pkts.Clear();
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 0);	//	0 alive
			}
			DBG_CHECK_EQ(AJAAncillaryData::GetNumConstructed(), 3);	//	3 created
			DBG_CHECK_EQ(AJAAncillaryData::GetNumDestructed(), 3);	//	3 destroyed
		}	//	TEST_CASE("BFT_AnalogGUMP")

		TEST_CASE("BFT_StripNativeGUMP")
		{
			static const uint64_t GumpF1[] = {	//	NOTE: Grabbed from an Anc buffer on Intel, then NTV2Buffer::ByteSwap64, then NTV2Buffer::Dump:
				0x98881000F405B0FF, 0x000000009C4C6CB8, 0xFF30000000000000, 0x20003010606008B0, 0x4000302020504000, 0xB0FF900000003000, 0x0100058104014109, 0x8C0302610AA0FFCD, 0x07410DA0FFF28080, 0x00000000051D6010, 0x0000000000000000, 0xFF000015E0FFDA00, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010,
				0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x0010101010101010, 0x1010FF000015E0FF, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010,
				0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0xE0FF001010101010, 0x10101010D2000015, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010,
				0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x0000101010101010, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000
				};
			static const uint64_t GumpF2[] = {	//	NOTE: Grabbed from an Anc buffer on Intel, then NTV2Buffer::ByteSwap64, then NTV2Buffer::Dump:
				0x78641000F43EB2FF, 0x0000000080C0F894, 0xFFAC000000000000, 0x00058104014142B2, 0x03026143A2FFCD01, 0x004EE2FFF280808C, 0x101010101010FF00, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010,
				0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0xFF00004EE2FF0010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010,
				0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x0010101010101010, 0x1010D200004EE2FF, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010, 0x1010101010101010,
				0x1010101010101010, 0x1010101010101010, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
				0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000
				};
			AJADebug::Open();
			const NTV2Buffer gumpF1(GumpF1, sizeof(GumpF1)), gumpF2(GumpF2, sizeof(GumpF2));
			AJAAncList pktsPre, pktsPost;
			CHECK(AJA_SUCCESS(AJAAncList::SetFromDeviceAncBuffers(gumpF1, gumpF2, pktsPre)));
			cout << "Before Stripping: " << pktsPre << endl;
			CHECK_EQ(pktsPre.CountAncillaryData(), 10);										//	10 packets total
			CHECK_EQ(pktsPre.CountAncillaryDataWithType(AJAAncDataType_Cea608_Vanc), 2);	//	2 CEA608 packets
			CHECK_EQ(pktsPre.CountAncillaryDataWithType(AJAAncDataType_Timecode_ATC), 1);	//	1 RP188 packet
			CHECK_EQ(pktsPre.CountAncillaryDataWithID(0x41, 0x01), 2);						//	2 VPID packets
			CHECK_EQ(pktsPre.CountAncillaryDataWithID(0x41, 0x07), 1);						//	1 SCTE104 packet

			NTV2Buffer stripF1(gumpF1.GetByteCount()), stripF2(gumpF2.GetByteCount());
			CHECK(AJAAncList::StripNativeInserterGUMPPackets (gumpF1, stripF1));
			CHECK_FALSE(gumpF1.IsContentEqual(stripF1));
			CHECK(AJAAncList::StripNativeInserterGUMPPackets (gumpF2, stripF2));
			CHECK_FALSE(gumpF2.IsContentEqual(stripF2));

			CHECK(AJA_SUCCESS(AJAAncList::SetFromDeviceAncBuffers(stripF1, stripF2, pktsPost)));
			cout << "After Stripping: " << pktsPost << endl;
			CHECK_EQ(pktsPost.CountAncillaryDataWithType(AJAAncDataType_Timecode_ATC), 0);	//	0 RP188 packets
			CHECK_EQ(pktsPost.CountAncillaryDataWithID(0x41, 0x01), 0);						//	0 VPID packets
			CHECK_EQ(pktsPost.CountAncillaryDataWithID(0x41, 0x07), 1);						//	1 SCTE104 packet
			CHECK_EQ(pktsPost.CountAncillaryDataWithType(AJAAncDataType_Cea608_Vanc), 2);	//	2 CEA608 packets
			CHECK_EQ(pktsPost.CountAncillaryData(), 3);										//	3 packets total
		}	//	TEST_CASE("BFT_StripNativeGUMP")

		TEST_CASE("BFT_AncillaryData")
		{
			uint32_t numConst(0), numDest(0);
			AJAAncillaryData::ResetInstanceCounts();
			AJAAncillaryData *pDefaultPkt(AJA_NULL), *pClone(AJA_NULL);
			const AJAAncDataLoc nullLocation;
			vector<uint8_t> buffer4K;
			buffer4K.resize(4096, 0x65);

			{	AJAAncillaryData defaultPkt;	//	constr +1  ==> 1
				//	Verify default values...
				CHECK_FALSE (defaultPkt.GotValidReceiveData());
				CHECK_EQ (defaultPkt.GetDID(), 0x00);
				CHECK_EQ (defaultPkt.GetSID(), 0x00);
				CHECK (defaultPkt.IsEmpty());
				CHECK_EQ (defaultPkt.GetPayloadByteCount(), 0);
				CHECK (defaultPkt.GetPayloadData() == AJA_NULL);
				CHECK_NE (defaultPkt.GetDataLocation(), nullLocation);
				CHECK (defaultPkt.GetDataLocation().IsValid());
				CHECK (defaultPkt.IsLumaChannel());
				CHECK_FALSE (defaultPkt.IsChromaChannel());
				CHECK (defaultPkt.IsDigital());
				CHECK_FALSE (defaultPkt.IsRaw());
				CHECK (defaultPkt.IsVanc());
				CHECK_FALSE (defaultPkt.IsHanc());
				CHECK_EQ (defaultPkt.GetLocationLineNumber(), 0);
				CHECK_EQ(defaultPkt.GetLocationVideoLink(), AJAAncDataLink_A);
				CHECK_EQ(defaultPkt.GetLocationDataStream(), AJAAncDataStream_1);
				CHECK_EQ(defaultPkt.GetLocationVideoSpace(), AJAAncDataSpace_VANC);
				CHECK_EQ(defaultPkt.GetLocationDataChannel(), AJAAncDataChannel_Y);
				AJAAncillaryData::GetInstanceCounts(numConst, numDest);
				cerr << "BFT_AncillaryData:  Validated AJAAncillaryData default values (" << DEC(numConst-numDest) << " alive)" << endl;
				DBG_CHECK_EQ(numConst, 1);	DBG_CHECK_EQ(numDest, 0);

				//	Verify that GetDataLocation/SetDataLocation works...
				if (true)
				{
					const AJAAncDataLoc savedLoc(defaultPkt.GetDataLocation());
						CHECK(AJA_SUCCESS(defaultPkt.SetLocationVideoLink(AJAAncDataLink_B)));
						CHECK(AJA_SUCCESS(defaultPkt.SetLocationDataStream(AJAAncDataStream_4)));
						CHECK(AJA_SUCCESS(defaultPkt.SetLocationHorizOffset(AJAAncDataHorizOffset_AnyHanc)));
						CHECK(AJA_SUCCESS(defaultPkt.SetLocationDataChannel(AJAAncDataChannel_C)));
						CHECK(AJA_SUCCESS(defaultPkt.SetLocationLineNumber(225)));
						CHECK_EQ(defaultPkt.GetLocationLineNumber(), 225);
						CHECK_EQ(defaultPkt.GetLocationVideoLink(), AJAAncDataLink_B);
						CHECK_EQ(defaultPkt.GetLocationDataStream(), AJAAncDataStream_4);
						CHECK_EQ(defaultPkt.GetLocationVideoSpace(), AJAAncDataSpace_HANC);
						CHECK_EQ(defaultPkt.GetLocationDataChannel(), AJAAncDataChannel_C);
					CHECK(AJA_SUCCESS(defaultPkt.SetDataLocation(savedLoc)));
					cerr << "BFT_AncillaryData:  Validated AJAAncillaryData::GetDataLocation, SetDataLocation" << endl;
				}

				//	Validate new "Unknown" packet from AJAAncillaryDataFactory::Create...
				CHECK (pDefaultPkt == AJA_NULL);
				pDefaultPkt = AJAAncillaryDataFactory::Create(AJAAncDataType_Unknown);	//	constr +1  ==> 2
				CHECK(pDefaultPkt != AJA_NULL);
				CHECK_FALSE (pDefaultPkt->GotValidReceiveData());
				CHECK_EQ (pDefaultPkt->GetDID(), 0x00);
				CHECK_EQ (pDefaultPkt->GetSID(), 0x00);
				CHECK (pDefaultPkt->IsEmpty());
				CHECK_EQ (pDefaultPkt->GetPayloadByteCount(), 0);
				CHECK (pDefaultPkt->GetPayloadData() == AJA_NULL);
				CHECK_NE (pDefaultPkt->GetDataLocation(), nullLocation);
				CHECK (pDefaultPkt->GetDataLocation().IsValid());
				CHECK (pDefaultPkt->IsLumaChannel());
				CHECK_FALSE (pDefaultPkt->IsChromaChannel());
				CHECK (pDefaultPkt->IsDigital());
				CHECK_FALSE (pDefaultPkt->IsRaw());
				CHECK (pDefaultPkt->IsVanc());
				CHECK_FALSE (pDefaultPkt->IsHanc());
				CHECK_EQ (pDefaultPkt->GetLocationLineNumber(), 0);
				cerr << "BFT_AncillaryData:  Validated AJAAncillaryData created by AJAAncillaryDataFactory::Create(AJAAncDataType_Unknown)" << endl;

				//	Validate AJAAncillaryData operator == comparing two NULL packets...
				CHECK_EQ(*pDefaultPkt, defaultPkt);

				//	Validate AJAAncillaryData payload data modification...
				static const UByte		pTestBytes []	=	{	0x10,	0x11,	0x12,	0x13,	0x14,	0x15,	0x16,	0x17,	0x18,	0x19,	0x1A,	0x1B,	0x1C,	0x1D,	0x1E,	0x1F,
																0x20,	0x21,	0x22,	0x23,	0x24,	0x25,	0x26,	0x27,	0x28,	0x29,	0x2A,	0x2B,	0x2C,	0x2D,	0x2E,	0x2F	};
				CHECK(AJA_FAILURE(pDefaultPkt->SetPayloadData (NULL, sizeof(pTestBytes))));
				CHECK(AJA_FAILURE(pDefaultPkt->SetPayloadData (pTestBytes, 0)));
				CHECK(AJA_SUCCESS(pDefaultPkt->SetPayloadData (pTestBytes, sizeof(pTestBytes))));
				CHECK(AJA_SUCCESS(pDefaultPkt->SetDID(0x40)));
				CHECK(AJA_SUCCESS(pDefaultPkt->SetSID(0x02)));
				CHECK_EQ (pDefaultPkt->GetDID(), 0x40);
				CHECK_EQ (pDefaultPkt->GetSID(), 0x02);
				CHECK_FALSE (pDefaultPkt->IsEmpty());
				CHECK_EQ (pDefaultPkt->GetPayloadByteCount(), sizeof(pTestBytes));
				CHECK(pDefaultPkt->GetPayloadData() != AJA_NULL);
				CHECK_EQ (pDefaultPkt->GetDC(), sizeof(pTestBytes));
				CHECK_FALSE (pDefaultPkt->ChecksumOK());
				CHECK(AJA_SUCCESS(pDefaultPkt->SetChecksum(pDefaultPkt->Calculate8BitChecksum())));
				CHECK (pDefaultPkt->ChecksumOK());
				CHECK(AJA_SUCCESS(pDefaultPkt->SetDID(0x45)));
				CHECK_NE (pDefaultPkt->GetChecksum(), pDefaultPkt->Calculate8BitChecksum());	//	Unequal, since DID changed
				CHECK_FALSE (pDefaultPkt->ChecksumOK());	//	Fail, since DID changed
				CHECK(AJA_SUCCESS(pDefaultPkt->SetChecksum(pDefaultPkt->Calculate8BitChecksum())));	//	Fix CS
				CHECK_EQ (pDefaultPkt->GetPayloadByteAtIndex(4), pTestBytes[4]);
				CHECK_EQ (pDefaultPkt->GetPayloadByteAtIndex(pDefaultPkt->GetDC()), 0);
				AJAAncillaryData::GetInstanceCounts(numConst, numDest);
				DBG_CHECK_EQ(numConst, 2);	DBG_CHECK_EQ(numDest, 0);

				//	Validate AJAAncillaryData operators...
				CHECK_NE(*pDefaultPkt, defaultPkt);		//	Validate AJAAncillaryData operator !=
				defaultPkt = *pDefaultPkt;				//	Validate AJAAncillaryData operator =
				CHECK_EQ(*pDefaultPkt, defaultPkt);		//	Validate AJAAncillaryData operator ==
				defaultPkt.SetPayloadByteAtIndex(0x1F, defaultPkt.GetDC()/2);
				CHECK_NE(*pDefaultPkt, defaultPkt);
				cerr << "BFT_AncillaryData:  Validated AJAAncillaryData payload data modification" << endl;
				AJAAncillaryData::GetInstanceCounts(numConst, numDest);
				DBG_CHECK_EQ(numConst, 2);	DBG_CHECK_EQ(numDest, 0);

				if (true)	/////	Clone, GeneratePayloadData and GuessAncillaryDataType Tests
				{
					pClone = defaultPkt.Clone();	//	constr +1  ==> 3
					CHECK_EQ(*pClone, defaultPkt);

					//	Validate that Clear works...
					pClone->Clear();
					CHECK_NE(*pClone, defaultPkt);
					CHECK_EQ(*pClone, AJAAncillaryData());	//	constr +1  ==> 4		destr +1  ==> 1
					CHECK_EQ (pClone->GetDID(), 0);
					CHECK_EQ (pClone->GetSID(), 0);
					CHECK (pClone->IsEmpty());
					CHECK_EQ(pClone->GetDataCoding(), AJAAncDataCoding_Digital);
					CHECK(AJA_SUCCESS(pClone->SetDataCoding(AJAAncDataCoding_Raw)));
					CHECK_EQ(pClone->GetDataCoding(), AJAAncDataCoding_Raw);
					CHECK_EQ(pClone->GetLocationLineNumber(), 0);
					CHECK(AJA_SUCCESS(pClone->SetLocationLineNumber(100)));
					CHECK_EQ(pClone->GetLocationLineNumber(), 100);
					CHECK_EQ(pClone->GetLocationVideoSpace(), AJAAncDataSpace_VANC);
					CHECK(AJA_SUCCESS(pClone->SetLocationHorizOffset(AJAAncDataHorizOffset_AnyHanc)));
					CHECK_EQ(pClone->GetLocationVideoSpace(), AJAAncDataSpace_HANC);
					CHECK_EQ(pClone->GetLocationDataChannel(), AJAAncDataChannel_Y);
					CHECK(AJA_SUCCESS(pClone->SetLocationDataChannel(AJAAncDataChannel_C)));
					CHECK_EQ(pClone->GetLocationDataChannel(), AJAAncDataChannel_C);
					CHECK_EQ(pClone->GetLocationVideoLink(), AJAAncDataLink_A);
					CHECK(AJA_SUCCESS(pClone->SetLocationVideoLink(AJAAncDataLink_B)));
					CHECK_EQ(pClone->GetLocationVideoLink(), AJAAncDataLink_B);
					AJAAncillaryData::GetInstanceCounts(numConst, numDest);
					cerr << "BFT_AncillaryData:  Validated AJAAncillaryData::Clone" << " C" << DEC(numConst) << " D" << DEC(numDest) << endl;
					DBG_CHECK_EQ(numConst, 4);	DBG_CHECK_EQ(numDest, 1);

					//	Test generating default packet using AJAAncillaryDataFactory::Create for these AJAAncDataTypes...
					//	...then GeneratePayloadData for it, then clone it, and test AJAAncillaryDataFactory::GuessAncillaryDataType using the clone.
					//	The detected data type should match the original's...
					static const AJAAncDataType	gDataTypes[]	=	{	//AJAAncDataType_Unknown,
																				//	AJAAncDataType_Smpte2016_3,
																				AJAAncDataType_Timecode_ATC
																				,AJAAncDataType_Timecode_VITC
																				,AJAAncDataType_Cea708
																				,AJAAncDataType_Cea608_Vanc
																				,AJAAncDataType_Cea608_Line21
																				//	,AJAAncDataType_Smpte352
																				//	,AJAAncDataType_Smpte2051
																				//	,AJAAncDataType_FrameStatusInfo524D
																				//	,AJAAncDataType_FrameStatusInfo5251
																				//	,AJAAncDataType_HDR_SDR
																				//	,AJAAncDataType_HDR_HDR10
																				//	,AJAAncDataType_HDR_HLG
																				};
					const size_t numTypes (sizeof(gDataTypes) / sizeof(AJAAncDataType));
					for (unsigned ndx(0);  ndx < numTypes;  ndx++)
					{
						const AJAAncDataType	dataType (gDataTypes[ndx]);
						AJAAncillaryData *	pDefaultPkt2 = AJAAncillaryDataFactory::Create(dataType);	//	constr +1  ==> 5, 7, 9, 11, 13
						CHECK(pDefaultPkt2 != AJA_NULL);
						CHECK(AJA_SUCCESS(pDefaultPkt2->GeneratePayloadData()));
						AJAAncillaryData *	pClonePkt	= pDefaultPkt2->Clone();	//	constr +1  ==> 6, 8, 10, 12, 14
						CHECK(pClonePkt != AJA_NULL);
						CHECK_NE(pDefaultPkt2, pClonePkt);	//	Different objects in memory
						CHECK_EQ(*pDefaultPkt2, *pClonePkt);
						CHECK_EQ (AJAAncillaryDataFactory::GuessAncillaryDataType(pClonePkt), dataType);
						delete pClonePkt;		//	destr +1  ==> 2,  4,  6,  8,  10
						delete pDefaultPkt2;	//	destr +1  ==>   3,  5,  7,  9,  11
						AJAAncillaryData::GetInstanceCounts(numConst, numDest);
						cerr << "BFT_AncillaryData:  Validated AJAAncillaryDataFactory::GuessAncillaryDataType for '" << ::AJAAncDataTypeToString(dataType) << "' C" << DEC(numConst) << " D" << DEC(numDest) << endl;
						DBG_CHECK_EQ(numConst, (4 + (ndx+1)*2));	DBG_CHECK_EQ(numDest, 1 + (ndx+1)*2);
					}
				}	//	Clone Tests
				AJAAncillaryData::GetInstanceCounts(numConst, numDest);
				DBG_CHECK_EQ(numConst, 4 + 10);	DBG_CHECK_EQ(numDest, 1 + 10);

				//	Validate AJAAncillaryData payload append...
				CHECK(AJA_SUCCESS(defaultPkt.SetPayloadData(buffer4K.data(), uint32_t(buffer4K.size()))));
				CHECK_EQ (defaultPkt.GetDC(), 4096);
				CHECK(AJA_SUCCESS(defaultPkt.AppendPayload(*pDefaultPkt)));
				CHECK_EQ (defaultPkt.GetDC(), 4096+sizeof(pTestBytes));
				CHECK(AJA_FAILURE(defaultPkt.AppendPayloadData(NULL, 50)));
				CHECK(AJA_FAILURE(defaultPkt.AppendPayloadData(pTestBytes, 0)));
				CHECK(AJA_SUCCESS(defaultPkt.AppendPayloadData(pTestBytes, sizeof(pTestBytes))));
				CHECK_EQ (defaultPkt.GetDC(), 4096 + 2*sizeof(pTestBytes));
				CHECK(AJA_SUCCESS(defaultPkt.AppendPayload(defaultPkt)));
				CHECK_EQ (defaultPkt.GetDC(), 2*(4096 + 2*sizeof(pTestBytes)));

				CHECK(AJA_SUCCESS(defaultPkt.SetDataCoding(AJAAncDataCoding_Digital)));
				CHECK(AJA_SUCCESS(defaultPkt.SetLocationLineNumber(9)));
				CHECK(AJA_SUCCESS(defaultPkt.SetLocationHorizOffset(AJAAncDataHorizOffset_AnyVanc)));
				CHECK(AJA_SUCCESS(defaultPkt.SetLocationDataChannel(AJAAncDataChannel_Y)));
				CHECK(AJA_SUCCESS(defaultPkt.SetLocationDataStream(AJAAncDataStream_1)));
				CHECK(AJA_SUCCESS(defaultPkt.SetLocationVideoLink(AJAAncDataLink_A)));
				CHECK(AJA_SUCCESS(defaultPkt.SetDID(0x45)));
				CHECK(AJA_SUCCESS(defaultPkt.SetSID(0x01)));
				cerr << "BFT_AncillaryData:  Validated AJAAncillaryData::AppendPayloadData" << endl;

				if (true)
				{
					//	Test 10-bit UDW conversion (with parity)...
					static const uint8_t				p8_RDD6Pkt1 []	=	{	0x00,	0x01,	0x02,	0x03,	0x04,	0x05,	0x06,	0x07,	0x08,	0x09,	0x0A,	0x0B,	0x0C,	0x0D,	0x0E,	0x0F,
																				0x10,	0x11,	0x12,	0x13,	0x14,	0x15,	0x16,	0x17,	0x18,	0x19,	0x1A,	0x1B,	0x1C,	0x1D,	0x1E,	0x1F,
																				0x20,	0x21,	0x22,	0x23,	0x24,	0x25,	0x26,	0x27,	0x28,	0x29,	0x2A,	0x2B,	0x2C,	0x2D,	0x2E,	0x2F,
																				0x30,	0x31,	0x32,	0x33,	0x34,	0x35,	0x36,	0x37,	0x38,	0x39,	0x3A,	0x3B,	0x3C,	0x3D,	0x3E,	0x3F,
																				0x40,	0x41,	0x42,	0x43,	0x44,	0x45,	0x46,	0x47,	0x48,	0x49,	0x4A,	0x4B,	0x4C,	0x4D,	0x4E,	0x4F,
																				0x50,	0x51,	0x52,	0x53,	0x54,	0x55,	0x56,	0x57,	0x58,	0x59,	0x5A,	0x5B,	0x5C,	0x5D,	0x5E,	0x5F,
																				0x60,	0x61,	0x62,	0x63,	0x64,	0x65,	0x66,	0x67,	0x68,	0x69,	0x6A,	0x6B,	0x6C,	0x6D,	0x6E,	0x6F,
																				0x70,	0x71,	0x72,	0x73,	0x74,	0x75,	0x76,	0x77,	0x78,	0x79,	0x7A,	0x7B,	0x7C,	0x7D,	0x7E,	0x7F,
																				0x80,	0x81,	0x82,	0x83,	0x84,	0x85,	0x86,	0x87,	0x88,	0x89,	0x8A,	0x8B,	0x8C,	0x8D,	0x8E,	0x8F,
																				0x90,	0x91,	0x92,	0x93,	0x94,	0x95,	0x96,	0x97,	0x98,	0x99,	0x9A,	0x9B,	0x9C,	0x9D,	0x9E,	0x9F,
																				0xA0,	0xA1,	0xA2,	0xA3,	0xA4,	0xA5,	0xA6,	0xA7,	0xA8,	0xA9,	0xAA,	0xAB,	0xAC,	0xAD,	0xAE,	0xAF,
																				0xB0,	0xB1,	0xB2,	0xB3,	0xB4,	0xB5,	0xB6,	0xB7,	0xB8,	0xB9,	0xBA,	0xBB,	0xBC,	0xBD,	0xBE,	0xBF,
																				0xC0,	0xC1,	0xC2,	0xC3,	0xC4,	0xC5,	0xC6,	0xC7,	0xC8,	0xC9,	0xCA,	0xCB,	0xCC,	0xCD,	0xCE,	0xCF,
																				0xD0,	0xD1,	0xD2,	0xD3,	0xD4,	0xD5,	0xD6,	0xD7,	0xD8,	0xD9,	0xDA,	0xDB,	0xDC,	0xDD,	0xDE,	0xDF,
																				0xE0,	0xE1,	0xE2,	0xE3,	0xE4,	0xE5,	0xE6,	0xE7,	0xE8,	0xE9,	0xEA,	0xEB,	0xEC,	0xED,	0xEE,	0xEF,
																				0xF0,	0xF1,	0xF2,	0xF3,	0xF4,	0xF5,	0xF6,	0xF7,	0xF8,	0xF9,	0xFA,	0xFB,	0xFC,	0xFD,	0xFE	};
					static const uint16_t				p16_RDD6Pkt1 [] =	{	0x200,	0x101,	0x102,	0x203,	0x104,	0x205,	0x206,	0x107,	0x108,	0x209,	0x20A,	0x10B,	0x20C,	0x10D,	0x10E,	0x20F,
																				0x110,	0x211,	0x212,	0x113,	0x214,	0x115,	0x116,	0x217,	0x218,	0x119,	0x11A,	0x21B,	0x11C,	0x21D,	0x21E,	0x11F,
																				0x120,	0x221,	0x222,	0x123,	0x224,	0x125,	0x126,	0x227,	0x228,	0x129,	0x12A,	0x22B,	0x12C,	0x22D,	0x22E,	0x12F,
																				0x230,	0x131,	0x132,	0x233,	0x134,	0x235,	0x236,	0x137,	0x138,	0x239,	0x23A,	0x13B,	0x23C,	0x13D,	0x13E,	0x23F,
																				0x140,	0x241,	0x242,	0x143,	0x244,	0x145,	0x146,	0x247,	0x248,	0x149,	0x14A,	0x24B,	0x14C,	0x24D,	0x24E,	0x14F,
																				0x250,	0x151,	0x152,	0x253,	0x154,	0x255,	0x256,	0x157,	0x158,	0x259,	0x25A,	0x15B,	0x25C,	0x15D,	0x15E,	0x25F,
																				0x260,	0x161,	0x162,	0x263,	0x164,	0x265,	0x266,	0x167,	0x168,	0x269,	0x26A,	0x16B,	0x26C,	0x16D,	0x16E,	0x26F,
																				0x170,	0x271,	0x272,	0x173,	0x274,	0x175,	0x176,	0x277,	0x278,	0x179,	0x17A,	0x27B,	0x17C,	0x27D,	0x27E,	0x17F,
																				0x180,	0x281,	0x282,	0x183,	0x284,	0x185,	0x186,	0x287,	0x288,	0x189,	0x18A,	0x28B,	0x18C,	0x28D,	0x28E,	0x18F,
																				0x290,	0x191,	0x192,	0x293,	0x194,	0x295,	0x296,	0x197,	0x198,	0x299,	0x29A,	0x19B,	0x29C,	0x19D,	0x19E,	0x29F,
																				0x2A0,	0x1A1,	0x1A2,	0x2A3,	0x1A4,	0x2A5,	0x2A6,	0x1A7,	0x1A8,	0x2A9,	0x2AA,	0x1AB,	0x2AC,	0x1AD,	0x1AE,	0x2AF,
																				0x1B0,	0x2B1,	0x2B2,	0x1B3,	0x2B4,	0x1B5,	0x1B6,	0x2B7,	0x2B8,	0x1B9,	0x1BA,	0x2BB,	0x1BC,	0x2BD,	0x2BE,	0x1BF,
																				0x2C0,	0x1C1,	0x1C2,	0x2C3,	0x1C4,	0x2C5,	0x2C6,	0x1C7,	0x1C8,	0x2C9,	0x2CA,	0x1CB,	0x2CC,	0x1CD,	0x1CE,	0x2CF,
																				0x1D0,	0x2D1,	0x2D2,	0x1D3,	0x2D4,	0x1D5,	0x1D6,	0x2D7,	0x2D8,	0x1D9,	0x1DA,	0x2DB,	0x1DC,	0x2DD,	0x2DE,	0x1DF,
																				0x1E0,	0x2E1,	0x2E2,	0x1E3,	0x2E4,	0x1E5,	0x1E6,	0x2E7,	0x2E8,	0x1E9,	0x1EA,	0x2EB,	0x1EC,	0x2ED,	0x2EE,	0x1EF,
																				0x2F0,	0x1F1,	0x1F2,	0x2F3,	0x1F4,	0x2F5,	0x2F6,	0x1F7,	0x1F8,	0x2F9,	0x2FA,	0x1FB,	0x2FC,	0x1FD,	0x1FE	};
					vector<uint16_t>	UDWs, componentData;
					CHECK(AJA_SUCCESS(defaultPkt.SetPayloadData(p8_RDD6Pkt1, sizeof(p8_RDD6Pkt1)/sizeof(uint8_t))));
					CHECK(AJA_SUCCESS(defaultPkt.GetPayloadData(UDWs, true)));
					//for (unsigned ndx(0);  ndx < UDWs.size();  )	{	cerr << xHEX0N(UDWs[ndx],3) << ",\t";	if (++ndx % 16 == 0) cerr << endl;	}
					CHECK_EQ(UDWs.size(), sizeof(p16_RDD6Pkt1)/sizeof(uint16_t));
					CHECK_EQ(::memcmp(UDWs.data(), p16_RDD6Pkt1, sizeof(p16_RDD6Pkt1)), 0);

					//	Test GenerateTransmitData...
					CHECK(AJA_SUCCESS(defaultPkt.GenerateTransmitData(componentData)));
					CHECK_EQ(componentData.size(), 262);
					CHECK_EQ(componentData.at(0), 0x000);
					CHECK_EQ(componentData.at(1), 0x3FF);
					CHECK_EQ(componentData.at(2), 0x3FF);
					CHECK_EQ(componentData.at(3)&0x0FF, defaultPkt.GetDID());
					CHECK_EQ(componentData.at(4)&0x0FF, defaultPkt.GetSID());
					CHECK_EQ(componentData.at(5)&0x0FF, defaultPkt.GetDC());
					CHECK_EQ(componentData.at(6), p16_RDD6Pkt1[0]);
					CHECK_EQ(::memcmp(&componentData[6], p16_RDD6Pkt1, sizeof(p16_RDD6Pkt1)), 0);
					CHECK_EQ(componentData.back()&0x0FF, defaultPkt.Calculate8BitChecksum());
					cerr << "BFT_AncillaryData:  Validated AJAAncillaryData::GenerateTransmitData(vector<uint16_t>&) 10-bit UDW conversion (with parity)" << endl;
				}
			}	//	defaultPkt destroyed:	destr +1  ==> 12
			AJAAncillaryData::GetInstanceCounts(numConst, numDest);
//			cerr << "BFT_AncillaryData passed" << " C" << DEC(numConst) << " D" << DEC(numDest) << endl;
			DBG_CHECK_EQ(numConst, 4 + 10);	DBG_CHECK_EQ(numDest, 1 + 1 + 10);
		}	//	TEST_CASE("BFT_AncillaryData")


#ifdef AJANTV2_PROPRIETARY
		//	We now test SMPTEAncData from the AJACC library here
		//	because it's now used extensively by AJAANC.
		TEST_CASE("BFT_SMPTEAncData")
		{
			const vector<uint16_t> & in10BitYUVReferenceLine(SD10BitYUVComponents);
			static const uint16_t	pv210YSamples [] =	{	0x000,	0x3FF,	0x3FF,	0x161,	0x101,	0x152,	0x296,	0x269,	0x152,	0x14F,	0x167,	0x2A9,	0x27E,	0x272,	0x1F4,	0x2FC,
															0x120,	0x173,	0x2F9,	0x200,	0x200,	0x2FF,	0x146,	0x228,	0x1FE,	0x173,	0x265,	0x1FE,	0x16D,	0x269,	0x1FE,	0x16E,
															0x161,	0x1FE,	0x272,	0x179,	0x1FE,	0x200,	0x200,	0x2FA,	0x200,	0x200,	0x2FA,	0x200,	0x200,	0x2FA,	0x200,	0x200,
															0x2FA,	0x200,	0x200,	0x2FA,	0x200,	0x200,	0x2FA,	0x200,	0x200,	0x2FA,	0x200,	0x200,	0x2FA,	0x200,	0x200,	0x2FA,
															0x200,	0x200,	0x2FA,	0x200,	0x200,	0x2FA,	0x200,	0x200,	0x2FA,	0x200,	0x200,	0x173,	0x191,	0x2E1,	0x200,	0x200,
															0x200,	0x1C1,	0x23F,	0x2FF,	0x274,	0x2A9,	0x27E,	0x2E2,	0x2B4,	//	end of packet
															0x040,	0x040,	0x040,	0x040,	0x040,	0x040,	0x040,	0x040,	0x040,	0x040,	0x040,	0x040,	0x040,	0x040,	0x040,	0x040,	0x040,	0x040,	0x040};
			UWordSequence				v210VancLine;
			UWordVANCPacketList			u16Pkts;
			AJAAncillaryList			pktList;
			AJAAncillaryData *			pPkt	(AJA_NULL);
			AJAAncillaryData_Cea708 *	p708Pkt	(AJA_NULL);
			UWordSequence				u16s;
			uint32_t numConst(0), numDest(0);
			AJAAncillaryData::ResetInstanceCounts();

			///////////////////////////////////////////////////////////////////////	BEGIN TEST SECTION 1
			//	The following 3 tests perform a round-trip validation of:
			//		UWordSequence (aka vector<uint16_t>)
			//			...into...
			//				CNTV2SMPTEAncData::GetAncPacketsFromVANCLine
			//					...resulting in...
			//						UWordVANCPacketList
			//							...into...
			//								AJAAncillaryList::AddVANCData
			//									...resulting in...
			//										AJAAncillaryList containing one AJAAncillaryData packet
			//											...into...
			//												AJAAncillaryDataFactory::GuessAncillaryDataType
			//												AJAAncillaryDataFactory::Create
			//													...resulting in...
			//														AJAAncillaryData_Cea708 instance
			//															...to call...
			//																AJAAncillaryData::GetPayloadData
			//																	...resulting in...
			//																		UWordSequence (aka vector<uint16_t>)

			//	TEST 1:		Y-Channel-only CEA708 PACKET
			for (unsigned ndx(0);  ndx < sizeof(pv210YSamples);  ndx++)
			{
				v210VancLine.push_back(0x040);				//	Chroma
				v210VancLine.push_back(pv210YSamples[ndx]);	//	Luma
			}
			UWordSequence	hOffsets;
			CHECK_FALSE (CNTV2SMPTEAncData::GetAncPacketsFromVANCLine (UWordSequence(), kNTV2SMPTEAncChannel_Y, u16Pkts, hOffsets));	//	This should fail (empty UWordSequence)
			CHECK (u16Pkts.empty());		//	Returned UWordSequence should be empty
			CHECK (CNTV2SMPTEAncData::GetAncPacketsFromVANCLine (v210VancLine, kNTV2SMPTEAncChannel_C, u16Pkts, hOffsets));	//	Should succeed, but no C-channel packets
			CHECK (u16Pkts.empty());		//	Expect no C-channel packets
			CHECK (CNTV2SMPTEAncData::GetAncPacketsFromVANCLine (v210VancLine, kNTV2SMPTEAncChannel_Y, u16Pkts, hOffsets));	//	Should succeed, 1 Y-channel packet
			CHECK_FALSE (u16Pkts.empty());		//	Expect 1 Y-channel packet
			CHECK_EQ (u16Pkts.size(), 1);	//	Expect 1 Y-channel packet
			CHECK(AJA_SUCCESS(pktList.AddVANCData(u16Pkts.front(), AJAAncDataLoc(AJAAncDataLink_A, AJAAncDataChannel_Y, AJAAncDataSpace_VANC, 9))));	//	Make a packet list from it
			CHECK_EQ(pktList.CountAncillaryData(), 1);	//	List should contain 1 packet
			pPkt = pktList.GetAncillaryDataAtIndex(0);			//	Get a pointer to the 1 and only packet
			CHECK(pPkt != AJA_NULL);							//	Pointer should be non-NULL
			CHECK_EQ(AJAAncDataType_Cea708, AJAAncillaryDataFactory::GuessAncillaryDataType(pPkt));	//	Guessed Anc type should be CEA708
			p708Pkt = reinterpret_cast <AJAAncillaryData_Cea708 *> (AJAAncillaryDataFactory::Create(AJAAncDataType_Cea708, pPkt));	//	Make a 708-specific packet instance
			CHECK(p708Pkt != AJA_NULL);								//	708-specific packet instance creation should work
			CHECK(AJA_SUCCESS(p708Pkt->GetPayloadData(u16s)));			//	Get its packet data as uint16_t vector (with parity)
			CHECK_EQ(uint32_t(u16s.size()), p708Pkt->GetDC());	//	Vector element count should match packet data count
			CHECK(size_t(u16s.size()) <= sizeof(pv210YSamples));//	Vector element count should be <= original pkt data count
			for (UWordSequence::size_type ndx(0);  ndx < u16s.size();  ndx++)
				CHECK_EQ(pv210YSamples[ndx+6], u16s.at(ndx));	//	Each element should match original

			//	Start over...
			v210VancLine.clear();	u16Pkts.clear();	u16s.clear();	pktList.Clear();
			delete p708Pkt;			p708Pkt = NULL;		pPkt = AJA_NULL;
			AJAAncillaryData::GetInstanceCounts(numConst, numDest);
			DBG_CHECK_EQ(numConst, 3);	DBG_CHECK_EQ(numDest, 3);

			//	TEST 2:		C-channel-only CEA708 PACKET
			for (unsigned ndx(0);  ndx < sizeof(pv210YSamples);  ndx++)
			{
				v210VancLine.push_back(pv210YSamples[ndx]);	//	Chroma
				v210VancLine.push_back(0x040);				//	Luma
			}
			CHECK (CNTV2SMPTEAncData::GetAncPacketsFromVANCLine (v210VancLine, kNTV2SMPTEAncChannel_Y, u16Pkts, hOffsets));	//	Should succeed, but no Y-channel packets
			CHECK (u16Pkts.empty());		//	Expect no Y-channel packets
			CHECK (CNTV2SMPTEAncData::GetAncPacketsFromVANCLine (v210VancLine, kNTV2SMPTEAncChannel_C, u16Pkts, hOffsets));	//	Should succeed, 1 C-channel packet
			CHECK_FALSE (u16Pkts.empty());		//	Expect 1 C-channel packet
			CHECK_EQ (u16Pkts.size(), 1);	//	Expect 1 C-channel packet
			CHECK(AJA_SUCCESS(pktList.AddVANCData(u16Pkts.front(), AJAAncDataLoc(AJAAncDataLink_A, AJAAncDataChannel_C, AJAAncDataSpace_VANC, 9))));	//	Make a packet list from it
			CHECK_EQ(pktList.CountAncillaryData(), 1);	//	List should contain 1 packet
			pPkt = pktList.GetAncillaryDataAtIndex(0);			//	Get a pointer to the 1 and only packet
			CHECK(pPkt != AJA_NULL);							//	Pointer should be non-NULL
			CHECK_EQ(AJAAncDataType_Cea708, AJAAncillaryDataFactory::GuessAncillaryDataType(pPkt));	//	This used to fail because it's not in Y channel, but changed in SDK 16.1 to succeed & log warning
			CHECK(AJA_SUCCESS(pPkt->GetPayloadData(u16s)));				//	Get its packet data as uint16_t vector (with parity)
			CHECK_EQ(uint32_t(u16s.size()), pPkt->GetDC());		//	Vector element count should match packet data count
			CHECK(size_t(u16s.size()) <= sizeof(pv210YSamples));//	Vector element count should be <= original pkt data count
			for (UWordSequence::size_type ndx(0);  ndx < u16s.size();  ndx++)
				CHECK_EQ(pv210YSamples[ndx+6], u16s.at(ndx));	//	Each element should match original

			//	Start over...
			v210VancLine.clear();	u16Pkts.clear();	u16s.clear();	pktList.Clear();	pPkt = AJA_NULL;
			AJAAncillaryData::GetInstanceCounts(numConst, numDest);
			DBG_CHECK_EQ(numConst, 5);	DBG_CHECK_EQ(numDest, 5);

			//	TEST 3:		Y&C-channel CEA708 PACKET
			for (unsigned ndx(0);  ndx < sizeof(pv210YSamples);  ndx++)
				v210VancLine.push_back(pv210YSamples[ndx]);	//	Both Chroma & Luma
			CHECK (CNTV2SMPTEAncData::GetAncPacketsFromVANCLine (v210VancLine, kNTV2SMPTEAncChannel_Y, u16Pkts, hOffsets));	//	Should succeed, but no Y-channel-only packets
			CHECK (u16Pkts.empty());		//	Expect no Y-channel-only packets
			CHECK (CNTV2SMPTEAncData::GetAncPacketsFromVANCLine (v210VancLine, kNTV2SMPTEAncChannel_C, u16Pkts, hOffsets));	//	Should succeed, but no C-channel-only packets
			CHECK (u16Pkts.empty());		//	Expect no C-channel-only packets
			CHECK (CNTV2SMPTEAncData::GetAncPacketsFromVANCLine (v210VancLine, kNTV2SMPTEAncChannel_Both, u16Pkts, hOffsets));	//	Should succeed, 1 Y&C-channel packet
			CHECK_EQ (u16Pkts.size(), 1);	//	Expect 1 Y&C-channel packet
			CHECK(AJA_SUCCESS(pktList.AddVANCData(u16Pkts.front(), AJAAncDataLoc(AJAAncDataLink_A, AJAAncDataChannel_Both, AJAAncDataSpace_VANC, 9))));	//	Make a packet list from it
			CHECK_EQ(pktList.CountAncillaryData(), 1);	//	List should contain 1 packet
			pPkt = pktList.GetAncillaryDataAtIndex(0);			//	Get a pointer to the 1 and only packet
			CHECK(pPkt != AJA_NULL);							//	Pointer should be non-NULL
			CHECK_EQ(AJAAncDataType_Cea708, AJAAncillaryDataFactory::GuessAncillaryDataType(pPkt));	//	This used to fail, but changed in SDK 16.1 to succeed as CEA708 starting to be carried in SD
			CHECK(AJA_SUCCESS(pPkt->GetPayloadData(u16s)));		//	Get its packet data as uint16_t vector (with parity)
			CHECK_EQ(uint32_t(u16s.size()), pPkt->GetDC());		//	Vector element count should match packet data count
			CHECK(size_t(u16s.size()) <= sizeof(pv210YSamples));//	Vector element count should be <= original pkt data count
			for (UWordSequence::size_type ndx(0);  ndx < u16s.size();  ndx++)
				CHECK_EQ(pv210YSamples[ndx+6], u16s.at(ndx));	//	Each element should match original
			//p708Pkt->Print(cerr, true);
			AJAAncillaryData::GetInstanceCounts(numConst, numDest);
			DBG_CHECK_EQ(numConst, 7);	DBG_CHECK_EQ(numDest, 6);
			///////////////////////////////////////////////////////////////////////	END TEST SECTION 1

			if (false)		//	** MrBill **	NOT QUITE READY FOR PRIME-TIME (SD NOT YET PASSING THIS TEST)
			{
				const uint8_t	TEST_DID	(0xAB);
				const uint8_t	TEST_SID	(0xCD);
				///////////////////////////////////////////////////////////////////////	BEGIN  CNTV2SMPTEAncData::FindAnc  TEST
				//	For each of a few NTV2VideoFormats (SD 525i, HD 720p, HD 1080i)
				//		For each of two NTV2FrameBufferFormats '2vuy' and 'v210' . . .
				//			For each VANC line in "Tall" mode . . .
				//				For each channel C then Y (unless SD, in which case C means "both")...
				//					Generate a test packet
				//					Stuff the test packet into the VANC line
				//					Call FindAnc at the start of the frame buffer...
				//						-	Once looking in the C channel (or "both" channels for SD)
				//						-	If HD, look again in the Y channel
				//					Confirm that FindAnc...
				//						-	found the packet in the channel it was supposed to;
				//							-	on the correct line
				//							-	at the correct pixel offset
				//						-	failed to find the packet in the channel it wasn't supposed to
				//
				const NTV2VideoFormat				VFs[]	=	{NTV2_FORMAT_525_5994, NTV2_FORMAT_720p_5994, NTV2_FORMAT_1080i_5994};
				const NTV2PixelFormat				FBFs[]	=	{NTV2_FBF_8BIT_YCBCR, NTV2_FBF_10BIT_YCBCR};
				const AJAAncDataChannel				CHLs[]	=	{AJAAncDataChannel_C, AJAAncDataChannel_Y};
				const NTV2_SMPTEAncChannelSelect	CHSs[]	=	{kNTV2SMPTEAncChannel_Both, kNTV2SMPTEAncChannel_Y, kNTV2SMPTEAncChannel_C};
				static const string					sCHSs[]	=	{"Y", "C", "Y+C", ""};
				for (unsigned VFndx(0);  VFndx < sizeof(VFs)/sizeof(VFs[0]);  VFndx++)
				{
					const NTV2VideoFormat	vf		(VFs[VFndx]);
					const bool				isSD	(NTV2_IS_SD_VIDEO_FORMAT(vf));
					for (unsigned FBFndx(0);  FBFndx < sizeof(FBFs)/sizeof(FBFs[0]);  FBFndx++)
					{
						const NTV2PixelFormat		fbf	(FBFs[FBFndx]);
						const NTV2FormatDescriptor	fd	(vf, fbf, NTV2_VANCMODE_TALL);
						if (!fd.IsValid()) cerr << ::NTV2FrameBufferFormatToString(fbf) << endl;
						CHECK(fd.IsValid());
						CHECK(fd.IsVANC());
						NTV2Buffer					fb	(size_t(fd.GetTotalRasterBytes() - fd.GetVisibleRasterBytes()));	//	Just VANC lines
						fb.Fill(UWord(0x8080));
cerr << endl << endl << "===========================================================================================================================================" << endl;
AJA_sREPORT(AJA_DebugUnit_SMPTEAnc, AJA_DebugSeverity_Notice,	__FUNCTION__ << ":  " << fd);
cerr << __FUNCTION__ << ":  " << fd << endl;
						for (UWord pktLineOffset(0);  pktLineOffset < fd.GetFirstActiveLine();  pktLineOffset++)
						{
							AJAAncillaryData pkt;
							AJAAncillaryList	pkts;
							ULWord				smpteLine	(0);
							bool				isF2		(false);
							for (unsigned CHLndx(0);  CHLndx < sizeof(CHLs)/sizeof(CHLs[0]);  CHLndx++)
							{
								const AJAAncDataChannel	chan	(CHLs[CHLndx]);
								CHECK(fd.GetSMPTELineNumber(pktLineOffset, smpteLine, isF2));
								CHECK(AJA_SUCCESS(pkt.SetDID(TEST_DID)));	CHECK(AJA_SUCCESS(pkt.SetSID(TEST_SID)));
								CHECK(AJA_SUCCESS(pkt.SetDataCoding(AJAAncDataCoding_Digital)));
								CHECK(AJA_SUCCESS(pkt.SetLocationVideoLink(AJAAncDataLink_A)));
								CHECK(AJA_SUCCESS(pkt.SetLocationHorizOffset(AJAAncDataHorizOffset_AnyVanc)));
								CHECK(AJA_SUCCESS(pkt.SetLocationDataChannel(chan)));
								CHECK(AJA_SUCCESS(pkt.SetLocationLineNumber(smpteLine)));
								const uint8_t		pTestData[]	=	{0xAA, 0xBB, uint8_t(vf), uint8_t(fbf), uint8_t(pkt.GetLocationVideoLink()), uint8_t(pkt.GetLocationDataChannel()), uint8_t(pkt.GetLocationVideoSpace()), uint8_t(smpteLine), uint8_t(pkt.GetDataCoding()), 0xBB, 0xAA};
								CHECK(AJA_SUCCESS(pkt.SetPayloadData (pTestData, sizeof(pTestData))));
								CHECK(AJA_SUCCESS(pkts.AddAncillaryData(pkt)));
//cerr << "Line offset " << pktLineOffset << "(" << smpteLine << ") BEFORE GetVANCTransmitData:" << endl;
//CNTV2CaptionLogConfig::DumpMemory(fd.GetRowAddress(fb.GetHostPointer(), pktLineOffset), fd.GetBytesPerRow(), cerr, 16/*inRadix*/, NTV2_IS_FBF_8BIT(fbf)?1:4/*bytesPerGroup*/,NTV2_IS_FBF_8BIT(fbf)?64:16/*groupsPerRow*/,0/*addressRadix*/,false/*ascii*/);
fb.Fill(UWord(0x8080));	//	** MrBill **	FOR NOW
								CHECK(AJA_SUCCESS(pkts.GetVANCTransmitData (fb,  fd)));
								//	At this point, the packet should be in the frame buffer's VANC area.
AJA_sDEBUG(AJA_DebugUnit_SMPTEAnc, "PKT SHOULD BE FOUND AT lineOffset=" << pktLineOffset << " SMPTELine=" << smpteLine << " chan=" << (isSD?(chan==AJAAncDataChannel_C?"Y+C":"ILLEGAL"):(chan==AJAAncDataChannel_C?"C":"Y")));
cerr << "Line offset " << pktLineOffset << "(" << smpteLine << ") AFTER GetVANCTransmitData:" << endl;
CNTV2CaptionLogConfig::DumpMemory(fd.GetRowAddress(fb.GetHostPointer(), pktLineOffset), fd.GetBytesPerRow(), cerr, 16/*inRadix*/, NTV2_IS_FBF_8BIT(fbf)?1:4/*bytesPerGroup*/,NTV2_IS_FBF_8BIT(fbf)?64:16/*groupsPerRow*/,0/*addressRadix*/,false/*ascii*/);
AJA_sDEBUG(AJA_DebugUnit_SMPTEAnc, ":  PKT SHOULD BE FOUND AT lineOffset=" << pktLineOffset << " SMPTELine=" << smpteLine << " chan=" << (isSD?(chan==AJAAncDataChannel_C?"Y+C":"ILLEGAL"):(chan==AJAAncDataChannel_C?"C":"Y")));
vector<uint16_t>	u16pktComponents;

CHECK(AJA_SUCCESS(pkt.GenerateTransmitData(u16pktComponents)));
cerr <<  "U16 Packet Components returned from GenerateTransmitData:" << endl;  for(unsigned n(0);  n < u16pktComponents.size();  n++)	cerr << " " << HEX0N(u16pktComponents[n],4); ;

//	Round-trip test:
	{
		AJAAncillaryList	compPkts;
		UWordSequence		uwords;
		unsigned			ndx	(0);
		if (NTV2_IS_FBF_8BIT(fbf))
			CNTV2SMPTEAncData::UnpackLine_8BitYUVtoUWordSequence (fd.GetRowAddress(fb.GetHostPointer(), pktLineOffset), uwords, fd.GetRasterWidth());
		else
			::UnpackLine_10BitYUVtoUWordSequence (fd.GetRowAddress(fb.GetHostPointer(), pktLineOffset), uwords, fd.GetRasterWidth());
//cerr << "UWordSequence returned from " << (NTV2_IS_FBF_8BIT(fbf)?"CNTV2SMPTEAncData::UnpackLine_8BitYUVtoUWordSequence:":"UnpackLine_10BitYUVtoUWordSequence:") << endl << uwords << endl;

		UWordVANCPacketList	cPackets, yPackets;
		UWordSequence		cHOffsets, yHOffsets;
		CNTV2SMPTEAncData::GetAncPacketsFromVANCLine (uwords, isSD ? kNTV2SMPTEAncChannel_Both : kNTV2SMPTEAncChannel_C, cPackets, cHOffsets);
		if (!isSD)
			CNTV2SMPTEAncData::GetAncPacketsFromVANCLine (uwords, kNTV2SMPTEAncChannel_Y, yPackets, yHOffsets);
		AJAAncDataLoc loc	(AJAAncDataLink_Unknown, isSD ? AJAAncDataChannel_Both : AJAAncDataChannel_C,
							AJAAncDataSpace_VANC, uint16_t(smpteLine));
		for (UWordVANCPacketListConstIter it(cPackets.begin());  it != cPackets.end();  ++it, ndx++)
			compPkts.AddVANCData (*it, loc.SetHorizontalOffset(cHOffsets[ndx]));
		ndx = 0;
		loc.SetDataChannel(AJAAncDataChannel_Y);
		for (UWordVANCPacketListConstIter it(yPackets.begin());  it != yPackets.end();  ++it, ndx++)
			compPkts.AddVANCData (*it, loc.SetHorizontalOffset(yHOffsets[ndx]));
AJA_sDEBUG(AJA_DebugUnit_SMPTEAnc, "cPackets: " << cPackets);
AJA_sDEBUG(AJA_DebugUnit_SMPTEAnc, "yPackets: " << yPackets);
AJA_sDEBUG(AJA_DebugUnit_SMPTEAnc, "compPkts: " << compPkts);
		CHECK_EQ(compPkts.CountAncillaryDataWithID (TEST_DID, TEST_SID), 1);
		AJAAncillaryData *	pCompPkt	(compPkts.GetAncillaryDataWithID (TEST_DID, TEST_SID));
		CHECK(pCompPkt != AJA_NULL);
		CHECK_EQ(*pCompPkt, pkt);
	}

								//	Search for the packet using FindAnc...
								if (false)	//	** MrBill **	FOR NOW
								for (UWord srchLineOffset(0);  srchLineOffset < fd.GetFirstActiveLine();  srchLineOffset++)
								{
									for (unsigned CHSndx(0);  CHSndx < sizeof(CHSs)/sizeof(CHSs[0]);  CHSndx++)
									{
										const NTV2_SMPTEAncChannelSelect	srchChan(CHSs[CHSndx]);
										//UWord	wordBuffer[256];
										UWordSequence	wordBuffer;
										ULWord	numWordsCopied	(0);
										UWord	lineOffset		(srchLineOffset);
										UWord	pixelOffset		(0);
										ULWord	foundSmpteLine	(0);
										bool	foundIsF2		(false);
										bool	hasParityErrors(false);
										bool	bFound	(CNTV2SMPTEAncData::FindAnc (pkt.GetDID(),				//	inAncDID
																					pkt.GetSID(),				//	inAncSID
																					fb,							//	inFrameBuffer
																					fd,							//	inFormatDesc
																					srchChan,					//	inAncChannel
																					wordBuffer,					//	outWords
																					hasParityErrors,			//	outHasParityErrors
																					1,							//	inLineIncrement
																					lineOffset,					//	inOutLineStart
																					pixelOffset));				//	inOutPixelStart
/*										bool	bFound	(CNTV2SMPTEAncData::FindAnc (pkt.GetDID(),				//	inAncDID
																					pkt.GetSID(),				//	inAncSID
																					reinterpret_cast<const ULWord*>(fb.GetHostPointer()),	//	pInFrameBuffer
																					srchChan,					//	inAncChannel
																					vf,							//	inVideoFormat
																					fbf,						//	inFBFormat
																					wordBuffer,					//	pOutBuff
																					numWordsCopied,				//	outWordCount
																					ULWord(sizeof(wordBuffer)),	//	inWordCountMax
																					hasParityErrors,			//	outHasParityErrors
																					1,							//	inLineIncrement
																					lineOffset,					//	inOutLineStart
																					pixelOffset));				//	inOutPixelStart
*/										CHECK(fd.GetSMPTELineNumber (lineOffset, foundSmpteLine, foundIsF2));
										bool	isAMatch	= foundSmpteLine == smpteLine
																&&	pktLineOffset == lineOffset
																&&	numWordsCopied == pkt.GetDC();
AJA_sREPORT(AJA_DebugUnit_SMPTEAnc, AJA_DebugSeverity_Notice,	__FUNCTION__ << ":  " << (bFound?"FOUND":"NOT FOUND") << ": srchCh=" << sCHSs[srchChan] << " srchLn=" << srchLineOffset << " ln=" << lineOffset << " px=" << pixelOffset << " words=" << numWordsCopied);
cerr << __FUNCTION__ << ": " << (bFound?"FOUND":"NOT FOUND") << ": srchCh=" << sCHSs[srchChan] << " srchLn=" << srchLineOffset << " ln=" << lineOffset << " px=" << pixelOffset << " words=" << numWordsCopied << endl;
										//CHECK_EQ(bFound, isAMatch);
										if (bFound && isAMatch)
											cerr << "HOORAY!" << endl;
									}	//	vary the search channel select
									//break;	//	** MrBill **	FOR NOW
								}	//	vary the starting search line offset
								pkts.Clear();
								pkt.Clear();
								if (NTV2_IS_SD_VIDEO_FORMAT(vf))
									break;	//	For SD, only one channel: AJAAncDataChannel_Both == AJAAncDataChannel_C
							}	//	permute Y+C/Y/C channel
						}	//	permute line offset
					}	//	permute FBF
				}	//	permute VF
				///////////////////////////////////////////////////////////////////////	END  CNTV2SMPTEAncData::FindAnc  TEST
			}	//	if test FindAnc

			if (true)
			{
				//	Start over:
				v210VancLine.clear();	u16Pkts.clear();	u16s.clear();	pktList.Clear();	pPkt = AJA_NULL;

				//	Test CNTV2SMPTEAncData::GetAncPacketsFromVANCLine...
				UWordSequence	hOffsets;
				AJAAncDataLoc	loc	(AJAAncDataLink_A, AJAAncDataChannel_Both, AJAAncDataSpace_VANC, 9);
				::SetDefaultCaptionLogOutputStream(cerr);
				//::SetDefaultCaptionLogMask(kCaptionLog_SMPTEAncErrors | kCaptionLog_SMPTEAncSuccess | kCaptionLog_SMPTEAncDebug);
				CHECK(CNTV2SMPTEAncData::GetAncPacketsFromVANCLine (in10BitYUVReferenceLine, kNTV2SMPTEAncChannel_Both, u16Pkts, hOffsets));
				CHECK_EQ(u16Pkts.size(), 2);
				CHECK_EQ(u16Pkts.size(), hOffsets.size());
				const UWordSequence	u16s_1	(u16Pkts.at(0));
				const UWordSequence	u16s_2	(u16Pkts.at(1));
				CHECK_EQ(u16s_1.size(), 223);
				CHECK_EQ(u16s_2.size(), 9);
				//cerr << "BFT_SMPTEAncData:  PACKETS:  " << u16Pkts << endl
				//	 << "HOFFSETS:  " << hOffsets << endl
				//	 << "RESULTING PACKET LIST:" << endl;
				AJAAncillaryData::GetInstanceCounts(numConst, numDest);
				DBG_CHECK_EQ(numConst, 7);	DBG_CHECK_EQ(numDest, 7);
				for (UWordVANCPacketList::size_type ndx(0);  ndx < u16Pkts.size();  ndx++)	//	+2 more pkt instances ==> 11
					CHECK(AJA_SUCCESS(pktList.AddVANCData(u16Pkts.at(ndx), loc.SetHorizontalOffset(hOffsets.at(ndx)))));	//	Add to packet list
				//pktList.Print(cerr, true) << endl;
				AJAAncillaryData::GetInstanceCounts(numConst, numDest);
				DBG_CHECK_EQ(numConst, 11);	DBG_CHECK_EQ(numDest, 9);
				pktList.Clear();
				AJAAncillaryData::GetInstanceCounts(numConst, numDest);
				DBG_CHECK_EQ(numDest, 11);
			}
		}	//	TEST_CASE("BFT_SMPTEAncData")
#endif // AJANTV2_PROPRIETARY

#if 0	//	** MrBill **	NOT READY FOR PRIME-TIME
		TEST_CASE("BFT_SDSetFromVANCData")
		{
			const vector<uint16_t> & in10BitYUVReferenceLine(SD10BitYUVComponents);
			//	Test SetFromVANCData
			NTV2FormatDescriptor	fd	(NTV2_STANDARD_525, NTV2_FBF_10BIT_YCBCR, NTV2_VANCMODE_TALLER);
			NTV2Buffer				fb (size_t(fd.GetTotalRasterBytes()));	//	Tall SD frame buffer, in which a single VANC line will receive the packed YUV component data

			cerr << "WILL ITERATE OVER " << fd.GetFirstActiveLine() << " VANC LINES AND " << in10BitYUVReferenceLine.size() << " YUV COMPONENTS" << endl;
			//	For each VANC line...
			for (UWord fbRowOffset(0);  fbRowOffset < fd.GetFirstActiveLine();  fbRowOffset++)
			{
				//	This test repeatedly resets the frame buffer and packs an ever-shrinking number of YUV components
				//	into one of its tall VANC lines (fbRowOffset)... then AJAAncillaryList::SetFromVANCData is called
				//	to parse as many AJAAncillaryData packets from the VANC area as possible... then each packet's
				//	data and metadata are validated.
				cerr << endl << endl << "VANC LINE OFFSET " << fbRowOffset << endl;
				vector<uint16_t>	YUVLine	(in10BitYUVReferenceLine);
				while (YUVLine.size() > 11)
				{
					ULWord				smpteLineNum	(0);
					bool				isField2		(false);
					AJAAncillaryList	pktList;	//	Receives decoded Anc packets
					CHECK(fd.GetSMPTELineNumber (fbRowOffset, smpteLineNum, isField2));
					fb.Fill(UByte(0x80));			//	Reset the frame buffer contents to 0x80808080....
					CHECK(::YUVComponentsTo10BitYUVPackedBuffer (YUVLine,  fb,  fd,  fbRowOffset));
					CHECK(AJA_SUCCESS(AJAAncillaryList::SetFromVANCData (fb, fd, pktList)));
					if (YUVLine.size() < 228)	//	1st packet disappears around here
					{
						if (pktList.CountAncillaryData() != 0)	cerr << "IN ITERATION YUVLine.size=" << YUVLine.size() << " IN fbRowOffset=" << fbRowOffset << "..." << endl;
						CHECK_EQ(pktList.CountAncillaryData(), 0);	//	Should have no packets
					}
					else
					{
						AJAAncillaryData * pPkt1	(NULL);
						if (pktList.CountAncillaryData() != ((YUVLine.size() < 240) ? 1 : 2))	cerr << "IN ITERATION YUVLine.size=" << YUVLine.size() << " IN fbRowOffset=" << fbRowOffset << "..." << endl;
						CHECK_EQ(pktList.CountAncillaryData(),  YUVLine.size() < 240  ?  1  :  2);
						pPkt1 = pktList.GetAncillaryDataWithID (0x45, 0x01);
						CHECK(pPkt1 != AJA_NULL);	//	Should have packet DID=0x41/SID=0x01
						CHECK_EQ(pPkt1->GetDID(), 0x45);	//	Should have DID=0x41
						CHECK_EQ(pPkt1->GetSID(), 0x01);	//	Should have SID=0x01
						CHECK_EQ(pPkt1->GetDC(), 216);	//	Should have DC=216
						CHECK_EQ(pPkt1->GetChecksum(), 0xD2);	//	Should have CS=0xD2
						CHECK_EQ(pPkt1->GetDataCoding(), AJAAncDataCoding_Digital);		//	Should be Digital
						CHECK_EQ(pPkt1->GetLocationVideoLink(), AJAAncDataLink_Unknown);	//	Should be unknown Link
						CHECK_EQ(pPkt1->GetLocationDataStream(), AJAAncDataStream_1);		//	Should be DS1
						CHECK_EQ(pPkt1->GetLocationDataChannel(), AJAAncDataChannel_Both);	//	Should be C/both
						CHECK_EQ(pPkt1->GetLocationVideoSpace(), AJAAncDataSpace_VANC);	//	Should be VANC
						CHECK_EQ(pPkt1->GetLocationLineNumber(), uint16_t(smpteLineNum));		//	Should match smpteLineNum
						CHECK_EQ(pPkt1->GetLocationHorizOffset(), 0);							//	Should be zero
						if (YUVLine.size() >= 240)	//	2nd packet appears around here
						{
							AJAAncillaryData *	pPkt2		(NULL);
							const UWord			pkt2HOffset	(pPkt1->GetDC() + 7);	//	Pkt2 HOffset should be 7 components more than Pkt1's DC
							if (pktList.CountAncillaryData() != 2)	cerr << "IN ITERATION YUVLine.size=" << YUVLine.size() << " IN fbRowOffset=" << fbRowOffset << "..." << endl;
							CHECK_EQ(pktList.CountAncillaryData(), 2);	//	Should have two packets
							pPkt2 = pktList.GetAncillaryDataWithID (0x45, 0x02);
							CHECK(pPkt2 != AJA_NULL);	//	Should have packet DID=0x41/SID=0x02
							CHECK_EQ(pPkt2->GetDID(), 0x45);	//	Should have DID=0x41
							CHECK_EQ(pPkt2->GetSID(), 0x02);	//	Should have SID=0x02
							CHECK_EQ(pPkt2->GetDC(), 2);	//	Should have DC=2
							CHECK_EQ(pPkt2->GetChecksum(), 0x62);	//	Should have CS=0x62
							CHECK_EQ(pPkt2->GetDataCoding(), AJAAncDataCoding_Digital);		//	Should be Digital
							CHECK_EQ(pPkt2->GetLocationVideoLink(), AJAAncDataLink_Unknown);	//	Should be unknown Link
							CHECK_EQ(pPkt2->GetLocationDataStream(), AJAAncDataStream_1);		//	Should be DS1
							CHECK_EQ(pPkt2->GetLocationDataChannel(), AJAAncDataChannel_Both);	//	Should be C/both
							CHECK_EQ(pPkt2->GetLocationVideoSpace(), AJAAncDataSpace_VANC);	//	Should be VANC
							CHECK_EQ(pPkt2->GetLocationLineNumber(), uint16_t(smpteLineNum));		//	Should match smpteLineNum
							if (pPkt2->GetLocationHorizOffset() != pkt2HOffset)	cerr << "IN ITERATION YUVLine.size=" << YUVLine.size() << " IN fbRowOffset=" << fbRowOffset << "..." << endl;
							CHECK_EQ(pPkt2->GetLocationHorizOffset(), pkt2HOffset);					//	Should match hOffset
						}
					}	//	else YUVLine.size() >= 228
					//cerr << "LINE OFFSET " << fbRowOffset << " (YUVLine " << YUVLine.size() << "):" << endl << pktList << endl;
					YUVLine.pop_back();	//	Lop off last component value
				}	//	while YUVLine has at least 12 components
			}	//	for each VANC line
		}	//	TEST_CASE("BFT_SDSetFromVANCData")
#endif	//	DISABLE FOR NOW

		TEST_CASE("RTPHeaderBFT")
		{
			AJARTPAncPayloadHeader	hdrA, hdrB;
			CHECK(hdrA.IsValid());
			CHECK(hdrA == hdrB);
			CHECK_EQ(hdrA, hdrB);
			//	Set stuff the hardware ordinarily sets...
			hdrB.SetCCBits(0x5).SetPayloadType(0xF5).SetSequenceNumber(0x12345678).SetTimeStamp(0x6789ABCD).SetSyncSourceID(0xBaadF00d);
			//	Set stuff the client needs to set...
			hdrB.SetPayloadLength(0x4321).SetAncPacketCount(20).SetField2();
			CHECK(hdrB.IsValid());
			CHECK_NE(hdrA, hdrB);
			CHECK_FALSE(hdrB.IsField1());
			CHECK(hdrB.IsField2());
			CHECK_FALSE(hdrB.IsProgressive());
			CHECK_EQ(hdrB.GetPayloadLength(), 0x4321);
			CHECK_EQ(hdrB.GetAncPacketCount(), 20);
			CHECK_EQ(hdrB.GetSyncSourceID(), 0xBaadF00d);
			CHECK_EQ(hdrB.GetPayloadType(), 0x75);
			NTV2Buffer	nullBuffer;
			CHECK_FALSE(hdrB.WriteToBuffer(nullBuffer));

			//	HdrB => buffer => HdrC	. . .	verify HdrB == HdrC
			AJARTPAncPayloadHeader	hdrC, hdrD, hdrE;
			NTV2Buffer	bBuffer(4096);
			CHECK(hdrB.WriteToBuffer(bBuffer));
			if (gIsVerbose)
				cerr << "hdrB: " << hdrB << endl << bBuffer.AsString(20) << endl;
			CHECK(hdrC.ReadFromBuffer(bBuffer));
			if (gIsVerbose)
				cerr << "hdrC: " << hdrC << endl;

			//	bufferB => HdrD => bufferC . . .	verify bufferA == bufferB
			NTV2Buffer	cBuffer(bBuffer.GetByteCount());
			CHECK(hdrD.ReadFromBuffer(bBuffer));
			CHECK(hdrD.WriteToBuffer(cBuffer));
			CHECK(cBuffer.IsContentEqual(bBuffer));

			//	HdrB => u32vectorA => HdrE	. . .	verify HdrB == HdrE
			vector<uint32_t>	u32vectorA, u32vectorTooSmall;
			for (unsigned ndx(0);  ndx < 20;  ndx++)	u32vectorA.push_back(0xFFFFFFFF);
			CHECK_EQ(u32vectorA.size(), 20);	//	Size is 20
			vector<uint32_t>	u32vectorB (u32vectorA);
			CHECK_EQ(u32vectorB.size(), 20);	//	Size is 20
			CHECK(hdrB.WriteToULWordVector(u32vectorA, false));
			CHECK_EQ(u32vectorA.size(), 20);	//	Size is unchanged -- still 20
			CHECK(hdrB.WriteToULWordVector(u32vectorB, true));
			CHECK_EQ(u32vectorB.size(), 5);	//	Size is changed to 5 due to "reset" param being 'true'
			CHECK_FALSE(hdrE.ReadFromULWordVector(u32vectorTooSmall));
			//CHECK_FALSE(hdrE.SetFromPacketHeaderULWordAtIndex(5, 0xFFFFFFFF));
			CHECK(hdrE.ReadFromULWordVector(u32vectorA));
			CHECK_EQ(hdrB, hdrE);
		}	//	TEST_CASE("RTPHeaderBFT")


		/*
			ROUND-TRIP BFTS:

			AncList 'a' to buffer to AncList 'b':
				AJAAncillaryList 'a'	=>	GUMP				=>	AJAAncillaryList 'b'		Verify 'a' == 'b'					BFT_AncListToGumpToAncList
				AJAAncillaryList 'a'	=>	RTP					=>	AJAAncillaryList 'b'		Verify 'a' == 'b'					BFT_AncListToRTPToAncList
				AJAAncillaryList 'a'	=>	FB-VANC-8bitYC		=>	AJAAncillaryList 'b'		Verify 'a' == 'b'					BFT_AncListToFBYUV8ToAncList	TBD
				AJAAncillaryList 'a'	=>	FB-VANC-10bitYC		=>	AJAAncillaryList 'b'		Verify 'a' == 'b'					BFT_AncListToFBYUV10ToAncList

			Buffer 'a' to AncList to buffer 'b':
				GUMP 'a'				=>	AJAAncillaryList	=>	GUMP 'b'					Verify buffer contents 'a' == 'b'	BFT_GumpToAncListToGump
				RTP 'a'					=>	AJAAncillaryList	=>	RTP 'b'						Verify buffer contents 'a' == 'b'	BFT_RTPToAncListToRTP
				FB-VANC-8bitYC 'a'		=>	AJAAncillaryList	=>	FB-VANC-8bitYC 'b'			Verify buffer contents 'a' == 'b'	BFT_FBYUV8ToAncListToFBYUV8		TBD
				FB-VANC-10bitYC 'a'		=>	AJAAncillaryList	=>	FB-VANC-10bitYC 'b'			Verify buffer contents 'a' == 'b'	BFT_FBYUV10ToAncListToFBYUV10
		*/
		TEST_CASE("BFT_AncListToGumpToAncList")
		{
			AJAAncillaryData::ResetInstanceCounts();
			const NTV2VideoFormat	vFormats[]	=	{NTV2_FORMAT_525_5994, NTV2_FORMAT_625_5000, NTV2_FORMAT_720p_5994, NTV2_FORMAT_1080i_5994, NTV2_FORMAT_1080p_3000};
			for (unsigned ndx(0);  ndx < sizeof(vFormats)/sizeof(NTV2VideoFormat);  ndx++)
			{
				const NTV2VideoFormat		vFormat	(vFormats[ndx]);
				const NTV2FormatDescriptor	fd		(vFormat, NTV2_FBF_10BIT_YCBCR, NTV2_VANCMODE_OFF);
				ULWord						smpteLineF1(0), smpteLineF2(0);
				bool						isF2	(false);
				if (gIsVerbose)	cerr << "Trying " << ::NTV2VideoFormatToString(vFormat) << endl;
				CHECK(fd.GetSMPTELineNumber(0, smpteLineF1, isF2));
				if (isF2)
					smpteLineF2 = smpteLineF1;
				if (!NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat)  &&  !isF2)
					CHECK(fd.GetSMPTELineNumber(1, smpteLineF2, isF2));

				//	Make the transmit packets...
				AJAAncDataLoc	loc;
				loc.SetDataLink(AJAAncDataLink_A).SetHorizontalOffset(AJAAncDataHorizOffset_AnyVanc);
				AJAAncillaryData_Cea608_Vanc	pkt608F1;
				CHECK(AJA_SUCCESS(pkt608F1.SetLine(false/*isF1*/, 9)));
				CHECK(AJA_SUCCESS(pkt608F1.SetCEA608Bytes(AJAAncillaryData_Cea608::AddOddParity('A'), AJAAncillaryData_Cea608::AddOddParity('B'))));
				CHECK(AJA_SUCCESS(pkt608F1.SetDataLocation(loc.SetDataChannel(AJAAncDataChannel_Y).SetLineNumber(pkt608F1.IsField2() ? smpteLineF2+9 : 9))));
				CHECK(AJA_SUCCESS(pkt608F1.GeneratePayloadData()));
	
				AJAAncillaryData_Cea608_Vanc	pkt608F2;
				CHECK(AJA_SUCCESS(pkt608F2.SetLine(true/*isF2*/, 9)));
				CHECK(AJA_SUCCESS(pkt608F2.SetCEA608Bytes(AJAAncillaryData_Cea608::AddOddParity('a'), AJAAncillaryData_Cea608::AddOddParity('b'))));
				CHECK(AJA_SUCCESS(pkt608F2.SetDataLocation(loc.SetLineNumber(pkt608F2.IsField2() ? smpteLineF2+9 : 9))));
				CHECK(AJA_SUCCESS(pkt608F2.GeneratePayloadData()));
	
				AJAAncillaryData_HDR_HLG		pktHDR;
				CHECK(AJA_SUCCESS(pktHDR.GeneratePayloadData()));
	
				AJAAncillaryData				pktCustomY;
				CHECK(AJA_SUCCESS(pktCustomY.SetDataLocation(loc.SetLineNumber(10))));
				CHECK(AJA_SUCCESS(pktCustomY.SetDataCoding(AJAAncDataCoding_Digital)));
				CHECK(AJA_SUCCESS(pktCustomY.SetDID(0x7A)));
				CHECK(AJA_SUCCESS(pktCustomY.SetSID(0x01)));
				static const uint8_t	pCustomDataY[]	=	{	0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x09, 0x0A	};
				CHECK(AJA_SUCCESS(pktCustomY.SetPayloadData(pCustomDataY, sizeof(pCustomDataY))));
	
				AJAAncillaryData				pktCustomC;
				CHECK(AJA_SUCCESS(pktCustomC.SetDataLocation(loc.SetDataChannel(AJAAncDataChannel_C).SetLineNumber(11))));
				CHECK(AJA_SUCCESS(pktCustomC.SetDataCoding(AJAAncDataCoding_Digital)));
				CHECK(AJA_SUCCESS(pktCustomC.SetDID(0x8A)));
				CHECK(AJA_SUCCESS(pktCustomC.SetSID(0x02)));
				static const uint8_t	pCustomDataC[]	=	{	0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F	};
				CHECK(AJA_SUCCESS(pktCustomC.SetPayloadData(pCustomDataC, sizeof(pCustomDataC))));
	
				//	Make the transmit packet list...
				AJAAncillaryList	txPkts;
				CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pkt608F1)));
				if (!NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat))
					CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pkt608F2)));
				CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pktHDR)));
				CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pktCustomY)));
				CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pktCustomC)));
				CHECK(AJA_SUCCESS(txPkts.SortListByLocation()));
				//cerr << "Tx: " << txPkts << endl;

				//	Transmit the packets into the 8-bit GUMP buffer...
				NTV2Buffer	gumpF1(4096), gumpF2(4096);
				CHECK(AJA_SUCCESS(txPkts.GetTransmitData (gumpF1, gumpF2, NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat), smpteLineF2)));
				if (gIsVerbose)	cerr << "GUMP F1: " << gumpF1.AsString(64) << endl << "GUMP F2: " << gumpF2.AsString(64) << endl;

				//	NOTE:	This test saves the F1 GUMP buffers for use later by BFT_GumpToAncListToGump...
				gGumpBuffers[vFormat] = NTV2Buffer(gumpF1);

				//	Receive packets from the 8-bit GUMP buffer...
				AJAAncillaryList	rxPkts;
				CHECK(AJA_SUCCESS(AJAAncillaryList::SetFromDeviceAncBuffers(gumpF1, gumpF2, rxPkts)));
				//cerr << "Rx: " << rxPkts << endl;
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 5 + txPkts.CountAncillaryData() * 2);

				//	Compare the Tx and Rx packet lists.
				//	The txPkts all have zero checksums because we never set them.
				//	The rxPkts all have valid checksums because txPkts.GetTransmitData generates them,
				//	and SetFromDeviceAncBuffers uses the checksums when it creates packets.
				//	Therefore, when comparing the two lists, we must ignore checksums...
				CHECK(AJA_SUCCESS(txPkts.Compare(rxPkts, false/*ignoreLocation*/, true/*ignoreChecksum*/)));
cerr << ::NTV2VideoFormatToString(vFormat) << ": " << AJAAncillaryData::GetNumActiveInstances() << " pkts alive" << endl;
				rxPkts.Clear();
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 5 + txPkts.CountAncillaryData());
				txPkts.Clear();
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 5);
			}	//	for each video format
			DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 0);
//			cerr << "BFT_AncListToGumpToAncList passed -- C" << AJAAncillaryData::GetNumConstructed() << " D" << AJAAncillaryData::GetNumDestructed() << endl;
		}	//	TEST_CASE("BFT_AncListToGumpToAncList")

		TEST_CASE("BFT_GumpToAncListToGump")
		{
			//	NOTE:	This test relies on GUMP buffers generated by BFT_AncListToGumpToAncList
			AJAAncillaryData::ResetInstanceCounts();
			for (NTV2VideoFormat vFormat(NTV2_FORMAT_UNKNOWN);  vFormat < NTV2_MAX_NUM_VIDEO_FORMATS;  vFormat = NTV2VideoFormat(vFormat+1))
			{
				if (gGumpBuffers[vFormat].IsNULL())
					{LOGMYWARN("BFT_GumpToAncListToGump: Missing GUMP buffer for '" << ::NTV2VideoFormatToString(vFormat) << "' -- skipped"); continue;}

				const NTV2FormatDescriptor	fd		(vFormat, NTV2_FBF_10BIT_YCBCR, NTV2_VANCMODE_OFF);
				ULWord						smpteLineF1(0), smpteLineF2(0);
				bool						isF2	(false);
				if (gIsVerbose)	cerr << "Trying " << ::NTV2VideoFormatToString(vFormat) << endl;
				CHECK(fd.GetSMPTELineNumber(0, smpteLineF1, isF2));
				if (isF2)
					smpteLineF2 = smpteLineF1;
				if (!NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat)  &&  !isF2)
					CHECK(fd.GetSMPTELineNumber(1, smpteLineF2, isF2));

				{	//	Receive packets from the 8-bit GUMP buffer...
					AJAAncillaryList	pkts;
					CHECK(AJA_SUCCESS(AJAAncillaryList::SetFromDeviceAncBuffers(gGumpBuffers[vFormat], NTV2Buffer(), pkts)));
					CHECK_EQ(pkts.CountAncillaryData(), 4);
					if (gIsVerbose)
						cerr	<< "Received buffer: " << gGumpBuffers[vFormat].AsString(64) << " has " << DEC(pkts.CountAncillaryData()) << " packet(s)"
								//  << ": " << pkts
								<< endl;

					//	Transmit the packets into another 8-bit GUMP buffer...
					NTV2Buffer	gumpF1(4096), gumpF2;
					CHECK(AJA_SUCCESS(pkts.GetTransmitData (gumpF1, gumpF2, NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat), smpteLineF2)));
					if (gIsVerbose)	cerr << "        GUMP F1: " << gumpF1.AsString(64) << endl;
					CHECK(gGumpBuffers[vFormat].IsContentEqual(gumpF1));
					DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 4);
				}
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 0);
			}	//	for each video format
//			cerr << "BFT_GumpToAncListToGump passed -- C" << AJAAncillaryData::GetNumConstructed() << " D" << AJAAncillaryData::GetNumDestructed() << endl;
		}	//	TEST_CASE("BFT_GumpToAncListToGump")


		TEST_CASE("BFT_AncListToSortToAncList")
		{
			AJAAncillaryData::ResetInstanceCounts();
			{
				const NTV2VideoFormat		vFormat	(NTV2_FORMAT_1080i_5994);
				const NTV2FormatDescriptor	fd		(vFormat, NTV2_FBF_10BIT_YCBCR);
				ULWord						smpteLineF1(0), smpteLineF2(0);
				bool						isF2	(false);
				CHECK(fd.GetSMPTELineNumber(0, smpteLineF1, isF2));
				if (isF2)
					smpteLineF2 = smpteLineF1;
				if (!NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat)  &&  !isF2)
					CHECK(fd.GetSMPTELineNumber(1, smpteLineF2, isF2));

				//	Make the transmit packets...
				AJAAncDataLoc loc;	loc.SetDataLink(AJAAncDataLink_A).SetHorizontalOffset(AJAAncDataHorizOffset_AnyVanc);
				AJAAncillaryData_Cea608_Vanc	pkt608F1;
				CHECK(AJA_SUCCESS(pkt608F1.SetLine(false/*isF1*/, 9)));
				CHECK(AJA_SUCCESS(pkt608F1.SetCEA608Bytes(AJAAncillaryData_Cea608::AddOddParity('A'), AJAAncillaryData_Cea608::AddOddParity('B'))));
				CHECK(AJA_SUCCESS(pkt608F1.SetDataLocation(loc.SetDataChannel(AJAAncDataChannel_Y).SetLineNumber(pkt608F1.IsField2() ? smpteLineF2+9 : 9))));
				CHECK(AJA_SUCCESS(pkt608F1.GeneratePayloadData()));

				AJAAncillaryData_Cea608_Vanc	pkt608F2;
				CHECK(AJA_SUCCESS(pkt608F2.SetLine(true/*isF2*/, 9)));
				CHECK(AJA_SUCCESS(pkt608F2.SetCEA608Bytes(AJAAncillaryData_Cea608::AddOddParity('a'), AJAAncillaryData_Cea608::AddOddParity('b'))));
				CHECK(AJA_SUCCESS(pkt608F2.SetDataLocation(loc.SetLineNumber(pkt608F2.IsField2() ? smpteLineF2+9 : 9))));
				CHECK(AJA_SUCCESS(pkt608F2.GeneratePayloadData()));

				AJAAncillaryData_HDR_HLG		pktHDR;
				CHECK(AJA_SUCCESS(pktHDR.GeneratePayloadData()));

				AJAAncillaryData				pktCustomY;
				CHECK(AJA_SUCCESS(pktCustomY.SetDataLocation(loc.SetLineNumber(10))));
				CHECK(AJA_SUCCESS(pktCustomY.SetDataCoding(AJAAncDataCoding_Digital)));
				CHECK(AJA_SUCCESS(pktCustomY.SetDID(0x7A)));
				CHECK(AJA_SUCCESS(pktCustomY.SetSID(0x01)));
				static const uint8_t	pCustomDataY[]	=	{	0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x09, 0x0A	};
				CHECK(AJA_SUCCESS(pktCustomY.SetPayloadData(pCustomDataY, sizeof(pCustomDataY))));

				AJAAncillaryData				pktCustomC;		//	==> 5 active
				CHECK(AJA_SUCCESS(pktCustomC.SetDataLocation(loc.SetDataChannel(AJAAncDataChannel_C).SetLineNumber(11))));
				CHECK(AJA_SUCCESS(pktCustomC.SetDataCoding(AJAAncDataCoding_Digital)));
				CHECK(AJA_SUCCESS(pktCustomC.SetDID(0x8A)));
				CHECK(AJA_SUCCESS(pktCustomC.SetSID(0x02)));
				static const uint8_t	pCustomDataC[]	=	{	0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F	};
				CHECK(AJA_SUCCESS(pktCustomC.SetPayloadData(pCustomDataC, sizeof(pCustomDataC))));
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 5);

				//	Make the transmit packet list...
				AJAAncillaryList	origPkts;
				CHECK(AJA_SUCCESS(origPkts.AddAncillaryData(pktCustomC)));
				CHECK(AJA_SUCCESS(origPkts.AddAncillaryData(pktCustomY)));
				CHECK(AJA_SUCCESS(origPkts.AddAncillaryData(pktHDR)));
				if (!NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat))
					CHECK(AJA_SUCCESS(origPkts.AddAncillaryData(pkt608F2)));
				CHECK(AJA_SUCCESS(origPkts.AddAncillaryData(pkt608F1)));
				LOGMYDEBUG("BFT_AncListToSortToAncList: ORIG: " << origPkts);
				CHECK_EQ(origPkts.CountAncillaryData(), 5);
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 10);	//	==> 10 active

				//	Sort...
				AJAAncillaryList	sortedPkts(origPkts);	//	==> 15 active
				CHECK_EQ(sortedPkts.CountAncillaryData(), 5);
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 15);
				CHECK(AJA_SUCCESS(sortedPkts.SortListByLocation()));
				LOGMYDEBUG("BFT_AncListToSortToAncList: SORTED: " << sortedPkts);
				origPkts.Clear();
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 10);
				sortedPkts.Clear();
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 5);	//	pktCustomC, pktCustomY, pktHDR, pkt608F2, pkt608F1 still alive
			}	//	for each video format
			DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 0);
//			cerr << "BFT_AncListToSortToAncList passed -- C" << AJAAncillaryData::GetNumConstructed() << " D" << AJAAncillaryData::GetNumDestructed() << endl;
		}	//	TEST_CASE("BFT_AncListToSortToAncList")


		TEST_CASE("BFT_AncListToRTPToAncList")
		{
			AJAAncillaryData::ResetInstanceCounts();
			LOGMYNOTE("BFT_AncListToRTPToAncList: Starting");
			for (unsigned isSingleRTPPkt(0);  isSingleRTPPkt < 2;  isSingleRTPPkt++)
			{
				LOGMYDEBUG("BFT_AncListToRTPToAncList: " << (isSingleRTPPkt ? "TEST SINGLE RTP PKT CONTAINING MULTIPLE ANC PKTS" : "TEST MULTI RTP PKT (ONE RTP PKT PER ANC PKT)"));
				const NTV2VideoFormat	vFormats[]	=	{NTV2_FORMAT_525_5994, NTV2_FORMAT_625_5000, NTV2_FORMAT_720p_5994, NTV2_FORMAT_1080i_5994, NTV2_FORMAT_1080p_3000};
				for (unsigned ndx(0);  ndx < sizeof(vFormats)/sizeof(NTV2VideoFormat);  ndx++)
				{
					const NTV2VideoFormat		vFormat	(vFormats[ndx]);
					const NTV2FormatDescriptor	fd		(vFormat, NTV2_FBF_10BIT_YCBCR, NTV2_VANCMODE_OFF);
					ULWord						smpteLineF1(0), smpteLineF2(0);
					bool						isF2	(false);
//if (vFormat != NTV2_FORMAT_1080p_3000)	continue;	//	TEST 1080p ONLY??
					LOGMYDEBUG("BFT_AncListToRTPToAncList: Trying " << (isSingleRTPPkt?"SingleRTP ":"MultiRTP ") << fd);
					if (gIsVerbose)	cerr << "Trying " << fd << endl;
					CHECK(fd.GetSMPTELineNumber(0, smpteLineF1, isF2));
					if (isF2)
						smpteLineF2 = smpteLineF1;
					if (!NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat)  &&  !isF2)
						CHECK(fd.GetSMPTELineNumber(1, smpteLineF2, isF2));

					//	Make the transmit packets...
					AJAAncillaryList	txPkts;
					txPkts.SetAllowMultiRTPTransmit(!isSingleRTPPkt);
					DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 0);
					if (true)
					{
						AJAAncDataLoc loc;
						loc.SetDataLink(AJAAncDataLink_A).SetDataChannel(AJAAncDataChannel_Y).SetHorizontalOffset(AJAAncDataHorizOffset_AnyVanc);
						AJAAncillaryData_Cea608_Vanc	pkt608F1;
						CHECK(AJA_SUCCESS(pkt608F1.SetLine(false/*isF1*/, 9)));
						CHECK(AJA_SUCCESS(pkt608F1.SetCEA608Bytes(AJAAncillaryData_Cea608::AddOddParity('A'), AJAAncillaryData_Cea608::AddOddParity('B'))));
						CHECK(AJA_SUCCESS(pkt608F1.SetDataLocation(loc.SetLineNumber(pkt608F1.IsField2() ? smpteLineF2+9 : 9))));
						CHECK(AJA_SUCCESS(pkt608F1.GeneratePayloadData()));
	
						AJAAncillaryData_Cea608_Vanc	pkt608F2;
						CHECK(AJA_SUCCESS(pkt608F2.SetLine(true/*isF2*/, 9)));
						CHECK(AJA_SUCCESS(pkt608F2.SetCEA608Bytes(AJAAncillaryData_Cea608::AddOddParity('a'), AJAAncillaryData_Cea608::AddOddParity('b'))));
						CHECK(AJA_SUCCESS(pkt608F2.SetDataLocation(loc.SetLineNumber(pkt608F2.IsField2() ? smpteLineF2+9 : 9))));
						CHECK(AJA_SUCCESS(pkt608F2.GeneratePayloadData()));

						AJAAncillaryData_HDR_HLG		pktHDR;
						CHECK(AJA_SUCCESS(pktHDR.GeneratePayloadData()));

						AJAAncillaryData_Timecode_ATC	pktATC;
						{
							const NTV2_RP188				regTC (0, 0, 0);
							AJATimeCode						tc;
							NTV2FrameRate					ntv2Rate	(::GetNTV2FrameRateFromVideoFormat(vFormat));
							const AJA_FrameRate				ajaRate		(sNTV2Rate2AJARate[ntv2Rate]);
							const AJATimeBase				ajaTB		(ajaRate);
							tc.SetRP188 (regTC.fDBB, regTC.fLo, regTC.fHi, ajaTB);
							pktATC.SetTimecode (tc, ajaTB, ajaTB.IsNonIntegralRatio());
							pktATC.SetDBB1PayloadType(AJAAncillaryData_Timecode_ATC_DBB1PayloadType_VITC1);
							//pktATC.SetDBB1PayloadType(AJAAncillaryData_Timecode_ATC_DBB1PayloadType_LTC);
							pktATC.GeneratePayloadData();
						}
	
						AJAAncillaryData				pktCustomY;
						CHECK(AJA_SUCCESS(pktCustomY.SetDataLocation(loc.SetLineNumber(10))));
						CHECK(AJA_SUCCESS(pktCustomY.SetDataCoding(AJAAncDataCoding_Digital)));
						CHECK(AJA_SUCCESS(pktCustomY.SetDID(0x7A)));
						CHECK(AJA_SUCCESS(pktCustomY.SetSID(0x01)));
						static const uint8_t	pCustomDataY[]	=	{	0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20	};
						CHECK(AJA_SUCCESS(pktCustomY.SetPayloadData(pCustomDataY, sizeof(pCustomDataY))));
	
						AJAAncillaryData				pktCustomC;
						CHECK(AJA_SUCCESS(pktCustomC.SetDataLocation(loc.SetDataChannel(AJAAncDataChannel_C).SetLineNumber(11))));
						CHECK(AJA_SUCCESS(pktCustomC.SetDataCoding(AJAAncDataCoding_Digital)));
						CHECK(AJA_SUCCESS(pktCustomC.SetDID(0x8A)));
						CHECK(AJA_SUCCESS(pktCustomC.SetSID(0x02)));
						static const uint8_t	pCustomDataC[]	=	{	0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F	};
						CHECK(AJA_SUCCESS(pktCustomC.SetPayloadData(pCustomDataC, sizeof(pCustomDataC))));
	
						//	Make the transmit packet list...
						CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pkt608F1)));
						if (!NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat))
							CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pkt608F2)));
						CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pktHDR)));
						CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pktATC)));
						CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pktCustomY)));
						CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pktCustomC)));
						if (NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat))
							DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 1 + txPkts.CountAncillaryData() * 2);
						else
							DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), txPkts.CountAncillaryData() * 2);
					}
//gIsVerbose = true;
					DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), txPkts.CountAncillaryData());
					LOGMYDEBUG("BFT_AncListToRTPToAncList: Tx: " << txPkts);
					if (gIsVerbose)	cerr << "Tx: " << txPkts << endl;

					//	Transmit the packets into the IP buffer...
					NTV2Buffer	IPF1(2048), IPF2(2048);
					uint32_t	IPF1bytes(0), IPF2bytes(0);
LOGMYDEBUG("BFT_AncListToRTPToAncList: Call GetIPTransmitDataLength:");
					CHECK(AJA_SUCCESS(txPkts.GetIPTransmitDataLength (IPF1bytes, IPF2bytes, NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat), smpteLineF2)));
LOGMYDEBUG("BFT_AncListToRTPToAncList: IPF1bytes=" << DEC(IPF1bytes) << ", IPF2bytes=" << DEC(IPF2bytes) << " -- Call GetIPTransmitData:");
					CHECK(AJA_SUCCESS(txPkts.GetIPTransmitData (IPF1, IPF2, NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat), smpteLineF2)));
//cerr << "POST-TRANSMIT:" << endl << "IP F1 (" << DEC(IPF1bytes) << " bytes):" << endl;	IPF1.Dump(cerr, 0, IPF1bytes, 16, 4, 16);
//if (!NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat))
//	{cerr << "IP F2 (" << DEC(IPF2bytes) << " bytes):" << endl;	IPF2.Dump(cerr, 0, IPF2bytes, 16, 4, 16);}

					//	NOTE:	This test saves the F1 RTP buffers for use later by BFT_RTPToAncListToRTP...
					gRTPBuffers[vFormat] = NTV2Buffer(IPF1);	//	Deep copy
//cerr << endl << "gRTPBuffers[" << ::NTV2VideoFormatToString(vFormat) << "]:" << endl << IPF1.GetU32s(/*u32offset*/0, /*u32count*/IPF1bytes/4, /*byteSwap*/true) << endl;
//if (!NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat))
//	LOGMYDEBUG("F2 RTP Buffer:" << IPF2.GetU32s(/*u32offset*/0, /*u32count*/10, /*byteSwap*/true));

					//	Receive packets from the IP buffer...
					AJAAncillaryList	rxPkts;
					rxPkts.SetAllowMultiRTPReceive(!isSingleRTPPkt);
					LOGMYDEBUG("BFT_AncListToRTPToAncList: Receive packets from IP buffers");
					CHECK(AJA_SUCCESS(AJAAncillaryList::SetFromDeviceAncBuffers(IPF1, IPF2, rxPkts)));
					LOGMYDEBUG("BFT_AncListToRTPToAncList: Rx: " << rxPkts);	if (gIsVerbose)	cerr << "Rx: " << rxPkts << endl;
					DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), txPkts.CountAncillaryData() * 2);

					//	Compare the Tx and Rx packet lists...
					LOGMYDEBUG("BFT_AncListToRTPToAncList: Compare Tx vs Rx packets for " << fd);
					const string	cmpInfo	(txPkts.CompareWithInfo(rxPkts, /*ignoreLocation*/false, /*ignoreChecksum*/true));
					if (!cmpInfo.empty())
						LOGMYWARN("BFT_AncListToRTPToAncList: " << cmpInfo);
					CHECK(cmpInfo.empty());
				}	//	for each video format
			}	//	for singleRTP, multiRTP
			LOGMYNOTE("BFT_AncListToRTPToAncList: Passed");
//			cerr << "BFT_AncListToRTPToAncList passed -- C" << AJAAncillaryData::GetNumConstructed() << " D" << AJAAncillaryData::GetNumDestructed() << endl;
			DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 0);
		}	//	TEST_CASE("BFT_AncListToRTPToAncList")


		TEST_CASE("BFT_RTPToAncListToRTP")
		{
			AJAAncillaryData::ResetInstanceCounts();
			const NTV2VideoFormat	vFormats[]	=	{NTV2_FORMAT_525_5994, NTV2_FORMAT_625_5000, NTV2_FORMAT_720p_5994, NTV2_FORMAT_1080i_5994, NTV2_FORMAT_1080p_3000};
			for (unsigned ndx(0);  ndx < sizeof(vFormats)/sizeof(NTV2VideoFormat);  ndx++)
			{
				const NTV2VideoFormat		vFormat	(vFormats[ndx]);
				const NTV2FormatDescriptor	fd		(vFormat, NTV2_FBF_10BIT_YCBCR, NTV2_VANCMODE_OFF);
				ULWord						smpteLineF1(0), smpteLineF2(0);
				bool						isF2	(false);
				AJAAncillaryList			rxPkts;
				LOGMYNOTE("Trying " << fd);
				CHECK(fd.GetSMPTELineNumber(0, smpteLineF1, isF2));
				if (isF2)
					smpteLineF2 = smpteLineF1;
				if (!NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat)  &&  !isF2)
					CHECK(fd.GetSMPTELineNumber(1, smpteLineF2, isF2));

				//	NOTE:	Use the F1 RTP buffer we saved in BFT_AncListToRTPToAncList...
				if (gRTPBuffers[vFormat].IsNULL())	{LOGMYINFO("Skipping " << fd << " because " << gRTPBuffers[vFormat]);	continue;}
				const NTV2Buffer &	F1RTP_a	(gRTPBuffers[vFormat]);
				NTV2Buffer			F1RTP_b	(F1RTP_a.GetByteCount());
				NTV2Buffer			F2RTP_b	(F1RTP_a.GetByteCount());
				//	Unpack into an AJAAncillaryList of anc packets...
				CHECK(AJA_SUCCESS(AJAAncillaryList::SetFromDeviceAncBuffers(F1RTP_a, NTV2Buffer(), rxPkts)));
				CHECK_EQ(rxPkts.CountAncillaryData(), 5);	//	Should have 5 packets
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), rxPkts.CountAncillaryData());
				//	Re-pack into RTP...
				CHECK(AJA_SUCCESS(rxPkts.GetIPTransmitData (F1RTP_b, F2RTP_b, NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat), smpteLineF2)));
				//	Content of F1RTP "a" and "b" buffers should match...
				if (!F1RTP_a.IsContentEqual(F1RTP_b, 0))
				{
					LOGMYERROR("MIS-COMPARE: Rx RTP: " << F1RTP_a.GetU32s(0, 32, true));
					LOGMYERROR("MIS-COMPARE: CmpRTP: " << F1RTP_b.GetU32s(0, 32, true));
					AJAAncillaryList	b_pkts;
					AJAAncillaryList::SetFromDeviceAncBuffers(F1RTP_b, NTV2Buffer(), b_pkts);
					const string	info	(rxPkts.CompareWithInfo(b_pkts, false, false));
					if (!info.empty())
						LOGMYERROR("RxPkts != B-Pkts --- MISCOMPARE: " << info);
					LOGMYDEBUG("RxPkts: " << rxPkts);
					LOGMYDEBUG("B Pkts: " << b_pkts);
					const AJAAncillaryData *	rxA(rxPkts.GetAncillaryDataAtIndex(1));
					const AJAAncillaryData *	rxB(b_pkts.GetAncillaryDataAtIndex(1));
					if (*rxA != *rxB)
						LOGMYDEBUG(rxA->AsString() << " != " << rxB->AsString());
					else
						LOGMYDEBUG(rxA->AsString() << " == " << rxB->AsString());
					LOGMYDEBUG(rxA->CompareWithInfo(*rxB, false, false));
					CHECK_EQ(b_pkts.CountAncillaryData(), rxPkts.CountAncillaryData());
					DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), rxPkts.CountAncillaryData()*2);
				}
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), rxPkts.CountAncillaryData());
				rxPkts.Clear();
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 0);
				CHECK(F1RTP_a.IsContentEqual(F1RTP_b, 0));
			}	//	for each vFormat
//			cerr << "BFT_RTPToAncListToRTP passed -- C" << AJAAncillaryData::GetNumConstructed() << " D" << AJAAncillaryData::GetNumDestructed() << endl;
		}	//	TEST_CASE("BFT_RTPToAncListToRTP")


		TEST_CASE("BFT_RTPXmitTooManyPackets")
		{
			AJAAncillaryData::ResetInstanceCounts();
			//	"TOO MANY PACKETS" TEST
			//	Validates that GetIPTransmitData will correctly encode no more than 255 packets from a list that contains more.
			for (unsigned oneRTP(0);  oneRTP < 2;  oneRTP++)
			{
				const bool	isSingleRTPPacket	(oneRTP ? true : false);
				LOGMYNOTE("BFT_RTPXmitTooManyPackets: Starting test: " << (isSingleRTPPacket ? "SINGLE RTP PACKET" : "MULTIPLE RTP PACKETS"));

				//	Create a packet list having more than 255 packets...
				AJAAncillaryList	pkts;
				ULWord				pktNum(0);
				AJAAncDataLoc		loc;
				//	This test requires a 5K anc buffer for single-RTP, 10K for multiple-RTP:
				NTV2Buffer			F1Buffer((isSingleRTPPacket?5:10)*1024), F2Buffer;
				pkts.SetAllowMultiRTPTransmit(!isSingleRTPPacket);
				for (UWord lineNum(9);  lineNum < 42;  lineNum++)
				{
					for (UWord pktInLine(0);  pktInLine < 8;  pktInLine++)
					{
						AJAAncillaryData pkt;
						loc.Reset().SetDataLink(AJAAncDataLink_A)
									.SetDataChannel(AJAAncDataChannel_Y)
									.SetDataSpace(AJAAncDataSpace_VANC)
									.SetDataStream(AJAAncDataStream_1)
									.SetLineNumber(lineNum)
									.SetHorizontalOffset(AJAAncDataHorizOffset_Anywhere);
						pkt.SetDataLocation(loc);
						pkt.SetDataCoding(AJAAncDataCoding_Digital);
						pkt.SetBufferFormat(AJAAncBufferFormat_RTP);
						pkt.SetDID(0xCC);
						pkt.SetSID(0xDD);
						const ULWord pktNumBE (ENDIAN_32HtoN(pktNum++));
						pkt.SetPayloadData(reinterpret_cast<const UByte*>(&pktNumBE), sizeof(ULWord));
						pkts.AddAncillaryData(pkt);

						CHECK(AJA_SUCCESS(pkts.GetIPTransmitData (F1Buffer, F2Buffer, /*isProgressive=*/true, /*F2StartLine=*/0)));
					}	//	7 pkts per line
				}	//	for lineNum from 8 thru 41

				//	At this point, pkts should contain 264 packets, but F1Buffer will only have 255 (max for RTP).
				CHECK_EQ(pkts.CountAncillaryData(), 264);
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 264);
				{	//	Convert F1Buffer into a comparison packet list...
					AJAAncillaryList	cmpPkts;
					cmpPkts.SetAllowMultiRTPReceive(!isSingleRTPPacket);
					CHECK(AJA_SUCCESS(AJAAncillaryList::SetFromDeviceAncBuffers (F1Buffer, F2Buffer, cmpPkts)));
					CHECK_NE(pkts.CountAncillaryData(), cmpPkts.CountAncillaryData());
					CHECK_EQ(cmpPkts.CountAncillaryData(), 255);
					DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 255+264);

					//	Truncate the original pkts list until its count matches cmpPkts...
					while (pkts.CountAncillaryData() > cmpPkts.CountAncillaryData())
						pkts.DeleteAncillaryData(pkts.GetAncillaryDataAtIndex(pkts.CountAncillaryData()-1));
					DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 255+255);

					//	Now the two lists should match (ignoring checksums, since original list's packets all have zero checksums)...
					string	cmpResults (pkts.CompareWithInfo(cmpPkts, /*ignoreLocation=*/false));
					if (!cmpResults.empty())
						LOGMYNOTE(cmpResults);
					CHECK(cmpResults.empty());
				}
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 255);
				LOGMYNOTE("BFT_RTPXmitTooManyPackets: Passed test: " << (isSingleRTPPacket ? "SINGLE RTP PACKET" : "MULTIPLE RTP PACKETS"));
				pkts.Clear();
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 0);
			}	//	permute multiRTP & singleRTP
//			cerr << "BFT_RTPXmitTooManyPackets passed -- C" << AJAAncillaryData::GetNumConstructed() << " D" << AJAAncillaryData::GetNumDestructed() << endl;
		}	//	TEST_CASE("BFT_RTPXmitTooManyPackets")


		TEST_CASE("BFT_RTPXmitTooMuchData")
		{
			//	"TOO MUCH DATA" TEST
			//	Validates that GetIPTransmitData will correctly encode no more than 64K of packet data from a list that contains more.
			cerr << "BFT_RTPXmitTooMuchData: Starting test: SINGLE RTP PACKET" << endl;
			AJAAncillaryData::ResetInstanceCounts();

			//	Create a packet list having more than 255 packets...
			AJAAncillaryList	pkts;
			ULWord				pktNum(0);
			ULWord				totalPayloadBytes(0);
			AJAAncDataLoc		loc;
			for (UWord lineNum(9);  lineNum < 42;  lineNum++)
			{
				for (UWord pktInLine(0);  pktInLine < 8;  pktInLine++)
				{
					AJAAncillaryData pkt;
					loc.Reset().SetDataLink(AJAAncDataLink_A)
								.SetDataChannel(AJAAncDataChannel_Y)
								.SetDataSpace(AJAAncDataSpace_VANC)
								.SetDataStream(AJAAncDataStream_1)
								.SetLineNumber(lineNum)
								.SetHorizontalOffset(AJAAncDataHorizOffset_Anywhere);
					pkt.SetDataLocation(loc);
					pkt.SetDataCoding(AJAAncDataCoding_Digital);
					pkt.SetBufferFormat(AJAAncBufferFormat_RTP);
					pkt.SetDID(0xCC);
					pkt.SetSID(0xDD);
					totalPayloadBytes += 255;
					ostringstream	oss, oss2;  oss << "Packet " << DEC(++pktNum) << ", totalBytes=" << DEC(totalPayloadBytes);
					oss2 << setw(255) << left << oss.str();
					NTV2_ASSERT(oss2.str().length() == 255);
					pkt.SetPayloadData(reinterpret_cast<const UByte*>(oss2.str().c_str()), uint32_t(oss2.str().length()));
					pkts.AddAncillaryData(pkt);
				}	//	7 pkts per line
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), pkts.CountAncillaryData());
			}	//	for lineNum from 8 thru 41

			//	How big should our F1 anc buffer be?
			uint32_t	F1TotalBytes(0), F2TotalBytes(0);
			CHECK(AJA_SUCCESS(pkts.GetIPTransmitDataLength (F1TotalBytes, F2TotalBytes)));	//	Progressive, single RTP pkt
			LOGMYNOTE(DEC(F1TotalBytes) << " F1 Total Bytes Required, totalPayloadBytes=" << DEC(totalPayloadBytes));

			//	Allocate the F1 anc buffer:   (requires almost 100K!)
			NTV2Buffer	F1Buffer((F1TotalBytes / 1024 + 2) * 1024), F2Buffer;

			//	Fill it with RTP xmit data:
			CHECK(AJA_SUCCESS(pkts.GetIPTransmitData (F1Buffer, F2Buffer)));	//	Progressive, single RTP pkt

			//	At this point, pkts should contain ~264 packets, but F1Buffer will only have ~199 (max for RTP).
			CHECK_EQ(pkts.CountAncillaryData(), 264);
			DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 264);
			{	//	Produce a comparison packet list from the F1Buffer contents...
				AJAAncillaryList	cmpPkts;
				CHECK(AJA_SUCCESS(AJAAncillaryList::SetFromDeviceAncBuffers (F1Buffer, F2Buffer, cmpPkts)));
				CHECK_NE(pkts.CountAncillaryData(), cmpPkts.CountAncillaryData());
				CHECK_EQ(cmpPkts.CountAncillaryData(), 199);

				//	Truncate the original pkts list until its count matches cmpPkts...
				while (pkts.CountAncillaryData() > cmpPkts.CountAncillaryData())
					pkts.DeleteAncillaryData(pkts.GetAncillaryDataAtIndex(pkts.CountAncillaryData()-1));

				//	Now the two lists should match (ignoring checksums, since original list's packets all have zero checksums)...
				CHECK_EQ(pkts.CountAncillaryData(), cmpPkts.CountAncillaryData());
				string	cmpResults (pkts.CompareWithInfo(cmpPkts, /*ignoreLocation=*/false));
				if (!cmpResults.empty())
					LOGMYNOTE(cmpResults);
				CHECK(cmpResults.empty());
				DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), pkts.CountAncillaryData() + cmpPkts.CountAncillaryData());
			}
			DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), pkts.CountAncillaryData());
			pkts.Clear();
			DBG_CHECK_EQ(AJAAncillaryData::GetNumActiveInstances(), 0);
//			cerr << "BFT_RTPXmitTooMuchData: SINGLE RTP PACKET passed -- C" << AJAAncillaryData::GetNumConstructed() << " D" << AJAAncillaryData::GetNumDestructed() << endl;
		}	//	TEST_CASE("BFT_RTPXmitTooMuchData")


		TEST_CASE("BFT_AncListToFBYUV8ToAncList")
		{
			const NTV2VideoFormat	vFormats[]	=	{/*NTV2_FORMAT_525_5994, NTV2_FORMAT_625_5000,*/ NTV2_FORMAT_720p_5994, NTV2_FORMAT_1080i_5994, NTV2_FORMAT_1080p_3000};
			for (unsigned ndx(0);  ndx < sizeof(vFormats)/sizeof(NTV2VideoFormat);  ndx++)
			{
				const NTV2VideoFormat		vFormat	(vFormats[ndx]);
				const NTV2FormatDescriptor	fd		(vFormat, NTV2_FBF_8BIT_YCBCR, NTV2_VANCMODE_TALLER);
				ULWord						smpteLineF1(0), smpteLineF2(0);
				bool						isF2	(false);
				if (gIsVerbose)	cerr << "Trying " << ::NTV2VideoFormatToString(vFormat) << endl;
				CHECK(fd.GetSMPTELineNumber(0, smpteLineF1, isF2));
				if (isF2)
					smpteLineF2 = smpteLineF1;
				if (!NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat)  &&  !isF2)
					CHECK(fd.GetSMPTELineNumber(1, smpteLineF2, isF2));

				AJAAncillaryList	txPkts;
				if (true)
				{
					//	Make the transmit packets...
					AJAAncDataLoc	loc;
					loc.SetDataLink(AJAAncDataLink_A).SetHorizontalOffset(AJAAncDataHorizOffset_AnyVanc);
					AJAAncillaryData_Cea608_Vanc	pkt608F1;
					CHECK(AJA_SUCCESS(pkt608F1.SetLine(false/*isF1*/, 10)));
					CHECK(AJA_SUCCESS(pkt608F1.SetCEA608Bytes(AJAAncillaryData_Cea608::AddOddParity('A'), AJAAncillaryData_Cea608::AddOddParity('B'))));
					CHECK(AJA_SUCCESS(pkt608F1.SetDataLocation(loc.SetDataChannel(fd.IsSD() ? AJAAncDataChannel_Both : AJAAncDataChannel_Y).SetLineNumber(pkt608F1.IsField2() ? smpteLineF2+10 : 10))));
					CHECK(AJA_SUCCESS(pkt608F1.GeneratePayloadData()));

					AJAAncillaryData_Cea608_Vanc	pkt608F2;
					CHECK(AJA_SUCCESS(pkt608F2.SetLine(true/*isF2*/, 11)));
					CHECK(AJA_SUCCESS(pkt608F2.SetCEA608Bytes(AJAAncillaryData_Cea608::AddOddParity('a'), AJAAncillaryData_Cea608::AddOddParity('b'))));
					CHECK(AJA_SUCCESS(pkt608F2.SetDataLocation(loc.SetDataChannel(fd.IsSD() ? AJAAncDataChannel_Both : AJAAncDataChannel_Y).SetLineNumber(!NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat) && pkt608F2.IsField2() ? smpteLineF2+11 : 11))));
					CHECK(AJA_SUCCESS(pkt608F2.GeneratePayloadData()));
	
					AJAAncillaryData_HDR_HLG		pktHDR;
					CHECK(AJA_SUCCESS(pktHDR.GeneratePayloadData()));
	
					AJAAncillaryData				pktCustomY;
					CHECK(AJA_SUCCESS(pktCustomY.SetDataLocation(loc.SetDataChannel(fd.IsSD() ? AJAAncDataChannel_Both : AJAAncDataChannel_Y).SetLineNumber(12))));
					CHECK(AJA_SUCCESS(pktCustomY.SetDataCoding(AJAAncDataCoding_Digital)));
					CHECK(AJA_SUCCESS(pktCustomY.SetDID(0x7A)));
					CHECK(AJA_SUCCESS(pktCustomY.SetSID(0x01)));
					static const uint8_t	pCustomDataY[]	=	{	0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x09, 0x0A	};
					CHECK(AJA_SUCCESS(pktCustomY.SetPayloadData(pCustomDataY, sizeof(pCustomDataY))));
	
					AJAAncillaryData				pktCustomC;
					CHECK(AJA_SUCCESS(pktCustomC.SetDataLocation(loc.SetDataChannel(fd.IsSD() ? AJAAncDataChannel_Both : AJAAncDataChannel_C).SetLineNumber(13))));
					CHECK(AJA_SUCCESS(pktCustomC.SetDataCoding(AJAAncDataCoding_Digital)));
					CHECK(AJA_SUCCESS(pktCustomC.SetDID(0x8A)));
					CHECK(AJA_SUCCESS(pktCustomC.SetSID(0x02)));
					static const uint8_t	pCustomDataC[]	=	{	0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F	};
					CHECK(AJA_SUCCESS(pktCustomC.SetPayloadData(pCustomDataC, sizeof(pCustomDataC))));
	
					//	Make the transmit packet list...
					CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pkt608F1)));
					if (!NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat))
						CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pkt608F2)));
					CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pktHDR)));
					CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pktCustomY)));
					CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pktCustomC)));
				}
				txPkts.SortListByLocation();	//	To match rxPkts order at Compare (below)
				//	Transmit the packets into the 8-bit YCbCr frame buffer...
				NTV2Buffer	vanc8(fd.GetTotalRasterBytes() - fd.GetVisibleRasterBytes());	//	Just the VANC lines
				vanc8.Fill(uint8_t(0x80));
				CHECK(AJA_SUCCESS(txPkts.GetVANCTransmitData (vanc8, fd)));
/* cerr << ::NTV2VideoFormatToString(vFormat) << " " << ::NTV2FrameBufferFormatToString(fd.GetPixelFormat()) << " VANC: " << endl;
for (unsigned lineOffset(0);  lineOffset < fd.GetFirstActiveLine();  lineOffset++) {
	NTV2Buffer	blankLine(fd.GetBytesPerRow());
	NTV2Buffer	oneLine(fd.GetRowAddress(vanc8.GetHostPointer(), lineOffset), fd.GetBytesPerRow());
	blankLine.Fill(uint8_t(0x80));
	if (!oneLine.IsContentEqual(blankLine)) {
		cerr << "+" << DEC0N(lineOffset,2) << ": ";  fd.PrintSMPTELineNumber(cerr, lineOffset, false);  cerr << ":" << endl;
		CNTV2CaptionLogConfig::DumpMemory(oneLine.GetHostPointer(), 32*4, cerr, 16, 4, 32,0,false);
} } */

				//	NOTE:	This test saves the VANC buffer for use later by BFT_FBYUV8ToAncListToFBYUV8...
				gVanc8Buffers[vFormat] = NTV2Buffer(vanc8);

				//	Receive packets from the 8-bit GUMP buffer...
				AJAAncillaryList	rxPkts;
				CHECK(AJA_SUCCESS(AJAAncillaryList::SetFromVANCData(vanc8, fd, rxPkts)));
				rxPkts.SortListByLocation();	//	To match txPkts order -- for Compare below

				//	Compare the Tx and Rx packet lists...
				//	The txPkts have zero checksums because we never set them.
				//	The rxPkts all have valid checksums because rxPkts.SetFromVANCData sets them.
				//	Also, the rxPkts have unknown link values in their Locations.
				//	Therefore, when comparing the two lists, we must ignore location and checksums...
				AJAStatus	result	(txPkts.Compare(rxPkts,  true/*ignoreLocation?*/,  true/*ignoreChecksum?*/));
				if (AJA_FAILURE(result))
					cerr << "MISCOMPARE:" << endl << "Tx: " << txPkts << endl	<< "Rx: " << rxPkts << endl;
				CHECK(AJA_SUCCESS(result));
			}	//	for each video format
		}	//	TEST_CASE("BFT_AncListToFBYUV8ToAncList")


		TEST_CASE("BFT_FBYUV8ToAncListToFBYUV8")
		{
			CHECK(true);	//	TBD
		}	//	TEST_CASE("BFT_FBYUV8ToAncListToFBYUV8")


		TEST_CASE("BFT_AncListToFBYUV10ToAncList")
		{
			const NTV2VideoFormat	vFormats[]	=	{NTV2_FORMAT_525_5994, /*NTV2_FORMAT_625_5000,*/ NTV2_FORMAT_720p_5994, NTV2_FORMAT_1080i_5994, NTV2_FORMAT_1080p_3000};
			for (unsigned ndx(0);  ndx < sizeof(vFormats)/sizeof(NTV2VideoFormat);  ndx++)
			{
				const NTV2VideoFormat		vFormat	(vFormats[ndx]);
				const NTV2FormatDescriptor	fd		(vFormat, NTV2_FBF_10BIT_YCBCR, NTV2_VANCMODE_TALLER);
				ULWord						smpteLineF1(0), smpteLineF2(0);
				bool						isF2	(false);
				if (gIsVerbose)	cerr << "Trying " << ::NTV2VideoFormatToString(vFormat) << endl;
				CHECK(fd.GetSMPTELineNumber(0, smpteLineF1, isF2));
				if (isF2)
					smpteLineF2 = smpteLineF1;
				if (!NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat)  &&  !isF2)
					CHECK(fd.GetSMPTELineNumber(1, smpteLineF2, isF2));

				AJAAncillaryList txPkts;
				if (true)
				{
					//	Make the transmit packets...
					AJAAncDataLoc	loc;
					loc.SetDataLink(AJAAncDataLink_A).SetHorizontalOffset(AJAAncDataHorizOffset_AnyVanc);
					AJAAncillaryData_Cea608_Vanc	pkt608F1;
					CHECK(AJA_SUCCESS(pkt608F1.SetLine(false/*isF1*/, 10)));
					CHECK(AJA_SUCCESS(pkt608F1.SetCEA608Bytes(AJAAncillaryData_Cea608::AddOddParity('A'), AJAAncillaryData_Cea608::AddOddParity('B'))));
					CHECK(AJA_SUCCESS(pkt608F1.SetDataLocation(loc.SetDataChannel(fd.IsSD() ? AJAAncDataChannel_Both : AJAAncDataChannel_Y).SetLineNumber(pkt608F1.IsField2() ? smpteLineF2+10 : 10))));
					CHECK(AJA_SUCCESS(pkt608F1.GeneratePayloadData()));

					AJAAncillaryData_Cea608_Vanc	pkt608F2;
					CHECK(AJA_SUCCESS(pkt608F2.SetLine(true/*isF2*/, 11)));
					CHECK(AJA_SUCCESS(pkt608F2.SetCEA608Bytes(AJAAncillaryData_Cea608::AddOddParity('a'), AJAAncillaryData_Cea608::AddOddParity('b'))));
					CHECK(AJA_SUCCESS(pkt608F2.SetDataLocation(loc.SetDataChannel(fd.IsSD() ? AJAAncDataChannel_Both : AJAAncDataChannel_Y).SetLineNumber(!NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat) && pkt608F2.IsField2() ? smpteLineF2+11 : 11))));
					CHECK(AJA_SUCCESS(pkt608F2.GeneratePayloadData()));
	
					AJAAncillaryData_HDR_HLG		pktHDR;
					CHECK(AJA_SUCCESS(pktHDR.GeneratePayloadData()));
	
					AJAAncillaryData				pktCustomY;
					CHECK(AJA_SUCCESS(pktCustomY.SetDataLocation(loc.SetDataChannel(fd.IsSD() ? AJAAncDataChannel_Both : AJAAncDataChannel_Y).SetLineNumber(12))));
					CHECK(AJA_SUCCESS(pktCustomY.SetDataCoding(AJAAncDataCoding_Digital)));
					CHECK(AJA_SUCCESS(pktCustomY.SetDID(0x7A)));
					CHECK(AJA_SUCCESS(pktCustomY.SetSID(0x01)));
					static const uint8_t	pCustomDataY[]	=	{	0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x09, 0x0A	};
					CHECK(AJA_SUCCESS(pktCustomY.SetPayloadData(pCustomDataY, sizeof(pCustomDataY))));
	
					AJAAncillaryData				pktCustomC;
					CHECK(AJA_SUCCESS(pktCustomC.SetDataLocation(loc.SetDataChannel(fd.IsSD() ? AJAAncDataChannel_Both : AJAAncDataChannel_C).SetLineNumber(13))));
					CHECK(AJA_SUCCESS(pktCustomC.SetDataCoding(AJAAncDataCoding_Digital)));
					CHECK(AJA_SUCCESS(pktCustomC.SetDID(0x8A)));
					CHECK(AJA_SUCCESS(pktCustomC.SetSID(0x02)));
					static const uint8_t	pCustomDataC[]	=	{	0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F	};
					CHECK(AJA_SUCCESS(pktCustomC.SetPayloadData(pCustomDataC, sizeof(pCustomDataC))));
	
					//	Make the transmit packet list...
					CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pkt608F1)));
					if (!NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat))
						CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pkt608F2)));
					CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pktHDR)));
					CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pktCustomY)));
					CHECK(AJA_SUCCESS(txPkts.AddAncillaryData(pktCustomC)));
				}

				//	Transmit the packets into the 10-bit YCbCr frame buffer...
				NTV2Buffer	vanc10(fd.GetTotalRasterBytes() - fd.GetVisibleRasterBytes());	//	Just the VANC lines
				vanc10.Fill(uint8_t(0x80));
				CHECK(AJA_SUCCESS(txPkts.GetVANCTransmitData (vanc10, fd)));
/* cerr << ::NTV2VideoFormatToString(vFormat) << " " << ::NTV2FrameBufferFormatToString(fd.GetPixelFormat()) << " VANC: " << endl;
for (unsigned lineOffset(0);  lineOffset < fd.GetFirstActiveLine();  lineOffset++) {
	NTV2Buffer	blankLine(fd.GetBytesPerRow());
	NTV2Buffer	oneLine(fd.GetRowAddress(vanc10.GetHostPointer(), lineOffset), fd.GetBytesPerRow());
	blankLine.Fill(uint8_t(0x80));
	if (!oneLine.IsContentEqual(blankLine)) {
		cerr << "+" << DEC0N(lineOffset,2) << ": ";  fd.PrintSMPTELineNumber(cerr, lineOffset, false);  cerr << ":" << endl;
		CNTV2CaptionLogConfig::DumpMemory(oneLine.GetHostPointer(), 32*4, cerr, 16, 4, 32,0,false);
} } */

				//	NOTE:	This test saves the VANC buffer for use later by BFT_FBYUV10ToAncListToFBYUV10...
				gVanc10Buffers[vFormat] = NTV2Buffer(vanc10);

				//	Receive packets from the 8-bit GUMP buffer...
				AJAAncillaryList	rxPkts;
				CHECK(AJA_SUCCESS(AJAAncillaryList::SetFromVANCData (vanc10, fd, rxPkts)));
				rxPkts.SortListByLocation();	//	To match txPkts order for Compare (below)

				//	Compare the Tx and Rx packet lists...
				AJAStatus	result	(txPkts.Compare(rxPkts,  true /*ignoreLocation*/,  true /*ignoreChecksum*/));
				if (AJA_FAILURE(result))
					cerr << "MISCOMPARE:" << endl << "Tx: " << txPkts << endl	<< "Rx: " << rxPkts << endl;
				CHECK(AJA_SUCCESS(result));
			}	//	for each video format
		}	//	TEST_CASE("BFT_AncListToFBYUV10ToAncList")


		TEST_CASE("BFT_FBYUV10ToAncListToFBYUV10")
		{
			//	NOTE:	This test relies on YUV10 buffers generated by BFT_AncListToFBYUV10ToAncList
			for (NTV2VideoFormat vFormat(NTV2_FORMAT_UNKNOWN);  vFormat < NTV2_MAX_NUM_VIDEO_FORMATS;  vFormat = NTV2VideoFormat(vFormat+1))
			{
				if (gVanc10Buffers[vFormat].IsNULL())
					continue;

				const NTV2FormatDescriptor	fd		(vFormat, NTV2_FBF_10BIT_YCBCR, NTV2_VANCMODE_TALLER);
				ULWord						smpteLineF1(0), smpteLineF2(0);
				bool						isF2	(false);
				CHECK(fd.GetSMPTELineNumber(0, smpteLineF1, isF2));
				if (isF2)
					smpteLineF2 = smpteLineF1;
				if (!NTV2_VIDEO_FORMAT_HAS_PROGRESSIVE_PICTURE(vFormat)  &&  !isF2)
					CHECK(fd.GetSMPTELineNumber(1, smpteLineF2, isF2));


				//	Receive packets from the YUV10 frame buffer's VANC lines...
				AJAAncillaryList	pkts;
				CHECK(AJA_SUCCESS(AJAAncillaryList::SetFromVANCData(gVanc10Buffers[vFormat], fd, pkts)));
				if (gIsVerbose)
					cerr	<< ::NTV2VideoFormatToString(vFormat) << ": " << "Rx: " << pkts << endl;

				//	Transmit the packets into another YUV10 frame buffer...
				NTV2Buffer	yuv10vanc(fd.GetTotalRasterBytes() - fd.GetVisibleRasterBytes());	//	Just the VANC lines
				yuv10vanc.Fill(uint8_t(0x80));
				CHECK(AJA_SUCCESS(pkts.GetVANCTransmitData(yuv10vanc, fd)));
				CHECK(gVanc10Buffers[vFormat].IsContentEqual(yuv10vanc));
			}	//	for each video format
		}	//	TEST_CASE("BFT_FBYUV10ToAncListToFBYUV10")


		TEST_CASE("BFT_AddFromDeviceAncBuffer")
		{	//	This test is intended to elicit crashes (access violations), not to validate outcomes
			AJAAncillaryList pkts;
			NTV2Buffer nullBuffer;
			AJARTPAncPayloadHeader rtpHdr;
			CHECK(AJA_SUCCESS(AJAAncillaryList::SetFromDeviceAncBuffers(nullBuffer, nullBuffer, pkts)));
			//	Exercise RTP (variable-length runs of zeroes)
			for (ULWord sz(20);  sz < 65;  sz++)
			{
				NTV2Buffer buff(sz*4);
				CHECK_EQ(buff.GetByteCount(), sz*4);
				CHECK(rtpHdr.WriteToBuffer(buff, /*offset*/sz/4%8));
				CHECK(AJA_SUCCESS(AJAAncillaryList::SetFromDeviceAncBuffers(buff, nullBuffer, pkts)));
				CHECK_EQ(pkts.CountAncillaryData(), 0);
			}
			//	Exercise RTP (variable-length runs of random data)
			for (ULWord sz(15);  sz < 65;  sz++)
			{
				NTV2Buffer buff(sz*4);
				CHECK_EQ(buff.GetByteCount(), sz*4);
				std::random_device rd;	//	Seeds the random number engine
				std::mt19937 gen(rd());	//	Standard mersenne_twister_engine seeded with rd()
				std::uniform_int_distribution<> distrib(0, 255);
				for (ULWord ndx(0);  ndx < sz;  ++ndx)
					CHECK(buff.PutU8s(UByteSequence{UByte(distrib(gen))}, ndx));
				CHECK(rtpHdr.WriteToBuffer(buff, /*offset*/sz/4%8));
				if (AJA_FAILURE(AJAAncillaryList::SetFromDeviceAncBuffers(buff, nullBuffer, pkts)))
					LOGMYINFO("whoa, got AJAAncillaryList::SetFromDeviceAncBuffers to fail!");
				if (pkts.CountAncillaryData())
					LOGMYINFO("whoa, got one or more packets! " << pkts);
			}
			//	Exercise GUMP (variable-length runs of random data)
			for (ULWord sz(8);  sz < 200;  sz++)
			{
				NTV2Buffer buff(sz);
				CHECK_EQ(buff.GetByteCount(), sz);
				std::random_device rd;	//	Seeds the random number engine
				std::mt19937 gen(rd());	//	Standard mersenne_twister_engine seeded with rd()
				std::uniform_int_distribution<> distrib(0, 255);
				for (ULWord ndx(0);  ndx < sz;  ++ndx)
					CHECK(buff.PutU8s(UByteSequence{UByte(distrib(gen))}, ndx));
				CHECK(buff.PutU8s(UByteSequence{0xFF}, 0));
				if (AJAAncillaryList::SetFromDeviceAncBuffers(buff, nullBuffer, pkts))
					LOGMYINFO("whoa, a successful parse! " << pkts);
				if (pkts.CountAncillaryData())
					LOGMYINFO(pkts);
			}
		}	//	TEST_CASE("BFT_AddFromDeviceAncBuffer")


		TEST_CASE("BFT_AncDataCEA608Vanc")
		{
			static const uint8_t pGump608Vanc[]	= {	0xFF,	0xA0,	0x09,	0x61,	0x02,	0x03,
													0x09,	0xC1,	0xC2,	0xF2,	//	end of packet
													//	Extra bytes
													0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,
													0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10};
			uint32_t						packetByteCount	(0);
			AJAAncillaryData_Cea608_Vanc	pktRX, pktTX;
			AJAAncillaryData				defaultPkt;

			//	Test AJAAncillaryData_Cea608_Vanc  and GUMP decoding of it
			CHECK(AJA_SUCCESS(defaultPkt.InitWithReceivedData (pGump608Vanc, sizeof(pGump608Vanc), AJAAncDataLoc(), packetByteCount)));
			CHECK_EQ (defaultPkt.GetDC(), 3);
			CHECK_EQ (defaultPkt.GetDID(), 0x61);
			CHECK_EQ (defaultPkt.GetSID(), 0x02);
			CHECK_EQ (defaultPkt.GetChecksum(), 0xF2);
			CHECK_EQ (defaultPkt.GetLocationVideoLink(), AJAAncDataLink_Unknown);
			CHECK_EQ (defaultPkt.GetLocationDataChannel(), AJAAncDataChannel_Y);
			CHECK_EQ (defaultPkt.GetLocationVideoSpace(), AJAAncDataSpace_VANC);
			CHECK_EQ (defaultPkt.GetLocationLineNumber(), 9);
			CHECK_EQ (defaultPkt.GetLocationHorizOffset(), AJAAncDataHorizOffset_AnyVanc);
			CHECK_EQ (defaultPkt.GetDataCoding(), AJAAncDataCoding_Digital);
			CHECK_FALSE (defaultPkt.GotValidReceiveData());		//	False, because wasn't vetted by specific subclass
			CHECK_EQ (defaultPkt.GetAncillaryDataType(), AJAAncDataType_Unknown);
			CHECK_NE (defaultPkt.GetAncillaryDataType(), AJAAncDataType_Cea608_Vanc);
			CHECK_EQ (AJAAncillaryData_Cea608_Vanc::RecognizeThisAncillaryData(&defaultPkt), AJAAncDataType_Cea608_Vanc);
			CHECK_EQ (AJAAncillaryData_Cea608_Line21::RecognizeThisAncillaryData(&defaultPkt), AJAAncDataType_Unknown);
			CHECK_EQ (AJAAncillaryData_Cea708::RecognizeThisAncillaryData(&defaultPkt), AJAAncDataType_Unknown);
			CHECK(AJA_SUCCESS(pktRX.InitWithReceivedData (pGump608Vanc, sizeof(pGump608Vanc), AJAAncDataLoc(), packetByteCount)));
			CHECK(AJA_SUCCESS(pktRX.ParsePayloadData()));
			CHECK (pktRX.GotValidReceiveData());
			uint8_t	fieldNumber(0), lineNumber(0), char1(0), char2(0);	bool isValid(false);
			CHECK(AJA_SUCCESS(pktRX.GetLine(fieldNumber, lineNumber)));
			CHECK_EQ (fieldNumber, 1);	//	F2 == NTV2_FIELD1
			CHECK (pktRX.IsField2());	//	F2
			CHECK_EQ (lineNumber, 9);			//	SMPTE line 9
			CHECK_EQ (pktRX.GetLineNumber(), 9);	//	SMPTE line 9
			CHECK(AJA_SUCCESS(pktRX.GetCEA608Characters(char1, char2, isValid)));
			CHECK_EQ (char1, 'A');
			CHECK_EQ (char2, 'B');
			CHECK (isValid);
		}	//	TEST_CASE("BFT_AncDataCEA608Vanc")

		TEST_CASE("SDYUV8TallerVANCDecode")
		{
			//	525i5994  720x514  SD  YUV-8  28-line "taller" VANC Region   1440 bytes/line   40,320 bytes
			static const UByteSequence s525iYUV8TallerVANCLineNdx678 =
			{	//	3 of the 28 "taller" VANC lines:  line offsets 6, 7, and 8:
		/* 6   F2 L272*/0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
/*608	   7   F1 L10 */0x00,0xFF,0xFF,0x61,0x02,0x03,0x8C,0x80,0x80,0xF2,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
/*608	   8   F2 L273*/0x00,0xFF,0xFF,0x61,0x02,0x03,0x8C,0xB3,0x20,0xC5,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
						0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,0x80,0x10,
			};	//	s525iYUV8TallerVANCLineNdx678
			const NTV2Buffer bs525iYUV8TallerVANCLineNdx678(&s525iYUV8TallerVANCLineNdx678[0], s525iYUV8TallerVANCLineNdx678.size());

			const NTV2FormatDesc fd(NTV2_STANDARD_525, NTV2_FBF_8BIT_YCBCR, NTV2_VANCMODE_TALLER);
			//	Sanity checks...
			CHECK_EQ(fd.GetBytesPerRow(), 1440);
			CHECK_EQ(fd.GetRasterWidth(), 720);
			CHECK_EQ(fd.GetFullRasterHeight() - fd.GetVisibleRasterHeight(), 28);

			//	Explicitly test AJAAncillaryData::Unpack8BitYCbCrToU16sVANCLineSD
			UWordSequence u16Line;
			NTV2Buffer lineWith608Pkt;
			CHECK(fd.GetRowBuffer (bs525iYUV8TallerVANCLineNdx678, lineWith608Pkt, 1));
			CHECK(AJAAncillaryData::Unpack8BitYCbCrToU16sVANCLineSD (lineWith608Pkt, u16Line, fd.GetRasterWidth()));
			CHECK_EQ(u16Line.at(0), 0x000);	//	0x00
			CHECK_EQ(u16Line.at(1), 0x3FF);	//	0xFF
			CHECK_EQ(u16Line.at(2), 0x3FF);	//	0xFF
			CHECK_EQ(u16Line.at(3), 0x161);	//	0x61 DID
			CHECK_EQ(u16Line.at(4), 0x102);	//	0x02 SID
			CHECK_EQ(u16Line.at(5), 0x203);	//	0x03 DC
			CHECK_EQ(u16Line.at(6), 0x18C);	//	0x8C	dw[0]
			CHECK_EQ(u16Line.at(7), 0x180);	//	0x80	dw[1]
			CHECK_EQ(u16Line.at(8), 0x180);	//	0x80	dw[2]
			CHECK_EQ(u16Line.at(9), 0x2F2);	//	0xF2 CS

			//	Now build a full "taller" SD raster buffer
			NTV2Buffer FullRaster(fd.GetTotalBytes());
			//	Fill in FullRaster's VANC region
			//		- only line offsets 7 and 8 have a CEA608 packet (using lines 1 and 2 in s525iYUV8TallerVANCLineNdx678)
			//		- all others lines are copied from line 0 in s525iYUV8TallerVANCLineNdx678
			for (ULWord lineNdx(0);  lineNdx < fd.GetFirstActiveLine();  lineNdx++)
			{
				NTV2Buffer srcRowBuf, destRowBuf;
				CHECK(fd.GetRowBuffer (bs525iYUV8TallerVANCLineNdx678, srcRowBuf, lineNdx == 7 ? 1 : (lineNdx == 8 ? 2 : 0)));
				CHECK(fd.GetRowBuffer (FullRaster, destRowBuf,  lineNdx));
				CHECK(destRowBuf.SetFrom(srcRowBuf));
			}
			//	Now reference FullRaster's VANC region:
			NTV2Buffer FullRasterVANCPortion;
			FullRaster.Segment (FullRasterVANCPortion, 0, fd.GetTotalBytes() - fd.GetVisibleRasterBytes());

			//	Parse the VANC region pixel data for anc packets...
			AJAAncillaryList pkts;
			CHECK_EQ(FullRasterVANCPortion.GetByteCount(), fd.GetTotalBytes() - fd.GetVisibleRasterBytes());
			CHECK(AJA_SUCCESS(AJAAncillaryList::SetFromVANCData (FullRasterVANCPortion, fd, pkts, 234)));
			CHECK_EQ(pkts.CountAncillaryData(), 2);	//	There should be two CEA608 packets

//			cout << pkts << endl;
//0:  [Dig|A|DS1|C|L10|VANC|x61x02|CSF2|DC=3|FRx000000EA|FBVANC|SMPTE-334 SD CEA-608 CC]: 8C8080
			AJAAncillaryData & pkt = *(pkts.GetAncillaryDataAtIndex(0));
				CHECK_EQ(pkt.GetDataCoding(), AJAAncDataCoding_Digital);
				CHECK_EQ(pkt.GetLocationVideoLink(), AJAAncDataLink_A);
				CHECK_EQ(pkt.GetLocationDataStream(), AJAAncDataStream_1);
				CHECK_EQ(pkt.GetLocationDataChannel(), AJAAncDataChannel_C);
				CHECK_EQ(pkt.GetLocationLineNumber(), 10);
				CHECK_EQ(pkt.GetLocationVideoSpace(), AJAAncDataSpace_VANC);
				CHECK_EQ(pkt.GetDID(), 0x61);
				CHECK_EQ(pkt.GetSID(), 0x02);
				CHECK_EQ(pkt.GetChecksum(), 0xF2);
				CHECK_EQ(pkt.GetPayloadByteCount(), 3);
				CHECK_EQ(pkt.GetBufferFormat(), AJAAncBufferFormat_FBVANC);
				CHECK_EQ(pkt.GetFrameID(), 234);
				CHECK_EQ(pkt.GetAncillaryDataType(), AJAAncDataType_Cea608_Vanc);
				CHECK_NE(pkt.AsString(6).find("8C8080"), string::npos);
//2: [Dig|A|DS1|C|L273|VANC|x61x02|CSC5|DC=3|FRx000000EA|FBVANC|SMPTE-334 SD CEA-608 CC]: 8CB320
			pkt = *(pkts.GetAncillaryDataAtIndex(1));
				CHECK_EQ(pkt.GetDataCoding(), AJAAncDataCoding_Digital);
				CHECK_EQ(pkt.GetLocationVideoLink(), AJAAncDataLink_A);
				CHECK_EQ(pkt.GetLocationDataStream(), AJAAncDataStream_1);
				CHECK_EQ(pkt.GetLocationDataChannel(), AJAAncDataChannel_C);
				CHECK_EQ(pkt.GetLocationLineNumber(), 273);
				CHECK_EQ(pkt.GetLocationVideoSpace(), AJAAncDataSpace_VANC);
				CHECK_EQ(pkt.GetDID(), 0x61);
				CHECK_EQ(pkt.GetSID(), 0x02);
				CHECK_EQ(pkt.GetChecksum(), 0xC5);
				CHECK_EQ(pkt.GetPayloadByteCount(), 3);
				CHECK_EQ(pkt.GetBufferFormat(), AJAAncBufferFormat_FBVANC);
				CHECK_EQ(pkt.GetFrameID(), 234);
				CHECK_EQ(pkt.GetAncillaryDataType(), AJAAncDataType_Cea608_Vanc);
				CHECK_NE(pkt.AsString(6).find("8CB320"), string::npos);
		}	//	TEST_CASE("SDYUV8TallerVANCDecode")

		//	This test case reproduces a long-standing bug in which we'd see 3 "analog" packets of size 255, 255, 210
		//	in the Field2 anc buffer, instead of just one "analog" packet of size 720, as is expected.
		//	The anc buffer content here originated in a Corvid88 that was receiving SD from an Io4K+:
		//		$ ./path/to/ntv2ccplayer -d2 -m -vsd -c3 --no608 --stats --end loop,loop --rate 1200,1200 --608mode roll4,roll4 --608chan cc1,cc3  cc1.txt cc3.txt
		TEST_CASE("AnalogTest")
		{
			static const UByteSequence sCORVID880Ch1Input720x4868BITYCBCRFrm3F1 = {
			0xFF,0xB0,0x09,0xF4,0x00,0x10,0x70,0xC4,0x90,0x30,0xB4,0xA4,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x50,0xFF,0xB0,0x0B,0x60,0x60,0x10,0x30,0x00,0x40,0x00,0x40,0x00,0x10,0x30,0x20,0x00,0x50,0x00,0x00,0x00,0x00,0x00,0x30,0xFF,0xB0,0x0D,0x41,0x01,0x04,0x81,0x06,0x00,0x01,0xCE,0xFF,0xE0,0x15,0x00,0x00,0xFF,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,0x1D,0x16,0x11,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,
			0x1D,0x16,0x11,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,0x1D,0x16,0x11,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,0x1D,0x16,0x11,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,
			0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,0x1D,0x16,0x11,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,0x1D,0x16,0x11,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,
			0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,0x1D,0x16,0x11,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x24,0x47,0x6A,0x7E,0x7E,0x7E,0x7E,0x00,0xFF,
			0xE0,0x15,0x00,0x00,0xFF,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x6A,0x47,0x24,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x24,0x47,0x6A,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x6A,0x47,0x24,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x00,0xFF,0xE0,0x15,0x00,0x00,0xD2,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x24,0x47,0x6A,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,
			0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x6A,0x47,0x24,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x00,0x00,0x00,0x57,0x38,0xFA,0x3D,0xAF,0x7F,0xF9,0xFF,0xDE,0x9E,0x8E,0xF3,0xB6,0x2A,0xAC,0xE8,0xEB,0xE9,0x6F,0xEB,0xEF,0x9A,0xB1,0xBB,0xFF,0xC3,0xB2,0xC7,0xFE,0x3C,0x43,0xF8,
			0x3F,0xF5,0x9D,0xBF,0x7F,0xEF,0x2F,0xEB,0xBE,0xFE,0xF8,0xBB,0x3A,0xF8,0xFE,0x7F,0xFD,0xC9,0xF3,0x43,0xDE,0xFB,0xFB,0x2E,0xFF,0xFD,0xEE,0x2B,0xDE,0xFF,0x5C,0xBC,0xFF,0xDF,0xBA,0xDF,0xFD,0xF5,0xE5,0xE9,0x9B,0xFE,0xA6,0xBD,0xE2,0xF7,0xF6,0xAF,0xEF,0xD4,0xEC,0xDF,0x26,0xC2,0x6F,0xF3,0x9F,0x3F,0x4B,0xB0,0x6F,0xAC,0xF3,0x66,
			0x37,0xFA,0x77,0x8C,0x25,0x1D,0x26,0xE6,0x39,0x3C,0x1B,0x3C,0x3F,0x29,0x8B,0xFF,0x2E,0xB6,0x6F,0xDD,0x56,0x22,0xFE,0xCB,0xD7,0x6F,0x6E,0x7C,0xFD,0xC4,0xD7,0x75,0xBF,0xFB,0xFC,0x4F,0xAE,0xCB,0xF9,0xC7,0x93,0xAF,0xF9,0x76,0x91,0xA7,0x6E,0x1E,0x75,0xFB,0xEE,0x77,0x68,0xEF,0x47,0xCF,0xDD,0x3B,0xA3,0x4D,0xEF,0x5F,0xBF,0x73,
			0xDF,0xBC,0xE7,0x7D,0x36,0x7A,0xCF,0x73,0x9D,0xCF,0x2E,0xE7,0x79,0x2D,0xAA,0x39,0xDF,0xB0,0xA6,0xFB,0x38,0xBF,0xBF,0xE0,0x3F,0xF9,0xB8,0xEB,0xDE,0x7F,0x9F,0x72,0xE2,0x3B,0x7D,0xEF,0xBB,0xF5,0xFE,0x7A,0x77,0x17,0xFE,0x59,0xFE,0x6B,0x4F,0xAD,0x89,0x79,0xED,0xFD,0x7E,0xEF,0xFF,0xFA,0xEE,0xF8,0x75,0xE7,0x3A,0x63,0xE7,0x7F,
			0xEB,0xFF,0x31,0x9A,0x22,0xCF,0xFF,0xED,0xAB,0x9F,0xF3,0xDD,0xEC,0x25,0xFB,0x1C,0xFB,0xFD,0xFF,0xEF,0x7B,0xAB,0xEF,0xEE,0x8B,0x4F,0xE5,0xDE,0xE0,0x65,0x6F,0x34,0x3B,0x93,0x5F,0xBF,0x6F,0x1B,0xE7,0xFD,0xA1,0xEF,0xFB,0xCD,0xF7,0xEE,0x9D,0xFB,0xBE,0x7E,0xDB,0x06,0x7F,0xF1,0xBE,0x0B,0xB0,0xDC,0xB7,0xC6,0x3B,0x21,0xAF,0xB3,
			0xE7,0x5F,0x3B,0xCD,0xFF,0xFB,0xB4,0x5F,0x9E,0xFB,0x56,0xFC,0xB2,0x7B,0x30,0xDC,0x6E,0x66,0xF5,0xE6,0x38,0x2F,0xDA,0xDA,0x3F,0xF3,0xFB,0x8C,0xFF,0xAD,0xDC,0xDF,0xF7,0x9F,0x4F,0xD6,0xDF,0x5E,0x55,0x1E,0x1F,0xF7,0xEF,0x4A,0xEF,0x0F,0xD9,0xF3,0xAB,0xB0,0x2F,0xFB,0xEB,0xA3,0x8D,0x96,0xA7,0xBF,0xBB,0xD7,0xD7,0x7C,0xF6,0xBB,
			0xEB,0x6F,0xE2,0xFB,0xD9,0x7E,0x47,0xCD,0x7F,0x8B,0x6D,0xEF,0xEB,0xCA,0xCB,0xF8,0x87,0xEF,0x6F,0xE9,0xD7,0x50,0x3C,0xA4,0xD6,0xDA,0xED,0xF7,0x73,0x3A,0xEA,0xEA,0xB3,0x7F,0xFD,0xED,0xBA,0xDE,0xFA,0x7F,0x13,0xFB,0xFF,0x5B,0xE7,0xDE,0x6C,0xFE,0x65,0x6B,0x95,0xFE,0xDF,0xF6,0xFC,0xCF,0xB5,0xFD,0x7F,0x87,0xA6,0x99,0xBC,0xDA,
			0x97,0xCE,0xAF,0x3A,0xF3,0xDE,0xC3,0x6A,0xAF,0xD2,0xA2,0xFF,0xEE,0xFB,0xE2,0xF7,0x9E,0xDE,0x7F,0x2B,0xB7,0xFC,0x6F,0x5B,0xFA,0xEF,0x9B,0xBA,0xE1,0xED,0xAF,0x36,0xFC,0xDF,0xA4,0xFE,0x76,0x88,0x38,0xFD,0x6B,0xA4,0xC6,0x5B,0xCF,0x36,0xF9,0xBF,0xD4,0xBD,0xFA,0xFF,0xFA,0xF9,0x67,0xBE,0xD2,0xDD,0xEA,0xFA,0x85,0x99,0xF1,0xA1,
			0x6F,0xF1,0xE6,0x77,0x63,0x6F,0xEF,0xBB,0xAD,0x8B,0xCC,0xB8,0xF5,0x7E,0x9D,0x1F,0x06,0xDD,0x71,0xA7,0xD5,0x35,0xDB,0xBB,0xF9,0xE9,0x9A,0xBF,0xD3,0x3F,0x05,0x2E,0x78,0x79,0xDE,0xF1,0x70,0xFE,0xCF,0xBF,0x5E,0xFD,0x03,0xD7,0xBE,0xDE,0xF4,0xAD,0x27,0x7E,0xEF,0xBF,0xBE,0xBE,0x77,0xFE,0x77,0x1F,0xFE,0x2C,0x93,0x4E,0xE7,0xBD,
			0x09,0xFF,0xE4,0xBF,0x3F,0xAF,0xA3,0xE8,0x37,0xBB,0x29,0xA9,0x9E,0xF9,0xD3,0xC3,0xDF,0xF7,0xF9,0xBF,0xBD,0xFF,0xEE,0xDF,0xAF,0xE3,0xAF,0xAB,0xFD,0x3F,0xA8,0x2D,0x7E,0xEF,0xAD,0x6C,0xDF,0xC6,0xFF,0x2F,0xFF,0xFE,0xBE,0xEB,0xA7,0x8B,0x76,0xAD,0x3A,0xEA,0x8E,0xDF,0xDB,0xDA,0xFE,0x8F,0x2A,0x57,0xBB,0xD2,0xD2,0xF3,0xB7,0xBB,
			0x6D,0xFA,0x67,0xAE,0x78,0xEB,0x8E,0xF7,0xD6,0xFC,0xA7,0xFF,0xEF,0xCE,0xEF,0xAB,0xE3,0x38,0xDD,0x1B,0xFB,0xFF,0xAD,0x77,0xFA,0xD9,0x1D,0xF0,0xF5,0xB5,0xA9,0xF9,0xFE,0x7F,0x55,0x99,0xB0,0xE9,0xB7,0xE2,0xE6,0x7B,0xBB,0x9B,0xCE,0x3F,0xD3,0x7D,0xCB,0xF9,0xAF,0xBF,0xEF,0xE2,0xB7,0xBF,0xA4,0xE7,0xA8,0xFE,0x9F,0xDF,0x52,0xFD,
			0x37,0xD7,0xEF,0x77,0x67,0xFF,0x8E,0x8A,0xFE,0xF8,0xEE,0xCC,0xF7,0xCD,0xFB,0xFD,0xE9,0xDE,0xFE,0x9D,0xBF,0xBC,0x3E,0x74,0xBE,0x5F,0xEC,0x94,0x8E,0xED,0xAE,0xDA,0xAE,0xE6,0x59,0x7F,0x6F,0x13,0xFA,0x3F,0xBD,0xEF,0xF0,0x3E,0x6C,0xFE,0x6E,0xDF,0xF4,0x99,0xAA,0x2A,0x6F,0x94,0xE2,0xFC,0xFD,0xB7,0xFE,0xA7,0xAA,0x7B,0xFA,0x86,
			0xC7,0x7B,0xD2,0xFF,0xD9,0xA3,0x66,0x5F,0xE6,0xEF,0xBA,0x1E,0xFF,0xF7,0x5F,0xFA,0xDD,0xFF,0xF9,0xEF,0x31,0xFF,0xAC,0x5D,0xDF,0x82,0x4F,0x6D,0xBF,0x37,0x35,0xF7,0xFF,0x6D,0x31,0x2B,0xB7,0xC1,0xDB,0xF7,0xF3,0x26,0x9F,0xFC,0xEB,0x76,0xFB,0xFB,0xDA,0x65,0xDB,0xBD,0xBE,0xDA,0x9F,0xFF,0xDD,0x6C,0xB2,0xFE,0x9F,0xFF,0x8E,0xF7,
			0x92,0xCB,0xFA,0xFD,0xE7,0xBB,0xFE,0x7E,0x1D,0xCE,0xB8,0xF5,0xF7,0x7D,0xFE,0xFF,0xFE,0xFC,0xEE,0x9F,0xCB,0xFE,0xAA,0xFE,0x2E,0xFA,0x29,0x69,0x71,0x6A,0x5F,0xF6,0x9D,0xAF,0x88,0xF9,0x2E,0x98,0xF3,0xE3,0xE7,0xAA,0xB8,0xB3,0x7D,0xEF,0xBD,0xDF,0x8B,0xEB,0xDC,0xFE,0x1E,0x83,0xBF,0x75,0xFB,0xDE,0x8B,0xEB,0x7A,0x5A,0x9E,0xE3,
			0x0E,0x2E,0x93,0x4F,0xBA,0xFC,0x89,0xFF,0xFE,0x7D,0x9D,0x9B,0xEC,0x8E,0xDB,0xE3,0xEF,0xBB,0xED,0xFB,0x7A,0xBF,0x97,0xF7,0x3E,0xD9,0x3F,0xF7,0xF9,0xEE,0xE9,0xCF,0x36,0xF7,0xDB,0x14,0xFA,0x4B,0xFD,0xFF,0xC4,0xBA,0xDA,0x9A,0xC8,0x73,0x2F,0xBD,0x8D,0xBF,0x57,0xFB,0xAF,0x8F,0xAF,0xBF,0x77,0xE4,0xEF,0xBF,0xFF,0x7F,0x77,0x6B,
			0xFF,0xBB,0xBA,0xF5,0xFE,0x3F,0x8F,0xFF,0x88,0x5B,0x8E,0xBA,0x6E,0xF4,0x6B,0xB0,0x2E,0xAF,0x7E,0xEA,0x7F,0xAD,0xDA,0x50,0xF6,0xFF,0xE9,0xF7,0x3F,0xDA,0xC6,0xED,0x3B,0xDD,0xF6,0x9F,0xAF,0xFE,0xBC,0x7F,0x05,0xBD,0x2D,0xBD,0x6B,0x6D,0xFA,0xD7,0xFE,0xEB,0x8D,0xF6,0xBB,0xFF,0xE7,0xF9,0x4E,0xC7,0xBF,0xD9,0xEF,0x3B,0xBE,0xC8,
			0xB5,0xBF,0xB8,0xD3,0xFF,0xAF,0xDF,0xFB,0xCF,0xDE,0x7F,0xFB,0x7C,0x8F,0xEF,0xD7,0xBF,0x56,0x7F,0xEF,0xB0,0xAF,0x6E,0xCF,0xEB,0x7F,0xBF,0xFE,0xB9,0xEB,0xF9,0x73,0xFB,0x6A,0x21,0xEF,0xEA,0xB4,0x9A,0x67,0xFD,0x38,0xBB,0xF7,0xBB,0xFB,0x7B,0xA4,0xCF,0xEA,0xB3,0x7F,0x9A,0x8B,0xFE,0xED,0xFD,0xE9,0x9D,0x66,0xA1,0x5B,0xFF,0xFE,
			0xF8,0x64,0xF7,0x2F,0x6B,0x33,0xF8,0x3D,0xB7,0xDE,0xFA,0xD5,0x2B,0xAF,0x26,0xFB,0x92,0x8F,0xF2,0xFA,0xF8,0x43,0x7B,0xF2,0xEF,0xAC,0xDD,0xB0,0x40,0xF7,0x73,0x7F,0x4F,0x7F,0xEB,0x3C,0xDD,0x3A,0x46,0x9D,0xEB,0xFF,0x79,0x3F,0xAD,0x6F,0xEC,0xFE,0xF7,0xAC,0xEE,0x8B,0x59,0xBB,0xE7,0xDF,0x1A,0xBF,0x3F,0xFC,0xE9,0xF7,0xF4,0xF2,
			0xFD,0xF9,0xF6,0xDE,0x7B,0xBD,0xF6,0x5E,0x56,0xDF,0x5A,0x5C,0x53,0xD8,0xCF,0x6B,0x16,0xEF,0x27,0xFE,0xF9,0xAA,0x7F,0xDE,0xCE,0xFF,0xDF,0xB6,0xFF,0xC9,0xF6,0xFF,0xEC,0x9F,0x6F,0xBA,0x6F,0x86,0xEF,0xBE,0xC6,0xEC,0xFB,0xFB,0xE7,0x59,0xCA,0xFE,0xB6,0x9F,0x6B,0xF2,0xEF,0x6F,0x7E,0xF5,0xCE,0xCA,0x6F,0x63,0xFE,0x7E,0x77,0xDE,
			0x34,0x58,0x7F,0x67,0x73,0x6E,0x67,0x6B,0xAE,0xDD,0xD7,0xE5,0xB7,0x7B,0xE6,0xF6,0xA5,0x91,0x7E,0x7B,0x2D,0x1B,0xEF,0xBE,0x2E,0x78,0x6F,0xEF,0x6B,0xB6,0xAF,0xEE,0xE3,0x37,0xDB,0xAB,0xBF,0xEC,0x9E,0xFF,0x6F,0xFA,0xEF,0xF7,0x7F,0xEB,0xD1,0xBB,0xF7,0xA3,0xFB,0x6B,0xAB,0xA6,0x2C,0xFB,0xBE,0xDD,0x3B,0xFF,0xF3,0x0E,0xCF,0xB7,
			0xB5,0xFF,0xF7,0xB7,0xED,0x59,0x5F,0x66,0xB2,0xE8,0x87,0xAC,0xD5,0x1A,0x8C,0xEF,0xF1,0xAF,0xBA,0xC3,0xA0,0x70,0x8E,0x9F,0x53,0x3F,0xF3,0xFE,0xBD,0x5A,0x35,0xE0,0x7C,0xF5,0xBC,0xF6,0xD6,0xBA,0xE9,0x6A,0x7E,0x3D,0x3B,0xA7,0xC6,0xDD,0x5D,0x5E,0xE7,0x27,0x9B,0xAF,0xFF,0x7B,0xE7,0xFE,0xEF,0xAD,0x9F,0xED,0x3A,0x54,0xD5,0xA8,
			0xE8,0xFA,0x96,0xAE,0xA6,0xAF,0xFB,0xCA,0xFE,0xEF,0x8C,0x6C,0xDE,0xF6,0x7B,0x3F,0x7A,0x9F,0xAC,0xBB,0xAC,0x2E,0xCF,0x17,0xCF,0xBD,0xFE,0xF3,0xB7,0x52,0xFE,0x7F,0xE5,0xCD,0xBF,0x08,0xBC,0x3D,0x9F,0xB8,0xAA,0xBA,0xF1,0xF3,0x3B,0xAA,0xF7,0xB7,0xBE,0x1D,0xFB,0x83,0xF6,0xEE,0x8C,0xB9,0x9E,0xF2,0x5F,0xF5,0xC7,0x4B,0x34,0xBE,
			0x11,0xFF,0xBB,0x9F,0xD3,0x7B,0xFF,0xDF,0xE7,0xEF,0xEF,0xB7,0xAD,0xED,0xCB,0xFE,0x5F,0xDB,0x93,0xBB,0xE6,0xFF,0xE2,0xEE,0x7D,0x7F,0xA3,0x6D,0x09,0xAF,0xDC,0x7F,0x2F,0xF5,0x9E,0x8E,0x26,0xA3,0x1E,0x36,0xFF,0xE1,0xBE,0x77,0x9E,0xC6,0xAE,0xDD,0xFE,0xED,0xFF,0x07,0x3F,0xA3,0x0F,0xE1,0xBF,0xED,0xFB,0xEB,0xBF,0x3A,0x3F,0xFB,
			0xE5,0xF1,0x4C,0xF6,0xF7,0x1A,0x7A,0x8F,0x3B,0x69,0xD3,0xAF,0x67,0xCF,0xDB,0xF7,0xE3,0x77,0xE4,0xEB,0xEE,0x9C,0x8E,0x7B,0x9F,0x4B,0xBD,0xAF,0xF7,0x3F,0xBF,0x3A,0xFB,0x5A,0xD9,0x9F,0xF2,0x9D,0xFF,0xD7,0xDA,0x9B,0xFF,0xEA,0xCE,0xF7,0xFE,0xFB,0x65,0xFE,0xBD,0x5F,0x6A,0xF9,0xD3,0xC7,0xE3,0xBE,0xEF,0x3F,0x48,0x7C,0xEE,0xDA,
			0xDD,0x5B,0x9D,0xD8,0xE7,0xD9,0xBC,0xB8,0xF7,0xFF,0x7E,0xFD,0xB3,0xFA,0x9F,0x6B,0xEE,0xDF,0xB2,0xDC,0xE7,0xEF,0xD6,0xE6,0x9D,0xEB,0xB9,0x9F,0xC6,0xBC,0x8F,0xEE,0xD7,0x5F,0x7D,0xBD,0xB3,0x8D,0xBE,0xFC,0xFA,0xDC,0xF7,0xDB,0x8B,0x4F,0xB0,0x52,0xBC,0x1E,0xF4,0xAB,0xFF,0xFB,0xF7,0xE3,0xC6,0x47,0xCF,0x33,0xF5,0xF6,0x62,0xCF,
			0xEB,0xBF,0xDA,0xFC,0x3B,0xDE,0xEC,0x7B,0x8F,0xB7,0xEE,0x3D,0x2A,0xBF,0xEF,0x9B,0xB6,0xF3,0x7F,0x7D,0xFB,0x73,0xE9,0x5F,0xEF,0xED,0xF8,0xEC,0xBB,0xE7,0xBF,0x6E,0xEF,0xAD,0x1E,0xE3,0xDE,0xDF,0xAE,0x77,0x93,0x35,0x7B,0xFD,0xE5,0x2B,0xF7,0xF4,0xB7,0xAB,0x2C,0xBE,0x5F,0xFF,0xFF,0xBF,0x7B,0x97,0xE8,0xFF,0xDC,0xBB,0xE2,0xAD,
			0xDF,0x3E,0xBE,0x6D,0xED,0xD9,0xE7,0xF7,0x52,0xFD,0xA7,0xAF,0x7E,0xEA,0xB2,0xE7,0x38,0x9A,0xBC,0xBB,0xBD,0xC8,0x5E,0xFB,0xD8,0xBF,0x5D,0x76,0x2E,0x95,0xBA,0xBC,0x1F,0x33,0xC9,0x61,0xAA,0x4A,0xBF,0xF9,0xA3,0xA9,0x77,0x9B,0xF5,0xE9,0xAF,0xDE,0x9F,0xD6,0x1E,0x5B,0xF3,0x7E,0xF6,0xCE,0xD7,0x1D,0x5F,0xFB,0xF9,0x8E,0xE6,0xDD,
			0x3C,0x3B,0xCB,0xB7,0xBE,0x7E,0xDF,0x9F,0x7F,0xE3,0x4A,0x3A,0xFF,0xBD,0xCA,0xD6,0xFD,0xCE,0xF5,0xBE,0x6B,0x52,0xF7,0xBF,0x7D,0x69,0x5E,0xAB,0xD7,0xDF,0xC6,0xB7,0xFF,0xEC,0x63,0x9D,0x3D,0xCA,0xEC,0x3F,0x2E,0xA1,0xEF,0xFF,0xEF,0xE8,0xAA,0xFB,0xFE,0x4B,0xE3,0x16,0xAF,0xB7,0xC9,0xFD,0x9D,0xA1,0x2F,0xFD,0x77,0x3F,0xF4,0xAE,
			0x5D,0xF5,0xEE,0xF6,0xC7,0x74,0xDF,0xAE,0xDE,0xA1,0xB7,0xAF,0xE7,0xA7,0x3F,0xDF,0x6F,0xA2,0xE6,0xFF,0x63,0xD5,0xBF,0x3F,0xCB,0xB3,0xEA,0xFF,0xFC,0xED,0x5F,0x8B,0xFD,0xDA,0xAA,0x80,0xB7,0xFF,0xF7,0xDF,0xCF,0xB3,0xEE,0xEB,0xED,0xA1,0xAF,0xF8,0x7F,0xED,0x2A,0xB9,0xFF,0x49,0xFF,0xDC,0x97,0xDB,0x9E,0xF2,0xFB,0xFF,0xBC,0xFF,
			0xFD,0x98,0xD9,0x5B,0x5F,0xEA,0x77,0xFE,0xE5,0xFB,0xFE,0xE4,0x7F,0xB6,0x33,0xDF,0xDA,0xFF,0xEF,0x19,0x6A,0xEF,0xDB,0xAF,0xCB,0xDE,0xE1,0xFD,0xAF,0xFD,0xF9,0x56,0xBF,0xB6,0x7E,0xBF,0xDD,0xE9,0x6F,0x7F,0xF8,0xDF,0xFE,0x7F,0xFF,0xEF,0xDC,0x79,0x1A,0xB2,0xAE,0xFA,0x9E,0x2E,0xEA,0x6D,0x75,0x3D,0xE0,0x3E,0xAF,0xBA,0xD7,0xA9,
			0xFA,0x2D,0xF7,0x29,0xF9,0xAE,0x81,0xAC,0xFE,0x0A,0x2E,0xDB,0xEA,0x59,0xDF,0xFB,0xDC,0x7F,0xAD,0xEF,0x7F,0x8A,0xF5,0xD7,0xDF,0x7E,0xA9,0x3F,0xBF,0xAD,0xDF,0xB6,0xED,0xD7,0x63,0xBB,0xDE,0xFF,0xFD,0xF3,0xCB,0xF9,0xE9,0xBF,0x9E,0xFB,0x55,0xB8,0xE7,0xCB,0x7F,0xD7,0x7F,0x1F,0xD7,0x3B,0x9A,0xF7,0x99,0x9F,0xFF,0xB6,0xFD,0xEF,
			0xF7,0xBF,0xEF,0xB9,0xEB,0xED,0x2F,0xFA,0x7C,0xDE,0xBB,0xF6,0xA1,0xE7,0xF1,0xDF,0xE9,0x7F,0xF9,0x3F,0xBF,0xCA,0xBA,0x34,0x5B,0xFE,0xBE,0xE6,0x39,0x9C,0xE7,0xB3,0xEE,0x8F,0xE6,0xC5,0x9F,0xE7,0xE9,0x5D,0x2C,0x4F,0xEC,0xAA,0xDC,0xFC,0xFA,0xD9,0xEE,0x4B,0xDE,0xA5,0xA8,0x3F,0xCC,0xBD,0x3D,0x5F,0xF6,0x65,0x9C,0x6D,0xFB,0xF6,
			0x7F,0xBB,0x7D,0x73,0xFB,0xEE,0x26,0xA3,0xAF,0xBF,0xF7,0xAB,0xAD,0xED,0x6F,0xB3,0xC4,0x73,0x6F,0xBB,0x3E,0x7A,0x97,0xAD,0xBF,0xDF,0xFE,0xFB,0xCB,0x73,0xBC,0x7D,0xF3,0xF7,0x1F,0x9B,0xF8,0xC6,0x49,0xEB,0xF7,0x69,0xB7,0xEF,0xE7,0xFE,0x8F,0x63,0xC5,0xF7,0xFA,0x50,0xCA,0x67,0xFF,0xDC,0xF4,0xDF,0xAA,0xF5,0xCF,0xB9,0xFB,0x42,
			0xCE,0x93,0x4E,0xEF,0xF3,0xF6,0xED,0xDB,0x4B,0xFF,0x65,0x27,0x57,0xA9,0xBD,0xDB,0xFB,0xAD,0xF5,0xED,0x5D,0xF7,0xEF,0xEE,0x2F,0x59,0xFF,0x0F,0x35,0xE8,0xDD,0xF7,0xB7,0x8E,0x21,0xB7,0xEA,0x3B,0xF8,0xBB,0x23,0xF5,0xD8,0x7F,0x4A,0x2E,0x6E,0xB7,0x4D,0x3F,0x24,0x7F,0xF1,0xFE,0xAE,0xBF,0x65,0xAE,0xC8,0x2D,0xEA,0x3D,0xCD,0x7B,
			0xE5,0x90,0x7D,0xB7,0xCF,0xFC,0xB1,0x6F,0xA0,0xF7,0xFF,0xDD,0xFE,0xBE,0xF3,0x41,0xF5,0x76,0x7F,0xBA,0xEB,0x42,0xC9,0xCF,0xB7,0xCB,0xDE,0x7E,0x66,0x78,0xD3,0x99,0x5B,0xCB,0xFB,0xF7,0x14,0xAC,0x3F,0xF7,0x6F,0xCC,0xFF,0xEE,0x86,0xF7,0xEF,0x8F,0x78,0xEB,0xAF,0x7B,0x88,0x99,0xDF,0xE8,0xAB,0xCD,0xFF,0x6D,0x84,0xE9,0xDB,0xDE,
			0x31,0x86,0xF4,0xF4,0xFF,0xF6,0xBF,0xAA,0xC6,0x94,0xFB,0x5A,0x7A,0xEF,0xB2,0xF7,0xBB,0x3E,0xF9,0xFD,0xB7,0xFF,0x6D,0xFF,0xD7,0x9F,0xE4,0xDB,0x5D,0xFF,0xEE,0xB1,0xFF,0x72,0xDF,0xFF,0xDD,0x7B,0x9F,0x7E,0xFB,0xDF,0x13,0xAD,0xBB,0xFF,0xFB,0xBF,0xDE,0xA7,0xFE,0x37,0x9C,0xBC,0x39,0x67,0x7D,0xFF,0xFF,0x1D,0xBD,0x5F,0xF5,0x77,
			0xF5,0x7E,0xF8,0x27,0xF6,0xFB,0xDB,0x5D,0xDD,0xDE,0xEF,0xEE,0xF7,0xCA,0x3F,0xFE,0xFF,0x7A,0xFB,0x10,0x36,0x9F,0xD7,0xAE,0xA7,0xFB,0xE9,0xBF,0xEC,0xDD,0xAF,0xFE,0xDD,0x17,0xAB,0xE1,0xCF,0xAB,0xBC,0xFB,0xAE,0x5A,0xBA,0x66,0xEF,0xFA,0xDC,0xA6,0xE8,0x9B,0xBD,0x2C,0xF8,0x88,0xF1,0xFF,0xEC,0xDF,0xFB,0xFA,0xB7,0x2F,0x3E,0x7E,
			0xFB,0xF5,0xDE,0xFF,0xB3,0xDC,0xCC,0xE6,0x9D,0x4F,0x7E,0xC7,0xEA,0x49,0x7F,0xAB,0x86,0xF7,0xFF,0x7E,0x4E,0xFF,0xE5,0x7F,0xFE,0x57,0x6A,0xBD,0xF9,0x7D,0xA7,0xDA,0xFF,0x95,0x7B,0xBF,0xAB,0xAB,0xB3,0xA6,0xE6,0xFB,0xEB,0xFE,0xF9,0xEE,0xF7,0xE7,0xF6,0x8A,0x7B,0xBB,0x69,0x3D,0xFE,0x3C,0x77,0xAF,0xF3,0x8F,0xD9,0x83,0xDF,0xFC,
			0xDD,0x7B,0xAE,0xEB,0xFD,0xAB,0x39,0xFF,0xEB,0x9F,0x46,0xB7,0xBA,0x73,0xBB,0xDF,0xFA,0xFF,0x77,0xED,0x36,0x8B,0xB1,0xFA,0xF2,0xFF,0x6D,0x9B,0xBD,0x2F,0xC1,0xAF,0x67,0xFD,0x03,0xEF,0x6F,0x7A,0x3D,0xCF,0x77,0x7F,0xBA,0xEF,0xD4,0xFC,0xFE,0x26,0x8F,0xDF,0x9B,0xB2,0xC9,0x76,0x1F,0xFF,0xBF,0xF7,0x5A,0xBB,0xB8,0xD2,0x4B,0x7E,
			0xEB,0xA4,0xA2,0xDA,0x6E,0x89,0xDF,0xFE,0xB4,0x88,0xFD,0x75,0xA7,0x99,0xB3,0xFF,0xD3,0xCB,0x73,0x73,0xF2,0x94,0x52,0x7D,0xD7,0x1D,0x3B,0x2F,0x3F,0x2F,0x8F,0xFF,0x85,0xBE,0xEC,0xFD,0xAB,0xF2,0xBF,0xFD,0xEB,0xCF,0xBF,0xE6,0xAD,0x25,0xEE,0x22,0x7B,0xBF,0x3C,0x93,0xEF,0xE5,0xAE,0x6E,0xDE,0xFD,0xFF,0xF7,0xB6,0xE7,0xD3,0x5A,
			0xED,0xAD,0x3D,0x67,0xB5,0xFE,0xDF,0xFB,0x6B,0x3D,0xDF,0xAE,0x2F,0x85,0xCB,0x78,0x9D,0x2B,0xEF,0xE4,0x7F,0xF4,0xDF,0xD7,0xF7,0xAF,0xFB,0xFE,0xD5,0xF5,0xCE,0xBE,0xC7,0xE9,0xDD,0xEF,0x3F,0xEA,0x77,0xDE,0xDE,0xF9,0xD3,0xBF,0x30,0x36,0xE6,0xF8,0xF6,0x7B,0x7E,0xF7,0x7F,0xDB,0xAF,0xF4,0xBB,0x6D,0xBC,0x73,0xFE,0xE9,0xB5,0xBF,
			0xC7,0x7A,0x58,0xBA,0xBD,0x7F,0xFF,0xEB,0xA0,0x3A,0xEE,0xBB,0xEF,0xB5,0x9F,0xDB,0x33,0x9A,0x2D,0xFE,0xF7,0xAE,0xE7,0xCA,0xEF,0xF6,0xEF,0xCE,0xE0,0xAD,0xFF,0xFE,0xD2,0x65,0x7F,0x8B,0x8A,0xB1,0xEE,0x9F,0xFC,0xFF,0x7F,0x67,0xFF,0xBE,0x1B,0xFE,0xCE,0xEA,0x6C,0xFC,0xB3,0xC7,0x3A,0xFB,0xCD,0x5C,0xF3,0x70,0xE1,0xED,0xEB,0xFC,
			0x6C,0xB4,0x6D,0xDD,0x5A,0xEB,0x2B,0xB9,0xB7,0xFF,0xFA,0xFF,0xF4,0xDC,0xEE,0x6B,0x8E,0x8B,0x2E,0xFF,0xE4,0xFB,0xF5,0xD3,0xAB,0xDF,0x7C,0xFF,0xBF,0x7A,0xC8,0x7E,0xF3,0xFE,0xAE,0xFA,0xB3,0xF9,0x76,0xBA,0xEF,0x7D,0xD9,0x6F,0xE2,0x6A,0x7D,0x3F,0xBF,0x71,0xEE,0x7F,0xD3,0xBB,0xFF,0xBE,0xB7,0xF7,0xBD,0xFE,0xAD,0x9E,0x9F,0xE5,
			0xFB,0xBB,0xF3,0xCD,0x9C,0xFF,0xF4,0xF0,0xE9,0xED,0x72,0x8F,0x78,0x3B,0xA8,0xFF,0xFE,0xCF,0x71,0x23,0xFD,0xCC,0xB9,0x5C,0x7E,0xA9,0xDE,0xB7,0x9B,0xEB,0xFE,0x6F,0xB8,0xA7,0x97,0xDF,0xBE,0xAA,0xF5,0x7B,0xBF,0x83,0xA1,0xC9,0xBF,0x25,0xBD,0xF5,0x7A,0xD7,0xEE,0xC3,0xBE,0xCC,0x97,0xBF,0xAE,0x6F,0x7B,0xEB,0xFE,0x3D,0x6F,0xF7,
			0xEA,0xA6,0xC4,0xA9,0x3E,0x9F,0x72,0xFE,0xD7,0x32,0xD5,0x78,0x79,0x2C,0xEF,0xAC,0x6F,0xEB,0xE5,0xAE,0x77,0xDD,0xB3,0xF9,0xFF,0xE3,0x6D,0x95,0xE7,0xBF,0xFB,0xDF,0xBE,0xFE,0xF7,0xC4,0x3F,0xB5,0x69,0x6F,0x7B,0xFD,0x7B,0xB3,0x7F,0x3F,0xBB,0x6A,0x9E,0x6B,0xE7,0xA1,0xDE,0x67,0xDF,0xED,0x6F,0xFF,0xDA,0xBF,0xCE,0x9F,0xFD,0xBF,
			0xEE,0xFE,0x98,0xFA,0xBB,0x3E,0xEF,0x15,0xBE,0x7F,0x6D,0xEF,0x51,0xBF,0x6F,0xFF,0x36,0xF5,0x18,0xAD,0xC9,0xFD,0x69,0xEE,0x0D,0xFF,0x2B,0xBC,0xDB,0xDC,0x51,0xF3,0xFF,0x57,0x43,0xDA,0xB7,0xAB,0xEA,0xA5,0xFB,0xF5,0x15,0xEE,0x9F,0x34,0xCF,0xFE,0xA7,0x9F,0x6F,0x2B,0xEF,0xBF,0x76,0xE4,0xFF,0xFF,0x7A,0xAF,0x9F,0x23,0x75,0xF3,
			0xF1,0xB9,0x66,0xFF,0xEB,0x9A,0xE8,0x82,0xBF,0xFB,0xCA,0x8F,0xAE,0xDF,0xFD,0xAD,0x6B,0x4F,0x6F,0xDA,0xEF,0xEE,0xFC,0xEF,0x6B,0x59,0xE1,0xFE,0xFE,0xA2,0xF7,0x2F,0x14,0xDA,0xD1,0xF3,0xCB,0xE5,0xF3,0xDE,0x21,0x7F,0xAF,0xED,0xFB,0xD7,0xE1,0xBF,0xCF,0x1F,0xEF,0xCD,0xFF,0xBE,0xEB,0xFF,0xFF,0xF8,0x5F,0x6E,0x93,0x9B,0xFE,0xF5,
			0x67,0xCF,0xBD,0x57,0xFE,0xEB,0x3F,0xB6,0x7D,0x7B,0x6C,0xFB,0x38,0xFF,0xB7,0x76,0x6F,0xEA,0x9F,0x7C,0xDA,0xFB,0xF5,0xFB,0x97,0xBE,0xB5,0xDA,0xAF,0x6C,0xB6,0xFF,0x99,0xEE,0x17,0xFA,0xE5,0xDF,0xFF,0xDD,0x3A,0xEA,0xFB,0x26,0xFC,0xBF,0x67,0x6F,0xFE,0xF4,0x7F,0xFA,0x7D,0x6F,0x58,0xDA,0xB7,0xFE,0x9E,0xDE,0xA7,0xC7,0xA6,0x3F,
			0xBF,0xF7,0x77,0xD1,0xFA,0x77,0xF1,0x3B,0xFF,0xFE,0xFE,0xB1,0xFC,0x9E,0xD7,0xFE,0xEB,0xF9,0xAB,0xAF,0x25,0xEF,0xDB,0x93,0xFB,0x1A,0xF8,0xF7,0xF6,0xFB,0xDD,0xFF,0xD9,0x67,0x7F,0xD3,0xEF,0xFC,0xB7,0xFC,0xFE,0xEF,0xDB,0x9A,0xBB,0xFA,0x6E,0xCB,0xAE,0xEE,0xD0,0x16,0x2C,0x82,0x27,0xB7,0xBF,0xC6,0x9F,0xAB,0xEC,0x9A,0xA7,0x2E,
			0xBD,0xCF,0xB3,0xCC,0x0F,0x1E,0xF6,0xED,0x7D,0xE9,0xFF,0xBF,0xA7,0xF1,0xDF,0xF6,0x7E,0x47,0x3A,0xDE,0xEE,0xFF,0xF3,0xAB,0xA8,0x9E,0xFF,0x63,0xA2,0xDA,0xB5,0x5E,0xF7,0xFB,0x0E,0x37,0xB3,0x3F,0x3C,0xBD,0xE5,0xCF,0xF9,0x97,0x20,0xEF,0xEC,0x5B,0x3B,0x7F,0x27,0x40,0xB8,0xFA,0xE8,0x79,0xA4,0xD5,0xEB,0xEE,0xFA,0xB9,0xFF,0xFA,
			0x1D,0xF8,0xF7,0x3D,0xDB,0xD9,0xBF,0xBE,0xBE,0xD2,0x2B,0x5F,0x5B,0xEE,0xED,0x2F,0x5F,0xFF,0xA5,0x27,0xFF,0xED,0xFC,0xBF,0xFC,0xF2,0x6D,0x74,0xEE,0xDF,0xFF,0xBF,0xFF,0xFC,0x7E,0xBF,0xFE,0xFE,0x7F,0x7F,0x6F,0xF6,0xFF,0x73,0xEC,0xAA,0xBB,0xEF,0x7F,0xFE,0xFD,0xBB,0xE7,0xB2,0x8B,0xED,0xFF,0x3E,0x7F,0xDD,0x73,0xBA,0x7E,0xB9,
			0xF6,0xFF,0xCB,0x79,0xDA,0xDE,0xAE,0xBE,0xFD,0xEF,0x4F,0xE6,0xAE,0xFE,0x76,0xB6,0xF5,0xEE,0x39,0x9F,0xFF,0xEF,0xED,0x1F,0xFF,0x07,0x53,0xF2,0x3F,0xC4,0xDD,0xFF,0xFE,0x92,0x5E,0xF1,0x85,0xFF,0xDE,0xBF,0xF6,0xFE,0xFE,0xFE,0x96,0xDA,0xE0,0xAA,0xFD,0x56,0xF6,0x29,0xA7,0xFF,0xFD,0xEF,0x5C,0x6F,0xFE,0xAF,0xFD,0xF9,0x78,0xF9,
			0xFA,0xF3,0x07,0x9B,0x5E,0xB7,0xE6,0xAE,0xF8,0x5D,0xFF,0x86,0xEB,0xCE,0xEA,0xFC,0xFC,0xAB,0x23,0x08,0xFF,0x61,0xBE,0xFE,0xEF,0x3A,0xF9,0x4F,0xEA,0xAB,0x4E,0x1C,0xE6,0xAD,0xFE,0xD1,0xCB,0xFF,0xAA,0x59,0xAF,0x37,0xEA,0xAF,0x25,0xA7,0xED,0x3B,0x0F,0xEE,0xFC,0xEE,0xEF,0x3B,0xBB,0xFC,0x55,0xD8,0xFF,0xE3,0xD7,0xEE,0x76,0x04,
			0x7D,0xBD,0xFA,0xFF,0xF7,0x7F,0xBF,0x5F,0xED,0xF7,0xAE,0xAF,0xF7,0xCA,0xFE,0xBA,0xC8,0x2B,0x4E,0xCA,0xFA,0xAA,0x9B,0x6E,0xF7,0xFC,0xBA,0xAF,0xAE,0x2B,0xFE,0xEE,0xDA,0xBD,0x3E,0xEF,0x55,0xD6,0x79,0xC0,0xFA,0xD7,0x91,0x3F,0xEF,0xB5,0x7E,0xF6,0x32,0xBE,0xB0,0xFF,0x0F,0x67,0x9F,0xFE,0xF5,0x6C,0xB9,0x4F,0x6E,0xAF,0x8A,0xEF,
			0x4B,0xBD,0x99,0x67,0xF7,0xBD,0xDE,0xFB,0x7E,0xEE,0xE7,0xFD,0xB8,0xBC,0xCA,0xD7,0xC4,0xBB,0xAA,0xA7,0xEA,0xAD,0xFB,0xEB,0xF2,0xDF,0x77,0xFD,0x6B,0xBA,0xB8,0xAD,0x7C,0xFF,0xBF,0x7E,0xD9,0xBE,0xDA,0xD7,0xBE,0xB3,0xEA,0x6F,0xB9,0xF2,0x93,0x2E,0xBF,0x9F,0xE1,0x83,0xBE,0xC4,0x9E,0x87,0xE4,0xB7,0xB3,0xFB,0x3F,0xBA,0xBA,0xD8,
			0xFF,0x97,0x8B,0x96,0xFF,0xBD,0xBF,0xAB,0xFB,0x9F,0xEF,0xBD,0x37,0xBD,0xEF,0xF7,0x5B,0x9D,0xBD,0xBD,0x5F,0xDE,0xC8,0xFB,0xFC,0xBA,0xF3,0x9E,0x4A,0xA3,0xAF,0x66,0xAB,0xF3,0xDC,0x6F,0xE7,0x97,0x3F,0x9B,0xA7,0xFE,0xFE,0xBB,0xF6,0x7F,0xF7,0xFF,0x3D,0xF1,0xEE,0xB3,0xEA,0xFF,0xFB,0x2F,0x69,0x6C,0x27,0x93,0xBA,0x6E,0xBB,0xEF,
			0xB8,0xFF,0xAA,0xD7,0x9E,0x4F,0xFE,0xAF,0xAE,0xD5,0x7B,0xF6,0xEF,0x9E,0x7F,0xFD,0xB3,0x7F,0x4D,0xB6,0xAA,0x92,0xD3,0xAB,0xC7,0x6F,0xE3,0x59,0xDB,0xD7,0xB5,0x3F,0x67,0xDE,0xE3,0xD9,0xFF,0xEF,0xE7,0x8A,0xDD,0xBE,0x9B,0xE6,0x3F,0x7E,0xFF,0xF7,0xFE,0xF4,0x37,0xA4,0xF9,0xBE,0xF7,0x5E,0xBD,0xD8,0xAE,0x9E,0xE3,0xA3,0x6E,0x6F,
			0xFF,0x9F,0x5D,0x7B,0x08,0xEB,0x9D,0xB7,0xEE,0xB3,0xDB,0x6B,0xEC,0xEE,0xF9,0x81,0xC3,0xBB,0xAE,0xE4,0xE5,0xD8,0x6B,0xA1,0xF6,0x7D,0x8D,0x72,0xDE,0xAA,0x67,0xEB,0xDD,0xEA,0xFA,0x3F,0xBC,0x2B,0xBA,0x3F,0xF8,0xFE,0xDE,0xAD,0x8D,0xFE,0xE0,0x76,0xDD,0xEB,0x5F,0x2F,0xEB,0x2D,0xFB,0xCF,0x6C,0x3C,0x9F,0xD5,0xBC,0xFD,0xCE,0x5D,
			0xF7,0x2F,0x76,0x7B,0x7E,0x6F,0xFD,0xE7,0xFE,0x9F,0xA0,0xAB,0xA4,0xFC,0xB7,0xFA,0xFF,0xEB,0x33,0xBF,0xD6,0xB2,0xFD,0xCB,0xFF,0x0A,0xFE,0x77,0xD6,0x37,0x7E,0xEB,0xCF,0x68,0xE2,0xBD,0x2B,0x8D,0x8E,0xFA,0x89,0xEF,0xEA,0xCE,0xD7,0xA3,0xBF,0x83,0xD7,0x68,0x39,0xE3,0xBD,0x77,0xF6,0x9A,0xD3,0xAD,0xE3,0x0F,0x3B,0xBE,0xFE,0x86,
			0xF3,0x6F,0x7F,0xAA,0x7F,0x86,0xFD,0x7F,0x30,0x7A,0xBE,0x3A,0xF8,0xB8,0xFE,0xF9,0xEA,0x76,0x67,0x93,0x26,0x5E,0x79,0xEB,0xB9,0x7B,0xAC,0xE2,0xA5,0xAB,0xBF,0xA2,0xD3,0x5B,0xEE,0x2C,0xBE,0x6A,0xD7,0x39,0xEC,0xC1,0xE3,0xDF,0xFD,0x73,0x1C,0xCA,0xBA,0xDB,0xF5,0xDF,0xDE,0xEE,0xAD,0xDB,0xDB,0xED,0xF7,0xBA,0xFE,0xF7,0x0D,0x6A,
			0x3C,0xB5,0x65,0x2A,0x13,0xB4,0x75,0x6A,0x39,0x07,0xB3,0xFC,0xAC,0xB3,0xAD,0xEE,0xFD,0xFF,0xF0,0xA2,0xBB,0xCB,0xBD,0xEC,0x8B,0x7E,0xC3,0x71,0x78,0xBA,0xFE,0xB8,0xF5,0xE5,0x5F,0x34,0x0D,0xD7,0xFB,0xFA,0xE9,0x4D,0xEF,0x3D,0xF7,0x27,0x16,0xE8,0xBF,0xA6,0xFD,0xFB,0x37,0xCF,0x3B,0x93,0x74,0xE6,0xED,0xFE,0xCF,0x3E,0xA6,0xA4,
			0xAE,0xEA,0xBC,0x7E,0xD6,0x91,0xEB,0x9E,0xEE,0xFD,0x7B,0xDF,0xFD,0xFB,0xED,0xCF,0xFB,0xED,0xF9,0x40,0x5F,0x3A,0xEA,0xFC,0xFF,0xC3,0xFD,0xED,0xF9,0xBD,0x6B,0x3A,0xEA,0xFE,0xFA,0xAC,0xBF,0x6F,0x10,0x3D,0x93,0xEB,0xBD,0x1B,0xCA,0xBD,0xC5,0x5B,0xFB,0xDA,0xBE,0xAD,0xD6,0xF3,0x9A,0xEE,0xFB,0xF9,0xA3,0xDE,0xE3,0x9C,0x83,0xFB,
			0xB9,0xCB,0xF7,0xB9,0x44,0xEE,0xBF,0xFF,0x7B,0x87,0x9E,0x98,0x9D,0xEE,0x57,0xF7,0xCD,0xFF,0xFD,0xBD,0xF9,0x0A,0x7F,0xF1,0xFF,0xE3,0x4D,0xFF,0xCF,0x7C,0x03,0x57,0xEE,0xFF,0xEE,0xDE,0xD7,0x9F,0xFD,0x75,0x7C,0xBB,0xAC,0xEE,0xA9,0xCF,0x03,0xBF,0xE9,0xFE,0xE7,0xB7,0xBC,0xBD,0xB9,0xF7,0xD7,0xF7,0xBF,0xE7,0xF7,0xEE,0x70,0xFB,
			0xDB,0xA5,0xEF,0xFF,0x36,0x34,0x9D,0xB7,0xF6,0x73,0xBA,0xB9,0xEE,0xEB,0xF3,0x2C,0xDE,0x95,0xEF,0x6B,0xF7,0xEA,0xCD,0x1D,0xD1,0xDA,0x6B,0xB8,0xA3,0x69,0x73,0xD2,0xE5,0xDB,0x3E,0xF5,0xEC,0xFF,0xB8,0xE7,0xFE,0xFA,0xAB,0x1F,0x8F,0xBD,0xF6,0xAF,0x5D,0xE0,0x59,0xBF,0xEB,0xDF,0xBF,0xB8,0xDE,0x0F,0x8F,0x7E,0xFD,0x6B,0xF3,0xB3,
			0xCA,0x3B,0xAB,0x50,0xF6,0x3B,0xFD,0x77,0x73,0x9B,0xBF,0x6A,0xDF,0xE6,0xB2,0xBA,0xCF,0xBF,0x27,0x3D,0xBB,0x4D,0x79,0xAE,0xEF,0xD3,0xF2,0x5B,0xD7,0xAC,0x70,0xEA,0xE0,0xEF,0xFF,0xCF,0xE8,0xB9,0x7F,0xAD,0xFB,0x2D,0xEE,0x7B,0xFA,0xEF,0xFE,0x9E,0xA7,0xFF,0x7F,0xD8,0xAF,0xCA,0xBF,0xF3,0x8E,0xF9,0xAD,0xBA,0xBD,0xCA,0xFF,0xCF,
			0xE6,0xB6,0x7F,0xD0,0x5F,0x7B,0xE9,0xF1,0xF5,0xEC,0x57,0xFA,0xCA,0x38,0xF5,0x52,0xE2,0x6F,0xE6,0xF9,0xFE,0xF2,0x9F,0xBF,0x79,0x77,0xA8,0xD6,0xEA,0x8F,0x8E,0x3F,0xF2,0x5E,0x1E,0xDD,0xBD,0x3E,0x7A,0xCF,0xFF,0x5B,0x1F,0x62,0x7D,0x8F,0xB3,0x2C,0x74,0x6E,0x7F,0xDF,0x90,0xBF,0xEA,0x6B,0x77,0x3A,0xFC,0xCF,0x62,0xDC,0x36,0x9D,
			0x2D,0xED,0xEC,0xDF,0xAD,0xD2,0xF3,0xCF,0x7F,0xB7,0xEA,0x77,0x6A,0x8D,0xDE,0xDF,0xB9,0x6B,0xBF,0x34,0x6A,0xF7,0x5D,0xFF,0x38,0xFE,0x33,0xF5,0xE7,0x9D,0xFD,0xEF,0x24,0x76,0x7F,0xBF,0xA2,0x5B,0x17,0x63,0xDF,0x3E,0xB7,0x6E,0xB6,0x3F,0xB7,0xFF,0xF0,0xBF,0xBB,0xB4,0x7F,0x7E,0xBB,0x6F,0xFA,0xFF,0x7A,0x6F,0xF4,0xDF,0xBE,0xF7,
			0x8F,0x7F,0xFE,0xF8,0x2B,0x94,0xFB,0xF5,0xCF,0x7B,0x7F,0xFA,0xFA,0xBD,0xD7,0x2D,0xAB,0xE6,0xBA,0xCA,0xBE,0xEE,0x3B,0xD4,0xC6,0xEB,0xFB,0xB2,0xDF,0xB9,0xEE,0x37,0x5E,0xFF,0xFB,0xBE,0xFB,0x0E,0xB3,0x99,0xFD,0x9E,0xFE,0xE3,0xFD,0x6E,0xFF,0xD3,0x2F,0xFF,0xAE,0xFB,0xFF,0xDF,0xFB,0x9B,0xBD,0x9B,0xBB,0xAB,0x67,0xF5,0xDF,0x2F,
			0x7E,0xEE,0xF5,0xDE,0x69,0xDB,0xB9,0xE6,0x5F,0xAF,0xFF,0xA8,0xEE,0xEF,0xFE,0x63,0x4F,0xF1,0xF1,0xE5,0x05,0x8E,0x7A,0xB6,0xFE,0x9F,0x2F,0x99,0x2F,0xDF,0xBE,0x7D,0xEC,0xF9,0x9E,0x3A,0xB2,0x37,0x4F,0xBD,0x8F,0xAB,0xB9,0xE5,0x87,0xA7,0x3F,0xB5,0xFF,0x7B,0x36,0x25,0x3F,0x99,0xDD,0xEF,0x91,0xDB,0xDA,0x9F,0xE3,0x3B,0xB9,0xB5,
			0xC5,0xDD,0x99,0xDA,0xBF,0xAF,0xB6,0xDB,0xF4,0xBE,0x9F,0xE3,0xA7,0xD3,0xD8,0xAF,0x86,0x97,0xAA,0xDA,0xBF,0xEF,0xAF,0x86,0xEE,0xF8,0xF7,0xEA,0x97,0xEC,0xE7,0xEA,0xDF,0xDF,0xAB,0xFA,0xBF,0xBF,0xFA,0xFF,0x74,0xA2,0xF9,0xE9,0xAF,0xBF,0x0E,0xBF,0xFF,0x8F,0xBB,0xF7,0xDF,0xEB,0xBB,0x6E,0x77,0xAA,0xDF,0x40,0xFD,0x9A,0x17,0xDF,
			0xC7,0xEB,0xF8,0xAA,0x7E,0xEF,0x68,0xBB,0x9D,0x7F,0xCC,0xBC,0xEB,0xAC,0xF6,0xFC,0xEF,0xF7,0x1A,0xFB,0xDC,0xFF,0x37,0x69,0xDE,0xFE,0xEC,0xFF,0xFB,0xDC,0xB8,0xFA,0xD2,0xAE,0x7F,0xDE,0xBE,0xB2,0x7D,0xAF,0xA9,0xEF,0x7B,0xB3,0xEE,0xEC,0x7A,0xFA,0xBA,0xB8,0xE3,0xE6,0x6F,0xEE,0x7E,0x4F,0x79,0xBD,0xFA,0xB3,0xEF,0xE8,0xF6,0x38,
			0x70,0xD5,0xEF,0xBB,0x33,0xBC,0xF6,0x7A,0xBF,0x6C,0x6E,0xD2,0xFC,0xFE,0xD6,0xFE,0xB9,0xE9,0x7B,0xFE,0xEF,0xDF,0xD3,0x3A,0x8F,0x7D,0xF7,0x6F,0xFB,0xEA,0x36,0xBA,0xF5,0xB5,0xEF,0x13,0xB1,0xF9,0xED,0xD6,0xD3,0xC8,0xF4,0xDE,0x5E,0x9B,0xDF,0x5A,0xFF,0xF9,0xFE,0xD9,0xBF,0x9F,0xD2,0xC1,0x9C,0xF2,0xB8,0xEF,0xBC,0xDA,0x7B,0xFE,
			0x5F,0xBD,0xBC,0xBD,0x9E,0xFD,0xDC,0xDE,0xCD,0xDE,0x2A,0x62,0xDF,0xDF,0xCF,0xAB,0xD5,0xE3,0xAE,0x7B,0xCF,0xFF,0xEE,0xDB,0xFB,0xAF,0xE9,0xB1,0xBA,0xDA,0x2E,0x7F,0xCB,0xE5,0xC5,0xFF,0xBC,0xEC,0xFC,0xCF,0x45,0xFF,0x2F,0x3A,0xB6,0x2A,0xA7,0xB3,0x0A,0xFA,0x63,0xFD,0x2E,0xF6,0xD7,0x08,0xF6,0xFB,0xA1,0xF8,0xDD,0x7E,0xCE,0xBB,
			0xFF,0x3A,0xAB,0xF6,0x7F,0x7D,0xF7,0x4F,0xF7,0xED,0xBB,0x92,0xC2,0xF5,0xFF,0x2B,0xF7,0x1B,0xC6,0x3E,0x8F,0xDB,0xEF,0xDA,0x8F,0xD7,0xDD,0x7C,0xBB,0x3F,0x66,0xF9,0xFF,0xDE,0x4F,0xB5,0xE2,0xBF,0xE9,0xDD,0x0B,0xB7,0xFE,0xBE,0xED,0xDD,0xFE,0xA5,0x96,0xF3,0xEA,0x3D,0xFE,0x87,0xF6,0x5C,0xDA,0x8E,0xF3,0xE3,0xFE,0xFE,0xF2,0xEF,
			0xF7,0x5B,0x7F,0x6E,0xAB,0xFE,0xE5,0xC8,0xDF,0xFD,0xFC,0x4F,0x3F,0xB6,0xD6,0x0F,0xAD,0xDF,0x6D,0x7D,0x9F,0xEB,0x26,0xE8,0xFF,0xF5,0xDF,0xFA,0x3D,0xDB,0xAF,0xF5,0x5A,0x1F,0x3B,0xA7,0x69,0xE8,0x3C,0xAF,0xFA,0x7F,0x9A,0x97,0xAD,0xB3,0xE5,0x6C,0x58,0x37,0xB8,0x7A,0xAA,0xB1,0xFB,0x6F,0xBF,0xF4,0x8B,0x76,0xFF,0xA5,0xEC,0xBA,
			0x7F,0xF9,0xB9,0xBF,0xF2,0xEF,0xA2,0xF9,0x7E,0xCC,0xEB,0xBB,0xEE,0xF2,0xDF,0x7A,0xDB,0xBF,0x7D,0xAE,0xFC,0x0A,0xFC,0x7C,0xD7,0x7F,0xFF,0x1D,0xB5,0xF9,0xEE,0xFF,0x1F,0xF7,0xF5,0xBB,0xDA,0xCE,0x56,0xBF,0x7B,0xBF,0xBD,0xBF,0xCD,0xAF,0x8A,0xFF,0xBD,0xF7,0xA3,0xC2,0xE5,0xDB,0x6F,0x3E,0xBC,0xF5,0xFF,0xDD,0xC5,0xAB,0xFD,0xAF,
			0x7F,0xD5,0x7F,0xB7,0x0D,0xF6,0x3A,0x67,0xEF,0xCA,0xE8,0x3D,0xBF,0x87,0x49,0x7E,0xFB,0xEA,0x2B,0xB0,0xAB,0x9F,0xBA,0x7D,0xCC,0xFD,0x0B,0x8B,0xFE,0xFF,0x56,0xF9,0xB2,0xCA,0xF6,0xBB,0xE9,0xCE,0x9A,0x83,0x78,0xCD,0xE6,0xCE,0xC9,0xFF,0xF5,0xF7,0xBB,0xF7,0xD7,0xCF,0x7A,0xFA,0x9F,0x8F,0xB4,0x0F,0xDD,0x9C,0xDB,0xA6,0xC0,0xA7,
			0x73,0xF6,0x2E,0x1B,0xD4,0x1A,0x9A,0xF8,0xB4,0x5E,0x8D,0x7E,0x63,0xB9,0x3A,0xCA,0xF5,0x36,0xF3,0xFF,0xCE,0xDD,0xEE,0xF6,0xEE,0xFF,0xF9,0x39,0xF6,0x58,0xBB,0xFF,0xF2,0xE3,0xBF,0xFF,0xD7,0xDF,0xE0,0xF7,0x29,0x5B,0xFE,0xAF,0x2B,0xEA,0xED,0xB7,0x9E,0xEE,0xF3,0x7F,0xC7,0xEE,0xDA,0x3E,0x9C,0x6F,0xAC,0xFE,0xEB,0xE2,0xED,0xF6,
			0xFB,0xE5,0x87,0xF3,0x37,0xFA,0xEB,0xFF,0x68,0x50,0x3A,0x8A,0xBD,0x4F,0xFA,0x9F,0xF3,0xFE,0xBF,0xD7,0xEB,0xCB,0xFF,0xAC,0x9C,0xBF,0xEB,0xE3,0x95,0xE7,0xFC,0xBE,0xE6,0x5F,0x16,0x3F,0x51,0xBB,0xB4,0xED,0xE8,0x6B,0xEF,0xD5,0xE9,0xEE,0xCC,0xD1,0x2F,0x5F,0x0E,0x7F,0xF3,0x5B,0xA5,0xDB,0x9E,0xF9,0xAB,0x74,0xFB,0xBA,0xCE,0xFE,
			0x8B,0xE3,0xF3,0xB9,0xA3,0xEA,0xB6,0xFF,0xE7,0x66,0xF7,0x6D,0xCE,0xAC,0xE8,0xFD,0x27,0x7F,0xBF,0xEA,0x37,0x7A,0xFC,0xBE,0xBD,0xF5,0x3C,0x7F,0x07,0xBB,0xCF,0x7A,0xF7,0xAF,0x5C,0xBE,0x7B,0xFB,0xB9,0xBE,0xB9,0xFA,0xDE,0xC9,0x61,0x7A,0xD3,0xBF,0xDF,0xB6,0x05,0xFF,0xEE,0xD2,0x7B,0xBF,0xE4,0x7C,0xBF,0x9F,0xC1,0xAA,0xFE,0x6F,
			0xFE,0xF7,0x8B,0xF6,0xCF,0x9B,0x37,0xEB,0xED,0xFD,0x3F,0xAF,0x13,0xFB,0xEB,0xFF,0x1C,0xB3,0xAB,0xF7,0xFB,0x58,0xAF,0xEC,0x19,0xE7,0xBE,0xBB,0x5B,0xFB,0xFF,0x7F,0x3A,0x63,0xBF,0xC2,0xBC,0xE4,0xCA,0x75,0x8D,0xDF,0x8B,0xEA,0x29,0xFB,0xED,0xF3,0xEF,0xD0,0x08,0x88,0xB5,0xDB,0xA9,0x95,0xBA,0xFF,0xAC,0xAD,0xBB,0x9F,0xF2,0xDB,
			0x7F,0x3D,0xBF,0xE2,0xEF,0x6E,0x77,0xDF,0x5D,0x6A,0x35,0xC6,0x9D,0xBF,0xEE,0x96,0x9D,0xB9,0xFF,0xFC,0xDD,0x0E,0xF2,0xBF,0x1B,0xBD,0x9F,0x1A,0x6F,0x8A,0xCF,0xDE,0xCA,0xBB,0xA2,0x7A,0x99,0x9F,0xCA,0x4B,0x9F,0xFD,0xCE,0xC6,0xDB,0xEB,0xFD,0x22,0x9B,0xA9,0x5F,0x7B,0xFB,0xB4,0x59,0xFF,0x0F,0xBF,0xFB,0xF7,0x83,0xED,0xFF,0xAF,
			0xB9,0xF6,0xDB,0xB8,0xEC,0xBF,0xFA,0xF7,0xEC,0xF9,0xFF,0xF2,0xE5,0xF5,0xEB,0x9B,0xAF,0xB6,0x3F,0x0A,0xFE,0x1F,0xBE,0x8F,0xA6,0x3E,0x7E,0xBB,0xF3,0x99,0xEC,0xDB,0x86,0xFE,0x6D,0xEB,0x7A,0xDF,0xAB,0xBE,0x1F,0xA7,0xBF,0xEE,0xBF,0xE3,0xFB,0x7A,0xCF,0xB6,0xF3,0x67,0x4B,0x7F,0xE2,0x6F,0xFF,0x87,0x8F,0xFF,0x1E,0xE8,0x1B,0x8F,
			0xED,0xAD,0xBD,0x98,0xFC,0xBF,0x7C,0x4E,0x7E,0xF9,0x2B,0x52,0xEC,0xB4,0xE9,0x2B,0xF3,0x2E,0x69,0xDC,0x6A,0xFF,0x05,0xB9,0x39,0xEE,0x9A,0x7E,0x5F,0xEA,0xDA,0x3A,0xF7,0xE5,0xDA,0xF9,0xF9,0x98,0xDF,0xA9,0xB7,0x4B,0xF3,0xFF,0x37,0xAB,0x7E,0xEC,0xAF,0x87,0xB7,0xA2,0xB3,0xF3,0x4A,0xF1,0xF8,0x5F,0x5B,0xB6,0x3E,0x96,0xFB,0xFE,
			0xBC,0xF8,0x4E,0xDF,0x6D,0xFF,0xCF,0x57,0x77,0x39,0xE0,0x4E,0x27,0x9C,0xFD,0xCB,0x3F,0x7A,0x66,0xCA,0xC9,0xDE,0x5E,0x83,0x3D,0x3E,0xA8,0xDD,0x07,0xEF,0xAE,0x9B,0xEA,0x97,0x8B,0xED,0x6F,0x57,0xF8,0xDE,0xA8,0x55,0x9C,0x9B,0xDC,0xCB,0xF9,0xC5,0x3A,0xEA,0xCB,0x38,0xFA,0xCF,0xDB,0x12,0xDE,0x5F,0xE8,0xF4,0x8A,0xFB,0x91,0x5C,
			0xFB,0xFC,0xAE,0xAF,0xEA,0xE1,0xB7,0xFF,0xC7,0xF9,0xFE,0xF9,0xFC,0x9F,0x6F,0xEA,0xDD,0xCF,0x41,0xFA,0xE2,0x5E,0xEF,0xDF,0x7E,0xED,0xD3,0x27,0xD1,0xEF,0xFB,0x1B,0xB8,0x7B,0x47,0x3B,0xD3,0x67,0x9C,0xCD,0x2E,0xB3,0x5C,0x59,0x2D,0xBF,0xFA,0xEE,0xEF,0xAB,0xFE,0x01,0x8A,0xA7,0xEE,0x9E,0xBF,0xCB,0xDA,0xB2,0xFE,0xAB,0xEF,0x9B,
			0xBB,0xDA,0xBC,0x5A,0xFF,0xBF,0x8B,0xBB,0xC5,0x3F,0xF6,0xBE,0x7A,0xE7,0xBE,0xB3,0xDB,0xA1,0x93,0xEB,0xB3,0xFE,0x0A,0xFB,0x9B,0xFE,0x74,0xE7,0xB3,0x8F,0x76,0xB1,0xF5,0xEB,0xEE,0xBB,0xFB,0xCD,0x0D,0x9D,0x37,0xFF,0xCC,0xEF,0xA7,0xEA,0xBB,0xB3,0xF5,0x7B,0x76,0x70,0xFD,0xE6,0x9E,0xFD,0x5B,0xF9,0x89,0x7A,0x2F,0xCB,0x1B,0xC2,
			0xD9,0x53,0x3F,0xB6,0x9D,0xAF,0x5B,0xBE,0x5F,0xB5,0xB7,0xBE,0xAD,0x2B,0x83,0xAF,0xFE,0x7F,0xF1,0x6B,0xFC,0x53,0x4D,0x7F,0xF9,0x83,0x7A,0xA2,0xEC,0xBC,0x9D,0xED,0x8F,0x35,0xFE,0x31,0xE8,0x4C,0x6D,0x39,0x7E,0x86,0xBB,0x3B,0xE7,0x2E,0xBE,0x67,0xA6,0xBF,0xCB,0xDB,0x7F,0xEE,0xFF,0xE7,0x23,0x6F,0xF7,0xFF,0x2E,0xFF,0xFE,0xF8,
			0xFF,0xF8,0xDB,0xBD,0xFF,0x27,0x76,0xC6,0xBF,0xFB,0x3C,0xBB,0xFB,0x9C,0xF7,0x93,0xBB,0xCF,0xFD,0xC3,0x3E,0xFA,0xEE,0xB2,0xD5,0xBB,0x6F,0x8B,0x1E,0xBF,0xFE,0xFA,0x5F,0xBF,0xEB,0x9F,0xA1,0x23,0xE2,0xBD,0xF7,0x2F,0xCE,0x38,0xB7,0xE6,0x31,0xE9,0xEF,0xFE,0x93,0xBF,0x6B,0x84,0xFF,0xB9,0x0D,0xB7,0x7A,0xF7,0x7D,0x5F,0xD8,0x5B,
			0xDA,0xEF,0xE3,0xCD,0x6F,0x2E,0xF9,0xE2,0xFA,0xB9,0x11,0x1B,0x9B,0x22,0x3D,0xDA,0xE2,0xDD,0xF6,0xEF,0x3B,0xBC,0xFC,0xAE,0x0F,0xE8,0x2A,0x67,0xD9,0xAE,0xF8,0xFF,0xDB,0xFA,0x39,0xEC,0x18,0xC7,0xDF,0x0E,0xF5,0x59,0xFB,0x90,0xAC,0xE7,0xEF,0xDB,0xFE,0x4F,0xCB,0xFB,0xDB,0xBF,0xBF,0xFF,0xB6,0x63,0xEB,0xB2,0x6A,0x7F,0x77,0x1F,
			0xDF,0xFB,0xEF,0xCC,0xC6,0x6A,0x79,0x72,0x1E,0xF3,0xF8,0x72,0xCD,0x3C,0x36,0xF4,0xD3,0xA7,0x9B,0xCB,0xDE,0xE3,0x7A,0xFB,0x2D,0xFF,0x9C,0x4F,0xFF,0xBF,0xB6,0x76,0x9D,0xD7,0x3C,0x95,0x3F,0x99,0x1F,0xEF,0x5F,0xF9,0xF7,0xA6,0xA5,0x5E,0x6F,0xB8,0x38,0x7F,0x7C,0xBA,0xC0,0xCB,0xFB,0xFD,0xFF,0x66,0xF8,0xEF,0xFF,0xB3,0x4A,0xF8,
			0xB5,0xFD,0xCB,0x1D,0xBF,0xCE,0x9E,0x6F,0xBB,0xFB,0xFD,0xA5,0x9C,0xAA,0x9E,0xFB,0x5F,0x9B,0x67,0x7E,0xEF,0x93,0xF4,0xEF,0xC7,0xDB,0xF7,0x1A,0xFA,0x9B,0xAA,0x4B,0xD7,0xBD,0x6B,0xAF,0xFF,0xFB,0xA7,0xF4,0x63,0xDF,0xD5,0x4A,0xBA,0xAB,0xFE,0xBF,0xE7,0xA6,0x9B,0x83,0xAB,0xF3,0xF5,0xCC,0xFD,0xFF,0x3D,0x69,0x9E,0x89,0xDC,0x7B,
			0x98,0xEB,0x3F,0x22,0xDB,0xF3,0x2F,0x05,0xC3,0xFC,0xFB,0xBF,0xCE,0x6B,0x88,0xB2,0xBF,0x5C,0xFE,0xEB,0x6C,0x7F,0x6F,0x7D,0x1B,0x7A,0x6D,0xE1,0xA6,0x2D,0x2F,0x98,0x3F,0xF6,0xA5,0xE7,0x5B,0xB6,0xED,0xB5,0xCE,0x7F,0xC1,0xE5,0x7F,0xF7,0x57,0xB9,0xEA,0xD7,0x55,0xB5,0x5F,0xFF,0xEF,0xFF,0x5F,0xE7,0xAF,0x57,0xD3,0x7F,0xEF,0x65,
			0x3F,0xEF,0x6F,0x7F,0xB8,0xAE,0x51,0xDF,0xBF,0xEB,0xA2,0x1A,0xFA,0xAD,0xFB,0xBB,0xD7,0x7A,0xF3,0x8F,0xFC,0x2F,0xE7,0x9F,0xD2,0xEE,0xFD,0xF6,0xBA,0x3F,0x6B,0x3D,0xFF,0x5E,0xEB,0x62,0xF2,0x45,0x9A,0x9E,0xF8,0x77,0x93,0x9B,0xFE,0xB7,0xF9,0xCD,0x5D,0x2C,0xCF,0xFF,0x97,0xB7,0xFB,0xEE,0x7E,0xDF,0x2E,0xBE,0x5A,0xB7,0xCD,0xCF,
			0x3F,0xBC,0x2E,0xFA,0xCB,0xF8,0xBF,0xF4,0x7F,0xBF,0xAD,0x8B,0x9C,0xB6,0xB2,0x2C,0xF7,0xFE,0xFF,0x27,0xFB,0xB7,0x5F,0xFD,0xBF,0xFE,0xC8,0xF3,0xEF,0x3F,0xAF,0xFE,0x63,0xF5,0x7C,0xFF,0x49,0x9D,0xAF,0x7F,0xC3,0xDF,0x91,0xAE,0xEA,0xAF,0xEB,0xDB,0xE6,0x94,0xE5,0x62,0x6A,0x8A,0x6F,0xDA,0x3B,0x7F,0xF9,0xE9,0x6E,0xFB,0xFA,0xFA,
			0x2F,0xFA,0x48,0x92,0xF7,0x6C,0xA9,0xE2,0x3E,0xBA,0x2F,0xCF,0xE6,0xF1,0x3E,0x7A,0xF6,0x5B,0x8E,0x79,0xB7,0x7B,0xB3,0xEF,0x2A,0xB3,0x0F,0xAF,0xE6,0xCE,0x1F,0xDF,0x86,0xC3,0x5F,0xF4,0xF6,0xFE,0x2C,0xE7,0xAD,0x78,0x1B,0x7C,0x4E,0x92,0xE7,0xBE,0xA7,0xCE,0xB2,0xF6,0x3A,0xF2,0xA7,0xCD,0xBF,0x7C,0x12,0x1D,0xFE,0xBE,0xFE,0xF0,
			0x77,0x5C,0x73,0xFF,0x67,0xFF,0xFB,0xDA,0x37,0xEF,0x99,0xAC,0x9B,0xEA,0xFA,0xFB,0xBF,0xDE,0xFB,0xBE,0x7A,0xFB,0xFD,0xF3,0xBD,0xEF,0xA3,0x36,0x62,0x3A,0xBD,0xCD,0xF7,0xF0,0x69,0xEA,0x4B,0x8B,0xAC,0x6E,0xE7,0xE5,0xBE,0xBA,0xB6,0xCF,0xEA,0xDA,0x47,0x57,0xBE,0xEB,0xAF,0xAF,0x87,0x6D,0xFD,0x7E,0xBB,0xEB,0xEB,0xDD,0xF8,0xF3,
			0xE6,0xEB,0x96,0x38,0xFF,0xF5,0xF7,0x3B,0xFE,0xED,0xBF,0xBF,0xFC,0x92,0xF6,0xA7,0xF6,0xE7,0xFE,0xD7,0xBA,0xFB,0xFE,0xF0,0xF8,0xDF,0xBF,0xF9,0xBB,0xF3,0xEF,0x97,0x7F,0x5D,0x44,0xF7,0xBF,0xE6,0xEB,0xF7,0x6C,0x3E,0x71,0xEA,0xE1,0xA1,0xA0,0x6F,0x6A,0x66,0xB7,0xFF,0xFD,0xEB,0x2E,0xFF,0xF9,0xFF,0xDA,0xD8,0x6E,0xB7,0x3D,0x8C,
			0x7F,0x77,0xFF,0xB8,0x7F,0x7C,0x9D,0xEF,0xEC,0xFE,0x1A,0xEA,0x64,0xEE,0xDF,0x9F,0xBD,0xFE,0x29,0x6A,0x3A,0xB1,0xFC,0xFB,0x39,0xD3,0xEF,0xB7,0x79,0xE7,0x7F,0xD4,0xF4,0xEE,0x8A,0xAF,0xA7,0xAC,0xDA,0xD1,0xEA,0xFA,0xAD,0xAF,0xBF,0x6E,0x7F,0xB9,0xEE,0xB4,0xDF,0xB7,0xBF,0xCA,0xB4,0x76,0xCA,0x23,0xAD,0xBE,0x3B,0x1D,0x7E,0xBD,
			0xE4,0xAF,0x76,0x5F,0x13,0xAF,0xC8,0x8E,0xEA,0xCB,0x97,0x2B,0xA7,0xAD,0x8F,0xFE,0x46,0xEE,0xE1,0xCF,0xBF,0xEB,0x67,0x47,0xE2,0xCF,0xDF,0xED,0xCD,0xEF,0xBE,0x8F,0x08,0xFF,0x73,0xAC,0x0A,0xEF,0x9A,0x3F,0x67,0x2E,0xF2,0xCF,0xFA,0xEF,0xDC,0x2E,0x6D,0xEC,0xF6,0xAB,0x9F,0x6E,0xBC,0x6A,0x4F,0x2E,0x6F,0xFD,0x6F,0x69,0xBE,0x7C,
			0x7B,0xF4,0xE3,0xFB,0x95,0xFE,0xF7,0xA1,0xA9,0x9B,0xBF,0x0A,0x6B,0xE0,0xFF,0xDF,0xC4,0x04,0xAC,0xBB,0xF1,0x2A,0xEE,0xF3,0xD4,0xFF,0xBC,0xAD,0xE7,0xCF,0x66,0xE1,0x9D,0xF7,0xDF,0x97,0xCF,0xEF,0xD3,0xBE,0x8F,0xF3,0xBE,0xBF,0xDF,0x83,0xEF,0xFB,0xF8,0x5F,0x79,0xAF,0xBB,0xBB,0xF8,0xB0,0xDF,0xC7,0xCE,0xEE,0x54,0xF7,0x7F,0x11,
			0xFE,0xF2,0xFB,0xDA,0xBE,0x7B,0xDE,0x5F,0xA7,0xBD,0xAA,0xAF,0xCE,0xCA,0xFB,0x9B,0xA9,0xF9,0x7D,0xBA,0xF2,0xAE,0x7F,0x57,0x9A,0xFF,0xF6,0x2A,0xF2,0xB2,0xE2,0x8D,0x8F,0x7F,0xBF,0xE7,0xCD,0xBF,0x9C,0xFB,0xFB,0xCE,0xDB,0x9E,0xFA,0xFF,0xFA,0xFF,0xF9,0xC6,0xF8,0x2A,0xEE,0x68,0xF6,0x88,0xEA,0x73,0xEB,0x7A,0x6E,0x9F,0xFA,0xAE,
			0x9D,0xB6,0xFF,0xF7,0x2C,0x6B,0xD7,0xF3,0xDF,0xCA,0xF7,0xD7,0xFC,0x7C,0x9C,0xD7,0xF9,0x6B,0x7A,0xFF,0x5F,0x35,0xBC,0x72,0x79,0x7B,0x3B,0x7F,0x75,0x20,0x76,0xA2,0x9D,0x1C,0xB6,0xD4,0xF1,0xD7,0x66,0x7B,0x03,0x3E,0xEC,0xFC,0x8F,0xDE,0x7B,0x8E,0xF3,0xDC,0xAA,0xFF,0xF8,0x17,0x9F,0x7F,0x9A,0x7B,0x38,0xB6,0x72,0xFF,0xFA,0xFF,
			0x53,0xEE,0x0D,0x47,0x9A,0x3F,0xE9,0x77,0xEE,0x7A,0xDF,0xEA,0xFA,0xE6,0x7C,0xDB,0xF7,0xFB,0x8B,0xC7,0xBB,0xEF,0xB9,0xEF,0x51,0x9B,0x9F,0xEA,0x95,0xAE,0x26,0xEB,0x3F,0xB7,0xEF,0xF2,0x91,0x77,0xD3,0xA6,0xEB,0x7D,0xE3,0xF6,0xB4,0xDF,0xEB,0xAE,0xFF,0x6C,0xED,0x6E,0xAE,0xBE,0xFD,0x6F,0xF1,0xF7,0xF2,0x85,0x8E,0xC0,0x3D,0x7F,
			0xFE,0x9C,0xDD,0xDC,0x8D,0x7E,0x36,0x2F,0xA7,0xBF,0xDA,0xEE,0xEF,0xBF,0xBB,0x96,0xFE,0x78,0xD7,0xD5,0x9F,0xAF,0x75,0x87,0x77,0xBF,0xFF,0xF9,0xA9,0x3E,0xEF,0xF7,0x2A,0x7F,0xAF,0xEA,0xEE,0xFF,0xFF,0x7E,0xB6,0x5F,0xFC,0xEA,0x87,0xFF,0x44,0x92,0xAC,0xD5,0x92,0x98,0xAF,0x19,0xCF,0xFB,0xBA,0xC9,0x2A,0x3E,0xBE,0xBF,0x3D,0x9E,
			0xEF,0xFA,0xAB,0x7F,0xEF,0xE7,0xAF,0xFB,0xB9,0x56,0x06,0xFE,0xB8,0xD9,0xAB,0xAF,0xBF,0x2B,0x36,0xF6,0x68,0xBF,0xA9,0xBB,0x3F,0xA6,0xD1,0x8D,0x48,0xBB,0xF3,0x9E,0xFD,0xDF,0xDA,0xF6,0xAF,0xFF,0x9E,0xFD,0x3F,0xCD,0xEF,0xFE,0xC6,0xFB,0x75,0xFB,0xEB,0xEA,0x6F,0xF6,0xF6,0x0F,0xFA,0xE7,0x5D,0xDE,0xEF,0xAF,0xF7,0xAA,0xE5,0xBA,
			0x9B,0x7D,0xB2,0xFB,0xBF,0xBE,0xFE,0xDF,0x7F,0xE1,0x97,0xE3,0xBC,0xFE,0xAD,0xEB,0x7B,0xFB,0xAE,0xD7,0xAF,0x6F,0x90,0x1B,0x67,0xDD,0x47,0x59,0xF9,0xFF,0xFF,0xFA,0x43,0x7B,0x2C,0x9B,0xF2,0x1D,0xAE,0xEF,0xB5,0xBB,0xBD,0xBF,0xFB,0xFA,0xF8,0xFA,0xC2,0xE7,0x1F,0x49,0x5E,0x6C,0xB6,0xA2,0xBE,0xFA,0x39,0xEF,0xED,0x2B,0xF6,0xFB,
			0xCF,0x79,0xFB,0x77,0xFF,0x7E,0x3F,0x66,0x75,0xFB,0x58,0xDF,0x2A,0xBA,0xEF,0x90,0xDB,0xEC,0xAF,0x1F,0xEF,0xEB,0x4D,0x76,0xCD,0xDB,0xBE,0x5F,0xF1,0xFE,0x7D,0x9D,0x97,0x64,0x8D,0x31,0x39,0x7C,0xF5,0x7B,0xFB,0xFE,0xEF,0x4C,0xBE,0xEF,0xCE,0xF7,0x4D,0xEE,0xDB,0xBD,0x67,0xE2,0xAF,0xF6,0xEF,0xFE,0xFE,0xA7,0xF6,0xEF,0xFD,0xEB,
			0xD6,0xBB,0xE2,0xFF,0xF6,0x7F,0xF9,0xBE,0xFE,0xF7,0xB6,0x3E,0xAB,0xE3,0x84,0xF1,0x5C,0x33,0x6F,0x9A,0x5C,0x3A,0xBD,0xBB,0xF3,0xBF,0xF9,0xAF,0x8C,0xA3,0xCA,0xFF,0xFC,0x41,0xE7,0xD7,0x7F,0xCB,0xAF,0x7E,0xB7,0xED,0x7E,0xAE,0xBF,0xE2,0x5B,0xEF,0x12,0xF1,0xFB,0x5F,0x7A,0xAB,0xCE,0xFC,0x3F,0x2F,0xFF,0xFB,0xCA,0xC6,0xB5,0xF7,
			0x3B,0x06,0x68,0xBB,0xAF,0xA6,0x6F,0x9B,0xF7,0xBA,0x7C,0xBF,0x73,0xCD,0x37,0xDE,0xC3,0x83,0xF5,0xF7,0xBB,0xEF,0x3D,0xBF,0xCF,0xDA,0xBE,0x6D,0xFB,0xFF,0x5F,0xE4,0x9E,0x1B,0x9F,0xFF,0x5A,0x3E,0x6F,0xF8,0xFF,0xBF,0x79,0x40,0xF5,0xBB,0x7C,0xFF,0xD4,0xBB,0xB4,0xF3,0x5F,0xFF,0x9F,0xF9,0xFF,0xBF,0x7F,0x16,0xF3,0x3F,0xF8,0x7B,
			0x4B,0xCD,0xB2,0xFF,0x7A,0x77,0xCD,0xCD,0xFE,0x3E,0x4E,0x2B,0xED,0xBF,0xBF,0x3F,0x63,0xCB,0xF6,0x6F,0xFE,0x64,0x6F,0xAB,0xC7,0xE2,0x8E,0xE7,0x9F,0x7F,0x9E,0xBD,0xEF,0xCA,0xB9,0x4B,0xF8,0xEB,0x7B,0x7B,0xE4,0xFB,0x28,0xCE,0x3F,0x0F,0x9E,0x7F,0x8F,0xDB,0xDE,0x28,0xEB,0xC6,0x3F,0xBF,0xAD,0xEE,0xFE,0x7F,0xD5,0xFE,0xF6,0x9D,
			0x9F,0xFF,0x7F,0xFF,0xE5,0x3A,0xD2,0xF7,0xE5,0xEF,0xEF,0xC7,0xF6,0xDD,0xFF,0xFD,0x4E,0x35,0xEB,0x57,0x63,0x6C,0xAB,0xAA,0x64,0xAD,0xFB,0x9F,0x7E,0x9D,0x6E,0x5F,0x20,0x3E,0x6A,0x1F,0xFB,0xCF,0xFC,0xCB,0xB7,0xEA,0xCF,0xF3,0xBF,0x9E,0x97,0xEB,0x34,0x73,0xA9,0xBE,0x6A,0xBF,0x27,0xD6,0x53,0xFE,0xFF,0xF3,0xB7,0xCA,0xFF,0x68,
			0xB0,0x6B,0xBF,0xFF,0x3F,0xEB,0xDF,0xBA,0xF2,0xF8,0x65,0x73,0xFD,0xF9,0xFE,0xFF,0xDE,0xF7,0xBF,0x7A,0x7A,0xFA,0x9F,0xF6,0x9F,0xAF,0x7B,0xBF,0xBE,0x6A,0xBB,0x7B,0xEF,0xF5,0xBB,0xD7,0x7B,0xF2,0xF5,0xC5,0xEF,0xFF,0x9F,0xA7,0xF4,0x22,0xED,0xB6,0xED,0x7E,0xFB,0xAF,0xFE,0x79,0x9E,0x98,0x7F,0xEF,0x97,0x25,0x4A,0xB0,0xEA,0xAA,
			0x7F,0xAF,0x7F,0xA0,0xBF,0xBA,0xFB,0xFB,0xF3,0x95,0xFD,0x3A,0xDB,0x3A,0xF3,0xFF,0xBB,0xFF,0xE8,0xD8,0x9D,0xCD,0xCB,0xB7,0xBD,0xCA,0xAE,0xAE,0xF6,0xFF,0xB5,0x7B,0xDF,0x9C,0xEA,0xB6,0xF7,0x6B,0x77,0xE5,0x27,0xF7,0x62,0xF6,0xFA,0x69,0xD6,0xFB,0x1B,0x1E,0xAE,0xFA,0xD9,0xEB,0x6B,0xF3,0xFE,0xF7,0xF7,0xBA,0x7B,0x3B,0xEE,0xCF,
			0xDC,0xF7,0xFF,0x4B,0xBD,0xFD,0xFF,0xBE,0xBD,0xB6,0x45,0xDC,0xAB,0xBC,0x2F,0xBD,0xB7,0x9D,0x2E,0x9F,0x3B,0x6E,0x79,0xAD,0xBD,0xFF,0x7F,0xFF,0x64,0x9A,0x07,0xFF,0xF5,0xA3,0xBC,0xD7,0x95,0xEF,0xFE,0xBF,0x9E,0x53,0x5F,0xDD,0xDF,0xEA,0xBF,0x3B,0xB6,0x70,0xAA,0xF7,0xDF,0x33,0xE4,0xAD,0x8F,0x4B,0xDE,0x52,0xF3,0xA5,0xF6,0xFF,
			0x73,0x9F,0xEA,0xDB,0xBF,0xED,0x7F,0xBB,0xFD,0x77,0x7E,0xB9,0x83,0xEF,0x92,0xEF,0xE3,0xFB,0x6B,0x36,0x6D,0xFF,0x6F,0x7F,0x7F,0xD7,0xFA,0x7F,0x20,0xFE,0x6F,0x6E,0x76,0xAF,0xE5,0x96,0x65,0xEE,0x1D,0xA3,0x85,0x85,0x2A,0xB7,0x8F,0xFF,0xE9,0x8E,0x79,0xCD,0x29,0xB5,0xF6,0xFB,0xEA,0xBF,0xC7,0xB3,0x7B,0xF7,0xB8,0xC3,0xAB,0x93
			};

			static const UByteSequence sCORVID880Ch1Input720x4868BITYCBCRFrm3F2 =
			{
			0xFF,0xB2,0x10,0xF4,0x00,0x10,0x8C,0xA8,0xA4,0xC4,0x1C,0x90,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x4C,0xFF,0xB2,0x14,0x41,0x01,0x04,0x81,0x06,0x00,0x01,0xCE,0xFF,0xE2,0x1C,0x00,0x00,0xFF,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,0x4A,0x56,0x62,
			0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,0x1D,0x16,0x11,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,0x1D,0x16,0x11,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,0x56,
			0x4A,0x3D,0x31,0x26,0x1D,0x16,0x11,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,0x1D,0x16,0x11,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,0x1D,0x16,0x11,0x10,0x11,0x16,
			0x1D,0x26,0x31,0x3D,0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,0x1D,0x16,0x11,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,0x1D,0x16,0x11,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x24,0x47,0x6A,0x7E,0x7E,0x7E,0x7E,0x00,0xFF,0xE2,0x1C,0x00,0x00,0xFF,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,
			0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x6A,0x47,0x24,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x24,0x47,0x6A,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,
			0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x6A,0x47,0x24,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x24,0x47,0x6A,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x6A,0x47,0x24,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x00,0xFF,0xE2,0x1C,0x00,0x00,0xD2,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x24,0x47,0x6A,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,
			0x7E,0x7E,0x7E,0x7E,0x6A,0x47,0x24,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x00,0x00,0xC9,0x6E,0xB9,0x07,0x7F,0xC3,0x91,0x7D,0x53,0x67,0xB0,0xF7,0x7F,0x7D,0x63,0x3F,0x71,0xF7,0xFC,0xFF,0x5F,0xD6,0xBD,0x8B,0xD7,0x5B,0xD8,0xDF,0xBF,0xD4,0xD4,0x67,0x7F,0x9D,0x11,0xF7,0xCF,0x19,0xF5,0x6F,0x47,0xDF,0xFF,0x7D,0x47,0xC7,0x7E,0xEC,0xFD,0xFB,0x5D,0xD7,0xBD,0xCA,0xFA,0xD9,
			0x53,0xBC,0xCF,0x5E,0xF1,0xFF,0x7F,0x5F,0xEE,0xE9,0xC8,0xE1,0x74,0xC7,0x7C,0xF7,0xFC,0xAC,0x07,0xFB,0x86,0xCF,0x93,0x1F,0xEC,0x23,0xD5,0x43,0xF5,0x1E,0x97,0xEE,0x7F,0x01,0xB1,0xB5,0xF1,0xCF,0x3E,0xAF,0x5C,0x7E,0x37,0xF4,0x56,0xDD,0x9F,0x53,0xCF,0xB5,0xE5,0x35,0x8F,0xBE,0x1D,0xAE,0xFB,0xA4,0x59,0xB7,0xF5,0xBF,0x4F,0x37,
			0x3C,0x7F,0xFF,0x5F,0x47,0xEF,0x79,0xF6,0x5F,0xBE,0xE3,0xD7,0xF4,0x3F,0xF1,0xC7,0x4F,0xBF,0xEE,0xDF,0xC1,0xF1,0x1D,0x9B,0xFA,0xAF,0x59,0x1D,0x33,0x5A,0x0D,0x1F,0xA9,0xD3,0x94,0xFF,0xC4,0x7D,0xDE,0xEB,0xF3,0x7A,0x5C,0x5F,0xF7,0x12,0xA7,0xCF,0xBA,0xFB,0xFF,0x87,0xFE,0xAA,0xDE,0x8F,0x8F,0x12,0x5B,0xEF,0x6F,0x27,0xBF,0xEE,
			0x77,0x9E,0xC3,0xD7,0xCF,0xAE,0xDB,0x09,0x7F,0x7D,0x71,0xEC,0x1F,0xAF,0xA8,0xDF,0xFD,0x3C,0xEE,0xB5,0x69,0x37,0xBF,0xBE,0x97,0xDB,0x0B,0xC7,0x5B,0x8F,0x4E,0x85,0xEF,0xFD,0xFF,0xF3,0xB7,0xDD,0xE5,0x57,0x78,0x76,0x5F,0xD3,0x6D,0xBD,0x73,0xCB,0x6E,0x9D,0xFB,0x9B,0xC6,0xFE,0x55,0x5D,0xEA,0x9F,0x10,0xCF,0x5F,0xFA,0x7E,0x49,
			0x79,0x5B,0x90,0x5F,0xD2,0xD6,0x6F,0xF1,0xFF,0xFF,0xFB,0xEE,0xDF,0x9C,0xE6,0x3A,0xFF,0xE9,0x77,0xE7,0xE7,0x8E,0x7D,0xE7,0xFB,0xFB,0xDF,0xC5,0xF2,0xDD,0xED,0x2F,0x7C,0x7E,0x7B,0xDD,0xD5,0x6F,0x78,0xF9,0x5B,0xE1,0xD7,0xB8,0x5E,0x5F,0xFF,0xF9,0xD0,0x3D,0xCF,0xFF,0x73,0x9F,0xF7,0xBA,0x77,0x2B,0xAF,0xEC,0x7F,0x4D,0x7F,0xCD,
			0x91,0xA8,0xFB,0x35,0xA5,0xF5,0x9F,0x4C,0xD8,0xA5,0xDB,0xE5,0x9F,0x77,0xD9,0x8F,0xC3,0xB0,0xB1,0xCF,0xFD,0xD3,0xBB,0xE5,0xDB,0xD3,0x77,0x9D,0x33,0x65,0xBB,0x9B,0xAF,0xBA,0xF3,0x7F,0x4B,0xAD,0xBE,0x9B,0x27,0x2B,0xD8,0x7E,0x51,0xF0,0xFE,0xB2,0x7F,0xDB,0x7C,0xDD,0xED,0x49,0xB6,0xBF,0xBC,0xC3,0x4F,0xFD,0x9D,0xE3,0xC7,0x36,
			0xEF,0xDD,0xFF,0xE6,0xB7,0xD7,0x1F,0xBF,0x1D,0xF9,0xAB,0xDD,0xFC,0xC7,0xE3,0x27,0x6E,0x05,0xB9,0x77,0x35,0x7F,0xB9,0xDD,0xFE,0xFA,0x5B,0x5B,0xDF,0x57,0x97,0xBB,0x6F,0xEA,0xF1,0xE2,0xBB,0xDF,0xDF,0xFB,0xEB,0x63,0xEF,0xFF,0x9B,0xF7,0xF6,0x39,0x99,0xBD,0xF7,0x67,0xD2,0x79,0xCF,0xDE,0x9F,0x9F,0xFD,0xD7,0x76,0xDF,0xFD,0xE8,
			0xFA,0x9F,0x53,0xF5,0xF7,0x31,0x83,0xE9,0xFF,0x12,0xC7,0x75,0xDF,0xC9,0xFE,0x4C,0x76,0x7D,0xC5,0xE9,0x57,0xE7,0xC9,0x77,0xE4,0xFA,0xFE,0x7F,0xF3,0x45,0x6C,0x68,0x99,0xFE,0x2F,0xCD,0x8E,0xE5,0xD5,0x70,0xF7,0x7F,0xFF,0x79,0x11,0xC4,0x7E,0x76,0xD5,0xEE,0xDE,0xCE,0x94,0xE5,0x7B,0x7C,0x55,0x7F,0xD9,0xFD,0x5D,0xE6,0x6F,0x6A,
			0x54,0x3F,0xEF,0x61,0x5F,0xE7,0xBC,0x69,0x35,0xFF,0xCD,0xF4,0xDF,0x55,0xED,0xED,0xC3,0x5F,0xB4,0xD5,0x9F,0xBA,0x1B,0x2F,0x46,0x1D,0xFD,0x7F,0x6D,0x4E,0x27,0x49,0x3B,0x6A,0x17,0x17,0x87,0x27,0xBE,0xBE,0x1B,0xBB,0x76,0x1B,0xF8,0x46,0xD8,0x2D,0x7B,0xEB,0xDB,0xC5,0x57,0x54,0xFB,0xD3,0xB9,0xFB,0x67,0x8B,0xDF,0xDE,0x9D,0x7B,
			0xFF,0x53,0xF7,0x7D,0x67,0x8F,0x7E,0x1E,0x95,0xEA,0x5D,0x1C,0xE3,0x7F,0x5B,0x21,0xF3,0x7E,0x87,0xFF,0x23,0xDE,0xC9,0x46,0xFC,0xC5,0xF5,0x5B,0xED,0xCB,0xFF,0x7D,0xF3,0xBA,0xDC,0x19,0x5F,0xFD,0x5E,0xE2,0xEF,0x47,0x52,0xBD,0xF3,0x68,0x0F,0xF5,0x79,0xF7,0x7D,0xF9,0xBA,0xE6,0x6C,0x87,0xBB,0xF5,0x57,0xBF,0xEF,0xEF,0x3B,0xF1,
			0xDF,0xDB,0x70,0x77,0x9F,0xFF,0x58,0xAE,0xFB,0x7E,0xF6,0x35,0x5F,0x26,0xB5,0x7A,0xFE,0xF1,0xB1,0x76,0x07,0xB5,0xD3,0x6D,0x5B,0xD5,0xF3,0xF7,0xD1,0x5F,0x3B,0x24,0x7B,0xF7,0xFD,0xDE,0xD3,0x1D,0xFC,0xFC,0x73,0xF3,0x6F,0x11,0x57,0x51,0xFD,0x7E,0xB2,0x75,0x3F,0xDF,0x86,0x5F,0xC6,0xFF,0x36,0x7A,0x91,0xB4,0xDD,0xB2,0xCB,0xF7,
			0x7B,0xFD,0xE9,0xFF,0xBB,0xFE,0xFF,0xBE,0xE1,0xE7,0xBF,0xDF,0xE7,0xFF,0xB5,0xA4,0xCA,0xF0,0xF3,0x6D,0x5D,0xDF,0xF9,0xF4,0xF7,0x73,0x53,0xB9,0x0F,0xA3,0xA3,0xCF,0xB4,0x53,0xE4,0xD8,0xAD,0x37,0xAD,0xBA,0xF6,0x99,0xFB,0xDB,0xBE,0x77,0xFF,0xF5,0x0F,0xB7,0x37,0xCF,0x75,0xDF,0xFB,0xAB,0xDB,0xF0,0x47,0x6B,0xE7,0x77,0x0D,0xA6,
			0xE7,0xBB,0xCA,0xF9,0xDF,0xF3,0xF6,0xA6,0x34,0x34,0x1C,0xF7,0x5F,0x97,0x3B,0xEB,0xDF,0xE0,0xCF,0x0F,0xDF,0x77,0x6E,0xAE,0xCF,0x39,0xBF,0xF7,0xDC,0xFF,0x7B,0xEB,0xFD,0x06,0xEE,0xD0,0xBF,0xF7,0xEA,0xEC,0xDF,0x7F,0xEF,0xC7,0xDA,0xBD,0xEF,0xFF,0xBF,0xFB,0x73,0x5F,0x65,0x57,0xFC,0x47,0x77,0x2B,0x3E,0x7F,0xFD,0xF7,0x55,0xF5,
			0x6D,0xDB,0xFB,0x76,0xE3,0xBD,0xDD,0xF7,0xFF,0xCF,0x7D,0x65,0x15,0x95,0xCC,0x83,0x7D,0xB9,0x6E,0x37,0x77,0xAF,0x59,0x4F,0xBF,0xBA,0x7A,0xCF,0xB6,0x7B,0xF7,0xE3,0xFA,0xF7,0x77,0x5F,0xBC,0xA7,0xDB,0x5E,0xF7,0xBD,0x57,0xDF,0xAD,0x72,0x75,0xDD,0xEC,0xE6,0xDB,0xDB,0x7B,0x7D,0xED,0x9B,0x57,0x98,0x04,0xFF,0xED,0x97,0x77,0xFB,
			0xFF,0x57,0x57,0xF3,0x50,0x1F,0x56,0x4C,0xEB,0x9D,0xBF,0xF5,0x15,0x59,0xF4,0x35,0x7F,0xD7,0x9B,0xD5,0xF5,0x8F,0x58,0xFB,0xF3,0x1A,0x6F,0xCB,0x85,0x37,0xFB,0xBE,0xBF,0xBF,0x95,0x3F,0x3D,0xCF,0x9F,0xFC,0x5D,0xCF,0x39,0x63,0xEF,0xE5,0xFD,0xC3,0xB9,0xEE,0xF5,0xFE,0xEF,0x9F,0xC7,0x5F,0x6D,0x2F,0x77,0x3B,0x83,0x3E,0xEF,0xF5,
			0xF5,0xF9,0x10,0x8F,0xE8,0x67,0x5F,0xEF,0xFF,0xFF,0x34,0xE7,0xDD,0xBB,0xDF,0x8C,0x1D,0xDF,0xD3,0xDA,0x7E,0xA5,0xF3,0x74,0xF9,0xF7,0xAD,0xED,0x17,0xD9,0x3F,0xFF,0xC8,0x2E,0x0A,0xB8,0xDF,0xDF,0xFF,0xFD,0xDE,0xFD,0x77,0x95,0xEE,0xAF,0xD9,0xCA,0xFB,0xFB,0x7F,0xD9,0xD7,0x5F,0xFB,0xFB,0x9C,0xF2,0x78,0xEC,0x77,0xCF,0xF7,0xFF,
			0xEF,0x73,0x5F,0x8E,0xF5,0xF6,0x77,0x27,0xED,0xFA,0xE6,0xFF,0xD5,0x46,0xF6,0xEF,0xED,0xE1,0xD7,0xCF,0x77,0x42,0x5E,0xEB,0x8D,0x7F,0x64,0x9B,0xDC,0xDE,0xDF,0x37,0x96,0xFF,0xD3,0x37,0x15,0xE3,0x3E,0xFD,0x79,0xEF,0xFC,0x3F,0xB7,0xB4,0xED,0xD4,0xAE,0xBF,0x9F,0x16,0x74,0xC8,0xDF,0x5D,0x6F,0xFB,0xF1,0xDD,0xFD,0x19,0x77,0xFF,
			0xBC,0x7A,0x34,0x45,0xDD,0xF3,0xBE,0x7F,0x09,0xCD,0xD9,0x1D,0xBD,0xB1,0x7D,0xED,0x31,0x5E,0x57,0x3F,0xF7,0xBB,0xE7,0xF2,0xB1,0xBE,0x5B,0x77,0xF5,0xFF,0x5D,0x7F,0xB4,0xBF,0x10,0x95,0x8D,0x79,0x5D,0xA6,0xF5,0x54,0xF1,0x6F,0xF7,0xF2,0xFF,0xAE,0xF8,0xDD,0x97,0x43,0x53,0xFC,0x37,0x87,0xE9,0x73,0x75,0x1D,0xB2,0xB7,0x6F,0xA5,
			0x17,0xA4,0xE7,0x4C,0xF9,0x98,0xB7,0x9F,0x6B,0xBC,0xB7,0x75,0x55,0xDF,0x95,0xBB,0x6F,0xFA,0xDF,0x75,0x7F,0xDC,0xDE,0xC5,0x39,0xD4,0x8D,0x95,0x73,0xE6,0x54,0xDB,0x26,0x2D,0xF7,0xEB,0xF6,0xBF,0x07,0x2F,0x6C,0x2D,0xF5,0xD5,0x64,0xD6,0xE7,0xA1,0xB6,0xCF,0x5D,0x49,0xDF,0xBB,0x9C,0xB5,0xA5,0x9F,0x72,0x6D,0xED,0x06,0xDF,0xDD,
			0xFB,0x9F,0x5F,0x5F,0x9D,0xF9,0xCF,0x5E,0xFF,0xAC,0xD4,0x83,0x97,0x56,0x37,0xCB,0xAF,0xC3,0xB7,0xC4,0xF8,0xF2,0xF5,0x7E,0x3D,0x7C,0xF5,0x3D,0xFD,0xD9,0x77,0xAF,0x7E,0xFE,0x1F,0x9D,0xE9,0x79,0x7E,0x7F,0xF5,0xC6,0xF3,0xEF,0x5E,0x50,0xA8,0x5D,0x7E,0x4B,0x03,0xFD,0xFB,0xFD,0xDF,0xE9,0x9F,0x21,0x84,0x1C,0xE6,0x6F,0xBA,0x07,
			0xFE,0xDF,0xDD,0x7D,0x1F,0xF3,0xFD,0x7F,0x7A,0x5B,0x85,0xCD,0x3F,0x1F,0x7B,0x32,0xCA,0xE7,0x6F,0xFF,0x1F,0x90,0xD0,0xF8,0xAB,0x7F,0xBA,0x3B,0xDD,0xAF,0xF7,0xF3,0x7F,0x9C,0x40,0x1C,0x6F,0x7F,0xDF,0xE7,0xDE,0xBA,0x51,0x1D,0xB7,0xFF,0x9F,0x4B,0xDF,0x59,0xC3,0x5F,0xC5,0x5F,0x77,0x0F,0xCD,0x7E,0x17,0x7F,0x6F,0xA1,0x1E,0x0D,
			0xCF,0xE7,0x4F,0x9F,0x5C,0x4E,0xEE,0x5F,0xEF,0x79,0xB9,0xF7,0x9F,0x6A,0xAE,0x42,0x4E,0x89,0xFA,0x37,0x5F,0x7F,0xAA,0x3D,0xAD,0x76,0x91,0x6F,0x39,0x6C,0xEC,0x3F,0x75,0x63,0x9C,0x5E,0xFF,0x7C,0x49,0xA7,0xCC,0x4C,0xD8,0x39,0xF4,0x7F,0xFD,0x1D,0x37,0xDE,0xF3,0x29,0xCD,0x17,0xFB,0x6D,0x1C,0x2F,0x6F,0xF7,0xBF,0x7B,0x73,0x9F,
			0xF2,0x7B,0xE5,0x9F,0xE7,0x5F,0x7B,0xDF,0x72,0xE5,0xFF,0xDE,0x13,0x1D,0x71,0xFF,0xD9,0xE3,0xF7,0x31,0xE5,0xFD,0x7B,0xBF,0xAF,0xF4,0xF5,0xFA,0xC5,0xFC,0xEA,0x77,0xF2,0xDD,0xBC,0xBF,0xCD,0x7E,0x67,0x6F,0x4E,0xF4,0xE5,0xFD,0x5E,0x7E,0xFA,0xF9,0xFC,0xFD,0xFF,0xE9,0xED,0x94,0xB7,0x85,0x6F,0xB8,0xE7,0xF8,0xE2,0x9B,0xDF,0xFF,
			0xC5,0xE8,0x7E,0xF7,0xDD,0xCF,0x1F,0xD9,0xDF,0xFD,0x77,0x24,0xDF,0x2C,0xE5,0x9D,0x7F,0xFB,0xFF,0xE5,0x75,0xC0,0xBF,0xEF,0xF5,0xED,0x95,0x7B,0xE3,0x54,0xE7,0xFF,0xFE,0x37,0xFB,0x6D,0xF7,0x7D,0xAE,0xFF,0xFF,0xB3,0x1D,0xEF,0xFC,0x75,0xB7,0xB5,0xBB,0x6F,0xD5,0x76,0xF6,0xF5,0xF7,0x7C,0x5D,0xCB,0xB7,0x37,0xFC,0x3F,0xF3,0xBD,
			0x75,0x6E,0x71,0xB4,0xFC,0xDA,0x3C,0xF2,0x35,0xFC,0xF5,0xE7,0xDD,0xFB,0xF9,0x9A,0xFE,0xEE,0x74,0x13,0xFF,0xD7,0x74,0xD1,0x1C,0x2B,0xB7,0xE4,0x6E,0x74,0x7D,0x6F,0xB1,0xEF,0xDC,0xD5,0xD9,0x6A,0x9B,0xFE,0x33,0xF7,0x29,0xC5,0xB5,0x16,0x7F,0xBC,0xF7,0xEF,0x7D,0xB7,0x7F,0xFC,0xFB,0xB2,0xFD,0x47,0xFF,0xA5,0xD4,0xDD,0x53,0xE9,
			0x7D,0x65,0x73,0xDF,0xF9,0xDD,0x3F,0x7E,0xFF,0x5E,0x67,0x79,0x7D,0x4F,0xB3,0xEE,0x8E,0xB6,0xFB,0x3F,0xF5,0xFF,0xDF,0x5F,0x25,0x75,0xBF,0xF7,0x75,0xF7,0x76,0xFD,0x13,0xDF,0x9F,0xCB,0xE7,0xCE,0xAC,0xEB,0x75,0x7D,0xE1,0xF7,0xDF,0xF5,0x5B,0x7B,0xB7,0x7B,0xD4,0x5D,0xFC,0x7D,0x0D,0xD7,0x7F,0xFD,0xD6,0xC1,0xF9,0x37,0x97,0xD9,
			0x57,0x9F,0xDF,0xFF,0xF6,0xF5,0xF6,0xB3,0x7F,0x7F,0xDF,0xF4,0x5E,0xDA,0x7E,0xF9,0x95,0xFD,0xEE,0x78,0x67,0xF7,0xF2,0x7E,0xBF,0xDF,0x0B,0xFD,0xAE,0x0D,0xED,0xC7,0x63,0x87,0x31,0xFD,0x74,0x9C,0x57,0x8D,0xF8,0xE2,0x5F,0x56,0x1F,0xBF,0x73,0xAD,0xFF,0xFF,0xEF,0x8C,0xD7,0xFF,0xC2,0x3F,0xF6,0xB2,0x4D,0xC4,0x89,0xFD,0xC7,0xC8,
			0x7D,0xE2,0xB7,0x9D,0xFD,0xF9,0xEF,0xC7,0xEB,0xBA,0xFB,0xD9,0x7C,0x57,0xF5,0xFD,0x53,0xFC,0xFC,0x3D,0xEF,0x7D,0xE5,0xD3,0xCB,0xFF,0xA5,0x57,0x7A,0xE9,0x2F,0xBB,0x9B,0xDB,0xDC,0x3F,0x3D,0x8F,0xD6,0x3C,0x93,0x9C,0x77,0x71,0xC7,0xEB,0xFA,0x21,0x2B,0x7F,0xFC,0xB7,0xD5,0xFA,0xFF,0x4D,0xDB,0xFF,0xFE,0x9B,0xC7,0xCF,0x1E,0xF8,
			0x3E,0x35,0x95,0xEF,0xF9,0xFF,0xDF,0xF9,0xDF,0xF6,0xA7,0xDC,0xC1,0xD6,0x7F,0xB2,0xD7,0xF2,0xEB,0x4D,0xB9,0xDA,0x7D,0xBB,0x9F,0xDF,0x13,0xCF,0xB9,0x6F,0x8D,0xBF,0xCD,0x72,0x73,0xF7,0x61,0xAE,0xEF,0x7D,0x0E,0xFF,0xCF,0x7B,0xAD,0x7D,0xF7,0x8E,0xFE,0xBF,0x90,0x6B,0xC4,0x05,0x75,0x5F,0x7F,0xFF,0xBE,0x6F,0xFF,0xB6,0x8C,0x71,
			0xDF,0xFD,0x6F,0xB0,0x53,0x67,0xE0,0xAF,0xBF,0xF3,0x17,0xD1,0x15,0xDF,0x7C,0xFF,0xAF,0xE1,0x6F,0xEF,0x14,0xFD,0xFE,0xEC,0x77,0xF3,0x45,0x52,0x50,0x9A,0xCB,0xFF,0xF7,0x9A,0x18,0x74,0xF2,0xF7,0xB1,0xFF,0xAF,0xFD,0xED,0xEF,0xC9,0xCF,0x5F,0xE7,0x0F,0x20,0xDD,0x7D,0xDD,0xAE,0x34,0x6A,0x7C,0xFF,0xC1,0xFF,0x2F,0xF3,0x3F,0xDE,
			0xCD,0x3E,0xC7,0x99,0x6B,0x7B,0xFB,0x39,0xC1,0x29,0x3F,0x7D,0xFE,0xFD,0xDF,0xE1,0x0F,0xFE,0xA7,0x77,0xF7,0x65,0xFF,0x33,0x7F,0xB7,0x63,0xC6,0xE9,0x9D,0x63,0xB7,0xAF,0x7F,0xED,0x3B,0x69,0xFE,0x65,0xE8,0x3D,0xFE,0x05,0x32,0xF7,0x15,0xBD,0xCD,0xEE,0x65,0x3D,0x26,0xEF,0x35,0xFF,0x2E,0xD3,0x0A,0xFF,0xB6,0x9F,0x58,0xD8,0x73,
			0xF6,0xFC,0x1B,0xD5,0x9D,0xD9,0x7F,0xF2,0xFB,0xED,0xFC,0xFF,0x7E,0x7F,0xD1,0x73,0xED,0xFC,0xF9,0x7E,0xFB,0x5F,0xF7,0xF7,0xFD,0x2C,0xBC,0xBD,0x7C,0xDA,0xEF,0x67,0x6B,0xF7,0x03,0x9F,0xFE,0x9E,0x1F,0x5D,0x3F,0x98,0xBF,0xFF,0xF7,0xBD,0x5D,0x8F,0xAB,0x71,0x75,0xBF,0xFC,0x9D,0xE5,0xB7,0x7D,0xFF,0xFE,0x97,0xDF,0xF6,0x76,0x97,
			0xDE,0x0F,0xDF,0x91,0x4D,0x1D,0x7E,0xC7,0x29,0x73,0x7D,0xD7,0xB7,0xDD,0xFF,0xFB,0x2F,0xEB,0x53,0x1F,0xF7,0x4B,0xBB,0xF8,0xBA,0x76,0x62,0xD7,0xFB,0x8E,0x8F,0xDB,0x6E,0xDD,0xEE,0xFF,0xF4,0x3D,0xFD,0xFF,0x6C,0xD3,0x59,0xCD,0xFC,0xDF,0x2E,0x01,0x7D,0xFE,0xDB,0x7E,0xD1,0x59,0xFF,0x7F,0x59,0xBE,0x7B,0xFA,0x3C,0xF6,0xE6,0x7A,
			0xFF,0xDD,0x05,0xD7,0x5F,0x94,0x79,0xEB,0x44,0x6F,0x73,0xFD,0xCD,0x6F,0xE5,0x2F,0x8E,0x4F,0xCF,0xD6,0x8B,0xDB,0xF2,0xFA,0x66,0x8F,0x7E,0x0D,0xBA,0x2F,0xCB,0x5E,0xE4,0xC7,0xC2,0x5F,0x4F,0x3F,0x47,0xF5,0xE3,0x76,0x77,0xC4,0x20,0xDD,0xBB,0xEE,0xFF,0xD6,0xFF,0x3D,0x7B,0xF9,0xFF,0x67,0x96,0x24,0x5F,0x75,0xF7,0xD1,0xD6,0xBA,
			0x36,0xBF,0xFA,0xFB,0x5F,0x37,0xFB,0xFE,0x6F,0xFB,0x6B,0x7C,0x5F,0xCF,0xBD,0x5B,0xFF,0xDE,0xBA,0xB7,0xF4,0x17,0x3F,0x50,0x3F,0xFE,0xDD,0xFC,0xFF,0x95,0xEF,0xF6,0xB1,0xDE,0x2F,0x39,0x9D,0x50,0x48,0x7B,0xDB,0xDD,0xFE,0x9F,0x1D,0xF5,0xB7,0xE7,0xD7,0xE3,0xAB,0x55,0x54,0xD9,0x24,0x7F,0xFF,0x44,0xBD,0x7F,0xBF,0x3F,0xEF,0x6E,
			0xE6,0xB7,0xF7,0x73,0x1F,0x57,0xD7,0xFF,0xFA,0xEE,0xDB,0xD7,0xE9,0x76,0xBA,0xCB,0x2F,0x87,0xD6,0xD1,0xF7,0x97,0x7F,0xEE,0x6B,0xC9,0x7B,0xD9,0xBD,0xFF,0x32,0xE5,0xDB,0xEE,0xE7,0xF8,0xF4,0x99,0xDB,0xCB,0x57,0xDD,0xF0,0xBB,0xBF,0xFC,0xEF,0xDB,0xD9,0xF7,0x59,0x5C,0x7B,0xA4,0x0F,0x0E,0xBF,0x92,0xA4,0xBF,0xEF,0xF1,0xC7,0x6A,
			0x70,0xF7,0x5D,0xF7,0xE5,0x78,0x77,0x95,0xF5,0x3F,0x65,0xD3,0x45,0x56,0xF9,0x7D,0x66,0xF2,0x48,0xBE,0xF8,0xD5,0xF5,0x92,0x7F,0x70,0x67,0xF3,0xFA,0x96,0xFB,0x6F,0x5B,0xE4,0xA2,0xF5,0x3F,0xF0,0x85,0xEE,0xE5,0x5B,0xE9,0xEA,0xF4,0xFF,0xF4,0xFD,0x9F,0x73,0x41,0x76,0x58,0xFE,0xF9,0xFF,0x77,0xFB,0x7D,0x54,0xF7,0xCE,0xAA,0x76,
			0xDA,0x77,0xF6,0x73,0xFE,0x37,0xFD,0x6E,0xFF,0x6B,0xC9,0xDB,0xFD,0x5B,0xD4,0xFB,0xB6,0xE3,0x5C,0xF7,0x55,0xFF,0xF5,0xA9,0xB7,0xF3,0xF5,0x37,0xDF,0xB3,0x71,0xFD,0x9F,0x6F,0xEF,0x11,0x73,0xFF,0xFF,0x1E,0x7E,0xFF,0x53,0xF7,0xD6,0x5F,0xFE,0xAE,0xFF,0x6D,0x5B,0x5F,0xB5,0xB5,0xFE,0x39,0xDF,0x5F,0xAF,0x97,0xD8,0xFF,0x47,0xFF,
			0x96,0xAE,0xFD,0xAF,0xB9,0xF1,0x16,0x36,0xDE,0xBD,0xFF,0xA7,0x7F,0x6F,0x5C,0x7B,0xC5,0x11,0xD7,0xEF,0x07,0x48,0xDF,0xB3,0xFF,0xBF,0xE9,0x7B,0x45,0x1F,0x79,0x6D,0xFE,0xA3,0x5F,0xF3,0x79,0x9D,0xF7,0xE2,0xF7,0xFF,0x5F,0x6C,0xD7,0xDD,0x5E,0x7D,0x6F,0xA9,0x4C,0xD3,0xF9,0x93,0xBD,0xF0,0x7A,0xF6,0x5E,0x79,0x77,0xF5,0xFB,0x3D,
			0x3F,0x77,0x77,0x7E,0xE7,0xD5,0xA7,0xAD,0x3D,0xFF,0xFF,0xFE,0x3F,0x70,0x9B,0x6F,0x9E,0xCF,0x3D,0xF7,0x3E,0x77,0xF3,0xF2,0x7E,0xDB,0xDA,0xF7,0xFF,0xE5,0xE3,0xB2,0x6F,0xCA,0xFC,0x2D,0x77,0xCF,0x57,0xAF,0xC5,0xD6,0x3D,0xD9,0x0F,0x0F,0x1B,0xCD,0x2F,0x97,0x6D,0x41,0x17,0xDB,0x1B,0xE6,0xCB,0xEF,0x5F,0x55,0x5E,0xF7,0x8C,0xED,
			0xBF,0xED,0xDF,0x73,0x35,0xF1,0xFA,0x4E,0x57,0xFE,0xCE,0x76,0xDF,0xEC,0xC1,0xB9,0x7F,0xFF,0x65,0xD6,0xFB,0x77,0xC7,0xDB,0xFB,0x6F,0xDC,0xC1,0x55,0x65,0x6D,0xE9,0xFD,0xFE,0xAF,0xE3,0x6B,0xDF,0x7F,0xFF,0x9D,0xEF,0xFF,0x79,0xFD,0xBE,0xDD,0xDB,0xF5,0xE7,0x7D,0xFC,0xF5,0xCF,0x5E,0xBB,0xFB,0xE7,0xEF,0x96,0x4F,0x68,0x6E,0xAD,
			0x1B,0x7A,0x9B,0x1D,0xF5,0xFB,0xFD,0xBE,0x72,0xA8,0xA4,0xFB,0xD5,0x7F,0xD5,0xFE,0xDE,0xF4,0xCD,0xFD,0x07,0x75,0xE9,0xD7,0x63,0x7F,0xF7,0x97,0xC5,0x1B,0xFE,0xBF,0xF4,0x97,0x92,0xFB,0x67,0x30,0x4D,0xCB,0x5A,0xE3,0xCB,0x75,0x6F,0x4C,0xF3,0xFB,0xB9,0xD7,0x6F,0xDF,0x7B,0xFD,0xED,0x76,0xED,0xE1,0x94,0xCD,0x1B,0xD9,0x93,0xD3,
			0x7F,0xB6,0xA9,0xEC,0xF4,0x75,0xEF,0xF8,0xBA,0xFD,0xFF,0x9B,0xC5,0xBF,0xD5,0xFD,0xFD,0x9B,0x37,0xD5,0x57,0xFF,0xFF,0xFD,0xFF,0xBB,0xFF,0x33,0xAB,0x52,0xB1,0xF1,0xFD,0xFF,0x58,0xBC,0xD7,0xA3,0xA7,0xBF,0x6F,0x7F,0x67,0x1D,0x17,0xF7,0xE7,0x77,0xD3,0x49,0x0D,0x7F,0xB5,0xA5,0x7A,0xF4,0xF7,0xEE,0x55,0xF6,0xBC,0xBB,0xFF,0x26,
			0x5D,0xFE,0x1F,0xFD,0x06,0xBD,0xF5,0xBF,0x2E,0x2F,0x7E,0xFD,0xDF,0xDF,0x7D,0xBD,0xFD,0x24,0xDD,0xB4,0xB7,0xFF,0x5E,0x35,0xFE,0xB7,0xF5,0x67,0xC2,0x77,0xBC,0xB6,0xFF,0x8E,0xFD,0xC5,0x9F,0xFD,0x85,0xF6,0xF7,0x7D,0xF1,0x7E,0xB7,0xD1,0x95,0xF3,0x77,0x88,0x3F,0x36,0x7B,0x55,0x2F,0xC7,0xED,0xF7,0xFD,0x15,0xB5,0x57,0xD7,0xF7,
			0x35,0x8B,0xEB,0x5E,0x5F,0x75,0x36,0x1E,0xFF,0xB1,0xE9,0xDD,0xDD,0x5F,0x5A,0xEB,0x6F,0xA6,0xD5,0xEE,0xFF,0xFD,0x5E,0x0C,0x77,0xDF,0xFD,0x87,0xFA,0xCF,0x46,0x77,0x3F,0x4B,0x05,0x4A,0x71,0xF5,0x33,0xB6,0xFB,0x79,0xFF,0xF0,0xAD,0x55,0x57,0x6B,0xED,0xFB,0x65,0x75,0xFD,0xD3,0x7F,0x78,0xD7,0x3F,0x2E,0x3D,0x7E,0x9A,0xFF,0x3B,
			0xFB,0xBF,0xEE,0x5E,0xC7,0x9D,0x9F,0xC6,0xDF,0x7F,0xFB,0xB0,0xEC,0xAC,0x23,0xDD,0xD9,0x33,0x47,0xEB,0xEB,0x4C,0xAF,0x37,0x8E,0xFF,0xE7,0xBB,0xFF,0x6B,0xF2,0xDF,0x5F,0xE8,0x66,0x31,0x73,0x67,0xAF,0x0F,0xED,0xFF,0xD5,0x5A,0xF8,0x60,0x7F,0xF4,0xB0,0x89,0x3D,0x1F,0x45,0xEC,0xBB,0x37,0xFC,0xE8,0xEE,0x5F,0xF6,0x48,0xB9,0x57,
			0x3E,0x98,0x7F,0x0C,0xFF,0x89,0x5C,0x2F,0x1D,0xE7,0x7B,0x7C,0x1E,0xF7,0x73,0xFB,0xF9,0xEF,0xFB,0xFC,0x1D,0x6E,0x7D,0xE7,0xDF,0x55,0x6D,0xC3,0xF7,0xEE,0x77,0xF7,0x5F,0x3A,0x4C,0x9D,0x4D,0x8F,0x6B,0x7E,0xCF,0xBD,0xF5,0xFD,0xEF,0x59,0x62,0x3D,0xEA,0x6E,0x6F,0xCC,0x2F,0xDD,0xFF,0x7D,0xF9,0x89,0xEC,0x3D,0x6F,0x3D,0xBF,0x67,
			0x56,0x2F,0xFF,0x25,0x3D,0x48,0x6B,0x7B,0x3D,0x1B,0x69,0xF5,0xD6,0x1C,0xEC,0xEC,0xFA,0x2F,0xDD,0x31,0xB7,0x9D,0xEC,0xCF,0x37,0xB7,0x5F,0xFD,0xBF,0x31,0xB7,0xFC,0xF3,0x97,0xBE,0xD3,0x7F,0xFF,0x53,0xFA,0xB1,0xF4,0xF7,0xFF,0x72,0xFD,0xE3,0x93,0x5F,0xEF,0x3D,0x59,0xDB,0xD5,0x8B,0xF6,0xDC,0xDE,0x7F,0xF5,0xE8,0xFE,0xFF,0x0D,
			0xCE,0xC3,0x6C,0x74,0xFB,0xBE,0xC5,0xCF,0xF0,0x47,0x2F,0x95,0xBA,0xBF,0xFB,0x7E,0xAF,0x93,0xB5,0xBF,0x81,0x3F,0xCE,0x4B,0xFE,0x1A,0x9F,0xB4,0xEE,0x3F,0xD3,0xFE,0xBE,0x2C,0x79,0x6F,0xD7,0xE7,0x33,0x3F,0x5E,0x85,0xBC,0xFE,0xF3,0xDD,0xA5,0xA3,0xEB,0x47,0x1C,0xCD,0x87,0xF3,0xB7,0xE3,0xFF,0x7A,0xCC,0xBF,0xFA,0xBD,0xEE,0xF6,
			0x8C,0x7D,0x66,0x3F,0x7C,0x59,0xD6,0xFD,0xBF,0xBB,0xDF,0xBF,0x5F,0xEF,0xCC,0x6D,0xFD,0x3D,0xFD,0xDF,0x7E,0x9D,0xE7,0xF6,0x9F,0xBD,0xF3,0x77,0xF5,0x9E,0x8D,0xD6,0x3F,0x12,0xDA,0xD7,0xF5,0xFF,0x0F,0xFC,0x67,0xE4,0xFF,0x9F,0xE5,0x17,0xF7,0x5F,0x3D,0x7F,0xAF,0xA5,0xF7,0x9E,0xFF,0x4C,0xF7,0x23,0xF3,0xD4,0xBA,0x0E,0xF8,0x3F,
			0xED,0xED,0x3D,0x1F,0xB6,0x9C,0x73,0xEC,0x67,0xF8,0xC7,0xD7,0xF8,0xF9,0x75,0xFB,0xB7,0x35,0x9E,0x9A,0xD3,0xBF,0xD5,0x2B,0x37,0xDC,0xFF,0x7E,0xD6,0x6E,0x6E,0x17,0xDB,0x7D,0xD3,0x5E,0xDE,0xFB,0x67,0x3C,0x76,0xE8,0xFF,0x7F,0x57,0x2F,0x87,0x5E,0xDF,0xFB,0x57,0xFB,0xEB,0xFB,0xB8,0x6F,0xFB,0x38,0xDF,0x35,0x87,0xB3,0x5E,0x7C,
			0x7F,0xED,0x3F,0xFC,0xE9,0xBF,0xBC,0x87,0x1B,0x3B,0xB2,0x59,0xFF,0x52,0xF9,0xDD,0xEB,0x27,0xB7,0x1F,0x8F,0x5E,0x2F,0xBF,0xFF,0xBA,0x7F,0xF9,0xD9,0x0E,0xF7,0xF6,0x42,0xDF,0x8F,0xB7,0x6D,0x9F,0xFE,0x5B,0xFE,0xE3,0x27,0x55,0xFD,0xE4,0xED,0x1E,0x57,0x7D,0xBF,0xDF,0xF5,0xD3,0x3B,0xFD,0x73,0x3F,0xDD,0xF7,0xFF,0x7D,0xBB,0x5F,
			0xEA,0xFA,0xE6,0x58,0xAF,0x75,0xDD,0xDE,0x9B,0x5D,0x4F,0x52,0x52,0x79,0xCF,0xFD,0xDF,0xFA,0x65,0x33,0xFD,0xEC,0xB9,0xAD,0x19,0xC5,0x74,0x5F,0x97,0x3D,0xEE,0xEF,0xFF,0xA5,0x75,0xF7,0x55,0x77,0x99,0xB2,0x77,0x75,0xD6,0x56,0xFF,0x5B,0x9C,0xF7,0x5B,0x45,0xDD,0x7E,0x9F,0xD5,0x6D,0x7D,0x97,0x5B,0xB7,0xF6,0xAF,0xFB,0x37,0x7D,
			0xDD,0xA8,0xFF,0x97,0x67,0xAB,0xBD,0xF8,0xEF,0xB3,0xA9,0xED,0x46,0xBC,0x04,0xDA,0xFC,0xFF,0xBD,0xF8,0x0A,0xD7,0x24,0xDF,0xDE,0x55,0x26,0xDD,0x37,0x79,0xDD,0xEE,0x6F,0xFB,0xDC,0xF5,0x5A,0xB4,0xAB,0xA2,0xF5,0x8E,0x75,0x1A,0x3B,0x2A,0xE5,0x38,0xCF,0xCB,0x6E,0xE1,0x4C,0xF7,0xB4,0xDD,0x57,0xEB,0x79,0xFF,0x3C,0x62,0xD5,0xCD,
			0x7B,0x7F,0xE2,0x41,0xF5,0xF9,0xE7,0x7F,0xF6,0xDD,0xCD,0xF7,0x0D,0x70,0xCD,0x6D,0xD3,0xFD,0xFC,0x55,0x5C,0xD9,0xEA,0x9D,0xB2,0xF7,0xD6,0xF9,0x5F,0x47,0xF0,0xEE,0xFF,0x53,0xDE,0x37,0xED,0xE5,0x8A,0x7F,0xE9,0xFE,0xFE,0xC7,0xFF,0x64,0xA3,0x97,0x66,0x7F,0xF8,0x7D,0x5F,0xAC,0xAE,0x7B,0xDA,0x3C,0x5D,0xD3,0xDA,0x96,0x94,0xEE,
			0xD5,0x19,0x38,0xFD,0xF5,0x7B,0xBB,0x72,0xFD,0xEA,0xDF,0xC5,0xF5,0x7E,0xFD,0xFF,0x4C,0xFE,0x64,0xEF,0x55,0xDD,0x71,0xB8,0xA5,0x69,0xED,0xB3,0xD5,0x5E,0x7B,0xEF,0x51,0xF9,0x7B,0x75,0xCE,0xFE,0xFF,0xB5,0xDF,0x2C,0x1E,0xD3,0x9F,0xAD,0xD3,0xE5,0x0D,0xFE,0xFE,0x4B,0xCD,0x76,0xE7,0x4F,0xB7,0xFF,0x1F,0x49,0xAD,0xFF,0x74,0xDD,
			0xF0,0xFE,0xB1,0xCD,0x59,0x22,0xB9,0x11,0x73,0xCF,0xF4,0x43,0xEC,0x54,0x7F,0x92,0x2C,0x3B,0x6D,0xA9,0xCF,0xF6,0x39,0x4B,0xE2,0xEF,0xA6,0x71,0xF4,0xDC,0xDD,0xB3,0x6D,0x67,0xDF,0xED,0x5F,0x9D,0xBE,0xAF,0xFA,0x7E,0x34,0xCC,0x55,0xED,0xF7,0x97,0x96,0xF4,0xDF,0x6D,0xCF,0x9D,0x3A,0xCD,0xD2,0xEF,0x7C,0x6D,0x9D,0x65,0xBD,0xD6,
			0xC8,0x5C,0xD7,0x7D,0x52,0xA7,0xE4,0xEA,0xED,0x4F,0x12,0xEF,0xAD,0xDE,0x7A,0xDF,0xFE,0x46,0xD5,0x5F,0x97,0x6A,0x7F,0x97,0x27,0x17,0xF2,0x5F,0x77,0x1E,0xC2,0xFD,0xEF,0xFF,0xFB,0x9C,0xDB,0xBB,0xFB,0xAF,0xF4,0x5F,0x9B,0x16,0xE2,0x54,0xAB,0x7A,0x9F,0xEE,0xD5,0x5D,0x4A,0xDB,0x76,0x6E,0xFD,0xB2,0xFC,0xEF,0x8C,0xDF,0xF8,0xDF,
			0x3A,0xDF,0xFE,0x53,0x5F,0x5D,0x5A,0xB7,0x72,0xB7,0xC0,0xFE,0xDC,0xDE,0xBE,0x3F,0xBE,0xFF,0xFA,0x7F,0xFB,0x97,0xBB,0x9F,0xED,0xFB,0x7A,0xEA,0x5F,0x97,0x6E,0xFB,0x9F,0x7B,0x77,0x33,0xFF,0x0F,0xAF,0xFF,0xBC,0xEE,0x75,0x1D,0xFA,0xAB,0xFE,0xFE,0x7B,0xCC,0x73,0xC7,0xBF,0xB9,0xFF,0x96,0xFD,0x36,0x33,0x4D,0x51,0x9D,0x6E,0xEF,
			0xEF,0xA9,0x2F,0xE1,0x9E,0x5F,0xEC,0x9C,0x6D,0xEF,0x27,0xEB,0x75,0xF1,0xF9,0xF0,0xF9,0xAB,0xF5,0xDF,0x5B,0x3B,0xED,0xB2,0xDD,0x83,0xE6,0xE3,0xD7,0x3B,0xFF,0xFB,0xFD,0xD6,0x67,0xBD,0xAF,0xDB,0xBB,0xAB,0xBD,0xAE,0x70,0xD8,0xBF,0xD9,0x9C,0x9F,0x7B,0x3F,0x5F,0x79,0x01,0x6F,0x8A,0xF3,0xFB,0xF7,0x7B,0xF1,0x7D,0x55,0x1D,0x3A,
			0xFD,0xE9,0xFD,0x75,0xC5,0x1E,0xAA,0xB7,0xFB,0xEF,0xFB,0x67,0xFE,0x79,0xD8,0x72,0xFC,0x4B,0xB7,0xED,0xBC,0xFD,0x7E,0x66,0xFB,0xF7,0x66,0x17,0x7E,0xD7,0xF2,0x23,0x67,0xCA,0xE7,0xFF,0xFA,0x26,0x4F,0xF3,0xB9,0xFF,0xE7,0x7F,0xCD,0x9E,0xF7,0x6E,0xDF,0xCF,0x16,0xDF,0xBB,0xC7,0xDB,0xE5,0xD7,0xF7,0xED,0x9F,0xFF,0x4C,0xFE,0xDE,
			0xF5,0x1B,0x6C,0x7B,0xFA,0x0E,0xFA,0xD3,0xCD,0xFE,0xFD,0x97,0x77,0xCC,0x97,0xBE,0x3F,0x18,0x61,0x33,0xC7,0x37,0x65,0x4F,0xDF,0xF8,0xF7,0x79,0xBF,0xCD,0xD4,0xA7,0x59,0xDB,0x9D,0xD0,0x7F,0xD5,0x1D,0xFB,0x9B,0xFF,0xFD,0x5D,0x77,0x53,0xD3,0x5F,0xEF,0xBF,0xE5,0x5C,0x9B,0x75,0x37,0xBE,0xDE,0xED,0x1F,0xBD,0x57,0x5F,0x87,0x7D,
			0xBE,0xFE,0xCF,0xB7,0xD3,0x02,0x0F,0xCF,0x7A,0x9B,0xD3,0x94,0xD4,0x75,0xFD,0xFF,0x3B,0xCE,0xFC,0x5D,0xFD,0xFB,0x7D,0xDE,0xCF,0xEB,0x5F,0xAB,0xFD,0x3E,0x49,0xFD,0xF7,0x76,0xBF,0xF6,0xBF,0x17,0xAF,0xB7,0xED,0x03,0x9C,0xAE,0x3F,0xE5,0xEF,0xD4,0x77,0x1B,0xEF,0x0E,0xAF,0xEC,0xEB,0x5F,0xAB,0xBF,0x0F,0xF1,0x64,0x8F,0xDA,0xCB,
			0x7D,0xF7,0x8F,0xA9,0x2F,0xE7,0x5D,0xDD,0x7F,0x26,0x7D,0x7E,0x55,0xAD,0x67,0xE5,0xDF,0x9B,0xEF,0xBE,0xBF,0xDF,0xD2,0x3E,0xCD,0xBF,0xD5,0x47,0x94,0xD9,0x63,0xFF,0x77,0x9F,0x7F,0xDF,0xFC,0x7D,0x6B,0x7F,0xFD,0x7D,0xC6,0xD1,0xB7,0x86,0xD7,0xEB,0xFE,0x7F,0xF7,0x95,0xFF,0xF4,0xC0,0x4F,0x9F,0xC5,0x7A,0xE9,0xB1,0x69,0xFC,0xDB,
			0x9D,0xB9,0xED,0xE7,0xED,0xF9,0xFB,0xEE,0x0C,0x1D,0x15,0xD7,0x79,0xBD,0x9C,0xDF,0xEB,0xBA,0xB4,0xF9,0xEF,0xBD,0xC5,0xCF,0xDC,0xA7,0x33,0x1F,0x4D,0xDF,0x46,0x4E,0x1B,0x9B,0xE7,0xBF,0xF5,0xCF,0xFE,0xE3,0xA3,0xE2,0xB1,0xB9,0x3B,0xF2,0xBE,0x7E,0xD1,0x7E,0x6F,0xE0,0x35,0xD7,0xFF,0x78,0xF7,0x35,0x95,0xE6,0x6A,0xFD,0xDB,0xFE,
			0xED,0xFF,0xFF,0xAC,0xF5,0x1D,0x73,0x7E,0x94,0x7F,0x5E,0x3D,0xDF,0x5F,0x3A,0xB7,0xF1,0x2F,0x97,0xF1,0x7F,0x5E,0xFE,0xD5,0xC9,0x58,0x77,0xFF,0x57,0x9F,0x3E,0xB8,0x59,0x7F,0xD1,0xEF,0x39,0xBD,0x7E,0x45,0x6D,0xDD,0xBB,0x66,0xDF,0xFE,0x71,0xB1,0x2B,0xEF,0x57,0xDD,0xFE,0xFB,0xF2,0x8F,0x75,0xF3,0x94,0x72,0x7E,0xE7,0xFF,0xFF,
			0xDD,0x9B,0x9F,0x9F,0xBB,0xD5,0xDF,0xB3,0x36,0x7E,0x31,0xB9,0xC9,0xDF,0x57,0xDA,0xD6,0x7B,0xBF,0x76,0x51,0xE5,0xB7,0x7F,0x65,0xD6,0x1D,0xFB,0x27,0xFC,0xFE,0x67,0xDB,0xDA,0xD7,0x83,0xBB,0xBF,0xC4,0xF7,0xFD,0xFD,0x77,0x25,0xB9,0xF9,0xFB,0x79,0xF7,0x97,0x57,0x57,0xAF,0xBF,0x79,0x56,0xED,0xDA,0xFC,0xAF,0xE5,0xB3,0xF3,0xDB,
			0xD7,0xD5,0xF9,0xC6,0x3D,0xC7,0x55,0x1C,0xDC,0x38,0xA3,0xD6,0x07,0x5E,0x73,0xEB,0x17,0xDB,0xFE,0xFC,0xC7,0xD7,0x5E,0x35,0xDD,0x33,0x4E,0x7B,0x15,0x3C,0xFE,0x89,0xBF,0x4D,0x4D,0xFE,0xA9,0xD6,0x3B,0xED,0xF5,0xCD,0xAD,0xE8,0x7A,0x91,0xF6,0xFF,0x55,0xCD,0x17,0xFD,0x67,0xAC,0xEA,0x9F,0xF5,0x3B,0xF3,0xE7,0x3D,0x75,0xA3,0xEC,
			0x72,0x8F,0xF4,0xFE,0xD6,0x07,0x7D,0x7B,0xDF,0xAD,0xFB,0xD5,0xFE,0x7E,0x71,0x3B,0x7F,0xE9,0x5F,0x7D,0x7E,0xF2,0x5F,0x7F,0x7F,0xBB,0x8F,0x58,0x7D,0xEB,0xE5,0x94,0x5F,0x75,0x7E,0x67,0x1D,0x84,0x6F,0xD3,0x38,0xFB,0x72,0x65,0x3B,0xE7,0xF3,0xFD,0xFC,0xE1,0xFE,0xCF,0x35,0x73,0x9B,0x30,0xFD,0xFF,0xD3,0xFF,0xD5,0xFD,0x77,0xBF,
			0xD7,0x5A,0x6E,0x7F,0x33,0x67,0x5B,0xEF,0x5F,0xFF,0xD8,0xD3,0xCF,0x3D,0x6B,0x63,0x27,0xF2,0x77,0x6D,0xD7,0x78,0xFB,0xFE,0x2F,0xCF,0xF7,0x6E,0xF7,0x31,0x61,0xF6,0xBF,0x73,0xFF,0x19,0xEB,0xDC,0xEE,0x9B,0xCF,0x29,0x17,0xD4,0x4E,0x5F,0xE9,0xBF,0xCF,0x67,0x7F,0xDD,0x75,0x6F,0xE7,0x9F,0xFE,0x7E,0xFF,0x3E,0x4F,0xF0,0xE9,0x31,
			0x1E,0xDD,0xBF,0x61,0xDE,0x78,0xDF,0x8E,0x9C,0x03,0x7F,0x56,0x5D,0x4A,0xFE,0x71,0xFF,0x47,0x97,0x71,0xF7,0x33,0xF5,0xA2,0xD2,0x7F,0x7F,0xD6,0x5F,0x6F,0xE7,0x29,0xEA,0xD7,0x03,0x60,0xD3,0x95,0x7D,0x2E,0xBC,0xDE,0xB8,0x6A,0xDB,0xE6,0xF5,0xCE,0xDC,0xA6,0xBF,0xAA,0x5F,0xFC,0x3B,0x4D,0xDD,0xFA,0xFB,0xA3,0xEB,0x6E,0xF4,0x7B,
			0xF5,0xE3,0xBF,0xDD,0xE9,0x3E,0xFF,0x7E,0x95,0x7E,0x16,0xAC,0xC7,0xCF,0xB6,0xC4,0xD2,0x6D,0x1F,0xFF,0xFF,0x7F,0x89,0xB5,0xEE,0xDB,0xF5,0xE9,0x1B,0xAF,0xF4,0x59,0x5C,0xBC,0x85,0x3D,0x3F,0xB3,0xFF,0xF1,0x7A,0xFF,0xC7,0xF5,0x7E,0xA7,0x6D,0xFA,0xC2,0x63,0x4D,0xBD,0xBF,0xFB,0xFF,0xA8,0xEE,0xFB,0x5E,0x59,0xFB,0x3B,0x7B,0x35,
			0x95,0xAF,0xDB,0xE3,0x59,0x5D,0xEC,0x96,0x5F,0xFF,0xF9,0x06,0x5D,0xD3,0xBF,0x7F,0xEE,0x4D,0x95,0xDC,0x5D,0xB7,0xE9,0x67,0xF9,0xFB,0xF0,0x4D,0x7A,0xF5,0x48,0xDF,0xE9,0xED,0x9F,0xFF,0xB5,0x1F,0x64,0xAB,0x9F,0x61,0xFD,0xD5,0x9F,0x57,0x2F,0x2E,0x6D,0xCE,0xFE,0xF5,0xD9,0x5F,0x2D,0xB3,0x5E,0xBF,0x5D,0xE5,0x43,0xF7,0x2F,0xFD,
			0xD7,0xDB,0xF5,0xD5,0x43,0x2E,0x1D,0xDC,0xEA,0x7F,0x9D,0xEF,0x77,0x2F,0xC3,0xDF,0xCF,0xC3,0x3C,0xB7,0x3F,0x95,0xDC,0xD7,0xA0,0x7F,0xF7,0xFE,0x7D,0x36,0x25,0xF7,0xDB,0x9D,0xF9,0x89,0xCB,0x06,0xFB,0xF7,0x7F,0x5F,0xC7,0xDB,0xDC,0xDE,0x37,0x6F,0xED,0x77,0xF5,0xE9,0x85,0x6F,0x1F,0xF6,0xD1,0xFF,0xFD,0xD9,0xEE,0x53,0x15,0xCD,
			0x98,0xE3,0xF5,0xD7,0xF5,0xD5,0x91,0xBD,0x3F,0xC7,0x55,0x4B,0x7F,0xFE,0xBD,0x73,0x2F,0xEE,0xF7,0xF1,0x77,0xD6,0x59,0x9F,0x8E,0xEB,0x6F,0xDF,0x6B,0x7F,0xDF,0xA7,0xCF,0xA7,0x62,0xA4,0xD0,0x5F,0xDB,0x83,0x5F,0xC5,0xED,0xB9,0x5F,0x3E,0xDB,0x0B,0x1E,0xBD,0x7F,0x6A,0xC7,0x9F,0xF3,0x9B,0xAB,0xFB,0xB7,0x7D,0x45,0x23,0xD3,0xFD,
			0xEE,0xC9,0xDD,0xFD,0xF7,0xB0,0x64,0xE7,0xCF,0xEF,0x1E,0x7E,0x5F,0x1F,0x5D,0xEB,0xCD,0xD2,0x98,0xFD,0xD3,0x9D,0x7F,0xF7,0xFD,0xBB,0x33,0xBF,0x7F,0x2A,0x75,0xE7,0xE3,0xF5,0xCC,0xDB,0xAF,0x4F,0xF7,0xB6,0x1F,0xFD,0x27,0x59,0x3D,0x67,0x97,0x55,0xFF,0x75,0xE5,0xFF,0xEF,0xDF,0x8F,0x13,0xFD,0x77,0x8F,0x7F,0xFD,0xC5,0x79,0x5F,
			0xDF,0x1C,0xD7,0xBC,0x17,0xB7,0xDB,0xDE,0x19,0x3E,0xCF,0xBD,0xF5,0xDF,0xDB,0x76,0x85,0xEF,0x7F,0xE5,0xD4,0xED,0xDF,0x62,0xE7,0xDF,0xDF,0xBF,0xF5,0xF7,0xE7,0xCE,0x55,0xF2,0x77,0x5F,0xE9,0x7B,0x59,0x27,0xE6,0x9E,0xFF,0xA3,0x32,0x39,0xD4,0xE7,0xEB,0xFF,0xDF,0x5F,0xA9,0x5D,0xEC,0x3D,0xD8,0xDF,0x86,0xD0,0x53,0xCD,0xF8,0xF4,
			0xAF,0x7F,0x3F,0xD9,0x5B,0x1B,0xBD,0xBF,0x6E,0x2F,0x69,0x3D,0xBE,0x4F,0xB0,0xFF,0xCB,0x77,0x97,0x3D,0xDD,0xFF,0xCD,0x5D,0x97,0x99,0xDF,0x47,0xC5,0x6F,0xFF,0x7E,0xB8,0xBE,0xC7,0xE7,0x6A,0x35,0x9F,0xAB,0xFF,0x2B,0x2C,0xAD,0xBD,0x5F,0x3C,0x6F,0xFB,0x8B,0xB3,0xED,0xE9,0x58,0x1A,0x6D,0xDB,0x56,0x55,0xB1,0xF3,0x79,0x8F,0x57,
			0xC7,0xDF,0xFD,0xEE,0xBF,0xDF,0x7F,0xCB,0xDF,0xF3,0xE2,0xED,0x3E,0x5F,0x1C,0xFF,0xC1,0x33,0x90,0x4D,0x5D,0xDF,0xDD,0xE6,0xE6,0xED,0xF9,0xDF,0xFC,0x32,0xB6,0xDA,0xFF,0xFF,0xB7,0xA7,0xFC,0x97,0x67,0x15,0x3C,0xC5,0x7E,0xDC,0xF9,0x07,0xDB,0xD6,0x3C,0xAE,0x1D,0x08,0x5D,0x27,0xED,0x18,0xDA,0xFF,0xFF,0xB7,0xFF,0x43,0xE5,0xDF,
			0xFC,0x4F,0x3F,0xD3,0x1F,0xDB,0x64,0x7F,0x3F,0x9B,0xFF,0xBA,0xF4,0xD1,0xE3,0x7F,0xF7,0x7F,0xFA,0xD9,0xFE,0x6B,0xC7,0xFC,0xDB,0x4F,0xBF,0xD6,0x5D,0xA4,0xF6,0xF1,0x91,0x4F,0xBE,0xD7,0x7D,0xB6,0x07,0xF2,0x51,0x9A,0xD9,0x5B,0x5D,0x65,0x77,0xA9,0xB9,0x77,0xFE,0x9F,0x7C,0xA1,0x9F,0xFB,0x68,0x0F,0xC9,0xBD,0x55,0x97,0xCB,0xFB,
			0xBE,0x38,0xCA,0xDF,0xE3,0x14,0x77,0x76,0x97,0x7F,0x3E,0x75,0x3D,0xF6,0xEE,0x75,0x6A,0x44,0xE2,0x96,0xE1,0xDE,0xCB,0x4F,0x33,0x73,0x2E,0x7F,0x7E,0xB6,0x08,0x6F,0xBC,0xBF,0xFD,0x5D,0xFB,0xD3,0x67,0xDF,0xCF,0x7B,0x31,0x1D,0x1E,0x54,0xC5,0x5D,0xFA,0x3A,0x73,0x5C,0x69,0xD7,0x7D,0xBA,0xEB,0xFB,0x5D,0x68,0xDC,0x65,0xEC,0x77,
			0xD6,0xFF,0x3E,0xF7,0x5D,0xAF,0x8B,0xA1,0xFF,0x8F,0x7D,0xBE,0x5D,0xA1,0x23,0xBD,0x76,0xAE,0x2B,0xF7,0x37,0x7F,0xDE,0xF1,0xBE,0xBF,0x6C,0x91,0xFF,0xD8,0xA7,0xFC,0xED,0xFE,0xF6,0x8E,0xCF,0x58,0xAA,0xF1,0x4A,0x7D,0xB4,0xDF,0xFF,0x51,0xD4,0xDF,0x0D,0xBF,0xFD,0xD6,0xD7,0x51,0x6D,0x2F,0xFF,0x7F,0xDE,0x59,0xE7,0xD7,0xFF,0x6D,
			0x9E,0x13,0xDE,0xFB,0x7E,0xDF,0xFF,0xFB,0xD3,0xBF,0xFE,0x55,0x47,0x3E,0xF8,0xE7,0xF9,0xBF,0x36,0x7D,0x77,0x8F,0xA2,0x7E,0xBE,0xD4,0x7B,0x4C,0xDF,0xF4,0xFE,0xFA,0x5D,0xCF,0xBF,0xFA,0x5F,0x8F,0x7F,0x76,0xBF,0xA7,0xC5,0xFE,0xDF,0x44,0xF1,0xD1,0x72,0xD9,0xFD,0xF9,0x27,0xFD,0xFF,0xF7,0x8F,0x75,0xC4,0xDD,0xB2,0xFD,0xFB,0xC5,
			0xBF,0xEE,0xD6,0xF5,0x68,0xE7,0xA6,0xF4,0xFF,0x8F,0xAF,0xFD,0xF3,0xBD,0x9C,0x23,0xE7,0x69,0x57,0x6F,0xB9,0xFD,0xFF,0x75,0xFF,0x99,0x35,0xDC,0x57,0xDF,0xFC,0xA4,0x58,0xFA,0xEA,0x6D,0xAD,0x51,0xF1,0x20,0xF9,0x38,0xE6,0xDF,0xF6,0xF3,0x7C,0x2E,0x8F,0xFF,0x1F,0x57,0x7F,0x51,0x54,0x7F,0x9B,0x9A,0xCF,0xFD,0x73,0x2B,0xF7,0xEF,
			0xBA,0xAF,0x7B,0x3B,0xEE,0x8D,0xE6,0xEE,0x06,0x77,0x47,0x4F,0xEF,0xDD,0x83,0xFF,0xF5,0xDE,0x77,0xC3,0x2C,0xFB,0xCC,0x77,0x0C,0x78,0x37,0xA7,0x57,0x2D,0xBF,0xFB,0xC0,0x09,0x37,0xB9,0xF5,0x73,0xFE,0xCA,0x7F,0x7A,0x65,0x7F,0x27,0x4F,0xA7,0x6F,0x6F,0x1D,0xF7,0x6C,0xF5,0xDB,0xED,0x7B,0xCE,0xFF,0xE7,0xF1,0xFD,0xDB,0xF7,0x6F,
			0xEE,0xEE,0xFD,0xFF,0xDD,0x7D,0xFF,0x9C,0xAE,0xE8,0xFE,0xD3,0xDB,0xDC,0x6A,0x75,0x1B,0xDB,0xAB,0xE6,0xE5,0xF5,0xBF,0x26,0x6D,0xFF,0x9D,0x41,0xB5,0xED,0x3E,0x66,0xF3,0xFF,0xD1,0x37,0x7F,0x1F,0xF7,0x3E,0x7F,0xDB,0x7B,0xEF,0x7B,0xD5,0xDF,0xDF,0x73,0xD4,0xFE,0x56,0xFD,0x3F,0xDA,0xD2,0xFD,0xEC,0x5F,0x5D,0x76,0x5D,0x6B,0xBB,
			0xC3,0xF7,0x2F,0x14,0x3C,0x76,0xD2,0x9F,0xEF,0xAA,0xF6,0x75,0x4F,0x1F,0x5D,0xF7,0x6E,0x19,0xAE,0xDB,0xAE,0xF5,0x41,0x54,0xEE,0xFB,0xFD,0xE7,0xDD,0xF1,0xEF,0xF7,0xD5,0x77,0xAF,0x7D,0x6D,0xEF,0xAE,0x9F,0x4B,0xFE,0x96,0x4D,0x51,0xFF,0x5B,0xF4,0x3E,0xFD,0xDD,0xD7,0xFF,0x7E,0x7F,0xDF,0x71,0x7F,0x0F,0xBE,0x55,0xFE,0x3D,0x3D,
			0x2D,0x6B,0x53,0xC6,0x57,0xCB,0xF3,0xAB,0xC1,0x49,0xF8,0xD5,0xDD,0xD5,0xDA,0x8D,0xE3,0xFE,0x15,0xDF,0x9F,0x37,0x7A,0xA9,0xAD,0x15,0x6B,0x77,0x5E,0x58,0xFE,0xF3,0xF9,0xDF,0xF7,0x57,0xF7,0xEF,0x7A,0xF7,0xD3,0xEF,0xDF,0xDF,0x17,0xD7,0xF1,0xF7,0xBD,0xFF,0xDF,0x3E,0x99,0xD9,0x9E,0xC7,0xFF,0xF3,0xF3,0x5F,0x46,0xD7,0xCD,0xE6,
			0x7B,0x7F,0xFE,0xFD,0x8F,0xFF,0xEF,0xB4,0x77,0xEA,0xCF,0xFB,0x52,0x6F,0xE3,0x57,0xF7,0x5D,0xD7,0xED,0xD7,0xEE,0x45,0x9F,0xDB,0x4F,0xF1,0xED,0x2F,0x95,0xFF,0xDD,0xF6,0xDF,0xFD,0x4D,0xEF,0xFF,0xEE,0xD4,0xDC,0xFD,0xBD,0x7F,0x7C,0x4D,0x7F,0xFF,0xD2,0x3A,0xFF,0x77,0x17,0x1D,0xFF,0xD5,0x5F,0xBC,0xE7,0x47,0xF7,0x7F,0xFF,0x87,
			0xFB,0x1D,0xC7,0xFF,0x44,0x7D,0xAF,0xBA,0x9D,0xB3,0xB5,0xE5,0xCF,0xF7,0xDB,0xBD,0x79,0xFD,0x45,0xD5,0x67,0x54,0x9D,0x7E,0xF4,0xEB,0x7C,0xEE,0xF7,0xFF,0xD4,0xFF,0xDF,0xEE,0xB7,0x77,0x66,0x59,0xEE,0xA7,0x4E,0xB5,0xDD,0xB7,0xEB,0xFD,0xBF,0x2D,0xE0,0xF5,0xF6,0x76,0xDD,0x7F,0xE1,0xAA,0xE9,0xB9,0xF6,0x47,0x65,0xF6,0xEE,0xEF,
			0xEC,0xEB,0xFF,0x74,0xD7,0x75,0xD2,0x15,0xDF,0xB7,0xF4,0xDD,0x95,0xF1,0xB0,0x08,0xC7,0x67,0x75,0xF4,0x3E,0xED,0xFC,0xED,0xCB,0xFF,0x74,0x7B,0x9B,0x07,0xFF,0x72,0x17,0xF9,0x6D,0x7D,0x66,0xF3,0x67,0xB4,0x6B,0xFE,0xFE,0xCD,0x33,0x36,0x03,0xFE,0x23,0x1F,0xCF,0x77,0x35,0x2F,0x6F,0xEE,0xFF,0x7D,0xB5,0xAD,0xF3,0x7E,0x35,0xF7,
			0x0F,0xAF,0xDC,0xDD,0xDF,0x6D,0xE2,0xBE,0xF1,0xCD,0xBC,0xCF,0x57,0x39,0xDF,0xA5,0xAC,0x7D,0x45,0xBF,0xE1,0xDC,0xDC,0xFF,0xED,0xEF,0x67,0x5F,0xFB,0xFD,0x3F,0xFB,0x5E,0xDE,0x26,0x77,0x35,0x3D,0x7B,0xC7,0xD9,0x71,0xCF,0x61,0xDF,0x57,0xFD,0x6F,0x74,0xF3,0x1E,0x37,0x77,0x5D,0xB5,0xAC,0x5D,0x71,0x7D,0x7E,0xF9,0x7D,0xE9,0x7E,
			0x49,0xC7,0xF3,0x8F,0x3F,0xF3,0x5D,0x2D,0xDF,0x6B,0x59,0x55,0x11,0xDF,0xF1,0x76,0x7F,0xBD,0xF7,0xEF,0x7D,0xAD,0xFF,0xC9,0x7D,0x31,0xFB,0xCF,0xC7,0xF7,0xAE,0xF5,0x7F,0x7B,0x96,0x44,0x5E,0x7F,0x6D,0xCD,0xE9,0xEF,0xDB,0xFD,0x35,0x41,0x57,0xFF,0xF8,0xFB,0x1D,0x50,0x97,0x1B,0x5F,0x99,0x9B,0x57,0xF2,0x19,0xBF,0x07,0xE7,0x32,
			0xEE,0xAF,0x7D,0x9B,0xDF,0xD3,0xCE,0xF3,0x75,0x35,0x5A,0x94,0xD9,0xD0,0x3F,0xD7,0xF6,0xE4,0xFB,0x5B,0x69,0x8F,0x4F,0x75,0xB3,0xEC,0xAD,0xF7,0xFF,0xD1,0xF7,0x47,0x4F,0xEF,0xFC,0xDE,0x93,0xA7,0x7F,0xA6,0x94,0x9F,0xE9,0xF1,0x73,0xDF,0x13,0x3F,0xBF,0xEA,0x5D,0x37,0x7D,0xF5,0x3D,0xF0,0x88,0xC4,0x55,0xCD,0x4A,0xAF,0xF2,0x9F,
			0xFA,0xDA,0xC9,0xC5,0x2F,0x13,0xA0,0xD9,0x8F,0xCB,0xAF,0xBF,0x56,0x56,0x92,0x53,0xDF,0xB7,0xEC,0xF7,0xDF,0xF3,0x37,0x77,0xFD,0x15,0xD5,0x7D,0xB6,0x71,0xD7,0x6F,0x77,0xFE,0xDC,0xDF,0x6F,0x75,0xD5,0xF1,0x56,0xF6,0xBF,0xC9,0x63,0x72,0xDF,0xF4,0xD5,0x57,0xF7,0x5F,0x77,0xC3,0x15,0xE1,0x1F,0x3D,0x37,0xDF,0xDF,0x73,0xF7,0x78,
			0x8E,0xF9,0xFB,0x25,0xE9,0xCB,0xFF,0x67,0xDE,0x63,0xFF,0xDB,0xCE,0x55,0xF7,0x7F,0x45,0x71,0xEC,0xD5,0xCB,0x17,0xFD,0xCA,0xDF,0xD9,0xFE,0xBF,0x07,0x38,0xB7,0xFE,0xE9,0x7F,0xFC,0xF4,0x36,0x7D,0xE3,0x87,0x60,0xBC,0xF3,0x3F,0xFF,0x2E,0xCF,0x5B,0xBB,0xE6,0x5D,0x47,0xEF,0x7B,0xFC,0xE2,0xEF,0xB9,0x6A,0xDB,0xDB,0xD9,0xFB,0x73,
			0x4F,0x37,0xB5,0x7E,0xD8,0xA7,0xF4,0x2E,0xB0,0xC8,0x57,0x6F,0x6C,0xF5,0x33,0x57,0x95,0xB5,0x47,0x6F,0xF7,0xA7,0xA6,0x59,0xF1,0xD4,0x7C,0xD9,0x5D,0xFB,0x59,0xCE,0x3E,0xE8,0xF1,0x61,0xFF,0x5B,0xFE,0xD7,0xEE,0xBB,0xEF,0xFD,0xF3,0xEF,0x55,0x76,0x6F,0xE5,0xBD,0x4C,0x2D,0x97,0xF7,0x9F,0x76,0xBF,0xBD,0xE6,0x77,0x78,0xFB,0x75,
			0x69,0xF7,0x7A,0xFD,0xDF,0xBB,0xF5,0x6B,0x5A,0x79,0xDA,0xD8,0x45,0xFE,0x9E,0x7B,0x9F,0x1F,0xFB,0x87,0xFF,0xFD,0x6F,0xF9,0xB7,0xFF,0xF3,0x19,0x66,0xDF,0xD7,0xBD,0xC6,0xCB,0x52,0xFE,0x05,0xDF,0xB4,0xDF,0x58,0xF7,0xFD,0xEF,0x5E,0xF3,0xFB,0x6B,0xBE,0x87,0xFC,0xD9,0x47,0x5D,0xEF,0xBC,0xFA,0x92,0x25,0xB3,0xAF,0xA6,0x7C,0x57,
			0x0D,0x67,0xEE,0x1E,0x7F,0x97,0xEF,0xDD,0xD7,0x7F,0xFB,0xF2,0x57,0x5F,0x9A,0x77,0x1F,0xFF,0x6B,0xEE,0x79,0x5C,0x5A,0x2E,0x17,0x3F,0xCB,0x5D,0x8B,0x0B,0x7B,0x5D,0x97,0xF6,0xFE,0xD6,0xBA,0xD1,0xFD,0x73,0xF8,0xFF,0xEA,0xFF,0x41,0xF3,0xB1,0xFA,0x77,0xFD,0x9E,0x76,0xC8,0xDB,0x9E,0xFA,0x52,0xBF,0x8F,0x1B,0x4B,0x30,0x5F,0x51,
			0xCD,0xDA,0x7F,0x8D,0x71,0x7A,0x6B,0xBB,0xCD,0x9F,0xD7,0x57,0x73,0xC0,0xFD,0x93,0xAF,0xFB,0xE4,0x5F,0x35,0x60,0xEF,0x73,0xEA,0x2F,0xEB,0xDF,0xFD,0x7F,0xB3,0x6E,0x6F,0x5B,0xA7,0x4C,0x1D,0x26,0xD7,0xF7,0x9E,0xDA,0xEF,0xFD,0x55,0xED,0xF9,0xBC,0xEA,0x5B,0xE8,0xB5,0xFB,0xBF,0xEB,0xC6,0xAA,0xEF,0x7D,0xFF,0x25,0x39,0x1B,0xDF,
			0xAF,0xD5,0x9D,0x55,0xE1,0x7E,0xFB,0xFC,0xFA,0xFE,0xBF,0xDF,0xCF,0x73,0xBE,0xBE,0xBC,0xCF,0xFA,0x7D,0xD1,0xFF,0xBB,0xAF,0x65,0xDF,0xFF,0x79,0xCD,0xF9,0xA6,0x3F,0x06,0x7A,0xF7,0x92,0xDF,0x55,0xBD,0xC3,0x27,0xEF,0xF1,0x4F,0xAF,0xF6,0xDE,0xEF,0x66,0xBE,0x99,0x9F,0xE7,0xD3,0x7F,0xF7,0x9B,0xD7,0x9D,0x76,0x97,0xC3,0x7C,0x7C,
			0xB0,0xEF,0xB6,0xA7,0xFB,0x7D,0xFB,0xDF,0x46,0x87,0xBD,0xCD,0x72,0xDF,0xEB,0x3E,0x6B,0xFB,0xD1,0xC3,0xF7,0x3B,0xC5,0xD5,0xFD,0xF9,0xAC,0xBD,0xB7,0x9D,0xBF,0x7B,0x8A,0xFB,0xD2,0x93,0xFF,0xCF,0xD5,0xE5,0xE6,0xAD,0xD5,0xFC,0xCE,0xF6,0x9F,0xF6,0xBD,0x1F,0x56,0xF4,0xFF,0xD7,0xC7,0xF4,0xF6,0xE7,0xEE,0x77,0xFD,0x55,0xEF,0xD7,
			0x3B,0x6F,0xCB,0xAD,0xED,0xB9,0x89,0xF7,0xFF,0x77,0xF7,0xEF,0x1F,0x1E,0x6D,0xF5,0x3F,0xEF,0xF7,0x79,0x55,0xFF,0x83,0xB0,0xFB,0xCF,0x9C,0xB1,0xDB,0xFF,0x6F,0x6F,0xF5,0xA1,0x0C,0xEF,0x75,0xBE,0x94,0xF2,0xCE,0x88,0x0E,0xD9,0x36,0x3F,0xEF,0xEF,0xBB,0xEF,0xF6,0x4F,0xFC,0xB4,0xEE,0xFE,0xBE,0xFE,0xF4,0xFC,0xF8,0x8D,0x9F,0xF7,
			0x8F,0xDE,0xA7,0xC1,0xE1,0xF3,0xE4,0xBE,0x73,0xD5,0xDA,0x4D,0x33,0x77,0x77,0xBB,0xFF,0x57,0xCA,0xCD,0xFD,0xD5,0xF5,0xD8,0xF7,0x89,0xCD,0x6F,0xD7,0x31,0x1F,0xDB,0xFB,0x7E,0xDE,0x78,0xD5,0x5A,0x67,0xFB,0xFE,0x6F,0x30,0x7C,0x9F,0x7D,0xE6,0x7F,0xF8,0xFB,0xE7,0x55,0xD4,0xF7,0xF7,0x7B,0x3F,0x7F,0x77,0x08,0xFD,0xB4,0x4F,0xE9,
			0xD7,0xB2,0x36,0x55,0xEE,0xDF,0xF3,0xD8,0xCD,0xDF,0xC5,0xFC,0x59,0x6D,0xEF,0xBC,0xFA,0x7E,0x78,0xEB,0x7B,0x5E,0x7F,0x47,0xE6,0xBE,0x5B,0xBE,0x8A,0x7B,0x7B,0xF8,0xAB,0x25,0xF5,0x9C,0x77,0xE4,0x6D,0xB7,0x3D,0x83,0xFF,0x57,0x77,0xB5,0x4F,0xFD,0xAE,0x9E,0x2D,0xD7,0xF1,0x15,0xB6,0xAB,0x74,0x3B,0xF1,0xED,0x7D,0x87,0x8A,0xD6,
			0xDE,0xE5,0x5D,0xDB,0xFF,0xF8,0x75,0x5E,0x6D,0xFF,0x10,0x53,0x67,0x07,0xD6,0xFF,0xBA,0xF9,0xDF,0x09,0x76,0x7D,0xD3,0xE3,0xFF,0x8F,0x2B,0xCD,0xFF,0xD9,0xC6,0xBE,0xCB,0x6F,0xFB,0x05,0x92,0xDD,0x3F,0xFE,0x93,0x7F,0xD9,0xF5,0x4D,0xEF,0x77,0x77,0xE7,0xDE,0x9B,0x5F,0xD7,0x69,0xBE,0x74,0x7B,0xFF,0x5D,0xFE,0x75,0xA7,0xBF,0xF5,
			0x77,0xC8,0x99,0x77,0xD1,0x46,0x78,0x6E,0xDE,0x7E,0xD9,0xD7,0x5F,0x9F,0xEE,0xA2,0xFF,0xCF,0x5F,0xF5,0x91,0xD8,0x97,0xC5,0x54,0xE5,0x15,0x75,0x6F,0xE1,0xED,0xD6,0xB7,0x4D,0x12,0xDF,0xD5,0xFE,0x6C,0xF1,0xF9,0xA6,0x71,0xDF,0x57,0x77,0x7D,0xFD,0xDF,0x25,0xFE,0x6F,0xC0,0xDF,0xEB,0xFD,0xF3,0xDA,0x6D,0x7B,0xB7,0x3A,0x17,0x7D,
			0xBF,0x6F,0xEE,0xFF,0x07,0xFF,0xE7,0xFB,0xFF,0x6E,0xFC,0xDD,0xD5,0xF7,0xFD,0x2E,0x67,0xF3,0xD7,0xFF,0xF6,0xAF,0x6D,0xDF,0x9B,0x79,0xDD,0xF3,0xDF,0xE3,0xDB,0xEC,0x5E,0x75,0x88,0xEF,0xC4,0xFC,0x47,0x58,0xFF,0x27,0x93,0x74,0x7F,0x09,0xEC,0x32,0xFD,0x6A,0x1E,0xAF,0xFF,0xFD,0x5E,0x7C,0x7F,0xD4,0xF6,0x7E,0x7C,0x7E,0xF1,0xBE,
			0xFE,0xEA,0xF2,0x65,0x63,0xE5,0x47,0xBB,0xFE,0xC5,0x63,0xAF,0xD7,0xD6,0xD7,0xCF,0xDF,0xD2,0x37,0xBD,0xF7,0xB9,0x72,0x78,0x73,0xFA,0xF1,0xF7,0xB7,0xDB,0xDD,0xD3,0x9C,0xB3,0xB9,0x55,0x7B,0xD9,0xB7,0xFC,0x7F,0xE8,0xFF,0x3E,0x1F,0xBE,0x15,0xAD,0xC5,0xBD,0x19,0x3A,0x6B,0xEE,0x05,0xFD,0x57,0xFD,0x35,0xFC,0x7F,0x7D,0x67,0xF9,
			0xF5,0xFD,0x4C,0xBB,0x7F,0xFC,0xE1,0x09,0x77,0xF7,0xF7,0xE7,0xDC,0x3D,0x77,0x34,0xFA,0xED,0x3D,0x6E,0x5F,0xBD,0xD5,0xFD,0xE5,0xEE,0xE3,0xA4,0x4E,0xA7,0xC1,0x2F,0xDF,0xA7,0x5D,0xFD,0x47,0x5C,0x5B,0xFB,0x9F,0x9B,0x77,0x7F,0xEF,0x83,0xB3,0xFF,0x7F,0xF6,0x3F,0x63,0x7D,0xFB,0x7E,0x55,0xF1,0xB7,0x6F,0xAF,0x59,0x8E,0xE1,0xFF,
			0xD7,0x72,0x44,0xF9,0xF0,0xEE,0xFF,0xDD,0xAB,0x6E,0xFF,0xED,0x97,0x98,0xB6,0x2B,0xFE,0xFD,0xF5,0xD7,0xF3,0xFB,0xF9,0xBD,0xF1,0xFF,0x5A,0xBF,0xD2,0xDC,0xBE,0xEB,0xFF,0xEB,0x3C,0x73,0x91,0x43,0xFF,0x9A,0x45,0xDC,0xFC,0xC1,0xF3,0xF0,0x5F,0x3E,0x4D,0x7F,0xA6,0x7B,0x06,0x5E,0xBE,0x57,0x66,0xCA,0x3F,0xFF,0xD3,0x7F,0xEB,0x3E,
			0xFB,0xF9,0xC5,0xF3,0xDC,0xBC,0xFF,0xCA,0xB6,0x3B,0xC2,0xEC,0x5D,0xEF,0xF7,0x63,0x77,0xDD,0xD9,0x54,0x7B,0xD9,0xFF,0x5F,0xEF,0x3D,0xE6,0xB3,0xF6,0x65,0x74,0xF1,0x7F,0x76,0xBD,0xED,0x99,0x1F,0x7E,0xE3,0xF7,0x3D,0x52,0x7D,0x37,0xF7,0xFB,0xFB,0xDF,0xF9,0x1F,0x5D,0x39,0xFC,0xEF,0x53,0xF6,0x92,0x43,0x5F,0xBC,0xBD,0x63,0xFF,
			0x4F,0xF5,0x50,0x7F,0x5F,0x09,0x1A,0x8C,0x9F,0xFC,0x9B,0x7F,0xD3,0x9A,0x7C,0xB3,0xFD,0xF4,0xA0,0xCF,0xDF,0xED,0xB8,0xFB,0xFB,0xFF,0x7B,0xF7,0xC5,0xFF,0x6D,0xC6,0xAD,0x7A,0x97,0x7F,0x5E,0x3A,0x70,0x3E,0x5C,0xEA,0x52,0x1F,0xD6,0x7D,0xF5,0x0C,0xFF,0xF7,0x6E,0x1D,0x9D,0xB5,0xE9,0xDC,0xE4,0xBF,0x51,0xA7,0xF9,0xF9,0x55,0xAD,
			0xBF,0xD7,0xFF,0xFA,0x53,0x73,0x6C,0x77,0xFF,0xEF,0x37,0xEA,0xAB,0xC8,0xCF,0x13,0xFF,0xB3,0xF6,0xFD,0x27,0x9B,0x55,0x7A,0x27,0xD5,0x57,0xDE,0x57,0xF3,0x97,0x93,0x95,0x1C,0x9F,0x75,0x15,0x75,0x74,0xEE,0xF2,0xBB,0xF9,0xFB,0xF4,0x78,0x23,0xF5,0xBB,0x0B,0x0F,0x27,0xF4,0x7A,0x63,0xA7,0xCD,0xFB,0xF2,0x76,0x5F,0xFD,0x7E,0xFB,
			0xF1,0xFB,0xC7,0xFD,0x6D,0xFB,0xE5,0xFF,0x9F,0xFB,0xA3,0xF4,0x45,0x5D,0xFC,0xC9,0x73,0x88,0xED,0x75,0x5B,0xD5,0x98,0xFA,0xF8,0xFE,0xD7,0xEE,0x7D,0xC0,0xFC,0x9B,0xE9,0xBE,0x73,0x6D,0xBC,0xF7,0xAA,0x62,0x9E,0xCF,0xB2,0xF7,0xDF,0xBD,0x77,0x5D,0xFF,0xBB,0x77,0xA1,0xDD,0x9E,0xDA,0x7E,0xA6,0x36,0x52,0xC5,0x1D,0x79,0xEB,0x1F,
			0x7F,0xDD,0x4E,0x7F,0x14,0xDF,0x37,0x9B,0x77,0x77,0xDB,0x7B,0xDC,0x76,0xFF,0xF8,0xE1,0xDF,0xD3,0x5F,0x0D,0xC7,0xFC,0xDB,0x1B,0xDB,0x8E,0xF5,0xC9,0x46,0x9E,0x7F,0xCF,0xF6,0xBE,0x7B,0x6F,0xDD,0x5A,0xFF,0xDD,0x3D,0xF5,0x77,0x67,0xB9,0x54,0x9F,0xC3,0x6F,0xC1,0x7D,0x8D,0x2D,0xF6,0xE6,0xEC,0xF6,0xF8,0xFF,0x5F,0xFD,0x86,0x8E
			};
static const UByteSequence sSD8BitYUVF2WithAudio = {
			0xFF,0xB2,0x0A,0xFF,0xC7,0x24,0x38,0x18,0x00,0x92,0xCF,0x1F,0x84,0xEB,0x01,0x96,0x20,0x01,0x28,0x10,0x00,0xB2,0xDF,0x1F,0x44,0xDF,0x01,0x16,0x48,0x01,0x10,0x08,
			0x00,0xD2,0xEF,0x1F,0x34,0xD2,0x01,0x0E,0x6F,0x01,0xC9,0xFF,0xB2,0x0A,0xFD,0xC7,0x24,0x78,0x3E,0x01,0x42,0x00,0x1D,0xA4,0x9D,0x01,0xC6,0x2D,0x1E,0x98,0x75,0x01,
			0xF2,0x3A,0x1D,0x44,0xF5,0x00,0xC6,0x1C,0x1E,0xE0,0xA8,0x01,0x5A,0x7C,0x1D,0xAC,0x46,0x00,0xAE,0x1C,0x1F,0x38,0xFF,0xB2,0x0A,0xFB,0xC7,0x24,0x28,0xA4,0x1E,0x92,
			0x45,0x1B,0xFC,0x9D,0x1C,0x46,0x93,0x04,0x88,0x32,0x1E,0x2A,0x47,0x1B,0x84,0xF6,0x1B,0xCE,0x60,0x05,0x88,0xE6,0x1D,0x6A,0xE0,0x1B,0xD4,0x66,0x1B,0x66,0xFA,0x04,
			0x29,0xFF,0xB2,0x0A,0xF9,0xC7,0x24,0x40,0x33,0x05,0x12,0x75,0x05,0x3C,0x5E,0x19,0x66,0x19,0x05,0x18,0x29,0x04,0x7A,0xC1,0x01,0xC4,0x22,0x1E,0x06,0xF6,0x1B,0xD0,
			0xB6,0x01,0x02,0x46,0x1C,0x34,0x23,0x04,0xC6,0x45,0x1A,0x26,0xFF,0xB2,0x0B,0xFF,0xC8,0x24,0x00,0x00,0x00,0x02,0x00,0x00,0x5C,0xC4,0x01,0x4E,0x94,0x01,0xE8,0xF7,
			0x1F,0x2A,0x10,0x00,0xD4,0xB5,0x01,0xB6,0xB7,0x01,0xD0,0xEF,0x1F,0x4A,0x20,0x00,0x9C,0xA6,0x01,0x1E,0xDA,0x01,0xA5,0xFF,0xB2,0x0B,0xFD,0xC8,0x24,0xF0,0xD7,0x01,
			0x8A,0xC6,0x1D,0x2C,0x93,0x00,0x7E,0x26,0x1F,0x78,0x03,0x02,0x8A,0x18,0x1E,0x14,0xDF,0x1F,0xBE,0x34,0x00,0x08,0x2B,0x02,0x3A,0x70,0x1E,0xC4,0x2A,0x1F,0xAE,0x42,
			0x00,0xDB,0xFF,0xB2,0x0B,0xFB,0xC8,0x24,0xA8,0xCB,0x1C,0x82,0x06,0x1C,0xAC,0xFD,0x1B,0x5E,0xAB,0x02,0x98,0xEE,0x1C,0x5A,0xA6,0x1D,0x94,0xAA,0x1C,0x06,0x29,0x00,
			0x48,0x56,0x1C,0x72,0xA3,0x1E,0x4C,0x39,0x1D,0x9E,0x67,0x1E,0xE3,0xFF,0xB2,0x0B,0xF9,0xC8,0x24,0x48,0xE2,0x1E,0x82,0x22,0x1A,0xAC,0x1F,0x06,0x16,0xA9,0x02,0x78,
			0x36,0x1B,0x12,0xA5,0x1B,0x7C,0xD1,0x00,0x86,0xE0,0x06,0x80,0xD0,0x1A,0xD2,0x31,0x1F,0x2C,0x84,0x1A,0xDE,0xF2,0x1E,0x15,0xFF,0xB2,0x0C,0xFF,0xC9,0x30,0xC0,0xE7,
			0x1F,0x6A,0x30,0x00,0xB4,0x95,0x01,0x56,0xFA,0x01,0xB8,0xDF,0x1F,0x72,0x40,0x00,0x34,0x84,0x01,0x46,0x19,0x01,0xB0,0xD7,0x1F,0x62,0x50,0x00,0x24,0x72,0x01,0xC6,
			0x36,0x01,0xB0,0xCF,0x1F,0x3A,0x60,0x00,0x8C,0x5F,0x01,0xB6,0x52,0x01,0x8D,0xFF,0xB2,0x0C,0xFD,0xC9,0x30,0x68,0x4F,0x02,0x62,0xCF,0x1E,0xAC,0x79,0x1F,0xAE,0x48,
			0x01,0x48,0x6E,0x02,0xB2,0x33,0x1E,0x0C,0xCC,0x1E,0x6E,0x42,0x01,0x70,0x89,0x02,0xDA,0x9C,0x1E,0x24,0x26,0x1E,0x1E,0x2A,0x02,0xA0,0xA0,0x02,0x5A,0x09,0x1F,0xF4,
			0x89,0x1E,0xBE,0xFB,0x02,0xF6,0xFF,0xB2,0x0C,0xFB,0xC9,0x30,0x70,0x09,0x1C,0x82,0xDB,0x1F,0x3C,0x5B,0x1F,0x0E,0x69,0x1C,0xF0,0x0A,0x1C,0x62,0x26,0x00,0x5C,0xB0,
			0x00,0x5E,0x07,0x1B,0xB0,0x59,0x1C,0x32,0x5E,0x01,0x74,0xD0,0x02,0x66,0xC3,0x1A,0x98,0xF4,0x1C,0xC2,0x5A,0x02,0xDC,0x5C,0x03,0x06,0xAB,0x1B,0x84,0xFF,0xB2,0x0C,
			0xF9,0xC9,0x30,0x98,0x93,0x1B,0x1A,0x42,0x04,0xB4,0xB5,0x1A,0xEE,0x81,0x19,0x28,0xD8,0x1D,0x52,0xFF,0x05,0x5C,0x47,0x00,0x16,0x51,0x00,0x70,0xA8,0x01,0xB2,0xFF,
			0x03,0xDC,0x8A,0x05,0xEE,0x8E,0x06,0x20,0x7C,0x04,0xBA,0xD6,0x1F,0x2C,0x26,0x05,0xA6,0x6E,0x01,0x95,0xFF,0xB2,0x0D,0xFF,0xCA,0x30,0xC0,0xC7,0x1F,0xEA,0x6F,0x00,
			0x74,0x4C,0x01,0xF6,0x6B,0x01,0xD8,0xBF,0x1F,0x7A,0x7F,0x00,0xF4,0x37,0x01,0x66,0x83,0x01,0xF8,0xB7,0x1F,0xDA,0x8F,0x00,0x0C,0x23,0x01,0xF6,0x99,0x01,0x28,0xB0,
			0x1F,0x0A,0x9F,0x00,0xD4,0x0D,0x01,0x86,0xAD,0x01,0x8D,0xFF,0xB2,0x0D,0xFD,0xCA,0x30,0xB0,0xB2,0x02,0xBA,0x7A,0x1F,0x54,0xF7,0x1D,0x46,0xB0,0x02,0x88,0xBF,0x02,
			0x72,0xED,0x1F,0xD4,0x71,0x1D,0xE6,0x46,0x03,0xF8,0xC8,0x02,0xEA,0x62,0x1F,0xB4,0xF9,0x1D,0x1E,0xB9,0x03,0x00,0xCC,0x02,0x82,0xD9,0x1F,0xDC,0x91,0x1D,0xF6,0x07,
			0x03,0x09,0xFF,0xB2,0x0D,0xFB,0xCA,0x30,0xD0,0xD3,0x1C,0xE2,0xFA,0x03,0x8C,0x06,0x04,0xF6,0x58,0x1D,0x38,0xF0,0x1D,0x1A,0x20,0x04,0x34,0x99,0x04,0xBE,0x02,0x00,
			0xF8,0x3D,0x1E,0x5A,0xB9,0x04,0x54,0x05,0x04,0x86,0xAB,0x02,0x88,0xB0,0x1E,0xEA,0xB9,0x04,0xA4,0x5A,0x03,0xA6,0x57,0x04,0xA1,0xFF,0xB2,0x0D,0xF9,0xCA,0x30,0x78,
			0x27,0x05,0xC2,0x6D,0x1B,0xAC,0xA1,0x1E,0x4E,0xEF,0x19,0x08,0xAC,0x04,0x62,0x31,0x1A,0x5C,0xB5,0x19,0xC6,0xEB,0x1D,0x08,0x97,0x02,0xD2,0x92,0x1C,0xFC,0x42,0x1B,
			0xC6,0x09,0x06,0x40,0xCD,0x1E,0xA2,0x19,0x01,0x9C,0x65,0x02,0x16,0x84,0x03,0xD5,0xFF,0xB2,0x0E,0xFF,0xCB,0x30,0x70,0xA8,0x1F,0xFA,0xAE,0x00,0x4C,0xF7,0x00,0xFE,
			0xBE,0x01,0xC0,0xA0,0x1F,0xBA,0xBD,0x00,0x94,0xE0,0x00,0x56,0xCE,0x01,0x28,0x98,0x9F,0x2A,0xCD,0x80,0xAC,0xC8,0x80,0x6E,0xDC,0x81,0xA0,0x90,0x1F,0x5A,0xDC,0x00,
			0xA4,0xB0,0x00,0x46,0xE7,0x01,0x04,0xFF,0xB2,0x0E,0xFD,0xCB,0x30,0x98,0xCB,0x02,0xAA,0x4F,0x00,0xCC,0x3A,0x1D,0x16,0x2E,0x03,0xB0,0xC6,0x02,0xBA,0xC5,0x00,0xAC,
			0xF5,0x1C,0xC6,0x2D,0x03,0x68,0xBC,0x82,0x12,0x3A,0x80,0x1C,0xC3,0x9C,0x16,0x05,0x83,0xC8,0xAD,0x02,0x1A,0xAD,0x00,0x6C,0xA4,0x1C,0xC6,0xB6,0x03,0xDE,0xFF,0xB2,
			0x0E,0xFB,0xCB,0x30,0x68,0x3A,0x1F,0xAA,0x22,0x04,0xCC,0xCD,0x02,0x46,0x3D,0x05,0x78,0xCB,0x00,0x32,0xFE,0x03,0xEC,0xAC,0x00,0xF6,0xF7,0x04,0xF0,0x54,0x81,0x4A,
			0x5F,0x82,0xCC,0x58,0x9F,0x76,0x93,0x83,0xA8,0xC6,0x01,0x4A,0x63,0x01,0x9C,0x36,0x1D,0x7E,0x94,0x01,0x07,0xFF,0xB2,0x0E,0xF9,0xCB,0x30,0xF0,0xD4,0x1B,0x52,0xA0,
			0x05,0xD4,0xB5,0x06,0x36,0x7C,0x1A,0xB8,0xE6,0x1A,0xFA,0x4C,0x05,0x4C,0x17,0x03,0x26,0x5A,0x1B,0x20,0x19,0x9B,0xF2,0xAE,0x82,0xC4,0xB2,0x9C,0xC6,0x8E,0x84,0xF8,
			0xFA,0x1D,0xBA,0x0B,0x1D,0x5C,0x36,0x19,0xCE,0x61,0x05,0xD9,0xFF,0xB2,0x0F,0xFF,0xCC,0x30,0x30,0x89,0x9F,0x3A,0xEB,0x80,0x94,0x98,0x80,0xC6,0xF0,0x81,0xD8,0x81,
			0x1F,0xCA,0xF9,0x00,0x7C,0x7F,0x00,0xE6,0xF7,0x01,0x98,0x79,0x9F,0x02,0x08,0x80,0x7C,0x65,0x80,0x9E,0xFC,0x81,0x70,0x72,0x9F,0xDA,0x16,0x80,0x9C,0x4B,0x80,0xEE,
			0xFF,0x81,0x6B,0xFF,0xB2,0x0F,0xFD,0xCC,0x30,0xF0,0x99,0x82,0x42,0x1C,0x81,0x44,0x99,0x9C,0x36,0x42,0x83,0x00,0x82,0x02,0x0A,0x88,0x01,0xF4,0xA2,0x1C,0x6E,0xAB,
			0x02,0x38,0x65,0x82,0xF2,0xEF,0x81,0x44,0xBF,0x9C,0xF6,0xF4,0x82,0xC0,0x44,0x82,0x8A,0x52,0x81,0x84,0xEF,0x9C,0xB6,0x23,0x82,0x14,0xFF,0xB2,0x0F,0xFB,0xCC,0x30,
			0x40,0x14,0x82,0x4A,0x2C,0x80,0x5C,0xA8,0x9C,0x8E,0xD2,0x9F,0x20,0x2F,0x03,0x6A,0xE1,0x1F,0x6C,0xFC,0x1B,0xB6,0x50,0x1D,0x90,0x0E,0x83,0x8A,0xA9,0x9E,0xD4,0x66,
			0x9B,0xD6,0x03,0x9B,0x80,0xA7,0x83,0xEA,0xAB,0x9D,0xC4,0xF7,0x9B,0x9E,0x9E,0x9A,0x07,0xFF,0xB2,0x0F,0xF9,0xCC,0x30,0x08,0xBB,0x80,0x32,0x9A,0x9A,0x3C,0xBF,0x9D,
			0x2E,0xBF,0x9C,0x70,0xD6,0x03,0x82,0x83,0x1A,0xE4,0x0A,0x04,0x9E,0x59,0x19,0xE0,0x06,0x85,0xD2,0x30,0x9E,0x14,0x84,0x86,0x76,0xB0,0x81,0x90,0x1D,0x85,0x82,0xAD,
			0x83,0xC4,0x45,0x81,0x16,0x65,0x86,0x68,0xFF,0xB2,0x10,0xFF,0xCD,0x30,0x68,0x6A,0x9F,0x52,0x25,0x80,0xF4,0x31,0x80,0xCE,0xFF,0x81,0x81,0x63,0x9F,0x63,0x33,0x80,
			0x85,0x16,0x80,0x3F,0xFE,0x81,0xB0,0x5B,0x1F,0x0A,0x41,0x00,0x64,0xFB,0x00,0x3E,0xFA,0x01,0x08,0x54,0x9F,0x42,0x4E,0x80,0xAC,0xE0,0x80,0xDE,0xF3,0x81,0x3F,0xFF,
			0xB2,0x10,0xFD,0xCD,0x30,0xE0,0x1F,0x82,0x82,0xAE,0x81,0x8C,0x32,0x9D,0xAE,0x3A,0x81,0xD9,0xF6,0x81,0x93,0x04,0x82,0xAD,0x87,0x9D,0xEF,0x40,0x81,0x08,0xCA,0x01,
			0x92,0x53,0x02,0xDC,0xED,0x1D,0x46,0x39,0x00,0xB8,0x99,0x81,0x62,0x9B,0x82,0xA4,0x63,0x9D,0x16,0x2C,0x80,0xB0,0xFF,0xB2,0x10,0xFB,0xCD,0x30,0x20,0xF6,0x83,0xDA,
			0x0A,0x9C,0x34,0x9F,0x9C,0x76,0x6E,0x9B,0x79,0xF6,0x83,0x0B,0xE2,0x9B,0x25,0x2A,0x9D,0x47,0x45,0x9C,0x88,0xA8,0x03,0xC2,0x47,0x1B,0xF4,0x4A,0x1F,0x4E,0x8A,0x1E,
			0x40,0x0F,0x83,0x0A,0x45,0x9B,0x0C,0x9F,0x80,0x7E,0x55,0x81,0x72,0xFF,0xB2,0x10,0xF9,0xCD,0x30,0x68,0x70,0x82,0x0A,0xDB,0x85,0x5C,0x70,0x9B,0x7E,0x06,0x9F,0x39,
			0xBD,0x9F,0x5B,0x63,0x84,0x35,0x16,0x9A,0xE7,0x67,0x99,0x50,0x82,0x1C,0x9A,0xDD,0x00,0x5C,0xCC,0x1F,0x1E,0x38,0x1F,0x98,0x10,0x9A,0x5A,0xC8,0x9B,0x74,0xD3,0x85,
			0x4E,0x3B,0x86,0xE2,0xFF,0xB2,0x10,0xF4,0x00,0x10,0x50,0x84,0x9C,0x10,0x90,0x94,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xA8,0xFF,0xB2,0x14,0xFF,0xCE,
			0x30,0x78,0x4D,0x1F,0xFA,0x5B,0x00,0x6C,0xC4,0x00,0x16,0xEB,0x01,0x10,0x46,0x1F,0x42,0x69,0x00,0xAC,0xA8,0x00,0xEE,0xE0,0x01,0xD0,0x3E,0x1F,0x0A,0x76,0x00,0x84,
			0x8C,0x00,0x86,0xD4,0x01,0xB8,0x37,0x1F,0x4A,0x82,0x00,0x04,0x70,0x00,0xD6,0xC5,0x01,0xAD,0xFF,0xB2,0x14,0xFD,0xCE,0x30,0x58,0x65,0x01,0x1A,0xDA,0x02,0x24,0xE7,
			0x1D,0xFE,0x1D,0x1F,0x40,0x2D,0x01,0xCA,0x11,0x02,0x1C,0x78,0x1E,0xAE,0x13,0x1F,0xE8,0xF2,0x01,0xBA,0x3F,0x02,0x0C,0x14,0x1E,0x9E,0x14,0x1E,0xC0,0xB5,0x01,0x42,
			0x64,0x02,0x2C,0xB9,0x1E,0xD6,0x25,0x1E,0x0C,0xFF,0xB2,0x14,0xFB,0xCE,0x30,0x60,0x30,0x03,0x42,0xDA,0x1B,0x44,0xC1,0x02,0x26,0xA1,0x03,0xF8,0x15,0x02,0x1A,0xFD,
			0x1C,0x6C,0x51,0x03,0x56,0x84,0x04,0xC8,0xC8,0x01,0x12,0x9A,0x1D,0x94,0x00,0x04,0x1E,0x63,0x05,0x58,0x56,0x01,0xFA,0x94,0x1E,0x1C,0x99,0x04,0x06,0x0D,0x04,0x17,
			0xFF,0xB2,0x14,0xF9,0xCE,0x30,0xE8,0xB0,0x1A,0x4A,0x00,0x1A,0x2C,0xFC,0x05,0xDE,0x76,0x02,0x20,0x27,0x1D,0x5A,0xF5,0x1B,0xF4,0x19,0x1F,0xB6,0xAB,0x19,0x50,0xCA,
			0x00,0xE2,0x1A,0x00,0x14,0x34,0x1A,0x06,0x0A,0x1C,0x08,0x22,0x03,0x62,0x89,0x04,0xFC,0x3C,0x1B,0x16,0xEB,0x05,0x1D,0xFF,0xB2,0x14,0x41,0x01,0x04,0x81,0x06,0x00,
			0x01,0xCE,0xFF,0xB2,0x15,0xFF,0xCF,0x30,0xC0,0x30,0x9F,0x0A,0x8F,0x80,0x3C,0x54,0x80,0xFE,0xB4,0x81,0xF8,0x29,0x1F,0x3A,0x9B,0x00,0x44,0x37,0x00,0x06,0xA1,0x01,
			0x50,0x23,0x1F,0xDA,0xA6,0x00,0x24,0x1B,0x00,0x06,0x8C,0x01,0xE0,0x1C,0x1F,0xE2,0xB2,0x00,0xF4,0xFE,0x1F,0x16,0x75,0x01,0x51,0xFF,0xB2,0x15,0xFD,0xCF,0x30,0x40,
			0x75,0x81,0xF2,0x7F,0x82,0x7C,0x65,0x9F,0xA6,0x4C,0x9D,0xE8,0x32,0x01,0x5A,0x91,0x02,0xBC,0x16,0x1F,0x76,0x8C,0x1D,0x40,0xED,0x00,0x42,0x99,0x02,0xAC,0xCA,0x1F,
			0x8E,0xEB,0x1C,0xC0,0xA6,0x00,0x8A,0x97,0x02,0xE4,0x7F,0x00,0xDE,0x6B,0x1C,0xD1,0xFF,0xB2,0x15,0xFB,0xCF,0x30,0x00,0xCD,0x80,0xE2,0xCB,0x9F,0x1C,0x0B,0x84,0x56,
			0xCB,0x82,0xF0,0x3B,0x1F,0xBA,0x16,0x00,0xEC,0x65,0x03,0xF6,0x4E,0x00,0xF0,0xB2,0x1E,0xAA,0x4F,0x01,0xE4,0xDC,0x02,0x9E,0x8A,0x1E,0x20,0x3F,0x1E,0x5A,0x4E,0x02,
			0x2C,0xBE,0x00,0x0E,0x83,0x1C,0x13,0xFF,0xB2,0x15,0xF9,0xCF,0x30,0xE8,0xD2,0x85,0xCA,0xD1,0x85,0x5C,0xF9,0x81,0xC6,0x2C,0x84,0x18,0x7C,0x05,0x0A,0x79,0x03,0xA4,
			0x73,0x06,0xAE,0xF0,0x1B,0xE8,0x3E,0x03,0xDA,0xF4,0x1E,0x34,0x46,0x04,0x4E,0xFD,0x1A,0xD0,0xAE,0x1F,0x72,0x66,0x1A,0x5C,0x07,0x1D,0x1E,0xD4,0x03,0x94,0xFF,0xB2,
			0x16,0xFF,0xD0,0x24,0x90,0x15,0x9F,0x52,0xBD,0x80,0xCC,0xE1,0x9F,0x56,0x5C,0x81,0x78,0x0F,0x1F,0x2A,0xC8,0x00,0xAC,0xC5,0x1F,0xD6,0x41,0x01,0x90,0x08,0x1F,0x62,
			0xD3,0x00,0xBC,0xA8,0x1F,0xBE,0x25,0x01,0xD8,0xFF,0xB2,0x16,0xFD,0xD0,0x24,0x08,0x5E,0x80,0x3A,0x8B,0x82,0x04,0x32,0x80,0xF6,0x0F,0x9C,0x98,0x14,0x00,0x82,0x76,
			0x02,0xB4,0xE1,0x00,0xCE,0xDA,0x1C,0x10,0xCA,0x00,0xAA,0x57,0x02,0xAC,0x8B,0x01,0x9E,0xCD,0x1C,0x10,0xFF,0xB2,0x16,0xFB,0xD0,0x24,0xF8,0xF1,0x9D,0x5A,0xF0,0x83,
			0x1C,0x69,0x9F,0x4E,0x12,0x9B,0x10,0xD5,0x1C,0xBA,0x1A,0x04,0xDC,0x45,0x1D,0xA6,0xBC,0x1A,0x40,0xF5,0x1C,0x0A,0xB7,0x04,0xE4,0xB3,0x1C,0x46,0x95,0x1B,0xB3,0xFF,
			0xB2,0x16,0xF9,0xD0,0x24,0x18,0x3D,0x9C,0x22,0xAE,0x9A,0x64,0x3B,0x99,0x26,0x64,0x86,0x78,0x4F,0x1A,0xE2,0x43,0x1D,0xD4,0x6E,0x1C,0xAE,0xAE,0x1D,0x70,0x5B,0x1A,
			0x5A,0xE7,0x02,0x4C,0xD5,0x03,0xF6,0xBB,0x19,0xA0,0xFF,0xB2,0x17,0xFF,0xD1,0x24,0xD0,0x02,0x1F,0xF2,0xDD,0x00,0xFC,0x8C,0x1F,0x26,0x07,0x01,0x48,0xFC,0x9F,0xDA,
			0xE7,0x80,0x84,0x70,0x9F,0x2E,0xE7,0x81,0xF8,0xF5,0x1F,0x1A,0xF1,0x00,0x64,0x54,0x1F,0x06,0xC5,0x01,0x90,0xFF,0xB2,0x17,0xFD,0xD1,0x24,0xF0,0x7E,0x00,0x1A,0x2F,
			0x02,0xC4,0x2C,0x01,0x06,0xE7,0x1C,0xD8,0x32,0x80,0x5A,0xFD,0x82,0x24,0xC4,0x81,0xEE,0x28,0x9C,0x60,0xE6,0x1F,0x2A,0xC3,0x02,0x04,0x50,0x02,0x76,0x90,0x1C,0xEF,
			0xFF,0xB2,0x17,0xFB,0xD1,0x24,0xB8,0x5A,0x1C,0xEA,0xBB,0x04,0x84,0x02,0x1B,0xBE,0x36,0x1D,0x48,0x0A,0x9C,0xBA,0x28,0x84,0x0C,0x66,0x9B,0x46,0xDC,0x9F,0x18,0x09,
			0x1C,0x82,0x07,0x03,0x0C,0xF2,0x1B,0xD6,0x89,0x02,0xDE,0xFF,0xB2,0x17,0xF9,0xD1,0x24,0xB8,0x5D,0x1C,0x7A,0x60,0x05,0x64,0xBE,0x06,0xC6,0xA0,0x00,0xB0,0xD7,0x9F,
			0xCA,0x83,0x85,0x94,0xAE,0x82,0x06,0x99,0x86,0x88,0x60,0x03,0xDA,0xDD,0x01,0x54,0x7C,0x1B,0x06,0x20,0x00,0x21,0xFF,0xB2,0x18,0xFF,0xD2,0x24,0xD8,0xEF,0x1F,0xAA,
			0xFA,0x00,0xB4,0x38,0x1F,0xC6,0xA2,0x01,0xE8,0xE9,0x1F,0x8A,0x03,0x01,0x7C,0x1C,0x1F,0x9E,0x7D,0x01,0x38,0xE4,0x9F,0xBA,0x0B,0x81,0xD4,0x01,0x9F,0xAE,0x57,0x81,
			0x3F,0xFF,0xB2,0x18,0xFD,0xD2,0x24,0x18,0x9A,0x1F,0x42,0x81,0x02,0x0C,0xCF,0x02,0x26,0x1B,0x1D,0x98,0x4E,0x1F,0x92,0x36,0x02,0x2C,0x3E,0x02,0x0E,0xC6,0x1D,0x78,
			0x03,0x9F,0x22,0xE5,0x81,0x9C,0x9D,0x82,0xB6,0x8E,0x9D,0x2E,0xFF,0xB2,0x18,0xFB,0xD2,0x24,0x48,0x55,0x1C,0x82,0x6C,0x02,0xFC,0x94,0x1C,0x26,0x40,0x04,0xF0,0xED,
			0x1C,0xB2,0x72,0x01,0x1C,0x1B,0x1D,0x96,0x36,0x05,0x70,0xCA,0x9C,0xEA,0x3C,0x80,0xB4,0x39,0x9F,0xFE,0x01,0x85,0x7F,0xFF,0xB2,0x18,0xF9,0xD2,0x24,0x58,0x8A,0x05,
			0x0A,0x5F,0x1C,0x9C,0xA0,0x19,0xEE,0xA1,0x19,0x40,0xC7,0x05,0x92,0x26,0x1A,0x9C,0x54,0x1E,0xE6,0x29,0x1E,0x88,0x02,0x83,0x52,0x93,0x9B,0x9C,0xF6,0x84,0x0E,0x97,
			0x86,0x3F,0xFF,0xB2,0x19,0xFF,0xD3,0x24,0xB8,0xDE,0x1F,0x2A,0x14,0x01,0xC4,0xE6,0x1F,0x2E,0x30,0x01,0x78,0xD8,0x9F,0xE2,0x1B,0x81,0x6C,0xCB,0x9F,0x36,0x08,0x81,
			0x70,0xD3,0x1F,0xDA,0x23,0x01,0xD4,0xB1,0x1F,0x06,0xDE,0x00,0xFC,0xFF,0xB2,0x19,0xFD,0xD3,0x24,0x48,0xB9,0x1F,0x0A,0x8C,0x01,0x1C,0xEB,0x02,0x7E,0x6F,0x1E,0xA0,
			0x70,0x9F,0x82,0x2D,0x81,0xBC,0x26,0x83,0xAE,0x64,0x9E,0x08,0x29,0x1F,0xDA,0xC9,0x01,0x14,0x4F,0x03,0xB6,0x67,0x1F,0x49,0xFF,0xB2,0x19,0xFB,0xD3,0x24,0xC8,0xE4,
			0x1D,0x1A,0xF1,0x1F,0xB4,0x8D,0x00,0xC6,0xAD,0x03,0x68,0x31,0x9E,0x22,0xB8,0x9E,0xF4,0xB2,0x82,0x2E,0xB7,0x81,0xC8,0xA2,0x1E,0x62,0xB8,0x1D,0xC4,0x46,0x03,0x9E,
			0xF7,0x1F,0x59,0xFF,0xB2,0x19,0xF9,0xD3,0x24,0x98,0xA1,0x00,0xF2,0x13,0x1F,0xA4,0xAB,0x05,0x8E,0x63,0x03,0x98,0x04,0x9D,0x12,0x2D,0x84,0x64,0x95,0x80,0xB6,0xB3,
			0x9A,0x70,0xA1,0x1A,0xEA,0xFF,0x05,0xE4,0xDB,0x1A,0x4E,0x52,0x1B,0xBA,0xFF,0xB2,0x1A,0xFF,0xD4,0x24,0xA0,0xCD,0x1F,0x1A,0x2A,0x01,0x0C,0x97,0x1F,0xBE,0xB3,0x00,
			0x10,0xC8,0x1F,0x92,0x31,0x01,0x24,0x7E,0x1F,0x8E,0x88,0x00,0xB8,0xC3,0x1F,0x42,0x37,0x01,0x2C,0x65,0x1F,0xB6,0x5B,0x00,0x62,0xFF,0xB2,0x1A,0xFD,0xD4,0x24,0x18,
			0xE3,0x1E,0x72,0x60,0x01,0x24,0x63,0x03,0x66,0x73,0x1F,0x48,0xA0,0x1E,0xB2,0xF2,0x00,0x74,0x64,0x03,0x26,0x82,0x00,0x28,0x5F,0x1E,0x22,0x82,0x00,0xF4,0x50,0x03,
			0x4E,0x8E,0x00,0xFC,0xFF,0xB2,0x1A,0xFB,0xD4,0x24,0x88,0x2B,0x1F,0x7A,0x14,0x1C,0x54,0xFA,0x04,0x36,0x70,0x1D,0x90,0xBC,0x00,0x82,0xE8,0x1B,0xC4,0x99,0x04,0x46,
			0x16,0x1B,0x20,0x46,0x01,0x32,0x4A,0x1B,0xB4,0x10,0x04,0xDE,0xA1,0x1A,0x8C,0xFF,0xB2,0x1A,0xF9,0xD4,0x24,0xE0,0x19,0x1A,0x82,0x15,0x04,0x14,0x58,0x1A,0x0E,0x8A,
			0x04,0x90,0xA0,0x1C,0x82,0xF3,0x1F,0xAC,0x83,0x00,0x6E,0x9C,0x05,0xA8,0xE5,0x1F,0x92,0x7F,0x1B,0xDC,0x05,0x06,0xCE,0x9C,0x1C,0x24,0xFF,0xB2,0x1B,0xFF,0xD5,0x24,
			0xA8,0xBE,0x1F,0x2A,0x3D,0x01,0x34,0x4C,0x1F,0x56,0x2E,0x00,0xD0,0xB9,0x1F,0x52,0x43,0x01,0x4C,0x34,0x1F,0xA6,0x00,0x00,0x40,0xB4,0x1F,0xAA,0x48,0x01,0x84,0x1D,
			0x1F,0xDE,0xD2,0x00,0x01,0xFF,0xB2,0x1B,0xFD,0xD5,0x24,0x38,0x20,0x1E,0x42,0x0E,0x00,0x24,0x2A,0x03,0x3E,0x91,0x01,0xF8,0xE4,0x1E,0xB2,0x99,0x00,0xF4,0xEF,0x02,
			0xC6,0x86,0x01,0xD0,0xAC,0x1E,0x12,0x23,0x00,0xCC,0xA3,0x02,0x56,0x68,0x02,0x54,0xFF,0xB2,0x1B,0xFB,0xD5,0x24,0x08,0xBA,0x01,0x22,0x43,0x1B,0x0C,0x71,0x03,0x66,
			0x5F,0x1B,0xC0,0x08,0x02,0x4A,0xD4,0x1B,0xDC,0xEB,0x02,0x16,0x29,0x1C,0xA0,0x26,0x03,0xEA,0xF3,0x1C,0x6C,0xCF,0x00,0x16,0x66,0x1E,0x55,0xFF,0xB2,0x1B,0xF9,0xD5,
			0x24,0xB8,0x93,0x02,0x22,0x2C,0x1A,0x2C,0x59,0x04,0xF6,0xE4,0x19,0x38,0x2E,0x05,0x22,0x7A,0x1C,0x2C,0x6D,0x1E,0x06,0xEF,0x02,0xF8,0xFF,0x05,0x92,0xFD,0x01,0x44,
			0x6B,0x19,0x26,0x1B,0x06,0x8F,0xFF,0x92,0x1C,0xFF,0xD6,0x24,0xE8,0xAF,0x1F,0x32,0x4D,0x01,0xEC,0x06,0x1F,0x26,0xA3,0x00,0xD8,0xAB,0x1F,0xEA,0x51,0x01,0x8C,0xF0,
			0x1E,0xBE,0x74,0x00,0x08,0xA7,0x1F,0xDA,0x55,0x01,0x74,0xDA,0x1E,0xCE,0x44,0x00,0x2F,0xFF,0x92,0x1C,0xFD,0xD6,0x24,0x38,0x77,0x1E,0xEA,0xAC,0x1F,0x8C,0x45,0x02,
			0x1E,0x32,0x02,0x90,0x45,0x1E,0xEA,0x36,0x1F,0x7C,0xD6,0x02,0x6E,0xDE,0x02,0x50,0x17,0x1E,0xB2,0xC1,0x1F,0x4C,0x59,0x02,0xDE,0x6A,0x03,0x7B,0xFF,0x92,0x1C,0xFB,
			0xD6,0x24,0xC8,0x07,0x03,0xF2,0x8D,0x1D,0x7C,0x7A,0x1F,0xD6,0x30,0x01,0xF8,0xA3,0x03,0x9A,0x86,0x1E,0x44,0x55,0x1D,0xA6,0x83,0x03,0x30,0xF4,0x03,0x42,0xBB,0x1F,
			0x9C,0xBF,0x1C,0x76,0x75,0x04,0xE6,0xFF,0x92,0x1C,0xF9,0xD6,0x24,0x88,0xB8,0x03,0xAA,0x92,0x05,0x8C,0x34,0x1C,0xEE,0xB8,0x1E,0x08,0x92,0x00,0x9A,0x56,0x05,0xB4,
			0x89,0x02,0x66,0x6E,0x19,0x08,0xD6,0x1D,0x32,0xC9,0x02,0x14,0xCC,0x06,0x36,0x87,0x1F,0x8C,0xFF,0xE2,0x1C,0x00,0x00,0xFF,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,0x1D,0x16,
			0x11,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,0x1D,0x16,0x11,0x10,0x11,0x16,0x1D,
			0x26,0x31,0x3D,0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,0x1D,0x16,0x11,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,0x4A,0x56,
			0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,0x1D,0x16,0x11,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,
			0x7D,0x7A,0x74,0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,0x1D,0x16,0x11,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,
			0x56,0x4A,0x3D,0x31,0x26,0x1D,0x16,0x11,0x10,0x11,0x16,0x1D,0x26,0x31,0x3D,0x4A,0x56,0x62,0x6C,0x74,0x7A,0x7D,0x7D,0x7A,0x74,0x6C,0x62,0x56,0x4A,0x3D,0x31,0x26,
			0x1D,0x16,0x11,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x24,0x47,0x6A,0x7E,0x7E,0x7E,0x7E,0x00,0xFF,0xE2,0x1C,0x00,0x00,0xFF,0x7E,0x7E,
			0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x6A,0x47,0x24,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x24,0x47,0x6A,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,
			0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x6A,0x47,0x24,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x00,0xFF,0x92,
			0x1D,0xFF,0xD7,0x24,0x80,0xA2,0x1F,0xF2,0x58,0x01,0xB4,0xC5,0x1E,0x96,0x15,0x00,0x38,0x9E,0x1F,0x3A,0x5C,0x01,0x54,0xB1,0x1E,0x4E,0xE5,0x1F,0x40,0x9A,0x1F,0xAA,
			0x5E,0x01,0x64,0x9E,0x1E,0x16,0xB6,0x1F,0xD6,0xFF,0xE2,0x1C,0x00,0x00,0xD2,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x24,0x47,0x6A,0x7E,0x7E,0x7E,
			0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x6A,0x47,0x24,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
			0xFF,0xB2,0x1D,0xFF,0xD7,0x24,0x80,0xA2,0x1F,0xF2,0x58,0x01,0xB4,0xC5,0x1E,0x96,0x15,0x00,0x38,0x9E,0x1F,0x3A,0x5C,0x01,0x54,0xB1,0x1E,0x4E,0xE5,0x1F,0x40,0x9A,
			0x1F,0xAA,0x5E,0x01,0x64,0x9E,0x1E,0x16,0xB6,0x1F,0xD6,0xFF,0x92,0x1D,0xFD,0xD7,0x24,0xB8,0xEC,0x1D,0xD2,0x4F,0x1F,0x0C,0xCE,0x01,0x4E,0xD3,0x03,0x30,0xC6,0x1D,
			0xDA,0xDF,0x1E,0x1C,0x37,0x01,0x46,0x16,0x03,0xF8,0xA4,0x1D,0x5A,0x74,0x1E,0x2C,0x96,0x01,0xD6,0x32,0x03,0x08,0xFF,0xB2,0x1D,0xFD,0xD7,0x24,0xB8,0xEC,0x1D,0xD2,
			0x4F,0x1F,0x0C,0xCE,0x01,0x4E,0xD3,0x03,0x30,0xC6,0x1D,0xDA,0xDF,0x1E,0x1C,0x37,0x01,0x46,0x16,0x03,0xF8,0xA4,0x1D,0x5A,0x74,0x1E,0x2C,0x96,0x01,0xD6,0x32,0x03,
			0x08,0xFF,0x92,0x1D,0xFB,0xD7,0x24,0x38,0xF7,0x03,0x02,0x07,0x00,0xDC,0x08,0x1B,0x5E,0x64,0x05,0xD0,0xAC,0x03,0x02,0x40,0x01,0x84,0x66,0x1B,0xBE,0x1F,0x04,0xD0,
			0x15,0x03,0xD2,0x41,0x02,0x9C,0xEC,0x1B,0xB6,0xEA,0x02,0xE1,0xFF,0xB2,0x1D,0xFB,0xD7,0x24,0x38,0xF7,0x03,0x02,0x07,0x00,0xDC,0x08,0x1B,0x5E,0x64,0x05,0xD0,0xAC
			};	//	sSD8BitYUVF2WithAudio
			NTV2Buffer F1Buffer(&sCORVID880Ch1Input720x4868BITYCBCRFrm3F1[0], sCORVID880Ch1Input720x4868BITYCBCRFrm3F1.size()),
						F2Buffer(&sCORVID880Ch1Input720x4868BITYCBCRFrm3F2[0], sCORVID880Ch1Input720x4868BITYCBCRFrm3F2.size());
			AJAAncillaryList pkts;
			CHECK(AJA_SUCCESS(AJAAncillaryList::SetFromDeviceAncBuffers (F1Buffer, F2Buffer, pkts, 1234)));
//cout << "AnalogTest -- " << pkts << endl;
			CHECK_EQ(pkts.CountAncillaryData(),7);	//	There should be 7 packets
//0: [Dig|A|DS1|Y|L9|HANC|xF4x00|CS50|DC=16|SDI|RP-165 Error Detect/Checkwords]: 70C49030B4A400000000000000000000
			AJAAncillaryData & pkt = *(pkts.GetAncillaryDataAtIndex(0));
				CHECK_EQ(pkt.GetDataCoding(), AJAAncDataCoding_Digital);
				CHECK_EQ(pkt.GetLocationVideoLink(), AJAAncDataLink_A);
				CHECK_EQ(pkt.GetLocationDataStream(), AJAAncDataStream_1);
				CHECK_EQ(pkt.GetLocationDataChannel(), AJAAncDataChannel_Y);
				CHECK_EQ(pkt.GetLocationLineNumber(), 9);
				CHECK_EQ(pkt.GetLocationVideoSpace(), AJAAncDataSpace_HANC);
				CHECK_EQ(pkt.GetDID(), 0xF4);
				CHECK_EQ(pkt.GetSID(), 0x00);
				CHECK_EQ(pkt.GetChecksum(), 0x50);
				CHECK_EQ(pkt.GetPayloadByteCount(), 16);
				CHECK_EQ(pkt.GetBufferFormat(), AJAAncBufferFormat_SDI);
				CHECK_EQ(pkt.GetFrameID(), 1234);
				CHECK_NE(pkt.AsString(16).find("70C49030B4A400000000000000000000"), string::npos);
//1: [Dig|A|DS1|Y|L11|HANC|x60x60|CS30|DC=16|SDI|SMPTE-12M ATC Timecode]: 30004000400010302000500000000000
			pkt = *(pkts.GetAncillaryDataAtIndex(1));
				CHECK_EQ(pkt.GetDataCoding(), AJAAncDataCoding_Digital);
				CHECK_EQ(pkt.GetLocationVideoLink(), AJAAncDataLink_A);
				CHECK_EQ(pkt.GetLocationDataStream(), AJAAncDataStream_1);
				CHECK_EQ(pkt.GetLocationDataChannel(), AJAAncDataChannel_Y);
				CHECK_EQ(pkt.GetLocationLineNumber(), 11);
				CHECK_EQ(pkt.GetLocationVideoSpace(), AJAAncDataSpace_HANC);
				CHECK_EQ(pkt.GetDID(), 0x60);
				CHECK_EQ(pkt.GetSID(), 0x60);
				CHECK_EQ(pkt.GetChecksum(), 0x30);
				CHECK_EQ(pkt.GetPayloadByteCount(), 16);
				CHECK_EQ(pkt.GetBufferFormat(), AJAAncBufferFormat_SDI);
				CHECK_EQ(pkt.GetFrameID(), 1234);
				CHECK_NE(pkt.AsString(16).find("30004000400010302000500000000000"), string::npos);
//2: [Dig|A|DS1|Y|L13|HANC|x41x01|CSCE|DC=4|SDI|SMPTE-352M Payload ID]: 81060001
			pkt = *(pkts.GetAncillaryDataAtIndex(2));
				CHECK_EQ(pkt.GetDataCoding(), AJAAncDataCoding_Digital);
				CHECK_EQ(pkt.GetLocationVideoLink(), AJAAncDataLink_A);
				CHECK_EQ(pkt.GetLocationDataStream(), AJAAncDataStream_1);
				CHECK_EQ(pkt.GetLocationDataChannel(), AJAAncDataChannel_Y);
				CHECK_EQ(pkt.GetLocationLineNumber(), 13);
				CHECK_EQ(pkt.GetLocationVideoSpace(), AJAAncDataSpace_HANC);
				CHECK_EQ(pkt.GetDID(), 0x41);
				CHECK_EQ(pkt.GetSID(), 0x01);
				CHECK_EQ(pkt.GetChecksum(), 0xCE);
				CHECK_EQ(pkt.GetPayloadByteCount(), 4);
				CHECK_EQ(pkt.GetBufferFormat(), AJAAncBufferFormat_SDI);
				CHECK_EQ(pkt.GetFrameID(), 1234);
				CHECK_NE(pkt.AsString(16).find("81060001"), string::npos);
//3: [Ana|A|DS1|Y|L21|VANC|x00x00|CS00|DC=720|SDI|SMPTE-291 Control Packet]: 10101010101010101010101010101011...
			pkt = *(pkts.GetAncillaryDataAtIndex(3));
				CHECK_EQ(pkt.GetDataCoding(), AJAAncDataCoding_Raw);
				CHECK_EQ(pkt.GetLocationVideoLink(), AJAAncDataLink_A);
				CHECK_EQ(pkt.GetLocationDataStream(), AJAAncDataStream_1);
				CHECK_EQ(pkt.GetLocationDataChannel(), AJAAncDataChannel_Y);
				CHECK_EQ(pkt.GetLocationLineNumber(), 21);
				CHECK_EQ(pkt.GetLocationVideoSpace(), AJAAncDataSpace_VANC);
				CHECK_EQ(pkt.GetDID(), 0);
				CHECK_EQ(pkt.GetSID(), 0);
				CHECK_EQ(pkt.GetChecksum(), 0);
				CHECK_EQ(pkt.GetPayloadByteCount(), 720);
				CHECK_EQ(pkt.GetBufferFormat(), AJAAncBufferFormat_SDI);
				CHECK_EQ(pkt.GetFrameID(), 1234);
				CHECK_NE(pkt.AsString(16).find("10101010101010101010101010101011"), string::npos);
//4: [Dig|A|DS1|Y|L272|HANC|xF4x00|CS4C|DC=16|SDI|RP-165 Error Detect/Checkwords]: 8CA8A4C41C9000000000000000000000
			pkt = *(pkts.GetAncillaryDataAtIndex(4));
				CHECK_EQ(pkt.GetDataCoding(), AJAAncDataCoding_Digital);
				CHECK_EQ(pkt.GetLocationVideoLink(), AJAAncDataLink_A);
				CHECK_EQ(pkt.GetLocationDataStream(), AJAAncDataStream_1);
				CHECK_EQ(pkt.GetLocationDataChannel(), AJAAncDataChannel_Y);
				CHECK_EQ(pkt.GetLocationLineNumber(), 272);
				CHECK_EQ(pkt.GetLocationVideoSpace(), AJAAncDataSpace_HANC);
				CHECK_EQ(pkt.GetDID(), 0xF4);
				CHECK_EQ(pkt.GetSID(), 0x00);
				CHECK_EQ(pkt.GetChecksum(), 0x4C);
				CHECK_EQ(pkt.GetPayloadByteCount(), 16);
				CHECK_EQ(pkt.GetBufferFormat(), AJAAncBufferFormat_SDI);
				CHECK_EQ(pkt.GetFrameID(), 1234);
				CHECK_NE(pkt.AsString(16).find("8CA8A4C41C9000000000000000000000"), string::npos);
//5: [Dig|A|DS1|Y|L276|HANC|x41x01|CSCE|DC=4|SDI|SMPTE-352M Payload ID]: 81060001
			pkt = *(pkts.GetAncillaryDataAtIndex(5));
				CHECK_EQ(pkt.GetDataCoding(), AJAAncDataCoding_Digital);
				CHECK_EQ(pkt.GetLocationVideoLink(), AJAAncDataLink_A);
				CHECK_EQ(pkt.GetLocationDataStream(), AJAAncDataStream_1);
				CHECK_EQ(pkt.GetLocationDataChannel(), AJAAncDataChannel_Y);
				CHECK_EQ(pkt.GetLocationLineNumber(), 276);
				CHECK_EQ(pkt.GetLocationVideoSpace(), AJAAncDataSpace_HANC);
				CHECK_EQ(pkt.GetDID(), 0x41);
				CHECK_EQ(pkt.GetSID(), 0x01);
				CHECK_EQ(pkt.GetChecksum(), 0xCE);
				CHECK_EQ(pkt.GetPayloadByteCount(), 4);
				CHECK_EQ(pkt.GetBufferFormat(), AJAAncBufferFormat_SDI);
				CHECK_EQ(pkt.GetFrameID(), 1234);
				CHECK_NE(pkt.AsString(16).find("81060001"), string::npos);
//6: [Ana|A|DS1|Y|L284|VANC|x00x00|CS00|DC=720|SDI|SMPTE-291 Control Packet]: 10101010101010101010101010101011...
			pkt = *(pkts.GetAncillaryDataAtIndex(6));
				CHECK_EQ(pkt.GetDataCoding(), AJAAncDataCoding_Raw);
				CHECK_EQ(pkt.GetLocationVideoLink(), AJAAncDataLink_A);
				CHECK_EQ(pkt.GetLocationDataStream(), AJAAncDataStream_1);
				CHECK_EQ(pkt.GetLocationDataChannel(), AJAAncDataChannel_Y);
				CHECK_EQ(pkt.GetLocationLineNumber(), 284);
				CHECK_EQ(pkt.GetLocationVideoSpace(), AJAAncDataSpace_VANC);
				CHECK_EQ(pkt.GetDID(), 0);
				CHECK_EQ(pkt.GetSID(), 0);
				CHECK_EQ(pkt.GetChecksum(), 0);
				CHECK_EQ(pkt.GetPayloadByteCount(), 720);
				CHECK_EQ(pkt.GetBufferFormat(), AJAAncBufferFormat_SDI);
				CHECK_EQ(pkt.GetFrameID(), 1234);
				CHECK_NE(pkt.AsString(16).find("10101010101010101010101010101011"), string::npos);
			F1Buffer.Set(AJA_NULL, 0);
			F2Buffer.Set(&sSD8BitYUVF2WithAudio[0], sSD8BitYUVF2WithAudio.size());
			pkts.Clear();
			CHECK(AJA_SUCCESS(AJAAncillaryList::SetFromDeviceAncBuffers (F1Buffer, F2Buffer, pkts, 4321)));
cout << "AnalogTest -- " << pkts << endl;
			CHECK_EQ(pkts.CountAncillaryData(),68);	//	There should be 68 packets
		}	//	TEST_CASE("AnalogTest")

#if 0	//	** MrBill **	NOT READY FOR PRIME-TIME
		TEST_CASE("BFT_AncDataCEA608Analog")	
		{
			static const uint8_t			pGump608Raw[]	=	{	0xFF,	0xE0,	0x00,	0x00,	0x00,	0xFF,
																	0x24,	0x47,	0x6a,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x11,	0x16,	0x1d,	0x26,	0x31,	0x3d,	0x4a,	0x56,	0x62,	0x6c,	0x74,
																	0x7a,	0x7d,	0x7d,	0x7a,	0x74,	0x6c,	0x62,	0x56,	0x4a,	0x3d,	0x31,	0x26,	0x1d,	0x16,	0x11,	0x10,	0x11,	0x16,	0x1d,	0x26,	0x31,	0x3d,	0x4a,	0x56,	0x62,	0x6c,	0x74,	0x7a,	0x7d,	0x7d,	0x7a,	0x74,
																	0x6c,	0x62,	0x56,	0x4a,	0x3d,	0x31,	0x26,	0x1d,	0x16,	0x11,	0x10,	0x11,	0x16,	0x1d,	0x26,	0x31,	0x3d,	0x4a,	0x56,	0x62,	0x6c,	0x74,	0x7a,	0x7d,	0x7d,	0x7a,	0x74,	0x6c,	0x62,	0x56,	0x4a,	0x3d,
																	0x31,	0x26,	0x1d,	0x16,	0x11,	0x10,	0x11,	0x16,	0x1d,	0x26,	0x31,	0x3d,	0x4a,	0x56,	0x62,	0x6c,	0x74,	0x7a,	0x7d,	0x7d,	0x7a,	0x74,	0x6c,	0x62,	0x56,	0x4a,	0x3d,	0x31,	0x26,	0x1d,	0x16,	0x11,
																	0x10,	0x11,	0x16,	0x1d,	0x26,	0x31,	0x3d,	0x4a,	0x56,	0x62,	0x6c,	0x74,	0x7a,	0x7d,	0x7d,	0x7a,	0x74,	0x6c,	0x62,	0x56,	0x4a,	0x3d,	0x31,	0x26,	0x1d,	0x16,	0x11,	0x10,	0x11,	0x16,	0x1d,	0x26,
																	0x31,	0x3d,	0x4a,	0x56,	0x62,	0x6c,	0x74,	0x7a,	0x7d,	0x7d,	0x7a,	0x74,	0x6c,	0x62,	0x56,	0x4a,	0x3d,	0x31,	0x26,	0x1d,	0x16,	0x11,	0x10,	0x11,	0x16,	0x1d,	0x26,	0x31,	0x3d,	0x4a,	0x56,	0x62,
																	0x6c,	0x74,	0x7a,	0x7d,	0x7d,	0x7a,	0x74,	0x6c,	0x62,	0x56,	0x4a,	0x3d,	0x31,	0x26,	0x1d,	0x16,	0x11,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,
																	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x7e,	0x7e,
																	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x00,	0xff,	0xe0,	0x00,	0x00,	0x00,	0xff,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x10,
																	0x10,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x6a,	0x47,	0x24,	0x10,
																	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,
																	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,
																	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,
																	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,
																	0x10,	0x10,	0x10,	0x24,	0x47,	0x6a,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,
																	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x6a,	0x47,	0x24,	0x10,	0x10,	0x10,	0x10,
																	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x00,	0xff,	0xe0,	0x00,	0x00,	0x00,	0xd2,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x24,	0x47,	0x6a,	0x7e,	0x7e,
																	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,
																	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,
																	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x6a,	0x47,	0x24,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,
																	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x24,	0x47,	0x6a,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,
																	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x7e,	0x6a,	0x47,	0x24,
																	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,
																	0x10,	0x10,	0x10,	0x10,	0x00};
			uint32_t						packetByteCount	(0);
			AJAAncillaryData_Cea608_Line21	pktRX, pktTX;
			AJAAncillaryData				defaultPkt;
			uint8_t							char1(0), char2(0);
			bool							isValid(false);

			//	Test AJAAncillaryData_Cea608_Line21  and GUMP decoding of it
			CHECK(AJA_SUCCESS(defaultPkt.InitWithReceivedData (pGump608Raw, sizeof(pGump608Raw), AJAAncDataLoc(), packetByteCount)));
			CHECK_EQ (defaultPkt.GetDC(), 255);
			CHECK_EQ (defaultPkt.GetDID(), 0);
			CHECK_EQ (defaultPkt.GetSID(), 0);
			CHECK_EQ (defaultPkt.GetChecksum(), 0);
			CHECK_EQ (defaultPkt.GetLocationVideoLink(), AJAAncDataLink_Unknown);
			CHECK_EQ (defaultPkt.GetLocationDataChannel(), AJAAncDataChannel_Y);
			CHECK_EQ (defaultPkt.GetLocationVideoSpace(), AJAAncDataSpace_VANC);
			CHECK_EQ (defaultPkt.GetLocationLineNumber(), 9);
			CHECK_EQ (defaultPkt.GetDataCoding(), AJAAncDataCoding_Raw);
			CHECK_FALSE (defaultPkt.GotValidReceiveData());		//	False, because wasn't vetted by specific subclass
			CHECK_EQ (defaultPkt.GetAncillaryDataType(), AJAAncDataType_Unknown);
			CHECK_NE (defaultPkt.GetAncillaryDataType(), AJAAncDataType_Cea608_Line21);
			CHECK_EQ (AJAAncillaryData_Cea608_Vanc::RecognizeThisAncillaryData(&defaultPkt), AJAAncDataType_Unknown);
			CHECK_EQ (AJAAncillaryData_Cea608_Line21::RecognizeThisAncillaryData(&defaultPkt), AJAAncDataType_Cea608_Line21);
			CHECK_EQ (AJAAncillaryData_Cea708::RecognizeThisAncillaryData(&defaultPkt), AJAAncDataType_Unknown);
			CHECK(AJA_SUCCESS(pktRX.InitWithReceivedData (pGump608Raw, sizeof(pGump608Raw), AJAAncDataLoc(), packetByteCount)));
			CHECK(AJA_SUCCESS(pktRX.ParsePayloadData()));
			CHECK (pktRX.GotValidReceiveData());
			CHECK(AJA_SUCCESS(pktRX.GetCEA608Characters(char1, char2, isValid)));
			CHECK_EQ (char1, 'A');
			CHECK_EQ (char2, 'n');
			CHECK (isValid);

			AJAAncillaryList	packetList;
			CHECK(AJA_SUCCESS(packetList.AddReceivedAncillaryData (pGump608Raw, sizeof(pGump608Raw))));
			cerr << endl << packetList << endl;


			AJAAncillaryData_Cea608_Line21 * pLine21Packet = reinterpret_cast <AJAAncillaryData_Cea608_Line21 *> (AJAAncillaryDataFactory::Create(AJAAncDataType_Cea608_Line21));
			CHECK(pLine21Packet != AJA_NULL);
			CHECK(AJA_SUCCESS(pLine21Packet->GeneratePayloadData()));
			CHECK(AJA_SUCCESS(pLine21Packet->SetDataLocation (AJAAncDataLoc(AJAAncDataLink_A, AJAAncDataChannel_Y, AJAAncDataSpace_VANC, 21))));
			CHECK(AJA_SUCCESS(pLine21Packet->SetCEA608Bytes (AJAAncillaryData_Cea608::AddOddParity('A'), AJAAncillaryData_Cea608::AddOddParity('b'))));
			CHECK(AJA_SUCCESS(pLine21Packet->SetDID(AJAAncillaryData_Cea608_Vanc_DID)));
			CHECK(AJA_SUCCESS(pLine21Packet->SetSID(AJAAncillaryData_Cea608_Vanc_SID)));
			//pLine21Packet->Print (cerr, true);
			CHECK_EQ (AJAAncillaryDataFactory::GuessAncillaryDataType(pLine21Packet), AJAAncDataType_Cea608_Line21);
		}	//	TEST_CASE("BFT_AncDataCEA608Analog")
#endif	//	DISABLED FOR NOW

		TEST_CASE("BFT_AncDataCEA708")
		{
			static const uint8_t		pGump708	[]=	{	0xFF,	0xA0,	0x09,	0x61,	0x01,	0x52,	0x96,	0x69,	0x52,	0x4F,	0x67,	0xA7,	0x9A,	0x72,	0xF4,	0xFC,	0x80,
															0x80,	0xF9,	0x00,	0x00,	0xFB,	0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,
															0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,
															0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,	0x00,
															0x00,	0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,	0x73,	0x91,	0xE1,	0x00,	0x00,	0x00,
															0xC1,	0x3F,	0xFF,	0x74,	0xA7,	0x9A,	0x2F,	0xB4,	//	end of packet
															0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10};
			uint32_t					packetByteCount	(0);
			AJAAncillaryData_Cea708		pktRX, pktTX;
			AJAAncillaryData			defaultPkt;

			//	Test AJAAncillaryData_Cea708  and GUMP decoding of it
			CHECK(AJA_SUCCESS(defaultPkt.InitWithReceivedData (pGump708, sizeof(pGump708), AJAAncDataLoc(), packetByteCount)));
			CHECK_EQ (defaultPkt.GetDC(), 82);
			CHECK_EQ (defaultPkt.GetDID(), 0x61);
			CHECK_EQ (defaultPkt.GetSID(), 0x01);
			CHECK_EQ (defaultPkt.GetChecksum(), 0xB4);
			CHECK_EQ (defaultPkt.GetLocationVideoLink(), AJAAncDataLink_Unknown);
			CHECK_EQ (defaultPkt.GetLocationDataChannel(), AJAAncDataChannel_Y);
			CHECK_EQ (defaultPkt.GetLocationVideoSpace(), AJAAncDataSpace_VANC);
			CHECK_EQ (defaultPkt.GetLocationLineNumber(), 9);
			CHECK_EQ (defaultPkt.GetDataCoding(), AJAAncDataCoding_Digital);
			CHECK_FALSE (defaultPkt.GotValidReceiveData());		//	False, because wasn't vetted by specific subclass
			CHECK_EQ (defaultPkt.GetAncillaryDataType(), AJAAncDataType_Unknown);
			CHECK_NE (defaultPkt.GetAncillaryDataType(), AJAAncDataType_Cea608_Vanc);
			CHECK_EQ (AJAAncillaryData_Cea608_Vanc::RecognizeThisAncillaryData(&defaultPkt), AJAAncDataType_Unknown);
			CHECK_EQ (AJAAncillaryData_Cea608_Line21::RecognizeThisAncillaryData(&defaultPkt), AJAAncDataType_Unknown);
			CHECK_EQ (AJAAncillaryData_Cea708::RecognizeThisAncillaryData(&defaultPkt), AJAAncDataType_Cea708);
			CHECK(AJA_SUCCESS(pktRX.InitWithReceivedData (pGump708, sizeof(pGump708), AJAAncDataLoc(), packetByteCount)));
			CHECK(AJA_SUCCESS(pktRX.ParsePayloadData()));
			CHECK (pktRX.GotValidReceiveData());

			//	Test GUMP encoding...
			CHECK(AJA_SUCCESS(pktTX.SetPayloadData(pktRX.GetPayloadData(), uint32_t(pktRX.GetPayloadByteCount()))));
			CHECK_EQ(::memcmp(pktTX.GetPayloadData(), pktRX.GetPayloadData(), pktTX.GetDC()), 0);
			//CHECK_EQ(pktRX, pktTX);
		}	//	TEST_CASE("BFT_AncDataCEA708")


		TEST_CASE("BFT_YUVComponentsTo10BitYUVPackedBuffer")
		{
			const vector<uint16_t> & in10BitYUVReferenceLine(SD10BitYUVComponents);
			NTV2FormatDescriptor	fd		(NTV2_STANDARD_525, NTV2_FBF_10BIT_YCBCR, NTV2_VANCMODE_TALLER);
			NTV2Buffer				fb		(size_t(fd.GetTotalRasterBytes() - fd.GetVisibleRasterBytes()));	//	Just the VANC lines
			UWord					fbRow	(0);
			vector<uint16_t>		YUVLine;

			//	Validate "YUVLine too small" failures...
			fb.Fill(UByte(0x80));  YUVLine = in10BitYUVReferenceLine;  YUVLine.resize(5);
			CHECK_FALSE(::YUVComponentsTo10BitYUVPackedBuffer (YUVLine,  fb,  fd,  fbRow));	//	YUVLine too small
			fb.Fill(UByte(0x80));  YUVLine = in10BitYUVReferenceLine;  YUVLine.resize(10);
			CHECK_FALSE(::YUVComponentsTo10BitYUVPackedBuffer (YUVLine,  fb,  fd,  fbRow));	//	YUVLine too small
			fb.Fill(UByte(0x80));  YUVLine = in10BitYUVReferenceLine;  YUVLine.resize(15);
			CHECK(::YUVComponentsTo10BitYUVPackedBuffer (YUVLine,  fb,  fd,  fbRow));

			//	Validate buffer size failures...
			YUVLine = in10BitYUVReferenceLine;		//	Valid, full input line
			CHECK(fb.Allocate(0));		//	NULL buffer
			CHECK_FALSE(::YUVComponentsTo10BitYUVPackedBuffer (YUVLine,  fb,  fd,  fbRow));	//	Buffer is NULL
			CHECK(fb.Allocate(256));	//	256-byte buffer
			CHECK_FALSE(::YUVComponentsTo10BitYUVPackedBuffer (YUVLine,  fb,  fd,  fbRow));	//	Buffer too small
			CHECK(fb.Allocate(512));	//	512-byte buffer
			CHECK_FALSE(::YUVComponentsTo10BitYUVPackedBuffer (YUVLine,  fb,  fd,  fbRow));	//	Buffer too small
			CHECK(fb.Allocate(fd.GetBytesPerRow()));	//	One-line buffer
			CHECK(::YUVComponentsTo10BitYUVPackedBuffer (YUVLine,  fb,  fd,  fbRow));
			CHECK(fb.Allocate(size_t(fd.GetTotalRasterBytes() - fd.GetVisibleRasterBytes())));	//	Back to a full-VANC buffer

			//	Validate line number failures...
			CHECK_FALSE(::YUVComponentsTo10BitYUVPackedBuffer (YUVLine,  fb,  fd,  fd.GetFirstActiveLine()+20));	//	Line out of bounds for given buffer
			CHECK_FALSE(::YUVComponentsTo10BitYUVPackedBuffer (YUVLine,  fb,  fd,  fd.GetFirstActiveLine()));		//	Line out of bounds for given buffer
			CHECK(::YUVComponentsTo10BitYUVPackedBuffer (YUVLine,  fb,  fd,  fd.GetFirstActiveLine()/2));	//	OK: Line within VANC

			//	Validate format descriptor failures...
			CHECK_FALSE(::YUVComponentsTo10BitYUVPackedBuffer (YUVLine,  fb,  NTV2FormatDescriptor(),  fbRow));	//	Bad format descriptor

			//	Validate packed data...
			fb.Fill(UByte(0x80));
			CHECK(::YUVComponentsTo10BitYUVPackedBuffer (YUVLine,  fb,  fd,  0));
			//	Uncomment the following 2 lines to generate the static kCompPackedData array from the contents of line 0 in "fd" buffer...
			//	const uint64_t *	pData	((const uint64_t *) fb.GetHostPointer());
			//	cerr << "\t\t\tstatic const uint64_t kCompPackedData64[] = {";  for (UWord ndx(0);  ndx < fd.GetBytesPerRow()/sizeof(uint64_t);  ndx++)	{cerr << "\t" << xHEX0N(pData[ndx],16) << ",";  if((ndx+1)%6==0)cerr << endl << "\t\t\t\t\t\t\t\t\t\t\t\t\t\t";}	cerr << "\t};" << endl;
			static const uint64_t kCompPackedData64[] = {	0x2D8405453FFFFC00,	0x1258912322288520,	0x22B4A92922889D26,	0x1318C12F22E8B52C,	0x1378DA351348CD32,	0x13D8F13B23A8E538,
															0x14390A411408FD3E,	0x1499224714651644,	0x14F93A4D14C92D4A,	0x2555525315254650,	0x15B96A5915855E56,	0x1619825F15E5765C,
															0x16799A6516498D62,	0x16D9B16B26A9A568,	0x1739CA711709BD6E,	0x1799E2771765D674,	0x17F9FA7D17C9ED7A,	0x185A1183282A0580,
															0x28B62989288A1D86,	0x191A418F28EA358C,	0x197A5A95194A4D92,	0x19DA719B29AA6598,	0x2A3689A12A0A7D9E,	0x2A96A1A72A6A95A4,
															0x2AF6B9AD2AC6AEAA,	0x1B5AD1B32B2AC5B0,	0x2BB6E9B92B8ADDB6,	0x1C1B01BF2BEAF5BC,	0x1C7B1AC51C4B0DC2,	0x1CDB31CB2CAB25C8,
															0x1D3B4AD11D0B3DCE,	0x1D9B62D71D6756D4,	0x1DFB7ADD1DCB6DDA,	0x1E5B91E32E2B85E0,	0x2EB7A9E92E8B9DE6,	0x1F1BC1EF2EEBB5EC,
															0x1F7BDAF51F4BCDF2,	0x102517FF3FF002D2,	0x0408016220A83D02,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,
															0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200,	0x0408004020010200	};
			CHECK_EQ(::memcmp(kCompPackedData64, fb.GetHostPointer(), fd.GetBytesPerRow()), 0);
		}	//	TEST_CASE("BFT_YUVComponentsTo10BitYUVPackedBuffer")

#if 0	//	DISABLED BY DEFAULT
		TEST_CASE("RTPTimingTest")	//	Normally Disabled
		{
			unsigned numRoundTrips(1000);

			{	// How fast is AJAAncillaryList at adding and removing packets?
				AJAPerformance perfAncList("AncListTimingTest");
				perfAncList.Start();
				for (unsigned tripNum(0);  tripNum < numRoundTrips;  tripNum++)
				{
					AJAAncillaryList pktListA;
					for (unsigned pktNum(0);  pktNum < 1000;  pktNum++)
						pktListA.AddAncillaryData(AJAAncillaryData());
					AJAAncillaryList pktListB(pktListA);
					while (pktListA.CountAncillaryData())
						pktListA.DeleteAncillaryData(pktListA.GetAncillaryDataAtIndex(0));
					while (pktListB.CountAncillaryData())
						pktListB.DeleteAncillaryData(pktListB.GetAncillaryDataAtIndex(pktListB.CountAncillaryData()-1));
				}	//	for numRoundTrips
				perfAncList.Stop();
				perfAncList.Report();
			}

			// Make some test packets for RTP conversion...
			unsigned numF1Packets(90), numF2Packets(163);
			vector<uint8_t>	payloadData;
			for (unsigned ndx(0);  ndx < 255;  ndx++)
				payloadData.push_back(ndx);
			NTV2Buffer	payloadBuffer(255);
			payloadBuffer.PutU8s(payloadData);
			AJAAncillaryList	ancPkts;
			AJAAncDataLoc	loc;
			for (unsigned pktNum(0);  pktNum < numF1Packets;  pktNum++)
			{
				loc.Reset().SetDataLink(pktNum & 1 ? AJAAncDataLink_A : AJAAncDataLink_B);
				loc.SetDataChannel(pktNum & 1 ? AJAAncDataChannel_C : AJAAncDataChannel_Y);
				loc.SetDataSpace(AJAAncDataSpace_VANC).SetDataStream(AJAAncDataStream(pktNum & 3));
				loc.SetLineNumber(9+(pktNum&0x0F));
				loc.SetHorizontalOffset(AJAAncDataHorizOffset_Anywhere);
				AJAAncillaryData pkt;
				pkt.SetDID(pktNum);
				pkt.SetSID(200-pktNum);
				pkt.SetDataCoding(AJAAncDataCoding_Digital);
				pkt.SetDataLocation(loc);
				pkt.SetPayloadData(&payloadData[0], uint32_t(payloadData.size()));
				ancPkts.AddAncillaryData(pkt);
			}	//	for numF1Packets
			for (unsigned pktNum(0);  pktNum < numF2Packets;  pktNum++)
			{
				loc.Reset().SetDataLink(pktNum & 1 ? AJAAncDataLink_A : AJAAncDataLink_B);
				loc.SetDataChannel(pktNum & 1 ? AJAAncDataChannel_C : AJAAncDataChannel_Y);
				loc.SetDataSpace(AJAAncDataSpace_VANC).SetDataStream(AJAAncDataStream(pktNum & 3));
				loc.SetLineNumber(567+(pktNum&0x0F));
				loc.SetHorizontalOffset(AJAAncDataHorizOffset_Anywhere);
				AJAAncillaryData pkt;
				pkt.SetDID(pktNum);
				pkt.SetSID(200-pktNum);
				pkt.SetDataCoding(AJAAncDataCoding_Digital);
				pkt.SetDataLocation(loc);
				pkt.SetPayloadData(&payloadData[0], uint32_t(payloadData.size()));
				ancPkts.AddAncillaryData(pkt);
			}	//	for numF2Packets

			NTV2Buffer			F1Buffer(4096),	F2Buffer(4096);
			AJAAncillaryList	packetList;
	        AJAPerformance		perfOverall("RTPTimingTest"), perfTx("RTPTimingTx"), perfRx("RTPTimingRx");
	        perfOverall.Start();
			for (unsigned tripNum(0);  tripNum < numRoundTrips;  tripNum++)
			{
				perfTx.Start();
				ancPkts.GetIPTransmitData (F1Buffer, F2Buffer, false, 564);
				perfTx.Stop();
				perfRx.Start();
				AJAAncillaryList::SetFromDeviceAncBuffers (F1Buffer, F2Buffer, packetList);
				perfRx.Stop();
			}	//	for numRoundTrips
			perfOverall.Stop();
			perfTx.Report();
			perfRx.Report();
			perfOverall.Report();
		}	//	TEST_CASE("RTPTimingTest")
#endif	//	DISABLED FOR NOW


#if defined(AJA_USE_CPLUSPLUS11)
		TEST_CASE("BFT_AncListMoveSemantics")
		{
			const string myPacketData ("This is a test packet to be used with the new move semantics added to AJAAncillaryList");
			AJAAncillaryData pkt;
			pkt.SetDID(0xAA);  pkt.SetSID(0xBB);
			pkt.SetPayloadData(reinterpret_cast<const uint8_t*>(myPacketData.c_str()), myPacketData.length());
			AJAAncillaryList pktsC;

			//	Create "A" list of 1024 packets...
			AJAAncillaryList pktsA;
			while (pktsA.CountAncillaryData() < 1024)
				pktsA.AddAncillaryData(pkt);
			CHECK_EQ(pktsA.CountAncillaryData(), 1024);
			CHECK_EQ(pktsC.CountAncillaryData(), 0);

			//	Move A's packets into new "B" list...
			AJAAncillaryList pktsB(std::move(pktsA));
			CHECK_EQ(pktsC.CountAncillaryData(), 0);
			CHECK_EQ(pktsA.CountAncillaryData(), 0);
			CHECK_EQ(pktsB.CountAncillaryData(), 1024);

			//	Move B's packets into "C"...
			pktsC = std::move(pktsB);
			CHECK_EQ(pktsB.CountAncillaryData(), 0);
			CHECK_EQ(pktsC.CountAncillaryData(), 1024);
		}	//	TEST_CASE("BFT_AncDataCompare")
#endif	//	defined(AJA_USE_CPLUSPLUS11)


//	This explicitly tests AJAAncillaryData::GenerateTransmitData:
#if 0
static bool GenerateIPPacketWordsBFT (void)
{
	static const string	pkts[]	=	{	"\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0A\x0B\x0C\x0D\x0E\x0F\x10\x11\x12",
										"\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0A\x0B\x0C\x0D\x0E\x0F\x10\x11",
										"\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0A\x0B\x0C\x0D\x0E\x0F\x10",
										"\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0A\x0B\x0C\x0D\x0E\x0F",
										"\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0A\x0B\x0C\x0D\x0E",
										"\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0A\x0B\x0C\x0D",
										"\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0A\x0B\x0C",
										"\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0A\x0B",
										"\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0A",
										"\x01\x02\x03\x04\x05\x06\x07\x08\x09",
										"\x01\x02\x03\x04\x05\x06\x07\x08",
										"\x01\x02\x03\x04\x05\x06\x07",
										"\x01\x02\x03\x04\x05\x06",
										"\x01\x02\x03\x04\x05",
										"\x01\x02\x03\x04",
										"\x01\x02\x03",
										"\x01\x02",
										"\x01",
										""};
	vector<uint32_t>	pktWords;
	unsigned			ndx(0);
	while(true)
	{
		AJAAncillaryData pkt;
		const string & data(pkts[ndx++]);
		pkt.SetDID(0x33);  pkt.SetSID(0x55);  pkt.SetLocationLineNumber(9);
		if (!data.empty())
			CHECK(AJA_SUCCESS(pkt.SetPayloadData(reinterpret_cast<const uint8_t *>(data.c_str()), uint32_t(data.length()))));
		CHECK(AJA_SUCCESS(pkt.GenerateTransmitData(pktWords)));
		if (data.empty())
			break;	//	Done
	};
	return true;
}
#endif


#if 0	///////////////	BEGIN TAILING DUMP
	//AJAAncillaryData_Cea608_Line21		line21Packet;
	/*static const uint8_t				p2vuyLumaSamples[]=	{	0x00,	0xFF,	0xFF,	0x61,	0x01,	0x52,	0x96,	0x69,	0x52,	0x4F,	0x67,	0xA9,	0x7E,	0x72,	0xF4,	0xFC,
																0x20,	0x73,	0xF9,	0x00,	0x00,	0xFF,	0x46,	0x28,	0xFE,	0x73,	0x65,	0xFE,	0x6D,	0x69,	0xFE,	0x6E,
																0x61,	0xFE,	0x72,	0x79,	0xFE,	0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,
																0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,
																0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,	0xFA,	0x00,	0x00,	0x73,	0x91,	0xE1,	0x00,	0x00,
																0x00,	0xC1,	0x3F,	0xFF,	0x74,	0xA9,	0x7E,	0xE2,	0xB4,	//	end of packet
																0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10,	0x10};	*/
	//static const uint8_t				p8_RDD6Pkt2 []	=	{	/*		0x00,	0xFF,	0xFF,	0x45,	0x01,	0x02,*/	0x0F,	0x0A	/*		0xXX	*/	};
	//static const uint16_t				p16_RDD6Pkt2 []	=	{	0x200,	0x20F,	0x20A	};
#endif	///////////////	END TAILING DUMP
