/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.collect;

import com.google.common.annotations.GwtCompatible;
import com.google.common.base.Preconditions;
import com.google.common.collect.NullnessCasts;
import com.google.common.collect.Ordering;
import com.google.common.collect.ParametricNullness;
import com.google.common.math.IntMath;
import java.math.RoundingMode;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import org.jspecify.annotations.Nullable;

@GwtCompatible
final class TopKSelector<T> {
    private final int k;
    private final Comparator<? super T> comparator;
    private final @Nullable T[] buffer;
    private int bufferSize;
    private @Nullable T threshold;

    public static <T extends Comparable<? super T>> TopKSelector<T> least(int k15) {
        return TopKSelector.least(k15, Ordering.natural());
    }

    public static <T> TopKSelector<T> least(int k15, Comparator<? super T> comparator) {
        return new TopKSelector<T>(comparator, k15);
    }

    public static <T extends Comparable<? super T>> TopKSelector<T> greatest(int k15) {
        return TopKSelector.greatest(k15, Ordering.natural());
    }

    public static <T> TopKSelector<T> greatest(int k15, Comparator<? super T> comparator) {
        return new TopKSelector(Ordering.from(comparator).reverse(), k15);
    }

    private TopKSelector(Comparator<? super T> comparator, int k15) {
        this.comparator = Preconditions.checkNotNull(comparator, "comparator");
        this.k = k15;
        Preconditions.checkArgument(k15 >= 0, "k (%s) must be >= 0", k15);
        Preconditions.checkArgument(k15 <= 0x3FFFFFFF, "k (%s) must be <= Integer.MAX_VALUE / 2", k15);
        this.buffer = new Object[Math.multiplyExact(k15, 2)];
        this.bufferSize = 0;
        this.threshold = null;
    }

    public void offer(@ParametricNullness T elem) {
        if (this.k == 0) {
            return;
        }
        if (this.bufferSize == 0) {
            this.buffer[0] = elem;
            this.threshold = elem;
            this.bufferSize = 1;
        } else if (this.bufferSize < this.k) {
            this.buffer[this.bufferSize++] = elem;
            if (this.comparator.compare(elem, NullnessCasts.uncheckedCastNullableTToT(this.threshold)) > 0) {
                this.threshold = elem;
            }
        } else if (this.comparator.compare(elem, NullnessCasts.uncheckedCastNullableTToT(this.threshold)) < 0) {
            this.buffer[this.bufferSize++] = elem;
            if (this.bufferSize == 2 * this.k) {
                this.trim();
            }
        }
    }

    private void trim() {
        int left = 0;
        int right = 2 * this.k - 1;
        int minThresholdPosition = 0;
        int iterations = 0;
        int maxIterations = IntMath.log2(right - left, RoundingMode.CEILING) * 3;
        while (left < right) {
            int pivotIndex = left + right + 1 >>> 1;
            int pivotNewIndex = this.partition(left, right, pivotIndex);
            if (pivotNewIndex > this.k) {
                right = pivotNewIndex - 1;
            } else {
                if (pivotNewIndex >= this.k) break;
                left = Math.max(pivotNewIndex, left + 1);
                minThresholdPosition = pivotNewIndex;
            }
            if (++iterations < maxIterations) continue;
            T[] castBuffer = this.buffer;
            Arrays.sort(castBuffer, left, right + 1, this.comparator);
            break;
        }
        this.bufferSize = this.k;
        this.threshold = NullnessCasts.uncheckedCastNullableTToT(this.buffer[minThresholdPosition]);
        for (int i15 = minThresholdPosition + 1; i15 < this.k; ++i15) {
            if (this.comparator.compare(NullnessCasts.uncheckedCastNullableTToT(this.buffer[i15]), NullnessCasts.uncheckedCastNullableTToT(this.threshold)) <= 0) continue;
            this.threshold = this.buffer[i15];
        }
    }

    private int partition(int left, int right, int pivotIndex) {
        T pivotValue = NullnessCasts.uncheckedCastNullableTToT(this.buffer[pivotIndex]);
        this.buffer[pivotIndex] = this.buffer[right];
        int pivotNewIndex = left;
        for (int i15 = left; i15 < right; ++i15) {
            if (this.comparator.compare(NullnessCasts.uncheckedCastNullableTToT(this.buffer[i15]), pivotValue) >= 0) continue;
            this.swap(pivotNewIndex, i15);
            ++pivotNewIndex;
        }
        this.buffer[right] = this.buffer[pivotNewIndex];
        this.buffer[pivotNewIndex] = pivotValue;
        return pivotNewIndex;
    }

    private void swap(int i15, int j15) {
        T tmp = this.buffer[i15];
        this.buffer[i15] = this.buffer[j15];
        this.buffer[j15] = tmp;
    }

    TopKSelector<T> combine(TopKSelector<T> other) {
        for (int i15 = 0; i15 < other.bufferSize; ++i15) {
            this.offer(NullnessCasts.uncheckedCastNullableTToT(other.buffer[i15]));
        }
        return this;
    }

    public void offerAll(Iterable<? extends T> elements) {
        this.offerAll(elements.iterator());
    }

    public void offerAll(Iterator<? extends T> elements) {
        while (elements.hasNext()) {
            this.offer(elements.next());
        }
    }

    public List<T> topK() {
        T[] castBuffer = this.buffer;
        Arrays.sort(castBuffer, 0, this.bufferSize, this.comparator);
        if (this.bufferSize > this.k) {
            Arrays.fill(this.buffer, this.k, this.buffer.length, null);
            this.bufferSize = this.k;
            this.threshold = this.buffer[this.k - 1];
        }
        T[] topK = Arrays.copyOf(castBuffer, this.bufferSize);
        return Collections.unmodifiableList(Arrays.asList(topK));
    }
}

