﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/s3/S3Request.h>
#include <aws/s3/S3_EXPORTS.h>
#include <aws/s3/model/EncodingType.h>
#include <aws/s3/model/OptionalObjectAttributes.h>
#include <aws/s3/model/RequestPayer.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace S3 {
namespace Model {

/**
 */
class ListObjectVersionsRequest : public S3Request {
 public:
  AWS_S3_API ListObjectVersionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListObjectVersions"; }

  AWS_S3_API Aws::String SerializePayload() const override;

  AWS_S3_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  AWS_S3_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  AWS_S3_API bool HasEmbeddedError(IOStream& body, const Http::HeaderValueCollection& header) const override;
  /**
   * Helper function to collect parameters (configurable and static hardcoded) required for endpoint computation.
   */
  AWS_S3_API EndpointParameters GetEndpointContextParams() const override;

  ///@{
  /**
   * <p>The bucket name that contains the objects. </p>
   */
  inline const Aws::String& GetBucket() const { return m_bucket; }
  inline bool BucketHasBeenSet() const { return m_bucketHasBeenSet; }
  template <typename BucketT = Aws::String>
  void SetBucket(BucketT&& value) {
    m_bucketHasBeenSet = true;
    m_bucket = std::forward<BucketT>(value);
  }
  template <typename BucketT = Aws::String>
  ListObjectVersionsRequest& WithBucket(BucketT&& value) {
    SetBucket(std::forward<BucketT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A delimiter is a character that you specify to group keys. All keys that
   * contain the same string between the <code>prefix</code> and the first occurrence
   * of the delimiter are grouped under a single result element in
   * <code>CommonPrefixes</code>. These groups are counted as one result against the
   * <code>max-keys</code> limitation. These keys are not returned elsewhere in the
   * response.</p> <p> <code>CommonPrefixes</code> is filtered out from results if it
   * is not lexicographically greater than the key-marker.</p>
   */
  inline const Aws::String& GetDelimiter() const { return m_delimiter; }
  inline bool DelimiterHasBeenSet() const { return m_delimiterHasBeenSet; }
  template <typename DelimiterT = Aws::String>
  void SetDelimiter(DelimiterT&& value) {
    m_delimiterHasBeenSet = true;
    m_delimiter = std::forward<DelimiterT>(value);
  }
  template <typename DelimiterT = Aws::String>
  ListObjectVersionsRequest& WithDelimiter(DelimiterT&& value) {
    SetDelimiter(std::forward<DelimiterT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline EncodingType GetEncodingType() const { return m_encodingType; }
  inline bool EncodingTypeHasBeenSet() const { return m_encodingTypeHasBeenSet; }
  inline void SetEncodingType(EncodingType value) {
    m_encodingTypeHasBeenSet = true;
    m_encodingType = value;
  }
  inline ListObjectVersionsRequest& WithEncodingType(EncodingType value) {
    SetEncodingType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the key to start with when listing objects in a bucket.</p>
   */
  inline const Aws::String& GetKeyMarker() const { return m_keyMarker; }
  inline bool KeyMarkerHasBeenSet() const { return m_keyMarkerHasBeenSet; }
  template <typename KeyMarkerT = Aws::String>
  void SetKeyMarker(KeyMarkerT&& value) {
    m_keyMarkerHasBeenSet = true;
    m_keyMarker = std::forward<KeyMarkerT>(value);
  }
  template <typename KeyMarkerT = Aws::String>
  ListObjectVersionsRequest& WithKeyMarker(KeyMarkerT&& value) {
    SetKeyMarker(std::forward<KeyMarkerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Sets the maximum number of keys returned in the response. By default, the
   * action returns up to 1,000 key names. The response might contain fewer keys but
   * will never contain more. If additional keys satisfy the search criteria, but
   * were not returned because <code>max-keys</code> was exceeded, the response
   * contains <code>&lt;isTruncated&gt;true&lt;/isTruncated&gt;</code>. To return the
   * additional keys, see <code>key-marker</code> and
   * <code>version-id-marker</code>.</p>
   */
  inline int GetMaxKeys() const { return m_maxKeys; }
  inline bool MaxKeysHasBeenSet() const { return m_maxKeysHasBeenSet; }
  inline void SetMaxKeys(int value) {
    m_maxKeysHasBeenSet = true;
    m_maxKeys = value;
  }
  inline ListObjectVersionsRequest& WithMaxKeys(int value) {
    SetMaxKeys(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Use this parameter to select only those keys that begin with the specified
   * prefix. You can use prefixes to separate a bucket into different groupings of
   * keys. (You can think of using <code>prefix</code> to make groups in the same way
   * that you'd use a folder in a file system.) You can use <code>prefix</code> with
   * <code>delimiter</code> to roll up numerous objects into a single result under
   * <code>CommonPrefixes</code>. </p>
   */
  inline const Aws::String& GetPrefix() const { return m_prefix; }
  inline bool PrefixHasBeenSet() const { return m_prefixHasBeenSet; }
  template <typename PrefixT = Aws::String>
  void SetPrefix(PrefixT&& value) {
    m_prefixHasBeenSet = true;
    m_prefix = std::forward<PrefixT>(value);
  }
  template <typename PrefixT = Aws::String>
  ListObjectVersionsRequest& WithPrefix(PrefixT&& value) {
    SetPrefix(std::forward<PrefixT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the object version you want to start listing from.</p>
   */
  inline const Aws::String& GetVersionIdMarker() const { return m_versionIdMarker; }
  inline bool VersionIdMarkerHasBeenSet() const { return m_versionIdMarkerHasBeenSet; }
  template <typename VersionIdMarkerT = Aws::String>
  void SetVersionIdMarker(VersionIdMarkerT&& value) {
    m_versionIdMarkerHasBeenSet = true;
    m_versionIdMarker = std::forward<VersionIdMarkerT>(value);
  }
  template <typename VersionIdMarkerT = Aws::String>
  ListObjectVersionsRequest& WithVersionIdMarker(VersionIdMarkerT&& value) {
    SetVersionIdMarker(std::forward<VersionIdMarkerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The account ID of the expected bucket owner. If the account ID that you
   * provide does not match the actual owner of the bucket, the request fails with
   * the HTTP status code <code>403 Forbidden</code> (access denied).</p>
   */
  inline const Aws::String& GetExpectedBucketOwner() const { return m_expectedBucketOwner; }
  inline bool ExpectedBucketOwnerHasBeenSet() const { return m_expectedBucketOwnerHasBeenSet; }
  template <typename ExpectedBucketOwnerT = Aws::String>
  void SetExpectedBucketOwner(ExpectedBucketOwnerT&& value) {
    m_expectedBucketOwnerHasBeenSet = true;
    m_expectedBucketOwner = std::forward<ExpectedBucketOwnerT>(value);
  }
  template <typename ExpectedBucketOwnerT = Aws::String>
  ListObjectVersionsRequest& WithExpectedBucketOwner(ExpectedBucketOwnerT&& value) {
    SetExpectedBucketOwner(std::forward<ExpectedBucketOwnerT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline RequestPayer GetRequestPayer() const { return m_requestPayer; }
  inline bool RequestPayerHasBeenSet() const { return m_requestPayerHasBeenSet; }
  inline void SetRequestPayer(RequestPayer value) {
    m_requestPayerHasBeenSet = true;
    m_requestPayer = value;
  }
  inline ListObjectVersionsRequest& WithRequestPayer(RequestPayer value) {
    SetRequestPayer(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the optional fields that you want returned in the response. Fields
   * that you do not specify are not returned.</p>
   */
  inline const Aws::Vector<OptionalObjectAttributes>& GetOptionalObjectAttributes() const { return m_optionalObjectAttributes; }
  inline bool OptionalObjectAttributesHasBeenSet() const { return m_optionalObjectAttributesHasBeenSet; }
  template <typename OptionalObjectAttributesT = Aws::Vector<OptionalObjectAttributes>>
  void SetOptionalObjectAttributes(OptionalObjectAttributesT&& value) {
    m_optionalObjectAttributesHasBeenSet = true;
    m_optionalObjectAttributes = std::forward<OptionalObjectAttributesT>(value);
  }
  template <typename OptionalObjectAttributesT = Aws::Vector<OptionalObjectAttributes>>
  ListObjectVersionsRequest& WithOptionalObjectAttributes(OptionalObjectAttributesT&& value) {
    SetOptionalObjectAttributes(std::forward<OptionalObjectAttributesT>(value));
    return *this;
  }
  inline ListObjectVersionsRequest& AddOptionalObjectAttributes(OptionalObjectAttributes value) {
    m_optionalObjectAttributesHasBeenSet = true;
    m_optionalObjectAttributes.push_back(value);
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::Map<Aws::String, Aws::String>& GetCustomizedAccessLogTag() const { return m_customizedAccessLogTag; }
  inline bool CustomizedAccessLogTagHasBeenSet() const { return m_customizedAccessLogTagHasBeenSet; }
  template <typename CustomizedAccessLogTagT = Aws::Map<Aws::String, Aws::String>>
  void SetCustomizedAccessLogTag(CustomizedAccessLogTagT&& value) {
    m_customizedAccessLogTagHasBeenSet = true;
    m_customizedAccessLogTag = std::forward<CustomizedAccessLogTagT>(value);
  }
  template <typename CustomizedAccessLogTagT = Aws::Map<Aws::String, Aws::String>>
  ListObjectVersionsRequest& WithCustomizedAccessLogTag(CustomizedAccessLogTagT&& value) {
    SetCustomizedAccessLogTag(std::forward<CustomizedAccessLogTagT>(value));
    return *this;
  }
  template <typename CustomizedAccessLogTagKeyT = Aws::String, typename CustomizedAccessLogTagValueT = Aws::String>
  ListObjectVersionsRequest& AddCustomizedAccessLogTag(CustomizedAccessLogTagKeyT&& key, CustomizedAccessLogTagValueT&& value) {
    m_customizedAccessLogTagHasBeenSet = true;
    m_customizedAccessLogTag.emplace(std::forward<CustomizedAccessLogTagKeyT>(key), std::forward<CustomizedAccessLogTagValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_bucket;
  bool m_bucketHasBeenSet = false;

  Aws::String m_delimiter;
  bool m_delimiterHasBeenSet = false;

  EncodingType m_encodingType{EncodingType::NOT_SET};
  bool m_encodingTypeHasBeenSet = false;

  Aws::String m_keyMarker;
  bool m_keyMarkerHasBeenSet = false;

  int m_maxKeys{0};
  bool m_maxKeysHasBeenSet = false;

  Aws::String m_prefix;
  bool m_prefixHasBeenSet = false;

  Aws::String m_versionIdMarker;
  bool m_versionIdMarkerHasBeenSet = false;

  Aws::String m_expectedBucketOwner;
  bool m_expectedBucketOwnerHasBeenSet = false;

  RequestPayer m_requestPayer{RequestPayer::NOT_SET};
  bool m_requestPayerHasBeenSet = false;

  Aws::Vector<OptionalObjectAttributes> m_optionalObjectAttributes;
  bool m_optionalObjectAttributesHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_customizedAccessLogTag;
  bool m_customizedAccessLogTagHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3
}  // namespace Aws
