﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/globalaccelerator/GlobalAcceleratorRequest.h>
#include <aws/globalaccelerator/GlobalAccelerator_EXPORTS.h>
#include <aws/globalaccelerator/model/PortRange.h>

#include <utility>

namespace Aws {
namespace GlobalAccelerator {
namespace Model {

/**
 */
class UpdateCustomRoutingListenerRequest : public GlobalAcceleratorRequest {
 public:
  AWS_GLOBALACCELERATOR_API UpdateCustomRoutingListenerRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateCustomRoutingListener"; }

  AWS_GLOBALACCELERATOR_API Aws::String SerializePayload() const override;

  AWS_GLOBALACCELERATOR_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the listener to update.</p>
   */
  inline const Aws::String& GetListenerArn() const { return m_listenerArn; }
  inline bool ListenerArnHasBeenSet() const { return m_listenerArnHasBeenSet; }
  template <typename ListenerArnT = Aws::String>
  void SetListenerArn(ListenerArnT&& value) {
    m_listenerArnHasBeenSet = true;
    m_listenerArn = std::forward<ListenerArnT>(value);
  }
  template <typename ListenerArnT = Aws::String>
  UpdateCustomRoutingListenerRequest& WithListenerArn(ListenerArnT&& value) {
    SetListenerArn(std::forward<ListenerArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The updated port range to support for connections from clients to your
   * accelerator. If you remove ports that are currently being used by a subnet
   * endpoint, the call fails.</p> <p>Separately, you set port ranges for endpoints.
   * For more information, see <a
   * href="https://docs.aws.amazon.com/global-accelerator/latest/dg/about-custom-routing-endpoints.html">About
   * endpoints for custom routing accelerators</a>.</p>
   */
  inline const Aws::Vector<PortRange>& GetPortRanges() const { return m_portRanges; }
  inline bool PortRangesHasBeenSet() const { return m_portRangesHasBeenSet; }
  template <typename PortRangesT = Aws::Vector<PortRange>>
  void SetPortRanges(PortRangesT&& value) {
    m_portRangesHasBeenSet = true;
    m_portRanges = std::forward<PortRangesT>(value);
  }
  template <typename PortRangesT = Aws::Vector<PortRange>>
  UpdateCustomRoutingListenerRequest& WithPortRanges(PortRangesT&& value) {
    SetPortRanges(std::forward<PortRangesT>(value));
    return *this;
  }
  template <typename PortRangesT = PortRange>
  UpdateCustomRoutingListenerRequest& AddPortRanges(PortRangesT&& value) {
    m_portRangesHasBeenSet = true;
    m_portRanges.emplace_back(std::forward<PortRangesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_listenerArn;

  Aws::Vector<PortRange> m_portRanges;
  bool m_listenerArnHasBeenSet = false;
  bool m_portRangesHasBeenSet = false;
};

}  // namespace Model
}  // namespace GlobalAccelerator
}  // namespace Aws
