/* $Id: DnDURIList.cpp $ */
/** @file
 * DnD: URI list class.
 */

/*
 * Copyright (C) 2014-2015 Oracle Corporation
 *
 * This file is part of VirtualBox Open Source Edition (OSE), as
 * available from http://www.virtualbox.org. This file is free software;
 * you can redistribute it and/or modify it under the terms of the GNU
 * General Public License (GPL) as published by the Free Software
 * Foundation, in version 2 as it comes in the "COPYING" file of the
 * VirtualBox OSE distribution. VirtualBox OSE is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY of any kind.
 */

/******************************************************************************
 *   Header Files                                                             *
 ******************************************************************************/

#include <iprt/dir.h>
#include <iprt/file.h>
#include <iprt/fs.h>
#include <iprt/path.h>
#include <iprt/symlink.h>
#include <iprt/uri.h>

#ifdef LOG_GROUP
 #undef LOG_GROUP
#endif
#define LOG_GROUP LOG_GROUP_GUEST_DND
#include <VBox/log.h>

#include <VBox/GuestHost/DragAndDrop.h>

DnDURIList::DnDURIList(void)
    : m_cTotal(0)
    , m_cbTotal(0)
{
}

DnDURIList::~DnDURIList(void)
{
}

int DnDURIList::addEntry(const char *pcszSource, const char *pcszTarget, uint32_t fFlags)
{
    AssertPtrReturn(pcszSource, VERR_INVALID_POINTER);
    AssertPtrReturn(pcszTarget, VERR_INVALID_POINTER);

    LogFlowFunc(("pcszSource=%s, pcszTarget=%s\n", pcszSource, pcszTarget));

    RTFSOBJINFO objInfo;
    int rc = RTPathQueryInfo(pcszSource, &objInfo, RTFSOBJATTRADD_NOTHING);
    if (RT_SUCCESS(rc))
    {
        if (RTFS_IS_FILE(objInfo.Attr.fMode))
        {
            LogFlowFunc(("File '%s' -> '%s'\n", pcszSource, pcszTarget));

            m_lstTree.append(DnDURIObject(DnDURIObject::File, pcszSource, pcszTarget,
                             objInfo.Attr.fMode, (uint64_t)objInfo.cbObject));
            m_cTotal++;
            m_cbTotal += (uint64_t)objInfo.cbObject;
        }
        else if (RTFS_IS_DIRECTORY(objInfo.Attr.fMode))
        {
            LogFlowFunc(("Directory '%s' -> '%s' \n", pcszSource, pcszTarget));

            m_lstTree.append(DnDURIObject(DnDURIObject::Directory, pcszSource, pcszTarget,
                             objInfo.Attr.fMode, 0 /* Size */));
            m_cTotal++;
        }
        /* Note: Symlinks already should have been resolved at this point. */
        else
            rc = VERR_NOT_SUPPORTED;
    }

    LogFlowFuncLeaveRC(rc);
    return rc;
}

int DnDURIList::appendPathRecursive(const char *pcszSrcPath,
                                    const char *pcszDstPath, const char *pcszDstBase, size_t cchDstBase, uint32_t fFlags)
{
    AssertPtrReturn(pcszSrcPath, VERR_INVALID_POINTER);
    AssertPtrReturn(pcszDstBase, VERR_INVALID_POINTER);
    AssertPtrReturn(pcszDstPath, VERR_INVALID_POINTER);

    LogFlowFunc(("pcszSrcPath=%s, pcszDstPath=%s, pcszDstBase=%s, cchDstBase=%zu\n",
                 pcszSrcPath, pcszDstPath, pcszDstBase, cchDstBase));

    RTFSOBJINFO objInfo;
    int rc = RTPathQueryInfo(pcszSrcPath, &objInfo, RTFSOBJATTRADD_NOTHING);
    if (RT_SUCCESS(rc))
    {
        if (RTFS_IS_DIRECTORY(objInfo.Attr.fMode))
        {
            rc = addEntry(pcszSrcPath, &pcszDstPath[cchDstBase], fFlags);

            PRTDIR hDir;
            if (RT_SUCCESS(rc))
                rc = RTDirOpen(&hDir, pcszSrcPath);
            if (RT_SUCCESS(rc))
            {
                do
                {
                    RTDIRENTRY DirEntry;
                    rc = RTDirRead(hDir, &DirEntry, NULL);
                    if (RT_FAILURE(rc))
                    {
                        if (rc == VERR_NO_MORE_FILES)
                            rc = VINF_SUCCESS;
                        break;
                    }

                    switch (DirEntry.enmType)
                    {
                        case RTDIRENTRYTYPE_DIRECTORY:
                        {
                            /* Skip "." and ".." entries. */
                            if (   RTStrCmp(DirEntry.szName, ".")  == 0
                                || RTStrCmp(DirEntry.szName, "..") == 0)
                                break;

                            char *pszSrc = RTPathJoinA(pcszSrcPath, DirEntry.szName);
                            if (pszSrc)
                            {
                                char *pszDst = RTPathJoinA(pcszDstPath, DirEntry.szName);
                                if (pszDst)
                                {
                                    rc = appendPathRecursive(pszSrc, pszDst, pcszDstBase, cchDstBase, fFlags);
                                    RTStrFree(pszDst);
                                }
                                else
                                    rc = VERR_NO_MEMORY;

                                RTStrFree(pszSrc);
                            }
                            else
                                rc = VERR_NO_MEMORY;
                            break;
                        }

                        case RTDIRENTRYTYPE_FILE:
                        {
                            char *pszSrc = RTPathJoinA(pcszSrcPath, DirEntry.szName);
                            if (pszSrc)
                            {
                                char *pszDst = RTPathJoinA(pcszDstPath, DirEntry.szName);
                                if (pszDst)
                                {
                                    rc = addEntry(pszSrc, &pszDst[cchDstBase], fFlags);
                                    RTStrFree(pszDst);
                                }
                                else
                                    rc = VERR_NO_MEMORY;
                                RTStrFree(pszSrc);
                            }
                            else
                                rc = VERR_NO_MEMORY;
                            break;
                        }
                        case RTDIRENTRYTYPE_SYMLINK:
                        {
                            if (fFlags & DNDURILIST_FLAGS_RESOLVE_SYMLINKS)
                            {
                                char *pszSrc = RTPathRealDup(pcszDstBase);
                                if (pszSrc)
                                {
                                    rc = RTPathQueryInfo(pszSrc, &objInfo, RTFSOBJATTRADD_NOTHING);
                                    if (RT_SUCCESS(rc))
                                    {
                                        if (RTFS_IS_DIRECTORY(objInfo.Attr.fMode))
                                        {
                                            LogFlowFunc(("Directory entry is symlink to directory\n"));
                                            rc = appendPathRecursive(pszSrc, pcszDstPath, pcszDstBase, cchDstBase, fFlags);
                                        }
                                        else if (RTFS_IS_FILE(objInfo.Attr.fMode))
                                        {
                                            LogFlowFunc(("Directory entry is symlink to file\n"));
                                            rc = addEntry(pszSrc, &pcszDstPath[cchDstBase], fFlags);
                                        }
                                        else
                                            rc = VERR_NOT_SUPPORTED;
                                    }

                                    RTStrFree(pszSrc);
                                }
                                else
                                    rc = VERR_NO_MEMORY;
                            }
                            break;
                        }

                        default:
                            break;
                    }

                } while (RT_SUCCESS(rc));

                RTDirClose(hDir);
            }
        }
        else if (RTFS_IS_FILE(objInfo.Attr.fMode))
        {
            rc = addEntry(pcszSrcPath, &pcszDstPath[cchDstBase], fFlags);
        }
        else if (RTFS_IS_SYMLINK(objInfo.Attr.fMode))
        {
            if (fFlags & DNDURILIST_FLAGS_RESOLVE_SYMLINKS)
            {
                char *pszSrc = RTPathRealDup(pcszSrcPath);
                if (pszSrc)
                {
                    rc = RTPathQueryInfo(pszSrc, &objInfo, RTFSOBJATTRADD_NOTHING);
                    if (RT_SUCCESS(rc))
                    {
                        if (RTFS_IS_DIRECTORY(objInfo.Attr.fMode))
                        {
                            LogFlowFunc(("Symlink to directory\n"));
                            rc = appendPathRecursive(pszSrc, pcszDstPath, pcszDstBase, cchDstBase, fFlags);
                        }
                        else if (RTFS_IS_FILE(objInfo.Attr.fMode))
                        {
                            LogFlowFunc(("Symlink to file\n"));
                            rc = addEntry(pszSrc, &pcszDstPath[cchDstBase], fFlags);
                        }
                        else
                            rc = VERR_NOT_SUPPORTED;
                    }

                    RTStrFree(pszSrc);
                }
                else
                    rc = VERR_NO_MEMORY;
            }
        }
        else
            rc = VERR_NOT_SUPPORTED;
    }

    LogFlowFuncLeaveRC(rc);
    return rc;
}

int DnDURIList::AppendNativePath(const char *pszPath, uint32_t fFlags)
{
    AssertPtrReturn(pszPath, VERR_INVALID_POINTER);

    int rc;
    char *pszPathNative = RTStrDup(pszPath);
    if (pszPathNative)
    {
        RTPathChangeToUnixSlashes(pszPathNative, true /* fForce */);

        char *pszPathURI = RTUriCreate("file" /* pszScheme */, "/" /* pszAuthority */,
                                       pszPathNative, NULL /* pszQuery */, NULL /* pszFragment */);
        if (pszPathURI)
        {
            rc = AppendURIPath(pszPathURI, fFlags);
            RTStrFree(pszPathURI);
        }
        else
            rc = VERR_INVALID_PARAMETER;

        RTStrFree(pszPathNative);
    }
    else
        rc = VERR_NO_MEMORY;

    return rc;
}

int DnDURIList::AppendNativePathsFromList(const char *pszNativePaths, size_t cbNativePaths,
                                          uint32_t fFlags)
{
    AssertPtrReturn(pszNativePaths, VERR_INVALID_POINTER);
    AssertReturn(cbNativePaths, VERR_INVALID_PARAMETER);

    RTCList<RTCString> lstPaths
        = RTCString(pszNativePaths, cbNativePaths - 1).split("\r\n");
    return AppendNativePathsFromList(lstPaths, fFlags);
}

int DnDURIList::AppendNativePathsFromList(const RTCList<RTCString> &lstNativePaths,
                                          uint32_t fFlags)
{
    int rc = VINF_SUCCESS;

    for (size_t i = 0; i < lstNativePaths.size(); i++)
    {
        const RTCString &strPath = lstNativePaths.at(i);
        rc = AppendNativePath(strPath.c_str(), fFlags);
        if (RT_FAILURE(rc))
            break;
    }

    LogFlowFuncLeaveRC(rc);
    return rc;
}

int DnDURIList::AppendURIPath(const char *pszURI, uint32_t fFlags)
{
    AssertPtrReturn(pszURI, VERR_INVALID_POINTER);

    /** @todo Check for string termination?  */
#ifdef DEBUG_andy
    LogFlowFunc(("pszPath=%s, fFlags=0x%x\n", pszURI, fFlags));
#endif
    int rc = VINF_SUCCESS;

    /* Query the path component of a file URI. If this hasn't a
     * file scheme NULL is returned. */
    char *pszSrcPath = RTUriFilePath(pszURI, URI_FILE_FORMAT_AUTO);
    if (pszSrcPath)
    {
        /* Add the path to our internal file list (recursive in
         * the case of a directory). */
        size_t cbPathLen = RTPathStripTrailingSlash(pszSrcPath);
        if (cbPathLen)
        {
            char *pszFileName = RTPathFilename(pszSrcPath);
            if (pszFileName)
            {
                Assert(pszFileName >= pszSrcPath);
                size_t cchDstBase = (fFlags & DNDURILIST_FLAGS_ABSOLUTE_PATHS)
                                  ? 0 /* Use start of path as root. */
                                  : pszFileName - pszSrcPath;
                char *pszDstPath = &pszSrcPath[cchDstBase];
                m_lstRoot.append(pszDstPath);

                LogFlowFunc(("pszFilePath=%s, pszFileName=%s, pszRoot=%s\n",
                             pszSrcPath, pszFileName, pszDstPath));

                rc = appendPathRecursive(pszSrcPath, pszSrcPath, pszSrcPath, cchDstBase, fFlags);
            }
            else
                rc = VERR_PATH_NOT_FOUND;
        }
        else
            rc = VERR_INVALID_PARAMETER;

        RTStrFree(pszSrcPath);
    }
    else
        rc = VERR_INVALID_PARAMETER;

    LogFlowFuncLeaveRC(rc);
    return rc;
}

int DnDURIList::AppendURIPathsFromList(const char *pszURIPaths, size_t cbURIPaths,
                                       uint32_t fFlags)
{
    AssertPtrReturn(pszURIPaths, VERR_INVALID_POINTER);
    AssertReturn(cbURIPaths, VERR_INVALID_PARAMETER);

    RTCList<RTCString> lstPaths
        = RTCString(pszURIPaths, cbURIPaths - 1).split("\r\n");
    return AppendURIPathsFromList(lstPaths, fFlags);
}

int DnDURIList::AppendURIPathsFromList(const RTCList<RTCString> &lstURI,
                                       uint32_t fFlags)
{
    int rc = VINF_SUCCESS;

    for (size_t i = 0; i < lstURI.size(); i++)
    {
        RTCString strURI = lstURI.at(i);
        rc = AppendURIPath(strURI.c_str(), fFlags);

        if (RT_FAILURE(rc))
            break;
    }

    LogFlowFuncLeaveRC(rc);
    return rc;
}

void DnDURIList::Clear(void)
{
    m_lstRoot.clear();
    m_lstTree.clear();

    m_cbTotal = 0;
}

void DnDURIList::RemoveFirst(void)
{
    if (m_lstTree.isEmpty())
        return;

    DnDURIObject &curPath = m_lstTree.first();

    uint64_t cbSize = curPath.GetSize();
    Assert(m_cbTotal >= cbSize);
    m_cbTotal -= cbSize; /* Adjust total size. */

    m_lstTree.removeFirst();
}

int DnDURIList::RootFromURIData(const void *pvData, size_t cbData, uint32_t fFlags)
{
    AssertPtrReturn(pvData, VERR_INVALID_POINTER);
    AssertReturn(cbData, VERR_INVALID_PARAMETER);

    RTCList<RTCString> lstURI =
        RTCString(static_cast<const char*>(pvData), cbData - 1).split("\r\n");
    if (lstURI.isEmpty())
        return VINF_SUCCESS;

    int rc = VINF_SUCCESS;

    for (size_t i = 0; i < lstURI.size(); ++i)
    {
        /* Query the path component of a file URI. If this hasn't a
         * file scheme, NULL is returned. */
        const char *pszURI = lstURI.at(i).c_str();
        char *pszFilePath = RTUriFilePath(pszURI,
                                          URI_FILE_FORMAT_AUTO);
#ifdef DEBUG_andy
        LogFlowFunc(("pszURI=%s, pszFilePath=%s\n", pszURI, pszFilePath));
#endif
        if (pszFilePath)
        {
            rc = DnDPathSanitize(pszFilePath, strlen(pszFilePath));
            if (RT_SUCCESS(rc))
            {
                m_lstRoot.append(pszFilePath);
                m_cTotal++;
            }

            RTStrFree(pszFilePath);
        }
        else
            rc = VERR_INVALID_PARAMETER;

        if (RT_FAILURE(rc))
            break;
    }

    return rc;
}

RTCString DnDURIList::RootToString(const RTCString &strPathBase /* = "" */,
                                   const RTCString &strSeparator /* = "\r\n" */)
{
    RTCString strRet;
    for (size_t i = 0; i < m_lstRoot.size(); i++)
    {
        const char *pszCurRoot = m_lstRoot.at(i).c_str();
#ifdef DEBUG_andy
        LogFlowFunc(("pszCurRoot=%s\n", pszCurRoot));
#endif
        if (strPathBase.isNotEmpty())
        {
            char *pszPath = RTPathJoinA(strPathBase.c_str(), pszCurRoot);
            if (pszPath)
            {
                char *pszPathURI = RTUriFileCreate(pszPath);
                if (pszPathURI)
                {
                    strRet += RTCString(pszPathURI) + strSeparator;
                    LogFlowFunc(("URI: %s\n", strRet.c_str()));
                    RTStrFree(pszPathURI);
                }

                RTStrFree(pszPath);

                if (!pszPathURI)
                    break;
            }
            else
                break;
        }
        else
        {
            char *pszPathURI = RTUriFileCreate(pszCurRoot);
            if (pszPathURI)
            {
                strRet += RTCString(pszPathURI) + strSeparator;
                LogFlowFunc(("URI: %s\n", strRet.c_str()));
                RTStrFree(pszPathURI);
            }
            else
                break;
        }
    }

    return strRet;
}

